// Copyright 2011 The Go Authors. All rights reserved. BSD license.
// https://github.com/golang/go/blob/master/LICENSE
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Port of the Go
 * [encoding/csv](https://github.com/golang/go/blob/go1.12.5/src/encoding/csv/)
 * library.
 *
 * @module
 */ import { assert } from "../_util/assert.ts";
import { Parser } from "./csv/_parser.ts";
export { ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError } from "./csv/_io.ts";
const QUOTE = '"';
export const NEWLINE = "\r\n";
export class StringifyError extends Error {
    name = "StringifyError";
}
function getEscapedString(value, sep) {
    if (value === undefined || value === null) return "";
    let str = "";
    if (typeof value === "object") str = JSON.stringify(value);
    else str = String(value);
    // Is regex.test more performant here? If so, how to dynamically create?
    // https://stackoverflow.com/questions/3561493/
    if (str.includes(sep) || str.includes(NEWLINE) || str.includes(QUOTE)) {
        return `${QUOTE}${str.replaceAll(QUOTE, `${QUOTE}${QUOTE}`)}${QUOTE}`;
    }
    return str;
}
function normalizeColumn(column) {
    let fn, header, prop;
    if (typeof column === "object") {
        if (Array.isArray(column)) {
            header = String(column[column.length - 1]);
            prop = column;
        } else {
            ({ fn  } = column);
            prop = Array.isArray(column.prop) ? column.prop : [
                column.prop
            ];
            header = typeof column.header === "string" ? column.header : String(prop[prop.length - 1]);
        }
    } else {
        header = String(column);
        prop = [
            column
        ];
    }
    return {
        fn,
        header,
        prop
    };
}
/**
 * Returns an array of values from an object using the property accessors
 * (and optional transform function) in each column
 */ async function getValuesFromItem(item, normalizedColumns) {
    const values = [];
    for (const column of normalizedColumns){
        let value = item;
        for (const prop of column.prop){
            if (typeof value !== "object" || value === null) continue;
            if (Array.isArray(value)) {
                if (typeof prop === "number") value = value[prop];
                else {
                    throw new StringifyError('Property accessor is not of type "number"');
                }
            } else value = value[prop];
        }
        if (typeof column.fn === "function") value = await column.fn(value);
        values.push(value);
    }
    return values;
}
/**
 * @param data The array of objects to encode
 * @param columns Array of values specifying which data to include in the output
 * @param options Output formatting options
 */ export async function stringify(data, columns, options = {}) {
    const { headers , separator: sep  } = {
        headers: true,
        separator: ",",
        ...options
    };
    if (sep.includes(QUOTE) || sep.includes(NEWLINE)) {
        const message = [
            "Separator cannot include the following strings:",
            '  - U+0022: Quotation mark (")',
            "  - U+000D U+000A: Carriage Return + Line Feed (\\r\\n)", 
        ].join("\n");
        throw new StringifyError(message);
    }
    const normalizedColumns = columns.map(normalizeColumn);
    let output = "";
    if (headers) {
        output += normalizedColumns.map((column)=>getEscapedString(column.header, sep)).join(sep);
        output += NEWLINE;
    }
    for (const item of data){
        const values = await getValuesFromItem(item, normalizedColumns);
        output += values.map((value)=>getEscapedString(value, sep)).join(sep);
        output += NEWLINE;
    }
    return output;
}
export function parse(input, opt = {
    skipFirstRow: false
}) {
    const parser = new Parser(opt);
    const r = parser.parse(input);
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head.map((e)=>{
                return {
                    name: e
                };
            });
            i++;
        }
        if (opt.columns) {
            if (typeof opt.columns[0] !== "string") {
                headers = opt.columns;
            } else {
                const h = opt.columns;
                headers = h.map((e)=>{
                    return {
                        name: e
                    };
                });
            }
        }
        return r.map((e)=>{
            if (e.length !== headers.length) {
                throw new Error(`Error number of fields line: ${i}\nNumber of fields found: ${headers.length}\nExpected number of fields: ${e.length}`);
            }
            i++;
            const out = {};
            for(let j = 0; j < e.length; j++){
                out[headers[j].name] = e[j];
            }
            return out;
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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