// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { getLevelByName, LogLevels } from "./levels.ts";
import { blue, bold, red, yellow } from "../fmt/colors.ts";
import { exists, existsSync } from "../fs/exists.ts";
import { BufWriterSync } from "../io/buffer.ts";
const DEFAULT_FORMATTER = "{levelName} {msg}";
export class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {}){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{([^\s}]+)}/g, (match, p1)=>{
            const value = logRecord[p1];
            // do not interpolate missing values
            if (value == null) {
                return match;
            }
            return String(value);
        });
    }
    log(_msg) {}
    async setup() {}
    async destroy() {}
}
export class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow(msg);
                break;
            case LogLevels.ERROR:
                msg = red(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold(red(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
export class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
export class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        // default to append mode, write only
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    async setup() {
        this._file = await Deno.open(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        // Immediately flush if log level is higher than ERROR
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        if (this._encoder.encode(msg).byteLength + 1 > this._buf.available()) {
            this.flush();
        }
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
        return Promise.resolve();
    }
}
export class RotatingFileHandler extends FileHandler {
    #maxBytes;
    #maxBackupCount;
    #currentFileSize = 0;
    constructor(levelName, options){
        super(levelName, options);
        this.#maxBytes = options.maxBytes;
        this.#maxBackupCount = options.maxBackupCount;
    }
    async setup() {
        if (this.#maxBytes < 1) {
            this.destroy();
            throw new Error("maxBytes cannot be less than 1");
        }
        if (this.#maxBackupCount < 1) {
            this.destroy();
            throw new Error("maxBackupCount cannot be less than 1");
        }
        await super.setup();
        if (this._mode === "w") {
            // Remove old backups too as it doesn't make sense to start with a clean
            // log file, but old backups
            for(let i = 1; i <= this.#maxBackupCount; i++){
                if (await exists(this._filename + "." + i)) {
                    await Deno.remove(this._filename + "." + i);
                }
            }
        } else if (this._mode === "x") {
            // Throw if any backups also exist
            for(let i1 = 1; i1 <= this.#maxBackupCount; i1++){
                if (await exists(this._filename + "." + i1)) {
                    this.destroy();
                    throw new Deno.errors.AlreadyExists("Backup log file " + this._filename + "." + i1 + " already exists");
                }
            }
        } else {
            this.#currentFileSize = (await Deno.stat(this._filename)).size;
        }
    }
    log(msg) {
        const msgByteLength = this._encoder.encode(msg).byteLength + 1;
        if (this.#currentFileSize + msgByteLength > this.#maxBytes) {
            this.rotateLogFiles();
            this.#currentFileSize = 0;
        }
        super.log(msg);
        this.#currentFileSize += msgByteLength;
    }
    rotateLogFiles() {
        this._buf.flush();
        Deno.close(this._file.rid);
        for(let i = this.#maxBackupCount - 1; i >= 0; i--){
            const source = this._filename + (i === 0 ? "" : "." + i);
            const dest = this._filename + "." + (i + 1);
            if (existsSync(source)) {
                Deno.renameSync(source, dest);
            }
        }
        this._file = Deno.openSync(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
    }
}
//# sourceMappingURL=data:application/json;base64,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