import { Type } from "../type.ts";
import { isNegativeZero } from "../utils.ts";
function isHexCode(c) {
    return ((0x30 <= c && c <= 0x39) ||
        (0x41 <= c && c <= 0x46) ||
        (0x61 <= c && c <= 0x66));
}
function isOctCode(c) {
    return 0x30 <= c && c <= 0x37;
}
function isDecCode(c) {
    return 0x30 <= c && c <= 0x39;
}
function resolveYamlInteger(data) {
    const max = data.length;
    let index = 0;
    let hasDigits = false;
    if (!max)
        return false;
    let ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max)
            return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for (; index < max; index++) {
                ch = data[index];
                if (ch === "_")
                    continue;
                if (ch !== "0" && ch !== "1")
                    return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for (; index < max; index++) {
                ch = data[index];
                if (ch === "_")
                    continue;
                if (!isHexCode(data.charCodeAt(index)))
                    return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for (; index < max; index++) {
            ch = data[index];
            if (ch === "_")
                continue;
            if (!isOctCode(data.charCodeAt(index)))
                return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_")
        return false;
    for (; index < max; index++) {
        ch = data[index];
        if (ch === "_")
            continue;
        if (ch === ":")
            break;
        if (!isDecCode(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_")
        return false;
    if (ch !== ":")
        return true;
    return /^(:[0-5]?[0-9])+$/.test(data.slice(index));
}
function constructYamlInteger(data) {
    let value = data;
    const digits = [];
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-")
            sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0")
        return 0;
    if (ch === "0") {
        if (value[1] === "b")
            return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x")
            return sign * parseInt(value, 16);
        return sign * parseInt(value, 8);
    }
    if (value.indexOf(":") !== -1) {
        value.split(":").forEach((v) => {
            digits.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base = 1;
        digits.forEach((d) => {
            valueInt += d * base;
            base *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value, 10);
}
function isInteger(object) {
    return (Object.prototype.toString.call(object) === "[object Number]" &&
        object % 1 === 0 &&
        !isNegativeZero(object));
}
export const int = new Type("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger,
    represent: {
        binary(obj) {
            return obj >= 0
                ? `0b${obj.toString(2)}`
                : `-0b${obj.toString(2).slice(1)}`;
        },
        octal(obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal(obj) {
            return obj.toString(10);
        },
        hexadecimal(obj) {
            return obj >= 0
                ? `0x${obj.toString(16).toUpperCase()}`
                : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        },
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [2, "bin"],
        decimal: [10, "dec"],
        hexadecimal: [16, "hex"],
        octal: [8, "oct"],
    },
});
//# sourceMappingURL=data:application/json;base64,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