import { MultiReader } from "../io/readers.ts";
import { Buffer, PartialReadError } from "../io/buffer.ts";
import { assert } from "../_util/assert.ts";
import { readAll } from "../streams/conversion.ts";
const recordSize = 512;
const ustar = "ustar\u000000";
const initialChecksum = 8 * 32;
async function readBlock(reader, p) {
    let bytesRead = 0;
    while (bytesRead < p.length) {
        const rr = await reader.read(p.subarray(bytesRead));
        if (rr === null) {
            if (bytesRead === 0) {
                return null;
            }
            else {
                throw new PartialReadError();
            }
        }
        bytesRead += rr;
    }
    return bytesRead;
}
class FileReader {
    filePath;
    file;
    constructor(filePath) {
        this.filePath = filePath;
    }
    async read(p) {
        if (!this.file) {
            this.file = await Deno.open(this.filePath, { read: true });
        }
        const res = await Deno.read(this.file.rid, p);
        if (res === null) {
            Deno.close(this.file.rid);
            this.file = undefined;
        }
        return res;
    }
}
function trim(buffer) {
    const index = buffer.findIndex((v) => v === 0);
    if (index < 0)
        return buffer;
    return buffer.subarray(0, index);
}
function clean(length) {
    const buffer = new Uint8Array(length);
    buffer.fill(0, 0, length - 1);
    return buffer;
}
function pad(num, bytes, base = 8) {
    const numString = num.toString(base);
    return "000000000000".substr(numString.length + 12 - bytes) + numString;
}
var FileTypes;
(function (FileTypes) {
    FileTypes[FileTypes["file"] = 0] = "file";
    FileTypes[FileTypes["link"] = 1] = "link";
    FileTypes[FileTypes["symlink"] = 2] = "symlink";
    FileTypes[FileTypes["character-device"] = 3] = "character-device";
    FileTypes[FileTypes["block-device"] = 4] = "block-device";
    FileTypes[FileTypes["directory"] = 5] = "directory";
    FileTypes[FileTypes["fifo"] = 6] = "fifo";
    FileTypes[FileTypes["contiguous-file"] = 7] = "contiguous-file";
})(FileTypes || (FileTypes = {}));
const ustarStructure = [
    {
        field: "fileName",
        length: 100,
    },
    {
        field: "fileMode",
        length: 8,
    },
    {
        field: "uid",
        length: 8,
    },
    {
        field: "gid",
        length: 8,
    },
    {
        field: "fileSize",
        length: 12,
    },
    {
        field: "mtime",
        length: 12,
    },
    {
        field: "checksum",
        length: 8,
    },
    {
        field: "type",
        length: 1,
    },
    {
        field: "linkName",
        length: 100,
    },
    {
        field: "ustar",
        length: 8,
    },
    {
        field: "owner",
        length: 32,
    },
    {
        field: "group",
        length: 32,
    },
    {
        field: "majorNumber",
        length: 8,
    },
    {
        field: "minorNumber",
        length: 8,
    },
    {
        field: "fileNamePrefix",
        length: 155,
    },
    {
        field: "padding",
        length: 12,
    },
];
function formatHeader(data) {
    const encoder = new TextEncoder(), buffer = clean(512);
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const entry = encoder.encode(data[value.field] || "");
        buffer.set(entry, offset);
        offset += value.length;
    });
    return buffer;
}
function parseHeader(buffer) {
    const data = {};
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const arr = buffer.subarray(offset, offset + value.length);
        data[value.field] = arr;
        offset += value.length;
    });
    return data;
}
export class Tar {
    data;
    constructor() {
        this.data = [];
    }
    async append(fn, opts) {
        if (typeof fn !== "string") {
            throw new Error("file name not specified");
        }
        let fileName = fn;
        let fileNamePrefix;
        if (fileName.length > 100) {
            let i = fileName.length;
            while (i >= 0) {
                i = fileName.lastIndexOf("/", i);
                if (i <= 155) {
                    fileNamePrefix = fileName.substr(0, i);
                    fileName = fileName.substr(i + 1);
                    break;
                }
                i--;
            }
            const errMsg = "ustar format does not allow a long file name (length of [file name" +
                "prefix] + / + [file name] must be shorter than 256 bytes)";
            if (i < 0 || fileName.length > 100) {
                throw new Error(errMsg);
            }
            else {
                assert(fileNamePrefix != null);
                if (fileNamePrefix.length > 155) {
                    throw new Error(errMsg);
                }
            }
        }
        opts = opts || {};
        let info;
        if (opts.filePath) {
            info = await Deno.stat(opts.filePath);
            if (info.isDirectory) {
                info.size = 0;
                opts.reader = new Buffer();
            }
        }
        const mode = opts.fileMode || (info && info.mode) ||
            parseInt("777", 8) & 0xfff, mtime = Math.floor(opts.mtime ?? (info?.mtime ?? new Date()).valueOf() / 1000), uid = opts.uid || 0, gid = opts.gid || 0;
        if (typeof opts.owner === "string" && opts.owner.length >= 32) {
            throw new Error("ustar format does not allow owner name length >= 32 bytes");
        }
        if (typeof opts.group === "string" && opts.group.length >= 32) {
            throw new Error("ustar format does not allow group name length >= 32 bytes");
        }
        const fileSize = info?.size ?? opts.contentSize;
        assert(fileSize != null, "fileSize must be set");
        const type = opts.type
            ? FileTypes[opts.type]
            : (info?.isDirectory ? FileTypes.directory : FileTypes.file);
        const tarData = {
            fileName,
            fileNamePrefix,
            fileMode: pad(mode, 7),
            uid: pad(uid, 7),
            gid: pad(gid, 7),
            fileSize: pad(fileSize, 11),
            mtime: pad(mtime, 11),
            checksum: "        ",
            type: type.toString(),
            ustar,
            owner: opts.owner || "",
            group: opts.group || "",
            filePath: opts.filePath,
            reader: opts.reader,
        };
        let checksum = 0;
        const encoder = new TextEncoder();
        Object.keys(tarData)
            .filter((key) => ["filePath", "reader"].indexOf(key) < 0)
            .forEach(function (key) {
            checksum += encoder
                .encode(tarData[key])
                .reduce((p, c) => p + c, 0);
        });
        tarData.checksum = pad(checksum, 6) + "\u0000 ";
        this.data.push(tarData);
    }
    getReader() {
        const readers = [];
        this.data.forEach((tarData) => {
            let { reader } = tarData;
            const { filePath } = tarData;
            const headerArr = formatHeader(tarData);
            readers.push(new Buffer(headerArr));
            if (!reader) {
                assert(filePath != null);
                reader = new FileReader(filePath);
            }
            readers.push(reader);
            assert(tarData.fileSize != null, "fileSize must be set");
            readers.push(new Buffer(clean(recordSize -
                (parseInt(tarData.fileSize, 8) % recordSize || recordSize))));
        });
        readers.push(new Buffer(clean(recordSize * 2)));
        return new MultiReader(readers);
    }
}
class TarEntry {
    #header;
    #reader;
    #size;
    #read = 0;
    #consumed = false;
    #entrySize;
    constructor(meta, header, reader) {
        Object.assign(this, meta);
        this.#header = header;
        this.#reader = reader;
        this.#size = this.fileSize || 0;
        const blocks = Math.ceil(this.#size / recordSize);
        this.#entrySize = blocks * recordSize;
    }
    get consumed() {
        return this.#consumed;
    }
    async read(p) {
        const entryBytesLeft = this.#entrySize - this.#read;
        const bufSize = Math.min(p.length, entryBytesLeft);
        if (entryBytesLeft <= 0) {
            this.#consumed = true;
            return null;
        }
        const block = new Uint8Array(bufSize);
        const n = await readBlock(this.#reader, block);
        const bytesLeft = this.#size - this.#read;
        this.#read += n || 0;
        if (n === null || bytesLeft <= 0) {
            if (n === null)
                this.#consumed = true;
            return null;
        }
        const offset = bytesLeft < n ? bytesLeft : n;
        p.set(block.subarray(0, offset), 0);
        return offset < 0 ? n - Math.abs(offset) : offset;
    }
    async discard() {
        if (this.#consumed)
            return;
        this.#consumed = true;
        if (typeof this.#reader.seek === "function") {
            await this.#reader.seek(this.#entrySize - this.#read, Deno.SeekMode.Current);
            this.#read = this.#entrySize;
        }
        else {
            await readAll(this);
        }
    }
}
export class Untar {
    reader;
    block;
    #entry;
    constructor(reader) {
        this.reader = reader;
        this.block = new Uint8Array(recordSize);
    }
    #checksum(header) {
        let sum = initialChecksum;
        for (let i = 0; i < 512; i++) {
            if (i >= 148 && i < 156) {
                continue;
            }
            sum += header[i];
        }
        return sum;
    }
    async #getHeader() {
        await readBlock(this.reader, this.block);
        const header = parseHeader(this.block);
        const decoder = new TextDecoder();
        const checksum = this.#checksum(this.block);
        if (parseInt(decoder.decode(header.checksum), 8) !== checksum) {
            if (checksum === initialChecksum) {
                return null;
            }
            throw new Error("checksum error");
        }
        const magic = decoder.decode(header.ustar);
        if (magic.indexOf("ustar")) {
            throw new Error(`unsupported archive format: ${magic}`);
        }
        return header;
    }
    #getMetadata(header) {
        const decoder = new TextDecoder();
        const meta = {
            fileName: decoder.decode(trim(header.fileName)),
        };
        const fileNamePrefix = trim(header.fileNamePrefix);
        if (fileNamePrefix.byteLength > 0) {
            meta.fileName = decoder.decode(fileNamePrefix) + "/" + meta.fileName;
        }
        ["fileMode", "mtime", "uid", "gid"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = parseInt(decoder.decode(arr), 8);
            }
        });
        ["owner", "group", "type"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = decoder.decode(arr);
            }
        });
        meta.fileSize = parseInt(decoder.decode(header.fileSize), 8);
        meta.type = FileTypes[parseInt(meta.type)] ?? meta.type;
        return meta;
    }
    async extract() {
        if (this.#entry && !this.#entry.consumed) {
            await this.#entry.discard();
        }
        const header = await this.#getHeader();
        if (header === null)
            return null;
        const meta = this.#getMetadata(header);
        this.#entry = new TarEntry(meta, header, this.reader);
        return this.#entry;
    }
    async *[Symbol.asyncIterator]() {
        while (true) {
            const entry = await this.extract();
            if (entry === null)
                return;
            yield entry;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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