import { equals, indexOfNeedle, lastIndexOfNeedle, startsWith, } from "../bytes/mod.ts";
import { Buffer, BufReader, BufWriter } from "../io/buffer.ts";
import { copy } from "../streams/conversion.ts";
import { copyN } from "../io/util.ts";
import { MultiReader } from "../io/readers.ts";
import { extname } from "../path/mod.ts";
import { assert } from "../_util/assert.ts";
import { TextProtoReader } from "../textproto/mod.ts";
const { hasOwn } = Object;
export function isFormFile(x) {
    return hasOwn(x, "filename") && hasOwn(x, "type");
}
function randomBoundary() {
    let boundary = "--------------------------";
    for (let i = 0; i < 24; i++) {
        boundary += Math.floor(Math.random() * 16).toString(16);
    }
    return boundary;
}
const encoder = new TextEncoder();
export function matchAfterPrefix(buf, prefix, eof) {
    if (buf.length === prefix.length) {
        return eof ? 1 : 0;
    }
    const c = buf[prefix.length];
    if (c === " ".charCodeAt(0) ||
        c === "\t".charCodeAt(0) ||
        c === "\r".charCodeAt(0) ||
        c === "\n".charCodeAt(0) ||
        c === "-".charCodeAt(0)) {
        return 1;
    }
    return -1;
}
export function scanUntilBoundary(buf, dashBoundary, newLineDashBoundary, total, eof) {
    if (total === 0) {
        if (startsWith(buf, dashBoundary)) {
            switch (matchAfterPrefix(buf, dashBoundary, eof)) {
                case -1:
                    return dashBoundary.length;
                case 0:
                    return 0;
                case 1:
                    return null;
            }
        }
        if (startsWith(dashBoundary, buf)) {
            return 0;
        }
    }
    const i = indexOfNeedle(buf, newLineDashBoundary);
    if (i >= 0) {
        switch (matchAfterPrefix(buf.slice(i), newLineDashBoundary, eof)) {
            case -1:
                return i + newLineDashBoundary.length;
            case 0:
                return i;
            case 1:
                return i > 0 ? i : null;
        }
    }
    if (startsWith(newLineDashBoundary, buf)) {
        return 0;
    }
    const j = lastIndexOfNeedle(buf, newLineDashBoundary.slice(0, 1));
    if (j >= 0 && startsWith(newLineDashBoundary, buf.slice(j))) {
        return j;
    }
    return buf.length;
}
class PartReader {
    mr;
    headers;
    n = 0;
    total = 0;
    constructor(mr, headers) {
        this.mr = mr;
        this.headers = headers;
    }
    async read(p) {
        const br = this.mr.bufReader;
        let peekLength = 1;
        while (this.n === 0) {
            peekLength = Math.max(peekLength, br.buffered());
            const peekBuf = await br.peek(peekLength);
            if (peekBuf === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            const eof = peekBuf.length < peekLength;
            this.n = scanUntilBoundary(peekBuf, this.mr.dashBoundary, this.mr.newLineDashBoundary, this.total, eof);
            if (this.n === 0) {
                assert(eof === false);
                peekLength++;
            }
        }
        if (this.n === null) {
            return null;
        }
        const nread = Math.min(p.length, this.n);
        const buf = p.subarray(0, nread);
        const r = await br.readFull(buf);
        assert(r === buf);
        this.n -= nread;
        this.total += nread;
        return nread;
    }
    close() { }
    contentDisposition;
    contentDispositionParams;
    getContentDispositionParams() {
        if (this.contentDispositionParams)
            return this.contentDispositionParams;
        const cd = this.headers.get("content-disposition");
        const params = {};
        assert(cd != null, "content-disposition must be set");
        const comps = decodeURI(cd).split(";");
        this.contentDisposition = comps[0];
        comps
            .slice(1)
            .map((v) => v.trim())
            .map((kv) => {
            const [k, v] = kv.split("=");
            if (v) {
                const s = v.charAt(0);
                const e = v.charAt(v.length - 1);
                if ((s === e && s === '"') || s === "'") {
                    params[k] = v.substr(1, v.length - 2);
                }
                else {
                    params[k] = v;
                }
            }
        });
        return (this.contentDispositionParams = params);
    }
    get fileName() {
        return this.getContentDispositionParams()["filename"];
    }
    get formName() {
        const p = this.getContentDispositionParams();
        if (this.contentDisposition === "form-data") {
            return p["name"];
        }
        return "";
    }
}
function skipLWSPChar(u) {
    const ret = new Uint8Array(u.length);
    const sp = " ".charCodeAt(0);
    const ht = "\t".charCodeAt(0);
    let j = 0;
    for (let i = 0; i < u.length; i++) {
        if (u[i] === sp || u[i] === ht)
            continue;
        ret[j++] = u[i];
    }
    return ret.slice(0, j);
}
export class MultipartReader {
    boundary;
    newLine;
    newLineDashBoundary;
    dashBoundaryDash;
    dashBoundary;
    bufReader;
    constructor(reader, boundary) {
        this.boundary = boundary;
        this.newLine = encoder.encode("\r\n");
        this.newLineDashBoundary = encoder.encode(`\r\n--${boundary}`);
        this.dashBoundaryDash = encoder.encode(`--${this.boundary}--`);
        this.dashBoundary = encoder.encode(`--${this.boundary}`);
        this.bufReader = new BufReader(reader);
    }
    async readForm(maxMemoryOrOptions) {
        const options = typeof maxMemoryOrOptions === "number"
            ? { maxMemory: maxMemoryOrOptions }
            : maxMemoryOrOptions;
        let maxMemory = options?.maxMemory ?? 10 << 20;
        const fileMap = new Map();
        const valueMap = new Map();
        let maxValueBytes = maxMemory + (10 << 20);
        const buf = new Buffer(new Uint8Array(maxValueBytes));
        for (;;) {
            const p = await this.nextPart();
            if (p === null) {
                break;
            }
            if (p.formName === "") {
                continue;
            }
            buf.reset();
            if (!p.fileName) {
                const n = await copyN(p, buf, maxValueBytes);
                maxValueBytes -= n;
                if (maxValueBytes < 0) {
                    throw new RangeError("message too large");
                }
                const value = new TextDecoder().decode(buf.bytes());
                const mapVal = valueMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(value);
                }
                else {
                    valueMap.set(p.formName, [value]);
                }
                continue;
            }
            let formFile;
            const n = await copyN(p, buf, maxValueBytes);
            const contentType = p.headers.get("content-type");
            assert(contentType != null, "content-type must be set");
            if (n > maxMemory) {
                const ext = extname(p.fileName);
                const filepath = await Deno.makeTempFile({
                    dir: options?.dir ?? ".",
                    prefix: options?.prefix ?? "multipart-",
                    suffix: options?.suffix ?? ext,
                });
                const file = await Deno.open(filepath, { write: true });
                try {
                    const size = await copy(new MultiReader([buf, p]), file);
                    file.close();
                    formFile = {
                        filename: p.fileName,
                        type: contentType,
                        tempfile: filepath,
                        size,
                    };
                }
                catch (e) {
                    await Deno.remove(filepath);
                    throw e;
                }
            }
            else {
                formFile = {
                    filename: p.fileName,
                    type: contentType,
                    content: buf.bytes(),
                    size: buf.length,
                };
                maxMemory -= n;
                maxValueBytes -= n;
            }
            if (formFile) {
                const mapVal = fileMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(formFile);
                }
                else {
                    fileMap.set(p.formName, [formFile]);
                }
            }
        }
        return multipartFormData(fileMap, valueMap);
    }
    currentPart;
    partsRead = 0;
    async nextPart() {
        if (this.currentPart) {
            this.currentPart.close();
        }
        if (equals(this.dashBoundary, encoder.encode("--"))) {
            throw new Error("boundary is empty");
        }
        let expectNewPart = false;
        for (;;) {
            const line = await this.bufReader.readSlice("\n".charCodeAt(0));
            if (line === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            if (this.isBoundaryDelimiterLine(line)) {
                this.partsRead++;
                const r = new TextProtoReader(this.bufReader);
                const headers = await r.readMIMEHeader();
                if (headers === null) {
                    throw new Deno.errors.UnexpectedEof();
                }
                const np = new PartReader(this, headers);
                this.currentPart = np;
                return np;
            }
            if (this.isFinalBoundary(line)) {
                return null;
            }
            if (expectNewPart) {
                throw new Error(`expecting a new Part; got line ${line}`);
            }
            if (this.partsRead === 0) {
                continue;
            }
            if (equals(line, this.newLine)) {
                expectNewPart = true;
                continue;
            }
            throw new Error(`unexpected line in nextPart(): ${line}`);
        }
    }
    isFinalBoundary(line) {
        if (!startsWith(line, this.dashBoundaryDash)) {
            return false;
        }
        const rest = line.slice(this.dashBoundaryDash.length, line.length);
        return rest.length === 0 || equals(skipLWSPChar(rest), this.newLine);
    }
    isBoundaryDelimiterLine(line) {
        if (!startsWith(line, this.dashBoundary)) {
            return false;
        }
        const rest = line.slice(this.dashBoundary.length);
        return equals(skipLWSPChar(rest), this.newLine);
    }
}
function multipartFormData(fileMap, valueMap) {
    function files(key) {
        return fileMap.get(key);
    }
    function values(key) {
        return valueMap.get(key);
    }
    function* entries() {
        yield* fileMap;
        yield* valueMap;
    }
    async function removeAll() {
        const promises = [];
        for (const val of fileMap.values()) {
            for (const subVal of val) {
                if (!subVal.tempfile)
                    continue;
                promises.push(Deno.remove(subVal.tempfile));
            }
        }
        await Promise.all(promises);
    }
    return {
        files,
        values,
        entries,
        removeAll,
        [Symbol.iterator]() {
            return entries();
        },
    };
}
class PartWriter {
    writer;
    boundary;
    headers;
    closed = false;
    partHeader;
    headersWritten = false;
    constructor(writer, boundary, headers, isFirstBoundary) {
        this.writer = writer;
        this.boundary = boundary;
        this.headers = headers;
        let buf = "";
        if (isFirstBoundary) {
            buf += `--${boundary}\r\n`;
        }
        else {
            buf += `\r\n--${boundary}\r\n`;
        }
        for (const [key, value] of headers.entries()) {
            buf += `${key}: ${value}\r\n`;
        }
        buf += `\r\n`;
        this.partHeader = buf;
    }
    close() {
        this.closed = true;
    }
    async write(p) {
        if (this.closed) {
            throw new Error("part is closed");
        }
        if (!this.headersWritten) {
            await this.writer.write(encoder.encode(this.partHeader));
            this.headersWritten = true;
        }
        return this.writer.write(p);
    }
}
function checkBoundary(b) {
    if (b.length < 1 || b.length > 70) {
        throw new Error(`invalid boundary length: ${b.length}`);
    }
    const end = b.length - 1;
    for (let i = 0; i < end; i++) {
        const c = b.charAt(i);
        if (!c.match(/[a-zA-Z0-9'()+_,\-./:=?]/) || (c === " " && i !== end)) {
            throw new Error("invalid boundary character: " + c);
        }
    }
    return b;
}
export class MultipartWriter {
    writer;
    _boundary;
    get boundary() {
        return this._boundary;
    }
    lastPart;
    bufWriter;
    isClosed = false;
    constructor(writer, boundary) {
        this.writer = writer;
        if (boundary !== void 0) {
            this._boundary = checkBoundary(boundary);
        }
        else {
            this._boundary = randomBoundary();
        }
        this.bufWriter = new BufWriter(writer);
    }
    formDataContentType() {
        return `multipart/form-data; boundary=${this.boundary}`;
    }
    createPart(headers) {
        if (this.isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.lastPart) {
            this.lastPart.close();
        }
        const part = new PartWriter(this.writer, this.boundary, headers, !this.lastPart);
        this.lastPart = part;
        return part;
    }
    createFormFile(field, filename) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"; filename="${filename}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    createFormField(field) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    async writeField(field, value) {
        const f = await this.createFormField(field);
        await f.write(encoder.encode(value));
    }
    async writeFile(field, filename, file) {
        const f = await this.createFormFile(field, filename);
        await copy(file, f);
    }
    flush() {
        return this.bufWriter.flush();
    }
    async close() {
        if (this.isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.lastPart) {
            this.lastPart.close();
            this.lastPart = void 0;
        }
        await this.writer.write(encoder.encode(`\r\n--${this.boundary}--\r\n`));
        await this.flush();
        this.isClosed = true;
    }
}
//# sourceMappingURL=data:application/json;base64,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