const lookup = [];
const revLookup = [];
const code = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
for (let i = 0, len = code.length; i < len; ++i) {
    lookup[i] = code[i];
    revLookup[code.charCodeAt(i)] = i;
}
const placeHolderPadLookup = [0, 1, , 2, 3, , 4];
function _getPadLen(placeHoldersLen) {
    const maybeLen = placeHolderPadLookup[placeHoldersLen];
    if (typeof maybeLen !== "number") {
        throw new Error("Invalid pad length");
    }
    return maybeLen;
}
function getLens(b32) {
    const len = b32.length;
    if (len % 8 > 0) {
        throw new Error("Invalid string. Length must be a multiple of 8");
    }
    let validLen = b32.indexOf("=");
    if (validLen === -1)
        validLen = len;
    const placeHoldersLen = validLen === len ? 0 : 8 - (validLen % 8);
    return [validLen, placeHoldersLen];
}
export function byteLength(b32) {
    const [validLen, placeHoldersLen] = getLens(b32);
    return _byteLength(validLen, placeHoldersLen);
}
function _byteLength(validLen, placeHoldersLen) {
    return ((validLen + placeHoldersLen) * 5) / 8 - _getPadLen(placeHoldersLen);
}
export function decode(b32) {
    let tmp;
    const [validLen, placeHoldersLen] = getLens(b32);
    const arr = new Uint8Array(_byteLength(validLen, placeHoldersLen));
    let curByte = 0;
    const len = placeHoldersLen > 0 ? validLen - 8 : validLen;
    let i;
    for (i = 0; i < len; i += 8) {
        tmp = (revLookup[b32.charCodeAt(i)] << 20) |
            (revLookup[b32.charCodeAt(i + 1)] << 15) |
            (revLookup[b32.charCodeAt(i + 2)] << 10) |
            (revLookup[b32.charCodeAt(i + 3)] << 5) |
            revLookup[b32.charCodeAt(i + 4)];
        arr[curByte++] = (tmp >> 17) & 0xff;
        arr[curByte++] = (tmp >> 9) & 0xff;
        arr[curByte++] = (tmp >> 1) & 0xff;
        tmp = ((tmp & 1) << 15) |
            (revLookup[b32.charCodeAt(i + 5)] << 10) |
            (revLookup[b32.charCodeAt(i + 6)] << 5) |
            revLookup[b32.charCodeAt(i + 7)];
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    if (placeHoldersLen === 1) {
        tmp = (revLookup[b32.charCodeAt(i)] << 20) |
            (revLookup[b32.charCodeAt(i + 1)] << 15) |
            (revLookup[b32.charCodeAt(i + 2)] << 10) |
            (revLookup[b32.charCodeAt(i + 3)] << 5) |
            revLookup[b32.charCodeAt(i + 4)];
        arr[curByte++] = (tmp >> 17) & 0xff;
        arr[curByte++] = (tmp >> 9) & 0xff;
        arr[curByte++] = (tmp >> 1) & 0xff;
        tmp = ((tmp & 1) << 7) |
            (revLookup[b32.charCodeAt(i + 5)] << 2) |
            (revLookup[b32.charCodeAt(i + 6)] >> 3);
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 3) {
        tmp = (revLookup[b32.charCodeAt(i)] << 19) |
            (revLookup[b32.charCodeAt(i + 1)] << 14) |
            (revLookup[b32.charCodeAt(i + 2)] << 9) |
            (revLookup[b32.charCodeAt(i + 3)] << 4) |
            (revLookup[b32.charCodeAt(i + 4)] >> 1);
        arr[curByte++] = (tmp >> 16) & 0xff;
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 4) {
        tmp = (revLookup[b32.charCodeAt(i)] << 11) |
            (revLookup[b32.charCodeAt(i + 1)] << 6) |
            (revLookup[b32.charCodeAt(i + 2)] << 1) |
            (revLookup[b32.charCodeAt(i + 3)] >> 4);
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 6) {
        tmp = (revLookup[b32.charCodeAt(i)] << 3) |
            (revLookup[b32.charCodeAt(i + 1)] >> 2);
        arr[curByte++] = tmp & 0xff;
    }
    return arr;
}
function encodeChunk(uint8, start, end) {
    let tmp;
    const output = [];
    for (let i = start; i < end; i += 5) {
        tmp = ((uint8[i] << 16) & 0xff0000) |
            ((uint8[i + 1] << 8) & 0xff00) |
            (uint8[i + 2] & 0xff);
        output.push(lookup[(tmp >> 19) & 0x1f]);
        output.push(lookup[(tmp >> 14) & 0x1f]);
        output.push(lookup[(tmp >> 9) & 0x1f]);
        output.push(lookup[(tmp >> 4) & 0x1f]);
        tmp = ((tmp & 0xf) << 16) |
            ((uint8[i + 3] << 8) & 0xff00) |
            (uint8[i + 4] & 0xff);
        output.push(lookup[(tmp >> 15) & 0x1f]);
        output.push(lookup[(tmp >> 10) & 0x1f]);
        output.push(lookup[(tmp >> 5) & 0x1f]);
        output.push(lookup[tmp & 0x1f]);
    }
    return output.join("");
}
export function encode(uint8) {
    let tmp;
    const len = uint8.length;
    const extraBytes = len % 5;
    const parts = [];
    const maxChunkLength = 16385;
    const len2 = len - extraBytes;
    for (let i = 0; i < len2; i += maxChunkLength) {
        parts.push(encodeChunk(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
    }
    if (extraBytes === 4) {
        tmp = ((uint8[len2] & 0xff) << 16) |
            ((uint8[len2 + 1] & 0xff) << 8) |
            (uint8[len2 + 2] & 0xff);
        parts.push(lookup[(tmp >> 19) & 0x1f]);
        parts.push(lookup[(tmp >> 14) & 0x1f]);
        parts.push(lookup[(tmp >> 9) & 0x1f]);
        parts.push(lookup[(tmp >> 4) & 0x1f]);
        tmp = ((tmp & 0xf) << 11) | (uint8[len2 + 3] << 3);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("=");
    }
    else if (extraBytes === 3) {
        tmp = ((uint8[len2] & 0xff) << 17) |
            ((uint8[len2 + 1] & 0xff) << 9) |
            ((uint8[len2 + 2] & 0xff) << 1);
        parts.push(lookup[(tmp >> 20) & 0x1f]);
        parts.push(lookup[(tmp >> 15) & 0x1f]);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("===");
    }
    else if (extraBytes === 2) {
        tmp = ((uint8[len2] & 0xff) << 12) | ((uint8[len2 + 1] & 0xff) << 4);
        parts.push(lookup[(tmp >> 15) & 0x1f]);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("====");
    }
    else if (extraBytes === 1) {
        tmp = (uint8[len2] & 0xff) << 2;
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("======");
    }
    return parts.join("");
}
//# sourceMappingURL=data:application/json;base64,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