// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import * as path from "../path/mod.ts";
import { ensureDir, ensureDirSync } from "./ensure_dir.ts";
import { getFileInfoType } from "./_util.ts";
import { isWindows } from "../_util/os.ts";
/**
 * Ensures that the link exists.
 * If the directory structure does not exist, it is created.
 *
 * @param src the source file path
 * @param dest the destination link path
 */ export async function ensureSymlink(src, dest) {
    const srcStatInfo = await Deno.lstat(src);
    const srcFilePathType = getFileInfoType(srcStatInfo);
    await ensureDir(path.dirname(dest));
    const options = isWindows ? {
        type: srcFilePathType === "dir" ? "dir" : "file"
    } : undefined;
    await Deno.symlink(src, dest, options);
}
/**
 * Ensures that the link exists.
 * If the directory structure does not exist, it is created.
 *
 * @param src the source file path
 * @param dest the destination link path
 */ export function ensureSymlinkSync(src, dest) {
    const srcStatInfo = Deno.lstatSync(src);
    const srcFilePathType = getFileInfoType(srcStatInfo);
    ensureDirSync(path.dirname(dest));
    const options = isWindows ? {
        type: srcFilePathType === "dir" ? "dir" : "file"
    } : undefined;
    Deno.symlinkSync(src, dest, options);
}
//# sourceMappingURL=data:application/json;base64,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