// Copyright 2014-2021 Sindre Sorhus. All rights reserved. MIT license.
// Copyright 2021 Yoshiya Hinosawa. All rights reserved. MIT license.
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/** Pretty print bytes.
 *
 * Based on [pretty-bytes](https://github.com/sindresorhus/pretty-bytes).
 *
 * This module is browser compatible.
 *
 * @module
 */ /**
 * Convert bytes to a human readable string: 1337 → 1.34 kB
 *
 * @param num The number to format
 * @param options The options
 */ export function prettyBytes(num, options = {}) {
    if (!Number.isFinite(num)) {
        throw new TypeError(`Expected a finite number, got ${typeof num}: ${num}`);
    }
    const UNITS_FIRSTLETTER = (options.bits ? "b" : "B") + "kMGTPEZY";
    if (options.signed && num === 0) {
        return ` 0 ${UNITS_FIRSTLETTER[0]}`;
    }
    const prefix = num < 0 ? "-" : options.signed ? "+" : "";
    num = Math.abs(num);
    const localeOptions = getLocaleOptions(options);
    if (num < 1) {
        const numberString = toLocaleString(num, options.locale, localeOptions);
        return prefix + numberString + " " + UNITS_FIRSTLETTER[0];
    }
    const exponent = Math.min(Math.floor(options.binary ? Math.log(num) / Math.log(1024) : Math.log10(num) / 3), UNITS_FIRSTLETTER.length - 1);
    num /= Math.pow(options.binary ? 1024 : 1000, exponent);
    if (!localeOptions) {
        num = Number(num.toPrecision(3));
    }
    const numberString1 = toLocaleString(num, options.locale, localeOptions);
    let unit = UNITS_FIRSTLETTER[exponent];
    if (exponent > 0) {
        unit += options.binary ? "i" : "";
        unit += options.bits ? "bit" : "B";
    }
    return prefix + numberString1 + " " + unit;
}
function getLocaleOptions({ maximumFractionDigits , minimumFractionDigits  }) {
    if (maximumFractionDigits || minimumFractionDigits) {
        return {
            maximumFractionDigits,
            minimumFractionDigits
        };
    }
}
/**
 * Formats the given number using `Number#toLocaleString`.
 * - If locale is a string, the value is expected to be a locale-key (for example: `de`).
 * - If locale is true, the system default locale is used for translation.
 * - If no value for locale is specified, the number is returned unmodified.
 */ function toLocaleString(num, locale, options) {
    if (typeof locale === "string" || Array.isArray(locale)) {
        return num.toLocaleString(locale, options);
    } else if (locale === true || options !== undefined) {
        return num.toLocaleString(undefined, options);
    }
    return num.toString();
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjE1My4wL2ZtdC9ieXRlcy50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgMjAxNC0yMDIxIFNpbmRyZSBTb3JodXMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuLy8gQ29weXJpZ2h0IDIwMjEgWW9zaGl5YSBIaW5vc2F3YS4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vLyBDb3B5cmlnaHQgMjAxOC0yMDIyIHRoZSBEZW5vIGF1dGhvcnMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuXG4vKiogUHJldHR5IHByaW50IGJ5dGVzLlxuICpcbiAqIEJhc2VkIG9uIFtwcmV0dHktYnl0ZXNdKGh0dHBzOi8vZ2l0aHViLmNvbS9zaW5kcmVzb3JodXMvcHJldHR5LWJ5dGVzKS5cbiAqXG4gKiBUaGlzIG1vZHVsZSBpcyBicm93c2VyIGNvbXBhdGlibGUuXG4gKlxuICogQG1vZHVsZVxuICovXG5cbnR5cGUgTG9jYWxlT3B0aW9ucyA9IHtcbiAgbWluaW11bUZyYWN0aW9uRGlnaXRzPzogbnVtYmVyO1xuICBtYXhpbXVtRnJhY3Rpb25EaWdpdHM/OiBudW1iZXI7XG59O1xuXG4vKipcbiAqIFRoZSBvcHRpb25zIGZvciBwcmV0dHkgcHJpbnRpbmcgdGhlIGJ5dGUgbnVtYmVycy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcmV0dHlCeXRlc09wdGlvbnMge1xuICAvKiogVXNlcyBiaXRzIHJlcHJlc2VudGF0aW9uLiBEZWZhdWx0IGlzIGZhbHNlLiAqL1xuICBiaXRzPzogYm9vbGVhbjtcbiAgLyoqIFVzZXMgYmluYXJ5IGJ5dGVzIChlLmcuIGtpYmlieXRlKS4gRGVmYXVsdCBpcyBmYWxzZS4gKi9cbiAgYmluYXJ5PzogYm9vbGVhbjtcbiAgLyoqIEluY2x1ZGUgcGx1cyBzaWduIGZvciBwb3NpdGl2ZSBudW1iZXJzLiAqL1xuICBzaWduZWQ/OiBib29sZWFuO1xuICAvKiogVXNlcyBsb2NhbGl6ZWQgbnVtYmVyIGZvcm1hdHRpbmcuIElmIGl0IGlzIHNldCB0byB0cnVlLCB1c2VzIGRlZmF1bHQgbG9jYWxlIG9uIHRoZSBzeXN0ZW0uIElmIGl0J3Mgc2V0IHRvIHN0cmluZywgdXNlcyB0aGF0IGxvY2FsZS4gVGhlIGdpdmVuIHN0cmluZyBzaG91bGQgYmUgQkNQIDQ3IGxhbmd1YWdlIHRhZyAocmVmOiBodHRwczovL2VuLndpa2lwZWRpYS5vcmcvd2lraS9JRVRGX2xhbmd1YWdlX3RhZykuIFlvdSBjYW4gYWxzbyBnaXZlIHRoZSBsaXN0IG9mIGxhbmd1YWdlIHRhZ3MuICovXG4gIGxvY2FsZT86IGJvb2xlYW4gfCBzdHJpbmcgfCBzdHJpbmdbXTtcbiAgLyoqIFRoZSBtaW5pbXVtIG51bWJlciBvZiBmcmFjdGlvbiBkaWdpdHMgdG8gZGlzcGxheS4gSWYgbmVpdGhlciBtaW5pbXVtRnJhY3Rpb25EaWdpdHMgb3IgbWF4aW11bUZyYWN0aW9uRGlnaXRzIGFyZSBzZXQsIHRoZSBkZWZhdWx0IGJlaGF2aW9yIGlzIHRvIHJvdW5kIHRvIDMgc2lnbmlmaWNhbnQgZGlnaXRzLiAqL1xuICBtaW5pbXVtRnJhY3Rpb25EaWdpdHM/OiBudW1iZXI7XG4gIC8qKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgZnJhY3Rpb24gZGlnaXRzIHRvIGRpc3BsYXkuIElmIG5laXRoZXIgbWluaW11bUZyYWN0aW9uRGlnaXRzIG9yIG1heGltdW1GcmFjdGlvbkRpZ2l0cyBhcmUgc2V0LCB0aGUgZGVmYXVsdCBiZWhhdmlvciBpcyB0byByb3VuZCB0byAzIHNpZ25pZmljYW50IGRpZ2l0cy4gKi9cbiAgbWF4aW11bUZyYWN0aW9uRGlnaXRzPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIENvbnZlcnQgYnl0ZXMgdG8gYSBodW1hbiByZWFkYWJsZSBzdHJpbmc6IDEzMzcg4oaSIDEuMzQga0JcbiAqXG4gKiBAcGFyYW0gbnVtIFRoZSBudW1iZXIgdG8gZm9ybWF0XG4gKiBAcGFyYW0gb3B0aW9ucyBUaGUgb3B0aW9uc1xuICovXG5leHBvcnQgZnVuY3Rpb24gcHJldHR5Qnl0ZXMoXG4gIG51bTogbnVtYmVyLFxuICBvcHRpb25zOiBQcmV0dHlCeXRlc09wdGlvbnMgPSB7fSxcbik6IHN0cmluZyB7XG4gIGlmICghTnVtYmVyLmlzRmluaXRlKG51bSkpIHtcbiAgICB0aHJvdyBuZXcgVHlwZUVycm9yKGBFeHBlY3RlZCBhIGZpbml0ZSBudW1iZXIsIGdvdCAke3R5cGVvZiBudW19OiAke251bX1gKTtcbiAgfVxuXG4gIGNvbnN0IFVOSVRTX0ZJUlNUTEVUVEVSID0gKG9wdGlvbnMuYml0cyA/IFwiYlwiIDogXCJCXCIpICsgXCJrTUdUUEVaWVwiO1xuXG4gIGlmIChvcHRpb25zLnNpZ25lZCAmJiBudW0gPT09IDApIHtcbiAgICByZXR1cm4gYCAwICR7VU5JVFNfRklSU1RMRVRURVJbMF19YDtcbiAgfVxuXG4gIGNvbnN0IHByZWZpeCA9IG51bSA8IDAgPyBcIi1cIiA6IChvcHRpb25zLnNpZ25lZCA/IFwiK1wiIDogXCJcIik7XG4gIG51bSA9IE1hdGguYWJzKG51bSk7XG5cbiAgY29uc3QgbG9jYWxlT3B0aW9ucyA9IGdldExvY2FsZU9wdGlvbnMob3B0aW9ucyk7XG5cbiAgaWYgKG51bSA8IDEpIHtcbiAgICBjb25zdCBudW1iZXJTdHJpbmcgPSB0b0xvY2FsZVN0cmluZyhudW0sIG9wdGlvbnMubG9jYWxlLCBsb2NhbGVPcHRpb25zKTtcbiAgICByZXR1cm4gcHJlZml4ICsgbnVtYmVyU3RyaW5nICsgXCIgXCIgKyBVTklUU19GSVJTVExFVFRFUlswXTtcbiAgfVxuXG4gIGNvbnN0IGV4cG9uZW50ID0gTWF0aC5taW4oXG4gICAgTWF0aC5mbG9vcihcbiAgICAgIG9wdGlvbnMuYmluYXJ5ID8gTWF0aC5sb2cobnVtKSAvIE1hdGgubG9nKDEwMjQpIDogTWF0aC5sb2cxMChudW0pIC8gMyxcbiAgICApLFxuICAgIFVOSVRTX0ZJUlNUTEVUVEVSLmxlbmd0aCAtIDEsXG4gICk7XG4gIG51bSAvPSBNYXRoLnBvdyhvcHRpb25zLmJpbmFyeSA/IDEwMjQgOiAxMDAwLCBleHBvbmVudCk7XG5cbiAgaWYgKCFsb2NhbGVPcHRpb25zKSB7XG4gICAgbnVtID0gTnVtYmVyKG51bS50b1ByZWNpc2lvbigzKSk7XG4gIH1cblxuICBjb25zdCBudW1iZXJTdHJpbmcgPSB0b0xvY2FsZVN0cmluZyhcbiAgICBudW0sXG4gICAgb3B0aW9ucy5sb2NhbGUsXG4gICAgbG9jYWxlT3B0aW9ucyxcbiAgKTtcblxuICBsZXQgdW5pdCA9IFVOSVRTX0ZJUlNUTEVUVEVSW2V4cG9uZW50XTtcbiAgaWYgKGV4cG9uZW50ID4gMCkge1xuICAgIHVuaXQgKz0gb3B0aW9ucy5iaW5hcnkgPyBcImlcIiA6IFwiXCI7XG4gICAgdW5pdCArPSBvcHRpb25zLmJpdHMgPyBcImJpdFwiIDogXCJCXCI7XG4gIH1cblxuICByZXR1cm4gcHJlZml4ICsgbnVtYmVyU3RyaW5nICsgXCIgXCIgKyB1bml0O1xufVxuXG5mdW5jdGlvbiBnZXRMb2NhbGVPcHRpb25zKFxuICB7IG1heGltdW1GcmFjdGlvbkRpZ2l0cywgbWluaW11bUZyYWN0aW9uRGlnaXRzIH06IFByZXR0eUJ5dGVzT3B0aW9ucyxcbik6IExvY2FsZU9wdGlvbnMgfCB1bmRlZmluZWQge1xuICBpZiAobWF4aW11bUZyYWN0aW9uRGlnaXRzIHx8IG1pbmltdW1GcmFjdGlvbkRpZ2l0cykge1xuICAgIHJldHVybiB7XG4gICAgICBtYXhpbXVtRnJhY3Rpb25EaWdpdHMsXG4gICAgICBtaW5pbXVtRnJhY3Rpb25EaWdpdHMsXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIEZvcm1hdHMgdGhlIGdpdmVuIG51bWJlciB1c2luZyBgTnVtYmVyI3RvTG9jYWxlU3RyaW5nYC5cbiAqIC0gSWYgbG9jYWxlIGlzIGEgc3RyaW5nLCB0aGUgdmFsdWUgaXMgZXhwZWN0ZWQgdG8gYmUgYSBsb2NhbGUta2V5IChmb3IgZXhhbXBsZTogYGRlYCkuXG4gKiAtIElmIGxvY2FsZSBpcyB0cnVlLCB0aGUgc3lzdGVtIGRlZmF1bHQgbG9jYWxlIGlzIHVzZWQgZm9yIHRyYW5zbGF0aW9uLlxuICogLSBJZiBubyB2YWx1ZSBmb3IgbG9jYWxlIGlzIHNwZWNpZmllZCwgdGhlIG51bWJlciBpcyByZXR1cm5lZCB1bm1vZGlmaWVkLlxuICovXG5mdW5jdGlvbiB0b0xvY2FsZVN0cmluZyhcbiAgbnVtOiBudW1iZXIsXG4gIGxvY2FsZTogYm9vbGVhbiB8IHN0cmluZyB8IHN0cmluZ1tdIHwgdW5kZWZpbmVkLFxuICBvcHRpb25zOiBMb2NhbGVPcHRpb25zIHwgdW5kZWZpbmVkLFxuKTogc3RyaW5nIHtcbiAgaWYgKHR5cGVvZiBsb2NhbGUgPT09IFwic3RyaW5nXCIgfHwgQXJyYXkuaXNBcnJheShsb2NhbGUpKSB7XG4gICAgcmV0dXJuIG51bS50b0xvY2FsZVN0cmluZyhsb2NhbGUsIG9wdGlvbnMpO1xuICB9IGVsc2UgaWYgKGxvY2FsZSA9PT0gdHJ1ZSB8fCBvcHRpb25zICE9PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gbnVtLnRvTG9jYWxlU3RyaW5nKHVuZGVmaW5lZCwgb3B0aW9ucyk7XG4gIH1cblxuICByZXR1cm4gbnVtLnRvU3RyaW5nKCk7XG59XG4iXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsdUVBQXVFO0FBQ3ZFLHFFQUFxRTtBQUNyRSwwRUFBMEU7QUFFMUU7Ozs7Ozs7Q0FPQyxHQUVELEFBdUJBOzs7OztDQUtDLEdBQ0QsT0FBTyxTQUFTLFdBQVcsQ0FDekIsR0FBVyxFQUNYLE9BQTJCLEdBQUcsRUFBRSxFQUN4QjtJQUNSLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3pCLE1BQU0sSUFBSSxTQUFTLENBQUMsQ0FBQyw4QkFBOEIsRUFBRSxPQUFPLEdBQUcsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRCxNQUFNLGlCQUFpQixHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksR0FBRyxHQUFHLEdBQUcsR0FBRyxDQUFDLEdBQUcsVUFBVSxBQUFDO0lBRWxFLElBQUksT0FBTyxDQUFDLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxFQUFFO1FBQy9CLE9BQU8sQ0FBQyxHQUFHLEVBQUUsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRCxNQUFNLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxHQUFHLEdBQUcsR0FBSSxPQUFPLENBQUMsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFLEFBQUMsQUFBQztJQUMzRCxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUVwQixNQUFNLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQUFBQztJQUVoRCxJQUFJLEdBQUcsR0FBRyxDQUFDLEVBQUU7UUFDWCxNQUFNLFlBQVksR0FBRyxjQUFjLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLEFBQUM7UUFDeEUsT0FBTyxNQUFNLEdBQUcsWUFBWSxHQUFHLEdBQUcsR0FBRyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FDdkIsSUFBSSxDQUFDLEtBQUssQ0FDUixPQUFPLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FDdEUsRUFDRCxpQkFBaUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUM3QixBQUFDO0lBQ0YsR0FBRyxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLE1BQU0sR0FBRyxJQUFJLEdBQUcsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBRXhELElBQUksQ0FBQyxhQUFhLEVBQUU7UUFDbEIsR0FBRyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVELE1BQU0sYUFBWSxHQUFHLGNBQWMsQ0FDakMsR0FBRyxFQUNILE9BQU8sQ0FBQyxNQUFNLEVBQ2QsYUFBYSxDQUNkLEFBQUM7SUFFRixJQUFJLElBQUksR0FBRyxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsQUFBQztJQUN2QyxJQUFJLFFBQVEsR0FBRyxDQUFDLEVBQUU7UUFDaEIsSUFBSSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUUsQ0FBQztRQUNsQyxJQUFJLElBQUksT0FBTyxDQUFDLElBQUksR0FBRyxLQUFLLEdBQUcsR0FBRyxDQUFDO0lBQ3JDLENBQUM7SUFFRCxPQUFPLE1BQU0sR0FBRyxhQUFZLEdBQUcsR0FBRyxHQUFHLElBQUksQ0FBQztBQUM1QyxDQUFDO0FBRUQsU0FBUyxnQkFBZ0IsQ0FDdkIsRUFBRSxxQkFBcUIsQ0FBQSxFQUFFLHFCQUFxQixDQUFBLEVBQXNCLEVBQ3pDO0lBQzNCLElBQUkscUJBQXFCLElBQUkscUJBQXFCLEVBQUU7UUFDbEQsT0FBTztZQUNMLHFCQUFxQjtZQUNyQixxQkFBcUI7U0FDdEIsQ0FBQztJQUNKLENBQUM7QUFDSCxDQUFDO0FBRUQ7Ozs7O0NBS0MsR0FDRCxTQUFTLGNBQWMsQ0FDckIsR0FBVyxFQUNYLE1BQStDLEVBQy9DLE9BQWtDLEVBQzFCO0lBQ1IsSUFBSSxPQUFPLE1BQU0sS0FBSyxRQUFRLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUN2RCxPQUFPLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzdDLE9BQU8sSUFBSSxNQUFNLEtBQUssSUFBSSxJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7UUFDbkQsT0FBTyxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQsT0FBTyxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUM7QUFDeEIsQ0FBQyJ9