// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { getLevelByName, getLevelName, LogLevels } from "./levels.ts";
export class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
export class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    /** If the level of the logger is greater than the level to log, then nothing
   * is logged, otherwise a log record is passed to each log handler.  `msg` data
   * passed in is returned.  If a function is passed in, it is only evaluated
   * if the msg will be logged and the return value will be the result of the
   * function, not the function itself, unless the function isn't called, in which
   * case undefined is returned.  All types are coerced to strings for logging.
   */ #_log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data) {
        if (typeof data === "string") {
            return data;
        } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
            return String(data);
        } else if (data instanceof Error) {
            return data.stack;
        } else if (typeof data === "object") {
            return JSON.stringify(data);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this.#_log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this.#_log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this.#_log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this.#_log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this.#_log(LogLevels.CRITICAL, msg, ...args);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjE1My4wL2xvZy9sb2dnZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IDIwMTgtMjAyMiB0aGUgRGVubyBhdXRob3JzLiBBbGwgcmlnaHRzIHJlc2VydmVkLiBNSVQgbGljZW5zZS5cbmltcG9ydCB7IGdldExldmVsQnlOYW1lLCBnZXRMZXZlbE5hbWUsIExvZ0xldmVscyB9IGZyb20gXCIuL2xldmVscy50c1wiO1xuaW1wb3J0IHR5cGUgeyBMZXZlbE5hbWUgfSBmcm9tIFwiLi9sZXZlbHMudHNcIjtcbmltcG9ydCB0eXBlIHsgQmFzZUhhbmRsZXIgfSBmcm9tIFwiLi9oYW5kbGVycy50c1wiO1xuXG4vLyBkZW5vLWxpbnQtaWdub3JlIG5vLWV4cGxpY2l0LWFueVxuZXhwb3J0IHR5cGUgR2VuZXJpY0Z1bmN0aW9uID0gKC4uLmFyZ3M6IGFueVtdKSA9PiBhbnk7XG5cbmV4cG9ydCBpbnRlcmZhY2UgTG9nUmVjb3JkT3B0aW9ucyB7XG4gIG1zZzogc3RyaW5nO1xuICBhcmdzOiB1bmtub3duW107XG4gIGxldmVsOiBudW1iZXI7XG4gIGxvZ2dlck5hbWU6IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIExvZ1JlY29yZCB7XG4gIHJlYWRvbmx5IG1zZzogc3RyaW5nO1xuICAjYXJnczogdW5rbm93bltdO1xuICAjZGF0ZXRpbWU6IERhdGU7XG4gIHJlYWRvbmx5IGxldmVsOiBudW1iZXI7XG4gIHJlYWRvbmx5IGxldmVsTmFtZTogc3RyaW5nO1xuICByZWFkb25seSBsb2dnZXJOYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogTG9nUmVjb3JkT3B0aW9ucykge1xuICAgIHRoaXMubXNnID0gb3B0aW9ucy5tc2c7XG4gICAgdGhpcy4jYXJncyA9IFsuLi5vcHRpb25zLmFyZ3NdO1xuICAgIHRoaXMubGV2ZWwgPSBvcHRpb25zLmxldmVsO1xuICAgIHRoaXMubG9nZ2VyTmFtZSA9IG9wdGlvbnMubG9nZ2VyTmFtZTtcbiAgICB0aGlzLiNkYXRldGltZSA9IG5ldyBEYXRlKCk7XG4gICAgdGhpcy5sZXZlbE5hbWUgPSBnZXRMZXZlbE5hbWUob3B0aW9ucy5sZXZlbCk7XG4gIH1cbiAgZ2V0IGFyZ3MoKTogdW5rbm93bltdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuI2FyZ3NdO1xuICB9XG4gIGdldCBkYXRldGltZSgpOiBEYXRlIHtcbiAgICByZXR1cm4gbmV3IERhdGUodGhpcy4jZGF0ZXRpbWUuZ2V0VGltZSgpKTtcbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIExvZ2dlck9wdGlvbnMge1xuICBoYW5kbGVycz86IEJhc2VIYW5kbGVyW107XG59XG5cbmV4cG9ydCBjbGFzcyBMb2dnZXIge1xuICAjbGV2ZWw6IExvZ0xldmVscztcbiAgI2hhbmRsZXJzOiBCYXNlSGFuZGxlcltdO1xuICByZWFkb25seSAjbG9nZ2VyTmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIGxvZ2dlck5hbWU6IHN0cmluZyxcbiAgICBsZXZlbE5hbWU6IExldmVsTmFtZSxcbiAgICBvcHRpb25zOiBMb2dnZXJPcHRpb25zID0ge30sXG4gICkge1xuICAgIHRoaXMuI2xvZ2dlck5hbWUgPSBsb2dnZXJOYW1lO1xuICAgIHRoaXMuI2xldmVsID0gZ2V0TGV2ZWxCeU5hbWUobGV2ZWxOYW1lKTtcbiAgICB0aGlzLiNoYW5kbGVycyA9IG9wdGlvbnMuaGFuZGxlcnMgfHwgW107XG4gIH1cblxuICBnZXQgbGV2ZWwoKTogTG9nTGV2ZWxzIHtcbiAgICByZXR1cm4gdGhpcy4jbGV2ZWw7XG4gIH1cbiAgc2V0IGxldmVsKGxldmVsOiBMb2dMZXZlbHMpIHtcbiAgICB0aGlzLiNsZXZlbCA9IGxldmVsO1xuICB9XG5cbiAgZ2V0IGxldmVsTmFtZSgpOiBMZXZlbE5hbWUge1xuICAgIHJldHVybiBnZXRMZXZlbE5hbWUodGhpcy4jbGV2ZWwpO1xuICB9XG4gIHNldCBsZXZlbE5hbWUobGV2ZWxOYW1lOiBMZXZlbE5hbWUpIHtcbiAgICB0aGlzLiNsZXZlbCA9IGdldExldmVsQnlOYW1lKGxldmVsTmFtZSk7XG4gIH1cblxuICBnZXQgbG9nZ2VyTmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLiNsb2dnZXJOYW1lO1xuICB9XG5cbiAgc2V0IGhhbmRsZXJzKGhuZGxzOiBCYXNlSGFuZGxlcltdKSB7XG4gICAgdGhpcy4jaGFuZGxlcnMgPSBobmRscztcbiAgfVxuICBnZXQgaGFuZGxlcnMoKTogQmFzZUhhbmRsZXJbXSB7XG4gICAgcmV0dXJuIHRoaXMuI2hhbmRsZXJzO1xuICB9XG5cbiAgLyoqIElmIHRoZSBsZXZlbCBvZiB0aGUgbG9nZ2VyIGlzIGdyZWF0ZXIgdGhhbiB0aGUgbGV2ZWwgdG8gbG9nLCB0aGVuIG5vdGhpbmdcbiAgICogaXMgbG9nZ2VkLCBvdGhlcndpc2UgYSBsb2cgcmVjb3JkIGlzIHBhc3NlZCB0byBlYWNoIGxvZyBoYW5kbGVyLiAgYG1zZ2AgZGF0YVxuICAgKiBwYXNzZWQgaW4gaXMgcmV0dXJuZWQuICBJZiBhIGZ1bmN0aW9uIGlzIHBhc3NlZCBpbiwgaXQgaXMgb25seSBldmFsdWF0ZWRcbiAgICogaWYgdGhlIG1zZyB3aWxsIGJlIGxvZ2dlZCBhbmQgdGhlIHJldHVybiB2YWx1ZSB3aWxsIGJlIHRoZSByZXN1bHQgb2YgdGhlXG4gICAqIGZ1bmN0aW9uLCBub3QgdGhlIGZ1bmN0aW9uIGl0c2VsZiwgdW5sZXNzIHRoZSBmdW5jdGlvbiBpc24ndCBjYWxsZWQsIGluIHdoaWNoXG4gICAqIGNhc2UgdW5kZWZpbmVkIGlzIHJldHVybmVkLiAgQWxsIHR5cGVzIGFyZSBjb2VyY2VkIHRvIHN0cmluZ3MgZm9yIGxvZ2dpbmcuXG4gICAqL1xuICAjX2xvZzxUPihcbiAgICBsZXZlbDogbnVtYmVyLFxuICAgIG1zZzogKFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQpIHwgKCgpID0+IFQpLFxuICAgIC4uLmFyZ3M6IHVua25vd25bXVxuICApOiBUIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodGhpcy5sZXZlbCA+IGxldmVsKSB7XG4gICAgICByZXR1cm4gbXNnIGluc3RhbmNlb2YgRnVuY3Rpb24gPyB1bmRlZmluZWQgOiBtc2c7XG4gICAgfVxuXG4gICAgbGV0IGZuUmVzdWx0OiBUIHwgdW5kZWZpbmVkO1xuICAgIGxldCBsb2dNZXNzYWdlOiBzdHJpbmc7XG4gICAgaWYgKG1zZyBpbnN0YW5jZW9mIEZ1bmN0aW9uKSB7XG4gICAgICBmblJlc3VsdCA9IG1zZygpO1xuICAgICAgbG9nTWVzc2FnZSA9IHRoaXMuYXNTdHJpbmcoZm5SZXN1bHQpO1xuICAgIH0gZWxzZSB7XG4gICAgICBsb2dNZXNzYWdlID0gdGhpcy5hc1N0cmluZyhtc2cpO1xuICAgIH1cbiAgICBjb25zdCByZWNvcmQ6IExvZ1JlY29yZCA9IG5ldyBMb2dSZWNvcmQoe1xuICAgICAgbXNnOiBsb2dNZXNzYWdlLFxuICAgICAgYXJnczogYXJncyxcbiAgICAgIGxldmVsOiBsZXZlbCxcbiAgICAgIGxvZ2dlck5hbWU6IHRoaXMubG9nZ2VyTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuI2hhbmRsZXJzLmZvckVhY2goKGhhbmRsZXIpID0+IHtcbiAgICAgIGhhbmRsZXIuaGFuZGxlKHJlY29yZCk7XG4gICAgfSk7XG5cbiAgICByZXR1cm4gbXNnIGluc3RhbmNlb2YgRnVuY3Rpb24gPyBmblJlc3VsdCA6IG1zZztcbiAgfVxuXG4gIGFzU3RyaW5nKGRhdGE6IHVua25vd24pOiBzdHJpbmcge1xuICAgIGlmICh0eXBlb2YgZGF0YSA9PT0gXCJzdHJpbmdcIikge1xuICAgICAgcmV0dXJuIGRhdGE7XG4gICAgfSBlbHNlIGlmIChcbiAgICAgIGRhdGEgPT09IG51bGwgfHxcbiAgICAgIHR5cGVvZiBkYXRhID09PSBcIm51bWJlclwiIHx8XG4gICAgICB0eXBlb2YgZGF0YSA9PT0gXCJiaWdpbnRcIiB8fFxuICAgICAgdHlwZW9mIGRhdGEgPT09IFwiYm9vbGVhblwiIHx8XG4gICAgICB0eXBlb2YgZGF0YSA9PT0gXCJ1bmRlZmluZWRcIiB8fFxuICAgICAgdHlwZW9mIGRhdGEgPT09IFwic3ltYm9sXCJcbiAgICApIHtcbiAgICAgIHJldHVybiBTdHJpbmcoZGF0YSk7XG4gICAgfSBlbHNlIGlmIChkYXRhIGluc3RhbmNlb2YgRXJyb3IpIHtcbiAgICAgIHJldHVybiBkYXRhLnN0YWNrITtcbiAgICB9IGVsc2UgaWYgKHR5cGVvZiBkYXRhID09PSBcIm9iamVjdFwiKSB7XG4gICAgICByZXR1cm4gSlNPTi5zdHJpbmdpZnkoZGF0YSk7XG4gICAgfVxuICAgIHJldHVybiBcInVuZGVmaW5lZFwiO1xuICB9XG5cbiAgZGVidWc8VD4obXNnOiAoKSA9PiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUIHwgdW5kZWZpbmVkO1xuICBkZWJ1ZzxUPihtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQ7XG4gIGRlYnVnPFQ+KFxuICAgIG1zZzogKFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQpIHwgKCgpID0+IFQpLFxuICAgIC4uLmFyZ3M6IHVua25vd25bXVxuICApOiBUIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy4jX2xvZyhMb2dMZXZlbHMuREVCVUcsIG1zZywgLi4uYXJncyk7XG4gIH1cblxuICBpbmZvPFQ+KG1zZzogKCkgPT4gVCwgLi4uYXJnczogdW5rbm93bltdKTogVCB8IHVuZGVmaW5lZDtcbiAgaW5mbzxUPihtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQ7XG4gIGluZm88VD4oXG4gICAgbXNnOiAoVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCkgfCAoKCkgPT4gVCksXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLiNfbG9nKExvZ0xldmVscy5JTkZPLCBtc2csIC4uLmFyZ3MpO1xuICB9XG5cbiAgd2FybmluZzxUPihtc2c6ICgpID0+IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQgfCB1bmRlZmluZWQ7XG4gIHdhcm5pbmc8VD4obXNnOiBUIGV4dGVuZHMgR2VuZXJpY0Z1bmN0aW9uID8gbmV2ZXIgOiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUO1xuICB3YXJuaW5nPFQ+KFxuICAgIG1zZzogKFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQpIHwgKCgpID0+IFQpLFxuICAgIC4uLmFyZ3M6IHVua25vd25bXVxuICApOiBUIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy4jX2xvZyhMb2dMZXZlbHMuV0FSTklORywgbXNnLCAuLi5hcmdzKTtcbiAgfVxuXG4gIGVycm9yPFQ+KG1zZzogKCkgPT4gVCwgLi4uYXJnczogdW5rbm93bltdKTogVCB8IHVuZGVmaW5lZDtcbiAgZXJyb3I8VD4obXNnOiBUIGV4dGVuZHMgR2VuZXJpY0Z1bmN0aW9uID8gbmV2ZXIgOiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUO1xuICBlcnJvcjxUPihcbiAgICBtc2c6IChUIGV4dGVuZHMgR2VuZXJpY0Z1bmN0aW9uID8gbmV2ZXIgOiBUKSB8ICgoKSA9PiBUKSxcbiAgICAuLi5hcmdzOiB1bmtub3duW11cbiAgKTogVCB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuI19sb2coTG9nTGV2ZWxzLkVSUk9SLCBtc2csIC4uLmFyZ3MpO1xuICB9XG5cbiAgY3JpdGljYWw8VD4obXNnOiAoKSA9PiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUIHwgdW5kZWZpbmVkO1xuICBjcml0aWNhbDxUPihcbiAgICBtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQ7XG4gIGNyaXRpY2FsPFQ+KFxuICAgIG1zZzogKFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQpIHwgKCgpID0+IFQpLFxuICAgIC4uLmFyZ3M6IHVua25vd25bXVxuICApOiBUIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy4jX2xvZyhMb2dMZXZlbHMuQ1JJVElDQUwsIG1zZywgLi4uYXJncyk7XG4gIH1cbn1cbiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSwwRUFBMEU7QUFDMUUsU0FBUyxjQUFjLEVBQUUsWUFBWSxFQUFFLFNBQVMsUUFBUSxhQUFhLENBQUM7QUFjdEUsT0FBTyxNQUFNLFNBQVM7SUFDcEIsQUFBUyxHQUFHLENBQVM7SUFDckIsQ0FBQyxJQUFJLENBQVk7SUFDakIsQ0FBQyxRQUFRLENBQU87SUFDaEIsQUFBUyxLQUFLLENBQVM7SUFDdkIsQUFBUyxTQUFTLENBQVM7SUFDM0IsQUFBUyxVQUFVLENBQVM7SUFFNUIsWUFBWSxPQUF5QixDQUFFO1FBQ3JDLElBQUksQ0FBQyxHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUN2QixJQUFJLENBQUMsQ0FBQyxJQUFJLEdBQUc7ZUFBSSxPQUFPLENBQUMsSUFBSTtTQUFDLENBQUM7UUFDL0IsSUFBSSxDQUFDLEtBQUssR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDO1FBQzNCLElBQUksQ0FBQyxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUNyQyxJQUFJLENBQUMsQ0FBQyxRQUFRLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUM1QixJQUFJLENBQUMsU0FBUyxHQUFHLFlBQVksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDL0M7UUFDSSxJQUFJLEdBQWM7UUFDcEIsT0FBTztlQUFJLElBQUksQ0FBQyxDQUFDLElBQUk7U0FBQyxDQUFDO0lBQ3pCO1FBQ0ksUUFBUSxHQUFTO1FBQ25CLE9BQU8sSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDNUM7Q0FDRDtBQU1ELE9BQU8sTUFBTSxNQUFNO0lBQ2pCLENBQUMsS0FBSyxDQUFZO0lBQ2xCLENBQUMsUUFBUSxDQUFnQjtJQUNoQixDQUFDLFVBQVUsQ0FBUztJQUU3QixZQUNFLFVBQWtCLEVBQ2xCLFNBQW9CLEVBQ3BCLE9BQXNCLEdBQUcsRUFBRSxDQUMzQjtRQUNBLElBQUksQ0FBQyxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7UUFDOUIsSUFBSSxDQUFDLENBQUMsS0FBSyxHQUFHLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUM7SUFDMUM7UUFFSSxLQUFLLEdBQWM7UUFDckIsT0FBTyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUM7SUFDckI7UUFDSSxLQUFLLENBQUMsS0FBZ0IsRUFBRTtRQUMxQixJQUFJLENBQUMsQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ3RCO1FBRUksU0FBUyxHQUFjO1FBQ3pCLE9BQU8sWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25DO1FBQ0ksU0FBUyxDQUFDLFNBQW9CLEVBQUU7UUFDbEMsSUFBSSxDQUFDLENBQUMsS0FBSyxHQUFHLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMxQztRQUVJLFVBQVUsR0FBVztRQUN2QixPQUFPLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQztJQUMxQjtRQUVJLFFBQVEsQ0FBQyxLQUFvQixFQUFFO1FBQ2pDLElBQUksQ0FBQyxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUM7SUFDekI7UUFDSSxRQUFRLEdBQWtCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDO0lBQ3hCO0lBRUE7Ozs7OztHQU1DLEdBQ0QsQ0FBQyxJQUFJLENBQ0gsS0FBYSxFQUNiLEdBQXdELEVBQ3hELEdBQUcsSUFBSSxBQUFXLEVBQ0g7UUFDZixJQUFJLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxFQUFFO1lBQ3RCLE9BQU8sR0FBRyxZQUFZLFFBQVEsR0FBRyxTQUFTLEdBQUcsR0FBRyxDQUFDO1FBQ25ELENBQUM7UUFFRCxJQUFJLFFBQVEsQUFBZSxBQUFDO1FBQzVCLElBQUksVUFBVSxBQUFRLEFBQUM7UUFDdkIsSUFBSSxHQUFHLFlBQVksUUFBUSxFQUFFO1lBQzNCLFFBQVEsR0FBRyxHQUFHLEVBQUUsQ0FBQztZQUNqQixVQUFVLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN2QyxPQUFPO1lBQ0wsVUFBVSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEMsQ0FBQztRQUNELE1BQU0sTUFBTSxHQUFjLElBQUksU0FBUyxDQUFDO1lBQ3RDLEdBQUcsRUFBRSxVQUFVO1lBQ2YsSUFBSSxFQUFFLElBQUk7WUFDVixLQUFLLEVBQUUsS0FBSztZQUNaLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtTQUM1QixDQUFDLEFBQUM7UUFFSCxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxHQUFLO1lBQ2xDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekIsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLEdBQUcsWUFBWSxRQUFRLEdBQUcsUUFBUSxHQUFHLEdBQUcsQ0FBQztJQUNsRCxDQUFDO0lBRUQsUUFBUSxDQUFDLElBQWEsRUFBVTtRQUM5QixJQUFJLE9BQU8sSUFBSSxLQUFLLFFBQVEsRUFBRTtZQUM1QixPQUFPLElBQUksQ0FBQztRQUNkLE9BQU8sSUFDTCxJQUFJLEtBQUssSUFBSSxJQUNiLE9BQU8sSUFBSSxLQUFLLFFBQVEsSUFDeEIsT0FBTyxJQUFJLEtBQUssUUFBUSxJQUN4QixPQUFPLElBQUksS0FBSyxTQUFTLElBQ3pCLE9BQU8sSUFBSSxLQUFLLFdBQVcsSUFDM0IsT0FBTyxJQUFJLEtBQUssUUFBUSxFQUN4QjtZQUNBLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RCLE9BQU8sSUFBSSxJQUFJLFlBQVksS0FBSyxFQUFFO1lBQ2hDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBRTtRQUNyQixPQUFPLElBQUksT0FBTyxJQUFJLEtBQUssUUFBUSxFQUFFO1lBQ25DLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM5QixDQUFDO1FBQ0QsT0FBTyxXQUFXLENBQUM7SUFDckI7SUFJQSxLQUFLLENBQ0gsR0FBd0QsRUFDeEQsR0FBRyxJQUFJLEFBQVcsRUFDSDtRQUNmLE9BQU8sSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsR0FBRyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ25EO0lBSUEsSUFBSSxDQUNGLEdBQXdELEVBQ3hELEdBQUcsSUFBSSxBQUFXLEVBQ0g7UUFDZixPQUFPLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLEdBQUcsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUNsRDtJQUlBLE9BQU8sQ0FDTCxHQUF3RCxFQUN4RCxHQUFHLElBQUksQUFBVyxFQUNIO1FBQ2YsT0FBTyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxHQUFHLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDckQ7SUFJQSxLQUFLLENBQ0gsR0FBd0QsRUFDeEQsR0FBRyxJQUFJLEFBQVcsRUFDSDtRQUNmLE9BQU8sSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsR0FBRyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ25EO0lBT0EsUUFBUSxDQUNOLEdBQXdELEVBQ3hELEdBQUcsSUFBSSxBQUFXLEVBQ0g7UUFDZixPQUFPLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLEdBQUcsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUN0RDtDQUNEIn0=