// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { isSubdir } from "./_util.ts";
const EXISTS_ERROR = new Error("dest already exists.");
/** Moves a file or directory */ export async function move(src, dest, { overwrite =false  } = {}) {
    const srcStat = await Deno.stat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (overwrite) {
        try {
            await Deno.remove(dest, {
                recursive: true
            });
        } catch (error) {
            if (!(error instanceof Deno.errors.NotFound)) {
                throw error;
            }
        }
    } else {
        try {
            await Deno.lstat(dest);
            return Promise.reject(EXISTS_ERROR);
        } catch  {
        // Do nothing...
        }
    }
    await Deno.rename(src, dest);
    return;
}
/** Moves a file or directory synchronously */ export function moveSync(src, dest, { overwrite =false  } = {}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (overwrite) {
        try {
            Deno.removeSync(dest, {
                recursive: true
            });
        } catch (error) {
            if (!(error instanceof Deno.errors.NotFound)) {
                throw error;
            }
        }
    } else {
        try {
            Deno.lstatSync(dest);
            throw EXISTS_ERROR;
        } catch (error1) {
            if (error1 === EXISTS_ERROR) {
                throw error1;
            }
        }
    }
    Deno.renameSync(src, dest);
}
//# sourceMappingURL=data:application/json;base64,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