// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/** {@linkcode parse} and {@linkcode stringify} for handling
 * [TOML](https://toml.io/en/) encoded data.
 *
 * This module is browser compatible.
 *
 * @module
 */ export { parse } from "./_toml/parser.ts";
// Bare keys may only contain ASCII letters,
// ASCII digits, underscores, and dashes (A-Za-z0-9_-).
function joinKeys(keys) {
    // Dotted keys are a sequence of bare or quoted keys joined with a dot.
    // This allows for grouping similar properties together:
    return keys.map((str)=>{
        return str.match(/[^A-Za-z0-9_-]/) ? JSON.stringify(str) : str;
    }).join(".");
}
var ArrayType;
(function(ArrayType) {
    ArrayType[ArrayType["ONLY_PRIMITIVE"] = 0] = "ONLY_PRIMITIVE";
    ArrayType[ArrayType["ONLY_OBJECT_EXCLUDING_ARRAY"] = 1] = "ONLY_OBJECT_EXCLUDING_ARRAY";
    ArrayType[ArrayType["MIXED"] = 2] = "MIXED";
})(ArrayType || (ArrayType = {}));
class Dumper {
    maxPad = 0;
    srcObject;
    output = [];
    #arrayTypeCache = new Map();
    constructor(srcObjc){
        this.srcObject = srcObjc;
    }
    dump(fmtOptions = {}) {
        // deno-lint-ignore no-explicit-any
        this.output = this.#printObject(this.srcObject);
        this.output = this.#format(fmtOptions);
        return this.output;
    }
    #printObject(obj, keys = []) {
        const out = [];
        const props = Object.keys(obj);
        const inlineProps = [];
        const multilineProps = [];
        for (const prop of props){
            if (this.#isSimplySerializable(obj[prop])) {
                inlineProps.push(prop);
            } else {
                multilineProps.push(prop);
            }
        }
        const sortedProps = inlineProps.concat(multilineProps);
        for(let i = 0; i < sortedProps.length; i++){
            const prop1 = sortedProps[i];
            const value = obj[prop1];
            if (value instanceof Date) {
                out.push(this.#dateDeclaration([
                    prop1
                ], value));
            } else if (typeof value === "string" || value instanceof RegExp) {
                out.push(this.#strDeclaration([
                    prop1
                ], value.toString()));
            } else if (typeof value === "number") {
                out.push(this.#numberDeclaration([
                    prop1
                ], value));
            } else if (typeof value === "boolean") {
                out.push(this.#boolDeclaration([
                    prop1
                ], value));
            } else if (value instanceof Array) {
                const arrayType = this.#getTypeOfArray(value);
                if (arrayType === ArrayType.ONLY_PRIMITIVE) {
                    out.push(this.#arrayDeclaration([
                        prop1
                    ], value));
                } else if (arrayType === ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY) {
                    // array of objects
                    for(let i1 = 0; i1 < value.length; i1++){
                        out.push("");
                        out.push(this.#headerGroup([
                            ...keys,
                            prop1
                        ]));
                        out.push(...this.#printObject(value[i1], [
                            ...keys,
                            prop1
                        ]));
                    }
                } else {
                    // this is a complex array, use the inline format.
                    const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
                    out.push(`${this.#declaration([
                        prop1
                    ])}[${str}]`);
                }
            } else if (typeof value === "object") {
                out.push("");
                out.push(this.#header([
                    ...keys,
                    prop1
                ]));
                if (value) {
                    const toParse = value;
                    out.push(...this.#printObject(toParse, [
                        ...keys,
                        prop1
                    ]));
                }
            // out.push(...this._parse(value, `${path}${prop}.`));
            }
        }
        out.push("");
        return out;
    }
    #isPrimitive(value1) {
        return value1 instanceof Date || value1 instanceof RegExp || [
            "string",
            "number",
            "boolean"
        ].includes(typeof value1);
    }
    #getTypeOfArray(arr) {
        if (this.#arrayTypeCache.has(arr)) {
            return this.#arrayTypeCache.get(arr);
        }
        const type = this.#doGetTypeOfArray(arr);
        this.#arrayTypeCache.set(arr, type);
        return type;
    }
    #doGetTypeOfArray(arr1) {
        if (!arr1.length) {
            // any type should be fine
            return ArrayType.ONLY_PRIMITIVE;
        }
        const onlyPrimitive = this.#isPrimitive(arr1[0]);
        if (arr1[0] instanceof Array) {
            return ArrayType.MIXED;
        }
        for(let i2 = 1; i2 < arr1.length; i2++){
            if (onlyPrimitive !== this.#isPrimitive(arr1[i2]) || arr1[i2] instanceof Array) {
                return ArrayType.MIXED;
            }
        }
        return onlyPrimitive ? ArrayType.ONLY_PRIMITIVE : ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY;
    }
    #printAsInlineValue(value2) {
        if (value2 instanceof Date) {
            return `"${this.#printDate(value2)}"`;
        } else if (typeof value2 === "string" || value2 instanceof RegExp) {
            return JSON.stringify(value2.toString());
        } else if (typeof value2 === "number") {
            return value2;
        } else if (typeof value2 === "boolean") {
            return value2.toString();
        } else if (value2 instanceof Array) {
            const str1 = value2.map((x)=>this.#printAsInlineValue(x)).join(",");
            return `[${str1}]`;
        } else if (typeof value2 === "object") {
            if (!value2) {
                throw new Error("should never reach");
            }
            const str2 = Object.keys(value2).map((key)=>{
                // deno-lint-ignore no-explicit-any
                return `${key} = ${this.#printAsInlineValue(value2[key])}`;
            }).join(",");
            return `{${str2}}`;
        }
        throw new Error("should never reach");
    }
    #isSimplySerializable(value3) {
        return typeof value3 === "string" || typeof value3 === "number" || typeof value3 === "boolean" || value3 instanceof RegExp || value3 instanceof Date || value3 instanceof Array && this.#getTypeOfArray(value3) !== ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY;
    }
    #header(keys1) {
        return `[${joinKeys(keys1)}]`;
    }
    #headerGroup(keys2) {
        return `[[${joinKeys(keys2)}]]`;
    }
    #declaration(keys3) {
        const title = joinKeys(keys3);
        if (title.length > this.maxPad) {
            this.maxPad = title.length;
        }
        return `${title} = `;
    }
    #arrayDeclaration(keys4, value4) {
        return `${this.#declaration(keys4)}${JSON.stringify(value4)}`;
    }
    #strDeclaration(keys5, value5) {
        return `${this.#declaration(keys5)}${JSON.stringify(value5)}`;
    }
    #numberDeclaration(keys6, value6) {
        switch(value6){
            case Infinity:
                return `${this.#declaration(keys6)}inf`;
            case -Infinity:
                return `${this.#declaration(keys6)}-inf`;
            default:
                return `${this.#declaration(keys6)}${value6}`;
        }
    }
    #boolDeclaration(keys7, value7) {
        return `${this.#declaration(keys7)}${value7}`;
    }
    #printDate(value8) {
        function dtPad(v, lPad = 2) {
            return v.padStart(lPad, "0");
        }
        const m = dtPad((value8.getUTCMonth() + 1).toString());
        const d = dtPad(value8.getUTCDate().toString());
        const h = dtPad(value8.getUTCHours().toString());
        const min = dtPad(value8.getUTCMinutes().toString());
        const s = dtPad(value8.getUTCSeconds().toString());
        const ms = dtPad(value8.getUTCMilliseconds().toString(), 3);
        // formatted date
        const fData = `${value8.getUTCFullYear()}-${m}-${d}T${h}:${min}:${s}.${ms}`;
        return fData;
    }
    #dateDeclaration(keys8, value9) {
        return `${this.#declaration(keys8)}${this.#printDate(value9)}`;
    }
    #format(options = {}) {
        const { keyAlignment =false  } = options;
        const rDeclaration = /^(\".*\"|[^=]*)\s=/;
        const out1 = [];
        for(let i3 = 0; i3 < this.output.length; i3++){
            const l = this.output[i3];
            // we keep empty entry for array of objects
            if (l[0] === "[" && l[1] !== "[") {
                // empty object
                if (this.output[i3 + 1] === "") {
                    i3 += 1;
                    continue;
                }
                out1.push(l);
            } else {
                if (keyAlignment) {
                    const m1 = rDeclaration.exec(l);
                    if (m1) {
                        out1.push(l.replace(m1[1], m1[1].padEnd(this.maxPad)));
                    } else {
                        out1.push(l);
                    }
                } else {
                    out1.push(l);
                }
            }
        }
        // Cleaning multiple spaces
        const cleanedOutput = [];
        for(let i4 = 0; i4 < out1.length; i4++){
            const l1 = out1[i4];
            if (!(l1 === "" && out1[i4 + 1] === "")) {
                cleanedOutput.push(l1);
            }
        }
        return cleanedOutput;
    }
}
/**
 * Stringify dumps source object into TOML string and returns it.
 * @param srcObj
 * @param [fmtOptions] format options
 * @param [fmtOptions.keyAlignment] whether to algin key
 */ export function stringify(srcObj, fmtOptions) {
    return new Dumper(srcObj).dump(fmtOptions).join("\n");
}
//# sourceMappingURL=data:application/json;base64,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