// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * {@linkcode encode} and {@linkcode decode} for
 * [base64 URL safe](https://en.wikipedia.org/wiki/Base64#URL_applications) encoding.
 *
 * This module is browser compatible.
 *
 * @module
 */ import * as base64 from "./base64.ts";
/*
 * Some variants allow or require omitting the padding '=' signs:
 * https://en.wikipedia.org/wiki/Base64#The_URL_applications
 * @param base64url
 */ function addPaddingToBase64url(base64url) {
    if (base64url.length % 4 === 2) return base64url + "==";
    if (base64url.length % 4 === 3) return base64url + "=";
    if (base64url.length % 4 === 1) {
        throw new TypeError("Illegal base64url string!");
    }
    return base64url;
}
function convertBase64urlToBase64(b64url) {
    if (!/^[-_A-Z0-9]*?={0,2}$/i.test(b64url)) {
        // Contains characters not part of base64url spec.
        throw new TypeError("Failed to decode base64url: invalid character");
    }
    return addPaddingToBase64url(b64url).replace(/\-/g, "+").replace(/_/g, "/");
}
function convertBase64ToBase64url(b64) {
    return b64.replace(/=/g, "").replace(/\+/g, "-").replace(/\//g, "_");
}
/**
 * Encodes a given ArrayBuffer or string into a base64url representation
 * @param data
 */ export function encode(data) {
    return convertBase64ToBase64url(base64.encode(data));
}
/**
 * Converts given base64url encoded data back to original
 * @param b64url
 */ export function decode(b64url) {
    return base64.decode(convertBase64urlToBase64(b64url));
}
//# sourceMappingURL=data:application/json;base64,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