import { filterInPlace } from "./_utils.ts";
const { hasOwn } = Object;
export function deepMerge(record, other, options) {
    return deepMergeInternal(record, other, new Set(), options);
}
function deepMergeInternal(record, other, seen, options) {
    const result = {};
    const keys = new Set([
        ...getKeys(record),
        ...getKeys(other),
    ]);
    for (const key of keys) {
        if (key === "__proto__") {
            continue;
        }
        const a = record[key];
        if (!hasOwn(other, key)) {
            result[key] = a;
            continue;
        }
        const b = other[key];
        if (isNonNullObject(a) && isNonNullObject(b) && !seen.has(a) && !seen.has(b)) {
            seen.add(a);
            seen.add(b);
            result[key] = mergeObjects(a, b, seen, options);
            continue;
        }
        result[key] = b;
    }
    return result;
}
function mergeObjects(left, right, seen, options = {
    arrays: "merge",
    sets: "merge",
    maps: "merge",
}) {
    if (isMergeable(left) && isMergeable(right)) {
        return deepMergeInternal(left, right, seen);
    }
    if (isIterable(left) && isIterable(right)) {
        if ((Array.isArray(left)) && (Array.isArray(right))) {
            if (options.arrays === "merge") {
                return left.concat(right);
            }
            return right;
        }
        if ((left instanceof Map) && (right instanceof Map)) {
            if (options.maps === "merge") {
                return new Map([
                    ...left,
                    ...right,
                ]);
            }
            return right;
        }
        if ((left instanceof Set) && (right instanceof Set)) {
            if (options.sets === "merge") {
                return new Set([
                    ...left,
                    ...right,
                ]);
            }
            return right;
        }
    }
    return right;
}
function isMergeable(value) {
    return Object.getPrototypeOf(value) === Object.prototype;
}
function isIterable(value) {
    return typeof value[Symbol.iterator] === "function";
}
function isNonNullObject(value) {
    return value !== null && typeof value === "object";
}
function getKeys(record) {
    const ret = Object.getOwnPropertySymbols(record);
    filterInPlace(ret, (key) => Object.prototype.propertyIsEnumerable.call(record, key));
    ret.push(...Object.keys(record));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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