// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { parseMediaType } from "./parse_media_type.ts";
import { getCharset } from "./get_charset.ts";
import { formatMediaType } from "./format_media_type.ts";
import { typeByExtension } from "./type_by_extension.ts";
/**
 * Returns the full `Content-Type` or `Content-Disposition` header value for the
 * given extension or media type.
 *
 * The function will treat the `extensionOrType` as a media type when it
 * contains a `/`, otherwise it will process it as an extension, with or without
 * the leading `.`.
 *
 * Returns `undefined` if unable to resolve the media type.
 *
 * @typeParam T Type of the extension or media type to resolve.
 *
 * @param extensionOrType The extension or media type to resolve.
 *
 * @returns The full `Content-Type` or `Content-Disposition` header value, or
 * `undefined` if unable to resolve the media type.
 *
 * @example Usage
 * ```ts
 * import { contentType } from "@std/media-types/content-type";
 * import { assertEquals } from "@std/assert";
 *
 * assertEquals(contentType(".json"), "application/json; charset=UTF-8");
 * assertEquals(contentType("text/html"), "text/html; charset=UTF-8");
 * assertEquals(contentType("text/html; charset=UTF-8"), "text/html; charset=UTF-8");
 * assertEquals(contentType("txt"), "text/plain; charset=UTF-8");
 * assertEquals(contentType("foo"), undefined);
 * assertEquals(contentType("file.json"), undefined);
 * ```
 */ export function contentType(extensionOrType) {
  try {
    const [mediaType, params = {}] = extensionOrType.includes("/") ? parseMediaType(extensionOrType) : [
      typeByExtension(extensionOrType),
      undefined
    ];
    if (!mediaType) {
      return undefined;
    }
    if (!("charset" in params)) {
      const charset = getCharset(mediaType);
      if (charset) {
        params.charset = charset;
      }
    }
    return formatMediaType(mediaType, params);
  } catch  {
  // just swallow returning undefined
  }
  return undefined;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=3060368166612659189,13467338511935921152