// deno-fmt-ignore-file
// deno-lint-ignore-file
// This code was bundled using `deno bundle` and it's not recommended to edit it manually

function isWindows() {
    return Deno.build.os === "windows";
}
function isRStudio() {
    return !!Deno.env.get("RSTUDIO");
}
function isVSCodeOutputChannel() {
    return !!Deno.env.get("VSCODE_PID");
}
function isVSCodeTerminal() {
    return Deno.env.get("TERM_PROGRAM") === "vscode";
}
function isVSCodeServer() {
    return !!vsCodeServerProxyUri();
}
function isRStudioWorkbench() {
    return !!Deno.env.get("RS_SERVER_URL") && !!Deno.env.get("RS_SESSION_URL");
}
function isRStudioTerminal() {
    return !!Deno.env.get("RSTUDIO_TERM");
}
function isServerSession() {
    return isRStudioServer() || isRStudioWorkbench() || isJupyterServer() || isJupyterHubServer();
}
function isRStudioServer() {
    return isRStudio() && Deno.env.get("RSTUDIO_PROGRAM_MODE") === "server";
}
function isRStudioPreview() {
    return isRStudio() && !isRStudioTerminal();
}
function isJupyterServer() {
    return !!Deno.env.get("JUPYTER_SERVER_URL");
}
function isJupyterHubServer() {
    return jupyterHubUser() !== undefined;
}
function vsCodeServerProxyUri() {
    return Deno.env.get("VSCODE_PROXY_URI");
}
function jupyterHubHttpReferrer() {
    return Deno.env.get("JUPYTERHUB_HTTP_REFERER");
}
function jupyterHubUser() {
    return Deno.env.get("JUPYTERHUB_USER");
}
function isInteractiveTerminal() {
    return Deno.isatty(Deno.stderr.rid);
}
function isInteractiveSession() {
    return isRStudio() || isInteractiveTerminal() || isVSCodeOutputChannel();
}
function haveArrowKeys() {
    return !isWindows();
}
const osType = (()=>{
    const { Deno: Deno1  } = globalThis;
    if (typeof Deno1?.build?.os === "string") {
        return Deno1.build.os;
    }
    const { navigator  } = globalThis;
    if (navigator?.appVersion?.includes?.("Win")) {
        return "windows";
    }
    return "linux";
})();
const isWindows1 = osType === "windows";
const CHAR_FORWARD_SLASH = 47;
function assertPath(path) {
    if (typeof path !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path)}`);
    }
}
function isPosixPathSeparator(code) {
    return code === 47;
}
function isPathSeparator(code) {
    return isPosixPathSeparator(code) || code === 92;
}
function isWindowsDeviceRoot(code) {
    return code >= 97 && code <= 122 || code >= 65 && code <= 90;
}
function normalizeString(path, allowAboveRoot, separator, isPathSeparator) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code;
    for(let i1 = 0, len = path.length; i1 <= len; ++i1){
        if (i1 < len) code = path.charCodeAt(i1);
        else if (isPathSeparator(code)) break;
        else code = CHAR_FORWARD_SLASH;
        if (isPathSeparator(code)) {
            if (lastSlash === i1 - 1 || dots === 1) {} else if (lastSlash !== i1 - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path.slice(lastSlash + 1, i1);
                else res = path.slice(lastSlash + 1, i1);
                lastSegmentLength = i1 - lastSlash - 1;
            }
            lastSlash = i1;
            dots = 0;
        } else if (code === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function _format(sep, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base;
    if (dir === pathObject.root) return dir + base;
    return dir + sep + base;
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string) {
    return string.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
class DenoStdInternalError extends Error {
    constructor(message){
        super(message);
        this.name = "DenoStdInternalError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new DenoStdInternalError(msg);
    }
}
const sep = "\\";
const delimiter = ";";
function resolve(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1; i1--){
        let path;
        const { Deno: Deno1  } = globalThis;
        if (i1 >= 0) {
            path = pathSegments[i1];
        } else if (!resolvedDevice) {
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path = Deno1.cwd();
        } else {
            if (typeof Deno1?.env?.get !== "function" || typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
            if (path === undefined || path.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path = `${resolvedDevice}\\`;
            }
        }
        assertPath(path);
        const len = path.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute = false;
        const code = path.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code)) {
                isAbsolute = true;
                if (isPathSeparator(path.charCodeAt(1))) {
                    let j = 2;
                    let last = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        const firstPart = path.slice(last, j);
                        last = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last) {
                            last = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path.slice(last)}`;
                                rootEnd = j;
                            } else if (j !== last) {
                                device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code)) {
                if (path.charCodeAt(1) === 58) {
                    device = path.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path.charCodeAt(2))) {
                            isAbsolute = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code)) {
            rootEnd = 1;
            isAbsolute = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function normalize(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return ".";
    let rootEnd = 0;
    let device;
    let isAbsolute = false;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            isAbsolute = true;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    const firstPart = path.slice(last, j);
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path.slice(last)}\\`;
                        } else if (j !== last) {
                            device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                device = path.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        isAbsolute = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return "\\";
    }
    let tail;
    if (rootEnd < len) {
        tail = normalizeString(path.slice(rootEnd), !isAbsolute, "\\", isPathSeparator);
    } else {
        tail = "";
    }
    if (tail.length === 0 && !isAbsolute) tail = ".";
    if (tail.length > 0 && isPathSeparator(path.charCodeAt(len - 1))) {
        tail += "\\";
    }
    if (device === undefined) {
        if (isAbsolute) {
            if (tail.length > 0) return `\\${tail}`;
            else return "\\";
        } else if (tail.length > 0) {
            return tail;
        } else {
            return "";
        }
    } else if (isAbsolute) {
        if (tail.length > 0) return `${device}\\${tail}`;
        else return `${device}\\`;
    } else if (tail.length > 0) {
        return device + tail;
    } else {
        return device;
    }
}
function isAbsolute(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return false;
    const code = path.charCodeAt(0);
    if (isPathSeparator(code)) {
        return true;
    } else if (isWindowsDeviceRoot(code)) {
        if (len > 2 && path.charCodeAt(1) === 58) {
            if (isPathSeparator(path.charCodeAt(2))) return true;
        }
    }
    return false;
}
function join(...paths) {
    const pathsCount = paths.length;
    if (pathsCount === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i1 = 0; i1 < pathsCount; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (joined === undefined) joined = firstPart = path;
            else joined += `\\${path}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart != null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize(joined);
}
function relative(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    const fromOrig = resolve(from);
    const toOrig = resolve(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i1) === 92) {
                    return toOrig.slice(toStart + i1 + 1);
                } else if (i1 === 2) {
                    return toOrig.slice(toStart + i1);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i1) === 92) {
                    lastCommonSep = i1;
                } else if (i1 === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i1;
    }
    if (i1 !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || from.charCodeAt(i1) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
function toNamespacedPath(path) {
    if (typeof path !== "string") return path;
    if (path.length === 0) return "";
    const resolvedPath = resolve(path);
    if (resolvedPath.length >= 3) {
        if (resolvedPath.charCodeAt(0) === 92) {
            if (resolvedPath.charCodeAt(1) === 92) {
                const code = resolvedPath.charCodeAt(2);
                if (code !== 63 && code !== 46) {
                    return `\\\\?\\UNC\\${resolvedPath.slice(2)}`;
                }
            }
        } else if (isWindowsDeviceRoot(resolvedPath.charCodeAt(0))) {
            if (resolvedPath.charCodeAt(1) === 58 && resolvedPath.charCodeAt(2) === 92) {
                return `\\\\?\\${resolvedPath}`;
            }
        }
    }
    return path;
}
function dirname(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return ".";
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path;
                        }
                        if (j !== last) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return path;
    }
    for(let i1 = len - 1; i1 >= offset; --i1){
        if (isPathSeparator(path.charCodeAt(i1))) {
            if (!matchedSlash) {
                end = i1;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return path.slice(0, end);
}
function basename(path, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i1;
    if (path.length >= 2) {
        const drive = path.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path.charCodeAt(1) === 58) start = 2;
        }
    }
    if (ext !== undefined && ext.length > 0 && ext.length <= path.length) {
        if (ext.length === path.length && ext === path) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i1 = path.length - 1; i1 >= start; --i1){
            const code = path.charCodeAt(i1);
            if (isPathSeparator(code)) {
                if (!matchedSlash) {
                    start = i1 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i1 + 1;
                }
                if (extIdx >= 0) {
                    if (code === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i1;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path.length;
        return path.slice(start, end);
    } else {
        for(i1 = path.length - 1; i1 >= start; --i1){
            if (isPathSeparator(path.charCodeAt(i1))) {
                if (!matchedSlash) {
                    start = i1 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i1 + 1;
            }
        }
        if (end === -1) return "";
        return path.slice(start, end);
    }
}
function extname(path) {
    assertPath(path);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path.length >= 2 && path.charCodeAt(1) === 58 && isWindowsDeviceRoot(path.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i1 = path.length - 1; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (isPathSeparator(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("\\", pathObject);
}
function parse(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    const len = path.length;
    if (len === 0) return ret;
    let rootEnd = 0;
    let code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            rootEnd = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            rootEnd = j;
                        } else if (j !== last) {
                            rootEnd = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        if (len === 3) {
                            ret.root = ret.dir = path;
                            return ret;
                        }
                        rootEnd = 3;
                    }
                } else {
                    ret.root = ret.dir = path;
                    return ret;
                }
            }
        }
    } else if (isPathSeparator(code)) {
        ret.root = ret.dir = path;
        return ret;
    }
    if (rootEnd > 0) ret.root = path.slice(0, rootEnd);
    let startDot = -1;
    let startPart = rootEnd;
    let end = -1;
    let matchedSlash = true;
    let i1 = path.length - 1;
    let preDotState = 0;
    for(; i1 >= rootEnd; --i1){
        code = path.charCodeAt(i1);
        if (isPathSeparator(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            ret.base = ret.name = path.slice(startPart, end);
        }
    } else {
        ret.name = path.slice(startPart, startDot);
        ret.base = path.slice(startPart, end);
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0 && startPart !== rootEnd) {
        ret.dir = path.slice(0, startPart - 1);
    } else ret.dir = ret.root;
    return ret;
}
function fromFileUrl(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    let path = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname != "") {
        path = `\\\\${url.hostname}${path}`;
    }
    return path;
}
function toFileUrl(path) {
    if (!isAbsolute(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname != null && hostname != "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
const mod = {
    sep: sep,
    delimiter: delimiter,
    resolve: resolve,
    normalize: normalize,
    isAbsolute: isAbsolute,
    join: join,
    relative: relative,
    toNamespacedPath: toNamespacedPath,
    dirname: dirname,
    basename: basename,
    extname: extname,
    format: format,
    parse: parse,
    fromFileUrl: fromFileUrl,
    toFileUrl: toFileUrl
};
const sep1 = "/";
const delimiter1 = ":";
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1 && !resolvedAbsolute; i1--){
        let path;
        if (i1 >= 0) path = pathSegments[i1];
        else {
            const { Deno: Deno1  } = globalThis;
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
        }
        assertPath(path);
        if (path.length === 0) {
            continue;
        }
        resolvedPath = `${path}/${resolvedPath}`;
        resolvedAbsolute = path.charCodeAt(0) === CHAR_FORWARD_SLASH;
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function normalize1(path) {
    assertPath(path);
    if (path.length === 0) return ".";
    const isAbsolute = path.charCodeAt(0) === 47;
    const trailingSeparator = path.charCodeAt(path.length - 1) === 47;
    path = normalizeString(path, !isAbsolute, "/", isPosixPathSeparator);
    if (path.length === 0 && !isAbsolute) path = ".";
    if (path.length > 0 && trailingSeparator) path += "/";
    if (isAbsolute) return `/${path}`;
    return path;
}
function isAbsolute1(path) {
    assertPath(path);
    return path.length > 0 && path.charCodeAt(0) === 47;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i1 = 0, len = paths.length; i1 < len; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `/${path}`;
        }
    }
    if (!joined) return ".";
    return normalize1(joined);
}
function relative1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 47) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 47) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i1) === 47) {
                    return to.slice(toStart + i1 + 1);
                } else if (i1 === 0) {
                    return to.slice(toStart + i1);
                }
            } else if (fromLen > length) {
                if (from.charCodeAt(fromStart + i1) === 47) {
                    lastCommonSep = i1;
                } else if (i1 === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (fromCode === 47) lastCommonSep = i1;
    }
    let out = "";
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || from.charCodeAt(i1) === 47) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (to.charCodeAt(toStart) === 47) ++toStart;
        return to.slice(toStart);
    }
}
function toNamespacedPath1(path) {
    return path;
}
function dirname1(path) {
    assertPath(path);
    if (path.length === 0) return ".";
    const hasRoot = path.charCodeAt(0) === 47;
    let end = -1;
    let matchedSlash = true;
    for(let i1 = path.length - 1; i1 >= 1; --i1){
        if (path.charCodeAt(i1) === 47) {
            if (!matchedSlash) {
                end = i1;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) return hasRoot ? "/" : ".";
    if (hasRoot && end === 1) return "//";
    return path.slice(0, end);
}
function basename1(path, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i1;
    if (ext !== undefined && ext.length > 0 && ext.length <= path.length) {
        if (ext.length === path.length && ext === path) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i1 = path.length - 1; i1 >= 0; --i1){
            const code = path.charCodeAt(i1);
            if (code === 47) {
                if (!matchedSlash) {
                    start = i1 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i1 + 1;
                }
                if (extIdx >= 0) {
                    if (code === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i1;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path.length;
        return path.slice(start, end);
    } else {
        for(i1 = path.length - 1; i1 >= 0; --i1){
            if (path.charCodeAt(i1) === 47) {
                if (!matchedSlash) {
                    start = i1 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i1 + 1;
            }
        }
        if (end === -1) return "";
        return path.slice(start, end);
    }
}
function extname1(path) {
    assertPath(path);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i1 = path.length - 1; i1 >= 0; --i1){
        const code = path.charCodeAt(i1);
        if (code === 47) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("/", pathObject);
}
function parse1(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path.length === 0) return ret;
    const isAbsolute = path.charCodeAt(0) === 47;
    let start;
    if (isAbsolute) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i1 = path.length - 1;
    let preDotState = 0;
    for(; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (code === 47) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute) {
                ret.base = ret.name = path.slice(1, end);
            } else {
                ret.base = ret.name = path.slice(startPart, end);
            }
        }
    } else {
        if (startPart === 0 && isAbsolute) {
            ret.name = path.slice(1, startDot);
            ret.base = path.slice(1, end);
        } else {
            ret.name = path.slice(startPart, startDot);
            ret.base = path.slice(startPart, end);
        }
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0) ret.dir = path.slice(0, startPart - 1);
    else if (isAbsolute) ret.dir = "/";
    return ret;
}
function fromFileUrl1(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function toFileUrl1(path) {
    if (!isAbsolute1(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
const mod1 = {
    sep: sep1,
    delimiter: delimiter1,
    resolve: resolve1,
    normalize: normalize1,
    isAbsolute: isAbsolute1,
    join: join1,
    relative: relative1,
    toNamespacedPath: toNamespacedPath1,
    dirname: dirname1,
    basename: basename1,
    extname: extname1,
    format: format1,
    parse: parse1,
    fromFileUrl: fromFileUrl1,
    toFileUrl: toFileUrl1
};
const SEP = isWindows1 ? "\\" : "/";
const SEP_PATTERN = isWindows1 ? /[\\/]+/ : /\/+/;
const path = isWindows1 ? mod : mod1;
const { join: join2 , normalize: normalize2  } = path;
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|", 
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function globToRegExp(glob, { extended =true , globstar: globstarOption = true , os =osType , caseInsensitive =false  } = {}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep = os == "windows" ? "(?:\\\\|/)+" : "/+";
    const sepMaybe = os == "windows" ? "(?:\\\\|/)*" : "/*";
    const seps = os == "windows" ? [
        "\\",
        "/"
    ] : [
        "/"
    ];
    const globstar = os == "windows" ? "(?:[^\\\\/]*(?:\\\\|/|$)+)*" : "(?:[^/]*(?:/|$)+)*";
    const wildcard = os == "windows" ? "[^\\\\/]*" : "[^/]*";
    const escapePrefix = os == "windows" ? "`" : "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] == "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] == "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] == ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i1 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type == "!") {
                    segment += wildcard;
                } else if (type != "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "+" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "@" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "?") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] == "!" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "*") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] == "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? sep : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar =false  } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize2(glob);
    }
    const s = SEP_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize2(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended =true , globstar =false  } = {}) {
    if (!globstar || globs.length == 0) {
        return join2(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path = glob;
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `${SEP}${path}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
const path1 = isWindows1 ? mod : mod1;
const posix = mod1;
const { basename: basename2 , delimiter: delimiter2 , dirname: dirname2 , extname: extname2 , format: format2 , fromFileUrl: fromFileUrl2 , isAbsolute: isAbsolute2 , join: join3 , normalize: normalize3 , parse: parse2 , relative: relative2 , resolve: resolve2 , sep: sep2 , toFileUrl: toFileUrl2 , toNamespacedPath: toNamespacedPath2 ,  } = path1;
if (isWindows()) {
    Deno.realPathSync = function(path) {
        let file = path instanceof URL ? fromFileUrl2(path) : path;
        if (!isAbsolute2(file)) {
            file = join3(Deno.cwd(), file);
        }
        file = normalize3(file);
        return file.replace(/^\w:\\/, (m)=>m[0].toUpperCase() + ":\\");
    };
}
function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i1 = 0; i1 <= b.length; i1++){
        matrix[i1] = [
            i1
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i11 = 1; i11 <= b.length; i11++){
        for(let j1 = 1; j1 <= a.length; j1++){
            if (b.charAt(i11 - 1) == a.charAt(j1 - 1)) {
                matrix[i11][j1] = matrix[i11 - 1][j1 - 1];
            } else {
                matrix[i11][j1] = Math.min(matrix[i11 - 1][j1 - 1] + 1, Math.min(matrix[i11][j1 - 1] + 1, matrix[i11 - 1][j1] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str) {
    return str.replace(/-([a-z])/g, (g)=>g[1].toUpperCase());
}
function underscoreToCamelCase(str) {
    return str.replace(/([a-z])([A-Z])/g, "$1_$2").toLowerCase().replace(/_([a-z])/g, (g)=>g[1].toUpperCase());
}
function getOption(flags, name) {
    while(name[0] === "-"){
        name = name.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option, options) {
    const optionNames = options.map((option)=>[
            option.name,
            ...option.aliases ?? []
        ]).flat().map((option)=>getFlag(option));
    return didYouMean(" Did you mean option", getFlag(option), optionNames);
}
function didYouMeanType(type, types) {
    return didYouMean(" Did you mean type", type, types);
}
function didYouMean(message, type, types) {
    const match = closest(type, types);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name) {
    if (name.startsWith("-")) {
        return name;
    }
    if (name.length > 1) {
        return `--${name}`;
    }
    return `-${name}`;
}
function isOption(option, name) {
    return option.name === name || option.aliases && option.aliases.indexOf(name) !== -1;
}
function matchWildCardOptions(name, flags) {
    for (const option of flags){
        if (option.name.indexOf("*") === -1) {
            continue;
        }
        let matched = matchWildCardOption(name, option);
        if (matched) {
            matched = {
                ...matched,
                name
            };
            flags.push(matched);
            return matched;
        }
    }
}
function matchWildCardOption(name, option) {
    const parts = option.name.split(".");
    const parts2 = name.split(".");
    if (parts.length !== parts2.length) {
        return false;
    }
    const count = Math.max(parts.length, parts2.length);
    for(let i1 = 0; i1 < count; i1++){
        if (parts[i1] !== parts2[i1] && parts[i1] !== "*") {
            return false;
        }
    }
    return option;
}
function closest(str, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i1 = 0; i1 < arr.length; i1++){
        const dist = distance(str, arr[i1]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i1;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOption extends FlagsError {
    constructor(option, options){
        super(`Unknown required option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownRequiredOption.prototype);
    }
}
class UnknownConflictingOption extends FlagsError {
    constructor(option, options){
        super(`Unknown conflicting option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownConflictingOption.prototype);
    }
}
class UnknownType extends FlagsError {
    constructor(type, types){
        super(`Unknown type "${type}".${didYouMeanType(type, types)}`);
        Object.setPrototypeOf(this, UnknownType.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOption extends ValidationError {
    constructor(name){
        super(`Option "${getFlag(name).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOption.prototype);
    }
}
class InvalidOption extends ValidationError {
    constructor(option, options){
        super(`Invalid option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, InvalidOption.prototype);
    }
}
class UnknownOption extends ValidationError {
    constructor(option, options){
        super(`Unknown option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownOption.prototype);
    }
}
class MissingOptionValue extends ValidationError {
    constructor(option){
        super(`Missing value for option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
class InvalidOptionValue extends ValidationError {
    constructor(option, expected, value){
        super(`Option "${getFlag(option)}" must be of type "${expected}", but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValue.prototype);
    }
}
class OptionNotCombinable extends ValidationError {
    constructor(option){
        super(`Option "${getFlag(option)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinable.prototype);
    }
}
class ConflictingOption extends ValidationError {
    constructor(option, conflictingOption){
        super(`Option "${getFlag(option)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOption.prototype);
    }
}
class DependingOption extends ValidationError {
    constructor(option, dependingOption){
        super(`Option "${getFlag(option)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOption.prototype);
    }
}
class MissingRequiredOption extends ValidationError {
    constructor(option){
        super(`Missing required option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingRequiredOption.prototype);
    }
}
class RequiredArgumentFollowsOptionalArgument extends ValidationError {
    constructor(arg){
        super(`An required argument cannot follow an optional argument, but "${arg}"  is defined as required.`);
        Object.setPrototypeOf(this, RequiredArgumentFollowsOptionalArgument.prototype);
    }
}
class ArgumentFollowsVariadicArgument extends ValidationError {
    constructor(arg){
        super(`An argument cannot follow an variadic argument, but got "${arg}".`);
        Object.setPrototypeOf(this, ArgumentFollowsVariadicArgument.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label , name , value , type  }, expected){
        super(`${label} "${name}" must be of type "${type}", but got "${value}".` + (expected ? ` Expected values: ${expected.map((value)=>`"${value}"`).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
var OptionType;
(function(OptionType) {
    OptionType["STRING"] = "string";
    OptionType["NUMBER"] = "number";
    OptionType["INTEGER"] = "integer";
    OptionType["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {}));
function didYouMeanCommand(command, commands, excludes = []) {
    const commandNames = commands.map((command)=>command.getName()).filter((command)=>!excludes.includes(command));
    return didYouMean(" Did you mean command", command, commandNames);
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition,
        equalsSign: args.includes("=")
    };
}
function parseArgumentsDefinition(argsDefinition, validate = true, all) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg of parts){
        if (validate && hasVariadic) {
            throw new ArgumentFollowsVariadicArgument(arg);
        }
        const parts1 = arg.split(ARGUMENT_DETAILS_REGEX);
        if (!parts1[1]) {
            if (all) {
                argumentDetails.push(parts1[0]);
            }
            continue;
        }
        const type = parts1[2] || OptionType.STRING;
        const details = {
            optionalValue: arg[0] === "[",
            requiredValue: arg[0] === "<",
            name: parts1[1],
            action: parts1[3] || type,
            variadic: false,
            list: type ? arg.indexOf(type + "[]") !== -1 : false,
            type
        };
        if (validate && !details.optionalValue && hasOptional) {
            throw new RequiredArgumentFollowsOptionalArgument(details.name);
        }
        if (arg[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        argumentDetails.push(details);
    }
    return argumentDetails;
}
function dedent(str) {
    const lines = str.split(/\r?\n|\r/g);
    let text = "";
    let indent = 0;
    for (const line of lines){
        if (text || line.trim()) {
            if (!text) {
                text = line.trimStart();
                indent = line.length - text.length;
            } else {
                text += line.slice(indent);
            }
            text += "\n";
        }
    }
    return text.trimEnd();
}
function getDescription(description, __short) {
    return __short ? description.trim().split("\n", 1)[0] : dedent(description);
}
class CommandError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    constructor(message, { exitCode  } = {}){
        super(message);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 1;
    }
}
class DuplicateOptionName extends CommandError {
    constructor(name){
        super(`Option with name "${getFlag(name)}" already exists.`);
        Object.setPrototypeOf(this, DuplicateOptionName.prototype);
    }
}
class MissingCommandName extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandName.prototype);
    }
}
class DuplicateCommandName extends CommandError {
    constructor(name){
        super(`Duplicate command name "${name}".`);
        Object.setPrototypeOf(this, DuplicateCommandName.prototype);
    }
}
class DuplicateCommandAlias extends CommandError {
    constructor(alias){
        super(`Duplicate command alias "${alias}".`);
        Object.setPrototypeOf(this, DuplicateCommandAlias.prototype);
    }
}
class CommandNotFound extends CommandError {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class DuplicateType extends CommandError {
    constructor(name){
        super(`Type with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateType.prototype);
    }
}
class DuplicateCompletion extends CommandError {
    constructor(name){
        super(`Completion with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletion.prototype);
    }
}
class DuplicateExample extends CommandError {
    constructor(name){
        super(`Example with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExample.prototype);
    }
}
class DuplicateEnvironmentVariable extends CommandError {
    constructor(name){
        super(`Environment variable with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvironmentVariable.prototype);
    }
}
class MissingRequiredEnvVar extends ValidationError1 {
    constructor(envVar){
        super(`Missing required environment variable "${envVar.names[0]}".`);
        Object.setPrototypeOf(this, MissingRequiredEnvVar.prototype);
    }
}
class EnvironmentVariableSingleValue extends CommandError {
    constructor(name){
        super(`An environment variable can only have one value, but "${name}" has more than one.`);
        Object.setPrototypeOf(this, EnvironmentVariableSingleValue.prototype);
    }
}
class EnvironmentVariableOptionalValue extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an optional value, but "${name}" is defined as optional.`);
        Object.setPrototypeOf(this, EnvironmentVariableOptionalValue.prototype);
    }
}
class EnvironmentVariableVariadicValue extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an variadic value, but "${name}" is defined as variadic.`);
        Object.setPrototypeOf(this, EnvironmentVariableVariadicValue.prototype);
    }
}
class DefaultCommandNotFound extends CommandError {
    constructor(name, commands){
        super(`Default command "${name}" not found.${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFound.prototype);
    }
}
class CommandExecutableNotFound extends CommandError {
    constructor(name){
        super(`Command executable not found: ${name}`);
        Object.setPrototypeOf(this, CommandExecutableNotFound.prototype);
    }
}
class UnknownCompletionCommand extends CommandError {
    constructor(name, commands){
        super(`Auto-completion failed. Unknown command "${name}".${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommand.prototype);
    }
}
class UnknownCommand extends ValidationError1 {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class NoArgumentsAllowed extends ValidationError1 {
    constructor(name){
        super(`No arguments allowed for command "${name}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowed.prototype);
    }
}
class MissingArguments extends ValidationError1 {
    constructor(args){
        super("Missing argument(s): " + args.join(", "));
        Object.setPrototypeOf(this, MissingArguments.prototype);
    }
}
class MissingArgument extends ValidationError1 {
    constructor(arg){
        super(`Missing argument "${arg}".`);
        Object.setPrototypeOf(this, MissingArgument.prototype);
    }
}
class TooManyArguments extends ValidationError1 {
    constructor(args){
        super(`Too many arguments: ${args.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArguments.prototype);
    }
}
const __boolean = (type)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type.value)) {
        return false;
    }
    throw new InvalidTypeError(type);
};
const number = (type)=>{
    const value = Number(type.value);
    if (Number.isFinite(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const string = ({ value  })=>{
    return value;
};
function validateFlags(opts, values, optionNameMap = {}) {
    if (!opts.flags?.length) {
        return;
    }
    const defaultValues = setDefaultValues(opts, values, optionNameMap);
    const optionNames = Object.keys(values);
    if (!optionNames.length && opts.allowEmpty) {
        return;
    }
    const options = optionNames.map((name)=>({
            name,
            option: getOption(opts.flags, optionNameMap[name])
        }));
    for (const { name , option  } of options){
        if (!option) {
            throw new UnknownOption(name, opts.flags);
        }
        if (validateStandaloneOption(option, options, optionNames, defaultValues)) {
            return;
        }
        validateConflictingOptions(option, values);
        validateDependingOptions(option, values, defaultValues);
        validateRequiredValues(option, values, name);
    }
    validateRequiredOptions(options, values, opts);
}
function setDefaultValues(opts, values, optionNameMap = {}) {
    const defaultValues = {};
    if (!opts.flags?.length) {
        return defaultValues;
    }
    for (const option of opts.flags){
        let name;
        let defaultValue = undefined;
        if (option.name.startsWith("no-")) {
            const propName = option.name.replace(/^no-/, "");
            if (propName in values) {
                continue;
            }
            const positiveOption = getOption(opts.flags, propName);
            if (positiveOption) {
                continue;
            }
            name = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name) {
            name = paramCaseToCamelCase(option.name);
        }
        if (!(name in optionNameMap)) {
            optionNameMap[name] = option.name;
        }
        const hasDefaultValue = (!opts.ignoreDefaults || typeof opts.ignoreDefaults[name] === "undefined") && typeof values[name] === "undefined" && (typeof option.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            values[name] = getDefaultValue(option) ?? defaultValue;
            defaultValues[option.name] = true;
            if (typeof option.value === "function") {
                values[name] = option.value(values[name]);
            }
        }
    }
    return defaultValues;
}
function validateStandaloneOption(option, options, optionNames, defaultValues) {
    if (!option.standalone) {
        return false;
    }
    if (optionNames.length === 1) {
        return true;
    }
    if (options.every((opt)=>opt.option && (option === opt.option || defaultValues[opt.option.name]))) {
        return true;
    }
    throw new OptionNotCombinable(option.name);
}
function validateConflictingOptions(option, values) {
    option.conflicts?.forEach((flag)=>{
        if (isset(flag, values)) {
            throw new ConflictingOption(option.name, flag);
        }
    });
}
function validateDependingOptions(option, values, defaultValues) {
    option.depends?.forEach((flag)=>{
        if (!isset(flag, values) && !defaultValues[option.name]) {
            throw new DependingOption(option.name, flag);
        }
    });
}
function validateRequiredValues(option, values, name) {
    const isArray = (option.args?.length || 0) > 1;
    option.args?.forEach((arg, i1)=>{
        if (arg.requiredValue && (typeof values[name] === "undefined" || isArray && typeof values[name][i1] === "undefined")) {
            throw new MissingOptionValue(option.name);
        }
    });
}
function validateRequiredOptions(options, values, opts) {
    if (!opts.flags?.length) {
        return;
    }
    for (const option of opts.flags){
        if (option.required && !(paramCaseToCamelCase(option.name) in values)) {
            if ((!option.conflicts || !option.conflicts.find((flag)=>!!values[flag])) && !options.find((opt)=>opt.option?.conflicts?.find((flag)=>flag === option.name))) {
                throw new MissingRequiredOption(option.name);
            }
        }
    }
}
function isset(flag, values) {
    const name = paramCaseToCamelCase(flag);
    return typeof values[name] !== "undefined";
}
const integer = (type)=>{
    const value = Number(type.value);
    if (Number.isInteger(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const Types = {
    [OptionType.STRING]: string,
    [OptionType.NUMBER]: number,
    [OptionType.INTEGER]: integer,
    [OptionType.BOOLEAN]: __boolean
};
function parseFlags(args, opts = {}) {
    args = args.slice();
    !opts.flags && (opts.flags = []);
    let inLiteral = false;
    let negate = false;
    const flags = {};
    const optionNameMap = {};
    let literal = [];
    let unknown = [];
    let stopEarly = null;
    opts.flags.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOption(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOption(flag, opts.flags ?? []);
            }
        });
    });
    for(let argsIndex = 0; argsIndex < args.length; argsIndex++){
        let option;
        let optionArgs;
        let current = args[argsIndex];
        let currentValue;
        if (inLiteral) {
            literal.push(current);
            continue;
        }
        if (current === "--") {
            inLiteral = true;
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        const next = ()=>currentValue ?? args[argsIndex + 1];
        if (isFlag) {
            const isShort = current[1] !== "-";
            const isLong = isShort ? false : current.length > 3 && current[2] !== "-";
            if (!isShort && !isLong) {
                throw new InvalidOption(current, opts.flags);
            }
            const equalSignIndex = current.indexOf("=");
            if (equalSignIndex > -1) {
                currentValue = current.slice(equalSignIndex + 1) || undefined;
                current = current.slice(0, equalSignIndex);
            }
            if (isShort && current.length > 2 && current[2] !== ".") {
                args.splice(argsIndex, 1, ...splitFlags(current));
                current = args[argsIndex];
            } else if (isLong && current.startsWith("--no-")) {
                negate = true;
            }
            option = getOption(opts.flags, current);
            if (!option) {
                if (opts.flags.length) {
                    const name = current.replace(/^-+/g, "");
                    option = matchWildCardOptions(name, opts.flags);
                    if (!option) {
                        throw new UnknownOption(current, opts.flags);
                    }
                }
                if (!option) {
                    option = {
                        name: current.replace(/^-+/, ""),
                        optionalValue: true,
                        type: OptionType.STRING
                    };
                }
            }
            const positiveName = negate ? option.name.replace(/^no-?/, "") : option.name;
            const propName = paramCaseToCamelCase(positiveName);
            if (typeof flags[propName] !== "undefined") {
                if (!opts.flags.length) {
                    option.collect = true;
                } else if (!option.collect) {
                    throw new DuplicateOption(current);
                }
            }
            optionArgs = option.args?.length ? option.args : [
                {
                    type: option.type,
                    requiredValue: option.requiredValue,
                    optionalValue: option.optionalValue,
                    variadic: option.variadic,
                    list: option.list,
                    separator: option.separator
                }
            ];
            let optionArgsIndex = 0;
            let inOptionalArg = false;
            const previous = flags[propName];
            parseNext(option, optionArgs);
            if (typeof flags[propName] === "undefined") {
                if (optionArgs[optionArgsIndex].requiredValue) {
                    throw new MissingOptionValue(option.name);
                } else if (typeof option.default !== "undefined") {
                    flags[propName] = getDefaultValue(option);
                } else {
                    flags[propName] = true;
                }
            }
            if (option.value) {
                flags[propName] = option.value(flags[propName], previous);
            } else if (option.collect) {
                const value = typeof previous !== "undefined" ? Array.isArray(previous) ? previous : [
                    previous
                ] : [];
                value.push(flags[propName]);
                flags[propName] = value;
            }
            optionNameMap[propName] = option.name;
            opts.option?.(option, flags[propName]);
            function parseNext(option, optionArgs) {
                const arg = optionArgs[optionArgsIndex];
                if (!arg) {
                    const flag = next();
                    throw new UnknownOption(flag, opts.flags ?? []);
                }
                if (!arg.type) {
                    arg.type = OptionType.BOOLEAN;
                }
                if (option.args?.length) {
                    if ((typeof arg.optionalValue === "undefined" || arg.optionalValue === false) && typeof arg.requiredValue === "undefined") {
                        arg.requiredValue = true;
                    }
                } else {
                    if (arg.type !== OptionType.BOOLEAN && (typeof arg.optionalValue === "undefined" || arg.optionalValue === false) && typeof arg.requiredValue === "undefined") {
                        arg.requiredValue = true;
                    }
                }
                if (arg.requiredValue) {
                    if (inOptionalArg) {
                        throw new RequiredArgumentFollowsOptionalArgument(option.name);
                    }
                } else {
                    inOptionalArg = true;
                }
                if (negate) {
                    flags[propName] = false;
                    return;
                }
                let result1;
                let increase = false;
                if (arg.list && hasNext(arg)) {
                    const parsed = next().split(arg.separator || ",").map((nextValue)=>{
                        const value = parseValue(option, arg, nextValue);
                        if (typeof value === "undefined") {
                            throw new InvalidOptionValue(option.name, arg.type ?? "?", nextValue);
                        }
                        return value;
                    });
                    if (parsed?.length) {
                        result1 = parsed;
                    }
                } else {
                    if (hasNext(arg)) {
                        result1 = parseValue(option, arg, next());
                    } else if (arg.optionalValue && arg.type === OptionType.BOOLEAN) {
                        result1 = true;
                    }
                }
                if (increase && typeof currentValue === "undefined") {
                    argsIndex++;
                    if (!arg.variadic) {
                        optionArgsIndex++;
                    } else if (optionArgs[optionArgsIndex + 1]) {
                        throw new ArgumentFollowsVariadicArgument(next());
                    }
                }
                if (typeof result1 !== "undefined" && (optionArgs.length > 1 || arg.variadic)) {
                    if (!flags[propName]) {
                        flags[propName] = [];
                    }
                    flags[propName].push(result1);
                    if (hasNext(arg)) {
                        parseNext(option, optionArgs);
                    }
                } else {
                    flags[propName] = result1;
                }
                function hasNext(arg) {
                    const nextValue = currentValue ?? args[argsIndex + 1];
                    if (!nextValue) {
                        return false;
                    }
                    if (optionArgs.length > 1 && optionArgsIndex >= optionArgs.length) {
                        return false;
                    }
                    if (arg.requiredValue) {
                        return true;
                    }
                    if (option.equalsSign && arg.optionalValue && !arg.variadic && typeof currentValue === "undefined") {
                        return false;
                    }
                    if (arg.optionalValue || arg.variadic) {
                        return nextValue[0] !== "-" || arg.type === OptionType.NUMBER && !isNaN(Number(nextValue));
                    }
                    return false;
                }
                function parseValue(option, arg, value) {
                    const type = arg.type || OptionType.STRING;
                    const result1 = opts.parse ? opts.parse({
                        label: "Option",
                        type,
                        name: `--${option.name}`,
                        value
                    }) : parseFlagValue(option, arg, value);
                    if (typeof result1 !== "undefined") {
                        increase = true;
                    }
                    return result1;
                }
            }
        } else {
            if (opts.stopEarly) {
                stopEarly = current;
                break;
            }
            unknown.push(current);
        }
    }
    if (stopEarly) {
        const stopEarlyArgIndex = args.indexOf(stopEarly);
        if (stopEarlyArgIndex !== -1) {
            const doubleDashIndex = args.indexOf("--");
            unknown = args.slice(stopEarlyArgIndex, doubleDashIndex === -1 ? undefined : doubleDashIndex);
            if (doubleDashIndex !== -1) {
                literal = args.slice(doubleDashIndex + 1);
            }
        }
    }
    validateFlags(opts, flags, optionNameMap);
    const result1 = Object.keys(flags).reduce((result1, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result1, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result1[subKey] = flags[key];
                } else {
                    result1[subKey] = result1[subKey] ?? {};
                }
                return result1[subKey];
            }, result1);
        } else {
            result1[key] = flags[key];
        }
        return result1;
    }, {});
    return {
        flags: result1,
        unknown,
        literal
    };
}
function splitFlags(flag) {
    const normalized = [];
    const flags = flag.slice(1).split("");
    if (isNaN(Number(flag[flag.length - 1]))) {
        flags.forEach((val)=>normalized.push(`-${val}`));
    } else {
        normalized.push(`-${flags.shift()}`);
        if (flags.length) {
            normalized.push(flags.join(""));
        }
    }
    return normalized;
}
function parseFlagValue(option, arg, value) {
    const type = arg.type || OptionType.STRING;
    const parseType = Types[type];
    if (!parseType) {
        throw new UnknownType(type, Object.keys(Types));
    }
    return parseType({
        label: "Option",
        type,
        name: `--${option.name}`,
        value
    });
}
const { Deno: Deno1  } = globalThis;
const noColor = typeof Deno1?.noColor === "boolean" ? Deno1.noColor : true;
let enabled = !noColor;
function setColorEnabled(value) {
    if (noColor) {
        return;
    }
    enabled = value;
}
function getColorEnabled() {
    return enabled;
}
function code(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run(str, code) {
    return enabled ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function reset(str) {
    return run(str, code([
        0
    ], 0));
}
function bold(str) {
    return run(str, code([
        1
    ], 22));
}
function dim(str) {
    return run(str, code([
        2
    ], 22));
}
function italic(str) {
    return run(str, code([
        3
    ], 23));
}
function underline(str) {
    return run(str, code([
        4
    ], 24));
}
function inverse(str) {
    return run(str, code([
        7
    ], 27));
}
function hidden(str) {
    return run(str, code([
        8
    ], 28));
}
function strikethrough(str) {
    return run(str, code([
        9
    ], 29));
}
function black(str) {
    return run(str, code([
        30
    ], 39));
}
function red(str) {
    return run(str, code([
        31
    ], 39));
}
function green(str) {
    return run(str, code([
        32
    ], 39));
}
function yellow(str) {
    return run(str, code([
        33
    ], 39));
}
function blue(str) {
    return run(str, code([
        34
    ], 39));
}
function magenta(str) {
    return run(str, code([
        35
    ], 39));
}
function cyan(str) {
    return run(str, code([
        36
    ], 39));
}
function white(str) {
    return run(str, code([
        37
    ], 39));
}
function gray(str) {
    return brightBlack(str);
}
function brightBlack(str) {
    return run(str, code([
        90
    ], 39));
}
function brightRed(str) {
    return run(str, code([
        91
    ], 39));
}
function brightGreen(str) {
    return run(str, code([
        92
    ], 39));
}
function brightYellow(str) {
    return run(str, code([
        93
    ], 39));
}
function brightBlue(str) {
    return run(str, code([
        94
    ], 39));
}
function brightMagenta(str) {
    return run(str, code([
        95
    ], 39));
}
function brightCyan(str) {
    return run(str, code([
        96
    ], 39));
}
function brightWhite(str) {
    return run(str, code([
        97
    ], 39));
}
function bgBlack(str) {
    return run(str, code([
        40
    ], 49));
}
function bgRed(str) {
    return run(str, code([
        41
    ], 49));
}
function bgGreen(str) {
    return run(str, code([
        42
    ], 49));
}
function bgYellow(str) {
    return run(str, code([
        43
    ], 49));
}
function bgBlue(str) {
    return run(str, code([
        44
    ], 49));
}
function bgMagenta(str) {
    return run(str, code([
        45
    ], 49));
}
function bgCyan(str) {
    return run(str, code([
        46
    ], 49));
}
function bgWhite(str) {
    return run(str, code([
        47
    ], 49));
}
function bgBrightBlack(str) {
    return run(str, code([
        100
    ], 49));
}
function bgBrightRed(str) {
    return run(str, code([
        101
    ], 49));
}
function bgBrightGreen(str) {
    return run(str, code([
        102
    ], 49));
}
function bgBrightYellow(str) {
    return run(str, code([
        103
    ], 49));
}
function bgBrightBlue(str) {
    return run(str, code([
        104
    ], 49));
}
function bgBrightMagenta(str) {
    return run(str, code([
        105
    ], 49));
}
function bgBrightCyan(str) {
    return run(str, code([
        106
    ], 49));
}
function bgBrightWhite(str) {
    return run(str, code([
        107
    ], 49));
}
function clampAndTruncate(n, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n, max), min));
}
function rgb8(str, color) {
    return run(str, code([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str, color) {
    return run(str, code([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str, color) {
    if (typeof color === "number") {
        return run(str, code([
            38,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 39));
    }
    return run(str, code([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 39));
}
function bgRgb24(str, color) {
    if (typeof color === "number") {
        return run(str, code([
            48,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 49));
    }
    return run(str, code([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 49));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-nq-uy=><~]))", 
].join("|"), "g");
function stripColor(string) {
    return string.replace(ANSI_PATTERN, "");
}
const mod2 = {
    setColorEnabled: setColorEnabled,
    getColorEnabled: getColorEnabled,
    reset: reset,
    bold: bold,
    dim: dim,
    italic: italic,
    underline: underline,
    inverse: inverse,
    hidden: hidden,
    strikethrough: strikethrough,
    black: black,
    red: red,
    green: green,
    yellow: yellow,
    blue: blue,
    magenta: magenta,
    cyan: cyan,
    white: white,
    gray: gray,
    brightBlack: brightBlack,
    brightRed: brightRed,
    brightGreen: brightGreen,
    brightYellow: brightYellow,
    brightBlue: brightBlue,
    brightMagenta: brightMagenta,
    brightCyan: brightCyan,
    brightWhite: brightWhite,
    bgBlack: bgBlack,
    bgRed: bgRed,
    bgGreen: bgGreen,
    bgYellow: bgYellow,
    bgBlue: bgBlue,
    bgMagenta: bgMagenta,
    bgCyan: bgCyan,
    bgWhite: bgWhite,
    bgBrightBlack: bgBrightBlack,
    bgBrightRed: bgBrightRed,
    bgBrightGreen: bgBrightGreen,
    bgBrightYellow: bgBrightYellow,
    bgBrightBlue: bgBrightBlue,
    bgBrightMagenta: bgBrightMagenta,
    bgBrightCyan: bgBrightCyan,
    bgBrightWhite: bgBrightWhite,
    rgb8: rgb8,
    bgRgb8: bgRgb8,
    rgb24: rgb24,
    bgRgb24: bgRgb24,
    stripColor: stripColor
};
class Type {
}
class BooleanType extends Type {
    parse(type) {
        return __boolean(type);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class StringType extends Type {
    parse(type) {
        return string(type);
    }
}
class FileType extends StringType {
    constructor(){
        super();
    }
}
class NumberType extends Type {
    parse(type) {
        return number(type);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    options;
    get length() {
        return this.toString().length;
    }
    static from(value) {
        const cell = new this(value);
        if (value instanceof Cell) {
            cell.options = {
                ...value.options
            };
        }
        return cell;
    }
    constructor(value){
        this.value = value;
        this.options = {};
    }
    toString() {
        return this.value.toString();
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        const cell = new Cell(value ?? this);
        cell.options = {
            ...this.options
        };
        return cell;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
    value;
}
class Row extends Array {
    options = {};
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder());
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
function consumeWords(length, content) {
    let consumed = "";
    const words = content.split("\n")[0]?.split(/ /g);
    for(let i1 = 0; i1 < words.length; i1++){
        const word = words[i1];
        if (consumed) {
            const nextLength = strLength(word);
            const consumedLength = strLength(consumed);
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i1 > 0 ? " " : "") + word;
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    const cellLengths = rows.map((row)=>{
        const cell = row[index];
        const cellValue = cell instanceof Cell && cell.getColSpan() > 1 ? "" : cell?.toString() || "";
        return cellValue.split("\n").map((line)=>{
            const str = typeof maxWidth === "undefined" ? line : consumeWords(maxWidth, line);
            return strLength(str) || 0;
        });
    }).flat();
    return Math.max(...cellLengths);
}
const strLength = (str)=>{
    str = stripColor(str);
    let length = 0;
    for(let i1 = 0; i1 < str.length; i1++){
        const charCode = str.charCodeAt(i1);
        if (charCode >= 19968 && charCode <= 40869) {
            length += 2;
        } else {
            length += 1;
        }
    }
    return length;
};
class TableLayout {
    constructor(table, options){
        this.table = table;
        this.options = options;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const rows = this.#getRows();
        const columns = Math.max(...rows.map((row)=>row.length));
        for (const row of rows){
            const length = row.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i1 = 0; i1 < diff; i1++){
                    row.push(this.createCell(null, row));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const minColWidth = Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth;
            const maxColWidth = Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth;
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding;
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    #getRows() {
        const header = this.table.getHeader();
        const rows = header ? [
            header,
            ...this.table
        ] : this.table.slice();
        const hasSpan = rows.some((row)=>row.some((cell)=>cell instanceof Cell && (cell.getColSpan() > 1 || cell.getRowSpan() > 1)));
        if (hasSpan) {
            return this.spanRows(rows);
        }
        return rows.map((row)=>{
            const newRow = this.createRow(row);
            return newRow.map((cell)=>this.createCell(cell, newRow));
        });
    }
    spanRows(_rows, rowIndex = 0, colIndex = 0, rowSpan = [], colSpan = 1) {
        const rows = _rows;
        if (rowIndex >= rows.length && rowSpan.every((span)=>span === 1)) {
            return rows;
        } else if (rows[rowIndex] && colIndex >= rows[rowIndex].length && colIndex >= rowSpan.length && colSpan === 1) {
            return this.spanRows(rows, ++rowIndex, 0, rowSpan, 1);
        }
        if (colSpan > 1) {
            colSpan--;
            rowSpan[colIndex] = rowSpan[colIndex - 1];
            rows[rowIndex].splice(colIndex - 1, 0, rows[rowIndex][colIndex - 1]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        if (colIndex === 0) {
            rows[rowIndex] = this.createRow(rows[rowIndex] || []);
        }
        if (rowSpan[colIndex] > 1) {
            rowSpan[colIndex]--;
            rows[rowIndex].splice(colIndex, 0, rows[rowIndex - 1][colIndex]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        rows[rowIndex][colIndex] = this.createCell(rows[rowIndex][colIndex] || null, rows[rowIndex]);
        colSpan = rows[rowIndex][colIndex].getColSpan();
        rowSpan[colIndex] = rows[rowIndex][colIndex].getRowSpan();
        return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row) {
        return Cell.from(cell ?? "").border(row.getBorder(), false).align(row.getAlign(), false);
    }
    renderRows(opts) {
        let result1 = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result1 += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result1.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result1 = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result1 += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result1 += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result1 += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result1 += this.options.chars.right;
            } else if (opts.hasBorder) {
                result1 += " ";
            }
        }
        result1 += "\n";
        if (isMultilineRow) {
            return result1 + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder) {
            result1 += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result1 += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result1;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result1 = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result1 += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result1 += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current , next  } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next);
        if (opts.hasBorder) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        result1 += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        return result1;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, strLength(cell.toString()));
        let words = consumeWords(length, cell.toString());
        const breakWord = strLength(words) > length;
        if (breakWord) {
            words = words.slice(0, length);
        }
        const next = cell.toString().slice(words.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - strLength(words);
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words;
        } else if (align === "left") {
            current = words + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result1 = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result1 += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result1.length ? " ".repeat(this.options.indent) + result1 + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1;
        let result1 = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += " ";
                }
            } else if (b1Border && b2Border) {
                result1 += this.options.chars.leftMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result1 += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result1 += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result1 += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result1 += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result1 += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result1 += this.options.chars.rightMid;
                } else {
                    result1 += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result1 += this.options.chars.bottom;
                } else {
                    result1 += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result1 += this.options.chars.top;
                } else {
                    result1 += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result1 += this.options.chars.topRight;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result1 += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += " ";
                }
                return result1;
            }
        } else if (b1Border && b2Border) {
            result1 += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result1 += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result1 += this.options.chars.top.repeat(length);
        } else {
            result1 += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result1 += this.options.chars.rightMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b2Border) {
                result1 += this.options.chars.topRight;
            } else {
                result1 += " ";
            }
        }
        return result1;
    }
    table;
    options;
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        }
    };
    headerRow;
    static from(rows) {
        const table = new this(...rows);
        if (rows instanceof Table) {
            table.options = {
                ...rows.options
            };
            table.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)));
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()));
        table.options = {
            ...this.options
        };
        table.headerRow = this.headerRow?.clone();
        return table;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        console.log(this.toString());
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder : false));
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class HelpGenerator {
    indent;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd, options = {}){
        this.cmd = cmd;
        this.indent = 2;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            long: false,
            ...options
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result1 = this.generateHeader() + this.generateMeta() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples();
        setColorEnabled(areColorsEnabled);
        return result1;
    }
    generateHeader() {
        const usage = this.cmd.getUsage();
        const rows = [
            [
                bold("Usage:"),
                magenta(this.cmd.getPath() + (usage ? " " + highlightArguments(usage, this.options.types) : "")), 
            ], 
        ];
        const version = this.cmd.getVersion();
        if (version) {
            rows.push([
                bold("Version:"),
                yellow(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).indent(this.indent).padding(1).toString() + "\n";
    }
    generateMeta() {
        const meta = Object.entries(this.cmd.getMeta());
        if (!meta.length) {
            return "";
        }
        const rows = [];
        for (const [name, value] of meta){
            rows.push([
                bold(`${name}: `) + value
            ]);
        }
        return "\n" + Table.from(rows).indent(this.indent).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                dedent(this.cmd.getDescription())
            ], 
        ]).indent(this.indent * 2).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options = this.cmd.getOptions(false);
        if (!options.length) {
            return "";
        }
        let groups = [];
        const hasGroups = options.some((option)=>option.groupName);
        if (hasGroups) {
            for (const option of options){
                let group = groups.find((group)=>group.name === option.groupName);
                if (!group) {
                    group = {
                        name: option.groupName,
                        options: []
                    };
                    groups.push(group);
                }
                group.options.push(option);
            }
        } else {
            groups = [
                {
                    name: "Options",
                    options
                }
            ];
        }
        let result1 = "";
        for (const group1 of groups){
            result1 += this.generateOptionGroup(group1);
        }
        return result1;
    }
    generateOptionGroup(group) {
        if (!group.options.length) {
            return "";
        }
        const hasTypeDefinitions = !!group.options.find((option)=>!!option.typeDefinition);
        if (hasTypeDefinitions) {
            return this.label(group.name ?? "Options") + Table.from([
                ...group.options.map((option)=>[
                        option.flags.map((flag)=>blue(flag)).join(", "),
                        highlightArguments(option.typeDefinition || "", this.options.types),
                        red(bold("-")),
                        getDescription(option.description, !this.options.long),
                        this.generateHints(option), 
                    ]), 
            ]).padding([
                2,
                2,
                1,
                2
            ]).indent(this.indent * 2).maxColWidth([
                60,
                60,
                1,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label(group.name ?? "Options") + Table.from([
            ...group.options.map((option)=>[
                    option.flags.map((flag)=>blue(flag)).join(", "),
                    red(bold("-")),
                    getDescription(option.description, !this.options.long),
                    this.generateHints(option), 
                ]), 
        ]).indent(this.indent * 2).maxColWidth([
            60,
            1,
            80,
            60
        ]).padding([
            2,
            1,
            2
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands = this.cmd.getCommands(false);
        if (!commands.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands.find((command)=>!!command.getArgsDefinition());
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name)=>blue(name)).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red(bold("-")),
                        command.getShortDescription(), 
                    ]), 
            ]).indent(this.indent * 2).maxColWidth([
                60,
                60,
                1,
                80
            ]).padding([
                2,
                2,
                1,
                2
            ]).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name)=>blue(name)).join(", "),
                    red(bold("-")),
                    command.getShortDescription(), 
                ]), 
        ]).maxColWidth([
            60,
            1,
            80
        ]).padding([
            2,
            1,
            2
        ]).indent(this.indent * 2).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name)=>blue(name)).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    red(bold("-")),
                    this.options.long ? dedent(envVar.description) : envVar.description.trim().split("\n", 1)[0], 
                ]), 
        ]).padding([
            2,
            2,
            1
        ]).indent(this.indent * 2).maxColWidth([
            60,
            60,
            1,
            80
        ]).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim(bold(`${capitalize(example.name)}:`)),
                dedent(example.description), 
            ])).padding(1).indent(this.indent * 2).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow(`required`));
        typeof option.default !== "undefined" && hints.push(bold(`Default: `) + inspect(option.default, this.options.colors));
        option.depends?.length && hints.push(yellow(bold(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red(bold(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type instanceof Type) {
            const possibleValues = type.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold(`Values: `) + possibleValues.map((value)=>inspect(value, this.options.colors)).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + " ".repeat(this.indent) + bold(`${label}:`) + "\n\n";
    }
    cmd;
}
function capitalize(string) {
    return (string?.charAt(0).toUpperCase() + string.slice(1)) ?? "";
}
function inspect(value, colors) {
    return Deno.inspect(value, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition, false, true).map((arg)=>typeof arg === "string" ? arg : highlightArgumentDetails(arg, types)).join(" ");
}
function highlightArgumentDetails(arg, types = true) {
    let str = "";
    str += yellow(arg.optionalValue ? "[" : "<");
    let name = "";
    name += arg.name;
    if (arg.variadic) {
        name += "...";
    }
    name = magenta(name);
    str += name;
    if (types) {
        str += yellow(":");
        str += red(arg.type);
        if (arg.list) {
            str += green("[]");
        }
    }
    str += yellow(arg.optionalValue ? "]" : ">");
    return str;
}
class IntegerType extends Type {
    parse(type) {
        return integer(type);
    }
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    _usage;
    fn;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = false;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOption;
    _helpOption;
    _help;
    _shouldExit;
    _meta = {};
    _groupName;
    versionOption(flags, desc, opts) {
        this._versionOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        this.reset();
        const result1 = splitArguments(nameAndArguments);
        const name = result1.flags.shift();
        const aliases = result1.flags;
        if (!name) {
            throw new MissingCommandName();
        }
        if (this.getBaseCommand(name, true)) {
            if (!override) {
                throw new DuplicateCommandName(name);
            }
            this.removeCommand(name);
        }
        let description;
        let cmd;
        if (typeof cmdOrDescription === "string") {
            description = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd = cmdOrDescription.reset();
        } else {
            cmd = new Command();
        }
        cmd._name = name;
        cmd._parent = this;
        if (description) {
            cmd.description(description);
        }
        if (result1.typeDefinition) {
            cmd.arguments(result1.typeDefinition);
        }
        aliases.forEach((alias)=>cmd.alias(alias));
        this.commands.set(name, cmd);
        this.select(name);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAlias(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this._groupName = undefined;
        this.cmd = this;
        return this;
    }
    select(name) {
        const cmd = this.getBaseCommand(name, true);
        if (!cmd) {
            throw new CommandNotFound(name, this.getBaseCommands(true));
        }
        this.cmd = cmd;
        return this;
    }
    name(name) {
        this.cmd._name = name;
        return this;
    }
    version(version) {
        if (typeof version === "string") {
            this.cmd.ver = ()=>version;
        } else if (typeof version === "function") {
            this.cmd.ver = version;
        }
        return this;
    }
    meta(name, value) {
        this.cmd._meta[name] = value;
        return this;
    }
    getMeta(name) {
        return typeof name === "undefined" ? this._meta : this._meta[name];
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd, options)=>HelpGenerator.generate(cmd, {
                    ...help,
                    ...options
                });
        }
        return this;
    }
    description(description) {
        this.cmd.desc = description;
        return this;
    }
    usage(usage) {
        this.cmd._usage = usage;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.fn = fn;
        return this;
    }
    allowEmpty(allowEmpty = true) {
        this.cmd._allowEmpty = allowEmpty;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name) {
        this.cmd.defaultCommand = name;
        return this;
    }
    globalType(name, handler, options) {
        return this.type(name, handler, {
            ...options,
            global: true
        });
    }
    type(name, handler, options) {
        if (this.cmd.types.get(name) && !options?.override) {
            throw new DuplicateType(name);
        }
        this.cmd.types.set(name, {
            ...options,
            name,
            handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd, parent)=>handler.complete?.(cmd, parent) || [];
            this.complete(name, completeHandler, options);
        }
        return this;
    }
    globalComplete(name, complete, options) {
        return this.complete(name, complete, {
            ...options,
            global: true
        });
    }
    complete(name, complete, options) {
        if (this.cmd.completions.has(name) && !options?.override) {
            throw new DuplicateCompletion(name);
        }
        this.cmd.completions.set(name, {
            name,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    noExit() {
        this.cmd._shouldExit = false;
        this.throwErrors();
        return this;
    }
    shouldThrowErrors() {
        return this.cmd.throwOnError || !!this.cmd._parent?.shouldThrowErrors();
    }
    shouldExit() {
        return this.cmd._shouldExit ?? this.cmd._parent?.shouldExit() ?? true;
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    group(name) {
        this.cmd._groupName = name;
        return this;
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts
            });
        }
        const result1 = splitArguments(flags);
        const args = result1.typeDefinition ? parseArgumentsDefinition(result1.typeDefinition) : [];
        const option = {
            ...opts,
            name: "",
            description: desc,
            args,
            flags: result1.flags,
            equalsSign: result1.equalsSign,
            typeDefinition: result1.typeDefinition,
            groupName: this._groupName
        };
        if (option.separator) {
            for (const arg of args){
                if (arg.list) {
                    arg.separator = option.separator;
                }
            }
        }
        for (const part of option.flags){
            const arg1 = part.trim();
            const isLong = /^--/.test(arg1);
            const name = isLong ? arg1.slice(2) : arg1.slice(1);
            if (this.cmd.getBaseOption(name, true)) {
                if (opts?.override) {
                    this.removeOption(name);
                } else {
                    throw new DuplicateOptionName(name);
                }
            }
            if (!option.name && isLong) {
                option.name = name;
            } else if (!option.aliases) {
                option.aliases = [
                    name
                ];
            } else {
                option.aliases.push(name);
            }
        }
        if (option.prepend) {
            this.cmd.options.unshift(option);
        } else {
            this.cmd.options.push(option);
        }
        return this;
    }
    example(name, description) {
        if (this.cmd.hasExample(name)) {
            throw new DuplicateExample(name);
        }
        this.cmd.examples.push({
            name,
            description
        });
        return this;
    }
    globalEnv(name, description, options) {
        return this.env(name, description, {
            ...options,
            global: true
        });
    }
    env(name, description, options) {
        const result1 = splitArguments(name);
        if (!result1.typeDefinition) {
            result1.typeDefinition = "<value:boolean>";
        }
        if (result1.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true))) {
            throw new DuplicateEnvironmentVariable(name);
        }
        const details = parseArgumentsDefinition(result1.typeDefinition);
        if (details.length > 1) {
            throw new EnvironmentVariableSingleValue(name);
        } else if (details.length && details[0].optionalValue) {
            throw new EnvironmentVariableOptionalValue(name);
        } else if (details.length && details[0].variadic) {
            throw new EnvironmentVariableVariadicValue(name);
        }
        this.cmd.envVars.push({
            name: result1.flags[0],
            names: result1.flags,
            description,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    async parse(args = Deno.args) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = args;
            if (args.length > 0) {
                const subCommand = this.getCommand(args[0], true);
                if (subCommand) {
                    subCommand._globalParent = this;
                    return subCommand.parse(this.rawArgs.slice(1));
                }
            }
            if (this.isExecutable) {
                await this.executeExecutable(this.rawArgs);
                return {
                    options: {},
                    args: [],
                    cmd: this,
                    literal: []
                };
            } else if (this._useRawArgs) {
                const env = await this.parseEnvVars();
                return this.execute(env, ...this.rawArgs);
            } else {
                const env1 = await this.parseEnvVars();
                const { actionOption , flags , unknown , literal  } = this.parseFlags(this.rawArgs, env1);
                this.literalArgs = literal;
                const options = {
                    ...env1,
                    ...flags
                };
                const params = this.parseArguments(unknown, options);
                if (actionOption) {
                    await actionOption.action.call(this, options, ...params);
                    if (actionOption.standalone) {
                        return {
                            options,
                            args: params,
                            cmd: this,
                            literal: this.literalArgs
                        };
                    }
                }
                return this.execute(options, ...params);
            }
        } catch (error) {
            if (error instanceof Error) {
                throw this.error(error);
            } else {
                throw this.error(new Error(`[non-error-thrown] ${error}`));
            }
        }
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        !this.types.has("file") && this.type("file", new FileType(), {
            global: true
        });
        if (!this._help) {
            this.help({
                hints: true,
                types: false
            });
        }
        if (this._versionOption !== false && (this._versionOption || this.ver)) {
            this.option(this._versionOption?.flags || "-V, --version", this._versionOption?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${versionOption.name}`);
                    if (__long) {
                        await this.checkVersion();
                        this.showLongVersion();
                    } else {
                        this.showVersion();
                    }
                    this.exit();
                },
                ...this._versionOption?.opts ?? {}
            });
            const versionOption = this.options[0];
        }
        if (this._helpOption !== false) {
            this.option(this._helpOption?.flags || "-h, --help", this._helpOption?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${helpOption.name}`);
                    await this.checkVersion();
                    this.showHelp({
                        long: __long
                    });
                    this.exit();
                },
                ...this._helpOption?.opts ?? {}
            });
            const helpOption = this.options[0];
        }
        return this;
    }
    async execute(options, ...args) {
        if (this.fn) {
            await this.fn(options, ...args);
        } else if (this.defaultCommand) {
            const cmd = this.getCommand(this.defaultCommand, true);
            if (!cmd) {
                throw new DefaultCommandNotFound(this.defaultCommand, this.getCommands());
            }
            cmd._globalParent = this;
            await cmd.execute(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeExecutable(args) {
        const command = this.getPath().replace(/\s+/g, "-");
        await Deno.permissions.request({
            name: "run",
            command
        });
        try {
            const process = Deno.run({
                cmd: [
                    command,
                    ...args
                ]
            });
            const status = await process.status();
            if (!status.success) {
                Deno.exit(status.code);
            }
        } catch (error) {
            if (error instanceof Deno.errors.NotFound) {
                throw new CommandExecutableNotFound(command);
            }
            throw error;
        }
    }
    parseFlags(args, env) {
        try {
            let actionOption;
            const result1 = parseFlags(args, {
                stopEarly: this._stopEarly,
                allowEmpty: this._allowEmpty,
                flags: this.getOptions(true),
                ignoreDefaults: env,
                parse: (type)=>this.parseType(type),
                option: (option)=>{
                    if (!actionOption && option.action) {
                        actionOption = option;
                    }
                }
            });
            return {
                ...result1,
                actionOption
            };
        } catch (error) {
            if (error instanceof ValidationError) {
                throw new ValidationError1(error.message);
            }
            throw error;
        }
    }
    parseType(type) {
        const typeSettings = this.getType(type.type);
        if (!typeSettings) {
            throw new UnknownType(type.type, this.getTypes().map((type)=>type.name));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type) : typeSettings.handler(type);
    }
    async parseEnvVars() {
        const envVars = this.getEnvVars(true);
        const result1 = {};
        if (!envVars.length) {
            return result1;
        }
        const hasEnvPermissions = (await Deno.permissions.query({
            name: "env"
        })).state === "granted";
        for (const env of envVars){
            const name = hasEnvPermissions && env.names.find((name)=>!!Deno.env.get(name));
            if (name) {
                const propertyName = underscoreToCamelCase(env.prefix ? env.names[0].replace(new RegExp(`^${env.prefix}`), "") : env.names[0]);
                if (env.details.list) {
                    const values = Deno.env.get(name)?.split(env.details.separator ?? ",") ?? [
                        ""
                    ];
                    result1[propertyName] = values.map((value)=>this.parseType({
                            label: "Environment variable",
                            type: env.type,
                            name,
                            value
                        }));
                } else {
                    result1[propertyName] = this.parseType({
                        label: "Environment variable",
                        type: env.type,
                        name,
                        value: Deno.env.get(name) ?? ""
                    });
                }
                if (env.value && typeof result1[propertyName] !== "undefined") {
                    result1[propertyName] = env.value(result1[propertyName]);
                }
            } else if (env.required) {
                throw new MissingRequiredEnvVar(env);
            }
        }
        return result1;
    }
    parseArguments(args, flags) {
        const params = [];
        args = args.slice(0);
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    throw new UnknownCommand(args[0], this.getCommands());
                } else {
                    throw new NoArgumentsAllowed(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required = this.getArguments().filter((expectedArg)=>!expectedArg.optionalValue).map((expectedArg)=>expectedArg.name);
                if (required.length) {
                    const flagNames = Object.keys(flags);
                    const hasStandaloneOption = !!flagNames.find((name)=>this.getOption(name, true)?.standalone);
                    if (!hasStandaloneOption) {
                        throw new MissingArguments(required);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optionalValue) {
                            break;
                        }
                        throw new MissingArgument(`Missing argument: ${expectedArg.name}`);
                    }
                    let arg;
                    if (expectedArg.variadic) {
                        arg = args.splice(0, args.length).map((value)=>this.parseType({
                                label: "Argument",
                                type: expectedArg.type,
                                name: expectedArg.name,
                                value
                            }));
                    } else {
                        arg = this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value: args.shift()
                        });
                    }
                    if (typeof arg !== "undefined") {
                        params.push(arg);
                    }
                }
                if (args.length) {
                    throw new TooManyArguments(args);
                }
            }
        }
        return params;
    }
    error(error) {
        if (this.shouldThrowErrors() || !(error instanceof ValidationError1)) {
            return error;
        }
        this.showHelp();
        console.error(red(`  ${bold("error")}: ${error.message}\n`));
        Deno.exit(error instanceof ValidationError1 ? error.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath() {
        return this._parent ? this._parent.getPath() + " " + this._name : this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name) {
        return this.getArguments().find((arg)=>arg.name === name);
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getUsage() {
        return this._usage ?? this.getArgsDefinition();
    }
    getShortDescription() {
        return getDescription(this.getDescription(), true);
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        console.log(this.getVersion());
    }
    getLongVersion() {
        return `${bold(this.getMainCommand().getName())} ${blue(this.getVersion() ?? "")}` + Object.entries(this.getMeta()).map(([k, v])=>`\n${bold(k)} ${blue(v)}`).join("");
    }
    showLongVersion() {
        console.log(this.getLongVersion());
    }
    showHelp(options) {
        console.log(this.getHelp(options));
    }
    getHelp(options) {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this, options ?? {});
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    exit(code = 0) {
        if (this.shouldExit()) {
            Deno.exit(code);
        }
    }
    async checkVersion() {
        const mainCommand = this.getMainCommand();
        const upgradeCommand = mainCommand.getCommand("upgrade");
        if (isUpgradeCommand(upgradeCommand)) {
            const latestVersion = await upgradeCommand.getLatestVersion();
            const currentVersion = mainCommand.getVersion();
            if (currentVersion !== latestVersion) {
                mainCommand.version(`${currentVersion}  ${bold(yellow(`(New version available: ${latestVersion}. Run '${mainCommand.getName()} upgrade' to upgrade to the latest version!)`))}`);
            }
        }
    }
    hasOptions(hidden) {
        return this.getOptions(hidden).length > 0;
    }
    getOptions(hidden) {
        return this.getGlobalOptions(hidden).concat(this.getBaseOptions(hidden));
    }
    getBaseOptions(hidden) {
        if (!this.options.length) {
            return [];
        }
        return hidden ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden);
    }
    getGlobalOptions(hidden) {
        const getOptions = (cmd, options = [], names = [])=>{
            if (cmd) {
                if (cmd.options.length) {
                    cmd.options.forEach((option)=>{
                        if (option.global && !this.options.find((opt)=>opt.name === option.name) && names.indexOf(option.name) === -1 && (hidden || !option.hidden)) {
                            names.push(option.name);
                            options.push(option);
                        }
                    });
                }
                return getOptions(cmd._parent, options, names);
            }
            return options;
        };
        return getOptions(this._parent);
    }
    hasOption(name, hidden) {
        return !!this.getOption(name, hidden);
    }
    getOption(name, hidden) {
        return this.getBaseOption(name, hidden) ?? this.getGlobalOption(name, hidden);
    }
    getBaseOption(name, hidden) {
        const option = this.options.find((option)=>option.name === name);
        return option && (hidden || !option.hidden) ? option : undefined;
    }
    getGlobalOption(name, hidden) {
        if (!this._parent) {
            return;
        }
        const option = this._parent.getBaseOption(name, hidden);
        if (!option || !option.global) {
            return this._parent.getGlobalOption(name, hidden);
        }
        return option;
    }
    removeOption(name) {
        const index = this.options.findIndex((option)=>option.name === name);
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden) {
        return this.getCommands(hidden).length > 0;
    }
    getCommands(hidden) {
        return this.getGlobalCommands(hidden).concat(this.getBaseCommands(hidden));
    }
    getBaseCommands(hidden) {
        const commands = Array.from(this.commands.values());
        return hidden ? commands : commands.filter((cmd)=>!cmd.isHidden);
    }
    getGlobalCommands(hidden) {
        const getCommands = (cmd, commands = [], names = [])=>{
            if (cmd) {
                if (cmd.commands.size) {
                    cmd.commands.forEach((cmd)=>{
                        if (cmd.isGlobal && this !== cmd && !this.commands.has(cmd._name) && names.indexOf(cmd._name) === -1 && (hidden || !cmd.isHidden)) {
                            names.push(cmd._name);
                            commands.push(cmd);
                        }
                    });
                }
                return getCommands(cmd._parent, commands, names);
            }
            return commands;
        };
        return getCommands(this._parent);
    }
    hasCommand(name, hidden) {
        return !!this.getCommand(name, hidden);
    }
    getCommand(name, hidden) {
        return this.getBaseCommand(name, hidden) ?? this.getGlobalCommand(name, hidden);
    }
    getBaseCommand(name, hidden) {
        for (const cmd of this.commands.values()){
            if (cmd._name === name || cmd.aliases.includes(name)) {
                return cmd && (hidden || !cmd.isHidden) ? cmd : undefined;
            }
        }
    }
    getGlobalCommand(name, hidden) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseCommand(name, hidden);
        if (!cmd?.isGlobal) {
            return this._parent.getGlobalCommand(name, hidden);
        }
        return cmd;
    }
    removeCommand(name) {
        const command = this.getBaseCommand(name, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd, types = [], names = [])=>{
            if (cmd) {
                if (cmd.types.size) {
                    cmd.types.forEach((type)=>{
                        if (type.global && !this.types.has(type.name) && names.indexOf(type.name) === -1) {
                            names.push(type.name);
                            types.push(type);
                        }
                    });
                }
                return getTypes(cmd._parent, types, names);
            }
            return types;
        };
        return getTypes(this._parent);
    }
    getType(name) {
        return this.getBaseType(name) ?? this.getGlobalType(name);
    }
    getBaseType(name) {
        return this.types.get(name);
    }
    getGlobalType(name) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseType(name);
        if (!cmd?.global) {
            return this._parent.getGlobalType(name);
        }
        return cmd;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd, completions = [], names = [])=>{
            if (cmd) {
                if (cmd.completions.size) {
                    cmd.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name) {
        return this.getBaseCompletion(name) ?? this.getGlobalCompletion(name);
    }
    getBaseCompletion(name) {
        return this.completions.get(name);
    }
    getGlobalCompletion(name) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name);
        }
        return completion;
    }
    hasEnvVars(hidden) {
        return this.getEnvVars(hidden).length > 0;
    }
    getEnvVars(hidden) {
        return this.getGlobalEnvVars(hidden).concat(this.getBaseEnvVars(hidden));
    }
    getBaseEnvVars(hidden) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden);
    }
    getGlobalEnvVars(hidden) {
        const getEnvVars = (cmd, envVars = [], names = [])=>{
            if (cmd) {
                if (cmd.envVars.length) {
                    cmd.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]) && names.indexOf(envVar.names[0]) === -1 && (hidden || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name, hidden) {
        return !!this.getEnvVar(name, hidden);
    }
    getEnvVar(name, hidden) {
        return this.getBaseEnvVar(name, hidden) ?? this.getGlobalEnvVar(name, hidden);
    }
    getBaseEnvVar(name, hidden) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name) !== -1);
        return envVar && (hidden || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name, hidden) {
        if (!this._parent) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name, hidden);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name, hidden);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name) {
        return !!this.getExample(name);
    }
    getExample(name) {
        return this.examples.find((example)=>example.name === name);
    }
}
function isUpgradeCommand(command) {
    return command instanceof Command && "getLatestVersion" in command;
}
class BashCompletionsGenerator {
    static generate(cmd) {
        return new BashCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path}${version}

_${replaceSpecialChars(path)}() {
  local word cur prev listFiles
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()
  listFiles=0

  _${replaceSpecialChars(this.cmd.getName())}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.cmd.getName()} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  _${replaceSpecialChars(this.cmd.getName())}_expand() {
    [ "$cur" != "\${cur%\\\\}" ] && cur="$cur\\\\"
  
    # expand ~username type directory specifications
    if [[ "$cur" == \\~*/* ]]; then
      eval cur=$cur
      
    elif [[ "$cur" == \\~* ]]; then
      cur=\${cur#\\~}
      COMPREPLY=( $( compgen -P '~' -u $cur ) )
      return \${#COMPREPLY[@]}
    fi
  }

  _${replaceSpecialChars(this.cmd.getName())}_file_dir() {
    listFiles=1
    local IFS=$'\\t\\n' xspec #glob
    _${replaceSpecialChars(this.cmd.getName())}_expand || return 0
  
    if [ "\${1:-}" = -d ]; then
      COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -d -- $cur ) )
      #eval "$glob"    # restore glob setting.
      return 0
    fi
  
    xspec=\${1:+"!*.$1"}	# set only if glob passed in as $1
    COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -f -X "$xspec" -- "$cur" ) \
          $( compgen -d -- "$cur" ) )
  }

  ${this.generateCompletions(this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ listFiles -eq 1 ]]; then
    return 0
  fi

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars(path)} -o bashdefault -o default ${path}`;
    }
    generateCompletions(command, path = "", index = 1) {
        path = (path ? path + " " : "") + command.getName();
        const commandCompletions = this.generateCommandCompletions(command, path, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand, path, index + 1)).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName());
        const completionsPath = ~path.indexOf(" ") ? " " + path.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command, completionsPath);
        return `  __${replaceSpecialChars(path)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option)=>option.flags).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options = command.getOptions(false);
        if (options.length) {
            opts += 'case "${prev}" in';
            for (const option of options){
                const flags = option.flags.map((flag)=>flag.trim()).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(command, option.args, completionsPath, {
                    standalone: option.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(command, path) {
        const args = command.getArguments();
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `_${replaceSpecialChars(this.cmd.getName())}_file_dir`;
            }
            return `_${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(command, args, path, opts) {
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `opts=(); _${replaceSpecialChars(this.cmd.getName())}_file_dir`;
            }
            return `opts=(); _${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
    cmd;
}
function replaceSpecialChars(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim(italic("~/.bashrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(BashCompletionsGenerator.generate(baseCmd));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd){
        super();
        return this.description("Get completions for given action from given command.").arguments("<action:string> [command...:string]").action(async (_, action, commandNames)=>{
            let parent;
            const completeCommand = commandNames?.reduce((cmd, name)=>{
                parent = cmd;
                const childCmd = cmd.getCommand(name, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommand(name, cmd.getCommands());
                }
                return childCmd;
            }, cmd || this.getMainCommand()) ?? (cmd || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result1 = await completion?.complete(completeCommand, parent) ?? [];
            if (result1?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result1.join("\n")));
            }
        }).reset();
    }
}
class FishCompletionsGenerator {
    static generate(cmd) {
        return new FishCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path}${version}

function __fish_${replaceSpecialChars1(this.cmd.getName())}_using_command
  set cmds ${getCommandFnNames(this.cmd).join(" ")}
  set words (commandline -opc)
  set cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if [ "$cmd" = "$argv[1]" ]
    return 0
  end
  return 1
end

${this.generateCompletions(this.cmd).trim()}`;
    }
    generateCompletions(command) {
        const parent = command.getParent();
        let result1 = ``;
        if (parent) {
            result1 += "\n" + this.complete(parent, {
                description: command.getShortDescription(),
                arguments: command.getName()
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result1 += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(command, commandArgs[0]) : undefined
            });
        }
        for (const option of command.getOptions(false)){
            result1 += "\n" + this.completeOption(command, option);
        }
        for (const subCommand of command.getCommands(false)){
            result1 += this.generateCompletions(subCommand);
        }
        return result1;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2)?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2)?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: getDescription(option.description),
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(command, option.args[0]) : undefined
        });
    }
    complete(command, options) {
        const cmd = [
            "complete"
        ];
        cmd.push("-c", this.cmd.getName());
        cmd.push("-n", `'__fish_${replaceSpecialChars1(this.cmd.getName())}_using_command __${replaceSpecialChars1(command.getPath())}'`);
        options.shortOption && cmd.push("-s", options.shortOption);
        options.longOption && cmd.push("-l", options.longOption);
        options.standalone && cmd.push("-x");
        cmd.push("-k");
        cmd.push("-f");
        if (options.arguments) {
            options.required && cmd.push("-r");
            cmd.push("-a", options.arguments);
        }
        options.description && cmd.push("-d", `'${getDescription(options.description, true)}'`);
        return cmd.join(" ");
    }
    getCompletionCommand(cmd, arg) {
        const type = cmd.getType(arg.type);
        if (type && type.handler instanceof FileType) {
            return `'(__fish_complete_path)'`;
        }
        return `'(${this.cmd.getName()} completions complete ${arg.action + " " + getCompletionsPath(cmd)})'`;
    }
    cmd;
}
function getCommandFnNames(cmd, cmds = []) {
    cmds.push(`__${replaceSpecialChars1(cmd.getPath())}`);
    cmd.getCommands(false).forEach((command)=>{
        getCommandFnNames(command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars1(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim(italic("~/.config/fish/config.fish"))}:

    ${dim(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(FishCompletionsGenerator.generate(baseCmd));
        });
    }
}
class ZshCompletionsGenerator {
    actions;
    static generate(cmd) {
        return new ZshCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
        this.actions = new Map();
    }
    generate() {
        const path = this.cmd.getPath();
        const name = this.cmd.getName();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path}${version}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars2(name)}_complete] )) ||
function __${replaceSpecialChars2(name)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${name} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.cmd).trim()}

# _${replaceSpecialChars2(path)} "\${@}"

compdef _${replaceSpecialChars2(path)} ${path}`;
    }
    generateCompletions(command, path = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path = (path ? path + " " : "") + command.getName();
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars2(path)}] )) ||
function _${replaceSpecialChars2(path)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path) + this.generateSubCommandCompletions(command, path) + this.generateArgumentCompletions(command, path) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand, path)).join("");
    }
    generateCommandCompletions(command, path) {
        const commands = command.getCommands(false);
        let completions = commands.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription()}'`).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path.split(" ").slice(1).join(" ");
            const arg = command.getArguments()[0];
            const action = this.addAction(arg, completionsPath);
            if (action && command.getCompletion(arg.action)) {
                completions += `\n    __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command, path) {
        if (command.hasCommands(false)) {
            const actions = command.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars2(path + " " + command.getName())} ;;`).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path) {
        this.actions.clear();
        const options = this.generateOptions(command, path);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg)=>command.getCompletion(arg.action)).length) {
            argsCommand += ` \\\n    '${++argIndex}:command:_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args = [];
            for (const arg of command.getArguments().slice(1)){
                const type = command.getType(arg.type);
                if (type && type.handler instanceof FileType) {
                    const fileCompletions = this.getFileCompletions(type);
                    if (arg.variadic) {
                        argIndex++;
                        for(let i1 = 0; i1 < 5; i1++){
                            args.push(`${argIndex + i1}${arg.optionalValue ? "::" : ":"}${arg.name}:${fileCompletions}`);
                        }
                    } else {
                        args.push(`${++argIndex}${arg.optionalValue ? "::" : ":"}${arg.name}:${fileCompletions}`);
                    }
                } else {
                    const completionsPath = path.split(" ").slice(1).join(" ");
                    const action = this.addAction(arg, completionsPath);
                    args.push(`${++argIndex}${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`);
                }
            }
            argsCommand += args.map((arg)=>`\\\n    '${arg}'`).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*::sub command:->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path) {
        const options = [];
        const cmdArgs = path.split(" ");
        cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option)=>option.standalone ? option.flags : false).flat().filter((flag)=>typeof flag === "string");
        for (const option of command.getOptions(false)){
            options.push(this.generateOption(command, option, completionsPath, excludedFlags));
        }
        return options;
    }
    generateOption(command, option, completionsPath, excludedOptions) {
        let excludedFlags = option.conflicts?.length ? [
            ...excludedOptions,
            ...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")), 
        ] : excludedOptions;
        excludedFlags = option.collect ? excludedFlags : [
            ...excludedFlags,
            ...option.flags, 
        ];
        let args = "";
        for (const arg of option.args){
            const type = command.getType(arg.type);
            if (type && type.handler instanceof FileType) {
                const fileCompletions = this.getFileCompletions(type);
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:${fileCompletions}`;
            } else {
                const action = this.addAction(arg, completionsPath);
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`;
            }
        }
        const description = getDescription(option.description, true).replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        const equalsSign = option.equalsSign ? "=" : "";
        const flags = option.flags.map((flag)=>`${flag}${equalsSign}`).join(",");
        if (option.standalone) {
            return `'(- *)'{${collect}${flags}}'[${description}]${args}'`;
        } else {
            const excluded = excludedFlags.length ? `'(${excludedFlags.join(" ")})'` : "";
            if (collect || flags.length > 1) {
                return `${excluded}{${collect}${flags}}'[${description}]${args}'`;
            } else {
                return `${excluded}${flags}'[${description}]${args}'`;
            }
        }
    }
    getFileCompletions(type) {
        if (!(type.handler instanceof FileType)) {
            return "";
        }
        return "_files";
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name, action])=>`${name}) __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`);
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
    cmd;
}
function replaceSpecialChars2(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim(italic("~/.zshrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(ZshCompletionsGenerator.generate(baseCmd));
        });
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim(italic("~/.bashrc"))} or similar:

    ${dim(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).action(()=>this.showHelp()).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd).hidden()).reset();
    }
}
class CommandType extends StringType {
    complete(_cmd, parent) {
        return parent?.getCommands(false).map((cmd)=>cmd.getName()) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd){
        super();
        return this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").action(async (_, name)=>{
            if (!cmd) {
                cmd = name ? this.getGlobalParent()?.getBaseCommand(name) : this.getGlobalParent();
            }
            if (!cmd) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommand(name ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases(), 
                ]);
            }
            await cmd.checkVersion();
            cmd.showHelp();
            Deno.exit(0);
        });
    }
}
function isSubdir(src, dest, sep = sep2) {
    if (src === dest) {
        return false;
    }
    const srcArray = src.split(sep);
    const destArray = dest.split(sep);
    return srcArray.every((current, i1)=>destArray[i1] === current);
}
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
function createWalkEntrySync(path) {
    path = normalize3(path);
    const name = basename2(path);
    const info = Deno.statSync(path);
    return {
        path,
        name,
        isFile: info.isFile,
        isDirectory: info.isDirectory,
        isSymlink: info.isSymlink
    };
}
function include(path, exts, match, skip) {
    if (exts && !exts.some((ext)=>path.endsWith(ext))) {
        return false;
    }
    if (match && !match.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    if (skip && skip.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    return true;
}
function wrapErrorWithRootPath(err, root) {
    if (err instanceof Error && "root" in err) return err;
    const e = new Error();
    e.root = root;
    e.message = err instanceof Error ? `${err.message} for path "${root}"` : `[non-error thrown] for path "${root}"`;
    e.stack = err instanceof Error ? err.stack : undefined;
    e.cause = err instanceof Error ? err.cause : undefined;
    return e;
}
function* walkSync(root, { maxDepth =Infinity , includeFiles =true , includeDirs =true , followSymlinks =false , exts =undefined , match =undefined , skip =undefined  } = {}) {
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include(root, exts, match, skip)) {
        yield createWalkEntrySync(root);
    }
    if (maxDepth < 1 || !include(root, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root);
    } catch (err) {
        throw wrapErrorWithRootPath(err, normalize3(root));
    }
    for (const entry of entries){
        assert(entry.name != null);
        let path = join3(root, entry.name);
        let { isSymlink , isDirectory  } = entry;
        if (isSymlink) {
            if (!followSymlinks) continue;
            path = Deno.realPathSync(path);
            ({ isSymlink , isDirectory  } = Deno.lstatSync(path));
        }
        if (isSymlink || isDirectory) {
            yield* walkSync(path, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                followSymlinks,
                exts,
                match,
                skip
            });
        } else if (includeFiles && include(path, exts, match, skip)) {
            yield {
                path,
                ...entry
            };
        }
    }
}
function split(path) {
    const s = SEP_PATTERN.source;
    const segments = path.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEP_PATTERN);
    const isAbsolute_ = isAbsolute2(path);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path.match(new RegExp(`${s}$`)),
        winRoot: isWindows1 && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error) {
    if (!(error instanceof Deno.errors.NotFound)) {
        throw error;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root =Deno.cwd() , exclude =[] , includeDirs =true , extended =true , globstar =false , caseInsensitive  } = {}) {
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = resolve2(root);
    const resolveFromRoot = (path)=>resolve2(absRoot, path);
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp(s, globOptions));
    const shouldInclude = (path)=>!excludePatterns.some((p)=>!!path.match(p));
    const { segments , isAbsolute: isGlobAbsolute , hasTrailingSep , winRoot  } = split(glob);
    let fixedRoot = isGlobAbsolute ? winRoot != undefined ? winRoot : "/" : absRoot;
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg != null);
        fixedRoot = joinGlobs([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = createWalkEntrySync(fixedRoot);
    } catch (error) {
        return throwUnlessNotFound(error);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment == "..") {
            const parentPath = joinGlobs([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude(parentPath)) {
                    return yield createWalkEntrySync(parentPath);
                }
            } catch (error) {
                throwUnlessNotFound(error);
            }
            return;
        } else if (globSegment == "**") {
            return yield* walkSync(walkInfo.path, {
                skip: excludePatterns
            });
        }
        const globPattern = globToRegExp(globSegment, globOptions);
        for (const walkEntry of walkSync(walkInfo.path, {
            maxDepth: 1,
            skip: excludePatterns
        })){
            if (walkEntry.path != walkInfo.path && walkEntry.name.match(globPattern)) {
                yield walkEntry;
            }
        }
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory);
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory);
    }
    yield* currentMatches;
}
var LogLevels;
(function(LogLevels) {
    LogLevels[LogLevels["NOTSET"] = 0] = "NOTSET";
    LogLevels[LogLevels["DEBUG"] = 10] = "DEBUG";
    LogLevels[LogLevels["INFO"] = 20] = "INFO";
    LogLevels[LogLevels["WARNING"] = 30] = "WARNING";
    LogLevels[LogLevels["ERROR"] = 40] = "ERROR";
    LogLevels[LogLevels["CRITICAL"] = 50] = "CRITICAL";
})(LogLevels || (LogLevels = {}));
Object.keys(LogLevels).filter((key)=>isNaN(Number(key)));
const byLevel = {
    [String(LogLevels.NOTSET)]: "NOTSET",
    [String(LogLevels.DEBUG)]: "DEBUG",
    [String(LogLevels.INFO)]: "INFO",
    [String(LogLevels.WARNING)]: "WARNING",
    [String(LogLevels.ERROR)]: "ERROR",
    [String(LogLevels.CRITICAL)]: "CRITICAL"
};
function getLevelByName(name) {
    switch(name){
        case "NOTSET":
            return LogLevels.NOTSET;
        case "DEBUG":
            return LogLevels.DEBUG;
        case "INFO":
            return LogLevels.INFO;
        case "WARNING":
            return LogLevels.WARNING;
        case "ERROR":
            return LogLevels.ERROR;
        case "CRITICAL":
            return LogLevels.CRITICAL;
        default:
            throw new Error(`no log level found for "${name}"`);
    }
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`no level name found for level: ${level}`);
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    #_log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data) {
        if (typeof data === "string") {
            return data;
        } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
            return String(data);
        } else if (data instanceof Error) {
            return data.stack;
        } else if (typeof data === "object") {
            return JSON.stringify(data);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this.#_log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this.#_log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this.#_log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this.#_log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this.#_log(LogLevels.CRITICAL, msg, ...args);
    }
}
const { Deno: Deno2  } = globalThis;
const noColor1 = typeof Deno2?.noColor === "boolean" ? Deno2.noColor : true;
let enabled1 = !noColor1;
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str, code) {
    return enabled1 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function bold1(str) {
    return run1(str, code1([
        1
    ], 22));
}
function dim1(str) {
    return run1(str, code1([
        2
    ], 22));
}
function underline1(str) {
    return run1(str, code1([
        4
    ], 24));
}
function red1(str) {
    return run1(str, code1([
        31
    ], 39));
}
function green1(str) {
    return run1(str, code1([
        32
    ], 39));
}
function yellow1(str) {
    return run1(str, code1([
        33
    ], 39));
}
function blue1(str) {
    return run1(str, code1([
        34
    ], 39));
}
function brightRed1(str) {
    return run1(str, code1([
        91
    ], 39));
}
const ANSI_PATTERN1 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-nq-uy=><~]))", 
].join("|"), "g");
function stripColor1(string) {
    return string.replace(ANSI_PATTERN1, "");
}
async function exists(filePath) {
    try {
        await Deno.lstat(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function existsSync(filePath) {
    try {
        Deno.lstatSync(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
class BytesList {
    #len = 0;
    #chunks = [];
    constructor(){}
    size() {
        return this.#len;
    }
    add(value, start = 0, end = value.byteLength) {
        if (value.byteLength === 0 || end - start === 0) {
            return;
        }
        checkRange(start, end, value.byteLength);
        this.#chunks.push({
            value,
            end,
            start,
            offset: this.#len
        });
        this.#len += end - start;
    }
    shift(n) {
        if (n === 0) {
            return;
        }
        if (this.#len <= n) {
            this.#chunks = [];
            this.#len = 0;
            return;
        }
        const idx = this.getChunkIndex(n);
        this.#chunks.splice(0, idx);
        const [chunk] = this.#chunks;
        if (chunk) {
            const diff = n - chunk.offset;
            chunk.start += diff;
        }
        let offset = 0;
        for (const chunk1 of this.#chunks){
            chunk1.offset = offset;
            offset += chunk1.end - chunk1.start;
        }
        this.#len = offset;
    }
    getChunkIndex(pos) {
        let max = this.#chunks.length;
        let min = 0;
        while(true){
            const i1 = min + Math.floor((max - min) / 2);
            if (i1 < 0 || this.#chunks.length <= i1) {
                return -1;
            }
            const { offset , start , end  } = this.#chunks[i1];
            const len = end - start;
            if (offset <= pos && pos < offset + len) {
                return i1;
            } else if (offset + len <= pos) {
                min = i1 + 1;
            } else {
                max = i1 - 1;
            }
        }
    }
    get(i1) {
        if (i1 < 0 || this.#len <= i1) {
            throw new Error("out of range");
        }
        const idx = this.getChunkIndex(i1);
        const { value , offset , start  } = this.#chunks[idx];
        return value[start + i1 - offset];
    }
    *iterator(start = 0) {
        const startIdx = this.getChunkIndex(start);
        if (startIdx < 0) return;
        const first = this.#chunks[startIdx];
        let firstOffset = start - first.offset;
        for(let i1 = startIdx; i1 < this.#chunks.length; i1++){
            const chunk = this.#chunks[i1];
            for(let j = chunk.start + firstOffset; j < chunk.end; j++){
                yield chunk.value[j];
            }
            firstOffset = 0;
        }
    }
    slice(start, end = this.#len) {
        if (end === start) {
            return new Uint8Array();
        }
        checkRange(start, end, this.#len);
        const result1 = new Uint8Array(end - start);
        const startIdx = this.getChunkIndex(start);
        const endIdx = this.getChunkIndex(end - 1);
        let written = 0;
        for(let i1 = startIdx; i1 < endIdx; i1++){
            const chunk = this.#chunks[i1];
            const len = chunk.end - chunk.start;
            result1.set(chunk.value.subarray(chunk.start, chunk.end), written);
            written += len;
        }
        const last = this.#chunks[endIdx];
        const rest = end - start - written;
        result1.set(last.value.subarray(last.start, last.start + rest), written);
        return result1;
    }
    concat() {
        const result1 = new Uint8Array(this.#len);
        let sum = 0;
        for (const { value , start , end  } of this.#chunks){
            result1.set(value.subarray(start, end), sum);
            sum += end - start;
        }
        return result1;
    }
}
function checkRange(start, end, len) {
    if (start < 0 || len < start || end < 0 || len < end || end < start) {
        throw new Error("invalid range");
    }
}
function indexOfNeedle(source, needle, start = 0) {
    if (start >= source.length) {
        return -1;
    }
    if (start < 0) {
        start = Math.max(0, source.length + start);
    }
    const s = needle[0];
    for(let i1 = start; i1 < source.length; i1++){
        if (source[i1] !== s) continue;
        const pin = i1;
        let matched = 1;
        let j = i1;
        while(matched < needle.length){
            j++;
            if (source[j] !== needle[j - pin]) {
                break;
            }
            matched++;
        }
        if (matched === needle.length) {
            return pin;
        }
    }
    return -1;
}
function concat(...buf) {
    let length = 0;
    for (const b of buf){
        length += b.length;
    }
    const output = new Uint8Array(length);
    let index = 0;
    for (const b1 of buf){
        output.set(b1, index);
        index += b1.length;
    }
    return output;
}
function copy(src, dst, off = 0) {
    off = Math.max(0, Math.min(off, dst.byteLength));
    const dstBytesAvailable = dst.byteLength - off;
    if (src.byteLength > dstBytesAvailable) {
        src = src.subarray(0, dstBytesAvailable);
    }
    dst.set(src, off);
    return src.byteLength;
}
const MIN_READ = 32 * 1024;
const MAX_SIZE = 2 ** 32 - 2;
class Buffer1 {
    #buf;
    #off = 0;
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice(n) {
        const l1 = this.#buf.byteLength;
        if (n <= this.capacity - l1) {
            this.#reslice(l1 + n);
            return l1;
        }
        return -1;
    }
    #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    readSync(p) {
        if (this.empty()) {
            this.reset();
            if (p.byteLength === 0) {
                return 0;
            }
            return null;
        }
        const nread = copy(this.#buf.subarray(this.#off), p);
        this.#off += nread;
        return nread;
    }
    read(p) {
        const rr = this.readSync(p);
        return Promise.resolve(rr);
    }
    writeSync(p) {
        const m = this.#grow(p.byteLength);
        return copy(p, this.#buf, m);
    }
    write(p) {
        const n = this.writeSync(p);
        return Promise.resolve(n);
    }
    #grow(n1) {
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i1 = this.#tryGrowByReslice(n1);
        if (i1 >= 0) {
            return i1;
        }
        const c = this.capacity;
        if (n1 <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n1 > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n1, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n1, MAX_SIZE));
        return m;
    }
    grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
    async readFrom(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = await r.read(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
    readFromSync(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = r.readSync(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
}
const MIN_BUF_SIZE = 16;
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
class BufferFullError extends Error {
    name;
    constructor(partial){
        super("Buffer full");
        this.partial = partial;
        this.name = "BufferFullError";
    }
    partial;
}
class PartialReadError extends Error {
    name = "PartialReadError";
    partial;
    constructor(){
        super("Encountered UnexpectedEof, data only partially read");
    }
}
class BufReader {
    #buf;
    #rd;
    #r = 0;
    #w = 0;
    #eof = false;
    static create(r, size = 4096) {
        return r instanceof BufReader ? r : new BufReader(r, size);
    }
    constructor(rd, size = 4096){
        if (size < 16) {
            size = MIN_BUF_SIZE;
        }
        this.#reset(new Uint8Array(size), rd);
    }
    size() {
        return this.#buf.byteLength;
    }
    buffered() {
        return this.#w - this.#r;
    }
    #fill = async ()=>{
        if (this.#r > 0) {
            this.#buf.copyWithin(0, this.#r, this.#w);
            this.#w -= this.#r;
            this.#r = 0;
        }
        if (this.#w >= this.#buf.byteLength) {
            throw Error("bufio: tried to fill full buffer");
        }
        for(let i1 = 100; i1 > 0; i1--){
            const rr = await this.#rd.read(this.#buf.subarray(this.#w));
            if (rr === null) {
                this.#eof = true;
                return;
            }
            assert(rr >= 0, "negative read");
            this.#w += rr;
            if (rr > 0) {
                return;
            }
        }
        throw new Error(`No progress after ${100} read() calls`);
    };
    reset(r) {
        this.#reset(this.#buf, r);
    }
    #reset = (buf, rd)=>{
        this.#buf = buf;
        this.#rd = rd;
        this.#eof = false;
    };
    async read(p) {
        let rr = p.byteLength;
        if (p.byteLength === 0) return rr;
        if (this.#r === this.#w) {
            if (p.byteLength >= this.#buf.byteLength) {
                const rr1 = await this.#rd.read(p);
                const nread = rr1 ?? 0;
                assert(nread >= 0, "negative read");
                return rr1;
            }
            this.#r = 0;
            this.#w = 0;
            rr = await this.#rd.read(this.#buf);
            if (rr === 0 || rr === null) return rr;
            assert(rr >= 0, "negative read");
            this.#w += rr;
        }
        const copied = copy(this.#buf.subarray(this.#r, this.#w), p, 0);
        this.#r += copied;
        return copied;
    }
    async readFull(p) {
        let bytesRead = 0;
        while(bytesRead < p.length){
            try {
                const rr = await this.read(p.subarray(bytesRead));
                if (rr === null) {
                    if (bytesRead === 0) {
                        return null;
                    } else {
                        throw new PartialReadError();
                    }
                }
                bytesRead += rr;
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = p.subarray(0, bytesRead);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = p.subarray(0, bytesRead);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return p;
    }
    async readByte() {
        while(this.#r === this.#w){
            if (this.#eof) return null;
            await this.#fill();
        }
        const c = this.#buf[this.#r];
        this.#r++;
        return c;
    }
    async readString(delim) {
        if (delim.length !== 1) {
            throw new Error("Delimiter should be a single character");
        }
        const buffer = await this.readSlice(delim.charCodeAt(0));
        if (buffer === null) return null;
        return new TextDecoder().decode(buffer);
    }
    async readLine() {
        let line = null;
        try {
            line = await this.readSlice(LF);
        } catch (err) {
            if (err instanceof Deno.errors.BadResource) {
                throw err;
            }
            let partial;
            if (err instanceof PartialReadError) {
                partial = err.partial;
                assert(partial instanceof Uint8Array, "bufio: caught error from `readSlice()` without `partial` property");
            }
            if (!(err instanceof BufferFullError)) {
                throw err;
            }
            partial = err.partial;
            if (!this.#eof && partial && partial.byteLength > 0 && partial[partial.byteLength - 1] === CR) {
                assert(this.#r > 0, "bufio: tried to rewind past start of buffer");
                this.#r--;
                partial = partial.subarray(0, partial.byteLength - 1);
            }
            if (partial) {
                return {
                    line: partial,
                    more: !this.#eof
                };
            }
        }
        if (line === null) {
            return null;
        }
        if (line.byteLength === 0) {
            return {
                line,
                more: false
            };
        }
        if (line[line.byteLength - 1] == LF) {
            let drop = 1;
            if (line.byteLength > 1 && line[line.byteLength - 2] === CR) {
                drop = 2;
            }
            line = line.subarray(0, line.byteLength - drop);
        }
        return {
            line,
            more: false
        };
    }
    async readSlice(delim) {
        let s = 0;
        let slice;
        while(true){
            let i1 = this.#buf.subarray(this.#r + s, this.#w).indexOf(delim);
            if (i1 >= 0) {
                i1 += s;
                slice = this.#buf.subarray(this.#r, this.#r + i1 + 1);
                this.#r += i1 + 1;
                break;
            }
            if (this.#eof) {
                if (this.#r === this.#w) {
                    return null;
                }
                slice = this.#buf.subarray(this.#r, this.#w);
                this.#r = this.#w;
                break;
            }
            if (this.buffered() >= this.#buf.byteLength) {
                this.#r = this.#w;
                const oldbuf = this.#buf;
                const newbuf = this.#buf.slice(0);
                this.#buf = newbuf;
                throw new BufferFullError(oldbuf);
            }
            s = this.#w - this.#r;
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = slice;
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = slice;
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return slice;
    }
    async peek(n) {
        if (n < 0) {
            throw Error("negative count");
        }
        let avail = this.#w - this.#r;
        while(avail < n && avail < this.#buf.byteLength && !this.#eof){
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = this.#buf.subarray(this.#r, this.#w);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = this.#buf.subarray(this.#r, this.#w);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
            avail = this.#w - this.#r;
        }
        if (avail === 0 && this.#eof) {
            return null;
        } else if (avail < n && this.#eof) {
            return this.#buf.subarray(this.#r, this.#r + avail);
        } else if (avail < n) {
            throw new BufferFullError(this.#buf.subarray(this.#r, this.#w));
        }
        return this.#buf.subarray(this.#r, this.#r + n);
    }
}
class AbstractBufBase {
    buf;
    usedBufferBytes = 0;
    err = null;
    constructor(buf){
        this.buf = buf;
    }
    size() {
        return this.buf.byteLength;
    }
    available() {
        return this.buf.byteLength - this.usedBufferBytes;
    }
    buffered() {
        return this.usedBufferBytes;
    }
}
class BufWriterSync extends AbstractBufBase {
    #writer;
    static create(writer, size = 4096) {
        return writer instanceof BufWriterSync ? writer : new BufWriterSync(writer, size);
    }
    constructor(writer, size = 4096){
        super(new Uint8Array(size <= 0 ? 4096 : size));
        this.#writer = writer;
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.#writer = w;
    }
    flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            const p = this.buf.subarray(0, this.usedBufferBytes);
            let nwritten = 0;
            while(nwritten < p.length){
                nwritten += this.#writer.writeSync(p.subarray(nwritten));
            }
        } catch (e) {
            if (e instanceof Error) {
                this.err = e;
            }
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    writeSync(data) {
        if (this.err !== null) throw this.err;
        if (data.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = this.#writer.writeSync(data);
                } catch (e) {
                    if (e instanceof Error) {
                        this.err = e;
                    }
                    throw e;
                }
            } else {
                numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data = data.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
async function* readLines(reader, decoderOpts) {
    const bufReader = new BufReader(reader);
    let chunks = [];
    const decoder = new TextDecoder(decoderOpts?.encoding, decoderOpts);
    while(true){
        const res = await bufReader.readLine();
        if (!res) {
            if (chunks.length > 0) {
                yield decoder.decode(concat(...chunks));
            }
            break;
        }
        chunks.push(res.line);
        if (!res.more) {
            yield decoder.decode(concat(...chunks));
            chunks = [];
        }
    }
}
const DEFAULT_FORMATTER = "{levelName} {msg}";
class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {}){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{([^\s}]+)}/g, (match, p1)=>{
            const value = logRecord[p1];
            if (value == null) {
                return match;
            }
            return String(value);
        });
    }
    log(_msg) {}
    async setup() {}
    async destroy() {}
}
class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue1(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow1(msg);
                break;
            case LogLevels.ERROR:
                msg = red1(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold1(red1(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    async setup() {
        this._file = await Deno.open(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        if (this._encoder.encode(msg).byteLength + 1 > this._buf.available()) {
            this.flush();
        }
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
        return Promise.resolve();
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger(name) {
    if (!name) {
        const d = state.loggers.get("default");
        assert(d != null, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result1 = state.loggers.get(name);
    if (!result1) {
        const logger = new Logger(name, "NOTSET", {
            handlers: []
        });
        state.loggers.set(name, logger);
        return logger;
    }
    return result1;
}
function debug(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
function info(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
function warning(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").warning(msg, ...args);
    }
    return getLogger("default").warning(msg, ...args);
}
function error(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
async function setup(config) {
    state.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state.handlers.clear();
    const handlers = state.config.handlers || {};
    for(const handlerName in handlers){
        const handler = handlers[handlerName];
        await handler.setup();
        state.handlers.set(handlerName, handler);
    }
    state.loggers.clear();
    const loggers = state.config.loggers || {};
    for(const loggerName in loggers){
        const loggerConfig = loggers[loggerName];
        const handlerNames = loggerConfig.handlers || [];
        const handlers1 = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state.handlers.get(handlerName);
            if (handler) {
                handlers1.push(handler);
            }
        });
        const levelName = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers1
        });
        state.loggers.set(loggerName, logger);
    }
}
await setup(DEFAULT_CONFIG);
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            await Deno.mkdir(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            Deno.mkdirSync(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
new Error("dest already exists.");
function utime(...args) {
    if (typeof Deno.utime == "function") {
        return Deno.utime(...args);
    } else {
        return Promise.reject(new TypeError("Requires --unstable"));
    }
}
async function ensureValidCopy(src, dest, options) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    await Deno.copyFile(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await utime(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    const originSrcFilePath = await Deno.readLink(src);
    const type = getFileInfoType(await Deno.lstat(src));
    if (isWindows1) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await utime(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for await (const entry of Deno.readDir(src)){
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            await copySymLink(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            await copyDir(srcPath, destPath, options);
        } else if (entry.isFile) {
            await copyFile(srcPath, destPath, options);
        }
    }
}
async function copy1(src, dest, options = {}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options);
    }
}
var EOL;
(function(EOL) {
    EOL["LF"] = "\n";
    EOL["CRLF"] = "\r\n";
})(EOL || (EOL = {}));
class YAMLError extends Error {
    constructor(message = "(unknown reason)", mark = ""){
        super(`${message} ${mark}`);
        this.mark = mark;
        this.name = this.constructor.name;
    }
    toString(_compact) {
        return `${this.name}: ${this.message} ${this.mark}`;
    }
    mark;
}
function isBoolean(value) {
    return typeof value === "boolean" || value instanceof Boolean;
}
function isObject(value) {
    return value !== null && typeof value === "object";
}
function repeat(str, count) {
    let result1 = "";
    for(let cycle = 0; cycle < count; cycle++){
        result1 += str;
    }
    return result1;
}
function isNegativeZero(i1) {
    return i1 === 0 && Number.NEGATIVE_INFINITY === 1 / i1;
}
class Mark {
    constructor(name, buffer, position, line, column){
        this.name = name;
        this.buffer = buffer;
        this.position = position;
        this.line = line;
        this.column = column;
    }
    getSnippet(indent = 4, maxLength = 75) {
        if (!this.buffer) return null;
        let head = "";
        let start = this.position;
        while(start > 0 && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(start - 1)) === -1){
            start -= 1;
            if (this.position - start > maxLength / 2 - 1) {
                head = " ... ";
                start += 5;
                break;
            }
        }
        let tail = "";
        let end = this.position;
        while(end < this.buffer.length && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(end)) === -1){
            end += 1;
            if (end - this.position > maxLength / 2 - 1) {
                tail = " ... ";
                end -= 5;
                break;
            }
        }
        const snippet = this.buffer.slice(start, end);
        return `${repeat(" ", indent)}${head}${snippet}${tail}\n${repeat(" ", indent + this.position - start + head.length)}^`;
    }
    toString(compact) {
        let snippet, where = "";
        if (this.name) {
            where += `in "${this.name}" `;
        }
        where += `at line ${this.line + 1}, column ${this.column + 1}`;
        if (!compact) {
            snippet = this.getSnippet();
            if (snippet) {
                where += `:\n${snippet}`;
            }
        }
        return where;
    }
    name;
    buffer;
    position;
    line;
    column;
}
function compileList(schema, name, result1) {
    const exclude = [];
    for (const includedSchema of schema.include){
        result1 = compileList(includedSchema, name, result1);
    }
    for (const currentType of schema[name]){
        for(let previousIndex = 0; previousIndex < result1.length; previousIndex++){
            const previousType = result1[previousIndex];
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind) {
                exclude.push(previousIndex);
            }
        }
        result1.push(currentType);
    }
    return result1.filter((_type, index)=>!exclude.includes(index));
}
function compileMap(...typesList) {
    const result1 = {
        fallback: {},
        mapping: {},
        scalar: {},
        sequence: {}
    };
    for (const types of typesList){
        for (const type of types){
            if (type.kind !== null) {
                result1[type.kind][type.tag] = result1["fallback"][type.tag] = type;
            }
        }
    }
    return result1;
}
class Schema {
    static SCHEMA_DEFAULT;
    implicit;
    explicit;
    include;
    compiledImplicit;
    compiledExplicit;
    compiledTypeMap;
    constructor(definition){
        this.explicit = definition.explicit || [];
        this.implicit = definition.implicit || [];
        this.include = definition.include || [];
        for (const type of this.implicit){
            if (type.loadKind && type.loadKind !== "scalar") {
                throw new YAMLError("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
            }
        }
        this.compiledImplicit = compileList(this, "implicit", []);
        this.compiledExplicit = compileList(this, "explicit", []);
        this.compiledTypeMap = compileMap(this.compiledImplicit, this.compiledExplicit);
    }
    extend(definition) {
        return new Schema({
            implicit: [
                ...new Set([
                    ...this.implicit,
                    ...definition?.implicit ?? []
                ]), 
            ],
            explicit: [
                ...new Set([
                    ...this.explicit,
                    ...definition?.explicit ?? []
                ]), 
            ],
            include: [
                ...new Set([
                    ...this.include,
                    ...definition?.include ?? []
                ])
            ]
        });
    }
    static create() {}
}
const DEFAULT_RESOLVE = ()=>true;
const DEFAULT_CONSTRUCT = (data)=>data;
function checkTagFormat(tag) {
    return tag;
}
class Type1 {
    tag;
    kind = null;
    instanceOf;
    predicate;
    represent;
    defaultStyle;
    styleAliases;
    loadKind;
    constructor(tag, options){
        this.tag = checkTagFormat(tag);
        if (options) {
            this.kind = options.kind;
            this.resolve = options.resolve || DEFAULT_RESOLVE;
            this.construct = options.construct || DEFAULT_CONSTRUCT;
            this.instanceOf = options.instanceOf;
            this.predicate = options.predicate;
            this.represent = options.represent;
            this.defaultStyle = options.defaultStyle;
            this.styleAliases = options.styleAliases;
        }
    }
    resolve = ()=>true;
    construct = (data)=>data;
}
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data) {
    if (data === null) return false;
    let code;
    let bitlen = 0;
    const max = data.length;
    const map = BASE64_MAP;
    for(let idx = 0; idx < max; idx++){
        code = map.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data) {
    const input = data.replace(/[\r\n=]/g, "");
    const max = input.length;
    const map = BASE64_MAP;
    const result1 = [];
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 0xff);
            result1.push(bits >> 8 & 0xff);
            result1.push(bits & 0xff);
        }
        bits = bits << 6 | map.indexOf(input.charAt(idx));
    }
    const tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 0xff);
        result1.push(bits >> 8 & 0xff);
        result1.push(bits & 0xff);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 0xff);
        result1.push(bits >> 2 & 0xff);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 0xff);
    }
    return new Buffer1(new Uint8Array(result1));
}
function representYamlBinary(object) {
    const max = object.length;
    const map = BASE64_MAP;
    let result1 = "";
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map[bits >> 18 & 0x3f];
            result1 += map[bits >> 12 & 0x3f];
            result1 += map[bits >> 6 & 0x3f];
            result1 += map[bits & 0x3f];
        }
        bits = (bits << 8) + object[idx];
    }
    const tail = max % 3;
    if (tail === 0) {
        result1 += map[bits >> 18 & 0x3f];
        result1 += map[bits >> 12 & 0x3f];
        result1 += map[bits >> 6 & 0x3f];
        result1 += map[bits & 0x3f];
    } else if (tail === 2) {
        result1 += map[bits >> 10 & 0x3f];
        result1 += map[bits >> 4 & 0x3f];
        result1 += map[bits << 2 & 0x3f];
        result1 += map[64];
    } else if (tail === 1) {
        result1 += map[bits >> 2 & 0x3f];
        result1 += map[bits << 4 & 0x3f];
        result1 += map[64];
        result1 += map[64];
    }
    return result1;
}
function isBinary(obj) {
    const buf = new Buffer1();
    try {
        if (0 > buf.readFromSync(obj)) return true;
        return false;
    } catch  {
        return false;
    } finally{
        buf.reset();
    }
}
const binary = new Type1("tag:yaml.org,2002:binary", {
    construct: constructYamlBinary,
    kind: "scalar",
    predicate: isBinary,
    represent: representYamlBinary,
    resolve: resolveYamlBinary
});
function resolveYamlBoolean(data) {
    const max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
const bool = new Type1("tag:yaml.org,2002:bool", {
    construct: constructYamlBoolean,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isBoolean,
    represent: {
        lowercase (object) {
            return object ? "true" : "false";
        },
        uppercase (object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase (object) {
            return object ? "True" : "False";
        }
    },
    resolve: resolveYamlBoolean
});
const YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + "|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + "|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + "|[-+]?\\.(?:inf|Inf|INF)" + "|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data) {
    if (!YAML_FLOAT_PATTERN.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data) {
    let value = data.replace(/_/g, "").toLowerCase();
    const sign = value[0] === "-" ? -1 : 1;
    const digits = [];
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    }
    if (value === ".nan") {
        return NaN;
    }
    if (value.indexOf(":") >= 0) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseFloat(v));
        });
        let valueNb = 0.0;
        let base = 1;
        digits.forEach((d)=>{
            valueNb += d * base;
            base *= 60;
        });
        return sign * valueNb;
    }
    return sign * parseFloat(value);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (isNegativeZero(object)) {
        return "-0.0";
    }
    const res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || isNegativeZero(object));
}
const __float = new Type1("tag:yaml.org,2002:float", {
    construct: constructYamlFloat,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isFloat,
    represent: representYamlFloat,
    resolve: resolveYamlFloat
});
function reconstructFunction(code) {
    const func = new Function(`return ${code}`)();
    if (!(func instanceof Function)) {
        throw new TypeError(`Expected function but got ${typeof func}: ${code}`);
    }
    return func;
}
new Type1("tag:yaml.org,2002:js/function", {
    kind: "scalar",
    resolve (data) {
        if (data === null) {
            return false;
        }
        try {
            reconstructFunction(`${data}`);
            return true;
        } catch (_err) {
            return false;
        }
    },
    construct (data) {
        return reconstructFunction(data);
    },
    predicate (object) {
        return object instanceof Function;
    },
    represent (object) {
        return object.toString();
    }
});
function isHexCode(c) {
    return 0x30 <= c && c <= 0x39 || 0x41 <= c && c <= 0x46 || 0x61 <= c && c <= 0x66;
}
function isOctCode(c) {
    return 0x30 <= c && c <= 0x37;
}
function isDecCode(c) {
    return 0x30 <= c && c <= 0x39;
}
function resolveYamlInteger(data) {
    const max = data.length;
    let index = 0;
    let hasDigits = false;
    if (!max) return false;
    let ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for(; index < max; index++){
            ch = data[index];
            if (ch === "_") continue;
            if (!isOctCode(data.charCodeAt(index))) return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (ch === ":") break;
        if (!isDecCode(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    if (ch !== ":") return true;
    return /^(:[0-5]?[0-9])+$/.test(data.slice(index));
}
function constructYamlInteger(data) {
    let value = data;
    const digits = [];
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value, 16);
        return sign * parseInt(value, 8);
    }
    if (value.indexOf(":") !== -1) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base = 1;
        digits.forEach((d)=>{
            valueInt += d * base;
            base *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value, 10);
}
function isInteger(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !isNegativeZero(object);
}
const __int = new Type1("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger,
    represent: {
        binary (obj) {
            return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
        },
        octal (obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal (obj) {
            return obj.toString(10);
        },
        hexadecimal (obj) {
            return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        }
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ],
        octal: [
            8,
            "oct"
        ]
    }
});
const map = new Type1("tag:yaml.org,2002:map", {
    construct (data) {
        return data !== null ? data : {};
    },
    kind: "mapping"
});
function resolveYamlMerge(data) {
    return data === "<<" || data === null;
}
const merge = new Type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
function resolveYamlNull(data) {
    const max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull(object) {
    return object === null;
}
const nil = new Type1("tag:yaml.org,2002:null", {
    construct: constructYamlNull,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isNull,
    represent: {
        canonical () {
            return "~";
        },
        lowercase () {
            return "null";
        },
        uppercase () {
            return "NULL";
        },
        camelcase () {
            return "Null";
        }
    },
    resolve: resolveYamlNull
});
const { hasOwn  } = Object;
const _toString = Object.prototype.toString;
function resolveYamlOmap(data) {
    const objectKeys = [];
    let pairKey = "";
    let pairHasKey = false;
    for (const pair of data){
        pairHasKey = false;
        if (_toString.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (hasOwn(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data) {
    return data !== null ? data : [];
}
const omap = new Type1("tag:yaml.org,2002:omap", {
    construct: constructYamlOmap,
    kind: "sequence",
    resolve: resolveYamlOmap
});
const _toString1 = Object.prototype.toString;
function resolveYamlPairs(data) {
    const result1 = Array.from({
        length: data.length
    });
    for(let index = 0; index < data.length; index++){
        const pair = data[index];
        if (_toString1.call(pair) !== "[object Object]") return false;
        const keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data) {
    if (data === null) return [];
    const result1 = Array.from({
        length: data.length
    });
    for(let index = 0; index < data.length; index += 1){
        const pair = data[index];
        const keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
const pairs = new Type1("tag:yaml.org,2002:pairs", {
    construct: constructYamlPairs,
    kind: "sequence",
    resolve: resolveYamlPairs
});
const REGEXP = /^\/(?<regexp>[\s\S]+)\/(?<modifiers>[gismuy]*)$/;
const regexp = new Type1("tag:yaml.org,2002:js/regexp", {
    kind: "scalar",
    resolve (data) {
        if (data === null || !data.length) {
            return false;
        }
        const regexp = `${data}`;
        if (regexp.charAt(0) === "/") {
            if (!REGEXP.test(data)) {
                return false;
            }
            const modifiers = [
                ...regexp.match(REGEXP)?.groups?.modifiers ?? ""
            ];
            if (new Set(modifiers).size < modifiers.length) {
                return false;
            }
        }
        return true;
    },
    construct (data) {
        const { regexp =`${data}` , modifiers =""  } = `${data}`.match(REGEXP)?.groups ?? {};
        return new RegExp(regexp, modifiers);
    },
    predicate (object) {
        return object instanceof RegExp;
    },
    represent (object) {
        return object.toString();
    }
});
const seq = new Type1("tag:yaml.org,2002:seq", {
    construct (data) {
        return data !== null ? data : [];
    },
    kind: "sequence"
});
const { hasOwn: hasOwn1  } = Object;
function resolveYamlSet(data) {
    if (data === null) return true;
    for(const key in data){
        if (hasOwn1(data, key)) {
            if (data[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data) {
    return data !== null ? data : {};
}
const set = new Type1("tag:yaml.org,2002:set", {
    construct: constructYamlSet,
    kind: "mapping",
    resolve: resolveYamlSet
});
const str = new Type1("tag:yaml.org,2002:str", {
    construct (data) {
        return data !== null ? data : "";
    },
    kind: "scalar"
});
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9])" + "-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9]?)" + "-([0-9][0-9]?)" + "(?:[Tt]|[ \\t]+)" + "([0-9][0-9]?)" + ":([0-9][0-9])" + ":([0-9][0-9])" + "(?:\\.([0-9]*))?" + "(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + "(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp(data) {
    let match = YAML_DATE_REGEXP.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
    if (match === null) throw new Error("Date resolve error");
    const year = +match[1];
    const month = +match[2] - 1;
    const day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    const hour = +match[4];
    const minute = +match[5];
    const second = +match[6];
    let fraction = 0;
    if (match[7]) {
        let partFraction = match[7].slice(0, 3);
        while(partFraction.length < 3){
            partFraction += "0";
        }
        fraction = +partFraction;
    }
    let delta = null;
    if (match[9]) {
        const tzHour = +match[10];
        const tzMinute = +(match[11] || 0);
        delta = (tzHour * 60 + tzMinute) * 60000;
        if (match[9] === "-") delta = -delta;
    }
    const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp(date) {
    return date.toISOString();
}
const timestamp = new Type1("tag:yaml.org,2002:timestamp", {
    construct: constructYamlTimestamp,
    instanceOf: Date,
    kind: "scalar",
    represent: representYamlTimestamp,
    resolve: resolveYamlTimestamp
});
const undefinedType = new Type1("tag:yaml.org,2002:js/undefined", {
    kind: "scalar",
    resolve () {
        return true;
    },
    construct () {
        return undefined;
    },
    predicate (object) {
        return typeof object === "undefined";
    },
    represent () {
        return "";
    }
});
const failsafe = new Schema({
    explicit: [
        str,
        seq,
        map
    ]
});
const json = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ]
});
const core = new Schema({
    include: [
        json
    ]
});
const def = new Schema({
    explicit: [
        binary,
        omap,
        pairs,
        set
    ],
    implicit: [
        timestamp,
        merge
    ],
    include: [
        core
    ]
});
new Schema({
    explicit: [
        regexp,
        undefinedType
    ],
    include: [
        def
    ]
});
class State {
    constructor(schema = def){
        this.schema = schema;
    }
    schema;
}
class LoaderState extends State {
    documents;
    length;
    lineIndent;
    lineStart;
    position;
    line;
    filename;
    onWarning;
    legacy;
    json;
    listener;
    implicitTypes;
    typeMap;
    version;
    checkLineBreaks;
    tagMap;
    anchorMap;
    tag;
    anchor;
    kind;
    result;
    constructor(input, { filename , schema , onWarning , legacy =false , json =false , listener =null  }){
        super(schema);
        this.input = input;
        this.documents = [];
        this.lineIndent = 0;
        this.lineStart = 0;
        this.position = 0;
        this.line = 0;
        this.result = "";
        this.filename = filename;
        this.onWarning = onWarning;
        this.legacy = legacy;
        this.json = json;
        this.listener = listener;
        this.implicitTypes = this.schema.compiledImplicit;
        this.typeMap = this.schema.compiledTypeMap;
        this.length = input.length;
    }
    input;
}
const { hasOwn: hasOwn2  } = Object;
const CONTEXT_BLOCK_IN = 3;
const CONTEXT_BLOCK_OUT = 4;
const CHOMPING_STRIP = 2;
const CHOMPING_KEEP = 3;
const PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
const PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
const PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
const PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
const PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function isEOL(c) {
    return c === 0x0a || c === 0x0d;
}
function isWhiteSpace(c) {
    return c === 0x09 || c === 0x20;
}
function isWsOrEol(c) {
    return c === 0x09 || c === 0x20 || c === 0x0a || c === 0x0d;
}
function isFlowIndicator(c) {
    return c === 0x2c || c === 0x5b || c === 0x5d || c === 0x7b || c === 0x7d;
}
function fromHexCode(c) {
    if (0x30 <= c && c <= 0x39) {
        return c - 0x30;
    }
    const lc = c | 0x20;
    if (0x61 <= lc && lc <= 0x66) {
        return lc - 0x61 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 0x78) {
        return 2;
    }
    if (c === 0x75) {
        return 4;
    }
    if (c === 0x55) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (0x30 <= c && c <= 0x39) {
        return c - 0x30;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 0x30 ? "\x00" : c === 0x61 ? "\x07" : c === 0x62 ? "\x08" : c === 0x74 ? "\x09" : c === 0x09 ? "\x09" : c === 0x6e ? "\x0A" : c === 0x76 ? "\x0B" : c === 0x66 ? "\x0C" : c === 0x72 ? "\x0D" : c === 0x65 ? "\x1B" : c === 0x20 ? " " : c === 0x22 ? "\x22" : c === 0x2f ? "/" : c === 0x5c ? "\x5C" : c === 0x4e ? "\x85" : c === 0x5f ? "\xA0" : c === 0x4c ? "\u2028" : c === 0x50 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 0xffff) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 0x010000 >> 10) + 0xd800, (c - 0x010000 & 0x03ff) + 0xdc00);
}
const simpleEscapeCheck = Array.from({
    length: 256
});
const simpleEscapeMap = Array.from({
    length: 256
});
for(let i2 = 0; i2 < 256; i2++){
    simpleEscapeCheck[i2] = simpleEscapeSequence(i2) ? 1 : 0;
    simpleEscapeMap[i2] = simpleEscapeSequence(i2);
}
function generateError(state, message) {
    return new YAMLError(message, new Mark(state.filename, state.input, state.position, state.line, state.position - state.lineStart));
}
function throwError(state, message) {
    throw generateError(state, message);
}
function throwWarning(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError(state, message));
    }
}
const directiveHandlers = {
    YAML (state, _name, ...args) {
        if (state.version !== null) {
            return throwError(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            return throwError(state, "YAML directive accepts exactly one argument");
        }
        const match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            return throwError(state, "ill-formed argument of the YAML directive");
        }
        const major = parseInt(match[1], 10);
        const minor = parseInt(match[2], 10);
        if (major !== 1) {
            return throwError(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            return throwWarning(state, "unsupported YAML version of the document");
        }
    },
    TAG (state, _name, ...args) {
        if (args.length !== 2) {
            return throwError(state, "TAG directive accepts exactly two arguments");
        }
        const handle = args[0];
        const prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            return throwError(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (state.tagMap && hasOwn2(state.tagMap, handle)) {
            return throwError(state, `there is a previously declared suffix for "${handle}" tag handle`);
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            return throwError(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        if (typeof state.tagMap === "undefined") {
            state.tagMap = {};
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment(state, start, end, checkJson) {
    let result1;
    if (start < end) {
        result1 = state.input.slice(start, end);
        if (checkJson) {
            for(let position = 0, length = result1.length; position < length; position++){
                const character = result1.charCodeAt(position);
                if (!(character === 0x09 || 0x20 <= character && character <= 0x10ffff)) {
                    return throwError(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(result1)) {
            return throwError(state, "the stream contains non-printable characters");
        }
        state.result += result1;
    }
}
function mergeMappings(state, destination, source, overridableKeys) {
    if (!isObject(source)) {
        return throwError(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    const keys = Object.keys(source);
    for(let i1 = 0, len = keys.length; i1 < len; i1++){
        const key = keys[i1];
        if (!hasOwn2(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode, startLine, startPos) {
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(let index = 0, quantity = keyNode.length; index < quantity; index++){
            if (Array.isArray(keyNode[index])) {
                return throwError(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (result1 === null) {
        result1 = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(let index1 = 0, quantity1 = valueNode.length; index1 < quantity1; index1++){
                mergeMappings(state, result1, valueNode[index1], overridableKeys);
            }
        } else {
            mergeMappings(state, result1, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !hasOwn2(overridableKeys, keyNode) && hasOwn2(result1, keyNode)) {
            state.line = startLine || state.line;
            state.position = startPos || state.position;
            return throwError(state, "duplicated mapping key");
        }
        result1[keyNode] = valueNode;
        delete overridableKeys[keyNode];
    }
    return result1;
}
function readLineBreak(state) {
    const ch = state.input.charCodeAt(state.position);
    if (ch === 0x0a) {
        state.position++;
    } else if (ch === 0x0d) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 0x0a) {
            state.position++;
        }
    } else {
        return throwError(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
}
function skipSeparationSpace(state, allowComments, checkIndent) {
    let lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(isWhiteSpace(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 0x23) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 0x0a && ch !== 0x0d && ch !== 0)
        }
        if (isEOL(ch)) {
            readLineBreak(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 0x20){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state) {
    let _position = state.position;
    let ch = state.input.charCodeAt(_position);
    if ((ch === 0x2d || ch === 0x2e) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || isWsOrEol(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += repeat("\n", count - 1);
    }
}
function readPlainScalar(state, nodeIndent, withinFlowCollection) {
    const kind = state.kind;
    const result1 = state.result;
    let ch = state.input.charCodeAt(state.position);
    if (isWsOrEol(ch) || isFlowIndicator(ch) || ch === 0x23 || ch === 0x26 || ch === 0x2a || ch === 0x21 || ch === 0x7c || ch === 0x3e || ch === 0x27 || ch === 0x22 || ch === 0x25 || ch === 0x40 || ch === 0x60) {
        return false;
    }
    let following;
    if (ch === 0x3f || ch === 0x2d) {
        following = state.input.charCodeAt(state.position + 1);
        if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    let captureEnd, captureStart = captureEnd = state.position;
    let hasPendingContent = false;
    let line = 0;
    while(ch !== 0){
        if (ch === 0x3a) {
            following = state.input.charCodeAt(state.position + 1);
            if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
                break;
            }
        } else if (ch === 0x23) {
            const preceding = state.input.charCodeAt(state.position - 1);
            if (isWsOrEol(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator(state) || withinFlowCollection && isFlowIndicator(ch)) {
            break;
        } else if (isEOL(ch)) {
            line = state.line;
            const lineStart = state.lineStart;
            const lineIndent = state.lineIndent;
            skipSeparationSpace(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = line;
                state.lineStart = lineStart;
                state.lineIndent = lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state, captureStart, captureEnd, false);
            writeFoldedLines(state, state.line - line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!isWhiteSpace(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = kind;
    state.result = result1;
    return false;
}
function readSingleQuotedScalar(state, nodeIndent) {
    let ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 0x27) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 0x27) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 0x27) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (isEOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            return throwError(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    return throwError(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state, nodeIndent) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x22) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    let captureEnd, captureStart = captureEnd = state.position;
    let tmp;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 0x22) {
            captureSegment(state, captureStart, state.position, true);
            state.position++;
            return true;
        }
        if (ch === 0x5c) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (isEOL(ch)) {
                skipSeparationSpace(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state.result += simpleEscapeMap[ch];
                state.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                let hexLength = tmp;
                let hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        return throwError(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint(hexResult);
                state.position++;
            } else {
                return throwError(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (isEOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            return throwError(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    return throwError(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state, nodeIndent) {
    let ch = state.input.charCodeAt(state.position);
    let terminator;
    let isMapping = true;
    let result1 = {};
    if (ch === 0x5b) {
        terminator = 0x5d;
        isMapping = false;
        result1 = [];
    } else if (ch === 0x7b) {
        terminator = 0x7d;
    } else {
        return false;
    }
    if (state.anchor !== null && typeof state.anchor != "undefined" && typeof state.anchorMap != "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(++state.position);
    const tag = state.tag, anchor = state.anchor;
    let readNext = true;
    let valueNode, keyNode, keyTag = keyNode = valueNode = null, isExplicitPair, isPair = isExplicitPair = false;
    let following = 0, line = 0;
    const overridableKeys = {};
    while(ch !== 0){
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = tag;
            state.anchor = anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = result1;
            return true;
        }
        if (!readNext) {
            return throwError(state, "missed comma between flow collection entries");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 0x3f) {
            following = state.input.charCodeAt(state.position + 1);
            if (isWsOrEol(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace(state, true, nodeIndent);
            }
        }
        line = state.line;
        composeNode(state, nodeIndent, 1, false, true);
        keyTag = state.tag || null;
        keyNode = state.result;
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === line) && ch === 0x3a) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace(state, true, nodeIndent);
            composeNode(state, nodeIndent, 1, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode);
        } else if (isPair) {
            result1.push(storeMappingPair(state, null, overridableKeys, keyTag, keyNode, valueNode));
        } else {
            result1.push(keyNode);
        }
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 0x2c) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    return throwError(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state, nodeIndent) {
    let chomping = 1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false;
    let ch = state.input.charCodeAt(state.position);
    let folding = false;
    if (ch === 0x7c) {
        folding = false;
    } else if (ch === 0x3e) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    let tmp = 0;
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 0x2b || ch === 0x2d) {
            if (1 === chomping) {
                chomping = ch === 0x2b ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                return throwError(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                return throwError(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                return throwError(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (isWhiteSpace(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (isWhiteSpace(ch))
        if (ch === 0x23) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!isEOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 0x20){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (isEOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === 3) {
                state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === 1) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (isWhiteSpace(ch)) {
                atMoreIndented = true;
                state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += repeat("\n", emptyLines);
            }
        } else {
            state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        const captureStart = state.position;
        while(!isEOL(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence(state, nodeIndent) {
    let line, following, detected = false, ch;
    const tag = state.tag, anchor = state.anchor, result1 = [];
    if (state.anchor !== null && typeof state.anchor !== "undefined" && typeof state.anchorMap !== "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (ch !== 0x2d) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!isWsOrEol(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                result1.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        line = state.line;
        composeNode(state, nodeIndent, 3, false, true);
        result1.push(state.result);
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === line || state.lineIndent > nodeIndent) && ch !== 0) {
            return throwError(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = tag;
        state.anchor = anchor;
        state.kind = "sequence";
        state.result = result1;
        return true;
    }
    return false;
}
function readBlockMapping(state, nodeIndent, flowIndent) {
    const tag = state.tag, anchor = state.anchor, result1 = {}, overridableKeys = {};
    let following, allowCompact = false, line, pos, keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.anchor !== null && typeof state.anchor !== "undefined" && typeof state.anchorMap !== "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        following = state.input.charCodeAt(state.position + 1);
        line = state.line;
        pos = state.position;
        if ((ch === 0x3f || ch === 0x3a) && isWsOrEol(following)) {
            if (ch === 0x3f) {
                if (atExplicitKey) {
                    storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                return throwError(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else if (composeNode(state, flowIndent, 2, false, true)) {
            if (state.line === line) {
                ch = state.input.charCodeAt(state.position);
                while(isWhiteSpace(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 0x3a) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!isWsOrEol(ch)) {
                        return throwError(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    return throwError(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = tag;
                    state.anchor = anchor;
                    return true;
                }
            } else if (detected) {
                return throwError(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = tag;
                state.anchor = anchor;
                return true;
            }
        } else {
            break;
        }
        if (state.line === line || state.lineIndent > nodeIndent) {
            if (composeNode(state, nodeIndent, 4, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode, line, pos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if (state.lineIndent > nodeIndent && ch !== 0) {
            return throwError(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
    }
    if (detected) {
        state.tag = tag;
        state.anchor = anchor;
        state.kind = "mapping";
        state.result = result1;
    }
    return detected;
}
function readTagProperty(state) {
    let position, isVerbatim = false, isNamed = false, tagHandle = "", tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 0x21) return false;
    if (state.tag !== null) {
        return throwError(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 0x3c) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 0x21) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 0x3e)
        if (state.position < state.length) {
            tagName = state.input.slice(position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            return throwError(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !isWsOrEol(ch)){
            if (ch === 0x21) {
                if (!isNamed) {
                    tagHandle = state.input.slice(position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        return throwError(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    position = state.position + 1;
                } else {
                    return throwError(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(position, state.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            return throwError(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        return throwError(state, `tag name cannot contain such characters: ${tagName}`);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (typeof state.tagMap !== "undefined" && hasOwn2(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = `!${tagName}`;
    } else if (tagHandle === "!!") {
        state.tag = `tag:yaml.org,2002:${tagName}`;
    } else {
        return throwError(state, `undeclared tag handle "${tagHandle}"`);
    }
    return true;
}
function readAnchorProperty(state) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x26) return false;
    if (state.anchor !== null) {
        return throwError(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    const position = state.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === position) {
        return throwError(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(position, state.position);
    return true;
}
function readAlias(state) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x2a) return false;
    ch = state.input.charCodeAt(++state.position);
    const _position = state.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        return throwError(state, "name of an alias node must contain at least one character");
    }
    const alias = state.input.slice(_position, state.position);
    if (typeof state.anchorMap !== "undefined" && !hasOwn2(state.anchorMap, alias)) {
        return throwError(state, `unidentified alias "${alias}"`);
    }
    if (typeof state.anchorMap !== "undefined") {
        state.result = state.anchorMap[alias];
    }
    skipSeparationSpace(state, true, -1);
    return true;
}
function composeNode(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    let allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, type, flowIndent, blockIndent;
    if (state.listener && state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    const allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state) || readAnchorProperty(state)){
            if (skipSeparationSpace(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || 4 === nodeContext) {
        const cond = 1 === nodeContext || 2 === nodeContext;
        flowIndent = cond ? parentIndent : parentIndent + 1;
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state, blockIndent) || readBlockMapping(state, blockIndent, flowIndent)) || readFlowCollection(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state, flowIndent) || readSingleQuotedScalar(state, flowIndent) || readDoubleQuotedScalar(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        return throwError(state, "alias node should not have Any properties");
                    }
                } else if (readPlainScalar(state, flowIndent, 1 === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state, blockIndent);
        }
    }
    if (state.tag !== null && state.tag !== "!") {
        if (state.tag === "?") {
            for(let typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex++){
                type = state.implicitTypes[typeIndex];
                if (type.resolve(state.result)) {
                    state.result = type.construct(state.result);
                    state.tag = type.tag;
                    if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                        state.anchorMap[state.anchor] = state.result;
                    }
                    break;
                }
            }
        } else if (hasOwn2(state.typeMap[state.kind || "fallback"], state.tag)) {
            type = state.typeMap[state.kind || "fallback"][state.tag];
            if (state.result !== null && type.kind !== state.kind) {
                return throwError(state, `unacceptable node kind for !<${state.tag}> tag; it should be "${type.kind}", not "${state.kind}"`);
            }
            if (!type.resolve(state.result)) {
                return throwError(state, `cannot resolve a node with !<${state.tag}> explicit tag`);
            } else {
                state.result = type.construct(state.result);
                if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else {
            return throwError(state, `unknown tag !<${state.tag}>`);
        }
    }
    if (state.listener && state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument(state) {
    const documentStart = state.position;
    let position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = {};
    state.anchorMap = {};
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 0x25) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        position = state.position;
        while(ch !== 0 && !isWsOrEol(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            return throwError(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(isWhiteSpace(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 0x23) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !isEOL(ch))
                break;
            }
            if (isEOL(ch)) break;
            position = state.position;
            while(ch !== 0 && !isWsOrEol(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(position, state.position));
        }
        if (ch !== 0) readLineBreak(state);
        if (hasOwn2(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state, directiveName, ...directiveArgs);
        } else {
            throwWarning(state, `unknown document directive "${directiveName}"`);
        }
    }
    skipSeparationSpace(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 0x2d && state.input.charCodeAt(state.position + 1) === 0x2d && state.input.charCodeAt(state.position + 2) === 0x2d) {
        state.position += 3;
        skipSeparationSpace(state, true, -1);
    } else if (hasDirectives) {
        return throwError(state, "directives end mark is expected");
    }
    composeNode(state, state.lineIndent - 1, 4, false, true);
    skipSeparationSpace(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state.input.slice(documentStart, state.position))) {
        throwWarning(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator(state)) {
        if (state.input.charCodeAt(state.position) === 0x2e) {
            state.position += 3;
            skipSeparationSpace(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        return throwError(state, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 0x0a && input.charCodeAt(input.length - 1) !== 0x0d) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 0xfeff) {
            input = input.slice(1);
        }
    }
    const state = new LoaderState(input, options);
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 0x20){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument(state);
    }
    return state.documents;
}
function load(input, options) {
    const documents = loadDocuments(input, options);
    if (documents.length === 0) {
        return;
    }
    if (documents.length === 1) {
        return documents[0];
    }
    throw new YAMLError("expected a single document in the stream, but found more");
}
function parse3(content, options) {
    return load(content, options);
}
const { hasOwn: hasOwn3  } = Object;
function compileStyleMap(schema, map) {
    if (typeof map === "undefined" || map === null) return {};
    let type;
    const result1 = {};
    const keys = Object.keys(map);
    let tag, style;
    for(let index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = `tag:yaml.org,2002:${tag.slice(2)}`;
        }
        type = schema.compiledTypeMap.fallback[tag];
        if (type && typeof type.styleAliases !== "undefined" && hasOwn3(type.styleAliases, style)) {
            style = type.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
class DumperState extends State {
    indent;
    noArrayIndent;
    skipInvalid;
    flowLevel;
    sortKeys;
    lineWidth;
    noRefs;
    noCompatMode;
    condenseFlow;
    implicitTypes;
    explicitTypes;
    tag = null;
    result = "";
    duplicates = [];
    usedDuplicates = [];
    styleMap;
    dump;
    constructor({ schema , indent =2 , noArrayIndent =false , skipInvalid =false , flowLevel =-1 , styles =null , sortKeys =false , lineWidth =80 , noRefs =false , noCompatMode =false , condenseFlow =false  }){
        super(schema);
        this.indent = Math.max(1, indent);
        this.noArrayIndent = noArrayIndent;
        this.skipInvalid = skipInvalid;
        this.flowLevel = flowLevel;
        this.styleMap = compileStyleMap(this.schema, styles);
        this.sortKeys = sortKeys;
        this.lineWidth = lineWidth;
        this.noRefs = noRefs;
        this.noCompatMode = noCompatMode;
        this.condenseFlow = condenseFlow;
        this.implicitTypes = this.schema.compiledImplicit;
        this.explicitTypes = this.schema.compiledExplicit;
    }
}
const _toString2 = Object.prototype.toString;
const { hasOwn: hasOwn4  } = Object;
const ESCAPE_SEQUENCES = {};
ESCAPE_SEQUENCES[0x00] = "\\0";
ESCAPE_SEQUENCES[0x07] = "\\a";
ESCAPE_SEQUENCES[0x08] = "\\b";
ESCAPE_SEQUENCES[0x09] = "\\t";
ESCAPE_SEQUENCES[0x0a] = "\\n";
ESCAPE_SEQUENCES[0x0b] = "\\v";
ESCAPE_SEQUENCES[0x0c] = "\\f";
ESCAPE_SEQUENCES[0x0d] = "\\r";
ESCAPE_SEQUENCES[0x1b] = "\\e";
ESCAPE_SEQUENCES[0x22] = '\\"';
ESCAPE_SEQUENCES[0x5c] = "\\\\";
ESCAPE_SEQUENCES[0x85] = "\\N";
ESCAPE_SEQUENCES[0xa0] = "\\_";
ESCAPE_SEQUENCES[0x2028] = "\\L";
ESCAPE_SEQUENCES[0x2029] = "\\P";
const DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF", 
];
function encodeHex(character) {
    const string = character.toString(16).toUpperCase();
    let handle;
    let length;
    if (character <= 0xff) {
        handle = "x";
        length = 2;
    } else if (character <= 0xffff) {
        handle = "u";
        length = 4;
    } else if (character <= 0xffffffff) {
        handle = "U";
        length = 8;
    } else {
        throw new YAMLError("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return `\\${handle}${repeat("0", length - string.length)}${string}`;
}
function indentString(string, spaces) {
    const ind = repeat(" ", spaces), length = string.length;
    let position = 0, next = -1, result1 = "", line;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine(state, level) {
    return `\n${repeat(" ", state.indent * level)}`;
}
function testImplicitResolving(state, str) {
    let type;
    for(let index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type = state.implicitTypes[index];
        if (type.resolve(str)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === 0x20 || c === 0x09;
}
function isPrintable(c) {
    return 0x00020 <= c && c <= 0x00007e || 0x000a1 <= c && c <= 0x00d7ff && c !== 0x2028 && c !== 0x2029 || 0x0e000 <= c && c <= 0x00fffd && c !== 0xfeff || 0x10000 <= c && c <= 0x10ffff;
}
function isPlainSafe(c) {
    return isPrintable(c) && c !== 0xfeff && c !== 0x2c && c !== 0x5b && c !== 0x5d && c !== 0x7b && c !== 0x7d && c !== 0x3a && c !== 0x23;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== 0xfeff && !isWhitespace(c) && c !== 0x2d && c !== 0x3f && c !== 0x3a && c !== 0x2c && c !== 0x5b && c !== 0x5d && c !== 0x7b && c !== 0x7d && c !== 0x23 && c !== 0x26 && c !== 0x2a && c !== 0x21 && c !== 0x7c && c !== 0x3e && c !== 0x27 && c !== 0x22 && c !== 0x25 && c !== 0x40 && c !== 0x60;
}
function needIndentIndicator(string) {
    const leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
const STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType) {
    const shouldTrackWidth = lineWidth !== -1;
    let hasLineBreak = false, hasFoldableLine = false, previousLineBreak = -1, plain = isPlainSafeFirst(string.charCodeAt(0)) && !isWhitespace(string.charCodeAt(string.length - 1));
    let __char, i1;
    if (singleLineOnly) {
        for(i1 = 0; i1 < string.length; i1++){
            __char = string.charCodeAt(i1);
            if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
    } else {
        for(i1 = 0; i1 < string.length; i1++){
            __char = string.charCodeAt(i1);
            if (__char === 0x0a) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        return plain && !testAmbiguousType(string) ? 1 : 2;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string)) {
        return 5;
    }
    return hasFoldableLine ? 4 : 3;
}
function foldLine(line, width) {
    if (line === "" || line[0] === " ") return line;
    const breakRe = / [^ ]/g;
    let match;
    let start = 0, end, curr = 0, next = 0;
    let result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += `\n${line.slice(start, end)}`;
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += `${line.slice(start, curr)}\n${line.slice(curr + 1)}`;
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function dropEndingNewline(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString(string, width) {
    const lineRe = /(\n+)([^\n]*)/g;
    let result1 = (()=>{
        let nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string.slice(0, nextLF), width);
    })();
    let prevMoreIndented = string[0] === "\n" || string[0] === " ";
    let moreIndented;
    let match;
    while(match = lineRe.exec(string)){
        const prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function escapeString(string) {
    let result1 = "";
    let __char, nextChar;
    let escapeSeq;
    for(let i1 = 0; i1 < string.length; i1++){
        __char = string.charCodeAt(i1);
        if (__char >= 0xd800 && __char <= 0xdbff) {
            nextChar = string.charCodeAt(i1 + 1);
            if (nextChar >= 0xdc00 && nextChar <= 0xdfff) {
                result1 += encodeHex((__char - 0xd800) * 0x400 + nextChar - 0xdc00 + 0x10000);
                i1++;
                continue;
            }
        }
        escapeSeq = ESCAPE_SEQUENCES[__char];
        result1 += !escapeSeq && isPrintable(__char) ? string[i1] : escapeSeq || encodeHex(__char);
    }
    return result1;
}
function blockHeader(string, indentPerLevel) {
    const indentIndicator = needIndentIndicator(string) ? String(indentPerLevel) : "";
    const clip = string[string.length - 1] === "\n";
    const keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    const chomp = keep ? "+" : clip ? "" : "-";
    return `${indentIndicator}${chomp}\n`;
}
function writeScalar(state, string, level, iskey) {
    state.dump = (()=>{
        if (string.length === 0) {
            return "''";
        }
        if (!state.noCompatMode && DEPRECATED_BOOLEANS_SYNTAX.indexOf(string) !== -1) {
            return `'${string}'`;
        }
        const indent = state.indent * Math.max(1, level);
        const lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        const singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(str) {
            return testImplicitResolving(state, str);
        }
        switch(chooseScalarStyle(string, singleLineOnly, state.indent, lineWidth, testAmbiguity)){
            case STYLE_PLAIN:
                return string;
            case STYLE_SINGLE:
                return `'${string.replace(/'/g, "''")}'`;
            case STYLE_LITERAL:
                return `|${blockHeader(string, state.indent)}${dropEndingNewline(indentString(string, indent))}`;
            case STYLE_FOLDED:
                return `>${blockHeader(string, state.indent)}${dropEndingNewline(indentString(foldString(string, lineWidth), indent))}`;
            case STYLE_DOUBLE:
                return `"${escapeString(string)}"`;
            default:
                throw new YAMLError("impossible error: invalid scalar style");
        }
    })();
}
function writeFlowSequence(state, level, object) {
    let _result = "";
    const _tag = state.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state, level, object[index], false, false)) {
            if (index !== 0) _result += `,${!state.condenseFlow ? " " : ""}`;
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = `[${_result}]`;
}
function writeBlockSequence(state, level, object, compact = false) {
    let _result = "";
    const _tag = state.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state, level + 1, object[index], true, true)) {
            if (!compact || index !== 0) {
                _result += generateNextLine(state, level);
            }
            if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping(state, level, object) {
    let _result = "";
    const _tag = state.tag, objectKeyList = Object.keys(object);
    let pairBuffer, objectKey, objectValue;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = state.condenseFlow ? '"' : "";
        if (index !== 0) pairBuffer += ", ";
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += `${state.dump}${state.condenseFlow ? '"' : ""}:${state.condenseFlow ? "" : " "}`;
        if (!writeNode(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = `{${_result}}`;
}
function writeBlockMapping(state, level, object, compact = false) {
    const _tag = state.tag, objectKeyList = Object.keys(object);
    let _result = "";
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new YAMLError("sortKeys must be a boolean or a function");
    }
    let pairBuffer = "", objectKey, objectValue, explicitPair;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || index !== 0) {
            pairBuffer += generateNextLine(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state, level);
        }
        if (!writeNode(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType(state, object, explicit = false) {
    const typeList = explicit ? state.explicitTypes : state.implicitTypes;
    let type;
    let style;
    let _result;
    for(let index = 0, length = typeList.length; index < length; index += 1){
        type = typeList[index];
        if ((type.instanceOf || type.predicate) && (!type.instanceOf || typeof object === "object" && object instanceof type.instanceOf) && (!type.predicate || type.predicate(object))) {
            state.tag = explicit ? type.tag : "?";
            if (type.represent) {
                style = state.styleMap[type.tag] || type.defaultStyle;
                if (_toString2.call(type.represent) === "[object Function]") {
                    _result = type.represent(object, style);
                } else if (hasOwn4(type.represent, style)) {
                    _result = type.represent[style](object, style);
                } else {
                    throw new YAMLError(`!<${type.tag}> tag resolver accepts not "${style}" style`);
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state, level, object, block, compact, iskey = false) {
    state.tag = null;
    state.dump = object;
    if (!detectType(state, object, false)) {
        detectType(state, object, true);
    }
    const type = _toString2.call(state.dump);
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    const objectOrArray = type === "[object Object]" || type === "[object Array]";
    let duplicateIndex = -1;
    let duplicate = false;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = `*ref_${duplicateIndex}`;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex}${state.dump}`;
                }
            } else {
                writeFlowMapping(state, level, state.dump);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex} ${state.dump}`;
                }
            }
        } else if (type === "[object Array]") {
            const arrayLevel = state.noArrayIndent && level > 0 ? level - 1 : level;
            if (block && state.dump.length !== 0) {
                writeBlockSequence(state, arrayLevel, state.dump, compact);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex}${state.dump}`;
                }
            } else {
                writeFlowSequence(state, arrayLevel, state.dump);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex} ${state.dump}`;
                }
            }
        } else if (type === "[object String]") {
            if (state.tag !== "?") {
                writeScalar(state, state.dump, level, iskey);
            }
        } else {
            if (state.skipInvalid) return false;
            throw new YAMLError(`unacceptable kind of an object to dump ${type}`);
        }
        if (state.tag !== null && state.tag !== "?") {
            state.dump = `!<${state.tag}> ${state.dump}`;
        }
    }
    return true;
}
function inspectNode(object, objects, duplicatesIndexes) {
    if (object !== null && typeof object === "object") {
        const index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(let idx = 0, length = object.length; idx < length; idx += 1){
                    inspectNode(object[idx], objects, duplicatesIndexes);
                }
            } else {
                const objectKeyList = Object.keys(object);
                for(let idx1 = 0, length1 = objectKeyList.length; idx1 < length1; idx1 += 1){
                    inspectNode(object[objectKeyList[idx1]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function getDuplicateReferences(object, state) {
    const objects = [], duplicatesIndexes = [];
    inspectNode(object, objects, duplicatesIndexes);
    const length = duplicatesIndexes.length;
    for(let index = 0; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = Array.from({
        length
    });
}
function dump(input, options) {
    options = options || {};
    const state = new DumperState(options);
    if (!state.noRefs) getDuplicateReferences(input, state);
    if (writeNode(state, 0, input, true, true)) return `${state.dump}\n`;
    return "";
}
function stringify(obj, options) {
    return dump(obj, options);
}
var State1;
(function(State) {
    State[State["PASSTHROUGH"] = 0] = "PASSTHROUGH";
    State[State["PERCENT"] = 1] = "PERCENT";
    State[State["POSITIONAL"] = 2] = "POSITIONAL";
    State[State["PRECISION"] = 3] = "PRECISION";
    State[State["WIDTH"] = 4] = "WIDTH";
})(State1 || (State1 = {}));
var WorP;
(function(WorP) {
    WorP[WorP["WIDTH"] = 0] = "WIDTH";
    WorP[WorP["PRECISION"] = 1] = "PRECISION";
})(WorP || (WorP = {}));
class Flags {
    plus;
    dash;
    sharp;
    space;
    zero;
    lessthan;
    width = -1;
    precision = -1;
}
const min = Math.min;
const UNICODE_REPLACEMENT_CHARACTER = "\ufffd";
const FLOAT_REGEXP = /(-?)(\d)\.?(\d*)e([+-])(\d+)/;
var F;
(function(F) {
    F[F["sign"] = 1] = "sign";
    F[F["mantissa"] = 2] = "mantissa";
    F[F["fractional"] = 3] = "fractional";
    F[F["esign"] = 4] = "esign";
    F[F["exponent"] = 5] = "exponent";
})(F || (F = {}));
class Printf {
    format;
    args;
    i;
    state = State1.PASSTHROUGH;
    verb = "";
    buf = "";
    argNum = 0;
    flags = new Flags();
    haveSeen;
    tmpError;
    constructor(format, ...args){
        this.format = format;
        this.args = args;
        this.haveSeen = Array.from({
            length: args.length
        });
        this.i = 0;
    }
    doPrintf() {
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PASSTHROUGH:
                    if (c === "%") {
                        this.state = State1.PERCENT;
                    } else {
                        this.buf += c;
                    }
                    break;
                case State1.PERCENT:
                    if (c === "%") {
                        this.buf += c;
                        this.state = State1.PASSTHROUGH;
                    } else {
                        this.handleFormat();
                    }
                    break;
                default:
                    throw Error("Should be unreachable, certainly a bug in the lib.");
            }
        }
        let extras = false;
        let err = "%!(EXTRA";
        for(let i1 = 0; i1 !== this.haveSeen.length; ++i1){
            if (!this.haveSeen[i1]) {
                extras = true;
                err += ` '${Deno.inspect(this.args[i1])}'`;
            }
        }
        err += ")";
        if (extras) {
            this.buf += err;
        }
        return this.buf;
    }
    handleFormat() {
        this.flags = new Flags();
        const flags = this.flags;
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PERCENT:
                    switch(c){
                        case "[":
                            this.handlePositional();
                            this.state = State1.POSITIONAL;
                            break;
                        case "+":
                            flags.plus = true;
                            break;
                        case "<":
                            flags.lessthan = true;
                            break;
                        case "-":
                            flags.dash = true;
                            flags.zero = false;
                            break;
                        case "#":
                            flags.sharp = true;
                            break;
                        case " ":
                            flags.space = true;
                            break;
                        case "0":
                            flags.zero = !flags.dash;
                            break;
                        default:
                            if ("1" <= c && c <= "9" || c === "." || c === "*") {
                                if (c === ".") {
                                    this.flags.precision = 0;
                                    this.state = State1.PRECISION;
                                    this.i++;
                                } else {
                                    this.state = State1.WIDTH;
                                }
                                this.handleWidthAndPrecision(flags);
                            } else {
                                this.handleVerb();
                                return;
                            }
                    }
                    break;
                case State1.POSITIONAL:
                    if (c === "*") {
                        const worp = this.flags.precision === -1 ? WorP.WIDTH : WorP.PRECISION;
                        this.handleWidthOrPrecisionRef(worp);
                        this.state = State1.PERCENT;
                        break;
                    } else {
                        this.handleVerb();
                        return;
                    }
                default:
                    throw new Error(`Should not be here ${this.state}, library bug!`);
            }
        }
    }
    handleWidthOrPrecisionRef(wOrP) {
        if (this.argNum >= this.args.length) {
            return;
        }
        const arg = this.args[this.argNum];
        this.haveSeen[this.argNum] = true;
        if (typeof arg === "number") {
            switch(wOrP){
                case WorP.WIDTH:
                    this.flags.width = arg;
                    break;
                default:
                    this.flags.precision = arg;
            }
        } else {
            const tmp = wOrP === WorP.WIDTH ? "WIDTH" : "PREC";
            this.tmpError = `%!(BAD ${tmp} '${this.args[this.argNum]}')`;
        }
        this.argNum++;
    }
    handleWidthAndPrecision(flags) {
        const fmt = this.format;
        for(; this.i !== this.format.length; ++this.i){
            const c = fmt[this.i];
            switch(this.state){
                case State1.WIDTH:
                    switch(c){
                        case ".":
                            this.flags.precision = 0;
                            this.state = State1.PRECISION;
                            break;
                        case "*":
                            this.handleWidthOrPrecisionRef(WorP.WIDTH);
                            break;
                        default:
                            {
                                const val = parseInt(c);
                                if (isNaN(val)) {
                                    this.i--;
                                    this.state = State1.PERCENT;
                                    return;
                                }
                                flags.width = flags.width == -1 ? 0 : flags.width;
                                flags.width *= 10;
                                flags.width += val;
                            }
                    }
                    break;
                case State1.PRECISION:
                    {
                        if (c === "*") {
                            this.handleWidthOrPrecisionRef(WorP.PRECISION);
                            break;
                        }
                        const val1 = parseInt(c);
                        if (isNaN(val1)) {
                            this.i--;
                            this.state = State1.PERCENT;
                            return;
                        }
                        flags.precision *= 10;
                        flags.precision += val1;
                        break;
                    }
                default:
                    throw new Error("can't be here. bug.");
            }
        }
    }
    handlePositional() {
        if (this.format[this.i] !== "[") {
            throw new Error("Can't happen? Bug.");
        }
        let positional = 0;
        const format = this.format;
        this.i++;
        let err = false;
        for(; this.i !== this.format.length; ++this.i){
            if (format[this.i] === "]") {
                break;
            }
            positional *= 10;
            const val = parseInt(format[this.i]);
            if (isNaN(val)) {
                this.tmpError = "%!(BAD INDEX)";
                err = true;
            }
            positional += val;
        }
        if (positional - 1 >= this.args.length) {
            this.tmpError = "%!(BAD INDEX)";
            err = true;
        }
        this.argNum = err ? this.argNum : positional - 1;
        return;
    }
    handleLessThan() {
        const arg = this.args[this.argNum];
        if ((arg || {}).constructor.name !== "Array") {
            throw new Error(`arg ${arg} is not an array. Todo better error handling`);
        }
        let str = "[ ";
        for(let i1 = 0; i1 !== arg.length; ++i1){
            if (i1 !== 0) str += ", ";
            str += this._handleVerb(arg[i1]);
        }
        return str + " ]";
    }
    handleVerb() {
        const verb = this.format[this.i];
        this.verb = verb;
        if (this.tmpError) {
            this.buf += this.tmpError;
            this.tmpError = undefined;
            if (this.argNum < this.haveSeen.length) {
                this.haveSeen[this.argNum] = true;
            }
        } else if (this.args.length <= this.argNum) {
            this.buf += `%!(MISSING '${verb}')`;
        } else {
            const arg = this.args[this.argNum];
            this.haveSeen[this.argNum] = true;
            if (this.flags.lessthan) {
                this.buf += this.handleLessThan();
            } else {
                this.buf += this._handleVerb(arg);
            }
        }
        this.argNum++;
        this.state = State1.PASSTHROUGH;
    }
    _handleVerb(arg) {
        switch(this.verb){
            case "t":
                return this.pad(arg.toString());
            case "b":
                return this.fmtNumber(arg, 2);
            case "c":
                return this.fmtNumberCodePoint(arg);
            case "d":
                return this.fmtNumber(arg, 10);
            case "o":
                return this.fmtNumber(arg, 8);
            case "x":
                return this.fmtHex(arg);
            case "X":
                return this.fmtHex(arg, true);
            case "e":
                return this.fmtFloatE(arg);
            case "E":
                return this.fmtFloatE(arg, true);
            case "f":
            case "F":
                return this.fmtFloatF(arg);
            case "g":
                return this.fmtFloatG(arg);
            case "G":
                return this.fmtFloatG(arg, true);
            case "s":
                return this.fmtString(arg);
            case "T":
                return this.fmtString(typeof arg);
            case "v":
                return this.fmtV(arg);
            case "j":
                return this.fmtJ(arg);
            default:
                return `%!(BAD VERB '${this.verb}')`;
        }
    }
    pad(s) {
        const padding = this.flags.zero ? "0" : " ";
        if (this.flags.dash) {
            return s.padEnd(this.flags.width, padding);
        }
        return s.padStart(this.flags.width, padding);
    }
    padNum(nStr, neg) {
        let sign;
        if (neg) {
            sign = "-";
        } else if (this.flags.plus || this.flags.space) {
            sign = this.flags.plus ? "+" : " ";
        } else {
            sign = "";
        }
        const zero = this.flags.zero;
        if (!zero) {
            nStr = sign + nStr;
        }
        const pad = zero ? "0" : " ";
        const len = zero ? this.flags.width - sign.length : this.flags.width;
        if (this.flags.dash) {
            nStr = nStr.padEnd(len, pad);
        } else {
            nStr = nStr.padStart(len, pad);
        }
        if (zero) {
            nStr = sign + nStr;
        }
        return nStr;
    }
    fmtNumber(n, radix, upcase = false) {
        let num = Math.abs(n).toString(radix);
        const prec = this.flags.precision;
        if (prec !== -1) {
            this.flags.zero = false;
            num = n === 0 && prec === 0 ? "" : num;
            while(num.length < prec){
                num = "0" + num;
            }
        }
        let prefix = "";
        if (this.flags.sharp) {
            switch(radix){
                case 2:
                    prefix += "0b";
                    break;
                case 8:
                    prefix += num.startsWith("0") ? "" : "0";
                    break;
                case 16:
                    prefix += "0x";
                    break;
                default:
                    throw new Error("cannot handle base: " + radix);
            }
        }
        num = num.length === 0 ? num : prefix + num;
        if (upcase) {
            num = num.toUpperCase();
        }
        return this.padNum(num, n < 0);
    }
    fmtNumberCodePoint(n) {
        let s = "";
        try {
            s = String.fromCodePoint(n);
        } catch  {
            s = UNICODE_REPLACEMENT_CHARACTER;
        }
        return this.pad(s);
    }
    fmtFloatSpecial(n) {
        if (isNaN(n)) {
            this.flags.zero = false;
            return this.padNum("NaN", false);
        }
        if (n === Number.POSITIVE_INFINITY) {
            this.flags.zero = false;
            this.flags.plus = true;
            return this.padNum("Inf", false);
        }
        if (n === Number.NEGATIVE_INFINITY) {
            this.flags.zero = false;
            return this.padNum("Inf", true);
        }
        return "";
    }
    roundFractionToPrecision(fractional, precision) {
        let round = false;
        if (fractional.length > precision) {
            fractional = "1" + fractional;
            let tmp = parseInt(fractional.substr(0, precision + 2)) / 10;
            tmp = Math.round(tmp);
            fractional = Math.floor(tmp).toString();
            round = fractional[0] === "2";
            fractional = fractional.substr(1);
        } else {
            while(fractional.length < precision){
                fractional += "0";
            }
        }
        return [
            fractional,
            round
        ];
    }
    fmtFloatE(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen, bug");
        }
        let fractional = m[F.fractional];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        let rounding = false;
        [fractional, rounding] = this.roundFractionToPrecision(fractional, precision);
        let e = m[F.exponent];
        let esign = m[F.esign];
        let mantissa = parseInt(m[F.mantissa]);
        if (rounding) {
            mantissa += 1;
            if (10 <= mantissa) {
                mantissa = 1;
                const r = parseInt(esign + e) + 1;
                e = r.toString();
                esign = r < 0 ? "-" : "+";
            }
        }
        e = e.length == 1 ? "0" + e : e;
        const val = `${mantissa}.${fractional}${upcase ? "E" : "e"}${esign}${e}`;
        return this.padNum(val, n < 0);
    }
    fmtFloatF(n) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        function expandNumber(n) {
            if (Number.isSafeInteger(n)) {
                return n.toString() + ".";
            }
            const t = n.toExponential().split("e");
            let m = t[0].replace(".", "");
            const e = parseInt(t[1]);
            if (e < 0) {
                let nStr = "0.";
                for(let i1 = 0; i1 !== Math.abs(e) - 1; ++i1){
                    nStr += "0";
                }
                return nStr += m;
            } else {
                const splIdx = e + 1;
                while(m.length < splIdx){
                    m += "0";
                }
                return m.substr(0, splIdx) + "." + m.substr(splIdx);
            }
        }
        const val = expandNumber(Math.abs(n));
        const arr = val.split(".");
        let dig = arr[0];
        let fractional = arr[1];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        let round = false;
        [fractional, round] = this.roundFractionToPrecision(fractional, precision);
        if (round) {
            dig = (parseInt(dig) + 1).toString();
        }
        return this.padNum(`${dig}.${fractional}`, n < 0);
    }
    fmtFloatG(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        let P = this.flags.precision !== -1 ? this.flags.precision : 6;
        P = P === 0 ? 1 : P;
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen");
        }
        const X = parseInt(m[F.exponent]) * (m[F.esign] === "-" ? -1 : 1);
        let nStr = "";
        if (P > X && X >= -4) {
            this.flags.precision = P - (X + 1);
            nStr = this.fmtFloatF(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*$/, "");
            }
        } else {
            this.flags.precision = P - 1;
            nStr = this.fmtFloatE(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*e/, upcase ? "E" : "e");
            }
        }
        return nStr;
    }
    fmtString(s) {
        if (this.flags.precision !== -1) {
            s = s.substr(0, this.flags.precision);
        }
        return this.pad(s);
    }
    fmtHex(val, upper = false) {
        switch(typeof val){
            case "number":
                return this.fmtNumber(val, 16, upper);
            case "string":
                {
                    const sharp = this.flags.sharp && val.length !== 0;
                    let hex = sharp ? "0x" : "";
                    const prec = this.flags.precision;
                    const end = prec !== -1 ? min(prec, val.length) : val.length;
                    for(let i1 = 0; i1 !== end; ++i1){
                        if (i1 !== 0 && this.flags.space) {
                            hex += sharp ? " 0x" : " ";
                        }
                        const c = (val.charCodeAt(i1) & 0xff).toString(16);
                        hex += c.length === 1 ? `0${c}` : c;
                    }
                    if (upper) {
                        hex = hex.toUpperCase();
                    }
                    return this.pad(hex);
                }
            default:
                throw new Error("currently only number and string are implemented for hex");
        }
    }
    fmtV(val) {
        if (this.flags.sharp) {
            const options = this.flags.precision !== -1 ? {
                depth: this.flags.precision
            } : {};
            return this.pad(Deno.inspect(val, options));
        } else {
            const p = this.flags.precision;
            return p === -1 ? val.toString() : val.toString().substr(0, p);
        }
    }
    fmtJ(val) {
        return JSON.stringify(val);
    }
}
function glb(array, value, compare) {
    compare = compare || ((a, b)=>a - b);
    if (array.length === 0) {
        return -1;
    }
    if (array.length === 1) {
        if (compare(value, array[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array.length - 1;
    const vLeft = array[left], vRight = array[right];
    if (compare(value, vRight) >= 0) {
        return right;
    }
    if (compare(value, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array[center];
        const cmp = compare(value, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
let Deno3;
try {
    Deno3 = globalThis.Deno;
} catch (_e) {}
const noColor2 = typeof (Deno3 && Deno3.noColor) === "boolean" ? Deno3.noColor : true;
let enabled2 = !noColor2;
function code2(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run2(str, code) {
    return enabled2 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function red2(str) {
    return run2(str, code2([
        31
    ], 39));
}
function blue2(str) {
    return run2(str, code2([
        34
    ], 39));
}
const ANSI_PATTERN2 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor2(string) {
    return string.replace(ANSI_PATTERN2, "");
}
function platformHasNonAsciiCharacters() {
    try {
        return Deno.build.os !== "windows";
    } catch (_e) {
        return false;
    }
}
function tidyverseInfo(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${blue2("ℹ")} ${msg}`;
    } else {
        return `${blue2("i")} ${msg}`;
    }
}
function tidyverseError(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${red2("✖")} ${msg}`;
    } else {
        return `${red2("x")} ${msg}`;
    }
}
function tidyverseFormatError(msg) {
    let { heading , error , info  } = msg;
    if (msg.location) {
        heading = `${locationString(msg.location)} ${heading}`;
    }
    if (msg.fileName) {
        heading = `In file ${msg.fileName}\n${heading}`;
    }
    const strings = [
        heading,
        msg.sourceContext,
        ...error.map(tidyverseError),
        ...Object.values(info).map(tidyverseInfo),
        "", 
    ];
    return strings.join("\n");
}
function quotedStringColor(msg) {
    return blue2(msg);
}
function addFileInfo(msg, src) {
    if (src.fileName !== undefined) {
        msg.fileName = src.fileName;
    }
}
function addInstancePathInfo(msg, instancePath) {
    if (instancePath.length) {
        const niceInstancePath = instancePath.map((s)=>blue2(String(s))).join(":");
        msg.info["instance-path-location"] = `The error happened in location ${niceInstancePath}.`;
    }
}
function locationString(loc) {
    const { start , end  } = loc;
    if (start.line === end.line) {
        if (start.column === end.column) {
            return `(line ${start.line + 1}, column ${start.column + 1})`;
        } else {
            return `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})`;
        }
    } else {
        return `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
    }
}
function errorKey(err) {
    const positionKey = (pos)=>`${pos.line}-${pos.column}`;
    return `${err.fileName || ""}-${positionKey(err.location.start)}-${positionKey(err.location.end)}`;
}
function reportOnce(reporter, reportSet) {
    const errorsReported = reportSet || new Set();
    return (err)=>{
        const key = errorKey(err);
        if (errorsReported.has(key)) {
            return;
        }
        errorsReported.add(key);
        reporter(err);
    };
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function* matchAll(text, regexp) {
    if (!regexp.global) {
        throw new Error("matchAll requires global regexps");
    }
    let match;
    while((match = regexp.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function indexToLineCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function lineColToIndex(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(position) {
        return offsets[position.line] + position.column;
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad = " ".repeat(lineWidth);
    const ls = lines(text);
    const result1 = [];
    for(let i1 = firstLine; i1 <= lastLine; ++i1){
        const numberStr = `${pad}${i1 + 1}: `.slice(-(lineWidth + 2));
        const lineStr = ls[i1];
        result1.push({
            lineNumber: i1,
            content: numberStr + quotedStringColor(lineStr),
            rawLine: ls[i1]
        });
    }
    return {
        prefixWidth: lineWidth + 2,
        lines: result1
    };
}
function editDistance(w1, w2) {
    const cost = (c)=>{
        if ("_-".indexOf(c) !== -1) {
            return 1;
        }
        return 10;
    };
    const cost2 = (c1, c2)=>{
        if (c1 === c2) {
            return 0;
        }
        if ("_-".indexOf(c1) !== -1 && "_-".indexOf(c2) !== -1) {
            return 1;
        }
        if (c1.toLocaleLowerCase() === c2.toLocaleLowerCase()) {
            return 1;
        }
        const cc1 = c1.charCodeAt(0);
        const cc2 = c2.charCodeAt(0);
        if (cc1 >= 48 && cc1 <= 57 && cc2 >= 48 && cc2 <= 57) {
            return 1;
        }
        return 10;
    };
    const s1 = w1.length + 1;
    const s2 = w2.length + 1;
    const v = new Int32Array(s1 * s2);
    for(let i1 = 0; i1 < s1; ++i1){
        for(let j = 0; j < s2; ++j){
            if (i1 === 0 && j === 0) {
                continue;
            } else if (i1 === 0) {
                v[i1 * s2 + j] = v[i1 * s2 + (j - 1)] + cost(w2[j - 1]);
            } else if (j === 0) {
                v[i1 * s2 + j] = v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]);
            } else {
                v[i1 * s2 + j] = Math.min(v[(i1 - 1) * s2 + (j - 1)] + cost2(w1[i1 - 1], w2[j - 1]), v[i1 * s2 + (j - 1)] + cost(w2[j - 1]), v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]));
            }
        }
    }
    return v[(w1.length + 1) * (w2.length + 1) - 1];
}
function detectCaseConvention(key) {
    if (key.toLocaleLowerCase() !== key) {
        return "capitalizationCase";
    }
    if (key.indexOf("_") !== -1) {
        return "underscore_case";
    }
    if (key.indexOf("-") !== -1) {
        return "dash-case";
    }
    return undefined;
}
function resolveCaseConventionRegex(keys, conventions) {
    if (conventions !== undefined) {
        if (conventions.length === 0) {
            throw new Error("Internal Error: resolveCaseConventionRegex requires nonempty `conventions`");
        }
        return {
            pattern: conventions.map((c)=>`(${c})`).join("|"),
            list: conventions
        };
    }
    const disallowedNearMisses = [];
    const foundConventions = new Set();
    for (const key of keys){
        const found = detectCaseConvention(key);
        if (found) {
            foundConventions.add(found);
        }
        switch(found){
            case "capitalizationCase":
                disallowedNearMisses.push(toUnderscoreCase(key), toDashCase(key));
                break;
            case "dash-case":
                disallowedNearMisses.push(toUnderscoreCase(key), toCapitalizationCase(key));
                break;
            case "underscore_case":
                disallowedNearMisses.push(toDashCase(key), toCapitalizationCase(key));
                break;
        }
    }
    if (foundConventions.size === 0) {
        return {
            pattern: undefined,
            list: []
        };
    }
    return {
        pattern: `^(?!(${disallowedNearMisses.join("|")}))`,
        list: Array.from(foundConventions)
    };
}
function toDashCase(str) {
    return toUnderscoreCase(str).replace(/_/g, "-");
}
function toUnderscoreCase(str) {
    return str.replace(/([A-Z]+)/g, (_match, p1)=>`-${p1}`).replace(/-/g, "_").split("_").filter((x)=>x.length).join("_").toLocaleLowerCase();
}
function toCapitalizationCase(str) {
    return toUnderscoreCase(str).replace(/_(.)/g, (_match, p1)=>p1.toLocaleUpperCase());
}
function capitalizeWord(str) {
    return str.slice(0, 1).toUpperCase() + str.slice(1);
}
function capitalizeTitle(str) {
    return str.split(/\s+/).map((str, index, arr)=>{
        if (index === 0 || index === arr.length - 1 || !isNotCapitalized(str)) {
            return capitalizeWord(str);
        } else {
            return str;
        }
    }).join(" ");
}
function isNotCapitalized(str) {
    return [
        "a",
        "an",
        "the",
        "for",
        "and",
        "nor",
        "but",
        "or",
        "yet",
        "so",
        "with",
        "at",
        "by",
        "to",
        "in",
        "for",
        "from",
        "of",
        "on", 
    ].includes(str);
}
const kLastPunctuationRegex = /([\S\s]*)[\.\?\!]/;
function trimSentence(text) {
    const match = text.match(kLastPunctuationRegex);
    if (match) {
        return {
            text: match[0],
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function trimLength(text, length) {
    if (text.length < length) {
        return {
            text,
            trimmed: false
        };
    } else {
        return {
            text: text.substring(0, length),
            trimmed: true
        };
    }
}
function trimSpace(text) {
    const lastSpace = text.lastIndexOf(" ");
    if (lastSpace > 0) {
        return {
            text: text.substring(0, lastSpace),
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function truncateText(text, length, breakAt) {
    const trimEnd = (text)=>{
        if ([
            ",",
            "/",
            ":"
        ].includes(text.charAt(text.length - 1))) {
            return text.substring(0, text.length - 1);
        } else {
            return text;
        }
    };
    const trimAtSpace = (text)=>{
        const spaceResult = trimSpace(text.substring(0, text.length - 1));
        return trimEnd(spaceResult.text) + "…";
    };
    const trimPunc = (text)=>{
        const puncResult = trimSentence(text);
        if (puncResult.trimmed) {
            return puncResult.text;
        } else {
            return trimAtSpace(puncResult.text);
        }
    };
    const lengthResult = trimLength(text, length);
    if (lengthResult.trimmed) {
        if (breakAt === "punctuation") {
            return trimPunc(lengthResult.text);
        } else {
            return trimAtSpace(lengthResult.text);
        }
    } else {
        return lengthResult.text;
    }
}
function matchAll1(str, regex) {
    let match;
    regex = new RegExp(regex);
    const result1 = [];
    while((match = regex.exec(str)) != null){
        result1.push(match);
    }
    return result1;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result1 = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result1.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result1.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result1;
    } else {
        const matches = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r1 of matches){
            const stringEnd = r1.index + r1[0].length;
            result1.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result1.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result1;
    }
}
function mappedSubstring(source, start, end) {
    if (typeof source === "string") {
        source = asMappedString(source);
    }
    const value = source.value.substring(start, end);
    const mappedSource = source;
    return {
        value,
        map: (index, closest)=>{
            if (closest) {
                index = Math.max(0, Math.min(value.length, index - 1));
            }
            if (index === 0 && index === value.length) {
                return mappedSource.map(index + start, closest);
            }
            if (index < 0 || index >= value.length) {
                return undefined;
            }
            return mappedSource.map(index + start, closest);
        }
    };
}
function mappedString(source, pieces, fileName) {
    if (typeof source === "string") {
        source = asMappedString(source, fileName);
    }
    const mappedPieces = pieces.map((piece)=>{
        if (typeof piece === "string") {
            return asMappedString(piece);
        } else if (piece.value !== undefined) {
            return piece;
        } else {
            const { start , end  } = piece;
            return mappedSubstring(source, start, end);
        }
    });
    return mappedConcat(mappedPieces);
}
function asMappedString(str, fileName) {
    if (typeof str === "string") {
        return {
            value: str,
            fileName,
            map: function(index, closest) {
                if (closest) {
                    index = Math.min(str.length - 1, Math.max(0, index));
                }
                if (index < 0 || index >= str.length) {
                    return undefined;
                }
                return {
                    index,
                    originalString: this
                };
            }
        };
    } else if (fileName !== undefined) {
        throw new Error("Internal error: can't change the fileName of an existing MappedString");
    } else {
        return str;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        return {
            value: "",
            map: (_index, _closest)=>undefined
        };
    }
    if (strings.every((s)=>typeof s === "string")) {
        return asMappedString(strings.join(""));
    }
    const mappedStrings = strings.map((s)=>{
        if (typeof s === "string") {
            return asMappedString(s);
        } else return s;
    });
    let currentOffset = 0;
    const offsets = [
        0
    ];
    for (const s of mappedStrings){
        currentOffset += s.value.length;
        offsets.push(currentOffset);
    }
    const value = mappedStrings.map((s)=>s.value).join("");
    return {
        value,
        map: (offset, closest)=>{
            if (closest) {
                offset = Math.max(0, Math.min(offset, value.length - 1));
            }
            if (offset === 0 && offset == value.length && mappedStrings.length) {
                return mappedStrings[0].map(0, closest);
            }
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            const v = mappedStrings[ix];
            return v.map(offset - offsets[ix]);
        }
    };
}
function mappedIndexToLineCol(eitherText) {
    const text = asMappedString(eitherText);
    return function(offset) {
        const mapResult = text.map(offset, true);
        if (mapResult === undefined) {
            throw new Error("Internal Error: bad offset in mappedIndexRowCol");
        }
        const { index , originalString  } = mapResult;
        return indexToLineCol(originalString.value)(index);
    };
}
function join4(mappedStrs, sep) {
    const innerStrings = [];
    const mappedSep = asMappedString(sep);
    for(let i1 = 0; i1 < mappedStrs.length; ++i1){
        const mappedStr = mappedStrs[i1];
        if (typeof mappedStr === "string") {
            innerStrings.push(asMappedString(mappedStr));
        } else {
            innerStrings.push(mappedStr);
        }
        if (i1 < mappedStrs.length) {
            innerStrings.push(mappedSep);
        }
    }
    return mappedConcat(innerStrings);
}
function mappedLines(str, keepNewLines = false) {
    const lines = rangedLines(str.value, keepNewLines);
    return lines.map((v)=>mappedString(str, [
            v.range
        ]));
}
function findSpaceStart(string) {
    const result1 = string.value.match(/^\s+/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function findSpaceEnd(string) {
    const result1 = string.value.match(/\s+$/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function mappedTrim(string) {
    const start = findSpaceStart(string);
    const end = findSpaceEnd(string);
    if (start === 0 && end === 0) {
        return string;
    }
    if (start > string.value.length - end) {
        return mappedSubstring(string, 0, 0);
    }
    return mappedSubstring(string, start, string.value.length - end);
}
class ErrorEx extends Error {
    constructor(name, message, printName = true, printStack = true){
        super(message);
        this.name = name;
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
class LocalizedError extends Error {
    constructor(name, message, source, position = 0, printName = true, printStack = true){
        const fileName = source.map(position)?.originalString?.fileName;
        if (fileName) {
            const { line , column  } = mappedIndexToLineCol(source)(position);
            message = `In file ${fileName} (${line + 1}:${column + 1}):
${message}`;
        }
        super(message);
        this.name = name;
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
function asErrorEx(e) {
    if (e instanceof ErrorEx) {
        return e;
    } else if (e instanceof Error) {
        e.printName = e.name !== "Error";
        e.printStack = !!e.message;
        return e;
    } else {
        return new ErrorEx("Error", String(e), false, true);
    }
}
const booleanSchema = {
    "type": "boolean",
    "description": "be `true` or `false`",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const numberSchema = {
    "type": "number",
    "description": "be a number"
};
const stringSchema = {
    "type": "string",
    "description": "be a string"
};
const nullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function schemaType(schema) {
    if (schema === false) {
        return "false";
    }
    if (schema === true) {
        return "true";
    }
    return schema.type;
}
function schemaDispatch(s, d) {
    const st = schemaType(s);
    if (d[st]) {
        d[st](s);
    }
}
function schemaCall(s, d, other) {
    const st = schemaType(s);
    if (d[st]) {
        return d[st](s);
    }
    if (other) {
        return other(s);
    }
    throw new Error(`Internal Error: dispatch failed for type ${st}`);
}
function schemaDocString(d) {
    if (typeof d === "string") {
        return d;
    }
    if (d.short) {
        return d.short;
    }
    return "";
}
function schemaDescription(schema) {
    if (schema === true) {
        return `be anything`;
    } else if (schema === false) {
        return `be no possible value`;
    } else {
        return schema.description || `be ${schemaType(schema)}`;
    }
}
let globalInternalIdCounter = 0;
function internalId() {
    return {
        _internalId: ++globalInternalIdCounter
    };
}
function tagSchema(schema, tags) {
    return {
        ...schema,
        tags: {
            ...schema.tags || {},
            ...tags
        }
    };
}
function anySchema(description) {
    return {
        ...internalId(),
        description,
        "type": "any"
    };
}
function enumSchema(...args) {
    if (args.length === 0) {
        throw new Error("Internal Error: Empty enum schema not supported.");
    }
    return {
        ...internalId(),
        "type": "enum",
        "enum": args,
        "description": args.length > 1 ? `be one of: ${args.map((x)=>"`" + x + "`").join(", ")}` : `be '${args[0]}'`,
        "completions": args.map(String),
        "exhaustiveCompletions": true
    };
}
function regexSchema(arg, description) {
    const result1 = {
        ...internalId(),
        "type": "string",
        "pattern": arg
    };
    if (description) {
        result1.description = description;
    } else {
        result1.description = `be a string that satisfies regex "${arg}"`;
    }
    return result1;
}
function anyOfSchema(...args) {
    return {
        ...internalId(),
        "type": "anyOf",
        "anyOf": args,
        "description": `be at least one of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function allOfSchema(...args) {
    return {
        ...internalId(),
        "type": "allOf",
        "allOf": args,
        "description": `be all of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function objectSchema(params = {}) {
    let { properties , patternProperties , required , additionalProperties , description , baseSchema , exhaustive , completions: completionsParam , namingConvention , propertyNames: propertyNamesSchema , closed ,  } = params;
    required = required || [];
    properties = properties || {};
    patternProperties = patternProperties || {};
    const tags = {};
    let tagsAreSet = false;
    let propertyNames = propertyNamesSchema;
    const objectKeys = Object.getOwnPropertyNames(completionsParam || properties);
    if (namingConvention !== "ignore") {
        const { pattern , list  } = resolveCaseConventionRegex(objectKeys, namingConvention);
        if (pattern !== undefined) {
            if (propertyNames === undefined) {
                propertyNames = {
                    "type": "string",
                    pattern
                };
            } else {
                propertyNames = allOfSchema(propertyNames, {
                    "type": "string",
                    pattern
                });
            }
            tags["case-convention"] = list;
            tagsAreSet = true;
        }
    }
    if (completionsParam) {
        tags["completions"] = completionsParam;
        tagsAreSet = true;
    }
    const hasDescription = description !== undefined;
    description = description || "be an object";
    let result1 = undefined;
    if (baseSchema) {
        if (!Array.isArray(baseSchema)) {
            baseSchema = [
                baseSchema
            ];
        }
        if (baseSchema.some((s)=>s.type !== "object")) {
            throw new Error("Internal Error: can only extend other object Schema");
        }
        if (baseSchema.length <= 0) {
            throw new Error("Internal Error: base schema must be non-empty");
        }
        let temp = {
            ...internalId()
        };
        for (const base of baseSchema){
            temp = Object.assign(temp, base);
        }
        result1 = temp;
        if (result1 === undefined) {
            throw new Error("Internal Error: result should not be undefined");
        }
        if (result1.$id) {
            delete result1.$id;
        }
        for (const base1 of baseSchema){
            if (base1.exhaustiveCompletions) {
                result1.exhaustiveCompletions = true;
            }
        }
        if (hasDescription) {
            result1.description = description;
        }
        result1.properties = Object.assign({}, ...baseSchema.map((s)=>s.properties), properties);
        result1.patternProperties = Object.assign({}, ...baseSchema.map((s)=>s.patternProperties), patternProperties);
        result1.required = [
            ...baseSchema.map((s)=>s.required || []),
            required || [], 
        ].flat();
        if (result1.required && result1.required.length === 0) {
            result1.required = undefined;
        }
        const additionalPropArray = baseSchema.map((s)=>s.additionalProperties).filter((s)=>s !== undefined);
        if (additionalProperties) {
            additionalPropArray.push(additionalProperties);
        }
        if (additionalPropArray.length) {
            result1.additionalProperties = allOfSchema(...additionalPropArray);
        }
        const propNamesArray = baseSchema.map((s)=>s.propertyNames).filter((s)=>s !== undefined);
        if (propertyNames) {
            propNamesArray.push(propertyNames);
        }
        if (propNamesArray.length === baseSchema.length + 1) {
            result1.propertyNames = anyOfSchema(...propNamesArray);
        }
        result1.closed = closed || baseSchema.some((s)=>s.closed);
    } else {
        result1 = {
            ...internalId(),
            "type": "object",
            description
        };
        if (exhaustive) {
            result1.exhaustiveCompletions = true;
        }
        if (properties) {
            result1.properties = properties;
        }
        if (patternProperties) {
            result1.patternProperties = patternProperties;
        }
        if (required && required.length > 0) {
            result1.required = required;
        }
        result1.closed = closed;
        if (additionalProperties !== undefined) {
            result1.additionalProperties = additionalProperties;
        }
        if (propertyNames !== undefined) {
            result1.propertyNames = propertyNames;
        }
    }
    if (tagsAreSet) {
        result1.tags = tags;
    }
    return result1;
}
function arraySchema(items) {
    if (items) {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values, where each element must ${schemaDescription(items)}`,
            items
        };
    } else {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema, doc) {
    const result1 = Object.assign({}, schema);
    result1.documentation = doc;
    return result1;
}
function describeSchema(schema, description) {
    const result1 = Object.assign({}, schema);
    result1.description = `be ${description}`;
    return result1;
}
function completeSchema(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    const prevCompletions = (schema.completions || []).slice();
    prevCompletions.push(...completions);
    result1.completions = prevCompletions;
    return result1;
}
function completeSchemaOverwrite(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    result1.completions = completions;
    return result1;
}
function idSchema(schema, id) {
    const result1 = Object.assign({}, schema);
    result1["$id"] = id;
    return result1;
}
function errorMessageSchema(schema, errorMessage) {
    return {
        ...schema,
        errorMessage
    };
}
function refSchema($ref, description) {
    return {
        ...internalId(),
        "type": "ref",
        $ref,
        description
    };
}
function valueSchema(val, description) {
    return {
        ...internalId(),
        "type": "enum",
        "enum": [
            val
        ],
        "description": description || `be ${JSON.stringify(val)}`
    };
}
const definitionsObject = {};
function hasSchemaDefinition(key) {
    return definitionsObject[key] !== undefined;
}
function getSchemaDefinition(key) {
    if (definitionsObject[key] === undefined) {
        throw new Error(`Internal Error: Schema ${key} not found.`);
    }
    return definitionsObject[key];
}
function setSchemaDefinition(schema) {
    if (schema.$id === undefined) {
        throw new Error("Internal Error, setSchemaDefinition needs $id");
    }
    if (definitionsObject[schema.$id] === undefined) {
        definitionsObject[schema.$id] = schema;
    }
}
function getSchemaDefinitionsObject() {
    return Object.assign({}, definitionsObject);
}
function expandAliasesFrom(lst, defs) {
    const aliases = defs;
    const result1 = [];
    lst = lst.slice();
    for(let i1 = 0; i1 < lst.length; ++i1){
        const el = lst[i1];
        if (el.startsWith("$")) {
            const v = aliases[el.slice(1)];
            if (v === undefined) {
                throw new Error(`Internal Error: ${el} doesn't have an entry in the aliases map`);
            }
            lst.push(...v);
        } else {
            result1.push(el);
        }
    }
    return result1;
}
function resolveSchema(schema, visit, hasRef, next) {
    if (schema === false || schema === true) {
        return schema;
    }
    if (hasRef === undefined) {
        hasRef = (cursor)=>{
            return schemaCall(cursor, {
                ref: (_s)=>true
            }, (_s)=>false);
        };
    }
    if (!hasRef(schema)) {
        return schema;
    }
    if (visit === undefined) {
        visit = (_schema)=>{};
    }
    if (next === undefined) {
        next = (cursor)=>{
            const result1 = schemaCall(cursor, {
                ref: (s)=>getSchemaDefinition(s.$ref)
            });
            if (result1 === undefined) {
                throw new Error("Internal Error, couldn't resolve schema ${JSON.stringify(cursor)}");
            }
            return result1;
        };
    }
    let cursor1 = schema;
    let cursor2 = schema;
    let stopped = false;
    do {
        cursor1 = next(cursor1);
        visit(cursor1);
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (!stopped && cursor1 === cursor2) {
            throw new Error(`reference cycle detected at ${JSON.stringify(cursor1)}`);
        }
    }while (hasRef(cursor1))
    return cursor1;
}
Object.freeze({
    __proto__: null
});
class DataSet {
    constructor(raw2018, raw2019, raw2020, raw2021){
        this._raw2018 = raw2018;
        this._raw2019 = raw2019;
        this._raw2020 = raw2020;
        this._raw2021 = raw2021;
    }
    get es2018() {
        return this._set2018 || (this._set2018 = new Set(this._raw2018.split(" ")));
    }
    get es2019() {
        return this._set2019 || (this._set2019 = new Set(this._raw2019.split(" ")));
    }
    get es2020() {
        return this._set2020 || (this._set2020 = new Set(this._raw2020.split(" ")));
    }
    get es2021() {
        return this._set2021 || (this._set2021 = new Set(this._raw2021.split(" ")));
    }
}
new Set([
    "General_Category",
    "gc"
]);
new Set([
    "Script",
    "Script_Extensions",
    "sc",
    "scx"
]);
new DataSet("C Cased_Letter Cc Cf Close_Punctuation Cn Co Combining_Mark Connector_Punctuation Control Cs Currency_Symbol Dash_Punctuation Decimal_Number Enclosing_Mark Final_Punctuation Format Initial_Punctuation L LC Letter Letter_Number Line_Separator Ll Lm Lo Lowercase_Letter Lt Lu M Mark Math_Symbol Mc Me Mn Modifier_Letter Modifier_Symbol N Nd Nl No Nonspacing_Mark Number Open_Punctuation Other Other_Letter Other_Number Other_Punctuation Other_Symbol P Paragraph_Separator Pc Pd Pe Pf Pi Po Private_Use Ps Punctuation S Sc Separator Sk Sm So Space_Separator Spacing_Mark Surrogate Symbol Titlecase_Letter Unassigned Uppercase_Letter Z Zl Zp Zs cntrl digit punct", "", "", "");
new DataSet("Adlam Adlm Aghb Ahom Anatolian_Hieroglyphs Arab Arabic Armenian Armi Armn Avestan Avst Bali Balinese Bamu Bamum Bass Bassa_Vah Batak Batk Beng Bengali Bhaiksuki Bhks Bopo Bopomofo Brah Brahmi Brai Braille Bugi Buginese Buhd Buhid Cakm Canadian_Aboriginal Cans Cari Carian Caucasian_Albanian Chakma Cham Cher Cherokee Common Copt Coptic Cprt Cuneiform Cypriot Cyrillic Cyrl Deseret Deva Devanagari Dsrt Dupl Duployan Egyp Egyptian_Hieroglyphs Elba Elbasan Ethi Ethiopic Geor Georgian Glag Glagolitic Gonm Goth Gothic Gran Grantha Greek Grek Gujarati Gujr Gurmukhi Guru Han Hang Hangul Hani Hano Hanunoo Hatr Hatran Hebr Hebrew Hira Hiragana Hluw Hmng Hung Imperial_Aramaic Inherited Inscriptional_Pahlavi Inscriptional_Parthian Ital Java Javanese Kaithi Kali Kana Kannada Katakana Kayah_Li Khar Kharoshthi Khmer Khmr Khoj Khojki Khudawadi Knda Kthi Lana Lao Laoo Latin Latn Lepc Lepcha Limb Limbu Lina Linb Linear_A Linear_B Lisu Lyci Lycian Lydi Lydian Mahajani Mahj Malayalam Mand Mandaic Mani Manichaean Marc Marchen Masaram_Gondi Meetei_Mayek Mend Mende_Kikakui Merc Mero Meroitic_Cursive Meroitic_Hieroglyphs Miao Mlym Modi Mong Mongolian Mro Mroo Mtei Mult Multani Myanmar Mymr Nabataean Narb Nbat New_Tai_Lue Newa Nko Nkoo Nshu Nushu Ogam Ogham Ol_Chiki Olck Old_Hungarian Old_Italic Old_North_Arabian Old_Permic Old_Persian Old_South_Arabian Old_Turkic Oriya Orkh Orya Osage Osge Osma Osmanya Pahawh_Hmong Palm Palmyrene Pau_Cin_Hau Pauc Perm Phag Phags_Pa Phli Phlp Phnx Phoenician Plrd Prti Psalter_Pahlavi Qaac Qaai Rejang Rjng Runic Runr Samaritan Samr Sarb Saur Saurashtra Sgnw Sharada Shavian Shaw Shrd Sidd Siddham SignWriting Sind Sinh Sinhala Sora Sora_Sompeng Soyo Soyombo Sund Sundanese Sylo Syloti_Nagri Syrc Syriac Tagalog Tagb Tagbanwa Tai_Le Tai_Tham Tai_Viet Takr Takri Tale Talu Tamil Taml Tang Tangut Tavt Telu Telugu Tfng Tglg Thaa Thaana Thai Tibetan Tibt Tifinagh Tirh Tirhuta Ugar Ugaritic Vai Vaii Wara Warang_Citi Xpeo Xsux Yi Yiii Zanabazar_Square Zanb Zinh Zyyy", "Dogr Dogra Gong Gunjala_Gondi Hanifi_Rohingya Maka Makasar Medefaidrin Medf Old_Sogdian Rohg Sogd Sogdian Sogo", "Elym Elymaic Hmnp Nand Nandinagari Nyiakeng_Puachue_Hmong Wancho Wcho", "Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi");
new DataSet("AHex ASCII ASCII_Hex_Digit Alpha Alphabetic Any Assigned Bidi_C Bidi_Control Bidi_M Bidi_Mirrored CI CWCF CWCM CWKCF CWL CWT CWU Case_Ignorable Cased Changes_When_Casefolded Changes_When_Casemapped Changes_When_Lowercased Changes_When_NFKC_Casefolded Changes_When_Titlecased Changes_When_Uppercased DI Dash Default_Ignorable_Code_Point Dep Deprecated Dia Diacritic Emoji Emoji_Component Emoji_Modifier Emoji_Modifier_Base Emoji_Presentation Ext Extender Gr_Base Gr_Ext Grapheme_Base Grapheme_Extend Hex Hex_Digit IDC IDS IDSB IDST IDS_Binary_Operator IDS_Trinary_Operator ID_Continue ID_Start Ideo Ideographic Join_C Join_Control LOE Logical_Order_Exception Lower Lowercase Math NChar Noncharacter_Code_Point Pat_Syn Pat_WS Pattern_Syntax Pattern_White_Space QMark Quotation_Mark RI Radical Regional_Indicator SD STerm Sentence_Terminal Soft_Dotted Term Terminal_Punctuation UIdeo Unified_Ideograph Upper Uppercase VS Variation_Selector White_Space XIDC XIDS XID_Continue XID_Start space", "Extended_Pictographic", "", "EBase EComp EMod EPres ExtPict");
function navigateSchemaBySchemaPathSingle(schema, path) {
    const ensurePathFragment = (fragment, expected)=>{
        if (fragment !== expected) {
            throw new Error(`Internal Error in navigateSchemaBySchemaPathSingle: ${fragment} !== ${expected}`);
        }
    };
    const inner = (subschema, index)=>{
        subschema = resolveSchema(subschema);
        if (subschema === undefined) {
            throw new Error(`Internal Error in navigateSchemaBySchemaPathSingle: invalid path navigation`);
        }
        if (index === path.length) {
            return subschema;
        }
        const st = schemaType(subschema);
        switch(st){
            case "anyOf":
                ensurePathFragment(path[index], "anyOf");
                return inner(subschema.anyOf[path[index + 1]], index + 2);
            case "allOf":
                ensurePathFragment(path[index], "allOf");
                return inner(subschema.allOf[path[index + 1]], index + 2);
            case "array":
                ensurePathFragment(path[index], "array");
                return inner(subschema.arrayOf.schema, index + 2);
            case "object":
                ensurePathFragment(path[index], "object");
                if (path[index + 1] === "properties") {
                    return inner(subschema.properties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "patternProperties") {
                    return inner(subschema.patternProperties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "additionalProperties") {
                    return inner(subschema.additionalProperties, index + 2);
                } else {
                    throw new Error(`Internal Error in navigateSchemaBySchemaPathSingle: bad path fragment ${path[index]} in object navigation`);
                }
            default:
                throw new Error(`Internal Error in navigateSchemaBySchemaPathSingle: can't navigate schema type ${st}`);
        }
    };
    return inner(schema, 0);
}
function resolveDescription(s) {
    if (typeof s === "string") {
        return s;
    }
    const valueS = resolveSchema(s);
    if (valueS === false || valueS === true) {
        return "";
    }
    if (valueS.documentation === undefined) {
        return "";
    }
    if (typeof valueS.documentation === "string") {
        return valueS.documentation;
    }
    if (valueS.documentation.short) {
        return valueS.documentation.short;
    } else {
        return "";
    }
}
function schemaCompletions(s) {
    if (s === true || s === false) {
        return [];
    }
    let schema = resolveSchema(s);
    schema = resolveSchema(schema, (_schema)=>{}, (schema)=>{
        return schema.tags !== undefined && schema.tags["complete-from"] !== undefined;
    }, (schema)=>{
        return navigateSchemaBySchemaPathSingle(schema, schema.tags["complete-from"]);
    });
    if (schema === true || schema === false) {
        return [];
    }
    const normalize = (completions)=>{
        const result1 = (completions || []).map((c)=>{
            if (typeof c === "string") {
                return {
                    type: "value",
                    display: c,
                    value: c,
                    description: "",
                    suggest_on_accept: false,
                    schema
                };
            }
            return {
                ...c,
                description: resolveDescription(c.description),
                schema
            };
        });
        return result1;
    };
    if (schema.completions && schema.completions.length) {
        return normalize(schema.completions);
    }
    if (schema.tags && schema.tags.completions) {
        if (Array.isArray(schema.tags.completions) && schema.tags.completions.length) {
            return normalize(schema.tags.completions);
        } else {
            return normalize(Object.values(schema.tags.completions));
        }
    }
    return schemaCall(schema, {
        array: (s)=>{
            if (s.items) {
                return schemaCompletions(s.items);
            } else {
                return [];
            }
        },
        anyOf: (s)=>{
            return s.anyOf.map(schemaCompletions).flat();
        },
        allOf: (s)=>{
            return s.allOf.map(schemaCompletions).flat();
        },
        "object": (s)=>{
            s.cachedCompletions = getObjectCompletions(s);
            return normalize(s.cachedCompletions);
        }
    }, (_)=>[]);
}
function getObjectCompletions(s) {
    const completionsParam = s.tags && s.tags.completions || [];
    return schemaCall(s, {
        "object": (schema)=>{
            const properties = schema.properties;
            const objectKeys = completionsParam.length ? completionsParam : Object.getOwnPropertyNames(properties);
            const completions = [];
            for (const k of objectKeys){
                const schema1 = properties && properties[k];
                const maybeDescriptions = [];
                let hidden = false;
                if (schema1 !== undefined && schema1 !== true && schema1 !== false) {
                    if (schema1.documentation) {
                        maybeDescriptions.push(schemaDocString(schema1.documentation));
                    } else {
                        let described = false;
                        const visitor = (schema)=>{
                            if (schema === false || schema === true) {
                                return;
                            }
                            if (schema.hidden) {
                                hidden = true;
                            }
                            if (described) {
                                return;
                            }
                            if (schema.documentation) {
                                maybeDescriptions.push(schemaDocString(schema.documentation));
                                described = true;
                            }
                        };
                        try {
                            resolveSchema(schema1, visitor);
                        } catch (_e) {}
                        if (!described) {
                            schemaDispatch(schema1, {
                                ref: (schema)=>maybeDescriptions.push({
                                        $ref: schema.$ref
                                    })
                            });
                        }
                    }
                }
                if (hidden) {
                    continue;
                }
                let description = "";
                for (const md of maybeDescriptions){
                    if (md !== undefined) {
                        description = md;
                        break;
                    }
                }
                completions.push({
                    type: "key",
                    display: "",
                    value: `${k}: `,
                    description,
                    suggest_on_accept: true
                });
            }
            return completions;
        }
    }, (_)=>completionsParam.map((c)=>({
                type: "value",
                display: "",
                value: c,
                description: "",
                suggest_on_accept: false
            })));
}
function possibleSchemaKeys(schema) {
    const precomputedCompletions = schemaCompletions(schema).filter((c)=>c.type === "key").map((c)=>c.value.split(":")[0]);
    if (precomputedCompletions.length) {
        return precomputedCompletions;
    }
    const results = [];
    walkSchema(schema, {
        "object": (s)=>{
            results.push(...Object.keys(s.properties || {}));
            return true;
        },
        "array": (_s)=>true
    });
    return results;
}
function possibleSchemaValues(schema) {
    const results = [];
    walkSchema(schema, {
        "enum": (s)=>{
            results.push(...s["enum"].map(String));
            return true;
        },
        "array": (_s)=>true,
        "object": (_s)=>true
    });
    return results;
}
function walkSchema(schema, f) {
    const recur = {
        "anyOf": (ss)=>{
            for (const s of ss.anyOf){
                walkSchema(s, f);
            }
        },
        "allOf": (ss)=>{
            for (const s of ss.allOf){
                walkSchema(s, f);
            }
        },
        "array": (x)=>{
            if (x.items) {
                walkSchema(x.items, f);
            }
        },
        "object": (x)=>{
            if (x.properties) {
                for (const ss of Object.values(x.properties)){
                    walkSchema(ss, f);
                }
            }
            if (x.patternProperties) {
                for (const ss1 of Object.values(x.patternProperties)){
                    walkSchema(ss1, f);
                }
            }
            if (x.propertyNames) {
                walkSchema(x.propertyNames, f);
            }
        }
    };
    if (typeof f === "function") {
        if (f(schema) === true) {
            return;
        }
    } else {
        if (schemaCall(schema, f, (_)=>false) === true) {
            return;
        }
    }
    schemaCall(schema, recur, (_)=>false);
}
function setDefaultErrorHandlers(validator) {
    validator.addHandler(ignoreExprViolations);
    validator.addHandler(expandEmptySpan);
    validator.addHandler(improveErrorHeadingForValueErrors);
    validator.addHandler(checkForTypeMismatch);
    validator.addHandler(checkForBadBoolean);
    validator.addHandler(checkForBadColon);
    validator.addHandler(checkForBadEquals);
    validator.addHandler(identifyKeyErrors);
    validator.addHandler(checkForNearbyCorrection);
    validator.addHandler(checkForNearbyRequired);
    validator.addHandler(schemaDefinedErrors);
}
function errorKeyword(error) {
    if (error.schemaPath.length === 0) {
        return "";
    }
    return String(error.schemaPath[error.schemaPath.length - 1]);
}
function schemaPathMatches(error, strs) {
    const schemaPath = error.schemaPath.slice(-strs.length);
    if (schemaPath.length !== strs.length) {
        return false;
    }
    return strs.every((str, i1)=>str === schemaPath[i1]);
}
function getBadKey(error) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return undefined;
    }
    const result1 = error.violatingObject.result;
    if (typeof result1 !== "string") {
        throw new Error("Internal Error: propertyNames error has a violating non-string.");
    }
    return result1;
}
function getVerbatimInput(error) {
    return error.source.value;
}
function navigate(path, annotation, returnKey = false, pathIndex = 0) {
    if (annotation === undefined) {
        throw new Error("Can't navigate an undefined annotation");
    }
    if (pathIndex >= path.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components  } = annotation;
        const searchKey = path[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i1 = lastKeyIndex; i1 >= 0; i1 -= 2){
            const key = components[i1].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path.length - 1) {
                    return navigate(path, components[i1], returnKey, pathIndex + 1);
                } else {
                    return navigate(path, components[i1 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        return annotation;
    } else if ([
        "sequence",
        "block_sequence",
        "flow_sequence"
    ].indexOf(annotation.kind) !== -1) {
        const searchKey1 = Number(path[pathIndex]);
        if (isNaN(searchKey1) || searchKey1 < 0 || searchKey1 >= annotation.components.length) {
            return annotation;
        }
        return navigate(path, annotation.components[searchKey1], returnKey, pathIndex + 1);
    } else {
        return annotation;
    }
}
function isEmptyValue(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    return rawVerbatimInput.trim().length === 0;
}
function getLastFragment(instancePath) {
    if (instancePath.length === 0) {
        return undefined;
    }
    return instancePath[instancePath.length - 1];
}
function reindent(str) {
    const s = new Set();
    const ls = lines(str);
    for (const l1 of ls){
        const r = l1.match("^[ ]+");
        if (r) {
            s.add(r[0].length);
        }
    }
    if (s.size === 0) {
        return str;
    } else if (s.size === 1) {
        const v = Array.from(s)[0];
        const oldIndent = " ".repeat(v);
        if (v <= 2) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent) ? l1.slice(v - 2) : l1).join("\n");
    } else {
        const [first, second] = Array.from(s);
        const oldIndent1 = " ".repeat(first);
        const newIndent = second - first;
        if (newIndent >= first) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent1) ? l1.slice(first - newIndent) : l1).join("\n");
    }
}
function ignoreExprViolations(error, _parse, _schema) {
    const { result: result1  } = error.violatingObject;
    if (typeof result1 !== "object" || Array.isArray(result1) || result1 === null || error.schemaPath.slice(-1)[0] !== "type") {
        return error;
    }
    if (result1.tag === "!expr" && typeof result1.value === "string") {
        return null;
    } else {
        return error;
    }
}
function formatHeadingForKeyError(_error, _parse, _schema, key) {
    return `property name ${blue2(key)} is invalid`;
}
function formatHeadingForValueError(error, _parse, _schema) {
    const rawVerbatimInput = reindent(getVerbatimInput(error));
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const empty = isEmptyValue(error);
    const lastFragment = getLastFragment(error.instancePath);
    switch(typeof lastFragment){
        case "undefined":
            if (empty) {
                return "YAML value is missing.";
            } else {
                return `YAML value ${verbatimInput} must ${schemaDescription(error.schema)}.`;
            }
        case "number":
            if (empty) {
                return `Array entry ${lastFragment + 1} is empty but it must instead ${schemaDescription(error.schema)}.`;
            } else {
                return `Array entry ${lastFragment + 1} with value ${verbatimInput} failed to ${schemaDescription(error.schema)}.`;
            }
        case "string":
            {
                const formatLastFragment = '"' + blue2(lastFragment) + '"';
                if (empty) {
                    return `Field ${formatLastFragment} has empty value but it must instead ${schemaDescription(error.schema)}`;
                } else {
                    if (verbatimInput.indexOf("\n") !== -1) {
                        return `Field ${formatLastFragment} has value

${verbatimInput}

The value must instead ${schemaDescription(error.schema)}.`;
                    } else {
                        return `Field ${formatLastFragment} has value ${verbatimInput}, which must instead ${schemaDescription(error.schema)}`;
                    }
                }
            }
    }
}
function identifyKeyErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return error;
    }
    const badKey = getBadKey(error);
    if (badKey) {
        if (error.instancePath.length && error.instancePath[error.instancePath.length - 1] !== badKey) {
            addInstancePathInfo(error.niceError, [
                ...error.instancePath,
                badKey
            ]);
        } else {
            addInstancePathInfo(error.niceError, error.instancePath);
        }
        error.niceError.heading = formatHeadingForKeyError(error, parse, schema, badKey);
    }
    return error;
}
function improveErrorHeadingForValueErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") !== -1 || error.schemaPath.indexOf("closed") !== -1 || errorKeyword(error) === "required") {
        return error;
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema)
        }
    };
}
function expandEmptySpan(error, parse, _schema) {
    if (error.location.start.line !== error.location.end.line || error.location.start.column !== error.location.end.column || !isEmptyValue(error) || typeof getLastFragment(error.instancePath) === "undefined") {
        return error;
    }
    const lastKey = navigate(error.instancePath, parse, true);
    const locF = mappedIndexToLineCol(parse.source);
    try {
        const location = {
            start: locF(lastKey.start),
            end: locF(lastKey.end)
        };
        return {
            ...error,
            location,
            niceError: {
                ...error.niceError,
                location
            }
        };
    } catch (_e) {
        return error;
    }
}
function checkForTypeMismatch(error, parse, schema) {
    const rawVerbatimInput = getVerbatimInput(error);
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const goodType = (obj)=>{
        if (Array.isArray(obj)) {
            return "an array";
        }
        if (obj === null) {
            return "a null value";
        }
        return typeof obj;
    };
    if (errorKeyword(error) === "type" && rawVerbatimInput.length > 0) {
        const reindented = reindent(verbatimInput);
        const subject = reindented.indexOf("\n") === -1 ? `The value ${reindented} ` : `The value

${reindented}

`;
        const newError = {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema),
            error: [
                `${subject}is of type ${goodType(error.violatingObject.result)}.`, 
            ],
            info: {},
            location: error.niceError.location
        };
        addInstancePathInfo(newError, error.instancePath);
        addFileInfo(newError, error.source);
        return {
            ...error,
            niceError: newError
        };
    }
    return error;
}
function checkForBadBoolean(error, parse, _schema) {
    const schema = error.schema;
    if (!(typeof error.violatingObject.result === "string" && errorKeyword(error) === "type" && schemaType(schema) === "boolean")) {
        return error;
    }
    const strValue = error.violatingObject.result;
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const yesses = new Set("y|Y|yes|Yes|YES|true|True|TRUE|on|On|ON".split("|"));
    const nos = new Set("n|N|no|No|NO|false|False|FALSE|off|Off|OFF".split("|"));
    let fix;
    if (yesses.has(strValue)) {
        fix = true;
    } else if (nos.has(strValue)) {
        fix = false;
    } else {
        return error;
    }
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `Quarto uses YAML 1.2, which interprets booleans strictly.`;
    const suggestion2 = `Try using ${quotedStringColor(String(fix))} instead.`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-version-1.2"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadColon(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+:[^ ].*$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/:/g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadEquals(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ]) && !schemaPathMatches(error, [
        "object",
        "propertyNames",
        "string",
        "pattern"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+ *= *.+$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a colon and a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/ *= */g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function createErrorFragments(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    const verbatimInput = quotedStringColor(reindent(rawVerbatimInput));
    const pathFragments = error.instancePath.map((s)=>blue2(String(s)));
    return {
        location: locationString(error.location),
        fullPath: pathFragments.join(":"),
        key: pathFragments[pathFragments.length - 1],
        value: verbatimInput
    };
}
function schemaDefinedErrors(error, _parse, _schema) {
    const schema = error.schema;
    if (schema === true || schema === false) {
        return error;
    }
    if (schema.errorMessage === undefined) {
        return error;
    }
    if (typeof schema.errorMessage !== "string") {
        return error;
    }
    let result1 = schema.errorMessage;
    for (const [k, v] of Object.entries(createErrorFragments(error))){
        result1 = result1.replace("${" + k + "}", v);
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: result1
        }
    };
}
function checkForNearbyRequired(error, _parse, _schema) {
    const schema = error.schema;
    if (errorKeyword(error) !== "required") {
        return error;
    }
    const missingKeys = [];
    const errObj = error.violatingObject.result;
    const keys = Object.keys(errObj);
    schemaCall(schema, {
        object (s) {
            if (s.required === undefined) {
                throw new Error("Internal Error: required schema error without a required field");
            }
            for (const r of s.required){
                if (keys.indexOf(r) === -1) {
                    missingKeys.push(r);
                }
            }
        }
    }, (_)=>{
        throw new Error("Internal Error: required error on a non-object schema");
    });
    for (const missingKey of missingKeys){
        let bestCorrection;
        let bestDistance = Infinity;
        for (const correction of keys){
            const d = editDistance(correction, missingKey);
            if (d < bestDistance) {
                bestCorrection = [
                    correction
                ];
                bestDistance = d;
            } else if (d === bestDistance) {
                bestCorrection.push(correction);
                bestDistance = d;
            }
        }
        if (bestDistance > missingKey.length * 10 * 0.3) {
            continue;
        }
        const suggestions = bestCorrection.map((s)=>blue2(s));
        if (suggestions.length === 1) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} a typo of ${blue2(missingKey)}?`;
        } else if (suggestions.length === 2) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} or ${suggestions[1]} a typo of ${blue2(missingKey)}?`;
        } else {
            suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
            error.niceError.info[`did-you-mean-key`] = `Is one of ${suggestions.join(", ")} a typo of ${blue2(missingKey)}?`;
        }
    }
    return error;
}
function checkForNearbyCorrection(error, parse, _schema) {
    const schema = error.schema;
    const corrections = [];
    let errVal = "";
    let keyOrValue = "";
    const key = getBadKey(error);
    if (key) {
        errVal = key;
        corrections.push(...possibleSchemaKeys(schema));
        keyOrValue = "key";
    } else {
        const val = navigate(error.instancePath, parse);
        if (typeof val.result !== "string") {
            return error;
        }
        errVal = val.result;
        corrections.push(...possibleSchemaValues(schema));
        keyOrValue = "value";
    }
    if (corrections.length === 0) {
        return error;
    }
    let bestCorrection;
    let bestDistance = Infinity;
    for (const correction of corrections){
        const d = editDistance(correction, errVal);
        if (d < bestDistance) {
            bestCorrection = [
                correction
            ];
            bestDistance = d;
        } else if (d === bestDistance) {
            bestCorrection.push(correction);
            bestDistance = d;
        }
    }
    if (bestDistance > errVal.length * 10 * 0.3) {
        return error;
    }
    const suggestions = bestCorrection.map((s)=>blue2(s));
    if (suggestions.length === 1) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]}?`;
    } else if (suggestions.length === 2) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]} or ${suggestions[1]}?`;
    } else {
        suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions.join(", ")}?`;
    }
    return error;
}
function createSourceContext(src, location) {
    const startMapResult = src.map(location.start, true);
    const endMapResult = src.map(location.end, true);
    const locF = mappedIndexToLineCol(src);
    let sourceLocation;
    try {
        sourceLocation = {
            start: locF(location.start),
            end: locF(location.end)
        };
    } catch (_e) {
        sourceLocation = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    if (startMapResult === undefined || endMapResult === undefined) {
        throw new Error("Internal Error: createSourceContext called with bad location.");
    }
    if (startMapResult.originalString !== endMapResult.originalString) {
        throw new Error("Internal Error: don't know how to create source context across different source files");
    }
    const originalString = startMapResult.originalString;
    const nLines = lines(originalString.value).length;
    const { start , end ,  } = sourceLocation;
    const { prefixWidth , lines: formattedLines ,  } = formatLineRange(originalString.value, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
    const contextLines = [];
    let mustPrintEllipsis = true;
    for (const { lineNumber , content , rawLine  } of formattedLines){
        if (lineNumber < start.line || lineNumber > end.line) {
            if (rawLine.trim().length) {
                contextLines.push(content);
            }
        } else {
            if (lineNumber >= start.line + 2 && lineNumber <= end.line - 2) {
                if (mustPrintEllipsis) {
                    mustPrintEllipsis = false;
                    contextLines.push("...");
                }
            } else {
                const startColumn = lineNumber > start.line ? 0 : start.column;
                const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                contextLines.push(content);
                contextLines.push(" ".repeat(prefixWidth + startColumn - 1) + "~".repeat(endColumn - startColumn + 1));
            }
        }
    }
    return contextLines.join("\n");
}
function createLocalizedError(obj) {
    const { violatingObject , instancePath , schemaPath , source , message , schema ,  } = obj;
    const locF = mappedIndexToLineCol(source);
    let location;
    try {
        location = {
            start: locF(violatingObject.start),
            end: locF(violatingObject.end)
        };
    } catch (_e) {
        location = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    const mapResult = source.map(violatingObject.start);
    const fileName = mapResult ? mapResult.originalString.fileName : undefined;
    return {
        source: mappedString(source, [
            {
                start: violatingObject.start,
                end: violatingObject.end
            }
        ]),
        violatingObject: violatingObject,
        instancePath,
        schemaPath,
        schema,
        message,
        location: location,
        niceError: {
            heading: message,
            error: [],
            info: {},
            fileName,
            location: location,
            sourceContext: createSourceContext(violatingObject.source, {
                start: violatingObject.start,
                end: violatingObject.end
            })
        }
    };
}
class ValidationContext {
    instancePath;
    root;
    nodeStack;
    currentNode;
    constructor(){
        this.instancePath = [];
        this.currentNode = {
            edge: "#",
            errors: [],
            children: []
        };
        this.nodeStack = [
            this.currentNode
        ];
        this.root = this.currentNode;
    }
    error(value, schema, message) {
        this.currentNode.errors.push({
            value,
            schema,
            message,
            instancePath: this.instancePath.slice(),
            schemaPath: this.nodeStack.map((node)=>node.edge)
        });
    }
    pushSchema(schemaPath) {
        const newNode = {
            edge: schemaPath,
            errors: [],
            children: []
        };
        this.currentNode.children.push(newNode);
        this.currentNode = newNode;
        this.nodeStack.push(newNode);
    }
    popSchema(success) {
        this.nodeStack.pop();
        this.currentNode = this.nodeStack[this.nodeStack.length - 1];
        if (success) {
            this.currentNode.children.pop();
        }
        return success;
    }
    pushInstance(instance) {
        this.instancePath.push(instance);
    }
    popInstance() {
        this.instancePath.pop();
    }
    withSchemaPath(schemaPath, chunk) {
        this.pushSchema(schemaPath);
        return this.popSchema(chunk());
    }
    validate(schema, source, value, pruneErrors = true) {
        if (validateGeneric(value, schema, this)) {
            return [];
        }
        return this.collectErrors(schema, source, value, pruneErrors);
    }
    collectErrors(_schema, source, _value, pruneErrors = true) {
        const inner = (node)=>{
            const result1 = [];
            if (node.edge === "anyOf" && pruneErrors) {
                const innerResults = node.children.map(inner);
                const isRequiredError = (e)=>e.schemaPath.indexOf("required") === e.schemaPath.length - 1;
                const isPropertyNamesError = (e)=>e.schemaPath.indexOf("propertyNames") !== -1;
                if (innerResults.some((el)=>el.length && isRequiredError(el[0])) && innerResults.some((el)=>el.length && isPropertyNamesError(el[0]))) {
                    return innerResults.filter((r)=>{
                        return r.length && r[0].schemaPath.slice(-1)[0] === "required";
                    })[0];
                }
                const errorTypeQuality = (e)=>{
                    const t = e.schemaPath.slice().reverse();
                    if (e.schemaPath.indexOf("propertyNames") !== -1) {
                        return 10;
                    }
                    if (t[0] === "required") {
                        return 0;
                    }
                    if (t[0] === "type") {
                        if (t[1] === "null") {
                            return 10;
                        }
                        return 1;
                    }
                    return 1;
                };
                const errorComparator = (a, b)=>{
                    for(let i1 = 0; i1 < a.length; ++i1){
                        if (a[i1] < b[i1]) {
                            return -1;
                        }
                        if (a[i1] > b[i1]) {
                            return 1;
                        }
                    }
                    return 0;
                };
                let bestResults = [];
                let bestError = [
                    Infinity,
                    Infinity
                ];
                for (const resultGroup of innerResults){
                    let maxQuality = -Infinity;
                    let totalSpan = 0;
                    for (const result11 of resultGroup){
                        totalSpan += result11.value.end - result11.value.start;
                        maxQuality = Math.max(maxQuality, errorTypeQuality(result11));
                    }
                    const thisError = [
                        maxQuality,
                        totalSpan
                    ];
                    if (errorComparator(thisError, bestError) === -1) {
                        bestError = thisError;
                        bestResults = resultGroup;
                    }
                }
                return bestResults;
            } else {
                result1.push(...node.errors);
                for (const child of node.children){
                    result1.push(...inner(child));
                }
                return result1;
            }
        };
        const errors = inner(this.root);
        const result1 = errors.map((validationError)=>createLocalizedError({
                violatingObject: validationError.value,
                instancePath: validationError.instancePath,
                schemaPath: validationError.schemaPath,
                schema: validationError.schema,
                message: validationError.message,
                source
            }));
        return result1;
    }
}
function validateGeneric(value, s, context) {
    s = resolveSchema(s);
    const st = schemaType(s);
    return context.withSchemaPath(st, ()=>schemaCall(s, {
            "false": (schema)=>{
                context.error(value, schema, "false");
                return false;
            },
            "true": (_)=>true,
            "any": (schema)=>validateAny(value, schema, context),
            "boolean": (schema)=>validateBoolean(value, schema, context),
            "number": (schema)=>validateNumber(value, schema, context),
            "string": (schema)=>validateString(value, schema, context),
            "null": (schema)=>validateNull(value, schema, context),
            "enum": (schema)=>validateEnum(value, schema, context),
            "anyOf": (schema)=>validateAnyOf(value, schema, context),
            "allOf": (schema)=>validateAllOf(value, schema, context),
            "array": (schema)=>validateArray(value, schema, context),
            "object": (schema)=>validateObject(value, schema, context),
            "ref": (schema)=>validateGeneric(value, resolveSchema(schema), context)
        }));
}
function typeIsValid(value, schema, context, valid) {
    if (!valid) {
        return context.withSchemaPath("type", ()=>{
            context.error(value, schema, "type mismatch");
            return false;
        });
    }
    return valid;
}
function validateAny(_value, _schema, _context) {
    return true;
}
function validateBoolean(value, schema, context) {
    return typeIsValid(value, schema, context, typeof value.result === "boolean");
}
function validateNumber(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "number")) {
        return false;
    }
    let result1 = true;
    if (schema.minimum !== undefined) {
        result1 = context.withSchemaPath("minimum", ()=>{
            const v = value.result;
            if (!(v >= schema.minimum)) {
                context.error(value, schema, `value ${value.result} is less than required minimum ${schema.minimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.maximum !== undefined) {
        result1 = context.withSchemaPath("maximum", ()=>{
            const v = value.result;
            if (!(v <= schema.maximum)) {
                context.error(value, schema, `value ${value.result} is greater than required maximum ${schema.maximum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMinimum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMinimum", ()=>{
            const v = value.result;
            if (!(v > schema.exclusiveMinimum)) {
                context.error(value, schema, `value ${value.result} is less than or equal to required (exclusive) minimum ${schema.exclusiveMinimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMaximum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMaximum", ()=>{
            const v = value.result;
            if (!(v < schema.exclusiveMaximum)) {
                context.error(value, schema, `value ${value.result} is greater than or equal to required (exclusive) maximum ${schema.exclusiveMaximum}`);
                return false;
            }
            return true;
        });
    }
    return result1;
}
function validateString(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "string")) {
        return false;
    }
    if (schema.pattern !== undefined) {
        if (schema.compiledPattern === undefined) {
            schema.compiledPattern = new RegExp(schema.pattern);
        }
        if (!value.result.match(schema.compiledPattern)) {
            return context.withSchemaPath("pattern", ()=>{
                context.error(value, schema, `value doesn't match pattern`);
                return false;
            });
        }
    }
    return true;
}
function validateNull(value, schema, context) {
    if (!typeIsValid(value, schema, context, value.result === null)) {
        return false;
    }
    return true;
}
function validateEnum(value, schema, context) {
    for (const enumValue of schema["enum"]){
        if (enumValue === value.result) {
            return true;
        }
    }
    context.error(value, schema, `must match one of the values`);
    return false;
}
function validateAnyOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.anyOf.length; ++i1){
        const subSchema = schema.anyOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas > 0;
}
function validateAllOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.allOf.length; ++i1){
        const subSchema = schema.allOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas === schema.allOf.length;
}
function validateArray(value, schema, context) {
    let result1 = true;
    if (!typeIsValid(value, schema, context, Array.isArray(value.result))) {
        return false;
    }
    const length = value.result.length;
    if (schema.minItems !== undefined && length < schema.minItems) {
        context.withSchemaPath("minItems", ()=>{
            context.error(value, schema, `array should have at least ${schema.minItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.maxItems !== undefined && length > schema.maxItems) {
        context.withSchemaPath("maxItems", ()=>{
            context.error(value, schema, `array should have at most ${schema.maxItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.items !== undefined) {
        result1 = context.withSchemaPath("items", ()=>{
            let result1 = true;
            for(let i1 = 0; i1 < value.components.length; ++i1){
                context.pushInstance(i1);
                result1 = validateGeneric(value.components[i1], schema.items, context) && result1;
                context.popInstance();
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validateObject(value, schema, context) {
    const isObject = typeof value.result === "object" && !Array.isArray(value.result) && value.result !== null;
    if (!typeIsValid(value, schema, context, isObject)) {
        return false;
    }
    let result1 = true;
    const ownProperties = new Set(Object.getOwnPropertyNames(value.result));
    const objResult = value.result;
    const locate = (key, keyOrValue = "value")=>{
        for(let i1 = 0; i1 < value.components.length; i1 += 2){
            if (String(value.components[i1].result) === key) {
                if (keyOrValue === "value") {
                    return value.components[i1 + 1];
                } else {
                    return value.components[i1];
                }
            }
        }
        throw new Error(`Internal Error, couldn't locate key ${key}`);
    };
    const inspectedProps = new Set();
    if (schema.closed) {
        result1 = context.withSchemaPath("closed", ()=>{
            if (schema.properties === undefined) {
                throw new Error("Internal Error: closed schemas need properties");
            }
            let innerResult = true;
            for (const key of ownProperties){
                if (!schema.properties[key]) {
                    context.error(locate(key, "key"), schema, `object has invalid field ${key}`);
                    innerResult = false;
                }
            }
            return innerResult;
        }) && result1;
    }
    if (schema.properties !== undefined) {
        result1 = context.withSchemaPath("properties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.properties)){
                if (ownProperties.has(key)) {
                    inspectedProps.add(key);
                    context.pushInstance(key);
                    result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(key), subSchema, context)) && result1;
                    context.popInstance();
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.patternProperties !== undefined) {
        result1 = context.withSchemaPath("patternProperties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.patternProperties)){
                if (schema.compiledPatterns === undefined) {
                    schema.compiledPatterns = {};
                }
                if (schema.compiledPatterns[key] === undefined) {
                    schema.compiledPatterns[key] = new RegExp(key);
                }
                const regexp = schema.compiledPatterns[key];
                for (const [objectKey, _val] of Object.entries(objResult)){
                    if (objectKey.match(regexp)) {
                        inspectedProps.add(objectKey);
                        context.pushInstance(objectKey);
                        result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(objectKey), subSchema, context)) && result1;
                        context.popInstance();
                    }
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.additionalProperties !== undefined) {
        result1 = context.withSchemaPath("additionalProperties", ()=>{
            return Object.keys(objResult).filter((objectKey)=>!inspectedProps.has(objectKey)).every((objectKey)=>validateGeneric(locate(objectKey), schema.additionalProperties, context));
        }) && result1;
    }
    if (schema.propertyNames !== undefined) {
        result1 = context.withSchemaPath("propertyNames", ()=>{
            return Array.from(ownProperties).every((key)=>validateGeneric(locate(key, "key"), schema.propertyNames, context));
        }) && result1;
    }
    if (schema.required !== undefined) {
        result1 = context.withSchemaPath("required", ()=>{
            let result1 = true;
            for (const reqKey of schema.required){
                if (!ownProperties.has(reqKey)) {
                    context.error(value, schema, `object is missing required property ${reqKey}`);
                    result1 = false;
                }
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validate(value, schema, source, pruneErrors = true) {
    const context = new ValidationContext();
    return context.validate(schema, source, value, pruneErrors);
}
class YAMLSchema {
    schema;
    errorHandlers;
    constructor(schema){
        this.errorHandlers = [];
        this.schema = schema;
    }
    addHandler(handler) {
        this.errorHandlers.push(handler);
    }
    transformErrors(annotation, errors) {
        return errors.map((error)=>{
            for (const handler of this.errorHandlers){
                const localError = handler(error, annotation, this.schema);
                if (localError === null) {
                    return null;
                }
                error = localError;
            }
            return error;
        }).filter((error)=>error !== null);
    }
    async validateParse(src, annotation, pruneErrors = true) {
        const validationErrors = validate(annotation, this.schema, src, pruneErrors);
        if (validationErrors.length) {
            const localizedErrors = this.transformErrors(annotation, validationErrors);
            return {
                result: annotation.result,
                errors: localizedErrors
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result1, _src, message, error, log) {
        if (result1.errors.length) {
            if (message.length) {
                error(message);
            }
            for (const err of result1.errors){
                log(err.niceError);
            }
        }
        return result1;
    }
    async validateParseWithErrors(src, annotation, message, error, log) {
        const result1 = await this.validateParse(src, annotation);
        this.reportErrorsInSource(result1, src, message, error, log);
        return result1;
    }
}
const yamlValidators = {};
function getSchemaName(schema) {
    if (schema === true || schema === false) {
        throw new Error("Expected schema to be named");
    }
    let schemaName = schema["$id"];
    if (schemaName !== undefined) {
        return schemaName;
    }
    if (schemaType(schema) === "ref") {
        schemaName = schema["$ref"];
    }
    if (schemaName !== undefined) {
        return schemaName;
    }
    throw new Error("Expected schema to be named");
}
function getValidator(schema) {
    const schemaName = getSchemaName(schema);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema);
    yamlValidators[schemaName] = validator;
    setDefaultErrorHandlers(validator);
    return validator;
}
async function withValidator(schema, fun) {
    let result1;
    let error;
    try {
        const validator = getValidator(schema);
        result1 = await fun(validator);
    } catch (e) {
        error = e;
    }
    if (error !== undefined) {
        throw error;
    }
    return result1;
}
function addValidatorErrorHandler(schema, handler) {
    return withValidator(schema, async (validator)=>{
        validator.addHandler(handler);
    });
}
function memoize(f, keyMemoizer) {
    const memo = {};
    const inner = (...args)=>{
        const key = keyMemoizer(...args);
        const v = memo[key];
        if (v !== undefined) {
            return v;
        }
        memo[key] = f(...args);
        return memo[key];
    };
    return inner;
}
const regExpEscapeChars1 = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|", 
];
const rangeEscapeChars1 = [
    "-",
    "\\",
    "]"
];
function globToRegExp1(glob, { extended =true , globstar: globstarOption = true , caseInsensitive =false  } = {}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep = "/+";
    const sepMaybe = "/*";
    const seps = [
        "/"
    ];
    const globstar = "(?:[^/]*(?:/|$)+)*";
    const wildcard = "[^/]*";
    const escapePrefix = "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars1 : regExpEscapeChars1;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] == "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] == "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] == ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i1 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type == "!") {
                    segment += wildcard;
                } else if (type != "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "+" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "@" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "?") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] == "!" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "*") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] == "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars1.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars1.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? sep : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
const _resources = {};
function setYamlIntelligenceResources(resources) {
    for (const [key, value] of Object.entries(resources)){
        _resources[key] = value;
    }
}
function exportYamlIntelligenceResources(pretty = false) {
    if (pretty) {
        return JSON.stringify(_resources, null, 2);
    } else {
        return JSON.stringify(_resources);
    }
}
function getYamlIntelligenceResource(filename) {
    if (_resources[filename] === undefined) {
        throw new Error(`Internal Error: getYamlIntelligenceResource called with missing resource ${filename}`);
    }
    return _resources[filename];
}
function expandResourceGlob(glob) {
    return Object.keys(_resources).filter((key)=>key.match(globToRegExp1(glob))).map((key)=>[
            key,
            getYamlIntelligenceResource(key)
        ]);
}
function fromEntries(iterable) {
    return [
        ...iterable
    ].reduce((obj, [key, val])=>{
        obj[key] = val;
        return obj;
    }, {});
}
let _parser;
function setTreeSitter(parser) {
    _parser = parser;
}
function getTreeSitterSync() {
    if (_parser === undefined) {
        throw new Error("tree-sitter uninitialized");
    }
    return _parser;
}
function isNothing(subject) {
    return typeof subject === "undefined" || subject === null;
}
function isObject1(subject) {
    return typeof subject === "object" && subject !== null;
}
function toArray(sequence) {
    if (Array.isArray(sequence)) return sequence;
    else if (isNothing(sequence)) return [];
    return [
        sequence
    ];
}
function extend(target, source) {
    var index, length, key, sourceKeys;
    if (source) {
        sourceKeys = Object.keys(source);
        for(index = 0, length = sourceKeys.length; index < length; index += 1){
            key = sourceKeys[index];
            target[key] = source[key];
        }
    }
    return target;
}
function repeat1(string, count) {
    var result1 = "", cycle;
    for(cycle = 0; cycle < count; cycle += 1){
        result1 += string;
    }
    return result1;
}
function isNegativeZero1(number) {
    return number === 0 && Number.NEGATIVE_INFINITY === 1 / number;
}
var isNothing_1 = isNothing;
var isObject_1 = isObject1;
var toArray_1 = toArray;
var repeat_1 = repeat1;
var isNegativeZero_1 = isNegativeZero1;
var extend_1 = extend;
var common = {
    isNothing: isNothing_1,
    isObject: isObject_1,
    toArray: toArray_1,
    repeat: repeat_1,
    isNegativeZero: isNegativeZero_1,
    extend: extend_1
};
function formatError(exception2, compact) {
    var where = "", message = exception2.reason || "(unknown reason)";
    if (!exception2.mark) return message;
    if (exception2.mark.name) {
        where += 'in "' + exception2.mark.name + '" ';
    }
    where += "(" + (exception2.mark.line + 1) + ":" + (exception2.mark.column + 1) + ")";
    if (!compact && exception2.mark.snippet) {
        where += "\n\n" + exception2.mark.snippet;
    }
    return message + " " + where;
}
function YAMLException$1(reason, mark) {
    Error.call(this);
    this.name = "YAMLException";
    this.reason = reason;
    this.mark = mark;
    this.message = formatError(this, false);
    if (Error.captureStackTrace) {
        Error.captureStackTrace(this, this.constructor);
    } else {
        this.stack = new Error().stack || "";
    }
}
YAMLException$1.prototype = Object.create(Error.prototype);
YAMLException$1.prototype.constructor = YAMLException$1;
YAMLException$1.prototype.toString = function toString(compact) {
    return this.name + ": " + formatError(this, compact);
};
var exception = YAMLException$1;
function getLine(buffer, lineStart, lineEnd, position, maxLineLength) {
    var head = "";
    var tail = "";
    var maxHalfLength = Math.floor(maxLineLength / 2) - 1;
    if (position - lineStart > maxHalfLength) {
        head = " ... ";
        lineStart = position - maxHalfLength + head.length;
    }
    if (lineEnd - position > maxHalfLength) {
        tail = " ...";
        lineEnd = position + maxHalfLength - tail.length;
    }
    return {
        str: head + buffer.slice(lineStart, lineEnd).replace(/\t/g, "\u2192") + tail,
        pos: position - lineStart + head.length
    };
}
function padStart(string, max) {
    return common.repeat(" ", max - string.length) + string;
}
function makeSnippet(mark, options) {
    options = Object.create(options || null);
    if (!mark.buffer) return null;
    if (!options.maxLength) options.maxLength = 79;
    if (typeof options.indent !== "number") options.indent = 1;
    if (typeof options.linesBefore !== "number") options.linesBefore = 3;
    if (typeof options.linesAfter !== "number") options.linesAfter = 2;
    var re = /\r?\n|\r|\0/g;
    var lineStarts = [
        0
    ];
    var lineEnds = [];
    var match;
    var foundLineNo = -1;
    while(match = re.exec(mark.buffer)){
        lineEnds.push(match.index);
        lineStarts.push(match.index + match[0].length);
        if (mark.position <= match.index && foundLineNo < 0) {
            foundLineNo = lineStarts.length - 2;
        }
    }
    if (foundLineNo < 0) foundLineNo = lineStarts.length - 1;
    var result1 = "", i1, line;
    var lineNoLength = Math.min(mark.line + options.linesAfter, lineEnds.length).toString().length;
    var maxLineLength = options.maxLength - (options.indent + lineNoLength + 3);
    for(i1 = 1; i1 <= options.linesBefore; i1++){
        if (foundLineNo - i1 < 0) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo - i1], lineEnds[foundLineNo - i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo - i1]), maxLineLength);
        result1 = common.repeat(" ", options.indent) + padStart((mark.line - i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n" + result1;
    }
    line = getLine(mark.buffer, lineStarts[foundLineNo], lineEnds[foundLineNo], mark.position, maxLineLength);
    result1 += common.repeat(" ", options.indent) + padStart((mark.line + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    result1 += common.repeat("-", options.indent + lineNoLength + 3 + line.pos) + "^\n";
    for(i1 = 1; i1 <= options.linesAfter; i1++){
        if (foundLineNo + i1 >= lineEnds.length) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo + i1], lineEnds[foundLineNo + i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo + i1]), maxLineLength);
        result1 += common.repeat(" ", options.indent) + padStart((mark.line + i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    }
    return result1.replace(/\n$/, "");
}
var snippet = makeSnippet;
var TYPE_CONSTRUCTOR_OPTIONS = [
    "kind",
    "multi",
    "resolve",
    "construct",
    "instanceOf",
    "predicate",
    "represent",
    "representName",
    "defaultStyle",
    "styleAliases", 
];
var YAML_NODE_KINDS = [
    "scalar",
    "sequence",
    "mapping"
];
function compileStyleAliases(map2) {
    var result1 = {};
    if (map2 !== null) {
        Object.keys(map2).forEach(function(style) {
            map2[style].forEach(function(alias) {
                result1[String(alias)] = style;
            });
        });
    }
    return result1;
}
function Type$1(tag, options) {
    options = options || {};
    Object.keys(options).forEach(function(name) {
        if (TYPE_CONSTRUCTOR_OPTIONS.indexOf(name) === -1) {
            throw new exception('Unknown option "' + name + '" is met in definition of "' + tag + '" YAML type.');
        }
    });
    this.options = options;
    this.tag = tag;
    this.kind = options["kind"] || null;
    this.resolve = options["resolve"] || function() {
        return true;
    };
    this.construct = options["construct"] || function(data) {
        return data;
    };
    this.instanceOf = options["instanceOf"] || null;
    this.predicate = options["predicate"] || null;
    this.represent = options["represent"] || null;
    this.representName = options["representName"] || null;
    this.defaultStyle = options["defaultStyle"] || null;
    this.multi = options["multi"] || false;
    this.styleAliases = compileStyleAliases(options["styleAliases"] || null);
    if (YAML_NODE_KINDS.indexOf(this.kind) === -1) {
        throw new exception('Unknown kind "' + this.kind + '" is specified for "' + tag + '" YAML type.');
    }
}
var type = Type$1;
function compileList1(schema2, name) {
    var result1 = [];
    schema2[name].forEach(function(currentType) {
        var newIndex = result1.length;
        result1.forEach(function(previousType, previousIndex) {
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind && previousType.multi === currentType.multi) {
                newIndex = previousIndex;
            }
        });
        result1[newIndex] = currentType;
    });
    return result1;
}
function compileMap1() {
    var result1 = {
        scalar: {},
        sequence: {},
        mapping: {},
        fallback: {},
        multi: {
            scalar: [],
            sequence: [],
            mapping: [],
            fallback: []
        }
    }, index, length;
    function collectType(type2) {
        if (type2.multi) {
            result1.multi[type2.kind].push(type2);
            result1.multi["fallback"].push(type2);
        } else {
            result1[type2.kind][type2.tag] = result1["fallback"][type2.tag] = type2;
        }
    }
    for(index = 0, length = arguments.length; index < length; index += 1){
        arguments[index].forEach(collectType);
    }
    return result1;
}
function Schema$1(definition) {
    return this.extend(definition);
}
Schema$1.prototype.extend = function extend2(definition) {
    var implicit = [];
    var explicit = [];
    if (definition instanceof type) {
        explicit.push(definition);
    } else if (Array.isArray(definition)) {
        explicit = explicit.concat(definition);
    } else if (definition && (Array.isArray(definition.implicit) || Array.isArray(definition.explicit))) {
        if (definition.implicit) implicit = implicit.concat(definition.implicit);
        if (definition.explicit) explicit = explicit.concat(definition.explicit);
    } else {
        throw new exception("Schema.extend argument should be a Type, [ Type ], or a schema definition ({ implicit: [...], explicit: [...] })");
    }
    implicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
        if (type$1.loadKind && type$1.loadKind !== "scalar") {
            throw new exception("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
        }
        if (type$1.multi) {
            throw new exception("There is a multi type in the implicit list of a schema. Multi tags can only be listed as explicit.");
        }
    });
    explicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
    });
    var result1 = Object.create(Schema$1.prototype);
    result1.implicit = (this.implicit || []).concat(implicit);
    result1.explicit = (this.explicit || []).concat(explicit);
    result1.compiledImplicit = compileList1(result1, "implicit");
    result1.compiledExplicit = compileList1(result1, "explicit");
    result1.compiledTypeMap = compileMap1(result1.compiledImplicit, result1.compiledExplicit);
    return result1;
};
var schema = Schema$1;
var str1 = new type("tag:yaml.org,2002:str", {
    kind: "scalar",
    construct: function(data) {
        return data !== null ? data : "";
    }
});
var seq1 = new type("tag:yaml.org,2002:seq", {
    kind: "sequence",
    construct: function(data) {
        return data !== null ? data : [];
    }
});
var map1 = new type("tag:yaml.org,2002:map", {
    kind: "mapping",
    construct: function(data) {
        return data !== null ? data : {};
    }
});
var failsafe1 = new schema({
    explicit: [
        str1,
        seq1,
        map1
    ]
});
function resolveYamlNull1(data) {
    if (data === null) return true;
    var max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull1() {
    return null;
}
function isNull1(object) {
    return object === null;
}
var _null = new type("tag:yaml.org,2002:null", {
    kind: "scalar",
    resolve: resolveYamlNull1,
    construct: constructYamlNull1,
    predicate: isNull1,
    represent: {
        canonical: function() {
            return "~";
        },
        lowercase: function() {
            return "null";
        },
        uppercase: function() {
            return "NULL";
        },
        camelcase: function() {
            return "Null";
        },
        empty: function() {
            return "";
        }
    },
    defaultStyle: "lowercase"
});
function resolveYamlBoolean1(data) {
    if (data === null) return false;
    var max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean1(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
function isBoolean1(object) {
    return Object.prototype.toString.call(object) === "[object Boolean]";
}
var bool1 = new type("tag:yaml.org,2002:bool", {
    kind: "scalar",
    resolve: resolveYamlBoolean1,
    construct: constructYamlBoolean1,
    predicate: isBoolean1,
    represent: {
        lowercase: function(object) {
            return object ? "true" : "false";
        },
        uppercase: function(object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase: function(object) {
            return object ? "True" : "False";
        }
    },
    defaultStyle: "lowercase"
});
function isHexCode1(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode1(c) {
    return 48 <= c && c <= 55;
}
function isDecCode1(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger1(data) {
    if (data === null) return false;
    var max = data.length, index = 0, hasDigits = false, ch;
    if (!max) return false;
    ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "o") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isOctCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (!isDecCode1(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    return true;
}
function constructYamlInteger1(data) {
    var value = data, sign = 1, ch;
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value.slice(2), 16);
        if (value[1] === "o") return sign * parseInt(value.slice(2), 8);
    }
    return sign * parseInt(value, 10);
}
function isInteger1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !common.isNegativeZero(object);
}
var __int1 = new type("tag:yaml.org,2002:int", {
    kind: "scalar",
    resolve: resolveYamlInteger1,
    construct: constructYamlInteger1,
    predicate: isInteger1,
    represent: {
        binary: function(obj) {
            return obj >= 0 ? "0b" + obj.toString(2) : "-0b" + obj.toString(2).slice(1);
        },
        octal: function(obj) {
            return obj >= 0 ? "0o" + obj.toString(8) : "-0o" + obj.toString(8).slice(1);
        },
        decimal: function(obj) {
            return obj.toString(10);
        },
        hexadecimal: function(obj) {
            return obj >= 0 ? "0x" + obj.toString(16).toUpperCase() : "-0x" + obj.toString(16).toUpperCase().slice(1);
        }
    },
    defaultStyle: "decimal",
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        octal: [
            8,
            "oct"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ]
    }
});
var YAML_FLOAT_PATTERN1 = new RegExp("^(?:[-+]?(?:[0-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?|[-+]?\\.(?:inf|Inf|INF)|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat1(data) {
    if (data === null) return false;
    if (!YAML_FLOAT_PATTERN1.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat1(data) {
    var value, sign;
    value = data.replace(/_/g, "").toLowerCase();
    sign = value[0] === "-" ? -1 : 1;
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    } else if (value === ".nan") {
        return NaN;
    }
    return sign * parseFloat(value, 10);
}
var SCIENTIFIC_WITHOUT_DOT1 = /^[-+]?[0-9]+e/;
function representYamlFloat1(object, style) {
    var res;
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (common.isNegativeZero(object)) {
        return "-0.0";
    }
    res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT1.test(res) ? res.replace("e", ".e") : res;
}
function isFloat1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || common.isNegativeZero(object));
}
var __float1 = new type("tag:yaml.org,2002:float", {
    kind: "scalar",
    resolve: resolveYamlFloat1,
    construct: constructYamlFloat1,
    predicate: isFloat1,
    represent: representYamlFloat1,
    defaultStyle: "lowercase"
});
var json1 = failsafe1.extend({
    implicit: [
        _null,
        bool1,
        __int1,
        __float1
    ]
});
var core1 = json1;
var YAML_DATE_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9])-([0-9][0-9])$");
var YAML_TIMESTAMP_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9]?)-([0-9][0-9]?)(?:[Tt]|[ \\t]+)([0-9][0-9]?):([0-9][0-9]):([0-9][0-9])(?:\\.([0-9]*))?(?:[ \\t]*(Z|([-+])([0-9][0-9]?)(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp1(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP1.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP1.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp1(data) {
    var match, year, month, day, hour, minute, second, fraction = 0, delta = null, tz_hour, tz_minute, date;
    match = YAML_DATE_REGEXP1.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP1.exec(data);
    if (match === null) throw new Error("Date resolve error");
    year = +match[1];
    month = +match[2] - 1;
    day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    hour = +match[4];
    minute = +match[5];
    second = +match[6];
    if (match[7]) {
        fraction = match[7].slice(0, 3);
        while(fraction.length < 3){
            fraction += "0";
        }
        fraction = +fraction;
    }
    if (match[9]) {
        tz_hour = +match[10];
        tz_minute = +(match[11] || 0);
        delta = (tz_hour * 60 + tz_minute) * 6e4;
        if (match[9] === "-") delta = -delta;
    }
    date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp1(object) {
    return object.toISOString();
}
var timestamp1 = new type("tag:yaml.org,2002:timestamp", {
    kind: "scalar",
    resolve: resolveYamlTimestamp1,
    construct: constructYamlTimestamp1,
    instanceOf: Date,
    represent: representYamlTimestamp1
});
function resolveYamlMerge1(data) {
    return data === "<<" || data === null;
}
var merge1 = new type("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge1
});
var BASE64_MAP1 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary1(data) {
    if (data === null) return false;
    var code, idx, bitlen = 0, max = data.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        code = map2.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary1(data) {
    var idx, tailbits, input = data.replace(/[\r\n=]/g, ""), max = input.length, map2 = BASE64_MAP1, bits = 0, result1 = [];
    for(idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 255);
            result1.push(bits >> 8 & 255);
            result1.push(bits & 255);
        }
        bits = bits << 6 | map2.indexOf(input.charAt(idx));
    }
    tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 255);
        result1.push(bits >> 8 & 255);
        result1.push(bits & 255);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 255);
        result1.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 255);
    }
    return new Uint8Array(result1);
}
function representYamlBinary1(object) {
    var result1 = "", bits = 0, idx, tail, max = object.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map2[bits >> 18 & 63];
            result1 += map2[bits >> 12 & 63];
            result1 += map2[bits >> 6 & 63];
            result1 += map2[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    tail = max % 3;
    if (tail === 0) {
        result1 += map2[bits >> 18 & 63];
        result1 += map2[bits >> 12 & 63];
        result1 += map2[bits >> 6 & 63];
        result1 += map2[bits & 63];
    } else if (tail === 2) {
        result1 += map2[bits >> 10 & 63];
        result1 += map2[bits >> 4 & 63];
        result1 += map2[bits << 2 & 63];
        result1 += map2[64];
    } else if (tail === 1) {
        result1 += map2[bits >> 2 & 63];
        result1 += map2[bits << 4 & 63];
        result1 += map2[64];
        result1 += map2[64];
    }
    return result1;
}
function isBinary1(obj) {
    return Object.prototype.toString.call(obj) === "[object Uint8Array]";
}
var binary1 = new type("tag:yaml.org,2002:binary", {
    kind: "scalar",
    resolve: resolveYamlBinary1,
    construct: constructYamlBinary1,
    predicate: isBinary1,
    represent: representYamlBinary1
});
var _hasOwnProperty$3 = Object.prototype.hasOwnProperty;
var _toString$2 = Object.prototype.toString;
function resolveYamlOmap1(data) {
    if (data === null) return true;
    var objectKeys = [], index, length, pair, pairKey, pairHasKey, object = data;
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        pairHasKey = false;
        if (_toString$2.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (_hasOwnProperty$3.call(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap1(data) {
    return data !== null ? data : [];
}
var omap1 = new type("tag:yaml.org,2002:omap", {
    kind: "sequence",
    resolve: resolveYamlOmap1,
    construct: constructYamlOmap1
});
var _toString$1 = Object.prototype.toString;
function resolveYamlPairs1(data) {
    if (data === null) return true;
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        if (_toString$1.call(pair) !== "[object Object]") return false;
        keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs1(data) {
    if (data === null) return [];
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
var pairs1 = new type("tag:yaml.org,2002:pairs", {
    kind: "sequence",
    resolve: resolveYamlPairs1,
    construct: constructYamlPairs1
});
var _hasOwnProperty$2 = Object.prototype.hasOwnProperty;
function resolveYamlSet1(data) {
    if (data === null) return true;
    var key, object = data;
    for(key in object){
        if (_hasOwnProperty$2.call(object, key)) {
            if (object[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet1(data) {
    return data !== null ? data : {};
}
var set1 = new type("tag:yaml.org,2002:set", {
    kind: "mapping",
    resolve: resolveYamlSet1,
    construct: constructYamlSet1
});
var _default = core1.extend({
    implicit: [
        timestamp1,
        merge1
    ],
    explicit: [
        binary1,
        omap1,
        pairs1,
        set1
    ]
});
var _hasOwnProperty$1 = Object.prototype.hasOwnProperty;
var CONTEXT_FLOW_IN = 1;
var CONTEXT_FLOW_OUT = 2;
var CONTEXT_BLOCK_IN1 = 3;
var CONTEXT_BLOCK_OUT1 = 4;
var CHOMPING_CLIP = 1;
var CHOMPING_STRIP1 = 2;
var CHOMPING_KEEP1 = 3;
var PATTERN_NON_PRINTABLE1 = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
var PATTERN_NON_ASCII_LINE_BREAKS1 = /[\x85\u2028\u2029]/;
var PATTERN_FLOW_INDICATORS1 = /[,\[\]\{\}]/;
var PATTERN_TAG_HANDLE1 = /^(?:!|!!|![a-z\-]+!)$/i;
var PATTERN_TAG_URI1 = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class1(obj) {
    return Object.prototype.toString.call(obj);
}
function is_EOL(c) {
    return c === 10 || c === 13;
}
function is_WHITE_SPACE(c) {
    return c === 9 || c === 32;
}
function is_WS_OR_EOL(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function is_FLOW_INDICATOR(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode1(c) {
    var lc;
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen1(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode1(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence1(c) {
    return c === 48 ? "\0" : c === 97 ? "\x07" : c === 98 ? "\b" : c === 116 ? "	" : c === 9 ? "	" : c === 110 ? "\n" : c === 118 ? "\v" : c === 102 ? "\f" : c === 114 ? "\r" : c === 101 ? "" : c === 32 ? " " : c === 34 ? '"' : c === 47 ? "/" : c === 92 ? "\\" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint1(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
var simpleEscapeCheck1 = new Array(256);
var simpleEscapeMap1 = new Array(256);
for(var i3 = 0; i3 < 256; i3++){
    simpleEscapeCheck1[i3] = simpleEscapeSequence1(i3) ? 1 : 0;
    simpleEscapeMap1[i3] = simpleEscapeSequence1(i3);
}
function State$1(input, options) {
    this.input = input;
    this.filename = options["filename"] || null;
    this.schema = options["schema"] || _default;
    this.onWarning = options["onWarning"] || null;
    this.legacy = options["legacy"] || false;
    this.json = options["json"] || false;
    this.listener = options["listener"] || null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.typeMap = this.schema.compiledTypeMap;
    this.length = input.length;
    this.position = 0;
    this.line = 0;
    this.lineStart = 0;
    this.lineIndent = 0;
    this.firstTabInLine = -1;
    this.documents = [];
}
function generateError1(state, message) {
    var mark = {
        name: state.filename,
        buffer: state.input.slice(0, -1),
        position: state.position,
        line: state.line,
        column: state.position - state.lineStart
    };
    mark.snippet = snippet(mark);
    return new exception(message, mark);
}
function throwError1(state, message) {
    throw generateError1(state, message);
}
function throwWarning1(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError1(state, message));
    }
}
var directiveHandlers1 = {
    YAML: function handleYamlDirective(state, name, args) {
        var match, major, minor;
        if (state.version !== null) {
            throwError1(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            throwError1(state, "YAML directive accepts exactly one argument");
        }
        match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            throwError1(state, "ill-formed argument of the YAML directive");
        }
        major = parseInt(match[1], 10);
        minor = parseInt(match[2], 10);
        if (major !== 1) {
            throwError1(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            throwWarning1(state, "unsupported YAML version of the document");
        }
    },
    TAG: function handleTagDirective(state, name, args) {
        var handle, prefix;
        if (args.length !== 2) {
            throwError1(state, "TAG directive accepts exactly two arguments");
        }
        handle = args[0];
        prefix = args[1];
        if (!PATTERN_TAG_HANDLE1.test(handle)) {
            throwError1(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (_hasOwnProperty$1.call(state.tagMap, handle)) {
            throwError1(state, 'there is a previously declared suffix for "' + handle + '" tag handle');
        }
        if (!PATTERN_TAG_URI1.test(prefix)) {
            throwError1(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        try {
            prefix = decodeURIComponent(prefix);
        } catch (err) {
            throwError1(state, "tag prefix is malformed: " + prefix);
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment1(state, start, end, checkJson) {
    var _position, _length, _character, _result;
    if (start < end) {
        _result = state.input.slice(start, end);
        if (checkJson) {
            for(_position = 0, _length = _result.length; _position < _length; _position += 1){
                _character = _result.charCodeAt(_position);
                if (!(_character === 9 || 32 <= _character && _character <= 1114111)) {
                    throwError1(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE1.test(_result)) {
            throwError1(state, "the stream contains non-printable characters");
        }
        state.result += _result;
    }
}
function mergeMappings1(state, destination, source, overridableKeys) {
    var sourceKeys, key, index, quantity;
    if (!common.isObject(source)) {
        throwError1(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    sourceKeys = Object.keys(source);
    for(index = 0, quantity = sourceKeys.length; index < quantity; index += 1){
        key = sourceKeys[index];
        if (!_hasOwnProperty$1.call(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, startLine, startLineStart, startPos) {
    var index, quantity;
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(index = 0, quantity = keyNode.length; index < quantity; index += 1){
            if (Array.isArray(keyNode[index])) {
                throwError1(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class1(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class1(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (_result === null) {
        _result = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(index = 0, quantity = valueNode.length; index < quantity; index += 1){
                mergeMappings1(state, _result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings1(state, _result, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !_hasOwnProperty$1.call(overridableKeys, keyNode) && _hasOwnProperty$1.call(_result, keyNode)) {
            state.line = startLine || state.line;
            state.lineStart = startLineStart || state.lineStart;
            state.position = startPos || state.position;
            throwError1(state, "duplicated mapping key");
        }
        if (keyNode === "__proto__") {
            Object.defineProperty(_result, keyNode, {
                configurable: true,
                enumerable: true,
                writable: true,
                value: valueNode
            });
        } else {
            _result[keyNode] = valueNode;
        }
        delete overridableKeys[keyNode];
    }
    return _result;
}
function readLineBreak1(state) {
    var ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 10) {
        state.position++;
    } else if (ch === 13) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 10) {
            state.position++;
        }
    } else {
        throwError1(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
    state.firstTabInLine = -1;
}
function skipSeparationSpace1(state, allowComments, checkIndent) {
    var lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(is_WHITE_SPACE(ch)){
            if (ch === 9 && state.firstTabInLine === -1) {
                state.firstTabInLine = state.position;
            }
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (is_EOL(ch)) {
            readLineBreak1(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 32){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning1(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator1(state) {
    var _position = state.position, ch;
    ch = state.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || is_WS_OR_EOL(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines1(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += common.repeat("\n", count - 1);
    }
}
function readPlainScalar1(state, nodeIndent, withinFlowCollection) {
    var preceding, following, captureStart, captureEnd, hasPendingContent, _line, _lineStart, _lineIndent, _kind = state.kind, _result = state.result, ch;
    ch = state.input.charCodeAt(state.position);
    if (is_WS_OR_EOL(ch) || is_FLOW_INDICATOR(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    if (ch === 63 || ch === 45) {
        following = state.input.charCodeAt(state.position + 1);
        if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    captureStart = captureEnd = state.position;
    hasPendingContent = false;
    while(ch !== 0){
        if (ch === 58) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
                break;
            }
        } else if (ch === 35) {
            preceding = state.input.charCodeAt(state.position - 1);
            if (is_WS_OR_EOL(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator1(state) || withinFlowCollection && is_FLOW_INDICATOR(ch)) {
            break;
        } else if (is_EOL(ch)) {
            _line = state.line;
            _lineStart = state.lineStart;
            _lineIndent = state.lineIndent;
            skipSeparationSpace1(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = _line;
                state.lineStart = _lineStart;
                state.lineIndent = _lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment1(state, captureStart, captureEnd, false);
            writeFoldedLines1(state, state.line - _line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!is_WHITE_SPACE(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment1(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = _kind;
    state.result = _result;
    return false;
}
function readSingleQuotedScalar1(state, nodeIndent) {
    var ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 39) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 39) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 39) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (is_EOL(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar1(state, nodeIndent) {
    var captureStart, captureEnd, hexLength, hexResult, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 34) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 34) {
            captureSegment1(state, captureStart, state.position, true);
            state.position++;
            return true;
        } else if (ch === 92) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (is_EOL(ch)) {
                skipSeparationSpace1(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck1[ch]) {
                state.result += simpleEscapeMap1[ch];
                state.position++;
            } else if ((tmp = escapedHexLen1(ch)) > 0) {
                hexLength = tmp;
                hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode1(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        throwError1(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint1(hexResult);
                state.position++;
            } else {
                throwError1(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (is_EOL(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection1(state, nodeIndent) {
    var readNext = true, _line, _lineStart, _pos, _tag = state.tag, _result, _anchor = state.anchor, following, terminator, isPair, isExplicitPair, isMapping, overridableKeys = Object.create(null), keyNode, keyTag, valueNode, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        _result = [];
    } else if (ch === 123) {
        terminator = 125;
        isMapping = true;
        _result = {};
    } else {
        return false;
    }
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(++state.position);
    while(ch !== 0){
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = _tag;
            state.anchor = _anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = _result;
            return true;
        } else if (!readNext) {
            throwError1(state, "missed comma between flow collection entries");
        } else if (ch === 44) {
            throwError1(state, "expected the node content, but found ','");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace1(state, true, nodeIndent);
            }
        }
        _line = state.line;
        _lineStart = state.lineStart;
        _pos = state.position;
        composeNode1(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
        keyTag = state.tag;
        keyNode = state.result;
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === _line) && ch === 58) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace1(state, true, nodeIndent);
            composeNode1(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos);
        } else if (isPair) {
            _result.push(storeMappingPair1(state, null, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos));
        } else {
            _result.push(keyNode);
        }
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 44) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    throwError1(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar1(state, nodeIndent) {
    var captureStart, folding, chomping = CHOMPING_CLIP, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 43 || ch === 45) {
            if (CHOMPING_CLIP === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP1 : CHOMPING_STRIP1;
            } else {
                throwError1(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode1(ch)) >= 0) {
            if (tmp === 0) {
                throwError1(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                throwError1(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (is_WHITE_SPACE(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (is_WHITE_SPACE(ch))
        if (ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!is_EOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak1(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 32){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (is_EOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === CHOMPING_KEEP1) {
                state.result += common.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === CHOMPING_CLIP) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (is_WHITE_SPACE(ch)) {
                atMoreIndented = true;
                state.result += common.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += common.repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += common.repeat("\n", emptyLines);
            }
        } else {
            state.result += common.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        captureStart = state.position;
        while(!is_EOL(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment1(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence1(state, nodeIndent) {
    var _line, _tag = state.tag, _anchor = state.anchor, _result = [], following, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        if (ch !== 45) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!is_WS_OR_EOL(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace1(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                _result.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        _line = state.line;
        composeNode1(state, nodeIndent, CONTEXT_BLOCK_IN1, false, true);
        _result.push(state.result);
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "sequence";
        state.result = _result;
        return true;
    }
    return false;
}
function readBlockMapping1(state, nodeIndent, flowIndent) {
    var following, allowCompact, _line, _keyLine, _keyLineStart, _keyPos, _tag = state.tag, _anchor = state.anchor, _result = {}, overridableKeys = Object.create(null), keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (!atExplicitKey && state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        following = state.input.charCodeAt(state.position + 1);
        _line = state.line;
        if ((ch === 63 || ch === 58) && is_WS_OR_EOL(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                throwError1(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else {
            _keyLine = state.line;
            _keyLineStart = state.lineStart;
            _keyPos = state.position;
            if (!composeNode1(state, flowIndent, CONTEXT_FLOW_OUT, false, true)) {
                break;
            }
            if (state.line === _line) {
                ch = state.input.charCodeAt(state.position);
                while(is_WHITE_SPACE(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 58) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!is_WS_OR_EOL(ch)) {
                        throwError1(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    throwError1(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = _tag;
                    state.anchor = _anchor;
                    return true;
                }
            } else if (detected) {
                throwError1(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = _tag;
                state.anchor = _anchor;
                return true;
            }
        }
        if (state.line === _line || state.lineIndent > nodeIndent) {
            if (atExplicitKey) {
                _keyLine = state.line;
                _keyLineStart = state.lineStart;
                _keyPos = state.position;
            }
            if (composeNode1(state, nodeIndent, CONTEXT_BLOCK_OUT1, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _keyLine, _keyLineStart, _keyPos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace1(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "mapping";
        state.result = _result;
    }
    return detected;
}
function readTagProperty1(state) {
    var _position, isVerbatim = false, isNamed = false, tagHandle, tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 33) return false;
    if (state.tag !== null) {
        throwError1(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    _position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 62)
        if (state.position < state.length) {
            tagName = state.input.slice(_position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            throwError1(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state.input.slice(_position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE1.test(tagHandle)) {
                        throwError1(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    _position = state.position + 1;
                } else {
                    throwError1(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(_position, state.position);
        if (PATTERN_FLOW_INDICATORS1.test(tagName)) {
            throwError1(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI1.test(tagName)) {
        throwError1(state, "tag name cannot contain such characters: " + tagName);
    }
    try {
        tagName = decodeURIComponent(tagName);
    } catch (err) {
        throwError1(state, "tag name is malformed: " + tagName);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (_hasOwnProperty$1.call(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = "!" + tagName;
    } else if (tagHandle === "!!") {
        state.tag = "tag:yaml.org,2002:" + tagName;
    } else {
        throwError1(state, 'undeclared tag handle "' + tagHandle + '"');
    }
    return true;
}
function readAnchorProperty1(state) {
    var _position, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 38) return false;
    if (state.anchor !== null) {
        throwError1(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(_position, state.position);
    return true;
}
function readAlias1(state) {
    var _position, alias, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 42) return false;
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an alias node must contain at least one character");
    }
    alias = state.input.slice(_position, state.position);
    if (!_hasOwnProperty$1.call(state.anchorMap, alias)) {
        throwError1(state, 'unidentified alias "' + alias + '"');
    }
    state.result = state.anchorMap[alias];
    skipSeparationSpace1(state, true, -1);
    return true;
}
function composeNode1(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    var allowBlockStyles, allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, typeIndex, typeQuantity, typeList, type2, flowIndent, blockIndent;
    if (state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT1 === nodeContext || CONTEXT_BLOCK_IN1 === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace1(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty1(state) || readAnchorProperty1(state)){
            if (skipSeparationSpace1(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || CONTEXT_BLOCK_OUT1 === nodeContext) {
        if (CONTEXT_FLOW_IN === nodeContext || CONTEXT_FLOW_OUT === nodeContext) {
            flowIndent = parentIndent;
        } else {
            flowIndent = parentIndent + 1;
        }
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence1(state, blockIndent) || readBlockMapping1(state, blockIndent, flowIndent)) || readFlowCollection1(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar1(state, flowIndent) || readSingleQuotedScalar1(state, flowIndent) || readDoubleQuotedScalar1(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias1(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        throwError1(state, "alias node should not have any properties");
                    }
                } else if (readPlainScalar1(state, flowIndent, CONTEXT_FLOW_IN === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence1(state, blockIndent);
        }
    }
    if (state.tag === null) {
        if (state.anchor !== null) {
            state.anchorMap[state.anchor] = state.result;
        }
    } else if (state.tag === "?") {
        if (state.result !== null && state.kind !== "scalar") {
            throwError1(state, 'unacceptable node kind for !<?> tag; it should be "scalar", not "' + state.kind + '"');
        }
        for(typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex += 1){
            type2 = state.implicitTypes[typeIndex];
            if (type2.resolve(state.result)) {
                state.result = type2.construct(state.result);
                state.tag = type2.tag;
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
                break;
            }
        }
    } else if (state.tag !== "!") {
        if (_hasOwnProperty$1.call(state.typeMap[state.kind || "fallback"], state.tag)) {
            type2 = state.typeMap[state.kind || "fallback"][state.tag];
        } else {
            type2 = null;
            typeList = state.typeMap.multi[state.kind || "fallback"];
            for(typeIndex = 0, typeQuantity = typeList.length; typeIndex < typeQuantity; typeIndex += 1){
                if (state.tag.slice(0, typeList[typeIndex].tag.length) === typeList[typeIndex].tag) {
                    type2 = typeList[typeIndex];
                    break;
                }
            }
        }
        if (!type2) {
            throwError1(state, "unknown tag !<" + state.tag + ">");
        }
        if (state.result !== null && type2.kind !== state.kind) {
            throwError1(state, "unacceptable node kind for !<" + state.tag + '> tag; it should be "' + type2.kind + '", not "' + state.kind + '"');
        }
        if (!type2.resolve(state.result, state.tag)) {
            throwError1(state, "cannot resolve a node with !<" + state.tag + "> explicit tag");
        } else {
            state.result = type2.construct(state.result, state.tag);
            if (state.anchor !== null) {
                state.anchorMap[state.anchor] = state.result;
            }
        }
    }
    if (state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument1(state) {
    var documentStart = state.position, _position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = Object.create(null);
    state.anchorMap = Object.create(null);
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        _position = state.position;
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(_position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            throwError1(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(is_WHITE_SPACE(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 35) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !is_EOL(ch))
                break;
            }
            if (is_EOL(ch)) break;
            _position = state.position;
            while(ch !== 0 && !is_WS_OR_EOL(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(_position, state.position));
        }
        if (ch !== 0) readLineBreak1(state);
        if (_hasOwnProperty$1.call(directiveHandlers1, directiveName)) {
            directiveHandlers1[directiveName](state, directiveName, directiveArgs);
        } else {
            throwWarning1(state, 'unknown document directive "' + directiveName + '"');
        }
    }
    skipSeparationSpace1(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 45 && state.input.charCodeAt(state.position + 1) === 45 && state.input.charCodeAt(state.position + 2) === 45) {
        state.position += 3;
        skipSeparationSpace1(state, true, -1);
    } else if (hasDirectives) {
        throwError1(state, "directives end mark is expected");
    }
    composeNode1(state, state.lineIndent - 1, CONTEXT_BLOCK_OUT1, false, true);
    skipSeparationSpace1(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS1.test(state.input.slice(documentStart, state.position))) {
        throwWarning1(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator1(state)) {
        if (state.input.charCodeAt(state.position) === 46) {
            state.position += 3;
            skipSeparationSpace1(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        throwError1(state, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments1(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    var state = new State$1(input, options);
    var nullpos = input.indexOf("\0");
    if (nullpos !== -1) {
        state.position = nullpos;
        throwError1(state, "null byte is not allowed in input");
    }
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 32){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument1(state);
    }
    return state.documents;
}
function loadAll$1(input, iterator, options) {
    if (iterator !== null && typeof iterator === "object" && typeof options === "undefined") {
        options = iterator;
        iterator = null;
    }
    var documents = loadDocuments1(input, options);
    if (typeof iterator !== "function") {
        return documents;
    }
    for(var index = 0, length = documents.length; index < length; index += 1){
        iterator(documents[index]);
    }
}
function load$1(input, options) {
    var documents = loadDocuments1(input, options);
    if (documents.length === 0) {
        return void 0;
    } else if (documents.length === 1) {
        return documents[0];
    }
    throw new exception("expected a single document in the stream, but found more");
}
var loadAll_1 = loadAll$1;
var load_1 = load$1;
var loader = {
    loadAll: loadAll_1,
    load: load_1
};
var _toString3 = Object.prototype.toString;
var _hasOwnProperty = Object.prototype.hasOwnProperty;
var CHAR_BOM = 65279;
var CHAR_TAB = 9;
var CHAR_LINE_FEED = 10;
var CHAR_CARRIAGE_RETURN = 13;
var CHAR_SPACE = 32;
var CHAR_EXCLAMATION = 33;
var CHAR_DOUBLE_QUOTE = 34;
var CHAR_SHARP = 35;
var CHAR_PERCENT = 37;
var CHAR_AMPERSAND = 38;
var CHAR_SINGLE_QUOTE = 39;
var CHAR_ASTERISK = 42;
var CHAR_COMMA = 44;
var CHAR_MINUS = 45;
var CHAR_COLON = 58;
var CHAR_EQUALS = 61;
var CHAR_GREATER_THAN = 62;
var CHAR_QUESTION = 63;
var CHAR_COMMERCIAL_AT = 64;
var CHAR_LEFT_SQUARE_BRACKET = 91;
var CHAR_RIGHT_SQUARE_BRACKET = 93;
var CHAR_GRAVE_ACCENT = 96;
var CHAR_LEFT_CURLY_BRACKET = 123;
var CHAR_VERTICAL_LINE = 124;
var CHAR_RIGHT_CURLY_BRACKET = 125;
var ESCAPE_SEQUENCES1 = {};
ESCAPE_SEQUENCES1[0] = "\\0";
ESCAPE_SEQUENCES1[7] = "\\a";
ESCAPE_SEQUENCES1[8] = "\\b";
ESCAPE_SEQUENCES1[9] = "\\t";
ESCAPE_SEQUENCES1[10] = "\\n";
ESCAPE_SEQUENCES1[11] = "\\v";
ESCAPE_SEQUENCES1[12] = "\\f";
ESCAPE_SEQUENCES1[13] = "\\r";
ESCAPE_SEQUENCES1[27] = "\\e";
ESCAPE_SEQUENCES1[34] = '\\"';
ESCAPE_SEQUENCES1[92] = "\\\\";
ESCAPE_SEQUENCES1[133] = "\\N";
ESCAPE_SEQUENCES1[160] = "\\_";
ESCAPE_SEQUENCES1[8232] = "\\L";
ESCAPE_SEQUENCES1[8233] = "\\P";
var DEPRECATED_BOOLEANS_SYNTAX1 = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF", 
];
var DEPRECATED_BASE60_SYNTAX = /^[-+]?[0-9_]+(?::[0-9_]+)+(?:\.[0-9_]*)?$/;
function compileStyleMap1(schema2, map2) {
    var result1, keys, index, length, tag, style, type2;
    if (map2 === null) return {};
    result1 = {};
    keys = Object.keys(map2);
    for(index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map2[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = "tag:yaml.org,2002:" + tag.slice(2);
        }
        type2 = schema2.compiledTypeMap["fallback"][tag];
        if (type2 && _hasOwnProperty.call(type2.styleAliases, style)) {
            style = type2.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
function encodeHex1(character) {
    var string, handle, length;
    string = character.toString(16).toUpperCase();
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new exception("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return "\\" + handle + common.repeat("0", length - string.length) + string;
}
var QUOTING_TYPE_SINGLE = 1, QUOTING_TYPE_DOUBLE = 2;
function State2(options) {
    this.schema = options["schema"] || _default;
    this.indent = Math.max(1, options["indent"] || 2);
    this.noArrayIndent = options["noArrayIndent"] || false;
    this.skipInvalid = options["skipInvalid"] || false;
    this.flowLevel = common.isNothing(options["flowLevel"]) ? -1 : options["flowLevel"];
    this.styleMap = compileStyleMap1(this.schema, options["styles"] || null);
    this.sortKeys = options["sortKeys"] || false;
    this.lineWidth = options["lineWidth"] || 80;
    this.noRefs = options["noRefs"] || false;
    this.noCompatMode = options["noCompatMode"] || false;
    this.condenseFlow = options["condenseFlow"] || false;
    this.quotingType = options["quotingType"] === '"' ? QUOTING_TYPE_DOUBLE : QUOTING_TYPE_SINGLE;
    this.forceQuotes = options["forceQuotes"] || false;
    this.replacer = typeof options["replacer"] === "function" ? options["replacer"] : null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.explicitTypes = this.schema.compiledExplicit;
    this.tag = null;
    this.result = "";
    this.duplicates = [];
    this.usedDuplicates = null;
}
function indentString1(string, spaces) {
    var ind = common.repeat(" ", spaces), position = 0, next = -1, result1 = "", line, length = string.length;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine1(state, level) {
    return "\n" + common.repeat(" ", state.indent * level);
}
function testImplicitResolving1(state, str2) {
    var index, length, type2;
    for(index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type2 = state.implicitTypes[index];
        if (type2.resolve(str2)) {
            return true;
        }
    }
    return false;
}
function isWhitespace1(c) {
    return c === CHAR_SPACE || c === CHAR_TAB;
}
function isPrintable1(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== CHAR_BOM || 65536 <= c && c <= 1114111;
}
function isNsCharOrWhitespace(c) {
    return isPrintable1(c) && c !== CHAR_BOM && c !== CHAR_CARRIAGE_RETURN && c !== CHAR_LINE_FEED;
}
function isPlainSafe1(c, prev, inblock) {
    var cIsNsCharOrWhitespace = isNsCharOrWhitespace(c);
    var cIsNsChar = cIsNsCharOrWhitespace && !isWhitespace1(c);
    return (inblock ? cIsNsCharOrWhitespace : cIsNsCharOrWhitespace && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET) && c !== CHAR_SHARP && !(prev === CHAR_COLON && !cIsNsChar) || isNsCharOrWhitespace(prev) && !isWhitespace1(prev) && c === CHAR_SHARP || prev === CHAR_COLON && cIsNsChar;
}
function isPlainSafeFirst1(c) {
    return isPrintable1(c) && c !== CHAR_BOM && !isWhitespace1(c) && c !== CHAR_MINUS && c !== CHAR_QUESTION && c !== CHAR_COLON && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET && c !== CHAR_SHARP && c !== CHAR_AMPERSAND && c !== CHAR_ASTERISK && c !== CHAR_EXCLAMATION && c !== CHAR_VERTICAL_LINE && c !== CHAR_EQUALS && c !== CHAR_GREATER_THAN && c !== CHAR_SINGLE_QUOTE && c !== CHAR_DOUBLE_QUOTE && c !== CHAR_PERCENT && c !== CHAR_COMMERCIAL_AT && c !== CHAR_GRAVE_ACCENT;
}
function isPlainSafeLast(c) {
    return !isWhitespace1(c) && c !== CHAR_COLON;
}
function codePointAt(string, pos) {
    var first = string.charCodeAt(pos), second;
    if (first >= 55296 && first <= 56319 && pos + 1 < string.length) {
        second = string.charCodeAt(pos + 1);
        if (second >= 56320 && second <= 57343) {
            return (first - 55296) * 1024 + second - 56320 + 65536;
        }
    }
    return first;
}
function needIndentIndicator1(string) {
    var leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
var STYLE_PLAIN1 = 1, STYLE_SINGLE1 = 2, STYLE_LITERAL1 = 3, STYLE_FOLDED1 = 4, STYLE_DOUBLE1 = 5;
function chooseScalarStyle1(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType, quotingType, forceQuotes, inblock) {
    var i1;
    var __char = 0;
    var prevChar = null;
    var hasLineBreak = false;
    var hasFoldableLine = false;
    var shouldTrackWidth = lineWidth !== -1;
    var previousLineBreak = -1;
    var plain = isPlainSafeFirst1(codePointAt(string, 0)) && isPlainSafeLast(codePointAt(string, string.length - 1));
    if (singleLineOnly || forceQuotes) {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
    } else {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (__char === CHAR_LINE_FEED) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        if (plain && !forceQuotes && !testAmbiguousType(string)) {
            return STYLE_PLAIN1;
        }
        return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE1 : STYLE_SINGLE1;
    }
    if (indentPerLevel > 9 && needIndentIndicator1(string)) {
        return STYLE_DOUBLE1;
    }
    if (!forceQuotes) {
        return hasFoldableLine ? STYLE_FOLDED1 : STYLE_LITERAL1;
    }
    return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE1 : STYLE_SINGLE1;
}
function writeScalar1(state, string, level, iskey, inblock) {
    state.dump = function() {
        if (string.length === 0) {
            return state.quotingType === QUOTING_TYPE_DOUBLE ? '""' : "''";
        }
        if (!state.noCompatMode) {
            if (DEPRECATED_BOOLEANS_SYNTAX1.indexOf(string) !== -1 || DEPRECATED_BASE60_SYNTAX.test(string)) {
                return state.quotingType === QUOTING_TYPE_DOUBLE ? '"' + string + '"' : "'" + string + "'";
            }
        }
        var indent = state.indent * Math.max(1, level);
        var lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        var singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(string2) {
            return testImplicitResolving1(state, string2);
        }
        switch(chooseScalarStyle1(string, singleLineOnly, state.indent, lineWidth, testAmbiguity, state.quotingType, state.forceQuotes && !iskey, inblock)){
            case STYLE_PLAIN1:
                return string;
            case STYLE_SINGLE1:
                return "'" + string.replace(/'/g, "''") + "'";
            case STYLE_LITERAL1:
                return "|" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(string, indent));
            case STYLE_FOLDED1:
                return ">" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(foldString1(string, lineWidth), indent));
            case STYLE_DOUBLE1:
                return '"' + escapeString1(string) + '"';
            default:
                throw new exception("impossible error: invalid scalar style");
        }
    }();
}
function blockHeader1(string, indentPerLevel) {
    var indentIndicator = needIndentIndicator1(string) ? String(indentPerLevel) : "";
    var clip = string[string.length - 1] === "\n";
    var keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    var chomp = keep ? "+" : clip ? "" : "-";
    return indentIndicator + chomp + "\n";
}
function dropEndingNewline1(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString1(string, width) {
    var lineRe = /(\n+)([^\n]*)/g;
    var result1 = function() {
        var nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine1(string.slice(0, nextLF), width);
    }();
    var prevMoreIndented = string[0] === "\n" || string[0] === " ";
    var moreIndented;
    var match;
    while(match = lineRe.exec(string)){
        var prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine1(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function foldLine1(line, width) {
    if (line === "" || line[0] === " ") return line;
    var breakRe = / [^ ]/g;
    var match;
    var start = 0, end, curr = 0, next = 0;
    var result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += "\n" + line.slice(start, end);
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += line.slice(start, curr) + "\n" + line.slice(curr + 1);
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function escapeString1(string) {
    var result1 = "";
    var __char = 0;
    var escapeSeq;
    for(var i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
        __char = codePointAt(string, i1);
        escapeSeq = ESCAPE_SEQUENCES1[__char];
        if (!escapeSeq && isPrintable1(__char)) {
            result1 += string[i1];
            if (__char >= 65536) result1 += string[i1 + 1];
        } else {
            result1 += escapeSeq || encodeHex1(__char);
        }
    }
    return result1;
}
function writeFlowSequence1(state, level, object) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level, value, false, false) || typeof value === "undefined" && writeNode1(state, level, null, false, false)) {
            if (_result !== "") _result += "," + (!state.condenseFlow ? " " : "");
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = "[" + _result + "]";
}
function writeBlockSequence1(state, level, object, compact) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level + 1, value, true, true, false, true) || typeof value === "undefined" && writeNode1(state, level + 1, null, true, true, false, true)) {
            if (!compact || _result !== "") {
                _result += generateNextLine1(state, level);
            }
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping1(state, level, object) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, pairBuffer;
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (_result !== "") pairBuffer += ", ";
        if (state.condenseFlow) pairBuffer += '"';
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += state.dump + (state.condenseFlow ? '"' : "") + ":" + (state.condenseFlow ? "" : " ");
        if (!writeNode1(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = "{" + _result + "}";
}
function writeBlockMapping1(state, level, object, compact) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, explicitPair, pairBuffer;
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new exception("sortKeys must be a boolean or a function");
    }
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || _result !== "") {
            pairBuffer += generateNextLine1(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine1(state, level);
        }
        if (!writeNode1(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType1(state, object, explicit) {
    var _result, typeList, index, length, type2, style;
    typeList = explicit ? state.explicitTypes : state.implicitTypes;
    for(index = 0, length = typeList.length; index < length; index += 1){
        type2 = typeList[index];
        if ((type2.instanceOf || type2.predicate) && (!type2.instanceOf || typeof object === "object" && object instanceof type2.instanceOf) && (!type2.predicate || type2.predicate(object))) {
            if (explicit) {
                if (type2.multi && type2.representName) {
                    state.tag = type2.representName(object);
                } else {
                    state.tag = type2.tag;
                }
            } else {
                state.tag = "?";
            }
            if (type2.represent) {
                style = state.styleMap[type2.tag] || type2.defaultStyle;
                if (_toString3.call(type2.represent) === "[object Function]") {
                    _result = type2.represent(object, style);
                } else if (_hasOwnProperty.call(type2.represent, style)) {
                    _result = type2.represent[style](object, style);
                } else {
                    throw new exception("!<" + type2.tag + '> tag resolver accepts not "' + style + '" style');
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode1(state, level, object, block, compact, iskey, isblockseq) {
    state.tag = null;
    state.dump = object;
    if (!detectType1(state, object, false)) {
        detectType1(state, object, true);
    }
    var type2 = _toString3.call(state.dump);
    var inblock = block;
    var tagStr;
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    var objectOrArray = type2 === "[object Object]" || type2 === "[object Array]", duplicateIndex, duplicate;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = "*ref_" + duplicateIndex;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type2 === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping1(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowMapping1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object Array]") {
            if (block && state.dump.length !== 0) {
                if (state.noArrayIndent && !isblockseq && level > 0) {
                    writeBlockSequence1(state, level - 1, state.dump, compact);
                } else {
                    writeBlockSequence1(state, level, state.dump, compact);
                }
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowSequence1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object String]") {
            if (state.tag !== "?") {
                writeScalar1(state, state.dump, level, iskey, inblock);
            }
        } else if (type2 === "[object Undefined]") {
            return false;
        } else {
            if (state.skipInvalid) return false;
            throw new exception("unacceptable kind of an object to dump " + type2);
        }
        if (state.tag !== null && state.tag !== "?") {
            tagStr = encodeURI(state.tag[0] === "!" ? state.tag.slice(1) : state.tag).replace(/!/g, "%21");
            if (state.tag[0] === "!") {
                tagStr = "!" + tagStr;
            } else if (tagStr.slice(0, 18) === "tag:yaml.org,2002:") {
                tagStr = "!!" + tagStr.slice(18);
            } else {
                tagStr = "!<" + tagStr + ">";
            }
            state.dump = tagStr + " " + state.dump;
        }
    }
    return true;
}
function getDuplicateReferences1(object, state) {
    var objects = [], duplicatesIndexes = [], index, length;
    inspectNode1(object, objects, duplicatesIndexes);
    for(index = 0, length = duplicatesIndexes.length; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = new Array(length);
}
function inspectNode1(object, objects, duplicatesIndexes) {
    var objectKeyList, index, length;
    if (object !== null && typeof object === "object") {
        index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(index = 0, length = object.length; index < length; index += 1){
                    inspectNode1(object[index], objects, duplicatesIndexes);
                }
            } else {
                objectKeyList = Object.keys(object);
                for(index = 0, length = objectKeyList.length; index < length; index += 1){
                    inspectNode1(object[objectKeyList[index]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function dump$1(input, options) {
    options = options || {};
    var state = new State2(options);
    if (!state.noRefs) getDuplicateReferences1(input, state);
    var value = input;
    if (state.replacer) {
        value = state.replacer.call({
            "": value
        }, "", value);
    }
    if (writeNode1(state, 0, value, true, true)) return state.dump + "\n";
    return "";
}
var dump_1 = dump$1;
var dumper = {
    dump: dump_1
};
function renamed(from, to) {
    return function() {
        throw new Error("Function yaml." + from + " is removed in js-yaml 4. Use yaml." + to + " instead, which is now safe by default.");
    };
}
var Type2 = type;
var Schema1 = schema;
var load1 = loader.load;
loader.loadAll;
dumper.dump;
renamed("safeLoad", "load");
renamed("safeLoadAll", "loadAll");
renamed("safeDump", "dump");
const QuartoJSONSchema = new Schema1({
    implicit: [
        _null,
        bool1,
        __int1,
        __float1
    ],
    include: [
        failsafe1
    ],
    explicit: [
        new Type2("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        }), 
    ]
});
function postProcessAnnotation(parse) {
    if (parse.components.length === 1 && parse.start === parse.components[0].start && parse.end === parse.components[0].end) {
        return postProcessAnnotation(parse.components[0]);
    } else {
        return {
            ...parse,
            components: parse.components.map(postProcessAnnotation)
        };
    }
}
function jsYamlParseLenient(yml) {
    try {
        return load1(yml, {
            schema: QuartoJSONSchema
        });
    } catch (_e) {
        return yml;
    }
}
function readAnnotatedYamlFromMappedString(mappedSource, lenient = false) {
    if (lenient) {
        const parser = getTreeSitterSync();
        const tree = parser.parse(mappedSource.value);
        const treeSitterAnnotation = buildTreeSitterAnnotation(tree, mappedSource);
        if (treeSitterAnnotation) {
            return treeSitterAnnotation;
        }
    }
    try {
        return buildJsYamlAnnotation(mappedSource);
    } catch (e) {
        const m = e.stack.split("\n")[0].match(/^.+ \((\d+):(\d+)\)$/);
        if (m) {
            const f = lineColToIndex(mappedSource.value);
            const location = {
                line: Number(m[1]) - 1,
                column: Number(m[2] - 1)
            };
            const offset = f(location);
            const { originalString  } = mappedSource.map(offset, true);
            const filename = originalString.fileName;
            const f2 = mappedIndexToLineCol(mappedSource);
            const { line , column  } = f2(offset);
            const sourceContext = createSourceContext(mappedSource, {
                start: offset,
                end: offset + 1
            });
            e.stack = `${e.reason} (${filename}, ${line + 1}:${column + 1})\n${sourceContext}`;
            e.message = e.stack;
            if (mappedLines(mappedSource)[location.line].value.indexOf("!expr") !== -1 && e.reason.match(/bad indentation of a mapping entry/)) {
                e.message = `${e.message}\n${tidyverseInfo("YAML tags like !expr must be followed by YAML strings.")}\n${tidyverseInfo("Is it possible you need to quote the value you passed to !expr ?")}`;
            }
            e.stack = "";
        }
        throw e;
    }
}
function buildJsYamlAnnotation(mappedYaml) {
    const yml = mappedYaml.value;
    const stack = [];
    const results = [];
    function listener(what, state) {
        const { result: result1 , position , kind  } = state;
        if (what === "close") {
            const { position: openPosition  } = stack.pop();
            if (results.length > 0) {
                const last = results[results.length - 1];
                if (last.start === openPosition && last.end === position) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last1 = results[results.length - 1];
                if (last1.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position);
            const leftTrim = rawRange.length - rawRange.trimLeft().length;
            const rightTrim = rawRange.length - rawRange.trimRight().length;
            if (rawRange.trim().length === 0) {
                results.push({
                    start: position - rightTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            } else {
                results.push({
                    start: openPosition + leftTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            }
        } else {
            stack.push({
                position
            });
        }
    }
    load1(yml, {
        listener,
        schema: QuartoJSONSchema
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: [],
            source: mappedYaml
        };
    }
    if (results.length !== 1) {
        throw new Error(`Internal Error - expected a single result, got ${results.length} instead`);
    }
    JSON.stringify(results[0]);
    return postProcessAnnotation(results[0]);
}
function buildTreeSitterAnnotation(tree, mappedSource) {
    const errors = [];
    const singletonBuild = (node)=>{
        let tag = undefined;
        for (const child of node.children){
            if (child.type === "tag") {
                tag = child;
                continue;
            }
            if (child.type !== "comment") {
                const result1 = buildNode(child, node.endIndex);
                if (tag) {
                    return annotateTag(result1, tag, node);
                } else {
                    return result1;
                }
            }
        }
        return annotateEmpty(node.endIndex);
    };
    const buildNode = (node, endIndex)=>{
        if (node === null) {
            return annotateEmpty(endIndex === undefined ? -1 : endIndex);
        }
        if (dispatch[node.type] === undefined) {
            return annotateEmpty(endIndex || node.endIndex || -1);
        }
        return dispatch[node.type](node);
    };
    const annotateEmpty = (position)=>{
        return {
            start: position,
            end: position,
            result: null,
            kind: "<<EMPTY>>",
            components: [],
            source: mappedSource
        };
    };
    const annotate = (node, result1, components)=>{
        return {
            start: node.startIndex,
            end: node.endIndex,
            result: result1,
            kind: node.type,
            components,
            source: mappedSource
        };
    };
    const annotateTag = (innerParse, tagNode, outerNode)=>{
        const tagParse = annotate(tagNode, tagNode.text, []);
        const result1 = annotate(outerNode, {
            tag: tagNode.text,
            value: innerParse.result
        }, [
            tagParse,
            innerParse
        ]);
        return result1;
    };
    const buildPair = (node)=>{
        let key, value;
        const children = node.children.filter((n)=>n.type !== "comment");
        if (children.length === 3) {
            key = annotate(children[0], children[0].text, []);
            value = buildNode(children[2], node.endIndex);
        } else if (children.length === 2) {
            key = annotate(children[0], children[0].text, []);
            value = annotateEmpty(node.endIndex);
        } else {
            key = annotateEmpty(node.endIndex);
            value = annotateEmpty(node.endIndex);
        }
        return annotate(node, {
            key: key.result,
            value: value.result
        }, [
            key,
            value
        ]);
    };
    const dispatch = {
        "stream": singletonBuild,
        "document": singletonBuild,
        "block_node": singletonBuild,
        "flow_node": singletonBuild,
        "double_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "single_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "plain_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "block_sequence_item") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component && component.result);
            }
            return annotate(node, result1, components);
        },
        "block_sequence_item": (node)=>{
            if (node.childCount < 2) {
                return annotateEmpty(node.endIndex);
            } else {
                return buildNode(node.child(1), node.endIndex);
            }
        },
        "flow_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "flow_node") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component.result);
            }
            return annotate(node, result1, components);
        },
        "block_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                let component;
                if (child.type === "ERROR") {
                    result1[child.text] = "<<ERROR>>";
                    const key = annotate(child, child.text, []);
                    const value = annotateEmpty(child.endIndex);
                    component = annotate(child, {
                        key: key.result,
                        value: value.result
                    }, [
                        key,
                        value
                    ]);
                } else if (child.type !== "block_mapping_pair") {
                    continue;
                } else {
                    component = buildNode(child, node.endIndex);
                }
                const { key: key1 , value: value1  } = component.result;
                result1[String(key1)] = value1;
                components.push(...component.components);
            }
            return annotate(node, result1, components);
        },
        "flow_pair": buildPair,
        "flow_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type === "flow_node") {
                    continue;
                }
                if (child.type === "flow_pair") {
                    const component = buildNode(child, node.endIndex);
                    const { key , value  } = component.result;
                    result1[String(key)] = value;
                    components.push(...component.components);
                }
            }
            return annotate(node, result1, components);
        },
        "block_mapping_pair": buildPair
    };
    const result1 = buildNode(tree.rootNode, tree.rootNode.endIndex);
    if (errors.length) {
        result1.errors = errors;
    }
    const parsedSize = tree.rootNode.text.trim().length;
    const codeSize = mappedSource.value.trim().length;
    const lossage = parsedSize / codeSize;
    if (lossage < 0.95) {
        return null;
    }
    return result1;
}
function locateAnnotation(annotation, position, kind) {
    const originalSource = annotation.source;
    kind = kind || "value";
    for(let i1 = 0; i1 < position.length; ++i1){
        const value = position[i1];
        if (typeof value === "number") {
            const inner = annotation.components[value];
            if (inner === undefined) {
                throw new Error("Internal Error: invalid path for locateAnnotation");
            }
            annotation = inner;
        } else {
            let found = false;
            for(let j = 0; j < annotation.components.length; j += 2){
                if (originalSource.value.substring(annotation.components[j].start, annotation.components[j].end).trim() === value) {
                    if (i1 === position.length - 1) {
                        if (kind === "key") {
                            annotation = annotation.components[j];
                        } else {
                            annotation = annotation.components[j + 1];
                        }
                    }
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new Error("Internal Error: invalid path for locateAnnotation");
            }
        }
    }
    return annotation;
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg, validationErrors){
        super([
            msg,
            ...validationErrors.map((e)=>tidyverseFormatError(e.niceError))
        ].join("\n\n"));
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
const isObject2 = (value)=>{
    const type = typeof value;
    return value !== null && (type === "object" || type === "function");
};
async function readAndValidateYamlFromMappedString(mappedYaml, schema, pruneErrors = true, lenient = false) {
    const annotation = await readAnnotatedYamlFromMappedString(mappedYaml, lenient);
    if (annotation === null) {
        throw new Error("Parse error in readAnnotatedYamlFromMappedString");
    }
    const validateYaml = !isObject2(annotation.result) || annotation.result["validate-yaml"] !== false;
    if (!validateYaml) {
        return {
            yaml: annotation.result,
            yamlValidationErrors: []
        };
    }
    const validate = async (validator)=>{
        const valResult = await validator.validateParse(mappedYaml, annotation, pruneErrors);
        return {
            yaml: annotation.result,
            yamlValidationErrors: valResult.errors
        };
    };
    if (typeof annotation.result === "object" && !Array.isArray(annotation.result)) {
        const preCheckResult = await withValidator(getSchemaDefinition("bad-parse-schema"), validate);
        if (preCheckResult.yamlValidationErrors.length !== 0) {
            return preCheckResult;
        }
    }
    const result1 = await withValidator(schema, validate);
    return result1;
}
function setBaseSchemaProperties(yaml, schema) {
    if (yaml.additionalCompletions) {
        schema = completeSchema(schema, ...yaml.additionalCompletions);
    }
    if (yaml.completions) {
        schema = completeSchemaOverwrite(schema, ...yaml.completions);
    }
    if (yaml.id) {
        schema = idSchema(schema, yaml.id);
    }
    if (yaml.hidden === true) {
        schema = completeSchemaOverwrite(schema);
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    if (yaml.tags) {
        schema = tagSchema(schema, yaml.tags);
    }
    if (yaml.description) {
        schema = tagSchema(schema, {
            description: yaml.description
        });
        if (typeof yaml.description === "string") {
            schema = documentSchema(schema, yaml.description);
        } else if (typeof yaml.description === "object") {
            schema = documentSchema(schema, yaml.description.short);
        }
    }
    const result1 = Object.assign({}, schema);
    if (yaml.errorDescription) {
        result1.description = yaml.errorDescription;
    }
    if (yaml.errorMessage) {
        result1.errorMessage = yaml.errorMessage;
    }
    return result1;
}
function convertFromNull(yaml) {
    return setBaseSchemaProperties(yaml["null"], nullSchema);
}
function convertFromSchema(yaml) {
    const schema = convertFromYaml(yaml.schema);
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromString(yaml) {
    if (yaml["string"].pattern) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], regexSchema(yaml["string"].pattern)));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], stringSchema));
    }
}
function convertFromPattern(yaml) {
    if (typeof yaml.pattern === "string") {
        return setBaseSchemaProperties(yaml, regexSchema(yaml.pattern));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.pattern, regexSchema(yaml.pattern.regex)));
    }
}
function convertFromPath(yaml) {
    return setBaseSchemaProperties(yaml["path"], stringSchema);
}
function convertFromNumber(yaml) {
    return setBaseSchemaProperties(yaml["number"], numberSchema);
}
function convertFromBoolean(yaml) {
    return setBaseSchemaProperties(yaml["boolean"], booleanSchema);
}
function convertFromRef(yaml) {
    return setBaseSchemaProperties(yaml, refSchema(yaml.ref, `be ${yaml.ref}`));
}
function convertFromMaybeArrayOf(yaml) {
    const inner = convertFromYaml(yaml.maybeArrayOf);
    const schema = tagSchema(anyOfSchema(inner, arraySchema(inner)), {
        "complete-from": [
            "anyOf",
            0
        ]
    });
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromArrayOf(yaml) {
    if (yaml.arrayOf.schema) {
        const result1 = arraySchema(convertFromYaml(yaml.arrayOf.schema));
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.arrayOf, result1));
    } else {
        return setBaseSchemaProperties(yaml, arraySchema(convertFromYaml(yaml.arrayOf)));
    }
}
function convertFromAllOf(yaml) {
    if (yaml.allOf.schemas) {
        const inner = yaml.allOf.schemas.map((x)=>convertFromYaml(x));
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.allOf, schema));
    } else {
        const inner1 = yaml.allOf.map((x)=>convertFromYaml(x));
        const schema1 = allOfSchema(...inner1);
        return setBaseSchemaProperties(yaml, schema1);
    }
}
function convertFromAnyOf(yaml) {
    if (yaml.anyOf.schemas) {
        const inner = yaml.anyOf.schemas.map((x)=>convertFromYaml(x));
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.anyOf, schema));
    } else {
        const inner1 = yaml.anyOf.map((x)=>convertFromYaml(x));
        const schema1 = anyOfSchema(...inner1);
        return setBaseSchemaProperties(yaml, schema1);
    }
}
function convertFromEnum(yaml) {
    const schema = yaml["enum"];
    if (schema.hasOwnProperty("values")) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["enum"], enumSchema(...schema.values)));
    } else {
        return setBaseSchemaProperties(yaml, enumSchema(...schema));
    }
}
function convertFromRecord(yaml) {
    if (yaml.record.properties) {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record.properties,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.record, schema));
    } else {
        const schema1 = convertFromObject({
            "object": {
                "properties": yaml.record,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, schema1);
    }
}
function convertFromObject(yaml) {
    const schema = yaml["object"];
    const params = {};
    if (schema.namingConvention) {
        switch(schema.namingConvention){
            case "capitalizationCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "underscoreCase":
                params.namingConvention = "underscore_case";
                break;
            case "underscore-case":
                params.namingConvention = "underscore_case";
                break;
            case "underscore_case":
                params.namingConvention = "underscore_case";
                break;
            case "dashCase":
                params.namingConvention = "dash-case";
                break;
            case "dash-case":
                params.namingConvention = "dash-case";
                break;
            case "dash_case":
                params.namingConvention = "dash-case";
                break;
            case "camelCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "snakeCase":
                params.namingConvention = "underscore_case";
                break;
            case "snake-case":
                params.namingConvention = "underscore_case";
                break;
            case "snake_case":
                params.namingConvention = "underscore_case";
                break;
            case "kebabCase":
                params.namingConvention = "dash-case";
                break;
            case "kebab-case":
                params.namingConvention = "dash-case";
                break;
            case "kebab_case":
                params.namingConvention = "dash-case";
                break;
            default:
                throw new Error("Internal Error: this should have failed validation");
        }
        params.namingConvention = schema.namingConvention;
    }
    if (schema.properties) {
        params.properties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.patternProperties) {
        params.patternProperties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.propertyNames !== undefined) {
        params.propertyNames = convertFromYaml(schema.propertyNames);
    } else if (schema.closed === true) {
        const objectKeys = Object.keys(params.properties || {});
        if (objectKeys.length === 0) {
            throw new Error("object schema `closed` requires field `properties`.");
        }
        if (params.namingConvention !== undefined && params.namingConvention !== "ignore") {
            throw new Error("object schema `closed` is only supported with namingConvention: `ignore`");
        }
        params.namingConvention = "ignore";
        params.closed = true;
    }
    if (schema.additionalProperties !== undefined) {
        if (schema.additionalProperties === false) {
            params.additionalProperties = false;
        } else {
            params.additionalProperties = convertFromYaml(schema.additionalProperties);
        }
    }
    if (schema["super"]) {
        if (Array.isArray(schema["super"])) {
            params.baseSchema = schema["super"].map((s)=>convertFromYaml(s));
        } else {
            params.baseSchema = convertFromYaml(schema["super"]);
        }
    }
    if (schema["required"] === "all") {
        params.required = Object.keys(schema.properties || {});
    } else if (schema["required"]) {
        params.required = schema["required"];
    }
    if (schema["completions"]) {
        params.completions = schema["completions"];
    }
    return setBaseSchemaProperties(yaml, setBaseSchemaProperties(schema, objectSchema(params)));
}
function lookup(yaml) {
    if (!hasSchemaDefinition(yaml.resolveRef)) {
        throw new Error(`lookup of key ${yaml.resolveRef} in definitions failed`);
    }
    return getSchemaDefinition(yaml.resolveRef);
}
function convertFromYaml(yaml) {
    const literalValues = [
        {
            val: "object",
            schema: objectSchema()
        },
        {
            val: "path",
            schema: stringSchema
        },
        {
            val: "string",
            schema: stringSchema
        },
        {
            val: "number",
            schema: numberSchema
        },
        {
            val: "boolean",
            schema: booleanSchema
        },
        {
            val: "any",
            schema: anySchema()
        },
        {
            val: null,
            schema: nullSchema
        }, 
    ];
    for (const { val , schema  } of literalValues){
        if (yaml === val) {
            return schema;
        }
    }
    if (typeof yaml !== "object") {
        return valueSchema(yaml);
    }
    const schemaObjectKeyFunctions = [
        {
            key: "anyOf",
            value: convertFromAnyOf
        },
        {
            key: "allOf",
            value: convertFromAllOf
        },
        {
            key: "boolean",
            value: convertFromBoolean
        },
        {
            key: "arrayOf",
            value: convertFromArrayOf
        },
        {
            key: "enum",
            value: convertFromEnum
        },
        {
            key: "maybeArrayOf",
            value: convertFromMaybeArrayOf
        },
        {
            key: "null",
            value: convertFromNull
        },
        {
            key: "number",
            value: convertFromNumber
        },
        {
            key: "object",
            value: convertFromObject
        },
        {
            key: "path",
            value: convertFromPath
        },
        {
            key: "record",
            value: convertFromRecord
        },
        {
            key: "ref",
            value: convertFromRef
        },
        {
            key: "resolveRef",
            value: lookup
        },
        {
            key: "string",
            value: convertFromString
        },
        {
            key: "pattern",
            value: convertFromPattern
        },
        {
            key: "schema",
            value: convertFromSchema
        }, 
    ];
    for (const { key: objectKey , value: fun  } of schemaObjectKeyFunctions){
        if (yaml[objectKey] !== undefined) {
            return fun(yaml);
        }
    }
    throw new Error("Internal Error: Cannot convert object; this should have failed validation.");
}
function objectSchemaFromFieldsObject(fields, exclude) {
    exclude = exclude || ((_key)=>false);
    const properties = {};
    convertFromFieldsObject(fields, properties);
    for (const key of Object.keys(properties)){
        if (exclude(key)) {
            delete properties[key];
        }
    }
    return objectSchema({
        properties
    });
}
function annotateSchemaFromField(field, schema) {
    if (field.enabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.enabled
        });
    }
    if (field.disabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.disabled.map((x)=>`!${x}`)
        });
    }
    if (field.tags) {
        schema = tagSchema(schema, field.tags);
    }
    if (field.description) {
        if (typeof field.description === "string") {
            schema = documentSchema(schema, field.description);
        } else if (typeof field.description === "object") {
            schema = documentSchema(schema, field.description.short);
        }
        schema = tagSchema(schema, {
            description: field.description
        });
    }
    if (field.hidden) {
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    return schema;
}
function schemaFromField(entry) {
    const schema = convertFromYaml(entry.schema);
    return annotateSchemaFromField(entry, schema);
}
function convertFromFieldsObject(yaml, obj) {
    const result1 = obj || {};
    for (const field of yaml){
        let schema = convertFromYaml(field.schema);
        schema = annotateSchemaFromField(field, schema);
        result1[field.name] = schema;
        if (field.alias) {
            result1[field.alias] = schema;
        }
    }
    return result1;
}
function schemaFieldsFromGlob(globPath, testFun) {
    const result1 = [];
    testFun = testFun || ((_e, _p)=>true);
    for (const [file, fields] of expandResourceGlob(globPath)){
        for (const field of fields){
            const fieldName = field.name;
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            if (testFun(field, file)) {
                result1.push({
                    schemaId,
                    field
                });
            }
        }
    }
    return result1;
}
memoize(()=>{
    const groups = getYamlIntelligenceResource("schema/groups.yml");
    const result1 = [];
    for (const [topLevel, sub] of Object.entries(groups)){
        for (const key of Object.keys(sub)){
            result1.push(`${topLevel}-${key}`);
        }
    }
    return result1;
}, ()=>"const");
function objectRefSchemaFromContextGlob(contextGlob, testFun) {
    const regexp = globToRegExp1(contextGlob);
    return objectRefSchemaFromGlob("schema/{document,cell}-*.yml", (field, path)=>{
        if (testFun !== undefined && !testFun(field, path)) {
            return false;
        }
        const pathContext = path.split("/").slice(-1)[0].slice(0, -4);
        const schemaContexts = field !== undefined && field.tags !== undefined && field.tags.contexts || [];
        if (pathContext.match(regexp)) {
            return true;
        }
        return schemaContexts.some((c)=>c.match(regexp));
    });
}
function objectRefSchemaFromGlob(glob, testFun) {
    const properties = {};
    for (const { schemaId , field  } of schemaFieldsFromGlob(glob, testFun)){
        const schema = refSchema(schemaId, schemaId);
        properties[field.name] = schema;
        if (field.alias) {
            properties[field.alias] = schema;
        }
    }
    return objectSchema({
        properties
    });
}
async function buildResourceSchemas() {
    const path = "schema/{cell-*,document-*,project}.yml";
    for (const [file, fields] of expandResourceGlob(path)){
        const yaml = fields;
        const entries = Object.entries(convertFromFieldsObject(yaml));
        for (const [fieldName, fieldSchema] of entries){
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            const schema = idSchema(fieldSchema, schemaId);
            setSchemaDefinition(schema);
            await withValidator(schema, async (_validator)=>{});
        }
    }
}
function defineCached(thunk, schemaId) {
    let schema;
    return async ()=>{
        if (hasSchemaDefinition(schemaId)) {
            schema = getSchemaDefinition(schemaId);
            return refSchema(schema.$id, schema.description || `be a {schema['$id'] as string}`);
        }
        const result1 = await thunk();
        const { errorHandlers  } = result1;
        schema = result1.schema;
        if (schemaId !== schema.$id) {
            schema = idSchema(schema, schemaId);
        }
        define(schema);
        for (const fun of errorHandlers){
            addValidatorErrorHandler(schema, fun);
        }
        return refSchema(schema.$id, schema.description || `be a {schema['$id']}`);
    };
}
function define(schema) {
    if (schema !== true && schema !== false && schema.$id && !hasSchemaDefinition(schema.$id)) {
        setSchemaDefinition(schema);
    }
}
async function loadDefaultSchemaDefinitions() {
    await loadSchemaDefinitions(getYamlIntelligenceResource("schema/definitions.yml"));
    await buildResourceSchemas();
}
async function loadSchemaDefinitions(yaml) {
    await Promise.all(yaml.map(async (yamlSchema)=>{
        const schema = convertFromYaml(yamlSchema);
        if (schema.$id === undefined) {
            throw new Error(`Internal error: unnamed schema in definitions`);
        }
        setSchemaDefinition(schema);
    }));
}
function getFormatExecuteOptionsSchema() {
    const schema = idSchema(objectRefSchemaFromContextGlob("document-execute"), "front-matter-execute");
    define(schema);
    return refSchema("front-matter-execute", "be a front-matter-execute object");
}
let formatAliases = undefined;
function getFormatAliases() {
    if (formatAliases !== undefined) {
        return formatAliases;
    }
    formatAliases = getYamlIntelligenceResource("schema/format-aliases.yml").aliases;
    return formatAliases;
}
function expandFormatAliases(lst) {
    return expandAliasesFrom(lst, getFormatAliases());
}
function useSchema(schema, format) {
    const formats = schema && schema.tags && schema.tags.formats;
    if (formats === undefined) {
        return true;
    }
    const disabled = formats.filter((f)=>f.startsWith("!")).map((f)=>f.slice(1));
    const enabled = formats.filter((f)=>!f.startsWith("!"));
    if (disabled.length > 0 && expandFormatAliases(disabled).indexOf(format) !== -1) {
        return false;
    }
    if (enabled.length > 0 && expandFormatAliases(enabled).indexOf(format) === -1) {
        return false;
    }
    return true;
}
function getFormatSchema(format) {
    const schema = objectRefSchemaFromContextGlob("document-*", (field)=>{
        const schema = schemaFromField(field);
        return useSchema(schema, format);
    });
    return anyOfSchema(schema, enumSchema("default"));
}
function pandocFormatsResource() {
    return getYamlIntelligenceResource("pandoc/formats.yml");
}
async function makeFrontMatterFormatSchema(nonStrict = false) {
    const hideFormat = (format)=>{
        const hideList = [
            "html",
            "epub",
            "docbook"
        ];
        const hidden = hideList.some((h)=>format.startsWith(h) && format.length > h.length);
        return {
            name: format,
            hidden
        };
    };
    const formatSchemaDescriptorList = (await pandocFormatsResource()).concat("md", "hugo").map((format)=>{
        const { name , hidden ,  } = hideFormat(format);
        return {
            regex: `^(.+-)?${name}([-+].+)?$`,
            schema: getFormatSchema(name),
            name,
            hidden
        };
    });
    const formatSchemas = formatSchemaDescriptorList.map(({ regex , schema  })=>[
            regex,
            schema
        ]);
    const plusFormatStringSchemas = formatSchemaDescriptorList.map(({ regex , name , hidden  })=>{
        const schema = regexSchema(regex, `be '${name}'`);
        if (hidden) {
            return schema;
        }
        return completeSchema(schema, name);
    });
    const luaFilenameS = regexSchema("^.+\.lua$");
    plusFormatStringSchemas.push(luaFilenameS);
    const completionsObject = fromEntries(formatSchemaDescriptorList.filter(({ hidden  })=>!hidden).map(({ name  })=>[
            name,
            {
                type: "key",
                display: name,
                value: `${name}: `,
                description: `be '${name}'`,
                suggest_on_accept: true
            }
        ]));
    return errorMessageSchema(anyOfSchema(describeSchema(anyOfSchema(...plusFormatStringSchemas), "the name of a pandoc-supported output format"), objectSchema({
        propertyNames: luaFilenameS
    }), allOfSchema(objectSchema({
        patternProperties: fromEntries(formatSchemas),
        completions: completionsObject,
        additionalProperties: nonStrict
    }))), "${value} is not a valid output format.");
}
const getFrontMatterFormatSchema = defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(),
        errorHandlers: []
    };
}, "front-matter-format");
defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(true),
        errorHandlers: []
    };
}, "front-matter-format-nonstrict");
const getFrontMatterSchema = defineCached(async ()=>{
    const executeObjSchema = await getFormatExecuteOptionsSchema();
    return {
        schema: anyOfSchema(nullSchema, allOfSchema(objectSchema({
            properties: {
                execute: executeObjSchema,
                format: await getFrontMatterFormatSchema()
            },
            description: "be a Quarto YAML front matter object"
        }), objectRefSchemaFromContextGlob("document-*", (field)=>field.name !== "format"), executeObjSchema)),
        errorHandlers: []
    };
}, "front-matter");
function Diff() {}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
        var callback = options.callback;
        if (typeof options === 'function') {
            callback = options;
            options = {};
        }
        this.options = options;
        var self1 = this;
        function done(value) {
            return value;
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        if (options.maxEditLength) {
            maxEditLength = Math.min(maxEditLength, options.maxEditLength);
        }
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath = bestPath[diagonalPath - 1], removePath = bestPath[diagonalPath + 1], _oldPos = (removePath ? removePath.newPos : 0) - diagonalPath;
                if (addPath) {
                    bestPath[diagonalPath - 1] = undefined;
                }
                var canAdd = addPath && addPath.newPos + 1 < newLen, canRemove = removePath && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = undefined;
                    continue;
                }
                if (!canAdd || canRemove && addPath.newPos < removePath.newPos) {
                    basePath = clonePath(removePath);
                    self1.pushComponent(basePath.components, undefined, true);
                } else {
                    basePath = addPath;
                    basePath.newPos++;
                    self1.pushComponent(basePath.components, true, undefined);
                }
                _oldPos = self1.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self1, basePath.components, newString, oldString, self1.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        while(editLength <= maxEditLength){
            var ret = execEditLength();
            if (callback && callback(ret, editLength, maxEditLength)) {
                return ret;
            }
            if (ret) {
                return ret;
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last = components[components.length - 1];
        if (last && last.added === added && last.removed === removed) {
            components[components.length - 1] = {
                count: last.count + 1,
                added: added,
                removed: removed
            };
        } else {
            components.push({
                count: 1,
                added: added,
                removed: removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array) {
        var ret = [];
        for(var i1 = 0; i1 < array.length; i1++){
            if (array[i1]) {
                ret.push(array[i1]);
            }
        }
        return ret;
    },
    castInput: function castInput(value) {
        return value;
    },
    tokenize: function tokenize(value) {
        return value.split('');
    },
    join: function join(chars) {
        return chars.join('');
    }
};
function buildValues(diff, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value = newString.slice(newPos, newPos + component.count);
                value = value.map(function(value, i1) {
                    var oldValue = oldString[oldPos + i1];
                    return oldValue.length > value.length ? oldValue : value;
                });
                component.value = diff.join(value);
            } else {
                component.value = diff.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === 'string' && (lastComponent.added || lastComponent.removed) && diff.equals('', lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path) {
    return {
        newPos: path.newPos,
        components: path.components.slice(0)
    };
}
new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace.test(left) && !reWhitespace.test(right);
};
wordDiff.tokenize = function(value) {
    var tokens = value.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i1 = 0; i1 < tokens.length - 1; i1++){
        if (!tokens[i1 + 1] && tokens[i1 + 2] && extendedWordChars.test(tokens[i1]) && extendedWordChars.test(tokens[i1 + 2])) {
            tokens[i1] += tokens[i1 + 2];
            tokens.splice(i1 + 1, 2);
            i1--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value) {
    var retLines = [], linesAndNewlines = value.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i1 = 0; i1 < linesAndNewlines.length; i1++){
        var line = linesAndNewlines[i1];
        if (i1 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line;
        } else {
            if (this.options.ignoreWhitespace) {
                line = line.trim();
            }
            retLines.push(line);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value) {
    return value.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value) {
    return value.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj) {
            return typeof obj;
        };
    } else {
        _typeof = function(obj) {
            return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === 'undefined' ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value === 'string' ? value : JSON.stringify(canonicalize(value, null, null, stringifyReplacer), stringifyReplacer, '  ');
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, '$1'), right.replace(/,([\r\n])/g, '$1'));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i1;
    for(i1 = 0; i1 < stack.length; i1 += 1){
        if (stack[i1] === obj) {
            return replacementStack[i1];
        }
    }
    var canonicalizedObj;
    if ('[object Array]' === objectPrototypeToString.call(obj)) {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i1 = 0; i1 < obj.length; i1 += 1){
            canonicalizedObj[i1] = canonicalize(obj[i1], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === 'object' && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {};
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i1 = 0; i1 < sortedKeys.length; i1 += 1){
            _key = sortedKeys[i1];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value) {
    return value.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value) {
    return value;
};
function deferred() {
    let methods;
    let state = "pending";
    const promise = new Promise((resolve, reject)=>{
        methods = {
            async resolve (value) {
                await value;
                state = "fulfilled";
                resolve(value);
            },
            reject (reason) {
                state = "rejected";
                reject(reason);
            }
        };
    });
    Object.defineProperty(promise, "state", {
        get: ()=>state
    });
    return Object.assign(promise, methods);
}
class MuxAsyncIterator {
    #iteratorCount = 0;
    #yields = [];
    #throws = [];
    #signal = deferred();
    add(iterable) {
        ++this.#iteratorCount;
        this.#callIteratorNext(iterable[Symbol.asyncIterator]());
    }
    async #callIteratorNext(iterator) {
        try {
            const { value , done  } = await iterator.next();
            if (done) {
                --this.#iteratorCount;
            } else {
                this.#yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.#throws.push(e);
        }
        this.#signal.resolve();
    }
    async *iterate() {
        while(this.#iteratorCount > 0){
            await this.#signal;
            for(let i1 = 0; i1 < this.#yields.length; i1++){
                const { iterator , value  } = this.#yields[i1];
                yield value;
                this.#callIteratorNext(iterator);
            }
            if (this.#throws.length) {
                for (const e of this.#throws){
                    throw e;
                }
                this.#throws.length = 0;
            }
            this.#yields.length = 0;
            this.#signal = deferred();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
const ERROR_WHILE_MAPPING_MESSAGE = "Threw while mapping.";
function pooledMap(poolLimit, array, iteratorFn) {
    const res = new TransformStream({
        async transform (p, controller) {
            try {
                const s = await p;
                controller.enqueue(s);
            } catch (e) {
                if (e instanceof AggregateError && e.message == ERROR_WHILE_MAPPING_MESSAGE) {
                    controller.error(e);
                }
            }
        }
    });
    (async ()=>{
        const writer = res.writable.getWriter();
        const executing = [];
        try {
            for await (const item of array){
                const p = Promise.resolve().then(()=>iteratorFn(item));
                writer.write(p);
                const e = p.then(()=>executing.splice(executing.indexOf(e), 1));
                executing.push(e);
                if (executing.length >= poolLimit) {
                    await Promise.race(executing);
                }
            }
            await Promise.all(executing);
            writer.close();
        } catch  {
            const errors = [];
            for (const result1 of (await Promise.allSettled(executing))){
                if (result1.status == "rejected") {
                    errors.push(result1.reason);
                }
            }
            writer.write(Promise.reject(new AggregateError(errors, ERROR_WHILE_MAPPING_MESSAGE))).catch(()=>{});
        }
    })();
    return res.readable[Symbol.asyncIterator]();
}
const MAX_SIZE1 = 2 ** 32 - 2;
class Buffer2 {
    #buf;
    #off = 0;
    #readable = new ReadableStream({
        type: "bytes",
        pull: (controller)=>{
            const view = new Uint8Array(controller.byobRequest.view.buffer);
            if (this.empty()) {
                this.reset();
                controller.close();
                controller.byobRequest.respond(0);
                return;
            }
            const nread = copy(this.#buf.subarray(this.#off), view);
            this.#off += nread;
            controller.byobRequest.respond(nread);
        },
        autoAllocateChunkSize: 16_640
    });
    get readable() {
        return this.#readable;
    }
    #writable = new WritableStream({
        write: (chunk)=>{
            const m = this.#grow(chunk.byteLength);
            copy(chunk, this.#buf, m);
        }
    });
    get writable() {
        return this.#writable;
    }
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice(n2) {
        const l2 = this.#buf.byteLength;
        if (n2 <= this.capacity - l2) {
            this.#reslice(l2 + n2);
            return l2;
        }
        return -1;
    }
    #reslice(len1) {
        assert(len1 <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len1);
    }
    #grow(n11) {
        const m1 = this.length;
        if (m1 === 0 && this.#off !== 0) {
            this.reset();
        }
        const i4 = this.#tryGrowByReslice(n11);
        if (i4 >= 0) {
            return i4;
        }
        const c1 = this.capacity;
        if (n11 <= Math.floor(c1 / 2) - m1) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c1 + n11 > MAX_SIZE1) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf1 = new Uint8Array(Math.min(2 * c1 + n11, MAX_SIZE1));
            copy(this.#buf.subarray(this.#off), buf1);
            this.#buf = buf1;
        }
        this.#off = 0;
        this.#reslice(Math.min(m1 + n11, MAX_SIZE1));
        return m1;
    }
    grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
async function readAll(r) {
    const buf = new Buffer1();
    await buf.readFrom(r);
    return buf.bytes();
}
function readAllSync(r) {
    const buf = new Buffer1();
    buf.readFromSync(r);
    return buf.bytes();
}
async function writeAll(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += await w.write(arr.subarray(nwritten));
    }
}
function writeAllSync(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += w.writeSync(arr.subarray(nwritten));
    }
}
async function* iterateReader(r, options) {
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result1 = await r.read(b);
        if (result1 === null) {
            break;
        }
        yield b.subarray(0, result1);
    }
}
async function copy2(src, dst, options) {
    let n = 0;
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    let gotEOF = false;
    while(gotEOF === false){
        const result1 = await src.read(b);
        if (result1 === null) {
            gotEOF = true;
        } else {
            let nwritten = 0;
            while(nwritten < result1){
                nwritten += await dst.write(b.subarray(nwritten, result1));
            }
            n += nwritten;
        }
    }
    return n;
}
const CR1 = "\r".charCodeAt(0);
const LF1 = "\n".charCodeAt(0);
class LineStream extends TransformStream {
    #bufs = new BytesList();
    #prevHadCR = false;
    constructor(){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#mergeBufs(false));
            }
        });
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf(LF1);
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            if (lfIndex === 0) {
                controller.enqueue(this.#mergeBufs(true));
                this.#handle(chunk.subarray(1), controller);
                return;
            }
        }
        if (lfIndex === -1) {
            if (chunk.at(-1) === CR1) {
                this.#prevHadCR = true;
            }
            this.#bufs.add(chunk);
        } else {
            let crOrLfIndex = lfIndex;
            if (chunk[lfIndex - 1] === CR1) {
                crOrLfIndex--;
            }
            this.#bufs.add(chunk.subarray(0, crOrLfIndex));
            controller.enqueue(this.#mergeBufs(false));
            this.#handle(chunk.subarray(lfIndex + 1), controller);
        }
    }
    #mergeBufs(prevHadCR) {
        const mergeBuf = this.#bufs.concat();
        this.#bufs = new BytesList();
        if (prevHadCR) {
            return mergeBuf.subarray(0, -1);
        } else {
            return mergeBuf;
        }
    }
}
class DelimiterStream extends TransformStream {
    #bufs = new BytesList();
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLen;
    #delimLPS;
    constructor(delimiter){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#bufs.concat());
            }
        });
        this.#delimiter = delimiter;
        this.#delimLen = delimiter.length;
        this.#delimLPS = createLPS(delimiter);
    }
    #handle(chunk2, controller2) {
        this.#bufs.add(chunk2);
        let localIndex = 0;
        while(this.#inspectIndex < this.#bufs.size()){
            if (chunk2[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimLen) {
                    const matchEnd = this.#inspectIndex - this.#delimLen;
                    const readyBytes = this.#bufs.slice(0, matchEnd);
                    controller2.enqueue(readyBytes);
                    this.#bufs.shift(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            } else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                } else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i1 = 1;
    while(i1 < lps.length){
        if (pat[i1] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i1] = prefixEnd;
            i1++;
        } else if (prefixEnd === 0) {
            lps[i1] = 0;
            i1++;
        } else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
async function execProcess(options, stdin, mergeOutput, stderrFilter, respectStreams) {
    try {
        const process = Deno.run({
            ...options,
            stdin: stdin !== undefined ? "piped" : options.stdin,
            stdout: typeof options.stdout === "number" ? options.stdout : "piped",
            stderr: typeof options.stderr === "number" ? options.stderr : "piped"
        });
        if (stdin !== undefined) {
            if (!process.stdin) {
                throw new Error("Process stdin not available");
            }
            const buffer = new TextEncoder().encode(stdin);
            let offset = 0;
            while(offset < buffer.length){
                const end = Math.min(offset + 4096, buffer.length);
                const window1 = buffer.subarray(offset, end);
                const written = await process.stdin.write(window1);
                offset += written;
            }
            process.stdin.close();
        }
        let stdoutText = "";
        let stderrText = "";
        if (mergeOutput) {
            const multiplexIterator = new MuxAsyncIterator();
            const addStream = (stream, filter)=>{
                if (stream !== null) {
                    const streamIter = filter ? filteredAsyncIterator(iterateReader(stream), filter) : iterateReader(stream);
                    multiplexIterator.add(streamIter);
                }
            };
            addStream(process.stdout);
            addStream(process.stderr, stderrFilter);
            const allOutput = await processOutput(multiplexIterator, mergeOutput === "stderr>stdout" ? options.stdout : options.stderr);
            if (mergeOutput === "stderr>stdout") {
                stdoutText = allOutput;
            } else {
                stderrText = allOutput;
            }
            const closeStream = (stream)=>{
                if (stream) {
                    stream.close();
                }
            };
            closeStream(process.stdout);
            closeStream(process.stderr);
        } else {
            if (process.stdout !== null) {
                stdoutText = await processOutput(iterateReader(process.stdout), options.stdout, respectStreams ? "stdout" : undefined);
                process.stdout.close();
            }
            if (process.stderr != null) {
                const iterator = stderrFilter ? filteredAsyncIterator(iterateReader(process.stderr), stderrFilter) : iterateReader(process.stderr);
                stderrText = await processOutput(iterator, options.stderr, respectStreams ? "stderr" : undefined);
                process.stderr.close();
            }
        }
        const status = await process.status();
        process.close();
        return {
            success: status.success,
            code: status.code,
            stdout: stdoutText,
            stderr: stderrText
        };
    } catch (e) {
        throw new Error(`Error executing '${options.cmd[0]}': ${e.message}`);
    }
}
function processSuccessResult() {
    return {
        success: true,
        code: 0
    };
}
function filteredAsyncIterator(iterator, filter) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    return pooledMap(1, iterator, (data)=>{
        return Promise.resolve(encoder.encode(filter(decoder.decode(data))));
    });
}
async function processOutput(iterator, output, which) {
    const decoder = new TextDecoder();
    let outputText = "";
    for await (const chunk of iterator){
        if (output === "inherit" || output === undefined) {
            if (which === "stdout") {
                Deno.stdout.writeSync(chunk);
            } else if (which === "stderr") {
                Deno.stderr.writeSync(chunk);
            } else {
                info(decoder.decode(chunk), {
                    newline: false
                });
            }
        }
        const text = decoder.decode(chunk);
        outputText += text;
    }
    return outputText;
}
const nodeStack = [
    {
        name: "global",
        children: [
            {
                name: "deno-init",
                start: 0,
                end: performance.now(),
                elapsed: performance.now(),
                children: []
            }
        ],
        start: 0,
        end: 0
    }
];
function withTiming(name, fun) {
    push(name);
    try {
        const result1 = fun();
        return result1;
    } finally{
        pop();
    }
}
async function withTimingAsync(name, fun) {
    push(name);
    try {
        const result1 = await fun();
        pop();
        return result1;
    } catch (e) {
        pop();
        throw e;
    }
}
function push(name) {
    nodeStack.push({
        name,
        start: performance.now(),
        end: 0,
        children: []
    });
}
function pop(at) {
    if (nodeStack.length <= 1) {
        throw new Error("Can't pop top node");
    }
    const node = nodeStack.pop();
    node.end = at ?? performance.now();
    node.elapsed = node.end - node.start;
    nodeStack[nodeStack.length - 1].children.push(node);
}
function getData() {
    nodeStack[0].end = performance.now();
    nodeStack[0].elapsed = nodeStack[0].end - nodeStack[0].start;
    return nodeStack[0];
}
function insertExplicitTimingEntries(start, end, entries, groupName) {
    const [theirStart, ourStart] = start;
    const [theirEnd, _] = end;
    nodeStack.push({
        name: groupName,
        start: ourStart,
        end: 0,
        children: []
    });
    let prevTime = ourStart;
    if (entries.length && entries[0].name === "_start") {
        prevTime = entries[0].time - theirStart + ourStart;
        entries = entries.slice(1);
    }
    for (const entry of entries){
        nodeStack.push({
            name: entry.name,
            start: prevTime,
            end: 0,
            children: []
        });
        prevTime = entry.time - theirStart + ourStart;
        pop(prevTime);
    }
    prevTime = theirEnd - theirStart + ourStart;
    pop(prevTime);
}
async function getLuaTiming() {
    if (Deno.env.get("QUARTO_PROFILER_OUTPUT")) {
        return [
            Number((await execProcess({
                cmd: [
                    "python",
                    "-c",
                    "import time; print(time.time() * 1000)"
                ],
                stdout: "piped"
            })).stdout),
            performance.now(), 
        ];
    } else {
        return [
            0,
            0
        ];
    }
}
function mappedDiff(source, target) {
    return withTiming("mapped-diff", ()=>{
        const sourceLineRanges = rangedLines(source.value).map((x)=>x.range);
        let sourceCursor = 0;
        const resultChunks = [];
        const started = performance.now();
        const diffResult = diffLines(source.value, target, ()=>{
            const now = performance.now();
            if (now - started > 2000) {
                return true;
            }
        });
        if (diffResult === undefined) {
            warning("Warning: diff of engine output timed out. No source lines will be available.");
            return asMappedString(target);
        }
        for (const action of diffResult){
            if (action.removed) {
                sourceCursor += action.count;
            } else if (action.added) {
                resultChunks.push(action.value);
            } else {
                const start = sourceLineRanges[sourceCursor].start;
                const nextCursor = sourceCursor + action.count;
                const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
                sourceCursor = nextCursor;
                resultChunks.push({
                    start,
                    end
                });
            }
        }
        return mappedString(source, resultChunks, source.fileName);
    });
}
function mappedStringFromFile(filename) {
    const value = Deno.readTextFileSync(filename);
    if (filename.startsWith("/")) {
        filename = relative2(Deno.cwd(), filename);
    }
    return asMappedString(value, filename);
}
const kRegExBeginYAML = /^---[ \t]*$/;
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function isYamlPath(file) {
    return [
        ".yaml",
        ".yml"
    ].includes(extname2(file));
}
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        const result1 = parseWithNiceErrors(decoder.decode(yml));
        try {
            JSON.stringify(result1);
            return result1;
        } catch (e) {
            throw new Error(`Circular structures not allowed.\nFile ${file}\n${e.message.split("\n").slice(1).join("\n")}`);
        }
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function readYamlFromString(yml) {
    return parseWithNiceErrors(yml);
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            let yamlBlock = removeYamlDelimiters(match[2]);
            yamlBlock = lines(yamlBlock).map((x)=>x.trimEnd()).join("\n");
            if (!yamlBlock.match(/^\n\s*\n/m) && !yamlBlock.match(/^\n\s*\n---/m) && yamlBlock.trim().length > 0) {
                parseWithNiceErrors(yamlBlock, {
                    json: true,
                    schema: QuartoJSONSchema1
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata = parseWithNiceErrors(yaml, {
            json: true,
            schema: QuartoJSONSchema1
        });
        return metadata || {};
    } else {
        return {};
    }
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line, index)=>index > 0 && line.match(kRegExEndYAML));
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
class YAMLValidationError extends ErrorEx {
    constructor(message){
        super("YAMLValidationError", message, false, false);
    }
}
const QuartoJSONSchema1 = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ],
    explicit: [
        new Type1("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        }), 
    ]
});
function parseWithNiceErrors(content, options) {
    try {
        return parse3(content, options || {
            json: true,
            schema: QuartoJSONSchema1
        });
    } catch (e) {
        throw improveYamlParseErrorMessage(e);
    }
}
function improveYamlParseErrorMessage(e) {
    if (e.message.match(/unknown tag/)) {
        e.message = `${e.message}\nDid you try to use a '!' in a YAML string? If so, you need to add explicit quotes to your string.`;
    }
    return e;
}
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
var __VIRTUAL_FILE = listCacheClear;
function eq(value, other) {
    return value === other || value !== value && other !== other;
}
var __VIRTUAL_FILE1 = eq;
function assocIndexOf(array, key) {
    var length = array.length;
    while(length--){
        if (__VIRTUAL_FILE1(array[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var __VIRTUAL_FILE2 = assocIndexOf;
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data.length - 1;
    if (index == lastIndex) {
        data.pop();
    } else {
        splice.call(data, index, 1);
    }
    --this.size;
    return true;
}
var __VIRTUAL_FILE3 = listCacheDelete;
function listCacheGet(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    return index < 0 ? void 0 : data[index][1];
}
var __VIRTUAL_FILE4 = listCacheGet;
function listCacheHas(key) {
    return __VIRTUAL_FILE2(this.__data__, key) > -1;
}
var __VIRTUAL_FILE5 = listCacheHas;
function listCacheSet(key, value) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        ++this.size;
        data.push([
            key,
            value
        ]);
    } else {
        data[index][1] = value;
    }
    return this;
}
var __VIRTUAL_FILE6 = listCacheSet;
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = __VIRTUAL_FILE;
ListCache.prototype["delete"] = __VIRTUAL_FILE3;
ListCache.prototype.get = __VIRTUAL_FILE4;
ListCache.prototype.has = __VIRTUAL_FILE5;
ListCache.prototype.set = __VIRTUAL_FILE6;
var __VIRTUAL_FILE7 = ListCache;
function stackClear() {
    this.__data__ = new __VIRTUAL_FILE7();
    this.size = 0;
}
var __VIRTUAL_FILE8 = stackClear;
function stackDelete(key) {
    var data = this.__data__, result1 = data["delete"](key);
    this.size = data.size;
    return result1;
}
var __VIRTUAL_FILE9 = stackDelete;
function stackGet(key) {
    return this.__data__.get(key);
}
var __VIRTUAL_FILE10 = stackGet;
function stackHas(key) {
    return this.__data__.has(key);
}
var __VIRTUAL_FILE11 = stackHas;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
var freeGlobal = typeof commonjsGlobal == "object" && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;
var __VIRTUAL_FILE12 = freeGlobal;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = __VIRTUAL_FILE12 || freeSelf || Function("return this")();
var __VIRTUAL_FILE13 = root;
var Symbol1 = __VIRTUAL_FILE13.Symbol;
var __VIRTUAL_FILE14 = Symbol1;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag), tag = value[symToStringTag];
    try {
        value[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {}
    var result1 = nativeObjectToString.call(value);
    if (unmasked) {
        if (isOwn) {
            value[symToStringTag] = tag;
        } else {
            delete value[symToStringTag];
        }
    }
    return result1;
}
var __VIRTUAL_FILE15 = getRawTag;
var objectProto1 = Object.prototype;
var nativeObjectToString1 = objectProto1.toString;
function objectToString(value) {
    return nativeObjectToString1.call(value);
}
var __VIRTUAL_FILE16 = objectToString;
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function baseGetTag(value) {
    if (value == null) {
        return value === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag1 && symToStringTag1 in Object(value) ? __VIRTUAL_FILE15(value) : __VIRTUAL_FILE16(value);
}
var __VIRTUAL_FILE17 = baseGetTag;
function isObject3(value) {
    var type = typeof value;
    return value != null && (type == "object" || type == "function");
}
var __VIRTUAL_FILE18 = isObject3;
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value) {
    if (!__VIRTUAL_FILE18(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
}
var __VIRTUAL_FILE19 = isFunction;
var coreJsData = __VIRTUAL_FILE13["__core-js_shared__"];
var __VIRTUAL_FILE20 = coreJsData;
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(__VIRTUAL_FILE20 && __VIRTUAL_FILE20.keys && __VIRTUAL_FILE20.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func) {
    return !!maskSrcKey && maskSrcKey in func;
}
var __VIRTUAL_FILE21 = isMasked;
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func) {
    if (func != null) {
        try {
            return funcToString.call(func);
        } catch (e) {}
        try {
            return func + "";
        } catch (e1) {}
    }
    return "";
}
var __VIRTUAL_FILE22 = toSource;
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto1 = Function.prototype, objectProto2 = Object.prototype;
var funcToString1 = funcProto1.toString;
var hasOwnProperty1 = objectProto2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString1.call(hasOwnProperty1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value) {
    if (!__VIRTUAL_FILE18(value) || __VIRTUAL_FILE21(value)) {
        return false;
    }
    var pattern = __VIRTUAL_FILE19(value) ? reIsNative : reIsHostCtor;
    return pattern.test(__VIRTUAL_FILE22(value));
}
var __VIRTUAL_FILE23 = baseIsNative;
function getValue(object, key) {
    return object == null ? void 0 : object[key];
}
var __VIRTUAL_FILE24 = getValue;
function getNative(object, key) {
    var value = __VIRTUAL_FILE24(object, key);
    return __VIRTUAL_FILE23(value) ? value : void 0;
}
var __VIRTUAL_FILE25 = getNative;
var Map1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Map");
var __VIRTUAL_FILE26 = Map1;
var nativeCreate = __VIRTUAL_FILE25(Object, "create");
var __VIRTUAL_FILE27 = nativeCreate;
function hashClear() {
    this.__data__ = __VIRTUAL_FILE27 ? __VIRTUAL_FILE27(null) : {};
    this.size = 0;
}
var __VIRTUAL_FILE28 = hashClear;
function hashDelete(key) {
    var result1 = this.has(key) && delete this.__data__[key];
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE29 = hashDelete;
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto3 = Object.prototype;
var hasOwnProperty2 = objectProto3.hasOwnProperty;
function hashGet(key) {
    var data = this.__data__;
    if (__VIRTUAL_FILE27) {
        var result1 = data[key];
        return result1 === HASH_UNDEFINED ? void 0 : result1;
    }
    return hasOwnProperty2.call(data, key) ? data[key] : void 0;
}
var __VIRTUAL_FILE30 = hashGet;
var objectProto4 = Object.prototype;
var hasOwnProperty3 = objectProto4.hasOwnProperty;
function hashHas(key) {
    var data = this.__data__;
    return __VIRTUAL_FILE27 ? data[key] !== void 0 : hasOwnProperty3.call(data, key);
}
var __VIRTUAL_FILE31 = hashHas;
var HASH_UNDEFINED1 = "__lodash_hash_undefined__";
function hashSet(key, value) {
    var data = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data[key] = __VIRTUAL_FILE27 && value === void 0 ? HASH_UNDEFINED1 : value;
    return this;
}
var __VIRTUAL_FILE32 = hashSet;
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = __VIRTUAL_FILE28;
Hash.prototype["delete"] = __VIRTUAL_FILE29;
Hash.prototype.get = __VIRTUAL_FILE30;
Hash.prototype.has = __VIRTUAL_FILE31;
Hash.prototype.set = __VIRTUAL_FILE32;
var __VIRTUAL_FILE33 = Hash;
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new __VIRTUAL_FILE33(),
        map: new (__VIRTUAL_FILE26 || __VIRTUAL_FILE7)(),
        string: new __VIRTUAL_FILE33()
    };
}
var __VIRTUAL_FILE34 = mapCacheClear;
function isKeyable(value) {
    var type = typeof value;
    return type == "string" || type == "number" || type == "symbol" || type == "boolean" ? value !== "__proto__" : value === null;
}
var __VIRTUAL_FILE35 = isKeyable;
function getMapData(map, key) {
    var data = map.__data__;
    return __VIRTUAL_FILE35(key) ? data[typeof key == "string" ? "string" : "hash"] : data.map;
}
var __VIRTUAL_FILE36 = getMapData;
function mapCacheDelete(key) {
    var result1 = __VIRTUAL_FILE36(this, key)["delete"](key);
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE37 = mapCacheDelete;
function mapCacheGet(key) {
    return __VIRTUAL_FILE36(this, key).get(key);
}
var __VIRTUAL_FILE38 = mapCacheGet;
function mapCacheHas(key) {
    return __VIRTUAL_FILE36(this, key).has(key);
}
var __VIRTUAL_FILE39 = mapCacheHas;
function mapCacheSet(key, value) {
    var data = __VIRTUAL_FILE36(this, key), size = data.size;
    data.set(key, value);
    this.size += data.size == size ? 0 : 1;
    return this;
}
var __VIRTUAL_FILE40 = mapCacheSet;
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = __VIRTUAL_FILE34;
MapCache.prototype["delete"] = __VIRTUAL_FILE37;
MapCache.prototype.get = __VIRTUAL_FILE38;
MapCache.prototype.has = __VIRTUAL_FILE39;
MapCache.prototype.set = __VIRTUAL_FILE40;
var __VIRTUAL_FILE41 = MapCache;
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value) {
    var data = this.__data__;
    if (data instanceof __VIRTUAL_FILE7) {
        var pairs = data.__data__;
        if (!__VIRTUAL_FILE26 || pairs.length < LARGE_ARRAY_SIZE - 1) {
            pairs.push([
                key,
                value
            ]);
            this.size = ++data.size;
            return this;
        }
        data = this.__data__ = new __VIRTUAL_FILE41(pairs);
    }
    data.set(key, value);
    this.size = data.size;
    return this;
}
var __VIRTUAL_FILE42 = stackSet;
function Stack(entries) {
    var data = this.__data__ = new __VIRTUAL_FILE7(entries);
    this.size = data.size;
}
Stack.prototype.clear = __VIRTUAL_FILE8;
Stack.prototype["delete"] = __VIRTUAL_FILE9;
Stack.prototype.get = __VIRTUAL_FILE10;
Stack.prototype.has = __VIRTUAL_FILE11;
Stack.prototype.set = __VIRTUAL_FILE42;
var __VIRTUAL_FILE43 = Stack;
function arrayEach(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (iteratee(array[index], index, array) === false) {
            break;
        }
    }
    return array;
}
var __VIRTUAL_FILE44 = arrayEach;
var defineProperty = function() {
    try {
        var func = __VIRTUAL_FILE25(Object, "defineProperty");
        func({}, "", {});
        return func;
    } catch (e) {}
}();
var __VIRTUAL_FILE45 = defineProperty;
function baseAssignValue(object, key, value) {
    if (key == "__proto__" && __VIRTUAL_FILE45) {
        __VIRTUAL_FILE45(object, key, {
            configurable: true,
            enumerable: true,
            value,
            writable: true
        });
    } else {
        object[key] = value;
    }
}
var __VIRTUAL_FILE46 = baseAssignValue;
var objectProto5 = Object.prototype;
var hasOwnProperty4 = objectProto5.hasOwnProperty;
function assignValue(object, key, value) {
    var objValue = object[key];
    if (!(hasOwnProperty4.call(object, key) && __VIRTUAL_FILE1(objValue, value)) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE47 = assignValue;
function copyObject(source, props, object, customizer) {
    var isNew = !object;
    object || (object = {});
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object[key], source[key], key, object, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            __VIRTUAL_FILE46(object, key, newValue);
        } else {
            __VIRTUAL_FILE47(object, key, newValue);
        }
    }
    return object;
}
var __VIRTUAL_FILE48 = copyObject;
function baseTimes(n, iteratee) {
    var index = -1, result1 = Array(n);
    while(++index < n){
        result1[index] = iteratee(index);
    }
    return result1;
}
var __VIRTUAL_FILE49 = baseTimes;
function isObjectLike(value) {
    return value != null && typeof value == "object";
}
var __VIRTUAL_FILE50 = isObjectLike;
var argsTag = "[object Arguments]";
function baseIsArguments(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == argsTag;
}
var __VIRTUAL_FILE51 = baseIsArguments;
var objectProto6 = Object.prototype;
var hasOwnProperty5 = objectProto6.hasOwnProperty;
var propertyIsEnumerable = objectProto6.propertyIsEnumerable;
var isArguments = __VIRTUAL_FILE51(function() {
    return arguments;
}()) ? __VIRTUAL_FILE51 : function(value) {
    return __VIRTUAL_FILE50(value) && hasOwnProperty5.call(value, "callee") && !propertyIsEnumerable.call(value, "callee");
};
var __VIRTUAL_FILE52 = isArguments;
var isArray = Array.isArray;
var __VIRTUAL_FILE53 = isArray;
function stubFalse() {
    return false;
}
var __VIRTUAL_FILE54 = stubFalse;
function createCommonjsModule(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE55 = createCommonjsModule(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0;
    var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
    var isBuffer = nativeIsBuffer || __VIRTUAL_FILE54;
    module.exports = isBuffer;
});
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value, length) {
    var type = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type == "number" || type != "symbol" && reIsUint.test(value)) && value > -1 && value % 1 == 0 && value < length;
}
var __VIRTUAL_FILE56 = isIndex;
var MAX_SAFE_INTEGER1 = 9007199254740991;
function isLength(value) {
    return typeof value == "number" && value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER1;
}
var __VIRTUAL_FILE57 = isLength;
var argsTag1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE57(value.length) && !!typedArrayTags[__VIRTUAL_FILE17(value)];
}
var __VIRTUAL_FILE58 = baseIsTypedArray;
function baseUnary(func) {
    return function(value) {
        return func(value);
    };
}
var __VIRTUAL_FILE59 = baseUnary;
function createCommonjsModule1(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE60 = createCommonjsModule1(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var freeProcess = moduleExports && __VIRTUAL_FILE12.process;
    var nodeUtil = function() {
        try {
            var types = freeModule && freeModule.require && freeModule.require("util").types;
            if (types) {
                return types;
            }
            return freeProcess && freeProcess.binding && freeProcess.binding("util");
        } catch (e) {}
    }();
    module.exports = nodeUtil;
});
var nodeIsTypedArray = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isTypedArray;
var isTypedArray = nodeIsTypedArray ? __VIRTUAL_FILE59(nodeIsTypedArray) : __VIRTUAL_FILE58;
var __VIRTUAL_FILE61 = isTypedArray;
var objectProto7 = Object.prototype;
var hasOwnProperty6 = objectProto7.hasOwnProperty;
function arrayLikeKeys(value, inherited) {
    var isArr = __VIRTUAL_FILE53(value), isArg = !isArr && __VIRTUAL_FILE52(value), isBuff = !isArr && !isArg && __VIRTUAL_FILE55(value), isType = !isArr && !isArg && !isBuff && __VIRTUAL_FILE61(value), skipIndexes = isArr || isArg || isBuff || isType, result1 = skipIndexes ? __VIRTUAL_FILE49(value.length, String) : [], length = result1.length;
    for(var key in value){
        if ((inherited || hasOwnProperty6.call(value, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || __VIRTUAL_FILE56(key, length)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE62 = arrayLikeKeys;
var objectProto8 = Object.prototype;
function isPrototype(value) {
    var Ctor = value && value.constructor, proto = typeof Ctor == "function" && Ctor.prototype || objectProto8;
    return value === proto;
}
var __VIRTUAL_FILE63 = isPrototype;
function overArg(func, transform) {
    return function(arg) {
        return func(transform(arg));
    };
}
var __VIRTUAL_FILE64 = overArg;
var nativeKeys = __VIRTUAL_FILE64(Object.keys, Object);
var __VIRTUAL_FILE65 = nativeKeys;
var objectProto9 = Object.prototype;
var hasOwnProperty7 = objectProto9.hasOwnProperty;
function baseKeys(object) {
    if (!__VIRTUAL_FILE63(object)) {
        return __VIRTUAL_FILE65(object);
    }
    var result1 = [];
    for(var key in Object(object)){
        if (hasOwnProperty7.call(object, key) && key != "constructor") {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE66 = baseKeys;
function isArrayLike(value) {
    return value != null && __VIRTUAL_FILE57(value.length) && !__VIRTUAL_FILE19(value);
}
var __VIRTUAL_FILE67 = isArrayLike;
function keys(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object) : __VIRTUAL_FILE66(object);
}
var __VIRTUAL_FILE68 = keys;
function baseAssign(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE68(source), object);
}
var __VIRTUAL_FILE69 = baseAssign;
function nativeKeysIn(object) {
    var result1 = [];
    if (object != null) {
        for(var key in Object(object)){
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE70 = nativeKeysIn;
var objectProto10 = Object.prototype;
var hasOwnProperty8 = objectProto10.hasOwnProperty;
function baseKeysIn(object) {
    if (!__VIRTUAL_FILE18(object)) {
        return __VIRTUAL_FILE70(object);
    }
    var isProto = __VIRTUAL_FILE63(object), result1 = [];
    for(var key in object){
        if (!(key == "constructor" && (isProto || !hasOwnProperty8.call(object, key)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE71 = baseKeysIn;
function keysIn(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object, true) : __VIRTUAL_FILE71(object);
}
var __VIRTUAL_FILE72 = keysIn;
function baseAssignIn(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object);
}
var __VIRTUAL_FILE73 = baseAssignIn;
function createCommonjsModule2(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE74 = createCommonjsModule2(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0, allocUnsafe = Buffer1 ? Buffer1.allocUnsafe : void 0;
    function cloneBuffer(buffer, isDeep) {
        if (isDeep) {
            return buffer.slice();
        }
        var length = buffer.length, result1 = allocUnsafe ? allocUnsafe(length) : new buffer.constructor(length);
        buffer.copy(result1);
        return result1;
    }
    module.exports = cloneBuffer;
});
function copyArray(source, array) {
    var index = -1, length = source.length;
    array || (array = Array(length));
    while(++index < length){
        array[index] = source[index];
    }
    return array;
}
var __VIRTUAL_FILE75 = copyArray;
function arrayFilter(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length, resIndex = 0, result1 = [];
    while(++index < length){
        var value = array[index];
        if (predicate(value, index, array)) {
            result1[resIndex++] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE76 = arrayFilter;
function stubArray() {
    return [];
}
var __VIRTUAL_FILE77 = stubArray;
var objectProto11 = Object.prototype;
var propertyIsEnumerable1 = objectProto11.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? __VIRTUAL_FILE77 : function(object) {
    if (object == null) {
        return [];
    }
    object = Object(object);
    return __VIRTUAL_FILE76(nativeGetSymbols(object), function(symbol) {
        return propertyIsEnumerable1.call(object, symbol);
    });
};
var __VIRTUAL_FILE78 = getSymbols;
function copySymbols(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE78(source), object);
}
var __VIRTUAL_FILE79 = copySymbols;
function arrayPush(array, values) {
    var index = -1, length = values.length, offset = array.length;
    while(++index < length){
        array[offset + index] = values[index];
    }
    return array;
}
var __VIRTUAL_FILE80 = arrayPush;
var getPrototype = __VIRTUAL_FILE64(Object.getPrototypeOf, Object);
var __VIRTUAL_FILE81 = getPrototype;
var nativeGetSymbols1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols1 ? __VIRTUAL_FILE77 : function(object) {
    var result1 = [];
    while(object){
        __VIRTUAL_FILE80(result1, __VIRTUAL_FILE78(object));
        object = __VIRTUAL_FILE81(object);
    }
    return result1;
};
var __VIRTUAL_FILE82 = getSymbolsIn;
function copySymbolsIn(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE82(source), object);
}
var __VIRTUAL_FILE83 = copySymbolsIn;
function baseGetAllKeys(object, keysFunc, symbolsFunc) {
    var result1 = keysFunc(object);
    return __VIRTUAL_FILE53(object) ? result1 : __VIRTUAL_FILE80(result1, symbolsFunc(object));
}
var __VIRTUAL_FILE84 = baseGetAllKeys;
function getAllKeys(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE68, __VIRTUAL_FILE78);
}
var __VIRTUAL_FILE85 = getAllKeys;
function getAllKeysIn(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE72, __VIRTUAL_FILE82);
}
var __VIRTUAL_FILE86 = getAllKeysIn;
var DataView1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "DataView");
var __VIRTUAL_FILE87 = DataView1;
var Promise1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Promise");
var __VIRTUAL_FILE88 = Promise1;
var Set1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Set");
var __VIRTUAL_FILE89 = Set1;
var WeakMap1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "WeakMap");
var __VIRTUAL_FILE90 = WeakMap1;
var mapTag1 = "[object Map]", objectTag1 = "[object Object]", promiseTag = "[object Promise]", setTag1 = "[object Set]", weakMapTag1 = "[object WeakMap]";
var dataViewTag1 = "[object DataView]";
var dataViewCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE87), mapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE26), promiseCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE88), setCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE89), weakMapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE90);
var getTag = __VIRTUAL_FILE17;
if (__VIRTUAL_FILE87 && getTag(new __VIRTUAL_FILE87(new ArrayBuffer(1))) != dataViewTag1 || __VIRTUAL_FILE26 && getTag(new __VIRTUAL_FILE26()) != mapTag1 || __VIRTUAL_FILE88 && getTag(__VIRTUAL_FILE88.resolve()) != promiseTag || __VIRTUAL_FILE89 && getTag(new __VIRTUAL_FILE89()) != setTag1 || __VIRTUAL_FILE90 && getTag(new __VIRTUAL_FILE90()) != weakMapTag1) {
    getTag = function(value) {
        var result1 = __VIRTUAL_FILE17(value), Ctor = result1 == objectTag1 ? value.constructor : void 0, ctorString = Ctor ? __VIRTUAL_FILE22(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag1;
                case mapCtorString:
                    return mapTag1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag1;
                case weakMapCtorString:
                    return weakMapTag1;
            }
        }
        return result1;
    };
}
var __VIRTUAL_FILE91 = getTag;
var objectProto12 = Object.prototype;
var hasOwnProperty9 = objectProto12.hasOwnProperty;
function initCloneArray(array) {
    var length = array.length, result1 = new array.constructor(length);
    if (length && typeof array[0] == "string" && hasOwnProperty9.call(array, "index")) {
        result1.index = array.index;
        result1.input = array.input;
    }
    return result1;
}
var __VIRTUAL_FILE92 = initCloneArray;
var Uint8Array1 = __VIRTUAL_FILE13.Uint8Array;
var __VIRTUAL_FILE93 = Uint8Array1;
function cloneArrayBuffer(arrayBuffer) {
    var result1 = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new __VIRTUAL_FILE93(result1).set(new __VIRTUAL_FILE93(arrayBuffer));
    return result1;
}
var __VIRTUAL_FILE94 = cloneArrayBuffer;
function cloneDataView(dataView, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}
var __VIRTUAL_FILE95 = cloneDataView;
var reFlags = /\w*$/;
function cloneRegExp(regexp) {
    var result1 = new regexp.constructor(regexp.source, reFlags.exec(regexp));
    result1.lastIndex = regexp.lastIndex;
    return result1;
}
var __VIRTUAL_FILE96 = cloneRegExp;
var symbolProto = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf = symbolProto ? symbolProto.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {};
}
var __VIRTUAL_FILE97 = cloneSymbol;
function cloneTypedArray(typedArray, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}
var __VIRTUAL_FILE98 = cloneTypedArray;
var boolTag1 = "[object Boolean]", dateTag1 = "[object Date]", mapTag2 = "[object Map]", numberTag1 = "[object Number]", regexpTag1 = "[object RegExp]", setTag2 = "[object Set]", stringTag1 = "[object String]", symbolTag = "[object Symbol]";
var arrayBufferTag1 = "[object ArrayBuffer]", dataViewTag2 = "[object DataView]", float32Tag1 = "[object Float32Array]", float64Tag1 = "[object Float64Array]", int8Tag1 = "[object Int8Array]", int16Tag1 = "[object Int16Array]", int32Tag1 = "[object Int32Array]", uint8Tag1 = "[object Uint8Array]", uint8ClampedTag1 = "[object Uint8ClampedArray]", uint16Tag1 = "[object Uint16Array]", uint32Tag1 = "[object Uint32Array]";
function initCloneByTag(object, tag, isDeep) {
    var Ctor = object.constructor;
    switch(tag){
        case arrayBufferTag1:
            return __VIRTUAL_FILE94(object);
        case boolTag1:
        case dateTag1:
            return new Ctor(+object);
        case dataViewTag2:
            return __VIRTUAL_FILE95(object, isDeep);
        case float32Tag1:
        case float64Tag1:
        case int8Tag1:
        case int16Tag1:
        case int32Tag1:
        case uint8Tag1:
        case uint8ClampedTag1:
        case uint16Tag1:
        case uint32Tag1:
            return __VIRTUAL_FILE98(object, isDeep);
        case mapTag2:
            return new Ctor();
        case numberTag1:
        case stringTag1:
            return new Ctor(object);
        case regexpTag1:
            return __VIRTUAL_FILE96(object);
        case setTag2:
            return new Ctor();
        case symbolTag:
            return __VIRTUAL_FILE97(object);
    }
}
var __VIRTUAL_FILE99 = initCloneByTag;
var objectCreate = Object.create;
var baseCreate = function() {
    function object() {}
    return function(proto) {
        if (!__VIRTUAL_FILE18(proto)) {
            return {};
        }
        if (objectCreate) {
            return objectCreate(proto);
        }
        object.prototype = proto;
        var result1 = new object();
        object.prototype = void 0;
        return result1;
    };
}();
var __VIRTUAL_FILE100 = baseCreate;
function initCloneObject(object) {
    return typeof object.constructor == "function" && !__VIRTUAL_FILE63(object) ? __VIRTUAL_FILE100(__VIRTUAL_FILE81(object)) : {};
}
var __VIRTUAL_FILE101 = initCloneObject;
var mapTag3 = "[object Map]";
function baseIsMap(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == mapTag3;
}
var __VIRTUAL_FILE102 = baseIsMap;
var nodeIsMap = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isMap;
var isMap = nodeIsMap ? __VIRTUAL_FILE59(nodeIsMap) : __VIRTUAL_FILE102;
var __VIRTUAL_FILE103 = isMap;
var setTag3 = "[object Set]";
function baseIsSet(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == setTag3;
}
var __VIRTUAL_FILE104 = baseIsSet;
var nodeIsSet = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isSet;
var isSet = nodeIsSet ? __VIRTUAL_FILE59(nodeIsSet) : __VIRTUAL_FILE104;
var __VIRTUAL_FILE105 = isSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag2 = "[object Arguments]", arrayTag1 = "[object Array]", boolTag2 = "[object Boolean]", dateTag2 = "[object Date]", errorTag1 = "[object Error]", funcTag2 = "[object Function]", genTag1 = "[object GeneratorFunction]", mapTag4 = "[object Map]", numberTag2 = "[object Number]", objectTag2 = "[object Object]", regexpTag2 = "[object RegExp]", setTag4 = "[object Set]", stringTag2 = "[object String]", symbolTag1 = "[object Symbol]", weakMapTag2 = "[object WeakMap]";
var arrayBufferTag2 = "[object ArrayBuffer]", dataViewTag3 = "[object DataView]", float32Tag2 = "[object Float32Array]", float64Tag2 = "[object Float64Array]", int8Tag2 = "[object Int8Array]", int16Tag2 = "[object Int16Array]", int32Tag2 = "[object Int32Array]", uint8Tag2 = "[object Uint8Array]", uint8ClampedTag2 = "[object Uint8ClampedArray]", uint16Tag2 = "[object Uint16Array]", uint32Tag2 = "[object Uint32Array]";
var cloneableTags = {};
cloneableTags[argsTag2] = cloneableTags[arrayTag1] = cloneableTags[arrayBufferTag2] = cloneableTags[dataViewTag3] = cloneableTags[boolTag2] = cloneableTags[dateTag2] = cloneableTags[float32Tag2] = cloneableTags[float64Tag2] = cloneableTags[int8Tag2] = cloneableTags[int16Tag2] = cloneableTags[int32Tag2] = cloneableTags[mapTag4] = cloneableTags[numberTag2] = cloneableTags[objectTag2] = cloneableTags[regexpTag2] = cloneableTags[setTag4] = cloneableTags[stringTag2] = cloneableTags[symbolTag1] = cloneableTags[uint8Tag2] = cloneableTags[uint8ClampedTag2] = cloneableTags[uint16Tag2] = cloneableTags[uint32Tag2] = true;
cloneableTags[errorTag1] = cloneableTags[funcTag2] = cloneableTags[weakMapTag2] = false;
function baseClone(value, bitmask, customizer, key, object, stack) {
    var result1, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result1 = object ? customizer(value, key, object, stack) : customizer(value);
    }
    if (result1 !== void 0) {
        return result1;
    }
    if (!__VIRTUAL_FILE18(value)) {
        return value;
    }
    var isArr = __VIRTUAL_FILE53(value);
    if (isArr) {
        result1 = __VIRTUAL_FILE92(value);
        if (!isDeep) {
            return __VIRTUAL_FILE75(value, result1);
        }
    } else {
        var tag = __VIRTUAL_FILE91(value), isFunc = tag == funcTag2 || tag == genTag1;
        if (__VIRTUAL_FILE55(value)) {
            return __VIRTUAL_FILE74(value, isDeep);
        }
        if (tag == objectTag2 || tag == argsTag2 || isFunc && !object) {
            result1 = isFlat || isFunc ? {} : __VIRTUAL_FILE101(value);
            if (!isDeep) {
                return isFlat ? __VIRTUAL_FILE83(value, __VIRTUAL_FILE73(result1, value)) : __VIRTUAL_FILE79(value, __VIRTUAL_FILE69(result1, value));
            }
        } else {
            if (!cloneableTags[tag]) {
                return object ? value : {};
            }
            result1 = __VIRTUAL_FILE99(value, tag, isDeep);
        }
    }
    stack || (stack = new __VIRTUAL_FILE43());
    var stacked = stack.get(value);
    if (stacked) {
        return stacked;
    }
    stack.set(value, result1);
    if (__VIRTUAL_FILE105(value)) {
        value.forEach(function(subValue) {
            result1.add(baseClone(subValue, bitmask, customizer, subValue, value, stack));
        });
    } else if (__VIRTUAL_FILE103(value)) {
        value.forEach(function(subValue, key2) {
            result1.set(key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? __VIRTUAL_FILE86 : __VIRTUAL_FILE85 : isFlat ? __VIRTUAL_FILE72 : __VIRTUAL_FILE68;
    var props = isArr ? void 0 : keysFunc(value);
    __VIRTUAL_FILE44(props || value, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value[key2];
        }
        __VIRTUAL_FILE47(result1, key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
    });
    return result1;
}
var __VIRTUAL_FILE106 = baseClone;
var CLONE_DEEP_FLAG1 = 1, CLONE_SYMBOLS_FLAG1 = 4;
function cloneDeep(value) {
    return __VIRTUAL_FILE106(value, CLONE_DEEP_FLAG1 | CLONE_SYMBOLS_FLAG1);
}
var __VIRTUAL_FILE107 = cloneDeep;
var now = function() {
    return __VIRTUAL_FILE13.Date.now();
};
var __VIRTUAL_FILE108 = now;
var reWhitespace1 = /\s/;
function trimmedEndIndex(string) {
    var index = string.length;
    while(index-- && reWhitespace1.test(string.charAt(index))){}
    return index;
}
var __VIRTUAL_FILE109 = trimmedEndIndex;
var reTrimStart = /^\s+/;
function baseTrim(string) {
    return string ? string.slice(0, __VIRTUAL_FILE109(string) + 1).replace(reTrimStart, "") : string;
}
var __VIRTUAL_FILE110 = baseTrim;
var symbolTag2 = "[object Symbol]";
function isSymbol(value) {
    return typeof value == "symbol" || __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == symbolTag2;
}
var __VIRTUAL_FILE111 = isSymbol;
var NAN = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (__VIRTUAL_FILE111(value)) {
        return NAN;
    }
    if (__VIRTUAL_FILE18(value)) {
        var other = typeof value.valueOf == "function" ? value.valueOf() : value;
        value = __VIRTUAL_FILE18(other) ? other + "" : other;
    }
    if (typeof value != "string") {
        return value === 0 ? value : +value;
    }
    value = __VIRTUAL_FILE110(value);
    var isBinary = reIsBinary.test(value);
    return isBinary || reIsOctal.test(value) ? freeParseInt(value.slice(2), isBinary ? 2 : 8) : reIsBadHex.test(value) ? NAN : +value;
}
var __VIRTUAL_FILE112 = toNumber;
var FUNC_ERROR_TEXT = "Expected a function";
var nativeMax = Math.max, nativeMin = Math.min;
function debounce(func, wait, options) {
    var lastArgs, lastThis, maxWait, result1, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    wait = __VIRTUAL_FILE112(wait) || 0;
    if (__VIRTUAL_FILE18(options)) {
        leading = !!options.leading;
        maxing = "maxWait" in options;
        maxWait = maxing ? nativeMax(__VIRTUAL_FILE112(options.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    function invokeFunc(time) {
        var args = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result1 = func.apply(thisArg, args);
        return result1;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result1;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = __VIRTUAL_FILE108();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result1;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result1 : trailingEdge(__VIRTUAL_FILE108());
    }
    function debounced() {
        var time = __VIRTUAL_FILE108(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result1;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
var __VIRTUAL_FILE113 = debounce;
var HASH_UNDEFINED2 = "__lodash_hash_undefined__";
function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED2);
    return this;
}
var __VIRTUAL_FILE114 = setCacheAdd;
function setCacheHas(value) {
    return this.__data__.has(value);
}
var __VIRTUAL_FILE115 = setCacheHas;
function SetCache(values) {
    var index = -1, length = values == null ? 0 : values.length;
    this.__data__ = new __VIRTUAL_FILE41();
    while(++index < length){
        this.add(values[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = __VIRTUAL_FILE114;
SetCache.prototype.has = __VIRTUAL_FILE115;
var __VIRTUAL_FILE116 = SetCache;
function baseFindIndex(array, predicate, fromIndex, fromRight) {
    var length = array.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array[index], index, array)) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE117 = baseFindIndex;
function baseIsNaN(value) {
    return value !== value;
}
var __VIRTUAL_FILE118 = baseIsNaN;
function strictIndexOf(array, value, fromIndex) {
    var index = fromIndex - 1, length = array.length;
    while(++index < length){
        if (array[index] === value) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE119 = strictIndexOf;
function baseIndexOf(array, value, fromIndex) {
    return value === value ? __VIRTUAL_FILE119(array, value, fromIndex) : __VIRTUAL_FILE117(array, __VIRTUAL_FILE118, fromIndex);
}
var __VIRTUAL_FILE120 = baseIndexOf;
function arrayIncludes(array, value) {
    var length = array == null ? 0 : array.length;
    return !!length && __VIRTUAL_FILE120(array, value, 0) > -1;
}
var __VIRTUAL_FILE121 = arrayIncludes;
function arrayIncludesWith(array, value, comparator) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (comparator(value, array[index])) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE122 = arrayIncludesWith;
function arrayMap(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length, result1 = Array(length);
    while(++index < length){
        result1[index] = iteratee(array[index], index, array);
    }
    return result1;
}
var __VIRTUAL_FILE123 = arrayMap;
function cacheHas(cache, key) {
    return cache.has(key);
}
var __VIRTUAL_FILE124 = cacheHas;
var LARGE_ARRAY_SIZE1 = 200;
function baseDifference(array, values, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, isCommon = true, length = array.length, result1 = [], valuesLength = values.length;
    if (!length) {
        return result1;
    }
    if (iteratee) {
        values = __VIRTUAL_FILE123(values, __VIRTUAL_FILE59(iteratee));
    }
    if (comparator) {
        includes = __VIRTUAL_FILE122;
        isCommon = false;
    } else if (values.length >= LARGE_ARRAY_SIZE1) {
        includes = __VIRTUAL_FILE124;
        isCommon = false;
        values = new __VIRTUAL_FILE116(values);
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee == null ? value : iteratee(value);
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result1.push(value);
        } else if (!includes(values, computed, comparator)) {
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE125 = baseDifference;
var spreadableSymbol = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.isConcatSpreadable : void 0;
function isFlattenable(value) {
    return __VIRTUAL_FILE53(value) || __VIRTUAL_FILE52(value) || !!(spreadableSymbol && value && value[spreadableSymbol]);
}
var __VIRTUAL_FILE126 = isFlattenable;
function baseFlatten(array, depth, predicate, isStrict, result1) {
    var index = -1, length = array.length;
    predicate || (predicate = __VIRTUAL_FILE126);
    result1 || (result1 = []);
    while(++index < length){
        var value = array[index];
        if (depth > 0 && predicate(value)) {
            if (depth > 1) {
                baseFlatten(value, depth - 1, predicate, isStrict, result1);
            } else {
                __VIRTUAL_FILE80(result1, value);
            }
        } else if (!isStrict) {
            result1[result1.length] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE127 = baseFlatten;
function identity(value) {
    return value;
}
var __VIRTUAL_FILE128 = identity;
function apply(func, thisArg, args) {
    switch(args.length){
        case 0:
            return func.call(thisArg);
        case 1:
            return func.call(thisArg, args[0]);
        case 2:
            return func.call(thisArg, args[0], args[1]);
        case 3:
            return func.call(thisArg, args[0], args[1], args[2]);
    }
    return func.apply(thisArg, args);
}
var __VIRTUAL_FILE129 = apply;
var nativeMax1 = Math.max;
function overRest(func, start, transform) {
    start = nativeMax1(start === void 0 ? func.length - 1 : start, 0);
    return function() {
        var args = arguments, index = -1, length = nativeMax1(args.length - start, 0), array = Array(length);
        while(++index < length){
            array[index] = args[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args[index];
        }
        otherArgs[start] = transform(array);
        return __VIRTUAL_FILE129(func, this, otherArgs);
    };
}
var __VIRTUAL_FILE130 = overRest;
function constant(value) {
    return function() {
        return value;
    };
}
var __VIRTUAL_FILE131 = constant;
var baseSetToString = !__VIRTUAL_FILE45 ? __VIRTUAL_FILE128 : function(func, string) {
    return __VIRTUAL_FILE45(func, "toString", {
        configurable: true,
        enumerable: false,
        value: __VIRTUAL_FILE131(string),
        writable: true
    });
};
var __VIRTUAL_FILE132 = baseSetToString;
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func.apply(void 0, arguments);
    };
}
var __VIRTUAL_FILE133 = shortOut;
var setToString = __VIRTUAL_FILE133(__VIRTUAL_FILE132);
var __VIRTUAL_FILE134 = setToString;
function baseRest(func, start) {
    return __VIRTUAL_FILE134(__VIRTUAL_FILE130(func, start, __VIRTUAL_FILE128), func + "");
}
var __VIRTUAL_FILE135 = baseRest;
function isArrayLikeObject(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE67(value);
}
var __VIRTUAL_FILE136 = isArrayLikeObject;
var difference = __VIRTUAL_FILE135(function(array, values) {
    return __VIRTUAL_FILE136(array) ? __VIRTUAL_FILE125(array, __VIRTUAL_FILE127(values, 1, __VIRTUAL_FILE136, true)) : [];
});
var __VIRTUAL_FILE137 = difference;
function createBaseFor(fromRight) {
    return function(object, iteratee, keysFunc) {
        var index = -1, iterable = Object(object), props = keysFunc(object), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object;
    };
}
var __VIRTUAL_FILE138 = createBaseFor;
var baseFor = __VIRTUAL_FILE138();
var __VIRTUAL_FILE139 = baseFor;
function baseForOwn(object, iteratee) {
    return object && __VIRTUAL_FILE139(object, iteratee, __VIRTUAL_FILE68);
}
var __VIRTUAL_FILE140 = baseForOwn;
function createBaseEach(eachFunc, fromRight) {
    return function(collection, iteratee) {
        if (collection == null) {
            return collection;
        }
        if (!__VIRTUAL_FILE67(collection)) {
            return eachFunc(collection, iteratee);
        }
        var length = collection.length, index = fromRight ? length : -1, iterable = Object(collection);
        while(fromRight ? index-- : ++index < length){
            if (iteratee(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection;
    };
}
var __VIRTUAL_FILE141 = createBaseEach;
var baseEach = __VIRTUAL_FILE141(__VIRTUAL_FILE140);
var __VIRTUAL_FILE142 = baseEach;
function castFunction(value) {
    return typeof value == "function" ? value : __VIRTUAL_FILE128;
}
var __VIRTUAL_FILE143 = castFunction;
function forEach(collection, iteratee) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE44 : __VIRTUAL_FILE142;
    return func(collection, __VIRTUAL_FILE143(iteratee));
}
var __VIRTUAL_FILE144 = forEach;
var __VIRTUAL_FILE145 = __VIRTUAL_FILE144;
function assignMergeValue(object, key, value) {
    if (value !== void 0 && !__VIRTUAL_FILE1(object[key], value) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE146 = assignMergeValue;
var objectTag3 = "[object Object]";
var funcProto2 = Function.prototype, objectProto13 = Object.prototype;
var funcToString2 = funcProto2.toString;
var hasOwnProperty10 = objectProto13.hasOwnProperty;
var objectCtorString = funcToString2.call(Object);
function isPlainObject(value) {
    if (!__VIRTUAL_FILE50(value) || __VIRTUAL_FILE17(value) != objectTag3) {
        return false;
    }
    var proto = __VIRTUAL_FILE81(value);
    if (proto === null) {
        return true;
    }
    var Ctor = hasOwnProperty10.call(proto, "constructor") && proto.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString2.call(Ctor) == objectCtorString;
}
var __VIRTUAL_FILE147 = isPlainObject;
function safeGet(object, key) {
    if (key === "constructor" && typeof object[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object[key];
}
var __VIRTUAL_FILE148 = safeGet;
function toPlainObject(value) {
    return __VIRTUAL_FILE48(value, __VIRTUAL_FILE72(value));
}
var __VIRTUAL_FILE149 = toPlainObject;
function baseMergeDeep(object, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = __VIRTUAL_FILE148(object, key), srcValue = __VIRTUAL_FILE148(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        __VIRTUAL_FILE146(object, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = __VIRTUAL_FILE53(srcValue), isBuff = !isArr && __VIRTUAL_FILE55(srcValue), isTyped = !isArr && !isBuff && __VIRTUAL_FILE61(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (__VIRTUAL_FILE53(objValue)) {
                newValue = objValue;
            } else if (__VIRTUAL_FILE136(objValue)) {
                newValue = __VIRTUAL_FILE75(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = __VIRTUAL_FILE74(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = __VIRTUAL_FILE98(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (__VIRTUAL_FILE147(srcValue) || __VIRTUAL_FILE52(srcValue)) {
            newValue = objValue;
            if (__VIRTUAL_FILE52(objValue)) {
                newValue = __VIRTUAL_FILE149(objValue);
            } else if (!__VIRTUAL_FILE18(objValue) || __VIRTUAL_FILE19(objValue)) {
                newValue = __VIRTUAL_FILE101(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    __VIRTUAL_FILE146(object, key, newValue);
}
var __VIRTUAL_FILE150 = baseMergeDeep;
function baseMerge(object, source, srcIndex, customizer, stack) {
    if (object === source) {
        return;
    }
    __VIRTUAL_FILE139(source, function(srcValue, key) {
        stack || (stack = new __VIRTUAL_FILE43());
        if (__VIRTUAL_FILE18(srcValue)) {
            __VIRTUAL_FILE150(object, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(__VIRTUAL_FILE148(object, key), srcValue, key + "", object, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            __VIRTUAL_FILE146(object, key, newValue);
        }
    }, __VIRTUAL_FILE72);
}
var __VIRTUAL_FILE151 = baseMerge;
function isIterateeCall(value, index, object) {
    if (!__VIRTUAL_FILE18(object)) {
        return false;
    }
    var type = typeof index;
    if (type == "number" ? __VIRTUAL_FILE67(object) && __VIRTUAL_FILE56(index, object.length) : type == "string" && index in object) {
        return __VIRTUAL_FILE1(object[index], value);
    }
    return false;
}
var __VIRTUAL_FILE152 = isIterateeCall;
function createAssigner(assigner) {
    return __VIRTUAL_FILE135(function(object, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && __VIRTUAL_FILE152(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object = Object(object);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object, source, index, customizer);
            }
        }
        return object;
    });
}
var __VIRTUAL_FILE153 = createAssigner;
var mergeWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE151(object, source, srcIndex, customizer);
});
var __VIRTUAL_FILE154 = mergeWith;
var nativeFloor = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor(nativeRandom() * (upper - lower + 1));
}
var __VIRTUAL_FILE155 = baseRandom;
function shuffleSelf(array, size) {
    var index = -1, length = array.length, lastIndex = length - 1;
    size = size === void 0 ? length : size;
    while(++index < size){
        var rand = __VIRTUAL_FILE155(index, lastIndex), value = array[rand];
        array[rand] = array[index];
        array[index] = value;
    }
    array.length = size;
    return array;
}
var __VIRTUAL_FILE156 = shuffleSelf;
function arrayShuffle(array) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE75(array));
}
var __VIRTUAL_FILE157 = arrayShuffle;
function baseValues(object, props) {
    return __VIRTUAL_FILE123(props, function(key) {
        return object[key];
    });
}
var __VIRTUAL_FILE158 = baseValues;
function values(object) {
    return object == null ? [] : __VIRTUAL_FILE158(object, __VIRTUAL_FILE68(object));
}
var __VIRTUAL_FILE159 = values;
function baseShuffle(collection) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE159(collection));
}
var __VIRTUAL_FILE160 = baseShuffle;
function shuffle(collection) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE157 : __VIRTUAL_FILE160;
    return func(collection);
}
var __VIRTUAL_FILE161 = shuffle;
var assignInWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object, customizer);
});
var __VIRTUAL_FILE162 = assignInWith;
var domExcTag = "[object DOMException]", errorTag2 = "[object Error]";
function isError(value) {
    if (!__VIRTUAL_FILE50(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == errorTag2 || tag == domExcTag || typeof value.message == "string" && typeof value.name == "string" && !__VIRTUAL_FILE147(value);
}
var __VIRTUAL_FILE163 = isError;
var attempt = __VIRTUAL_FILE135(function(func, args) {
    try {
        return __VIRTUAL_FILE129(func, void 0, args);
    } catch (e) {
        return __VIRTUAL_FILE163(e) ? e : new Error(e);
    }
});
var __VIRTUAL_FILE164 = attempt;
var objectProto14 = Object.prototype;
var hasOwnProperty11 = objectProto14.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object) {
    if (objValue === void 0 || __VIRTUAL_FILE1(objValue, objectProto14[key]) && !hasOwnProperty11.call(object, key)) {
        return srcValue;
    }
    return objValue;
}
var __VIRTUAL_FILE165 = customDefaultsAssignIn;
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var __VIRTUAL_FILE166 = escapeStringChar;
var reInterpolate = /<%=([\s\S]+?)%>/g;
var __VIRTUAL_FILE167 = reInterpolate;
function basePropertyOf(object) {
    return function(key) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE168 = basePropertyOf;
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = __VIRTUAL_FILE168(htmlEscapes);
var __VIRTUAL_FILE169 = escapeHtmlChar;
var INFINITY = 1 / 0;
var symbolProto1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolToString = symbolProto1 ? symbolProto1.toString : void 0;
function baseToString(value) {
    if (typeof value == "string") {
        return value;
    }
    if (__VIRTUAL_FILE53(value)) {
        return __VIRTUAL_FILE123(value, baseToString) + "";
    }
    if (__VIRTUAL_FILE111(value)) {
        return symbolToString ? symbolToString.call(value) : "";
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY ? "-0" : result1;
}
var __VIRTUAL_FILE170 = baseToString;
function toString(value) {
    return value == null ? "" : __VIRTUAL_FILE170(value);
}
var __VIRTUAL_FILE171 = toString;
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string) {
    string = __VIRTUAL_FILE171(string);
    return string && reHasUnescapedHtml.test(string) ? string.replace(reUnescapedHtml, __VIRTUAL_FILE169) : string;
}
var __VIRTUAL_FILE172 = escape;
var reEscape = /<%-([\s\S]+?)%>/g;
var __VIRTUAL_FILE173 = reEscape;
var reEvaluate = /<%([\s\S]+?)%>/g;
var __VIRTUAL_FILE174 = reEvaluate;
var templateSettings = {
    escape: __VIRTUAL_FILE173,
    evaluate: __VIRTUAL_FILE174,
    interpolate: __VIRTUAL_FILE167,
    variable: "",
    imports: {
        _: {
            escape: __VIRTUAL_FILE172
        }
    }
};
var __VIRTUAL_FILE175 = templateSettings;
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto15 = Object.prototype;
var hasOwnProperty12 = objectProto15.hasOwnProperty;
function template(string, options, guard) {
    var settings = __VIRTUAL_FILE175.imports._.templateSettings || __VIRTUAL_FILE175;
    if (guard && __VIRTUAL_FILE152(string, options, guard)) {
        options = void 0;
    }
    string = __VIRTUAL_FILE171(string);
    options = __VIRTUAL_FILE162({}, options, settings, __VIRTUAL_FILE165);
    var imports = __VIRTUAL_FILE162({}, options.imports, settings.imports, __VIRTUAL_FILE165), importsKeys = __VIRTUAL_FILE68(imports), importsValues = __VIRTUAL_FILE158(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === __VIRTUAL_FILE167 ? reEsTemplate : reNoMatch).source + "|" + (options.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty12.call(options, "sourceURL") ? "//# sourceURL=" + (options.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string.slice(index, offset).replace(reUnescapedString, __VIRTUAL_FILE166);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty12.call(options, "variable") && options.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result1 = __VIRTUAL_FILE164(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result1.source = source;
    if (__VIRTUAL_FILE163(result1)) {
        throw result1;
    }
    return result1;
}
var __VIRTUAL_FILE176 = template;
function noop() {}
var __VIRTUAL_FILE177 = noop;
function setToArray(set) {
    var index = -1, result1 = Array(set.size);
    set.forEach(function(value) {
        result1[++index] = value;
    });
    return result1;
}
var __VIRTUAL_FILE178 = setToArray;
var INFINITY1 = 1 / 0;
var createSet = !(__VIRTUAL_FILE89 && 1 / __VIRTUAL_FILE178(new __VIRTUAL_FILE89([
    ,
    -0
]))[1] == INFINITY1) ? __VIRTUAL_FILE177 : function(values) {
    return new __VIRTUAL_FILE89(values);
};
var __VIRTUAL_FILE179 = createSet;
var LARGE_ARRAY_SIZE2 = 200;
function baseUniq(array, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, length = array.length, isCommon = true, result1 = [], seen = result1;
    if (comparator) {
        isCommon = false;
        includes = __VIRTUAL_FILE122;
    } else if (length >= LARGE_ARRAY_SIZE2) {
        var set = iteratee ? null : __VIRTUAL_FILE179(array);
        if (set) {
            return __VIRTUAL_FILE178(set);
        }
        isCommon = false;
        includes = __VIRTUAL_FILE124;
        seen = new __VIRTUAL_FILE116();
    } else {
        seen = iteratee ? [] : result1;
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee ? iteratee(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee) {
                seen.push(computed);
            }
            result1.push(value);
        } else if (!includes(seen, computed, comparator)) {
            if (seen !== result1) {
                seen.push(computed);
            }
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE180 = baseUniq;
function uniq(array) {
    return array && array.length ? __VIRTUAL_FILE180(array) : [];
}
var __VIRTUAL_FILE181 = uniq;
function arraySome(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (predicate(array[index], index, array)) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE182 = arraySome;
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array;
    }
    var index = -1, result1 = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new __VIRTUAL_FILE116() : void 0;
    stack.set(array, other);
    stack.set(other, array);
    while(++index < arrLength){
        var arrValue = array[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array, stack) : customizer(arrValue, othValue, index, array, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result1 = false;
            break;
        }
        if (seen) {
            if (!__VIRTUAL_FILE182(other, function(othValue2, othIndex) {
                if (!__VIRTUAL_FILE124(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result1 = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result1 = false;
            break;
        }
    }
    stack["delete"](array);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE183 = equalArrays;
function mapToArray(map) {
    var index = -1, result1 = Array(map.size);
    map.forEach(function(value, key) {
        result1[++index] = [
            key,
            value
        ];
    });
    return result1;
}
var __VIRTUAL_FILE184 = mapToArray;
var COMPARE_PARTIAL_FLAG1 = 1, COMPARE_UNORDERED_FLAG1 = 2;
var boolTag3 = "[object Boolean]", dateTag3 = "[object Date]", errorTag3 = "[object Error]", mapTag5 = "[object Map]", numberTag3 = "[object Number]", regexpTag3 = "[object RegExp]", setTag5 = "[object Set]", stringTag3 = "[object String]", symbolTag3 = "[object Symbol]";
var arrayBufferTag3 = "[object ArrayBuffer]", dataViewTag4 = "[object DataView]";
var symbolProto2 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf1 = symbolProto2 ? symbolProto2.valueOf : void 0;
function equalByTag(object, other, tag, bitmask, customizer, equalFunc, stack) {
    switch(tag){
        case dataViewTag4:
            if (object.byteLength != other.byteLength || object.byteOffset != other.byteOffset) {
                return false;
            }
            object = object.buffer;
            other = other.buffer;
        case arrayBufferTag3:
            if (object.byteLength != other.byteLength || !equalFunc(new __VIRTUAL_FILE93(object), new __VIRTUAL_FILE93(other))) {
                return false;
            }
            return true;
        case boolTag3:
        case dateTag3:
        case numberTag3:
            return __VIRTUAL_FILE1(+object, +other);
        case errorTag3:
            return object.name == other.name && object.message == other.message;
        case regexpTag3:
        case stringTag3:
            return object == other + "";
        case mapTag5:
            var convert = __VIRTUAL_FILE184;
        case setTag5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG1;
            convert || (convert = __VIRTUAL_FILE178);
            if (object.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG1;
            stack.set(object, other);
            var result1 = __VIRTUAL_FILE183(convert(object), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object);
            return result1;
        case symbolTag3:
            if (symbolValueOf1) {
                return symbolValueOf1.call(object) == symbolValueOf1.call(other);
            }
    }
    return false;
}
var __VIRTUAL_FILE185 = equalByTag;
var COMPARE_PARTIAL_FLAG2 = 1;
var objectProto16 = Object.prototype;
var hasOwnProperty13 = objectProto16.hasOwnProperty;
function equalObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG2, objProps = __VIRTUAL_FILE85(object), objLength = objProps.length, othProps = __VIRTUAL_FILE85(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty13.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object;
    }
    var result1 = true;
    stack.set(object, other);
    stack.set(other, object);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object, stack) : customizer(objValue, othValue, key, object, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result1 = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result1 && !skipCtor) {
        var objCtor = object.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result1 = false;
        }
    }
    stack["delete"](object);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE186 = equalObjects;
var COMPARE_PARTIAL_FLAG3 = 1;
var argsTag3 = "[object Arguments]", arrayTag2 = "[object Array]", objectTag4 = "[object Object]";
var objectProto17 = Object.prototype;
var hasOwnProperty14 = objectProto17.hasOwnProperty;
function baseIsEqualDeep(object, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = __VIRTUAL_FILE53(object), othIsArr = __VIRTUAL_FILE53(other), objTag = objIsArr ? arrayTag2 : __VIRTUAL_FILE91(object), othTag = othIsArr ? arrayTag2 : __VIRTUAL_FILE91(other);
    objTag = objTag == argsTag3 ? objectTag4 : objTag;
    othTag = othTag == argsTag3 ? objectTag4 : othTag;
    var objIsObj = objTag == objectTag4, othIsObj = othTag == objectTag4, isSameTag = objTag == othTag;
    if (isSameTag && __VIRTUAL_FILE55(object)) {
        if (!__VIRTUAL_FILE55(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new __VIRTUAL_FILE43());
        return objIsArr || __VIRTUAL_FILE61(object) ? __VIRTUAL_FILE183(object, other, bitmask, customizer, equalFunc, stack) : __VIRTUAL_FILE185(object, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG3)) {
        var objIsWrapped = objIsObj && hasOwnProperty14.call(object, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty14.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object.value() : object, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new __VIRTUAL_FILE43());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new __VIRTUAL_FILE43());
    return __VIRTUAL_FILE186(object, other, bitmask, customizer, equalFunc, stack);
}
var __VIRTUAL_FILE187 = baseIsEqualDeep;
function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
        return true;
    }
    if (value == null || other == null || !__VIRTUAL_FILE50(value) && !__VIRTUAL_FILE50(other)) {
        return value !== value && other !== other;
    }
    return __VIRTUAL_FILE187(value, other, bitmask, customizer, baseIsEqual, stack);
}
var __VIRTUAL_FILE188 = baseIsEqual;
var COMPARE_PARTIAL_FLAG4 = 1, COMPARE_UNORDERED_FLAG2 = 2;
function baseIsMatch(object, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object == null) {
        return !length;
    }
    object = Object(object);
    while(index--){
        var data = matchData[index];
        if (noCustomizer && data[2] ? data[1] !== object[data[0]] : !(data[0] in object)) {
            return false;
        }
    }
    while(++index < length){
        data = matchData[index];
        var key = data[0], objValue = object[key], srcValue = data[1];
        if (noCustomizer && data[2]) {
            if (objValue === void 0 && !(key in object)) {
                return false;
            }
        } else {
            var stack = new __VIRTUAL_FILE43();
            if (customizer) {
                var result1 = customizer(objValue, srcValue, key, object, source, stack);
            }
            if (!(result1 === void 0 ? __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG4 | COMPARE_UNORDERED_FLAG2, customizer, stack) : result1)) {
                return false;
            }
        }
    }
    return true;
}
var __VIRTUAL_FILE189 = baseIsMatch;
function isStrictComparable(value) {
    return value === value && !__VIRTUAL_FILE18(value);
}
var __VIRTUAL_FILE190 = isStrictComparable;
function getMatchData(object) {
    var result1 = __VIRTUAL_FILE68(object), length = result1.length;
    while(length--){
        var key = result1[length], value = object[key];
        result1[length] = [
            key,
            value,
            __VIRTUAL_FILE190(value)
        ];
    }
    return result1;
}
var __VIRTUAL_FILE191 = getMatchData;
function matchesStrictComparable(key, srcValue) {
    return function(object) {
        if (object == null) {
            return false;
        }
        return object[key] === srcValue && (srcValue !== void 0 || key in Object(object));
    };
}
var __VIRTUAL_FILE192 = matchesStrictComparable;
function baseMatches(source) {
    var matchData = __VIRTUAL_FILE191(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return __VIRTUAL_FILE192(matchData[0][0], matchData[0][1]);
    }
    return function(object) {
        return object === source || __VIRTUAL_FILE189(object, source, matchData);
    };
}
var __VIRTUAL_FILE193 = baseMatches;
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return false;
    }
    var type = typeof value;
    if (type == "number" || type == "symbol" || type == "boolean" || value == null || __VIRTUAL_FILE111(value)) {
        return true;
    }
    return reIsPlainProp.test(value) || !reIsDeepProp.test(value) || object != null && value in Object(object);
}
var __VIRTUAL_FILE194 = isKey;
var FUNC_ERROR_TEXT1 = "Expected a function";
function memoize1(func, resolver) {
    if (typeof func != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT1);
    }
    var memoized = function() {
        var args = arguments, key = resolver ? resolver.apply(this, args) : args[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result1 = func.apply(this, args);
        memoized.cache = cache.set(key, result1) || cache;
        return result1;
    };
    memoized.cache = new (memoize1.Cache || __VIRTUAL_FILE41)();
    return memoized;
}
memoize1.Cache = __VIRTUAL_FILE41;
var __VIRTUAL_FILE195 = memoize1;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func) {
    var result1 = __VIRTUAL_FILE195(func, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result1.cache;
    return result1;
}
var __VIRTUAL_FILE196 = memoizeCapped;
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = __VIRTUAL_FILE196(function(string) {
    var result1 = [];
    if (string.charCodeAt(0) === 46) {
        result1.push("");
    }
    string.replace(rePropName, function(match, number, quote, subString) {
        result1.push(quote ? subString.replace(reEscapeChar, "$1") : number || match);
    });
    return result1;
});
var __VIRTUAL_FILE197 = stringToPath;
function castPath(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return value;
    }
    return __VIRTUAL_FILE194(value, object) ? [
        value
    ] : __VIRTUAL_FILE197(__VIRTUAL_FILE171(value));
}
var __VIRTUAL_FILE198 = castPath;
var INFINITY2 = 1 / 0;
function toKey(value) {
    if (typeof value == "string" || __VIRTUAL_FILE111(value)) {
        return value;
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY2 ? "-0" : result1;
}
var __VIRTUAL_FILE199 = toKey;
function baseGet(object, path) {
    path = __VIRTUAL_FILE198(path, object);
    var index = 0, length = path.length;
    while(object != null && index < length){
        object = object[__VIRTUAL_FILE199(path[index++])];
    }
    return index && index == length ? object : void 0;
}
var __VIRTUAL_FILE200 = baseGet;
function get(object, path, defaultValue) {
    var result1 = object == null ? void 0 : __VIRTUAL_FILE200(object, path);
    return result1 === void 0 ? defaultValue : result1;
}
var __VIRTUAL_FILE201 = get;
function baseHasIn(object, key) {
    return object != null && key in Object(object);
}
var __VIRTUAL_FILE202 = baseHasIn;
function hasPath(object, path, hasFunc) {
    path = __VIRTUAL_FILE198(path, object);
    var index = -1, length = path.length, result1 = false;
    while(++index < length){
        var key = __VIRTUAL_FILE199(path[index]);
        if (!(result1 = object != null && hasFunc(object, key))) {
            break;
        }
        object = object[key];
    }
    if (result1 || ++index != length) {
        return result1;
    }
    length = object == null ? 0 : object.length;
    return !!length && __VIRTUAL_FILE57(length) && __VIRTUAL_FILE56(key, length) && (__VIRTUAL_FILE53(object) || __VIRTUAL_FILE52(object));
}
var __VIRTUAL_FILE203 = hasPath;
function hasIn(object, path) {
    return object != null && __VIRTUAL_FILE203(object, path, __VIRTUAL_FILE202);
}
var __VIRTUAL_FILE204 = hasIn;
var COMPARE_PARTIAL_FLAG5 = 1, COMPARE_UNORDERED_FLAG3 = 2;
function baseMatchesProperty(path, srcValue) {
    if (__VIRTUAL_FILE194(path) && __VIRTUAL_FILE190(srcValue)) {
        return __VIRTUAL_FILE192(__VIRTUAL_FILE199(path), srcValue);
    }
    return function(object) {
        var objValue = __VIRTUAL_FILE201(object, path);
        return objValue === void 0 && objValue === srcValue ? __VIRTUAL_FILE204(object, path) : __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG5 | COMPARE_UNORDERED_FLAG3);
    };
}
var __VIRTUAL_FILE205 = baseMatchesProperty;
function baseProperty(key) {
    return function(object) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE206 = baseProperty;
function basePropertyDeep(path) {
    return function(object) {
        return __VIRTUAL_FILE200(object, path);
    };
}
var __VIRTUAL_FILE207 = basePropertyDeep;
function property(path) {
    return __VIRTUAL_FILE194(path) ? __VIRTUAL_FILE206(__VIRTUAL_FILE199(path)) : __VIRTUAL_FILE207(path);
}
var __VIRTUAL_FILE208 = property;
function baseIteratee(value) {
    if (typeof value == "function") {
        return value;
    }
    if (value == null) {
        return __VIRTUAL_FILE128;
    }
    if (typeof value == "object") {
        return __VIRTUAL_FILE53(value) ? __VIRTUAL_FILE205(value[0], value[1]) : __VIRTUAL_FILE193(value);
    }
    return __VIRTUAL_FILE208(value);
}
var __VIRTUAL_FILE209 = baseIteratee;
function uniqBy(array, iteratee) {
    return array && array.length ? __VIRTUAL_FILE180(array, __VIRTUAL_FILE209(iteratee, 2)) : [];
}
var __VIRTUAL_FILE210 = uniqBy;
function isEqual(value, other) {
    return __VIRTUAL_FILE188(value, other);
}
var __VIRTUAL_FILE211 = isEqual;
function baseMap(collection, iteratee) {
    var index = -1, result1 = __VIRTUAL_FILE67(collection) ? Array(collection.length) : [];
    __VIRTUAL_FILE142(collection, function(value, key, collection2) {
        result1[++index] = iteratee(value, key, collection2);
    });
    return result1;
}
var __VIRTUAL_FILE212 = baseMap;
function baseSortBy(array, comparer) {
    var length = array.length;
    array.sort(comparer);
    while(length--){
        array[length] = array[length].value;
    }
    return array;
}
var __VIRTUAL_FILE213 = baseSortBy;
function compareAscending(value, other) {
    if (value !== other) {
        var valIsDefined = value !== void 0, valIsNull = value === null, valIsReflexive = value === value, valIsSymbol = __VIRTUAL_FILE111(value);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = __VIRTUAL_FILE111(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
var __VIRTUAL_FILE214 = compareAscending;
function compareMultiple(object, other, orders) {
    var index = -1, objCriteria = object.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result1 = __VIRTUAL_FILE214(objCriteria[index], othCriteria[index]);
        if (result1) {
            if (index >= ordersLength) {
                return result1;
            }
            var order = orders[index];
            return result1 * (order == "desc" ? -1 : 1);
        }
    }
    return object.index - other.index;
}
var __VIRTUAL_FILE215 = compareMultiple;
function baseOrderBy(collection, iteratees, orders) {
    if (iteratees.length) {
        iteratees = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            if (__VIRTUAL_FILE53(iteratee)) {
                return function(value) {
                    return __VIRTUAL_FILE200(value, iteratee.length === 1 ? iteratee[0] : iteratee);
                };
            }
            return iteratee;
        });
    } else {
        iteratees = [
            __VIRTUAL_FILE128
        ];
    }
    var index = -1;
    iteratees = __VIRTUAL_FILE123(iteratees, __VIRTUAL_FILE59(__VIRTUAL_FILE209));
    var result1 = __VIRTUAL_FILE212(collection, function(value, key, collection2) {
        var criteria = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            return iteratee(value);
        });
        return {
            criteria,
            index: ++index,
            value
        };
    });
    return __VIRTUAL_FILE213(result1, function(object, other) {
        return __VIRTUAL_FILE215(object, other, orders);
    });
}
var __VIRTUAL_FILE216 = baseOrderBy;
function orderBy(collection, iteratees, orders, guard) {
    if (collection == null) {
        return [];
    }
    if (!__VIRTUAL_FILE53(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!__VIRTUAL_FILE53(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return __VIRTUAL_FILE216(collection, iteratees, orders);
}
var __VIRTUAL_FILE217 = orderBy;
const cloneDeep1 = __VIRTUAL_FILE107;
const debounce1 = __VIRTUAL_FILE113;
const shuffle1 = __VIRTUAL_FILE161;
const toString1 = __VIRTUAL_FILE171;
const uniq1 = __VIRTUAL_FILE181;
const uniqBy1 = __VIRTUAL_FILE210;
const escape1 = __VIRTUAL_FILE172;
function generate() {
    return crypto.randomUUID();
}
function mergeConfigs(config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, mergeArrayCustomizer);
}
function mergeArrayCustomizer(objValue, srcValue) {
    if (__VIRTUAL_FILE53(objValue) || __VIRTUAL_FILE53(srcValue)) {
        if (!objValue) {
            return srcValue;
        } else if (!srcValue) {
            return objValue;
        } else {
            if (!__VIRTUAL_FILE53(objValue)) {
                objValue = [
                    objValue
                ];
            }
            if (!__VIRTUAL_FILE53(srcValue)) {
                srcValue = [
                    srcValue
                ];
            }
        }
        const combined = objValue.concat(srcValue);
        return __VIRTUAL_FILE210(combined, (value)=>{
            if (typeof value === "function") {
                return generate();
            } else {
                return JSON.stringify(value);
            }
        });
    }
}
function camelToKebab(camel) {
    const kebab = [];
    for(let i1 = 0; i1 < camel.length; i1++){
        const ch = camel.charAt(i1);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i1 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i1 = 0; i1 < kebab.length; i1++){
        const ch = kebab.charAt(i1);
        if (ch === "-") {
            camel.push(kebab.charAt(++i1).toUpperCase());
        } else if (i1 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
const kMetadataFormat = "format";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig-width";
const kFigHeight = "fig-height";
const kFigFormat = "fig-format";
const kFigDpi = "fig-dpi";
const kMermaidFormat = "mermaid-format";
const kDfPrint = "df-print";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kFormatResources = "format-resources";
const kKeepHidden = "keep-hidden";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kIpynbFilter = "ipynb-filter";
const kIpynbFilters = "ipynb-filters";
const kIPynbTitleBlockTemplate = "ipynb-title-block";
const kShortcodes = "shortcodes";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kFigEnv = "fig-env";
const kFigPos = "fig-pos";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kTblCap = "tbl-cap";
const kTblColwidths = "tbl-colwidths";
const kMergeIncludes = "merge-includes";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexInputPaths = "latex-input-paths";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTinyTex = "latex-tinytex";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kQuartoVersion = "quarto-version";
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kMermaidFormat,
    kDfPrint,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug,
    kIpynbFilter,
    kIpynbFilters, 
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepSource,
    kKeepHidden,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kFigPos,
    kFigEnv,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kShortcodes,
    kTblColwidths,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexInputPaths,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLatexTinyTex,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter,
    kFormatResources, 
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kCalloutTipCaption = "callout-tip-caption";
const kCalloutNoteCaption = "callout-note-caption";
const kCalloutWarningCaption = "callout-warning-caption";
const kCalloutImportantCaption = "callout-important-caption";
const kCalloutCautionCaption = "callout-caution-caption";
const kSectionTitleAbstract = "section-title-abstract";
const kSectionTitleFootnotes = "section-title-footnotes";
const kSectionTitleReferences = "section-title-references";
const kSectionTitleAppendices = "section-title-appendices";
const kSectionTitleReuse = "section-title-reuse";
const kSectionTitleCitation = "section-title-citation";
const kAppendixAttributionBibTex = "appendix-attribution-bibtex";
const kAppendixAttributionCiteAs = "appendix-attribution-cite-as";
const kTitleBlockAuthorSingle = "title-block-author-single";
const kTitleBlockAuthorPlural = "title-block-author-plural";
const kTitleBlockAffiliationSingle = "title-block-affiliation-single";
const kTitleBlockAffiliationPlural = "title-block-affiliation-plural";
const kTitleBlockPublished = "title-block-published";
const kTitleBlockModified = "title-block-modified";
const kCodeSummary = "code-summary";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kCopyButtonTooltip = "copy-button-tooltip";
const kCopyButtonTooltipSuccess = "copy-button-tooltip-success";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefChPrefix = "crossref-ch-prefix";
const kCrossrefApxPrefix = "crossref-apx-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kEnvironmentProofTitle = "environment-proof-title";
const kEnvironmentRemarkTitle = "environment-remark-title";
const kEnvironmentSolutionTitle = "environment-solution-title";
const kListingPageOrderBy = "listing-page-order-by";
const kListingPageOrderByDefault = "listing-page-order-by-default";
const kListingPageOrderByDateAsc = "listing-page-order-by-date-asc";
const kListingPageOrderByDateDesc = "listing-page-order-by-date-desc";
const kListingPageOrderByNumberAsc = "listing-page-order-by-number-asc";
const kListingPageOrderByNumberDesc = "listing-page-order-by-number-desc";
const kListingPageFieldDate = "listing-page-field-date";
const kListingPageFieldTitle = "listing-page-field-title";
const kListingPageFieldDescription = "listing-page-field-description";
const kListingPageFieldAuthor = "listing-page-field-author";
const kListingPageFieldFileName = "listing-page-field-filename";
const kListingPageFieldFileModified = "listing-page-field-filemodified";
const kListingPageFieldSubtitle = "listing-page-field-subtitle";
const kListingPageFieldReadingTime = "listing-page-field-readingtime";
const kListingPageFieldCategories = "listing-page-field-categories";
const kListingPageMinutesCompact = "listing-page-minutes-compact";
const kListingPageCategoryAll = "listing-page-category-all";
const kListingPageNoMatches = "listing-page-no-matches";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutCautionCaption,
    kSectionTitleAbstract,
    kSectionTitleFootnotes,
    kSectionTitleReferences,
    kSectionTitleAppendices,
    kSectionTitleReuse,
    kSectionTitleCitation,
    kAppendixAttributionBibTex,
    kAppendixAttributionCiteAs,
    kTitleBlockAuthorSingle,
    kTitleBlockPublished,
    kTitleBlockModified,
    kCodeSummary,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kSearchNoResultsText,
    kCopyButtonTooltip,
    kCopyButtonTooltipSuccess,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefChPrefix,
    kCrossrefApxPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle,
    kEnvironmentProofTitle,
    kEnvironmentRemarkTitle,
    kEnvironmentSolutionTitle,
    kListingPageOrderBy,
    kListingPageOrderByDefault,
    kListingPageOrderByDateAsc,
    kListingPageOrderByDateDesc,
    kListingPageOrderByNumberAsc,
    kListingPageOrderByNumberDesc,
    kListingPageFieldDate,
    kListingPageFieldTitle,
    kListingPageFieldDescription,
    kListingPageFieldAuthor,
    kListingPageFieldFileName,
    kListingPageFieldFileModified,
    kListingPageFieldSubtitle,
    kListingPageFieldReadingTime,
    kListingPageFieldCategories,
    kListingPageMinutesCompact,
    kListingPageCategoryAll,
    kListingPageNoMatches,
    kTitleBlockAuthorSingle,
    kTitleBlockAuthorPlural,
    kTitleBlockAffiliationSingle,
    kTitleBlockAffiliationPlural,
    kTitleBlockPublished,
    kTitleBlockModified, 
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kAtxHeaders = "atx-headers";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kWrap = "wrap";
const kStandalone = "standalone";
const kSelfContained = "self-contained";
const kEmbedResources = "embed-resources";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kDefaultImageExtension = "default-image-extension";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTocDepth = "toc-depth";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kQuartoTemplateParams = "quarto-template-params";
const kRevealJsScripts = "reveal-jsscripts";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDateFormat = "date-format";
const kDateModified = "date-modified";
const kDoi = "doi";
const kAbstract = "abstract";
const kAbstractTitle = "abstract-title";
const kDescription = "description";
const kHideDescription = "hide-description";
const kTocTitle = "toc-title";
const kTocLocation = "toc-location";
const kLang = "lang";
const kOrder = "order";
const kAuthors = "authors";
const kInstitute = "institute";
const kInstitutes = "institutes";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kNoCite = "nocite";
const kCss = "css";
const kBibliography = "bibliography";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAppendix = "chapters-appendix";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kFigResponsive = "fig-responsive";
const kCapLoc = "cap-location";
const kFigCapLoc = "fig-cap-location";
const kTblCapLoc = "tbl-cap-location";
const kCapTop = "top";
const kCapBottom = "bottom";
const kPositionedRefs = "positioned-refs";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kEmbedResources,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    "resource-path",
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    kWrap,
    "columns",
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    kTocDepth,
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "strip-empty-paragraphs",
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    kDefaultImageExtension,
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kAtxHeaders,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace", 
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellLanguage = "language";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
async function readAndValidateYamlFromFile(file, schema, errorMessage, defaultContents) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    let shortFileName = file;
    if (shortFileName.startsWith("/")) {
        shortFileName = relative2(Deno.cwd(), shortFileName);
    }
    let fileContents = Deno.readTextFileSync(file).trimEnd();
    if (fileContents.trim().length === 0 && defaultContents) {
        fileContents = defaultContents;
    }
    const contents = asMappedString(fileContents, shortFileName);
    const { yaml , yamlValidationErrors ,  } = await readAndValidateYamlFromMappedString(contents, schema);
    if (yamlValidationErrors.length) {
        throw new ValidationError2(errorMessage, yamlValidationErrors);
    }
    return yaml;
}
function getenv(name, defaultValue) {
    const value = Deno.env.get(name);
    if (value === undefined) {
        if (defaultValue === undefined) {
            throw new Error(`Required environment variable ${name} not specified.`);
        } else {
            return defaultValue;
        }
    } else {
        return value;
    }
}
const kSkipHidden = /[/\\][\.]/;
function withPath(paths) {
    const delimiter = Deno.build.os === "windows" ? ";" : ":";
    const currentPath = Deno.env.get("PATH") || "";
    if (paths.append !== undefined && paths.prepend !== undefined) {
        return currentPath;
    } else if (paths.append?.length === 0 && paths.prepend?.length === 0) {
        return currentPath;
    } else {
        const modifiedPaths = [
            currentPath
        ];
        if (paths.append) {
            modifiedPaths.unshift(...paths.append);
        }
        if (paths.prepend) {
            modifiedPaths.push(...paths.prepend);
        }
        return modifiedPaths.join(delimiter);
    }
}
function removeIfExists(file) {
    if (existsSync(file)) {
        Deno.removeSync(file, {
            recursive: true
        });
    }
}
function safeRemoveIfExists(file) {
    try {
        removeIfExists(file);
    } catch (error) {
        warning(`Error removing file ${file}: ${error.message}`);
    }
}
function removeIfEmptyDir(dir) {
    if (existsSync(dir)) {
        let empty = true;
        for (const _entry of Deno.readDirSync(dir)){
            empty = false;
            break;
        }
        if (empty) {
            Deno.removeSync(dir, {
                recursive: true
            });
        }
    }
}
function isModifiedAfter(file, otherFile) {
    if (!existsSync(file)) {
        throw new Error(`${file} does not exist`);
    }
    if (!existsSync(otherFile)) {
        return true;
    }
    const fileInfo = Deno.statSync(file);
    const otherfileInfo = Deno.statSync(otherFile);
    if (fileInfo.mtime === null || otherfileInfo.mtime === null) {
        return true;
    } else {
        return fileInfo.mtime > otherfileInfo.mtime;
    }
}
function dirAndStem(file) {
    return [
        dirname2(file),
        basename2(file, extname2(file)), 
    ];
}
function expandPath(path) {
    if (path === "~") {
        return getenv("HOME", "~");
    } else {
        return path.replace(/^~\//, getenv("HOME", "~") + "/");
    }
}
function suggestUserBinPaths() {
    if (Deno.build.os !== "windows") {
        const possiblePaths = [
            "/usr/local/bin",
            "~/.local/bin",
            "~/bin", 
        ];
        const pathRaw = Deno.env.get("PATH");
        const paths = pathRaw ? pathRaw.split(":") : [];
        return possiblePaths.filter((path)=>{
            return paths.includes(path) || paths.includes(expandPath(path));
        });
    } else {
        throw new Error("suggestUserBinPaths not currently supported on Windows");
    }
}
function safeExistsSync(path) {
    try {
        return existsSync(path);
    } catch  {
        return false;
    }
}
async function which(cmd) {
    const args = Deno.build.os === "windows" ? [
        "CMD",
        "/C",
        "where",
        cmd
    ] : [
        "which",
        cmd
    ];
    const result1 = await execProcess({
        cmd: args,
        stderr: "piped",
        stdout: "piped"
    });
    if (result1.code === 0) {
        return Deno.build.os === "windows" ? result1.stdout?.split("\n")[0].trim() : result1.stdout?.trim();
    } else {
        return undefined;
    }
}
function filterPaths(root, paths, globs, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            const needSlash = !root.endsWith("/") && !glob.startsWith("/");
            const regex = globToRegExp(`${root}${needSlash ? "/" : ""}${glob}`);
            const matchingFiles = paths.filter((path)=>{
                return regex.test(path);
            });
            expanded.push(...matchingFiles);
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function resolvePathGlobs(root, globs, exclude, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            for (const file of expandGlobSync(glob, {
                root,
                exclude,
                includeDirs: true,
                extended: true,
                globstar: true
            })){
                expanded.push(file.path);
            }
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function pathWithForwardSlashes(path) {
    return path.replace(/\\/g, "/");
}
function ensureTrailingSlash(path) {
    if (path && !path.endsWith("/")) {
        return path + "/";
    } else {
        return path;
    }
}
function resolveGlobs(root, globs, expandGlobs, options) {
    const includeGlobs = [];
    const excludeGlobs = [];
    const asFullGlob = (glob, preferSmart)=>{
        const useSmartGlobs = ()=>{
            if (options?.mode === "strict") {
                return false;
            } else if (options?.mode === "always") {
                return true;
            } else if (options?.mode === "auto") {
                if (preferSmart) {
                    return true;
                } else {
                    return isGlob(glob);
                }
            } else {
                return true;
            }
        };
        const smartGlob = useSmartGlobs();
        if (glob.startsWith("\\!")) {
            glob = glob.slice(1);
        }
        if (smartGlob) {
            if (glob.endsWith("/")) {
                glob = glob + "**/*";
            } else {
                const fullPath = join3(root, glob);
                try {
                    if (Deno.statSync(fullPath).isDirectory) {
                        glob = glob + "/**/*";
                    }
                } catch  {}
            }
        }
        if (!glob.startsWith("/")) {
            if (smartGlob) {
                return "**/" + glob;
            } else {
                return glob;
            }
        } else {
            return glob.slice(1);
        }
    };
    for (const glob of globs){
        if (glob.startsWith("!")) {
            excludeGlobs.push(asFullGlob(glob.slice(1), true));
        } else {
            includeGlobs.push(asFullGlob(glob));
        }
    }
    const includeFiles = expandGlobs(includeGlobs);
    const excludeFiles = expandGlobs(excludeGlobs);
    return {
        include: includeFiles,
        exclude: excludeFiles
    };
}
const kQuartoProfile = "QUARTO_PROFILE";
function activeProfiles() {
    return readProfile(Deno.env.get(kQuartoProfile));
}
function readProfile(profile) {
    if (profile) {
        return profile.split(/[ ,]+/);
    } else {
        return [];
    }
}
function setProfileFromArg(args) {
    if (args.profile) {
        Deno.env.set(kQuartoProfile, args.profile);
    }
}
function appendProfileArg(cmd) {
    return cmd.option("--profile", "Active project profile(s)", {
        global: true
    });
}
function removeEmptyValues(obj) {
    return Object.fromEntries(Object.entries(obj).filter(([, value])=>{
        if (value === null) return false;
        if (value === undefined) return false;
        if (value === "") return false;
        return true;
    }));
}
function difference1(arrA, arrB) {
    return arrA.filter((a)=>arrB.indexOf(a) < 0);
}
const RE_KeyValue = /^\s*(?:export\s+)?(?<key>[a-zA-Z_]+[a-zA-Z0-9_]*?)\s*=[\ \t]*('\n?(?<notInterpolated>(.|\n)*?)\n?'|"\n?(?<interpolated>(.|\n)*?)\n?"|(?<unquoted>[^\n#]*)) *#*.*$/gm;
const RE_ExpandValue = /(\${(?<inBrackets>.+?)(\:-(?<inBracketsDefault>.+))?}|(?<!\\)\$(?<notInBrackets>\w+)(\:-(?<notInBracketsDefault>.+))?)/g;
function parse4(rawDotenv) {
    const env = {};
    let match;
    const keysForExpandCheck = [];
    while((match = RE_KeyValue.exec(rawDotenv)) != null){
        const { key , interpolated , notInterpolated , unquoted  } = match?.groups;
        if (unquoted) {
            keysForExpandCheck.push(key);
        }
        env[key] = notInterpolated ? notInterpolated : interpolated ? expandCharacters(interpolated) : unquoted.trim();
    }
    const variablesMap = {
        ...env,
        ...Deno.env.toObject()
    };
    keysForExpandCheck.forEach((key)=>{
        env[key] = expand(env[key], variablesMap);
    });
    return env;
}
const defaultConfigOptions = {
    path: `.env`,
    export: false,
    safe: false,
    example: `.env.example`,
    allowEmptyValues: false,
    defaults: `.env.defaults`
};
async function config(options = {}) {
    const o = {
        ...defaultConfigOptions,
        ...options
    };
    const conf = await parseFileAsync(o.path);
    if (o.defaults) {
        const confDefaults = await parseFileAsync(o.defaults);
        for(const key in confDefaults){
            if (!(key in conf)) {
                conf[key] = confDefaults[key];
            }
        }
    }
    if (o.safe) {
        const confExample = await parseFileAsync(o.example);
        assertSafe(conf, confExample, o.allowEmptyValues);
    }
    if (o.export) {
        for(const key1 in conf){
            if (Deno.env.get(key1) !== undefined) continue;
            Deno.env.set(key1, conf[key1]);
        }
    }
    return conf;
}
async function parseFileAsync(filepath) {
    try {
        return parse4(new TextDecoder("utf-8").decode(await Deno.readFile(filepath)));
    } catch (e) {
        if (e instanceof Deno.errors.NotFound) return {};
        throw e;
    }
}
function expandCharacters(str) {
    const charactersMap = {
        "\\n": "\n",
        "\\r": "\r",
        "\\t": "\t"
    };
    return str.replace(/\\([nrt])/g, ($1)=>charactersMap[$1]);
}
function assertSafe(conf, confExample, allowEmptyValues) {
    const currentEnv = Deno.env.toObject();
    const confWithEnv = Object.assign({}, currentEnv, conf);
    const missing = difference1(Object.keys(confExample), Object.keys(allowEmptyValues ? confWithEnv : removeEmptyValues(confWithEnv)));
    if (missing.length > 0) {
        const errorMessages = [
            `The following variables were defined in the example file but are not present in the environment:\n  ${missing.join(", ")}`,
            `Make sure to add them to your env file.`,
            !allowEmptyValues && `If you expect any of these variables to be empty, you can set the allowEmptyValues option to true.`, 
        ];
        throw new MissingEnvVarsError(errorMessages.filter(Boolean).join("\n\n"), missing);
    }
}
class MissingEnvVarsError extends Error {
    missing;
    constructor(message, missing){
        super(message);
        this.name = "MissingEnvVarsError";
        this.missing = missing;
        Object.setPrototypeOf(this, new.target.prototype);
    }
}
function expand(str, variablesMap) {
    if (RE_ExpandValue.test(str)) {
        return expand(str.replace(RE_ExpandValue, function(...params) {
            const { inBrackets , inBracketsDefault , notInBrackets , notInBracketsDefault ,  } = params[params.length - 1];
            const expandValue = inBrackets || notInBrackets;
            const defaultValue = inBracketsDefault || notInBracketsDefault;
            return variablesMap[expandValue] || expand(defaultValue, variablesMap);
        }), variablesMap);
    } else {
        return str;
    }
}
function stringify1(object) {
    const lines = [];
    for (const [key, value] of Object.entries(object)){
        let quote;
        let escapedValue = value ?? "";
        if (key.startsWith("#")) {
            console.warn(`key starts with a '#' indicates a comment and is ignored: '${key}'`);
            continue;
        } else if (escapedValue.includes("\n")) {
            escapedValue = escapedValue.replaceAll("\n", "\\n");
            quote = `"`;
        } else if (escapedValue.match(/\W/)) {
            quote = "'";
        }
        if (quote) {
            escapedValue = escapedValue.replaceAll(quote, `\\${quote}`);
            escapedValue = `${quote}${escapedValue}${quote}`;
        }
        const line = `${key}=${escapedValue}`;
        lines.push(line);
    }
    return lines.join("\n");
}
let tempDir;
let tempContext;
function initSessionTempDir() {
    const tmpEnv = Deno.env.get("TMPDIR");
    if (tmpEnv) {
        try {
            if (!existsSync(tmpEnv)) {
                ensureDirSync(tmpEnv);
            }
        } catch (err) {
            if (err.message) {
                debug("Error attempting to create TMPDIR: " + err.message);
            }
        }
    }
    tempDir = Deno.makeTempDirSync({
        prefix: "quarto-session"
    });
}
function cleanupSessionTempDir() {
    if (tempContext) {
        tempContext.cleanup();
        tempContext = undefined;
    }
    if (tempDir) {
        removeIfExists(tempDir);
        tempDir = undefined;
    }
}
function globalTempContext() {
    if (tempContext === undefined) {
        tempContext = createTempContext();
    }
    return tempContext;
}
function createTempContext(options) {
    let dir = Deno.makeTempDirSync({
        ...options,
        dir: tempDir
    });
    return {
        createFile: (options)=>{
            return Deno.makeTempFileSync({
                ...options,
                dir
            });
        },
        createDir: (options)=>{
            return Deno.makeTempDirSync({
                ...options,
                dir
            });
        },
        cleanup: ()=>{
            if (dir) {
                safeRemoveIfExists(dir);
                dir = undefined;
            }
        }
    };
}
const kQuartoEnv = "_environment";
const kQuartoEnvLocal = `${kQuartoEnv}.local`;
const kQuartoEnvRequired = `${kQuartoEnv}.required`;
async function dotenvQuartoProfile(projectDir) {
    const conf = await config({
        defaults: join3(projectDir, kQuartoEnv),
        path: join3(projectDir, kQuartoEnvLocal)
    });
    return conf[kQuartoProfile];
}
const dotenvVariablesSet = [];
let prevDotenvVariablesDefined;
async function dotenvSetVariables(projectDir) {
    dotenvVariablesSet.forEach(Deno.env.delete);
    dotenvVariablesSet.splice(0, dotenvVariablesSet.length);
    const dotenvFiles = [
        join3(projectDir, kQuartoEnv),
        ...activeProfiles().reverse().map((profile)=>join3(projectDir, `_environment-${profile}`)),
        join3(projectDir, kQuartoEnvLocal), 
    ].filter(safeExistsSync).reverse();
    const dotenvVariablesDefined = {};
    for (const dotenvFile of dotenvFiles){
        const conf = await config({
            path: dotenvFile
        });
        for(const key in conf){
            if (Deno.env.get(key) === undefined) {
                Deno.env.set(key, conf[key]);
                dotenvVariablesSet.push(key);
            }
            if (dotenvVariablesDefined[key] === undefined) {
                dotenvVariablesDefined[key] = conf[key];
            }
        }
    }
    const dotenvRequired = join3(projectDir, kQuartoEnvRequired);
    if (existsSync(dotenvRequired)) {
        const definedEnvTempPath = globalTempContext().createFile({
            suffix: ".yml"
        });
        Deno.writeTextFileSync(definedEnvTempPath, stringify1(dotenvVariablesDefined));
        await config({
            path: definedEnvTempPath,
            example: dotenvRequired,
            safe: true,
            allowEmptyValues: true
        });
    }
    if (prevDotenvVariablesDefined && !__VIRTUAL_FILE211(dotenvVariablesDefined, prevDotenvVariablesDefined)) {
        fireDotenvChanged();
    }
    prevDotenvVariablesDefined = dotenvVariablesDefined;
    return dotenvFiles;
}
const listeners = new Array();
function fireDotenvChanged() {
    listeners.forEach((listener)=>listener());
}
function onDotenvChanged(listener) {
    listeners.push(listener);
}
const cleanupHandlers = [];
function onCleanup(handler) {
    cleanupHandlers.push(handler);
}
function exitWithCleanup(code) {
    for (const handler of cleanupHandlers){
        try {
            handler();
        } catch (error) {
            info("Error occurred during cleanup: " + error);
        }
    }
    Deno.exit(code);
}
function appendLogOptions(cmd) {
    return cmd.option("--log <level>", "Path to log file", {
        global: true
    }).option("--log-level <level>", "Log level (info, warning, error, critical)", {
        global: true
    }).option("--log-format <format>", "Log format (plain, json-stream)", {
        global: true
    }).option("--quiet", "Suppress console output.", {
        global: true
    });
}
function logOptions(args) {
    const logOptions = {};
    logOptions.log = args.l || args.log || Deno.env.get("QUARTO_LOG");
    if (logOptions.log) {
        ensureDirSync(dirname2(logOptions.log));
    }
    logOptions.level = args.ll || args["log-level"] || Deno.env.get("QUARTO_LOG_LEVEL");
    logOptions.quiet = args.q || args.quiet;
    logOptions.format = parseFormat(args.lf || args["log-format"] || Deno.env.get("QUARTO_LOG_FORMAT"));
    return logOptions;
}
let currentLogLevel = "INFO";
function logLevel() {
    return currentLogLevel;
}
class StdErrOutputHandler extends BaseHandler {
    format(logRecord, prefix = true) {
        const options = {
            newline: true,
            colorize: true,
            ...logRecord.args[0]
        };
        let msg = super.format(logRecord);
        if (prefix && logRecord.level >= LogLevels.WARNING) {
            msg = `${logRecord.levelName}: ${msg}`;
        }
        switch(logRecord.level){
            case LogLevels.INFO:
            case LogLevels.DEBUG:
                msg = applyMsgOptions(msg, options);
                break;
            case LogLevels.WARNING:
                if (options.colorize) {
                    msg = yellow1(msg);
                }
                break;
            case LogLevels.ERROR:
                if (options.colorize) {
                    msg = brightRed1(msg);
                }
                break;
            case LogLevels.CRITICAL:
                if (options.colorize) {
                    msg = bold1(red1(msg));
                }
                break;
            default:
                break;
        }
        if (options.newline) {
            msg = msg + "\n";
        }
        return msg;
    }
    log(msg) {
        Deno.stderr.writeSync(new TextEncoder().encode(msg));
    }
}
class LogEventsHandler extends StdErrOutputHandler {
    constructor(levelName){
        super(levelName, {
            formatter: "{msg}"
        });
    }
    handle(logRecord) {
        LogEventsHandler.handlers_.forEach((handler)=>handler(logRecord, super.format(logRecord, false)));
    }
    static onLog(handler) {
        LogEventsHandler.handlers_.push(handler);
    }
    static handlers_ = new Array();
}
class LogFileHandler extends FileHandler {
    constructor(levelName, options){
        super(levelName, options);
        this.msgFormat = options.format;
    }
    msgFormat;
    format(logRecord) {
        if (logRecord.msg.startsWith("\r")) {
            return "";
        }
        if (this.msgFormat === undefined || this.msgFormat === "plain") {
            const options = {
                newline: true,
                ...logRecord.args[0],
                bold: false,
                dim: false,
                format: undefined
            };
            let msg = applyMsgOptions(logRecord.msg, options);
            if (options.newline) {
                msg = msg + "\n";
            }
            if (logRecord.level >= LogLevels.WARNING) {
                return `(${logRecord.levelName}) ${msg}`;
            } else {
                return msg;
            }
        } else {
            return JSON.stringify(logRecord, undefined, 0) + "\n";
        }
    }
    log(msg) {
        if (msg !== "") {
            msg = stripColor1(msg);
            this._buf.writeSync(this._encoder.encode(msg));
            this._buf.flush();
        }
    }
}
async function initializeLogger(logOptions) {
    const handlers = {};
    const defaultHandlers = [];
    const file = logOptions.log;
    currentLogLevel = logOptions.level ? parseLevel(logOptions.level) : "INFO";
    handlers["events"] = new LogEventsHandler(logLevel());
    defaultHandlers.push("events");
    if (!logOptions.quiet) {
        handlers["console"] = new StdErrOutputHandler(logLevel(), {
            formatter: "{msg}"
        });
        defaultHandlers.push("console");
    }
    if (file) {
        handlers["file"] = new LogFileHandler(logLevel(), {
            filename: file,
            mode: "w",
            format: logOptions.format
        });
        defaultHandlers.push("file");
    }
    await setup({
        handlers,
        loggers: {
            default: {
                level: "DEBUG",
                handlers: defaultHandlers
            }
        }
    });
}
async function cleanupLogger() {
    const logger = getLogger();
    for (const handler of logger.handlers){
        await handler.destroy();
    }
    logger.handlers = [];
}
function logProgress(message) {
    info(bold1(blue1(message)));
}
function logError(e) {
    const err = asErrorEx(e);
    let message = err.printName ? `${err.name}: ${err.message}` : err.message;
    const isDebug = getenv("QUARTO_DEBUG", "false") === "true" || getenv("QUARTO_PRINT_STACK", "false") === "true";
    if (isDebug && err.stack && err.printStack) {
        message = message + "\n\n" + err.stack;
    }
    if (message) {
        error(message);
    }
}
function warnOnce(msg) {
    if (!warnings[msg]) {
        warnings[msg] = true;
        warning(msg);
    }
}
const warnings = {};
function applyMsgOptions(msg, options) {
    if (options.indent) {
        const pad = " ".repeat(options.indent);
        msg = lines(msg).map((msg)=>pad + msg).join("\n");
    }
    if (options.bold) {
        msg = bold1(msg);
    }
    if (options.dim) {
        msg = dim1(msg);
    }
    if (options.format) {
        msg = options.format(msg);
    }
    return msg;
}
function parseFormat(format) {
    if (format) {
        format = format.toLowerCase();
        switch(format){
            case "plain":
            case "json-stream":
                return format;
            default:
                return "plain";
        }
    } else {
        return "plain";
    }
}
function parseLevel(level) {
    const lvl = levelMap[level.toLowerCase()];
    if (lvl) {
        return lvl;
    } else {
        return "WARNING";
    }
}
const levelMap = {
    debug: "DEBUG",
    info: "INFO",
    warning: "WARNING",
    error: "ERROR"
};
const kHKeyCurrentUser = "HKCU";
const kHKeyLocalMachine = "HKLM";
async function registryReadString(root, key, value) {
    if (Array.isArray(root)) {
        for (const r of root){
            const val = await registryReadString(r, key, value);
            if (val !== undefined) {
                return val;
            }
        }
        return undefined;
    }
    const kTypeString = "REG_SZ";
    const cmd = [
        "reg",
        "query",
        `${root}\\${key}`,
        "/v",
        value, 
    ];
    const result1 = await execProcess({
        cmd,
        stdout: "piped",
        stderr: "null"
    });
    if (result1.success && result1.stdout) {
        const typePos = result1.stdout?.indexOf(kTypeString);
        if (typePos !== -1) {
            return result1.stdout.substring(typePos + kTypeString.length).trim();
        }
    }
    return undefined;
}
const kQmdExtensions = [
    ".qmd"
];
const kMarkdownEngine = "markdown";
const kKnitrEngine = "knitr";
const kJupyterEngine = "jupyter";
async function includedMetadata(dir, baseMetadata, schema) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join3(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join3(dir, file)));
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (await exists(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeFormatMetadata({}, ...filesMetadata),
        files: yamlFiles
    };
}
const kGfmCommonmarkExtensions = [
    "+autolink_bare_uris",
    "+emoji",
    "+footnotes",
    "+gfm_auto_identifiers",
    "+pipe_tables",
    "+strikeout",
    "+task_lists",
    "+tex_math_dollars", 
];
function createFormat(ext, ...formats) {
    return mergeConfigs(defaultFormat(), ...formats, {
        render: {
            [kOutputExt]: ext
        }
    });
}
const kLocalDevelopment = "99.9.9";
let dotenvConfig;
const quartoConfig = {
    binPath: ()=>getenv("QUARTO_BIN_PATH"),
    toolsPath: ()=>join3(getenv("QUARTO_BIN_PATH"), "tools"),
    sharePath: ()=>getenv("QUARTO_SHARE_PATH"),
    isDebug: ()=>getenv("QUARTO_DEBUG", "false") === "true",
    version: ()=>{
        const versionPath = join3(getenv("QUARTO_SHARE_PATH"), "version");
        if (existsSync(versionPath)) {
            return Deno.readTextFileSync(versionPath);
        } else {
            return kLocalDevelopment;
        }
    },
    dotenv: async ()=>{
        if (!dotenvConfig) {
            const options = {
                defaults: join3(quartoConfig.sharePath(), "env", "env.defaults")
            };
            if (quartoConfig.isDebug()) {
                options.path = join3(quartoConfig.sharePath(), "..", "..", ".env");
            } else {
                options.path = options.defaults;
            }
            dotenvConfig = await config(options);
        }
        return dotenvConfig;
    }
};
function resourcePath(resource) {
    const sharePath = quartoConfig.sharePath();
    if (resource) {
        return join3(sharePath, resource);
    } else {
        return sharePath;
    }
}
const kQuartoProfileConfig = "profile";
function formatFromMetadata(baseFormat, to, debug) {
    const typedFormat = {
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    let format = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format = metadataAsFormat({});
        } else if (configFormat instanceof Object) {
            format = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeFormatMetadata(baseFormat, format);
    if (debug) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
    }
    return mergedFormat;
}
function formatKeys(metadata) {
    if (typeof metadata[kMetadataFormat] === "string") {
        return [
            metadata[kMetadataFormat]
        ];
    } else if (metadata[kMetadataFormat] instanceof Object) {
        return Object.keys(metadata[kMetadataFormat]).filter((key)=>{
            const format = metadata[kMetadataFormat][key];
            return format !== null && format !== false;
        });
    } else {
        return [];
    }
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key) || [
        kKnitrEngine,
        kJupyterEngine
    ].includes(key);
}
function isIncludeMetadata(key) {
    return [
        kHeaderIncludes,
        kIncludeBefore,
        kIncludeAfter
    ].includes(key);
}
const kGfmCommonmarkVariant = kGfmCommonmarkExtensions.join("");
function metadataAsFormat(metadata) {
    const typedFormat = {
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    const format = typedFormat;
    Object.keys(metadata).forEach((key)=>{
        if ([
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata, 
        ].includes(key)) {
            if (typeof metadata[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format[key] = format[key] || {};
                    format[kExecuteDefaults][kExecuteEnabled] = metadata[key];
                }
            } else {
                format[key] = {
                    ...format[key],
                    ...metadata[key]
                };
            }
        } else {
            if (kRenderDefaultsKeys.includes(key)) {
                format.render[key] = metadata[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format.execute[key] = metadata[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format.pandoc[key] = metadata[key];
            } else if (kLanguageDefaultsKeys.includes(key)) {
                format.language[key] = metadata[key];
            } else {
                format.metadata[key] = metadata[key];
            }
        }
    });
    const filter = format.execute[kIpynbFilter];
    if (typeof filter === "string") {
        typedFormat.execute[kIpynbFilters] = typedFormat.execute[kIpynbFilters] || [];
        typedFormat.execute[kIpynbFilters]?.push(filter);
        delete typedFormat.execute[kIpynbFilter];
    }
    if (typeof typedFormat.render.variant === "string") {
        typedFormat.render.variant = typedFormat.render.variant.replace(/^gfm/, kGfmCommonmarkVariant);
    }
    return typedFormat;
}
function metadataGetDeep(metadata, property) {
    let values = [];
    __VIRTUAL_FILE145(metadata, (value, key)=>{
        if (key === property) {
            values.push(value);
        } else if (__VIRTUAL_FILE18(value)) {
            values = values.concat(metadataGetDeep(value, property));
        }
    });
    return values;
}
function mergeFormatMetadata(config, ...configs) {
    const kUnmergeableKeys = [
        kTblColwidths
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kUnmergeableKeys.includes(key)) {
            return srcValue;
        } else if (key === kVariant) {
            return mergePandocVariant(objValue, srcValue);
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeProjectMetadata(config, ...configs) {
    const kExandableStringKeys = [
        "contents"
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kExandableStringKeys.includes(key) && typeof objValue === "string") {
            return srcValue;
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeConfigsCustomized(customizer, config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, (objValue, srcValue, key)=>{
        const custom = customizer(objValue, srcValue, key);
        if (custom !== undefined) {
            return custom;
        } else {
            return mergeArrayCustomizer(objValue, srcValue);
        }
    });
}
function mergePandocVariant(objValue, srcValue) {
    if (typeof objValue === "string" && typeof srcValue === "string" && objValue !== srcValue) {
        const extensions = {};
        [
            ...parsePandocVariant(objValue),
            ...parsePandocVariant(srcValue)
        ].forEach((extension)=>{
            extensions[extension.name] = extension.enabled;
        });
        return Object.keys(extensions).map((name)=>`${extensions[name] ? "+" : "-"}${name}`).join("");
    } else {
        return undefined;
    }
}
function parsePandocVariant(variant) {
    variant = variant.split("\n").join();
    const extensions = [];
    const re = /([+-])([a-z_]+)/g;
    let match = re.exec(variant);
    while(match){
        extensions.push({
            name: match[2],
            enabled: match[1] === "+"
        });
        match = re.exec(variant);
    }
    return extensions;
}
`commonmark${kGfmCommonmarkVariant}`;
function requiresShortcodeUnescapePostprocessor(markdown) {
    return markdown.includes("{{{<");
}
function shortcodeUnescapePostprocessor(output) {
    Deno.writeTextFileSync(output, Deno.readTextFileSync(output).replaceAll("{{\\<", "{{<").replaceAll("\\>}}", ">}}"));
    return Promise.resolve();
}
function gfmFormat() {
    return createFormat("md", markdownFormat(), {
        pandoc: {
            to: "commonmark"
        },
        render: {
            [kVariant]: kGfmCommonmarkVariant
        }
    });
}
function markdownWithCommonmarkExtensionsFormat() {
    return createFormat("md", markdownFormat(), {
        pandoc: {
            to: [
                "markdown_strict",
                "+raw_html",
                "+all_symbols_escapable",
                "+backtick_code_blocks",
                "+fenced_code_blocks",
                "+space_in_atx_header",
                "+intraword_underscores",
                "+lists_without_preceding_blankline",
                "+shortcut_reference_links", 
            ].join("")
        }
    });
}
function commonmarkFormat(to) {
    return createFormat("md", markdownFormat(), {
        pandoc: {
            to
        }
    });
}
function createHtmlFormat(figwidth, figheight) {
    return createFormat("html", {
        metadata: {
            [kLang]: "en",
            [kFigResponsive]: true,
            [kQuartoVersion]: quartoConfig.version()
        },
        execute: {
            [kFigFormat]: "retina",
            [kFigWidth]: figwidth,
            [kFigHeight]: figheight
        },
        render: {
            [kTblColwidths]: "auto"
        },
        pandoc: {
            [kStandalone]: true,
            [kWrap]: "none",
            [kDefaultImageExtension]: "png"
        }
    });
}
function createHtmlPresentationFormat(figwidth, figheight) {
    return mergeConfigs(createHtmlFormat(figwidth, figheight), {
        metadata: {
            [kFigResponsive]: false
        },
        execute: {
            [kEcho]: false,
            [kWarning]: false
        }
    });
}
function formatResourcePath(format, resource) {
    return join3(resourcePath("formats"), format, resource);
}
function createEbookFormat(ext) {
    return createFormat(ext, {
        formatExtras: ()=>{
            return {
                [kIncludeInHeader]: [
                    formatResourcePath("html", "styles-callout.html"),
                    formatResourcePath("epub", "styles.html"), 
                ]
            };
        },
        render: {
            [kMergeIncludes]: false
        },
        execute: {
            [kFigWidth]: 5,
            [kFigHeight]: 4
        },
        pandoc: {
            [kDefaultImageExtension]: "png"
        }
    });
}
function createWordprocessorFormat(ext) {
    return createFormat(ext, {
        render: {
            [kPageWidth]: 6.5
        },
        execute: {
            [kFigWidth]: 5,
            [kFigHeight]: 4
        },
        pandoc: {
            [kDefaultImageExtension]: "png"
        }
    });
}
function plaintextFormat(ext) {
    return createFormat(ext, {
        pandoc: {
            standalone: true,
            [kDefaultImageExtension]: "png"
        }
    });
}
function pandocMarkdownFormat() {
    return createFormat("md", plaintextFormat("md"), {});
}
function markdownFormat() {
    return createFormat("md", plaintextFormat("md"), {
        render: {
            [kOutputDivs]: false
        }
    });
}
function defaultFormat() {
    return {
        execute: {
            [kFigWidth]: 7,
            [kFigHeight]: 5,
            [kFigFormat]: "png",
            [kFigDpi]: 96,
            [kMermaidFormat]: undefined,
            [kDfPrint]: "default",
            [kError]: false,
            [kEval]: true,
            [kCache]: null,
            [kFreeze]: false,
            [kEcho]: true,
            [kOutput]: true,
            [kWarning]: true,
            [kInclude]: true,
            [kKeepMd]: false,
            [kKeepIpynb]: false,
            [kExecuteIpynb]: null,
            [kExecuteEnabled]: null,
            [kExecuteDaemon]: null,
            [kExecuteDaemonRestart]: false,
            [kExecuteDebug]: false,
            [kIpynbFilters]: []
        },
        render: {
            [kKeepTex]: false,
            [kKeepSource]: false,
            [kKeepHidden]: false,
            [kPreferHtml]: false,
            [kOutputDivs]: true,
            [kOutputExt]: "html",
            [kFigAlign]: "default",
            [kFigPos]: null,
            [kFigEnv]: null,
            [kCodeFold]: "none",
            [kCodeOverflow]: "scroll",
            [kCodeLink]: false,
            [kCodeLineNumbers]: false,
            [kCodeTools]: false,
            [kTblColwidths]: true,
            [kMergeIncludes]: true,
            [kLatexAutoMk]: true,
            [kLatexAutoInstall]: true,
            [kLatexClean]: true,
            [kLatexMaxRuns]: 1,
            [kLatexMaxRuns]: 10,
            [kLatexMakeIndex]: "makeindex",
            [kLatexMakeIndexOpts]: [],
            [kLatexTlmgrOpts]: [],
            [kLatexInputPaths]: [],
            [kLatexOutputDir]: null,
            [kLinkExternalIcon]: false,
            [kLinkExternalNewwindow]: false,
            [kSelfContainedMath]: false,
            [kFormatResources]: []
        },
        pandoc: {},
        language: {},
        metadata: {}
    };
}
function toolsPath(binary) {
    const displayWarning = ()=>{
        warnOnce(`Specified ${binaryEnvKey} does not exist, using built in ${binary}`);
    };
    const binaryEnvKey = `QUARTO_${binary.toUpperCase()}`;
    const binaryPath = Deno.env.get(binaryEnvKey);
    if (binaryPath) {
        if (!existsSync(binaryPath)) {
            displayWarning();
        } else {
            if (Deno.statSync(binaryPath).isFile) {
                return binaryPath;
            } else {
                const fullPath = join3(binaryPath, binary);
                if (!existsSync(fullPath)) {
                    displayWarning();
                } else {
                    return fullPath;
                }
            }
        }
    }
    return join3(quartoConfig.toolsPath(), binary);
}
function pandocBinaryPath() {
    return toolsPath("pandoc");
}
async function rBinaryPath(binary) {
    const quartoR = Deno.env.get("QUARTO_R");
    if (quartoR && existsSync(quartoR)) {
        const rBinDir = Deno.statSync(quartoR).isDirectory ? quartoR : dirname2(quartoR);
        return join3(rBinDir, binary);
    }
    const rHome = Deno.env.get("R_HOME");
    if (rHome) {
        let rHomeBin = join3(rHome, "bin", binary);
        if (safeExistsSync(rHomeBin)) return rHomeBin;
        if (Deno.build.os === "windows") {
            rHomeBin = join3(rHome, "bin", "x64", binary);
            if (safeExistsSync(rHomeBin)) return rHomeBin;
        }
    }
    const path = await which(binary);
    if (path) {
        return path;
    }
    if (Deno.build.os === "windows") {
        const version = await registryReadString([
            kHKeyCurrentUser,
            kHKeyLocalMachine
        ], "Software\\R-core\\R", "Current Version");
        if (version) {
            const installPath = await registryReadString([
                kHKeyCurrentUser,
                kHKeyLocalMachine
            ], `Software\\R-core\\R\\${version}`, "InstallPath");
            if (installPath) {
                return join3(installPath, "bin", binary);
            }
        }
        const progFiles = Deno.env.get("programfiles");
        if (progFiles) {
            for (const entry of Deno.readDirSync(progFiles)){
                if (entry.isDirectory && entry.name === "R") {
                    for (const walk of walkSync(join3(progFiles, "R"))){
                        if (walk.isDirectory && walk.name === "bin") {
                            return join3(walk.path, binary);
                        }
                    }
                }
            }
        }
    }
    return binary;
}
function projectTypeResourcePath(projectType) {
    return resourcePath(join3("projects", projectType));
}
const kDarkSuffix = "dark";
const kLightSuffix = "light";
function textHighlightThemePath(theme, style) {
    const names = [
        `${theme}-${style === "dark" ? kDarkSuffix : kLightSuffix}`,
        theme, 
    ];
    const themePath = names.map((name)=>{
        return resourcePath(join3("pandoc", "highlight-styles", `${name}.theme`));
    }).find((path)=>existsSync(path));
    return themePath;
}
let baseQuartoProfile;
async function initializeProfileConfig(dir, config, schema) {
    const firstRun = baseQuartoProfile === undefined;
    if (firstRun) {
        baseQuartoProfile = Deno.env.get(kQuartoProfile) || "";
    }
    const profileConfig = __VIRTUAL_FILE18(config[kQuartoProfileConfig]) ? config[kQuartoProfileConfig] : undefined;
    delete config[kQuartoProfileConfig];
    let quartoProfile = baseQuartoProfile || await dotenvQuartoProfile(dir) || await localConfigQuartoProfile(dir, schema) || "";
    if (!quartoProfile) {
        if (Array.isArray(profileConfig?.default)) {
            quartoProfile = profileConfig.default.map((value)=>String(value)).join(",");
        } else if (typeof (profileConfig?.default) === "string") {
            quartoProfile = profileConfig.default;
        }
    }
    const active = readProfile(quartoProfile);
    if (active.length === 0) {
        if (Deno.env.get("RSTUDIO_PRODUCT") === "CONNECT") {
            active.push("connect");
        }
    }
    const groups = readProfileGroups(profileConfig);
    for (const group of groups){
        if (!group.some((name)=>active.includes(name))) {
            active.push(group[0]);
        }
    }
    const updatedQuartoProfile = active.join(",");
    if (!firstRun) {
        if (Deno.env.get(kQuartoProfile) !== updatedQuartoProfile) {
            fireActiveProfileChanged(updatedQuartoProfile);
        }
    }
    Deno.env.set(kQuartoProfile, active.join(","));
    return await mergeProfiles(dir, config, schema);
}
const listeners1 = new Array();
function fireActiveProfileChanged(profile) {
    listeners1.forEach((listener)=>listener(profile));
}
function onActiveProfileChanged(listener) {
    listeners1.push(listener);
}
function monitorPreviewTerminationConditions(cleanup) {
    onActiveProfileChanged(()=>{
        terminatePreview("active profile changed", cleanup);
    });
    onDotenvChanged(()=>{
        terminatePreview("environment variables changed", cleanup);
    });
    if (quartoConfig.isDebug()) {
        const srcDir = Deno.realPathSync(join3(quartoConfig.binPath(), "../../../src"));
        const watcher = Deno.watchFs([
            srcDir
        ], {
            recursive: true
        });
        const watchForChanges = async ()=>{
            for await (const event of watcher){
                if (event.paths.some((path)=>extname2(path).toLowerCase() === ".ts")) {
                    terminatePreview("quarto src code changed", cleanup);
                }
            }
        };
        watchForChanges();
    }
}
function terminatePreview(reason, cleanup) {
    info(bold1(blue1(`\n${reason}: preview terminating\n`)));
    if (cleanup) {
        cleanup();
    }
    exitWithCleanup(1);
}
async function localConfigQuartoProfile(dir, schema) {
    const localConfigPath = localProjectConfigFile(dir);
    if (localConfigPath) {
        const yaml = await readAndValidateYamlFromFile(localConfigPath, schema, `Validation of configuration profile file ${basename2(localConfigPath)} failed.`, "{}");
        const profile = yaml[kQuartoProfileConfig];
        if (Array.isArray(profile?.default)) {
            return profile?.default.join(",");
        } else if (typeof (profile?.default) === "string") {
            return profile?.default;
        } else {
            return undefined;
        }
    } else {
        return undefined;
    }
}
async function mergeProfiles(dir, config, schema) {
    const files = [];
    const mergeProfile = async (profilePath)=>{
        try {
            const yaml = await readAndValidateYamlFromFile(profilePath, schema, `Validation of configuration profile file ${basename2(profilePath)} failed.`, "{}");
            config = mergeProjectMetadata(config, yaml);
            files.push(profilePath);
        } catch (e) {
            error("\nError reading configuration profile file from " + basename2(profilePath) + "\n");
            throw e;
        }
    };
    for (const profileName of activeProfiles().reverse()){
        const profilePath = [
            ".yml",
            ".yaml"
        ].map((ext)=>join3(dir, `_quarto-${profileName}${ext}`)).find(safeExistsSync);
        if (profilePath) {
            await mergeProfile(profilePath);
        }
    }
    const localConfigPath = localProjectConfigFile(dir);
    if (localConfigPath) {
        await mergeProfile(localConfigPath);
    }
    return {
        config,
        files
    };
}
function localProjectConfigFile(dir) {
    return [
        ".yml",
        ".yaml"
    ].map((ext)=>join3(dir, `_quarto${ext}.local`)).find(safeExistsSync);
}
function readProfileGroups(profileConfig) {
    const groups = [];
    const configGroup = profileConfig?.group;
    if (Array.isArray(configGroup)) {
        if (configGroup.every((value)=>typeof value === "string")) {
            groups.push(configGroup);
        } else if (configGroup.every(Array.isArray)) {
            groups.push(...configGroup);
        }
    }
    return groups;
}
const kStdOut = "-";
async function parseRenderFlags(args) {
    const flags = {};
    const argsStack = [
        ...args
    ];
    let arg = argsStack.shift();
    while(arg !== undefined){
        switch(arg){
            case "-t":
            case "--to":
                arg = argsStack.shift();
                if (arg && !arg.startsWith("-")) {
                    flags.to = arg;
                }
                break;
            case "-o":
            case "--output":
                arg = argsStack.shift();
                if (!arg || arg.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    if (arg.match(SEP_PATTERN)) {
                        throw new Error("--output option cannot specify a relative or absolute path");
                    }
                    flags.output = arg;
                }
                break;
            case "--output-dir":
                arg = argsStack.shift();
                flags.outputDir = arg;
                break;
            case "--site-url":
                arg = argsStack.shift();
                flags.siteUrl = arg;
                break;
            case "--standalone":
                flags[kStandalone] = true;
                arg = argsStack.shift();
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg = argsStack.shift();
                break;
            case "--embed-resources":
                flags[kEmbedResources] = true;
                arg = argsStack.shift();
                break;
            case "--pdf-engine":
                arg = argsStack.shift();
                flags.pdfEngine = arg;
                break;
            case "--pdf-engine-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg);
                }
                break;
            case "--latex-makeindex-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg);
                }
                break;
            case "--latex-tlmgr-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg);
                }
                break;
            case "--natbib":
                arg = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg);
                break;
            case "--top-level-division":
                arg = argsStack.shift();
                flags[kTopLevelDivision] = arg;
                break;
            case "--shift-heading-level-by":
                arg = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include = arg.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include] = includeFlags[include] || [];
                    arg = argsStack.shift();
                    includeFlags[include].push(arg);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg = argsStack.shift();
                break;
            case "--execute-params":
                arg = argsStack.shift();
                flags.paramsFile = arg;
                break;
            case "--execute-dir":
                arg = argsStack.shift();
                if (arg) {
                    if (isAbsolute2(arg)) {
                        flags.executeDir = arg;
                    } else {
                        flags.executeDir = Deno.realPathSync(arg);
                    }
                }
                break;
            case "--execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = parseInt(arg, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--use-freezer":
                arg = argsStack.shift();
                flags.useFreezer = true;
                break;
            case "--cache":
                arg = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--clean":
                arg = argsStack.shift();
                flags.clean = true;
                break;
            case "--no-clean":
                arg = argsStack.shift();
                flags.clean = false;
                break;
            case "--debug":
                flags.debug = true;
                arg = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg = argsStack.shift();
                if (arg) {
                    const param = parseMetadataFlagValue(arg);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {};
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg = argsStack.shift();
                if (arg) {
                    const metadata = parseMetadataFlagValue(arg);
                    if (metadata) {
                        if (metadata.value !== undefined) {
                            if (isQuartoMetadata(metadata.name)) {
                                flags.metadata = flags.metadata || {};
                                flags.metadata[metadata.name] = metadata.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {};
                                flags.pandocMetadata[metadata.name] = metadata.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg = argsStack.shift();
                if (arg) {
                    if (existsSync(arg)) {
                        const metadata1 = await readYamlFromString(Deno.readTextFileSync(arg));
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata1
                        };
                    }
                }
                break;
            case "--reference-location":
                arg = argsStack.shift();
                if (arg) {
                    flags[kReferenceLocation] = arg;
                }
                break;
            default:
                arg = argsStack.shift();
                break;
        }
    }
    if (flags.clean === undefined) {
        flags.clean = true;
    }
    return flags;
}
function havePandocArg(pandocArgs, arg) {
    return pandocArgs.indexOf(arg) !== -1;
}
function replacePandocArg(pandocArgs, arg, value) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value;
    } else {
        newArgs.push(arg);
        newArgs.push(value);
    }
    return newArgs;
}
function getPandocArg(pandocArgs, arg) {
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1 && argIndex + 1 < pandocArgs.length) {
        return pandocArgs[argIndex + 1];
    } else {
        return undefined;
    }
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args, arg, index)=>{
        args.push(arg);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg === "-o" || arg === "--output")) {
            args.push(kStdOut);
        }
        return args;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--site-url", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--use-freezer", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--clean", false);
    removeArgs.set("--no-clean", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    removeArgs.set("--profile", true);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    let removeNext = false;
    return pandocArgs.reduce((args, arg)=>{
        if (!removeArgs.has(arg)) {
            if (!removeNext) {
                args.push(arg);
            }
            removeNext = false;
        } else {
            removeNext = removeArgs.get(arg);
        }
        return args;
    }, new Array());
}
function removePandocToArg(args) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args, removeArgs);
}
function removePandocTo(renderOptions) {
    renderOptions = cloneDeep1(renderOptions);
    delete renderOptions.flags?.to;
    if (renderOptions.pandocArgs) {
        renderOptions.pandocArgs = removePandocToArg(renderOptions.pandocArgs);
    }
    return renderOptions;
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args = [];
    for(let i1 = 0; i1 < pandocArgs.length; i1++){
        const arg = pandocArgs[i1];
        if (arg === "--metadata" || arg === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i1 + 1] || "");
            if (flagValue !== undefined && (isQuartoMetadata(flagValue.name) || kQuartoForwardedMetadataFields.includes(flagValue.name))) {
                i1++;
            } else {
                args.push(arg);
            }
        } else {
            args.push(arg);
        }
    }
    return args;
}
const kQuartoForwardedMetadataFields = [
    kAuthor,
    kDate
];
function parseMetadataFlagValue(arg) {
    const match = arg.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {};
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value) {
    if (value) {
        const numbers = value.split(/,/).map((number)=>parseInt(number.trim(), 10)).filter((number)=>!isNaN(number));
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
const importMeta = {
    url: "file:///home/runner/work/quarto-cli/quarto-cli/src/vendor/deno.land/x/deno_dom@v0.1.20-alpha/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder('utf-8');
const encodeString = typeof cachedTextEncoder.encodeInto === 'function' ? function(arg, view) {
    return cachedTextEncoder.encodeInto(arg, view);
} : function(arg, view) {
    const buf = cachedTextEncoder.encode(arg);
    view.set(buf);
    return {
        read: arg.length,
        written: buf.length
    };
};
function passStringToWasm0(arg, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len = arg.length;
    let ptr1 = malloc(len);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len; offset++){
        const code = arg.charCodeAt(offset);
        if (code > 0x7F) break;
        mem[ptr1 + offset] = code;
    }
    if (offset !== len) {
        if (offset !== 0) {
            arg = arg.slice(offset);
        }
        ptr1 = realloc(ptr1, len, len = offset + arg.length * 3);
        const view = getUint8Memory0().subarray(ptr1 + offset, ptr1 + len);
        const ret = encodeString(arg, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr1;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder('utf-8', {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len));
}
function parse5(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load2(module, imports) {
    if (typeof Response === 'function' && module instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === 'function') {
            try {
                return await WebAssembly.instantiateStreaming(module, imports);
            } catch (e) {
                if (module.headers.get('Content-Type') != 'application/wasm') {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module
            };
        } else {
            return instance;
        }
    }
}
async function init(input) {
    if (typeof input === 'undefined') {
        input = new URL('deno-wasm_bg.wasm', importMeta.url);
    }
    const imports = {};
    imports.env = {
        now () {}
    };
    input = Uint8Array.from(atob("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