const QUOTE = '"';
export const NEWLINE = "\r\n";
export class StringifyError extends Error {
    name = "StringifyError";
}
function getEscapedString(value, sep) {
    if (value === undefined || value === null)
        return "";
    let str = "";
    if (typeof value === "object")
        str = JSON.stringify(value);
    else
        str = String(value);
    if (str.includes(sep) || str.includes(NEWLINE) || str.includes(QUOTE)) {
        return `${QUOTE}${str.replaceAll(QUOTE, `${QUOTE}${QUOTE}`)}${QUOTE}`;
    }
    return str;
}
function normalizeColumn(column) {
    let fn, header, prop;
    if (typeof column === "object") {
        if (Array.isArray(column)) {
            header = String(column[column.length - 1]);
            prop = column;
        }
        else {
            ({ fn } = column);
            prop = Array.isArray(column.prop) ? column.prop : [column.prop];
            header = typeof column.header === "string"
                ? column.header
                : String(prop[prop.length - 1]);
        }
    }
    else {
        header = String(column);
        prop = [column];
    }
    return { fn, header, prop };
}
async function getValuesFromItem(item, normalizedColumns) {
    const values = [];
    for (const column of normalizedColumns) {
        let value = item;
        for (const prop of column.prop) {
            if (typeof value !== "object" || value === null)
                continue;
            if (Array.isArray(value)) {
                if (typeof prop === "number")
                    value = value[prop];
                else {
                    throw new StringifyError('Property accessor is not of type "number"');
                }
            }
            else
                value = value[prop];
        }
        if (typeof column.fn === "function")
            value = await column.fn(value);
        values.push(value);
    }
    return values;
}
export async function stringify(data, columns, options = {}) {
    const { headers, separator: sep } = {
        headers: true,
        separator: ",",
        ...options,
    };
    if (sep.includes(QUOTE) || sep.includes(NEWLINE)) {
        const message = [
            "Separator cannot include the following strings:",
            '  - U+0022: Quotation mark (")',
            "  - U+000D U+000A: Carriage Return + Line Feed (\\r\\n)",
        ].join("\n");
        throw new StringifyError(message);
    }
    const normalizedColumns = columns.map(normalizeColumn);
    let output = "";
    if (headers) {
        output += normalizedColumns
            .map((column) => getEscapedString(column.header, sep))
            .join(sep);
        output += NEWLINE;
    }
    for (const item of data) {
        const values = await getValuesFromItem(item, normalizedColumns);
        output += values
            .map((value) => getEscapedString(value, sep))
            .join(sep);
        output += NEWLINE;
    }
    return output;
}
//# sourceMappingURL=data:application/json;base64,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