import { assert } from "../_util/assert.ts";
import * as streams from "../streams/conversion.ts";
import * as files from "./files.ts";
const DEFAULT_BUFFER_SIZE = 32 * 1024;
export async function copyN(r, dest, size) {
    let bytesRead = 0;
    let buf = new Uint8Array(DEFAULT_BUFFER_SIZE);
    while (bytesRead < size) {
        if (size - bytesRead < DEFAULT_BUFFER_SIZE) {
            buf = new Uint8Array(size - bytesRead);
        }
        const result = await r.read(buf);
        const nread = result ?? 0;
        bytesRead += nread;
        if (nread > 0) {
            let n = 0;
            while (n < nread) {
                n += await dest.write(buf.slice(n, nread));
            }
            assert(n === nread, "could not write");
        }
        if (result === null) {
            break;
        }
    }
    return bytesRead;
}
export async function readShort(buf) {
    const high = await buf.readByte();
    if (high === null)
        return null;
    const low = await buf.readByte();
    if (low === null)
        throw new Deno.errors.UnexpectedEof();
    return (high << 8) | low;
}
export async function readInt(buf) {
    const high = await readShort(buf);
    if (high === null)
        return null;
    const low = await readShort(buf);
    if (low === null)
        throw new Deno.errors.UnexpectedEof();
    return (high << 16) | low;
}
const MAX_SAFE_INTEGER = BigInt(Number.MAX_SAFE_INTEGER);
export async function readLong(buf) {
    const high = await readInt(buf);
    if (high === null)
        return null;
    const low = await readInt(buf);
    if (low === null)
        throw new Deno.errors.UnexpectedEof();
    const big = (BigInt(high) << 32n) | BigInt(low);
    if (big > MAX_SAFE_INTEGER) {
        throw new RangeError("Long value too big to be represented as a JavaScript number.");
    }
    return Number(big);
}
export function sliceLongToBytes(d, dest = Array.from({ length: 8 })) {
    let big = BigInt(d);
    for (let i = 0; i < 8; i++) {
        dest[7 - i] = Number(big & 0xffn);
        big >>= 8n;
    }
    return dest;
}
export const readAll = streams.readAll;
export const readAllSync = streams.readAllSync;
export const writeAll = streams.writeAll;
export const writeAllSync = streams.writeAllSync;
export const iter = streams.iterateReader;
export const iterSync = streams.iterateReaderSync;
export const copy = streams.copy;
export const readRange = files.readRange;
export const readRangeSync = files.readRangeSync;
//# sourceMappingURL=data:application/json;base64,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