import { ascend, BSTree } from "./bs_tree.ts";
import { RBNode } from "./rb_node.ts";
export * from "./_comparators.ts";
export class RBTree extends BSTree {
    constructor(compare = ascend) {
        super(compare);
    }
    static from(collection, options) {
        let result;
        let unmappedValues = [];
        if (collection instanceof RBTree) {
            result = new RBTree(options?.compare ?? collection.compare);
            if (options?.compare || options?.map) {
                unmappedValues = collection;
            }
            else {
                const nodes = [];
                if (collection.root) {
                    result.root = RBNode.from(collection.root);
                    nodes.push(result.root);
                }
                while (nodes.length) {
                    const node = nodes.pop();
                    const left = node.left
                        ? RBNode.from(node.left)
                        : null;
                    const right = node.right
                        ? RBNode.from(node.right)
                        : null;
                    if (left) {
                        left.parent = node;
                        nodes.push(left);
                    }
                    if (right) {
                        right.parent = node;
                        nodes.push(right);
                    }
                }
            }
        }
        else {
            result = (options?.compare
                ? new RBTree(options.compare)
                : new RBTree());
            unmappedValues = collection;
        }
        const values = options?.map
            ? Array.from(unmappedValues, options.map, options.thisArg)
            : unmappedValues;
        for (const value of values)
            result.insert(value);
        return result;
    }
    removeFixup(parent, current) {
        while (parent && !current?.red) {
            const direction = parent.left === current ? "left" : "right";
            const siblingDirection = direction === "right"
                ? "left"
                : "right";
            let sibling = parent[siblingDirection];
            if (sibling?.red) {
                sibling.red = false;
                parent.red = true;
                this.rotateNode(parent, direction);
                sibling = parent[siblingDirection];
            }
            if (sibling) {
                if (!sibling.left?.red && !sibling.right?.red) {
                    sibling.red = true;
                    current = parent;
                    parent = current.parent;
                }
                else {
                    if (!sibling[siblingDirection]?.red) {
                        sibling[direction].red = false;
                        sibling.red = true;
                        this.rotateNode(sibling, siblingDirection);
                        sibling = parent[siblingDirection];
                    }
                    sibling.red = parent.red;
                    parent.red = false;
                    sibling[siblingDirection].red = false;
                    this.rotateNode(parent, direction);
                    current = this.root;
                    parent = null;
                }
            }
        }
        if (current)
            current.red = false;
    }
    insert(value) {
        let node = this.insertNode(RBNode, value);
        if (node) {
            while (node.parent?.red) {
                let parent = node.parent;
                const parentDirection = parent.directionFromParent();
                const uncleDirection = parentDirection === "right"
                    ? "left"
                    : "right";
                const uncle = parent.parent[uncleDirection] ?? null;
                if (uncle?.red) {
                    parent.red = false;
                    uncle.red = false;
                    parent.parent.red = true;
                    node = parent.parent;
                }
                else {
                    if (node === parent[uncleDirection]) {
                        node = parent;
                        this.rotateNode(node, parentDirection);
                        parent = node.parent;
                    }
                    parent.red = false;
                    parent.parent.red = true;
                    this.rotateNode(parent.parent, uncleDirection);
                }
            }
            this.root.red = false;
        }
        return !!node;
    }
    remove(value) {
        const node = this.removeNode(value);
        if (node && !node.red) {
            this.removeFixup(node.parent, node.left ?? node.right);
        }
        return !!node;
    }
}
//# sourceMappingURL=data:application/json;base64,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