import { MultiReader } from "../io/readers.ts";
import { Buffer, PartialReadError } from "../io/buffer.ts";
import { assert } from "../_util/assert.ts";
import { readAll } from "../streams/conversion.ts";
const recordSize = 512;
const ustar = "ustar\u000000";
const initialChecksum = 8 * 32;
async function readBlock(reader, p) {
    let bytesRead = 0;
    while (bytesRead < p.length) {
        const rr = await reader.read(p.subarray(bytesRead));
        if (rr === null) {
            if (bytesRead === 0) {
                return null;
            }
            else {
                throw new PartialReadError();
            }
        }
        bytesRead += rr;
    }
    return bytesRead;
}
class FileReader {
    filePath;
    #file;
    constructor(filePath) {
        this.filePath = filePath;
    }
    async read(p) {
        if (!this.#file) {
            this.#file = await Deno.open(this.filePath, { read: true });
        }
        const res = await Deno.read(this.#file.rid, p);
        if (res === null) {
            Deno.close(this.#file.rid);
            this.#file = undefined;
        }
        return res;
    }
}
function trim(buffer) {
    const index = buffer.findIndex((v) => v === 0);
    if (index < 0)
        return buffer;
    return buffer.subarray(0, index);
}
function clean(length) {
    const buffer = new Uint8Array(length);
    buffer.fill(0, 0, length - 1);
    return buffer;
}
function pad(num, bytes, base = 8) {
    const numString = num.toString(base);
    return "000000000000".substr(numString.length + 12 - bytes) + numString;
}
var FileTypes;
(function (FileTypes) {
    FileTypes[FileTypes["file"] = 0] = "file";
    FileTypes[FileTypes["link"] = 1] = "link";
    FileTypes[FileTypes["symlink"] = 2] = "symlink";
    FileTypes[FileTypes["character-device"] = 3] = "character-device";
    FileTypes[FileTypes["block-device"] = 4] = "block-device";
    FileTypes[FileTypes["directory"] = 5] = "directory";
    FileTypes[FileTypes["fifo"] = 6] = "fifo";
    FileTypes[FileTypes["contiguous-file"] = 7] = "contiguous-file";
})(FileTypes || (FileTypes = {}));
const ustarStructure = [
    {
        field: "fileName",
        length: 100,
    },
    {
        field: "fileMode",
        length: 8,
    },
    {
        field: "uid",
        length: 8,
    },
    {
        field: "gid",
        length: 8,
    },
    {
        field: "fileSize",
        length: 12,
    },
    {
        field: "mtime",
        length: 12,
    },
    {
        field: "checksum",
        length: 8,
    },
    {
        field: "type",
        length: 1,
    },
    {
        field: "linkName",
        length: 100,
    },
    {
        field: "ustar",
        length: 8,
    },
    {
        field: "owner",
        length: 32,
    },
    {
        field: "group",
        length: 32,
    },
    {
        field: "majorNumber",
        length: 8,
    },
    {
        field: "minorNumber",
        length: 8,
    },
    {
        field: "fileNamePrefix",
        length: 155,
    },
    {
        field: "padding",
        length: 12,
    },
];
function formatHeader(data) {
    const encoder = new TextEncoder(), buffer = clean(512);
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const entry = encoder.encode(data[value.field] || "");
        buffer.set(entry, offset);
        offset += value.length;
    });
    return buffer;
}
function parseHeader(buffer) {
    const data = {};
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const arr = buffer.subarray(offset, offset + value.length);
        data[value.field] = arr;
        offset += value.length;
    });
    return data;
}
export class Tar {
    data;
    constructor() {
        this.data = [];
    }
    async append(fn, opts) {
        if (typeof fn !== "string") {
            throw new Error("file name not specified");
        }
        let fileName = fn;
        let fileNamePrefix;
        if (fileName.length > 100) {
            let i = fileName.length;
            while (i >= 0) {
                i = fileName.lastIndexOf("/", i);
                if (i <= 155) {
                    fileNamePrefix = fileName.substr(0, i);
                    fileName = fileName.substr(i + 1);
                    break;
                }
                i--;
            }
            const errMsg = "ustar format does not allow a long file name (length of [file name" +
                "prefix] + / + [file name] must be shorter than 256 bytes)";
            if (i < 0 || fileName.length > 100) {
                throw new Error(errMsg);
            }
            else {
                assert(fileNamePrefix != null);
                if (fileNamePrefix.length > 155) {
                    throw new Error(errMsg);
                }
            }
        }
        opts = opts || {};
        let info;
        if (opts.filePath) {
            info = await Deno.stat(opts.filePath);
            if (info.isDirectory) {
                info.size = 0;
                opts.reader = new Buffer();
            }
        }
        const mode = opts.fileMode || (info && info.mode) ||
            parseInt("777", 8) & 0xfff, mtime = Math.floor(opts.mtime ?? (info?.mtime ?? new Date()).valueOf() / 1000), uid = opts.uid || 0, gid = opts.gid || 0;
        if (typeof opts.owner === "string" && opts.owner.length >= 32) {
            throw new Error("ustar format does not allow owner name length >= 32 bytes");
        }
        if (typeof opts.group === "string" && opts.group.length >= 32) {
            throw new Error("ustar format does not allow group name length >= 32 bytes");
        }
        const fileSize = info?.size ?? opts.contentSize;
        assert(fileSize != null, "fileSize must be set");
        const type = opts.type
            ? FileTypes[opts.type]
            : (info?.isDirectory ? FileTypes.directory : FileTypes.file);
        const tarData = {
            fileName,
            fileNamePrefix,
            fileMode: pad(mode, 7),
            uid: pad(uid, 7),
            gid: pad(gid, 7),
            fileSize: pad(fileSize, 11),
            mtime: pad(mtime, 11),
            checksum: "        ",
            type: type.toString(),
            ustar,
            owner: opts.owner || "",
            group: opts.group || "",
            filePath: opts.filePath,
            reader: opts.reader,
        };
        let checksum = 0;
        const encoder = new TextEncoder();
        Object.keys(tarData)
            .filter((key) => ["filePath", "reader"].indexOf(key) < 0)
            .forEach(function (key) {
            checksum += encoder
                .encode(tarData[key])
                .reduce((p, c) => p + c, 0);
        });
        tarData.checksum = pad(checksum, 6) + "\u0000 ";
        this.data.push(tarData);
    }
    getReader() {
        const readers = [];
        this.data.forEach((tarData) => {
            let { reader } = tarData;
            const { filePath } = tarData;
            const headerArr = formatHeader(tarData);
            readers.push(new Buffer(headerArr));
            if (!reader) {
                assert(filePath != null);
                reader = new FileReader(filePath);
            }
            readers.push(reader);
            assert(tarData.fileSize != null, "fileSize must be set");
            readers.push(new Buffer(clean(recordSize -
                (parseInt(tarData.fileSize, 8) % recordSize || recordSize))));
        });
        readers.push(new Buffer(clean(recordSize * 2)));
        return new MultiReader(readers);
    }
}
class TarEntry {
    #header;
    #reader;
    #size;
    #read = 0;
    #consumed = false;
    #entrySize;
    constructor(meta, header, reader) {
        Object.assign(this, meta);
        this.#header = header;
        this.#reader = reader;
        this.#size = this.fileSize || 0;
        const blocks = Math.ceil(this.#size / recordSize);
        this.#entrySize = blocks * recordSize;
    }
    get consumed() {
        return this.#consumed;
    }
    async read(p) {
        const entryBytesLeft = this.#entrySize - this.#read;
        const bufSize = Math.min(p.length, entryBytesLeft);
        if (entryBytesLeft <= 0) {
            this.#consumed = true;
            return null;
        }
        const block = new Uint8Array(bufSize);
        const n = await readBlock(this.#reader, block);
        const bytesLeft = this.#size - this.#read;
        this.#read += n || 0;
        if (n === null || bytesLeft <= 0) {
            if (n === null)
                this.#consumed = true;
            return null;
        }
        const offset = bytesLeft < n ? bytesLeft : n;
        p.set(block.subarray(0, offset), 0);
        return offset < 0 ? n - Math.abs(offset) : offset;
    }
    async discard() {
        if (this.#consumed)
            return;
        this.#consumed = true;
        if (typeof this.#reader.seek === "function") {
            await this.#reader.seek(this.#entrySize - this.#read, Deno.SeekMode.Current);
            this.#read = this.#entrySize;
        }
        else {
            await readAll(this);
        }
    }
}
export class Untar {
    reader;
    block;
    #entry;
    constructor(reader) {
        this.reader = reader;
        this.block = new Uint8Array(recordSize);
    }
    #checksum(header) {
        let sum = initialChecksum;
        for (let i = 0; i < 512; i++) {
            if (i >= 148 && i < 156) {
                continue;
            }
            sum += header[i];
        }
        return sum;
    }
    async #getHeader() {
        await readBlock(this.reader, this.block);
        const header = parseHeader(this.block);
        const decoder = new TextDecoder();
        const checksum = this.#checksum(this.block);
        if (parseInt(decoder.decode(header.checksum), 8) !== checksum) {
            if (checksum === initialChecksum) {
                return null;
            }
            throw new Error("checksum error");
        }
        const magic = decoder.decode(header.ustar);
        if (magic.indexOf("ustar")) {
            throw new Error(`unsupported archive format: ${magic}`);
        }
        return header;
    }
    #getMetadata(header) {
        const decoder = new TextDecoder();
        const meta = {
            fileName: decoder.decode(trim(header.fileName)),
        };
        const fileNamePrefix = trim(header.fileNamePrefix);
        if (fileNamePrefix.byteLength > 0) {
            meta.fileName = decoder.decode(fileNamePrefix) + "/" + meta.fileName;
        }
        ["fileMode", "mtime", "uid", "gid"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = parseInt(decoder.decode(arr), 8);
            }
        });
        ["owner", "group", "type"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = decoder.decode(arr);
            }
        });
        meta.fileSize = parseInt(decoder.decode(header.fileSize), 8);
        meta.type = FileTypes[parseInt(meta.type)] ?? meta.type;
        return meta;
    }
    async extract() {
        if (this.#entry && !this.#entry.consumed) {
            await this.#entry.discard();
        }
        const header = await this.#getHeader();
        if (header === null)
            return null;
        const meta = this.#getMetadata(header);
        this.#entry = new TarEntry(meta, header, this.reader);
        return this.#entry;
    }
    async *[Symbol.asyncIterator]() {
        while (true) {
            const entry = await this.extract();
            if (entry === null)
                return;
            yield entry;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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