import { BufReader } from "../io/buffer.ts";
import { TextProtoReader } from "../textproto/mod.ts";
import { StringReader } from "../io/readers.ts";
import { assert } from "../_util/assert.ts";
import { ERR_FIELD_COUNT, ERR_INVALID_DELIM, ParseError, readRecord, } from "./csv/_io.ts";
export { ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError, } from "./csv/_io.ts";
export { NEWLINE, stringify, StringifyError } from "./csv_stringify.ts";
class TextProtoLineReader {
    #tp;
    constructor(bufReader) {
        this.#tp = new TextProtoReader(bufReader);
    }
    async readLine() {
        let line;
        const r = await this.#tp.readLine();
        if (r === null)
            return null;
        line = r;
        if ((await this.isEOF()) && line.length > 0 && line[line.length - 1] === "\r") {
            line = line.substring(0, line.length - 1);
        }
        if (line.length >= 2 &&
            line[line.length - 2] === "\r" &&
            line[line.length - 1] === "\n") {
            line = line.substring(0, line.length - 2);
            line = line + "\n";
        }
        return line;
    }
    async isEOF() {
        return (await this.#tp.r.peek(0)) === null;
    }
}
const INVALID_RUNE = ["\r", "\n", '"'];
function chkOptions(opt) {
    if (!opt.separator) {
        opt.separator = ",";
    }
    if (!opt.trimLeadingSpace) {
        opt.trimLeadingSpace = false;
    }
    if (INVALID_RUNE.includes(opt.separator) ||
        (typeof opt.comment === "string" && INVALID_RUNE.includes(opt.comment)) ||
        opt.separator === opt.comment) {
        throw new Error(ERR_INVALID_DELIM);
    }
}
export async function readMatrix(reader, opt = {
    separator: ",",
    trimLeadingSpace: false,
    lazyQuotes: false,
}) {
    const result = [];
    let _nbFields;
    let lineResult;
    let first = true;
    let lineIndex = 0;
    chkOptions(opt);
    const lineReader = new TextProtoLineReader(reader);
    for (;;) {
        const r = await readRecord(lineIndex, lineReader, opt);
        if (r === null)
            break;
        lineResult = r;
        lineIndex++;
        if (first) {
            first = false;
            if (opt.fieldsPerRecord !== undefined) {
                if (opt.fieldsPerRecord === 0) {
                    _nbFields = lineResult.length;
                }
                else {
                    _nbFields = opt.fieldsPerRecord;
                }
            }
        }
        if (lineResult.length > 0) {
            if (_nbFields && _nbFields !== lineResult.length) {
                throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
            }
            result.push(lineResult);
        }
    }
    return result;
}
export async function parse(input, opt = {
    skipFirstRow: false,
}) {
    let r;
    if (input instanceof BufReader) {
        r = await readMatrix(input, opt);
    }
    else {
        r = await readMatrix(new BufReader(new StringReader(input)), opt);
    }
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head.map((e) => {
                return {
                    name: e,
                };
            });
            i++;
        }
        if (opt.columns) {
            if (typeof opt.columns[0] !== "string") {
                headers = opt.columns;
            }
            else {
                const h = opt.columns;
                headers = h.map((e) => {
                    return {
                        name: e,
                    };
                });
            }
        }
        return r.map((e) => {
            if (e.length !== headers.length) {
                throw new Error(`Error number of fields line: ${i}\nNumber of fields found: ${headers.length}\nExpected number of fields: ${e.length}`);
            }
            i++;
            const out = {};
            for (let j = 0; j < e.length; j++) {
                out[headers[j].name] = e[j];
            }
            return out;
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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