import { equals, indexOfNeedle, lastIndexOfNeedle, startsWith, } from "../bytes/mod.ts";
import { Buffer, BufReader, BufWriter } from "../io/buffer.ts";
import { copy } from "../streams/conversion.ts";
import { copyN } from "../io/util.ts";
import { MultiReader } from "../io/readers.ts";
import { extname } from "../path/mod.ts";
import { assert } from "../_util/assert.ts";
import { TextProtoReader } from "../textproto/mod.ts";
const { hasOwn } = Object;
export function isFormFile(x) {
    return hasOwn(x, "filename") && hasOwn(x, "type");
}
function randomBoundary() {
    let boundary = "--------------------------";
    for (let i = 0; i < 24; i++) {
        boundary += Math.floor(Math.random() * 16).toString(16);
    }
    return boundary;
}
const encoder = new TextEncoder();
export function matchAfterPrefix(buf, prefix, eof) {
    if (buf.length === prefix.length) {
        return eof ? 1 : 0;
    }
    const c = buf[prefix.length];
    if (c === " ".charCodeAt(0) ||
        c === "\t".charCodeAt(0) ||
        c === "\r".charCodeAt(0) ||
        c === "\n".charCodeAt(0) ||
        c === "-".charCodeAt(0)) {
        return 1;
    }
    return -1;
}
export function scanUntilBoundary(buf, dashBoundary, newLineDashBoundary, total, eof) {
    if (total === 0) {
        if (startsWith(buf, dashBoundary)) {
            switch (matchAfterPrefix(buf, dashBoundary, eof)) {
                case -1:
                    return dashBoundary.length;
                case 0:
                    return 0;
                case 1:
                    return null;
            }
        }
        if (startsWith(dashBoundary, buf)) {
            return 0;
        }
    }
    const i = indexOfNeedle(buf, newLineDashBoundary);
    if (i >= 0) {
        switch (matchAfterPrefix(buf.slice(i), newLineDashBoundary, eof)) {
            case -1:
                return i + newLineDashBoundary.length;
            case 0:
                return i;
            case 1:
                return i > 0 ? i : null;
        }
    }
    if (startsWith(newLineDashBoundary, buf)) {
        return 0;
    }
    const j = lastIndexOfNeedle(buf, newLineDashBoundary.slice(0, 1));
    if (j >= 0 && startsWith(newLineDashBoundary, buf.slice(j))) {
        return j;
    }
    return buf.length;
}
class PartReader {
    mr;
    headers;
    n = 0;
    total = 0;
    constructor(mr, headers) {
        this.mr = mr;
        this.headers = headers;
    }
    async read(p) {
        const br = this.mr.bufReader;
        let peekLength = 1;
        while (this.n === 0) {
            peekLength = Math.max(peekLength, br.buffered());
            const peekBuf = await br.peek(peekLength);
            if (peekBuf === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            const eof = peekBuf.length < peekLength;
            this.n = scanUntilBoundary(peekBuf, this.mr.dashBoundary, this.mr.newLineDashBoundary, this.total, eof);
            if (this.n === 0) {
                assert(eof === false);
                peekLength++;
            }
        }
        if (this.n === null) {
            return null;
        }
        const nread = Math.min(p.length, this.n);
        const buf = p.subarray(0, nread);
        const r = await br.readFull(buf);
        assert(r === buf);
        this.n -= nread;
        this.total += nread;
        return nread;
    }
    close() { }
    #contentDisposition;
    #contentDispositionParams;
    #getContentDispositionParams() {
        if (this.#contentDispositionParams)
            return this.#contentDispositionParams;
        const cd = this.headers.get("content-disposition");
        const params = {};
        assert(cd != null, "content-disposition must be set");
        const comps = decodeURI(cd).split(";");
        this.#contentDisposition = comps[0];
        comps
            .slice(1)
            .map((v) => v.trim())
            .map((kv) => {
            const [k, v] = kv.split("=");
            if (v) {
                const s = v.charAt(0);
                const e = v.charAt(v.length - 1);
                if ((s === e && s === '"') || s === "'") {
                    params[k] = v.substr(1, v.length - 2);
                }
                else {
                    params[k] = v;
                }
            }
        });
        return (this.#contentDispositionParams = params);
    }
    get fileName() {
        return this.#getContentDispositionParams()["filename"];
    }
    get formName() {
        const p = this.#getContentDispositionParams();
        if (this.#contentDisposition === "form-data") {
            return p["name"];
        }
        return "";
    }
}
function skipLWSPChar(u) {
    const ret = new Uint8Array(u.length);
    const sp = " ".charCodeAt(0);
    const ht = "\t".charCodeAt(0);
    let j = 0;
    for (let i = 0; i < u.length; i++) {
        if (u[i] === sp || u[i] === ht)
            continue;
        ret[j++] = u[i];
    }
    return ret.slice(0, j);
}
export class MultipartReader {
    boundary;
    newLine;
    newLineDashBoundary;
    dashBoundaryDash;
    dashBoundary;
    bufReader;
    constructor(reader, boundary) {
        this.boundary = boundary;
        this.newLine = encoder.encode("\r\n");
        this.newLineDashBoundary = encoder.encode(`\r\n--${boundary}`);
        this.dashBoundaryDash = encoder.encode(`--${this.boundary}--`);
        this.dashBoundary = encoder.encode(`--${this.boundary}`);
        this.bufReader = new BufReader(reader);
    }
    async readForm(maxMemoryOrOptions) {
        const options = typeof maxMemoryOrOptions === "number"
            ? { maxMemory: maxMemoryOrOptions }
            : maxMemoryOrOptions;
        let maxMemory = options?.maxMemory ?? 10 << 20;
        const fileMap = new Map();
        const valueMap = new Map();
        let maxValueBytes = maxMemory + (10 << 20);
        const buf = new Buffer(new Uint8Array(maxValueBytes));
        for (;;) {
            const p = await this.#nextPart();
            if (p === null) {
                break;
            }
            if (p.formName === "") {
                continue;
            }
            buf.reset();
            if (!p.fileName) {
                const n = await copyN(p, buf, maxValueBytes);
                maxValueBytes -= n;
                if (maxValueBytes < 0) {
                    throw new RangeError("message too large");
                }
                const value = new TextDecoder().decode(buf.bytes());
                const mapVal = valueMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(value);
                }
                else {
                    valueMap.set(p.formName, [value]);
                }
                continue;
            }
            let formFile;
            const n = await copyN(p, buf, maxValueBytes);
            const contentType = p.headers.get("content-type");
            assert(contentType != null, "content-type must be set");
            if (n > maxMemory) {
                const ext = extname(p.fileName);
                const filepath = await Deno.makeTempFile({
                    dir: options?.dir ?? ".",
                    prefix: options?.prefix ?? "multipart-",
                    suffix: options?.suffix ?? ext,
                });
                const file = await Deno.open(filepath, { write: true });
                try {
                    const size = await copy(new MultiReader([buf, p]), file);
                    file.close();
                    formFile = {
                        filename: p.fileName,
                        type: contentType,
                        tempfile: filepath,
                        size,
                    };
                }
                catch (e) {
                    await Deno.remove(filepath);
                    throw e;
                }
            }
            else {
                formFile = {
                    filename: p.fileName,
                    type: contentType,
                    content: buf.bytes(),
                    size: buf.length,
                };
                maxMemory -= n;
                maxValueBytes -= n;
            }
            if (formFile) {
                const mapVal = fileMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(formFile);
                }
                else {
                    fileMap.set(p.formName, [formFile]);
                }
            }
        }
        return multipartFormData(fileMap, valueMap);
    }
    #currentPart;
    #partsRead = 0;
    async #nextPart() {
        if (this.#currentPart) {
            this.#currentPart.close();
        }
        if (equals(this.dashBoundary, encoder.encode("--"))) {
            throw new Error("boundary is empty");
        }
        let expectNewPart = false;
        for (;;) {
            const line = await this.bufReader.readSlice("\n".charCodeAt(0));
            if (line === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            if (this.#isBoundaryDelimiterLine(line)) {
                this.#partsRead++;
                const r = new TextProtoReader(this.bufReader);
                const headers = await r.readMIMEHeader();
                if (headers === null) {
                    throw new Deno.errors.UnexpectedEof();
                }
                const np = new PartReader(this, headers);
                this.#currentPart = np;
                return np;
            }
            if (this.#isFinalBoundary(line)) {
                return null;
            }
            if (expectNewPart) {
                throw new Error(`expecting a new Part; got line ${line}`);
            }
            if (this.#partsRead === 0) {
                continue;
            }
            if (equals(line, this.newLine)) {
                expectNewPart = true;
                continue;
            }
            throw new Error(`unexpected line in nextPart(): ${line}`);
        }
    }
    #isFinalBoundary(line) {
        if (!startsWith(line, this.dashBoundaryDash)) {
            return false;
        }
        const rest = line.slice(this.dashBoundaryDash.length, line.length);
        return rest.length === 0 || equals(skipLWSPChar(rest), this.newLine);
    }
    #isBoundaryDelimiterLine(line) {
        if (!startsWith(line, this.dashBoundary)) {
            return false;
        }
        const rest = line.slice(this.dashBoundary.length);
        return equals(skipLWSPChar(rest), this.newLine);
    }
}
function multipartFormData(fileMap, valueMap) {
    function files(key) {
        return fileMap.get(key);
    }
    function values(key) {
        return valueMap.get(key);
    }
    function* entries() {
        yield* fileMap;
        yield* valueMap;
    }
    async function removeAll() {
        const promises = [];
        for (const val of fileMap.values()) {
            for (const subVal of val) {
                if (!subVal.tempfile)
                    continue;
                promises.push(Deno.remove(subVal.tempfile));
            }
        }
        await Promise.all(promises);
    }
    return {
        files,
        values,
        entries,
        removeAll,
        [Symbol.iterator]() {
            return entries();
        },
    };
}
class PartWriter {
    writer;
    boundary;
    headers;
    closed = false;
    #partHeader;
    #headersWritten = false;
    constructor(writer, boundary, headers, isFirstBoundary) {
        this.writer = writer;
        this.boundary = boundary;
        this.headers = headers;
        let buf = "";
        if (isFirstBoundary) {
            buf += `--${boundary}\r\n`;
        }
        else {
            buf += `\r\n--${boundary}\r\n`;
        }
        for (const [key, value] of headers.entries()) {
            buf += `${key}: ${value}\r\n`;
        }
        buf += `\r\n`;
        this.#partHeader = buf;
    }
    close() {
        this.closed = true;
    }
    async write(p) {
        if (this.closed) {
            throw new Error("part is closed");
        }
        if (!this.#headersWritten) {
            await this.writer.write(encoder.encode(this.#partHeader));
            this.#headersWritten = true;
        }
        return this.writer.write(p);
    }
}
function checkBoundary(b) {
    if (b.length < 1 || b.length > 70) {
        throw new Error(`invalid boundary length: ${b.length}`);
    }
    const end = b.length - 1;
    for (let i = 0; i < end; i++) {
        const c = b.charAt(i);
        if (!c.match(/[a-zA-Z0-9'()+_,\-./:=?]/) || (c === " " && i !== end)) {
            throw new Error("invalid boundary character: " + c);
        }
    }
    return b;
}
export class MultipartWriter {
    writer;
    #_boundary;
    get boundary() {
        return this.#_boundary;
    }
    #lastPart;
    #bufWriter;
    #isClosed = false;
    constructor(writer, boundary) {
        this.writer = writer;
        if (boundary !== void 0) {
            this.#_boundary = checkBoundary(boundary);
        }
        else {
            this.#_boundary = randomBoundary();
        }
        this.#bufWriter = new BufWriter(writer);
    }
    formDataContentType() {
        return `multipart/form-data; boundary=${this.boundary}`;
    }
    createPart(headers) {
        if (this.#isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.#lastPart) {
            this.#lastPart.close();
        }
        const part = new PartWriter(this.writer, this.boundary, headers, !this.#lastPart);
        this.#lastPart = part;
        return part;
    }
    createFormFile(field, filename) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"; filename="${filename}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    createFormField(field) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    async writeField(field, value) {
        const f = await this.createFormField(field);
        await f.write(encoder.encode(value));
    }
    async writeFile(field, filename, file) {
        const f = await this.createFormFile(field, filename);
        await copy(file, f);
    }
    #flush() {
        return this.#bufWriter.flush();
    }
    async close() {
        if (this.#isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.#lastPart) {
            this.#lastPart.close();
            this.#lastPart = void 0;
        }
        await this.writer.write(encoder.encode(`\r\n--${this.boundary}--\r\n`));
        await this.#flush();
        this.#isClosed = true;
    }
}
//# sourceMappingURL=data:application/json;base64,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