import { BufReader } from "../io/buffer.ts";
import { TextProtoReader } from "../textproto/mod.ts";
import { StringReader } from "../io/readers.ts";
import { assert } from "../_util/assert.ts";
import { ERR_FIELD_COUNT, ERR_INVALID_DELIM, ParseError, readRecord, } from "./csv/_io.ts";
export { ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError, } from "./csv/_io.ts";
export { NEWLINE, stringify, StringifyError } from "./csv_stringify.ts";
class TextProtoLineReader {
    #tp;
    constructor(bufReader) {
        this.#tp = new TextProtoReader(bufReader);
    }
    async readLine() {
        let line;
        const r = await this.#tp.readLine();
        if (r === null)
            return null;
        line = r;
        if ((await this.isEOF()) && line.length > 0 && line[line.length - 1] === "\r") {
            line = line.substring(0, line.length - 1);
        }
        if (line.length >= 2 &&
            line[line.length - 2] === "\r" &&
            line[line.length - 1] === "\n") {
            line = line.substring(0, line.length - 2);
            line = line + "\n";
        }
        return line;
    }
    async isEOF() {
        return (await this.#tp.r.peek(0)) === null;
    }
}
const INVALID_RUNE = ["\r", "\n", '"'];
function chkOptions(opt) {
    if (!opt.separator) {
        opt.separator = ",";
    }
    if (!opt.trimLeadingSpace) {
        opt.trimLeadingSpace = false;
    }
    if (INVALID_RUNE.includes(opt.separator) ||
        (typeof opt.comment === "string" && INVALID_RUNE.includes(opt.comment)) ||
        opt.separator === opt.comment) {
        throw new Error(ERR_INVALID_DELIM);
    }
}
export async function readMatrix(reader, opt = {
    separator: ",",
    trimLeadingSpace: false,
    lazyQuotes: false,
}) {
    const result = [];
    let _nbFields;
    let lineResult;
    let first = true;
    let lineIndex = 0;
    chkOptions(opt);
    const lineReader = new TextProtoLineReader(reader);
    for (;;) {
        const r = await readRecord(lineIndex, lineReader, opt);
        if (r === null)
            break;
        lineResult = r;
        lineIndex++;
        if (first) {
            first = false;
            if (opt.fieldsPerRecord !== undefined) {
                if (opt.fieldsPerRecord === 0) {
                    _nbFields = lineResult.length;
                }
                else {
                    _nbFields = opt.fieldsPerRecord;
                }
            }
        }
        if (lineResult.length > 0) {
            if (_nbFields && _nbFields !== lineResult.length) {
                throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
            }
            result.push(lineResult);
        }
    }
    return result;
}
export async function parse(input, opt = {
    skipFirstRow: false,
}) {
    let r;
    if (input instanceof BufReader) {
        r = await readMatrix(input, opt);
    }
    else {
        r = await readMatrix(new BufReader(new StringReader(input)), opt);
    }
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head.map((e) => {
                return {
                    name: e,
                };
            });
            i++;
        }
        if (opt.columns) {
            if (typeof opt.columns[0] !== "string") {
                headers = opt.columns;
            }
            else {
                const h = opt.columns;
                headers = h.map((e) => {
                    return {
                        name: e,
                    };
                });
            }
        }
        return r.map((e) => {
            if (e.length !== headers.length) {
                throw `Error number of fields line:${i}`;
            }
            i++;
            const out = {};
            for (let j = 0; j < e.length; j++) {
                out[headers[j].name] = e[j];
            }
            return out;
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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