const rawTypeSizes = {
    int8: 1,
    uint8: 1,
    int16: 2,
    uint16: 2,
    int32: 4,
    uint32: 4,
    int64: 8,
    uint64: 8,
    float32: 4,
    float64: 8,
};
export function sizeof(dataType) {
    return rawTypeSizes[dataType];
}
export async function readExact(r, b) {
    let totalRead = 0;
    do {
        const tmp = new Uint8Array(b.length - totalRead);
        const nRead = await r.read(tmp);
        if (nRead === null)
            throw new Deno.errors.UnexpectedEof();
        b.set(tmp, totalRead);
        totalRead += nRead;
    } while (totalRead < b.length);
}
export async function getNBytes(r, n) {
    const scratch = new Uint8Array(n);
    await readExact(r, scratch);
    return scratch;
}
export function varnum(b, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType))
        return null;
    const view = new DataView(b.buffer);
    switch (o.dataType) {
        case "int8":
            return view.getInt8(b.byteOffset);
        case "uint8":
            return view.getUint8(b.byteOffset);
        case "int16":
            return view.getInt16(b.byteOffset, littleEndian);
        case "uint16":
            return view.getUint16(b.byteOffset, littleEndian);
        case "int32":
            return view.getInt32(b.byteOffset, littleEndian);
        case "uint32":
            return view.getUint32(b.byteOffset, littleEndian);
        case "float32":
            return view.getFloat32(b.byteOffset, littleEndian);
        case "float64":
            return view.getFloat64(b.byteOffset, littleEndian);
    }
}
export function varbig(b, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType))
        return null;
    const view = new DataView(b.buffer);
    switch (o.dataType) {
        case "int8":
            return BigInt(view.getInt8(b.byteOffset));
        case "uint8":
            return BigInt(view.getUint8(b.byteOffset));
        case "int16":
            return BigInt(view.getInt16(b.byteOffset, littleEndian));
        case "uint16":
            return BigInt(view.getUint16(b.byteOffset, littleEndian));
        case "int32":
            return BigInt(view.getInt32(b.byteOffset, littleEndian));
        case "uint32":
            return BigInt(view.getUint32(b.byteOffset, littleEndian));
        case "int64":
            return view.getBigInt64(b.byteOffset, littleEndian);
        case "uint64":
            return view.getBigUint64(b.byteOffset, littleEndian);
    }
}
export function putVarnum(b, x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType))
        return 0;
    const view = new DataView(b.buffer);
    switch (o.dataType) {
        case "int8":
            view.setInt8(b.byteOffset, x);
            break;
        case "uint8":
            view.setUint8(b.byteOffset, x);
            break;
        case "int16":
            view.setInt16(b.byteOffset, x, littleEndian);
            break;
        case "uint16":
            view.setUint16(b.byteOffset, x, littleEndian);
            break;
        case "int32":
            view.setInt32(b.byteOffset, x, littleEndian);
            break;
        case "uint32":
            view.setUint32(b.byteOffset, x, littleEndian);
            break;
        case "float32":
            view.setFloat32(b.byteOffset, x, littleEndian);
            break;
        case "float64":
            view.setFloat64(b.byteOffset, x, littleEndian);
            break;
    }
    return sizeof(o.dataType);
}
export function putVarbig(b, x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const littleEndian = (o.endian ?? "big") === "little" ? true : false;
    if (b.length < sizeof(o.dataType))
        return 0;
    const view = new DataView(b.buffer);
    switch (o.dataType) {
        case "int8":
            view.setInt8(b.byteOffset, Number(x));
            break;
        case "uint8":
            view.setUint8(b.byteOffset, Number(x));
            break;
        case "int16":
            view.setInt16(b.byteOffset, Number(x), littleEndian);
            break;
        case "uint16":
            view.setUint16(b.byteOffset, Number(x), littleEndian);
            break;
        case "int32":
            view.setInt32(b.byteOffset, Number(x), littleEndian);
            break;
        case "uint32":
            view.setUint32(b.byteOffset, Number(x), littleEndian);
            break;
        case "int64":
            view.setBigInt64(b.byteOffset, x, littleEndian);
            break;
        case "uint64":
            view.setBigUint64(b.byteOffset, x, littleEndian);
            break;
    }
    return sizeof(o.dataType);
}
export async function readVarnum(r, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const scratch = await getNBytes(r, sizeof(o.dataType));
    return varnum(scratch, o);
}
export async function readVarbig(r, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const scratch = await getNBytes(r, sizeof(o.dataType));
    return varbig(scratch, o);
}
export function writeVarnum(w, x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const scratch = new Uint8Array(sizeof(o.dataType));
    putVarnum(scratch, x, o);
    return w.write(scratch);
}
export function writeVarbig(w, x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const scratch = new Uint8Array(sizeof(o.dataType));
    putVarbig(scratch, x, o);
    return w.write(scratch);
}
export function varnumBytes(x, o = {}) {
    o.dataType = o.dataType ?? "int32";
    const b = new Uint8Array(sizeof(o.dataType));
    putVarnum(b, x, o);
    return b;
}
export function varbigBytes(x, o = {}) {
    o.dataType = o.dataType ?? "int64";
    const b = new Uint8Array(sizeof(o.dataType));
    putVarbig(b, x, o);
    return b;
}
//# sourceMappingURL=data:application/json;base64,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