import { crypto as wasmCrypto, digestAlgorithms as wasmDigestAlgorithms, } from "../_wasm_crypto/mod.ts";
const webCrypto = ((crypto) => ({
    getRandomValues: crypto.getRandomValues?.bind(crypto),
    randomUUID: crypto.randomUUID?.bind(crypto),
    subtle: {
        decrypt: crypto.subtle?.decrypt?.bind(crypto.subtle),
        deriveBits: crypto.subtle?.deriveBits?.bind(crypto.subtle),
        deriveKey: crypto.subtle?.deriveKey?.bind(crypto.subtle),
        digest: crypto.subtle?.digest?.bind(crypto.subtle),
        encrypt: crypto.subtle?.encrypt?.bind(crypto.subtle),
        exportKey: crypto.subtle?.exportKey?.bind(crypto.subtle),
        generateKey: crypto.subtle?.generateKey?.bind(crypto.subtle),
        importKey: crypto.subtle?.importKey?.bind(crypto.subtle),
        sign: crypto.subtle?.sign?.bind(crypto.subtle),
        unwrapKey: crypto.subtle?.unwrapKey?.bind(crypto.subtle),
        verify: crypto.subtle?.verify?.bind(crypto.subtle),
        wrapKey: crypto.subtle?.wrapKey?.bind(crypto.subtle),
    },
}))(globalThis.crypto);
const bufferSourceBytes = (data) => {
    let bytes;
    if (data instanceof Uint8Array) {
        bytes = data;
    }
    else if (ArrayBuffer.isView(data)) {
        bytes = new Uint8Array(data.buffer, data.byteOffset, data.byteLength);
    }
    else if (data instanceof ArrayBuffer) {
        bytes = new Uint8Array(data);
    }
    return bytes;
};
const stdCrypto = ((x) => x)({
    ...webCrypto,
    subtle: {
        ...webCrypto.subtle,
        async digest(algorithm, data) {
            const { name, length } = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (webCrypto.subtle?.digest &&
                webCryptoDigestAlgorithms.includes(name) &&
                bytes) {
                return webCrypto.subtle.digest(algorithm, bytes);
            }
            else if (wasmDigestAlgorithms.includes(name)) {
                if (bytes) {
                    return stdCrypto.subtle.digestSync(algorithm, bytes);
                }
                else if (data[Symbol.iterator]) {
                    return stdCrypto.subtle.digestSync(algorithm, data);
                }
                else if (data[Symbol.asyncIterator]) {
                    const context = new wasmCrypto.DigestContext(name);
                    for await (const chunk of data) {
                        const chunkBytes = bufferSourceBytes(chunk);
                        if (!chunkBytes) {
                            throw new TypeError("data contained chunk of the wrong type");
                        }
                        context.update(chunkBytes);
                    }
                    return context.digestAndDrop(length).buffer;
                }
                else {
                    throw new TypeError("data must be a BufferSource or [Async]Iterable<BufferSource>");
                }
            }
            else if (webCrypto.subtle?.digest) {
                return webCrypto.subtle.digest(algorithm, data);
            }
            else {
                throw new TypeError(`unsupported digest algorithm: ${algorithm}`);
            }
        },
        digestSync(algorithm, data) {
            algorithm = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (bytes) {
                return wasmCrypto.digest(algorithm.name, bytes, algorithm.length)
                    .buffer;
            }
            else if (data[Symbol.iterator]) {
                const context = new wasmCrypto.DigestContext(algorithm.name);
                for (const chunk of data) {
                    const chunkBytes = bufferSourceBytes(chunk);
                    if (!chunkBytes) {
                        throw new TypeError("data contained chunk of the wrong type");
                    }
                    context.update(chunkBytes);
                }
                return context.digestAndDrop(algorithm.length).buffer;
            }
            else {
                throw new TypeError("data must be a BufferSource or Iterable<BufferSource>");
            }
        },
    },
});
const webCryptoDigestAlgorithms = [
    "SHA-384",
    "SHA-256",
    "SHA-512",
    "SHA-1",
];
const normalizeAlgorithm = (algorithm) => ((typeof algorithm === "string") ? { name: algorithm.toUpperCase() } : {
    ...algorithm,
    name: algorithm.name.toUpperCase(),
});
export { stdCrypto as crypto };
//# sourceMappingURL=data:application/json;base64,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