import { assert } from "../_util/assert.ts";
import { toIMF } from "../datetime/mod.ts";
const FIELD_CONTENT_REGEXP = /^(?=[\x20-\x7E]*$)[^()@<>,;:\\"\[\]?={}\s]+$/;
function toString(cookie) {
    if (!cookie.name) {
        return "";
    }
    const out = [];
    validateName(cookie.name);
    validateValue(cookie.name, cookie.value);
    out.push(`${cookie.name}=${cookie.value}`);
    if (cookie.name.startsWith("__Secure")) {
        cookie.secure = true;
    }
    if (cookie.name.startsWith("__Host")) {
        cookie.path = "/";
        cookie.secure = true;
        delete cookie.domain;
    }
    if (cookie.secure) {
        out.push("Secure");
    }
    if (cookie.httpOnly) {
        out.push("HttpOnly");
    }
    if (typeof cookie.maxAge === "number" && Number.isInteger(cookie.maxAge)) {
        assert(cookie.maxAge >= 0, "Max-Age must be an integer superior or equal to 0");
        out.push(`Max-Age=${cookie.maxAge}`);
    }
    if (cookie.domain) {
        validateDomain(cookie.domain);
        out.push(`Domain=${cookie.domain}`);
    }
    if (cookie.sameSite) {
        out.push(`SameSite=${cookie.sameSite}`);
    }
    if (cookie.path) {
        validatePath(cookie.path);
        out.push(`Path=${cookie.path}`);
    }
    if (cookie.expires) {
        const dateString = toIMF(cookie.expires);
        out.push(`Expires=${dateString}`);
    }
    if (cookie.unparsed) {
        out.push(cookie.unparsed.join("; "));
    }
    return out.join("; ");
}
function validateName(name) {
    if (name && !FIELD_CONTENT_REGEXP.test(name)) {
        throw new TypeError(`Invalid cookie name: "${name}".`);
    }
}
function validatePath(path) {
    if (path == null) {
        return;
    }
    for (let i = 0; i < path.length; i++) {
        const c = path.charAt(i);
        if (c < String.fromCharCode(0x20) || c > String.fromCharCode(0x7E) || c == ";") {
            throw new Error(path + ": Invalid cookie path char '" + c + "'");
        }
    }
}
function validateValue(name, value) {
    if (value == null || name == null)
        return;
    for (let i = 0; i < value.length; i++) {
        const c = value.charAt(i);
        if (c < String.fromCharCode(0x21) || c == String.fromCharCode(0x22) ||
            c == String.fromCharCode(0x2c) || c == String.fromCharCode(0x3b) ||
            c == String.fromCharCode(0x5c) || c == String.fromCharCode(0x7f)) {
            throw new Error("RFC2616 cookie '" + name + "' cannot have '" + c + "' as value");
        }
        if (c > String.fromCharCode(0x80)) {
            throw new Error("RFC2616 cookie '" + name + "' can only have US-ASCII chars as value" +
                c.charCodeAt(0).toString(16));
        }
    }
}
function validateDomain(domain) {
    if (domain == null) {
        return;
    }
    const char1 = domain.charAt(0);
    const charN = domain.charAt(domain.length - 1);
    if (char1 == "-" || charN == "." || charN == "-") {
        throw new Error("Invalid first/last char in cookie domain: " + domain);
    }
}
export function getCookies(headers) {
    const cookie = headers.get("Cookie");
    if (cookie != null) {
        const out = {};
        const c = cookie.split(";");
        for (const kv of c) {
            const [cookieKey, ...cookieVal] = kv.split("=");
            assert(cookieKey != null);
            const key = cookieKey.trim();
            out[key] = cookieVal.join("=");
        }
        return out;
    }
    return {};
}
export function setCookie(headers, cookie) {
    const v = toString(cookie);
    if (v) {
        headers.append("Set-Cookie", v);
    }
}
export function deleteCookie(headers, name, attributes) {
    setCookie(headers, {
        name: name,
        value: "",
        expires: new Date(0),
        ...attributes,
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29va2llLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29va2llLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUdBLE9BQU8sRUFBRSxNQUFNLEVBQUUsTUFBTSxvQkFBb0IsQ0FBQztBQUM1QyxPQUFPLEVBQUUsS0FBSyxFQUFFLE1BQU0sb0JBQW9CLENBQUM7QUE0QjNDLE1BQU0sb0JBQW9CLEdBQUcsOENBQThDLENBQUM7QUFFNUUsU0FBUyxRQUFRLENBQUMsTUFBYztJQUM5QixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRTtRQUNoQixPQUFPLEVBQUUsQ0FBQztLQUNYO0lBQ0QsTUFBTSxHQUFHLEdBQWEsRUFBRSxDQUFDO0lBQ3pCLFlBQVksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUIsYUFBYSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsSUFBSSxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBSTNDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDdEMsTUFBTSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUM7S0FDdEI7SUFDRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ3BDLE1BQU0sQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDO1FBQ2xCLE1BQU0sQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDO1FBQ3JCLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQztLQUN0QjtJQUVELElBQUksTUFBTSxDQUFDLE1BQU0sRUFBRTtRQUNqQixHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQ3BCO0lBQ0QsSUFBSSxNQUFNLENBQUMsUUFBUSxFQUFFO1FBQ25CLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7S0FDdEI7SUFDRCxJQUFJLE9BQU8sTUFBTSxDQUFDLE1BQU0sS0FBSyxRQUFRLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDeEUsTUFBTSxDQUNKLE1BQU0sQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUNsQixtREFBbUQsQ0FDcEQsQ0FBQztRQUNGLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztLQUN0QztJQUNELElBQUksTUFBTSxDQUFDLE1BQU0sRUFBRTtRQUNqQixjQUFjLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzlCLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztLQUNyQztJQUNELElBQUksTUFBTSxDQUFDLFFBQVEsRUFBRTtRQUNuQixHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksTUFBTSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDekM7SUFDRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLEVBQUU7UUFDZixZQUFZLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzFCLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxNQUFNLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztLQUNqQztJQUNELElBQUksTUFBTSxDQUFDLE9BQU8sRUFBRTtRQUNsQixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0tBQ25DO0lBQ0QsSUFBSSxNQUFNLENBQUMsUUFBUSxFQUFFO1FBQ25CLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztLQUN0QztJQUNELE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUN4QixDQUFDO0FBTUQsU0FBUyxZQUFZLENBQUMsSUFBK0I7SUFDbkQsSUFBSSxJQUFJLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDNUMsTUFBTSxJQUFJLFNBQVMsQ0FBQyx5QkFBeUIsSUFBSSxJQUFJLENBQUMsQ0FBQztLQUN4RDtBQUNILENBQUM7QUFPRCxTQUFTLFlBQVksQ0FBQyxJQUFtQjtJQUN2QyxJQUFJLElBQUksSUFBSSxJQUFJLEVBQUU7UUFDaEIsT0FBTztLQUNSO0lBQ0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7UUFDcEMsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN6QixJQUNFLENBQUMsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLEVBQzFFO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixJQUFJLEdBQUcsOEJBQThCLEdBQUcsQ0FBQyxHQUFHLEdBQUcsQ0FDaEQsQ0FBQztTQUNIO0tBQ0Y7QUFDSCxDQUFDO0FBT0QsU0FBUyxhQUFhLENBQUMsSUFBWSxFQUFFLEtBQW9CO0lBQ3ZELElBQUksS0FBSyxJQUFJLElBQUksSUFBSSxJQUFJLElBQUksSUFBSTtRQUFFLE9BQU87SUFDMUMsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7UUFDckMsTUFBTSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxQixJQUNFLENBQUMsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztZQUMvRCxDQUFDLElBQUksTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7WUFDaEUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLE1BQU0sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQ2hFO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixrQkFBa0IsR0FBRyxJQUFJLEdBQUcsaUJBQWlCLEdBQUcsQ0FBQyxHQUFHLFlBQVksQ0FDakUsQ0FBQztTQUNIO1FBQ0QsSUFBSSxDQUFDLEdBQUcsTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUNiLGtCQUFrQixHQUFHLElBQUksR0FBRyx5Q0FBeUM7Z0JBQ25FLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUMvQixDQUFDO1NBQ0g7S0FDRjtBQUNILENBQUM7QUFPRCxTQUFTLGNBQWMsQ0FBQyxNQUFjO0lBQ3BDLElBQUksTUFBTSxJQUFJLElBQUksRUFBRTtRQUNsQixPQUFPO0tBQ1I7SUFDRCxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQy9CLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztJQUMvQyxJQUFJLEtBQUssSUFBSSxHQUFHLElBQUksS0FBSyxJQUFJLEdBQUcsSUFBSSxLQUFLLElBQUksR0FBRyxFQUFFO1FBQ2hELE1BQU0sSUFBSSxLQUFLLENBQ2IsNENBQTRDLEdBQUcsTUFBTSxDQUN0RCxDQUFDO0tBQ0g7QUFDSCxDQUFDO0FBT0QsTUFBTSxVQUFVLFVBQVUsQ0FBQyxPQUFnQjtJQUN6QyxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3JDLElBQUksTUFBTSxJQUFJLElBQUksRUFBRTtRQUNsQixNQUFNLEdBQUcsR0FBMkIsRUFBRSxDQUFDO1FBQ3ZDLE1BQU0sQ0FBQyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDNUIsS0FBSyxNQUFNLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDbEIsTUFBTSxDQUFDLFNBQVMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDaEQsTUFBTSxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsQ0FBQztZQUMxQixNQUFNLEdBQUcsR0FBRyxTQUFTLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDN0IsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDaEM7UUFDRCxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBQ0QsT0FBTyxFQUFFLENBQUM7QUFDWixDQUFDO0FBT0QsTUFBTSxVQUFVLFNBQVMsQ0FBQyxPQUFnQixFQUFFLE1BQWM7SUFJeEQsTUFBTSxDQUFDLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzNCLElBQUksQ0FBQyxFQUFFO1FBQ0wsT0FBTyxDQUFDLE1BQU0sQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDLENBQUM7S0FDakM7QUFDSCxDQUFDO0FBUUQsTUFBTSxVQUFVLFlBQVksQ0FDMUIsT0FBZ0IsRUFDaEIsSUFBWSxFQUNaLFVBQStDO0lBRS9DLFNBQVMsQ0FBQyxPQUFPLEVBQUU7UUFDakIsSUFBSSxFQUFFLElBQUk7UUFDVixLQUFLLEVBQUUsRUFBRTtRQUNULE9BQU8sRUFBRSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDcEIsR0FBRyxVQUFVO0tBQ2QsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAyMDE4LTIwMjEgdGhlIERlbm8gYXV0aG9ycy4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vLyBTdHJ1Y3R1cmVkIHNpbWlsYXJseSB0byBHbydzIGNvb2tpZS5nb1xuLy8gaHR0cHM6Ly9naXRodWIuY29tL2dvbGFuZy9nby9ibG9iL21hc3Rlci9zcmMvbmV0L2h0dHAvY29va2llLmdvXG5pbXBvcnQgeyBhc3NlcnQgfSBmcm9tIFwiLi4vX3V0aWwvYXNzZXJ0LnRzXCI7XG5pbXBvcnQgeyB0b0lNRiB9IGZyb20gXCIuLi9kYXRldGltZS9tb2QudHNcIjtcblxuZXhwb3J0IGludGVyZmFjZSBDb29raWUge1xuICAvKiogTmFtZSBvZiB0aGUgY29va2llLiAqL1xuICBuYW1lOiBzdHJpbmc7XG4gIC8qKiBWYWx1ZSBvZiB0aGUgY29va2llLiAqL1xuICB2YWx1ZTogc3RyaW5nO1xuICAvKiogRXhwaXJhdGlvbiBkYXRlIG9mIHRoZSBjb29raWUuICovXG4gIGV4cGlyZXM/OiBEYXRlO1xuICAvKiogTWF4LUFnZSBvZiB0aGUgQ29va2llLiBNYXgtQWdlIG11c3QgYmUgYW4gaW50ZWdlciBzdXBlcmlvciBvciBlcXVhbCB0byAwLiAqL1xuICBtYXhBZ2U/OiBudW1iZXI7XG4gIC8qKiBTcGVjaWZpZXMgdGhvc2UgaG9zdHMgdG8gd2hpY2ggdGhlIGNvb2tpZSB3aWxsIGJlIHNlbnQuICovXG4gIGRvbWFpbj86IHN0cmluZztcbiAgLyoqIEluZGljYXRlcyBhIFVSTCBwYXRoIHRoYXQgbXVzdCBleGlzdCBpbiB0aGUgcmVxdWVzdC4gKi9cbiAgcGF0aD86IHN0cmluZztcbiAgLyoqIEluZGljYXRlcyBpZiB0aGUgY29va2llIGlzIG1hZGUgdXNpbmcgU1NMICYgSFRUUFMuICovXG4gIHNlY3VyZT86IGJvb2xlYW47XG4gIC8qKiBJbmRpY2F0ZXMgdGhhdCBjb29raWUgaXMgbm90IGFjY2Vzc2libGUgdmlhIEphdmFTY3JpcHQuICovXG4gIGh0dHBPbmx5PzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIEFsbG93cyBzZXJ2ZXJzIHRvIGFzc2VydCB0aGF0IGEgY29va2llIG91Z2h0IG5vdCB0b1xuICAgKiBiZSBzZW50IGFsb25nIHdpdGggY3Jvc3Mtc2l0ZSByZXF1ZXN0cy5cbiAgICovXG4gIHNhbWVTaXRlPzogXCJTdHJpY3RcIiB8IFwiTGF4XCIgfCBcIk5vbmVcIjtcbiAgLyoqIEFkZGl0aW9uYWwga2V5IHZhbHVlIHBhaXJzIHdpdGggdGhlIGZvcm0gXCJrZXk9dmFsdWVcIiAqL1xuICB1bnBhcnNlZD86IHN0cmluZ1tdO1xufVxuXG5jb25zdCBGSUVMRF9DT05URU5UX1JFR0VYUCA9IC9eKD89W1xceDIwLVxceDdFXSokKVteKClAPD4sOzpcXFxcXCJcXFtcXF0/PXt9XFxzXSskLztcblxuZnVuY3Rpb24gdG9TdHJpbmcoY29va2llOiBDb29raWUpOiBzdHJpbmcge1xuICBpZiAoIWNvb2tpZS5uYW1lKSB7XG4gICAgcmV0dXJuIFwiXCI7XG4gIH1cbiAgY29uc3Qgb3V0OiBzdHJpbmdbXSA9IFtdO1xuICB2YWxpZGF0ZU5hbWUoY29va2llLm5hbWUpO1xuICB2YWxpZGF0ZVZhbHVlKGNvb2tpZS5uYW1lLCBjb29raWUudmFsdWUpO1xuICBvdXQucHVzaChgJHtjb29raWUubmFtZX09JHtjb29raWUudmFsdWV9YCk7XG5cbiAgLy8gRmFsbGJhY2sgZm9yIGludmFsaWQgU2V0LUNvb2tpZVxuICAvLyByZWY6IGh0dHBzOi8vdG9vbHMuaWV0Zi5vcmcvaHRtbC9kcmFmdC1pZXRmLWh0dHBiaXMtY29va2llLXByZWZpeGVzLTAwI3NlY3Rpb24tMy4xXG4gIGlmIChjb29raWUubmFtZS5zdGFydHNXaXRoKFwiX19TZWN1cmVcIikpIHtcbiAgICBjb29raWUuc2VjdXJlID0gdHJ1ZTtcbiAgfVxuICBpZiAoY29va2llLm5hbWUuc3RhcnRzV2l0aChcIl9fSG9zdFwiKSkge1xuICAgIGNvb2tpZS5wYXRoID0gXCIvXCI7XG4gICAgY29va2llLnNlY3VyZSA9IHRydWU7XG4gICAgZGVsZXRlIGNvb2tpZS5kb21haW47XG4gIH1cblxuICBpZiAoY29va2llLnNlY3VyZSkge1xuICAgIG91dC5wdXNoKFwiU2VjdXJlXCIpO1xuICB9XG4gIGlmIChjb29raWUuaHR0cE9ubHkpIHtcbiAgICBvdXQucHVzaChcIkh0dHBPbmx5XCIpO1xuICB9XG4gIGlmICh0eXBlb2YgY29va2llLm1heEFnZSA9PT0gXCJudW1iZXJcIiAmJiBOdW1iZXIuaXNJbnRlZ2VyKGNvb2tpZS5tYXhBZ2UpKSB7XG4gICAgYXNzZXJ0KFxuICAgICAgY29va2llLm1heEFnZSA+PSAwLFxuICAgICAgXCJNYXgtQWdlIG11c3QgYmUgYW4gaW50ZWdlciBzdXBlcmlvciBvciBlcXVhbCB0byAwXCIsXG4gICAgKTtcbiAgICBvdXQucHVzaChgTWF4LUFnZT0ke2Nvb2tpZS5tYXhBZ2V9YCk7XG4gIH1cbiAgaWYgKGNvb2tpZS5kb21haW4pIHtcbiAgICB2YWxpZGF0ZURvbWFpbihjb29raWUuZG9tYWluKTtcbiAgICBvdXQucHVzaChgRG9tYWluPSR7Y29va2llLmRvbWFpbn1gKTtcbiAgfVxuICBpZiAoY29va2llLnNhbWVTaXRlKSB7XG4gICAgb3V0LnB1c2goYFNhbWVTaXRlPSR7Y29va2llLnNhbWVTaXRlfWApO1xuICB9XG4gIGlmIChjb29raWUucGF0aCkge1xuICAgIHZhbGlkYXRlUGF0aChjb29raWUucGF0aCk7XG4gICAgb3V0LnB1c2goYFBhdGg9JHtjb29raWUucGF0aH1gKTtcbiAgfVxuICBpZiAoY29va2llLmV4cGlyZXMpIHtcbiAgICBjb25zdCBkYXRlU3RyaW5nID0gdG9JTUYoY29va2llLmV4cGlyZXMpO1xuICAgIG91dC5wdXNoKGBFeHBpcmVzPSR7ZGF0ZVN0cmluZ31gKTtcbiAgfVxuICBpZiAoY29va2llLnVucGFyc2VkKSB7XG4gICAgb3V0LnB1c2goY29va2llLnVucGFyc2VkLmpvaW4oXCI7IFwiKSk7XG4gIH1cbiAgcmV0dXJuIG91dC5qb2luKFwiOyBcIik7XG59XG5cbi8qKlxuICogVmFsaWRhdGUgQ29va2llIE5hbWUuXG4gKiBAcGFyYW0gbmFtZSBDb29raWUgbmFtZS5cbiAqL1xuZnVuY3Rpb24gdmFsaWRhdGVOYW1lKG5hbWU6IHN0cmluZyB8IHVuZGVmaW5lZCB8IG51bGwpOiB2b2lkIHtcbiAgaWYgKG5hbWUgJiYgIUZJRUxEX0NPTlRFTlRfUkVHRVhQLnRlc3QobmFtZSkpIHtcbiAgICB0aHJvdyBuZXcgVHlwZUVycm9yKGBJbnZhbGlkIGNvb2tpZSBuYW1lOiBcIiR7bmFtZX1cIi5gKTtcbiAgfVxufVxuXG4vKipcbiAqIFZhbGlkYXRlIFBhdGggVmFsdWUuXG4gKiBTZWUge0BsaW5rIGh0dHBzOi8vdG9vbHMuaWV0Zi5vcmcvaHRtbC9yZmM2MjY1I3NlY3Rpb24tNC4xLjIuNH0uXG4gKiBAcGFyYW0gcGF0aCBQYXRoIHZhbHVlLlxuICovXG5mdW5jdGlvbiB2YWxpZGF0ZVBhdGgocGF0aDogc3RyaW5nIHwgbnVsbCk6IHZvaWQge1xuICBpZiAocGF0aCA9PSBudWxsKSB7XG4gICAgcmV0dXJuO1xuICB9XG4gIGZvciAobGV0IGkgPSAwOyBpIDwgcGF0aC5sZW5ndGg7IGkrKykge1xuICAgIGNvbnN0IGMgPSBwYXRoLmNoYXJBdChpKTtcbiAgICBpZiAoXG4gICAgICBjIDwgU3RyaW5nLmZyb21DaGFyQ29kZSgweDIwKSB8fCBjID4gU3RyaW5nLmZyb21DaGFyQ29kZSgweDdFKSB8fCBjID09IFwiO1wiXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIHBhdGggKyBcIjogSW52YWxpZCBjb29raWUgcGF0aCBjaGFyICdcIiArIGMgKyBcIidcIixcbiAgICAgICk7XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogVmFsaWRhdGUgQ29va2llIFZhbHVlLlxuICogU2VlIHtAbGluayBodHRwczovL3Rvb2xzLmlldGYub3JnL2h0bWwvcmZjNjI2NSNzZWN0aW9uLTQuMX0uXG4gKiBAcGFyYW0gdmFsdWUgQ29va2llIHZhbHVlLlxuICovXG5mdW5jdGlvbiB2YWxpZGF0ZVZhbHVlKG5hbWU6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IG51bGwpOiB2b2lkIHtcbiAgaWYgKHZhbHVlID09IG51bGwgfHwgbmFtZSA9PSBudWxsKSByZXR1cm47XG4gIGZvciAobGV0IGkgPSAwOyBpIDwgdmFsdWUubGVuZ3RoOyBpKyspIHtcbiAgICBjb25zdCBjID0gdmFsdWUuY2hhckF0KGkpO1xuICAgIGlmIChcbiAgICAgIGMgPCBTdHJpbmcuZnJvbUNoYXJDb2RlKDB4MjEpIHx8IGMgPT0gU3RyaW5nLmZyb21DaGFyQ29kZSgweDIyKSB8fFxuICAgICAgYyA9PSBTdHJpbmcuZnJvbUNoYXJDb2RlKDB4MmMpIHx8IGMgPT0gU3RyaW5nLmZyb21DaGFyQ29kZSgweDNiKSB8fFxuICAgICAgYyA9PSBTdHJpbmcuZnJvbUNoYXJDb2RlKDB4NWMpIHx8IGMgPT0gU3RyaW5nLmZyb21DaGFyQ29kZSgweDdmKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIlJGQzI2MTYgY29va2llICdcIiArIG5hbWUgKyBcIicgY2Fubm90IGhhdmUgJ1wiICsgYyArIFwiJyBhcyB2YWx1ZVwiLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKGMgPiBTdHJpbmcuZnJvbUNoYXJDb2RlKDB4ODApKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiUkZDMjYxNiBjb29raWUgJ1wiICsgbmFtZSArIFwiJyBjYW4gb25seSBoYXZlIFVTLUFTQ0lJIGNoYXJzIGFzIHZhbHVlXCIgK1xuICAgICAgICAgIGMuY2hhckNvZGVBdCgwKS50b1N0cmluZygxNiksXG4gICAgICApO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIFZhbGlkYXRlIENvb2tpZSBEb21haW4uXG4gKiBTZWUge0BsaW5rIGh0dHBzOi8vZGF0YXRyYWNrZXIuaWV0Zi5vcmcvZG9jL2h0bWwvcmZjNjI2NSNzZWN0aW9uLTQuMS4yLjN9LlxuICogQHBhcmFtIGRvbWFpbiBDb29raWUgZG9tYWluLlxuICovXG5mdW5jdGlvbiB2YWxpZGF0ZURvbWFpbihkb21haW46IHN0cmluZyk6IHZvaWQge1xuICBpZiAoZG9tYWluID09IG51bGwpIHtcbiAgICByZXR1cm47XG4gIH1cbiAgY29uc3QgY2hhcjEgPSBkb21haW4uY2hhckF0KDApO1xuICBjb25zdCBjaGFyTiA9IGRvbWFpbi5jaGFyQXQoZG9tYWluLmxlbmd0aCAtIDEpO1xuICBpZiAoY2hhcjEgPT0gXCItXCIgfHwgY2hhck4gPT0gXCIuXCIgfHwgY2hhck4gPT0gXCItXCIpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIkludmFsaWQgZmlyc3QvbGFzdCBjaGFyIGluIGNvb2tpZSBkb21haW46IFwiICsgZG9tYWluLFxuICAgICk7XG4gIH1cbn1cblxuLyoqXG4gKiBQYXJzZSBjb29raWVzIG9mIGEgaGVhZGVyXG4gKiBAcGFyYW0ge0hlYWRlcnN9IGhlYWRlcnMgVGhlIGhlYWRlcnMgaW5zdGFuY2UgdG8gZ2V0IGNvb2tpZXMgZnJvbVxuICogQHJldHVybiB7T2JqZWN0fSBPYmplY3Qgd2l0aCBjb29raWUgbmFtZXMgYXMga2V5c1xuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0Q29va2llcyhoZWFkZXJzOiBIZWFkZXJzKTogUmVjb3JkPHN0cmluZywgc3RyaW5nPiB7XG4gIGNvbnN0IGNvb2tpZSA9IGhlYWRlcnMuZ2V0KFwiQ29va2llXCIpO1xuICBpZiAoY29va2llICE9IG51bGwpIHtcbiAgICBjb25zdCBvdXQ6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPSB7fTtcbiAgICBjb25zdCBjID0gY29va2llLnNwbGl0KFwiO1wiKTtcbiAgICBmb3IgKGNvbnN0IGt2IG9mIGMpIHtcbiAgICAgIGNvbnN0IFtjb29raWVLZXksIC4uLmNvb2tpZVZhbF0gPSBrdi5zcGxpdChcIj1cIik7XG4gICAgICBhc3NlcnQoY29va2llS2V5ICE9IG51bGwpO1xuICAgICAgY29uc3Qga2V5ID0gY29va2llS2V5LnRyaW0oKTtcbiAgICAgIG91dFtrZXldID0gY29va2llVmFsLmpvaW4oXCI9XCIpO1xuICAgIH1cbiAgICByZXR1cm4gb3V0O1xuICB9XG4gIHJldHVybiB7fTtcbn1cblxuLyoqXG4gKiBTZXQgdGhlIGNvb2tpZSBoZWFkZXIgcHJvcGVybHkgaW4gdGhlIGhlYWRlcnNcbiAqIEBwYXJhbSB7SGVhZGVyc30gaGVhZGVycyBUaGUgaGVhZGVycyBpbnN0YW5jZSB0byBzZXQgdGhlIGNvb2tpZSB0b1xuICogQHBhcmFtIHtPYmplY3R9IGNvb2tpZSBDb29raWUgdG8gc2V0XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzZXRDb29raWUoaGVhZGVyczogSGVhZGVycywgY29va2llOiBDb29raWUpOiB2b2lkIHtcbiAgLy8gVE9ETyh6ZWt0aCkgOiBBZGQgcHJvcGVyIHBhcnNpbmcgb2YgU2V0LUNvb2tpZSBoZWFkZXJzXG4gIC8vIFBhcnNpbmcgY29va2llIGhlYWRlcnMgdG8gbWFrZSBjb25zaXN0ZW50IHNldC1jb29raWUgaGVhZGVyXG4gIC8vIHJlZjogaHR0cHM6Ly90b29scy5pZXRmLm9yZy9odG1sL3JmYzYyNjUjc2VjdGlvbi00LjEuMVxuICBjb25zdCB2ID0gdG9TdHJpbmcoY29va2llKTtcbiAgaWYgKHYpIHtcbiAgICBoZWFkZXJzLmFwcGVuZChcIlNldC1Db29raWVcIiwgdik7XG4gIH1cbn1cblxuLyoqXG4gKiBTZXQgdGhlIGNvb2tpZSBoZWFkZXIgd2l0aCBlbXB0eSB2YWx1ZSBpbiB0aGUgaGVhZGVycyB0byBkZWxldGUgaXRcbiAqIEBwYXJhbSB7SGVhZGVyc30gaGVhZGVycyBUaGUgaGVhZGVycyBpbnN0YW5jZSB0byBkZWxldGUgdGhlIGNvb2tpZSBmcm9tXG4gKiBAcGFyYW0ge3N0cmluZ30gbmFtZSBOYW1lIG9mIGNvb2tpZVxuICogQHBhcmFtIHtPYmplY3R9IGF0dHJpYnV0ZXMgQWRkaXRpb25hbCBjb29raWUgYXR0cmlidXRlc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZGVsZXRlQ29va2llKFxuICBoZWFkZXJzOiBIZWFkZXJzLFxuICBuYW1lOiBzdHJpbmcsXG4gIGF0dHJpYnV0ZXM/OiB7IHBhdGg/OiBzdHJpbmc7IGRvbWFpbj86IHN0cmluZyB9LFxuKTogdm9pZCB7XG4gIHNldENvb2tpZShoZWFkZXJzLCB7XG4gICAgbmFtZTogbmFtZSxcbiAgICB2YWx1ZTogXCJcIixcbiAgICBleHBpcmVzOiBuZXcgRGF0ZSgwKSxcbiAgICAuLi5hdHRyaWJ1dGVzLFxuICB9KTtcbn1cbiJdfQ==