import { BufReader } from "../io/buffer.ts";
import { TextProtoReader } from "../textproto/mod.ts";
import { StringReader } from "../io/readers.ts";
import { assert } from "../_util/assert.ts";
export { NEWLINE, stringify, StringifyError } from "./csv_stringify.ts";
const INVALID_RUNE = ["\r", "\n", '"'];
export const ERR_BARE_QUOTE = 'bare " in non-quoted-field';
export const ERR_QUOTE = 'extraneous or missing " in quoted-field';
export const ERR_INVALID_DELIM = "Invalid Delimiter";
export const ERR_FIELD_COUNT = "wrong number of fields";
export class ParseError extends Error {
    startLine;
    line;
    column;
    constructor(start, line, column, message) {
        super();
        this.startLine = start;
        this.column = column;
        this.line = line;
        if (message === ERR_FIELD_COUNT) {
            this.message = `record on line ${line}: ${message}`;
        }
        else if (start !== line) {
            this.message =
                `record on line ${start}; parse error on line ${line}, column ${column}: ${message}`;
        }
        else {
            this.message =
                `parse error on line ${line}, column ${column}: ${message}`;
        }
    }
}
function chkOptions(opt) {
    if (!opt.separator) {
        opt.separator = ",";
    }
    if (!opt.trimLeadingSpace) {
        opt.trimLeadingSpace = false;
    }
    if (INVALID_RUNE.includes(opt.separator) ||
        (typeof opt.comment === "string" && INVALID_RUNE.includes(opt.comment)) ||
        opt.separator === opt.comment) {
        throw new Error(ERR_INVALID_DELIM);
    }
}
async function readRecord(startLine, reader, opt = { separator: ",", trimLeadingSpace: false }) {
    const tp = new TextProtoReader(reader);
    let line = await readLine(tp);
    let lineIndex = startLine + 1;
    if (line === null)
        return null;
    if (line.length === 0) {
        return [];
    }
    if (opt.comment && line[0] === opt.comment) {
        return [];
    }
    assert(opt.separator != null);
    let fullLine = line;
    let quoteError = null;
    const quote = '"';
    const quoteLen = quote.length;
    const separatorLen = opt.separator.length;
    let recordBuffer = "";
    const fieldIndexes = [];
    parseField: for (;;) {
        if (opt.trimLeadingSpace) {
            line = line.trimLeft();
        }
        if (line.length === 0 || !line.startsWith(quote)) {
            const i = line.indexOf(opt.separator);
            let field = line;
            if (i >= 0) {
                field = field.substring(0, i);
            }
            if (!opt.lazyQuotes) {
                const j = field.indexOf(quote);
                if (j >= 0) {
                    const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
                    quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
                    break parseField;
                }
            }
            recordBuffer += field;
            fieldIndexes.push(recordBuffer.length);
            if (i >= 0) {
                line = line.substring(i + separatorLen);
                continue parseField;
            }
            break parseField;
        }
        else {
            line = line.substring(quoteLen);
            for (;;) {
                const i = line.indexOf(quote);
                if (i >= 0) {
                    recordBuffer += line.substring(0, i);
                    line = line.substring(i + quoteLen);
                    if (line.startsWith(quote)) {
                        recordBuffer += quote;
                        line = line.substring(quoteLen);
                    }
                    else if (line.startsWith(opt.separator)) {
                        line = line.substring(separatorLen);
                        fieldIndexes.push(recordBuffer.length);
                        continue parseField;
                    }
                    else if (0 === line.length) {
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    else if (opt.lazyQuotes) {
                        recordBuffer += quote;
                    }
                    else {
                        const col = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                }
                else if (line.length > 0 || !(await isEOF(tp))) {
                    recordBuffer += line;
                    const r = await readLine(tp);
                    lineIndex++;
                    line = r ?? "";
                    fullLine = line;
                    if (r === null) {
                        if (!opt.lazyQuotes) {
                            const col = runeCount(fullLine);
                            quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                            break parseField;
                        }
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    recordBuffer += "\n";
                }
                else {
                    if (!opt.lazyQuotes) {
                        const col = runeCount(fullLine);
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                    fieldIndexes.push(recordBuffer.length);
                    break parseField;
                }
            }
        }
    }
    if (quoteError) {
        throw quoteError;
    }
    const result = [];
    let preIdx = 0;
    for (const i of fieldIndexes) {
        result.push(recordBuffer.slice(preIdx, i));
        preIdx = i;
    }
    return result;
}
async function isEOF(tp) {
    return (await tp.r.peek(0)) === null;
}
function runeCount(s) {
    return Array.from(s).length;
}
async function readLine(tp) {
    let line;
    const r = await tp.readLine();
    if (r === null)
        return null;
    line = r;
    if ((await isEOF(tp)) && line.length > 0 && line[line.length - 1] === "\r") {
        line = line.substring(0, line.length - 1);
    }
    if (line.length >= 2 &&
        line[line.length - 2] === "\r" &&
        line[line.length - 1] === "\n") {
        line = line.substring(0, line.length - 2);
        line = line + "\n";
    }
    return line;
}
export async function readMatrix(reader, opt = {
    separator: ",",
    trimLeadingSpace: false,
    lazyQuotes: false,
}) {
    const result = [];
    let _nbFields;
    let lineResult;
    let first = true;
    let lineIndex = 0;
    chkOptions(opt);
    for (;;) {
        const r = await readRecord(lineIndex, reader, opt);
        if (r === null)
            break;
        lineResult = r;
        lineIndex++;
        if (first) {
            first = false;
            if (opt.fieldsPerRecord !== undefined) {
                if (opt.fieldsPerRecord === 0) {
                    _nbFields = lineResult.length;
                }
                else {
                    _nbFields = opt.fieldsPerRecord;
                }
            }
        }
        if (lineResult.length > 0) {
            if (_nbFields && _nbFields !== lineResult.length) {
                throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
            }
            result.push(lineResult);
        }
    }
    return result;
}
export async function parse(input, opt = {
    skipFirstRow: false,
}) {
    let r;
    if (input instanceof BufReader) {
        r = await readMatrix(input, opt);
    }
    else {
        r = await readMatrix(new BufReader(new StringReader(input)), opt);
    }
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head.map((e) => {
                return {
                    name: e,
                };
            });
            i++;
        }
        if (opt.columns) {
            if (typeof opt.columns[0] !== "string") {
                headers = opt.columns;
            }
            else {
                const h = opt.columns;
                headers = h.map((e) => {
                    return {
                        name: e,
                    };
                });
            }
        }
        return r.map((e) => {
            if (e.length !== headers.length) {
                throw `Error number of fields line:${i}`;
            }
            i++;
            const out = {};
            for (let j = 0; j < e.length; j++) {
                out[headers[j].name] = e[j];
            }
            return out;
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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