import { filterInPlace } from "./_utils.ts";
const { hasOwn } = Object;
export function deepMerge(record, other, options) {
    const result = {};
    const keys = new Set([
        ...getKeys(record),
        ...getKeys(other),
    ]);
    for (const key of keys) {
        if (key === "__proto__") {
            continue;
        }
        const a = record[key];
        if (!hasOwn(other, key)) {
            result[key] = a;
            continue;
        }
        const b = other[key];
        if (isNonNullObject(a) && isNonNullObject(b)) {
            result[key] = mergeObjects(a, b, options);
            continue;
        }
        result[key] = b;
    }
    return result;
}
function mergeObjects(left, right, options = {
    arrays: "merge",
    sets: "merge",
    maps: "merge",
}) {
    if (isMergeable(left) && isMergeable(right)) {
        return deepMerge(left, right);
    }
    if (isIterable(left) && isIterable(right)) {
        if ((Array.isArray(left)) && (Array.isArray(right))) {
            if (options.arrays === "merge") {
                return left.concat(right);
            }
            return right;
        }
        if ((left instanceof Map) && (right instanceof Map)) {
            if (options.maps === "merge") {
                return new Map([
                    ...left,
                    ...right,
                ]);
            }
            return right;
        }
        if ((left instanceof Set) && (right instanceof Set)) {
            if (options.sets === "merge") {
                return new Set([
                    ...left,
                    ...right,
                ]);
            }
            return right;
        }
    }
    return right;
}
function isMergeable(value) {
    return Object.getPrototypeOf(value) === Object.prototype;
}
function isIterable(value) {
    return typeof value[Symbol.iterator] === "function";
}
function isNonNullObject(value) {
    return value !== null && typeof value === "object";
}
function getKeys(record) {
    const ret = Object.getOwnPropertySymbols(record);
    filterInPlace(ret, (key) => Object.prototype.propertyIsEnumerable.call(record, key));
    ret.push(...Object.keys(record));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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