import { concat } from "../bytes/mod.ts";
const CHAR_SPACE = " ".charCodeAt(0);
const CHAR_TAB = "\t".charCodeAt(0);
const CHAR_COLON = ":".charCodeAt(0);
const WHITESPACES = [CHAR_SPACE, CHAR_TAB];
const decoder = new TextDecoder();
const invalidHeaderCharRegex = /[^\t\x20-\x7e\x80-\xff]/g;
function str(buf) {
    return !buf ? "" : decoder.decode(buf);
}
export class TextProtoReader {
    r;
    constructor(r) {
        this.r = r;
    }
    async readLine() {
        const s = await this.readLineSlice();
        return s === null ? null : str(s);
    }
    async readMIMEHeader() {
        const m = new Headers();
        let line;
        let buf = await this.r.peek(1);
        if (buf === null) {
            return null;
        }
        else if (WHITESPACES.includes(buf[0])) {
            line = (await this.readLineSlice());
        }
        buf = await this.r.peek(1);
        if (buf === null) {
            throw new Deno.errors.UnexpectedEof();
        }
        else if (WHITESPACES.includes(buf[0])) {
            throw new Deno.errors.InvalidData(`malformed MIME header initial line: ${str(line)}`);
        }
        while (true) {
            const kv = await this.readLineSlice();
            if (kv === null)
                throw new Deno.errors.UnexpectedEof();
            if (kv.byteLength === 0)
                return m;
            let i = kv.indexOf(CHAR_COLON);
            if (i < 0) {
                throw new Deno.errors.InvalidData(`malformed MIME header line: ${str(kv)}`);
            }
            const key = str(kv.subarray(0, i));
            if (key == "") {
                continue;
            }
            i++;
            while (i < kv.byteLength &&
                (WHITESPACES.includes(kv[i]))) {
                i++;
            }
            const value = str(kv.subarray(i)).replace(invalidHeaderCharRegex, encodeURI);
            try {
                m.append(key, value);
            }
            catch {
            }
        }
    }
    async readLineSlice() {
        let line = new Uint8Array(0);
        let r = null;
        do {
            r = await this.r.readLine();
            if (r !== null && this.skipSpace(r.line) !== 0) {
                line = concat(line, r.line);
            }
        } while (r !== null && r.more);
        return r === null ? null : line;
    }
    skipSpace(l) {
        let n = 0;
        for (const val of l) {
            if (!WHITESPACES.includes(val)) {
                n++;
            }
        }
        return n;
    }
}
//# sourceMappingURL=data:application/json;base64,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