const lookup = [];
const revLookup = [];
const code = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
for (let i = 0, len = code.length; i < len; ++i) {
    lookup[i] = code[i];
    revLookup[code.charCodeAt(i)] = i;
}
const placeHolderPadLookup = [0, 1, , 2, 3, , 4];
function _getPadLen(placeHoldersLen) {
    const maybeLen = placeHolderPadLookup[placeHoldersLen];
    if (typeof maybeLen !== "number") {
        throw new Error("Invalid pad length");
    }
    return maybeLen;
}
function getLens(b32) {
    const len = b32.length;
    if (len % 8 > 0) {
        throw new Error("Invalid string. Length must be a multiple of 8");
    }
    let validLen = b32.indexOf("=");
    if (validLen === -1)
        validLen = len;
    const placeHoldersLen = validLen === len ? 0 : 8 - (validLen % 8);
    return [validLen, placeHoldersLen];
}
export function byteLength(b32) {
    const [validLen, placeHoldersLen] = getLens(b32);
    return _byteLength(validLen, placeHoldersLen);
}
function _byteLength(validLen, placeHoldersLen) {
    return ((validLen + placeHoldersLen) * 5) / 8 - _getPadLen(placeHoldersLen);
}
export function decode(b32) {
    let tmp;
    const [validLen, placeHoldersLen] = getLens(b32);
    const arr = new Uint8Array(_byteLength(validLen, placeHoldersLen));
    let curByte = 0;
    const len = placeHoldersLen > 0 ? validLen - 8 : validLen;
    let i;
    for (i = 0; i < len; i += 8) {
        tmp = (revLookup[b32.charCodeAt(i)] << 20) |
            (revLookup[b32.charCodeAt(i + 1)] << 15) |
            (revLookup[b32.charCodeAt(i + 2)] << 10) |
            (revLookup[b32.charCodeAt(i + 3)] << 5) |
            revLookup[b32.charCodeAt(i + 4)];
        arr[curByte++] = (tmp >> 17) & 0xff;
        arr[curByte++] = (tmp >> 9) & 0xff;
        arr[curByte++] = (tmp >> 1) & 0xff;
        tmp = ((tmp & 1) << 15) |
            (revLookup[b32.charCodeAt(i + 5)] << 10) |
            (revLookup[b32.charCodeAt(i + 6)] << 5) |
            revLookup[b32.charCodeAt(i + 7)];
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    if (placeHoldersLen === 1) {
        tmp = (revLookup[b32.charCodeAt(i)] << 20) |
            (revLookup[b32.charCodeAt(i + 1)] << 15) |
            (revLookup[b32.charCodeAt(i + 2)] << 10) |
            (revLookup[b32.charCodeAt(i + 3)] << 5) |
            revLookup[b32.charCodeAt(i + 4)];
        arr[curByte++] = (tmp >> 17) & 0xff;
        arr[curByte++] = (tmp >> 9) & 0xff;
        arr[curByte++] = (tmp >> 1) & 0xff;
        tmp = ((tmp & 1) << 7) |
            (revLookup[b32.charCodeAt(i + 5)] << 2) |
            (revLookup[b32.charCodeAt(i + 6)] >> 3);
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 3) {
        tmp = (revLookup[b32.charCodeAt(i)] << 19) |
            (revLookup[b32.charCodeAt(i + 1)] << 14) |
            (revLookup[b32.charCodeAt(i + 2)] << 9) |
            (revLookup[b32.charCodeAt(i + 3)] << 4) |
            (revLookup[b32.charCodeAt(i + 4)] >> 1);
        arr[curByte++] = (tmp >> 16) & 0xff;
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 4) {
        tmp = (revLookup[b32.charCodeAt(i)] << 11) |
            (revLookup[b32.charCodeAt(i + 1)] << 6) |
            (revLookup[b32.charCodeAt(i + 2)] << 1) |
            (revLookup[b32.charCodeAt(i + 3)] >> 4);
        arr[curByte++] = (tmp >> 8) & 0xff;
        arr[curByte++] = tmp & 0xff;
    }
    else if (placeHoldersLen === 6) {
        tmp = (revLookup[b32.charCodeAt(i)] << 3) |
            (revLookup[b32.charCodeAt(i + 1)] >> 2);
        arr[curByte++] = tmp & 0xff;
    }
    return arr;
}
function encodeChunk(uint8, start, end) {
    let tmp;
    const output = [];
    for (let i = start; i < end; i += 5) {
        tmp = ((uint8[i] << 16) & 0xff0000) |
            ((uint8[i + 1] << 8) & 0xff00) |
            (uint8[i + 2] & 0xff);
        output.push(lookup[(tmp >> 19) & 0x1f]);
        output.push(lookup[(tmp >> 14) & 0x1f]);
        output.push(lookup[(tmp >> 9) & 0x1f]);
        output.push(lookup[(tmp >> 4) & 0x1f]);
        tmp = ((tmp & 0xf) << 16) |
            ((uint8[i + 3] << 8) & 0xff00) |
            (uint8[i + 4] & 0xff);
        output.push(lookup[(tmp >> 15) & 0x1f]);
        output.push(lookup[(tmp >> 10) & 0x1f]);
        output.push(lookup[(tmp >> 5) & 0x1f]);
        output.push(lookup[tmp & 0x1f]);
    }
    return output.join("");
}
export function encode(uint8) {
    let tmp;
    const len = uint8.length;
    const extraBytes = len % 5;
    const parts = [];
    const maxChunkLength = 16385;
    const len2 = len - extraBytes;
    for (let i = 0; i < len2; i += maxChunkLength) {
        parts.push(encodeChunk(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
    }
    if (extraBytes === 4) {
        tmp = ((uint8[len2] & 0xff) << 16) |
            ((uint8[len2 + 1] & 0xff) << 8) |
            (uint8[len2 + 2] & 0xff);
        parts.push(lookup[(tmp >> 19) & 0x1f]);
        parts.push(lookup[(tmp >> 14) & 0x1f]);
        parts.push(lookup[(tmp >> 9) & 0x1f]);
        parts.push(lookup[(tmp >> 4) & 0x1f]);
        tmp = ((tmp & 0xf) << 11) | (uint8[len2 + 3] << 3);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("=");
    }
    else if (extraBytes === 3) {
        tmp = ((uint8[len2] & 0xff) << 17) |
            ((uint8[len2 + 1] & 0xff) << 9) |
            ((uint8[len2 + 2] & 0xff) << 1);
        parts.push(lookup[(tmp >> 20) & 0x1f]);
        parts.push(lookup[(tmp >> 15) & 0x1f]);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("===");
    }
    else if (extraBytes === 2) {
        tmp = ((uint8[len2] & 0xff) << 12) | ((uint8[len2 + 1] & 0xff) << 4);
        parts.push(lookup[(tmp >> 15) & 0x1f]);
        parts.push(lookup[(tmp >> 10) & 0x1f]);
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("====");
    }
    else if (extraBytes === 1) {
        tmp = (uint8[len2] & 0xff) << 2;
        parts.push(lookup[(tmp >> 5) & 0x1f]);
        parts.push(lookup[tmp & 0x1f]);
        parts.push("======");
    }
    return parts.join("");
}
//# sourceMappingURL=data:application/json;base64,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