// deno-fmt-ignore-file
// deno-lint-ignore-file
// This code was bundled using `deno bundle` and it's not recommended to edit it manually

function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i2 = 0; i2 <= b.length; i2++){
        matrix[i2] = [
            i2
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i1 = 1; i1 <= b.length; i1++){
        for(let j = 1; j <= a.length; j++){
            if (b.charAt(i1 - 1) == a.charAt(j - 1)) {
                matrix[i1][j] = matrix[i1 - 1][j - 1];
            } else {
                matrix[i1][j] = Math.min(matrix[i1 - 1][j - 1] + 1, Math.min(matrix[i1][j - 1] + 1, matrix[i1 - 1][j] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str1) {
    return str1.replace(/-([a-z])/g, (g)=>g[1].toUpperCase()
    );
}
function getOption(flags, name1) {
    while(name1[0] === "-"){
        name1 = name1.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name1)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option1, options) {
    const optionNames = options.map((option)=>[
            option.name,
            ...option.aliases ?? []
        ]
    ).flat().map((option)=>getFlag(option)
    );
    return didYouMean(" Did you mean option", getFlag(option1), optionNames);
}
function didYouMeanType(type, types2) {
    return didYouMean(" Did you mean type", type, types2);
}
function didYouMean(message, type, types3) {
    const match = closest(type, types3);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name2) {
    if (name2.startsWith("-")) {
        return name2;
    }
    if (name2.length > 1) {
        return `--${name2}`;
    }
    return `-${name2}`;
}
function isOption(option, name3) {
    return option.name === name3 || option.aliases && option.aliases.indexOf(name3) !== -1;
}
function closest(str2, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i3 = 0; i3 < arr.length; i3++){
        const dist = distance(str2, arr[i3]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i3;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOption extends FlagsError {
    constructor(option, options){
        super(`Unknown required option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownRequiredOption.prototype);
    }
}
class UnknownConflictingOption extends FlagsError {
    constructor(option, options){
        super(`Unknown conflicting option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownConflictingOption.prototype);
    }
}
class UnknownType extends FlagsError {
    constructor(type, types4){
        super(`Unknown type "${type}".${didYouMeanType(type, types4)}`);
        Object.setPrototypeOf(this, UnknownType.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOption extends ValidationError {
    constructor(name4){
        super(`Option "${getFlag(name4).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOption.prototype);
    }
}
class UnknownOption extends ValidationError {
    constructor(option, options){
        super(`Unknown option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownOption.prototype);
    }
}
class MissingOptionValue extends ValidationError {
    constructor(option){
        super(`Missing value for option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
class InvalidOptionValue extends ValidationError {
    constructor(option, expected, value){
        super(`Option "${getFlag(option)}" must be of type "${expected}", but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValue.prototype);
    }
}
class OptionNotCombinable extends ValidationError {
    constructor(option){
        super(`Option "${getFlag(option)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinable.prototype);
    }
}
class ConflictingOption extends ValidationError {
    constructor(option, conflictingOption){
        super(`Option "${getFlag(option)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOption.prototype);
    }
}
class DependingOption extends ValidationError {
    constructor(option, dependingOption){
        super(`Option "${getFlag(option)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOption.prototype);
    }
}
class MissingRequiredOption extends ValidationError {
    constructor(option){
        super(`Missing required option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingRequiredOption.prototype);
    }
}
class RequiredArgumentFollowsOptionalArgument extends ValidationError {
    constructor(arg){
        super(`An required argument cannot follow an optional argument, but "${arg}"  is defined as required.`);
        Object.setPrototypeOf(this, RequiredArgumentFollowsOptionalArgument.prototype);
    }
}
class ArgumentFollowsVariadicArgument extends ValidationError {
    constructor(arg){
        super(`An argument cannot follow an variadic argument, but got "${arg}".`);
        Object.setPrototypeOf(this, ArgumentFollowsVariadicArgument.prototype);
    }
}
class NoArguments extends ValidationError {
    constructor(){
        super(`No arguments.`);
        Object.setPrototypeOf(this, NoArguments.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label , name: name5 , value: value1 , type  }, expected){
        super(`${label} "${name5}" must be of type "${type}", but got "${value1}".` + (expected ? ` Expected values: ${expected.map((value)=>`"${value}"`
        ).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
function normalize(args) {
    const normalized = [];
    let inLiteral = false;
    for (const arg of args){
        if (inLiteral) {
            normalized.push(arg);
        } else if (arg === "--") {
            inLiteral = true;
            normalized.push(arg);
        } else if (arg.length > 1 && arg[0] === "-") {
            const isLong = arg[1] === "-";
            const isDotted = !isLong && arg[2] === ".";
            if (arg.includes("=")) {
                const parts = arg.split("=");
                const flag = parts.shift();
                if (isLong) {
                    normalized.push(flag);
                } else {
                    normalizeShortFlags(flag);
                }
                normalized.push(parts.join("="));
            } else if (isLong || isDotted) {
                normalized.push(arg);
            } else {
                normalizeShortFlags(arg);
            }
        } else {
            normalized.push(arg);
        }
    }
    return normalized;
    function normalizeShortFlags(flag) {
        const flags = flag.slice(1).split("");
        if (isNaN(Number(flag[flag.length - 1]))) {
            flags.forEach((val)=>normalized.push(`-${val}`)
            );
        } else {
            normalized.push(`-${flags.shift()}`);
            normalized.push(flags.join(""));
        }
    }
}
var OptionType;
(function(OptionType1) {
    OptionType1["STRING"] = "string";
    OptionType1["NUMBER"] = "number";
    OptionType1["INTEGER"] = "integer";
    OptionType1["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {}));
const __boolean = (type)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type.value)) {
        return false;
    }
    throw new InvalidTypeError(type);
};
const number = (type)=>{
    const value = Number(type.value);
    if (Number.isFinite(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const string = ({ value  })=>{
    return value;
};
function validateFlags(flags, values1, _knownFlaks, allowEmpty, optionNames = {}) {
    const defaultValues = {};
    for (const option of flags){
        let name6;
        let defaultValue = undefined;
        if (option.name.startsWith("no-")) {
            const propName = option.name.replace(/^no-/, "");
            if (propName in values1) {
                continue;
            }
            const positiveOption = getOption(flags, propName);
            if (positiveOption) {
                continue;
            }
            name6 = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name6) {
            name6 = paramCaseToCamelCase(option.name);
        }
        if (!(name6 in optionNames)) {
            optionNames[name6] = option.name;
        }
        const hasDefaultValue = typeof values1[name6] === "undefined" && (typeof option.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            values1[name6] = getDefaultValue(option) ?? defaultValue;
            defaultValues[option.name] = true;
            if (typeof option.value === "function") {
                values1[name6] = option.value(values1[name6]);
            }
        }
    }
    const keys1 = Object.keys(values1);
    if (keys1.length === 0 && allowEmpty) {
        return;
    }
    const options = keys1.map((name7)=>({
            name: name7,
            option: getOption(flags, optionNames[name7])
        })
    );
    for (const { name: name1 , option: option1  } of options){
        if (!option1) {
            throw new UnknownOption(name1, flags);
        }
        if (option1.standalone) {
            if (keys1.length > 1) {
                if (options.every(({ option: opt  })=>opt && (option1 === opt || defaultValues[opt.name])
                )) {
                    return;
                }
                throw new OptionNotCombinable(option1.name);
            }
            return;
        }
        option1.conflicts?.forEach((flag)=>{
            if (isset(flag, values1)) {
                throw new ConflictingOption(option1.name, flag);
            }
        });
        option1.depends?.forEach((flag)=>{
            if (!isset(flag, values1) && !defaultValues[option1.name]) {
                throw new DependingOption(option1.name, flag);
            }
        });
        const isArray2 = (option1.args?.length || 0) > 1;
        option1.args?.forEach((arg, i4)=>{
            if (arg.requiredValue && (typeof values1[name1] === "undefined" || isArray2 && typeof values1[name1][i4] === "undefined")) {
                throw new MissingOptionValue(option1.name);
            }
        });
    }
    for (const option2 of flags){
        if (option2.required && !(paramCaseToCamelCase(option2.name) in values1)) {
            if ((!option2.conflicts || !option2.conflicts.find((flag)=>!!values1[flag]
            )) && !options.find((opt)=>opt.option?.conflicts?.find((flag)=>flag === option2.name
                )
            )) {
                throw new MissingRequiredOption(option2.name);
            }
        }
    }
    if (keys1.length === 0 && !allowEmpty) {
        throw new NoArguments();
    }
}
function isset(flag, values2) {
    const name8 = paramCaseToCamelCase(flag);
    return typeof values2[name8] !== "undefined";
}
const integer = (type)=>{
    const value = Number(type.value);
    if (Number.isInteger(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const Types = {
    [OptionType.STRING]: string,
    [OptionType.NUMBER]: number,
    [OptionType.INTEGER]: integer,
    [OptionType.BOOLEAN]: __boolean
};
function parseFlags(args2, opts = {}) {
    !opts.flags && (opts.flags = []);
    const normalized = normalize(args2);
    let inLiteral = false;
    let negate = false;
    const flags = {};
    const optionNames = {};
    let literal1 = [];
    let unknown = [];
    let stopEarly = null;
    opts.flags.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOption(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOption(flag, opts.flags ?? []);
            }
        });
    });
    for(let i5 = 0; i5 < normalized.length; i5++){
        let option1;
        let args1;
        const current = normalized[i5];
        if (inLiteral) {
            literal1.push(current);
            continue;
        }
        if (current === "--") {
            inLiteral = true;
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        const next = ()=>normalized[i5 + 1]
        ;
        if (isFlag) {
            if (current[2] === "-" || current[1] === "-" && current.length === 3) {
                throw new UnknownOption(current, opts.flags);
            }
            negate = current.startsWith("--no-");
            option1 = getOption(opts.flags, current);
            if (!option1) {
                if (opts.flags.length) {
                    throw new UnknownOption(current, opts.flags);
                }
                option1 = {
                    name: current.replace(/^-+/, ""),
                    optionalValue: true,
                    type: OptionType.STRING
                };
            }
            const positiveName = option1.name.replace(/^no-?/, "");
            const propName = paramCaseToCamelCase(positiveName);
            if (typeof flags[propName] !== "undefined" && !option1.collect) {
                throw new DuplicateOption(current);
            }
            args1 = option1.args?.length ? option1.args : [
                {
                    type: option1.type,
                    requiredValue: option1.requiredValue,
                    optionalValue: option1.optionalValue,
                    variadic: option1.variadic,
                    list: option1.list,
                    separator: option1.separator
                }
            ];
            let argIndex = 0;
            let inOptionalArg = false;
            const previous = flags[propName];
            parseNext(option1, args1);
            if (typeof flags[propName] === "undefined") {
                if (typeof option1.default !== "undefined") {
                    flags[propName] = getDefaultValue(option1);
                } else if (args1[argIndex].requiredValue) {
                    throw new MissingOptionValue(option1.name);
                } else {
                    flags[propName] = true;
                }
            }
            if (option1.value) {
                flags[propName] = option1.value(flags[propName], previous);
            } else if (option1.collect) {
                const value = Array.isArray(previous) ? previous : [];
                value.push(flags[propName]);
                flags[propName] = value;
            }
            optionNames[propName] = option1.name;
            opts.option?.(option1, flags[propName]);
            function parseNext(option3, args) {
                const arg1 = args[argIndex];
                if (!arg1) {
                    const flag = next();
                    throw new UnknownOption(flag, opts.flags ?? []);
                }
                if (!arg1.type) {
                    arg1.type = OptionType.BOOLEAN;
                }
                if (option3.args?.length) {
                    if ((typeof arg1.optionalValue === "undefined" || arg1.optionalValue === false) && typeof arg1.requiredValue === "undefined") {
                        arg1.requiredValue = true;
                    }
                } else {
                    if (arg1.type !== OptionType.BOOLEAN && (typeof arg1.optionalValue === "undefined" || arg1.optionalValue === false) && typeof arg1.requiredValue === "undefined") {
                        arg1.requiredValue = true;
                    }
                }
                if (arg1.requiredValue) {
                    if (inOptionalArg) {
                        throw new RequiredArgumentFollowsOptionalArgument(option3.name);
                    }
                } else {
                    inOptionalArg = true;
                }
                if (negate) {
                    flags[propName] = false;
                    return;
                }
                let result2;
                let increase = false;
                if (arg1.list && hasNext(arg1)) {
                    const parsed = next().split(arg1.separator || ",").map((nextValue)=>{
                        const value = parseValue(option3, arg1, nextValue);
                        if (typeof value === "undefined") {
                            throw new InvalidOptionValue(option3.name, arg1.type ?? "?", nextValue);
                        }
                        return value;
                    });
                    if (parsed?.length) {
                        result2 = parsed;
                    }
                } else {
                    if (hasNext(arg1)) {
                        result2 = parseValue(option3, arg1, next());
                    } else if (arg1.optionalValue && arg1.type === OptionType.BOOLEAN) {
                        result2 = true;
                    }
                }
                if (increase) {
                    i5++;
                    if (!arg1.variadic) {
                        argIndex++;
                    } else if (args[argIndex + 1]) {
                        throw new ArgumentFollowsVariadicArgument(next());
                    }
                }
                if (typeof result2 !== "undefined" && (args.length > 1 || arg1.variadic)) {
                    if (!flags[propName]) {
                        flags[propName] = [];
                    }
                    flags[propName].push(result2);
                    if (hasNext(arg1)) {
                        parseNext(option3, args);
                    }
                } else {
                    flags[propName] = result2;
                }
                function hasNext(arg) {
                    return !!(normalized[i5 + 1] && (arg.optionalValue || arg.requiredValue || arg.variadic) && (normalized[i5 + 1][0] !== "-" || arg.type === OptionType.NUMBER && !isNaN(Number(normalized[i5 + 1]))) && arg);
                }
                function parseValue(option, arg, value) {
                    const type = arg.type || OptionType.STRING;
                    const result = opts.parse ? opts.parse({
                        label: "Option",
                        type,
                        name: `--${option.name}`,
                        value
                    }) : parseFlagValue(option, arg, value);
                    if (typeof result !== "undefined") {
                        increase = true;
                    }
                    return result;
                }
            }
        } else {
            if (opts.stopEarly) {
                stopEarly = current;
                break;
            }
            unknown.push(current);
        }
    }
    if (stopEarly) {
        const stopEarlyArgIndex = args2.indexOf(stopEarly);
        if (stopEarlyArgIndex !== -1) {
            const doubleDashIndex = args2.indexOf("--");
            unknown = args2.slice(stopEarlyArgIndex, doubleDashIndex === -1 ? undefined : doubleDashIndex);
            if (doubleDashIndex !== -1) {
                literal1 = args2.slice(doubleDashIndex + 1);
            }
        }
    }
    if (opts.flags?.length) {
        validateFlags(opts.flags, flags, opts.knownFlaks, opts.allowEmpty, optionNames);
    }
    const result1 = Object.keys(flags).reduce((result3, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result[subKey] = flags[key];
                } else {
                    result[subKey] = result[subKey] ?? {};
                }
                return result[subKey];
            }, result3);
        } else {
            result3[key] = flags[key];
        }
        return result3;
    }, {});
    return {
        flags: result1,
        unknown,
        literal: literal1
    };
}
function parseFlagValue(option, arg, value) {
    const type = arg.type || OptionType.STRING;
    const parseType = Types[type];
    if (!parseType) {
        throw new UnknownType(type, Object.keys(Types));
    }
    return parseType({
        label: "Option",
        type,
        name: `--${option.name}`,
        value
    });
}
function getPermissions() {
    return hasPermissions([
        "env",
        "hrtime",
        "net",
        "plugin",
        "read",
        "run",
        "write", 
    ]);
}
function isUnstable() {
    return !!Deno.permissions;
}
function didYouMeanCommand(command1, commands1, excludes = []) {
    const commandNames = commands1.map((command)=>command.getName()
    ).filter((command)=>!excludes.includes(command)
    );
    return didYouMean(" Did you mean command", command1, commandNames);
}
async function hasPermission(permission) {
    try {
        return (await Deno.permissions?.query?.({
            name: permission
        }))?.state === "granted";
    } catch  {
        return false;
    }
}
async function hasPermissions(names) {
    const permissions = {};
    await Promise.all(names.map((name9)=>hasPermission(name9).then((hasPermission1)=>permissions[name9] = hasPermission1
        )
    ));
    return permissions;
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition
    };
}
function parseArgumentsDefinition(argsDefinition) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg of parts){
        if (hasVariadic) {
            throw new ArgumentFollowsVariadicArgument(arg);
        }
        const parts = arg.split(ARGUMENT_DETAILS_REGEX);
        const type = parts[2] || OptionType.STRING;
        const details = {
            optionalValue: arg[0] !== "<",
            name: parts[1],
            action: parts[3] || type,
            variadic: false,
            list: type ? arg.indexOf(type + "[]") !== -1 : false,
            type
        };
        if (!details.optionalValue && hasOptional) {
            throw new RequiredArgumentFollowsOptionalArgument(details.name);
        }
        if (arg[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        if (details.name) {
            argumentDetails.push(details);
        }
    }
    return argumentDetails;
}
const { Deno: Deno2  } = globalThis;
const noColor = typeof Deno2?.noColor === "boolean" ? Deno2.noColor : true;
let enabled = !noColor;
function setColorEnabled(value) {
    if (noColor) {
        return;
    }
    enabled = value;
}
function getColorEnabled() {
    return enabled;
}
function code(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run(str3, code1) {
    return enabled ? `${code1.open}${str3.replace(code1.regexp, code1.open)}${code1.close}` : str3;
}
function reset(str4) {
    return run(str4, code([
        0
    ], 0));
}
function bold(str5) {
    return run(str5, code([
        1
    ], 22));
}
function dim(str6) {
    return run(str6, code([
        2
    ], 22));
}
function italic(str7) {
    return run(str7, code([
        3
    ], 23));
}
function underline(str8) {
    return run(str8, code([
        4
    ], 24));
}
function inverse(str9) {
    return run(str9, code([
        7
    ], 27));
}
function hidden(str10) {
    return run(str10, code([
        8
    ], 28));
}
function strikethrough(str11) {
    return run(str11, code([
        9
    ], 29));
}
function black(str12) {
    return run(str12, code([
        30
    ], 39));
}
function red(str13) {
    return run(str13, code([
        31
    ], 39));
}
function green(str14) {
    return run(str14, code([
        32
    ], 39));
}
function yellow(str15) {
    return run(str15, code([
        33
    ], 39));
}
function blue(str16) {
    return run(str16, code([
        34
    ], 39));
}
function magenta(str17) {
    return run(str17, code([
        35
    ], 39));
}
function cyan(str18) {
    return run(str18, code([
        36
    ], 39));
}
function white(str19) {
    return run(str19, code([
        37
    ], 39));
}
function gray(str20) {
    return brightBlack(str20);
}
function brightBlack(str21) {
    return run(str21, code([
        90
    ], 39));
}
function brightRed(str22) {
    return run(str22, code([
        91
    ], 39));
}
function brightGreen(str23) {
    return run(str23, code([
        92
    ], 39));
}
function brightYellow(str24) {
    return run(str24, code([
        93
    ], 39));
}
function brightBlue(str25) {
    return run(str25, code([
        94
    ], 39));
}
function brightMagenta(str26) {
    return run(str26, code([
        95
    ], 39));
}
function brightCyan(str27) {
    return run(str27, code([
        96
    ], 39));
}
function brightWhite(str28) {
    return run(str28, code([
        97
    ], 39));
}
function bgBlack(str29) {
    return run(str29, code([
        40
    ], 49));
}
function bgRed(str30) {
    return run(str30, code([
        41
    ], 49));
}
function bgGreen(str31) {
    return run(str31, code([
        42
    ], 49));
}
function bgYellow(str32) {
    return run(str32, code([
        43
    ], 49));
}
function bgBlue(str33) {
    return run(str33, code([
        44
    ], 49));
}
function bgMagenta(str34) {
    return run(str34, code([
        45
    ], 49));
}
function bgCyan(str35) {
    return run(str35, code([
        46
    ], 49));
}
function bgWhite(str36) {
    return run(str36, code([
        47
    ], 49));
}
function bgBrightBlack(str37) {
    return run(str37, code([
        100
    ], 49));
}
function bgBrightRed(str38) {
    return run(str38, code([
        101
    ], 49));
}
function bgBrightGreen(str39) {
    return run(str39, code([
        102
    ], 49));
}
function bgBrightYellow(str40) {
    return run(str40, code([
        103
    ], 49));
}
function bgBrightBlue(str41) {
    return run(str41, code([
        104
    ], 49));
}
function bgBrightMagenta(str42) {
    return run(str42, code([
        105
    ], 49));
}
function bgBrightCyan(str43) {
    return run(str43, code([
        106
    ], 49));
}
function bgBrightWhite(str44) {
    return run(str44, code([
        107
    ], 49));
}
function clampAndTruncate(n, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n, max), min));
}
function rgb8(str45, color) {
    return run(str45, code([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str46, color) {
    return run(str46, code([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str47, color) {
    if (typeof color === "number") {
        return run(str47, code([
            38,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 39));
    }
    return run(str47, code([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 39));
}
function bgRgb24(str48, color) {
    if (typeof color === "number") {
        return run(str48, code([
            48,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 49));
    }
    return run(str48, code([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 49));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor(string1) {
    return string1.replace(ANSI_PATTERN, "");
}
const mod = {
    setColorEnabled: setColorEnabled,
    getColorEnabled: getColorEnabled,
    reset: reset,
    bold: bold,
    dim: dim,
    italic: italic,
    underline: underline,
    inverse: inverse,
    hidden: hidden,
    strikethrough: strikethrough,
    black: black,
    red: red,
    green: green,
    yellow: yellow,
    blue: blue,
    magenta: magenta,
    cyan: cyan,
    white: white,
    gray: gray,
    brightBlack: brightBlack,
    brightRed: brightRed,
    brightGreen: brightGreen,
    brightYellow: brightYellow,
    brightBlue: brightBlue,
    brightMagenta: brightMagenta,
    brightCyan: brightCyan,
    brightWhite: brightWhite,
    bgBlack: bgBlack,
    bgRed: bgRed,
    bgGreen: bgGreen,
    bgYellow: bgYellow,
    bgBlue: bgBlue,
    bgMagenta: bgMagenta,
    bgCyan: bgCyan,
    bgWhite: bgWhite,
    bgBrightBlack: bgBrightBlack,
    bgBrightRed: bgBrightRed,
    bgBrightGreen: bgBrightGreen,
    bgBrightYellow: bgBrightYellow,
    bgBrightBlue: bgBrightBlue,
    bgBrightMagenta: bgBrightMagenta,
    bgBrightCyan: bgBrightCyan,
    bgBrightWhite: bgBrightWhite,
    rgb8: rgb8,
    bgRgb8: bgRgb8,
    rgb24: rgb24,
    bgRgb24: bgRgb24,
    stripColor: stripColor
};
class CommandError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    constructor(message, { exitCode  } = {}){
        super(message);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 1;
    }
}
class DuplicateOptionName extends CommandError {
    constructor(name10){
        super(`Option with name "${getFlag(name10)}" already exists.`);
        Object.setPrototypeOf(this, DuplicateOptionName.prototype);
    }
}
class MissingCommandName extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandName.prototype);
    }
}
class DuplicateCommandName extends CommandError {
    constructor(name11){
        super(`Duplicate command name "${name11}".`);
        Object.setPrototypeOf(this, DuplicateCommandName.prototype);
    }
}
class DuplicateCommandAlias extends CommandError {
    constructor(alias){
        super(`Duplicate command alias "${alias}".`);
        Object.setPrototypeOf(this, DuplicateCommandAlias.prototype);
    }
}
class CommandNotFound extends CommandError {
    constructor(name12, commands2, excluded){
        super(`Unknown command "${name12}".${didYouMeanCommand(name12, commands2, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class DuplicateType extends CommandError {
    constructor(name13){
        super(`Type with name "${name13}" already exists.`);
        Object.setPrototypeOf(this, DuplicateType.prototype);
    }
}
class DuplicateCompletion extends CommandError {
    constructor(name14){
        super(`Completion with name "${name14}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletion.prototype);
    }
}
class DuplicateExample extends CommandError {
    constructor(name15){
        super(`Example with name "${name15}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExample.prototype);
    }
}
class DuplicateEnvironmentVariable extends CommandError {
    constructor(name16){
        super(`Environment variable with name "${name16}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvironmentVariable.prototype);
    }
}
class EnvironmentVariableSingleValue extends CommandError {
    constructor(name17){
        super(`An environment variable can only have one value, but "${name17}" has more than one.`);
        Object.setPrototypeOf(this, EnvironmentVariableSingleValue.prototype);
    }
}
class EnvironmentVariableOptionalValue extends CommandError {
    constructor(name18){
        super(`An environment variable cannot have an optional value, but "${name18}" is defined as optional.`);
        Object.setPrototypeOf(this, EnvironmentVariableOptionalValue.prototype);
    }
}
class EnvironmentVariableVariadicValue extends CommandError {
    constructor(name19){
        super(`An environment variable cannot have an variadic value, but "${name19}" is defined as variadic.`);
        Object.setPrototypeOf(this, EnvironmentVariableVariadicValue.prototype);
    }
}
class DefaultCommandNotFound extends CommandError {
    constructor(name20, commands3){
        super(`Default command "${name20}" not found.${didYouMeanCommand(name20, commands3)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFound.prototype);
    }
}
class CommandExecutableNotFound extends CommandError {
    constructor(name21, files){
        super(`Command executable not found: ${name21}:\n    - ${files.join("\\n    - ")}`);
        Object.setPrototypeOf(this, CommandExecutableNotFound.prototype);
    }
}
class UnknownCompletionCommand extends CommandError {
    constructor(name22, commands4){
        super(`Auto-completion failed. Unknown command "${name22}".${didYouMeanCommand(name22, commands4)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommand.prototype);
    }
}
class UnknownCommand extends ValidationError1 {
    constructor(name23, commands5, excluded){
        super(`Unknown command "${name23}".${didYouMeanCommand(name23, commands5, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class NoArgumentsAllowed extends ValidationError1 {
    constructor(name24){
        super(`No arguments allowed for command "${name24}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowed.prototype);
    }
}
class MissingArguments extends ValidationError1 {
    constructor(args){
        super("Missing argument(s): " + args.join(", "));
        Object.setPrototypeOf(this, MissingArguments.prototype);
    }
}
class MissingArgument extends ValidationError1 {
    constructor(arg){
        super(`Missing argument "${arg}".`);
        Object.setPrototypeOf(this, MissingArgument.prototype);
    }
}
class TooManyArguments extends ValidationError1 {
    constructor(args){
        super(`Too many arguments: ${args.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArguments.prototype);
    }
}
class Type {
}
class BooleanType extends Type {
    parse(type) {
        return __boolean(type);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class NumberType extends Type {
    parse(type) {
        return number(type);
    }
}
class StringType extends Type {
    parse(type) {
        return string(type);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    value;
    options = {};
    get length() {
        return this.toString().length;
    }
    static from(value) {
        const cell = new this(value);
        if (value instanceof Cell) {
            cell.options = {
                ...value.options
            };
        }
        return cell;
    }
    constructor(value){
        this.value = value;
    }
    toString() {
        return this.value.toString();
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        const cell = new Cell(value ?? this);
        cell.options = {
            ...this.options
        };
        return cell;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class Row extends Array {
    options = {};
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell
        ));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder()
        );
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
function consumeWords(length, content) {
    let consumed = "";
    const words = content.split(/ /g);
    for(let i6 = 0; i6 < words.length; i6++){
        let word = words[i6];
        const hasLineBreak = word.indexOf("\n") !== -1;
        if (hasLineBreak) {
            word = word.split("\n").shift();
        }
        if (consumed) {
            const nextLength = stripColor(word).length;
            const consumedLength = stripColor(consumed).length;
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i6 > 0 ? " " : "") + word;
        if (hasLineBreak) {
            break;
        }
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    return Math.max(...rows.map((row)=>(row[index] instanceof Cell && row[index].getColSpan() > 1 ? "" : row[index]?.toString() || "").split("\n").map((r)=>{
            const str49 = typeof maxWidth === "undefined" ? r : consumeWords(maxWidth, r);
            return stripColor(str49).length || 0;
        })
    ).flat());
}
class TableLayout {
    table;
    options;
    constructor(table, options){
        this.table = table;
        this.options = options;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const header = this.table.getHeader();
        const rows = this.spanRows(header ? [
            header,
            ...this.table
        ] : this.table.slice());
        const columns = Math.max(...rows.map((row)=>row.length
        ));
        for (const row1 of rows){
            const length = row1.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i7 = 0; i7 < diff; i7++){
                    row1.push(this.createCell(null, row1));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const minColWidth = Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth;
            const maxColWidth = Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth;
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding;
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    spanRows(_rows, rowIndex = 0, colIndex = 0, rowSpan = [], colSpan = 1) {
        const rows = _rows;
        if (rowIndex >= rows.length && rowSpan.every((span)=>span === 1
        )) {
            return rows;
        } else if (rows[rowIndex] && colIndex >= rows[rowIndex].length && colIndex >= rowSpan.length && colSpan === 1) {
            return this.spanRows(rows, ++rowIndex, 0, rowSpan, 1);
        }
        if (colSpan > 1) {
            colSpan--;
            rowSpan[colIndex] = rowSpan[colIndex - 1];
            rows[rowIndex].splice(colIndex - 1, 0, rows[rowIndex][colIndex - 1]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        if (colIndex === 0) {
            rows[rowIndex] = this.createRow(rows[rowIndex] || []);
        }
        if (rowSpan[colIndex] > 1) {
            rowSpan[colIndex]--;
            rows[rowIndex].splice(colIndex, 0, rows[rowIndex - 1][colIndex]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        rows[rowIndex][colIndex] = this.createCell(rows[rowIndex][colIndex] || null, rows[rowIndex]);
        colSpan = rows[rowIndex][colIndex].getColSpan();
        rowSpan[colIndex] = rows[rowIndex][colIndex].getRowSpan();
        return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row) {
        return Cell.from(cell ?? "").border(row.getBorder(), false).align(row.getAlign(), false);
    }
    renderRows(opts) {
        let result = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result += this.options.chars.right;
            } else if (opts.hasBorder) {
                result += " ";
            }
        }
        result += "\n";
        if (isMultilineRow) {
            return result + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder) {
            result += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current , next  } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next);
        if (opts.hasBorder) {
            result += " ".repeat(opts.padding[colIndex]);
        }
        result += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result += " ".repeat(opts.padding[colIndex]);
        }
        return result;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, stripColor(cell.toString()).length);
        let words = consumeWords(length, cell.toString());
        const breakWord = stripColor(words).length > length;
        if (breakWord) {
            words = words.slice(0, length);
        }
        const next = cell.toString().slice(words.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - stripColor(words).length;
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words;
        } else if (align === "left") {
            current = words + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result.length ? " ".repeat(this.options.indent) + result + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1
        ;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1
        ;
        let result = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result += this.options.chars.left;
                } else {
                    result += " ";
                }
            } else if (b1Border && b2Border) {
                result += this.options.chars.leftMid;
            } else if (b1Border) {
                result += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result += this.options.chars.topLeft;
            } else {
                result += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result += this.options.chars.rightMid;
                } else {
                    result += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result += this.options.chars.bottom;
                } else {
                    result += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result += this.options.chars.left;
                } else {
                    result += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result += this.options.chars.top;
                } else {
                    result += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result += this.options.chars.right;
                } else {
                    result += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result += this.options.chars.bottomRight;
            } else if (b1Border) {
                result += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result += this.options.chars.topRight;
            } else if (b2Border) {
                result += this.options.chars.topLeft;
            } else {
                result += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result += this.options.chars.right;
                } else {
                    result += " ";
                }
                return result;
            }
        } else if (b1Border && b2Border) {
            result += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result += this.options.chars.top.repeat(length);
        } else {
            result += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result += this.options.chars.rightMid;
            } else if (b1Border) {
                result += this.options.chars.bottomRight;
            } else if (b2Border) {
                result += this.options.chars.topRight;
            } else {
                result += " ";
            }
        }
        return result;
    }
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        }
    };
    headerRow;
    static from(rows) {
        const table = new this(...rows);
        if (rows instanceof Table) {
            table.options = {
                ...rows.options
            };
            table.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)
        ));
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()
        ));
        table.options = {
            ...this.options
        };
        table.headerRow = this.headerRow?.clone();
        return table;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.toString() + "\n"));
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder : false
            )
        );
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class HelpGenerator {
    cmd;
    indent = 2;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd, options = {}){
        this.cmd = cmd;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            ...options
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result = this.generateHeader() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples() + "\n";
        setColorEnabled(areColorsEnabled);
        return result;
    }
    generateHeader() {
        const rows = [
            [
                bold("Usage:"),
                magenta(`${this.cmd.getPath()}${this.cmd.getArgsDefinition() ? " " + this.cmd.getArgsDefinition() : ""}`), 
            ], 
        ];
        const version1 = this.cmd.getVersion();
        if (version1) {
            rows.push([
                bold("Version:"),
                yellow(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).indent(this.indent).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                this.cmd.getDescription()
            ], 
        ]).indent(this.indent * 2).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options = this.cmd.getOptions(false);
        if (!options.length) {
            return "";
        }
        const hasTypeDefinitions = !!options.find((option)=>!!option.typeDefinition
        );
        if (hasTypeDefinitions) {
            return this.label("Options") + Table.from([
                ...options.map((option)=>[
                        option.flags.map((flag)=>blue(flag)
                        ).join(", "),
                        highlightArguments(option.typeDefinition || "", this.options.types),
                        red(bold("-")) + " " + option.description.split("\n").shift(),
                        this.generateHints(option), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).maxColWidth([
                60,
                60,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label("Options") + Table.from([
            ...options.map((option)=>[
                    option.flags.map((flag)=>blue(flag)
                    ).join(", "),
                    red(bold("-")) + " " + option.description.split("\n").shift(),
                    this.generateHints(option), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).maxColWidth([
            60,
            80,
            60
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands6 = this.cmd.getCommands(false);
        if (!commands6.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands6.find((command)=>!!command.getArgsDefinition()
        );
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands6.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name25)=>blue(name25)
                        ).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands6.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name26)=>blue(name26)
                    ).join(", "),
                    red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name27)=>blue(name27)
                    ).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    `${red(bold("-"))} ${envVar.description}`, 
                ]
            ), 
        ]).padding(2).indent(this.indent * 2).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim(bold(`${capitalize(example.name)}:`)),
                example.description, 
            ]
        )).padding(1).indent(this.indent * 2).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow(`required`));
        typeof option.default !== "undefined" && hints.push(bold(`Default: `) + inspect(option.default, this.options.colors));
        option.depends?.length && hints.push(yellow(bold(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red(bold(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type instanceof Type) {
            const possibleValues = type.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold(`Values: `) + possibleValues.map((value)=>inspect(value, this.options.colors)
                ).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + " ".repeat(this.indent) + bold(`${label}:`) + "\n\n";
    }
}
function capitalize(string2) {
    return (string2?.charAt(0).toUpperCase() + string2.slice(1)) ?? "";
}
function inspect(value, colors) {
    return Deno.inspect(value, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types5 = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition).map((arg)=>highlightArgumentDetails(arg, types5)
    ).join(" ");
}
function highlightArgumentDetails(arg, types6 = true) {
    let str50 = "";
    str50 += yellow(arg.optionalValue ? "[" : "<");
    let name28 = "";
    name28 += arg.name;
    if (arg.variadic) {
        name28 += "...";
    }
    name28 = magenta(name28);
    str50 += name28;
    if (types6) {
        str50 += yellow(":");
        str50 += red(arg.type);
    }
    if (arg.list) {
        str50 += green("[]");
    }
    str50 += yellow(arg.optionalValue ? "]" : ">");
    return str50;
}
class IntegerType extends Type {
    parse(type) {
        return integer(type);
    }
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    fn;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = true;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOption;
    _helpOption;
    _help;
    versionOption(flags, desc, opts) {
        this._versionOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        const result = splitArguments(nameAndArguments);
        const name29 = result.flags.shift();
        const aliases = result.flags;
        if (!name29) {
            throw new MissingCommandName();
        }
        if (this.getBaseCommand(name29, true)) {
            if (!override) {
                throw new DuplicateCommandName(name29);
            }
            this.removeCommand(name29);
        }
        let description;
        let cmd;
        if (typeof cmdOrDescription === "string") {
            description = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd = cmdOrDescription.reset();
        } else {
            cmd = new Command();
        }
        cmd._name = name29;
        cmd._parent = this;
        if (description) {
            cmd.description(description);
        }
        if (result.typeDefinition) {
            cmd.arguments(result.typeDefinition);
        }
        aliases.forEach((alias)=>cmd.alias(alias)
        );
        this.commands.set(name29, cmd);
        this.select(name29);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAlias(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this.cmd = this;
        return this;
    }
    select(name30) {
        const cmd = this.getBaseCommand(name30, true);
        if (!cmd) {
            throw new CommandNotFound(name30, this.getBaseCommands(true));
        }
        this.cmd = cmd;
        return this;
    }
    name(name31) {
        this.cmd._name = name31;
        return this;
    }
    version(version2) {
        if (typeof version2 === "string") {
            this.cmd.ver = ()=>version2
            ;
        } else if (typeof version2 === "function") {
            this.cmd.ver = version2;
        }
        return this;
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help
            ;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd)=>HelpGenerator.generate(cmd, help)
            ;
        }
        return this;
    }
    description(description) {
        this.cmd.desc = description;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.fn = fn;
        return this;
    }
    allowEmpty(allowEmpty = true) {
        this.cmd._allowEmpty = allowEmpty;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name32) {
        this.cmd.defaultCommand = name32;
        return this;
    }
    globalType(name33, type, options) {
        return this.type(name33, type, {
            ...options,
            global: true
        });
    }
    type(name34, handler, options) {
        if (this.cmd.types.get(name34) && !options?.override) {
            throw new DuplicateType(name34);
        }
        this.cmd.types.set(name34, {
            ...options,
            name: name34,
            handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd, parent)=>handler.complete?.(cmd, parent) || []
            ;
            this.complete(name34, completeHandler, options);
        }
        return this;
    }
    globalComplete(name35, complete, options) {
        return this.complete(name35, complete, {
            ...options,
            global: true
        });
    }
    complete(name36, complete, options) {
        if (this.cmd.completions.has(name36) && !options?.override) {
            throw new DuplicateCompletion(name36);
        }
        this.cmd.completions.set(name36, {
            name: name36,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    shouldThrowErrors() {
        return this.cmd.throwOnError || !!this.cmd._parent?.shouldThrowErrors();
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts
            });
        }
        const result = splitArguments(flags);
        const args = result.typeDefinition ? parseArgumentsDefinition(result.typeDefinition) : [];
        const option = {
            ...opts,
            name: "",
            description: desc,
            args,
            flags: result.flags,
            typeDefinition: result.typeDefinition
        };
        if (option.separator) {
            for (const arg of args){
                if (arg.list) {
                    arg.separator = option.separator;
                }
            }
        }
        for (const part of option.flags){
            const arg = part.trim();
            const isLong = /^--/.test(arg);
            const name37 = isLong ? arg.slice(2) : arg.slice(1);
            if (this.cmd.getBaseOption(name37, true)) {
                if (opts?.override) {
                    this.removeOption(name37);
                } else {
                    throw new DuplicateOptionName(name37);
                }
            }
            if (!option.name && isLong) {
                option.name = name37;
            } else if (!option.aliases) {
                option.aliases = [
                    name37
                ];
            } else {
                option.aliases.push(name37);
            }
        }
        if (option.prepend) {
            this.cmd.options.unshift(option);
        } else {
            this.cmd.options.push(option);
        }
        return this;
    }
    example(name38, description) {
        if (this.cmd.hasExample(name38)) {
            throw new DuplicateExample(name38);
        }
        this.cmd.examples.push({
            name: name38,
            description
        });
        return this;
    }
    globalEnv(name39, description, options) {
        return this.env(name39, description, {
            ...options,
            global: true
        });
    }
    env(name40, description, options) {
        const result = splitArguments(name40);
        if (!result.typeDefinition) {
            result.typeDefinition = "<value:boolean>";
        }
        if (result.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true)
        )) {
            throw new DuplicateEnvironmentVariable(name40);
        }
        const details = parseArgumentsDefinition(result.typeDefinition);
        if (details.length > 1) {
            throw new EnvironmentVariableSingleValue(name40);
        } else if (details.length && details[0].optionalValue) {
            throw new EnvironmentVariableOptionalValue(name40);
        } else if (details.length && details[0].variadic) {
            throw new EnvironmentVariableVariadicValue(name40);
        }
        this.cmd.envVars.push({
            name: result.flags[0],
            names: result.flags,
            description,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    async parse(args = Deno.args, dry) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = args;
            const subCommand = args.length > 0 && this.getCommand(args[0], true);
            if (subCommand) {
                subCommand._globalParent = this;
                return await subCommand.parse(this.rawArgs.slice(1), dry);
            }
            const result = {
                options: {},
                args: this.rawArgs,
                cmd: this,
                literal: this.literalArgs
            };
            if (this.isExecutable) {
                if (!dry) {
                    await this.executeExecutable(this.rawArgs);
                }
                return result;
            } else if (this._useRawArgs) {
                if (dry) {
                    return result;
                }
                return await this.execute({}, ...this.rawArgs);
            } else {
                const { action , flags , unknown , literal: literal2  } = this.parseFlags(this.rawArgs);
                this.literalArgs = literal2;
                const params = this.parseArguments(unknown, flags);
                await this.validateEnvVars();
                if (dry || action) {
                    if (action) {
                        await action.call(this, flags, ...params);
                    }
                    return {
                        options: flags,
                        args: params,
                        cmd: this,
                        literal: this.literalArgs
                    };
                }
                return await this.execute(flags, ...params);
            }
        } catch (error1) {
            throw this.error(error1);
        }
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        if (!this._help) {
            this.help({
                hints: true,
                types: false
            });
        }
        if (this._versionOption !== false && (this._versionOption || this.ver)) {
            this.option(this._versionOption?.flags || "-V, --version", this._versionOption?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: function() {
                    this.showVersion();
                    Deno.exit(0);
                },
                ...this._versionOption?.opts ?? {}
            });
        }
        if (this._helpOption !== false) {
            this.option(this._helpOption?.flags || "-h, --help", this._helpOption?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: function() {
                    this.showHelp();
                    Deno.exit(0);
                },
                ...this._helpOption?.opts ?? {}
            });
        }
        return this;
    }
    async execute(options, ...args) {
        if (this.fn) {
            await this.fn(options, ...args);
        } else if (this.defaultCommand) {
            const cmd = this.getCommand(this.defaultCommand, true);
            if (!cmd) {
                throw new DefaultCommandNotFound(this.defaultCommand, this.getCommands());
            }
            cmd._globalParent = this;
            await cmd.execute(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeExecutable(args) {
        const permissions = await getPermissions();
        if (!permissions.read) {
            await Deno.permissions?.request({
                name: "read"
            });
        }
        if (!permissions.run) {
            await Deno.permissions?.request({
                name: "run"
            });
        }
        const [main1, ...names] = this.getPath().split(" ");
        names.unshift(main1.replace(/\.ts$/, ""));
        const executableName = names.join("-");
        const files = [];
        const parts = Deno.mainModule.replace(/^file:\/\//g, "").split("/");
        if (Deno.build.os === "windows" && parts[0] === "") {
            parts.shift();
        }
        parts.pop();
        const path2 = parts.join("/");
        files.push(path2 + "/" + executableName, path2 + "/" + executableName + ".ts");
        files.push(executableName, executableName + ".ts");
        const denoOpts = [];
        if (isUnstable()) {
            denoOpts.push("--unstable");
        }
        denoOpts.push("--allow-read", "--allow-run");
        Object.keys(permissions).forEach((name41)=>{
            if (name41 === "read" || name41 === "run") {
                return;
            }
            if (permissions[name41]) {
                denoOpts.push(`--allow-${name41}`);
            }
        });
        for (const file of files){
            try {
                Deno.lstatSync(file);
            } catch (error2) {
                if (error2 instanceof Deno.errors.NotFound) {
                    continue;
                }
                throw error2;
            }
            const cmd = [
                "deno",
                "run",
                ...denoOpts,
                file,
                ...args
            ];
            const process = Deno.run({
                cmd: cmd
            });
            const status = await process.status();
            if (!status.success) {
                Deno.exit(status.code);
            }
            return;
        }
        throw new CommandExecutableNotFound(executableName, files);
    }
    parseFlags(args) {
        try {
            let action;
            const result = parseFlags(args, {
                stopEarly: this._stopEarly,
                allowEmpty: this._allowEmpty,
                flags: this.getOptions(true),
                parse: (type)=>this.parseType(type)
                ,
                option: (option)=>{
                    if (!action && option.action) {
                        action = option.action;
                    }
                }
            });
            return {
                ...result,
                action
            };
        } catch (error3) {
            if (error3 instanceof ValidationError) {
                throw new ValidationError1(error3.message);
            }
            throw error3;
        }
    }
    parseType(type1) {
        const typeSettings = this.getType(type1.type);
        if (!typeSettings) {
            throw new UnknownType(type1.type, this.getTypes().map((type)=>type.name
            ));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type1) : typeSettings.handler(type1);
    }
    async validateEnvVars() {
        if (!await hasPermission("env")) {
            return;
        }
        const envVars = this.getEnvVars(true);
        if (!envVars.length) {
            return;
        }
        envVars.forEach((env)=>{
            const name1 = env.names.find((name42)=>!!Deno.env.get(name42)
            );
            if (name1) {
                this.parseType({
                    label: "Environment variable",
                    type: env.type,
                    name: name1,
                    value: Deno.env.get(name1) ?? ""
                });
            }
        });
    }
    parseArguments(args, flags) {
        const params = [];
        args = args.slice(0);
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    throw new UnknownCommand(args[0], this.getCommands());
                } else {
                    throw new NoArgumentsAllowed(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required = this.getArguments().filter((expectedArg)=>!expectedArg.optionalValue
                ).map((expectedArg)=>expectedArg.name
                );
                if (required.length) {
                    const flagNames = Object.keys(flags);
                    const hasStandaloneOption = !!flagNames.find((name43)=>this.getOption(name43, true)?.standalone
                    );
                    if (!hasStandaloneOption) {
                        throw new MissingArguments(required);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optionalValue) {
                            break;
                        }
                        throw new MissingArgument(`Missing argument: ${expectedArg.name}`);
                    }
                    let arg;
                    if (expectedArg.variadic) {
                        arg = args.splice(0, args.length).map((value)=>this.parseType({
                                label: "Argument",
                                type: expectedArg.type,
                                name: expectedArg.name,
                                value
                            })
                        );
                    } else {
                        arg = this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value: args.shift()
                        });
                    }
                    if (arg) {
                        params.push(arg);
                    }
                }
                if (args.length) {
                    throw new TooManyArguments(args);
                }
            }
        }
        return params;
    }
    error(error4) {
        if (this.shouldThrowErrors() || !(error4 instanceof ValidationError1)) {
            return error4;
        }
        this.showHelp();
        Deno.stderr.writeSync(new TextEncoder().encode(red(`  ${bold("error")}: ${error4.message}\n`) + "\n"));
        Deno.exit(error4 instanceof ValidationError1 ? error4.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath() {
        return this._parent ? this._parent.getPath() + " " + this._name : this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name44) {
        return this.getArguments().find((arg)=>arg.name === name44
        );
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getShortDescription() {
        return this.getDescription().trim().split("\n").shift();
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getVersion()));
    }
    showHelp() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getHelp()));
    }
    getHelp() {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this);
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    hasOptions(hidden1) {
        return this.getOptions(hidden1).length > 0;
    }
    getOptions(hidden2) {
        return this.getGlobalOptions(hidden2).concat(this.getBaseOptions(hidden2));
    }
    getBaseOptions(hidden3) {
        if (!this.options.length) {
            return [];
        }
        return hidden3 ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden
        );
    }
    getGlobalOptions(hidden4) {
        const getOptions1 = (cmd, options = [], names = [])=>{
            if (cmd) {
                if (cmd.options.length) {
                    cmd.options.forEach((option)=>{
                        if (option.global && !this.options.find((opt)=>opt.name === option.name
                        ) && names.indexOf(option.name) === -1 && (hidden4 || !option.hidden)) {
                            names.push(option.name);
                            options.push(option);
                        }
                    });
                }
                return getOptions1(cmd._parent, options, names);
            }
            return options;
        };
        return getOptions1(this._parent);
    }
    hasOption(name45, hidden5) {
        return !!this.getOption(name45, hidden5);
    }
    getOption(name46, hidden6) {
        return this.getBaseOption(name46, hidden6) ?? this.getGlobalOption(name46, hidden6);
    }
    getBaseOption(name47, hidden7) {
        const option1 = this.options.find((option)=>option.name === name47
        );
        return option1 && (hidden7 || !option1.hidden) ? option1 : undefined;
    }
    getGlobalOption(name48, hidden8) {
        if (!this._parent) {
            return;
        }
        const option = this._parent.getBaseOption(name48, hidden8);
        if (!option || !option.global) {
            return this._parent.getGlobalOption(name48, hidden8);
        }
        return option;
    }
    removeOption(name49) {
        const index = this.options.findIndex((option)=>option.name === name49
        );
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden9) {
        return this.getCommands(hidden9).length > 0;
    }
    getCommands(hidden10) {
        return this.getGlobalCommands(hidden10).concat(this.getBaseCommands(hidden10));
    }
    getBaseCommands(hidden11) {
        const commands7 = Array.from(this.commands.values());
        return hidden11 ? commands7 : commands7.filter((cmd)=>!cmd.isHidden
        );
    }
    getGlobalCommands(hidden12) {
        const getCommands = (cmd1, commands8 = [], names = [])=>{
            if (cmd1) {
                if (cmd1.commands.size) {
                    cmd1.commands.forEach((cmd)=>{
                        if (cmd.isGlobal && this !== cmd && !this.commands.has(cmd._name) && names.indexOf(cmd._name) === -1 && (hidden12 || !cmd.isHidden)) {
                            names.push(cmd._name);
                            commands8.push(cmd);
                        }
                    });
                }
                return getCommands(cmd1._parent, commands8, names);
            }
            return commands8;
        };
        return getCommands(this._parent);
    }
    hasCommand(name50, hidden13) {
        return !!this.getCommand(name50, hidden13);
    }
    getCommand(name51, hidden14) {
        return this.getBaseCommand(name51, hidden14) ?? this.getGlobalCommand(name51, hidden14);
    }
    getBaseCommand(name52, hidden15) {
        for (const cmd of this.commands.values()){
            if (cmd._name === name52 || cmd.aliases.includes(name52)) {
                return cmd && (hidden15 || !cmd.isHidden) ? cmd : undefined;
            }
        }
    }
    getGlobalCommand(name53, hidden16) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseCommand(name53, hidden16);
        if (!cmd?.isGlobal) {
            return this._parent.getGlobalCommand(name53, hidden16);
        }
        return cmd;
    }
    removeCommand(name54) {
        const command = this.getBaseCommand(name54, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd, types7 = [], names = [])=>{
            if (cmd) {
                if (cmd.types.size) {
                    cmd.types.forEach((type)=>{
                        if (type.global && !this.types.has(type.name) && names.indexOf(type.name) === -1) {
                            names.push(type.name);
                            types7.push(type);
                        }
                    });
                }
                return getTypes(cmd._parent, types7, names);
            }
            return types7;
        };
        return getTypes(this._parent);
    }
    getType(name55) {
        return this.getBaseType(name55) ?? this.getGlobalType(name55);
    }
    getBaseType(name56) {
        return this.types.get(name56);
    }
    getGlobalType(name57) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseType(name57);
        if (!cmd?.global) {
            return this._parent.getGlobalType(name57);
        }
        return cmd;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd, completions = [], names = [])=>{
            if (cmd) {
                if (cmd.completions.size) {
                    cmd.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name58) {
        return this.getBaseCompletion(name58) ?? this.getGlobalCompletion(name58);
    }
    getBaseCompletion(name59) {
        return this.completions.get(name59);
    }
    getGlobalCompletion(name60) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name60);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name60);
        }
        return completion;
    }
    hasEnvVars(hidden17) {
        return this.getEnvVars(hidden17).length > 0;
    }
    getEnvVars(hidden18) {
        return this.getGlobalEnvVars(hidden18).concat(this.getBaseEnvVars(hidden18));
    }
    getBaseEnvVars(hidden19) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden19 ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden
        );
    }
    getGlobalEnvVars(hidden20) {
        const getEnvVars = (cmd, envVars = [], names = [])=>{
            if (cmd) {
                if (cmd.envVars.length) {
                    cmd.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]
                        ) && names.indexOf(envVar.names[0]) === -1 && (hidden20 || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name61, hidden21) {
        return !!this.getEnvVar(name61, hidden21);
    }
    getEnvVar(name62, hidden22) {
        return this.getBaseEnvVar(name62, hidden22) ?? this.getGlobalEnvVar(name62, hidden22);
    }
    getBaseEnvVar(name63, hidden23) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name63) !== -1
        );
        return envVar && (hidden23 || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name64, hidden24) {
        if (!this._parent) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name64, hidden24);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name64, hidden24);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name65) {
        return !!this.getExample(name65);
    }
    getExample(name66) {
        return this.examples.find((example)=>example.name === name66
        );
    }
}
class BashCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new BashCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path3 = this.cmd.getPath();
        const version3 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path3}${version3}

_${replaceSpecialChars(path3)}() {
  local word cur prev
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()

  _${replaceSpecialChars(this.cmd.getName())}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.cmd.getName()} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  ${this.generateCompletions(this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars(path3)} -o bashdefault -o default ${path3}
`;
    }
    generateCompletions(command, path4 = "", index = 1) {
        path4 = (path4 ? path4 + " " : "") + command.getName();
        const commandCompletions = this.generateCommandCompletions(command, path4, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path4, index + 1)
        ).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path5, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName()
        );
        const completionsPath = ~path5.indexOf(" ") ? " " + path5.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command.getArguments(), completionsPath);
        return `  __${replaceSpecialChars(path5)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option)=>option.flags
        ).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options = command.getOptions(false);
        if (options.length) {
            opts += 'case "${prev}" in';
            for (const option of options){
                const flags = option.flags.map((flag)=>flag.trim()
                ).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(option.args, completionsPath, {
                    standalone: option.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(args, path6) {
        if (args.length) {
            return `_${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path6}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(args, path7, opts) {
        if (args.length) {
            return `opts=(); _${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path7}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
}
function replaceSpecialChars(str51) {
    return str51.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim(italic("~/.bashrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(BashCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd1){
        super();
        this.description("Get completions for given action from given command.").arguments("<action:string> [command...:string]").action(async (_, action, commandNames)=>{
            let parent;
            const completeCommand = commandNames?.reduce((cmd, name67)=>{
                parent = cmd;
                const childCmd = cmd.getCommand(name67, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommand(name67, cmd.getCommands());
                }
                return childCmd;
            }, cmd1 || this.getMainCommand()) ?? (cmd1 || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result = await completion?.complete(completeCommand, parent) ?? [];
            if (result?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result.join("\n")));
            }
        }).reset();
    }
}
class FishCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new FishCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path8 = this.cmd.getPath();
        const version4 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path8}${version4}

function __fish_${replaceSpecialChars1(this.cmd.getName())}_using_command
  set cmds ${getCommandFnNames(this.cmd).join(" ")}
  set words (commandline -opc)
  set cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if [ "$cmd" = "$argv[1]" ]
    return 0
  end
  return 1
end

${this.generateCompletions(this.cmd).trim()}
`;
    }
    generateCompletions(command) {
        const parent = command.getParent();
        let result = ``;
        if (parent) {
            result += "\n" + this.complete(parent, {
                description: command.getShortDescription(),
                arguments: command.getName()
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(commandArgs[0].action + " " + getCompletionsPath(command)) : undefined
            });
        }
        for (const option of command.getOptions(false)){
            result += "\n" + this.completeOption(command, option);
        }
        for (const subCommand of command.getCommands(false)){
            result += this.generateCompletions(subCommand);
        }
        return result;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2
        )?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2
        )?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: option.description,
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(option.args[0].action + " " + getCompletionsPath(command)) : undefined
        });
    }
    complete(command, options) {
        const cmd = [
            "complete"
        ];
        cmd.push("-c", this.cmd.getName());
        cmd.push("-n", `'__fish_${replaceSpecialChars1(this.cmd.getName())}_using_command __${replaceSpecialChars1(command.getPath())}'`);
        options.shortOption && cmd.push("-s", options.shortOption);
        options.longOption && cmd.push("-l", options.longOption);
        options.standalone && cmd.push("-x");
        cmd.push("-k");
        cmd.push("-f");
        if (options.arguments) {
            options.required && cmd.push("-r");
            cmd.push("-a", options.arguments);
        }
        options.description && cmd.push("-d", `'${options.description}'`);
        return cmd.join(" ");
    }
    getCompletionCommand(cmd) {
        return `'(${this.cmd.getName()} completions complete ${cmd.trim()})'`;
    }
}
function getCommandFnNames(cmd, cmds = []) {
    cmds.push(`__${replaceSpecialChars1(cmd.getPath())}`);
    cmd.getCommands(false).forEach((command)=>{
        getCommandFnNames(command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars1(str52) {
    return str52.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim(italic("~/.config/fish/config.fish"))}:

    ${dim(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(FishCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class ZshCompletionsGenerator {
    cmd;
    actions = new Map();
    static generate(cmd) {
        return new ZshCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path9 = this.cmd.getPath();
        const name68 = this.cmd.getName();
        const version5 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path9}${version5}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars2(name68)}_complete] )) ||
function __${replaceSpecialChars2(name68)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${name68} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.cmd).trim()}

# _${replaceSpecialChars2(path9)} "\${@}"

compdef _${replaceSpecialChars2(path9)} ${path9}

`;
    }
    generateCompletions(command, path10 = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path10 = (path10 ? path10 + " " : "") + command.getName();
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars2(path10)}] )) ||
function _${replaceSpecialChars2(path10)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path10) + this.generateSubCommandCompletions(command, path10) + this.generateArgumentCompletions(command, path10) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path10)
        ).join("");
    }
    generateCommandCompletions(command, path11) {
        const commands9 = command.getCommands(false);
        let completions = commands9.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription()}'`
        ).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path11.split(" ").slice(1).join(" ");
            const arg = command.getArguments()[0];
            const action = this.addAction(arg, completionsPath);
            if (action && command.getCompletion(arg.action)) {
                completions += `\n    __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command1, path12) {
        if (command1.hasCommands(false)) {
            const actions = command1.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars2(path12 + " " + command.getName())} ;;`
            ).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path13) {
        this.actions.clear();
        const options = this.generateOptions(command, path13);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg)=>command.getCompletion(arg.action)
        ).length) {
            argsCommand += ` \\\n    '${++argIndex}: :_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args = [];
            for (const arg1 of command.getArguments().slice(1)){
                const completionsPath = path13.split(" ").slice(1).join(" ");
                const action = this.addAction(arg1, completionsPath);
                args.push(`${++argIndex}${arg1.optionalValue ? "::" : ":"}${action.name}`);
            }
            argsCommand += args.map((arg)=>`\\\n    '${arg}'`
            ).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*:: :->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path14) {
        const options = [];
        const cmdArgs = path14.split(" ");
        cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option)=>option.standalone ? option.flags : false
        ).flat().filter((flag)=>typeof flag === "string"
        );
        for (const option1 of command.getOptions(false)){
            options.push(this.generateOption(option1, completionsPath, excludedFlags));
        }
        return options;
    }
    generateOption(option, completionsPath, excludedOptions) {
        const flags = option.flags;
        let excludedFlags = option.conflicts?.length ? [
            ...excludedOptions,
            ...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")
            ), 
        ] : excludedOptions;
        excludedFlags = option.collect ? excludedFlags : [
            ...excludedFlags,
            ...flags, 
        ];
        let args = "";
        for (const arg of option.args){
            const action = this.addAction(arg, completionsPath);
            if (arg.variadic) {
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`;
            } else {
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`;
            }
        }
        let description = option.description.trim().split("\n").shift();
        description = description.replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        if (option.standalone) {
            return `'(- *)'{${collect}${flags}}'[${description}]${args}'`;
        } else {
            const excluded = excludedFlags.length ? `'(${excludedFlags.join(" ")})'` : "";
            if (collect || flags.length > 1) {
                return `${excluded}{${collect}${flags}}'[${description}]${args}'`;
            } else {
                return `${excluded}${flags}'[${description}]${args}'`;
            }
        }
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name69, action])=>`${name69}) __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`
            );
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
}
function replaceSpecialChars2(str53) {
    return str53.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim(italic("~/.zshrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(ZshCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim(italic("~/.bashrc"))} or similar:

    ${dim(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).action(()=>this.showHelp()
        ).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd).hidden()).reset();
    }
}
class CommandType extends StringType {
    complete(_cmd, parent) {
        return parent?.getCommands(false).map((cmd)=>cmd.getName()
        ) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd){
        super();
        this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").action((_, name70)=>{
            if (!cmd) {
                cmd = name70 ? this.getGlobalParent()?.getBaseCommand(name70) : this.getGlobalParent();
            }
            if (!cmd) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommand(name70 ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases(), 
                ]);
            }
            cmd.showHelp();
            Deno.exit(0);
        });
    }
}
class EnumType extends Type {
    allowedValues;
    constructor(values3){
        super();
        this.allowedValues = values3;
    }
    parse(type) {
        for (const value of this.allowedValues){
            if (value.toString() === type.value) {
                return value;
            }
        }
        throw new InvalidTypeError(type, this.allowedValues.slice());
    }
    values() {
        return this.allowedValues.slice();
    }
    complete() {
        return this.values();
    }
}
const osType = (()=>{
    const { Deno  } = globalThis;
    if (typeof Deno?.build?.os === "string") {
        return Deno.build.os;
    }
    const { navigator  } = globalThis;
    if (navigator?.appVersion?.includes?.("Win") ?? false) {
        return "windows";
    }
    return "linux";
})();
const isWindows = osType === "windows";
const CHAR_FORWARD_SLASH = 47;
function assertPath(path15) {
    if (typeof path15 !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path15)}`);
    }
}
function isPosixPathSeparator(code2) {
    return code2 === 47;
}
function isPathSeparator(code3) {
    return isPosixPathSeparator(code3) || code3 === 92;
}
function isWindowsDeviceRoot(code4) {
    return code4 >= 97 && code4 <= 122 || code4 >= 65 && code4 <= 90;
}
function normalizeString(path16, allowAboveRoot, separator, isPathSeparator1) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code5;
    for(let i8 = 0, len = path16.length; i8 <= len; ++i8){
        if (i8 < len) code5 = path16.charCodeAt(i8);
        else if (isPathSeparator1(code5)) break;
        else code5 = CHAR_FORWARD_SLASH;
        if (isPathSeparator1(code5)) {
            if (lastSlash === i8 - 1 || dots === 1) {} else if (lastSlash !== i8 - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i8;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i8;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path16.slice(lastSlash + 1, i8);
                else res = path16.slice(lastSlash + 1, i8);
                lastSegmentLength = i8 - lastSlash - 1;
            }
            lastSlash = i8;
            dots = 0;
        } else if (code5 === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function _format(sep3, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base1 = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base1;
    if (dir === pathObject.root) return dir + base1;
    return dir + sep3 + base1;
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string3) {
    return string3.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
class DenoStdInternalError extends Error {
    constructor(message){
        super(message);
        this.name = "DenoStdInternalError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new DenoStdInternalError(msg);
    }
}
const sep = "\\";
const delimiter = ";";
function resolve(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i9 = pathSegments.length - 1; i9 >= -1; i9--){
        let path17;
        const { Deno  } = globalThis;
        if (i9 >= 0) {
            path17 = pathSegments[i9];
        } else if (!resolvedDevice) {
            if (typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path17 = Deno.cwd();
        } else {
            if (typeof Deno?.env?.get !== "function" || typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path17 = Deno.cwd();
            if (path17 === undefined || path17.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path17 = `${resolvedDevice}\\`;
            }
        }
        assertPath(path17);
        const len = path17.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute1 = false;
        const code6 = path17.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code6)) {
                isAbsolute1 = true;
                if (isPathSeparator(path17.charCodeAt(1))) {
                    let j = 2;
                    let last = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path17.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        const firstPart = path17.slice(last, j);
                        last = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path17.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last) {
                            last = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path17.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path17.slice(last)}`;
                                rootEnd = j;
                            } else if (j !== last) {
                                device = `\\\\${firstPart}\\${path17.slice(last, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code6)) {
                if (path17.charCodeAt(1) === 58) {
                    device = path17.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path17.charCodeAt(2))) {
                            isAbsolute1 = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code6)) {
            rootEnd = 1;
            isAbsolute1 = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path17.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute1;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function normalize1(path18) {
    assertPath(path18);
    const len = path18.length;
    if (len === 0) return ".";
    let rootEnd = 0;
    let device;
    let isAbsolute2 = false;
    const code7 = path18.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code7)) {
            isAbsolute2 = true;
            if (isPathSeparator(path18.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path18.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    const firstPart = path18.slice(last, j);
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path18.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path18.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path18.slice(last)}\\`;
                        } else if (j !== last) {
                            device = `\\\\${firstPart}\\${path18.slice(last, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code7)) {
            if (path18.charCodeAt(1) === 58) {
                device = path18.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path18.charCodeAt(2))) {
                        isAbsolute2 = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code7)) {
        return "\\";
    }
    let tail;
    if (rootEnd < len) {
        tail = normalizeString(path18.slice(rootEnd), !isAbsolute2, "\\", isPathSeparator);
    } else {
        tail = "";
    }
    if (tail.length === 0 && !isAbsolute2) tail = ".";
    if (tail.length > 0 && isPathSeparator(path18.charCodeAt(len - 1))) {
        tail += "\\";
    }
    if (device === undefined) {
        if (isAbsolute2) {
            if (tail.length > 0) return `\\${tail}`;
            else return "\\";
        } else if (tail.length > 0) {
            return tail;
        } else {
            return "";
        }
    } else if (isAbsolute2) {
        if (tail.length > 0) return `${device}\\${tail}`;
        else return `${device}\\`;
    } else if (tail.length > 0) {
        return device + tail;
    } else {
        return device;
    }
}
function isAbsolute(path19) {
    assertPath(path19);
    const len = path19.length;
    if (len === 0) return false;
    const code8 = path19.charCodeAt(0);
    if (isPathSeparator(code8)) {
        return true;
    } else if (isWindowsDeviceRoot(code8)) {
        if (len > 2 && path19.charCodeAt(1) === 58) {
            if (isPathSeparator(path19.charCodeAt(2))) return true;
        }
    }
    return false;
}
function join(...paths) {
    const pathsCount = paths.length;
    if (pathsCount === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i10 = 0; i10 < pathsCount; ++i10){
        const path20 = paths[i10];
        assertPath(path20);
        if (path20.length > 0) {
            if (joined === undefined) joined = firstPart = path20;
            else joined += `\\${path20}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart != null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize1(joined);
}
function relative(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    const fromOrig = resolve(from);
    const toOrig = resolve(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i11 = 0;
    for(; i11 <= length; ++i11){
        if (i11 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i11) === 92) {
                    return toOrig.slice(toStart + i11 + 1);
                } else if (i11 === 2) {
                    return toOrig.slice(toStart + i11);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i11) === 92) {
                    lastCommonSep = i11;
                } else if (i11 === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i11);
        const toCode = to.charCodeAt(toStart + i11);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i11;
    }
    if (i11 !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i11 = fromStart + lastCommonSep + 1; i11 <= fromEnd; ++i11){
        if (i11 === fromEnd || from.charCodeAt(i11) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
function toNamespacedPath(path21) {
    if (typeof path21 !== "string") return path21;
    if (path21.length === 0) return "";
    const resolvedPath = resolve(path21);
    if (resolvedPath.length >= 3) {
        if (resolvedPath.charCodeAt(0) === 92) {
            if (resolvedPath.charCodeAt(1) === 92) {
                const code9 = resolvedPath.charCodeAt(2);
                if (code9 !== 63 && code9 !== 46) {
                    return `\\\\?\\UNC\\${resolvedPath.slice(2)}`;
                }
            }
        } else if (isWindowsDeviceRoot(resolvedPath.charCodeAt(0))) {
            if (resolvedPath.charCodeAt(1) === 58 && resolvedPath.charCodeAt(2) === 92) {
                return `\\\\?\\${resolvedPath}`;
            }
        }
    }
    return path21;
}
function dirname(path22) {
    assertPath(path22);
    const len = path22.length;
    if (len === 0) return ".";
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code10 = path22.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code10)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path22.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path22.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path22.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path22.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path22;
                        }
                        if (j !== last) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code10)) {
            if (path22.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path22.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code10)) {
        return path22;
    }
    for(let i12 = len - 1; i12 >= offset; --i12){
        if (isPathSeparator(path22.charCodeAt(i12))) {
            if (!matchedSlash) {
                end = i12;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return path22.slice(0, end);
}
function basename(path23, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path23);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i13;
    if (path23.length >= 2) {
        const drive = path23.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path23.charCodeAt(1) === 58) start = 2;
        }
    }
    if (ext !== undefined && ext.length > 0 && ext.length <= path23.length) {
        if (ext.length === path23.length && ext === path23) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i13 = path23.length - 1; i13 >= start; --i13){
            const code11 = path23.charCodeAt(i13);
            if (isPathSeparator(code11)) {
                if (!matchedSlash) {
                    start = i13 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i13 + 1;
                }
                if (extIdx >= 0) {
                    if (code11 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i13;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path23.length;
        return path23.slice(start, end);
    } else {
        for(i13 = path23.length - 1; i13 >= start; --i13){
            if (isPathSeparator(path23.charCodeAt(i13))) {
                if (!matchedSlash) {
                    start = i13 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i13 + 1;
            }
        }
        if (end === -1) return "";
        return path23.slice(start, end);
    }
}
function extname(path24) {
    assertPath(path24);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path24.length >= 2 && path24.charCodeAt(1) === 58 && isWindowsDeviceRoot(path24.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i14 = path24.length - 1; i14 >= start; --i14){
        const code12 = path24.charCodeAt(i14);
        if (isPathSeparator(code12)) {
            if (!matchedSlash) {
                startPart = i14 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i14 + 1;
        }
        if (code12 === 46) {
            if (startDot === -1) startDot = i14;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path24.slice(startDot, end);
}
function format(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("\\", pathObject);
}
function parse(path25) {
    assertPath(path25);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    const len = path25.length;
    if (len === 0) return ret;
    let rootEnd = 0;
    let code13 = path25.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code13)) {
            rootEnd = 1;
            if (isPathSeparator(path25.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path25.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path25.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path25.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            rootEnd = j;
                        } else if (j !== last) {
                            rootEnd = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code13)) {
            if (path25.charCodeAt(1) === 58) {
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path25.charCodeAt(2))) {
                        if (len === 3) {
                            ret.root = ret.dir = path25;
                            return ret;
                        }
                        rootEnd = 3;
                    }
                } else {
                    ret.root = ret.dir = path25;
                    return ret;
                }
            }
        }
    } else if (isPathSeparator(code13)) {
        ret.root = ret.dir = path25;
        return ret;
    }
    if (rootEnd > 0) ret.root = path25.slice(0, rootEnd);
    let startDot = -1;
    let startPart = rootEnd;
    let end = -1;
    let matchedSlash = true;
    let i15 = path25.length - 1;
    let preDotState = 0;
    for(; i15 >= rootEnd; --i15){
        code13 = path25.charCodeAt(i15);
        if (isPathSeparator(code13)) {
            if (!matchedSlash) {
                startPart = i15 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i15 + 1;
        }
        if (code13 === 46) {
            if (startDot === -1) startDot = i15;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            ret.base = ret.name = path25.slice(startPart, end);
        }
    } else {
        ret.name = path25.slice(startPart, startDot);
        ret.base = path25.slice(startPart, end);
        ret.ext = path25.slice(startDot, end);
    }
    if (startPart > 0 && startPart !== rootEnd) {
        ret.dir = path25.slice(0, startPart - 1);
    } else ret.dir = ret.root;
    return ret;
}
function fromFileUrl(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    let path26 = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname != "") {
        path26 = `\\\\${url.hostname}${path26}`;
    }
    return path26;
}
function toFileUrl(path27) {
    if (!isAbsolute(path27)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path27.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname != null && hostname != "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
const mod1 = {
    sep: sep,
    delimiter: delimiter,
    resolve: resolve,
    normalize: normalize1,
    isAbsolute: isAbsolute,
    join: join,
    relative: relative,
    toNamespacedPath: toNamespacedPath,
    dirname: dirname,
    basename: basename,
    extname: extname,
    format: format,
    parse: parse,
    fromFileUrl: fromFileUrl,
    toFileUrl: toFileUrl
};
const sep1 = "/";
const delimiter1 = ":";
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i16 = pathSegments.length - 1; i16 >= -1 && !resolvedAbsolute; i16--){
        let path28;
        if (i16 >= 0) path28 = pathSegments[i16];
        else {
            const { Deno  } = globalThis;
            if (typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path28 = Deno.cwd();
        }
        assertPath(path28);
        if (path28.length === 0) {
            continue;
        }
        resolvedPath = `${path28}/${resolvedPath}`;
        resolvedAbsolute = path28.charCodeAt(0) === CHAR_FORWARD_SLASH;
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function normalize2(path29) {
    assertPath(path29);
    if (path29.length === 0) return ".";
    const isAbsolute1 = path29.charCodeAt(0) === 47;
    const trailingSeparator = path29.charCodeAt(path29.length - 1) === 47;
    path29 = normalizeString(path29, !isAbsolute1, "/", isPosixPathSeparator);
    if (path29.length === 0 && !isAbsolute1) path29 = ".";
    if (path29.length > 0 && trailingSeparator) path29 += "/";
    if (isAbsolute1) return `/${path29}`;
    return path29;
}
function isAbsolute1(path30) {
    assertPath(path30);
    return path30.length > 0 && path30.charCodeAt(0) === 47;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i17 = 0, len = paths.length; i17 < len; ++i17){
        const path31 = paths[i17];
        assertPath(path31);
        if (path31.length > 0) {
            if (!joined) joined = path31;
            else joined += `/${path31}`;
        }
    }
    if (!joined) return ".";
    return normalize2(joined);
}
function relative1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 47) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 47) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i18 = 0;
    for(; i18 <= length; ++i18){
        if (i18 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i18) === 47) {
                    return to.slice(toStart + i18 + 1);
                } else if (i18 === 0) {
                    return to.slice(toStart + i18);
                }
            } else if (fromLen > length) {
                if (from.charCodeAt(fromStart + i18) === 47) {
                    lastCommonSep = i18;
                } else if (i18 === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i18);
        const toCode = to.charCodeAt(toStart + i18);
        if (fromCode !== toCode) break;
        else if (fromCode === 47) lastCommonSep = i18;
    }
    let out = "";
    for(i18 = fromStart + lastCommonSep + 1; i18 <= fromEnd; ++i18){
        if (i18 === fromEnd || from.charCodeAt(i18) === 47) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (to.charCodeAt(toStart) === 47) ++toStart;
        return to.slice(toStart);
    }
}
function toNamespacedPath1(path32) {
    return path32;
}
function dirname1(path33) {
    assertPath(path33);
    if (path33.length === 0) return ".";
    const hasRoot = path33.charCodeAt(0) === 47;
    let end = -1;
    let matchedSlash = true;
    for(let i19 = path33.length - 1; i19 >= 1; --i19){
        if (path33.charCodeAt(i19) === 47) {
            if (!matchedSlash) {
                end = i19;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) return hasRoot ? "/" : ".";
    if (hasRoot && end === 1) return "//";
    return path33.slice(0, end);
}
function basename1(path34, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path34);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i20;
    if (ext !== undefined && ext.length > 0 && ext.length <= path34.length) {
        if (ext.length === path34.length && ext === path34) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i20 = path34.length - 1; i20 >= 0; --i20){
            const code14 = path34.charCodeAt(i20);
            if (code14 === 47) {
                if (!matchedSlash) {
                    start = i20 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i20 + 1;
                }
                if (extIdx >= 0) {
                    if (code14 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i20;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path34.length;
        return path34.slice(start, end);
    } else {
        for(i20 = path34.length - 1; i20 >= 0; --i20){
            if (path34.charCodeAt(i20) === 47) {
                if (!matchedSlash) {
                    start = i20 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i20 + 1;
            }
        }
        if (end === -1) return "";
        return path34.slice(start, end);
    }
}
function extname1(path35) {
    assertPath(path35);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i21 = path35.length - 1; i21 >= 0; --i21){
        const code15 = path35.charCodeAt(i21);
        if (code15 === 47) {
            if (!matchedSlash) {
                startPart = i21 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i21 + 1;
        }
        if (code15 === 46) {
            if (startDot === -1) startDot = i21;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path35.slice(startDot, end);
}
function format1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("/", pathObject);
}
function parse1(path36) {
    assertPath(path36);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path36.length === 0) return ret;
    const isAbsolute2 = path36.charCodeAt(0) === 47;
    let start;
    if (isAbsolute2) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i22 = path36.length - 1;
    let preDotState = 0;
    for(; i22 >= start; --i22){
        const code16 = path36.charCodeAt(i22);
        if (code16 === 47) {
            if (!matchedSlash) {
                startPart = i22 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i22 + 1;
        }
        if (code16 === 46) {
            if (startDot === -1) startDot = i22;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute2) {
                ret.base = ret.name = path36.slice(1, end);
            } else {
                ret.base = ret.name = path36.slice(startPart, end);
            }
        }
    } else {
        if (startPart === 0 && isAbsolute2) {
            ret.name = path36.slice(1, startDot);
            ret.base = path36.slice(1, end);
        } else {
            ret.name = path36.slice(startPart, startDot);
            ret.base = path36.slice(startPart, end);
        }
        ret.ext = path36.slice(startDot, end);
    }
    if (startPart > 0) ret.dir = path36.slice(0, startPart - 1);
    else if (isAbsolute2) ret.dir = "/";
    return ret;
}
function fromFileUrl1(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function toFileUrl1(path37) {
    if (!isAbsolute1(path37)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path37.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
const mod2 = {
    sep: sep1,
    delimiter: delimiter1,
    resolve: resolve1,
    normalize: normalize2,
    isAbsolute: isAbsolute1,
    join: join1,
    relative: relative1,
    toNamespacedPath: toNamespacedPath1,
    dirname: dirname1,
    basename: basename1,
    extname: extname1,
    format: format1,
    parse: parse1,
    fromFileUrl: fromFileUrl1,
    toFileUrl: toFileUrl1
};
const SEP = isWindows ? "\\" : "/";
const SEP_PATTERN = isWindows ? /[\\/]+/ : /\/+/;
const path = isWindows ? mod1 : mod2;
const { join: join2 , normalize: normalize3  } = path;
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|", 
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function globToRegExp(glob, { extended =true , globstar: globstarOption = true , os =osType , caseInsensitive =false  } = {}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep4 = os == "windows" ? "(?:\\\\|/)+" : "/+";
    const sepMaybe = os == "windows" ? "(?:\\\\|/)*" : "/*";
    const seps = os == "windows" ? [
        "\\",
        "/"
    ] : [
        "/"
    ];
    const globstar = os == "windows" ? "(?:[^\\\\/]*(?:\\\\|/|$)+)*" : "(?:[^/]*(?:/|$)+)*";
    const wildcard = os == "windows" ? "[^\\\\/]*" : "[^/]*";
    const escapePrefix = os == "windows" ? "`" : "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i23 = j;
        for(; i23 < glob.length && !seps.includes(glob[i23]); i23++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i23]) ? `\\${glob[i23]}` : glob[i23];
                continue;
            }
            if (glob[i23] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i23] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i23 + 1] == "!") {
                        i23++;
                        segment += "^";
                    } else if (glob[i23 + 1] == "^") {
                        i23++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i23 + 1] == ":") {
                    let k = i23 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i23 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i23] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i23] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i23];
                }
                continue;
            }
            if (glob[i23] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type == "!") {
                    segment += wildcard;
                } else if (type != "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i23] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i23] == "+" && extended && glob[i23 + 1] == "(") {
                i23++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i23] == "@" && extended && glob[i23 + 1] == "(") {
                i23++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i23] == "?") {
                if (extended && glob[i23 + 1] == "(") {
                    i23++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i23] == "!" && extended && glob[i23 + 1] == "(") {
                i23++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i23] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i23] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i23] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i23] == "*") {
                if (extended && glob[i23 + 1] == "(") {
                    i23++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i23 - 1];
                    let numStars = 1;
                    while(glob[i23 + 1] == "*"){
                        i23++;
                        numStars++;
                    }
                    const nextChar = glob[i23 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i23]) ? `\\${glob[i23]}` : glob[i23];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i23)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i23 < glob.length ? sep4 : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i23]))i23++;
        if (!(i23 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i23;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob(str54) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str54 === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str54)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str54.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str54 = str54.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar =false  } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize3(glob);
    }
    const s = SEP_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize3(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended =true , globstar =false  } = {}) {
    if (!globstar || globs.length == 0) {
        return join2(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path1 = glob;
        if (path1.length > 0) {
            if (!joined) joined = path1;
            else joined += `${SEP}${path1}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
const path1 = isWindows ? mod1 : mod2;
const posix = mod2;
const { basename: basename2 , delimiter: delimiter2 , dirname: dirname2 , extname: extname2 , format: format2 , fromFileUrl: fromFileUrl2 , isAbsolute: isAbsolute2 , join: join3 , normalize: normalize4 , parse: parse2 , relative: relative2 , resolve: resolve2 , sep: sep2 , toFileUrl: toFileUrl2 , toNamespacedPath: toNamespacedPath2 ,  } = path1;
function _createWalkEntrySync(path38) {
    path38 = normalize4(path38);
    const name71 = basename2(path38);
    const info1 = Deno.statSync(path38);
    return {
        path: path38,
        name: name71,
        isFile: info1.isFile,
        isDirectory: info1.isDirectory,
        isSymlink: info1.isSymlink
    };
}
function include(path39, exts, match, skip) {
    if (exts && !exts.some((ext)=>path39.endsWith(ext)
    )) {
        return false;
    }
    if (match && !match.some((pattern)=>!!path39.match(pattern)
    )) {
        return false;
    }
    if (skip && skip.some((pattern)=>!!path39.match(pattern)
    )) {
        return false;
    }
    return true;
}
function wrapErrorWithRootPath(err, root1) {
    if (err instanceof Error && "root" in err) return err;
    const e = new Error();
    e.root = root1;
    e.message = err instanceof Error ? `${err.message} for path "${root1}"` : `[non-error thrown] for path "${root1}"`;
    e.stack = err instanceof Error ? err.stack : undefined;
    e.cause = err instanceof Error ? err.cause : undefined;
    return e;
}
function* walkSync(root2, { maxDepth =Infinity , includeFiles =true , includeDirs =true , followSymlinks =false , exts =undefined , match =undefined , skip =undefined  } = {}) {
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include(root2, exts, match, skip)) {
        yield _createWalkEntrySync(root2);
    }
    if (maxDepth < 1 || !include(root2, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root2);
    } catch (err) {
        throw wrapErrorWithRootPath(err, normalize4(root2));
    }
    for (const entry of entries){
        assert(entry.name != null);
        let path40 = join3(root2, entry.name);
        let { isSymlink , isDirectory  } = entry;
        if (isSymlink) {
            if (!followSymlinks) continue;
            path40 = Deno.realPathSync(path40);
            ({ isSymlink , isDirectory  } = Deno.lstatSync(path40));
        }
        if (isSymlink || isDirectory) {
            yield* walkSync(path40, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                followSymlinks,
                exts,
                match,
                skip
            });
        } else if (includeFiles && include(path40, exts, match, skip)) {
            yield {
                path: path40,
                ...entry
            };
        }
    }
}
function split(path41) {
    const s = SEP_PATTERN.source;
    const segments = path41.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEP_PATTERN);
    const isAbsolute_ = isAbsolute2(path41);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path41.match(new RegExp(`${s}$`)),
        winRoot: isWindows && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error5) {
    if (!(error5 instanceof Deno.errors.NotFound)) {
        throw error5;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root: root3 = Deno.cwd() , exclude =[] , includeDirs =true , extended =true , globstar =false , caseInsensitive  } = {}) {
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = resolve2(root3);
    const resolveFromRoot = (path42)=>resolve2(absRoot, path42)
    ;
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp(s, globOptions)
    );
    const shouldInclude1 = (path43)=>!excludePatterns.some((p)=>!!path43.match(p)
        )
    ;
    const { segments , isAbsolute: isGlobAbsolute , hasTrailingSep , winRoot  } = split(glob);
    let fixedRoot = isGlobAbsolute ? winRoot != undefined ? winRoot : "/" : absRoot;
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg != null);
        fixedRoot = joinGlobs([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = _createWalkEntrySync(fixedRoot);
    } catch (error2) {
        return throwUnlessNotFound(error2);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment == "..") {
            const parentPath = joinGlobs([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude1(parentPath)) {
                    return yield _createWalkEntrySync(parentPath);
                }
            } catch (error6) {
                throwUnlessNotFound(error6);
            }
            return;
        } else if (globSegment == "**") {
            return yield* walkSync(walkInfo.path, {
                skip: excludePatterns
            });
        }
        const globPattern = globToRegExp(globSegment, globOptions);
        for (const walkEntry of walkSync(walkInfo.path, {
            maxDepth: 1,
            skip: excludePatterns
        })){
            if (walkEntry.path != walkInfo.path && walkEntry.name.match(globPattern)) {
                yield walkEntry;
            }
        }
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory
        );
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory
        );
    }
    yield* currentMatches;
}
var LogLevels;
(function(LogLevels1) {
    LogLevels1[LogLevels1["NOTSET"] = 0] = "NOTSET";
    LogLevels1[LogLevels1["DEBUG"] = 10] = "DEBUG";
    LogLevels1[LogLevels1["INFO"] = 20] = "INFO";
    LogLevels1[LogLevels1["WARNING"] = 30] = "WARNING";
    LogLevels1[LogLevels1["ERROR"] = 40] = "ERROR";
    LogLevels1[LogLevels1["CRITICAL"] = 50] = "CRITICAL";
})(LogLevels || (LogLevels = {}));
Object.keys(LogLevels).filter((key)=>isNaN(Number(key))
);
const byLevel = {
    [String(LogLevels.NOTSET)]: "NOTSET",
    [String(LogLevels.DEBUG)]: "DEBUG",
    [String(LogLevels.INFO)]: "INFO",
    [String(LogLevels.WARNING)]: "WARNING",
    [String(LogLevels.ERROR)]: "ERROR",
    [String(LogLevels.CRITICAL)]: "CRITICAL"
};
function getLevelByName(name72) {
    switch(name72){
        case "NOTSET":
            return LogLevels.NOTSET;
        case "DEBUG":
            return LogLevels.DEBUG;
        case "INFO":
            return LogLevels.INFO;
        case "WARNING":
            return LogLevels.WARNING;
        case "ERROR":
            return LogLevels.ERROR;
        case "CRITICAL":
            return LogLevels.CRITICAL;
        default:
            throw new Error(`no log level found for "${name72}"`);
    }
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`no level name found for level: ${level}`);
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    _log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data1) {
        if (typeof data1 === "string") {
            return data1;
        } else if (data1 === null || typeof data1 === "number" || typeof data1 === "bigint" || typeof data1 === "boolean" || typeof data1 === "undefined" || typeof data1 === "symbol") {
            return String(data1);
        } else if (data1 instanceof Error) {
            return data1.stack;
        } else if (typeof data1 === "object") {
            return JSON.stringify(data1);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this._log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this._log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this._log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this._log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this._log(LogLevels.CRITICAL, msg, ...args);
    }
}
async function exists(filePath) {
    try {
        await Deno.lstat(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function existsSync(filePath) {
    try {
        Deno.lstatSync(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function concat(...buf) {
    let length = 0;
    for (const b of buf){
        length += b.length;
    }
    const output = new Uint8Array(length);
    let index = 0;
    for (const b1 of buf){
        output.set(b1, index);
        index += b1.length;
    }
    return output;
}
function copy(src, dst, off = 0) {
    off = Math.max(0, Math.min(off, dst.byteLength));
    const dstBytesAvailable = dst.byteLength - off;
    if (src.byteLength > dstBytesAvailable) {
        src = src.subarray(0, dstBytesAvailable);
    }
    dst.set(src, off);
    return src.byteLength;
}
const MIN_READ = 32 * 1024;
const MAX_SIZE = 2 ** 32 - 2;
class Buffer {
    #buf;
    #off = 0;
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
     #tryGrowByReslice(n) {
        const l = this.#buf.byteLength;
        if (n <= this.capacity - l) {
            this.#reslice(l + n);
            return l;
        }
        return -1;
    }
     #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    readSync(p) {
        if (this.empty()) {
            this.reset();
            if (p.byteLength === 0) {
                return 0;
            }
            return null;
        }
        const nread = copy(this.#buf.subarray(this.#off), p);
        this.#off += nread;
        return nread;
    }
    read(p) {
        const rr = this.readSync(p);
        return Promise.resolve(rr);
    }
    writeSync(p) {
        const m = this.#grow(p.byteLength);
        return copy(p, this.#buf, m);
    }
    write(p) {
        const n1 = this.writeSync(p);
        return Promise.resolve(n1);
    }
     #grow(n2) {
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i = this.#tryGrowByReslice(n2);
        if (i >= 0) {
            return i;
        }
        const c = this.capacity;
        if (n2 <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n2 > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n2, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n2, MAX_SIZE));
        return m;
    }
    grow(n3) {
        if (n3 < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n3);
        this.#reslice(m);
    }
    async readFrom(r) {
        let n4 = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = await r.read(buf);
            if (nread === null) {
                return n4;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n4 += nread;
        }
    }
    readFromSync(r) {
        let n5 = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = r.readSync(buf);
            if (nread === null) {
                return n5;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n5 += nread;
        }
    }
}
const DEFAULT_BUF_SIZE = 4096;
const MIN_BUF_SIZE = 16;
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
class BufferFullError extends Error {
    partial;
    name = "BufferFullError";
    constructor(partial){
        super("Buffer full");
        this.partial = partial;
    }
}
class PartialReadError extends Error {
    name = "PartialReadError";
    partial;
    constructor(){
        super("Encountered UnexpectedEof, data only partially read");
    }
}
class BufReader {
    #buf;
    #rd;
    #r = 0;
    #w = 0;
    #eof = false;
    static create(r, size = 4096) {
        return r instanceof BufReader ? r : new BufReader(r, size);
    }
    constructor(rd, size = 4096){
        if (size < 16) {
            size = MIN_BUF_SIZE;
        }
        this.#reset(new Uint8Array(size), rd);
    }
    size() {
        return this.#buf.byteLength;
    }
    buffered() {
        return this.#w - this.#r;
    }
    #fill = async ()=>{
        if (this.#r > 0) {
            this.#buf.copyWithin(0, this.#r, this.#w);
            this.#w -= this.#r;
            this.#r = 0;
        }
        if (this.#w >= this.#buf.byteLength) {
            throw Error("bufio: tried to fill full buffer");
        }
        for(let i24 = 100; i24 > 0; i24--){
            const rr = await this.#rd.read(this.#buf.subarray(this.#w));
            if (rr === null) {
                this.#eof = true;
                return;
            }
            assert(rr >= 0, "negative read");
            this.#w += rr;
            if (rr > 0) {
                return;
            }
        }
        throw new Error(`No progress after ${100} read() calls`);
    };
    reset(r) {
        this.#reset(this.#buf, r);
    }
    #reset = (buf, rd)=>{
        this.#buf = buf;
        this.#rd = rd;
        this.#eof = false;
    };
    async read(p) {
        let rr = p.byteLength;
        if (p.byteLength === 0) return rr;
        if (this.#r === this.#w) {
            if (p.byteLength >= this.#buf.byteLength) {
                const rr = await this.#rd.read(p);
                const nread = rr ?? 0;
                assert(nread >= 0, "negative read");
                return rr;
            }
            this.#r = 0;
            this.#w = 0;
            rr = await this.#rd.read(this.#buf);
            if (rr === 0 || rr === null) return rr;
            assert(rr >= 0, "negative read");
            this.#w += rr;
        }
        const copied = copy(this.#buf.subarray(this.#r, this.#w), p, 0);
        this.#r += copied;
        return copied;
    }
    async readFull(p) {
        let bytesRead = 0;
        while(bytesRead < p.length){
            try {
                const rr = await this.read(p.subarray(bytesRead));
                if (rr === null) {
                    if (bytesRead === 0) {
                        return null;
                    } else {
                        throw new PartialReadError();
                    }
                }
                bytesRead += rr;
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = p.subarray(0, bytesRead);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = p.subarray(0, bytesRead);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return p;
    }
    async readByte() {
        while(this.#r === this.#w){
            if (this.#eof) return null;
            await this.#fill();
        }
        const c = this.#buf[this.#r];
        this.#r++;
        return c;
    }
    async readString(delim) {
        if (delim.length !== 1) {
            throw new Error("Delimiter should be a single character");
        }
        const buffer = await this.readSlice(delim.charCodeAt(0));
        if (buffer === null) return null;
        return new TextDecoder().decode(buffer);
    }
    async readLine() {
        let line = null;
        try {
            line = await this.readSlice(LF);
        } catch (err) {
            if (err instanceof Deno.errors.BadResource) {
                throw err;
            }
            let partial;
            if (err instanceof PartialReadError) {
                partial = err.partial;
                assert(partial instanceof Uint8Array, "bufio: caught error from `readSlice()` without `partial` property");
            }
            if (!(err instanceof BufferFullError)) {
                throw err;
            }
            partial = err.partial;
            if (!this.#eof && partial && partial.byteLength > 0 && partial[partial.byteLength - 1] === CR) {
                assert(this.#r > 0, "bufio: tried to rewind past start of buffer");
                this.#r--;
                partial = partial.subarray(0, partial.byteLength - 1);
            }
            if (partial) {
                return {
                    line: partial,
                    more: !this.#eof
                };
            }
        }
        if (line === null) {
            return null;
        }
        if (line.byteLength === 0) {
            return {
                line,
                more: false
            };
        }
        if (line[line.byteLength - 1] == LF) {
            let drop = 1;
            if (line.byteLength > 1 && line[line.byteLength - 2] === CR) {
                drop = 2;
            }
            line = line.subarray(0, line.byteLength - drop);
        }
        return {
            line,
            more: false
        };
    }
    async readSlice(delim) {
        let s = 0;
        let slice1;
        while(true){
            let i25 = this.#buf.subarray(this.#r + s, this.#w).indexOf(delim);
            if (i25 >= 0) {
                i25 += s;
                slice1 = this.#buf.subarray(this.#r, this.#r + i25 + 1);
                this.#r += i25 + 1;
                break;
            }
            if (this.#eof) {
                if (this.#r === this.#w) {
                    return null;
                }
                slice1 = this.#buf.subarray(this.#r, this.#w);
                this.#r = this.#w;
                break;
            }
            if (this.buffered() >= this.#buf.byteLength) {
                this.#r = this.#w;
                const oldbuf = this.#buf;
                const newbuf = this.#buf.slice(0);
                this.#buf = newbuf;
                throw new BufferFullError(oldbuf);
            }
            s = this.#w - this.#r;
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = slice1;
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = slice1;
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return slice1;
    }
    async peek(n6) {
        if (n6 < 0) {
            throw Error("negative count");
        }
        let avail = this.#w - this.#r;
        while(avail < n6 && avail < this.#buf.byteLength && !this.#eof){
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = this.#buf.subarray(this.#r, this.#w);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = this.#buf.subarray(this.#r, this.#w);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
            avail = this.#w - this.#r;
        }
        if (avail === 0 && this.#eof) {
            return null;
        } else if (avail < n6 && this.#eof) {
            return this.#buf.subarray(this.#r, this.#r + avail);
        } else if (avail < n6) {
            throw new BufferFullError(this.#buf.subarray(this.#r, this.#w));
        }
        return this.#buf.subarray(this.#r, this.#r + n6);
    }
}
class AbstractBufBase {
    buf;
    usedBufferBytes = 0;
    err = null;
    constructor(buf){
        this.buf = buf;
    }
    size() {
        return this.buf.byteLength;
    }
    available() {
        return this.buf.byteLength - this.usedBufferBytes;
    }
    buffered() {
        return this.usedBufferBytes;
    }
}
class BufWriter extends AbstractBufBase {
    #writer;
    static create(writer, size = 4096) {
        return writer instanceof BufWriter ? writer : new BufWriter(writer, size);
    }
    constructor(writer, size = 4096){
        if (size <= 0) {
            size = DEFAULT_BUF_SIZE;
        }
        const buf = new Uint8Array(size);
        super(buf);
        this.#writer = writer;
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.#writer = w;
    }
    async flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            const p = this.buf.subarray(0, this.usedBufferBytes);
            let nwritten = 0;
            while(nwritten < p.length){
                nwritten += await this.#writer.write(p.subarray(nwritten));
            }
        } catch (e) {
            if (e instanceof Error) {
                this.err = e;
            }
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    async write(data2) {
        if (this.err !== null) throw this.err;
        if (data2.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data2.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = await this.#writer.write(data2);
                } catch (e) {
                    if (e instanceof Error) {
                        this.err = e;
                    }
                    throw e;
                }
            } else {
                numBytesWritten = copy(data2, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                await this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data2 = data2.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data2, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
class BufWriterSync extends AbstractBufBase {
    #writer;
    static create(writer, size = 4096) {
        return writer instanceof BufWriterSync ? writer : new BufWriterSync(writer, size);
    }
    constructor(writer, size = 4096){
        if (size <= 0) {
            size = DEFAULT_BUF_SIZE;
        }
        const buf = new Uint8Array(size);
        super(buf);
        this.#writer = writer;
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.#writer = w;
    }
    flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            const p = this.buf.subarray(0, this.usedBufferBytes);
            let nwritten = 0;
            while(nwritten < p.length){
                nwritten += this.#writer.writeSync(p.subarray(nwritten));
            }
        } catch (e) {
            if (e instanceof Error) {
                this.err = e;
            }
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    writeSync(data3) {
        if (this.err !== null) throw this.err;
        if (data3.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data3.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = this.#writer.writeSync(data3);
                } catch (e) {
                    if (e instanceof Error) {
                        this.err = e;
                    }
                    throw e;
                }
            } else {
                numBytesWritten = copy(data3, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data3 = data3.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data3, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
async function* readLines(reader, decoderOpts) {
    const bufReader = new BufReader(reader);
    let chunks = [];
    const decoder = new TextDecoder(decoderOpts?.encoding, decoderOpts);
    while(true){
        const res = await bufReader.readLine();
        if (!res) {
            if (chunks.length > 0) {
                yield decoder.decode(concat(...chunks));
            }
            break;
        }
        chunks.push(res.line);
        if (!res.more) {
            yield decoder.decode(concat(...chunks));
            chunks = [];
        }
    }
}
const DEFAULT_FORMATTER = "{levelName} {msg}";
class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {}){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{(\S+)}/g, (match, p1)=>{
            const value = logRecord[p1];
            if (value == null) {
                return match;
            }
            return String(value);
        });
    }
    log(_msg) {}
    async setup() {}
    async destroy() {}
}
class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow(msg);
                break;
            case LogLevels.ERROR:
                msg = red(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold(red(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    async setup() {
        this._file = await Deno.open(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        if (this._encoder.encode(msg).byteLength + 1 > this._buf.available()) {
            this.flush();
        }
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
        return Promise.resolve();
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger(name73) {
    if (!name73) {
        const d = state.loggers.get("default");
        assert(d != null, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result = state.loggers.get(name73);
    if (!result) {
        const logger = new Logger(name73, "NOTSET", {
            handlers: []
        });
        state.loggers.set(name73, logger);
        return logger;
    }
    return result;
}
function debug(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
function info(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
function warning(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").warning(msg, ...args);
    }
    return getLogger("default").warning(msg, ...args);
}
function error(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
async function setup(config) {
    state.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state.handlers.clear();
    const handlers1 = state.config.handlers || {};
    for(const handlerName1 in handlers1){
        const handler = handlers1[handlerName1];
        await handler.setup();
        state.handlers.set(handlerName1, handler);
    }
    state.loggers.clear();
    const loggers = state.config.loggers || {};
    for(const loggerName in loggers){
        const loggerConfig = loggers[loggerName];
        const handlerNames = loggerConfig.handlers || [];
        const handlers2 = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state.handlers.get(handlerName);
            if (handler) {
                handlers2.push(handler);
            }
        });
        const levelName = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers2
        });
        state.loggers.set(loggerName, logger);
    }
}
await setup(DEFAULT_CONFIG);
function isSubdir(src, dest, sep5 = sep2) {
    if (src === dest) {
        return false;
    }
    const srcArray = src.split(sep5);
    const destArray = dest.split(sep5);
    return srcArray.every((current, i26)=>destArray[i26] === current
    );
}
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            await Deno.mkdir(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            Deno.mkdirSync(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function moveSync(src, dest, { overwrite =false  } = {}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (overwrite) {
        if (existsSync(dest)) {
            Deno.removeSync(dest, {
                recursive: true
            });
        }
    } else {
        if (existsSync(dest)) {
            throw new Error("dest already exists.");
        }
    }
    Deno.renameSync(src, dest);
}
var EOL;
(function(EOL1) {
    EOL1["LF"] = "\n";
    EOL1["CRLF"] = "\r\n";
})(EOL || (EOL = {}));
function getenv(name74, defaultValue) {
    const value = Deno.env.get(name74);
    if (value === undefined) {
        if (defaultValue === undefined) {
            throw new Error(`Required environment variable ${name74} not specified.`);
        } else {
            return defaultValue;
        }
    } else {
        return value;
    }
}
var State;
(function(State2) {
    State2[State2["PASSTHROUGH"] = 0] = "PASSTHROUGH";
    State2[State2["PERCENT"] = 1] = "PERCENT";
    State2[State2["POSITIONAL"] = 2] = "POSITIONAL";
    State2[State2["PRECISION"] = 3] = "PRECISION";
    State2[State2["WIDTH"] = 4] = "WIDTH";
})(State || (State = {}));
var WorP;
(function(WorP1) {
    WorP1[WorP1["WIDTH"] = 0] = "WIDTH";
    WorP1[WorP1["PRECISION"] = 1] = "PRECISION";
})(WorP || (WorP = {}));
var F;
(function(F1) {
    F1[F1["sign"] = 1] = "sign";
    F1[F1["mantissa"] = 2] = "mantissa";
    F1[F1["fractional"] = 3] = "fractional";
    F1[F1["esign"] = 4] = "esign";
    F1[F1["exponent"] = 5] = "exponent";
})(F || (F = {}));
function glb(array, value, compare) {
    compare = compare || ((a, b)=>a - b
    );
    if (array.length === 0) {
        return -1;
    }
    if (array.length === 1) {
        if (compare(value, array[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array.length - 1;
    const vLeft = array[left], vRight = array[right];
    if (compare(value, vRight) >= 0) {
        return right;
    }
    if (compare(value, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array[center];
        const cmp = compare(value, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
let Deno1;
try {
    Deno1 = globalThis.Deno;
} catch (_e) {}
const noColor1 = typeof (Deno1 && Deno1.noColor) === "boolean" ? Deno1.noColor : true;
let enabled1 = !noColor1;
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str55, code17) {
    return enabled1 ? `${code17.open}${str55.replace(code17.regexp, code17.open)}${code17.close}` : str55;
}
function red1(str56) {
    return run1(str56, code1([
        31
    ], 39));
}
function blue1(str57) {
    return run1(str57, code1([
        34
    ], 39));
}
function clampAndTruncate1(n1, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n1, max), min));
}
function rgb241(str58, color) {
    if (typeof color === "number") {
        return run1(str58, code1([
            38,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 39));
    }
    return run1(str58, code1([
        38,
        2,
        clampAndTruncate1(color.r),
        clampAndTruncate1(color.g),
        clampAndTruncate1(color.b), 
    ], 39));
}
new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function tidyverseInfo(msg) {
    return `${blue1("ℹ")} ${msg}`;
}
function tidyverseError(msg) {
    return `${red1("✖")} ${msg}`;
}
function tidyverseFormatError(msg) {
    let { heading , error: error7 , info: info2  } = msg;
    if (msg.location) {
        heading = `${locationString(msg.location)} ${heading}`;
    }
    if (msg.fileName) {
        heading = `In file ${msg.fileName}\n${heading}`;
    }
    const strings = [
        heading,
        msg.sourceContext,
        ...error7.map(tidyverseError),
        ...info2.map(tidyverseInfo), 
    ];
    return strings.join("\n");
}
function quotedStringColor(msg) {
    return rgb241(msg, 12369186);
}
function addFileInfo(msg, src) {
    if (src.fileName !== undefined) {
        msg.fileName = src.fileName;
    }
}
function addInstancePathInfo(msg, instancePath) {
    if (instancePath !== "") {
        const niceInstancePath = instancePath.trim().slice(1).split("/").map((s)=>blue1(s)
        ).join(":");
        msg.info.push(`The error happened in location ${niceInstancePath}.`);
    }
}
function locationString(loc) {
    const { start , end  } = loc;
    if (start.line === end.line) {
        if (start.column === end.column) {
            return `(line ${start.line + 1}, column ${start.column + 1})`;
        } else {
            return `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})`;
        }
    } else {
        return `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
    }
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function* matchAll(text, regexp1) {
    if (!regexp1.global) {
        throw new Error("matchAll requires global regexps");
    }
    let match;
    while((match = regexp1.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function* lineBreakPositions(text) {
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index;
    }
}
function indexToRowCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad = " ".repeat(lineWidth);
    const ls = lines(text);
    const result = [];
    for(let i27 = firstLine; i27 <= lastLine; ++i27){
        const numberStr = `${pad}${i27 + 1}: `.slice(-(lineWidth + 2));
        const lineStr = ls[i27];
        result.push({
            lineNumber: i27,
            content: numberStr + quotedStringColor(lineStr),
            rawLine: ls[i27]
        });
    }
    return {
        prefixWidth: lineWidth + 2,
        lines: result
    };
}
function appendLogOptions(cmd) {
    return cmd.option("--log <level>", "Path to log file", {
        global: true
    }).option("--log-level <level>", "Log level (info, warning, error, critical)", {
        global: true
    }).option("--log-format <format>", "Log format (plain, json-stream)", {
        global: true
    }).option("--quiet", "Suppress console output.", {
        global: true
    });
}
function logOptions(args) {
    const logOptions1 = {};
    logOptions1.log = args.l || args.log;
    logOptions1.level = args.ll || args["log-level"];
    logOptions1.quiet = args.q || args.quiet;
    logOptions1.format = parseFormat(args.lf || args["log-format"]);
    return logOptions1;
}
class StdErrOutputHandler extends BaseHandler {
    format(logRecord, prefix = true) {
        const options = {
            newline: true,
            colorize: true,
            ...logRecord.args[0]
        };
        let msg = super.format(logRecord);
        if (prefix && logRecord.level >= LogLevels.WARNING) {
            msg = `${logRecord.levelName}: ${msg}`;
        }
        switch(logRecord.level){
            case LogLevels.INFO:
            case LogLevels.DEBUG:
                msg = applyMsgOptions(msg, options);
                break;
            case LogLevels.WARNING:
                if (options.colorize) {
                    msg = yellow(msg);
                }
                break;
            case LogLevels.ERROR:
                if (options.colorize) {
                    msg = brightRed(msg);
                }
                break;
            case LogLevels.CRITICAL:
                if (options.colorize) {
                    msg = bold(red(msg));
                }
                break;
            default:
                break;
        }
        if (options.newline) {
            msg = msg + "\n";
        }
        return msg;
    }
    log(msg) {
        Deno.stderr.writeSync(new TextEncoder().encode(msg));
    }
}
class LogEventsHandler extends StdErrOutputHandler {
    constructor(levelName){
        super(levelName, {
            formatter: "{msg}"
        });
    }
    handle(logRecord) {
        LogEventsHandler.handlers_.forEach((handler)=>handler(logRecord, super.format(logRecord, false))
        );
    }
    static onLog(handler) {
        LogEventsHandler.handlers_.push(handler);
    }
    static handlers_ = new Array();
}
class LogFileHandler extends FileHandler {
    constructor(levelName, options){
        super(levelName, options);
        this.msgFormat = options.format;
    }
    msgFormat;
    format(logRecord) {
        if (logRecord.msg.startsWith("\r")) {
            return "";
        }
        if (this.msgFormat === undefined || this.msgFormat === "plain") {
            const options = {
                newline: true,
                ...logRecord.args[0],
                bold: false,
                dim: false,
                format: undefined
            };
            let msg = applyMsgOptions(logRecord.msg, options);
            if (options.newline) {
                msg = msg + "\n";
            }
            if (logRecord.level >= LogLevels.WARNING) {
                return `(${logRecord.levelName}) ${msg}`;
            } else {
                return msg;
            }
        } else {
            return JSON.stringify(logRecord, undefined, 0) + "\n";
        }
    }
    log(msg) {
        if (msg !== "") {
            msg = stripColor(msg);
            this._buf.writeSync(this._encoder.encode(msg));
            this._buf.flush();
        }
    }
}
async function initializeLogger(logOptions2) {
    const handlers = {};
    const defaultHandlers = [];
    const file = logOptions2.log;
    const logLevel = logOptions2.level ? parseLevel(logOptions2.level) : "INFO";
    handlers["events"] = new LogEventsHandler(logLevel);
    defaultHandlers.push("events");
    if (!logOptions2.quiet) {
        handlers["console"] = new StdErrOutputHandler(logLevel, {
            formatter: "{msg}"
        });
        defaultHandlers.push("console");
    }
    if (file) {
        handlers["file"] = new LogFileHandler(logLevel, {
            filename: file,
            mode: "w",
            format: logOptions2.format
        });
        defaultHandlers.push("file");
    }
    await setup({
        handlers,
        loggers: {
            default: {
                level: "DEBUG",
                handlers: defaultHandlers
            }
        }
    });
}
async function cleanupLogger() {
    const logger = getLogger();
    for (const handler of logger.handlers){
        await handler.destroy();
    }
    logger.handlers = [];
}
function logError(e) {
    let message = e.message ? `${e.name}: ${e.message}` : "";
    const isDebug = getenv("QUARTO_DEBUG", "false") === "true";
    if (isDebug && e.stack) {
        message = message + "\n\n" + e.stack;
    }
    if (message) {
        error(message);
    }
}
function errorOnce(msg) {
    if (!errors[msg]) {
        errors[msg] = true;
        error(msg);
        return true;
    }
    return false;
}
const errors = {};
function warnOnce(msg) {
    if (!warnings[msg]) {
        warnings[msg] = true;
        warning(msg);
    }
}
const warnings = {};
function applyMsgOptions(msg1, options) {
    if (options.indent) {
        const pad = " ".repeat(options.indent);
        msg1 = lines(msg1).map((msg)=>pad + msg
        ).join("\n");
    }
    if (options.bold) {
        msg1 = bold(msg1);
    }
    if (options.dim) {
        msg1 = dim(msg1);
    }
    if (options.format) {
        msg1 = options.format(msg1);
    }
    return msg1;
}
function parseFormat(format4) {
    if (format4) {
        format4 = format4.toLowerCase();
        switch(format4){
            case "plain":
            case "json-stream":
                return format4;
            default:
                return "plain";
        }
    } else {
        return "plain";
    }
}
function parseLevel(level) {
    const lvl = levelMap[level.toLowerCase()];
    if (lvl) {
        return lvl;
    } else {
        return "WARNING";
    }
}
const levelMap = {
    debug: "DEBUG",
    info: "INFO",
    warning: "WARNING",
    error: "ERROR",
    critical: "CRITICAL"
};
async function ensureValidCopy(src, dest, options) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
function ensureValidCopySync(src, dest, options) {
    let destStat;
    try {
        destStat = Deno.lstatSync(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    await Deno.copyFile(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copyFileSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    Deno.copyFileSync(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = Deno.statSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    const originSrcFilePath = await Deno.readLink(src);
    const type = getFileInfoType(await Deno.lstat(src));
    if (isWindows) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copySymlinkSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    const originSrcFilePath = Deno.readLinkSync(src);
    const type = getFileInfoType(Deno.lstatSync(src));
    if (isWindows) {
        Deno.symlinkSync(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        Deno.symlinkSync(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = Deno.lstatSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for await (const entry of Deno.readDir(src)){
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            await copySymLink(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            await copyDir(srcPath, destPath, options);
        } else if (entry.isFile) {
            await copyFile(srcPath, destPath, options);
        }
    }
}
function copyDirSync(src, dest, options) {
    const destStat = ensureValidCopySync(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        ensureDirSync(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = Deno.statSync(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for (const entry of Deno.readDirSync(src)){
        assert(entry.name != null, "file.name must be set");
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            copySymlinkSync(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            copyDirSync(srcPath, destPath, options);
        } else if (entry.isFile) {
            copyFileSync(srcPath, destPath, options);
        }
    }
}
async function copy1(src, dest, options = {}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options);
    }
}
function copySync(src, dest, options = {}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = Deno.lstatSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        copySymlinkSync(src, dest, options);
    } else if (srcStat.isDirectory) {
        copyDirSync(src, dest, options);
    } else if (srcStat.isFile) {
        copyFileSync(src, dest, options);
    }
}
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
var __VIRTUAL_FILE = listCacheClear;
function eq(value, other) {
    return value === other || value !== value && other !== other;
}
var __VIRTUAL_FILE1 = eq;
function assocIndexOf(array, key) {
    var length = array.length;
    while(length--){
        if (__VIRTUAL_FILE1(array[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var __VIRTUAL_FILE2 = assocIndexOf;
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data4 = this.__data__, index = __VIRTUAL_FILE2(data4, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data4.length - 1;
    if (index == lastIndex) {
        data4.pop();
    } else {
        splice.call(data4, index, 1);
    }
    --this.size;
    return true;
}
var __VIRTUAL_FILE3 = listCacheDelete;
function listCacheGet(key) {
    var data5 = this.__data__, index = __VIRTUAL_FILE2(data5, key);
    return index < 0 ? void 0 : data5[index][1];
}
var __VIRTUAL_FILE4 = listCacheGet;
function listCacheHas(key) {
    return __VIRTUAL_FILE2(this.__data__, key) > -1;
}
var __VIRTUAL_FILE5 = listCacheHas;
function listCacheSet(key, value) {
    var data6 = this.__data__, index = __VIRTUAL_FILE2(data6, key);
    if (index < 0) {
        ++this.size;
        data6.push([
            key,
            value
        ]);
    } else {
        data6[index][1] = value;
    }
    return this;
}
var __VIRTUAL_FILE6 = listCacheSet;
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = __VIRTUAL_FILE;
ListCache.prototype["delete"] = __VIRTUAL_FILE3;
ListCache.prototype.get = __VIRTUAL_FILE4;
ListCache.prototype.has = __VIRTUAL_FILE5;
ListCache.prototype.set = __VIRTUAL_FILE6;
var __VIRTUAL_FILE7 = ListCache;
function stackClear() {
    this.__data__ = new __VIRTUAL_FILE7();
    this.size = 0;
}
var __VIRTUAL_FILE8 = stackClear;
function stackDelete(key) {
    var data7 = this.__data__, result = data7["delete"](key);
    this.size = data7.size;
    return result;
}
var __VIRTUAL_FILE9 = stackDelete;
function stackGet(key) {
    return this.__data__.get(key);
}
var __VIRTUAL_FILE10 = stackGet;
function stackHas(key) {
    return this.__data__.has(key);
}
var __VIRTUAL_FILE11 = stackHas;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
var freeGlobal = typeof commonjsGlobal == "object" && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;
var __VIRTUAL_FILE12 = freeGlobal;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = __VIRTUAL_FILE12 || freeSelf || Function("return this")();
var __VIRTUAL_FILE13 = root;
var Symbol1 = __VIRTUAL_FILE13.Symbol;
var __VIRTUAL_FILE14 = Symbol1;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag), tag = value[symToStringTag];
    try {
        value[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {}
    var result = nativeObjectToString.call(value);
    if (unmasked) {
        if (isOwn) {
            value[symToStringTag] = tag;
        } else {
            delete value[symToStringTag];
        }
    }
    return result;
}
var __VIRTUAL_FILE15 = getRawTag;
var objectProto1 = Object.prototype;
var nativeObjectToString1 = objectProto1.toString;
function objectToString(value) {
    return nativeObjectToString1.call(value);
}
var __VIRTUAL_FILE16 = objectToString;
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function baseGetTag(value) {
    if (value == null) {
        return value === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag1 && symToStringTag1 in Object(value) ? __VIRTUAL_FILE15(value) : __VIRTUAL_FILE16(value);
}
var __VIRTUAL_FILE17 = baseGetTag;
function isObject(value) {
    var type = typeof value;
    return value != null && (type == "object" || type == "function");
}
var __VIRTUAL_FILE18 = isObject;
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value) {
    if (!__VIRTUAL_FILE18(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
}
var __VIRTUAL_FILE19 = isFunction;
var coreJsData = __VIRTUAL_FILE13["__core-js_shared__"];
var __VIRTUAL_FILE20 = coreJsData;
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(__VIRTUAL_FILE20 && __VIRTUAL_FILE20.keys && __VIRTUAL_FILE20.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func) {
    return !!maskSrcKey && maskSrcKey in func;
}
var __VIRTUAL_FILE21 = isMasked;
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func) {
    if (func != null) {
        try {
            return funcToString.call(func);
        } catch (e) {}
        try {
            return func + "";
        } catch (e1) {}
    }
    return "";
}
var __VIRTUAL_FILE22 = toSource;
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto1 = Function.prototype, objectProto2 = Object.prototype;
var funcToString1 = funcProto1.toString;
var hasOwnProperty1 = objectProto2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString1.call(hasOwnProperty1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value) {
    if (!__VIRTUAL_FILE18(value) || __VIRTUAL_FILE21(value)) {
        return false;
    }
    var pattern = __VIRTUAL_FILE19(value) ? reIsNative : reIsHostCtor;
    return pattern.test(__VIRTUAL_FILE22(value));
}
var __VIRTUAL_FILE23 = baseIsNative;
function getValue(object, key) {
    return object == null ? void 0 : object[key];
}
var __VIRTUAL_FILE24 = getValue;
function getNative(object, key) {
    var value = __VIRTUAL_FILE24(object, key);
    return __VIRTUAL_FILE23(value) ? value : void 0;
}
var __VIRTUAL_FILE25 = getNative;
var Map1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Map");
var __VIRTUAL_FILE26 = Map1;
var nativeCreate = __VIRTUAL_FILE25(Object, "create");
var __VIRTUAL_FILE27 = nativeCreate;
function hashClear() {
    this.__data__ = __VIRTUAL_FILE27 ? __VIRTUAL_FILE27(null) : {};
    this.size = 0;
}
var __VIRTUAL_FILE28 = hashClear;
function hashDelete(key) {
    var result = this.has(key) && delete this.__data__[key];
    this.size -= result ? 1 : 0;
    return result;
}
var __VIRTUAL_FILE29 = hashDelete;
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto3 = Object.prototype;
var hasOwnProperty2 = objectProto3.hasOwnProperty;
function hashGet(key) {
    var data8 = this.__data__;
    if (__VIRTUAL_FILE27) {
        var result = data8[key];
        return result === HASH_UNDEFINED ? void 0 : result;
    }
    return hasOwnProperty2.call(data8, key) ? data8[key] : void 0;
}
var __VIRTUAL_FILE30 = hashGet;
var objectProto4 = Object.prototype;
var hasOwnProperty3 = objectProto4.hasOwnProperty;
function hashHas(key) {
    var data9 = this.__data__;
    return __VIRTUAL_FILE27 ? data9[key] !== void 0 : hasOwnProperty3.call(data9, key);
}
var __VIRTUAL_FILE31 = hashHas;
var HASH_UNDEFINED1 = "__lodash_hash_undefined__";
function hashSet(key, value) {
    var data10 = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data10[key] = __VIRTUAL_FILE27 && value === void 0 ? HASH_UNDEFINED1 : value;
    return this;
}
var __VIRTUAL_FILE32 = hashSet;
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = __VIRTUAL_FILE28;
Hash.prototype["delete"] = __VIRTUAL_FILE29;
Hash.prototype.get = __VIRTUAL_FILE30;
Hash.prototype.has = __VIRTUAL_FILE31;
Hash.prototype.set = __VIRTUAL_FILE32;
var __VIRTUAL_FILE33 = Hash;
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new __VIRTUAL_FILE33(),
        map: new (__VIRTUAL_FILE26 || __VIRTUAL_FILE7)(),
        string: new __VIRTUAL_FILE33()
    };
}
var __VIRTUAL_FILE34 = mapCacheClear;
function isKeyable(value) {
    var type = typeof value;
    return type == "string" || type == "number" || type == "symbol" || type == "boolean" ? value !== "__proto__" : value === null;
}
var __VIRTUAL_FILE35 = isKeyable;
function getMapData(map1, key) {
    var data11 = map1.__data__;
    return __VIRTUAL_FILE35(key) ? data11[typeof key == "string" ? "string" : "hash"] : data11.map;
}
var __VIRTUAL_FILE36 = getMapData;
function mapCacheDelete(key) {
    var result = __VIRTUAL_FILE36(this, key)["delete"](key);
    this.size -= result ? 1 : 0;
    return result;
}
var __VIRTUAL_FILE37 = mapCacheDelete;
function mapCacheGet(key) {
    return __VIRTUAL_FILE36(this, key).get(key);
}
var __VIRTUAL_FILE38 = mapCacheGet;
function mapCacheHas(key) {
    return __VIRTUAL_FILE36(this, key).has(key);
}
var __VIRTUAL_FILE39 = mapCacheHas;
function mapCacheSet(key, value) {
    var data12 = __VIRTUAL_FILE36(this, key), size = data12.size;
    data12.set(key, value);
    this.size += data12.size == size ? 0 : 1;
    return this;
}
var __VIRTUAL_FILE40 = mapCacheSet;
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = __VIRTUAL_FILE34;
MapCache.prototype["delete"] = __VIRTUAL_FILE37;
MapCache.prototype.get = __VIRTUAL_FILE38;
MapCache.prototype.has = __VIRTUAL_FILE39;
MapCache.prototype.set = __VIRTUAL_FILE40;
var __VIRTUAL_FILE41 = MapCache;
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value) {
    var data13 = this.__data__;
    if (data13 instanceof __VIRTUAL_FILE7) {
        var pairs1 = data13.__data__;
        if (!__VIRTUAL_FILE26 || pairs1.length < LARGE_ARRAY_SIZE - 1) {
            pairs1.push([
                key,
                value
            ]);
            this.size = ++data13.size;
            return this;
        }
        data13 = this.__data__ = new __VIRTUAL_FILE41(pairs1);
    }
    data13.set(key, value);
    this.size = data13.size;
    return this;
}
var __VIRTUAL_FILE42 = stackSet;
function Stack(entries) {
    var data14 = this.__data__ = new __VIRTUAL_FILE7(entries);
    this.size = data14.size;
}
Stack.prototype.clear = __VIRTUAL_FILE8;
Stack.prototype["delete"] = __VIRTUAL_FILE9;
Stack.prototype.get = __VIRTUAL_FILE10;
Stack.prototype.has = __VIRTUAL_FILE11;
Stack.prototype.set = __VIRTUAL_FILE42;
var __VIRTUAL_FILE43 = Stack;
function arrayEach(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (iteratee(array[index], index, array) === false) {
            break;
        }
    }
    return array;
}
var __VIRTUAL_FILE44 = arrayEach;
var defineProperty = function() {
    try {
        var func = __VIRTUAL_FILE25(Object, "defineProperty");
        func({}, "", {});
        return func;
    } catch (e) {}
}();
var __VIRTUAL_FILE45 = defineProperty;
function baseAssignValue(object, key, value) {
    if (key == "__proto__" && __VIRTUAL_FILE45) {
        __VIRTUAL_FILE45(object, key, {
            configurable: true,
            enumerable: true,
            value,
            writable: true
        });
    } else {
        object[key] = value;
    }
}
var __VIRTUAL_FILE46 = baseAssignValue;
var objectProto5 = Object.prototype;
var hasOwnProperty4 = objectProto5.hasOwnProperty;
function assignValue(object, key, value) {
    var objValue = object[key];
    if (!(hasOwnProperty4.call(object, key) && __VIRTUAL_FILE1(objValue, value)) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE47 = assignValue;
function copyObject(source, props, object, customizer) {
    var isNew = !object;
    object || (object = {});
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object[key], source[key], key, object, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            __VIRTUAL_FILE46(object, key, newValue);
        } else {
            __VIRTUAL_FILE47(object, key, newValue);
        }
    }
    return object;
}
var __VIRTUAL_FILE48 = copyObject;
function baseTimes(n3, iteratee) {
    var index = -1, result = Array(n3);
    while(++index < n3){
        result[index] = iteratee(index);
    }
    return result;
}
var __VIRTUAL_FILE49 = baseTimes;
function isObjectLike(value) {
    return value != null && typeof value == "object";
}
var __VIRTUAL_FILE50 = isObjectLike;
var argsTag = "[object Arguments]";
function baseIsArguments(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == argsTag;
}
var __VIRTUAL_FILE51 = baseIsArguments;
var objectProto6 = Object.prototype;
var hasOwnProperty5 = objectProto6.hasOwnProperty;
var propertyIsEnumerable = objectProto6.propertyIsEnumerable;
var isArguments = __VIRTUAL_FILE51(function() {
    return arguments;
}()) ? __VIRTUAL_FILE51 : function(value) {
    return __VIRTUAL_FILE50(value) && hasOwnProperty5.call(value, "callee") && !propertyIsEnumerable.call(value, "callee");
};
var __VIRTUAL_FILE52 = isArguments;
var isArray = Array.isArray;
var __VIRTUAL_FILE53 = isArray;
function stubFalse() {
    return false;
}
var __VIRTUAL_FILE54 = stubFalse;
function createCommonjsModule(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE55 = createCommonjsModule(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0;
    var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
    var isBuffer = nativeIsBuffer || __VIRTUAL_FILE54;
    module.exports = isBuffer;
});
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value, length) {
    var type = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type == "number" || type != "symbol" && reIsUint.test(value)) && value > -1 && value % 1 == 0 && value < length;
}
var __VIRTUAL_FILE56 = isIndex;
var MAX_SAFE_INTEGER1 = 9007199254740991;
function isLength(value) {
    return typeof value == "number" && value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER1;
}
var __VIRTUAL_FILE57 = isLength;
var argsTag1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE57(value.length) && !!typedArrayTags[__VIRTUAL_FILE17(value)];
}
var __VIRTUAL_FILE58 = baseIsTypedArray;
function baseUnary(func) {
    return function(value) {
        return func(value);
    };
}
var __VIRTUAL_FILE59 = baseUnary;
function createCommonjsModule1(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE60 = createCommonjsModule1(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var freeProcess = moduleExports && __VIRTUAL_FILE12.process;
    var nodeUtil = function() {
        try {
            var types8 = freeModule && freeModule.require && freeModule.require("util").types;
            if (types8) {
                return types8;
            }
            return freeProcess && freeProcess.binding && freeProcess.binding("util");
        } catch (e) {}
    }();
    module.exports = nodeUtil;
});
var nodeIsTypedArray = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isTypedArray;
var isTypedArray = nodeIsTypedArray ? __VIRTUAL_FILE59(nodeIsTypedArray) : __VIRTUAL_FILE58;
var __VIRTUAL_FILE61 = isTypedArray;
var objectProto7 = Object.prototype;
var hasOwnProperty6 = objectProto7.hasOwnProperty;
function arrayLikeKeys(value, inherited) {
    var isArr = __VIRTUAL_FILE53(value), isArg = !isArr && __VIRTUAL_FILE52(value), isBuff = !isArr && !isArg && __VIRTUAL_FILE55(value), isType = !isArr && !isArg && !isBuff && __VIRTUAL_FILE61(value), skipIndexes = isArr || isArg || isBuff || isType, result = skipIndexes ? __VIRTUAL_FILE49(value.length, String) : [], length = result.length;
    for(var key in value){
        if ((inherited || hasOwnProperty6.call(value, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || __VIRTUAL_FILE56(key, length)))) {
            result.push(key);
        }
    }
    return result;
}
var __VIRTUAL_FILE62 = arrayLikeKeys;
var objectProto8 = Object.prototype;
function isPrototype(value) {
    var Ctor = value && value.constructor, proto1 = typeof Ctor == "function" && Ctor.prototype || objectProto8;
    return value === proto1;
}
var __VIRTUAL_FILE63 = isPrototype;
function overArg(func, transform) {
    return function(arg) {
        return func(transform(arg));
    };
}
var __VIRTUAL_FILE64 = overArg;
var nativeKeys = __VIRTUAL_FILE64(Object.keys, Object);
var __VIRTUAL_FILE65 = nativeKeys;
var objectProto9 = Object.prototype;
var hasOwnProperty7 = objectProto9.hasOwnProperty;
function baseKeys(object) {
    if (!__VIRTUAL_FILE63(object)) {
        return __VIRTUAL_FILE65(object);
    }
    var result = [];
    for(var key in Object(object)){
        if (hasOwnProperty7.call(object, key) && key != "constructor") {
            result.push(key);
        }
    }
    return result;
}
var __VIRTUAL_FILE66 = baseKeys;
function isArrayLike(value) {
    return value != null && __VIRTUAL_FILE57(value.length) && !__VIRTUAL_FILE19(value);
}
var __VIRTUAL_FILE67 = isArrayLike;
function keys(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object) : __VIRTUAL_FILE66(object);
}
var __VIRTUAL_FILE68 = keys;
function baseAssign(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE68(source), object);
}
var __VIRTUAL_FILE69 = baseAssign;
function nativeKeysIn(object) {
    var result = [];
    if (object != null) {
        for(var key in Object(object)){
            result.push(key);
        }
    }
    return result;
}
var __VIRTUAL_FILE70 = nativeKeysIn;
var objectProto10 = Object.prototype;
var hasOwnProperty8 = objectProto10.hasOwnProperty;
function baseKeysIn(object) {
    if (!__VIRTUAL_FILE18(object)) {
        return __VIRTUAL_FILE70(object);
    }
    var isProto = __VIRTUAL_FILE63(object), result = [];
    for(var key in object){
        if (!(key == "constructor" && (isProto || !hasOwnProperty8.call(object, key)))) {
            result.push(key);
        }
    }
    return result;
}
var __VIRTUAL_FILE71 = baseKeysIn;
function keysIn(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object, true) : __VIRTUAL_FILE71(object);
}
var __VIRTUAL_FILE72 = keysIn;
function baseAssignIn(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object);
}
var __VIRTUAL_FILE73 = baseAssignIn;
function createCommonjsModule2(fn) {
    var module = {
        exports: {}
    };
    return fn(module, module.exports), module.exports;
}
var __VIRTUAL_FILE74 = createCommonjsModule2(function(module, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module && !module.nodeType && module;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer2 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0, allocUnsafe = Buffer2 ? Buffer2.allocUnsafe : void 0;
    function cloneBuffer(buffer, isDeep) {
        if (isDeep) {
            return buffer.slice();
        }
        var length = buffer.length, result = allocUnsafe ? allocUnsafe(length) : new buffer.constructor(length);
        buffer.copy(result);
        return result;
    }
    module.exports = cloneBuffer;
});
function copyArray(source, array) {
    var index = -1, length = source.length;
    array || (array = Array(length));
    while(++index < length){
        array[index] = source[index];
    }
    return array;
}
var __VIRTUAL_FILE75 = copyArray;
function arrayFilter(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length, resIndex = 0, result = [];
    while(++index < length){
        var value = array[index];
        if (predicate(value, index, array)) {
            result[resIndex++] = value;
        }
    }
    return result;
}
var __VIRTUAL_FILE76 = arrayFilter;
function stubArray() {
    return [];
}
var __VIRTUAL_FILE77 = stubArray;
var objectProto11 = Object.prototype;
var propertyIsEnumerable1 = objectProto11.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? __VIRTUAL_FILE77 : function(object) {
    if (object == null) {
        return [];
    }
    object = Object(object);
    return __VIRTUAL_FILE76(nativeGetSymbols(object), function(symbol) {
        return propertyIsEnumerable1.call(object, symbol);
    });
};
var __VIRTUAL_FILE78 = getSymbols;
function copySymbols(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE78(source), object);
}
var __VIRTUAL_FILE79 = copySymbols;
function arrayPush(array, values4) {
    var index = -1, length = values4.length, offset = array.length;
    while(++index < length){
        array[offset + index] = values4[index];
    }
    return array;
}
var __VIRTUAL_FILE80 = arrayPush;
var getPrototype = __VIRTUAL_FILE64(Object.getPrototypeOf, Object);
var __VIRTUAL_FILE81 = getPrototype;
var nativeGetSymbols1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols1 ? __VIRTUAL_FILE77 : function(object) {
    var result = [];
    while(object){
        __VIRTUAL_FILE80(result, __VIRTUAL_FILE78(object));
        object = __VIRTUAL_FILE81(object);
    }
    return result;
};
var __VIRTUAL_FILE82 = getSymbolsIn;
function copySymbolsIn(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE82(source), object);
}
var __VIRTUAL_FILE83 = copySymbolsIn;
function baseGetAllKeys(object, keysFunc, symbolsFunc) {
    var result = keysFunc(object);
    return __VIRTUAL_FILE53(object) ? result : __VIRTUAL_FILE80(result, symbolsFunc(object));
}
var __VIRTUAL_FILE84 = baseGetAllKeys;
function getAllKeys(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE68, __VIRTUAL_FILE78);
}
var __VIRTUAL_FILE85 = getAllKeys;
function getAllKeysIn(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE72, __VIRTUAL_FILE82);
}
var __VIRTUAL_FILE86 = getAllKeysIn;
var DataView1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "DataView");
var __VIRTUAL_FILE87 = DataView1;
var Promise1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Promise");
var __VIRTUAL_FILE88 = Promise1;
var Set1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Set");
var __VIRTUAL_FILE89 = Set1;
var WeakMap = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "WeakMap");
var __VIRTUAL_FILE90 = WeakMap;
var mapTag1 = "[object Map]", objectTag1 = "[object Object]", promiseTag = "[object Promise]", setTag1 = "[object Set]", weakMapTag1 = "[object WeakMap]";
var dataViewTag1 = "[object DataView]";
var dataViewCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE87), mapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE26), promiseCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE88), setCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE89), weakMapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE90);
var getTag = __VIRTUAL_FILE17;
if (__VIRTUAL_FILE87 && getTag(new __VIRTUAL_FILE87(new ArrayBuffer(1))) != dataViewTag1 || __VIRTUAL_FILE26 && getTag(new __VIRTUAL_FILE26()) != mapTag1 || __VIRTUAL_FILE88 && getTag(__VIRTUAL_FILE88.resolve()) != promiseTag || __VIRTUAL_FILE89 && getTag(new __VIRTUAL_FILE89()) != setTag1 || __VIRTUAL_FILE90 && getTag(new __VIRTUAL_FILE90()) != weakMapTag1) {
    getTag = function(value) {
        var result = __VIRTUAL_FILE17(value), Ctor = result == objectTag1 ? value.constructor : void 0, ctorString = Ctor ? __VIRTUAL_FILE22(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag1;
                case mapCtorString:
                    return mapTag1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag1;
                case weakMapCtorString:
                    return weakMapTag1;
            }
        }
        return result;
    };
}
var __VIRTUAL_FILE91 = getTag;
var objectProto12 = Object.prototype;
var hasOwnProperty9 = objectProto12.hasOwnProperty;
function initCloneArray(array) {
    var length = array.length, result = new array.constructor(length);
    if (length && typeof array[0] == "string" && hasOwnProperty9.call(array, "index")) {
        result.index = array.index;
        result.input = array.input;
    }
    return result;
}
var __VIRTUAL_FILE92 = initCloneArray;
var Uint8Array1 = __VIRTUAL_FILE13.Uint8Array;
var __VIRTUAL_FILE93 = Uint8Array1;
function cloneArrayBuffer(arrayBuffer) {
    var result = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new __VIRTUAL_FILE93(result).set(new __VIRTUAL_FILE93(arrayBuffer));
    return result;
}
var __VIRTUAL_FILE94 = cloneArrayBuffer;
function cloneDataView(dataView, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}
var __VIRTUAL_FILE95 = cloneDataView;
var reFlags = /\w*$/;
function cloneRegExp(regexp2) {
    var result = new regexp2.constructor(regexp2.source, reFlags.exec(regexp2));
    result.lastIndex = regexp2.lastIndex;
    return result;
}
var __VIRTUAL_FILE96 = cloneRegExp;
var symbolProto = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf = symbolProto ? symbolProto.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {};
}
var __VIRTUAL_FILE97 = cloneSymbol;
function cloneTypedArray(typedArray, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}
var __VIRTUAL_FILE98 = cloneTypedArray;
var boolTag1 = "[object Boolean]", dateTag1 = "[object Date]", mapTag2 = "[object Map]", numberTag1 = "[object Number]", regexpTag1 = "[object RegExp]", setTag2 = "[object Set]", stringTag1 = "[object String]", symbolTag = "[object Symbol]";
var arrayBufferTag1 = "[object ArrayBuffer]", dataViewTag2 = "[object DataView]", float32Tag1 = "[object Float32Array]", float64Tag1 = "[object Float64Array]", int8Tag1 = "[object Int8Array]", int16Tag1 = "[object Int16Array]", int32Tag1 = "[object Int32Array]", uint8Tag1 = "[object Uint8Array]", uint8ClampedTag1 = "[object Uint8ClampedArray]", uint16Tag1 = "[object Uint16Array]", uint32Tag1 = "[object Uint32Array]";
function initCloneByTag(object, tag, isDeep) {
    var Ctor = object.constructor;
    switch(tag){
        case arrayBufferTag1:
            return __VIRTUAL_FILE94(object);
        case boolTag1:
        case dateTag1:
            return new Ctor(+object);
        case dataViewTag2:
            return __VIRTUAL_FILE95(object, isDeep);
        case float32Tag1:
        case float64Tag1:
        case int8Tag1:
        case int16Tag1:
        case int32Tag1:
        case uint8Tag1:
        case uint8ClampedTag1:
        case uint16Tag1:
        case uint32Tag1:
            return __VIRTUAL_FILE98(object, isDeep);
        case mapTag2:
            return new Ctor();
        case numberTag1:
        case stringTag1:
            return new Ctor(object);
        case regexpTag1:
            return __VIRTUAL_FILE96(object);
        case setTag2:
            return new Ctor();
        case symbolTag:
            return __VIRTUAL_FILE97(object);
    }
}
var __VIRTUAL_FILE99 = initCloneByTag;
var objectCreate = Object.create;
var baseCreate = function() {
    function object() {}
    return function(proto2) {
        if (!__VIRTUAL_FILE18(proto2)) {
            return {};
        }
        if (objectCreate) {
            return objectCreate(proto2);
        }
        object.prototype = proto2;
        var result = new object();
        object.prototype = void 0;
        return result;
    };
}();
var __VIRTUAL_FILE100 = baseCreate;
function initCloneObject(object) {
    return typeof object.constructor == "function" && !__VIRTUAL_FILE63(object) ? __VIRTUAL_FILE100(__VIRTUAL_FILE81(object)) : {};
}
var __VIRTUAL_FILE101 = initCloneObject;
var mapTag3 = "[object Map]";
function baseIsMap(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == mapTag3;
}
var __VIRTUAL_FILE102 = baseIsMap;
var nodeIsMap = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isMap;
var isMap = nodeIsMap ? __VIRTUAL_FILE59(nodeIsMap) : __VIRTUAL_FILE102;
var __VIRTUAL_FILE103 = isMap;
var setTag3 = "[object Set]";
function baseIsSet(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == setTag3;
}
var __VIRTUAL_FILE104 = baseIsSet;
var nodeIsSet = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isSet;
var isSet = nodeIsSet ? __VIRTUAL_FILE59(nodeIsSet) : __VIRTUAL_FILE104;
var __VIRTUAL_FILE105 = isSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag2 = "[object Arguments]", arrayTag1 = "[object Array]", boolTag2 = "[object Boolean]", dateTag2 = "[object Date]", errorTag1 = "[object Error]", funcTag2 = "[object Function]", genTag1 = "[object GeneratorFunction]", mapTag4 = "[object Map]", numberTag2 = "[object Number]", objectTag2 = "[object Object]", regexpTag2 = "[object RegExp]", setTag4 = "[object Set]", stringTag2 = "[object String]", symbolTag1 = "[object Symbol]", weakMapTag2 = "[object WeakMap]";
var arrayBufferTag2 = "[object ArrayBuffer]", dataViewTag3 = "[object DataView]", float32Tag2 = "[object Float32Array]", float64Tag2 = "[object Float64Array]", int8Tag2 = "[object Int8Array]", int16Tag2 = "[object Int16Array]", int32Tag2 = "[object Int32Array]", uint8Tag2 = "[object Uint8Array]", uint8ClampedTag2 = "[object Uint8ClampedArray]", uint16Tag2 = "[object Uint16Array]", uint32Tag2 = "[object Uint32Array]";
var cloneableTags = {};
cloneableTags[argsTag2] = cloneableTags[arrayTag1] = cloneableTags[arrayBufferTag2] = cloneableTags[dataViewTag3] = cloneableTags[boolTag2] = cloneableTags[dateTag2] = cloneableTags[float32Tag2] = cloneableTags[float64Tag2] = cloneableTags[int8Tag2] = cloneableTags[int16Tag2] = cloneableTags[int32Tag2] = cloneableTags[mapTag4] = cloneableTags[numberTag2] = cloneableTags[objectTag2] = cloneableTags[regexpTag2] = cloneableTags[setTag4] = cloneableTags[stringTag2] = cloneableTags[symbolTag1] = cloneableTags[uint8Tag2] = cloneableTags[uint8ClampedTag2] = cloneableTags[uint16Tag2] = cloneableTags[uint32Tag2] = true;
cloneableTags[errorTag1] = cloneableTags[funcTag2] = cloneableTags[weakMapTag2] = false;
function baseClone(value, bitmask, customizer, key, object, stack) {
    var result, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result = object ? customizer(value, key, object, stack) : customizer(value);
    }
    if (result !== void 0) {
        return result;
    }
    if (!__VIRTUAL_FILE18(value)) {
        return value;
    }
    var isArr = __VIRTUAL_FILE53(value);
    if (isArr) {
        result = __VIRTUAL_FILE92(value);
        if (!isDeep) {
            return __VIRTUAL_FILE75(value, result);
        }
    } else {
        var tag = __VIRTUAL_FILE91(value), isFunc = tag == funcTag2 || tag == genTag1;
        if (__VIRTUAL_FILE55(value)) {
            return __VIRTUAL_FILE74(value, isDeep);
        }
        if (tag == objectTag2 || tag == argsTag2 || isFunc && !object) {
            result = isFlat || isFunc ? {} : __VIRTUAL_FILE101(value);
            if (!isDeep) {
                return isFlat ? __VIRTUAL_FILE83(value, __VIRTUAL_FILE73(result, value)) : __VIRTUAL_FILE79(value, __VIRTUAL_FILE69(result, value));
            }
        } else {
            if (!cloneableTags[tag]) {
                return object ? value : {};
            }
            result = __VIRTUAL_FILE99(value, tag, isDeep);
        }
    }
    stack || (stack = new __VIRTUAL_FILE43());
    var stacked = stack.get(value);
    if (stacked) {
        return stacked;
    }
    stack.set(value, result);
    if (__VIRTUAL_FILE105(value)) {
        value.forEach(function(subValue) {
            result.add(baseClone(subValue, bitmask, customizer, subValue, value, stack));
        });
    } else if (__VIRTUAL_FILE103(value)) {
        value.forEach(function(subValue, key2) {
            result.set(key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? __VIRTUAL_FILE86 : __VIRTUAL_FILE85 : isFlat ? __VIRTUAL_FILE72 : __VIRTUAL_FILE68;
    var props = isArr ? void 0 : keysFunc(value);
    __VIRTUAL_FILE44(props || value, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value[key2];
        }
        __VIRTUAL_FILE47(result, key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
    });
    return result;
}
var __VIRTUAL_FILE106 = baseClone;
var CLONE_DEEP_FLAG1 = 1, CLONE_SYMBOLS_FLAG1 = 4;
function cloneDeep(value) {
    return __VIRTUAL_FILE106(value, CLONE_DEEP_FLAG1 | CLONE_SYMBOLS_FLAG1);
}
var __VIRTUAL_FILE107 = cloneDeep;
var now = function() {
    return __VIRTUAL_FILE13.Date.now();
};
var __VIRTUAL_FILE108 = now;
var reWhitespace = /\s/;
function trimmedEndIndex(string4) {
    var index = string4.length;
    while(index-- && reWhitespace.test(string4.charAt(index))){}
    return index;
}
var __VIRTUAL_FILE109 = trimmedEndIndex;
var reTrimStart = /^\s+/;
function baseTrim(string5) {
    return string5 ? string5.slice(0, __VIRTUAL_FILE109(string5) + 1).replace(reTrimStart, "") : string5;
}
var __VIRTUAL_FILE110 = baseTrim;
var symbolTag2 = "[object Symbol]";
function isSymbol(value) {
    return typeof value == "symbol" || __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == symbolTag2;
}
var __VIRTUAL_FILE111 = isSymbol;
var NAN = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (__VIRTUAL_FILE111(value)) {
        return NAN;
    }
    if (__VIRTUAL_FILE18(value)) {
        var other = typeof value.valueOf == "function" ? value.valueOf() : value;
        value = __VIRTUAL_FILE18(other) ? other + "" : other;
    }
    if (typeof value != "string") {
        return value === 0 ? value : +value;
    }
    value = __VIRTUAL_FILE110(value);
    var isBinary1 = reIsBinary.test(value);
    return isBinary1 || reIsOctal.test(value) ? freeParseInt(value.slice(2), isBinary1 ? 2 : 8) : reIsBadHex.test(value) ? NAN : +value;
}
var __VIRTUAL_FILE112 = toNumber;
var FUNC_ERROR_TEXT = "Expected a function";
var nativeMax = Math.max, nativeMin = Math.min;
function debounce(func, wait, options) {
    var lastArgs, lastThis, maxWait, result, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    wait = __VIRTUAL_FILE112(wait) || 0;
    if (__VIRTUAL_FILE18(options)) {
        leading = !!options.leading;
        maxing = "maxWait" in options;
        maxWait = maxing ? nativeMax(__VIRTUAL_FILE112(options.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    function invokeFunc(time) {
        var args = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result = func.apply(thisArg, args);
        return result;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = __VIRTUAL_FILE108();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result : trailingEdge(__VIRTUAL_FILE108());
    }
    function debounced() {
        var time = __VIRTUAL_FILE108(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
var __VIRTUAL_FILE113 = debounce;
var HASH_UNDEFINED2 = "__lodash_hash_undefined__";
function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED2);
    return this;
}
var __VIRTUAL_FILE114 = setCacheAdd;
function setCacheHas(value) {
    return this.__data__.has(value);
}
var __VIRTUAL_FILE115 = setCacheHas;
function SetCache(values5) {
    var index = -1, length = values5 == null ? 0 : values5.length;
    this.__data__ = new __VIRTUAL_FILE41();
    while(++index < length){
        this.add(values5[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = __VIRTUAL_FILE114;
SetCache.prototype.has = __VIRTUAL_FILE115;
var __VIRTUAL_FILE116 = SetCache;
function baseFindIndex(array, predicate, fromIndex, fromRight) {
    var length = array.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array[index], index, array)) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE117 = baseFindIndex;
function baseIsNaN(value) {
    return value !== value;
}
var __VIRTUAL_FILE118 = baseIsNaN;
function strictIndexOf(array, value, fromIndex) {
    var index = fromIndex - 1, length = array.length;
    while(++index < length){
        if (array[index] === value) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE119 = strictIndexOf;
function baseIndexOf(array, value, fromIndex) {
    return value === value ? __VIRTUAL_FILE119(array, value, fromIndex) : __VIRTUAL_FILE117(array, __VIRTUAL_FILE118, fromIndex);
}
var __VIRTUAL_FILE120 = baseIndexOf;
function arrayIncludes(array, value) {
    var length = array == null ? 0 : array.length;
    return !!length && __VIRTUAL_FILE120(array, value, 0) > -1;
}
var __VIRTUAL_FILE121 = arrayIncludes;
function arrayIncludesWith(array, value, comparator) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (comparator(value, array[index])) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE122 = arrayIncludesWith;
function arrayMap(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length, result = Array(length);
    while(++index < length){
        result[index] = iteratee(array[index], index, array);
    }
    return result;
}
var __VIRTUAL_FILE123 = arrayMap;
function cacheHas(cache, key) {
    return cache.has(key);
}
var __VIRTUAL_FILE124 = cacheHas;
var LARGE_ARRAY_SIZE1 = 200;
function baseDifference(array, values6, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, isCommon = true, length = array.length, result = [], valuesLength = values6.length;
    if (!length) {
        return result;
    }
    if (iteratee) {
        values6 = __VIRTUAL_FILE123(values6, __VIRTUAL_FILE59(iteratee));
    }
    if (comparator) {
        includes = __VIRTUAL_FILE122;
        isCommon = false;
    } else if (values6.length >= LARGE_ARRAY_SIZE1) {
        includes = __VIRTUAL_FILE124;
        isCommon = false;
        values6 = new __VIRTUAL_FILE116(values6);
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee == null ? value : iteratee(value);
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values6[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result.push(value);
        } else if (!includes(values6, computed, comparator)) {
            result.push(value);
        }
    }
    return result;
}
var __VIRTUAL_FILE125 = baseDifference;
var spreadableSymbol = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.isConcatSpreadable : void 0;
function isFlattenable(value) {
    return __VIRTUAL_FILE53(value) || __VIRTUAL_FILE52(value) || !!(spreadableSymbol && value && value[spreadableSymbol]);
}
var __VIRTUAL_FILE126 = isFlattenable;
function baseFlatten(array, depth, predicate, isStrict, result) {
    var index = -1, length = array.length;
    predicate || (predicate = __VIRTUAL_FILE126);
    result || (result = []);
    while(++index < length){
        var value = array[index];
        if (depth > 0 && predicate(value)) {
            if (depth > 1) {
                baseFlatten(value, depth - 1, predicate, isStrict, result);
            } else {
                __VIRTUAL_FILE80(result, value);
            }
        } else if (!isStrict) {
            result[result.length] = value;
        }
    }
    return result;
}
var __VIRTUAL_FILE127 = baseFlatten;
function identity(value) {
    return value;
}
var __VIRTUAL_FILE128 = identity;
function apply(func, thisArg, args) {
    switch(args.length){
        case 0:
            return func.call(thisArg);
        case 1:
            return func.call(thisArg, args[0]);
        case 2:
            return func.call(thisArg, args[0], args[1]);
        case 3:
            return func.call(thisArg, args[0], args[1], args[2]);
    }
    return func.apply(thisArg, args);
}
var __VIRTUAL_FILE129 = apply;
var nativeMax1 = Math.max;
function overRest(func, start, transform) {
    start = nativeMax1(start === void 0 ? func.length - 1 : start, 0);
    return function() {
        var args = arguments, index = -1, length = nativeMax1(args.length - start, 0), array = Array(length);
        while(++index < length){
            array[index] = args[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args[index];
        }
        otherArgs[start] = transform(array);
        return __VIRTUAL_FILE129(func, this, otherArgs);
    };
}
var __VIRTUAL_FILE130 = overRest;
function constant(value) {
    return function() {
        return value;
    };
}
var __VIRTUAL_FILE131 = constant;
var baseSetToString = !__VIRTUAL_FILE45 ? __VIRTUAL_FILE128 : function(func, string6) {
    return __VIRTUAL_FILE45(func, "toString", {
        configurable: true,
        enumerable: false,
        value: __VIRTUAL_FILE131(string6),
        writable: true
    });
};
var __VIRTUAL_FILE132 = baseSetToString;
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func.apply(void 0, arguments);
    };
}
var __VIRTUAL_FILE133 = shortOut;
var setToString = __VIRTUAL_FILE133(__VIRTUAL_FILE132);
var __VIRTUAL_FILE134 = setToString;
function baseRest(func, start) {
    return __VIRTUAL_FILE134(__VIRTUAL_FILE130(func, start, __VIRTUAL_FILE128), func + "");
}
var __VIRTUAL_FILE135 = baseRest;
function isArrayLikeObject(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE67(value);
}
var __VIRTUAL_FILE136 = isArrayLikeObject;
var difference = __VIRTUAL_FILE135(function(array, values7) {
    return __VIRTUAL_FILE136(array) ? __VIRTUAL_FILE125(array, __VIRTUAL_FILE127(values7, 1, __VIRTUAL_FILE136, true)) : [];
});
var __VIRTUAL_FILE137 = difference;
function createBaseFor(fromRight) {
    return function(object, iteratee, keysFunc) {
        var index = -1, iterable = Object(object), props = keysFunc(object), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object;
    };
}
var __VIRTUAL_FILE138 = createBaseFor;
var baseFor = __VIRTUAL_FILE138();
var __VIRTUAL_FILE139 = baseFor;
function baseForOwn(object, iteratee) {
    return object && __VIRTUAL_FILE139(object, iteratee, __VIRTUAL_FILE68);
}
var __VIRTUAL_FILE140 = baseForOwn;
function createBaseEach(eachFunc, fromRight) {
    return function(collection, iteratee) {
        if (collection == null) {
            return collection;
        }
        if (!__VIRTUAL_FILE67(collection)) {
            return eachFunc(collection, iteratee);
        }
        var length = collection.length, index = fromRight ? length : -1, iterable = Object(collection);
        while(fromRight ? index-- : ++index < length){
            if (iteratee(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection;
    };
}
var __VIRTUAL_FILE141 = createBaseEach;
var baseEach = __VIRTUAL_FILE141(__VIRTUAL_FILE140);
var __VIRTUAL_FILE142 = baseEach;
function castFunction(value) {
    return typeof value == "function" ? value : __VIRTUAL_FILE128;
}
var __VIRTUAL_FILE143 = castFunction;
function forEach(collection, iteratee) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE44 : __VIRTUAL_FILE142;
    return func(collection, __VIRTUAL_FILE143(iteratee));
}
var __VIRTUAL_FILE144 = forEach;
var __VIRTUAL_FILE145 = __VIRTUAL_FILE144;
function assignMergeValue(object, key, value) {
    if (value !== void 0 && !__VIRTUAL_FILE1(object[key], value) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE146 = assignMergeValue;
var objectTag3 = "[object Object]";
var funcProto2 = Function.prototype, objectProto13 = Object.prototype;
var funcToString2 = funcProto2.toString;
var hasOwnProperty10 = objectProto13.hasOwnProperty;
var objectCtorString = funcToString2.call(Object);
function isPlainObject(value) {
    if (!__VIRTUAL_FILE50(value) || __VIRTUAL_FILE17(value) != objectTag3) {
        return false;
    }
    var proto3 = __VIRTUAL_FILE81(value);
    if (proto3 === null) {
        return true;
    }
    var Ctor = hasOwnProperty10.call(proto3, "constructor") && proto3.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString2.call(Ctor) == objectCtorString;
}
var __VIRTUAL_FILE147 = isPlainObject;
function safeGet(object, key) {
    if (key === "constructor" && typeof object[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object[key];
}
var __VIRTUAL_FILE148 = safeGet;
function toPlainObject(value) {
    return __VIRTUAL_FILE48(value, __VIRTUAL_FILE72(value));
}
var __VIRTUAL_FILE149 = toPlainObject;
function baseMergeDeep(object, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = __VIRTUAL_FILE148(object, key), srcValue = __VIRTUAL_FILE148(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        __VIRTUAL_FILE146(object, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = __VIRTUAL_FILE53(srcValue), isBuff = !isArr && __VIRTUAL_FILE55(srcValue), isTyped = !isArr && !isBuff && __VIRTUAL_FILE61(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (__VIRTUAL_FILE53(objValue)) {
                newValue = objValue;
            } else if (__VIRTUAL_FILE136(objValue)) {
                newValue = __VIRTUAL_FILE75(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = __VIRTUAL_FILE74(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = __VIRTUAL_FILE98(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (__VIRTUAL_FILE147(srcValue) || __VIRTUAL_FILE52(srcValue)) {
            newValue = objValue;
            if (__VIRTUAL_FILE52(objValue)) {
                newValue = __VIRTUAL_FILE149(objValue);
            } else if (!__VIRTUAL_FILE18(objValue) || __VIRTUAL_FILE19(objValue)) {
                newValue = __VIRTUAL_FILE101(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    __VIRTUAL_FILE146(object, key, newValue);
}
var __VIRTUAL_FILE150 = baseMergeDeep;
function baseMerge(object, source, srcIndex, customizer, stack) {
    if (object === source) {
        return;
    }
    __VIRTUAL_FILE139(source, function(srcValue, key) {
        stack || (stack = new __VIRTUAL_FILE43());
        if (__VIRTUAL_FILE18(srcValue)) {
            __VIRTUAL_FILE150(object, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(__VIRTUAL_FILE148(object, key), srcValue, key + "", object, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            __VIRTUAL_FILE146(object, key, newValue);
        }
    }, __VIRTUAL_FILE72);
}
var __VIRTUAL_FILE151 = baseMerge;
function isIterateeCall(value, index, object) {
    if (!__VIRTUAL_FILE18(object)) {
        return false;
    }
    var type = typeof index;
    if (type == "number" ? __VIRTUAL_FILE67(object) && __VIRTUAL_FILE56(index, object.length) : type == "string" && index in object) {
        return __VIRTUAL_FILE1(object[index], value);
    }
    return false;
}
var __VIRTUAL_FILE152 = isIterateeCall;
function createAssigner(assigner) {
    return __VIRTUAL_FILE135(function(object, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && __VIRTUAL_FILE152(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object = Object(object);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object, source, index, customizer);
            }
        }
        return object;
    });
}
var __VIRTUAL_FILE153 = createAssigner;
var mergeWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE151(object, source, srcIndex, customizer);
});
var __VIRTUAL_FILE154 = mergeWith;
var nativeFloor = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor(nativeRandom() * (upper - lower + 1));
}
var __VIRTUAL_FILE155 = baseRandom;
function shuffleSelf(array, size) {
    var index = -1, length = array.length, lastIndex = length - 1;
    size = size === void 0 ? length : size;
    while(++index < size){
        var rand = __VIRTUAL_FILE155(index, lastIndex), value = array[rand];
        array[rand] = array[index];
        array[index] = value;
    }
    array.length = size;
    return array;
}
var __VIRTUAL_FILE156 = shuffleSelf;
function arrayShuffle(array) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE75(array));
}
var __VIRTUAL_FILE157 = arrayShuffle;
function baseValues(object, props) {
    return __VIRTUAL_FILE123(props, function(key) {
        return object[key];
    });
}
var __VIRTUAL_FILE158 = baseValues;
function values(object) {
    return object == null ? [] : __VIRTUAL_FILE158(object, __VIRTUAL_FILE68(object));
}
var __VIRTUAL_FILE159 = values;
function baseShuffle(collection) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE159(collection));
}
var __VIRTUAL_FILE160 = baseShuffle;
function shuffle(collection) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE157 : __VIRTUAL_FILE160;
    return func(collection);
}
var __VIRTUAL_FILE161 = shuffle;
var assignInWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object, customizer);
});
var __VIRTUAL_FILE162 = assignInWith;
var domExcTag = "[object DOMException]", errorTag2 = "[object Error]";
function isError(value) {
    if (!__VIRTUAL_FILE50(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == errorTag2 || tag == domExcTag || typeof value.message == "string" && typeof value.name == "string" && !__VIRTUAL_FILE147(value);
}
var __VIRTUAL_FILE163 = isError;
var attempt = __VIRTUAL_FILE135(function(func, args) {
    try {
        return __VIRTUAL_FILE129(func, void 0, args);
    } catch (e) {
        return __VIRTUAL_FILE163(e) ? e : new Error(e);
    }
});
var __VIRTUAL_FILE164 = attempt;
var objectProto14 = Object.prototype;
var hasOwnProperty11 = objectProto14.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object) {
    if (objValue === void 0 || __VIRTUAL_FILE1(objValue, objectProto14[key]) && !hasOwnProperty11.call(object, key)) {
        return srcValue;
    }
    return objValue;
}
var __VIRTUAL_FILE165 = customDefaultsAssignIn;
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var __VIRTUAL_FILE166 = escapeStringChar;
var reInterpolate = /<%=([\s\S]+?)%>/g;
var __VIRTUAL_FILE167 = reInterpolate;
function basePropertyOf(object) {
    return function(key) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE168 = basePropertyOf;
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = __VIRTUAL_FILE168(htmlEscapes);
var __VIRTUAL_FILE169 = escapeHtmlChar;
var INFINITY = 1 / 0;
var symbolProto1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolToString = symbolProto1 ? symbolProto1.toString : void 0;
function baseToString(value) {
    if (typeof value == "string") {
        return value;
    }
    if (__VIRTUAL_FILE53(value)) {
        return __VIRTUAL_FILE123(value, baseToString) + "";
    }
    if (__VIRTUAL_FILE111(value)) {
        return symbolToString ? symbolToString.call(value) : "";
    }
    var result = value + "";
    return result == "0" && 1 / value == -INFINITY ? "-0" : result;
}
var __VIRTUAL_FILE170 = baseToString;
function toString(value) {
    return value == null ? "" : __VIRTUAL_FILE170(value);
}
var __VIRTUAL_FILE171 = toString;
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string7) {
    string7 = __VIRTUAL_FILE171(string7);
    return string7 && reHasUnescapedHtml.test(string7) ? string7.replace(reUnescapedHtml, __VIRTUAL_FILE169) : string7;
}
var __VIRTUAL_FILE172 = escape;
var reEscape = /<%-([\s\S]+?)%>/g;
var __VIRTUAL_FILE173 = reEscape;
var reEvaluate = /<%([\s\S]+?)%>/g;
var __VIRTUAL_FILE174 = reEvaluate;
var templateSettings = {
    escape: __VIRTUAL_FILE173,
    evaluate: __VIRTUAL_FILE174,
    interpolate: __VIRTUAL_FILE167,
    variable: "",
    imports: {
        _: {
            escape: __VIRTUAL_FILE172
        }
    }
};
var __VIRTUAL_FILE175 = templateSettings;
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto15 = Object.prototype;
var hasOwnProperty12 = objectProto15.hasOwnProperty;
function template(string8, options, guard) {
    var settings = __VIRTUAL_FILE175.imports._.templateSettings || __VIRTUAL_FILE175;
    if (guard && __VIRTUAL_FILE152(string8, options, guard)) {
        options = void 0;
    }
    string8 = __VIRTUAL_FILE171(string8);
    options = __VIRTUAL_FILE162({}, options, settings, __VIRTUAL_FILE165);
    var imports = __VIRTUAL_FILE162({}, options.imports, settings.imports, __VIRTUAL_FILE165), importsKeys = __VIRTUAL_FILE68(imports), importsValues = __VIRTUAL_FILE158(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === __VIRTUAL_FILE167 ? reEsTemplate : reNoMatch).source + "|" + (options.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty12.call(options, "sourceURL") ? "//# sourceURL=" + (options.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string8.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string8.slice(index, offset).replace(reUnescapedString, __VIRTUAL_FILE166);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty12.call(options, "variable") && options.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result = __VIRTUAL_FILE164(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result.source = source;
    if (__VIRTUAL_FILE163(result)) {
        throw result;
    }
    return result;
}
var __VIRTUAL_FILE176 = template;
function noop() {}
var __VIRTUAL_FILE177 = noop;
function setToArray(set1) {
    var index = -1, result = Array(set1.size);
    set1.forEach(function(value) {
        result[++index] = value;
    });
    return result;
}
var __VIRTUAL_FILE178 = setToArray;
var INFINITY1 = 1 / 0;
var createSet = !(__VIRTUAL_FILE89 && 1 / __VIRTUAL_FILE178(new __VIRTUAL_FILE89([
    ,
    -0
]))[1] == INFINITY1) ? __VIRTUAL_FILE177 : function(values8) {
    return new __VIRTUAL_FILE89(values8);
};
var __VIRTUAL_FILE179 = createSet;
var LARGE_ARRAY_SIZE2 = 200;
function baseUniq(array, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, length = array.length, isCommon = true, result = [], seen = result;
    if (comparator) {
        isCommon = false;
        includes = __VIRTUAL_FILE122;
    } else if (length >= LARGE_ARRAY_SIZE2) {
        var set2 = iteratee ? null : __VIRTUAL_FILE179(array);
        if (set2) {
            return __VIRTUAL_FILE178(set2);
        }
        isCommon = false;
        includes = __VIRTUAL_FILE124;
        seen = new __VIRTUAL_FILE116();
    } else {
        seen = iteratee ? [] : result;
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee ? iteratee(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee) {
                seen.push(computed);
            }
            result.push(value);
        } else if (!includes(seen, computed, comparator)) {
            if (seen !== result) {
                seen.push(computed);
            }
            result.push(value);
        }
    }
    return result;
}
var __VIRTUAL_FILE180 = baseUniq;
function uniq(array) {
    return array && array.length ? __VIRTUAL_FILE180(array) : [];
}
var __VIRTUAL_FILE181 = uniq;
function arraySome(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (predicate(array[index], index, array)) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE182 = arraySome;
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array;
    }
    var index = -1, result = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new __VIRTUAL_FILE116() : void 0;
    stack.set(array, other);
    stack.set(other, array);
    while(++index < arrLength){
        var arrValue = array[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array, stack) : customizer(arrValue, othValue, index, array, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result = false;
            break;
        }
        if (seen) {
            if (!__VIRTUAL_FILE182(other, function(othValue2, othIndex) {
                if (!__VIRTUAL_FILE124(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result = false;
            break;
        }
    }
    stack["delete"](array);
    stack["delete"](other);
    return result;
}
var __VIRTUAL_FILE183 = equalArrays;
function mapToArray(map2) {
    var index = -1, result = Array(map2.size);
    map2.forEach(function(value, key) {
        result[++index] = [
            key,
            value
        ];
    });
    return result;
}
var __VIRTUAL_FILE184 = mapToArray;
var COMPARE_PARTIAL_FLAG1 = 1, COMPARE_UNORDERED_FLAG1 = 2;
var boolTag3 = "[object Boolean]", dateTag3 = "[object Date]", errorTag3 = "[object Error]", mapTag5 = "[object Map]", numberTag3 = "[object Number]", regexpTag3 = "[object RegExp]", setTag5 = "[object Set]", stringTag3 = "[object String]", symbolTag3 = "[object Symbol]";
var arrayBufferTag3 = "[object ArrayBuffer]", dataViewTag4 = "[object DataView]";
var symbolProto2 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf1 = symbolProto2 ? symbolProto2.valueOf : void 0;
function equalByTag(object, other, tag, bitmask, customizer, equalFunc, stack) {
    switch(tag){
        case dataViewTag4:
            if (object.byteLength != other.byteLength || object.byteOffset != other.byteOffset) {
                return false;
            }
            object = object.buffer;
            other = other.buffer;
        case arrayBufferTag3:
            if (object.byteLength != other.byteLength || !equalFunc(new __VIRTUAL_FILE93(object), new __VIRTUAL_FILE93(other))) {
                return false;
            }
            return true;
        case boolTag3:
        case dateTag3:
        case numberTag3:
            return __VIRTUAL_FILE1(+object, +other);
        case errorTag3:
            return object.name == other.name && object.message == other.message;
        case regexpTag3:
        case stringTag3:
            return object == other + "";
        case mapTag5:
            var convert = __VIRTUAL_FILE184;
        case setTag5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG1;
            convert || (convert = __VIRTUAL_FILE178);
            if (object.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG1;
            stack.set(object, other);
            var result = __VIRTUAL_FILE183(convert(object), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object);
            return result;
        case symbolTag3:
            if (symbolValueOf1) {
                return symbolValueOf1.call(object) == symbolValueOf1.call(other);
            }
    }
    return false;
}
var __VIRTUAL_FILE185 = equalByTag;
var COMPARE_PARTIAL_FLAG2 = 1;
var objectProto16 = Object.prototype;
var hasOwnProperty13 = objectProto16.hasOwnProperty;
function equalObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG2, objProps = __VIRTUAL_FILE85(object), objLength = objProps.length, othProps = __VIRTUAL_FILE85(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty13.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object;
    }
    var result = true;
    stack.set(object, other);
    stack.set(other, object);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object, stack) : customizer(objValue, othValue, key, object, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result && !skipCtor) {
        var objCtor = object.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result = false;
        }
    }
    stack["delete"](object);
    stack["delete"](other);
    return result;
}
var __VIRTUAL_FILE186 = equalObjects;
var COMPARE_PARTIAL_FLAG3 = 1;
var argsTag3 = "[object Arguments]", arrayTag2 = "[object Array]", objectTag4 = "[object Object]";
var objectProto17 = Object.prototype;
var hasOwnProperty14 = objectProto17.hasOwnProperty;
function baseIsEqualDeep(object, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = __VIRTUAL_FILE53(object), othIsArr = __VIRTUAL_FILE53(other), objTag = objIsArr ? arrayTag2 : __VIRTUAL_FILE91(object), othTag = othIsArr ? arrayTag2 : __VIRTUAL_FILE91(other);
    objTag = objTag == argsTag3 ? objectTag4 : objTag;
    othTag = othTag == argsTag3 ? objectTag4 : othTag;
    var objIsObj = objTag == objectTag4, othIsObj = othTag == objectTag4, isSameTag = objTag == othTag;
    if (isSameTag && __VIRTUAL_FILE55(object)) {
        if (!__VIRTUAL_FILE55(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new __VIRTUAL_FILE43());
        return objIsArr || __VIRTUAL_FILE61(object) ? __VIRTUAL_FILE183(object, other, bitmask, customizer, equalFunc, stack) : __VIRTUAL_FILE185(object, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG3)) {
        var objIsWrapped = objIsObj && hasOwnProperty14.call(object, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty14.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object.value() : object, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new __VIRTUAL_FILE43());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new __VIRTUAL_FILE43());
    return __VIRTUAL_FILE186(object, other, bitmask, customizer, equalFunc, stack);
}
var __VIRTUAL_FILE187 = baseIsEqualDeep;
function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
        return true;
    }
    if (value == null || other == null || !__VIRTUAL_FILE50(value) && !__VIRTUAL_FILE50(other)) {
        return value !== value && other !== other;
    }
    return __VIRTUAL_FILE187(value, other, bitmask, customizer, baseIsEqual, stack);
}
var __VIRTUAL_FILE188 = baseIsEqual;
var COMPARE_PARTIAL_FLAG4 = 1, COMPARE_UNORDERED_FLAG2 = 2;
function baseIsMatch(object, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object == null) {
        return !length;
    }
    object = Object(object);
    while(index--){
        var data15 = matchData[index];
        if (noCustomizer && data15[2] ? data15[1] !== object[data15[0]] : !(data15[0] in object)) {
            return false;
        }
    }
    while(++index < length){
        data15 = matchData[index];
        var key = data15[0], objValue = object[key], srcValue = data15[1];
        if (noCustomizer && data15[2]) {
            if (objValue === void 0 && !(key in object)) {
                return false;
            }
        } else {
            var stack = new __VIRTUAL_FILE43();
            if (customizer) {
                var result = customizer(objValue, srcValue, key, object, source, stack);
            }
            if (!(result === void 0 ? __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG4 | COMPARE_UNORDERED_FLAG2, customizer, stack) : result)) {
                return false;
            }
        }
    }
    return true;
}
var __VIRTUAL_FILE189 = baseIsMatch;
function isStrictComparable(value) {
    return value === value && !__VIRTUAL_FILE18(value);
}
var __VIRTUAL_FILE190 = isStrictComparable;
function getMatchData(object) {
    var result = __VIRTUAL_FILE68(object), length = result.length;
    while(length--){
        var key = result[length], value = object[key];
        result[length] = [
            key,
            value,
            __VIRTUAL_FILE190(value)
        ];
    }
    return result;
}
var __VIRTUAL_FILE191 = getMatchData;
function matchesStrictComparable(key, srcValue) {
    return function(object) {
        if (object == null) {
            return false;
        }
        return object[key] === srcValue && (srcValue !== void 0 || key in Object(object));
    };
}
var __VIRTUAL_FILE192 = matchesStrictComparable;
function baseMatches(source) {
    var matchData = __VIRTUAL_FILE191(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return __VIRTUAL_FILE192(matchData[0][0], matchData[0][1]);
    }
    return function(object) {
        return object === source || __VIRTUAL_FILE189(object, source, matchData);
    };
}
var __VIRTUAL_FILE193 = baseMatches;
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return false;
    }
    var type = typeof value;
    if (type == "number" || type == "symbol" || type == "boolean" || value == null || __VIRTUAL_FILE111(value)) {
        return true;
    }
    return reIsPlainProp.test(value) || !reIsDeepProp.test(value) || object != null && value in Object(object);
}
var __VIRTUAL_FILE194 = isKey;
var FUNC_ERROR_TEXT1 = "Expected a function";
function memoize(func, resolver) {
    if (typeof func != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT1);
    }
    var memoized = function() {
        var args = arguments, key = resolver ? resolver.apply(this, args) : args[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result = func.apply(this, args);
        memoized.cache = cache.set(key, result) || cache;
        return result;
    };
    memoized.cache = new (memoize.Cache || __VIRTUAL_FILE41)();
    return memoized;
}
memoize.Cache = __VIRTUAL_FILE41;
var __VIRTUAL_FILE195 = memoize;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func) {
    var result = __VIRTUAL_FILE195(func, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result.cache;
    return result;
}
var __VIRTUAL_FILE196 = memoizeCapped;
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = __VIRTUAL_FILE196(function(string9) {
    var result = [];
    if (string9.charCodeAt(0) === 46) {
        result.push("");
    }
    string9.replace(rePropName, function(match, number1, quote, subString) {
        result.push(quote ? subString.replace(reEscapeChar, "$1") : number1 || match);
    });
    return result;
});
var __VIRTUAL_FILE197 = stringToPath;
function castPath(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return value;
    }
    return __VIRTUAL_FILE194(value, object) ? [
        value
    ] : __VIRTUAL_FILE197(__VIRTUAL_FILE171(value));
}
var __VIRTUAL_FILE198 = castPath;
var INFINITY2 = 1 / 0;
function toKey(value) {
    if (typeof value == "string" || __VIRTUAL_FILE111(value)) {
        return value;
    }
    var result = value + "";
    return result == "0" && 1 / value == -INFINITY2 ? "-0" : result;
}
var __VIRTUAL_FILE199 = toKey;
function baseGet(object, path44) {
    path44 = __VIRTUAL_FILE198(path44, object);
    var index = 0, length = path44.length;
    while(object != null && index < length){
        object = object[__VIRTUAL_FILE199(path44[index++])];
    }
    return index && index == length ? object : void 0;
}
var __VIRTUAL_FILE200 = baseGet;
function get(object, path45, defaultValue) {
    var result = object == null ? void 0 : __VIRTUAL_FILE200(object, path45);
    return result === void 0 ? defaultValue : result;
}
var __VIRTUAL_FILE201 = get;
function baseHasIn(object, key) {
    return object != null && key in Object(object);
}
var __VIRTUAL_FILE202 = baseHasIn;
function hasPath(object, path46, hasFunc) {
    path46 = __VIRTUAL_FILE198(path46, object);
    var index = -1, length = path46.length, result = false;
    while(++index < length){
        var key = __VIRTUAL_FILE199(path46[index]);
        if (!(result = object != null && hasFunc(object, key))) {
            break;
        }
        object = object[key];
    }
    if (result || ++index != length) {
        return result;
    }
    length = object == null ? 0 : object.length;
    return !!length && __VIRTUAL_FILE57(length) && __VIRTUAL_FILE56(key, length) && (__VIRTUAL_FILE53(object) || __VIRTUAL_FILE52(object));
}
var __VIRTUAL_FILE203 = hasPath;
function hasIn(object, path47) {
    return object != null && __VIRTUAL_FILE203(object, path47, __VIRTUAL_FILE202);
}
var __VIRTUAL_FILE204 = hasIn;
var COMPARE_PARTIAL_FLAG5 = 1, COMPARE_UNORDERED_FLAG3 = 2;
function baseMatchesProperty(path48, srcValue) {
    if (__VIRTUAL_FILE194(path48) && __VIRTUAL_FILE190(srcValue)) {
        return __VIRTUAL_FILE192(__VIRTUAL_FILE199(path48), srcValue);
    }
    return function(object) {
        var objValue = __VIRTUAL_FILE201(object, path48);
        return objValue === void 0 && objValue === srcValue ? __VIRTUAL_FILE204(object, path48) : __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG5 | COMPARE_UNORDERED_FLAG3);
    };
}
var __VIRTUAL_FILE205 = baseMatchesProperty;
function baseProperty(key) {
    return function(object) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE206 = baseProperty;
function basePropertyDeep(path49) {
    return function(object) {
        return __VIRTUAL_FILE200(object, path49);
    };
}
var __VIRTUAL_FILE207 = basePropertyDeep;
function property(path50) {
    return __VIRTUAL_FILE194(path50) ? __VIRTUAL_FILE206(__VIRTUAL_FILE199(path50)) : __VIRTUAL_FILE207(path50);
}
var __VIRTUAL_FILE208 = property;
function baseIteratee(value) {
    if (typeof value == "function") {
        return value;
    }
    if (value == null) {
        return __VIRTUAL_FILE128;
    }
    if (typeof value == "object") {
        return __VIRTUAL_FILE53(value) ? __VIRTUAL_FILE205(value[0], value[1]) : __VIRTUAL_FILE193(value);
    }
    return __VIRTUAL_FILE208(value);
}
var __VIRTUAL_FILE209 = baseIteratee;
function uniqBy(array, iteratee) {
    return array && array.length ? __VIRTUAL_FILE180(array, __VIRTUAL_FILE209(iteratee, 2)) : [];
}
var __VIRTUAL_FILE210 = uniqBy;
function isEqual(value, other) {
    return __VIRTUAL_FILE188(value, other);
}
var __VIRTUAL_FILE211 = isEqual;
function baseMap(collection, iteratee) {
    var index = -1, result = __VIRTUAL_FILE67(collection) ? Array(collection.length) : [];
    __VIRTUAL_FILE142(collection, function(value, key, collection2) {
        result[++index] = iteratee(value, key, collection2);
    });
    return result;
}
var __VIRTUAL_FILE212 = baseMap;
function baseSortBy(array, comparer) {
    var length = array.length;
    array.sort(comparer);
    while(length--){
        array[length] = array[length].value;
    }
    return array;
}
var __VIRTUAL_FILE213 = baseSortBy;
function compareAscending(value, other) {
    if (value !== other) {
        var valIsDefined = value !== void 0, valIsNull = value === null, valIsReflexive = value === value, valIsSymbol = __VIRTUAL_FILE111(value);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = __VIRTUAL_FILE111(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
var __VIRTUAL_FILE214 = compareAscending;
function compareMultiple(object, other, orders) {
    var index = -1, objCriteria = object.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result = __VIRTUAL_FILE214(objCriteria[index], othCriteria[index]);
        if (result) {
            if (index >= ordersLength) {
                return result;
            }
            var order = orders[index];
            return result * (order == "desc" ? -1 : 1);
        }
    }
    return object.index - other.index;
}
var __VIRTUAL_FILE215 = compareMultiple;
function baseOrderBy(collection, iteratees, orders) {
    if (iteratees.length) {
        iteratees = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            if (__VIRTUAL_FILE53(iteratee)) {
                return function(value) {
                    return __VIRTUAL_FILE200(value, iteratee.length === 1 ? iteratee[0] : iteratee);
                };
            }
            return iteratee;
        });
    } else {
        iteratees = [
            __VIRTUAL_FILE128
        ];
    }
    var index = -1;
    iteratees = __VIRTUAL_FILE123(iteratees, __VIRTUAL_FILE59(__VIRTUAL_FILE209));
    var result = __VIRTUAL_FILE212(collection, function(value, key, collection2) {
        var criteria = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            return iteratee(value);
        });
        return {
            criteria,
            index: ++index,
            value
        };
    });
    return __VIRTUAL_FILE213(result, function(object, other) {
        return __VIRTUAL_FILE215(object, other, orders);
    });
}
var __VIRTUAL_FILE216 = baseOrderBy;
function orderBy(collection, iteratees, orders, guard) {
    if (collection == null) {
        return [];
    }
    if (!__VIRTUAL_FILE53(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!__VIRTUAL_FILE53(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return __VIRTUAL_FILE216(collection, iteratees, orders);
}
var __VIRTUAL_FILE217 = orderBy;
const cloneDeep1 = __VIRTUAL_FILE107;
const shuffle1 = __VIRTUAL_FILE161;
const toString1 = __VIRTUAL_FILE171;
const uniq1 = __VIRTUAL_FILE181;
const uniqBy1 = __VIRTUAL_FILE210;
const escape1 = __VIRTUAL_FILE172;
function deferred() {
    let methods;
    let state1 = "pending";
    const promise = new Promise((resolve3, reject)=>{
        methods = {
            async resolve (value) {
                await value;
                state1 = "fulfilled";
                resolve3(value);
            },
            reject (reason) {
                state1 = "rejected";
                reject(reason);
            }
        };
    });
    Object.defineProperty(promise, "state", {
        get: ()=>state1
    });
    return Object.assign(promise, methods);
}
class MuxAsyncIterator {
    iteratorCount = 0;
    yields = [];
    throws = [];
    signal = deferred();
    add(iterable) {
        ++this.iteratorCount;
        this.callIteratorNext(iterable[Symbol.asyncIterator]());
    }
    async callIteratorNext(iterator) {
        try {
            const { value , done  } = await iterator.next();
            if (done) {
                --this.iteratorCount;
            } else {
                this.yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.throws.push(e);
        }
        this.signal.resolve();
    }
    async *iterate() {
        while(this.iteratorCount > 0){
            await this.signal;
            for(let i28 = 0; i28 < this.yields.length; i28++){
                const { iterator , value  } = this.yields[i28];
                yield value;
                this.callIteratorNext(iterator);
            }
            if (this.throws.length) {
                for (const e of this.throws){
                    throw e;
                }
                this.throws.length = 0;
            }
            this.yields.length = 0;
            this.signal = deferred();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
function pooledMap(poolLimit, array, iteratorFn) {
    const res = new TransformStream({
        async transform (p, controller) {
            controller.enqueue(await p);
        }
    });
    (async ()=>{
        const writer = res.writable.getWriter();
        const executing = [];
        try {
            for await (const item of array){
                const p = Promise.resolve().then(()=>iteratorFn(item)
                );
                p.then((v)=>writer.write(Promise.resolve(v))
                ).catch(()=>{});
                const e = p.then(()=>executing.splice(executing.indexOf(e), 1)
                );
                executing.push(e);
                if (executing.length >= poolLimit) {
                    await Promise.race(executing);
                }
            }
            await Promise.all(executing);
            writer.close();
        } catch  {
            const errors1 = [];
            for (const result of (await Promise.allSettled(executing))){
                if (result.status == "rejected") {
                    errors1.push(result.reason);
                }
            }
            writer.write(Promise.reject(new AggregateError(errors1, "Threw while mapping."))).catch(()=>{});
        }
    })();
    return res.readable[Symbol.asyncIterator]();
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
function readAllSync(r) {
    const buf = new Buffer();
    buf.readFromSync(r);
    return buf.bytes();
}
async function writeAll(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += await w.write(arr.subarray(nwritten));
    }
}
function writeAllSync(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += w.writeSync(arr.subarray(nwritten));
    }
}
async function* iterateReader(r, options) {
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result = await r.read(b);
        if (result === null) {
            break;
        }
        yield b.subarray(0, result);
    }
}
"\r".charCodeAt(0);
"\n".charCodeAt(0);
async function execProcess(options, stdin, mergeOutput, stderrFilter) {
    try {
        const process = Deno.run({
            ...options,
            stdin: stdin !== undefined ? "piped" : options.stdin,
            stdout: typeof options.stdout === "number" ? options.stdout : "piped",
            stderr: typeof options.stderr === "number" ? options.stderr : "piped"
        });
        if (stdin !== undefined) {
            if (!process.stdin) {
                throw new Error("Process stdin not available");
            }
            const buffer = new TextEncoder().encode(stdin);
            let offset = 0;
            while(offset < buffer.length){
                const end = Math.min(offset + 4096, buffer.length);
                const window = buffer.subarray(offset, end);
                const written = await process.stdin.write(window);
                offset += written;
            }
            process.stdin.close();
        }
        let stdoutText = "";
        let stderrText = "";
        if (mergeOutput) {
            const multiplexIterator = new MuxAsyncIterator();
            const addStream = (stream, filter1)=>{
                if (stream !== null) {
                    const streamIter = filter1 ? filteredAsyncIterator(iterateReader(stream), filter1) : iterateReader(stream);
                    multiplexIterator.add(streamIter);
                }
            };
            addStream(process.stdout);
            addStream(process.stderr, stderrFilter);
            const allOutput = await processOutput(multiplexIterator, mergeOutput === "stderr>stdout" ? options.stdout : options.stderr);
            if (mergeOutput === "stderr>stdout") {
                stdoutText = allOutput;
            } else {
                stderrText = allOutput;
            }
            const closeStream = (stream)=>{
                if (stream) {
                    stream.close();
                }
            };
            closeStream(process.stdout);
            closeStream(process.stderr);
        } else {
            if (process.stdout !== null) {
                stdoutText = await processOutput(iterateReader(process.stdout), options.stdout);
                process.stdout.close();
            }
            if (process.stderr != null) {
                const iterator = stderrFilter ? filteredAsyncIterator(iterateReader(process.stderr), stderrFilter) : iterateReader(process.stderr);
                stderrText = await processOutput(iterator, options.stderr);
                process.stderr.close();
            }
        }
        const status = await process.status();
        process.close();
        return {
            success: status.success,
            code: status.code,
            stdout: stdoutText,
            stderr: stderrText
        };
    } catch (e) {
        throw new Error(`Error executing '${options.cmd[0]}': ${e.message}`);
    }
}
function processSuccessResult() {
    return {
        success: true,
        code: 0
    };
}
function filteredAsyncIterator(iterator, filter2) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    return pooledMap(1, iterator, (data16)=>{
        return Promise.resolve(encoder.encode(filter2(decoder.decode(data16))));
    });
}
async function processOutput(iterator, output) {
    const decoder = new TextDecoder();
    let outputText = "";
    for await (const chunk of iterator){
        if (output === "inherit" || output === undefined) {
            info(decoder.decode(chunk), {
                newline: false
            });
        }
        const text = decoder.decode(chunk);
        outputText += text;
    }
    return outputText;
}
const kSkipHidden = /[/\\][\.]/;
function removeIfExists(file) {
    if (existsSync(file)) {
        Deno.removeSync(file, {
            recursive: true
        });
    }
}
function removeIfEmptyDir(dir) {
    if (existsSync(dir)) {
        let empty1 = true;
        for (const _entry of Deno.readDirSync(dir)){
            empty1 = false;
            break;
        }
        if (empty1) {
            Deno.removeSync(dir, {
                recursive: true
            });
        }
    }
}
function isModifiedAfter(file, otherFile) {
    if (!existsSync(file)) {
        throw new Error(`${file} does not exist`);
    }
    if (!existsSync(otherFile)) {
        return true;
    }
    const fileInfo = Deno.statSync(file);
    const otherfileInfo = Deno.statSync(otherFile);
    if (fileInfo.mtime === null || otherfileInfo.mtime === null) {
        return true;
    } else {
        return fileInfo.mtime > otherfileInfo.mtime;
    }
}
function dirAndStem(file) {
    return [
        dirname2(file),
        basename2(file, extname2(file)), 
    ];
}
function expandPath(path51) {
    if (path51 === "~") {
        return getenv("HOME", "~");
    } else {
        return path51.replace(/^~\//, getenv("HOME", "~") + "/");
    }
}
function safeExistsSync(path52) {
    try {
        return existsSync(path52);
    } catch  {
        return false;
    }
}
async function which(cmd) {
    const args = Deno.build.os === "windows" ? [
        "CMD",
        "/C",
        "where",
        cmd
    ] : [
        "which",
        cmd
    ];
    const result = await execProcess({
        cmd: args,
        stderr: "piped",
        stdout: "piped"
    });
    if (result.code === 0) {
        return result.stdout?.trim();
    } else {
        return undefined;
    }
}
function filterPaths(root4, paths, globs) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            const regex = globToRegExp(`${root4}/${glob}`);
            const matchingFiles = paths.filter((path53)=>{
                return regex.test(path53);
            });
            expanded.push(...matchingFiles);
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root4, globs, expandGlobs);
}
function resolvePathGlobs(root5, globs, exclude) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            for (const file of expandGlobSync(glob, {
                root: root5,
                exclude,
                includeDirs: true,
                extended: true,
                globstar: true
            })){
                expanded.push(file.path);
            }
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root5, globs, expandGlobs);
}
function pathWithForwardSlashes(path54) {
    return path54.replace(/\\/g, "/");
}
function copyMinimal(srcDir, destDir, skip, filter3) {
    skip = skip || [];
    const srcFiles = [];
    for (const walk1 of walkSync(srcDir, {
        includeDirs: false,
        followSymlinks: false,
        skip
    })){
        const srcFile = walk1.path;
        if (filter3 && !filter3(srcFile)) {
            continue;
        }
        srcFiles.push(srcFile);
    }
    for (const srcFile of srcFiles){
        if (!existsSync(srcFile)) {
            continue;
        }
        const destFile = join3(destDir, relative2(srcDir, srcFile));
        copyFileIfNewer(srcFile, destFile);
    }
}
function copyFileIfNewer(srcFile, destFile) {
    ensureDirSync(dirname2(destFile));
    if (existsSync(destFile)) {
        const srcInfo = Deno.statSync(srcFile);
        const destInfo = Deno.statSync(destFile);
        if (!srcInfo.mtime || !destInfo.mtime || destInfo.mtime < srcInfo.mtime) {
            copySync(srcFile, destFile, {
                overwrite: true,
                preserveTimestamps: true
            });
        }
    } else {
        copySync(srcFile, destFile, {
            overwrite: true,
            preserveTimestamps: true
        });
    }
}
function resolveGlobs(root6, globs, expandGlobs) {
    const includeGlobs = [];
    const excludeGlobs = [];
    const asFullGlob = (glob)=>{
        if (glob.startsWith("\\!")) {
            glob = glob.slice(1);
        }
        if (glob.endsWith("/")) {
            glob = glob + "**/*";
        } else {
            const fullPath = join3(root6, glob);
            if (existsSync(fullPath) && Deno.statSync(fullPath).isDirectory) {
                glob = glob + "/**/*";
            }
        }
        if (!glob.startsWith("/")) {
            return "**/" + glob;
        } else {
            return join3(root6, glob.slice(1));
        }
    };
    for (const glob1 of globs){
        if (glob1.startsWith("!")) {
            excludeGlobs.push(asFullGlob(glob1.slice(1)));
        } else {
            includeGlobs.push(asFullGlob(glob1));
        }
    }
    const includeFiles = expandGlobs(includeGlobs);
    const excludeFiles = expandGlobs(excludeGlobs);
    return {
        include: includeFiles,
        exclude: excludeFiles
    };
}
const kLocalDevelopment = "99.9.9";
const quartoConfig = {
    binPath: ()=>getenv("QUARTO_BIN_PATH")
    ,
    sharePath: ()=>getenv("QUARTO_SHARE_PATH")
    ,
    isDebug: ()=>getenv("QUARTO_DEBUG", "false") === "true"
    ,
    version: ()=>{
        const versionPath = join3(getenv("QUARTO_SHARE_PATH"), "version");
        if (existsSync(versionPath)) {
            return Deno.readTextFileSync(versionPath);
        } else {
            return kLocalDevelopment;
        }
    }
};
const kHKeyCurrentUser = "HKCU";
const kHKeyLocalMachine = "HKLM";
async function registryReadString(root7, key, value) {
    if (Array.isArray(root7)) {
        for (const r of root7){
            const val = registryReadString(r, key, value);
            if (val !== undefined) {
                return val;
            }
        }
        return undefined;
    }
    const kTypeString = "REG_SZ";
    const cmd = [
        "reg",
        "query",
        `${root7}\\${key}`,
        "/v",
        value, 
    ];
    const result = await execProcess({
        cmd,
        stdout: "piped"
    });
    if (result.success && result.stdout) {
        const typePos = result.stdout?.indexOf(kTypeString);
        if (typePos !== -1) {
            return result.stdout.substring(typePos + kTypeString.length).trim();
        }
    }
    return undefined;
}
function resourcePath(resource) {
    const sharePath = quartoConfig.sharePath();
    if (resource) {
        return join3(sharePath, resource);
    } else {
        return sharePath;
    }
}
function formatResourcePath(format5, resource) {
    return join3(resourcePath("formats"), format5, resource);
}
function binaryPath(binary1) {
    return join3(quartoConfig.binPath(), binary1);
}
function pandocBinaryPath() {
    const quartoPandoc = Deno.env.get("QUARTO_PANDOC");
    if (quartoPandoc) {
        if (!existsSync(quartoPandoc)) {
            warnOnce("Specified QUARTO_PANDOC does not exist, using built in Pandoc");
        }
        if (Deno.statSync(quartoPandoc).isFile) {
            return quartoPandoc;
        } else {
            return join3(quartoPandoc, "pandoc");
        }
    }
    return binaryPath("pandoc");
}
async function rBinaryPath(binary2) {
    const rHome = Deno.env.get("R_HOME");
    if (rHome) {
        return join3(rHome, "bin", binary2);
    }
    const path55 = await which(binary2);
    if (path55) {
        return path55;
    }
    if (Deno.build.os === "windows") {
        const version6 = await registryReadString([
            kHKeyLocalMachine,
            kHKeyCurrentUser
        ], "Software\\R-core\\R", "Current Version");
        if (version6) {
            const installPath = await registryReadString([
                kHKeyLocalMachine,
                kHKeyCurrentUser
            ], `Software\\R-core\\R\\${version6}`, "InstallPath");
            if (installPath) {
                return join3(installPath, "bin", binary2);
            }
        }
        const progFiles = Deno.env.get("programfiles");
        if (progFiles) {
            for (const entry of Deno.readDirSync(progFiles)){
                if (entry.isDirectory && entry.name === "R") {
                    for (const walk2 of walkSync(join3(progFiles, "R"))){
                        if (walk2.isDirectory && walk2.name === "bin") {
                            return join3(walk2.path, binary2);
                        }
                    }
                }
            }
        }
    }
    return binary2;
}
function projectTypeResourcePath(projectType1) {
    return resourcePath(join3("projects", projectType1));
}
const kDarkSuffix = "dark";
const kLightSuffix = "light";
function textHighlightThemePath(theme, style) {
    const names = [
        `${theme}-${style === "dark" ? kDarkSuffix : kLightSuffix}`,
        theme, 
    ];
    const themePath = names.map((name75)=>{
        return resourcePath(join3("pandoc", "highlight-styles", `${name75}.theme`));
    }).find((path56)=>existsSync(path56)
    );
    return themePath;
}
function schemaType(schema) {
    const t = schema.type;
    if (t) {
        return t;
    }
    if (schema.anyOf) {
        return "anyOf";
    }
    if (schema.oneOf) {
        return "oneOf";
    }
    if (schema.allOf) {
        return "allOf";
    }
    if (schema.enum) {
        return "enum";
    }
    return "any";
}
function walkSchema(schema, f) {
    const t = schemaType(schema);
    if (typeof f === "function") {
        if (f(schema) === true) {
            return;
        }
    } else {
        if (f[t] !== undefined) {
            if (f[t](schema) === true) {
                return;
            }
        }
    }
    switch(t){
        case "array":
            if (schema.items) {
                walkSchema(schema.items, f);
            }
            break;
        case "anyOf":
            for (const s of schema.anyOf){
                walkSchema(s, f);
            }
            break;
        case "oneOf":
            for (const s1 of schema.oneOf){
                walkSchema(s1, f);
            }
            break;
        case "allOf":
            for (const s2 of schema.allOf){
                walkSchema(s2, f);
            }
            break;
        case "object":
            if (schema.properties) {
                for (const key of Object.getOwnPropertyNames(schema.properties)){
                    const s = schema.properties[key];
                    walkSchema(s, f);
                }
            }
            if (schema.patternProperties) {
                for (const key of Object.getOwnPropertyNames(schema.patternProperties)){
                    const s = schema.patternProperties[key];
                    walkSchema(s, f);
                }
            }
            if (schema.additionalProperties) {
                walkSchema(schema.additionalProperties, f);
            }
            break;
    }
}
function normalizeSchema(schema1) {
    const result = JSON.parse(JSON.stringify(schema1));
    walkSchema(result, (schema)=>{
        if (schema.completions) {
            delete schema.completions;
        }
        if (schema.exhaustiveCompletions) {
            delete schema.exhaustiveCompletions;
        }
        if (schema.documentation) {
            delete schema.documentation;
        }
        if (schema.tags) {
            delete schema.tags;
        }
    });
    return result;
}
const definitionsObject = {};
function hasSchemaDefinition(key) {
    return definitionsObject[key] !== undefined;
}
function getSchemaDefinition(key) {
    if (definitionsObject[key] === undefined) {
        throw new Error(`Internal Error: Schema ${key} not found.`);
    }
    return definitionsObject[key];
}
function setSchemaDefinition(schema) {
    if (definitionsObject[schema.$id] === undefined) {
        definitionsObject[schema.$id] = schema;
    }
}
function getSchemaDefinitionsObject() {
    return Object.assign({}, definitionsObject);
}
function expandAliasesFrom(lst, defs) {
    const aliases = defs;
    const result = [];
    lst = lst.slice();
    for(let i29 = 0; i29 < lst.length; ++i29){
        const el = lst[i29];
        if (el.startsWith("$")) {
            const v = aliases[el.slice(1)];
            if (v === undefined) {
                throw new Error(`Internal Error: ${el} doesn't have an entry in the aliases map`);
            }
            lst.push(...v);
        } else {
            result.push(el);
        }
    }
    return result;
}
Object.freeze({
    __proto__: null
});
let largeIdStartRanges = undefined;
let largeIdContinueRanges = undefined;
function isIdStart(cp) {
    if (cp < 65) return false;
    if (cp < 91) return true;
    if (cp < 97) return false;
    if (cp < 123) return true;
    return isLargeIdStart(cp);
}
function isIdContinue(cp) {
    if (cp < 48) return false;
    if (cp < 58) return true;
    if (cp < 65) return false;
    if (cp < 91) return true;
    if (cp === 95) return true;
    if (cp < 97) return false;
    if (cp < 123) return true;
    return isLargeIdStart(cp) || isLargeIdContinue(cp);
}
function isLargeIdStart(cp) {
    return isInRange(cp, largeIdStartRanges || (largeIdStartRanges = initLargeIdStartRanges()));
}
function isLargeIdContinue(cp) {
    return isInRange(cp, largeIdContinueRanges || (largeIdContinueRanges = initLargeIdContinueRanges()));
}
function initLargeIdStartRanges() {
    return restoreRanges("4q 0 b 0 5 0 6 m 2 u 2 cp 5 b f 4 8 0 2 0 3m 4 2 1 3 3 2 0 7 0 2 2 2 0 2 j 2 2a 2 3u 9 4l 2 11 3 0 7 14 20 q 5 3 1a 16 10 1 2 2q 2 0 g 1 8 1 b 2 3 0 h 0 2 t u 2g c 0 p w a 1 5 0 6 l 5 0 a 0 4 0 o o 8 a 1i k 2 h 1p 1h 4 0 j 0 8 9 g f 5 7 3 1 3 l 2 6 2 0 4 3 4 0 h 0 e 1 2 2 f 1 b 0 9 5 5 1 3 l 2 6 2 1 2 1 2 1 w 3 2 0 k 2 h 8 2 2 2 l 2 6 2 1 2 4 4 0 j 0 g 1 o 0 c 7 3 1 3 l 2 6 2 1 2 4 4 0 v 1 2 2 g 0 i 0 2 5 4 2 2 3 4 1 2 0 2 1 4 1 4 2 4 b n 0 1h 7 2 2 2 m 2 f 4 0 r 2 6 1 v 0 5 7 2 2 2 m 2 9 2 4 4 0 x 0 2 1 g 1 i 8 2 2 2 14 3 0 h 0 6 2 9 2 p 5 6 h 4 n 2 8 2 0 3 6 1n 1b 2 1 d 6 1n 1 2 0 2 4 2 n 2 0 2 9 2 1 a 0 3 4 2 0 m 3 x 0 1s 7 2 z s 4 38 16 l 0 h 5 5 3 4 0 4 1 8 2 5 c d 0 i 11 2 0 6 0 3 16 2 98 2 3 3 6 2 0 2 3 3 14 2 3 3 w 2 3 3 6 2 0 2 3 3 e 2 1k 2 3 3 1u 12 f h 2d 3 5 4 h7 3 g 2 p 6 22 4 a 8 c 2 3 f h f h f c 2 2 g 1f 10 0 5 0 1w 2g 8 14 2 0 6 1x b u 1e t 3 4 c 17 5 p 1j m a 1g 2b 0 2m 1a i 6 1k t e 1 b 17 r z 16 2 b z 3 8 8 16 3 2 16 3 2 5 2 1 4 0 6 5b 1t 7p 3 5 3 11 3 5 3 7 2 0 2 0 2 0 2 u 3 1g 2 6 2 0 4 2 2 6 4 3 3 5 5 c 6 2 2 6 39 0 e 0 h c 2u 0 5 0 3 9 2 0 3 5 7 0 2 0 2 0 2 f 3 3 6 4 5 0 i 14 22g 1a 2 1a 2 3o 7 3 4 1 d 11 2 0 6 0 3 1j 8 0 h m a 6 2 6 2 6 2 6 2 6 2 6 2 6 2 6 fb 2 q 8 8 4 3 4 5 2d 5 4 2 2h 2 3 6 16 2 2l i v 1d f e9 533 1t g70 4 wc 1w 19 3 7g 4 f b 1 l 1a h u 3 27 14 8 3 2u 3 1g 3 8 17 c 2 2 2 3 2 m u 1f f 1d 1r 5 4 0 2 1 c r b m q s 8 1a t 0 h 4 2 9 b 4 2 14 o 2 2 7 l m 4 0 4 1d 2 0 4 1 3 4 3 0 2 0 p 2 3 a 8 2 d 5 3 5 3 5 a 6 2 6 2 16 2 d 7 36 u 8mb d m 5 1c 6it a5 3 2x 13 6 d 4 6 0 2 9 2 c 2 4 2 0 2 1 2 1 2 2z y a2 j 1r 3 1h 15 b 39 4 2 3q 11 p 7 p c 2g 4 5 3 5 3 5 3 2 10 b 2 p 2 i 2 1 2 e 3 d z 3e 1y 1g 7g s 4 1c 1c v e t 6 11 b t 3 z 5 7 2 4 17 4d j z 5 z 5 13 9 1f 4d 8m a l b 7 49 5 3 0 2 17 2 1 4 0 3 m b m a u 1u i 2 1 b l b p 1z 1j 7 1 1t 0 g 3 2 2 2 s 17 s 4 s 10 7 2 r s 1h b l b i e h 33 20 1k 1e e 1e e z 9p 15 7 1 27 s b 0 9 l 2z k s m d 1g 24 18 x o r z u 0 3 0 9 y 4 0 d 1b f 3 m 0 2 0 10 h 2 o 2d 6 2 0 2 3 2 e 2 9 8 1a 13 7 3 1 3 l 2 6 2 1 2 4 4 0 j 0 d 4 4f 1g j 3 l 2 v 1b l 1 2 0 55 1a 16 3 11 1b l 0 1o 16 e 0 20 q 6e 17 39 1r w 7 3 0 3 7 2 1 2 n g 0 2 0 2n 7 3 12 h 0 2 0 t 0 b 13 8 0 m 0 c 19 k 0 z 1k 7c 8 2 10 i 0 1e t 35 6 2 1 2 11 m 0 q 5 2 1 2 v f 0 94 i 5a 0 28 pl 2v 32 i 5f 24d tq 34i g6 6nu fs 8 u 36 t j 1b h 3 w k 6 i j5 1r 3l 22 6 0 1v c 1t 1 2 0 t 4qf 9 yd 17 8 6wo 7y 1e 2 i 3 9 az 1s5 2y 6 c 4 8 8 9 4mf 2c 2 1y 2 1 3 0 3 1 3 3 2 b 2 0 2 6 2 1s 2 3 3 7 2 6 2 r 2 3 2 4 2 0 4 6 2 9f 3 o 2 o 2 u 2 o 2 u 2 o 2 u 2 o 2 u 2 o 2 7 1th 18 b 6 h 0 aa 17 105 5g 1o 1v 8 0 xh 3 2 q 2 1 2 0 3 0 2 9 2 3 2 0 2 0 7 0 5 0 2 0 2 0 2 2 2 1 2 0 3 0 2 0 2 0 2 0 2 0 2 1 2 0 3 3 2 6 2 3 2 3 2 0 2 9 2 g 6 2 2 4 2 g 3et wyl z 378 c 65 3 4g1 f 5rk 2e8 f1 15v 3t6");
}
function initLargeIdContinueRanges() {
    return restoreRanges("53 0 g9 33 o 0 70 4 7e 18 2 0 2 1 2 1 2 0 21 a 1d u 7 0 2u 6 3 5 3 1 2 3 3 9 o 0 v q 2k a g 9 y 8 a 0 p 3 2 8 2 2 2 4 18 2 3c e 2 w 1j 2 2 h 2 6 b 1 3 9 i 2 1l 0 2 6 3 1 3 2 a 0 b 1 3 9 f 0 3 2 1l 0 2 4 5 1 3 2 4 0 l b 4 0 c 2 1l 0 2 7 2 2 2 2 l 1 3 9 b 5 2 2 1l 0 2 6 3 1 3 2 8 2 b 1 3 9 j 0 1o 4 4 2 2 3 a 0 f 9 h 4 1m 6 2 2 2 3 8 1 c 1 3 9 i 2 1l 0 2 6 2 2 2 3 8 1 c 1 3 9 h 3 1k 1 2 6 2 2 2 3 a 0 b 1 3 9 i 2 1z 0 5 5 2 0 2 7 7 9 3 1 1q 0 3 6 d 7 2 9 2g 0 3 8 c 5 3 9 1r 1 7 9 c 0 2 0 2 0 5 1 1e j 2 1 6 a 2 z a 0 2t j 2 9 d 3 5 2 2 2 3 6 4 3 e b 2 e jk 2 a 8 pt 2 u 2 u 1 v 1 1t v a 0 3 9 y 2 3 9 40 0 3b b 5 b b 9 3l a 1p 4 1m 9 2 s 3 a 7 9 n d 2 1 1s 4 1c g c 9 i 8 d 2 v c 3 9 19 d 1d j 9 9 7 9 3b 2 2 k 5 0 7 0 3 2 5j 1l 2 4 g0 1 k 0 3g c 5 0 4 b 2db 2 3y 0 2p v ff 5 2y 1 n7q 9 1y 0 5 9 x 1 29 1 7l 0 4 0 5 0 o 4 5 0 2c 1 1f h b 9 7 h e a t 7 q c 19 3 1c d g 9 c 0 b 9 1c d d 0 9 1 3 9 y 2 1f 0 2 2 3 1 6 1 2 0 16 4 6 1 6l 7 2 1 3 9 fmt 0 ki f h f 4 1 p 2 5d 9 12 0 ji 0 6b 0 46 4 86 9 120 2 2 1 6 3 15 2 5 0 4m 1 fy 3 9 9 aa 1 4a a 4w 2 1i e w 9 g 3 1a a 1i 9 7 2 11 d 2 9 6 1 19 0 d 2 1d d 9 3 2 b 2b b 7 0 4h b 6 9 7 3 1k 1 2 6 3 1 3 2 a 0 b 1 3 6 4 4 5d h a 9 5 0 2a j d 9 5y 6 3 8 s 1 2b g g 9 2a c 9 9 2c e 5 9 6r e 4m 9 1z 5 2 1 3 3 2 0 2 1 d 9 3c 6 3 6 4 0 t 9 15 6 2 3 9 0 a a 1b f ba 7 2 7 h 9 1l l 2 d 3f 5 4 0 2 1 2 6 2 0 9 9 1d 4 2 1 2 4 9 9 96 3 ewa 9 3r 4 1o 6 q 9 s6 0 2 1i 8 3 2a 0 c 1 f58 1 43r 4 4 5 9 7 3 6 v 3 45 2 13e 1d e9 1i 5 1d 9 0 f 0 n 4 2 e 11t 6 2 g 3 6 2 1 2 4 7a 6 a 9 bn d 15j 6 32 6 6 9 3o7 9 gvt3 6n");
}
function isInRange(cp, ranges) {
    let l = 0, r = ranges.length / 2 | 0, i30 = 0, min = 0, max = 0;
    while(l < r){
        i30 = (l + r) / 2 | 0;
        min = ranges[2 * i30];
        max = ranges[2 * i30 + 1];
        if (cp < min) {
            r = i30;
        } else if (cp > max) {
            l = i30 + 1;
        } else {
            return true;
        }
    }
    return false;
}
function restoreRanges(data17) {
    let last = 0;
    return data17.split(" ").map((s)=>last += parseInt(s, 36) | 0
    );
}
class DataSet {
    constructor(raw2018, raw2019, raw2020, raw2021){
        this._raw2018 = raw2018;
        this._raw2019 = raw2019;
        this._raw2020 = raw2020;
        this._raw2021 = raw2021;
    }
    get es2018() {
        return this._set2018 || (this._set2018 = new Set(this._raw2018.split(" ")));
    }
    get es2019() {
        return this._set2019 || (this._set2019 = new Set(this._raw2019.split(" ")));
    }
    get es2020() {
        return this._set2020 || (this._set2020 = new Set(this._raw2020.split(" ")));
    }
    get es2021() {
        return this._set2021 || (this._set2021 = new Set(this._raw2021.split(" ")));
    }
}
const gcNameSet = new Set([
    "General_Category",
    "gc"
]);
const scNameSet = new Set([
    "Script",
    "Script_Extensions",
    "sc",
    "scx"
]);
const gcValueSets = new DataSet("C Cased_Letter Cc Cf Close_Punctuation Cn Co Combining_Mark Connector_Punctuation Control Cs Currency_Symbol Dash_Punctuation Decimal_Number Enclosing_Mark Final_Punctuation Format Initial_Punctuation L LC Letter Letter_Number Line_Separator Ll Lm Lo Lowercase_Letter Lt Lu M Mark Math_Symbol Mc Me Mn Modifier_Letter Modifier_Symbol N Nd Nl No Nonspacing_Mark Number Open_Punctuation Other Other_Letter Other_Number Other_Punctuation Other_Symbol P Paragraph_Separator Pc Pd Pe Pf Pi Po Private_Use Ps Punctuation S Sc Separator Sk Sm So Space_Separator Spacing_Mark Surrogate Symbol Titlecase_Letter Unassigned Uppercase_Letter Z Zl Zp Zs cntrl digit punct", "", "", "");
const scValueSets = new DataSet("Adlam Adlm Aghb Ahom Anatolian_Hieroglyphs Arab Arabic Armenian Armi Armn Avestan Avst Bali Balinese Bamu Bamum Bass Bassa_Vah Batak Batk Beng Bengali Bhaiksuki Bhks Bopo Bopomofo Brah Brahmi Brai Braille Bugi Buginese Buhd Buhid Cakm Canadian_Aboriginal Cans Cari Carian Caucasian_Albanian Chakma Cham Cher Cherokee Common Copt Coptic Cprt Cuneiform Cypriot Cyrillic Cyrl Deseret Deva Devanagari Dsrt Dupl Duployan Egyp Egyptian_Hieroglyphs Elba Elbasan Ethi Ethiopic Geor Georgian Glag Glagolitic Gonm Goth Gothic Gran Grantha Greek Grek Gujarati Gujr Gurmukhi Guru Han Hang Hangul Hani Hano Hanunoo Hatr Hatran Hebr Hebrew Hira Hiragana Hluw Hmng Hung Imperial_Aramaic Inherited Inscriptional_Pahlavi Inscriptional_Parthian Ital Java Javanese Kaithi Kali Kana Kannada Katakana Kayah_Li Khar Kharoshthi Khmer Khmr Khoj Khojki Khudawadi Knda Kthi Lana Lao Laoo Latin Latn Lepc Lepcha Limb Limbu Lina Linb Linear_A Linear_B Lisu Lyci Lycian Lydi Lydian Mahajani Mahj Malayalam Mand Mandaic Mani Manichaean Marc Marchen Masaram_Gondi Meetei_Mayek Mend Mende_Kikakui Merc Mero Meroitic_Cursive Meroitic_Hieroglyphs Miao Mlym Modi Mong Mongolian Mro Mroo Mtei Mult Multani Myanmar Mymr Nabataean Narb Nbat New_Tai_Lue Newa Nko Nkoo Nshu Nushu Ogam Ogham Ol_Chiki Olck Old_Hungarian Old_Italic Old_North_Arabian Old_Permic Old_Persian Old_South_Arabian Old_Turkic Oriya Orkh Orya Osage Osge Osma Osmanya Pahawh_Hmong Palm Palmyrene Pau_Cin_Hau Pauc Perm Phag Phags_Pa Phli Phlp Phnx Phoenician Plrd Prti Psalter_Pahlavi Qaac Qaai Rejang Rjng Runic Runr Samaritan Samr Sarb Saur Saurashtra Sgnw Sharada Shavian Shaw Shrd Sidd Siddham SignWriting Sind Sinh Sinhala Sora Sora_Sompeng Soyo Soyombo Sund Sundanese Sylo Syloti_Nagri Syrc Syriac Tagalog Tagb Tagbanwa Tai_Le Tai_Tham Tai_Viet Takr Takri Tale Talu Tamil Taml Tang Tangut Tavt Telu Telugu Tfng Tglg Thaa Thaana Thai Tibetan Tibt Tifinagh Tirh Tirhuta Ugar Ugaritic Vai Vaii Wara Warang_Citi Xpeo Xsux Yi Yiii Zanabazar_Square Zanb Zinh Zyyy", "Dogr Dogra Gong Gunjala_Gondi Hanifi_Rohingya Maka Makasar Medefaidrin Medf Old_Sogdian Rohg Sogd Sogdian Sogo", "Elym Elymaic Hmnp Nand Nandinagari Nyiakeng_Puachue_Hmong Wancho Wcho", "Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi");
const binPropertySets = new DataSet("AHex ASCII ASCII_Hex_Digit Alpha Alphabetic Any Assigned Bidi_C Bidi_Control Bidi_M Bidi_Mirrored CI CWCF CWCM CWKCF CWL CWT CWU Case_Ignorable Cased Changes_When_Casefolded Changes_When_Casemapped Changes_When_Lowercased Changes_When_NFKC_Casefolded Changes_When_Titlecased Changes_When_Uppercased DI Dash Default_Ignorable_Code_Point Dep Deprecated Dia Diacritic Emoji Emoji_Component Emoji_Modifier Emoji_Modifier_Base Emoji_Presentation Ext Extender Gr_Base Gr_Ext Grapheme_Base Grapheme_Extend Hex Hex_Digit IDC IDS IDSB IDST IDS_Binary_Operator IDS_Trinary_Operator ID_Continue ID_Start Ideo Ideographic Join_C Join_Control LOE Logical_Order_Exception Lower Lowercase Math NChar Noncharacter_Code_Point Pat_Syn Pat_WS Pattern_Syntax Pattern_White_Space QMark Quotation_Mark RI Radical Regional_Indicator SD STerm Sentence_Terminal Soft_Dotted Term Terminal_Punctuation UIdeo Unified_Ideograph Upper Uppercase VS Variation_Selector White_Space XIDC XIDS XID_Continue XID_Start space", "Extended_Pictographic", "", "EBase EComp EMod EPres ExtPict");
function isValidUnicodeProperty(version7, name76, value) {
    if (gcNameSet.has(name76)) {
        return version7 >= 2018 && gcValueSets.es2018.has(value);
    }
    if (scNameSet.has(name76)) {
        return version7 >= 2018 && scValueSets.es2018.has(value) || version7 >= 2019 && scValueSets.es2019.has(value) || version7 >= 2020 && scValueSets.es2020.has(value) || version7 >= 2021 && scValueSets.es2021.has(value);
    }
    return false;
}
function isValidLoneUnicodeProperty(version8, value) {
    return version8 >= 2018 && binPropertySets.es2018.has(value) || version8 >= 2019 && binPropertySets.es2019.has(value) || version8 >= 2021 && binPropertySets.es2021.has(value);
}
const Backspace = 8;
const CharacterTabulation = 9;
const LineFeed = 10;
const LineTabulation = 11;
const FormFeed = 12;
const CarriageReturn = 13;
const ExclamationMark = 33;
const HyphenMinus = 45;
const DigitZero = 48;
const QuestionMark = 63;
const LatinCapitalLetterP = 80;
const ReverseSolidus = 92;
function isLatinLetter(code18) {
    return code18 >= 65 && code18 <= 90 || code18 >= 97 && code18 <= 122;
}
function isDecimalDigit(code19) {
    return code19 >= 48 && code19 <= 57;
}
function isOctalDigit(code20) {
    return code20 >= 48 && code20 <= 55;
}
function isHexDigit(code21) {
    return code21 >= 48 && code21 <= 57 || code21 >= 65 && code21 <= 70 || code21 >= 97 && code21 <= 102;
}
function isLineTerminator(code22) {
    return code22 === 10 || code22 === 13 || code22 === 8232 || code22 === 8233;
}
function isValidUnicode(code23) {
    return code23 >= 0 && code23 <= 1114111;
}
function digitToInt(code24) {
    if (code24 >= 97 && code24 <= 102) {
        return code24 - 97 + 10;
    }
    if (code24 >= 65 && code24 <= 70) {
        return code24 - 65 + 10;
    }
    return code24 - 48;
}
function isLeadSurrogate(code25) {
    return code25 >= 55296 && code25 <= 56319;
}
function isTrailSurrogate(code26) {
    return code26 >= 56320 && code26 <= 57343;
}
function combineSurrogatePair(lead, trail) {
    return (lead - 55296) * 1024 + (trail - 56320) + 65536;
}
const legacyImpl = {
    at (s, end, i31) {
        return i31 < end ? s.charCodeAt(i31) : -1;
    },
    width (c) {
        return 1;
    }
};
const unicodeImpl = {
    at (s, end, i32) {
        return i32 < end ? s.codePointAt(i32) : -1;
    },
    width (c) {
        return c > 65535 ? 2 : 1;
    }
};
class Reader {
    constructor(){
        this._impl = legacyImpl;
        this._s = "";
        this._i = 0;
        this._end = 0;
        this._cp1 = -1;
        this._w1 = 1;
        this._cp2 = -1;
        this._w2 = 1;
        this._cp3 = -1;
        this._w3 = 1;
        this._cp4 = -1;
    }
    get source() {
        return this._s;
    }
    get index() {
        return this._i;
    }
    get currentCodePoint() {
        return this._cp1;
    }
    get nextCodePoint() {
        return this._cp2;
    }
    get nextCodePoint2() {
        return this._cp3;
    }
    get nextCodePoint3() {
        return this._cp4;
    }
    reset(source, start, end, uFlag) {
        this._impl = uFlag ? unicodeImpl : legacyImpl;
        this._s = source;
        this._end = end;
        this.rewind(start);
    }
    rewind(index) {
        const impl = this._impl;
        this._i = index;
        this._cp1 = impl.at(this._s, this._end, index);
        this._w1 = impl.width(this._cp1);
        this._cp2 = impl.at(this._s, this._end, index + this._w1);
        this._w2 = impl.width(this._cp2);
        this._cp3 = impl.at(this._s, this._end, index + this._w1 + this._w2);
        this._w3 = impl.width(this._cp3);
        this._cp4 = impl.at(this._s, this._end, index + this._w1 + this._w2 + this._w3);
    }
    advance() {
        if (this._cp1 !== -1) {
            const impl = this._impl;
            this._i += this._w1;
            this._cp1 = this._cp2;
            this._w1 = this._w2;
            this._cp2 = this._cp3;
            this._w2 = impl.width(this._cp2);
            this._cp3 = this._cp4;
            this._w3 = impl.width(this._cp3);
            this._cp4 = impl.at(this._s, this._end, this._i + this._w1 + this._w2 + this._w3);
        }
    }
    eat(cp) {
        if (this._cp1 === cp) {
            this.advance();
            return true;
        }
        return false;
    }
    eat2(cp1, cp2) {
        if (this._cp1 === cp1 && this._cp2 === cp2) {
            this.advance();
            this.advance();
            return true;
        }
        return false;
    }
    eat3(cp1, cp2, cp3) {
        if (this._cp1 === cp1 && this._cp2 === cp2 && this._cp3 === cp3) {
            this.advance();
            this.advance();
            this.advance();
            return true;
        }
        return false;
    }
}
class RegExpSyntaxError extends SyntaxError {
    constructor(source, uFlag, index, message){
        if (source) {
            if (!source.startsWith("/")) {
                source = `/${source}/${uFlag ? "u" : ""}`;
            }
            source = `: ${source}`;
        }
        super(`Invalid regular expression${source}: ${message}`);
        this.index = index;
    }
}
function isSyntaxCharacter(cp) {
    return cp === 94 || cp === 36 || cp === 92 || cp === 46 || cp === 42 || cp === 43 || cp === 63 || cp === 40 || cp === 41 || cp === 91 || cp === 93 || cp === 123 || cp === 125 || cp === 124;
}
function isRegExpIdentifierStart(cp) {
    return isIdStart(cp) || cp === 36 || cp === 95;
}
function isRegExpIdentifierPart(cp) {
    return isIdContinue(cp) || cp === 36 || cp === 95 || cp === 8204 || cp === 8205;
}
function isUnicodePropertyNameCharacter(cp) {
    return isLatinLetter(cp) || cp === 95;
}
function isUnicodePropertyValueCharacter(cp) {
    return isUnicodePropertyNameCharacter(cp) || isDecimalDigit(cp);
}
class RegExpValidator {
    constructor(options){
        this._reader = new Reader();
        this._uFlag = false;
        this._nFlag = false;
        this._lastIntValue = 0;
        this._lastMinValue = 0;
        this._lastMaxValue = 0;
        this._lastStrValue = "";
        this._lastKeyValue = "";
        this._lastValValue = "";
        this._lastAssertionIsQuantifiable = false;
        this._numCapturingParens = 0;
        this._groupNames = new Set();
        this._backreferenceNames = new Set();
        this._options = options || {};
    }
    validateLiteral(source, start = 0, end = source.length) {
        this._uFlag = this._nFlag = false;
        this.reset(source, start, end);
        this.onLiteralEnter(start);
        if (this.eat(47) && this.eatRegExpBody() && this.eat(47)) {
            const flagStart = this.index;
            const uFlag = source.includes("u", flagStart);
            this.validateFlags(source, flagStart, end);
            this.validatePattern(source, start + 1, flagStart - 1, uFlag);
        } else if (start >= end) {
            this.raise("Empty");
        } else {
            const c = String.fromCodePoint(this.currentCodePoint);
            this.raise(`Unexpected character '${c}'`);
        }
        this.onLiteralLeave(start, end);
    }
    validateFlags(source, start = 0, end = source.length) {
        const existingFlags = new Set();
        let global = false;
        let ignoreCase = false;
        let multiline = false;
        let sticky = false;
        let unicode = false;
        let dotAll = false;
        let hasIndices = false;
        for(let i33 = start; i33 < end; ++i33){
            const flag = source.charCodeAt(i33);
            if (existingFlags.has(flag)) {
                this.raise(`Duplicated flag '${source[i33]}'`);
            }
            existingFlags.add(flag);
            if (flag === 103) {
                global = true;
            } else if (flag === 105) {
                ignoreCase = true;
            } else if (flag === 109) {
                multiline = true;
            } else if (flag === 117 && this.ecmaVersion >= 2015) {
                unicode = true;
            } else if (flag === 121 && this.ecmaVersion >= 2015) {
                sticky = true;
            } else if (flag === 115 && this.ecmaVersion >= 2018) {
                dotAll = true;
            } else if (flag === 100 && this.ecmaVersion >= 2022) {
                hasIndices = true;
            } else {
                this.raise(`Invalid flag '${source[i33]}'`);
            }
        }
        this.onFlags(start, end, global, ignoreCase, multiline, unicode, sticky, dotAll, hasIndices);
    }
    validatePattern(source, start = 0, end = source.length, uFlag = false) {
        this._uFlag = uFlag && this.ecmaVersion >= 2015;
        this._nFlag = uFlag && this.ecmaVersion >= 2018;
        this.reset(source, start, end);
        this.consumePattern();
        if (!this._nFlag && this.ecmaVersion >= 2018 && this._groupNames.size > 0) {
            this._nFlag = true;
            this.rewind(start);
            this.consumePattern();
        }
    }
    get strict() {
        return Boolean(this._options.strict || this._uFlag);
    }
    get ecmaVersion() {
        return this._options.ecmaVersion || 2022;
    }
    onLiteralEnter(start) {
        if (this._options.onLiteralEnter) {
            this._options.onLiteralEnter(start);
        }
    }
    onLiteralLeave(start, end) {
        if (this._options.onLiteralLeave) {
            this._options.onLiteralLeave(start, end);
        }
    }
    onFlags(start, end, global, ignoreCase, multiline, unicode, sticky, dotAll, hasIndices) {
        if (this._options.onFlags) {
            this._options.onFlags(start, end, global, ignoreCase, multiline, unicode, sticky, dotAll, hasIndices);
        }
    }
    onPatternEnter(start) {
        if (this._options.onPatternEnter) {
            this._options.onPatternEnter(start);
        }
    }
    onPatternLeave(start, end) {
        if (this._options.onPatternLeave) {
            this._options.onPatternLeave(start, end);
        }
    }
    onDisjunctionEnter(start) {
        if (this._options.onDisjunctionEnter) {
            this._options.onDisjunctionEnter(start);
        }
    }
    onDisjunctionLeave(start, end) {
        if (this._options.onDisjunctionLeave) {
            this._options.onDisjunctionLeave(start, end);
        }
    }
    onAlternativeEnter(start, index) {
        if (this._options.onAlternativeEnter) {
            this._options.onAlternativeEnter(start, index);
        }
    }
    onAlternativeLeave(start, end, index) {
        if (this._options.onAlternativeLeave) {
            this._options.onAlternativeLeave(start, end, index);
        }
    }
    onGroupEnter(start) {
        if (this._options.onGroupEnter) {
            this._options.onGroupEnter(start);
        }
    }
    onGroupLeave(start, end) {
        if (this._options.onGroupLeave) {
            this._options.onGroupLeave(start, end);
        }
    }
    onCapturingGroupEnter(start, name77) {
        if (this._options.onCapturingGroupEnter) {
            this._options.onCapturingGroupEnter(start, name77);
        }
    }
    onCapturingGroupLeave(start, end, name78) {
        if (this._options.onCapturingGroupLeave) {
            this._options.onCapturingGroupLeave(start, end, name78);
        }
    }
    onQuantifier(start, end, min, max, greedy) {
        if (this._options.onQuantifier) {
            this._options.onQuantifier(start, end, min, max, greedy);
        }
    }
    onLookaroundAssertionEnter(start, kind, negate) {
        if (this._options.onLookaroundAssertionEnter) {
            this._options.onLookaroundAssertionEnter(start, kind, negate);
        }
    }
    onLookaroundAssertionLeave(start, end, kind, negate) {
        if (this._options.onLookaroundAssertionLeave) {
            this._options.onLookaroundAssertionLeave(start, end, kind, negate);
        }
    }
    onEdgeAssertion(start, end, kind) {
        if (this._options.onEdgeAssertion) {
            this._options.onEdgeAssertion(start, end, kind);
        }
    }
    onWordBoundaryAssertion(start, end, kind, negate) {
        if (this._options.onWordBoundaryAssertion) {
            this._options.onWordBoundaryAssertion(start, end, kind, negate);
        }
    }
    onAnyCharacterSet(start, end, kind) {
        if (this._options.onAnyCharacterSet) {
            this._options.onAnyCharacterSet(start, end, kind);
        }
    }
    onEscapeCharacterSet(start, end, kind, negate) {
        if (this._options.onEscapeCharacterSet) {
            this._options.onEscapeCharacterSet(start, end, kind, negate);
        }
    }
    onUnicodePropertyCharacterSet(start, end, kind, key, value, negate) {
        if (this._options.onUnicodePropertyCharacterSet) {
            this._options.onUnicodePropertyCharacterSet(start, end, kind, key, value, negate);
        }
    }
    onCharacter(start, end, value) {
        if (this._options.onCharacter) {
            this._options.onCharacter(start, end, value);
        }
    }
    onBackreference(start, end, ref1) {
        if (this._options.onBackreference) {
            this._options.onBackreference(start, end, ref1);
        }
    }
    onCharacterClassEnter(start, negate) {
        if (this._options.onCharacterClassEnter) {
            this._options.onCharacterClassEnter(start, negate);
        }
    }
    onCharacterClassLeave(start, end, negate) {
        if (this._options.onCharacterClassLeave) {
            this._options.onCharacterClassLeave(start, end, negate);
        }
    }
    onCharacterClassRange(start, end, min, max) {
        if (this._options.onCharacterClassRange) {
            this._options.onCharacterClassRange(start, end, min, max);
        }
    }
    get source() {
        return this._reader.source;
    }
    get index() {
        return this._reader.index;
    }
    get currentCodePoint() {
        return this._reader.currentCodePoint;
    }
    get nextCodePoint() {
        return this._reader.nextCodePoint;
    }
    get nextCodePoint2() {
        return this._reader.nextCodePoint2;
    }
    get nextCodePoint3() {
        return this._reader.nextCodePoint3;
    }
    reset(source, start, end) {
        this._reader.reset(source, start, end, this._uFlag);
    }
    rewind(index) {
        this._reader.rewind(index);
    }
    advance() {
        this._reader.advance();
    }
    eat(cp) {
        return this._reader.eat(cp);
    }
    eat2(cp1, cp2) {
        return this._reader.eat2(cp1, cp2);
    }
    eat3(cp1, cp2, cp3) {
        return this._reader.eat3(cp1, cp2, cp3);
    }
    raise(message) {
        throw new RegExpSyntaxError(this.source, this._uFlag, this.index, message);
    }
    eatRegExpBody() {
        const start = this.index;
        let inClass = false;
        let escaped = false;
        for(;;){
            const cp = this.currentCodePoint;
            if (cp === -1 || isLineTerminator(cp)) {
                const kind = inClass ? "character class" : "regular expression";
                this.raise(`Unterminated ${kind}`);
            }
            if (escaped) {
                escaped = false;
            } else if (cp === 92) {
                escaped = true;
            } else if (cp === 91) {
                inClass = true;
            } else if (cp === 93) {
                inClass = false;
            } else if (cp === 47 && !inClass || cp === 42 && this.index === start) {
                break;
            }
            this.advance();
        }
        return this.index !== start;
    }
    consumePattern() {
        const start = this.index;
        this._numCapturingParens = this.countCapturingParens();
        this._groupNames.clear();
        this._backreferenceNames.clear();
        this.onPatternEnter(start);
        this.consumeDisjunction();
        const cp = this.currentCodePoint;
        if (this.currentCodePoint !== -1) {
            if (cp === 41) {
                this.raise("Unmatched ')'");
            }
            if (cp === 92) {
                this.raise("\\ at end of pattern");
            }
            if (cp === 93 || cp === 125) {
                this.raise("Lone quantifier brackets");
            }
            const c = String.fromCodePoint(cp);
            this.raise(`Unexpected character '${c}'`);
        }
        for (const name79 of this._backreferenceNames){
            if (!this._groupNames.has(name79)) {
                this.raise("Invalid named capture referenced");
            }
        }
        this.onPatternLeave(start, this.index);
    }
    countCapturingParens() {
        const start = this.index;
        let inClass = false;
        let escaped = false;
        let count = 0;
        let cp = 0;
        while((cp = this.currentCodePoint) !== -1){
            if (escaped) {
                escaped = false;
            } else if (cp === 92) {
                escaped = true;
            } else if (cp === 91) {
                inClass = true;
            } else if (cp === 93) {
                inClass = false;
            } else if (cp === 40 && !inClass && (this.nextCodePoint !== 63 || this.nextCodePoint2 === 60 && this.nextCodePoint3 !== 61 && this.nextCodePoint3 !== 33)) {
                count += 1;
            }
            this.advance();
        }
        this.rewind(start);
        return count;
    }
    consumeDisjunction() {
        const start = this.index;
        let i34 = 0;
        this.onDisjunctionEnter(start);
        do {
            this.consumeAlternative(i34++);
        }while (this.eat(124))
        if (this.consumeQuantifier(true)) {
            this.raise("Nothing to repeat");
        }
        if (this.eat(123)) {
            this.raise("Lone quantifier brackets");
        }
        this.onDisjunctionLeave(start, this.index);
    }
    consumeAlternative(i35) {
        const start = this.index;
        this.onAlternativeEnter(start, i35);
        while(this.currentCodePoint !== -1 && this.consumeTerm()){}
        this.onAlternativeLeave(start, this.index, i35);
    }
    consumeTerm() {
        if (this._uFlag || this.strict) {
            return this.consumeAssertion() || this.consumeAtom() && this.consumeOptionalQuantifier();
        }
        return this.consumeAssertion() && (!this._lastAssertionIsQuantifiable || this.consumeOptionalQuantifier()) || this.consumeExtendedAtom() && this.consumeOptionalQuantifier();
    }
    consumeOptionalQuantifier() {
        this.consumeQuantifier();
        return true;
    }
    consumeAssertion() {
        const start = this.index;
        this._lastAssertionIsQuantifiable = false;
        if (this.eat(94)) {
            this.onEdgeAssertion(start, this.index, "start");
            return true;
        }
        if (this.eat(36)) {
            this.onEdgeAssertion(start, this.index, "end");
            return true;
        }
        if (this.eat2(92, 66)) {
            this.onWordBoundaryAssertion(start, this.index, "word", true);
            return true;
        }
        if (this.eat2(92, 98)) {
            this.onWordBoundaryAssertion(start, this.index, "word", false);
            return true;
        }
        if (this.eat2(40, 63)) {
            const lookbehind = this.ecmaVersion >= 2018 && this.eat(60);
            let negate = false;
            if (this.eat(61) || (negate = this.eat(ExclamationMark))) {
                const kind = lookbehind ? "lookbehind" : "lookahead";
                this.onLookaroundAssertionEnter(start, kind, negate);
                this.consumeDisjunction();
                if (!this.eat(41)) {
                    this.raise("Unterminated group");
                }
                this._lastAssertionIsQuantifiable = !lookbehind && !this.strict;
                this.onLookaroundAssertionLeave(start, this.index, kind, negate);
                return true;
            }
            this.rewind(start);
        }
        return false;
    }
    consumeQuantifier(noConsume = false) {
        const start = this.index;
        let min = 0;
        let max = 0;
        let greedy = false;
        if (this.eat(42)) {
            min = 0;
            max = Number.POSITIVE_INFINITY;
        } else if (this.eat(43)) {
            min = 1;
            max = Number.POSITIVE_INFINITY;
        } else if (this.eat(63)) {
            min = 0;
            max = 1;
        } else if (this.eatBracedQuantifier(noConsume)) {
            min = this._lastMinValue;
            max = this._lastMaxValue;
        } else {
            return false;
        }
        greedy = !this.eat(QuestionMark);
        if (!noConsume) {
            this.onQuantifier(start, this.index, min, max, greedy);
        }
        return true;
    }
    eatBracedQuantifier(noError) {
        const start = this.index;
        if (this.eat(123)) {
            this._lastMinValue = 0;
            this._lastMaxValue = Number.POSITIVE_INFINITY;
            if (this.eatDecimalDigits()) {
                this._lastMinValue = this._lastMaxValue = this._lastIntValue;
                if (this.eat(44)) {
                    this._lastMaxValue = this.eatDecimalDigits() ? this._lastIntValue : Number.POSITIVE_INFINITY;
                }
                if (this.eat(125)) {
                    if (!noError && this._lastMaxValue < this._lastMinValue) {
                        this.raise("numbers out of order in {} quantifier");
                    }
                    return true;
                }
            }
            if (!noError && (this._uFlag || this.strict)) {
                this.raise("Incomplete quantifier");
            }
            this.rewind(start);
        }
        return false;
    }
    consumeAtom() {
        return this.consumePatternCharacter() || this.consumeDot() || this.consumeReverseSolidusAtomEscape() || this.consumeCharacterClass() || this.consumeUncapturingGroup() || this.consumeCapturingGroup();
    }
    consumeDot() {
        if (this.eat(46)) {
            this.onAnyCharacterSet(this.index - 1, this.index, "any");
            return true;
        }
        return false;
    }
    consumeReverseSolidusAtomEscape() {
        const start = this.index;
        if (this.eat(92)) {
            if (this.consumeAtomEscape()) {
                return true;
            }
            this.rewind(start);
        }
        return false;
    }
    consumeUncapturingGroup() {
        const start = this.index;
        if (this.eat3(40, 63, 58)) {
            this.onGroupEnter(start);
            this.consumeDisjunction();
            if (!this.eat(41)) {
                this.raise("Unterminated group");
            }
            this.onGroupLeave(start, this.index);
            return true;
        }
        return false;
    }
    consumeCapturingGroup() {
        const start = this.index;
        if (this.eat(40)) {
            let name80 = null;
            if (this.ecmaVersion >= 2018) {
                if (this.consumeGroupSpecifier()) {
                    name80 = this._lastStrValue;
                }
            } else if (this.currentCodePoint === 63) {
                this.raise("Invalid group");
            }
            this.onCapturingGroupEnter(start, name80);
            this.consumeDisjunction();
            if (!this.eat(41)) {
                this.raise("Unterminated group");
            }
            this.onCapturingGroupLeave(start, this.index, name80);
            return true;
        }
        return false;
    }
    consumeExtendedAtom() {
        return this.consumeDot() || this.consumeReverseSolidusAtomEscape() || this.consumeReverseSolidusFollowedByC() || this.consumeCharacterClass() || this.consumeUncapturingGroup() || this.consumeCapturingGroup() || this.consumeInvalidBracedQuantifier() || this.consumeExtendedPatternCharacter();
    }
    consumeReverseSolidusFollowedByC() {
        const start = this.index;
        if (this.currentCodePoint === 92 && this.nextCodePoint === 99) {
            this._lastIntValue = this.currentCodePoint;
            this.advance();
            this.onCharacter(start, this.index, 92);
            return true;
        }
        return false;
    }
    consumeInvalidBracedQuantifier() {
        if (this.eatBracedQuantifier(true)) {
            this.raise("Nothing to repeat");
        }
        return false;
    }
    consumePatternCharacter() {
        const start = this.index;
        const cp = this.currentCodePoint;
        if (cp !== -1 && !isSyntaxCharacter(cp)) {
            this.advance();
            this.onCharacter(start, this.index, cp);
            return true;
        }
        return false;
    }
    consumeExtendedPatternCharacter() {
        const start = this.index;
        const cp = this.currentCodePoint;
        if (cp !== -1 && cp !== 94 && cp !== 36 && cp !== 92 && cp !== 46 && cp !== 42 && cp !== 43 && cp !== 63 && cp !== 40 && cp !== 41 && cp !== 91 && cp !== 124) {
            this.advance();
            this.onCharacter(start, this.index, cp);
            return true;
        }
        return false;
    }
    consumeGroupSpecifier() {
        if (this.eat(63)) {
            if (this.eatGroupName()) {
                if (!this._groupNames.has(this._lastStrValue)) {
                    this._groupNames.add(this._lastStrValue);
                    return true;
                }
                this.raise("Duplicate capture group name");
            }
            this.raise("Invalid group");
        }
        return false;
    }
    consumeAtomEscape() {
        if (this.consumeBackreference() || this.consumeCharacterClassEscape() || this.consumeCharacterEscape() || this._nFlag && this.consumeKGroupName()) {
            return true;
        }
        if (this.strict || this._uFlag) {
            this.raise("Invalid escape");
        }
        return false;
    }
    consumeBackreference() {
        const start = this.index;
        if (this.eatDecimalEscape()) {
            const n4 = this._lastIntValue;
            if (n4 <= this._numCapturingParens) {
                this.onBackreference(start - 1, this.index, n4);
                return true;
            }
            if (this.strict || this._uFlag) {
                this.raise("Invalid escape");
            }
            this.rewind(start);
        }
        return false;
    }
    consumeCharacterClassEscape() {
        const start = this.index;
        if (this.eat(100)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "digit", false);
            return true;
        }
        if (this.eat(68)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "digit", true);
            return true;
        }
        if (this.eat(115)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "space", false);
            return true;
        }
        if (this.eat(83)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "space", true);
            return true;
        }
        if (this.eat(119)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "word", false);
            return true;
        }
        if (this.eat(87)) {
            this._lastIntValue = -1;
            this.onEscapeCharacterSet(start - 1, this.index, "word", true);
            return true;
        }
        let negate = false;
        if (this._uFlag && this.ecmaVersion >= 2018 && (this.eat(112) || (negate = this.eat(LatinCapitalLetterP)))) {
            this._lastIntValue = -1;
            if (this.eat(123) && this.eatUnicodePropertyValueExpression() && this.eat(125)) {
                this.onUnicodePropertyCharacterSet(start - 1, this.index, "property", this._lastKeyValue, this._lastValValue || null, negate);
                return true;
            }
            this.raise("Invalid property name");
        }
        return false;
    }
    consumeCharacterEscape() {
        const start = this.index;
        if (this.eatControlEscape() || this.eatCControlLetter() || this.eatZero() || this.eatHexEscapeSequence() || this.eatRegExpUnicodeEscapeSequence() || !this.strict && !this._uFlag && this.eatLegacyOctalEscapeSequence() || this.eatIdentityEscape()) {
            this.onCharacter(start - 1, this.index, this._lastIntValue);
            return true;
        }
        return false;
    }
    consumeKGroupName() {
        const start = this.index;
        if (this.eat(107)) {
            if (this.eatGroupName()) {
                const groupName = this._lastStrValue;
                this._backreferenceNames.add(groupName);
                this.onBackreference(start - 1, this.index, groupName);
                return true;
            }
            this.raise("Invalid named reference");
        }
        return false;
    }
    consumeCharacterClass() {
        const start = this.index;
        if (this.eat(91)) {
            const negate = this.eat(94);
            this.onCharacterClassEnter(start, negate);
            this.consumeClassRanges();
            if (!this.eat(93)) {
                this.raise("Unterminated character class");
            }
            this.onCharacterClassLeave(start, this.index, negate);
            return true;
        }
        return false;
    }
    consumeClassRanges() {
        const strict = this.strict || this._uFlag;
        for(;;){
            const rangeStart = this.index;
            if (!this.consumeClassAtom()) {
                break;
            }
            const min = this._lastIntValue;
            if (!this.eat(45)) {
                continue;
            }
            this.onCharacter(this.index - 1, this.index, 45);
            if (!this.consumeClassAtom()) {
                break;
            }
            const max = this._lastIntValue;
            if (min === -1 || max === -1) {
                if (strict) {
                    this.raise("Invalid character class");
                }
                continue;
            }
            if (min > max) {
                this.raise("Range out of order in character class");
            }
            this.onCharacterClassRange(rangeStart, this.index, min, max);
        }
    }
    consumeClassAtom() {
        const start = this.index;
        const cp = this.currentCodePoint;
        if (cp !== -1 && cp !== 92 && cp !== 93) {
            this.advance();
            this._lastIntValue = cp;
            this.onCharacter(start, this.index, this._lastIntValue);
            return true;
        }
        if (this.eat(92)) {
            if (this.consumeClassEscape()) {
                return true;
            }
            if (!this.strict && this.currentCodePoint === 99) {
                this._lastIntValue = ReverseSolidus;
                this.onCharacter(start, this.index, this._lastIntValue);
                return true;
            }
            if (this.strict || this._uFlag) {
                this.raise("Invalid escape");
            }
            this.rewind(start);
        }
        return false;
    }
    consumeClassEscape() {
        const start = this.index;
        if (this.eat(98)) {
            this._lastIntValue = Backspace;
            this.onCharacter(start - 1, this.index, this._lastIntValue);
            return true;
        }
        if (this._uFlag && this.eat(45)) {
            this._lastIntValue = HyphenMinus;
            this.onCharacter(start - 1, this.index, this._lastIntValue);
            return true;
        }
        let cp = 0;
        if (!this.strict && !this._uFlag && this.currentCodePoint === 99 && (isDecimalDigit(cp = this.nextCodePoint) || cp === 95)) {
            this.advance();
            this.advance();
            this._lastIntValue = cp % 32;
            this.onCharacter(start - 1, this.index, this._lastIntValue);
            return true;
        }
        return this.consumeCharacterClassEscape() || this.consumeCharacterEscape();
    }
    eatGroupName() {
        if (this.eat(60)) {
            if (this.eatRegExpIdentifierName() && this.eat(62)) {
                return true;
            }
            this.raise("Invalid capture group name");
        }
        return false;
    }
    eatRegExpIdentifierName() {
        if (this.eatRegExpIdentifierStart()) {
            this._lastStrValue = String.fromCodePoint(this._lastIntValue);
            while(this.eatRegExpIdentifierPart()){
                this._lastStrValue += String.fromCodePoint(this._lastIntValue);
            }
            return true;
        }
        return false;
    }
    eatRegExpIdentifierStart() {
        const start = this.index;
        const forceUFlag = !this._uFlag && this.ecmaVersion >= 2020;
        let cp = this.currentCodePoint;
        this.advance();
        if (cp === 92 && this.eatRegExpUnicodeEscapeSequence(forceUFlag)) {
            cp = this._lastIntValue;
        } else if (forceUFlag && isLeadSurrogate(cp) && isTrailSurrogate(this.currentCodePoint)) {
            cp = combineSurrogatePair(cp, this.currentCodePoint);
            this.advance();
        }
        if (isRegExpIdentifierStart(cp)) {
            this._lastIntValue = cp;
            return true;
        }
        if (this.index !== start) {
            this.rewind(start);
        }
        return false;
    }
    eatRegExpIdentifierPart() {
        const start = this.index;
        const forceUFlag = !this._uFlag && this.ecmaVersion >= 2020;
        let cp = this.currentCodePoint;
        this.advance();
        if (cp === 92 && this.eatRegExpUnicodeEscapeSequence(forceUFlag)) {
            cp = this._lastIntValue;
        } else if (forceUFlag && isLeadSurrogate(cp) && isTrailSurrogate(this.currentCodePoint)) {
            cp = combineSurrogatePair(cp, this.currentCodePoint);
            this.advance();
        }
        if (isRegExpIdentifierPart(cp)) {
            this._lastIntValue = cp;
            return true;
        }
        if (this.index !== start) {
            this.rewind(start);
        }
        return false;
    }
    eatCControlLetter() {
        const start = this.index;
        if (this.eat(99)) {
            if (this.eatControlLetter()) {
                return true;
            }
            this.rewind(start);
        }
        return false;
    }
    eatZero() {
        if (this.currentCodePoint === 48 && !isDecimalDigit(this.nextCodePoint)) {
            this._lastIntValue = 0;
            this.advance();
            return true;
        }
        return false;
    }
    eatControlEscape() {
        if (this.eat(102)) {
            this._lastIntValue = FormFeed;
            return true;
        }
        if (this.eat(110)) {
            this._lastIntValue = LineFeed;
            return true;
        }
        if (this.eat(114)) {
            this._lastIntValue = CarriageReturn;
            return true;
        }
        if (this.eat(116)) {
            this._lastIntValue = CharacterTabulation;
            return true;
        }
        if (this.eat(118)) {
            this._lastIntValue = LineTabulation;
            return true;
        }
        return false;
    }
    eatControlLetter() {
        const cp = this.currentCodePoint;
        if (isLatinLetter(cp)) {
            this.advance();
            this._lastIntValue = cp % 32;
            return true;
        }
        return false;
    }
    eatRegExpUnicodeEscapeSequence(forceUFlag = false) {
        const start = this.index;
        const uFlag = forceUFlag || this._uFlag;
        if (this.eat(117)) {
            if (uFlag && this.eatRegExpUnicodeSurrogatePairEscape() || this.eatFixedHexDigits(4) || uFlag && this.eatRegExpUnicodeCodePointEscape()) {
                return true;
            }
            if (this.strict || uFlag) {
                this.raise("Invalid unicode escape");
            }
            this.rewind(start);
        }
        return false;
    }
    eatRegExpUnicodeSurrogatePairEscape() {
        const start = this.index;
        if (this.eatFixedHexDigits(4)) {
            const lead = this._lastIntValue;
            if (isLeadSurrogate(lead) && this.eat(92) && this.eat(117) && this.eatFixedHexDigits(4)) {
                const trail = this._lastIntValue;
                if (isTrailSurrogate(trail)) {
                    this._lastIntValue = combineSurrogatePair(lead, trail);
                    return true;
                }
            }
            this.rewind(start);
        }
        return false;
    }
    eatRegExpUnicodeCodePointEscape() {
        const start = this.index;
        if (this.eat(123) && this.eatHexDigits() && this.eat(125) && isValidUnicode(this._lastIntValue)) {
            return true;
        }
        this.rewind(start);
        return false;
    }
    eatIdentityEscape() {
        const cp = this.currentCodePoint;
        if (this.isValidIdentityEscape(cp)) {
            this._lastIntValue = cp;
            this.advance();
            return true;
        }
        return false;
    }
    isValidIdentityEscape(cp) {
        if (cp === -1) {
            return false;
        }
        if (this._uFlag) {
            return isSyntaxCharacter(cp) || cp === 47;
        }
        if (this.strict) {
            return !isIdContinue(cp);
        }
        if (this._nFlag) {
            return !(cp === 99 || cp === 107);
        }
        return cp !== 99;
    }
    eatDecimalEscape() {
        this._lastIntValue = 0;
        let cp = this.currentCodePoint;
        if (cp >= 49 && cp <= 57) {
            do {
                this._lastIntValue = 10 * this._lastIntValue + (cp - DigitZero);
                this.advance();
            }while ((cp = this.currentCodePoint) >= 48 && cp <= 57)
            return true;
        }
        return false;
    }
    eatUnicodePropertyValueExpression() {
        const start = this.index;
        if (this.eatUnicodePropertyName() && this.eat(61)) {
            this._lastKeyValue = this._lastStrValue;
            if (this.eatUnicodePropertyValue()) {
                this._lastValValue = this._lastStrValue;
                if (isValidUnicodeProperty(this.ecmaVersion, this._lastKeyValue, this._lastValValue)) {
                    return true;
                }
                this.raise("Invalid property name");
            }
        }
        this.rewind(start);
        if (this.eatLoneUnicodePropertyNameOrValue()) {
            const nameOrValue = this._lastStrValue;
            if (isValidUnicodeProperty(this.ecmaVersion, "General_Category", nameOrValue)) {
                this._lastKeyValue = "General_Category";
                this._lastValValue = nameOrValue;
                return true;
            }
            if (isValidLoneUnicodeProperty(this.ecmaVersion, nameOrValue)) {
                this._lastKeyValue = nameOrValue;
                this._lastValValue = "";
                return true;
            }
            this.raise("Invalid property name");
        }
        return false;
    }
    eatUnicodePropertyName() {
        this._lastStrValue = "";
        while(isUnicodePropertyNameCharacter(this.currentCodePoint)){
            this._lastStrValue += String.fromCodePoint(this.currentCodePoint);
            this.advance();
        }
        return this._lastStrValue !== "";
    }
    eatUnicodePropertyValue() {
        this._lastStrValue = "";
        while(isUnicodePropertyValueCharacter(this.currentCodePoint)){
            this._lastStrValue += String.fromCodePoint(this.currentCodePoint);
            this.advance();
        }
        return this._lastStrValue !== "";
    }
    eatLoneUnicodePropertyNameOrValue() {
        return this.eatUnicodePropertyValue();
    }
    eatHexEscapeSequence() {
        const start = this.index;
        if (this.eat(120)) {
            if (this.eatFixedHexDigits(2)) {
                return true;
            }
            if (this._uFlag || this.strict) {
                this.raise("Invalid escape");
            }
            this.rewind(start);
        }
        return false;
    }
    eatDecimalDigits() {
        const start = this.index;
        this._lastIntValue = 0;
        while(isDecimalDigit(this.currentCodePoint)){
            this._lastIntValue = 10 * this._lastIntValue + digitToInt(this.currentCodePoint);
            this.advance();
        }
        return this.index !== start;
    }
    eatHexDigits() {
        const start = this.index;
        this._lastIntValue = 0;
        while(isHexDigit(this.currentCodePoint)){
            this._lastIntValue = 16 * this._lastIntValue + digitToInt(this.currentCodePoint);
            this.advance();
        }
        return this.index !== start;
    }
    eatLegacyOctalEscapeSequence() {
        if (this.eatOctalDigit()) {
            const n1 = this._lastIntValue;
            if (this.eatOctalDigit()) {
                const n21 = this._lastIntValue;
                if (n1 <= 3 && this.eatOctalDigit()) {
                    this._lastIntValue = n1 * 64 + n21 * 8 + this._lastIntValue;
                } else {
                    this._lastIntValue = n1 * 8 + n21;
                }
            } else {
                this._lastIntValue = n1;
            }
            return true;
        }
        return false;
    }
    eatOctalDigit() {
        const cp = this.currentCodePoint;
        if (isOctalDigit(cp)) {
            this.advance();
            this._lastIntValue = cp - DigitZero;
            return true;
        }
        this._lastIntValue = 0;
        return false;
    }
    eatFixedHexDigits(length) {
        const start = this.index;
        this._lastIntValue = 0;
        for(let i36 = 0; i36 < length; ++i36){
            const cp = this.currentCodePoint;
            if (!isHexDigit(cp)) {
                this.rewind(start);
                return false;
            }
            this._lastIntValue = 16 * this._lastIntValue + digitToInt(cp);
            this.advance();
        }
        return true;
    }
}
const DummyPattern = {};
const DummyFlags = {};
const DummyCapturingGroup = {};
class RegExpParserState {
    constructor(options){
        this._node = DummyPattern;
        this._flags = DummyFlags;
        this._backreferences = [];
        this._capturingGroups = [];
        this.source = "";
        this.strict = Boolean(options && options.strict);
        this.ecmaVersion = options && options.ecmaVersion || 2022;
    }
    get pattern() {
        if (this._node.type !== "Pattern") {
            throw new Error("UnknownError");
        }
        return this._node;
    }
    get flags() {
        if (this._flags.type !== "Flags") {
            throw new Error("UnknownError");
        }
        return this._flags;
    }
    onFlags(start, end, global, ignoreCase, multiline, unicode, sticky, dotAll, hasIndices) {
        this._flags = {
            type: "Flags",
            parent: null,
            start,
            end,
            raw: this.source.slice(start, end),
            global,
            ignoreCase,
            multiline,
            unicode,
            sticky,
            dotAll,
            hasIndices
        };
    }
    onPatternEnter(start) {
        this._node = {
            type: "Pattern",
            parent: null,
            start,
            end: start,
            raw: "",
            alternatives: []
        };
        this._backreferences.length = 0;
        this._capturingGroups.length = 0;
    }
    onPatternLeave(start, end) {
        this._node.end = end;
        this._node.raw = this.source.slice(start, end);
        for (const reference of this._backreferences){
            const ref2 = reference.ref;
            const group = typeof ref2 === "number" ? this._capturingGroups[ref2 - 1] : this._capturingGroups.find((g)=>g.name === ref2
            );
            reference.resolved = group;
            group.references.push(reference);
        }
    }
    onAlternativeEnter(start) {
        const parent = this._node;
        if (parent.type !== "Assertion" && parent.type !== "CapturingGroup" && parent.type !== "Group" && parent.type !== "Pattern") {
            throw new Error("UnknownError");
        }
        this._node = {
            type: "Alternative",
            parent,
            start,
            end: start,
            raw: "",
            elements: []
        };
        parent.alternatives.push(this._node);
    }
    onAlternativeLeave(start, end) {
        const node = this._node;
        if (node.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        node.end = end;
        node.raw = this.source.slice(start, end);
        this._node = node.parent;
    }
    onGroupEnter(start) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        this._node = {
            type: "Group",
            parent,
            start,
            end: start,
            raw: "",
            alternatives: []
        };
        parent.elements.push(this._node);
    }
    onGroupLeave(start, end) {
        const node = this._node;
        if (node.type !== "Group" || node.parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        node.end = end;
        node.raw = this.source.slice(start, end);
        this._node = node.parent;
    }
    onCapturingGroupEnter(start, name81) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        this._node = {
            type: "CapturingGroup",
            parent,
            start,
            end: start,
            raw: "",
            name: name81,
            alternatives: [],
            references: []
        };
        parent.elements.push(this._node);
        this._capturingGroups.push(this._node);
    }
    onCapturingGroupLeave(start, end) {
        const node = this._node;
        if (node.type !== "CapturingGroup" || node.parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        node.end = end;
        node.raw = this.source.slice(start, end);
        this._node = node.parent;
    }
    onQuantifier(start, end, min, max, greedy) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        const element = parent.elements.pop();
        if (element == null || element.type === "Quantifier" || element.type === "Assertion" && element.kind !== "lookahead") {
            throw new Error("UnknownError");
        }
        const node = {
            type: "Quantifier",
            parent,
            start: element.start,
            end,
            raw: this.source.slice(element.start, end),
            min,
            max,
            greedy,
            element
        };
        parent.elements.push(node);
        element.parent = node;
    }
    onLookaroundAssertionEnter(start, kind, negate) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        const node = this._node = {
            type: "Assertion",
            parent,
            start,
            end: start,
            raw: "",
            kind,
            negate,
            alternatives: []
        };
        parent.elements.push(node);
    }
    onLookaroundAssertionLeave(start, end) {
        const node = this._node;
        if (node.type !== "Assertion" || node.parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        node.end = end;
        node.raw = this.source.slice(start, end);
        this._node = node.parent;
    }
    onEdgeAssertion(start, end, kind) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "Assertion",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            kind
        });
    }
    onWordBoundaryAssertion(start, end, kind, negate) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "Assertion",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            kind,
            negate
        });
    }
    onAnyCharacterSet(start, end, kind) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "CharacterSet",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            kind
        });
    }
    onEscapeCharacterSet(start, end, kind, negate) {
        const parent = this._node;
        if (parent.type !== "Alternative" && parent.type !== "CharacterClass") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "CharacterSet",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            kind,
            negate
        });
    }
    onUnicodePropertyCharacterSet(start, end, kind, key, value, negate) {
        const parent = this._node;
        if (parent.type !== "Alternative" && parent.type !== "CharacterClass") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "CharacterSet",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            kind,
            key,
            value,
            negate
        });
    }
    onCharacter(start, end, value) {
        const parent = this._node;
        if (parent.type !== "Alternative" && parent.type !== "CharacterClass") {
            throw new Error("UnknownError");
        }
        parent.elements.push({
            type: "Character",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            value
        });
    }
    onBackreference(start, end, ref3) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        const node = {
            type: "Backreference",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            ref: ref3,
            resolved: DummyCapturingGroup
        };
        parent.elements.push(node);
        this._backreferences.push(node);
    }
    onCharacterClassEnter(start, negate) {
        const parent = this._node;
        if (parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        this._node = {
            type: "CharacterClass",
            parent,
            start,
            end: start,
            raw: "",
            negate,
            elements: []
        };
        parent.elements.push(this._node);
    }
    onCharacterClassLeave(start, end) {
        const node = this._node;
        if (node.type !== "CharacterClass" || node.parent.type !== "Alternative") {
            throw new Error("UnknownError");
        }
        node.end = end;
        node.raw = this.source.slice(start, end);
        this._node = node.parent;
    }
    onCharacterClassRange(start, end) {
        const parent = this._node;
        if (parent.type !== "CharacterClass") {
            throw new Error("UnknownError");
        }
        const elements = parent.elements;
        const max = elements.pop();
        const hyphen = elements.pop();
        const min = elements.pop();
        if (!min || !max || !hyphen || min.type !== "Character" || max.type !== "Character" || hyphen.type !== "Character" || hyphen.value !== 45) {
            throw new Error("UnknownError");
        }
        const node = {
            type: "CharacterClassRange",
            parent,
            start,
            end,
            raw: this.source.slice(start, end),
            min,
            max
        };
        min.parent = node;
        max.parent = node;
        elements.push(node);
    }
}
class RegExpParser {
    constructor(options){
        this._state = new RegExpParserState(options);
        this._validator = new RegExpValidator(this._state);
    }
    parseLiteral(source, start = 0, end = source.length) {
        this._state.source = source;
        this._validator.validateLiteral(source, start, end);
        const pattern = this._state.pattern;
        const flags = this._state.flags;
        const literal3 = {
            type: "RegExpLiteral",
            parent: null,
            start,
            end,
            raw: source,
            pattern,
            flags
        };
        pattern.parent = literal3;
        flags.parent = literal3;
        return literal3;
    }
    parseFlags(source, start = 0, end = source.length) {
        this._state.source = source;
        this._validator.validateFlags(source, start, end);
        return this._state.flags;
    }
    parsePattern(source, start = 0, end = source.length, uFlag = false) {
        this._state.source = source;
        this._validator.validatePattern(source, start, end, uFlag);
        return this._state.pattern;
    }
}
function parseRegExpLiteral(source, options) {
    return new RegExpParser(options).parseLiteral(String(source));
}
function prefixesFromParse(parse9) {
    if (parse9.type === "Pattern" || parse9.type === "CapturingGroup") {
        const alternatives = parse9.alternatives.map(prefixesFromParse);
        return `(${alternatives.join("|")})`;
    } else if (parse9.type === "Alternative") {
        const result = [];
        for(let i37 = 0; i37 < parse9.elements.length; ++i37){
            const thisRe = [];
            for(let j = 0; j < i37; ++j){
                thisRe.push(parse9.elements[j].raw);
            }
            thisRe.push(prefixesFromParse(parse9.elements[i37]));
            result.push(thisRe.join(""));
        }
        return `(${result.join("|")})`;
    } else if (parse9.type === "RegExpLiteral") {
        return prefixesFromParse(parse9.pattern);
    } else if (parse9.type === "Character") {
        return `${parse9.raw}?`;
    } else if (parse9.type === "Quantifier") {
        if (parse9.min === 0 && parse9.max === 1) {
            return prefixesFromParse(parse9.element);
        }
        if (parse9.min === 1 && parse9.max === Infinity) {
            return `(${parse9.element.raw}*)` + prefixesFromParse(parse9.element);
        }
        if (parse9.min === 0 && parse9.max === Infinity) {
            return `(${parse9.element.raw}*)` + prefixesFromParse(parse9.element);
        } else {
            throw new Error(`Internal Error, can't handle quantifiers min=${parse9.min} max=${parse9.max}`);
        }
    } else if (parse9.type === "CharacterSet") {
        return `${parse9.raw}?`;
    }
    throw new Error(`Internal Error, don't know how to handle ${parse9.type}`);
}
function prefixes(regexp3) {
    regexp3 = regexp3.source;
    regexp3 = regexp3.slice(1, -1);
    return new RegExp("^" + prefixesFromParse(parseRegExpLiteral(new RegExp(regexp3))) + "$");
}
let _schemas = {
    schemas: {
        "front-matter": undefined,
        config: undefined,
        engines: undefined
    },
    aliases: {},
    definitions: {}
};
function setSchemas(schemas) {
    _schemas = schemas;
}
function getSchemas() {
    if (_schemas) {
        return _schemas;
    } else {
        throw new Error("Internal error: schemas not set");
    }
}
function matchPatternProperties(schema, key) {
    for (const [regexpStr, subschema] of Object.entries(schema.patternProperties || {})){
        const prefixPattern = prefixes(new RegExp(regexpStr));
        if (key.match(prefixPattern)) {
            return subschema;
        }
    }
    return false;
}
function navigateSchema(schema, path57) {
    const inner = (subSchema, index)=>{
        subSchema = resolveSchema(subSchema);
        if (index === path57.length) {
            return [
                subSchema
            ];
        }
        const st = schemaType(subSchema);
        if (st === "object") {
            const key = path57[index];
            if (subSchema.properties && subSchema.properties[key]) {
                return inner(subSchema.properties[key], index + 1);
            }
            const patternPropMatch = matchPatternProperties(subSchema, key);
            if (patternPropMatch) {
                return inner(patternPropMatch, index + 1);
            }
            if (index !== path57.length - 1) {
                return [];
            }
            const completions = Object.getOwnPropertyNames(subSchema.properties || {}).filter((name82)=>name82.startsWith(key)
            );
            if (completions.length === 0) {
                return [];
            }
            return [
                subSchema
            ];
        } else if (st === "array") {
            if (subSchema.items === undefined) {
                return [];
            }
            return inner(subSchema.items, index + 1);
        } else if (st === "anyOf") {
            return subSchema.anyOf.map((ss)=>inner(ss, index)
            );
        } else if (st === "allOf") {
            return subSchema.allOf.map((ss)=>inner(ss, index)
            );
        } else if (st === "oneOf") {
            return subSchema.oneOf.map((ss)=>inner(ss, index)
            );
        } else {
            return [];
        }
    };
    return inner(schema, 0).flat(Infinity);
}
function resolveSchema(schema, visit, hasRef, next) {
    if (hasRef === undefined) {
        hasRef = (cursor)=>{
            return cursor.$ref !== undefined;
        };
    }
    if (!hasRef(schema)) {
        return schema;
    }
    if (visit === undefined) {
        visit = (schema)=>{};
    }
    if (next === undefined) {
        next = (cursor)=>{
            const result = getSchemaDefinition(cursor.$ref);
            if (result === undefined) {
                throw new Error(`Internal Error: ref ${cursor.$ref} not in definitions`);
            }
            return result;
        };
    }
    let cursor1 = schema;
    let cursor2 = schema;
    let stopped = false;
    do {
        cursor1 = next(cursor1);
        visit(cursor1);
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (!stopped && cursor1 === cursor2) {
            throw new Error(`reference cycle detected at ${JSON.stringify(cursor1)}`);
        }
    }while (hasRef(cursor1))
    return cursor1;
}
function generate() {
    return crypto.randomUUID();
}
function mergeConfigs(config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, mergeArrayCustomizer);
}
function mergeArrayCustomizer(objValue, srcValue) {
    if (__VIRTUAL_FILE53(objValue) || __VIRTUAL_FILE53(srcValue)) {
        if (!objValue) {
            return srcValue;
        } else if (!srcValue) {
            return objValue;
        } else {
            if (!__VIRTUAL_FILE53(objValue)) {
                objValue = [
                    objValue
                ];
            }
            if (!__VIRTUAL_FILE53(srcValue)) {
                srcValue = [
                    srcValue
                ];
            }
        }
        const combined = objValue.concat(srcValue);
        return __VIRTUAL_FILE210(combined, (value)=>{
            if (typeof value === "function") {
                return generate();
            } else {
                return JSON.stringify(value);
            }
        });
    }
}
function camelToKebab(camel) {
    const kebab = [];
    for(let i38 = 0; i38 < camel.length; i38++){
        const ch = camel.charAt(i38);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i38 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i39 = 0; i39 < kebab.length; i39++){
        const ch = kebab.charAt(i39);
        if (ch === "-") {
            camel.push(kebab.charAt(++i39).toUpperCase());
        } else if (i39 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
const BooleanSchema = {
    "type": "boolean",
    "description": "be `true` or `false`",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const NumberSchema = {
    "type": "number",
    "description": "be a number"
};
function tagSchema(schema, tags) {
    return {
        ...schema,
        tags: mergeConfigs(schema?.tags ?? {}, tags)
    };
}
const StringSchema = {
    "type": "string",
    "description": "be a string"
};
const NullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function enumSchema(...args) {
    if (args.length === 0) {
        throw new Error("Internal Error: Empty enum schema not supported.");
    }
    return {
        "enum": args,
        "description": args.length > 1 ? `be one of: ${args.map((x)=>"`" + x + "`"
        ).join(", ")}` : `be '${args[0]}'`,
        "completions": args,
        "exhaustiveCompletions": true
    };
}
function regexSchema(arg, description) {
    const result = {
        "type": "string",
        "pattern": arg
    };
    if (description) {
        result.description = description;
    } else {
        result.description = `be a string that satisfies regex "${arg}"`;
    }
    return result;
}
function oneOfSchema(...args) {
    return {
        "oneOf": args,
        "description": `be exactly one of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function anyOfSchema(...args) {
    return {
        "anyOf": args,
        "description": `be at least one of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function allOfSchema(...args) {
    return {
        "allOf": args,
        "description": `be all of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function objectSchema(params = {}) {
    let { properties , patternProperties , required , additionalProperties , description , baseSchema , exhaustive , completions: completionsParam ,  } = params;
    required = required || [];
    properties = properties || {};
    patternProperties = patternProperties || {};
    const hasDescription = description !== undefined;
    description = description || "be an object";
    let result = undefined;
    const completions = [];
    const uniqueValues = (lst)=>{
        const obj = {};
        for (const c of lst){
            obj[c.value] = c;
        }
        return Object.getOwnPropertyNames(obj).map((k)=>obj[k]
        );
    };
    for (const k1 of Object.getOwnPropertyNames(completionsParam || properties)){
        const schema1 = properties[k1];
        const maybeDescriptions = [
            completionsParam?.[k1]
        ];
        let hidden25 = false;
        if (schema1 !== undefined) {
            if (schema1.documentation) {
                maybeDescriptions.push(schema1?.documentation?.short);
                maybeDescriptions.push(schema1?.documentation);
            } else {
                let described = false;
                const visitor = (schema)=>{
                    if (schema?.hidden) {
                        hidden25 = true;
                    }
                    if (described) {
                        return;
                    }
                    if (schema?.documentation?.short) {
                        maybeDescriptions.push(schema?.documentation?.short);
                        described = true;
                    } else if (schema?.documentation) {
                        maybeDescriptions.push(schema?.documentation);
                        described = true;
                    }
                };
                try {
                    resolveSchema(schema1, visitor);
                } catch (e) {}
                if (!described && schema1?.$ref) {
                    maybeDescriptions.push({
                        $ref: schema1?.$ref
                    });
                }
            }
        }
        if (hidden25) {
            continue;
        }
        let description = "";
        for (const md of maybeDescriptions){
            if (md !== undefined) {
                description = md;
                break;
            }
        }
        completions.push({
            type: "key",
            display: "",
            value: `${k1}: `,
            description,
            suggest_on_accept: schema1?.completions?.length !== 0
        });
    }
    if (baseSchema) {
        if (baseSchema.type !== "object") {
            throw new Error("Internal Error: can only extend other object Schema");
        }
        result = Object.assign({}, baseSchema);
        if (result.$id) {
            delete result.$id;
        }
        if (exhaustive && baseSchema.exhaustiveCompletions) {
            result.exhaustiveCompletions = true;
        }
        if (hasDescription) {
            result.description = description;
        }
        result.properties = Object.assign({}, result.properties, properties);
        result.patternProperties = Object.assign({}, result.patternProperties, patternProperties);
        if (required) {
            result.required = (result.required ?? []).slice();
            result.required.push(...required);
        }
        if (result.completions && result.completions.length || completions.length) {
            result.completions = (result.completions || []).slice();
            result.completions.push(...completions);
            result.completions = uniqueValues(result.completions);
        }
        if (additionalProperties !== undefined) {
            if (result.additionalProperties === false) {
                throw new Error("Internal Error: don't know how to subclass object schema with additionalProperties === false");
            }
            if (result.additionalProperties) {
                result.additionalProperties = allOfSchema(result.additionalProperties, additionalProperties);
            } else {
                result.additionalProperties = additionalProperties;
            }
        }
    } else {
        result = {
            "type": "object",
            description
        };
        if (exhaustive) {
            result.exhaustiveCompletions = true;
        }
        if (properties) {
            result.properties = properties;
        }
        if (patternProperties) {
            result.patternProperties = patternProperties;
        }
        if (required && required.length > 0) {
            result.required = required;
        }
        if (completions.length) {
            result.completions = completions;
        }
        if (additionalProperties !== undefined) {
            result.additionalProperties = additionalProperties;
        }
    }
    return result;
}
function arraySchema(items) {
    if (items) {
        return {
            "type": "array",
            "description": `be an array of values, where each element should ` + items.description,
            items
        };
    } else {
        return {
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema, doc) {
    const result = Object.assign({}, schema);
    result.documentation = doc;
    return result;
}
function describeSchema(schema, description) {
    const result = Object.assign({}, schema);
    result.description = `be ${description}`;
    return result;
}
function completeSchema(schema, ...completions) {
    const result = Object.assign({}, schema);
    const prevCompletions = (schema.completions ?? []).slice();
    prevCompletions.push(...completions);
    result.completions = prevCompletions;
    return result;
}
function completeSchemaOverwrite(schema, ...completions) {
    const result = Object.assign({}, schema);
    result.completions = completions;
    return result;
}
function idSchema(schema, id) {
    const result = Object.assign({}, schema);
    result["$id"] = id;
    return result;
}
function refSchema($ref, description) {
    return {
        $ref,
        description
    };
}
function valueSchema(val, description) {
    return {
        "enum": [
            val
        ],
        "description": description ?? `be ${JSON.stringify(val)}`
    };
}
function schemaPath(path58) {
    return resourcePath(join3("schema", path58));
}
function mappedString(source, pieces, fileName) {
    if (typeof source === "string") {
        const offsetInfo = [];
        let offset = 0;
        const resultList = pieces.filter((piece)=>typeof piece === "string" || piece.start !== piece.end
        ).map((piece)=>{
            if (typeof piece === "string") {
                offsetInfo.push({
                    fromSource: false,
                    length: piece.length,
                    offset
                });
                offset += piece.length;
                return piece;
            } else {
                const resultPiece = source.substring(piece.start, piece.end);
                offsetInfo.push({
                    fromSource: true,
                    length: resultPiece.length,
                    offset,
                    range: {
                        start: piece.start,
                        end: piece.end
                    }
                });
                offset += resultPiece.length;
                return resultPiece;
            }
        });
        const value = resultList.join("");
        const map3 = (targetOffset)=>{
            const ix = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            if (ix < 0) {
                return undefined;
            }
            const info3 = offsetInfo[ix];
            if (!info3.fromSource) {
                return undefined;
            }
            const localOffset = targetOffset - info3.offset;
            if (localOffset >= info3.length) {
                return undefined;
            }
            return info3.range.start + localOffset;
        };
        const mapClosest = (targetOffset)=>{
            if (offsetInfo.length === 0 || targetOffset < 0) {
                return undefined;
            }
            const firstIx = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            let ix = firstIx;
            let smallestSourceInfo = undefined;
            while(ix >= 0){
                const info4 = offsetInfo[ix];
                if (!info4.fromSource) {
                    ix--;
                    continue;
                }
                smallestSourceInfo = info4;
                if (ix === firstIx) {
                    const localOffset = targetOffset - info4.offset;
                    if (localOffset < info4.length) {
                        return info4.range.start + localOffset;
                    }
                }
                return info4.range.end - 1;
            }
            if (smallestSourceInfo === undefined) {
                return undefined;
            } else {
                return smallestSourceInfo.range.start;
            }
        };
        return {
            value,
            originalString: source,
            fileName,
            map: map3,
            mapClosest
        };
    } else {
        const { value , originalString , map: previousMap , mapClosest: previousMapClosest , fileName: previousFileName ,  } = source;
        const { value: resultValue , map: nextMap , mapClosest: nextMapClosest ,  } = mappedString(value, pieces);
        const composeMap = (offset)=>{
            const v = nextMap(offset);
            if (v === undefined) {
                return v;
            }
            return previousMap(v);
        };
        const composeMapClosest = (offset)=>{
            const v = nextMapClosest(offset);
            if (v === undefined) {
                return v;
            }
            return previousMapClosest(v);
        };
        return {
            value: resultValue,
            originalString,
            map: composeMap,
            mapClosest: composeMapClosest,
            fileName: previousFileName
        };
    }
}
function asMappedString(str59, fileName) {
    if (typeof str59 === "string") {
        return {
            value: str59,
            originalString: str59,
            map: (x)=>x
            ,
            mapClosest: (x)=>x
            ,
            fileName
        };
    } else if (fileName !== undefined) {
        throw new Error("Internal error: can't change the fileName of an existing MappedString");
    } else {
        return str59;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        throw new Error("strings must be non-empty");
    }
    let currentOffset = 0;
    const offsets = [];
    for (const s1 of strings){
        currentOffset += s1.value.length;
        offsets.push(currentOffset);
    }
    const value = "".concat(...strings.map((s)=>s.value
    ));
    return {
        value,
        originalString: strings[0].originalString,
        fileName: strings[0].fileName,
        map (offset) {
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].map(offset - offsets[ix]);
        },
        mapClosest (offset) {
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].mapClosest(offset - offsets[ix]);
        }
    };
}
function mappedIndexToRowCol(eitherText) {
    const text = asMappedString(eitherText);
    const f = indexToRowCol(text.originalString);
    return function(offset) {
        const n5 = text.mapClosest(offset);
        if (n5 === undefined) {
            throw new Error("Internal Error: bad offset in mappedIndexRowCol");
        }
        return f(n5);
    };
}
function mappedNormalizeNewlines(eitherText) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const offset of lineBreakPositions(text.value)){
        if (text.value[offset] !== "\r") {
            continue;
        }
        chunks.push({
            start,
            end: offset
        });
        chunks.push({
            start: offset + 1,
            end: offset + 2
        });
        start = offset + 2;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexpAll(eitherText, re) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const match of matchAll(text.value, re)){
        chunks.push({
            start,
            end: match.index
        });
        start = match[0].length + match.index;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexp(eitherText, re) {
    const text = asMappedString(eitherText);
    const m = text.value.match(re);
    if (m) {
        return mappedString(text, [
            {
                start: 0,
                end: m.index
            },
            {
                start: m.index + m[0].length,
                end: text.value.length
            }, 
        ]);
    } else {
        return text;
    }
}
function validateBoolean(value, _schema) {
    return typeof value === "boolean";
}
function validateNumber(value, _schema) {
    return typeof value === "number";
}
function validateString(value, schema) {
    if (typeof value !== "string") {
        return false;
    }
    if (schema.pattern === undefined) {
        return true;
    }
    if (value.match(new RegExp(schema.pattern))) {
        return true;
    } else {
        return false;
    }
}
function validateNull(value, _schema) {
    return value === null;
}
function validateEnum(value, schema) {
    return schema["enum"].indexOf(value) !== -1;
}
function validateOneOf(value, schema) {
    let count = 0;
    for (const subSchema of schema.oneOf){
        if (validate(value, subSchema)) {
            count += 1;
            if (count > 1) {
                return false;
            }
        }
    }
    return count === 1;
}
function validateAnyOf(value, schema) {
    for (const subSchema of schema.anyOf){
        if (validate(value, subSchema)) {
            return true;
        }
    }
    return false;
}
function validateAllOf(value, schema) {
    for (const subSchema of schema.allOf){
        if (!validate(value, subSchema)) {
            return false;
        }
    }
    return true;
}
function validateObject(value, schema) {
    if (typeof value !== "object" || Array.isArray(value) || value === null) {
        return false;
    }
    const inspectedProps = new Set();
    const ownProperties = new Set(Object.getOwnPropertyNames(value));
    if (schema.properties) {
        for (const [key, subSchema] of Object.entries(schema.properties)){
            if (ownProperties.has(key) && !validate(value[key], subSchema)) {
                return false;
            } else {
                inspectedProps.add(key);
            }
        }
    }
    if (schema.patternProperties) {
        for (const [key, subSchema] of Object.entries(schema.patternProperties)){
            const regexp4 = new RegExp(key);
            for (const [objectKey, val] of Object.entries(value)){
                if (objectKey.match(regexp4) && !validate(val, subSchema)) {
                    return false;
                } else {
                    inspectedProps.add(objectKey);
                }
            }
        }
    }
    if (schema.additionalProperties) {
        for (const [objectKey, val] of Object.entries(value)){
            if (inspectedProps.has(objectKey)) {
                continue;
            }
            if (!validate(val, schema.additionalProperties)) {
                return false;
            }
        }
    }
    for (const reqKey of schema.required || []){
        if (value[reqKey] === undefined) {
            return false;
        }
    }
    return true;
}
function validateArray(value, schema) {
    if (!Array.isArray(value)) {
        return false;
    }
    if (schema.items) {
        return value.every((entry)=>validate(entry, schema.items)
        );
    }
    return true;
}
function validate(value, schema) {
    if (schema === false) {
        return false;
    }
    if (schema === true) {
        return true;
    }
    const validators = {
        "boolean": validateBoolean,
        "number": validateNumber,
        "string": validateString,
        "null": validateNull,
        "enum": validateEnum,
        "oneOf": validateOneOf,
        "anyOf": validateAnyOf,
        "allOf": validateAllOf,
        "object": validateObject,
        "array": validateArray
    };
    schema = resolveSchema(schema);
    if (validators[schemaType(schema)]) {
        return validators[schemaType(schema)](value, schema);
    } else {
        throw new Error(`Don't know how to validate ${schema.type}`);
    }
}
const importMeta = {
    url: "file:///D:/a/quarto-cli/quarto-cli/src/core/lib/yaml-validation/staged-validator.ts",
    main: false
};
let _module = undefined;
let validatorModulePath = "";
async function ensureValidatorModule() {
    if (_module) {
        return _module;
    }
    if (validatorModulePath === "") {
        throw new Error("Internal Error: validator module path is not set");
    }
    const path59 = new URL(validatorModulePath, importMeta.url).href;
    const _mod = await import(path59);
    _module = _mod.default;
    return _module;
}
function setValidatorModulePath(path60) {
    validatorModulePath = path60;
}
let obtainFullValidator = (schema)=>undefined
;
function stagedValidator(schema) {
    schema = resolveSchema(schema);
    return async (value)=>{
        if (validate(value, schema)) {
            return [];
        }
        await ensureValidatorModule();
        const validator = _module[schema.$id] || obtainFullValidator(schema);
        if (validator === undefined) {
            throw new Error(`Internal error: ${schema.$id} not compiled and schema compiler not available`);
        }
        if (validator(value)) {
            throw new Error(`Internal error: validators disagree on schema ${schema.$id}`);
        }
        return JSON.parse(JSON.stringify(validator.errors));
    };
}
function getVerbatimInput(error8) {
    return error8.source.value.substring(error8.violatingObject.start, error8.violatingObject.end);
}
function navigate(path61, annotation, returnKey = false, pathIndex = 0) {
    if (annotation === undefined) {
        throw new Error("Can't navigate an undefined annotation");
    }
    if (pathIndex >= path61.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components  } = annotation;
        const searchKey = path61[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i40 = lastKeyIndex; i40 >= 0; i40 -= 2){
            const key = components[i40].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path61.length - 1) {
                    return navigate(path61, components[i40], returnKey, pathIndex + 1);
                } else {
                    return navigate(path61, components[i40 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        throw new Error(`Internal error: searchKey ${searchKey} (path: ${path61}) not found in mapping object`);
    } else if ([
        "sequence",
        "block_sequence",
        "flow_sequence"
    ].indexOf(annotation.kind) !== -1) {
        const searchKey = Number(path61[pathIndex]);
        return navigate(path61, annotation.components[searchKey], returnKey, pathIndex + 1);
    } else {
        throw new Error(`Internal error: unexpected kind ${annotation.kind}`);
    }
}
function navigateSchema1(path62, schema, pathIndex = 0) {
    if (schema.$ref) {
        schema = getSchemaDefinition(schema.$ref);
    }
    if (pathIndex >= path62.length - 1) {
        return [
            schema
        ];
    }
    const pathVal = path62[pathIndex];
    if (schema.allOf !== undefined) {
        return schema.allOf.map((s)=>navigateSchema1(path62, s, pathIndex)
        ).flat();
    } else if (pathVal === "patternProperties" && schema.patternProperties) {
        const key = path62[pathIndex + 1];
        const subSchema = schema.patternProperties[key];
        return navigateSchema1(path62, subSchema, pathIndex + 2);
    } else if (pathVal === "properties" && schema.properties) {
        const key = path62[pathIndex + 1];
        const subSchema = schema.properties[key];
        return navigateSchema1(path62, subSchema, pathIndex + 2);
    } else if (pathVal === "anyOf" && schema.anyOf) {
        const key = Number(path62[pathIndex + 1]);
        const subSchema = schema.anyOf[key];
        return navigateSchema1(path62, subSchema, pathIndex + 2);
    } else if (pathVal === "oneOf" && schema.oneOf) {
        const key = Number(path62[pathIndex + 1]);
        const subSchema = schema.oneOf[key];
        return navigateSchema1(path62, subSchema, pathIndex + 2);
    } else if (pathVal === "items" && schema.items) {
        const subSchema = schema.items;
        return navigateSchema1(path62, subSchema, pathIndex + 1);
    } else {
        return [];
    }
}
function isProperPrefix(a, b) {
    return b.length > a.length && b.substring(0, a.length) === a;
}
function groupBy(lst, f) {
    const record = {};
    const result = [];
    for (const el of lst){
        const key = f(el);
        if (record[key] === undefined) {
            const lst = [];
            const entry = {
                key,
                values: lst
            };
            record[key] = lst;
            result.push(entry);
        }
        record[key].push(el);
    }
    return result;
}
function groupByEntries(entries) {
    const result = [];
    for (const { values: values9  } of entries){
        result.push(...values9);
    }
    return result;
}
function narrowOneOfError(oneOf, suberrors) {
    const subschemaErrors = groupBy(suberrors.filter((error9)=>error9.schemaPath !== oneOf.schemaPath
    ), (error10)=>error10.schemaPath.substring(0, error10.schemaPath.lastIndexOf("/"))
    );
    const onlyAdditionalProperties = subschemaErrors.filter(({ values: values10  })=>values10.every((v)=>v.keyword === "additionalProperties"
        )
    );
    if (onlyAdditionalProperties.length) {
        return onlyAdditionalProperties[0].values;
    }
    return [];
}
function localizeAndPruneErrors(annotation, validationErrors, source, schema) {
    const result = [];
    const locF = mappedIndexToRowCol(source);
    let errorsPerInstanceList = groupBy(validationErrors, (error11)=>error11.instancePath
    );
    do {
        const newErrors = [];
        errorsPerInstanceList = errorsPerInstanceList.filter(({ key: pathA  })=>errorsPerInstanceList.filter(({ key: pathB  })=>isProperPrefix(pathA, pathB)
            ).length === 0
        );
        for (const { key: instancePath , values: errors2  } of errorsPerInstanceList){
            let errorsPerSchemaList = groupBy(errors2, (error12)=>error12.schemaPath
            );
            errorsPerSchemaList = errorsPerSchemaList.filter(({ key: pathA  })=>errorsPerSchemaList.filter(({ key: pathB  })=>isProperPrefix(pathB, pathA)
                ).length === 0
            );
            for (const error1 of groupByEntries(errorsPerSchemaList)){
                if (error1.hasBeenTransformed) {
                    continue;
                }
                if (error1.keyword === "oneOf") {
                    error1.hasBeenTransformed = true;
                    newErrors.push(...narrowOneOfError(error1, errors2));
                } else if (error1.keyword === "additionalProperties") {
                    error1.hasBeenTransformed = true;
                    newErrors.push({
                        ...error1,
                        instancePath: `${instancePath}/${error1.params.additionalProperty}`,
                        keyword: "_custom_invalidProperty",
                        message: `property ${error1.params.additionalProperty} not allowed in object`,
                        params: {
                            ...error1.params,
                            originalError: error1
                        },
                        schemaPath: error1.schemaPath.slice(0, -21)
                    });
                }
            }
        }
        if (newErrors.length) {
            errorsPerInstanceList.push(...groupBy(newErrors, (error13)=>error13.instancePath
            ));
        } else {
            break;
        }
    }while (true)
    for (const { key: instancePath , values: allErrors  } of errorsPerInstanceList){
        const path63 = instancePath.split("/").slice(1);
        const errors3 = allErrors.filter(({ schemaPath: pathA  })=>!(allErrors.filter(({ schemaPath: pathB  })=>isProperPrefix(pathB, pathA)
            ).length > 0)
        );
        for (const error14 of errors3){
            const returnKey = error14.keyword === "_custom_invalidProperty";
            const violatingObject = navigate(path63, annotation, returnKey);
            const schemaPath1 = error14.schemaPath.split("/").slice(1);
            let start = {
                line: 0,
                column: 0
            };
            let end = {
                line: 0,
                column: 0
            };
            if (source.value.length) {
                start = locF(violatingObject.start);
                end = locF(violatingObject.end);
            }
            let niceError = {
                heading: "",
                error: [],
                info: [],
                location: {
                    start,
                    end
                }
            };
            if (error14.keyword.startsWith("_custom_")) {
                niceError = {
                    ...niceError,
                    heading: error14.message === undefined ? "" : error14.message
                };
            } else {
                if (instancePath === "") {
                    niceError = {
                        ...niceError,
                        heading: `(top-level error) ${error14.message}`
                    };
                } else {
                    const errorSchema = error14.params && error14.params.schema || error14.parentSchema;
                    const innerSchema = errorSchema ? [
                        errorSchema
                    ] : navigateSchema1(schemaPath1.map(decodeURIComponent), schema);
                    if (innerSchema.length === 0) {
                        niceError = {
                            ...niceError,
                            heading: `Schema ${schemaPath1}: ${error14.message}`
                        };
                    } else {
                        const rawVerbatimInput = source.value.substring(violatingObject.start, violatingObject.end);
                        if (rawVerbatimInput.length === 0) {
                            niceError = {
                                ...niceError,
                                heading: `Empty value found where it must instead ${innerSchema.map((s)=>s.description
                                ).join(", ")}.`
                            };
                        } else {
                            const verbatimInput = quotedStringColor(source.value.substring(violatingObject.start, violatingObject.end));
                            niceError = {
                                ...niceError,
                                heading: `The value ${verbatimInput} must ${innerSchema.map((s)=>s.description
                                ).join(", ")}.`
                            };
                        }
                    }
                }
            }
            niceError.location = {
                start,
                end
            };
            addFileInfo(niceError, source);
            addInstancePathInfo(niceError, instancePath);
            result.push({
                instancePath,
                message: error14.message === undefined ? "" : error14.message,
                violatingObject,
                location: {
                    start,
                    end
                },
                source,
                ajvError: error14,
                niceError: niceError
            });
        }
    }
    result.sort((a, b)=>a.violatingObject.start - b.violatingObject.start
    );
    return result;
}
class YAMLSchema {
    schema;
    validate;
    errorHandlers;
    constructor(schema){
        this.errorHandlers = [];
        this.schema = schema;
        this.validate = stagedValidator(this.schema);
    }
    addHandler(handler) {
        this.errorHandlers.push(handler);
    }
    transformErrors(annotation, errors4) {
        return errors4.map((error15)=>{
            for (const handler of this.errorHandlers){
                error15 = handler(error15, annotation, this.schema);
            }
            return error15;
        });
    }
    async validateParse(src, annotation) {
        const validationErrors = await this.validate(annotation.result);
        if (validationErrors.length) {
            const localizedErrors = this.transformErrors(annotation, localizeAndPruneErrors(annotation, validationErrors, src, this.schema));
            return {
                result: annotation.result,
                errors: localizedErrors
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result, src, message, error16, log) {
        if (result.errors.length) {
            const locF = mappedIndexToRowCol(src);
            const nLines = lines(src.originalString).length;
            error16(message);
            for (const err of result.errors){
                let startO = err.violatingObject.start;
                let endO = err.violatingObject.end;
                while(src.mapClosest(startO) < src.originalString.length - 1 && src.originalString[src.mapClosest(startO)].match(/\s/)){
                    startO++;
                }
                while(src.mapClosest(endO) > src.mapClosest(startO) && src.originalString[src.mapClosest(endO)].match(/\s/)){
                    endO--;
                }
                const start = locF(startO);
                const end = locF(endO);
                const { prefixWidth , lines: lines1 ,  } = formatLineRange(src.originalString, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
                const contextLines = [];
                for (const { lineNumber , content , rawLine  } of lines1){
                    contextLines.push(content);
                    if (lineNumber >= start.line && lineNumber <= end.line) {
                        const startColumn = lineNumber > start.line ? 0 : start.column;
                        const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                        contextLines.push(" ".repeat(prefixWidth + startColumn) + blue1("~".repeat(endColumn - startColumn + 1)));
                    }
                }
                err.niceError.sourceContext = contextLines.join("\n");
                log(tidyverseFormatError(err.niceError));
            }
        }
        return result;
    }
    async validateParseWithErrors(src, annotation, message, error17, log) {
        const result = await this.validateParse(src, annotation);
        this.reportErrorsInSource(result, src, message, error17, log);
        return result;
    }
}
class PromiseQueue {
    queue = new Array();
    running = false;
    enqueue(promise, clearPending = false) {
        return new Promise((resolve4, reject)=>{
            if (clearPending) {
                this.queue.splice(0, this.queue.length);
            }
            this.queue.push({
                promise,
                resolve: resolve4,
                reject
            });
            this.dequeue();
        });
    }
    dequeue() {
        if (this.running) {
            return false;
        }
        const item = this.queue.shift();
        if (!item) {
            return false;
        }
        try {
            this.running = true;
            item.promise().then((value)=>{
                this.running = false;
                item.resolve(value);
                this.dequeue();
            }).catch((err)=>{
                this.running = false;
                item.reject(err);
                this.dequeue();
            });
        } catch (err) {
            this.running = false;
            item.reject(err);
            this.dequeue();
        }
        return true;
    }
}
function isEmptyValue(error18) {
    const rawVerbatimInput = getVerbatimInput(error18);
    return rawVerbatimInput.trim().length === 0;
}
function getLastFragment(instancePath) {
    const splitPath = instancePath.split("/");
    if (splitPath.length === 0) {
        return undefined;
    }
    const lastFragment = splitPath[splitPath.length - 1];
    if (lastFragment === "") {
        return undefined;
    }
    const maybeNumber = Number(lastFragment);
    if (!isNaN(maybeNumber)) {
        return maybeNumber;
    }
    return lastFragment;
}
function reindent(str60) {
    return str60;
}
function innerDescription(error19, parse, schema) {
    const schemaPath2 = error19.ajvError.schemaPath.split("/").slice(1);
    const errorSchema = error19.ajvError.params && error19.ajvError.params.schema || error19.ajvError.parentSchema;
    const innerSchema = errorSchema ? [
        errorSchema
    ] : navigateSchema(schemaPath2.map(decodeURIComponent), schema);
    return innerSchema.map((s)=>s.description
    ).join(", ");
}
function formatHeading(error20, parse10, schema) {
    const rawVerbatimInput = getVerbatimInput(error20);
    const verbatimInput = quotedStringColor(reindent(rawVerbatimInput));
    const empty2 = isEmptyValue(error20);
    const lastFragment = getLastFragment(error20.instancePath);
    switch(typeof lastFragment){
        case "undefined":
            if (empty2) {
                return "YAML object is missing.";
            } else {
                const innerDesc = innerDescription(error20, parse10, schema);
                return `YAML object ${verbatimInput} must instead ${innerDesc}`;
            }
        case "number":
            const innerDesc = innerDescription(error20, parse10, schema);
            if (empty2) {
                return `Array entry is empty but it must instead ${innerDesc}.`;
            } else {
                return `Array entry ${verbatimInput} must instead ${innerDesc}.`;
            }
        case "string":
            {
                const formatLastFragment = blue1(lastFragment);
                const innerDesc = innerDescription(error20, parse10, schema);
                if (empty2) {
                    return `Field ${formatLastFragment} is empty but it must instead ${innerDesc}`;
                } else {
                    return `Field ${formatLastFragment} is ${verbatimInput} but it must instead ${innerDesc}`;
                }
            }
    }
}
function improveErrorHeading(error21, parse11, schema) {
    return {
        ...error21,
        niceError: {
            ...error21.niceError,
            heading: formatHeading(error21, parse11, schema)
        }
    };
}
function setDefaultErrorHandlers(validator) {
    validator.addHandler(improveErrorHeading);
    validator.addHandler(checkForTypeMismatch);
    validator.addHandler(checkForBadBoolean);
}
function checkForTypeMismatch(error22, parse12, schema) {
    const rawVerbatimInput = getVerbatimInput(error22);
    const verbatimInput = quotedStringColor(rawVerbatimInput);
    if (error22.ajvError.keyword === "type" && rawVerbatimInput.length > 0) {
        const newError = {
            heading: formatHeading(error22, parse12, schema),
            error: [
                `The value ${verbatimInput} is a ${typeof error22.violatingObject.result}.`, 
            ],
            info: [],
            location: error22.niceError.location
        };
        addInstancePathInfo(newError, error22.ajvError.instancePath);
        addFileInfo(newError, error22.source);
        return {
            ...error22,
            niceError: newError
        };
    }
    return error22;
}
function checkForBadBoolean(error23, parse13, schema) {
    schema = error23.ajvError.params.schema;
    if (!(typeof error23.violatingObject.result === "string" && error23.ajvError.keyword === "type" && schema && schema.type === "boolean")) {
        return error23;
    }
    const strValue = error23.violatingObject.result;
    const verbatimInput = quotedStringColor(getVerbatimInput(error23));
    const yesses = new Set("y|Y|yes|Yes|YES|true|True|TRUE|on|On|ON".split("|"));
    const nos = new Set("n|N|no|No|NO|false|False|FALSE|off|Off|OFF".split("|"));
    let fix;
    if (yesses.has(strValue)) {
        fix = true;
    } else if (nos.has(strValue)) {
        fix = false;
    } else {
        return error23;
    }
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `Quarto uses YAML 1.2, which interprets booleans strictly.`;
    const suggestion2 = `Try using ${quotedStringColor(String(fix))} instead.`;
    const newError = {
        heading: formatHeading(error23, parse13, schema),
        error: [
            errorMessage
        ],
        info: [],
        location: error23.niceError.location
    };
    addInstancePathInfo(newError, error23.ajvError.instancePath);
    addFileInfo(newError, error23.source);
    newError.info.push(suggestion1, suggestion2);
    return {
        ...error23,
        niceError: newError
    };
}
const yamlValidators = {};
const validatorQueues = {};
function getSchemaName(schema) {
    const schemaName = schema["$id"] || schema["$ref"];
    if (schemaName === undefined) {
        throw new Error("Expected schema to be named");
    }
    return schemaName;
}
function getValidator(schema) {
    const schemaName = getSchemaName(schema);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema);
    yamlValidators[schemaName] = validator;
    setDefaultErrorHandlers(validator);
    return validator;
}
async function withValidator(schema, fun) {
    const schemaName = getSchemaName(schema);
    if (validatorQueues[schemaName] === undefined) {
        validatorQueues[schemaName] = new PromiseQueue();
    }
    const queue = validatorQueues[schemaName];
    let result;
    let error24;
    await queue.enqueue(async ()=>{
        try {
            const validator = getValidator(schema);
            result = await fun(validator);
        } catch (e) {
            error24 = e;
        }
    });
    if (error24 !== undefined) {
        throw error24;
    }
    return result;
}
function addValidatorErrorHandler(schema, handler) {
    return withValidator(schema, async (validator)=>{
        validator.addHandler(handler);
    });
}
class YAMLError extends Error {
    mark;
    constructor(message = "(unknown reason)", mark = ""){
        super(`${message} ${mark}`);
        this.mark = mark;
        this.name = this.constructor.name;
    }
    toString(_compact) {
        return `${this.name}: ${this.message} ${this.mark}`;
    }
}
function isBoolean(value) {
    return typeof value === "boolean" || value instanceof Boolean;
}
function isObject1(value) {
    return value !== null && typeof value === "object";
}
function repeat(str61, count) {
    let result = "";
    for(let cycle = 0; cycle < count; cycle++){
        result += str61;
    }
    return result;
}
function isNegativeZero(i41) {
    return i41 === 0 && Number.NEGATIVE_INFINITY === 1 / i41;
}
class Mark {
    name;
    buffer;
    position;
    line;
    column;
    constructor(name83, buffer, position, line, column){
        this.name = name83;
        this.buffer = buffer;
        this.position = position;
        this.line = line;
        this.column = column;
    }
    getSnippet(indent = 4, maxLength = 75) {
        if (!this.buffer) return null;
        let head = "";
        let start = this.position;
        while(start > 0 && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(start - 1)) === -1){
            start -= 1;
            if (this.position - start > maxLength / 2 - 1) {
                head = " ... ";
                start += 5;
                break;
            }
        }
        let tail = "";
        let end = this.position;
        while(end < this.buffer.length && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(end)) === -1){
            end += 1;
            if (end - this.position > maxLength / 2 - 1) {
                tail = " ... ";
                end -= 5;
                break;
            }
        }
        const snippet = this.buffer.slice(start, end);
        return `${repeat(" ", indent)}${head}${snippet}${tail}\n${repeat(" ", indent + this.position - start + head.length)}^`;
    }
    toString(compact) {
        let snippet, where = "";
        if (this.name) {
            where += `in "${this.name}" `;
        }
        where += `at line ${this.line + 1}, column ${this.column + 1}`;
        if (!compact) {
            snippet = this.getSnippet();
            if (snippet) {
                where += `:\n${snippet}`;
            }
        }
        return where;
    }
}
function compileList(schema, name84, result) {
    const exclude = [];
    for (const includedSchema of schema.include){
        result = compileList(includedSchema, name84, result);
    }
    for (const currentType of schema[name84]){
        for(let previousIndex = 0; previousIndex < result.length; previousIndex++){
            const previousType = result[previousIndex];
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind) {
                exclude.push(previousIndex);
            }
        }
        result.push(currentType);
    }
    return result.filter((_type, index)=>!exclude.includes(index)
    );
}
function compileMap(...typesList) {
    const result = {
        fallback: {},
        mapping: {},
        scalar: {},
        sequence: {}
    };
    for (const types9 of typesList){
        for (const type of types9){
            if (type.kind !== null) {
                result[type.kind][type.tag] = result["fallback"][type.tag] = type;
            }
        }
    }
    return result;
}
class Schema {
    static SCHEMA_DEFAULT;
    implicit;
    explicit;
    include;
    compiledImplicit;
    compiledExplicit;
    compiledTypeMap;
    constructor(definition){
        this.explicit = definition.explicit || [];
        this.implicit = definition.implicit || [];
        this.include = definition.include || [];
        for (const type of this.implicit){
            if (type.loadKind && type.loadKind !== "scalar") {
                throw new YAMLError("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
            }
        }
        this.compiledImplicit = compileList(this, "implicit", []);
        this.compiledExplicit = compileList(this, "explicit", []);
        this.compiledTypeMap = compileMap(this.compiledImplicit, this.compiledExplicit);
    }
    extend(definition) {
        return new Schema({
            implicit: [
                ...new Set([
                    ...this.implicit,
                    ...definition?.implicit ?? []
                ]), 
            ],
            explicit: [
                ...new Set([
                    ...this.explicit,
                    ...definition?.explicit ?? []
                ]), 
            ],
            include: [
                ...new Set([
                    ...this.include,
                    ...definition?.include ?? []
                ])
            ]
        });
    }
    static create() {}
}
const DEFAULT_RESOLVE = ()=>true
;
const DEFAULT_CONSTRUCT = (data18)=>data18
;
function checkTagFormat(tag) {
    return tag;
}
class Type1 {
    tag;
    kind = null;
    instanceOf;
    predicate;
    represent;
    defaultStyle;
    styleAliases;
    loadKind;
    constructor(tag, options){
        this.tag = checkTagFormat(tag);
        if (options) {
            this.kind = options.kind;
            this.resolve = options.resolve || DEFAULT_RESOLVE;
            this.construct = options.construct || DEFAULT_CONSTRUCT;
            this.instanceOf = options.instanceOf;
            this.predicate = options.predicate;
            this.represent = options.represent;
            this.defaultStyle = options.defaultStyle;
            this.styleAliases = options.styleAliases;
        }
    }
    resolve = ()=>true
    ;
    construct = (data19)=>data19
    ;
}
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data20) {
    if (data20 === null) return false;
    let code27;
    let bitlen = 0;
    const max = data20.length;
    const map4 = BASE64_MAP;
    for(let idx = 0; idx < max; idx++){
        code27 = map4.indexOf(data20.charAt(idx));
        if (code27 > 64) continue;
        if (code27 < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data21) {
    const input = data21.replace(/[\r\n=]/g, "");
    const max = input.length;
    const map5 = BASE64_MAP;
    const result = [];
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result.push(bits >> 16 & 255);
            result.push(bits >> 8 & 255);
            result.push(bits & 255);
        }
        bits = bits << 6 | map5.indexOf(input.charAt(idx));
    }
    const tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result.push(bits >> 16 & 255);
        result.push(bits >> 8 & 255);
        result.push(bits & 255);
    } else if (tailbits === 18) {
        result.push(bits >> 10 & 255);
        result.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result.push(bits >> 4 & 255);
    }
    return new Buffer(new Uint8Array(result));
}
function representYamlBinary(object) {
    const max = object.length;
    const map6 = BASE64_MAP;
    let result = "";
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result += map6[bits >> 18 & 63];
            result += map6[bits >> 12 & 63];
            result += map6[bits >> 6 & 63];
            result += map6[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    const tail = max % 3;
    if (tail === 0) {
        result += map6[bits >> 18 & 63];
        result += map6[bits >> 12 & 63];
        result += map6[bits >> 6 & 63];
        result += map6[bits & 63];
    } else if (tail === 2) {
        result += map6[bits >> 10 & 63];
        result += map6[bits >> 4 & 63];
        result += map6[bits << 2 & 63];
        result += map6[64];
    } else if (tail === 1) {
        result += map6[bits >> 2 & 63];
        result += map6[bits << 4 & 63];
        result += map6[64];
        result += map6[64];
    }
    return result;
}
function isBinary(obj) {
    const buf = new Buffer();
    try {
        if (0 > buf.readFromSync(obj)) return true;
        return false;
    } catch  {
        return false;
    } finally{
        buf.reset();
    }
}
const binary = new Type1("tag:yaml.org,2002:binary", {
    construct: constructYamlBinary,
    kind: "scalar",
    predicate: isBinary,
    represent: representYamlBinary,
    resolve: resolveYamlBinary
});
function resolveYamlBoolean(data22) {
    const max = data22.length;
    return max === 4 && (data22 === "true" || data22 === "True" || data22 === "TRUE") || max === 5 && (data22 === "false" || data22 === "False" || data22 === "FALSE");
}
function constructYamlBoolean(data23) {
    return data23 === "true" || data23 === "True" || data23 === "TRUE";
}
const bool = new Type1("tag:yaml.org,2002:bool", {
    construct: constructYamlBoolean,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isBoolean,
    represent: {
        lowercase (object) {
            return object ? "true" : "false";
        },
        uppercase (object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase (object) {
            return object ? "True" : "False";
        }
    },
    resolve: resolveYamlBoolean
});
const YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + "|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + "|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + "|[-+]?\\.(?:inf|Inf|INF)" + "|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data24) {
    if (!YAML_FLOAT_PATTERN.test(data24) || data24[data24.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data25) {
    let value = data25.replace(/_/g, "").toLowerCase();
    const sign = value[0] === "-" ? -1 : 1;
    const digits1 = [];
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    }
    if (value === ".nan") {
        return NaN;
    }
    if (value.indexOf(":") >= 0) {
        value.split(":").forEach((v)=>{
            digits1.unshift(parseFloat(v));
        });
        let valueNb = 0;
        let base2 = 1;
        digits1.forEach((d)=>{
            valueNb += d * base2;
            base2 *= 60;
        });
        return sign * valueNb;
    }
    return sign * parseFloat(value);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (isNegativeZero(object)) {
        return "-0.0";
    }
    const res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || isNegativeZero(object));
}
const __float = new Type1("tag:yaml.org,2002:float", {
    construct: constructYamlFloat,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isFloat,
    represent: representYamlFloat,
    resolve: resolveYamlFloat
});
function reconstructFunction(code28) {
    const func1 = new Function(`return ${code28}`)();
    if (!(func1 instanceof Function)) {
        throw new TypeError(`Expected function but got ${typeof func1}: ${code28}`);
    }
    return func1;
}
new Type1("tag:yaml.org,2002:js/function", {
    kind: "scalar",
    resolve (data26) {
        if (data26 === null) {
            return false;
        }
        try {
            reconstructFunction(`${data26}`);
            return true;
        } catch (_err) {
            return false;
        }
    },
    construct (data27) {
        return reconstructFunction(data27);
    },
    predicate (object) {
        return object instanceof Function;
    },
    represent (object) {
        return object.toString();
    }
});
function isHexCode(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode(c) {
    return 48 <= c && c <= 55;
}
function isDecCode(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger(data28) {
    const max = data28.length;
    let index = 0;
    let hasDigits = false;
    if (!max) return false;
    let ch = data28[index];
    if (ch === "-" || ch === "+") {
        ch = data28[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data28[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data28[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data28[index];
                if (ch === "_") continue;
                if (!isHexCode(data28.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for(; index < max; index++){
            ch = data28[index];
            if (ch === "_") continue;
            if (!isOctCode(data28.charCodeAt(index))) return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data28[index];
        if (ch === "_") continue;
        if (ch === ":") break;
        if (!isDecCode(data28.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    if (ch !== ":") return true;
    return /^(:[0-5]?[0-9])+$/.test(data28.slice(index));
}
function constructYamlInteger(data29) {
    let value = data29;
    const digits2 = [];
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value, 16);
        return sign * parseInt(value, 8);
    }
    if (value.indexOf(":") !== -1) {
        value.split(":").forEach((v)=>{
            digits2.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base3 = 1;
        digits2.forEach((d)=>{
            valueInt += d * base3;
            base3 *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value, 10);
}
function isInteger(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !isNegativeZero(object);
}
const __int = new Type1("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger,
    represent: {
        binary (obj) {
            return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
        },
        octal (obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal (obj) {
            return obj.toString(10);
        },
        hexadecimal (obj) {
            return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        }
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ],
        octal: [
            8,
            "oct"
        ]
    }
});
const map = new Type1("tag:yaml.org,2002:map", {
    construct (data30) {
        return data30 !== null ? data30 : {};
    },
    kind: "mapping"
});
function resolveYamlMerge(data31) {
    return data31 === "<<" || data31 === null;
}
const merge = new Type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
function resolveYamlNull(data32) {
    const max = data32.length;
    return max === 1 && data32 === "~" || max === 4 && (data32 === "null" || data32 === "Null" || data32 === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull(object) {
    return object === null;
}
const nil = new Type1("tag:yaml.org,2002:null", {
    construct: constructYamlNull,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isNull,
    represent: {
        canonical () {
            return "~";
        },
        lowercase () {
            return "null";
        },
        uppercase () {
            return "NULL";
        },
        camelcase () {
            return "Null";
        }
    },
    resolve: resolveYamlNull
});
const { hasOwn  } = Object;
const _toString = Object.prototype.toString;
function resolveYamlOmap(data33) {
    const objectKeys = [];
    let pairKey = "";
    let pairHasKey = false;
    for (const pair of data33){
        pairHasKey = false;
        if (_toString.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (hasOwn(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data34) {
    return data34 !== null ? data34 : [];
}
const omap = new Type1("tag:yaml.org,2002:omap", {
    construct: constructYamlOmap,
    kind: "sequence",
    resolve: resolveYamlOmap
});
const _toString1 = Object.prototype.toString;
function resolveYamlPairs(data35) {
    const result = Array.from({
        length: data35.length
    });
    for(let index = 0; index < data35.length; index++){
        const pair = data35[index];
        if (_toString1.call(pair) !== "[object Object]") return false;
        const keys2 = Object.keys(pair);
        if (keys2.length !== 1) return false;
        result[index] = [
            keys2[0],
            pair[keys2[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data36) {
    if (data36 === null) return [];
    const result = Array.from({
        length: data36.length
    });
    for(let index = 0; index < data36.length; index += 1){
        const pair = data36[index];
        const keys3 = Object.keys(pair);
        result[index] = [
            keys3[0],
            pair[keys3[0]]
        ];
    }
    return result;
}
const pairs = new Type1("tag:yaml.org,2002:pairs", {
    construct: constructYamlPairs,
    kind: "sequence",
    resolve: resolveYamlPairs
});
const REGEXP = /^\/(?<regexp>[\s\S]+)\/(?<modifiers>[gismuy]*)$/;
const regexp = new Type1("tag:yaml.org,2002:js/regexp", {
    kind: "scalar",
    resolve (data37) {
        if (data37 === null || !data37.length) {
            return false;
        }
        const regexp1 = `${data37}`;
        if (regexp1.charAt(0) === "/") {
            if (!REGEXP.test(data37)) {
                return false;
            }
            const modifiers = [
                ...regexp1.match(REGEXP)?.groups?.modifiers ?? ""
            ];
            if (new Set(modifiers).size < modifiers.length) {
                return false;
            }
        }
        return true;
    },
    construct (data38) {
        const { regexp: regexp2 = `${data38}` , modifiers =""  } = `${data38}`.match(REGEXP)?.groups ?? {};
        return new RegExp(regexp2, modifiers);
    },
    predicate (object) {
        return object instanceof RegExp;
    },
    represent (object) {
        return object.toString();
    }
});
const seq = new Type1("tag:yaml.org,2002:seq", {
    construct (data39) {
        return data39 !== null ? data39 : [];
    },
    kind: "sequence"
});
const { hasOwn: hasOwn1  } = Object;
function resolveYamlSet(data40) {
    if (data40 === null) return true;
    for(const key in data40){
        if (hasOwn1(data40, key)) {
            if (data40[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data41) {
    return data41 !== null ? data41 : {};
}
const set = new Type1("tag:yaml.org,2002:set", {
    construct: constructYamlSet,
    kind: "mapping",
    resolve: resolveYamlSet
});
const str = new Type1("tag:yaml.org,2002:str", {
    construct (data42) {
        return data42 !== null ? data42 : "";
    },
    kind: "scalar"
});
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9])" + "-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9]?)" + "-([0-9][0-9]?)" + "(?:[Tt]|[ \\t]+)" + "([0-9][0-9]?)" + ":([0-9][0-9])" + ":([0-9][0-9])" + "(?:\\.([0-9]*))?" + "(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + "(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data43) {
    if (data43 === null) return false;
    if (YAML_DATE_REGEXP.exec(data43) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data43) !== null) return true;
    return false;
}
function constructYamlTimestamp(data44) {
    let match = YAML_DATE_REGEXP.exec(data44);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data44);
    if (match === null) throw new Error("Date resolve error");
    const year = +match[1];
    const month = +match[2] - 1;
    const day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    const hour = +match[4];
    const minute = +match[5];
    const second = +match[6];
    let fraction = 0;
    if (match[7]) {
        let partFraction = match[7].slice(0, 3);
        while(partFraction.length < 3){
            partFraction += "0";
        }
        fraction = +partFraction;
    }
    let delta = null;
    if (match[9]) {
        const tzHour = +match[10];
        const tzMinute = +(match[11] || 0);
        delta = (tzHour * 60 + tzMinute) * 60000;
        if (match[9] === "-") delta = -delta;
    }
    const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp(date) {
    return date.toISOString();
}
const timestamp = new Type1("tag:yaml.org,2002:timestamp", {
    construct: constructYamlTimestamp,
    instanceOf: Date,
    kind: "scalar",
    represent: representYamlTimestamp,
    resolve: resolveYamlTimestamp
});
const undefinedType = new Type1("tag:yaml.org,2002:js/undefined", {
    kind: "scalar",
    resolve () {
        return true;
    },
    construct () {
        return undefined;
    },
    predicate (object) {
        return typeof object === "undefined";
    },
    represent () {
        return "";
    }
});
const failsafe = new Schema({
    explicit: [
        str,
        seq,
        map
    ]
});
const json = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ]
});
const core = new Schema({
    include: [
        json
    ]
});
const def = new Schema({
    explicit: [
        binary,
        omap,
        pairs,
        set
    ],
    implicit: [
        timestamp,
        merge
    ],
    include: [
        core
    ]
});
new Schema({
    explicit: [
        regexp,
        undefinedType
    ],
    include: [
        def
    ]
});
class State1 {
    schema;
    constructor(schema = def){
        this.schema = schema;
    }
}
class LoaderState extends State1 {
    input;
    documents = [];
    length;
    lineIndent = 0;
    lineStart = 0;
    position = 0;
    line = 0;
    filename;
    onWarning;
    legacy;
    json;
    listener;
    implicitTypes;
    typeMap;
    version;
    checkLineBreaks;
    tagMap;
    anchorMap;
    tag;
    anchor;
    kind;
    result = "";
    constructor(input, { filename , schema , onWarning , legacy =false , json: json1 = false , listener =null  }){
        super(schema);
        this.input = input;
        this.filename = filename;
        this.onWarning = onWarning;
        this.legacy = legacy;
        this.json = json1;
        this.listener = listener;
        this.implicitTypes = this.schema.compiledImplicit;
        this.typeMap = this.schema.compiledTypeMap;
        this.length = input.length;
    }
}
const { hasOwn: hasOwn2  } = Object;
const CONTEXT_BLOCK_IN = 3;
const CONTEXT_BLOCK_OUT = 4;
const CHOMPING_STRIP = 2;
const CHOMPING_KEEP = 3;
const PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
const PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
const PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
const PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
const PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function isEOL(c) {
    return c === 10 || c === 13;
}
function isWhiteSpace(c) {
    return c === 9 || c === 32;
}
function isWsOrEol(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function isFlowIndicator(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    const lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 48 ? "\x00" : c === 97 ? "\x07" : c === 98 ? "\x08" : c === 116 ? "\x09" : c === 9 ? "\x09" : c === 110 ? "\x0A" : c === 118 ? "\x0B" : c === 102 ? "\x0C" : c === 114 ? "\x0D" : c === 101 ? "\x1B" : c === 32 ? " " : c === 34 ? "\x22" : c === 47 ? "/" : c === 92 ? "\x5C" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
const simpleEscapeCheck = Array.from({
    length: 256
});
const simpleEscapeMap = Array.from({
    length: 256
});
for(let i77 = 0; i77 < 256; i77++){
    simpleEscapeCheck[i77] = simpleEscapeSequence(i77) ? 1 : 0;
    simpleEscapeMap[i77] = simpleEscapeSequence(i77);
}
function generateError(state2, message) {
    return new YAMLError(message, new Mark(state2.filename, state2.input, state2.position, state2.line, state2.position - state2.lineStart));
}
function throwError(state3, message) {
    throw generateError(state3, message);
}
function throwWarning(state4, message) {
    if (state4.onWarning) {
        state4.onWarning.call(null, generateError(state4, message));
    }
}
const directiveHandlers = {
    YAML (state5, _name, ...args) {
        if (state5.version !== null) {
            return throwError(state5, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            return throwError(state5, "YAML directive accepts exactly one argument");
        }
        const match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            return throwError(state5, "ill-formed argument of the YAML directive");
        }
        const major = parseInt(match[1], 10);
        const minor = parseInt(match[2], 10);
        if (major !== 1) {
            return throwError(state5, "unacceptable YAML version of the document");
        }
        state5.version = args[0];
        state5.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            return throwWarning(state5, "unsupported YAML version of the document");
        }
    },
    TAG (state6, _name, ...args) {
        if (args.length !== 2) {
            return throwError(state6, "TAG directive accepts exactly two arguments");
        }
        const handle = args[0];
        const prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            return throwError(state6, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (state6.tagMap && hasOwn2(state6.tagMap, handle)) {
            return throwError(state6, `there is a previously declared suffix for "${handle}" tag handle`);
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            return throwError(state6, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        if (typeof state6.tagMap === "undefined") {
            state6.tagMap = {};
        }
        state6.tagMap[handle] = prefix;
    }
};
function captureSegment(state7, start, end, checkJson) {
    let result;
    if (start < end) {
        result = state7.input.slice(start, end);
        if (checkJson) {
            for(let position = 0, length = result.length; position < length; position++){
                const character = result.charCodeAt(position);
                if (!(character === 9 || 32 <= character && character <= 1114111)) {
                    return throwError(state7, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(result)) {
            return throwError(state7, "the stream contains non-printable characters");
        }
        state7.result += result;
    }
}
function mergeMappings(state8, destination, source, overridableKeys) {
    if (!isObject1(source)) {
        return throwError(state8, "cannot merge mappings; the provided source object is unacceptable");
    }
    const keys4 = Object.keys(source);
    for(let i1 = 0, len1 = keys4.length; i1 < len1; i1++){
        const key = keys4[i1];
        if (!hasOwn2(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state9, result, overridableKeys, keyTag, keyNode, valueNode, startLine, startPos) {
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(let index = 0, quantity = keyNode.length; index < quantity; index++){
            if (Array.isArray(keyNode[index])) {
                return throwError(state9, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (result === null) {
        result = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(let index = 0, quantity = valueNode.length; index < quantity; index++){
                mergeMappings(state9, result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings(state9, result, valueNode, overridableKeys);
        }
    } else {
        if (!state9.json && !hasOwn2(overridableKeys, keyNode) && hasOwn2(result, keyNode)) {
            state9.line = startLine || state9.line;
            state9.position = startPos || state9.position;
            return throwError(state9, "duplicated mapping key");
        }
        result[keyNode] = valueNode;
        delete overridableKeys[keyNode];
    }
    return result;
}
function readLineBreak(state10) {
    const ch = state10.input.charCodeAt(state10.position);
    if (ch === 10) {
        state10.position++;
    } else if (ch === 13) {
        state10.position++;
        if (state10.input.charCodeAt(state10.position) === 10) {
            state10.position++;
        }
    } else {
        return throwError(state10, "a line break is expected");
    }
    state10.line += 1;
    state10.lineStart = state10.position;
}
function skipSeparationSpace(state11, allowComments, checkIndent) {
    let lineBreaks = 0, ch = state11.input.charCodeAt(state11.position);
    while(ch !== 0){
        while(isWhiteSpace(ch)){
            ch = state11.input.charCodeAt(++state11.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state11.input.charCodeAt(++state11.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (isEOL(ch)) {
            readLineBreak(state11);
            ch = state11.input.charCodeAt(state11.position);
            lineBreaks++;
            state11.lineIndent = 0;
            while(ch === 32){
                state11.lineIndent++;
                ch = state11.input.charCodeAt(++state11.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state11.lineIndent < checkIndent) {
        throwWarning(state11, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state12) {
    let _position = state12.position;
    let ch = state12.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state12.input.charCodeAt(_position + 1) && ch === state12.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state12.input.charCodeAt(_position);
        if (ch === 0 || isWsOrEol(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state13, count) {
    if (count === 1) {
        state13.result += " ";
    } else if (count > 1) {
        state13.result += repeat("\n", count - 1);
    }
}
function readPlainScalar(state14, nodeIndent, withinFlowCollection) {
    const kind = state14.kind;
    const result = state14.result;
    let ch = state14.input.charCodeAt(state14.position);
    if (isWsOrEol(ch) || isFlowIndicator(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    let following;
    if (ch === 63 || ch === 45) {
        following = state14.input.charCodeAt(state14.position + 1);
        if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
            return false;
        }
    }
    state14.kind = "scalar";
    state14.result = "";
    let captureEnd, captureStart = captureEnd = state14.position;
    let hasPendingContent = false;
    let line = 0;
    while(ch !== 0){
        if (ch === 58) {
            following = state14.input.charCodeAt(state14.position + 1);
            if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
                break;
            }
        } else if (ch === 35) {
            const preceding = state14.input.charCodeAt(state14.position - 1);
            if (isWsOrEol(preceding)) {
                break;
            }
        } else if (state14.position === state14.lineStart && testDocumentSeparator(state14) || withinFlowCollection && isFlowIndicator(ch)) {
            break;
        } else if (isEOL(ch)) {
            line = state14.line;
            const lineStart = state14.lineStart;
            const lineIndent = state14.lineIndent;
            skipSeparationSpace(state14, false, -1);
            if (state14.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state14.input.charCodeAt(state14.position);
                continue;
            } else {
                state14.position = captureEnd;
                state14.line = line;
                state14.lineStart = lineStart;
                state14.lineIndent = lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state14, captureStart, captureEnd, false);
            writeFoldedLines(state14, state14.line - line);
            captureStart = captureEnd = state14.position;
            hasPendingContent = false;
        }
        if (!isWhiteSpace(ch)) {
            captureEnd = state14.position + 1;
        }
        ch = state14.input.charCodeAt(++state14.position);
    }
    captureSegment(state14, captureStart, captureEnd, false);
    if (state14.result) {
        return true;
    }
    state14.kind = kind;
    state14.result = result;
    return false;
}
function readSingleQuotedScalar(state15, nodeIndent) {
    let ch, captureStart, captureEnd;
    ch = state15.input.charCodeAt(state15.position);
    if (ch !== 39) {
        return false;
    }
    state15.kind = "scalar";
    state15.result = "";
    state15.position++;
    captureStart = captureEnd = state15.position;
    while((ch = state15.input.charCodeAt(state15.position)) !== 0){
        if (ch === 39) {
            captureSegment(state15, captureStart, state15.position, true);
            ch = state15.input.charCodeAt(++state15.position);
            if (ch === 39) {
                captureStart = state15.position;
                state15.position++;
                captureEnd = state15.position;
            } else {
                return true;
            }
        } else if (isEOL(ch)) {
            captureSegment(state15, captureStart, captureEnd, true);
            writeFoldedLines(state15, skipSeparationSpace(state15, false, nodeIndent));
            captureStart = captureEnd = state15.position;
        } else if (state15.position === state15.lineStart && testDocumentSeparator(state15)) {
            return throwError(state15, "unexpected end of the document within a single quoted scalar");
        } else {
            state15.position++;
            captureEnd = state15.position;
        }
    }
    return throwError(state15, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state16, nodeIndent) {
    let ch = state16.input.charCodeAt(state16.position);
    if (ch !== 34) {
        return false;
    }
    state16.kind = "scalar";
    state16.result = "";
    state16.position++;
    let captureEnd, captureStart = captureEnd = state16.position;
    let tmp;
    while((ch = state16.input.charCodeAt(state16.position)) !== 0){
        if (ch === 34) {
            captureSegment(state16, captureStart, state16.position, true);
            state16.position++;
            return true;
        }
        if (ch === 92) {
            captureSegment(state16, captureStart, state16.position, true);
            ch = state16.input.charCodeAt(++state16.position);
            if (isEOL(ch)) {
                skipSeparationSpace(state16, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state16.result += simpleEscapeMap[ch];
                state16.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                let hexLength = tmp;
                let hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state16.input.charCodeAt(++state16.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        return throwError(state16, "expected hexadecimal character");
                    }
                }
                state16.result += charFromCodepoint(hexResult);
                state16.position++;
            } else {
                return throwError(state16, "unknown escape sequence");
            }
            captureStart = captureEnd = state16.position;
        } else if (isEOL(ch)) {
            captureSegment(state16, captureStart, captureEnd, true);
            writeFoldedLines(state16, skipSeparationSpace(state16, false, nodeIndent));
            captureStart = captureEnd = state16.position;
        } else if (state16.position === state16.lineStart && testDocumentSeparator(state16)) {
            return throwError(state16, "unexpected end of the document within a double quoted scalar");
        } else {
            state16.position++;
            captureEnd = state16.position;
        }
    }
    return throwError(state16, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state17, nodeIndent) {
    let ch = state17.input.charCodeAt(state17.position);
    let terminator;
    let isMapping = true;
    let result = {};
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        result = [];
    } else if (ch === 123) {
        terminator = 125;
    } else {
        return false;
    }
    if (state17.anchor !== null && typeof state17.anchor != "undefined" && typeof state17.anchorMap != "undefined") {
        state17.anchorMap[state17.anchor] = result;
    }
    ch = state17.input.charCodeAt(++state17.position);
    const tag = state17.tag, anchor = state17.anchor;
    let readNext = true;
    let valueNode, keyNode, keyTag = keyNode = valueNode = null, isExplicitPair, isPair = isExplicitPair = false;
    let following = 0, line = 0;
    const overridableKeys = {};
    while(ch !== 0){
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if (ch === terminator) {
            state17.position++;
            state17.tag = tag;
            state17.anchor = anchor;
            state17.kind = isMapping ? "mapping" : "sequence";
            state17.result = result;
            return true;
        }
        if (!readNext) {
            return throwError(state17, "missed comma between flow collection entries");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state17.input.charCodeAt(state17.position + 1);
            if (isWsOrEol(following)) {
                isPair = isExplicitPair = true;
                state17.position++;
                skipSeparationSpace(state17, true, nodeIndent);
            }
        }
        line = state17.line;
        composeNode(state17, nodeIndent, 1, false, true);
        keyTag = state17.tag || null;
        keyNode = state17.result;
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if ((isExplicitPair || state17.line === line) && ch === 58) {
            isPair = true;
            ch = state17.input.charCodeAt(++state17.position);
            skipSeparationSpace(state17, true, nodeIndent);
            composeNode(state17, nodeIndent, 1, false, true);
            valueNode = state17.result;
        }
        if (isMapping) {
            storeMappingPair(state17, result, overridableKeys, keyTag, keyNode, valueNode);
        } else if (isPair) {
            result.push(storeMappingPair(state17, null, overridableKeys, keyTag, keyNode, valueNode));
        } else {
            result.push(keyNode);
        }
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if (ch === 44) {
            readNext = true;
            ch = state17.input.charCodeAt(++state17.position);
        } else {
            readNext = false;
        }
    }
    return throwError(state17, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state18, nodeIndent) {
    let chomping = 1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false;
    let ch = state18.input.charCodeAt(state18.position);
    let folding = false;
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state18.kind = "scalar";
    state18.result = "";
    let tmp = 0;
    while(ch !== 0){
        ch = state18.input.charCodeAt(++state18.position);
        if (ch === 43 || ch === 45) {
            if (1 === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                return throwError(state18, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                return throwError(state18, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                return throwError(state18, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (isWhiteSpace(ch)) {
        do {
            ch = state18.input.charCodeAt(++state18.position);
        }while (isWhiteSpace(ch))
        if (ch === 35) {
            do {
                ch = state18.input.charCodeAt(++state18.position);
            }while (!isEOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state18);
        state18.lineIndent = 0;
        ch = state18.input.charCodeAt(state18.position);
        while((!detectedIndent || state18.lineIndent < textIndent) && ch === 32){
            state18.lineIndent++;
            ch = state18.input.charCodeAt(++state18.position);
        }
        if (!detectedIndent && state18.lineIndent > textIndent) {
            textIndent = state18.lineIndent;
        }
        if (isEOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state18.lineIndent < textIndent) {
            if (chomping === 3) {
                state18.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === 1) {
                if (didReadContent) {
                    state18.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (isWhiteSpace(ch)) {
                atMoreIndented = true;
                state18.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state18.result += repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state18.result += " ";
                }
            } else {
                state18.result += repeat("\n", emptyLines);
            }
        } else {
            state18.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        const captureStart = state18.position;
        while(!isEOL(ch) && ch !== 0){
            ch = state18.input.charCodeAt(++state18.position);
        }
        captureSegment(state18, captureStart, state18.position, false);
    }
    return true;
}
function readBlockSequence(state19, nodeIndent) {
    let line, following, detected = false, ch;
    const tag = state19.tag, anchor = state19.anchor, result = [];
    if (state19.anchor !== null && typeof state19.anchor !== "undefined" && typeof state19.anchorMap !== "undefined") {
        state19.anchorMap[state19.anchor] = result;
    }
    ch = state19.input.charCodeAt(state19.position);
    while(ch !== 0){
        if (ch !== 45) {
            break;
        }
        following = state19.input.charCodeAt(state19.position + 1);
        if (!isWsOrEol(following)) {
            break;
        }
        detected = true;
        state19.position++;
        if (skipSeparationSpace(state19, true, -1)) {
            if (state19.lineIndent <= nodeIndent) {
                result.push(null);
                ch = state19.input.charCodeAt(state19.position);
                continue;
            }
        }
        line = state19.line;
        composeNode(state19, nodeIndent, 3, false, true);
        result.push(state19.result);
        skipSeparationSpace(state19, true, -1);
        ch = state19.input.charCodeAt(state19.position);
        if ((state19.line === line || state19.lineIndent > nodeIndent) && ch !== 0) {
            return throwError(state19, "bad indentation of a sequence entry");
        } else if (state19.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state19.tag = tag;
        state19.anchor = anchor;
        state19.kind = "sequence";
        state19.result = result;
        return true;
    }
    return false;
}
function readBlockMapping(state20, nodeIndent, flowIndent) {
    const tag = state20.tag, anchor = state20.anchor, result = {}, overridableKeys = {};
    let following, allowCompact = false, line, pos, keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state20.anchor !== null && typeof state20.anchor !== "undefined" && typeof state20.anchorMap !== "undefined") {
        state20.anchorMap[state20.anchor] = result;
    }
    ch = state20.input.charCodeAt(state20.position);
    while(ch !== 0){
        following = state20.input.charCodeAt(state20.position + 1);
        line = state20.line;
        pos = state20.position;
        if ((ch === 63 || ch === 58) && isWsOrEol(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                return throwError(state20, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state20.position += 1;
            ch = following;
        } else if (composeNode(state20, flowIndent, 2, false, true)) {
            if (state20.line === line) {
                ch = state20.input.charCodeAt(state20.position);
                while(isWhiteSpace(ch)){
                    ch = state20.input.charCodeAt(++state20.position);
                }
                if (ch === 58) {
                    ch = state20.input.charCodeAt(++state20.position);
                    if (!isWsOrEol(ch)) {
                        return throwError(state20, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state20.tag;
                    keyNode = state20.result;
                } else if (detected) {
                    return throwError(state20, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state20.tag = tag;
                    state20.anchor = anchor;
                    return true;
                }
            } else if (detected) {
                return throwError(state20, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state20.tag = tag;
                state20.anchor = anchor;
                return true;
            }
        } else {
            break;
        }
        if (state20.line === line || state20.lineIndent > nodeIndent) {
            if (composeNode(state20, nodeIndent, 4, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state20.result;
                } else {
                    valueNode = state20.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, valueNode, line, pos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state20, true, -1);
            ch = state20.input.charCodeAt(state20.position);
        }
        if (state20.lineIndent > nodeIndent && ch !== 0) {
            return throwError(state20, "bad indentation of a mapping entry");
        } else if (state20.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
    }
    if (detected) {
        state20.tag = tag;
        state20.anchor = anchor;
        state20.kind = "mapping";
        state20.result = result;
    }
    return detected;
}
function readTagProperty(state21) {
    let position, isVerbatim = false, isNamed = false, tagHandle = "", tagName, ch;
    ch = state21.input.charCodeAt(state21.position);
    if (ch !== 33) return false;
    if (state21.tag !== null) {
        return throwError(state21, "duplication of a tag property");
    }
    ch = state21.input.charCodeAt(++state21.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state21.input.charCodeAt(++state21.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state21.input.charCodeAt(++state21.position);
    } else {
        tagHandle = "!";
    }
    position = state21.position;
    if (isVerbatim) {
        do {
            ch = state21.input.charCodeAt(++state21.position);
        }while (ch !== 0 && ch !== 62)
        if (state21.position < state21.length) {
            tagName = state21.input.slice(position, state21.position);
            ch = state21.input.charCodeAt(++state21.position);
        } else {
            return throwError(state21, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !isWsOrEol(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state21.input.slice(position - 1, state21.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        return throwError(state21, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    position = state21.position + 1;
                } else {
                    return throwError(state21, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state21.input.charCodeAt(++state21.position);
        }
        tagName = state21.input.slice(position, state21.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            return throwError(state21, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        return throwError(state21, `tag name cannot contain such characters: ${tagName}`);
    }
    if (isVerbatim) {
        state21.tag = tagName;
    } else if (typeof state21.tagMap !== "undefined" && hasOwn2(state21.tagMap, tagHandle)) {
        state21.tag = state21.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state21.tag = `!${tagName}`;
    } else if (tagHandle === "!!") {
        state21.tag = `tag:yaml.org,2002:${tagName}`;
    } else {
        return throwError(state21, `undeclared tag handle "${tagHandle}"`);
    }
    return true;
}
function readAnchorProperty(state22) {
    let ch = state22.input.charCodeAt(state22.position);
    if (ch !== 38) return false;
    if (state22.anchor !== null) {
        return throwError(state22, "duplication of an anchor property");
    }
    ch = state22.input.charCodeAt(++state22.position);
    const position = state22.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state22.input.charCodeAt(++state22.position);
    }
    if (state22.position === position) {
        return throwError(state22, "name of an anchor node must contain at least one character");
    }
    state22.anchor = state22.input.slice(position, state22.position);
    return true;
}
function readAlias(state23) {
    let ch = state23.input.charCodeAt(state23.position);
    if (ch !== 42) return false;
    ch = state23.input.charCodeAt(++state23.position);
    const _position = state23.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state23.input.charCodeAt(++state23.position);
    }
    if (state23.position === _position) {
        return throwError(state23, "name of an alias node must contain at least one character");
    }
    const alias = state23.input.slice(_position, state23.position);
    if (typeof state23.anchorMap !== "undefined" && !hasOwn2(state23.anchorMap, alias)) {
        return throwError(state23, `unidentified alias "${alias}"`);
    }
    if (typeof state23.anchorMap !== "undefined") {
        state23.result = state23.anchorMap[alias];
    }
    skipSeparationSpace(state23, true, -1);
    return true;
}
function composeNode(state24, parentIndent, nodeContext, allowToSeek, allowCompact) {
    let allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, type, flowIndent, blockIndent;
    if (state24.listener && state24.listener !== null) {
        state24.listener("open", state24);
    }
    state24.tag = null;
    state24.anchor = null;
    state24.kind = null;
    state24.result = null;
    const allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state24, true, -1)) {
            atNewLine = true;
            if (state24.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state24.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state24.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state24) || readAnchorProperty(state24)){
            if (skipSeparationSpace(state24, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state24.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state24.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state24.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || 4 === nodeContext) {
        const cond = 1 === nodeContext || 2 === nodeContext;
        flowIndent = cond ? parentIndent : parentIndent + 1;
        blockIndent = state24.position - state24.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state24, blockIndent) || readBlockMapping(state24, blockIndent, flowIndent)) || readFlowCollection(state24, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state24, flowIndent) || readSingleQuotedScalar(state24, flowIndent) || readDoubleQuotedScalar(state24, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state24)) {
                    hasContent = true;
                    if (state24.tag !== null || state24.anchor !== null) {
                        return throwError(state24, "alias node should not have Any properties");
                    }
                } else if (readPlainScalar(state24, flowIndent, 1 === nodeContext)) {
                    hasContent = true;
                    if (state24.tag === null) {
                        state24.tag = "?";
                    }
                }
                if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                    state24.anchorMap[state24.anchor] = state24.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state24, blockIndent);
        }
    }
    if (state24.tag !== null && state24.tag !== "!") {
        if (state24.tag === "?") {
            for(let typeIndex = 0, typeQuantity = state24.implicitTypes.length; typeIndex < typeQuantity; typeIndex++){
                type = state24.implicitTypes[typeIndex];
                if (type.resolve(state24.result)) {
                    state24.result = type.construct(state24.result);
                    state24.tag = type.tag;
                    if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                        state24.anchorMap[state24.anchor] = state24.result;
                    }
                    break;
                }
            }
        } else if (hasOwn2(state24.typeMap[state24.kind || "fallback"], state24.tag)) {
            type = state24.typeMap[state24.kind || "fallback"][state24.tag];
            if (state24.result !== null && type.kind !== state24.kind) {
                return throwError(state24, `unacceptable node kind for !<${state24.tag}> tag; it should be "${type.kind}", not "${state24.kind}"`);
            }
            if (!type.resolve(state24.result)) {
                return throwError(state24, `cannot resolve a node with !<${state24.tag}> explicit tag`);
            } else {
                state24.result = type.construct(state24.result);
                if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                    state24.anchorMap[state24.anchor] = state24.result;
                }
            }
        } else {
            return throwError(state24, `unknown tag !<${state24.tag}>`);
        }
    }
    if (state24.listener && state24.listener !== null) {
        state24.listener("close", state24);
    }
    return state24.tag !== null || state24.anchor !== null || hasContent;
}
function readDocument(state25) {
    const documentStart = state25.position;
    let position, directiveName, directiveArgs, hasDirectives = false, ch;
    state25.version = null;
    state25.checkLineBreaks = state25.legacy;
    state25.tagMap = {};
    state25.anchorMap = {};
    while((ch = state25.input.charCodeAt(state25.position)) !== 0){
        skipSeparationSpace(state25, true, -1);
        ch = state25.input.charCodeAt(state25.position);
        if (state25.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state25.input.charCodeAt(++state25.position);
        position = state25.position;
        while(ch !== 0 && !isWsOrEol(ch)){
            ch = state25.input.charCodeAt(++state25.position);
        }
        directiveName = state25.input.slice(position, state25.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            return throwError(state25, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(isWhiteSpace(ch)){
                ch = state25.input.charCodeAt(++state25.position);
            }
            if (ch === 35) {
                do {
                    ch = state25.input.charCodeAt(++state25.position);
                }while (ch !== 0 && !isEOL(ch))
                break;
            }
            if (isEOL(ch)) break;
            position = state25.position;
            while(ch !== 0 && !isWsOrEol(ch)){
                ch = state25.input.charCodeAt(++state25.position);
            }
            directiveArgs.push(state25.input.slice(position, state25.position));
        }
        if (ch !== 0) readLineBreak(state25);
        if (hasOwn2(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state25, directiveName, ...directiveArgs);
        } else {
            throwWarning(state25, `unknown document directive "${directiveName}"`);
        }
    }
    skipSeparationSpace(state25, true, -1);
    if (state25.lineIndent === 0 && state25.input.charCodeAt(state25.position) === 45 && state25.input.charCodeAt(state25.position + 1) === 45 && state25.input.charCodeAt(state25.position + 2) === 45) {
        state25.position += 3;
        skipSeparationSpace(state25, true, -1);
    } else if (hasDirectives) {
        return throwError(state25, "directives end mark is expected");
    }
    composeNode(state25, state25.lineIndent - 1, 4, false, true);
    skipSeparationSpace(state25, true, -1);
    if (state25.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state25.input.slice(documentStart, state25.position))) {
        throwWarning(state25, "non-ASCII line breaks are interpreted as content");
    }
    state25.documents.push(state25.result);
    if (state25.position === state25.lineStart && testDocumentSeparator(state25)) {
        if (state25.input.charCodeAt(state25.position) === 46) {
            state25.position += 3;
            skipSeparationSpace(state25, true, -1);
        }
        return;
    }
    if (state25.position < state25.length - 1) {
        return throwError(state25, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    const state26 = new LoaderState(input, options);
    state26.input += "\0";
    while(state26.input.charCodeAt(state26.position) === 32){
        state26.lineIndent += 1;
        state26.position += 1;
    }
    while(state26.position < state26.length - 1){
        readDocument(state26);
    }
    return state26.documents;
}
function load(input, options) {
    const documents = loadDocuments(input, options);
    if (documents.length === 0) {
        return;
    }
    if (documents.length === 1) {
        return documents[0];
    }
    throw new YAMLError("expected a single document in the stream, but found more");
}
function parse3(content, options) {
    return load(content, options);
}
const { hasOwn: hasOwn3  } = Object;
function compileStyleMap(schema, map7) {
    if (typeof map7 === "undefined" || map7 === null) return {};
    let type;
    const result = {};
    const keys5 = Object.keys(map7);
    let tag, style;
    for(let index = 0, length = keys5.length; index < length; index += 1){
        tag = keys5[index];
        style = String(map7[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = `tag:yaml.org,2002:${tag.slice(2)}`;
        }
        type = schema.compiledTypeMap.fallback[tag];
        if (type && typeof type.styleAliases !== "undefined" && hasOwn3(type.styleAliases, style)) {
            style = type.styleAliases[style];
        }
        result[tag] = style;
    }
    return result;
}
class DumperState extends State1 {
    indent;
    noArrayIndent;
    skipInvalid;
    flowLevel;
    sortKeys;
    lineWidth;
    noRefs;
    noCompatMode;
    condenseFlow;
    implicitTypes;
    explicitTypes;
    tag = null;
    result = "";
    duplicates = [];
    usedDuplicates = [];
    styleMap;
    dump;
    constructor({ schema , indent =2 , noArrayIndent =false , skipInvalid =false , flowLevel =-1 , styles =null , sortKeys =false , lineWidth =80 , noRefs =false , noCompatMode =false , condenseFlow =false  }){
        super(schema);
        this.indent = Math.max(1, indent);
        this.noArrayIndent = noArrayIndent;
        this.skipInvalid = skipInvalid;
        this.flowLevel = flowLevel;
        this.styleMap = compileStyleMap(this.schema, styles);
        this.sortKeys = sortKeys;
        this.lineWidth = lineWidth;
        this.noRefs = noRefs;
        this.noCompatMode = noCompatMode;
        this.condenseFlow = condenseFlow;
        this.implicitTypes = this.schema.compiledImplicit;
        this.explicitTypes = this.schema.compiledExplicit;
    }
}
const _toString2 = Object.prototype.toString;
const { hasOwn: hasOwn4  } = Object;
const ESCAPE_SEQUENCES = {};
ESCAPE_SEQUENCES[0] = "\\0";
ESCAPE_SEQUENCES[7] = "\\a";
ESCAPE_SEQUENCES[8] = "\\b";
ESCAPE_SEQUENCES[9] = "\\t";
ESCAPE_SEQUENCES[10] = "\\n";
ESCAPE_SEQUENCES[11] = "\\v";
ESCAPE_SEQUENCES[12] = "\\f";
ESCAPE_SEQUENCES[13] = "\\r";
ESCAPE_SEQUENCES[27] = "\\e";
ESCAPE_SEQUENCES[34] = '\\"';
ESCAPE_SEQUENCES[92] = "\\\\";
ESCAPE_SEQUENCES[133] = "\\N";
ESCAPE_SEQUENCES[160] = "\\_";
ESCAPE_SEQUENCES[8232] = "\\L";
ESCAPE_SEQUENCES[8233] = "\\P";
const DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF", 
];
function encodeHex(character) {
    const string10 = character.toString(16).toUpperCase();
    let handle;
    let length;
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new YAMLError("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return `\\${handle}${repeat("0", length - string10.length)}${string10}`;
}
function indentString(string11, spaces) {
    const ind = repeat(" ", spaces), length = string11.length;
    let position = 0, next = -1, result = "", line;
    while(position < length){
        next = string11.indexOf("\n", position);
        if (next === -1) {
            line = string11.slice(position);
            position = length;
        } else {
            line = string11.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result += ind;
        result += line;
    }
    return result;
}
function generateNextLine(state27, level) {
    return `\n${repeat(" ", state27.indent * level)}`;
}
function testImplicitResolving(state28, str62) {
    let type;
    for(let index = 0, length = state28.implicitTypes.length; index < length; index += 1){
        type = state28.implicitTypes[index];
        if (type.resolve(str62)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === 32 || c === 9;
}
function isPrintable(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== 65279 || 65536 <= c && c <= 1114111;
}
function isPlainSafe(c) {
    return isPrintable(c) && c !== 65279 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 58 && c !== 35;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== 65279 && !isWhitespace(c) && c !== 45 && c !== 63 && c !== 58 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 35 && c !== 38 && c !== 42 && c !== 33 && c !== 124 && c !== 62 && c !== 39 && c !== 34 && c !== 37 && c !== 64 && c !== 96;
}
function needIndentIndicator(string12) {
    const leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string12);
}
const STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string13, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType) {
    const shouldTrackWidth = lineWidth !== -1;
    let hasLineBreak = false, hasFoldableLine = false, previousLineBreak = -1, plain = isPlainSafeFirst(string13.charCodeAt(0)) && !isWhitespace(string13.charCodeAt(string13.length - 1));
    let __char, i42;
    if (singleLineOnly) {
        for(i42 = 0; i42 < string13.length; i42++){
            __char = string13.charCodeAt(i42);
            if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
    } else {
        for(i42 = 0; i42 < string13.length; i42++){
            __char = string13.charCodeAt(i42);
            if (__char === 10) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i42 - previousLineBreak - 1 > lineWidth && string13[previousLineBreak + 1] !== " ";
                    previousLineBreak = i42;
                }
            } else if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i42 - previousLineBreak - 1 > lineWidth && string13[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        return plain && !testAmbiguousType(string13) ? 1 : 2;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string13)) {
        return 5;
    }
    return hasFoldableLine ? 4 : 3;
}
function foldLine(line, width) {
    if (line === "" || line[0] === " ") return line;
    const breakRe = / [^ ]/g;
    let match;
    let start = 0, end, curr = 0, next = 0;
    let result = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result += `\n${line.slice(start, end)}`;
            start = end + 1;
        }
        curr = next;
    }
    result += "\n";
    if (line.length - start > width && curr > start) {
        result += `${line.slice(start, curr)}\n${line.slice(curr + 1)}`;
    } else {
        result += line.slice(start);
    }
    return result.slice(1);
}
function dropEndingNewline(string14) {
    return string14[string14.length - 1] === "\n" ? string14.slice(0, -1) : string14;
}
function foldString(string15, width) {
    const lineRe = /(\n+)([^\n]*)/g;
    let result = (()=>{
        let nextLF = string15.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string15.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string15.slice(0, nextLF), width);
    })();
    let prevMoreIndented = string15[0] === "\n" || string15[0] === " ";
    let moreIndented;
    let match;
    while(match = lineRe.exec(string15)){
        const prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine(line, width);
        prevMoreIndented = moreIndented;
    }
    return result;
}
function escapeString(string16) {
    let result = "";
    let __char, nextChar;
    let escapeSeq;
    for(let i43 = 0; i43 < string16.length; i43++){
        __char = string16.charCodeAt(i43);
        if (__char >= 55296 && __char <= 56319) {
            nextChar = string16.charCodeAt(i43 + 1);
            if (nextChar >= 56320 && nextChar <= 57343) {
                result += encodeHex((__char - 55296) * 1024 + nextChar - 56320 + 65536);
                i43++;
                continue;
            }
        }
        escapeSeq = ESCAPE_SEQUENCES[__char];
        result += !escapeSeq && isPrintable(__char) ? string16[i43] : escapeSeq || encodeHex(__char);
    }
    return result;
}
function blockHeader(string17, indentPerLevel) {
    const indentIndicator = needIndentIndicator(string17) ? String(indentPerLevel) : "";
    const clip = string17[string17.length - 1] === "\n";
    const keep = clip && (string17[string17.length - 2] === "\n" || string17 === "\n");
    const chomp = keep ? "+" : clip ? "" : "-";
    return `${indentIndicator}${chomp}\n`;
}
function writeScalar(state29, string18, level, iskey) {
    state29.dump = (()=>{
        if (string18.length === 0) {
            return "''";
        }
        if (!state29.noCompatMode && DEPRECATED_BOOLEANS_SYNTAX.indexOf(string18) !== -1) {
            return `'${string18}'`;
        }
        const indent = state29.indent * Math.max(1, level);
        const lineWidth = state29.lineWidth === -1 ? -1 : Math.max(Math.min(state29.lineWidth, 40), state29.lineWidth - indent);
        const singleLineOnly = iskey || state29.flowLevel > -1 && level >= state29.flowLevel;
        function testAmbiguity(str63) {
            return testImplicitResolving(state29, str63);
        }
        switch(chooseScalarStyle(string18, singleLineOnly, state29.indent, lineWidth, testAmbiguity)){
            case STYLE_PLAIN:
                return string18;
            case STYLE_SINGLE:
                return `'${string18.replace(/'/g, "''")}'`;
            case STYLE_LITERAL:
                return `|${blockHeader(string18, state29.indent)}${dropEndingNewline(indentString(string18, indent))}`;
            case STYLE_FOLDED:
                return `>${blockHeader(string18, state29.indent)}${dropEndingNewline(indentString(foldString(string18, lineWidth), indent))}`;
            case STYLE_DOUBLE:
                return `"${escapeString(string18)}"`;
            default:
                throw new YAMLError("impossible error: invalid scalar style");
        }
    })();
}
function writeFlowSequence(state30, level, object) {
    let _result = "";
    const _tag = state30.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state30, level, object[index], false, false)) {
            if (index !== 0) _result += `,${!state30.condenseFlow ? " " : ""}`;
            _result += state30.dump;
        }
    }
    state30.tag = _tag;
    state30.dump = `[${_result}]`;
}
function writeBlockSequence(state31, level, object, compact = false) {
    let _result = "";
    const _tag = state31.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state31, level + 1, object[index], true, true)) {
            if (!compact || index !== 0) {
                _result += generateNextLine(state31, level);
            }
            if (state31.dump && 10 === state31.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state31.dump;
        }
    }
    state31.tag = _tag;
    state31.dump = _result || "[]";
}
function writeFlowMapping(state32, level, object) {
    let _result = "";
    const _tag = state32.tag, objectKeyList = Object.keys(object);
    let pairBuffer, objectKey, objectValue;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = state32.condenseFlow ? '"' : "";
        if (index !== 0) pairBuffer += ", ";
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state32, level, objectKey, false, false)) {
            continue;
        }
        if (state32.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += `${state32.dump}${state32.condenseFlow ? '"' : ""}:${state32.condenseFlow ? "" : " "}`;
        if (!writeNode(state32, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state32.dump;
        _result += pairBuffer;
    }
    state32.tag = _tag;
    state32.dump = `{${_result}}`;
}
function writeBlockMapping(state33, level, object, compact = false) {
    const _tag = state33.tag, objectKeyList = Object.keys(object);
    let _result = "";
    if (state33.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state33.sortKeys === "function") {
        objectKeyList.sort(state33.sortKeys);
    } else if (state33.sortKeys) {
        throw new YAMLError("sortKeys must be a boolean or a function");
    }
    let pairBuffer = "", objectKey, objectValue, explicitPair;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || index !== 0) {
            pairBuffer += generateNextLine(state33, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state33, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state33.tag !== null && state33.tag !== "?" || state33.dump && state33.dump.length > 1024;
        if (explicitPair) {
            if (state33.dump && 10 === state33.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state33.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state33, level);
        }
        if (!writeNode(state33, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state33.dump && 10 === state33.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state33.dump;
        _result += pairBuffer;
    }
    state33.tag = _tag;
    state33.dump = _result || "{}";
}
function detectType(state34, object, explicit = false) {
    const typeList = explicit ? state34.explicitTypes : state34.implicitTypes;
    let type;
    let style;
    let _result;
    for(let index = 0, length = typeList.length; index < length; index += 1){
        type = typeList[index];
        if ((type.instanceOf || type.predicate) && (!type.instanceOf || typeof object === "object" && object instanceof type.instanceOf) && (!type.predicate || type.predicate(object))) {
            state34.tag = explicit ? type.tag : "?";
            if (type.represent) {
                style = state34.styleMap[type.tag] || type.defaultStyle;
                if (_toString2.call(type.represent) === "[object Function]") {
                    _result = type.represent(object, style);
                } else if (hasOwn4(type.represent, style)) {
                    _result = type.represent[style](object, style);
                } else {
                    throw new YAMLError(`!<${type.tag}> tag resolver accepts not "${style}" style`);
                }
                state34.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state35, level, object, block, compact, iskey = false) {
    state35.tag = null;
    state35.dump = object;
    if (!detectType(state35, object, false)) {
        detectType(state35, object, true);
    }
    const type = _toString2.call(state35.dump);
    if (block) {
        block = state35.flowLevel < 0 || state35.flowLevel > level;
    }
    const objectOrArray = type === "[object Object]" || type === "[object Array]";
    let duplicateIndex = -1;
    let duplicate = false;
    if (objectOrArray) {
        duplicateIndex = state35.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state35.tag !== null && state35.tag !== "?" || duplicate || state35.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state35.usedDuplicates[duplicateIndex]) {
        state35.dump = `*ref_${duplicateIndex}`;
    } else {
        if (objectOrArray && duplicate && !state35.usedDuplicates[duplicateIndex]) {
            state35.usedDuplicates[duplicateIndex] = true;
        }
        if (type === "[object Object]") {
            if (block && Object.keys(state35.dump).length !== 0) {
                writeBlockMapping(state35, level, state35.dump, compact);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex}${state35.dump}`;
                }
            } else {
                writeFlowMapping(state35, level, state35.dump);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex} ${state35.dump}`;
                }
            }
        } else if (type === "[object Array]") {
            const arrayLevel = state35.noArrayIndent && level > 0 ? level - 1 : level;
            if (block && state35.dump.length !== 0) {
                writeBlockSequence(state35, arrayLevel, state35.dump, compact);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex}${state35.dump}`;
                }
            } else {
                writeFlowSequence(state35, arrayLevel, state35.dump);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex} ${state35.dump}`;
                }
            }
        } else if (type === "[object String]") {
            if (state35.tag !== "?") {
                writeScalar(state35, state35.dump, level, iskey);
            }
        } else {
            if (state35.skipInvalid) return false;
            throw new YAMLError(`unacceptable kind of an object to dump ${type}`);
        }
        if (state35.tag !== null && state35.tag !== "?") {
            state35.dump = `!<${state35.tag}> ${state35.dump}`;
        }
    }
    return true;
}
function inspectNode(object, objects, duplicatesIndexes) {
    if (object !== null && typeof object === "object") {
        const index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(let idx = 0, length = object.length; idx < length; idx += 1){
                    inspectNode(object[idx], objects, duplicatesIndexes);
                }
            } else {
                const objectKeyList = Object.keys(object);
                for(let idx = 0, length = objectKeyList.length; idx < length; idx += 1){
                    inspectNode(object[objectKeyList[idx]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function getDuplicateReferences(object, state36) {
    const objects = [], duplicatesIndexes = [];
    inspectNode(object, objects, duplicatesIndexes);
    const length = duplicatesIndexes.length;
    for(let index = 0; index < length; index += 1){
        state36.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state36.usedDuplicates = Array.from({
        length
    });
}
function dump(input, options) {
    options = options || {};
    const state37 = new DumperState(options);
    if (!state37.noRefs) getDuplicateReferences(input, state37);
    if (writeNode(state37, 0, input, true, true)) return `${state37.dump}\n`;
    return "";
}
function stringify(obj, options) {
    return dump(obj, options);
}
function readAnnotatedYamlFromMappedString(yml) {
    return readAnnotatedYamlFromString(yml.value);
}
function readAnnotatedYamlFromString(yml) {
    const stack = [];
    const results = [];
    function listener(what, state38) {
        const { result , position , kind  } = state38;
        if (what === "close") {
            const { position: openPosition  } = stack.pop();
            if (results.length > 0) {
                const last = results[results.length - 1];
                if (last.start === openPosition && last.end === position) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last = results[results.length - 1];
                if (last.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position);
            const leftTrim = rawRange.length - rawRange.trimLeft().length;
            const rightTrim = rawRange.length - rawRange.trimRight().length;
            if (rawRange.trim().length === 0) {
                results.push({
                    start: position - rightTrim,
                    end: position - rightTrim,
                    result: result,
                    components,
                    kind
                });
            } else {
                results.push({
                    start: openPosition + leftTrim,
                    end: position - rightTrim,
                    result: result,
                    components,
                    kind
                });
            }
        } else {
            stack.push({
                position
            });
        }
    }
    parse3(yml, {
        listener,
        schema: json
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: []
        };
    }
    if (results.length !== 1) {
        throw new Error(`Internal Error - expected a single result, got ${results.length} instead`);
    }
    return results[0];
}
function memoize1(f, keyMemoizer) {
    const memo = {};
    const inner = (...args)=>{
        const key = keyMemoizer(...args);
        const v = memo[key];
        if (v !== undefined) {
            return v;
        }
        memo[key] = f(...args);
        return memo[key];
    };
    return inner;
}
async function pandocListFormats() {
    const result = await execProcess({
        cmd: [
            pandocBinaryPath(),
            "--list-output-formats"
        ],
        stdout: "piped"
    });
    if (result.success) {
        return lines(result.stdout).filter((line)=>line.length > 0
        );
    } else {
        return Promise.reject();
    }
}
function pandocFormatWith(format6, prepend, append) {
    const split1 = splitPandocFormatString(format6);
    return `${split1.format}${prepend}${split1.options}${append}`;
}
function splitPandocFormatString(format7) {
    let optionsPos = format7.indexOf("-");
    if (optionsPos === -1) {
        optionsPos = format7.indexOf("+");
    }
    const base4 = optionsPos === -1 ? format7 : format7.substr(0, optionsPos);
    const options = optionsPos === -1 ? "" : format7.substr(optionsPos);
    return {
        format: base4,
        options
    };
}
function Diff() {}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {};
        var callback = options.callback;
        if (typeof options === "function") {
            callback = options;
            options = {};
        }
        this.options = options;
        var self = this;
        function done(value) {
            if (callback) {
                setTimeout(function() {
                    callback(void 0, value);
                }, 0);
                return true;
            } else {
                return value;
            }
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath1 = bestPath[diagonalPath - 1], removePath1 = bestPath[diagonalPath + 1], _oldPos = (removePath1 ? removePath1.newPos : 0) - diagonalPath;
                if (addPath1) {
                    bestPath[diagonalPath - 1] = void 0;
                }
                var canAdd = addPath1 && addPath1.newPos + 1 < newLen, canRemove = removePath1 && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = void 0;
                    continue;
                }
                if (!canAdd || canRemove && addPath1.newPos < removePath1.newPos) {
                    basePath = clonePath(removePath1);
                    self.pushComponent(basePath.components, void 0, true);
                } else {
                    basePath = addPath1;
                    basePath.newPos++;
                    self.pushComponent(basePath.components, true, void 0);
                }
                _oldPos = self.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self, basePath.components, newString, oldString, self.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        if (callback) {
            (function exec1() {
                setTimeout(function() {
                    if (editLength > maxEditLength) {
                        return callback();
                    }
                    if (!execEditLength()) {
                        exec1();
                    }
                }, 0);
            })();
        } else {
            while(editLength <= maxEditLength){
                var ret = execEditLength();
                if (ret) {
                    return ret;
                }
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last = components[components.length - 1];
        if (last && last.added === added && last.removed === removed) {
            components[components.length - 1] = {
                count: last.count + 1,
                added,
                removed
            };
        } else {
            components.push({
                count: 1,
                added,
                removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array) {
        var ret = [];
        for(var i44 = 0; i44 < array.length; i44++){
            if (array[i44]) {
                ret.push(array[i44]);
            }
        }
        return ret;
    },
    castInput: function castInput(value) {
        return value;
    },
    tokenize: function tokenize(value) {
        return value.split("");
    },
    join: function join(chars) {
        return chars.join("");
    }
};
function buildValues(diff2, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value = newString.slice(newPos, newPos + component.count);
                value = value.map(function(value2, i45) {
                    var oldValue = oldString[oldPos + i45];
                    return oldValue.length > value2.length ? oldValue : value2;
                });
                component.value = diff2.join(value);
            } else {
                component.value = diff2.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff2.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === "string" && (lastComponent.added || lastComponent.removed) && diff2.equals("", lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path64) {
    return {
        newPos: path64.newPos,
        components: path64.components.slice(0)
    };
}
new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace1 = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace1.test(left) && !reWhitespace1.test(right);
};
wordDiff.tokenize = function(value) {
    var tokens = value.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i46 = 0; i46 < tokens.length - 1; i46++){
        if (!tokens[i46 + 1] && tokens[i46 + 2] && extendedWordChars.test(tokens[i46]) && extendedWordChars.test(tokens[i46 + 2])) {
            tokens[i46] += tokens[i46 + 2];
            tokens.splice(i46 + 1, 2);
            i46--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value) {
    var retLines = [], linesAndNewlines = value.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i47 = 0; i47 < linesAndNewlines.length; i47++){
        var line = linesAndNewlines[i47];
        if (i47 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line;
        } else {
            if (this.options.ignoreWhitespace) {
                line = line.trim();
            }
            retLines.push(line);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value) {
    return value.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value) {
    return value.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj2) {
            return typeof obj2;
        };
    } else {
        _typeof = function(obj2) {
            return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === "undefined" ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value === "string" ? value : JSON.stringify(canonicalize(value, null, null, stringifyReplacer), stringifyReplacer, "  ");
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, "$1"), right.replace(/,([\r\n])/g, "$1"));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i48;
    for(i48 = 0; i48 < stack.length; i48 += 1){
        if (stack[i48] === obj) {
            return replacementStack[i48];
        }
    }
    var canonicalizedObj;
    if (objectPrototypeToString.call(obj) === "[object Array]") {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i48 = 0; i48 < obj.length; i48 += 1){
            canonicalizedObj[i48] = canonicalize(obj[i48], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === "object" && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {};
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i48 = 0; i48 < sortedKeys.length; i48 += 1){
            _key = sortedKeys[i48];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value) {
    return value.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value) {
    return value;
};
function matchAll1(str64, regex) {
    let match;
    regex = new RegExp(regex);
    const result = [];
    while((match = regex.exec(str64)) != null){
        result.push(match);
    }
    return result;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result;
    } else {
        const matches = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r of matches){
            const stringEnd = r.index + 1;
            result.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result;
    }
}
function mappedDiff(source, target) {
    const sourceLineRanges = rangedLines(source.value).map((x)=>x.range
    );
    let sourceCursor = 0;
    const resultChunks = [];
    for (const action of diffLines(source.value, target)){
        if (action.removed) {
            sourceCursor += action.count;
        } else if (action.added) {
            resultChunks.push(action.value);
        } else {
            const start = sourceLineRanges[sourceCursor].start;
            const nextCursor = sourceCursor + action.count;
            const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
            sourceCursor = nextCursor;
            resultChunks.push({
                start,
                end
            });
        }
    }
    return mappedString(source, resultChunks, source.fileName);
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg, validationErrors){
        super(msg);
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
function readAndValidateYamlFromFile(file, schema, errorMessage) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    let shortFileName = file;
    if (shortFileName.startsWith("/")) {
        shortFileName = relative2(Deno.cwd(), shortFileName);
    }
    const contents = asMappedString(Deno.readTextFileSync(file), shortFileName);
    return readAndValidateYamlFromMappedString(contents, schema, errorMessage);
}
async function readAndValidateYamlFromMappedString(mappedYaml, schema, errorMessage) {
    const result = await withValidator(schema, async (validator)=>{
        const annotation = readAnnotatedYamlFromMappedString(mappedYaml);
        const validateYaml = !(annotation.result?.["validate-yaml"] === false);
        const yaml = annotation.result;
        if (validateYaml) {
            const valResult = await validator.validateParse(mappedYaml, annotation);
            if (valResult.errors.length) {
                validator.reportErrorsInSource({
                    result: yaml,
                    errors: valResult.errors
                }, mappedYaml, errorMessage, (msg)=>{
                    if (!errorOnce(msg)) {
                        info("");
                    }
                }, info);
            }
            return {
                yaml: yaml,
                yamlValidationErrors: valResult.errors
            };
        } else {
            return {
                yaml: yaml,
                yamlValidationErrors: []
            };
        }
    });
    if (result.yamlValidationErrors.length > 0) {
        throw new ValidationError2(errorMessage, result.yamlValidationErrors);
    }
    return result.yaml;
}
const kRegExBeginYAML = /^---[ \t]*$/;
function setBaseSchemaProperties(yaml, schema) {
    if (yaml.additionalCompletions) {
        schema = completeSchema(schema, ...yaml.additionalCompletions);
    }
    if (yaml.completions) {
        schema = completeSchemaOverwrite(schema, ...yaml.completions);
    }
    if (yaml.id) {
        schema = idSchema(schema, yaml.id);
    }
    if (yaml.hidden === true) {
        schema = completeSchemaOverwrite(schema);
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    if (yaml.tags) {
        schema = tagSchema(schema, yaml.tags);
    }
    if (yaml.description) {
        if (typeof yaml.description === "string") {
            schema = documentSchema(schema, yaml.description);
        } else if (typeof yaml.description === "object") {
            schema = documentSchema(schema, yaml.description.short);
        }
    }
    const result = Object.assign({}, schema);
    if (yaml.errorDescription) {
        result.description = yaml.errorDescription;
    }
    return result;
}
function convertFromNull(yaml) {
    return setBaseSchemaProperties(yaml["null"], NullSchema);
}
function convertFromSchema(yaml) {
    const schema = convertFromYaml(yaml.schema);
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromString(yaml) {
    if (yaml["string"].pattern) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], regexSchema(yaml["string"].pattern)));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], StringSchema));
    }
}
function convertFromPattern(yaml) {
    if (typeof yaml.pattern === "string") {
        return setBaseSchemaProperties(yaml, regexSchema(yaml.pattern));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.pattern, regexSchema(yaml.pattern.regex)));
    }
}
function convertFromPath(yaml) {
    return setBaseSchemaProperties(yaml["path"], StringSchema);
}
function convertFromNumber(yaml) {
    return setBaseSchemaProperties(yaml["number"], NumberSchema);
}
function convertFromBoolean(yaml) {
    return setBaseSchemaProperties(yaml["boolean"], BooleanSchema);
}
function convertFromRef(yaml) {
    return setBaseSchemaProperties(yaml, refSchema(yaml.ref, `be ${yaml.ref}`));
}
function convertFromMaybeArrayOf(yaml) {
    const inner = convertFromYaml(yaml.maybeArrayOf);
    const schema = tagSchema(oneOfSchema(inner, arraySchema(inner)), {
        "complete-from": [
            "oneOf",
            0
        ]
    });
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromArrayOf(yaml) {
    if (yaml.arrayOf.schema) {
        const result = arraySchema(convertFromYaml(yaml.arrayOf.schema));
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.arrayOf, result));
    } else {
        return setBaseSchemaProperties(yaml, arraySchema(convertFromYaml(yaml.arrayOf)));
    }
}
function convertFromOneOf(yaml) {
    if (yaml.oneOf.schemas) {
        const inner = yaml.oneOf.schemas.map((x)=>convertFromYaml(x)
        );
        const schema = oneOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.oneOf, schema));
    } else {
        const inner = yaml.oneOf.map((x)=>convertFromYaml(x)
        );
        const schema = oneOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromAllOf(yaml) {
    if (yaml.allOf.schemas) {
        const inner = yaml.allOf.schemas.map((x)=>convertFromYaml(x)
        );
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.allOf, schema));
    } else {
        const inner = yaml.allOf.map((x)=>convertFromYaml(x)
        );
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromAnyOf(yaml) {
    if (yaml.anyOf.schemas) {
        const inner = yaml.anyOf.schemas.map((x)=>convertFromYaml(x)
        );
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.anyOf, schema));
    } else {
        const inner = yaml.anyOf.map((x)=>convertFromYaml(x)
        );
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromEnum(yaml) {
    const schema = yaml["enum"];
    if (schema.hasOwnProperty("values")) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["enum"], enumSchema(...schema.values)));
    } else {
        return setBaseSchemaProperties(yaml, enumSchema(...schema));
    }
}
function convertFromRecord(yaml) {
    if (yaml.record.properties) {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record.properties,
                "additionalProperties": false,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.record, schema));
    } else {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record,
                "additionalProperties": false,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromObject(yaml) {
    const schema = yaml["object"];
    const params = {};
    if (schema.properties) {
        params.properties = Object.fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]
        ));
    }
    if (schema.patternProperties) {
        params.patternProperties = Object.fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]
        ));
    }
    if (schema.additionalProperties !== undefined) {
        if (schema.additionalProperties === false) {
            params.additionalProperties = false;
        } else {
            params.additionalProperties = convertFromYaml(schema.additionalProperties);
        }
    }
    if (schema["super"]) {
        params.baseSchema = convertFromYaml(schema["super"]);
    }
    if (schema["required"] === "all") {
        params.required = Object.keys(schema.properties || {});
    } else if (schema["required"]) {
        params.required = schema["required"];
    }
    if (schema["completions"]) {
        params.completions = schema["completions"];
    }
    return setBaseSchemaProperties(yaml, setBaseSchemaProperties(schema, objectSchema(params)));
}
function lookup(yaml) {
    if (!hasSchemaDefinition(yaml.resolveRef)) {
        throw new Error(`lookup of key ${yaml.resolveRef} in definitions failed`);
    }
    return getSchemaDefinition(yaml.resolveRef);
}
function convertFromYaml(yaml) {
    const literalValues = [
        [
            "object",
            objectSchema()
        ],
        [
            "path",
            StringSchema
        ],
        [
            "string",
            StringSchema
        ],
        [
            "number",
            NumberSchema
        ],
        [
            "boolean",
            BooleanSchema
        ],
        [
            null,
            NullSchema
        ], 
    ];
    for (const [testVal, result] of literalValues){
        if (yaml === testVal) {
            return result;
        }
    }
    if (typeof yaml !== "object") {
        return valueSchema(yaml);
    }
    const schemaObjectKeyFunctions = [
        {
            key: "anyOf",
            value: convertFromAnyOf
        },
        {
            key: "allOf",
            value: convertFromAllOf
        },
        {
            key: "boolean",
            value: convertFromBoolean
        },
        {
            key: "arrayOf",
            value: convertFromArrayOf
        },
        {
            key: "enum",
            value: convertFromEnum
        },
        {
            key: "maybeArrayOf",
            value: convertFromMaybeArrayOf
        },
        {
            key: "null",
            value: convertFromNull
        },
        {
            key: "number",
            value: convertFromNumber
        },
        {
            key: "object",
            value: convertFromObject
        },
        {
            key: "oneOf",
            value: convertFromOneOf
        },
        {
            key: "path",
            value: convertFromPath
        },
        {
            key: "record",
            value: convertFromRecord
        },
        {
            key: "ref",
            value: convertFromRef
        },
        {
            key: "resolveRef",
            value: lookup
        },
        {
            key: "string",
            value: convertFromString
        },
        {
            key: "pattern",
            value: convertFromPattern
        },
        {
            key: "schema",
            value: convertFromSchema
        }, 
    ];
    for (const { key: objectKey , value: fun  } of schemaObjectKeyFunctions){
        try {
            if (yaml[objectKey] !== undefined) {
                return fun(yaml);
            }
        } catch (e) {
            error({
                yaml
            });
            throw e;
        }
    }
    error(JSON.stringify(yaml, null, 2));
    throw new Error("Internal Error: Cannot convert object; this should have failed validation.");
}
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function isYamlPath(file) {
    return [
        ".yaml",
        ".yml"
    ].includes(extname2(file));
}
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        return parse3(decoder.decode(yml), {
            schema: json
        });
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function objectSchemaFromFieldsFile(file, exclude) {
    exclude = exclude ?? ((_key)=>false
    );
    const properties = {};
    const global = readYaml(file);
    convertFromFieldsObject(global, properties);
    for (const key of Object.keys(properties)){
        if (exclude(key)) {
            delete properties[key];
        }
    }
    return objectSchema({
        properties
    });
}
function annotateSchemaFromField(field, schema) {
    if (field.enabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.enabled
        });
    }
    if (field.disabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.disabled.map((x)=>`!${x}`
            )
        });
    }
    if (field.tags) {
        schema = tagSchema(schema, field.tags);
    }
    if (field.description) {
        if (typeof field.description === "string") {
            schema = documentSchema(schema, field.description);
        } else if (typeof field.description === "object") {
            schema = documentSchema(schema, field.description.short);
        }
    }
    if (field.hidden) {
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    return schema;
}
function schemaFromField(entry) {
    const schema = convertFromYaml(entry.schema);
    return annotateSchemaFromField(entry, schema);
}
function convertFromFieldsObject(yaml, obj) {
    const result = obj ?? {};
    for (const field of yaml){
        let schema = convertFromYaml(field.schema);
        schema = annotateSchemaFromField(field, schema);
        result[field.name] = schema;
        if (field.alias) {
            result[field.alias] = schema;
        }
    }
    return result;
}
function schemaFieldsFromGlob(globPath, testFun) {
    const result = [];
    testFun = testFun ?? ((_e, _p)=>true
    );
    for (const file of expandGlobSync(globPath)){
        for (const field of readYaml(file.path)){
            const fieldName = field.name;
            const schemaId = `quarto-resource-${file.name.slice(0, -4)}-${fieldName}`;
            if (testFun(field, file.path)) {
                result.push({
                    schemaId,
                    field
                });
            }
        }
    }
    return result;
}
memoize1(()=>{
    const groups = readYaml(schemaPath("groups.yml"));
    const result = [];
    for (const [topLevel, sub] of Object.entries(groups)){
        for (const key of Object.keys(sub)){
            result.push(`${topLevel}-${key}`);
        }
    }
    return result;
}, ()=>"const"
);
function objectRefSchemaFromContextGlob(contextGlob, testFun) {
    const regexp5 = globToRegExp(contextGlob);
    return objectRefSchemaFromGlob(schemaPath("{document,cell}-*.yml"), (field, path65)=>{
        if (testFun !== undefined && !testFun(field, path65)) {
            return false;
        }
        const pathContext = basename2(path65, ".yml");
        const schemaContexts = field?.tags?.contexts || [];
        if (pathContext.match(regexp5)) {
            return true;
        }
        return schemaContexts.some((c)=>c.match(regexp5)
        );
    });
}
let formatAliases = undefined;
function getFormatAliases() {
    if (formatAliases !== undefined) {
        return formatAliases;
    }
    formatAliases = readYaml(join3(resourcePath(), "schema/format-aliases.yml")).aliases;
    return formatAliases;
}
function expandFormatAliases(lst) {
    return expandAliasesFrom(lst, getFormatAliases());
}
function useSchema(schema, format8) {
    const formats1 = schema?.tags?.formats;
    if (formats1 === undefined) {
        return true;
    }
    const disabled = formats1.filter((f)=>f.startsWith("!")
    ).map((f)=>f.slice(1)
    );
    const enabled2 = formats1.filter((f)=>!f.startsWith("!")
    );
    if (disabled.length > 0 && expandFormatAliases(disabled).indexOf(format8) !== -1) {
        return false;
    }
    if (enabled2.length > 0 && expandFormatAliases(enabled2).indexOf(format8) === -1) {
        return false;
    }
    return true;
}
function getFormatSchema(format9) {
    const schema1 = objectRefSchemaFromContextGlob("document-*", (field)=>{
        const schema = schemaFromField(field);
        return useSchema(schema, format9);
    });
    return oneOfSchema(schema1, enumSchema("default"));
}
async function makeFrontMatterFormatSchema(nonStrict = false) {
    const hideFormat = (format10)=>{
        const hideList = [
            "html",
            "epub",
            "docbook"
        ];
        const hidden26 = hideList.some((h)=>format10.startsWith(h) && format10.length > h.length
        );
        return {
            name: format10,
            hidden: hidden26
        };
    };
    const formatSchemaDescriptorList = (await pandocListFormats()).map((format11)=>{
        const { name: name85 , hidden: hidden27  } = hideFormat(format11);
        return {
            regex: `^${name85}(\\+.+)?$`,
            schema: getFormatSchema(name85),
            name: name85,
            hidden: hidden27
        };
    });
    const formatSchemas = formatSchemaDescriptorList.map(({ regex , schema  })=>[
            regex,
            schema
        ]
    );
    const plusFormatStringSchemas = formatSchemaDescriptorList.map(({ regex , name: name86 , hidden: hidden28  })=>{
        const schema = regexSchema(regex, `be '${name86}'`);
        if (hidden28) {
            return schema;
        }
        return completeSchema(schema, {
            type: "value",
            display: "",
            suggest_on_accept: true,
            value: name86,
            description: ""
        });
    });
    const completionsObject = Object.fromEntries(formatSchemaDescriptorList.filter(({ hidden: hidden29  })=>!hidden29
    ).map(({ name: name87  })=>[
            name87,
            ""
        ]
    ));
    return oneOfSchema(describeSchema(oneOfSchema(...plusFormatStringSchemas), "the name of a pandoc-supported output format"), regexSchema("^hugo(\\+.+)?$", "be 'hugo'"), allOfSchema(objectSchema({
        patternProperties: Object.fromEntries(formatSchemas),
        completions: completionsObject,
        additionalProperties: nonStrict
    })));
}
function defineCached(thunk, schemaId) {
    let schema;
    return async ()=>{
        if (hasSchemaDefinition(schemaId)) {
            schema = getSchemaDefinition(schemaId);
            return refSchema(schema.$id, schema.description || `be a {schema['$id']}`);
        }
        const result = await thunk();
        const { errorHandlers  } = result;
        schema = result.schema;
        if (schemaId !== schema.$id) {
            schema = idSchema(schema, schemaId);
        }
        await define(schema);
        for (const fun of errorHandlers){
            addValidatorErrorHandler(schema, fun);
        }
        return refSchema(schema.$id, schema.description || `be a {schema['$id']}`);
    };
}
async function define(schema) {
    if (!hasSchemaDefinition(schema.$id)) {
        setSchemaDefinition(schema);
        await withValidator(normalizeSchema(schema), async (_validator)=>{});
    }
}
async function getFormatExecuteOptionsSchema() {
    const schema = idSchema(objectRefSchemaFromContextGlob("document-execute"), "front-matter-execute");
    await define(schema);
    return refSchema("front-matter-execute", "be a front-matter-execute object");
}
function readYamlFromString(yml) {
    return parse3(yml, {
        schema: json
    });
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            const yamlBlock = removeYamlDelimiters(match[2]);
            if (!yamlBlock.startsWith("\n\n") && !yamlBlock.startsWith("\n---") && yamlBlock.trim().length > 0) {
                parse3(yamlBlock, {
                    json: true,
                    schema: json
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata = parse3(yaml, {
            json: true,
            schema: json
        });
        return metadata || {};
    } else {
        return {};
    }
}
const getFrontMatterFormatSchema = defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(),
        errorHandlers: []
    };
}, "front-matter-format");
defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(true),
        errorHandlers: []
    };
}, "front-matter-format-nonstrict");
const getFrontMatterSchema = defineCached(async ()=>{
    const executeObjSchema = await getFormatExecuteOptionsSchema();
    return {
        schema: oneOfSchema(NullSchema, allOfSchema(objectSchema({
            properties: {
                execute: executeObjSchema,
                format: await getFrontMatterFormatSchema()
            },
            description: "be a Quarto YAML front matter object"
        }), objectRefSchemaFromContextGlob("document-*", (field)=>field.name !== "format"
        ), executeObjSchema)),
        errorHandlers: []
    };
}, "front-matter");
async function readAndValidateYamlFromMarkdown(eitherMarkdown) {
    let markdown = asMappedString(eitherMarkdown);
    if (!markdown.value) {
        return {};
    }
    markdown = mappedNormalizeNewlines(markdown);
    markdown = skipRegexpAll(markdown, kRegxHTMLComment);
    markdown = skipRegexpAll(markdown, kRegexFencedCode);
    const yaml = [];
    kRegExYAML.lastIndex = 0;
    for (const match of matchAll(markdown.value, kRegExYAML)){
        const yamlBlock = removeYamlDelimitersMapped(match[2]);
        const yamlBlockValue = yamlBlock.value;
        if (!yamlBlockValue.startsWith("\n\n") && !yamlBlockValue.startsWith("\n---") && yamlBlockValue.trim().length > 0) {
            parse3(yamlBlockValue, {
                json: true,
                schema: json
            });
            yaml.push(yamlBlock);
        }
    }
    kRegExYAML.lastIndex = 0;
    if (yaml.length === 0) {
        return {};
    }
    const mappedYaml = mappedConcat(yaml);
    const metadata = parse3(mappedYaml.value, {
        json: true,
        schema: json
    });
    if (metadata?.["validate-yaml"] === false) {
        return readAndValidateYamlFromMappedString(mappedYaml, await getFrontMatterSchema(), "YAML front matter validation failed");
    }
    return metadata;
}
async function readYamlFromMarkdownFile(file) {
    const markdown = Deno.readTextFileSync(file);
    const result = await readAndValidateYamlFromMarkdown(markdown);
    return result;
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line, index)=>index > 0 && line.match(kRegExEndYAML)
        );
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
function removeYamlDelimitersMapped(eitherYaml) {
    let yaml = asMappedString(eitherYaml);
    yaml = skipRegexp(yaml, /^---/);
    yaml = skipRegexp(yaml, /---\s*$/);
    return yaml;
}
function objectRefSchemaFromGlob(glob, testFun) {
    const properties = {};
    for (const { schemaId , field  } of schemaFieldsFromGlob(glob, testFun)){
        const schema = refSchema(schemaId, schemaId);
        properties[field.name] = schema;
        if (field.alias) {
            properties[field.alias] = schema;
        }
    }
    return objectSchema({
        properties
    });
}
async function buildSchemaResources() {
    const path66 = schemaPath("{cell-*,document-*,project}.yml");
    for (const file of expandGlobSync(path66)){
        const yaml = readYaml(file.path);
        const entries = Object.entries(convertFromFieldsObject(yaml));
        for (const [fieldName, fieldSchema] of entries){
            const schemaId = `quarto-resource-${file.name.slice(0, -4)}-${fieldName}`;
            const schema = idSchema(fieldSchema, schemaId);
            setSchemaDefinition(schema);
            await withValidator(schema, async (_validator)=>{});
        }
    }
}
async function loadDefaultSchemaDefinitions() {
    await loadSchemaDefinitions(schemaPath("definitions.yml"));
    await buildSchemaResources();
}
async function loadSchemaDefinitions(file) {
    const yaml = readYaml(file);
    await Promise.all(yaml.map(async (yamlSchema)=>{
        const schema = convertFromYaml(yamlSchema);
        if (schema.$id === undefined) {
            console.log(JSON.stringify(yamlSchema, null, 2));
            error(JSON.stringify(schema, null, 2));
            throw new Error(`Internal error: unnamed schema in definitions`);
        }
        setSchemaDefinition(schema);
        await withValidator(normalizeSchema(schema), async (_validator)=>{});
    }));
}
const kMetadataFormat = "format";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig-width";
const kFigHeight = "fig-height";
const kFigFormat = "fig-format";
const kFigDpi = "fig-dpi";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kKeepHidden = "keep-hidden";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepYaml = "keep-yaml";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kFigEnv = "fig-env";
const kFigPos = "fig-pos";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kTblColwidths = "tbl-colwidths";
const kMergeIncludes = "merge-includes";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug, 
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepYaml,
    kKeepSource,
    kKeepHidden,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kFigPos,
    kFigEnv,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kTblColwidths,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter, 
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kCalloutTipCaption = "callout-tip-caption";
const kCalloutNoteCaption = "callout-note-caption";
const kCalloutWarningCaption = "callout-warning-caption";
const kCalloutImportantCaption = "callout-important-caption";
const kCalloutDangerCaption = "callout-danger-caption";
const kSectionTitleAbstract = "section-title-abstract";
const kSectionTitleFootnotes = "section-title-footnotes";
const kSectionTitleAppendices = "section-title-appendices";
const kCodeSummary = "code-summary";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kCopyButtonTooltip = "copy-button-tooltip";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kListingPageOrderBy = "listing-page-order-by";
const kListingPageOrderByDefault = "listing-page-order-by-default";
const kListingPageOrderByDateAsc = "listing-page-order-by-date-asc";
const kListingPageOrderByDateDesc = "listing-page-order-by-date-desc";
const kListingPageOrderByNumberAsc = "listing-page-order-by-number-asc";
const kListingPageOrderByNumberDesc = "listing-page-order-by-number-desc";
const kListingPageFieldDate = "listing-page-field-date";
const kListingPageFieldTitle = "listing-page-field-title";
const kListingPageFieldDescription = "listing-page-field-description";
const kListingPageFieldAuthor = "listing-page-field-author";
const kListingPageFieldFileName = "listing-page-field-filename";
const kListingPageFieldFileModified = "listing-page-field-filemodified";
const kListingPageFieldSubtitle = "listing-page-field-subtitle";
const kListingPageFieldReadingTime = "listing-page-field-readingtime";
const kListingPageFieldCategories = "listing-page-field-categories";
const kListingPageMinutesCompact = "listing-page-minutes-compact";
const kListingPageCategoryAll = "listing-page-category-all";
const kListingPageNoMatches = "listing-page-no-matches";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutDangerCaption,
    kSectionTitleAbstract,
    kSectionTitleFootnotes,
    kSectionTitleAppendices,
    kCodeSummary,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kSearchNoResultsText,
    kCopyButtonTooltip,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle,
    kListingPageOrderBy,
    kListingPageOrderByDefault,
    kListingPageOrderByDateAsc,
    kListingPageOrderByDateDesc,
    kListingPageOrderByNumberAsc,
    kListingPageOrderByNumberDesc,
    kListingPageFieldDate,
    kListingPageFieldTitle,
    kListingPageFieldDescription,
    kListingPageFieldAuthor,
    kListingPageFieldFileName,
    kListingPageFieldFileModified,
    kListingPageFieldSubtitle,
    kListingPageFieldReadingTime,
    kListingPageFieldCategories,
    kListingPageMinutesCompact,
    kListingPageCategoryAll,
    kListingPageNoMatches, 
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kAtxHeaders = "atx-headers";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kStandalone = "standalone";
const kWrap = "wrap";
const kSelfContained = "self-contained";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kDefaultImageExtension = "default-image-extension";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDoi = "doi";
const kAbstract = "abstract";
const kAbstractTitle = "abstract-title";
const kDescription = "description";
const kTocTitle = "toc-title";
const kTocLocation = "toc-location";
const kLang = "lang";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kCss = "css";
const kBibliography = "bibliography";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kFigResponsive = "fig-responsive";
const kCapLoc = "cap-location";
const kFigCapLoc = "fig-cap-location";
const kTblCapLoc = "tbl-cap-location";
const kCapTop = "top";
const kCapBottom = "bottom";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    "resource-path",
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    kWrap,
    "columns",
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    "toc-depth",
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "strip-empty-paragraphs",
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    kDefaultImageExtension,
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kAtxHeaders,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace", 
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
async function includedMetadata(dir, baseMetadata, schema) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join3(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join3(dir, file))
        );
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (await exists(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeFormatMetadata({}, ...filesMetadata),
        files: yamlFiles
    };
}
function formatFromMetadata(baseFormat, to, debug1) {
    const typedFormat = {
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    let format12 = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format12 = metadataAsFormat({});
        } else if (configFormat instanceof Object) {
            format12 = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeFormatMetadata(baseFormat, format12);
    if (debug1) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
    }
    return mergedFormat;
}
function formatKeys(metadata) {
    if (typeof metadata[kMetadataFormat] === "string") {
        return [
            metadata[kMetadataFormat]
        ];
    } else if (metadata[kMetadataFormat] instanceof Object) {
        return Object.keys(metadata[kMetadataFormat]).filter((key)=>{
            const format13 = metadata[kMetadataFormat][key];
            return format13 !== null && format13 !== false;
        });
    } else {
        return [];
    }
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key);
}
function isIncludeMetadata(key) {
    return [
        kHeaderIncludes,
        kIncludeBefore,
        kIncludeAfter
    ].includes(key);
}
function metadataAsFormat(metadata) {
    const typedFormat = {
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    const format14 = typedFormat;
    Object.keys(metadata).forEach((key)=>{
        if ([
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata, 
        ].includes(key)) {
            if (typeof metadata[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format14[key] = format14[key] || {};
                    format14[kExecuteDefaults][kExecuteEnabled] = metadata[key];
                }
            } else {
                format14[key] = {
                    ...format14[key],
                    ...metadata[key]
                };
            }
        } else {
            if (kRenderDefaultsKeys.includes(key)) {
                format14.render[key] = metadata[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format14.execute[key] = metadata[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format14.pandoc[key] = metadata[key];
            } else if (kLanguageDefaultsKeys.includes(key)) {
                format14.language[key] = metadata[key];
            } else {
                format14.metadata[key] = metadata[key];
            }
        }
    });
    return typedFormat;
}
function metadataGetDeep(metadata, property1) {
    let values11 = [];
    __VIRTUAL_FILE145(metadata, (value, key)=>{
        if (key === property1) {
            values11.push(value);
        } else if (__VIRTUAL_FILE18(value)) {
            values11 = values11.concat(metadataGetDeep(value, property1));
        }
    });
    return values11;
}
const kUnmergeableKeys = [
    kTblColwidths
];
function mergeFormatMetadata(config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, (objValue, srcValue, key)=>{
        if (kUnmergeableKeys.includes(key)) {
            return srcValue;
        } else {
            return mergeArrayCustomizer(objValue, srcValue);
        }
    });
}
const kStdOut = "-";
function parseRenderFlags(args) {
    const flags = {};
    const argsStack = [
        ...args
    ];
    let arg = argsStack.shift();
    while(arg !== undefined){
        switch(arg){
            case "-t":
            case "--to":
                arg = argsStack.shift();
                if (arg && !arg.startsWith("-")) {
                    flags.to = arg;
                }
                break;
            case "-o":
            case "--output":
                arg = argsStack.shift();
                if (!arg || arg.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    flags.output = arg;
                }
                break;
            case "--output-dir":
                arg = argsStack.shift();
                flags.outputDir = arg;
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg = argsStack.shift();
                break;
            case "--pdf-engine":
                arg = argsStack.shift();
                flags.pdfEngine = arg;
                break;
            case "--pdf-engine-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg);
                }
                break;
            case "--latex-makeindex-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg);
                }
                break;
            case "--latex-tlmgr-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg);
                }
                break;
            case "--natbib":
                arg = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg);
                break;
            case "--top-level-division":
                arg = argsStack.shift();
                flags[kTopLevelDivision] = arg;
                break;
            case "--shift-heading-level-by":
                arg = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include1 = arg.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include1] = includeFlags[include1] || [];
                    arg = argsStack.shift();
                    includeFlags[include1].push(arg);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg = argsStack.shift();
                break;
            case "--execute-params":
                arg = argsStack.shift();
                flags.paramsFile = arg;
                break;
            case "--execute-dir":
                arg = argsStack.shift();
                flags.executeDir = arg;
                break;
            case "--execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = parseInt(arg, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--cache":
                arg = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--clean":
                arg = argsStack.shift();
                flags.clean = true;
                break;
            case "--no-clean":
                arg = argsStack.shift();
                flags.clean = false;
                break;
            case "--debug":
                flags.debug = true;
                arg = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg = argsStack.shift();
                if (arg) {
                    const param = parseMetadataFlagValue(arg);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {};
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg = argsStack.shift();
                if (arg) {
                    const metadata = parseMetadataFlagValue(arg);
                    if (metadata) {
                        if (metadata.value !== undefined) {
                            if (isQuartoMetadata(metadata.name)) {
                                flags.metadata = flags.metadata || {};
                                flags.metadata[metadata.name] = metadata.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {};
                                flags.pandocMetadata[metadata.name] = metadata.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg = argsStack.shift();
                if (arg) {
                    if (existsSync(arg)) {
                        const metadata = readYamlFromMarkdownFile(arg);
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata
                        };
                    }
                }
                break;
            case "--reference-location":
                arg = argsStack.shift();
                if (arg) {
                    flags[kReferenceLocation] = arg;
                }
                break;
            default:
                arg = argsStack.shift();
                break;
        }
    }
    if (flags.clean === undefined) {
        flags.clean = true;
    }
    return flags;
}
function havePandocArg(pandocArgs, arg) {
    return pandocArgs.indexOf(arg) !== -1;
}
function replacePandocArg(pandocArgs, arg, value) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value;
    } else {
        newArgs.push(arg);
        newArgs.push(value);
    }
    return newArgs;
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args, arg, index)=>{
        args.push(arg);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg === "-o" || arg === "--output")) {
            args.push(kStdOut);
        }
        return args;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--clean", false);
    removeArgs.set("--no-clean", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    let removeNext = false;
    return pandocArgs.reduce((args, arg)=>{
        if (!removeArgs.has(arg)) {
            if (!removeNext) {
                args.push(arg);
            }
            removeNext = false;
        } else {
            removeNext = removeArgs.get(arg);
        }
        return args;
    }, new Array());
}
function removePandocToArg(args) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args, removeArgs);
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args = [];
    for(let i49 = 0; i49 < pandocArgs.length; i49++){
        const arg = pandocArgs[i49];
        if (arg === "--metadata" || arg === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i49 + 1] || "");
            if (flagValue !== undefined && isQuartoMetadata(flagValue.name)) {
                i49++;
            } else {
                args.push(arg);
            }
        } else {
            args.push(arg);
        }
    }
    return args;
}
function parseMetadataFlagValue(arg) {
    const match = arg.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {};
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value) {
    if (value) {
        const numbers = value.split(/,/).map((number2)=>parseInt(number2.trim(), 10)
        ).filter((number3)=>!isNaN(number3)
        );
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
function isPdfOutput(format15) {
    if (typeof format15 !== "string") {
        format15 = format15?.to || "html";
    }
    return format15 === "pdf" || format15 === "beamer";
}
function isLatexOutput(format16) {
    return [
        "pdf",
        "latex",
        "beamer"
    ].includes(format16.to || "");
}
function isEpubOutput(format17) {
    if (typeof format17 !== "string") {
        format17 = format17?.to || "html";
    }
    return [
        "epub",
        "epub2",
        "epub3"
    ].includes(format17 || "");
}
function isHtmlFileOutput(format18) {
    if (typeof format18 !== "string") {
        format18 = format18?.to || "html";
    }
    return isHtmlDocOutput(format18) || isHtmlSlideOutput(format18);
}
function isHtmlOutput(format19, strict) {
    if (typeof format19 !== "string") {
        format19 = format19?.to;
    }
    format19 = format19 || "html";
    if (isHtmlDocOutput(format19)) {
        return true;
    } else if (!strict) {
        return isHtmlSlideOutput(format19) || isEpubOutput(format19);
    } else {
        return false;
    }
}
function isHtmlDocOutput(format20) {
    if (typeof format20 !== "string") {
        format20 = format20?.to || "html";
    }
    return [
        "html",
        "html4",
        "html5", 
    ].includes(format20);
}
function isHtmlSlideOutput(format21) {
    return [
        "s5",
        "dzslides",
        "slidy",
        "slideous",
        "revealjs", 
    ].includes(format21);
}
function isPresentationOutput(format22) {
    if (format22.to) {
        return [
            "s5",
            "dzslides",
            "slidy",
            "slideous",
            "revealjs",
            "beamer",
            "pptx"
        ].some((to)=>format22.to?.startsWith(to)
        );
    } else {
        return false;
    }
}
function isRevealjsOutput(format23) {
    if (typeof format23 !== "string") {
        format23 = format23?.to;
    }
    format23 = format23 || "html";
    return format23.startsWith("revealjs");
}
function isIpynbOutput(format24) {
    return !!format24.to && format24.to.startsWith("ipynb");
}
function isMarkdownOutput(format25, flavors = [
    "markdown",
    "gfm",
    "commonmark"
]) {
    const to = (format25.to || "").replace(/[\+\-_].*$/, "");
    return flavors.includes(to) || isIpynbOutput(format25);
}
function isHtmlCompatible(format26) {
    return isHtmlOutput(format26.pandoc) || isMarkdownOutput(format26.pandoc) && format26.render[kPreferHtml] || isIpynbOutput(format26.pandoc);
}
function isJavascriptCompatible(format27) {
    return isHtmlCompatible(format27) && !isEpubOutput(format27.pandoc);
}
function quartoDataDir(subdir, roaming = false) {
    return quartoDir(userDataDir, subdir, roaming);
}
function quartoCacheDir(subdir) {
    return quartoDir(userCacheDir, subdir);
}
function quartoRuntimeDir(subdir) {
    return quartoDir(userRuntimeDir, subdir);
}
function quartoDir(sourceFn, subdir, roaming) {
    const dir = sourceFn("quarto", roaming);
    const fullDir = subdir ? join3(dir, subdir) : dir;
    ensureDirSync(fullDir);
    return fullDir;
}
function userDataDir(appName, roaming = false) {
    switch(Deno.build.os){
        case "darwin":
            return darwinUserDataDir(appName);
        case "linux":
            return xdgUserDataDir(appName);
        case "windows":
            return windowsUserDataDir(appName, roaming);
    }
}
function userCacheDir(appName) {
    switch(Deno.build.os){
        case "darwin":
            return darwinUserCacheDir(appName);
        case "linux":
            return xdgUserCacheDir(appName);
        case "windows":
            return windowsUserDataDir(appName);
    }
}
function userRuntimeDir(appName) {
    switch(Deno.build.os){
        case "darwin":
            return darwinUserCacheDir(appName);
        case "linux":
            return xdgUserRuntimeDir(appName);
        case "windows":
            return windowsUserDataDir(appName);
    }
}
function darwinUserDataDir(appName) {
    return join3(Deno.env.get("HOME") || "", "Library", "Application Support", appName);
}
function darwinUserCacheDir(appName) {
    return join3(Deno.env.get("HOME") || "", "Library", "Caches", appName);
}
function xdgUserDataDir(appName) {
    const dataHome = Deno.env.get("XDG_DATA_HOME") || join3(Deno.env.get("HOME") || "", ".local", "share");
    return join3(dataHome, appName);
}
function xdgUserCacheDir(appName) {
    const cacheHome = Deno.env.get("XDG_CACHE_HOME") || join3(Deno.env.get("HOME") || "", ".cache");
    return join3(cacheHome, appName);
}
function xdgUserRuntimeDir(appName) {
    const runtimeDir = Deno.env.get("XDG_RUNTIME_DIR");
    if (runtimeDir) {
        return runtimeDir;
    } else {
        return xdgUserDataDir(appName);
    }
}
function windowsUserDataDir(appName, roaming = false) {
    const dir = (roaming ? Deno.env.get("APPDATA") : Deno.env.get("LOCALAPPDATA")) || "";
    return join3(dir, appName);
}
async function dartCompile(input, temp, loadPaths, compressed) {
    const inputFilePath = temp.createFile({
        suffix: "scss"
    });
    Deno.writeTextFileSync(inputFilePath, input);
    const args = [
        inputFilePath,
        "--style",
        compressed ? "compressed" : "expanded", 
    ];
    if (loadPaths) {
        loadPaths.forEach((loadPath)=>{
            args.push(`--load-path=${loadPath}`);
        });
    }
    return await dartCommand(args);
}
async function dartCommand(args) {
    const command = Deno.build.os === "windows" ? "sass.bat" : "sass";
    const sass = binaryPath(join3("dart-sass", command));
    const cmd = [
        sass,
        ...args, 
    ];
    const result = await execProcess({
        cmd,
        stdout: "piped",
        stderr: "piped"
    });
    if (result.success) {
        return result.stdout;
    } else {
        const errLines = lines(result.stderr || "");
        const errMsg = errLines.slice(0, errLines.length - 2).join("\n");
        throw new Error("Theme file compilation failed:\n\n" + errMsg);
    }
}
var commonjsGlobal1 = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
function createCommonjsModule3(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {},
        require: function(path67, base5) {
            return commonjsRequire(path67, base5 === void 0 || base5 === null ? module.path : base5);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var md5 = createCommonjsModule3(function(module) {
    (function($) {
        function safeAdd(x, y) {
            var lsw = (x & 65535) + (y & 65535);
            var msw = (x >> 16) + (y >> 16) + (lsw >> 16);
            return msw << 16 | lsw & 65535;
        }
        function bitRotateLeft(num, cnt) {
            return num << cnt | num >>> 32 - cnt;
        }
        function md5cmn(q, a, b, x, s, t) {
            return safeAdd(bitRotateLeft(safeAdd(safeAdd(a, q), safeAdd(x, t)), s), b);
        }
        function md5ff(a, b, c, d, x, s, t) {
            return md5cmn(b & c | ~b & d, a, b, x, s, t);
        }
        function md5gg(a, b, c, d, x, s, t) {
            return md5cmn(b & d | c & ~d, a, b, x, s, t);
        }
        function md5hh(a, b, c, d, x, s, t) {
            return md5cmn(b ^ c ^ d, a, b, x, s, t);
        }
        function md5ii(a, b, c, d, x, s, t) {
            return md5cmn(c ^ (b | ~d), a, b, x, s, t);
        }
        function binlMD5(x, len2) {
            x[len2 >> 5] |= 128 << len2 % 32;
            x[(len2 + 64 >>> 9 << 4) + 14] = len2;
            var i50;
            var olda;
            var oldb;
            var oldc;
            var oldd;
            var a = 1732584193;
            var b = -271733879;
            var c = -1732584194;
            var d = 271733878;
            for(i50 = 0; i50 < x.length; i50 += 16){
                olda = a;
                oldb = b;
                oldc = c;
                oldd = d;
                a = md5ff(a, b, c, d, x[i50], 7, -680876936);
                d = md5ff(d, a, b, c, x[i50 + 1], 12, -389564586);
                c = md5ff(c, d, a, b, x[i50 + 2], 17, 606105819);
                b = md5ff(b, c, d, a, x[i50 + 3], 22, -1044525330);
                a = md5ff(a, b, c, d, x[i50 + 4], 7, -176418897);
                d = md5ff(d, a, b, c, x[i50 + 5], 12, 1200080426);
                c = md5ff(c, d, a, b, x[i50 + 6], 17, -1473231341);
                b = md5ff(b, c, d, a, x[i50 + 7], 22, -45705983);
                a = md5ff(a, b, c, d, x[i50 + 8], 7, 1770035416);
                d = md5ff(d, a, b, c, x[i50 + 9], 12, -1958414417);
                c = md5ff(c, d, a, b, x[i50 + 10], 17, -42063);
                b = md5ff(b, c, d, a, x[i50 + 11], 22, -1990404162);
                a = md5ff(a, b, c, d, x[i50 + 12], 7, 1804603682);
                d = md5ff(d, a, b, c, x[i50 + 13], 12, -40341101);
                c = md5ff(c, d, a, b, x[i50 + 14], 17, -1502002290);
                b = md5ff(b, c, d, a, x[i50 + 15], 22, 1236535329);
                a = md5gg(a, b, c, d, x[i50 + 1], 5, -165796510);
                d = md5gg(d, a, b, c, x[i50 + 6], 9, -1069501632);
                c = md5gg(c, d, a, b, x[i50 + 11], 14, 643717713);
                b = md5gg(b, c, d, a, x[i50], 20, -373897302);
                a = md5gg(a, b, c, d, x[i50 + 5], 5, -701558691);
                d = md5gg(d, a, b, c, x[i50 + 10], 9, 38016083);
                c = md5gg(c, d, a, b, x[i50 + 15], 14, -660478335);
                b = md5gg(b, c, d, a, x[i50 + 4], 20, -405537848);
                a = md5gg(a, b, c, d, x[i50 + 9], 5, 568446438);
                d = md5gg(d, a, b, c, x[i50 + 14], 9, -1019803690);
                c = md5gg(c, d, a, b, x[i50 + 3], 14, -187363961);
                b = md5gg(b, c, d, a, x[i50 + 8], 20, 1163531501);
                a = md5gg(a, b, c, d, x[i50 + 13], 5, -1444681467);
                d = md5gg(d, a, b, c, x[i50 + 2], 9, -51403784);
                c = md5gg(c, d, a, b, x[i50 + 7], 14, 1735328473);
                b = md5gg(b, c, d, a, x[i50 + 12], 20, -1926607734);
                a = md5hh(a, b, c, d, x[i50 + 5], 4, -378558);
                d = md5hh(d, a, b, c, x[i50 + 8], 11, -2022574463);
                c = md5hh(c, d, a, b, x[i50 + 11], 16, 1839030562);
                b = md5hh(b, c, d, a, x[i50 + 14], 23, -35309556);
                a = md5hh(a, b, c, d, x[i50 + 1], 4, -1530992060);
                d = md5hh(d, a, b, c, x[i50 + 4], 11, 1272893353);
                c = md5hh(c, d, a, b, x[i50 + 7], 16, -155497632);
                b = md5hh(b, c, d, a, x[i50 + 10], 23, -1094730640);
                a = md5hh(a, b, c, d, x[i50 + 13], 4, 681279174);
                d = md5hh(d, a, b, c, x[i50], 11, -358537222);
                c = md5hh(c, d, a, b, x[i50 + 3], 16, -722521979);
                b = md5hh(b, c, d, a, x[i50 + 6], 23, 76029189);
                a = md5hh(a, b, c, d, x[i50 + 9], 4, -640364487);
                d = md5hh(d, a, b, c, x[i50 + 12], 11, -421815835);
                c = md5hh(c, d, a, b, x[i50 + 15], 16, 530742520);
                b = md5hh(b, c, d, a, x[i50 + 2], 23, -995338651);
                a = md5ii(a, b, c, d, x[i50], 6, -198630844);
                d = md5ii(d, a, b, c, x[i50 + 7], 10, 1126891415);
                c = md5ii(c, d, a, b, x[i50 + 14], 15, -1416354905);
                b = md5ii(b, c, d, a, x[i50 + 5], 21, -57434055);
                a = md5ii(a, b, c, d, x[i50 + 12], 6, 1700485571);
                d = md5ii(d, a, b, c, x[i50 + 3], 10, -1894986606);
                c = md5ii(c, d, a, b, x[i50 + 10], 15, -1051523);
                b = md5ii(b, c, d, a, x[i50 + 1], 21, -2054922799);
                a = md5ii(a, b, c, d, x[i50 + 8], 6, 1873313359);
                d = md5ii(d, a, b, c, x[i50 + 15], 10, -30611744);
                c = md5ii(c, d, a, b, x[i50 + 6], 15, -1560198380);
                b = md5ii(b, c, d, a, x[i50 + 13], 21, 1309151649);
                a = md5ii(a, b, c, d, x[i50 + 4], 6, -145523070);
                d = md5ii(d, a, b, c, x[i50 + 11], 10, -1120210379);
                c = md5ii(c, d, a, b, x[i50 + 2], 15, 718787259);
                b = md5ii(b, c, d, a, x[i50 + 9], 21, -343485551);
                a = safeAdd(a, olda);
                b = safeAdd(b, oldb);
                c = safeAdd(c, oldc);
                d = safeAdd(d, oldd);
            }
            return [
                a,
                b,
                c,
                d
            ];
        }
        function binl2rstr(input) {
            var i51;
            var output = "";
            var length32 = input.length * 32;
            for(i51 = 0; i51 < length32; i51 += 8){
                output += String.fromCharCode(input[i51 >> 5] >>> i51 % 32 & 255);
            }
            return output;
        }
        function rstr2binl(input) {
            var i52;
            var output = [];
            output[(input.length >> 2) - 1] = void 0;
            for(i52 = 0; i52 < output.length; i52 += 1){
                output[i52] = 0;
            }
            var length8 = input.length * 8;
            for(i52 = 0; i52 < length8; i52 += 8){
                output[i52 >> 5] |= (input.charCodeAt(i52 / 8) & 255) << i52 % 32;
            }
            return output;
        }
        function rstrMD5(s) {
            return binl2rstr(binlMD5(rstr2binl(s), s.length * 8));
        }
        function rstrHMACMD5(key, data45) {
            var i53;
            var bkey = rstr2binl(key);
            var ipad = [];
            var opad = [];
            var hash;
            ipad[15] = opad[15] = void 0;
            if (bkey.length > 16) {
                bkey = binlMD5(bkey, key.length * 8);
            }
            for(i53 = 0; i53 < 16; i53 += 1){
                ipad[i53] = bkey[i53] ^ 909522486;
                opad[i53] = bkey[i53] ^ 1549556828;
            }
            hash = binlMD5(ipad.concat(rstr2binl(data45)), 512 + data45.length * 8);
            return binl2rstr(binlMD5(opad.concat(hash), 512 + 128));
        }
        function rstr2hex(input) {
            var hexTab = "0123456789abcdef";
            var output = "";
            var x;
            var i54;
            for(i54 = 0; i54 < input.length; i54 += 1){
                x = input.charCodeAt(i54);
                output += hexTab.charAt(x >>> 4 & 15) + hexTab.charAt(x & 15);
            }
            return output;
        }
        function str2rstrUTF8(input) {
            return unescape(encodeURIComponent(input));
        }
        function rawMD5(s) {
            return rstrMD5(str2rstrUTF8(s));
        }
        function hexMD5(s) {
            return rstr2hex(rawMD5(s));
        }
        function rawHMACMD5(k, d) {
            return rstrHMACMD5(str2rstrUTF8(k), str2rstrUTF8(d));
        }
        function hexHMACMD5(k, d) {
            return rstr2hex(rawHMACMD5(k, d));
        }
        function md52(string19, key, raw) {
            if (!key) {
                if (!raw) {
                    return hexMD5(string19);
                }
                return rawMD5(string19);
            }
            if (!raw) {
                return hexHMACMD5(key, string19);
            }
            return rawHMACMD5(key, string19);
        }
        if (module.exports) {
            module.exports = md52;
        } else {
            $.md5 = md52;
        }
    })(commonjsGlobal1);
});
function md5Hash(content) {
    return md5(content);
}
function sassVariable(name88, value, formatter) {
    return {
        name: name88,
        value: formatter ? formatter(value) : value
    };
}
function outputVariable(variable, isDefault = true) {
    return `$${variable.name}: ${variable.value}${isDefault ? " !default" : ""};`;
}
async function compileSass(bundles, temp, minified = true) {
    const imports = __VIRTUAL_FILE181(bundles.flatMap((bundle)=>{
        return [
            ...bundle.user?.use || [],
            ...bundle.quarto?.use || [],
            ...bundle.framework?.use || [], 
        ];
    }));
    const useStatements = imports.map((use)=>{
        return `@use '${use}';`;
    }).join("\n");
    const frameworkFunctions = bundles.map((bundle)=>bundle.framework?.functions || ""
    );
    const frameworkDefaults = bundles.map((bundle)=>bundle.framework?.defaults || ""
    );
    const frameworkRules = bundles.map((bundle)=>bundle.framework?.rules || ""
    );
    const frameworkMixins = bundles.map((bundle)=>bundle.framework?.mixins || ""
    );
    const quartoFunctions1 = bundles.map((bundle)=>bundle.quarto?.functions || ""
    );
    const quartoDefaults1 = bundles.map((bundle)=>bundle.quarto?.defaults || ""
    );
    const quartoRules1 = bundles.map((bundle)=>bundle.quarto?.rules || ""
    );
    const quartoMixins = bundles.map((bundle)=>bundle.quarto?.mixins || ""
    );
    const userFunctions = bundles.map((bundle)=>bundle.user?.functions || ""
    );
    const userDefaults = bundles.map((bundle)=>bundle.user?.defaults || ""
    );
    const userRules = bundles.map((bundle)=>bundle.user?.rules || ""
    );
    const userMixins = bundles.map((bundle)=>bundle.user?.mixins || ""
    );
    const loadPaths = [];
    bundles.forEach((bundle)=>{
        if (bundle.loadPaths) {
            loadPaths.push(...bundle.loadPaths);
        }
    });
    const scssInput = [
        useStatements,
        ...frameworkFunctions,
        ...quartoFunctions1,
        ...userFunctions,
        ...userDefaults.reverse(),
        ...quartoDefaults1.reverse(),
        ...frameworkDefaults.reverse(),
        ...frameworkMixins,
        ...quartoMixins,
        ...userMixins,
        ...frameworkRules,
        ...quartoRules1,
        ...userRules, 
    ].join("\n\n");
    return await compileWithCache(scssInput, loadPaths, temp, minified, bundles.map((bundle)=>bundle.key
    ).join("|") + "-" + (minified ? "min" : "nomin"));
}
const layoutBoundary = "^\/\\*\\-\\-[ \\t]*scss:(functions|rules|defaults|mixins)[ \\t]*\\-\\-\\*\\/$";
const kLayerBoundaryLine = RegExp(layoutBoundary);
const kLayerBoundaryTest = RegExp(layoutBoundary, "m");
function mergeLayers(...layers) {
    const themeUse = [];
    const themeDefaults = [];
    const themeRules = [];
    const themeFunctions = [];
    const themeMixins = [];
    layers.forEach((theme)=>{
        if (theme.use) {
            themeUse.push(...theme.use);
        }
        if (theme.defaults) {
            themeDefaults.unshift(theme.defaults);
        }
        if (theme.rules) {
            themeRules.push(theme.rules);
        }
        if (theme.functions) {
            themeFunctions.push(theme.functions);
        }
        if (theme.mixins) {
            themeMixins.push(theme.mixins);
        }
    });
    return {
        use: __VIRTUAL_FILE181(themeUse),
        defaults: themeDefaults.join("\n"),
        functions: themeFunctions.join("\n"),
        mixins: themeMixins.join("\n"),
        rules: themeRules.join("\n")
    };
}
function sassLayer(path68) {
    if (Deno.statSync(path68).isFile) {
        return sassLayerFile(path68);
    } else {
        return sassLayerDir(path68, {
            functions: "_functions.scss",
            defaults: "_defaults.scss",
            mixins: "_mixins.scss",
            rules: "_rules.scss"
        });
    }
}
function sassLayerFile(theme) {
    const rawContents = Deno.readTextFileSync(theme);
    if (!kLayerBoundaryTest.test(rawContents)) {
        throw new Error(`The file ${theme} doesn't contain at least one layer boundary (/*-- scss:defaults --*/, /*-- scss:rules --*/, /*-- scss:mixins --*/ or /*-- scss:functions --*/)`);
    }
    const defaults = [];
    const rules = [];
    const functions = [];
    const mixins = [];
    let accum = defaults;
    lines(rawContents).forEach((line)=>{
        const scopeMatch = line.match(kLayerBoundaryLine);
        if (scopeMatch) {
            const scope = scopeMatch[1];
            switch(scope){
                case "defaults":
                    accum = defaults;
                    break;
                case "rules":
                    accum = rules;
                    break;
                case "functions":
                    accum = functions;
                    break;
                case "mixins":
                    accum = mixins;
                    break;
            }
        } else {
            accum.push(line);
        }
    });
    return {
        defaults: defaults.join("\n"),
        rules: rules.join("\n"),
        mixins: mixins.join("\n"),
        functions: functions.join("\n")
    };
}
function sassLayerDir(dir, names) {
    const read = (path69)=>{
        if (path69) {
            path69 = join3(dir, path69);
            if (existsSync(path69)) {
                return Deno.readTextFileSync(path69);
            } else {
                return "";
            }
        } else {
            return "";
        }
    };
    return {
        defaults: read(names.defaults),
        rules: read(names.rules),
        mixins: read(names.mixins),
        functions: read(names.functions)
    };
}
async function compileWithCache(input, loadPaths, temp, compressed, cacheIdentifier) {
    if (cacheIdentifier) {
        const identifierHash = md5Hash(cacheIdentifier);
        const inputHash = md5Hash(input);
        const cacheDir = quartoCacheDir("sass");
        const cacheIdxPath = join3(cacheDir, "index.json");
        const outputFile = `${identifierHash}.css`;
        const outputFilePath = join3(cacheDir, outputFile);
        let cacheIndex = {};
        let writeCache = true;
        if (existsSync(outputFilePath)) {
            cacheIndex = JSON.parse(Deno.readTextFileSync(cacheIdxPath));
            const existingEntry = cacheIndex[identifierHash];
            writeCache = !existingEntry || existingEntry.hash !== inputHash;
        }
        if (writeCache) {
            const cssOutput = await dartCompile(input, temp, loadPaths, compressed);
            if (cssOutput) {
                Deno.writeTextFileSync(outputFilePath, cssOutput || "");
            }
            cacheIndex[identifierHash] = {
                key: cacheIdentifier,
                hash: inputHash
            };
            Deno.writeTextFileSync(cacheIdxPath, JSON.stringify(cacheIndex));
        }
        return outputFilePath;
    } else {
        const outputFilePath = temp.createFile({
            suffix: "css"
        });
        const cssOutput = await dartCompile(input, temp, __VIRTUAL_FILE181(loadPaths), compressed);
        Deno.writeTextFileSync(outputFilePath, cssOutput || "");
        return outputFilePath;
    }
}
const kTabsets = "tabsets";
const kCodeCopy = "code-copy";
const kAnchorSections = "anchor-sections";
const kCitationsHover = "citations-hover";
const kFootnotesHover = "footnotes-hover";
const kMinimal = "minimal";
const kPageLayout = "page-layout";
const kPageLayoutArticle = "article";
const kPageLayoutCustom = "custom";
const kPageLayoutFull = "full";
const kComments = "comments";
const kHypothesis = "hypothesis";
const kUtterances = "utterances";
const kDocumentCss = "document-css";
const kBootstrapDependencyName = "bootstrap";
const clipboardDependency = ()=>{
    const dependency = {
        name: "clipboard"
    };
    dependency.scripts = [];
    dependency.scripts.push({
        name: "clipboard.min.js",
        path: formatResourcePath("html", join3("clipboard", "clipboard.min.js"))
    });
    return dependency;
};
const bootstrapFunctions = ()=>{
    return Deno.readTextFileSync(join3(bootstrapResourceDir(), "_functions.scss"));
};
const bootstrapMixins = ()=>{
    return Deno.readTextFileSync(join3(bootstrapResourceDir(), "_mixins.scss"));
};
const bootstrapVariables = ()=>{
    return Deno.readTextFileSync(join3(bootstrapResourceDir(), "_variables.scss"));
};
const bootstrapRules = ()=>{
    return Deno.readTextFileSync(join3(bootstrapResourceDir(), "bootstrap.scss"));
};
const bootstrapResourceDir = ()=>{
    return formatResourcePath("html", join3("bootstrap", "dist", "scss"));
};
const sassUtilFunctions = (name89)=>{
    const bootstrapDistDir = formatResourcePath("html", join3("bootstrap", "dist"));
    const path70 = join3(bootstrapDistDir, "sass-utils", name89);
    return Deno.readTextFileSync(path70);
};
const quartoRules = ()=>Deno.readTextFileSync(formatResourcePath("html", "_quarto-rules.scss"))
;
const quartoCopyCodeRules = ()=>Deno.readTextFileSync(formatResourcePath("html", "_quarto-rules-copy-code.scss"))
;
const quartoLinkExternalRules = ()=>Deno.readTextFileSync(formatResourcePath("html", "_quarto-rules-link-external.scss"))
;
const quartoTabbyRules = ()=>Deno.readTextFileSync(formatResourcePath("html", "_quarto-rules-tabby.scss"))
;
const quartoFigResponsiveRules = ()=>{
    return [
        ".img-fluid {",
        "  max-width: 100%;",
        "  height: auto;",
        "}", 
    ].join("\n");
};
const quartoGlobalCssVariableRules = ()=>{
    return `
  $font-family-monospace: SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace !default;
  /*! quarto-variables-start */
  :root {
    --quarto-font-monospace: #{inspect($font-family-monospace)};
  }
  /*! quarto-variables-end */
  `;
};
const quartoBootstrapCustomizationLayer = ()=>{
    const path71 = formatResourcePath("html", join3("bootstrap", "_bootstrap-customize.scss"));
    return sassLayer(path71);
};
const quartoBootstrapRules = ()=>Deno.readTextFileSync(formatResourcePath("html", join3("bootstrap", "_bootstrap-rules.scss")))
;
const quartoBootstrapMixins = ()=>Deno.readTextFileSync(formatResourcePath("html", join3("bootstrap", "_bootstrap-mixins.scss")))
;
const quartoBootstrapFunctions = ()=>Deno.readTextFileSync(formatResourcePath("html", join3("bootstrap", "_bootstrap-functions.scss")))
;
const quartoBaseLayer = (format28, codeCopy = false, tabby = false, figResponsive = false)=>{
    const rules = [
        quartoRules()
    ];
    if (codeCopy) {
        rules.push(quartoCopyCodeRules());
    }
    if (tabby) {
        rules.push(quartoTabbyRules());
    }
    if (figResponsive) {
        rules.push(quartoFigResponsiveRules());
    }
    if (format28.render[kLinkExternalIcon]) {
        rules.push(quartoLinkExternalRules());
    }
    return {
        use: [
            "sass:color",
            "sass:map",
            "sass:math"
        ],
        defaults: [
            quartoDefaults(format28), 
        ].join("\n"),
        functions: quartoFunctions(),
        mixins: "",
        rules: rules.join("\n")
    };
};
const quartoFunctions = ()=>Deno.readTextFileSync(formatResourcePath("html", "_quarto-functions.scss"))
;
const quartoDefaults = (format29)=>{
    const defaults = [];
    defaults.push(outputVariable(sassVariable("code-copy-selector", format29.metadata[kCodeCopy] === undefined || format29.metadata[kCodeCopy] === "hover" ? '"pre.sourceCode:hover > "' : '""')));
    defaults.push(outputVariable(sassVariable("code-white-space", format29.render[kCodeOverflow] === "wrap" ? "pre-wrap" : "pre")));
    defaults.push(outputVariable(sassVariable(kTblCapLoc, format29.metadata[kTblCapLoc] || format29.metadata[kCapLoc] || kCapTop)));
    return defaults.join("\n");
};
function insertFootnotesTitle(doc, footnotesEl, language, level = 2) {
    const heading = doc.createElement("h" + level);
    const title1 = language[kSectionTitleFootnotes];
    if (typeof title1 == "string" && title1 !== "none") {
        heading.innerHTML = title1;
    }
    footnotesEl.insertBefore(heading, footnotesEl.firstChild);
    const hr = footnotesEl.querySelector("hr");
    if (hr) {
        hr.remove();
    }
}
function removeFootnoteBacklinks(footnotesEl) {
    const backlinks = footnotesEl.querySelectorAll(".footnote-back");
    for (const backlink of backlinks){
        backlink.remove();
    }
}
function setMainColumn(doc, column) {
    const selectors = [
        "main.content",
        ".page-navigation"
    ];
    selectors.forEach((selector)=>{
        const el = doc.querySelector(selector);
        if (el) {
            for (const clz of el.classList){
                if (clz.startsWith("column-")) {
                    el.classList.remove(clz);
                }
            }
            el.classList.add(column);
        }
    });
}
const kWebsite = "website";
const kSiteUrl = "site-url";
const kSitePath = "site-path";
const kSiteTitle = "title";
const kSiteFavicon = "favicon";
const kSiteRepoUrl = "repo-url";
const kSiteRepoBranch = "repo-branch";
const kSiteRepoActions = "repo-actions";
const kSiteReaderMode = "reader-mode";
const kSiteNavbar = "navbar";
const kSiteSidebar = "sidebar";
const kSiteSidebarStyle = "style";
const kSiteSidebarHeader = "header";
const kSiteSidebarFooter = "footer";
const kSitePageNavigation = "page-navigation";
const kPageFooter = "page-footer";
const kMarginHeader = "margin-header";
const kMarginFooter = "margin-footer";
const kBodyHeader = "body-header";
const kBodyFooter = "body-footer";
const kContents = "contents";
const kTwitterCard = "twitter-card";
const kOpenGraph = "open-graph";
const kCardStyle = "card-style";
const kImage = "image";
const kImageWidth = "image-width";
const kImageHeight = "image-height";
const kCreator = "creator";
const kLocale = "locale";
const kSiteName = "site-name";
function websiteConfigBoolean(name90, defaultValue, project) {
    const config = websiteConfig(name90, project);
    if (typeof config === "string") {
        return !!config;
    } else if (typeof config == "boolean") {
        return config;
    } else {
        return defaultValue;
    }
}
function websiteConfigString(name91, project) {
    const config = websiteConfig(name91, project);
    if (typeof config === "string") {
        return config;
    } else {
        return undefined;
    }
}
function websiteConfigMetadata(name92, project) {
    const config = websiteConfig(name92, project);
    if (typeof config === "object" && !Array.isArray(config)) {
        return config;
    } else {
        return undefined;
    }
}
function websiteConfigArray(name93, project) {
    const config = websiteConfig(name93, project);
    if (Array.isArray(config)) {
        return config;
    } else {
        return undefined;
    }
}
function websiteConfig(name94, project) {
    const site = project?.[kWebsite];
    if (site) {
        return site[name94];
    } else {
        return undefined;
    }
}
function websiteTitle(project) {
    return websiteConfigString(kSiteTitle, project);
}
function websiteBaseurl(project) {
    return websiteConfigString(kSiteUrl, project);
}
function websitePath(project) {
    let path72 = websiteConfigString(kSitePath, project);
    if (path72) {
        if (!path72.endsWith("/")) {
            path72 = path72 + "/";
        }
        return path72;
    } else {
        const baseUrl = websiteBaseurl(project);
        if (baseUrl) {
            try {
                const url = new URL(baseUrl);
                let path73 = url.pathname;
                if (!path73.endsWith("/")) {
                    path73 = path73 + "/";
                }
                return path73;
            } catch  {
                return "/";
            }
        } else {
            return "/";
        }
    }
}
function websiteRepoUrl(project) {
    const repoUrl = websiteConfigString(kSiteRepoUrl, project);
    if (repoUrl) {
        if (!repoUrl.endsWith("/")) {
            return repoUrl + "/";
        } else {
            return repoUrl;
        }
    } else {
        return undefined;
    }
}
function websiteRepoBranch(project) {
    return websiteConfigString(kSiteRepoBranch, project) || "main";
}
function websiteMetadataFields() {
    return [
        kWebsite,
        "site"
    ];
}
function isGithubRepoUrl(url) {
    return !!url.match(/^http[s]?:\/\/github\.com/);
}
function websiteConfigActions(key, subkey, project) {
    const book = project?.[subkey];
    if (book) {
        const value = book[key];
        if (typeof value === "string") {
            if (value === "none") {
                return [];
            } else {
                return [
                    value
                ];
            }
        } else if (Array.isArray(value)) {
            return value.map((x)=>String(x)
            );
        } else {
            return [];
        }
    } else {
        return [];
    }
}
function websiteProjectConfig(_projectDir, config, forceHtml) {
    config = cloneDeep1(config);
    const format110 = config[kMetadataFormat];
    if (format110 !== undefined) {
        if (typeof format110 === "string") {
            if (!isHtmlOutput(format110, true) && forceHtml) {
                config[kMetadataFormat] = {
                    html: "default",
                    [format110]: "default"
                };
            }
        } else {
            const formats2 = Object.keys(format110);
            const orderedFormats = {};
            if (forceHtml) {
                const htmlFormatPos = formats2.findIndex((format30)=>isHtmlOutput(format30, true)
                );
                if (htmlFormatPos !== -1) {
                    const htmlFormatName = formats2.splice(htmlFormatPos, 1)[0];
                    orderedFormats[htmlFormatName] = format110[htmlFormatName];
                } else {
                    orderedFormats["html"] = "default";
                }
            }
            for (const formatName of formats2){
                orderedFormats[formatName] = format110[formatName];
            }
            config[kMetadataFormat] = orderedFormats;
        }
    } else {
        config[kMetadataFormat] = "html";
    }
    const ensureArray = (val)=>{
        if (Array.isArray(val)) {
            return val;
        } else if (typeof val === "string") {
            return [
                val
            ];
        }
    };
    const siteMeta = config[kWebsite] || {};
    if (siteMeta[kBodyHeader]) {
        siteMeta[kBodyHeader] = ensureArray(siteMeta[kBodyHeader]);
    }
    if (siteMeta[kBodyFooter]) {
        siteMeta[kBodyFooter] = ensureArray(siteMeta[kBodyFooter]);
    }
    if (siteMeta[kMarginHeader]) {
        siteMeta[kMarginHeader] = ensureArray(siteMeta[kMarginHeader]);
    }
    if (siteMeta[kMarginFooter]) {
        siteMeta[kMarginFooter] = ensureArray(siteMeta[kMarginFooter]);
    }
    config[kWebsite] = siteMeta;
    const sidebarRaw = siteMeta[kSiteSidebar];
    const sidebars = Array.isArray(sidebarRaw) ? sidebarRaw : typeof sidebarRaw == "object" ? [
        sidebarRaw
    ] : undefined;
    sidebars?.forEach((sidebar)=>{
        const headerRaw = sidebar[kSiteSidebarHeader];
        if (headerRaw) {
            sidebar[kSiteSidebarHeader] = ensureArray(sidebar[kSiteSidebarHeader]);
        }
    });
    sidebars?.forEach((sidebar)=>{
        const footerRaw = sidebar[kSiteSidebarFooter];
        if (footerRaw) {
            sidebar[kSiteSidebarFooter] = ensureArray(sidebar[kSiteSidebarFooter]);
        }
    });
    if (websiteConfigMetadata(kComments, config) && config[kComments] === undefined) {
        config[kComments] = websiteConfigMetadata(kComments, config);
    }
    return Promise.resolve(config);
}
function websiteHtmlFormat(project) {
    const projConfig = project.config || {};
    const baseFormat = metadataAsFormat(projConfig);
    const format31 = formatFromMetadata(baseFormat, formatKeys(projConfig)[0]);
    return mergeConfigs(baseFormat, format31);
}
const kGoogleAnalytics = "google-analytics";
const kTrackingId = "tracking-id";
const kStorage = "storage";
const kAnonymizeIp = "anonymize-ip";
const kVersion = "version";
const kCookieConsent = "cookie-consent";
const kCookieConsentType = "type";
const kCookieConsentStyle = "style";
const kCookieConsentPalette = "palette";
const kCookieConsentPolicyUrl = "policy-url";
const kCookiePrefsText = "prefs-text";
function scriptTagWithConsent(consentRequired, consentlevel, contents, src) {
    const srcAttr = src ? ` src="${src}"` : "";
    if (consentRequired) {
        return `
<script type="text/plain" cookie-consent="${consentlevel}"${srcAttr}>
${contents}
</script>`;
    } else {
        return `
<script type="text/javascript"${srcAttr}>
${contents}
</script>`;
    }
}
function websiteAnalyticsScriptFile(project, temp) {
    const siteMeta = project.config?.[kWebsite];
    let gaConfig = undefined;
    if (siteMeta) {
        const siteGa = siteMeta[kGoogleAnalytics];
        if (typeof siteGa === "object") {
            const siteGaMeta = siteGa;
            const trackingId = siteGaMeta[kTrackingId];
            const storage = siteGaMeta[kStorage];
            const anonymizedIp = siteGaMeta[kAnonymizeIp];
            const version9 = siteGaMeta[kVersion];
            gaConfig = googleAnalyticsConfig(project, trackingId, storage, anonymizedIp, version9);
        } else if (siteGa && typeof siteGa === "string") {
            gaConfig = googleAnalyticsConfig(project, siteGa);
        }
    }
    if (gaConfig) {
        const script = analyticsScript(gaConfig);
        if (script) {
            return scriptFile(script, temp);
        } else {
            return undefined;
        }
    } else {
        return undefined;
    }
}
function cookieConsentDependencies(project, temp) {
    const siteMeta = project.config?.[kWebsite];
    if (siteMeta) {
        const title2 = siteMeta[kTitle] || "";
        let configuration = undefined;
        let changePrefsText = undefined;
        const consent = siteMeta[kCookieConsent];
        if (typeof consent === "object") {
            const cookieMeta = consent;
            configuration = cookieConsentConfiguration(title2, cookieMeta[kCookieConsentType], cookieMeta[kCookieConsentStyle], cookieMeta[kCookieConsentPalette], cookieMeta[kCookieConsentPolicyUrl]);
            changePrefsText = cookieMeta[kCookiePrefsText];
        } else if (consent) {
            configuration = cookieConsentConfiguration(title2);
        }
        if (configuration) {
            const consentResourcesDir = join3(projectTypeResourcePath("website"), "cookie-consent");
            const name95 = "cookie-consent.js";
            const path74 = join3(consentResourcesDir, name95);
            const cssName = "cookie-consent.css";
            const cssPath = join3(consentResourcesDir, cssName);
            return {
                scriptFile: scriptFile(cookieConsentScript(configuration), temp),
                dependency: {
                    name: "cookie-consent",
                    scripts: [
                        {
                            name: name95,
                            path: path74
                        }
                    ],
                    stylesheets: [
                        {
                            name: cssName,
                            path: cssPath
                        }
                    ]
                },
                htmlPostProcessor: (doc)=>{
                    const anchorId = "open_preferences_center";
                    const prefsAnchor = doc.getElementById(anchorId);
                    if (!prefsAnchor) {
                        const footer = doc.querySelector("div.nav-footer .nav-footer-center");
                        if (footer) {
                            const anchor = doc.createElement("a");
                            anchor.setAttribute("href", "#");
                            anchor.setAttribute("id", anchorId);
                            anchor.innerHTML = changePrefsText || "Cookie Preferences";
                            const anchorContainer = doc.createElement("div");
                            anchorContainer.setAttribute("class", "cookie-consent-footer");
                            anchorContainer.appendChild(anchor);
                            footer.appendChild(anchorContainer);
                        }
                    }
                    return Promise.resolve([]);
                }
            };
        } else {
            return undefined;
        }
    } else {
        return undefined;
    }
}
function cookieConsentEnabled(project) {
    const siteMeta = project.config?.[kWebsite];
    if (siteMeta) {
        return !!siteMeta[kCookieConsent];
    } else {
        return false;
    }
}
function cookieConsentConfiguration(_siteName, type, style, palette, policyUrl) {
    return {
        siteName: "",
        type: type || "implied",
        style: style || "simple",
        palette: palette || "light",
        policyUrl
    };
}
function googleAnalyticsConfig(project, trackingId, storage, anoymizeIp, version10) {
    return {
        trackingId,
        consent: cookieConsentEnabled(project),
        storage: storage || "cookies",
        anonymizeIp: anoymizeIp === undefined ? true : !!anoymizeIp,
        version: version10 || (trackingId.startsWith("UA-") ? 3 : 4)
    };
}
function analyticsScript(config) {
    if (config.version === 3) {
        return ga3Script(config);
    } else if (config.version === 4) {
        return ga4Script(config);
    } else {
        return undefined;
    }
}
function ga3Script(config) {
    const scripts = [];
    scripts.push(`
(function(i,s,o,g,r,a,m){i['GoogleAnalyticsObject']=r;i[r]=i[r]||function(){
  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),
  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)
  })(window,document,'script','https://www.google-analytics.com/analytics.js','ga');`);
    if (config.storage === "none") {
        scripts.push(`ga('create', '${config.trackingId}', { 'storage': 'none' });`);
    } else {
        scripts.push(`ga('create', '${config.trackingId}', 'auto');`);
    }
    scripts.push(`
ga('send', {
  hitType: 'pageview',
  'anonymizeIp': ${config.anonymizeIp},
});`);
    return scriptTagWithConsent(!!config.consent, "tracking", scripts.join("\n"));
}
function ga4Script(config) {
    const scripts = [];
    scripts.push(`
window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}
gtag('js', new Date());`);
    if (config.storage === "none") {
        scripts.push(` 
  gtag('consent', 'default', {
    'ad_storage': 'denied',
    'analytics_storage': 'denied'
  });`);
    }
    scripts.push(`gtag('config', '${config.trackingId}', { 'anonymize_ip': ${config.anonymizeIp}});`);
    return [
        `<script async src="https://www.googletagmanager.com/gtag/js?id=${config.trackingId}"></script>`,
        scriptTagWithConsent(!!config.consent, "tracking", scripts.join("\n")), 
    ].join("\n");
}
function cookieConsentScript(config) {
    const consentLevels = config.type === "implied" ? `["strictly-necessary","functionality","tracking","targeting"]` : `["strictly-necessary"]`;
    const privacyJs = config.policyUrl !== undefined ? `,\n"website_privacy_policy_url":"${config.policyUrl}"` : "";
    return `
<script type="text/javascript" charset="UTF-8">
document.addEventListener('DOMContentLoaded', function () {
cookieconsent.run({
  "notice_banner_type":"${config.style}",
  "consent_type":"${config.type}",
  "palette":"${config.palette === "dark" ? "dark" : "light"}",
  "language":"en",
  "page_load_consent_levels":${consentLevels},
  "notice_banner_reject_button_hide":false,
  "preferences_center_close_button_hide":false,
  "website_name":"${config.siteName}"${privacyJs}
  });
});
</script> 
  `;
}
function scriptFile(script, temp) {
    const gaScriptFile = temp.createFile({
        suffix: ".js"
    });
    Deno.writeTextFileSync(gaScriptFile, script);
    return gaScriptFile;
}
function computePageTitle(format32, extras) {
    const meta = extras?.metadata || {};
    const pageTitle = meta[kPageTitle] || format32.metadata[kPageTitle];
    const titlePrefix = extras?.pandoc?.[kTitlePrefix] || format32.metadata[kWebsite]?.[kTitle];
    const title3 = format32.metadata[kTitle];
    if (pageTitle !== undefined) {
        return pageTitle;
    } else if (titlePrefix !== undefined) {
        if (titlePrefix === title3) {
            return title3;
        } else if (title3 !== undefined) {
            return titlePrefix + " - " + title3;
        } else {
            return undefined;
        }
    } else {
        return title3;
    }
}
function inputFileHref(href) {
    const [hrefDir, hrefStem] = dirAndStem(href);
    const htmlHref = "/" + join3(hrefDir, `${hrefStem}.html`);
    return pathWithForwardSlashes(htmlHref);
}
function websiteNavigationConfig(project) {
    let navbar = websiteConfigMetadata(kSiteNavbar, project.config);
    if (typeof navbar !== "object") {
        navbar = undefined;
    }
    const sidebar = websiteConfig(kSiteSidebar, project.config);
    const sidebars = Array.isArray(sidebar) ? sidebar : typeof sidebar == "object" ? [
        sidebar
    ] : undefined;
    if (sidebars && sidebars[0]) {
        if (sidebars[0].style === undefined) {
            sidebars[0].style = "floating";
        }
        if (sidebars[0].contents === undefined) {
            sidebars[0].contents = [];
        }
        if (sidebars[0].tools === undefined) {
            sidebars[0].tools = [];
        }
    }
    if (sidebars && sidebars.length > 1) {
        const sidebarOptions = __VIRTUAL_FILE107(sidebars[0]);
        delete sidebarOptions.id;
        delete sidebarOptions.title;
        sidebarOptions.contents.splice(0, sidebarOptions.contents.length);
        for(let i55 = 1; i55 < sidebars.length; i55++){
            sidebars[i55] = {
                ...sidebarOptions,
                ...sidebars[i55]
            };
        }
    }
    const pageNavigation = websiteConfigBoolean(kSitePageNavigation, false, project.config);
    const footerValue = (value)=>{
        if (typeof value === "string") {
            return value;
        } else if (Array.isArray(value)) {
            return value;
        } else {
            return undefined;
        }
    };
    const footer = {};
    const footerConfig = websiteConfig(kPageFooter, project.config);
    if (typeof footerConfig === "string") {
        footer.center = footerConfig;
    } else if (!Array.isArray(footerConfig)) {
        footer.left = footerValue(footerConfig?.left);
        footer.center = footerValue(footerConfig?.center);
        footer.right = footerValue(footerConfig?.right);
    }
    if (footer.center === undefined && cookieConsentEnabled(project)) {
        footer.center = " ";
    }
    const pageMargin = {};
    const headerVal = websiteConfigArray(kMarginHeader, project.config);
    if (headerVal) {
        pageMargin.header = headerVal;
    }
    const footerVal = websiteConfigArray(kMarginFooter, project.config);
    if (footerVal) {
        pageMargin.footer = footerVal;
    }
    const bodyDecorators = {};
    const bodyHeaderVal = websiteConfigArray(kBodyHeader, project.config);
    if (bodyHeaderVal) {
        bodyDecorators.header = bodyHeaderVal;
    }
    const bodyFooterVal = websiteConfigArray(kBodyFooter, project.config);
    if (bodyFooterVal) {
        bodyDecorators.footer = bodyFooterVal;
    }
    return {
        navbar,
        sidebars,
        pageNavigation,
        footer,
        pageMargin,
        bodyDecorators
    };
}
function flattenItems(sidebarItems, includeItem) {
    const items = [];
    const flatten = (sidebarItem)=>{
        if (includeItem(sidebarItem)) {
            items.push(sidebarItem);
        }
        if (sidebarItem.contents) {
            items.push(...flattenItems(sidebarItem.contents, includeItem));
        }
    };
    sidebarItems.forEach(flatten);
    return items;
}
const kDependencies = "dependencies";
const kSassBundles = "sass-bundles";
const kHtmlPostprocessors = "html-postprocessors";
const kHtmlFinalizers = "html-finalizers";
const kTemplatePatches = "template-patches";
const kBodyEnvelope = "body-envelope";
const kTextHighlightingMode = "text-highlighting-mode";
const kQuartoCssVariables = "css-variables";
const kMarkdownAfterBody = "render-after-body";
const kTextHtml = "text/html";
const kTextMarkdown = "text/markdown";
const kTextLatex = "text/latex";
const kTextPlain = "text/plain";
const kImagePng = "image/png";
const kImageJpeg = "image/jpeg";
const kImageSvg = "image/svg+xml";
const kApplicationPdf = "application/pdf";
const kApplicationJavascript = "application/javascript";
const kApplicationJupyterWidgetState = "application/vnd.jupyter.widget-state+json";
const kApplicationJupyterWidgetView = "application/vnd.jupyter.widget-view+json";
const kRestructuredText = "text/restructuredtext";
const kApplicationRtf = "application/rtf";
function extensionForMimeImageType(mimeType) {
    switch(mimeType){
        case kImagePng:
            return "png";
        case kImageJpeg:
            return "jpeg";
        case kImageSvg:
            return "svg";
        case kApplicationPdf:
            return "pdf";
        default:
            return "bin";
    }
}
function contentType(path75) {
    return MEDIA_TYPES[extname2(path75.toLowerCase())];
}
function isPdfContent(path76) {
    return path76 && contentType(path76) === kApplicationPdf;
}
function isHtmlContent(path77) {
    return path77 && contentType(path77) === kTextHtml;
}
const MEDIA_TYPES = {
    ".md": kTextMarkdown,
    ".html": kTextHtml,
    ".htm": kTextHtml,
    ".json": "application/json",
    ".map": "application/json",
    ".txt": kTextPlain,
    ".ts": "text/typescript",
    ".tsx": "text/tsx",
    ".js": "application/javascript",
    ".jsx": "text/jsx",
    ".gz": "application/gzip",
    ".css": "text/css",
    ".wasm": "application/wasm",
    ".mjs": "application/javascript",
    ".svg": kImageSvg,
    ".png": kImagePng,
    ".jpg": kImageJpeg,
    ".jpeg": kImageJpeg,
    ".pdf": kApplicationPdf,
    ".gif": "image/gif",
    ".wav": "audio/wav",
    ".mp4": "video/mp4",
    ".woff": "application/font-woff",
    ".ttf": "application/font-ttf",
    ".eot": "application/vnd.ms-fontobject",
    ".otf": "application/font-otf"
};
function isAbsoluteRef(href) {
    return /^(?:http|https)\:\/\/.+/.test(href);
}
function isFileRef(href) {
    return !/^\w+:/.test(href) && !href.startsWith("#");
}
function httpFileRequestHandler(options) {
    async function serveFile(filePath, req) {
        let fileContents;
        if (options.onFile) {
            fileContents = await options.onFile(filePath, req);
        }
        if (!fileContents) {
            fileContents = Deno.readFileSync(filePath);
        }
        return httpContentResponse(fileContents, contentType(filePath));
    }
    function serveFallback(req, e, fsPath) {
        const encoder = new TextEncoder();
        if (e instanceof URIError) {
            return Promise.resolve(new Response(encoder.encode("BadRequest"), {
                status: 400
            }));
        } else if (e instanceof Deno.errors.NotFound) {
            const url = normalizeURL(req.url);
            const handle404 = options.on404 ? options.on404(url, req) : {
                print: true,
                body: encoder.encode("Not Found")
            };
            handle404.print = handle404.print && !!options.printUrls && (!fsPath || basename2(fsPath) !== "favicon.ico" && extname2(fsPath) !== ".map");
            if (handle404.print) {
                printUrl(url, false);
            }
            return Promise.resolve(new Response(handle404.body, {
                status: 404,
                headers: {
                    "Content-Type": kTextHtml
                }
            }));
        } else {
            error(`500 (Internal Error): ${e.message}`, {
                bold: true
            });
            return Promise.resolve(new Response(encoder.encode("Internal server error"), {
                status: 500
            }));
        }
    }
    return async (req)=>{
        if (options.onRequest) {
            const response = await options.onRequest(req);
            if (response) {
                return response;
            }
        }
        let response;
        let fsPath;
        try {
            const baseDir = pathWithForwardSlashes(options.baseDir);
            const normalizedUrl = normalizeURL(req.url);
            fsPath = pathWithForwardSlashes(baseDir + normalizedUrl);
            if (fsPath.indexOf(baseDir) !== 0) {
                fsPath = baseDir;
            }
            const fileInfo = existsSync(fsPath) ? Deno.statSync(fsPath) : undefined;
            if (fileInfo && fileInfo.isDirectory) {
                fsPath = join3(fsPath, options.defaultFile || "index.html");
            }
            if (fileInfo?.isDirectory && !normalizedUrl.endsWith("/")) {
                response = serveRedirect(normalizedUrl + "/");
            } else {
                response = await serveFile(fsPath, req);
                if (options.printUrls === "all") {
                    printUrl(normalizedUrl);
                }
            }
        } catch (e) {
            response = await serveFallback(req, e, fsPath);
        }
        return response;
    };
}
function httpContentResponse(content, contentType1) {
    if (typeof content === "string") {
        content = new TextEncoder().encode(content);
    }
    const headers = new Headers();
    headers.set("Content-Length", content.byteLength.toString());
    if (contentType1) {
        headers.set("Content-Type", contentType1);
    }
    headers.set("Cache-Control", "no-store, max-age=0");
    return new Response(content, {
        status: 200,
        headers
    });
}
function normalizeURL(url) {
    let normalizedUrl = url;
    try {
        normalizedUrl = decodeURI(normalizedUrl);
    } catch (e) {
        if (!(e instanceof URIError)) {
            throw e;
        }
    }
    try {
        const absoluteURI = new URL(normalizedUrl);
        normalizedUrl = decodeURI(absoluteURI.pathname);
    } catch (e1) {
        if (!(e1 instanceof TypeError)) {
            throw e1;
        }
    }
    if (normalizedUrl[0] !== "/") {
        throw new URIError("The request URI is malformed.");
    }
    normalizedUrl = posix.normalize(normalizedUrl);
    const startOfParams = normalizedUrl.indexOf("?");
    return startOfParams > -1 ? normalizedUrl.slice(0, startOfParams) : normalizedUrl;
}
function maybeDisplaySocketError(e) {
    if (!(e instanceof Deno.errors.BrokenPipe) && !(e instanceof Deno.errors.ConnectionAborted) && !(e instanceof Deno.errors.ConnectionReset) && !(e instanceof DOMException)) {
        logError(e);
    }
}
function serveRedirect(url) {
    const headers = new Headers();
    headers.set("Location", url);
    return new Response(null, {
        status: 301,
        headers
    });
}
function printUrl(url, found = true) {
    const format33 = !found ? red : undefined;
    const urlDisplay = url + (found ? "" : " (404: Not Found)");
    if (isHtmlContent(url) || url.endsWith("/") || extname2(url) === "") {
        info(`GET: ${urlDisplay}`, {
            bold: false,
            format: format33 || green
        });
    } else if (!found) {
        info(urlDisplay, {
            dim: found,
            format: format33,
            indent: 2
        });
    }
}
const kMarkdownBlockSeparator = "\n\n<!-- -->\n\n";
const kDefaultHighlightStyle = "arrow";
const kCssUrlRegex = /url\((?!['"]?(?:data|https?):)(['"])?([^'"]*?)\1\)/g;
const kCssImportRegex = /@import\s(?!['"](?:data|https?):)(['"])([^'"]*?)\1/g;
function cssFileResourceReferences(files) {
    return files.reduce((allRefs, file)=>{
        if (extname2(file).toLowerCase() === ".css") {
            if (existsSync(file)) {
                file = Deno.realPathSync(file);
                const css = Deno.readTextFileSync(file);
                const cssRefs = cssFileRefs(css).map((ref4)=>join3(dirname2(file), ref4)
                );
                allRefs.push(...cssRefs);
                allRefs.push(...cssFileResourceReferences(cssRefs));
            }
        }
        return allRefs;
    }, []);
}
function cssFileRefs(css) {
    return cssImports(css).concat(cssResources(css)).filter(isFileRef);
}
function cssResources(css) {
    return matchCss(css, kCssUrlRegex);
}
function cssImports(css) {
    return matchCss(css, kCssImportRegex);
}
function matchCss(css, regex) {
    const matches = [];
    regex.lastIndex = 0;
    let match = regex.exec(css);
    while(match != null){
        matches.push(match[2]);
        match = regex.exec(css);
    }
    regex.lastIndex = 0;
    return matches;
}
function asCssFont(value) {
    if (!value) {
        return undefined;
    } else {
        const fontFamily = String(value).split(",").map((font)=>{
            font = font.trim();
            if (font.includes(" ")) {
                font = `"${font}"`;
            }
            return font;
        }).filter((font)=>font.length > 0
        ).join(", ");
        return `${fontFamily}`;
    }
}
function asCssNumber(value) {
    if (typeof value === "number") {
        return String(value);
    } else if (!value) {
        return undefined;
    } else {
        const str65 = String(value);
        const match = str65.match(/(^[0-9]*)/);
        if (match) {
            return match[1];
        } else {
            return undefined;
        }
    }
}
function asCssSize(value) {
    if (typeof value === "number") {
        if (value === 0) {
            return "0";
        } else {
            return value + "px";
        }
    } else if (!value) {
        return undefined;
    } else {
        const str66 = String(value);
        if (str66 !== "0" && !str66.match(/\w$/)) {
            return str66 + "px";
        } else {
            return str66;
        }
    }
}
function asCssColor(value) {
    if (typeof value === "string") {
        return value;
    }
}
const kBootstrapColors = [
    "primary",
    "secondary",
    "success",
    "info",
    "warning",
    "danger",
    "light",
    "dark", 
];
const kBootstrapPaletteRegex = RegExp(`gray\-[1-9]00`);
function asBootstrapColor(value) {
    if (typeof value === "string") {
        if (kBootstrapColors.includes(value) || value.match(kBootstrapPaletteRegex)) {
            return `$${value}`;
        } else {
            return asCssColor(value);
        }
    }
}
function renderEjs(file, data46, removeEmptyLines = true) {
    const template1 = compileTemplate(file, removeEmptyLines);
    return lines(template1(data46).trimLeft()).filter((line)=>!removeEmptyLines || line.trim().length > 0
    ).join("\n") + "\n";
}
const compiledTemplates = new Map();
function compileTemplate(file, removeEmptyLines) {
    if (!compiledTemplates.has(file)) {
        const template2 = `<% const partial = (file, data) => print(include(file, data)); %>
      ${Deno.readTextFileSync(file)}`;
        compiledTemplates.set(file, __VIRTUAL_FILE176(template2, {
            imports: {
                include: (includeFile, includeData)=>{
                    return renderEjs(join3(dirname2(file), includeFile), includeData, removeEmptyLines);
                }
            }
        }));
    }
    return compiledTemplates.get(file);
}
const base64abc = [
    "A",
    "B",
    "C",
    "D",
    "E",
    "F",
    "G",
    "H",
    "I",
    "J",
    "K",
    "L",
    "M",
    "N",
    "O",
    "P",
    "Q",
    "R",
    "S",
    "T",
    "U",
    "V",
    "W",
    "X",
    "Y",
    "Z",
    "a",
    "b",
    "c",
    "d",
    "e",
    "f",
    "g",
    "h",
    "i",
    "j",
    "k",
    "l",
    "m",
    "n",
    "o",
    "p",
    "q",
    "r",
    "s",
    "t",
    "u",
    "v",
    "w",
    "x",
    "y",
    "z",
    "0",
    "1",
    "2",
    "3",
    "4",
    "5",
    "6",
    "7",
    "8",
    "9",
    "+",
    "/", 
];
function encode(data47) {
    const uint8 = typeof data47 === "string" ? new TextEncoder().encode(data47) : data47 instanceof Uint8Array ? data47 : new Uint8Array(data47);
    let result = "", i56;
    const l = uint8.length;
    for(i56 = 2; i56 < l; i56 += 3){
        result += base64abc[uint8[i56 - 2] >> 2];
        result += base64abc[(uint8[i56 - 2] & 3) << 4 | uint8[i56 - 1] >> 4];
        result += base64abc[(uint8[i56 - 1] & 15) << 2 | uint8[i56] >> 6];
        result += base64abc[uint8[i56] & 63];
    }
    if (i56 === l + 1) {
        result += base64abc[uint8[i56 - 2] >> 2];
        result += base64abc[(uint8[i56 - 2] & 3) << 4];
        result += "==";
    }
    if (i56 === l) {
        result += base64abc[uint8[i56 - 2] >> 2];
        result += base64abc[(uint8[i56 - 2] & 3) << 4 | uint8[i56 - 1] >> 4];
        result += base64abc[(uint8[i56 - 1] & 15) << 2];
        result += "=";
    }
    return result;
}
function decode(b64) {
    const binString = atob(b64);
    const size = binString.length;
    const bytes = new Uint8Array(size);
    for(let i57 = 0; i57 < size; i57++){
        bytes[i57] = binString.charCodeAt(i57);
    }
    return bytes;
}
function mappedSource(source, substrs) {
    const params = [];
    for (const { range  } of substrs){
        params.push(range);
    }
    return mappedString(source, params);
}
async function partitionCellOptionsMapped(language, source, _validate = false, _engine = "") {
    const commentChars = langCommentChars(language);
    const optionPrefix = optionCommentPrefix(commentChars[0]);
    const optionSuffix = commentChars[1] || "";
    const optionsSource = [];
    const yamlLines = [];
    let endOfYaml = 0;
    for (const line of rangedLines(source.value, true)){
        if (line.substring.startsWith(optionPrefix)) {
            if (!optionSuffix || line.substring.trimRight().endsWith(optionSuffix)) {
                let yamlOption = line.substring.substring(optionPrefix.length);
                if (optionSuffix) {
                    yamlOption = yamlOption.trimRight();
                    yamlOption = yamlOption.substring(0, yamlOption.length - optionSuffix.length);
                }
                endOfYaml = line.range.start + optionPrefix.length + yamlOption.length - optionSuffix.length;
                const rangedYamlOption = {
                    substring: yamlOption,
                    range: {
                        start: line.range.start + optionPrefix.length,
                        end: endOfYaml
                    }
                };
                yamlLines.push(rangedYamlOption);
                optionsSource.push(line);
                continue;
            }
        }
        break;
    }
    const mappedYaml = yamlLines.length ? mappedSource(source, yamlLines) : undefined;
    return {
        yaml: mappedYaml,
        optionsSource,
        source: mappedString(source, [
            {
                start: endOfYaml,
                end: source.value.length
            }
        ]),
        sourceStartLine: yamlLines.length
    };
}
function langCommentChars(lang) {
    const chars = kLangCommentChars[lang] || "#";
    if (!Array.isArray(chars)) {
        return [
            chars
        ];
    } else {
        return chars;
    }
}
function optionCommentPrefix(comment) {
    return comment + "| ";
}
const kLangCommentChars = {
    r: "#",
    python: "#",
    julia: "#",
    scala: "//",
    matlab: "%",
    csharp: "//",
    fsharp: "//",
    c: [
        "/*",
        "*/"
    ],
    css: [
        "/*",
        "*/"
    ],
    sas: [
        "*",
        ";"
    ],
    powershell: "#",
    bash: "#",
    sql: "--",
    mysql: "--",
    psql: "--",
    lua: "--",
    cpp: "//",
    cc: "//",
    stan: "#",
    octave: "#",
    fortran: "!",
    fortran95: "!",
    awk: "#",
    gawk: "#",
    stata: "*",
    java: "//",
    groovy: "//",
    sed: "#",
    perl: "#",
    ruby: "#",
    tikz: "%",
    js: "//",
    d3: "//",
    node: "//",
    sass: "//",
    coffee: "#",
    go: "//",
    asy: "//",
    haskell: "--",
    dot: "//",
    ojs: "//"
};
function checkForEqualsInChunk(error25, _parse, _schema) {
    if (typeof error25.violatingObject.result !== "string") {
        return error25;
    }
    const badObject = error25.source.value.substring(error25.violatingObject.start, error25.violatingObject.end);
    if (error25.ajvError.keyword !== "type") {
        return error25;
    }
    let m;
    const heading = `${error25.location}: ${quotedStringColor(badObject)} must be a YAML mapping.`;
    const errorMsg = [
        `${quotedStringColor(badObject)} is a string.`
    ];
    const newError = {
        heading,
        error: errorMsg,
        info: []
    };
    addFileInfo(newError, error25.source);
    addInstancePathInfo(newError, error25.ajvError.instancePath);
    if (m = badObject.match(/= *TRUE/i)) {
        newError.info.push(`Try using ${quotedStringColor(": true")} instead of ${quotedStringColor(m[0])}.`);
    } else if (m = badObject.match(/= *FALSE/i)) {
        newError.info.push(`Try using ${quotedStringColor(": false")} instead of ${quotedStringColor(m[0])}.`);
    } else if (badObject.match("=")) {
        newError.info.push(`Try using ${quotedStringColor(":")} instead of ${quotedStringColor("=")}.`);
    } else {
        return error25;
    }
    return {
        ...error25,
        message: tidyverseFormatError(newError)
    };
}
const makeEngineSchema = (engine)=>idSchema(objectRefSchemaFromContextGlob("cell-*", (field, _path)=>{
        const engineTag = field?.tags?.engine;
        switch(typeof engineTag){
            case "undefined":
                return true;
            case "string":
                return engineTag === engine;
            case "object":
                return engineTag.indexOf(engine) !== -1;
            default:
                throw new Error(`Internal Error: bad engine tag ${engineTag}`);
        }
    }), `engine-${engine}`)
;
const markdownEngineSchema = defineCached(async ()=>{
    return {
        schema: makeEngineSchema("markdown"),
        errorHandlers: []
    };
}, "engine-markdown");
const knitrEngineSchema = defineCached(async ()=>{
    const result = await makeEngineSchema("knitr");
    return {
        schema: result,
        errorHandlers: [
            checkForEqualsInChunk
        ]
    };
}, "engine-knitr");
const jupyterEngineSchema = defineCached(async ()=>{
    return {
        schema: makeEngineSchema("jupyter"),
        errorHandlers: []
    };
}, "engine-jupyter");
async function getEngineOptionsSchema() {
    const obj = {
        markdown: await markdownEngineSchema(),
        knitr: await knitrEngineSchema(),
        jupyter: await jupyterEngineSchema()
    };
    return obj;
}
function guessChunkOptionsFormat(options) {
    const noIndentOrColon = /^[^:\s]+[^:]+$/;
    const chunkLines = lines(options);
    if (chunkLines.filter((l)=>l.match(noIndentOrColon)
    ).length === 0) {
        return "yaml";
    }
    if (chunkLines.some((l)=>l.trim() !== "" && !l.trimRight().endsWith(",") && l.indexOf("=") === -1
    )) {
        return "yaml";
    }
    return "knitr";
}
function partitionCellOptions(language, source) {
    const commentChars = langCommentChars(language);
    const optionPrefix = optionCommentPrefix(commentChars[0]);
    const optionSuffix = commentChars[1] || "";
    const optionsSource = [];
    const yamlLines = [];
    for (const line of source){
        if (line.startsWith(optionPrefix)) {
            if (!optionSuffix || line.trimRight().endsWith(optionSuffix)) {
                let yamlOption = line.substring(optionPrefix.length);
                if (optionSuffix) {
                    yamlOption = yamlOption.trimRight();
                    yamlOption = yamlOption.substring(0, yamlOption.length - optionSuffix.length);
                }
                yamlLines.push(yamlOption);
                optionsSource.push(line);
                continue;
            }
        }
        break;
    }
    if (guessChunkOptionsFormat(yamlLines.join("\n")) === "knitr") {
        return {
            yaml: undefined,
            optionsSource,
            source: source.slice(yamlLines.length),
            sourceStartLine: yamlLines.length
        };
    }
    let yaml = yamlLines.length > 0 ? readYamlFromString(yamlLines.join("\n")) : undefined;
    if (yaml !== undefined && (typeof yaml !== "object" || Array.isArray(yaml))) {
        warnOnce("Invalid YAML option format in cell:\n" + yamlLines.join("\n"));
        yaml = undefined;
    }
    return {
        yaml: yaml,
        optionsSource,
        source: source.slice(yamlLines.length),
        sourceStartLine: yamlLines.length
    };
}
async function parseAndValidateCellOptions(mappedYaml, _language, validate1 = false, engine = "") {
    if (mappedYaml.value.trim().length === 0) {
        return undefined;
    }
    const engineOptionsSchema = await getEngineOptionsSchema();
    const schema = engineOptionsSchema[engine];
    if (schema === undefined || !validate1) {
        return readYamlFromString(mappedYaml.value);
    }
    return readAndValidateYamlFromMappedString(mappedYaml, schema, `Validation of YAML chunk options for engine ${engine} failed`);
}
async function partitionCellOptionsMapped1(language, outerSource, validate2 = false, engine = "") {
    const { yaml: mappedYaml , optionsSource , source , sourceStartLine ,  } = await partitionCellOptionsMapped(language, outerSource);
    if (guessChunkOptionsFormat((mappedYaml ?? asMappedString("")).value) === "yaml") {
        const yaml = await parseAndValidateCellOptions(mappedYaml ?? asMappedString(""), language, validate2, engine);
        return {
            yaml: yaml,
            optionsSource,
            source,
            sourceStartLine
        };
    } else {
        return {
            yaml: undefined,
            optionsSource,
            source,
            sourceStartLine
        };
    }
}
function shortUuid() {
    return generate().replaceAll("-", "").slice(0, 8);
}
class PngImage {
    content = [];
    width = -1;
    height = -1;
    dpiX = -1;
    dpiY = -1;
    bitDepth = -1;
    colourType = -1;
    compressionMethod = -1;
    filterMethod = -1;
    interlaceMethod = -1;
    constructor(bytes, sizeOnly = true){
        const magicNumberBytes = bytes.slice(0, 8);
        const magicNumber = bytesToString(magicNumberBytes);
        if (magicNumber !== "13780787113102610") {
            throw new Error("Not a png file");
        }
        let pos = 8;
        while(pos < bytes.length){
            const chunk = new PngChunk(bytes.slice(pos));
            switch(chunk.type){
                case "IHDR":
                    this.parseIHDRChunk(chunk, sizeOnly);
                    break;
                case "pHYs":
                    this.parsePHYSChunk(chunk);
                    break;
                case "IDAT":
                    return;
            }
            pos += chunk.totalLength;
        }
    }
    parseIHDRChunk(chunk, sizeOnly) {
        this.width = bytesToUint32(chunk.data, 0, 4);
        this.height = bytesToUint32(chunk.data, 4, 4);
        if (sizeOnly) {
            return;
        }
        this.bitDepth = chunk.data.slice(8, 9)[0];
        if (this.bitDepth !== 8) throw new Error("bitDepth not supported");
        this.colourType = chunk.data.slice(9, 10)[0];
        if (this.colourType !== 6) throw new Error("colourType not supported");
        this.compressionMethod = chunk.data.slice(10, 11)[0];
        if (this.compressionMethod !== 0) {
            throw new Error("compressionMethod not supported");
        }
        this.filterMethod = chunk.data.slice(11, 12)[0];
        if (this.filterMethod !== 0) throw new Error("filterMethod not supported");
        this.interlaceMethod = chunk.data.slice(12, 13)[0];
        if (this.interlaceMethod !== 0) {
            throw new Error("Interlacing not supported");
        }
    }
    parsePHYSChunk(chunk) {
        const meter = chunk.data[8];
        if (meter === 1) {
            const x = bytesToUint32(chunk.data, 0, 4);
            const y = bytesToUint32(chunk.data, 4, 4);
            const kMetersPerInch = 0.0254;
            this.dpiX = Math.round(x * kMetersPerInch);
            this.dpiY = Math.round(y * kMetersPerInch);
        }
    }
}
class PngChunk {
    totalLength;
    dataLength;
    type;
    data;
    crc;
    constructor(bytes){
        this.dataLength = this.getLength(bytes);
        this.type = this.getType(bytes);
        this.data = this.getData(bytes);
        this.crc = this.getCRC(bytes);
        this.totalLength = this.dataLength + 12;
    }
    getLength(bytes) {
        const lengthBytes = bytes.slice(0, 4);
        return bytesToUint32(lengthBytes);
    }
    getType(bytes) {
        const typeByte = bytes.slice(4, 8);
        return new TextDecoder("ascii").decode(typeByte);
    }
    getData(bytes) {
        return bytes.slice(8, 8 + this.dataLength);
    }
    getCRC(bytes) {
        return bytes.slice(8 + this.dataLength, 8 + this.dataLength + 4);
    }
}
const MAX_SIGNIFICANT_SIZE = 127;
function bytesToUint32(byteArray, start = 0, count) {
    if (count === undefined) {
        count = byteArray.length;
    }
    if (count > 4) {
        throw new Error("Length cannot be greater than 4");
    }
    let position = start;
    let value = 0;
    if (count === 4) {
        let sigValue = byteArray[position];
        if (sigValue > 127) {
            value += MAX_SIGNIFICANT_SIZE << 24;
            sigValue -= MAX_SIGNIFICANT_SIZE;
        }
        value += sigValue << 24;
        position++;
    }
    for(let i58 = position; i58 < start + count; i58++){
        value += byteArray[i58] << 8 * (count - (i58 - start) - 1);
    }
    return value;
}
function bytesToString(byteArray, start = 0, count) {
    if (count === undefined) {
        count = byteArray.length;
    }
    let result = "";
    for(let i59 = start; i59 < start + count; i59++){
        const __byte = byteArray[i59];
        if (__byte === 0) {
            result += "00";
        } else if (__byte < 10) {
            result += `0${__byte.toString()}`;
        } else {
            result += __byte.toString();
        }
    }
    return result;
}
function hideCell(cell, options) {
    return shouldHide(cell, options, kInclude);
}
function hideCode(cell, options) {
    return shouldHide(cell, options, kEcho);
}
function hideOutput(cell, options) {
    return shouldHide(cell, options, kOutput);
}
function hideWarnings(cell, options) {
    if (options.execute[kOutput] === false && cell.options[kOutput] !== false) {
        return cell.options[kWarning] || false;
    } else {
        return shouldHide(cell, options, kWarning);
    }
}
function includeCell(cell, options) {
    return shouldInclude(cell, options, kInclude);
}
function includeCode(cell, options) {
    return shouldInclude(cell, options, kEcho);
}
function echoFenced(cell, options) {
    const fenced = cell.options.echo === "fenced" || cell.options.echo === undefined && options.execute.echo === "fenced";
    return fenced;
}
function includeOutput(cell, options) {
    return shouldInclude(cell, options, kOutput);
}
function includeWarnings(cell, options) {
    if (options.execute[kOutput] === false && cell.options[kOutput] !== false) {
        return cell.options[kWarning] || false;
    } else {
        return shouldInclude(cell, options, kWarning);
    }
}
function shouldHide(cell, options, context) {
    if (cell.options[context] !== undefined) {
        return !cell.options[context] && options.keepHidden;
    } else {
        return !options.execute[context] && options.keepHidden;
    }
}
function shouldInclude(cell, options, context) {
    if (cell.options[context] !== undefined) {
        return !!(cell.options[context] || options.keepHidden);
    } else {
        return !!(options.execute[context] || options.keepHidden);
    }
}
function pandocAutoIdentifier(text, asciify) {
    if (asciify) {
        text = pandocAsciify(text);
    }
    return text.replace(/[!"#$%&')*+,\/:;<=>?@\[\\\]^`{|}~]/g, "").replace(/\s/g, "-").toLowerCase().replace(/^[^A-Za-z]+/, "");
}
function pandocAsciify(text) {
    const chars = [];
    const len3 = text.length;
    let newch;
    for(let i60 = 0; i60 < len3; i60++){
        newch = "";
        const ch = text.charCodeAt(i60);
        switch(ch){
            case 192:
            case 193:
            case 194:
            case 195:
            case 196:
            case 197:
            case 256:
            case 258:
            case 260:
            case 461:
            case 512:
            case 514:
            case 550:
            case 7680:
            case 7840:
            case 7842:
                newch = "A";
                break;
            case 7682:
            case 7684:
            case 7686:
                newch = "B";
                break;
            case 199:
            case 262:
            case 264:
            case 266:
            case 268:
                newch = "C";
                break;
            case 270:
            case 7690:
            case 7692:
            case 7694:
            case 7696:
            case 7698:
                newch = "D";
                break;
            case 200:
            case 201:
            case 203:
            case 274:
            case 276:
            case 278:
            case 280:
            case 282:
            case 516:
            case 518:
            case 552:
            case 7704:
            case 7706:
            case 7864:
            case 7866:
            case 7868:
                newch = "E";
                break;
            case 7710:
                newch = "F";
                break;
            case 284:
            case 286:
            case 288:
            case 290:
            case 486:
            case 500:
            case 7712:
                newch = "G";
                break;
            case 292:
            case 542:
            case 7714:
            case 7716:
            case 7718:
            case 7720:
            case 7722:
                newch = "H";
                break;
            case 204:
            case 205:
            case 206:
            case 207:
            case 296:
            case 298:
            case 300:
            case 302:
            case 304:
            case 463:
            case 520:
            case 522:
            case 7724:
            case 7880:
            case 7882:
                newch = "I";
                break;
            case 308:
                newch = "J";
                break;
            case 310:
            case 488:
            case 7728:
            case 7730:
            case 7732:
            case 8490:
                newch = "K";
                break;
            case 313:
            case 315:
            case 317:
            case 7734:
            case 7738:
            case 7740:
                newch = "L";
                break;
            case 7742:
            case 7744:
            case 7746:
                newch = "M";
                break;
            case 209:
            case 323:
            case 325:
            case 327:
            case 504:
            case 7748:
            case 7750:
            case 7752:
            case 7754:
                newch = "N";
                break;
            case 210:
            case 211:
            case 212:
            case 213:
            case 214:
            case 332:
            case 334:
            case 336:
            case 416:
            case 465:
            case 490:
            case 524:
            case 526:
            case 558:
            case 7884:
            case 7886:
                newch = "O";
                break;
            case 7764:
            case 7766:
                newch = "P";
                break;
            case 340:
            case 342:
            case 344:
            case 528:
            case 530:
            case 7768:
            case 7770:
            case 7774:
                newch = "R";
                break;
            case 346:
            case 348:
            case 350:
            case 352:
            case 536:
            case 7776:
            case 7778:
                newch = "S";
                break;
            case 354:
            case 356:
            case 538:
            case 7786:
            case 7788:
            case 7790:
            case 7792:
                newch = "T";
                break;
            case 217:
            case 218:
            case 219:
            case 220:
            case 360:
            case 362:
            case 364:
            case 366:
            case 368:
            case 370:
            case 431:
            case 467:
            case 532:
            case 534:
            case 7794:
            case 7796:
            case 7798:
            case 7908:
            case 7910:
                newch = "U";
                break;
            case 7804:
            case 7806:
                newch = "V";
                break;
            case 372:
            case 7808:
            case 7810:
            case 7812:
            case 7814:
            case 7816:
                newch = "W";
                break;
            case 7818:
            case 7820:
                newch = "X";
                break;
            case 221:
            case 374:
            case 376:
            case 562:
            case 7822:
            case 7922:
            case 7924:
            case 7926:
            case 7928:
                newch = "Y";
                break;
            case 377:
            case 379:
            case 381:
            case 7824:
            case 7826:
            case 7828:
                newch = "Z";
                break;
            case 224:
            case 225:
            case 226:
            case 227:
            case 228:
            case 229:
            case 257:
            case 259:
            case 261:
            case 462:
            case 513:
            case 515:
            case 551:
            case 553:
            case 7681:
            case 7841:
            case 7843:
                newch = "a";
                break;
            case 7683:
            case 7685:
            case 7687:
                newch = "b";
                break;
            case 231:
            case 263:
            case 265:
            case 267:
            case 269:
                newch = "c";
                break;
            case 271:
            case 7691:
            case 7693:
            case 7695:
            case 7697:
            case 7699:
                newch = "d";
                break;
            case 232:
            case 233:
            case 234:
            case 235:
            case 275:
            case 277:
            case 279:
            case 281:
            case 283:
            case 517:
            case 519:
            case 7705:
            case 7707:
            case 7865:
            case 7867:
            case 7869:
                newch = "e";
                break;
            case 7711:
                newch = "f";
                break;
            case 285:
            case 287:
            case 289:
            case 291:
            case 487:
            case 501:
            case 7713:
                newch = "g";
                break;
            case 293:
            case 543:
            case 7715:
            case 7717:
            case 7719:
            case 7721:
            case 7723:
            case 7830:
                newch = "h";
                break;
            case 236:
            case 237:
            case 238:
            case 239:
            case 297:
            case 299:
            case 301:
            case 303:
            case 305:
            case 464:
            case 521:
            case 523:
            case 7725:
            case 7881:
            case 7883:
                newch = "i";
                break;
            case 309:
            case 496:
                newch = "j";
                break;
            case 311:
            case 489:
            case 7729:
            case 7731:
            case 7733:
                newch = "k";
                break;
            case 314:
            case 316:
            case 318:
            case 7735:
            case 7739:
            case 7741:
                newch = "l";
                break;
            case 7743:
            case 7745:
            case 7747:
                newch = "m";
                break;
            case 241:
            case 324:
            case 326:
            case 328:
            case 505:
            case 7749:
            case 7751:
            case 7753:
            case 7755:
                newch = "n";
                break;
            case 242:
            case 243:
            case 244:
            case 245:
            case 246:
            case 333:
            case 335:
            case 337:
            case 417:
            case 432:
            case 466:
            case 491:
            case 525:
            case 527:
            case 559:
            case 7885:
            case 7887:
                newch = "o";
                break;
            case 7765:
            case 7767:
                newch = "p";
                break;
            case 341:
            case 343:
            case 345:
            case 529:
            case 531:
            case 7769:
            case 7771:
            case 7775:
                newch = "r";
                break;
            case 347:
            case 349:
            case 351:
            case 353:
            case 537:
            case 7777:
            case 7779:
                newch = "s";
                break;
            case 355:
            case 357:
            case 539:
            case 7787:
            case 7789:
            case 7791:
            case 7793:
            case 7831:
                newch = "t";
                break;
            case 249:
            case 250:
            case 251:
            case 252:
            case 361:
            case 363:
            case 365:
            case 367:
            case 369:
            case 371:
            case 468:
            case 533:
            case 535:
            case 7795:
            case 7797:
            case 7799:
            case 7909:
            case 7911:
                newch = "u";
                break;
            case 7805:
            case 7807:
                newch = "v";
                break;
            case 373:
            case 7809:
            case 7811:
            case 7813:
            case 7815:
            case 7817:
            case 7832:
                newch = "w";
                break;
            case 7819:
            case 7821:
                newch = "x";
                break;
            case 253:
            case 255:
            case 375:
            case 563:
            case 7833:
            case 7923:
            case 7925:
            case 7927:
            case 7929:
                newch = "y";
                break;
            case 378:
            case 380:
            case 382:
            case 7825:
            case 7827:
            case 7829:
                newch = "z";
                break;
            case 894:
                newch = ";";
                break;
            case 8175:
                newch = "`";
                break;
            case 8800:
                newch = "=";
                break;
            case 8814:
                newch = "<";
                break;
            case 8815:
                newch = ">";
                break;
        }
        if (newch) {
            chars.push(newch.charCodeAt(0));
        } else if (ch < 128) {
            chars.push(ch);
        }
    }
    return String.fromCharCode(...chars);
}
function asHtmlId(text) {
    return pandocAutoIdentifier(text, false);
}
function getDecodedAttribute(element, attrib) {
    const value = element.getAttribute(attrib);
    if (value) {
        return decodeURI(value);
    } else {
        return value;
    }
}
function findParent(el, match) {
    let targetEl = el;
    do {
        if (targetEl.parentElement) {
            if (match(targetEl.parentElement)) {
                return targetEl.parentElement;
            } else {
                targetEl = targetEl.parentElement;
            }
        } else {
            return undefined;
        }
    }while (targetEl !== null && targetEl.nodeType === 1)
    return undefined;
}
const kHtmlResourceTags = {
    "a": [
        "href"
    ],
    "img": [
        "src",
        "data-src"
    ],
    "link": [
        "href"
    ],
    "script": [
        "src"
    ],
    "embed": [
        "src"
    ],
    "iframe": [
        "src"
    ],
    "section": [
        "data-background-image",
        "data-background-video"
    ]
};
function discoverResourceRefs(doc) {
    const refs = [];
    Object.keys(kHtmlResourceTags).forEach((tag)=>{
        for (const attrib of kHtmlResourceTags[tag]){
            refs.push(...resolveResourceTag(doc, tag, attrib));
        }
    });
    const styles = doc.querySelectorAll("style");
    for(let i61 = 0; i61 < styles.length; i61++){
        const style = styles[i61];
        if (style.innerHTML) {
            refs.push(...cssFileRefs(style.innerHTML));
        }
    }
    return Promise.resolve(refs);
}
function processFileResourceRefs(doc, tag, attrib, onRef) {
    const tags = doc.querySelectorAll(tag);
    for(let i62 = 0; i62 < tags.length; i62++){
        const tag = tags[i62];
        const href = getDecodedAttribute(tag, attrib);
        if (href !== null && href.length > 0 && isFileRef(href)) {
            onRef(tag, href);
        }
    }
}
function resolveResourceTag(doc, tag, attrib) {
    const refs = [];
    processFileResourceRefs(doc, tag, attrib, (_tag, ref5)=>refs.push(ref5)
    );
    return refs;
}
function isDisplayData(output) {
    return [
        "display_data",
        "execute_result"
    ].includes(output.output_type);
}
function isCaptionableData(output) {
    if (isDisplayData(output)) {
        const displayData = output;
        return !displayData.noCaption;
    } else {
        return false;
    }
}
function displayDataMimeType(output, options) {
    const displayPriority = [
        kTextMarkdown,
        kImageSvg,
        kImagePng,
        kImageJpeg, 
    ];
    if (options.toHtml) {
        const htmlFormats = [
            kApplicationJupyterWidgetState,
            kApplicationJupyterWidgetView,
            kApplicationJavascript,
            kTextHtml, 
        ];
        if (options.toMarkdown) {
            displayPriority.unshift(...htmlFormats);
        } else {
            displayPriority.push(...htmlFormats);
        }
        displayPriority.unshift(kApplicationJupyterWidgetState, kApplicationJupyterWidgetView, kApplicationJavascript, kTextHtml);
    } else if (options.toLatex) {
        displayPriority.push(kTextLatex, kApplicationPdf);
    } else if (options.toMarkdown) {
        displayPriority.push(kTextHtml);
    }
    displayPriority.push(kTextPlain);
    const availDisplay = Object.keys(output.data);
    for (const display of displayPriority){
        if (availDisplay.includes(display)) {
            return display;
        }
    }
    return null;
}
function displayDataIsImage(mimeType) {
    return [
        kImagePng,
        kImageJpeg,
        kImageSvg,
        kApplicationPdf
    ].includes(mimeType);
}
function displayDataIsTextPlain(mimeType) {
    return [
        kTextPlain
    ].includes(mimeType);
}
function displayDataIsMarkdown(mimeType) {
    return [
        kTextMarkdown
    ].includes(mimeType);
}
function displayDataIsLatex(mimeType) {
    return [
        kTextLatex
    ].includes(mimeType);
}
function displayDataIsHtml(mimeType) {
    return [
        kTextHtml
    ].includes(mimeType);
}
function displayDataIsJson(mimeType) {
    return [
        kApplicationJupyterWidgetState,
        kApplicationJupyterWidgetView
    ].includes(mimeType);
}
function displayDataIsJavascript(mimeType) {
    return [
        kApplicationJavascript
    ].includes(mimeType);
}
function cellLabel(cell) {
    const label = asHtmlId(cell.options[kCellLabel] || cell.metadata[kCellName] || "");
    if (label && !label.startsWith("#")) {
        return "#" + label;
    } else {
        return label;
    }
}
function cellLabelValidator() {
    const cellLabels = new Set();
    return function(cell) {
        const label = cellLabel(cell);
        if (label) {
            if (cellLabels.has(label)) {
                throw new Error("Cell label names must be unique (found duplicate '" + label + "')");
            } else {
                cellLabels.add(label);
            }
        }
    };
}
function shouldLabelCellContainer(cell, outputs, options) {
    if (!outputs) {
        return true;
    }
    if (!includeOutput(cell, options)) {
        return true;
    }
    const displayDataOutputs = outputs.filter(isDisplayData);
    if (displayDataOutputs.length === 0) {
        return true;
    }
    if (displayDataOutputs.length > 1 && !Array.isArray(cell.options[kCellFigCap])) {
        return true;
    }
    if (hasTableLabel(cell.options)) {
        return true;
    }
    return false;
}
function hasTableLabel(options) {
    return typeof options[kCellLabel] === "string" && options[kCellLabel]?.startsWith("tbl-");
}
function shouldLabelOutputContainer(output, cellOptions, options) {
    if (isDisplayData(output)) {
        if (hasTableLabel(cellOptions)) {
            return false;
        }
        if (!isCaptionableData(output)) {
            return false;
        }
        const mimeType = displayDataMimeType(output, options);
        if (mimeType) {
            if (displayDataIsImage(mimeType)) {
                return false;
            }
        }
        return true;
    } else {
        return false;
    }
}
function resolveCaptions(cell) {
    if (Array.isArray(cell.options[kCellFigCap])) {
        const figCap = cell.options[kCellFigCap];
        if (cell.outputs && cell.outputs.filter(isCaptionableData).length > 0) {
            return {
                cellCaption: undefined,
                outputCaptions: figCap
            };
        } else {
            return {
                cellCaption: undefined,
                outputCaptions: []
            };
        }
    } else if (cell.options[kCellFigCap]) {
        if (cell.options[kCellFigSubCap] !== undefined) {
            let subCap = cell.options[kCellFigSubCap];
            if (subCap === true) {
                subCap = [
                    ""
                ];
            }
            if (!Array.isArray(subCap)) {
                subCap = [
                    String(subCap)
                ];
            }
            return {
                cellCaption: cell.options[kCellFigCap],
                outputCaptions: subCap
            };
        } else {
            return {
                cellCaption: undefined,
                outputCaptions: [
                    cell.options[kCellFigCap]
                ]
            };
        }
    } else {
        return {
            cellCaption: undefined,
            outputCaptions: []
        };
    }
}
function extractJupyterWidgetDependencies(nb) {
    const jsWidgets = haveOutputType(nb, [
        kApplicationJavascript,
        kTextHtml
    ]);
    const jupyterWidgets = haveOutputType(nb, [
        kApplicationJupyterWidgetView
    ]);
    const htmlLibraries = [];
    nb.cells.forEach((cell)=>{
        if (cell.cell_type === "code") {
            cell.outputs = cell.outputs?.filter((output)=>{
                if (isDisplayData(output)) {
                    const displayOutput = output;
                    const html = displayOutput.data[kTextHtml];
                    const htmlText = Array.isArray(html) ? html.join("") : html;
                    if (html && isWidgetIncludeHtml(htmlText)) {
                        htmlLibraries.push(htmlText);
                        return false;
                    }
                }
                return true;
            });
        }
    });
    return {
        jsWidgets,
        jupyterWidgets,
        htmlLibraries,
        widgetsState: nb.metadata.widgets?.[kApplicationJupyterWidgetState]
    };
}
function includesForJupyterWidgetDependencies(dependencies, tempDir1) {
    let haveJavascriptWidgets = false;
    let haveJupyterWidgets = false;
    const htmlLibraries = [];
    let widgetsState;
    for (const dependency of dependencies){
        haveJavascriptWidgets = haveJavascriptWidgets || dependency.jsWidgets;
        haveJupyterWidgets = haveJupyterWidgets || dependency.jupyterWidgets;
        for (const htmlLib of dependency.htmlLibraries){
            if (!htmlLibraries.includes(htmlLib)) {
                htmlLibraries.push(htmlLib);
            }
        }
        if (dependency.widgetsState) {
            if (!widgetsState) {
                widgetsState = dependency.widgetsState;
            } else {
                widgetsState = mergeConfigs(widgetsState, dependency.widgetsState);
            }
        }
    }
    const head = [];
    if (haveJavascriptWidgets || haveJupyterWidgets) {
        head.push('<script src="https://cdnjs.cloudflare.com/ajax/libs/require.js/2.3.6/require.min.js" integrity="sha512-c3Nl8+7g4LMSTdrm621y7kf9v3SDPnhxLNhcjFJbKECVnmZHTdo+IRO05sNLTH/D3vA6u1X32ehoLC7WFVdheg==" crossorigin="anonymous"></script>');
        head.push('<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.5.1/jquery.min.js" integrity="sha512-bLT0Qm9VnAYZDflyKcBaQ2gg0hSYNQrJ8RilYldYQ1FxQYoCLtUjuuRuZo+fjqhx/qtq/1itJ0C2ejDxltZVFg==" crossorigin="anonymous"></script>');
        head.push("<script type=\"application/javascript\">define('jquery', [],function() {return window.jQuery;})</script>");
    }
    head.push(...htmlLibraries);
    if (haveJupyterWidgets) {
        head.push('<script src="https://unpkg.com/@jupyter-widgets/html-manager@*/dist/embed-amd.js" crossorigin="anonymous"></script>');
    }
    const afterBody = [];
    if (haveJupyterWidgets && widgetsState) {
        afterBody.push(`<script type=${kApplicationJupyterWidgetState}>`);
        afterBody.push(JSON.stringify(widgetsState));
        afterBody.push("</script>");
    }
    const widgetTempFile = (lines1)=>{
        const tempFile = Deno.makeTempFileSync({
            dir: tempDir1,
            prefix: "jupyter-widgets-",
            suffix: ".html"
        });
        Deno.writeTextFileSync(tempFile, lines1.join("\n") + "\n");
        return tempFile;
    };
    const result = {
        inHeader: "",
        afterBody: ""
    };
    if (head.length > 0) {
        result.inHeader = widgetTempFile(head);
    }
    if (afterBody.length > 0) {
        result.afterBody = widgetTempFile(afterBody);
    }
    return result;
}
function haveOutputType(nb, mimeTypes) {
    return nb.cells.some((cell)=>{
        if (cell.cell_type === "code" && cell.outputs) {
            return cell.outputs.some((output)=>{
                if (isDisplayData(output)) {
                    const outputTypes = Object.keys(output.data);
                    return outputTypes.some((type)=>mimeTypes.includes(type)
                    );
                } else {
                    return false;
                }
            });
        } else {
            return false;
        }
    });
}
function isWidgetIncludeHtml(html) {
    return isPlotlyLibrary(html);
}
function isPlotlyLibrary(html) {
    return /^\s*<script type="text\/javascript">/.test(html) && (/require\.undef\(["']plotly["']\)/.test(html) || /define\('plotly'/.test(html));
}
function removeAndPreserveHtml(nb) {
    const htmlPreserve = {};
    nb.cells.forEach((cell)=>{
        if (cell.cell_type === "code") {
            cell.outputs?.forEach((output)=>{
                if (isDisplayData(output)) {
                    const displayOutput = output;
                    const html = displayOutput.data[kTextHtml];
                    const htmlText = Array.isArray(html) ? html.join("") : html;
                    if (html && isPreservedHtml(htmlText)) {
                        const key = "preserve" + generate().replaceAll("-", "");
                        htmlPreserve[key] = htmlText;
                        displayOutput.data[kTextMarkdown] = [
                            key
                        ];
                        displayOutput.noCaption = true;
                        delete displayOutput.data[kTextHtml];
                    }
                }
            });
        }
    });
    if (Object.keys(htmlPreserve).length > 0) {
        return htmlPreserve;
    } else {
        return undefined;
    }
}
function restorePreservedHtml(html, preserve) {
    if (preserve) {
        Object.keys(preserve).forEach((key)=>{
            const keyLoc = html.indexOf(key);
            html = html.slice(0, keyLoc) + preserve[key] + html.slice(keyLoc + key.length);
        });
    }
    return html;
}
function isPreservedHtml(_html) {
    return false;
}
function isWindows1() {
    return Deno.build.os === "windows";
}
function isMac() {
    return Deno.build.os === "darwin";
}
function isRStudio() {
    return !!Deno.env.get("RSTUDIO");
}
function isRStudioServer() {
    return isRStudio() && Deno.env.get("RSTUDIO_PROGRAM_MODE") === "server";
}
function isJupyterHubServer() {
    return jupyterHubUser() !== undefined;
}
function jupyterHubHttpReferrer() {
    return Deno.env.get("JUPYTERHUB_HTTP_REFERER");
}
function jupyterHubUser() {
    return Deno.env.get("JUPYTERHUB_USER");
}
function isInteractiveTerminal() {
    return Deno.isatty(Deno.stderr.rid);
}
function isInteractiveSession() {
    return isRStudio() || isInteractiveTerminal();
}
let cachedJupyterCaps;
async function jupyterCapabilities() {
    if (!cachedJupyterCaps) {
        cachedJupyterCaps = await getQuartoJupyterCapabilities();
        if (cachedJupyterCaps) {
            return cachedJupyterCaps;
        }
        if (isWindows1() && pyPython()) {
            cachedJupyterCaps = await getPyLauncherJupyterCapabilities();
        }
        if (!cachedJupyterCaps) {
            cachedJupyterCaps = await getJupyterCapabilities([
                "python"
            ]);
            if (!cachedJupyterCaps?.conda) {
                const caps = isWindows1() ? await getPyLauncherJupyterCapabilities() : await getJupyterCapabilities([
                    "python3"
                ]);
                if (caps) {
                    cachedJupyterCaps = caps;
                }
            }
        }
    }
    return cachedJupyterCaps;
}
function getQuartoJupyterCapabilities() {
    const quartoJupyter = Deno.env.get("QUARTO_PYTHON");
    if (quartoJupyter) {
        if (existsSync(quartoJupyter)) {
            let quartoJupyterBin = quartoJupyter;
            if (Deno.statSync(quartoJupyter).isDirectory) {
                const bin1 = [
                    "python3",
                    "python",
                    "python3.exe",
                    "python.exe"
                ].find((bin)=>{
                    return existsSync(join3(quartoJupyter, bin));
                });
                if (bin1) {
                    quartoJupyterBin = join3(quartoJupyter, bin1);
                } else {
                    quartoJupyterBin = undefined;
                }
            }
            if (quartoJupyterBin) {
                return getJupyterCapabilities([
                    quartoJupyterBin
                ]);
            }
        }
        warning("Specified QUARTO_JUPYTER '" + quartoJupyter + "' does not exist.");
        return undefined;
    } else {
        return undefined;
    }
}
async function jupyterCapabilitiesNoConda() {
    const caps = await getQuartoJupyterCapabilities();
    if (caps && !caps.conda) {
        return caps;
    }
    if (isWindows1()) {
        return await getPyLauncherJupyterCapabilities();
    } else {
        const caps = await getJupyterCapabilities([
            "python3"
        ]);
        if (caps) {
            if (caps?.conda) {
                return await getJupyterCapabilities([
                    "/usr/local/bin/python3"
                ]) || await getJupyterCapabilities([
                    "/usr/bin/python3"
                ]);
            } else {
                return caps;
            }
        } else {
            return undefined;
        }
    }
}
function isEnvDir(dir) {
    return existsSync(join3(dir, "pyvenv.cfg")) || existsSync(join3(dir, "conda-meta"));
}
function pyPython() {
    return Deno.env.get("PY_PYTHON");
}
function getPyLauncherJupyterCapabilities() {
    return getJupyterCapabilities([
        "py"
    ], true);
}
async function getJupyterCapabilities(cmd, pyLauncher = false) {
    try {
        const result = await execProcess({
            cmd: [
                ...cmd,
                resourcePath("capabilities/jupyter.py"), 
            ],
            stdout: "piped",
            stderr: "piped"
        });
        if (result.success && result.stdout) {
            const caps = readYamlFromString(result.stdout);
            if (caps.versionMajor >= 3) {
                caps.pyLauncher = pyLauncher;
                return caps;
            } else {
                return undefined;
            }
        } else {
            return undefined;
        }
    } catch  {
        return undefined;
    }
}
async function pythonExec(binaryOnly = false) {
    const caps = await jupyterCapabilities();
    if (caps?.pyLauncher) {
        return [
            "py"
        ];
    } else if (isWindows1()) {
        return [
            binaryOnly ? "python" : caps?.executable || "python"
        ];
    } else {
        return [
            binaryOnly ? "python3" : caps?.executable || "python3"
        ];
    }
}
async function jupyterExec() {
    return [
        ...await pythonExec(),
        "-m",
        "jupyter", 
    ];
}
function isJupyterKernelspec(x) {
    if (x && typeof x === "object") {
        return typeof x.name === "string" && typeof x.language === "string" && typeof x.display_name === "string";
    } else {
        return false;
    }
}
async function jupyterKernelspec(name96) {
    const kernelspecs = await jupyterKernelspecs();
    return kernelspecs.get(name96);
}
async function jupyterKernelspecs() {
    try {
        const result = await execProcess({
            cmd: [
                ...await jupyterExec(),
                "--paths",
                "--json"
            ],
            stdout: "piped",
            stderr: "piped"
        });
        if (result.success) {
            const kernelmap = new Map();
            const dataPaths = JSON.parse(result.stdout).data;
            for (const path78 of dataPaths){
                if (!existsSync(path78)) {
                    continue;
                }
                const kernels = join3(path78, "kernels");
                if (!existsSync(kernels)) {
                    continue;
                }
                for (const walk3 of walkSync(kernels, {
                    maxDepth: 1
                })){
                    if (walk3.path === kernels || !walk3.isDirectory) {
                        continue;
                    }
                    const kernelConfig = join3(walk3.path, "kernel.json");
                    if (existsSync(kernelConfig)) {
                        const config = JSON.parse(Deno.readTextFileSync(kernelConfig));
                        const name97 = basename2(walk3.path);
                        kernelmap.set(name97, {
                            name: name97,
                            language: config.language,
                            display_name: config.display_name
                        });
                    }
                }
            }
            if (kernelmap.size > 0) {
                return kernelmap;
            } else {
                return kDefaultKernelspecs;
            }
        } else {
            return kDefaultKernelspecs;
        }
    } catch (e) {
        debug("Error reading kernelspecs: " + e.message);
        return kDefaultKernelspecs;
    }
}
const kDefaultKernelspecs = new Map();
kDefaultKernelspecs.set("python3", {
    "display_name": "Python 3",
    "language": "python",
    "name": "python3"
});
function inputFilesDir(input) {
    const [_, stem] = dirAndStem(input);
    return stem + "_files";
}
function figuresDir(pandocTo) {
    if (pandocTo === "html4") {
        pandocTo = "html";
    }
    pandocTo = (pandocTo || "html").replace(/[\+\-].*$/, "");
    return "figure-" + pandocTo;
}
const kJupyterNotebookExtensions = [
    ".ipynb", 
];
function isJupyterNotebook(file) {
    return kJupyterNotebookExtensions.includes(extname2(file).toLowerCase());
}
const kJupyterCellInternalOptionKeys = [
    kEval,
    kEcho,
    kWarning,
    kOutput,
    kInclude,
    kCellLabel,
    kCellClasses,
    kCellPanel,
    kCellColumn,
    kCellFigCap,
    kCellFigSubCap,
    kCellFigScap,
    kFigCapLoc,
    kTblCapLoc,
    kCapLoc,
    kCellFigColumn,
    kCellTblColumn,
    kCellFigLink,
    kCellFigAlign,
    kCellFigAlt,
    kCellFigEnv,
    kCellFigPos,
    kCellLstLabel,
    kCellLstCap,
    kCellOutWidth,
    kCellOutHeight,
    kCellMdIndent,
    kCodeFold,
    kCodeLineNumbers,
    kCodeSummary,
    kCodeOverflow, 
];
const kJupyterCellOptionKeys = kJupyterCellInternalOptionKeys.concat([
    kLayoutAlign,
    kLayoutVAlign,
    kLayoutNcol,
    kLayoutNrow,
    kLayout,
    kTblColwidths, 
]);
const kJupyterCellStandardMetadataKeys = [
    kCellCollapsed,
    kCellAutoscroll,
    kCellDeletable,
    kCellFormat,
    kCellName, 
];
const kJupyterCellThirdPartyMetadataKeys = [
    kCellId,
    kCellColab,
    kCellColabType,
    kCellColbOutputId,
    kCellLinesToNext, 
];
async function quartoMdToJupyter(input, includeIds) {
    const [kernelspec, metadata] = await jupyterKernelspecFromFile(input);
    const nb = {
        cells: [],
        metadata: {
            kernelspec,
            ...metadata
        },
        nbformat: 4,
        nbformat_minor: includeIds ? 5 : 4
    };
    const yamlRegEx = /^---\s*$/;
    const startCodeCellRegEx = new RegExp("^(\\s*)```+\\s*\\{" + kernelspec.language.toLowerCase() + "( *[ ,].*)?\\}\\s*$");
    const startCodeRegEx = /^(\s*)```/;
    const endCodeRegEx = (indent = "")=>{
        return new RegExp("^" + indent + "```\\s*$");
    };
    const inputContent = Deno.readTextFileSync(input);
    let codeIndent = "";
    const lineBuffer = [];
    const flushLineBuffer = (cell_type, frontMatter)=>{
        if (lineBuffer.length) {
            if (lineBuffer[0] === "") {
                lineBuffer.splice(0, 1);
            }
            if (lineBuffer[lineBuffer.length - 1] === "") {
                lineBuffer.splice(lineBuffer.length - 1, 1);
            }
            if (cell_type !== "code") {
                codeIndent = "";
            }
            const cell = {
                cell_type,
                metadata: codeIndent.length > 0 ? {
                    [kCellMdIndent]: codeIndent
                } : {},
                source: lineBuffer.map((line, index)=>{
                    if (codeIndent.length > 0) {
                        line = line.replace(codeIndent, "");
                    }
                    return line + (index < lineBuffer.length - 1 ? "\n" : "");
                })
            };
            if (includeIds) {
                cell.id = shortUuid();
            }
            if (cell_type === "raw" && frontMatter) {
                const yaml = readYamlFromMarkdown(cell.source.join(""));
                if (yaml.jupyter) {
                    delete yaml.jupyter;
                    if (Object.keys(yaml).length > 0) {
                        const yamlFrontMatter = mdTrimEmptyLines(lines(stringify(yaml, {
                            indent: 2,
                            sortKeys: false,
                            skipInvalid: true
                        })));
                        cell.source = [
                            "---\n",
                            ...yamlFrontMatter.map((line)=>line + "\n"
                            ),
                            "---", 
                        ];
                    } else {
                        cell.source = [];
                    }
                }
            } else if (cell_type === "code") {
                const { yaml , source  } = partitionCellOptions(kernelspec.language.toLowerCase(), cell.source);
                if (yaml) {
                    if (includeIds && yaml[kCellLabel] && !yaml[kCellId]) {
                        yaml[kCellId] = jupyterAutoIdentifier(String(yaml[kCellLabel]));
                    }
                    const yamlKeys = Object.keys(yaml);
                    yamlKeys.forEach((key)=>{
                        if (key === kCellId) {
                            if (includeIds) {
                                cell.id = String(yaml[key]);
                            }
                            delete yaml[key];
                        } else {
                            if (!kJupyterCellOptionKeys.includes(key)) {
                                cell.metadata[key] = yaml[key];
                                delete yaml[key];
                            }
                        }
                    });
                    if (Object.keys(yaml).length < yamlKeys.length) {
                        const yamlOutput = jupyterCellOptionsAsComment(kernelspec.language.toLowerCase(), yaml);
                        cell.source = yamlOutput.concat(source);
                    }
                }
                cell.execution_count = null;
                cell.outputs = [];
            }
            cell.source = mdTrimEmptyLines(cell.source);
            if (cell.source.length > 0) {
                nb.cells.push(cell);
            }
            lineBuffer.splice(0, lineBuffer.length);
        }
    };
    let parsedFrontMatter = false, inYaml = false, inCodeCell = false, inCode = false;
    for (const line1 of lines(inputContent)){
        if (yamlRegEx.test(line1) && !inCodeCell && !inCode) {
            if (inYaml) {
                lineBuffer.push(line1);
                flushLineBuffer("raw", !parsedFrontMatter);
                parsedFrontMatter = true;
                inYaml = false;
            } else {
                flushLineBuffer("markdown");
                lineBuffer.push(line1);
                inYaml = true;
            }
        } else if (startCodeCellRegEx.test(line1)) {
            flushLineBuffer("markdown");
            inCodeCell = true;
            codeIndent = line1.match(startCodeCellRegEx)[1];
        } else if (endCodeRegEx(codeIndent).test(line1)) {
            if (inCodeCell) {
                inCodeCell = false;
                flushLineBuffer("code");
                codeIndent = "";
            } else {
                inCode = !inCode;
                codeIndent = "";
                lineBuffer.push(line1);
            }
        } else if (startCodeRegEx.test(line1)) {
            codeIndent = line1.match(startCodeRegEx)[1];
            inCode = true;
            lineBuffer.push(line1);
        } else {
            lineBuffer.push(line1);
        }
    }
    flushLineBuffer("markdown");
    return nb;
}
async function jupyterKernelspecFromFile(file) {
    const yaml = await readYamlFromMarkdownFile(file);
    const yamlJupyter = yaml.jupyter;
    if (!yamlJupyter) {
        const languages = languagesInMarkdownFile(file);
        languages.add("python");
        const kernelspecs = await jupyterKernelspecs();
        for (const language of languages){
            for (const kernelspec of kernelspecs.values()){
                if (kernelspec.language.toLowerCase() === language) {
                    return [
                        kernelspec,
                        {}
                    ];
                }
            }
        }
    }
    if (typeof yamlJupyter === "string") {
        const kernel = yamlJupyter;
        const kernelspec = await jupyterKernelspec(kernel);
        if (kernelspec) {
            return [
                kernelspec,
                {}
            ];
        } else {
            return Promise.reject(new Error("Jupyter kernel '" + kernel + "' not found."));
        }
    } else if (typeof yamlJupyter === "object") {
        const jupyter = {
            ...yamlJupyter
        };
        if (isJupyterKernelspec(jupyter.kernelspec)) {
            const kernelspec = jupyter.kernelspec;
            delete jupyter.kernelspec;
            return [
                kernelspec,
                jupyter
            ];
        } else if (typeof jupyter.kernel === "string") {
            const kernelspec = await jupyterKernelspec(jupyter.kernel);
            if (kernelspec) {
                delete jupyter.kernel;
                return [
                    kernelspec,
                    jupyter
                ];
            } else {
                return Promise.reject(new Error("Jupyter kernel '" + jupyter.kernel + "' not found."));
            }
        } else {
            return Promise.reject(new Error("Invalid Jupyter kernelspec (must include name, language, & display_name)"));
        }
    } else {
        return Promise.reject(new Error("Invalid jupyter YAML metadata found in file (must be string or object)"));
    }
}
function jupyterFromFile(input) {
    const nbContents = Deno.readTextFileSync(input);
    const nbJSON = JSON.parse(nbContents);
    const nb = nbJSON;
    if (!nb.metadata.kernelspec.language) {
        nb.metadata.kernelspec.language = nbJSON.metadata.language_info?.name;
    }
    if (!nb.metadata.kernelspec.language) {
        throw new Error("No langage set for Jupyter notebook " + input);
    }
    if (!nb.cells) {
        throw new Error("No cells available in Jupyter notebook " + input);
    }
    return nb;
}
function languagesInMarkdownFile(file) {
    return languagesInMarkdown(Deno.readTextFileSync(file));
}
function languagesInMarkdown(markdown) {
    const languages = new Set();
    const kChunkRegex = /^[\t >]*```+\s*\{([a-zA-Z0-9_]+)( *[ ,].*)?\}\s*$/gm;
    kChunkRegex.lastIndex = 0;
    let match = kChunkRegex.exec(markdown);
    while(match){
        const language = match[1].toLowerCase();
        if (!languages.has(language)) {
            languages.add(language);
        }
        match = kChunkRegex.exec(markdown);
    }
    kChunkRegex.lastIndex = 0;
    return languages;
}
function jupyterAutoIdentifier(label) {
    label = pandocAsciify(label);
    label = label.replace(/\s/g, "-").replace(/[^a-zA-Z0-9-_]/g, "").replace(/^[^A-Za-z]+/, "");
    if (label.length > 0) {
        return label.slice(0, 64);
    } else {
        return shortUuid();
    }
}
function jupyterAssets(input, to) {
    input = Deno.realPathSync(input);
    const files_dir = join3(dirname2(input), inputFilesDir(input));
    const figures_dir = join3(files_dir, figuresDir(to));
    ensureDirSync(figures_dir);
    let supporting_dir = files_dir;
    for (const walk4 of walkSync(join3(files_dir), {
        maxDepth: 1
    })){
        if (walk4.path !== files_dir && walk4.path !== figures_dir) {
            supporting_dir = figures_dir;
            break;
        }
    }
    const base_dir = dirname2(input);
    return {
        base_dir,
        files_dir: relative2(base_dir, files_dir),
        figures_dir: relative2(base_dir, figures_dir),
        supporting_dir: relative2(base_dir, supporting_dir)
    };
}
function jupyterToMarkdown(nb, options) {
    const isHtml = options.toHtml && !options.toIpynb;
    const dependencies = isHtml ? extractJupyterWidgetDependencies(nb) : undefined;
    const htmlPreserve = isHtml ? removeAndPreserveHtml(nb) : undefined;
    const md = [];
    const validateCellLabel = cellLabelValidator();
    let codeCellIndex = 0;
    for(let i63 = 0; i63 < nb.cells.length; i63++){
        const cell = jupyterCellWithOptions(nb.metadata.kernelspec.language.toLowerCase(), nb.cells[i63]);
        validateCellLabel(cell);
        const slideType = options.toPresentation ? cell.metadata[kCellSlideshow]?.[kCellSlideshowSlideType] : undefined;
        if (slideType) {
            if (slideType === "skip") {
                continue;
            } else if (slideType == "slide" || slideType === "subslide") {
                md.push("\n---\n\n");
            } else if (slideType == "fragment") {
                md.push("\n. . .\n\n");
            } else if (slideType == "notes") {
                md.push("\n:::::::::: notes\n\n");
            }
        }
        switch(cell.cell_type){
            case "markdown":
                md.push(...mdFromContentCell(cell));
                break;
            case "raw":
                md.push(...mdFromRawCell(cell));
                break;
            case "code":
                md.push(...mdFromCodeCell(cell, ++codeCellIndex, options));
                break;
            default:
                throw new Error("Unexpected cell type " + cell.cell_type);
        }
        if (slideType === "notes") {
            md.push("\n::::::::::\n");
        }
        md.push("\n");
    }
    if (options.toIpynb) {
        md.push("---\n");
        const jupyterMetadata = {
            jupyter: {
                ...nb.metadata
            }
        };
        const yamlText = stringify(jupyterMetadata, {
            indent: 2,
            sortKeys: false,
            skipInvalid: true
        });
        md.push(yamlText);
        md.push("---\n");
    }
    return {
        markdown: md.join(""),
        dependencies,
        htmlPreserve
    };
}
function jupyterCellWithOptions(language, cell) {
    const { yaml , optionsSource , source  } = partitionCellOptions(language, cell.source);
    const metadataOptions = kJupyterCellOptionKeys.reduce((options, key)=>{
        if (cell.metadata[key]) {
            options[key] = cell.metadata[key];
        }
        return options;
    }, {});
    const options1 = {
        ...metadataOptions,
        ...yaml
    };
    [
        kLayout,
        kTblColwidths
    ].forEach((option)=>{
        if (options1[option] && typeof options1[option] !== "string") {
            options1[option] = JSON.stringify(options1[option]);
        }
    });
    return {
        ...cell,
        source,
        optionsSource,
        options: options1
    };
}
function jupyterCellOptionsAsComment(language, options, stringifyOptions) {
    if (Object.keys(options).length > 0) {
        const cellYaml = stringify(options, {
            indent: 2,
            sortKeys: false,
            skipInvalid: true,
            ...stringifyOptions
        });
        const commentChars = langCommentChars1(language);
        const yamlOutput = mdTrimEmptyLines(lines(cellYaml)).map((line)=>{
            line = optionCommentPrefix1(commentChars[0]) + line + optionCommentSuffix(commentChars[1]);
            return line + "\n";
        });
        return yamlOutput;
    } else {
        return [];
    }
}
function mdFromContentCell(cell) {
    return mdEnsureTrailingNewline(cell.source);
}
function mdFromRawCell(cell) {
    const mimeType = cell.metadata?.[kCellRawMimeType];
    if (mimeType) {
        switch(mimeType){
            case kTextHtml:
                return mdHtmlOutput(cell.source);
            case kTextLatex:
                return mdLatexOutput(cell.source);
            case kRestructuredText:
                return mdFormatOutput("rst", cell.source);
            case kApplicationRtf:
                return mdFormatOutput("rtf", cell.source);
            case kApplicationJavascript:
                return mdScriptOutput(mimeType, cell.source);
        }
    }
    return mdFromContentCell(cell);
}
function mdEnsureTrailingNewline(source) {
    if (source.length > 0 && !source[source.length - 1].endsWith("\n")) {
        return source.slice(0, source.length - 1).concat([
            source[source.length - 1] + "\n"
        ]);
    } else {
        return source;
    }
}
function optionCommentPrefix1(comment) {
    return comment + "| ";
}
function optionCommentSuffix(comment) {
    if (comment) {
        return " " + comment;
    } else {
        return "";
    }
}
function langCommentChars1(lang) {
    const chars = kLangCommentChars1[lang] || "#";
    if (!Array.isArray(chars)) {
        return [
            chars
        ];
    } else {
        return chars;
    }
}
const kLangCommentChars1 = {
    r: "#",
    python: "#",
    julia: "#",
    scala: "//",
    matlab: "%",
    csharp: "//",
    fsharp: "//",
    c: [
        "/*",
        "*/"
    ],
    css: [
        "/*",
        "*/"
    ],
    sas: [
        "*",
        ";"
    ],
    powershell: "#",
    bash: "#",
    sql: "--",
    mysql: "--",
    psql: "--",
    lua: "--",
    cpp: "//",
    cc: "//",
    stan: "#",
    octave: "#",
    fortran: "!",
    fortran95: "!",
    awk: "#",
    gawk: "#",
    stata: "*",
    java: "//",
    groovy: "//",
    sed: "#",
    perl: "#",
    ruby: "#",
    tikz: "%",
    js: "//",
    d3: "//",
    node: "//",
    sass: "//",
    coffee: "#",
    go: "//",
    asy: "//",
    haskell: "--",
    dot: "//"
};
function mdFromCodeCell(cell, cellIndex, options) {
    if (!includeCell(cell, options)) {
        return [];
    }
    const outputs = (cell.outputs || []).filter((output)=>{
        if (output.output_type === "stream" && output.name === "stderr" && !includeWarnings(cell, options)) {
            return false;
        }
        if (output.output_type === "execute_result") {
            const textPlain = output.data?.[kTextPlain];
            if (textPlain && textPlain[0].startsWith("[<matplotlib")) {
                return false;
            }
        }
        return true;
    });
    if (!cell.source.length && !outputs.length) {
        return [];
    }
    const asis = cell.options[kOutput] === "asis" || options.execute[kOutput] === "asis" && cell.options[kOutput] === undefined;
    const md = [];
    const divMd = [
        `::: {`
    ];
    const kCellOptionsFilter = kJupyterCellInternalOptionKeys.concat(kJupyterCellStandardMetadataKeys, kJupyterCellThirdPartyMetadataKeys);
    const label = cellLabel(cell);
    const labelCellContainer = shouldLabelCellContainer(cell, outputs, options);
    if (label && labelCellContainer) {
        divMd.push(`${label} `);
    }
    const { cellCaption , outputCaptions  } = resolveCaptions(cell);
    divMd.push(`.cell `);
    if (hideCell(cell, options)) {
        divMd.push(`.hidden `);
    }
    if (cell.options[kCellClasses] || cell.options[kCellPanel]) {
        const cellClasses = cell.options[kCellClasses] || new Array();
        const classes = Array.isArray(cellClasses) ? cellClasses : [
            cellClasses
        ];
        if (typeof cell.options[kCellPanel] === "string") {
            classes.push(`panel-${cell.options[kCellPanel]}`);
        }
        if (typeof cell.options[kCellColumn] === "string") {
            classes.push(`column-${cell.options[kCellColumn]}`);
        }
        if (typeof cell.options[kCellFigColumn] === "string") {
            classes.push(`fig-column-${cell.options[kCellFigColumn]}`);
        }
        if (typeof cell.options[kCellTblColumn] === "string") {
            classes.push(`tbl-column-${cell.options[kCellTblColumn]}`);
        }
        if (typeof cell.options[kCapLoc] === "string") {
            classes.push(`caption-${cell.options[kFigCapLoc]}`);
        }
        if (typeof cell.options[kFigCapLoc] === "string") {
            classes.push(`fig-cap-location-${cell.options[kFigCapLoc]}`);
        }
        if (typeof cell.options[kTblCapLoc] === "string") {
            classes.push(`tbl-cap-location-${cell.options[kTblCapLoc]}`);
        }
        const classText = classes.map((clz)=>{
            clz = toString1(clz);
            return clz.startsWith(".") ? clz : "." + clz;
        }).join(" ");
        divMd.push(classText + " ");
    }
    const cellOptions = {
        ...cell.metadata,
        ...cell.options
    };
    for (const key of Object.keys(cellOptions)){
        if (!kCellOptionsFilter.includes(key.toLowerCase())) {
            let value = cellOptions[key];
            if (value) {
                if (typeof value !== "string") {
                    value = JSON.stringify(value);
                }
                value = value.replaceAll("'", `\\'`);
                divMd.push(`${key}='${value}' `);
            }
        }
    }
    if (typeof cell.execution_count === "number") {
        divMd.push(`execution_count=${cell.execution_count} `);
    }
    const divBeginMd = divMd.join("").replace(/ $/, "").concat("}\n");
    if (includeCode(cell, options)) {
        const fenced = echoFenced(cell, options);
        const ticks = fenced ? "````" : "```";
        md.push(ticks + " {");
        if (typeof cell.options[kCellLstLabel] === "string") {
            let label = cell.options[kCellLstLabel];
            if (!label.startsWith("#")) {
                label = "#" + label;
            }
            md.push(label + " ");
        }
        if (!fenced) {
            md.push("." + options.language);
        }
        md.push(" .cell-code");
        if (hideCode(cell, options)) {
            md.push(" .hidden");
        }
        if (cell.options[kCodeOverflow] === "wrap") {
            md.push(" .code-overflow-wrap");
        } else if (cell.options[kCodeOverflow] === "scroll") {
            md.push(" .code-overflow-scroll");
        }
        if (typeof cell.options[kCellLstCap] === "string") {
            md.push(` caption=\"${cell.options[kCellLstCap]}\"`);
        }
        if (typeof cell.options[kCodeFold] !== "undefined") {
            md.push(` code-fold=\"${cell.options[kCodeFold]}\"`);
        }
        if (typeof cell.options[kCodeSummary] !== "undefined") {
            md.push(` code-summary=\"${cell.options[kCodeSummary]}\"`);
        }
        if (typeof cell.options[kCodeLineNumbers] !== "undefined") {
            md.push(` code-line-numbers=\"${cell.options[kCodeLineNumbers]}\"`);
        }
        md.push("}\n");
        let source = __VIRTUAL_FILE107(cell.source);
        if (fenced) {
            const optionsSource = cell.optionsSource.filter((line)=>line.search(/echo:\s+fenced/) === -1
            );
            if (optionsSource.length > 0) {
                source = mdTrimEmptyLines(source, "trailing");
            } else {
                source = mdTrimEmptyLines(source, "all");
            }
            source.unshift(...optionsSource);
            source.unshift("```{{" + options.language + "}}\n");
            source.push("\n```\n");
        } else if (cell.optionsSource.length > 0) {
            source = mdTrimEmptyLines(source, "leading");
        }
        md.push(...source, "\n");
        md.push(ticks + "\n");
    }
    if (includeOutput(cell, options)) {
        const labelName = label ? label.replace(/^#/, "").replaceAll(":", "-") : "cell-" + (cellIndex + 1);
        const outputName = pandocAutoIdentifier(labelName, true) + "-output";
        let nextOutputSuffix = 1;
        for (const { index: index1 , output  } of outputs.map((value, index)=>({
                index,
                output: value
            })
        )){
            const outputLabel = label && labelCellContainer && isDisplayData(output) ? label + "-" + nextOutputSuffix++ : label;
            if (!asis) {
                md.push("\n::: {");
                if (outputLabel && shouldLabelOutputContainer(output, cell.options, options)) {
                    md.push(outputLabel + " ");
                }
                if (output.output_type === "stream") {
                    const stream = output;
                    md.push(`.cell-output-${stream.name}`);
                } else {
                    md.push(`.${outputTypeCssClass(output.output_type)}`);
                }
                if (hideOutput(cell, options) || isWarningOutput(output) && hideWarnings(cell, options)) {
                    md.push(` .hidden`);
                }
                if (typeof output.execution_count === "number") {
                    md.push(` execution_count=${output.execution_count}`);
                }
                md.push("}\n");
            }
            if (options.toLatex && !options.figPos && !cell.options[kCellFigPos] && !hasLayoutOptions(cell) && includeCode(cell, options)) {
                cell.options[kCellFigPos] = "H";
            }
            const figureOptions = {};
            const broadcastFigureOption = (name98)=>{
                const value = cell.options[name98];
                if (value) {
                    if (Array.isArray(value)) {
                        return value[index1];
                    } else {
                        return value;
                    }
                } else {
                    return null;
                }
            };
            figureOptions[kCellFigAlign] = broadcastFigureOption(kCellFigAlign);
            figureOptions[kCellFigScap] = broadcastFigureOption(kCellFigScap);
            figureOptions[kCellFigLink] = broadcastFigureOption(kCellFigLink);
            figureOptions[kCellFigEnv] = broadcastFigureOption(kCellFigEnv);
            figureOptions[kCellFigPos] = broadcastFigureOption(kCellFigPos);
            figureOptions[kCellFigAlt] = broadcastFigureOption(kCellFigAlt);
            if (output.output_type === "stream") {
                const stream = output;
                if (asis && stream.name === "stdout") {
                    md.push(stream.text.join(""));
                } else {
                    md.push(mdOutputStream(stream));
                }
            } else if (output.output_type === "error") {
                md.push(mdOutputError(output));
            } else if (isDisplayData(output)) {
                const caption = isCaptionableData(output) ? outputCaptions.shift() || null : null;
                md.push(mdOutputDisplayData(outputLabel, caption, outputName + "-" + (index1 + 1), output, options, figureOptions));
                if (caption && !isImage(output, options)) {
                    md.push(`\n${caption}\n`);
                }
            } else {
                throw new Error("Unexpected output type " + output.output_type);
            }
            if (!asis) {
                md.push(`:::\n`);
            }
        }
    } else if (cell.outputs) {
        cell.outputs.filter(isDisplayData).filter((output)=>output.metadata.ojs_define
        ).forEach((ojs_define)=>{
            const ojs_html = ojs_define.data[kTextHtml];
            md.push("\n" + mdHtmlOutput(ojs_html));
        });
    }
    if (md.length > 0 && !asis) {
        md.unshift(divBeginMd);
        if (cellCaption) {
            md.push("\n" + cellCaption + "\n");
        }
        md.push(":::\n");
    }
    md.push("\n".repeat(cell.metadata.lines_to_next_cell || 1));
    if (cell.options[kCellMdIndent]) {
        const indent = String(cell.options[kCellMdIndent]);
        const mdWithIndent = md.join("").split("\n").map((line)=>indent + line
        ).join("\n");
        md.splice(0, md.length - 1);
        md.push(...mdWithIndent);
    }
    return md;
}
function hasLayoutOptions(cell) {
    return Object.keys(cell.options).some((key)=>key.startsWith("layout")
    );
}
function isDisplayDataType(output, options, checkFn) {
    if (isDisplayData(output)) {
        const mimeType = displayDataMimeType(output, options);
        if (mimeType) {
            if (checkFn(mimeType)) {
                return true;
            }
        }
    }
    return false;
}
function isImage(output, options) {
    return isDisplayDataType(output, options, displayDataIsImage);
}
function mdOutputStream(output) {
    if (output.name === "stderr") {
        if (output.text[0]) {
            const firstLine = output.text[0].replace(/<ipython-input.*?>:\d+:\s+/, "");
            return mdCodeOutput([
                firstLine,
                ...output.text.slice(1)
            ]);
        }
    }
    return mdCodeOutput(output.text);
}
function mdOutputError(output) {
    return mdCodeOutput([
        output.ename + ": " + output.evalue
    ]);
}
function mdOutputDisplayData(label, caption, filename, output, options, figureOptions) {
    const mimeType = displayDataMimeType(output, options);
    if (mimeType) {
        if (displayDataIsImage(mimeType)) {
            return mdImageOutput(label, caption, filename, mimeType, output, options, figureOptions);
        } else if (displayDataIsMarkdown(mimeType)) {
            return mdMarkdownOutput(output.data[mimeType]);
        } else if (displayDataIsLatex(mimeType)) {
            return mdLatexOutput(output.data[mimeType]);
        } else if (displayDataIsHtml(mimeType)) {
            return mdHtmlOutput(output.data[mimeType]);
        } else if (displayDataIsJson(mimeType)) {
            return mdJsonOutput(mimeType, output.data[mimeType], options);
        } else if (displayDataIsJavascript(mimeType)) {
            return mdScriptOutput(mimeType, output.data[mimeType]);
        } else if (displayDataIsTextPlain(mimeType)) {
            const lines1 = output.data[mimeType];
            if (lines1.length === 1 && lines1[0].startsWith("'<table") && lines1[0].endsWith("</table>'")) {
                lines1[0] = lines1[0].slice(1, -1);
                return mdMarkdownOutput(lines1);
            } else {
                return mdCodeOutput(lines1);
            }
        }
    }
    return mdWarningOutput("Unable to display output for mime type(s): " + Object.keys(output.data).join(", "));
}
function mdImageOutput(label, caption, filename, mimeType, output, options, figureOptions) {
    const data48 = output.data[mimeType];
    const metadata = output.metadata[mimeType];
    function metadataValue(key, defaultValue) {
        return metadata && metadata[key] ? metadata["key"] : defaultValue;
    }
    let width = metadataValue(kCellOutWidth, 0);
    let height = metadataValue(kCellOutHeight, 0);
    const alt = caption || "";
    const ext = extensionForMimeImageType(mimeType);
    const imageFile = options.assets.figures_dir + "/" + filename + "." + ext;
    const imageText = Array.isArray(data48) ? data48.join("") : data48;
    const outputFile = join3(options.assets.base_dir, imageFile);
    if (mimeType !== kImageSvg) {
        const imageData = decode(imageText);
        if (mimeType === kImagePng && options.figFormat === "retina" && options.figDpi) {
            const png = new PngImage(imageData);
            if (png.dpiX === options.figDpi * 2 && png.dpiY === options.figDpi * 2) {
                width = Math.round(png.width / 2);
                height = Math.round(png.height / 2);
            }
        }
        Deno.writeFileSync(outputFile, imageData);
    } else {
        Deno.writeTextFileSync(outputFile, imageText);
    }
    let image1 = `![${alt}](${imageFile})`;
    if (label || width || height) {
        image1 += "{";
        if (label) {
            image1 += `${label} `;
        }
        if (width) {
            image1 += `width=${width} `;
        }
        if (height) {
            image1 += `height=${height} `;
        }
        [
            kCellFigAlign,
            kCellFigEnv,
            kCellFigAlt,
            kCellFigPos,
            kCellFigScap
        ].forEach((attrib)=>{
            const value = figureOptions[attrib];
            if (value) {
                image1 += `${attrib}='${value}' `;
            }
        });
        image1 = image1.trimRight() + "}";
    }
    if (figureOptions[kCellFigLink]) {
        image1 = `[${image1}](${figureOptions[kCellFigLink]})`;
    }
    return mdMarkdownOutput([
        image1
    ]);
}
function mdMarkdownOutput(md) {
    return md.join("") + "\n";
}
function mdFormatOutput(format34, source) {
    return mdEnclosedOutput("```{=" + format34 + "}", source, "```");
}
function mdLatexOutput(latex) {
    return mdFormatOutput("tex", latex);
}
function mdHtmlOutput(html) {
    return mdFormatOutput("html", html);
}
function mdJsonOutput(mimeType, json2, options) {
    if (options.toIpynb) {
        return mdCodeOutput([
            JSON.stringify(json2)
        ], "json");
    } else {
        return mdScriptOutput(mimeType, [
            JSON.stringify(json2)
        ]);
    }
}
function mdScriptOutput(mimeType, script) {
    const scriptTag = [
        `<script type="${mimeType}">\n`,
        ...script,
        "\n</script>", 
    ];
    return mdHtmlOutput(scriptTag);
}
function mdTrimEmptyLines(lines2, trim = "all") {
    if (trim === "all" || trim === "leading") {
        const firstNonEmpty = lines2.findIndex((line)=>line.trim().length > 0
        );
        if (firstNonEmpty === -1) {
            return [];
        }
        lines2 = lines2.slice(firstNonEmpty);
    }
    if (trim === "all" || trim === "trailing") {
        let lastNonEmpty = -1;
        for(let i64 = lines2.length - 1; i64 >= 0; i64--){
            if (lines2[i64].trim().length > 0) {
                lastNonEmpty = i64;
                break;
            }
        }
        if (lastNonEmpty > -1) {
            lines2 = lines2.slice(0, lastNonEmpty + 1);
        }
    }
    return lines2;
}
function mdCodeOutput(code29, clz) {
    const open = "```" + (clz ? `{.${clz}}` : "");
    return mdEnclosedOutput(open, code29, "```");
}
function mdEnclosedOutput(begin, text, end) {
    const output = text.join("");
    const md = [
        begin + "\n",
        output + (output.endsWith("\n") ? "" : "\n"),
        end + "\n", 
    ];
    return md.join("");
}
function mdWarningOutput(msg) {
    return mdOutputStream({
        output_type: "stream",
        name: "stderr",
        text: [
            msg
        ]
    });
}
function isWarningOutput(output) {
    if (output.output_type === "stream") {
        const stream = output;
        return stream.name === "stderr";
    } else {
        return false;
    }
}
function outputTypeCssClass(output_type) {
    if ([
        "display_data",
        "execute_result"
    ].includes(output_type)) {
        output_type = "display";
    }
    return `cell-output-${output_type}`;
}
function pandocAttrParseText(attr) {
    attr = attr.trim();
    let id = "";
    const classes = [];
    let remainder = "";
    let current = "";
    const resolveCurrent = ()=>{
        const resolve5 = current;
        current = "";
        if (resolve5.length === 0) {
            return true;
        } else if (resolve5.startsWith("#")) {
            if (id.length === 0 && resolve5.length > 1) {
                id = resolve5.substr(1);
                return true;
            } else {
                return false;
            }
        } else if (resolve5.startsWith(".")) {
            if (resolve5.length > 1) {
                classes.push(resolve5.substr(1));
                return true;
            } else {
                return false;
            }
        } else if (resolve5 === "-") {
            classes.push("unnumbered");
            return true;
        } else {
            remainder = resolve5;
            return true;
        }
    };
    for(let i65 = 0; i65 < attr.length; i65++){
        let inQuotes = false;
        const ch = attr[i65];
        inQuotes = ch === '"' ? !inQuotes : inQuotes;
        if (ch !== " " && !inQuotes) {
            current += ch;
        } else if (resolveCurrent()) {
            if (remainder.length > 0) {
                remainder = remainder + attr.substr(i65);
                break;
            }
        } else {
            return null;
        }
    }
    if (resolveCurrent()) {
        if (id.length === 0 && classes.length === 0) {
            remainder = attr;
        }
        return {
            id,
            classes,
            keyvalue: remainder.length > 0 ? pandocAttrKeyvalueFromText(remainder, " ") : []
        };
    } else {
        return null;
    }
}
function pandocAttrKeyvalueFromText(text, separator) {
    if (separator === " ") {
        let convertedText = "";
        let inQuotes = false;
        for(let i66 = 0; i66 < text.length; i66++){
            let ch = text.charAt(i66);
            if (ch === '"') {
                inQuotes = !inQuotes;
            } else if (ch === " " && !inQuotes) {
                ch = "\n";
            }
            convertedText += ch;
        }
        text = convertedText;
    }
    const lines2 = text.trim().split("\n");
    return lines2.map((line)=>{
        const parts = line.trim().split("=");
        return [
            parts[0],
            (parts[1] || "").replace(/^"/, "").replace(/"$/, "")
        ];
    });
}
function parsePandocTitle(title4) {
    title4 = title4 ? title4.trim() : title4;
    let beginAttrPos = -1;
    let previousCh = undefined;
    let state39 = "none";
    for(let i67 = title4.length - 1; i67 > -1; i67--){
        const ch = title4.charAt(i67);
        if (previousCh === undefined && ch !== "}") {
            break;
        }
        if (previousCh === undefined && ch === "}") {
            state39 = "scanning";
            previousCh = ch;
            continue;
        }
        if (state39 === "scanning") {
            if ([
                "/",
                "}",
                "]"
            ].includes(ch)) {
                break;
            }
            state39 = "reading";
            previousCh = ch;
            continue;
        }
        if (state39 === "reading") {
            if (![
                "/"
            ].includes(ch) && previousCh === "{") {
                beginAttrPos = i67 + 1;
                break;
            }
            previousCh = ch;
        }
    }
    let markdownHeading = beginAttrPos !== -1 ? title4.slice(0, beginAttrPos) : title4;
    markdownHeading = markdownHeading.trim().replace(/^#{1,}\s*/, "");
    let markdownHeadingAttr;
    if (beginAttrPos !== -1) {
        const endAttrPos = title4.lastIndexOf("}");
        if (endAttrPos !== -1) {
            const attr = title4.slice(beginAttrPos + 1, endAttrPos);
            const parsed = pandocAttrParseText(attr);
            if (parsed) {
                markdownHeadingAttr = parsed;
            }
        }
    }
    return {
        heading: markdownHeading,
        attr: markdownHeadingAttr
    };
}
function partitionMarkdown(markdown) {
    const markdownLines = [];
    let markdownHeading;
    let markdownHeadingAttr;
    let markdownContainsRefs = false;
    const partitioned = partitionYamlFrontMatter(markdown);
    markdown = partitioned ? partitioned.markdown : markdown;
    for (const line of lines(markdown)){
        if (!markdownContainsRefs) {
            markdownContainsRefs = /^:::\s*{#refs([\s}]|.*?})\s*$/.test(line);
        }
        if (!markdownHeading) {
            if (line.startsWith("#")) {
                const parsedHeading = parsePandocTitle(line);
                markdownHeading = parsedHeading.heading;
                markdownHeadingAttr = parsedHeading.attr;
            } else if (line.match(/^=+\s*$/) || line.match(/^-+\s*$/)) {
                const prevLine = markdownLines[markdownLines.length - 1];
                if (prevLine) {
                    markdownHeading = prevLine;
                    markdownLines.splice(markdownLines.length - 1);
                } else {
                    markdownLines.push(line);
                }
            } else {
                markdownLines.push(line);
            }
        } else {
            markdownLines.push(line);
        }
    }
    return {
        yaml: partitioned ? readYamlFromMarkdown(partitioned.yaml) : undefined,
        headingText: markdownHeading,
        headingAttr: markdownHeadingAttr,
        containsRefs: markdownContainsRefs,
        markdown: markdownLines.join("\n")
    };
}
async function knitrCapabilities() {
    try {
        const result = await execProcess({
            cmd: [
                await rBinaryPath("Rscript"),
                resourcePath("capabilities/knitr.R"), 
            ],
            stdout: "piped"
        });
        if (result.success && result.stdout) {
            const caps = readYamlFromString(result.stdout);
            return caps;
        } else {
            return undefined;
        }
    } catch  {
        return undefined;
    }
}
function knitrCapabilitiesMessage(caps, indent = "") {
    const lines3 = [
        `Version: ${caps.versionMajor}.${caps.versionMinor}.${caps.versionPatch}`,
        `Path: ${caps.home}`,
        `LibPaths:`, 
    ];
    for (const path79 of caps.libPaths){
        lines3.push(`  - ${path79}`);
    }
    lines3.push(`rmarkdown: ${caps.rmarkdown || "(None)"}`);
    return lines3.map((line)=>`${indent}${line}`
    ).join("\n");
}
function knitrInstallationMessage(indent = "") {
    const lines4 = [
        "The rmarkdown package is not available in this R installation.",
        "Install with " + bold('install.packages("rmarkdown")'), 
    ];
    return lines4.map((line)=>`${indent}${line}`
    ).join("\n");
}
function rInstallationMessage(indent = "") {
    const lines5 = [
        "Unable to locate an installed version of R.",
        "Install R from " + bold("https://cloud.r-project.org/"), 
    ];
    return lines5.map((line)=>`${indent}${line}`
    ).join("\n");
}
const kQmdExtensions = [
    ".qmd"
];
const kMarkdownEngine = "markdown";
const kKnitrEngine = "knitr";
const kJupyterEngine = "jupyter";
function postProcessRestorePreservedHtml(options) {
    const outputPath = join3(dirname2(options.target.input), options.output);
    let output = Deno.readTextFileSync(outputPath);
    output = restorePreservedHtml(output, options.preserve);
    Deno.writeTextFileSync(outputPath, output);
}
const kRmdExtensions = [
    ".rmd",
    ".rmarkdown"
];
const knitrEngine = {
    name: kKnitrEngine,
    defaultExt: ".qmd",
    defaultYaml: ()=>[]
    ,
    defaultContent: ()=>[
            "```{r}",
            "1 + 1",
            "```", 
        ]
    ,
    validExtensions: ()=>kRmdExtensions.concat(kRmdExtensions)
    ,
    claimsExtension: (ext)=>{
        return kRmdExtensions.includes(ext.toLowerCase());
    },
    claimsLanguage: (language)=>{
        return language.toLowerCase() === "r";
    },
    target: (file, _quiet)=>{
        const markdown = Deno.readTextFileSync(file);
        return Promise.resolve({
            source: file,
            input: file,
            markdown,
            metadata: readYamlFromMarkdown(markdown)
        });
    },
    partitionedMarkdown: (file)=>{
        return Promise.resolve(partitionMarkdown(Deno.readTextFileSync(file)));
    },
    execute: (options)=>{
        return callR("execute", {
            ...options,
            target: undefined,
            input: options.target.input
        }, options.tempDir, options.quiet);
    },
    dependencies: (options)=>{
        return callR("dependencies", {
            ...options,
            target: undefined,
            input: options.target.input
        }, options.tempDir, options.quiet);
    },
    postprocess: async (options)=>{
        postProcessRestorePreservedHtml(options);
        if (options.format.render?.[kCodeLink]) {
            await callR("postprocess", {
                ...options,
                target: undefined,
                preserve: undefined,
                input: options.target.input
            }, options.tempDir, options.quiet, false).then(()=>{
                return Promise.resolve();
            }, ()=>{
                warning(`Unable to perform code-link (code-link requires R and the downlit package)`);
                return Promise.resolve();
            });
        }
    },
    canFreeze: true,
    generatesFigures: true,
    ignoreDirs: ()=>{
        return [
            "renv",
            "packrat",
            "rsconnect"
        ];
    },
    run: (options)=>{
        return callR("run", options, options.tempDir);
    }
};
async function callR(action, params, tempDir2, quiet, reportError = true) {
    const resultsFile = Deno.makeTempFileSync({
        dir: tempDir2,
        prefix: "r-results",
        suffix: ".json"
    });
    const input = JSON.stringify({
        action,
        params,
        results: resultsFile
    });
    try {
        const result = await execProcess({
            cmd: [
                await rBinaryPath("Rscript"),
                resourcePath("rmd/rmd.R"), 
            ],
            stderr: quiet ? "piped" : "inherit"
        }, input, "stdout>stderr", red);
        if (result.success) {
            const results = await Deno.readTextFile(resultsFile);
            await Deno.remove(resultsFile);
            const resultsJson = JSON.parse(results);
            return resultsJson;
        } else {
            if (quiet) {
                error(result.stderr || "");
            }
            if (reportError) {
                await printCallRDiagnostics();
            }
            return Promise.reject();
        }
    } catch (e) {
        if (reportError) {
            if (e?.message) {
                info("");
                error(e.message);
            }
            await printCallRDiagnostics();
        }
        return Promise.reject();
    }
}
async function printCallRDiagnostics() {
    const caps = await knitrCapabilities();
    if (caps && !caps.rmarkdown) {
        info("");
        info("R installation:");
        info(knitrCapabilitiesMessage(caps, "  "));
        info("");
        info(knitrInstallationMessage());
        info("");
    } else if (!caps) {
        info("");
        info(rInstallationMessage());
        info("");
    }
}
function runningInCI() {
    if ([
        "CI",
        "CONTINUOUS_INTEGRATION",
        "BUILD_NUMBER",
        "RUN_ID"
    ].some(checkEnv)) {
        return true;
    }
    return vendors.some((vendor)=>{
        const envs = Array.isArray(vendor.env) ? vendor.env : [
            vendor.env
        ];
        return envs.some(checkEnv);
    });
}
function checkEnv(obj) {
    if (typeof obj === "string") {
        return !!Deno.env.get(obj);
    } else {
        return Object.keys(obj).every(function(k) {
            return Deno.env.get(k) === obj[k];
        });
    }
}
const vendors = [
    {
        "name": "AppVeyor",
        "constant": "APPVEYOR",
        "env": "APPVEYOR",
        "pr": "APPVEYOR_PULL_REQUEST_NUMBER"
    },
    {
        "name": "Azure Pipelines",
        "constant": "AZURE_PIPELINES",
        "env": "SYSTEM_TEAMFOUNDATIONCOLLECTIONURI",
        "pr": "SYSTEM_PULLREQUEST_PULLREQUESTID"
    },
    {
        "name": "Appcircle",
        "constant": "APPCIRCLE",
        "env": "AC_APPCIRCLE"
    },
    {
        "name": "Bamboo",
        "constant": "BAMBOO",
        "env": "bamboo_planKey"
    },
    {
        "name": "Bitbucket Pipelines",
        "constant": "BITBUCKET",
        "env": "BITBUCKET_COMMIT",
        "pr": "BITBUCKET_PR_ID"
    },
    {
        "name": "Bitrise",
        "constant": "BITRISE",
        "env": "BITRISE_IO",
        "pr": "BITRISE_PULL_REQUEST"
    },
    {
        "name": "Buddy",
        "constant": "BUDDY",
        "env": "BUDDY_WORKSPACE_ID",
        "pr": "BUDDY_EXECUTION_PULL_REQUEST_ID"
    },
    {
        "name": "Buildkite",
        "constant": "BUILDKITE",
        "env": "BUILDKITE",
        "pr": {
            "env": "BUILDKITE_PULL_REQUEST",
            "ne": "false"
        }
    },
    {
        "name": "CircleCI",
        "constant": "CIRCLE",
        "env": "CIRCLECI",
        "pr": "CIRCLE_PULL_REQUEST"
    },
    {
        "name": "Cirrus CI",
        "constant": "CIRRUS",
        "env": "CIRRUS_CI",
        "pr": "CIRRUS_PR"
    },
    {
        "name": "AWS CodeBuild",
        "constant": "CODEBUILD",
        "env": "CODEBUILD_BUILD_ARN"
    },
    {
        "name": "Codefresh",
        "constant": "CODEFRESH",
        "env": "CF_BUILD_ID",
        "pr": {
            "any": [
                "CF_PULL_REQUEST_NUMBER",
                "CF_PULL_REQUEST_ID"
            ]
        }
    },
    {
        "name": "Codeship",
        "constant": "CODESHIP",
        "env": {
            "CI_NAME": "codeship"
        }
    },
    {
        "name": "Drone",
        "constant": "DRONE",
        "env": "DRONE",
        "pr": {
            "DRONE_BUILD_EVENT": "pull_request"
        }
    },
    {
        "name": "dsari",
        "constant": "DSARI",
        "env": "DSARI"
    },
    {
        "name": "GitHub Actions",
        "constant": "GITHUB_ACTIONS",
        "env": "GITHUB_ACTIONS",
        "pr": {
            "GITHUB_EVENT_NAME": "pull_request"
        }
    },
    {
        "name": "GitLab CI",
        "constant": "GITLAB",
        "env": "GITLAB_CI",
        "pr": "CI_MERGE_REQUEST_ID"
    },
    {
        "name": "GoCD",
        "constant": "GOCD",
        "env": "GO_PIPELINE_LABEL"
    },
    {
        "name": "LayerCI",
        "constant": "LAYERCI",
        "env": "LAYERCI",
        "pr": "LAYERCI_PULL_REQUEST"
    },
    {
        "name": "Hudson",
        "constant": "HUDSON",
        "env": "HUDSON_URL"
    },
    {
        "name": "Jenkins",
        "constant": "JENKINS",
        "env": [
            "JENKINS_URL",
            "BUILD_ID"
        ],
        "pr": {
            "any": [
                "ghprbPullId",
                "CHANGE_ID"
            ]
        }
    },
    {
        "name": "Magnum CI",
        "constant": "MAGNUM",
        "env": "MAGNUM"
    },
    {
        "name": "Netlify CI",
        "constant": "NETLIFY",
        "env": "NETLIFY",
        "pr": {
            "env": "PULL_REQUEST",
            "ne": "false"
        }
    },
    {
        "name": "Nevercode",
        "constant": "NEVERCODE",
        "env": "NEVERCODE",
        "pr": {
            "env": "NEVERCODE_PULL_REQUEST",
            "ne": "false"
        }
    },
    {
        "name": "Render",
        "constant": "RENDER",
        "env": "RENDER",
        "pr": {
            "IS_PULL_REQUEST": "true"
        }
    },
    {
        "name": "Sail CI",
        "constant": "SAIL",
        "env": "SAILCI",
        "pr": "SAIL_PULL_REQUEST_NUMBER"
    },
    {
        "name": "Semaphore",
        "constant": "SEMAPHORE",
        "env": "SEMAPHORE",
        "pr": "PULL_REQUEST_NUMBER"
    },
    {
        "name": "Screwdriver",
        "constant": "SCREWDRIVER",
        "env": "SCREWDRIVER",
        "pr": {
            "env": "SD_PULL_REQUEST",
            "ne": "false"
        }
    },
    {
        "name": "Shippable",
        "constant": "SHIPPABLE",
        "env": "SHIPPABLE",
        "pr": {
            "IS_PULL_REQUEST": "true"
        }
    },
    {
        "name": "Solano CI",
        "constant": "SOLANO",
        "env": "TDDIUM",
        "pr": "TDDIUM_PR_ID"
    },
    {
        "name": "Strider CD",
        "constant": "STRIDER",
        "env": "STRIDER"
    },
    {
        "name": "TaskCluster",
        "constant": "TASKCLUSTER",
        "env": [
            "TASK_ID",
            "RUN_ID"
        ]
    },
    {
        "name": "TeamCity",
        "constant": "TEAMCITY",
        "env": "TEAMCITY_VERSION"
    },
    {
        "name": "Travis CI",
        "constant": "TRAVIS",
        "env": "TRAVIS",
        "pr": {
            "env": "TRAVIS_PULL_REQUEST",
            "ne": "false"
        }
    },
    {
        "name": "Vercel",
        "constant": "VERCEL",
        "env": "NOW_BUILDER"
    },
    {
        "name": "Visual Studio App Center",
        "constant": "APPCENTER",
        "env": "APPCENTER_BUILD_ID"
    }, 
];
const sleep = (delay)=>{
    return new Promise((resolve6)=>{
        setTimeout(resolve6, delay);
    });
};
async function jupyterCapabilitiesMessage(caps, indent = "") {
    const lines6 = [
        `Version: ${caps.versionMajor}.${caps.versionMinor}.${caps.versionPatch}${caps.conda ? " (Conda)" : ""}`,
        `Path: ${caps.executable}`,
        `Jupyter: ${caps.jupyter_core || "(None)"}`, 
    ];
    if (caps.jupyter_core) {
        const kernels = Array.from((await jupyterKernelspecs()).values()).map((kernel)=>kernel.name
        ).join(", ");
        lines6.push(`Kernels: ${kernels}`);
    }
    return lines6.map((line)=>`${indent}${line}`
    ).join("\n");
}
async function jupyterInstallationMessage(caps, indent = "") {
    const lines7 = [
        "Jupyter is not available in this Python installation.",
        "Install with " + bold(`${caps.conda ? "conda" : (await pythonExec(true)).join(" ") + " -m pip"} install jupyter`), 
    ];
    return lines7.map((line)=>`${indent}${line}`
    ).join("\n");
}
function jupyterUnactivatedEnvMessage(caps, indent = "") {
    for (const path80 of Deno.readDirSync(Deno.cwd())){
        if (path80.isDirectory) {
            const targetPath = join3(Deno.cwd(), path80.name);
            if (isEnvDir(targetPath)) {
                try {
                    if (!pathWithForwardSlashes(caps.executable).startsWith(pathWithForwardSlashes(targetPath))) {
                        return indent + "There is an unactivated Python environment in " + bold(path80.name) + ". Did you forget to activate it?";
                    }
                } catch  {
                    return undefined;
                }
            }
        }
    }
    const kRequirementsTxt = "requirements.txt";
    const kEnvironmentYaml = "environment.yml";
    for (const envFile of [
        kRequirementsTxt,
        kEnvironmentYaml
    ]){
        if (existsSync(join3(Deno.cwd(), envFile))) {
            return indent + "There is a " + bold(envFile) + " file in this directory. " + "Is this for a " + (envFile === kRequirementsTxt ? "venv" : "conda env") + " that you need to restore?";
        }
    }
    return undefined;
}
function pythonInstallationMessage(indent = "") {
    const lines8 = [
        "Unable to locate an installed version of Python 3.",
        "Install Python 3 from " + bold("https://www.python.org/downloads/"), 
    ];
    return lines8.map((line)=>`${indent}${line}`
    ).join("\n");
}
async function executeKernelOneshot(options) {
    await abortKernel(options);
    if (!options.quiet) {
        messageStartingKernel();
    }
    trace(options, "Executing notebook with oneshot kernel");
    const debug2 = !!options.format.execute[kExecuteDebug];
    const result = await execJupyter("execute", {
        ...options,
        debug: debug2
    });
    if (!result.success) {
        return Promise.reject();
    }
}
async function executeKernelKeepalive(options) {
    let serverLogProcess;
    if (options.format.execute[kExecuteDebug]) {
        if (Deno.build.os !== "windows") {
            serverLogProcess = Deno.run({
                cmd: [
                    "tail",
                    "-F",
                    "-n",
                    "0",
                    kernelLogFile()
                ]
            });
        }
    }
    if (options.format.execute[kExecuteDaemonRestart]) {
        await abortKernel(options);
    }
    trace(options, "Connecting to kernel");
    const [conn, transport] = await connectToKernel(options);
    trace(options, "Kernel connection successful");
    try {
        trace(options, "Sending execute command to kernel");
        await writeKernelCommand(conn, "execute", transport.secret, {
            ...options
        });
        trace(options, "Execute command sent, reading response");
        let leftover = "";
        while(true){
            const buffer = new Uint8Array(512);
            const bytesRead = await conn.read(buffer);
            if (bytesRead === null) {
                break;
            }
            if (bytesRead > 0) {
                const payload = new TextDecoder().decode(buffer.slice(0, bytesRead));
                const jsonMessages = payload.split("\n");
                for (let jsonMessage of jsonMessages){
                    if (!jsonMessage) {
                        continue;
                    }
                    if (leftover) {
                        jsonMessage = leftover + jsonMessage;
                        leftover = "";
                    }
                    try {
                        const msg = JSON.parse(jsonMessage);
                        if (msg.type === "error") {
                            trace(options, "Error response received");
                            error(msg.data, {
                                colorize: false
                            });
                            printExecDiagnostics(msg.data);
                            return Promise.reject();
                        } else if (msg.type == "restart") {
                            trace(options, "Restart request received");
                            return executeKernelKeepalive(options);
                        } else {
                            info(msg.data, {
                                newline: false
                            });
                        }
                    } catch  {
                        leftover = jsonMessage;
                    }
                }
            }
        }
        trace(options, "Server request complete\n\n");
    } catch (e) {
        trace(options, "Error occurred receiving response from server");
        const transportFile = kernelTransportFile(options.target.input);
        if (existsSync(transportFile)) {
            Deno.removeSync(transportFile);
        }
        throw e;
    } finally{
        conn.close();
        if (serverLogProcess) {
            serverLogProcess.kill("SIGKILL");
        }
    }
}
async function abortKernel(options) {
    try {
        trace(options, "Checking for existing kernel");
        const [conn, transport] = await connectToKernel(options, false);
        trace(options, "Existing kernel found");
        try {
            trace(options, "Sending kernel abort request");
            await writeKernelCommand(conn, "abort", transport.secret, {});
            trace(options, "Abort request successful");
        } finally{
            const transportFile = kernelTransportFile(options.target.input);
            if (existsSync(transportFile)) {
                Deno.removeSync(transportFile);
            }
            conn.close();
        }
    } catch  {
        trace(options, "No existing kernel found");
    }
}
async function execJupyter(command, options) {
    try {
        const result = await execProcess({
            cmd: [
                ...await pythonExec(),
                resourcePath("jupyter/jupyter.py"), 
            ],
            stdout: "piped"
        }, kernelCommand(command, "", options));
        if (!result.success) {
            await printExecDiagnostics(result.stderr);
        }
        return result;
    } catch (e) {
        if (e?.message) {
            info("");
            error(e.message);
        }
        await printExecDiagnostics();
        return Promise.reject();
    }
}
async function printExecDiagnostics(stderr) {
    const caps = await jupyterCapabilities();
    if (caps && !caps.jupyter_core) {
        info("Python 3 installation:");
        info(await jupyterCapabilitiesMessage(caps, "  "));
        info("");
        info(await jupyterInstallationMessage(caps));
        info("");
        maybePrintUnactivatedEnvMessage(caps);
    } else if (!caps) {
        info(pythonInstallationMessage());
        info("");
    } else if (stderr && stderr.indexOf("ModuleNotFoundError") !== -1) {
        maybePrintUnactivatedEnvMessage(caps);
    }
}
function maybePrintUnactivatedEnvMessage(caps) {
    const envMessage = jupyterUnactivatedEnvMessage(caps);
    if (envMessage) {
        info(envMessage);
        info("");
    }
}
async function writeKernelCommand(conn, command, secret, options) {
    await conn.write(new TextEncoder().encode(kernelCommand(command, secret, options) + "\n"));
}
function kernelCommand(command, secret, options) {
    return JSON.stringify({
        command,
        secret,
        options: {
            ...options,
            log: kernelLogFile()
        }
    });
}
function kernelTransportFile(target) {
    const transportsDir = quartoRuntimeDir("jt");
    const targetFile = Deno.realPathSync(target);
    const hash = md5Hash(targetFile).slice(0, 20);
    return join3(transportsDir, hash);
}
function kernelLogFile() {
    const logsDir = quartoDataDir("logs");
    const kernelLog = join3(logsDir, "jupyter-kernel.log");
    if (!existsSync(kernelLog)) {
        Deno.writeTextFileSync(kernelLog, "");
    }
    return kernelLog;
}
function readKernelTransportFile(transportFile, type) {
    if (existsSync(transportFile)) {
        if (type === "tcp") {
            try {
                const transport = JSON.parse(Deno.readTextFileSync(transportFile));
                if (transport.port && transport.secret) {
                    return {
                        ...transport,
                        type
                    };
                } else {
                    throw new Error("Invalid file format");
                }
            } catch (e) {
                error("Error reading kernel transport file: " + e.toString() + "(removing file)");
                Deno.removeSync(transportFile);
                return null;
            }
        } else {
            return {
                port: transportFile,
                secret: "",
                type
            };
        }
    } else {
        return null;
    }
}
async function connectToKernel(options, startIfRequired = true) {
    const debug3 = !!options.format.execute[kExecuteDebug];
    const transportFile = kernelTransportFile(options.target.input);
    const type = "tcp";
    const transport = readKernelTransportFile(transportFile, type);
    if (transport) {
        try {
            return await denoConnectToKernel(transport);
        } catch  {
            Deno.removeSync(transportFile);
        }
    }
    if (!startIfRequired) {
        return Promise.reject();
    }
    if (!options.quiet) {
        messageStartingKernel();
    }
    const keepAlive = options.format.execute[kExecuteDaemon];
    const timeout = keepAlive === true || keepAlive === null || keepAlive === undefined ? 300 : keepAlive === false ? 0 : keepAlive;
    const result = await execJupyter("start", {
        transport: transportFile,
        timeout,
        type,
        debug: debug3
    });
    if (!result.success) {
        return Promise.reject();
    }
    for(let i68 = 1; i68 < 20; i68++){
        await sleep(i68 * 100);
        const kernelTransport = readKernelTransportFile(transportFile, type);
        if (kernelTransport) {
            try {
                return await denoConnectToKernel(kernelTransport);
            } catch (e) {
                Deno.removeSync(transportFile);
                error("Error connecting to Jupyter kernel: " + e.toString());
                return Promise.reject();
            }
        }
    }
    warning("Unable to start Jupyter kernel for " + options.target.input);
    return Promise.reject();
}
async function denoConnectToKernel(transport) {
    if (transport.type === "tcp") {
        const tcpConnectOptions = {
            transport: transport.type,
            hostname: "127.0.0.1",
            port: transport.port
        };
        return [
            await Deno.connect(tcpConnectOptions),
            transport, 
        ];
    } else {
        const unixConnectOptions = {
            transport: transport.type,
            path: transport.port
        };
        return [
            await Deno.connect(unixConnectOptions),
            transport, 
        ];
    }
}
function messageStartingKernel() {
    info("\nStarting Jupyter kernel...", {
        newline: false
    });
}
function trace(options, msg) {
    if (options.format.execute[kExecuteDebug]) {
        info("- " + msg, {
            bold: true
        });
    }
}
const jupyterEngine = {
    name: kJupyterEngine,
    defaultExt: ".qmd",
    defaultYaml: (kernel)=>[
            `jupyter: ${kernel || "python3"}`, 
        ]
    ,
    defaultContent: (kernel)=>{
        kernel = kernel || "python3";
        const lang = kernel.startsWith("python") ? "python" : kernel.startsWith("julia") ? "julia" : undefined;
        if (lang) {
            return [
                "```{" + lang + "}",
                "1 + 1",
                "```", 
            ];
        } else {
            return [];
        }
    },
    validExtensions: ()=>kJupyterNotebookExtensions.concat(kQmdExtensions)
    ,
    claimsExtension: (ext)=>{
        return kJupyterNotebookExtensions.includes(ext.toLowerCase());
    },
    claimsLanguage: (_language)=>{
        return false;
    },
    target: async (file)=>{
        const markdown = isJupyterNotebook(file) ? await markdownFromNotebook(file) : Deno.readTextFileSync(file);
        const metadata = readYamlFromMarkdown(markdown);
        if (isQmdFile(file)) {
            const [fileDir, fileStem] = dirAndStem(file);
            const notebook = join3(fileDir, fileStem + ".ipynb");
            const target = {
                source: file,
                input: notebook,
                markdown,
                metadata,
                data: {
                    transient: true
                }
            };
            await createNotebookforTarget(target);
            return target;
        } else if (isJupyterNotebook(file)) {
            return {
                source: file,
                input: file,
                markdown,
                metadata,
                data: {
                    transient: false
                }
            };
        } else {
            return undefined;
        }
    },
    partitionedMarkdown: async (file)=>{
        if (isJupyterNotebook(file)) {
            return partitionMarkdown(await markdownFromNotebook(file));
        } else {
            return partitionMarkdown(Deno.readTextFileSync(file));
        }
    },
    filterFormat: (source, options, format35)=>{
        if (isJupyterNotebook(source)) {
            let executeEnabled;
            if (options.devServerReload) {
                executeEnabled = false;
            } else if (typeof format35.execute[kExecuteIpynb] === "boolean") {
                executeEnabled = format35.execute[kExecuteIpynb];
            } else if (typeof format35.execute[kExecuteEnabled] == "boolean") {
                executeEnabled = format35.execute[kExecuteEnabled];
            } else {
                executeEnabled = false;
            }
            if (executeEnabled !== undefined) {
                return {
                    ...format35,
                    execute: {
                        ...format35.execute,
                        [kExecuteEnabled]: executeEnabled
                    }
                };
            } else {
                return format35;
            }
        } else {
            return format35;
        }
    },
    execute: async (options)=>{
        if (isQmdFile(options.target.source) && !existsSync(options.target.input)) {
            await createNotebookforTarget(options.target);
        }
        const execute = options.format.execute[kExecuteEnabled] !== false;
        if (execute) {
            const execOptions = {
                ...options,
                target: {
                    ...options.target,
                    input: Deno.realPathSync(options.target.input)
                }
            };
            let executeDaemon = options.format.execute[kExecuteDaemon];
            if (executeDaemon === null || executeDaemon === undefined) {
                executeDaemon = isInteractiveSession() && !runningInCI();
            }
            if (executeDaemon === false || executeDaemon === 0) {
                await executeKernelOneshot(execOptions);
            } else {
                await executeKernelKeepalive(execOptions);
            }
        }
        const nb = jupyterFromFile(options.target.input);
        const assets = jupyterAssets(options.target.input, options.format.pandoc.to);
        const result = jupyterToMarkdown(nb, {
            language: nb.metadata.kernelspec.language.toLowerCase(),
            assets,
            execute: options.format.execute,
            keepHidden: options.format.render[kKeepHidden],
            toHtml: isHtmlCompatible(options.format),
            toLatex: isLatexOutput(options.format.pandoc),
            toMarkdown: isMarkdownOutput(options.format.pandoc),
            toIpynb: isIpynbOutput(options.format.pandoc),
            toPresentation: isPresentationOutput(options.format.pandoc),
            figFormat: options.format.execute[kFigFormat],
            figDpi: options.format.execute[kFigDpi],
            figPos: options.format.render[kFigPos]
        });
        let includes;
        let engineDependencies;
        if (options.dependencies) {
            includes = executeResultIncludes(options.tempDir, result.dependencies);
        } else {
            const dependencies = executeResultEngineDependencies(result.dependencies);
            if (dependencies) {
                engineDependencies = {
                    [kJupyterEngine]: dependencies
                };
            }
        }
        cleanupNotebook(options.target, options.format);
        return {
            markdown: result.markdown,
            supporting: [
                join3(assets.base_dir, assets.supporting_dir)
            ],
            filters: [],
            pandoc: result.pandoc,
            includes,
            engineDependencies,
            preserve: result.htmlPreserve,
            postProcess: result.htmlPreserve && Object.keys(result.htmlPreserve).length > 0
        };
    },
    executeTargetSkipped: cleanupNotebook,
    dependencies: (options)=>{
        const includes = {};
        if (options.dependencies) {
            const includeFiles = includesForJupyterWidgetDependencies(options.dependencies, options.tempDir);
            if (includeFiles.inHeader) {
                includes[kIncludeInHeader] = [
                    includeFiles.inHeader
                ];
            }
            if (includeFiles.afterBody) {
                includes[kIncludeAfterBody] = [
                    includeFiles.afterBody
                ];
            }
        }
        return Promise.resolve({
            includes
        });
    },
    postprocess: (options)=>{
        postProcessRestorePreservedHtml(options);
        return Promise.resolve();
    },
    canFreeze: true,
    generatesFigures: true,
    ignoreDirs: ()=>{
        return [
            "venv",
            "env"
        ];
    },
    canKeepSource: (target)=>{
        return !isJupyterNotebook(target.source);
    },
    keepFiles: (input)=>{
        if (!isJupyterNotebook(input) && !input.endsWith(`.${kJupyterEngine}.md`)) {
            const [fileDir, fileStem] = dirAndStem(input);
            return [
                join3(fileDir, fileStem + ".ipynb")
            ];
        }
    }
};
function isQmdFile(file) {
    const ext = extname2(file);
    return kQmdExtensions.includes(ext);
}
async function createNotebookforTarget(target) {
    const nb = await quartoMdToJupyter(target.source, true);
    Deno.writeTextFileSync(target.input, JSON.stringify(nb, null, 2));
}
function cleanupNotebook(target, format36) {
    const data49 = target.data;
    if (data49.transient) {
        if (!format36.execute[kKeepIpynb]) {
            removeIfExists(target.input);
        }
    }
}
function executeResultIncludes(tempDir3, widgetDependencies) {
    if (widgetDependencies) {
        const includes = {};
        const includeFiles = includesForJupyterWidgetDependencies([
            widgetDependencies
        ], tempDir3);
        if (includeFiles.inHeader) {
            includes[kIncludeInHeader] = [
                includeFiles.inHeader
            ];
        }
        if (includeFiles.afterBody) {
            includes[kIncludeAfterBody] = [
                includeFiles.afterBody
            ];
        }
        return includes;
    } else {
        return undefined;
    }
}
function executeResultEngineDependencies(widgetDependencies) {
    if (widgetDependencies) {
        return [
            widgetDependencies
        ];
    } else {
        return undefined;
    }
}
async function markdownFromNotebook(file) {
    const decoder = new TextDecoder("utf-8");
    const nbContents = await Deno.readFile(file);
    const nb = JSON.parse(decoder.decode(nbContents));
    const cells = nb.cells;
    const markdown = cells.reduce((md, cell)=>{
        if ([
            "markdown",
            "raw"
        ].includes(cell.cell_type)) {
            return md + "\n" + cell.source.join("") + "\n";
        } else {
            return md;
        }
    }, "");
    return markdown;
}
const kMdExtensions = [
    ".md",
    ".markdown"
];
const markdownEngine = {
    name: kMarkdownEngine,
    defaultExt: ".qmd",
    defaultYaml: ()=>[]
    ,
    defaultContent: ()=>[]
    ,
    validExtensions: ()=>kQmdExtensions.concat(kMdExtensions)
    ,
    claimsExtension: (ext)=>{
        return kMdExtensions.includes(ext.toLowerCase());
    },
    claimsLanguage: (_language)=>{
        return false;
    },
    target: (file)=>{
        const markdown = Deno.readTextFileSync(file);
        return Promise.resolve({
            source: file,
            input: file,
            markdown,
            metadata: readYamlFromMarkdown(markdown)
        });
    },
    partitionedMarkdown: (file)=>{
        return Promise.resolve(partitionMarkdown(Deno.readTextFileSync(file)));
    },
    execute: (options)=>{
        const markdown = Deno.readTextFileSync(options.target.input);
        return Promise.resolve({
            markdown,
            supporting: [],
            filters: []
        });
    },
    dependencies: (_options)=>{
        return Promise.resolve({
            includes: {}
        });
    },
    postprocess: (_options)=>Promise.resolve()
    ,
    canFreeze: false,
    generatesFigures: false
};
const kEngines = [
    knitrEngine,
    jupyterEngine,
    markdownEngine, 
];
function executionEngines() {
    return kEngines.map((engine)=>engine.name
    );
}
function executionEngine(name99) {
    for (const engine of kEngines){
        if (engine.name === name99) {
            return engine;
        }
    }
}
function executionEngineKeepMd(input) {
    const keepSuffix = `.md`;
    if (!input.endsWith(keepSuffix)) {
        const [dir, stem] = dirAndStem(input);
        return join3(dir, stem + keepSuffix);
    }
}
function executionEngineCanKeepSource(engine, target) {
    return !engine.canKeepSource || engine.canKeepSource(target);
}
function executionEngineKeepFiles(engine, input) {
    const files = [];
    const keep = executionEngineKeepMd(input);
    if (keep) {
        files.push(keep);
    }
    const engineKeepFiles = engine.keepFiles ? engine.keepFiles(input) : undefined;
    if (engineKeepFiles) {
        return files.concat(engineKeepFiles);
    } else {
        return files;
    }
}
function engineValidExtensions() {
    return __VIRTUAL_FILE181(kEngines.flatMap((engine)=>engine.validExtensions()
    ));
}
function fileExecutionEngine(file) {
    const ext = extname2(file).toLowerCase();
    if (!kEngines.some((engine)=>engine.validExtensions().includes(ext)
    )) {
        return undefined;
    }
    for (const engine1 of kEngines){
        if (engine1.claimsExtension(ext)) {
            return engine1;
        }
    }
    const markdown = Deno.readTextFileSync(file);
    const result = partitionYamlFrontMatter(markdown);
    if (result) {
        const yaml = readYamlFromMarkdown(result.yaml);
        if (yaml) {
            for (const engine of kEngines){
                if (yaml[engine.name]) {
                    return engine;
                }
                const format37 = metadataAsFormat(yaml);
                if (format37.execute?.[kEngine] === engine.name) {
                    return engine;
                }
            }
        }
    }
    const languages = languagesInMarkdown(markdown);
    if (languages.size > 0) {
        for (const language of languages){
            for (const engine of kEngines){
                if (engine.claimsLanguage(language)) {
                    return engine;
                }
            }
        }
        if (languages.size === 0 || languages.size == 1 && languages.has("ojs")) {
            return engineForMarkdownWithNoLanguages(markdown);
        } else {
            return jupyterEngine;
        }
    } else {
        return engineForMarkdownWithNoLanguages(markdown);
    }
}
function engineForMarkdownWithNoLanguages(markdown) {
    if (markdown.match(/`r[ #]([^`]+)\s*`/)) {
        return knitrEngine;
    } else {
        return markdownEngine;
    }
}
function engineIgnoreDirs() {
    const ignoreDirs = [];
    executionEngines().forEach((name100)=>{
        const engine = executionEngine(name100);
        if (engine && engine.ignoreDirs) {
            const ignores = engine.ignoreDirs();
            if (ignores) {
                ignoreDirs.push(...ignores);
            }
        }
    });
    return ignoreDirs;
}
function engineIgnoreGlobs() {
    return engineIgnoreDirs().map((ignore1)=>`**/${ignore1}/**`
    );
}
function hasTableOfContents(flags, format38) {
    return !!(flags[kToc] || format38.pandoc[kToc] || format38.pandoc[kTableOfContents]);
}
function disabledTableOfContents(format39) {
    return format39.pandoc[kToc] === false || format39.pandoc[kTableOfContents] === false;
}
function layerQuartoScss(key, dependency, sassLayer1, format40, darkLayer, darkDefault) {
    const frameworkFunctions = [
        bootstrapFunctions(),
        sassUtilFunctions("color-contrast.scss"), 
    ].join("\n");
    const frameworkVariables = [
        bootstrapVariables(),
        pandocVariablesToThemeScss(format40.metadata), 
    ].join("\n");
    return {
        dependency,
        key,
        user: sassLayer1,
        quarto: {
            use: [
                "sass:color",
                "sass:map",
                "sass:math"
            ],
            defaults: [
                quartoDefaults(format40),
                quartoBootstrapDefaults(format40.metadata), 
            ].join("\n"),
            functions: [
                quartoFunctions(),
                quartoBootstrapFunctions()
            ].join("\n"),
            mixins: quartoBootstrapMixins(),
            rules: [
                quartoRules(),
                quartoCopyCodeRules(),
                quartoBootstrapRules(),
                quartoGlobalCssVariableRules(),
                quartoLinkExternalRules(), 
            ].join("\n")
        },
        framework: {
            defaults: frameworkVariables,
            functions: frameworkFunctions,
            mixins: bootstrapMixins(),
            rules: bootstrapRules()
        },
        loadPaths: [
            bootstrapResourceDir()
        ],
        dark: darkLayer ? {
            user: darkLayer,
            default: darkDefault
        } : undefined
    };
}
function resolveBootstrapScss(input, format41) {
    const quartoThemesDir = formatResourcePath("html", join3("bootstrap", "themes"));
    const theme = format41.metadata[kTheme] || [];
    const [themeSassLayers, defaultDark] = resolveThemeLayer(input, theme, quartoThemesDir);
    const sassBundles = [];
    sassBundles.push(layerQuartoScss("quarto-theme", kBootstrapDependencyName, themeSassLayers.light, format41, themeSassLayers.dark, defaultDark));
    return sassBundles;
}
function layerTheme(input, themes1, quartoThemesDir) {
    let injectedCustomization = false;
    const layers = themes1.flatMap((theme)=>{
        const resolvedThemePath = join3(quartoThemesDir, `${theme}.scss`);
        if (existsSync(resolvedThemePath)) {
            const themeLayer1 = sassLayer(resolvedThemePath);
            injectedCustomization = true;
            return [
                themeLayer1,
                quartoBootstrapCustomizationLayer()
            ];
        } else {
            const themePath = join3(dirname2(input), theme);
            if (existsSync(themePath)) {
                return sassLayer(themePath);
            } else {
                return {
                    defaults: "",
                    functions: "",
                    mixins: "",
                    rules: ""
                };
            }
        }
    });
    if (!injectedCustomization) {
        layers.unshift(quartoBootstrapCustomizationLayer());
    }
    return layers;
}
function resolveThemeLayer(input, themes1, quartoThemesDir) {
    let theme1 = undefined;
    let defaultDark = false;
    if (typeof themes1 === "string") {
        theme1 = {
            light: [
                themes1
            ]
        };
    } else if (Array.isArray(themes1)) {
        theme1 = {
            light: themes1
        };
    } else if (typeof themes1 === "object") {
        const themeArr = (theme)=>{
            const themes2 = [];
            if (theme) {
                if (typeof theme === "string") {
                    themes2.push(theme);
                } else if (Array.isArray(theme)) {
                    themes2.push(...theme);
                }
            }
            return themes2;
        };
        const themeObj = themes1;
        const keyList = Object.keys(themeObj);
        defaultDark = keyList.length > 1 && keyList[0] === "dark";
        theme1 = {
            light: themeArr(themeObj.light),
            dark: themeObj.dark ? themeArr(themeObj.dark) : undefined
        };
    } else {
        theme1 = {
            light: []
        };
    }
    const themeSassLayer = {
        light: mergeLayers(...layerTheme(input, theme1.light, quartoThemesDir)),
        dark: theme1.dark ? mergeLayers(...layerTheme(input, theme1.dark, quartoThemesDir)) : undefined
    };
    return [
        themeSassLayer,
        defaultDark
    ];
}
function pandocVariablesToThemeDefaults(metadata) {
    const explicitVars = [];
    const add = (defaults, name101, value, formatter)=>{
        if (value) {
            const sassVar = sassVariable(name101, value, formatter);
            defaults.push(sassVar);
        }
    };
    add(explicitVars, "line-height-base", metadata["linestretch"], asCssNumber);
    add(explicitVars, "font-size-root", metadata["fontsize"]);
    add(explicitVars, "body-bg", metadata["backgroundcolor"]);
    add(explicitVars, "body-color", metadata["fontcolor"]);
    add(explicitVars, "link-color", metadata["linkcolor"]);
    add(explicitVars, "font-family-base", metadata["mainfont"], asCssFont);
    add(explicitVars, "font-family-code", metadata["monofont"], asCssFont);
    add(explicitVars, "mono-background-color", metadata["monobackgroundcolor"]);
    const explicitSizes = [
        "max-width",
        "margin-top",
        "margin-bottom",
        "margin-left",
        "margin-right", 
    ];
    explicitSizes.forEach((attrib)=>{
        add(explicitVars, attrib, metadata[attrib], asCssSize);
    });
    return explicitVars;
}
function pandocVariablesToThemeScss(metadata, asDefaults = false) {
    return pandocVariablesToThemeDefaults(metadata).map((variable)=>{
        return outputVariable(variable, asDefaults);
    }).join("\n");
}
const kCodeBorderLeft = "code-block-border-left";
const kCodeBlockBackground = "code-block-bg";
const kBackground = "background";
const kForeground = "foreground";
const kColor = "color";
const kBorder = "border";
const quartoBootstrapDefaults = (metadata)=>{
    const varFilePath = formatResourcePath("html", join3("bootstrap", "_bootstrap-variables.scss"));
    const variables = [
        Deno.readTextFileSync(varFilePath)
    ];
    const navbar = metadata[kWebsite]?.[kSiteNavbar];
    if (navbar && typeof navbar === "object") {
        const navbarBackground = navbar[kBackground];
        if (navbarBackground !== undefined) {
            variables.push(outputVariable(sassVariable("navbar-bg", navbarBackground, typeof navbarBackground === "string" ? asBootstrapColor : undefined)));
        }
        const navbarForeground = navbar[kForeground];
        if (navbarForeground !== undefined) {
            variables.push(outputVariable(sassVariable("navbar-fg", navbarForeground, typeof navbarForeground === "string" ? asBootstrapColor : undefined)));
        }
    }
    const sidebars = metadata[kWebsite]?.[kSiteSidebar];
    const sidebar = Array.isArray(sidebars) ? sidebars[0] : typeof sidebars === "object" ? sidebars : undefined;
    if (sidebar) {
        const sidebarBackground = sidebar[kBackground];
        if (sidebarBackground !== undefined) {
            variables.push(outputVariable(sassVariable("sidebar-bg", sidebarBackground, typeof sidebarBackground === "string" ? asBootstrapColor : undefined)));
        } else if (sidebar.style === "floating" || navbar) {
            variables.push(`$sidebar-bg: if(variable-exists(body-bg), $body-bg, #fff) !default;`);
        }
        const sidebarForeground = sidebar[kForeground];
        if (sidebarForeground !== undefined) {
            variables.push(outputVariable(sassVariable("sidebar-fg", sidebarForeground, typeof sidebarForeground === "string" ? asBootstrapColor : undefined)));
        }
    }
    const footer = metadata[kWebsite]?.[kPageFooter];
    if (footer !== undefined && typeof footer === "object") {
        const footerBg = footer[kBackground];
        if (footerBg !== undefined) {
            variables.push(outputVariable(sassVariable("footer-bg", footerBg, typeof footerBg === "string" ? asBootstrapColor : undefined)));
        }
        const footerFg = footer[kForeground];
        if (footerFg !== undefined) {
            variables.push(outputVariable(sassVariable("footer-fg", footerFg, typeof footerFg === "string" ? asBootstrapColor : undefined)));
        }
        const footerBorder = footer[kBorder];
        if (footerBorder !== false) {
            variables.push(outputVariable(sassVariable("footer-border", true)));
        }
        if (footerBorder !== undefined && typeof footerBorder === "string") {
            variables.push(outputVariable(sassVariable("footer-border-color", footerBorder, asBootstrapColor)));
        }
        const footerColor = footer[kColor];
        if (footerColor && typeof footerColor === "string") {
            variables.push(outputVariable(sassVariable("footer-color", footerColor, asBootstrapColor)));
        }
    }
    const codeblockLeftBorder = metadata[kCodeBorderLeft];
    const codeblockBackground = metadata[kCodeBlockBackground];
    if (codeblockLeftBorder !== undefined) {
        variables.push(outputVariable(sassVariable(kCodeBorderLeft, codeblockLeftBorder, typeof codeblockLeftBorder === "string" ? asBootstrapColor : undefined)));
        if (codeblockBackground === undefined && codeblockLeftBorder !== false) {
            variables.push(outputVariable(sassVariable(kCodeBlockBackground, false)));
        }
    }
    if (codeblockBackground !== undefined) {
        variables.push(outputVariable(sassVariable(kCodeBlockBackground, codeblockBackground, typeof codeblockBackground === "string" ? asBootstrapColor : undefined)));
    }
    return variables.reverse().join("\n");
};
function formatHasBootstrap(format42) {
    if (format42 && isHtmlOutput(format42.pandoc, true)) {
        return hasBootstrapTheme(format42.metadata);
    } else {
        return false;
    }
}
function hasBootstrapTheme(metadata) {
    const theme = metadata[kTheme];
    return theme !== "none" && theme !== "pandoc";
}
function formatDarkMode(format43) {
    const isBootstrap = formatHasBootstrap(format43);
    if (isBootstrap) {
        return darkModeDefault(format43.metadata);
    }
    return undefined;
}
function darkModeDefault(metadata) {
    if (metadata !== undefined) {
        const theme = metadata[kTheme];
        if (theme && typeof theme === "object") {
            const keys6 = Object.keys(theme);
            if (keys6.includes("dark")) {
                if (keys6[0] === "dark") {
                    return true;
                } else {
                    return false;
                }
            }
        }
    }
    return undefined;
}
function formatPageLayout(format44) {
    return format44.metadata[kPageLayout] || kPageLayoutArticle;
}
function formatHasFullLayout(format45) {
    return format45.metadata[kPageLayout] === kPageLayoutFull;
}
function formatHasArticleLayout(format46) {
    return format46.metadata[kPageLayout] === undefined || format46.metadata[kPageLayout] === kPageLayoutArticle || format46.metadata[kPageLayout] === kPageLayoutFull;
}
function bootstrapFormatDependency() {
    const boostrapResource = (resource)=>formatResourcePath("html", join3("bootstrap", "dist", resource))
    ;
    const bootstrapDependency = (resource)=>({
            name: resource,
            path: boostrapResource(resource)
        })
    ;
    return {
        name: kBootstrapDependencyName,
        stylesheets: [
            bootstrapDependency("bootstrap-icons.css"), 
        ],
        scripts: [
            bootstrapDependency("bootstrap.min.js"), 
        ],
        resources: [
            bootstrapDependency("bootstrap-icons.woff"), 
        ]
    };
}
function boostrapExtras(input, flags, format47) {
    const toc = hasTableOfContents(flags, format47);
    const renderTemplate = (template3, pageLayout)=>{
        return renderEjs(formatResourcePath("html", `templates/${template3}`), {
            toc,
            pageLayout
        });
    };
    const pageLayout1 = formatPageLayout(format47);
    const bodyEnvelope = formatHasArticleLayout(format47) ? {
        before: renderTemplate("before-body-article.ejs", pageLayout1),
        afterPreamble: renderTemplate("after-body-article-preamble.ejs", kPageLayoutArticle),
        afterPostamble: renderTemplate("after-body-article-postamble.ejs", kPageLayoutArticle)
    } : {
        before: renderTemplate("before-body-custom.ejs", kPageLayoutCustom),
        afterPreamble: renderTemplate("after-body-custom-preamble.ejs", kPageLayoutCustom),
        afterPostamble: renderTemplate("after-body-custom-postamble.ejs", kPageLayoutCustom)
    };
    return {
        pandoc: {
            [kSectionDivs]: true,
            [kHtmlMathMethod]: "mathjax"
        },
        metadata: {
            [kDocumentCss]: false,
            [kLinkCitations]: true
        },
        html: {
            [kSassBundles]: resolveBootstrapScss(input, format47),
            [kDependencies]: [
                bootstrapFormatDependency()
            ],
            [kBodyEnvelope]: bodyEnvelope,
            [kHtmlPostprocessors]: [
                bootstrapHtmlPostprocessor(flags, format47), 
            ],
            [kHtmlFinalizers]: [
                bootstrapHtmlFinalizer(format47), 
            ]
        }
    };
}
const getColumnLayoutElements = (doc)=>{
    return doc.querySelectorAll('[class^="column-"], [class*=" column-"], aside, [class*="margin-caption"], [class*=" margin-caption"], [class*="margin-ref"], [class*=" margin-ref"]');
};
function bootstrapHtmlPostprocessor(flags, format48) {
    return (doc)=>{
        const title5 = doc.querySelector("header > .title");
        if (title5) {
            title5.classList.add("display-7");
        }
        const { citesInMargin , refsInMargin  } = processColumnElements(doc, format48, flags);
        const subtitle = doc.querySelector("header > .subtitle");
        if (subtitle) {
            subtitle.classList.add("lead");
        }
        const blockquotes = doc.querySelectorAll("blockquote");
        for(let i69 = 0; i69 < blockquotes.length; i69++){
            const classList = blockquotes[i69].classList;
            classList.add("blockquote");
        }
        const figures = doc.querySelectorAll("figure");
        for(let i1 = 0; i1 < figures.length; i1++){
            const figure = figures[i1];
            figure.classList.add("figure");
            const images = figure.querySelectorAll("img");
            for(let j = 0; j < images.length; j++){
                images[j].classList.add("figure-img");
            }
            const captions = figure.querySelectorAll("figcaption");
            for(let j1 = 0; j1 < captions.length; j1++){
                captions[j1].classList.add("figure-caption");
            }
        }
        const toc = doc.querySelector('nav[role="doc-toc"]');
        const tocTarget = doc.getElementById("quarto-toc-target");
        if (toc && tocTarget) {
            const tocLinks = doc.querySelectorAll('nav[role="doc-toc"] > ul a');
            for(let i70 = 0; i70 < tocLinks.length; i70++){
                const tocLink = tocLinks[i70];
                tocLink.classList.add("nav-link");
                if (i70 === 0) {
                    tocLink.classList.add("active");
                }
                if (!tocLink.hasAttribute("data-scroll-target")) {
                    tocLink.setAttribute("data-scroll-target", tocLink.getAttribute("href")?.replaceAll(":", "\\:"));
                }
            }
            const nestedUls = toc.querySelectorAll("ul ul");
            for(let i5 = 0; i5 < nestedUls.length; i5++){
                const ul = nestedUls[i5];
                ul.classList.add("collapse");
            }
            toc.remove();
            tocTarget.replaceWith(toc);
        } else {
            tocTarget?.remove();
        }
        const tableHeaders = doc.querySelectorAll("tbody > tr:first-child.odd");
        for(let i2 = 0; i2 < tableHeaders.length; i2++){
            const th = tableHeaders[i2];
            if (th.parentNode?.parentNode) {
                const table = th.parentNode.parentNode;
                table.removeAttribute("style");
                table.classList.add("table");
            }
        }
        const pandasTables = doc.querySelectorAll("table.dataframe");
        for(let i3 = 0; i3 < pandasTables.length; i3++){
            const table = pandasTables[i3];
            table.removeAttribute("border");
            table.classList.add("table");
            const headerRows = table.querySelectorAll("tr");
            for(let r = 0; r < headerRows.length; r++){
                headerRows[r].removeAttribute("style");
            }
            if (table.previousElementSibling && table.previousElementSibling.tagName === "STYLE") {
                table.previousElementSibling.remove();
            }
        }
        const sections = doc.querySelectorAll('section[class^="level"]');
        for(let i4 = 0; i4 < sections.length; i4++){
            const section = sections[i4];
            const heading = section.querySelector("h2") || section.querySelector("h3") || section.querySelector("h4") || section.querySelector("h5") || section.querySelector("h6");
            if (heading) {
                heading.setAttribute("data-anchor-id", section.id);
            }
        }
        const footnotes = doc.querySelectorAll('section[role="doc-endnotes"]');
        if (refsInMargin) {
            const footNoteSectionEl = doc.querySelector("section.footnotes");
            if (footNoteSectionEl) {
                footNoteSectionEl.remove();
            }
        } else if (footnotes.length === 1) {
            const footnotesEl = footnotes.item(0);
            insertFootnotesTitle(doc, footnotesEl, format48.language);
        }
        if (citesInMargin) {
            const bibliographyDiv = doc.querySelector("div#refs");
            if (bibliographyDiv) {
                bibliographyDiv.remove();
            }
        }
        return Promise.resolve([]);
    };
}
function bootstrapHtmlFinalizer(format49) {
    return (doc)=>{
        const fullLayout = formatHasFullLayout(format49);
        if (fullLayout) {
            const column = suggestColumn(doc);
            setMainColumn(doc, column);
        }
        const leftSidebar = doc.getElementById("quarto-sidebar");
        const hasLeftContent = leftSidebar && leftSidebar.children.length > 0;
        const rightSidebar = doc.getElementById("quarto-margin-sidebar");
        const hasRightContent = rightSidebar && rightSidebar.children.length > 0;
        if (rightSidebar && !hasRightContent) {
            rightSidebar.remove();
        }
        const hasColumnElements = getColumnLayoutElements(doc).length > 0;
        if (hasColumnElements) {
            if (hasLeftContent) {
                doc.body.classList.add("slimcontent");
            } else {}
        } else {
            if (!hasRightContent) {
                doc.body.classList.add("fullcontent");
            } else {}
        }
        return Promise.resolve();
    };
}
function processColumnElements(doc, format50, flags) {
    if (!formatHasArticleLayout(format50)) {
        return {
            citesInMargin: false,
            refsInMargin: false
        };
    }
    processMarginCaptions(doc);
    const marginProcessors = [
        simpleMarginProcessor, 
    ];
    const refsInMargin = format50.pandoc[kReferenceLocation] === "margin" || flags[kReferenceLocation] === "margin";
    if (refsInMargin) {
        marginProcessors.push(footnoteMarginProcessor);
    }
    const citesInMargin = format50.metadata[kCitationLocation] === "margin";
    if (citesInMargin) {
        marginProcessors.push(referenceMarginProcessor);
    }
    processMarginNodes(doc, marginProcessors);
    const columnLayouts = getColumnLayoutElements(doc);
    if (columnLayouts && columnLayouts.length > 0) {
        const ensureInGrid = (el, setLayout)=>{
            if (!el.classList.contains("page-columns")) {
                el.classList.add("page-columns");
            }
            if (setLayout && !el.classList.contains("page-full")) {
                el.classList.add("page-full");
            }
            if (el.tagName !== "MAIN") {
                const parent = el.parentElement;
                if (parent) {
                    ensureInGrid(parent, true);
                }
            }
        };
        columnLayouts.forEach((node)=>{
            const el = node;
            if (el.parentElement) {
                ensureInGrid(el.parentElement, true);
            }
        });
    }
    return {
        citesInMargin,
        refsInMargin
    };
}
const processMarginNodes = (doc, processors)=>{
    const marginSelector = processors.map((proc)=>proc.selector
    ).join(", ");
    const marginNodes = doc.querySelectorAll(marginSelector);
    marginNodes.forEach((marginNode)=>{
        const marginEl = marginNode;
        for (const processor of processors){
            if (processor.canProcess(marginEl)) {
                processor.process(marginEl, doc);
                break;
            }
        }
        marginEl.classList.remove("column-margin");
    });
};
const findQuartoFigure = (el)=>{
    if (el.classList.contains("quarto-figure")) {
        return el;
    } else if (el.parentElement) {
        return findQuartoFigure(el.parentElement);
    } else {
        return undefined;
    }
};
const moveClassToCaption = (container, sel)=>{
    const target = container.querySelector(sel);
    if (target) {
        target.classList.add("margin-caption");
        return true;
    } else {
        return false;
    }
};
const removeCaptionClass = (el)=>{
    el.classList.remove("margin-caption");
};
const processLayoutPanelMarginCaption = (captionContainer)=>{
    const figure = captionContainer.querySelector("figure");
    if (figure) {
        for (const child of figure.children){
            if (child.tagName === "FIGCAPTION") {
                child.classList.add("margin-caption");
                removeCaptionClass(captionContainer);
                break;
            }
        }
    } else {
        const caption = captionContainer.querySelector(".panel-caption");
        if (caption) {
            caption.classList.add("margin-caption");
            removeCaptionClass(captionContainer);
        }
    }
};
const processFigureMarginCaption = (captionContainer, doc)=>{
    const foundCaption = moveClassToCaption(captionContainer, "figcaption");
    if (!foundCaption) {
        const captionEl = captionContainer.querySelector("caption");
        if (captionEl) {
            const parentDivEl = captionEl?.parentElement?.parentElement;
            if (parentDivEl) {
                captionEl.classList.add("hidden");
                const divCopy = doc.createElement("div");
                divCopy.classList.add("figure-caption");
                divCopy.classList.add("margin-caption");
                divCopy.innerHTML = captionEl.innerHTML;
                parentDivEl.appendChild(divCopy);
                removeCaptionClass(captionContainer);
            }
        }
    } else {
        removeCaptionClass(captionContainer);
    }
};
const processMarginCaptions = (doc)=>{
    const marginCaptions = doc.querySelectorAll(".margin-caption");
    marginCaptions.forEach((node)=>{
        const figureEl = node;
        const captionContainer = findQuartoFigure(figureEl);
        if (captionContainer) {
            const isLayoutPanel = captionContainer.classList.contains("quarto-layout-panel");
            if (isLayoutPanel) {
                processLayoutPanelMarginCaption(captionContainer);
            } else {
                processFigureMarginCaption(captionContainer, doc);
            }
        }
        removeCaptionClass(figureEl);
    });
};
const simpleMarginProcessor = {
    selector: ".column-margin:not(.column-container)",
    canProcess (el) {
        return el.classList.contains("column-margin") && !el.classList.contains("column-container");
    },
    process (el, doc) {
        el.classList.remove("column-margin");
        addContentToMarginContainerForEl(el, el, doc);
    }
};
const footnoteMarginProcessor = {
    selector: ".footnote-ref",
    canProcess (el) {
        return el.classList.contains("footnote-ref");
    },
    process (el, doc) {
        if (el.hasAttribute("href")) {
            const target = el.getAttribute("href");
            if (target) {
                const refId = target.slice(1);
                const refContentsEl = doc.getElementById(refId);
                if (refContentsEl) {
                    const backLinkEl = refContentsEl.querySelector(".footnote-back");
                    if (backLinkEl) {
                        backLinkEl.remove();
                    }
                    if (refContentsEl.childNodes.length > 0) {
                        const firstChild = refContentsEl.childNodes[0];
                        firstChild.insertBefore(doc.createTextNode("\u00A0"), firstChild.firstChild);
                        firstChild.insertBefore(el.firstChild.cloneNode(true), firstChild.firstChild);
                    }
                    addContentToMarginContainerForEl(el, refContentsEl, doc);
                }
            }
        }
    }
};
const referenceMarginProcessor = {
    selector: "a[role='doc-biblioref']",
    canProcess (el) {
        return el.hasAttribute("role") && el.getAttribute("role") === "doc-biblioref";
    },
    process (el1, doc) {
        if (el1.hasAttribute("href")) {
            const target = el1.getAttribute("href");
            if (target) {
                const refId = target.slice(1);
                const refContentsEl = doc.getElementById(refId);
                const findCaptionEl = (el)=>{
                    if (el.parentElement?.tagName === "FIGCAPTION") {
                        return el.parentElement;
                    } else if (el.parentElement) {
                        return findCaptionEl(el.parentElement);
                    } else {
                        return undefined;
                    }
                };
                const parentCaptionEl = findCaptionEl(el1);
                if (refContentsEl && parentCaptionEl) {
                    addContentToMarginContainerForEl(parentCaptionEl, refContentsEl.cloneNode(true), doc);
                }
            }
        }
    }
};
const isContainer = (el)=>{
    return el && el.tagName === "DIV" && el.classList.contains("column-container") && el.classList.contains("column-margin");
};
const isAlreadyInMargin = (el)=>{
    const elInMargin = el.classList.contains("column-margin") || el.classList.contains("aside") || el.classList.contains("margin-caption");
    if (elInMargin) {
        return true;
    } else if (el.parentElement !== null) {
        return isAlreadyInMargin(el.parentElement);
    } else {
        return false;
    }
};
const createMarginContainer = (doc)=>{
    const container = doc.createElement("div");
    container.classList.add("no-row-height");
    container.classList.add("column-margin");
    container.classList.add("column-container");
    return container;
};
const marginContainerForEl = (el, doc)=>{
    if (el.parentElement && isAlreadyInMargin(el.parentElement)) {
        return el.parentElement;
    }
    if (el.nextElementSibling && isContainer(el.nextElementSibling)) {
        return el.nextElementSibling;
    }
    if (el.previousElementSibling && isContainer(el.previousElementSibling)) {
        return el.previousElementSibling;
    }
    const list = findOutermostParentElOfType(el, [
        "OL",
        "UL",
        "TABLE"
    ]);
    if (list) {
        if (list.nextElementSibling && isContainer(list.nextElementSibling)) {
            return list.nextElementSibling;
        } else {
            const container = createMarginContainer(doc);
            if (list.parentNode) {
                list.parentNode.insertBefore(container, list.nextElementSibling);
            }
            return container;
        }
    }
    const parentEl = el.parentElement;
    const cantContainBlockTags = [
        "P"
    ];
    if (parentEl && cantContainBlockTags.includes(parentEl.tagName)) {
        if (parentEl.parentElement && parentEl.parentElement.tagName === "DIV" && parentEl.nextElementSibling && isContainer(parentEl.nextElementSibling)) {
            return parentEl.nextElementSibling;
        } else {
            const container = createMarginContainer(doc);
            const wrapper = doc.createElement("div");
            parentEl.replaceWith(wrapper);
            wrapper.appendChild(parentEl);
            wrapper.appendChild(container);
            return container;
        }
    }
    const container = createMarginContainer(doc);
    el.parentNode?.insertBefore(container, el.nextElementSibling);
    return container;
};
const addContentToMarginContainerForEl = (el, content, doc)=>{
    const container = marginContainerForEl(el, doc);
    if (container) {
        container.appendChild(content);
    }
};
const findOutermostParentElOfType = (el, tagNames)=>{
    let outEl = undefined;
    if (el.parentElement) {
        if (el.parentElement.tagName === "MAIN") {
            return outEl;
        } else {
            if (tagNames.includes(el.parentElement.tagName)) {
                outEl = el.parentElement;
            }
            outEl = findOutermostParentElOfType(el.parentElement, tagNames) || outEl;
            return outEl;
        }
    } else {
        return undefined;
    }
};
function suggestColumn(doc) {
    const hasContents = (id)=>{
        const el = doc.getElementById(id);
        if (el === null) {
            return false;
        }
        if (el.children.length > 0) {
            return true;
        }
        return !!el.innerText.trim();
    };
    const leftSidebar = hasContents(kSidebarId);
    const rightSidebar = hasContents(kMarginSidebarId);
    const columnClasses = getColumnClasses(doc);
    const leftContent = [
        ...fullOccludeClz,
        ...leftOccludeClz
    ].some((clz)=>{
        return columnClasses.has(clz);
    });
    const rightContent = [
        ...fullOccludeClz,
        ...rightOccludeClz
    ].some((clz)=>{
        return columnClasses.has(clz);
    });
    const leftUsed = leftSidebar || leftContent;
    const rightUsed = rightSidebar || rightContent;
    if (leftUsed && rightUsed) {
        return "column-body";
    } else if (leftUsed) {
        return "column-page-right";
    } else if (rightUsed) {
        return "column-page-left";
    } else {
        return "column-page";
    }
}
const kSidebarId = "quarto-sidebar";
const kMarginSidebarId = "quarto-margin-sidebar";
const fullOccludeClz = [
    "column-page",
    "column-screen",
    "column-screen-inset", 
];
const leftOccludeClz = [
    "column-page-left",
    "column-screen-inset-left",
    "column-screen-left", 
];
const rightOccludeClz = [
    "column-margin",
    "column-page-right",
    "column-screen-inset-right",
    "column-screen-right",
    "margin-caption",
    "margin-ref", 
];
const getColumnClasses = (doc)=>{
    const classes = new Set();
    const colNodes = getColumnLayoutElements(doc);
    for (const colNode of colNodes){
        const colEl = colNode;
        colEl.classList.forEach((clz)=>{
            if (clz === "margin-caption" || clz === "margin-ref" || clz.startsWith("column-")) {
                classes.add(clz);
            }
        });
    }
    return classes;
};
const kHideAllCodeLinkId = "quarto-hide-all-code";
const kShowAllCodeLinkId = "quarto-show-all-code";
const kViewSourceLinkId = "quarto-view-source";
const kEmbeddedSourceClass = "quarto-embedded-source-code";
const kEmbeddedSourceModalId = kEmbeddedSourceClass + "-modal";
const kEmbeddedSourceModalLabelId = kEmbeddedSourceClass + "-modal-label";
const kKeepSourceSentinel = "quarto-executable-code-5450563D";
const kCodeToolsSourceButtonId = "quarto-code-tools-source";
const kCodeToolsMenuButtonId = "quarto-code-tools-menu";
const kDataQuartoSourceUrl = "data-quarto-source-url";
function formatHasCodeTools(format51) {
    const codeTools = format51.render?.[kCodeTools];
    return !!codeTools && isHtmlOutput(format51.pandoc, true) && formatHasBootstrap(format51);
}
function resolveKeepSource(format52, engine, target) {
    const codeTools = format52.render?.[kCodeTools];
    if (codeTools === true || typeof codeTools === "object" && (codeTools?.source === undefined || codeTools?.source === true)) {
        format52.render[kKeepSource] = true;
    }
    format52.render[kKeepSource] = format52.render[kKeepSource] && isHtmlOutput(format52.pandoc, true) && formatHasBootstrap(format52) && executionEngineCanKeepSource(engine, target);
}
function keepSourceBlock(format53, source) {
    if (format53.render[kKeepSource]) {
        let code30 = Deno.readTextFileSync(source).trimLeft();
        if (!code30.endsWith("\n")) {
            code30 = code30 + "\n";
        }
        code30 = code30.replaceAll(/\n```{(\w+)}\s*\n/g, "\n" + kKeepSourceSentinel + "\n\n```$1\n");
        const kKeepSourceBackticks = "```````````````````";
        return `${kMarkdownBlockSeparator}::: {.${kEmbeddedSourceClass}}\n${kKeepSourceBackticks}` + `{.markdown shortcodes="false"}\n${code30}` + `${kKeepSourceBackticks}\n:::\n`;
    } else {
        return "";
    }
}
function codeToolsPostprocessor(format54) {
    return (doc)=>{
        if (format54.render[kKeepSource]) {
            const lines9 = doc.querySelectorAll(`.${kEmbeddedSourceClass} > div.sourceCode > pre > code > span`);
            if (lines9.length > 0) {
                const newLines = [];
                for(let i71 = 0; i71 < lines9.length; i71++){
                    const line = lines9[i71];
                    if (line.innerText === kKeepSourceSentinel) {
                        i71 += 2;
                        const codeBlockLine = lines9[i71];
                        const codeSpan = codeBlockLine.lastChild;
                        if (codeSpan.innerHTML) {
                            codeSpan.innerHTML = codeSpan.innerHTML.replace(/```(\w+)/, "```{$1}");
                        }
                        newLines.push(codeBlockLine);
                    } else {
                        newLines.push(line);
                    }
                }
                if (newLines.length !== lines9.length) {
                    const parent = lines9[0].parentElement;
                    parent.innerHTML = "";
                    newLines.forEach((line)=>{
                        parent.appendChild(line);
                        parent.appendChild(doc.createTextNode("\n"));
                    });
                }
            }
        }
        if (formatHasCodeTools(format54)) {
            const codeTools = resolveCodeTools(format54, doc);
            if (codeTools.source || codeTools.toggle) {
                const title6 = doc.querySelector("#title-block-header > h1");
                if (title6) {
                    const header = title6.parentElement;
                    const titleDiv = doc.createElement("div");
                    titleDiv.classList.add("quarto-title-block");
                    const layoutDiv = doc.createElement("div");
                    titleDiv.appendChild(layoutDiv);
                    header?.replaceChild(titleDiv, title6);
                    layoutDiv.appendChild(title6);
                    const button = doc.createElement("button");
                    button.setAttribute("type", "button");
                    button.classList.add("btn").add("code-tools-button");
                    const icon = doc.createElement("i");
                    icon.classList.add("bi");
                    button.appendChild(icon);
                    if (codeTools.caption !== "none") {
                        button.appendChild(doc.createTextNode(" " + codeTools.caption));
                    }
                    layoutDiv.appendChild(button);
                    header.appendChild(titleDiv);
                    if (codeTools.toggle) {
                        button.setAttribute("id", kCodeToolsMenuButtonId);
                        button.classList.add("dropdown-toggle");
                        button.setAttribute("data-bs-toggle", "dropdown");
                        button.setAttribute("aria-expanded", "false");
                        const ul = doc.createElement("ul");
                        ul.classList.add("dropdown-menu").add("dropdown-menu-end");
                        ul.setAttribute("aria-labelelledby", kCodeToolsMenuButtonId);
                        const addListItem = (id, text)=>{
                            const a = doc.createElement("a");
                            a.setAttribute("id", id);
                            a.classList.add("dropdown-item");
                            a.setAttribute("href", "javascript:void(0)");
                            a.setAttribute("role", "button");
                            a.appendChild(doc.createTextNode(text));
                            const li = doc.createElement("li");
                            li.appendChild(a);
                            ul.appendChild(li);
                            return li;
                        };
                        const addDivider = ()=>{
                            const hr = doc.createElement("hr");
                            hr.classList.add("dropdown-divider");
                            const li = doc.createElement("li");
                            li.appendChild(hr);
                            ul.appendChild(li);
                        };
                        addListItem(kShowAllCodeLinkId, format54.language[kCodeToolsShowAllCode]);
                        addListItem(kHideAllCodeLinkId, format54.language[kCodeToolsHideAllCode]);
                        if (codeTools.source) {
                            addDivider();
                            const vsLi = addListItem(kViewSourceLinkId, format54.language[kCodeToolsViewSource]);
                            if (typeof codeTools.source === "string") {
                                vsLi.firstChild.setAttribute(kDataQuartoSourceUrl, codeTools.source);
                            }
                        }
                        layoutDiv.appendChild(ul);
                    } else {
                        button.setAttribute("id", kCodeToolsSourceButtonId);
                        if (typeof codeTools.source === "string") {
                            button.setAttribute(kDataQuartoSourceUrl, codeTools.source);
                        }
                    }
                }
                if (codeTools.source) {
                    const embeddedCode = doc.querySelector(`.${kEmbeddedSourceClass}`);
                    if (embeddedCode) {
                        const modalDiv = doc.createElement("div");
                        modalDiv.classList.add("modal").add("fade");
                        modalDiv.setAttribute("id", kEmbeddedSourceModalId);
                        modalDiv.setAttribute("tabindex", "-1");
                        modalDiv.setAttribute("aria-labelledby", kEmbeddedSourceModalLabelId);
                        modalDiv.setAttribute("aria-hidden", "true");
                        const modalDialogDiv = doc.createElement("div");
                        modalDialogDiv.classList.add("modal-dialog").add("modal-dialog-scrollable");
                        const modalContentDiv = doc.createElement("div");
                        modalContentDiv.classList.add("modal-content");
                        const modalDialogHeader = doc.createElement("div");
                        modalDialogHeader.classList.add("modal-header");
                        const h5 = doc.createElement("h5");
                        h5.classList.add("modal-title");
                        h5.setAttribute("id", kEmbeddedSourceModalLabelId);
                        h5.appendChild(doc.createTextNode(format54.language[kCodeToolsSourceCode]));
                        modalDialogHeader.appendChild(h5);
                        const button = doc.createElement("button");
                        button.classList.add("btn-close");
                        button.setAttribute("data-bs-dismiss", "modal");
                        modalDialogHeader.appendChild(button);
                        modalContentDiv.appendChild(modalDialogHeader);
                        const modalBody = doc.createElement("div");
                        modalBody.classList.add("modal-body");
                        modalContentDiv.appendChild(modalBody);
                        modalDialogDiv.appendChild(modalContentDiv);
                        modalDiv.appendChild(modalDialogDiv);
                        embeddedCode.parentElement?.insertBefore(modalDiv, embeddedCode);
                        modalBody.appendChild(embeddedCode);
                        embeddedCode.classList.delete(kEmbeddedSourceClass);
                    }
                }
            }
        }
        return Promise.resolve([]);
    };
}
function resolveCodeTools(format55, doc) {
    const kCodeCaption = format55.language[kCodeToolsMenuCaption];
    const codeTools = format55?.render[kCodeTools];
    const codeToolsResolved = {
        source: typeof codeTools === "boolean" ? codeTools : codeTools?.source !== undefined ? codeTools?.source : true,
        toggle: typeof codeTools === "boolean" ? codeTools : codeTools?.toggle !== undefined ? !!codeTools?.toggle : true,
        caption: typeof codeTools === "boolean" ? kCodeCaption : codeTools?.caption || kCodeCaption
    };
    if (codeToolsResolved.toggle) {
        const codeDetails = doc.querySelector(".cell > details > .sourceCode");
        const codeHidden = doc.querySelector(".cell .sourceCode.hidden");
        codeToolsResolved.toggle = codeToolsResolved.toggle && (!!codeDetails || !!codeHidden);
    }
    return codeToolsResolved;
}
function createFormat(ext, ...formats3) {
    return mergeConfigs(defaultFormat(), ...formats3, {
        render: {
            [kOutputExt]: ext
        }
    });
}
function createHtmlFormat(figwidth, figheight) {
    return createFormat("html", {
        metadata: {
            [kLang]: "en",
            [kFigResponsive]: true
        },
        execute: {
            [kFigFormat]: "retina",
            [kFigWidth]: figwidth,
            [kFigHeight]: figheight
        },
        render: {
            [kTblColwidths]: "auto"
        },
        pandoc: {
            [kStandalone]: true,
            [kWrap]: "none",
            [kDefaultImageExtension]: "png"
        }
    });
}
function createHtmlPresentationFormat(figwidth, figheight) {
    return mergeConfigs(createHtmlFormat(figwidth, figheight), {
        metadata: {
            [kFigResponsive]: false
        },
        execute: {
            [kEcho]: false,
            [kWarning]: false
        }
    });
}
function createEbookFormat(ext) {
    return createFormat(ext, {
        formatExtras: ()=>{
            return {
                [kIncludeInHeader]: [
                    formatResourcePath("html", "styles-callout.html"),
                    formatResourcePath("epub", "styles.html"), 
                ]
            };
        },
        render: {
            [kMergeIncludes]: false
        },
        execute: {
            [kFigWidth]: 5,
            [kFigHeight]: 4
        },
        pandoc: {
            [kDefaultImageExtension]: "png"
        }
    });
}
function createWordprocessorFormat(ext) {
    return createFormat(ext, {
        render: {
            [kPageWidth]: 6.5
        },
        execute: {
            [kFigWidth]: 5,
            [kFigHeight]: 4
        },
        pandoc: {
            [kDefaultImageExtension]: "png"
        }
    });
}
function defaultFormat() {
    return {
        execute: {
            [kFigWidth]: 7,
            [kFigHeight]: 5,
            [kFigFormat]: "png",
            [kFigDpi]: 96,
            [kError]: false,
            [kEval]: true,
            [kCache]: null,
            [kFreeze]: false,
            [kEcho]: true,
            [kOutput]: true,
            [kWarning]: true,
            [kInclude]: true,
            [kKeepMd]: false,
            [kKeepIpynb]: false,
            [kExecuteIpynb]: null,
            [kExecuteEnabled]: null,
            [kExecuteDaemon]: null,
            [kExecuteDaemonRestart]: false,
            [kExecuteDebug]: false
        },
        render: {
            [kKeepTex]: false,
            [kKeepYaml]: false,
            [kKeepSource]: false,
            [kKeepHidden]: false,
            [kPreferHtml]: false,
            [kOutputDivs]: true,
            [kOutputExt]: "html",
            [kFigAlign]: "default",
            [kFigPos]: null,
            [kFigEnv]: null,
            [kCodeFold]: "none",
            [kCodeOverflow]: "scroll",
            [kCodeLink]: false,
            [kCodeLineNumbers]: false,
            [kCodeTools]: false,
            [kTblColwidths]: true,
            [kMergeIncludes]: true,
            [kSelfContainedMath]: false,
            [kLatexAutoMk]: true,
            [kLatexAutoInstall]: true,
            [kLatexClean]: true,
            [kLatexMaxRuns]: 1,
            [kLatexMaxRuns]: 10,
            [kLatexMakeIndex]: "makeindex",
            [kLatexMakeIndexOpts]: [],
            [kLatexTlmgrOpts]: [],
            [kLatexOutputDir]: null,
            [kLinkExternalIcon]: false,
            [kLinkExternalNewwindow]: false
        },
        pandoc: {},
        language: {},
        metadata: {}
    };
}
function htmlFormat(figwidth, figheight) {
    return mergeConfigs(createHtmlFormat(figwidth, figheight), {
        resolveFormat: (format56)=>{
            if (format56.metadata[kMinimal] === true) {
                if (format56.metadata[kFigResponsive] === undefined) {
                    format56.metadata[kFigResponsive] = false;
                }
                if (format56.metadata[kTheme] === undefined) {
                    format56.metadata[kTheme] = "none";
                }
            }
        },
        formatExtras: (input, flags, format57, _libDir, temp)=>{
            const htmlFilterParams = htmlFormatFilterParams(format57);
            return mergeConfigs(htmlFormatExtras(format57, temp), themeFormatExtras(input, flags, format57), {
                [kFilterParams]: htmlFilterParams
            });
        },
        extensions: {
            book: {
                multiFile: true
            }
        }
    });
}
const kQuartoHtmlDependency = "quarto-html";
function htmlFormatExtras(format58, temp, featureDefaults, tippyOptions, scssOptions) {
    const bootstrap = formatHasBootstrap(format58);
    if (!featureDefaults) {
        featureDefaults = htmlFormatFeatureDefaults(format58);
    }
    if (!tippyOptions) {
        tippyOptions = {};
    }
    if (!tippyOptions.config) {
        tippyOptions.config = {};
    }
    if (!scssOptions) {
        scssOptions = {};
    }
    if (scssOptions.quartoBase === undefined) {
        scssOptions.quartoBase = true;
    }
    if (scssOptions.quartoCssVars === undefined) {
        scssOptions.quartoCssVars = true;
    }
    const scripts = [];
    const stylesheets = [];
    const sassBundles = [];
    const dependencies = [];
    const options = format58.metadata[kComments] ? {
        [kHypothesis]: format58.metadata[kComments][kHypothesis] || false,
        [kUtterances]: format58.metadata[kComments][kUtterances] || false
    } : {};
    options.codeLink = format58.metadata[kCodeLink] || false;
    if (featureDefaults.tabby) {
        options.tabby = format58.metadata[kTabsets] !== false;
    } else {
        options.tabby = format58.metadata[kTabsets] || false;
    }
    if (featureDefaults.copyCode) {
        options.copyCode = format58.metadata[kCodeCopy] !== false;
    } else {
        options.copyCode = format58.metadata[kCodeCopy] || false;
    }
    if (featureDefaults.anchors) {
        options.anchors = format58.metadata[kAnchorSections] !== false;
    } else {
        options.anchors = format58.metadata[kAnchorSections] || false;
    }
    if (featureDefaults.hoverCitations) {
        options.hoverCitations = format58.metadata[kCitationsHover] !== false;
    } else {
        options.hoverCitations = format58.metadata[kCitationsHover] || false;
    }
    if (featureDefaults.hoverFootnotes) {
        options.hoverFootnotes = format58.metadata[kFootnotesHover] !== false;
    } else {
        options.hoverFootnotes = format58.metadata[kFootnotesHover] || false;
    }
    if (featureDefaults.figResponsive) {
        options.figResponsive = format58.metadata[kFigResponsive] !== false;
    } else {
        options.figResponsive = format58.metadata[kFigResponsive] || false;
    }
    options.codeTools = formatHasCodeTools(format58);
    options.darkMode = formatDarkMode(format58);
    options.linkExternalIcon = format58.render[kLinkExternalIcon];
    options.linkExternalNewwindow = format58.render[kLinkExternalNewwindow];
    options.linkExternalFilter = format58.render[kLinkExternalFilter];
    const siteMetadata = format58.metadata[kWebsite];
    if (!options.linkExternalFilter && siteMetadata) {
        const siteUrl = siteMetadata[kSiteUrl];
        if (siteUrl) {
            options.linkExternalFilter = siteUrl.replaceAll(".", "\\.").replaceAll("/", "\\/");
        }
    }
    if (bootstrap) {
        scripts.push({
            name: "quarto.js",
            path: formatResourcePath("html", "quarto.js")
        });
    }
    if (options.tabby) {
        scripts.push({
            name: "tabby.min.js",
            path: formatResourcePath("html", join3("tabby", "js", "tabby.js"))
        });
    }
    options.tippy = options.hoverCitations || options.hoverFootnotes;
    if (bootstrap || options.tippy) {
        scripts.push({
            name: "popper.min.js",
            path: formatResourcePath("html", join3("popper", "popper.min.js"))
        });
    }
    if (options.tippy) {
        scripts.push({
            name: "tippy.umd.min.js",
            path: formatResourcePath("html", join3("tippy", "tippy.umd.min.js"))
        });
        stylesheets.push({
            name: "tippy.css",
            path: formatResourcePath("html", join3("tippy", "tippy.css"))
        });
        if (tippyOptions.theme === undefined) {
            if (bootstrap) {
                tippyOptions.theme = "quarto";
                sassBundles.push({
                    key: "tippy.scss",
                    dependency: kBootstrapDependencyName,
                    quarto: {
                        functions: "",
                        defaults: "",
                        mixins: "",
                        rules: Deno.readTextFileSync(formatResourcePath("html", join3("tippy", "_tippy.scss")))
                    }
                });
            } else {
                tippyOptions.theme = "light-border";
                stylesheets.push({
                    name: "light-border.css",
                    path: formatResourcePath("html", join3("tippy", "light-border.css"))
                });
            }
        }
    }
    options.tippyOptions = tippyOptions;
    if (options.copyCode) {
        dependencies.push(clipboardDependency());
    }
    if (options.anchors) {
        scripts.push({
            name: "anchor.min.js",
            path: formatResourcePath("html", join3("anchor", "anchor.min.js"))
        });
        options.anchors = typeof options.anchors === "string" ? options.anchors : true;
    }
    if (!bootstrap) {
        if (scssOptions.quartoBase) {
            sassBundles.push({
                dependency: kQuartoHtmlDependency,
                key: kQuartoHtmlDependency,
                quarto: quartoBaseLayer(format58, !!options.copyCode, !!options.tabby, !!options.figResponsive)
            });
        }
        if (scssOptions.quartoCssVars) {
            sassBundles.push({
                dependency: kQuartoHtmlDependency,
                key: kQuartoHtmlDependency,
                quarto: {
                    defaults: "",
                    functions: "",
                    mixins: "",
                    rules: quartoGlobalCssVariableRules()
                }
            });
        }
    }
    const includeInHeader = [];
    if (options.hypothesis) {
        const hypothesisHeader = temp.createFile({
            suffix: ".html"
        });
        Deno.writeTextFileSync(hypothesisHeader, renderEjs(formatResourcePath("html", join3("hypothesis", "hypothesis.ejs")), {
            hypothesis: options.hypothesis
        }));
        includeInHeader.push(hypothesisHeader);
    }
    const includeAfterBody = [];
    const quartoHtmlRequired = Object.keys(options).some((option)=>!!options[option]
    );
    if (quartoHtmlRequired) {
        const quartoHtmlScript = temp.createFile();
        Deno.writeTextFileSync(quartoHtmlScript, renderEjs(formatResourcePath("html", join3("templates", "quarto-html.ejs")), options));
        includeAfterBody.push(quartoHtmlScript);
    }
    if (options.utterances) {
        if (typeof options.utterances !== "object") {
            throw new Error("Invalid utterances configuration (must provide a repo");
        }
        const utterances = options.utterances;
        if (!utterances["repo"]) {
            throw new Error("Invalid utterances coniguration (must provide a repo)");
        }
        utterances["issue-term"] = utterances["issue-term"] || "pathname";
        utterances["theme"] = utterances["theme"] || "github-light";
        const utterancesAfterBody = temp.createFile({
            suffix: ".html"
        });
        Deno.writeTextFileSync(utterancesAfterBody, renderEjs(formatResourcePath("html", join3("utterances", "utterances.ejs")), {
            utterances
        }));
        includeAfterBody.push(utterancesAfterBody);
    }
    dependencies.push({
        name: kQuartoHtmlDependency,
        scripts,
        stylesheets
    });
    return {
        [kIncludeInHeader]: includeInHeader,
        [kIncludeAfterBody]: includeAfterBody,
        html: {
            [kDependencies]: dependencies,
            [kSassBundles]: sassBundles,
            [kHtmlPostprocessors]: [
                htmlFormatPostprocessor(format58, featureDefaults)
            ]
        }
    };
}
const kFormatHasBootstrap = "has-bootstrap";
function htmlFormatFilterParams(format59) {
    return {
        [kFormatHasBootstrap]: formatHasBootstrap(format59)
    };
}
function htmlFormatFeatureDefaults(format60) {
    const bootstrap = formatHasBootstrap(format60);
    const minimal = format60.metadata[kMinimal] === true;
    return {
        tabby: !minimal && !bootstrap,
        copyCode: !minimal,
        anchors: !minimal,
        hoverCitations: !minimal,
        hoverFootnotes: !minimal,
        figResponsive: !minimal
    };
}
function htmlFormatPostprocessor(format61, featureDefaults) {
    const haveBootstrap = formatHasBootstrap(format61);
    if (!featureDefaults) {
        featureDefaults = htmlFormatFeatureDefaults(format61);
    }
    const codeCopy = featureDefaults.copyCode ? format61.metadata[kCodeCopy] !== false : format61.metadata[kCodeCopy] || false;
    const anchors = featureDefaults.anchors ? format61.metadata[kAnchorSections] !== false : format61.metadata[kAnchorSections] || false;
    return (doc)=>{
        const codeBlocks = doc.querySelectorAll("pre.sourceCode");
        for(let i2 = 0; i2 < codeBlocks.length; i2++){
            const code31 = codeBlocks[i2];
            if (code31.classList.contains("hidden")) {
                code31.classList.delete("hidden");
                code31.parentElement?.classList.add("hidden");
            }
            if (codeCopy) {
                code31.classList.add("code-with-copy");
                const copyButton = doc.createElement("button");
                const title7 = format61.language[kCopyButtonTooltip];
                copyButton.setAttribute("title", title7);
                copyButton.classList.add("code-copy-button");
                const copyIcon = doc.createElement("i");
                copyIcon.classList.add("bi");
                copyButton.appendChild(copyIcon);
                code31.appendChild(copyButton);
            }
        }
        if (anchors) {
            const container = haveBootstrap ? doc.querySelector("main") : doc.querySelector("body");
            if (container) {
                [
                    "h2",
                    "h3",
                    "h4",
                    "h5",
                    "h6",
                    ".quarto-figure[id]",
                    "div[id^=tbl-]"
                ].forEach((selector)=>{
                    const headings = container.querySelectorAll(selector);
                    for(let i72 = 0; i72 < headings.length; i72++){
                        const heading = headings[i72];
                        if (heading.id !== "toc-title") {
                            if (!heading.classList.contains("no-anchor")) {
                                heading.classList.add("anchored");
                            }
                        }
                    }
                });
            }
        }
        const headerSections = doc.querySelectorAll(".header-section-number");
        for(let i1 = 0; i1 < headerSections.length; i1++){
            const secNumber = headerSections[i1];
            const prevElement = secNumber.previousElementSibling;
            if (prevElement && prevElement.classList.contains("toc-section-number")) {
                prevElement.remove();
            }
        }
        const doi = format61.metadata[kDoi];
        if (typeof doi === "string") {
            const header = doc.getElementById("title-block-header");
            if (header) {
                const p = doc.createElement("p");
                p.classList.add("doi");
                const doiUrl = `https://doi.org/${doi}`;
                const a = doc.createElement("a");
                a.innerHTML = doiUrl;
                a.setAttribute("href", doiUrl);
                p.appendChild(a);
                header.appendChild(p);
            }
        }
        return Promise.resolve([]);
    };
}
function themeFormatExtras(input, flags, format62) {
    const theme = format62.metadata[kTheme];
    if (theme === "none") {
        return {
            metadata: {
                [kDocumentCss]: false
            }
        };
    } else if (theme === "pandoc") {
        return pandocExtras(format62);
    } else {
        return boostrapExtras(input, flags, format62);
    }
}
function pandocExtras(format63) {
    const maxWidth = format63.metadata["max-width"];
    const headerIncludes = maxWidth ? `<style type="text/css">body { max-width: ${asCssSize(maxWidth)};}</style>` : undefined;
    return {
        metadata: {
            [kDocumentCss]: true,
            [kHeaderIncludes]: headerIncludes
        }
    };
}
async function resolveSassBundles(extras, pandoc, temp, formatBundles, projectBundles, project) {
    extras = cloneDeep1(extras);
    const mergedBundles = {};
    const group = (bundles, groupedBundles)=>{
        bundles.forEach((bundle)=>{
            if (!groupedBundles[bundle.dependency]) {
                groupedBundles[bundle.dependency] = [];
            }
            groupedBundles[bundle.dependency].push(bundle);
        });
    };
    if (projectBundles) {
        group(projectBundles, mergedBundles);
    }
    if (formatBundles) {
        group(formatBundles, mergedBundles);
    }
    let hasDarkStyles = false;
    let defaultStyle = undefined;
    for (const dependency of Object.keys(mergedBundles)){
        const bundles = mergedBundles[dependency];
        const hasDark = bundles.some((bundle)=>bundle.dark !== undefined
        );
        defaultStyle = bundles.some((bundle)=>bundle.dark !== undefined && bundle.dark.default
        ) ? "dark" : "light";
        const targets = [
            {
                name: `${dependency}.min.css`,
                bundles,
                attribs: {}
            }
        ];
        if (hasDark) {
            targets[0].attribs = {
                ...targets[0].attribs,
                ...attribForThemeStyle("light", defaultStyle)
            };
            const darkBundles = bundles.map((bundle)=>{
                bundle = cloneDeep1(bundle);
                bundle.user = bundle.dark?.user || bundle.user;
                bundle.quarto = bundle.dark?.quarto || bundle.quarto;
                bundle.framework = bundle.dark?.framework || bundle.framework;
                bundle.key = bundle.key + "-dark";
                return bundle;
            });
            targets.push({
                name: `${dependency}-dark.min.css`,
                bundles: darkBundles,
                attribs: attribForThemeStyle("dark", defaultStyle)
            });
            hasDarkStyles = true;
        }
        for (const target of targets){
            let cssPath = await compileSass(target.bundles, temp);
            cssPath = processCssIntoExtras(cssPath, extras, temp);
            const css = Deno.readTextFileSync(cssPath);
            const toDependencies = (paths)=>{
                return paths.map((path81)=>{
                    return {
                        name: path81,
                        path: project ? join3(project.dir, path81) : path81,
                        attribs: target.attribs
                    };
                });
            };
            const resources = toDependencies(cssResources(css));
            const imports = toDependencies(cssImports(css));
            const extraDeps = extras.html?.[kDependencies];
            if (extraDeps) {
                const existingDependency = extraDeps.find((extraDep)=>extraDep.name === dependency
                );
                if (existingDependency) {
                    if (!existingDependency.stylesheets) {
                        existingDependency.stylesheets = [];
                    }
                    existingDependency.stylesheets.push({
                        name: target.name,
                        path: cssPath,
                        attribs: target.attribs
                    });
                    existingDependency.stylesheets.push(...imports);
                    existingDependency.resources?.push(...resources);
                } else {
                    extraDeps.push({
                        name: dependency,
                        stylesheets: [
                            {
                                name: target.name,
                                path: cssPath,
                                attribs: target.attribs
                            },
                            ...imports
                        ],
                        resources
                    });
                }
            }
        }
    }
    extras = await resolveQuartoSyntaxHighlighting(extras, pandoc, temp, hasDarkStyles ? "light" : "default", defaultStyle);
    if (hasDarkStyles) {
        extras = await resolveQuartoSyntaxHighlighting(extras, pandoc, temp, "dark", defaultStyle);
    }
    setTextHighlightStyle("none", extras);
    return extras;
}
function cssHasDarkModeSentinel(css) {
    return !!css.match(/\/\*! dark \*\//g);
}
async function resolveQuartoSyntaxHighlighting(extras, pandoc, temp, style, defaultStyle) {
    extras = cloneDeep1(extras);
    const mediaAttr = attribForThemeStyle(style, defaultStyle);
    if (style === "default") {
        if (extras.html?.[kTextHighlightingMode] === "dark") {
            style = "dark";
        }
    }
    const cssFileName = `quarto-syntax-highlighting${style === "dark" ? "-dark" : ""}.css`;
    const theme = pandoc[kHighlightStyle] || kDefaultHighlightStyle;
    if (theme) {
        const themeRaw = readTheme(theme, style);
        if (themeRaw) {
            const themeJson = JSON.parse(themeRaw);
            const extraVariables = extras.html?.[kQuartoCssVariables] || [];
            const highlightCss = generateThemeCssVars(themeJson);
            if (highlightCss) {
                const rules = [
                    highlightCss,
                    "",
                    "/* other quarto variables */",
                    ...extraVariables, 
                ];
                const textHighlightCssRules = generateThemeCssClasses(themeJson);
                if (textHighlightCssRules) {
                    rules.push(...textHighlightCssRules);
                }
                const highlightCssPath = await compileSass([
                    {
                        key: cssFileName,
                        quarto: {
                            defaults: "",
                            functions: "",
                            mixins: "",
                            rules: rules.join("\n")
                        }
                    }
                ], temp, false);
                const extraDeps = extras.html?.[kDependencies];
                if (extraDeps) {
                    const existingDependency = extraDeps.find((extraDep)=>extraDep.name === kQuartoHtmlDependency
                    );
                    if (existingDependency) {
                        existingDependency.stylesheets = existingDependency.stylesheets || [];
                        existingDependency.stylesheets.push({
                            name: cssFileName,
                            path: highlightCssPath,
                            attribs: mediaAttr
                        });
                    }
                }
            }
        }
    }
    return extras;
}
function generateThemeCssVars(themeJson) {
    const textStyles = themeJson["text-styles"];
    if (textStyles) {
        const lines10 = [];
        lines10.push("/* quarto syntax highlight colors */");
        lines10.push(":root {");
        Object.keys(textStyles).forEach((styleName)=>{
            const abbr = kAbbrevs[styleName];
            if (abbr) {
                const textValues = textStyles[styleName];
                Object.keys(textValues).forEach((textAttr)=>{
                    switch(textAttr){
                        case "text-color":
                            lines10.push(`  --quarto-hl-${abbr}-color: ${textValues[textAttr] || "inherit"};`);
                            break;
                    }
                });
            }
        });
        lines10.push("}");
        return lines10.join("\n");
    }
    return undefined;
}
function generateThemeCssClasses(themeJson) {
    const textStyles = themeJson["text-styles"];
    if (textStyles) {
        const lines11 = [];
        Object.keys(textStyles).forEach((styleName)=>{
            const abbr = kAbbrevs[styleName];
            if (abbr !== undefined) {
                const textValues = textStyles[styleName];
                const cssValues = generateCssKeyValues(textValues);
                if (abbr !== "") {
                    lines11.push(`\ncode span.${abbr} {`);
                    lines11.push(...cssValues);
                    lines11.push("}\n");
                } else {
                    [
                        "code span",
                        "div.sourceCode"
                    ].forEach((selector)=>{
                        lines11.push(`\n${selector} {`);
                        lines11.push(...cssValues);
                        lines11.push("}\n");
                    });
                }
            }
        });
        return lines11;
    }
    return undefined;
}
function processCssIntoExtras(cssPath, extras, temp) {
    extras.html = extras.html || {};
    const css = Deno.readTextFileSync(cssPath);
    if (!extras.html[kTextHighlightingMode] && cssHasDarkModeSentinel(css)) {
        setTextHighlightStyle("dark", extras);
    }
    const matches = css.matchAll(kVariablesRegex);
    if (matches) {
        extras.html[kQuartoCssVariables] = extras.html[kQuartoCssVariables] || [];
        let dirty = false;
        for (const match of matches){
            const variables = match[1];
            extras.html[kQuartoCssVariables]?.push(variables);
            dirty = true;
        }
        extras.html[kQuartoCssVariables] = uniqBy1(extras.html[kQuartoCssVariables], (val)=>{
            return val;
        });
        if (dirty) {
            const cleanedCss = css.replaceAll(kVariablesRegex, "");
            const newCssPath = temp.createFile({
                suffix: ".css"
            });
            Deno.writeTextFileSync(newCssPath, cleanedCss);
            return newCssPath;
        }
    }
    return cssPath;
}
const kVariablesRegex = /\/\*\! quarto-variables-start \*\/([\S\s]*)\/\*\! quarto-variables-end \*\//g;
function generateCssKeyValues(textValues) {
    const lines12 = [];
    Object.keys(textValues).forEach((textAttr)=>{
        switch(textAttr){
            case "text-color":
                lines12.push(`color: ${textValues[textAttr]};`);
                break;
            case "background":
                lines12.push(`background-color: ${textValues[textAttr]};`);
                break;
            case "bold":
                if (textValues[textAttr]) {
                    lines12.push("font-weight: bold;");
                }
                break;
            case "italic":
                if (textValues[textAttr]) {
                    lines12.push("font-style: italic;");
                }
                break;
            case "underline":
                if (textValues[textAttr]) {
                    lines12.push("text-decoration: underline;");
                }
                break;
        }
    });
    return lines12;
}
const kAbbrevs = {
    "Keyword": "kw",
    "DataType": "dt",
    "DecVal": "dv",
    "BaseN": "bn",
    "Float": "fl",
    "Char": "ch",
    "String": "st",
    "Comment": "co",
    "Other": "ot",
    "Alert": "al",
    "Function": "fu",
    "RegionMarker": "re",
    "Error": "er",
    "Constant": "cn",
    "SpecialChar": "sc",
    "VerbatimString": "vs",
    "SpecialString": "ss",
    "Import": "im",
    "Documentation": "do",
    "Annotation": "an",
    "CommentVar": "cv",
    "Variable": "va",
    "ControlFlow": "cf",
    "Operator": "op",
    "BuiltIn": "bu",
    "Extension": "ex",
    "Preprocessor": "pp",
    "Attribute": "at",
    "Information": "in",
    "Warning": "wa",
    "Normal": ""
};
function attribForThemeStyle(style, defaultStyle) {
    const colorModeAttrs = (mode, disabled)=>{
        const attr = {
            class: `quarto-color-scheme${mode === "dark" ? " quarto-color-alternate" : ""}`
        };
        if (disabled) {
            attr.rel = "prefetch";
        }
        return attr;
    };
    switch(style){
        case "dark":
            return colorModeAttrs("dark", defaultStyle !== "dark");
        case "light":
            return colorModeAttrs("light", false);
        case "default":
        default:
            return {};
    }
}
function setTextHighlightStyle(style, extras) {
    extras.html = extras.html || {};
    extras.html[kTextHighlightingMode] = style;
}
function readTheme(theme, style) {
    const themeFile = textHighlightThemePath(theme, style === "default" ? undefined : style);
    if (themeFile && existsSync(themeFile)) {
        return Deno.readTextFileSync(themeFile);
    } else {
        return undefined;
    }
}
const importMeta1 = {
    url: "https://deno.land/x/deno_dom@v0.1.20-alpha/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder('utf-8');
const encodeString = typeof cachedTextEncoder.encodeInto === 'function' ? function(arg, view) {
    return cachedTextEncoder.encodeInto(arg, view);
} : function(arg, view) {
    const buf = cachedTextEncoder.encode(arg);
    view.set(buf);
    return {
        read: arg.length,
        written: buf.length
    };
};
function passStringToWasm0(arg, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len4 = arg.length;
    let ptr = malloc(len4);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len4; offset++){
        const code32 = arg.charCodeAt(offset);
        if (code32 > 127) break;
        mem[ptr + offset] = code32;
    }
    if (offset !== len4) {
        if (offset !== 0) {
            arg = arg.slice(offset);
        }
        ptr = realloc(ptr, len4, len4 = offset + arg.length * 3);
        const view = getUint8Memory0().subarray(ptr + offset, ptr + len4);
        const ret = encodeString(arg, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder('utf-8', {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len5) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len5));
}
function parse4(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load1(module, imports) {
    if (typeof Response === 'function' && module instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === 'function') {
            try {
                return await WebAssembly.instantiateStreaming(module, imports);
            } catch (e) {
                if (module.headers.get('Content-Type') != 'application/wasm') {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module
            };
        } else {
            return instance;
        }
    }
}
async function init(input) {
    if (typeof input === 'undefined') {
        input = new URL('deno-wasm_bg.wasm', importMeta1.url);
    }
    const imports = {};
    imports.env = {
        now () {}
    };
    input = Uint8Array.from(atob("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