import { ascend, BSTree } from "./bs_tree.ts";
import { RBNode } from "./rb_node.ts";
export * from "./_comparators.ts";
export class RBTree extends BSTree {
    constructor(compare = ascend) {
        super(compare);
    }
    static from(collection, options) {
        let result;
        let unmappedValues = [];
        if (collection instanceof RBTree) {
            result = new RBTree(options?.compare ?? collection.compare);
            if (options?.compare || options?.map) {
                unmappedValues = collection;
            }
            else {
                const nodes = [];
                if (collection.root) {
                    result.root = RBNode.from(collection.root);
                    nodes.push(result.root);
                }
                while (nodes.length) {
                    const node = nodes.pop();
                    const left = node.left
                        ? RBNode.from(node.left)
                        : null;
                    const right = node.right
                        ? RBNode.from(node.right)
                        : null;
                    if (left) {
                        left.parent = node;
                        nodes.push(left);
                    }
                    if (right) {
                        right.parent = node;
                        nodes.push(right);
                    }
                }
            }
        }
        else {
            result = (options?.compare
                ? new RBTree(options.compare)
                : new RBTree());
            unmappedValues = collection;
        }
        const values = options?.map
            ? Array.from(unmappedValues, options.map, options.thisArg)
            : unmappedValues;
        for (const value of values)
            result.insert(value);
        return result;
    }
    removeFixup(parent, current) {
        while (parent && !current?.red) {
            const direction = parent.left === current ? "left" : "right";
            const siblingDirection = direction === "right"
                ? "left"
                : "right";
            let sibling = parent[siblingDirection];
            if (sibling?.red) {
                sibling.red = false;
                parent.red = true;
                this.rotateNode(parent, direction);
                sibling = parent[siblingDirection];
            }
            if (sibling) {
                if (!sibling.left?.red && !sibling.right?.red) {
                    sibling.red = true;
                    current = parent;
                    parent = current.parent;
                }
                else {
                    if (!sibling[siblingDirection]?.red) {
                        sibling[direction].red = false;
                        sibling.red = true;
                        this.rotateNode(sibling, siblingDirection);
                        sibling = parent[siblingDirection];
                    }
                    sibling.red = parent.red;
                    parent.red = false;
                    sibling[siblingDirection].red = false;
                    this.rotateNode(parent, direction);
                    current = this.root;
                    parent = null;
                }
            }
        }
        if (current)
            current.red = false;
    }
    insert(value) {
        let node = this.insertNode(RBNode, value);
        if (node) {
            while (node.parent?.red) {
                let parent = node.parent;
                const parentDirection = parent.directionFromParent();
                const uncleDirection = parentDirection === "right"
                    ? "left"
                    : "right";
                const uncle = parent.parent[uncleDirection] ?? null;
                if (uncle?.red) {
                    parent.red = false;
                    uncle.red = false;
                    parent.parent.red = true;
                    node = parent.parent;
                }
                else {
                    if (node === parent[uncleDirection]) {
                        node = parent;
                        this.rotateNode(node, parentDirection);
                        parent = node.parent;
                    }
                    parent.red = false;
                    parent.parent.red = true;
                    this.rotateNode(parent.parent, uncleDirection);
                }
            }
            this.root.red = false;
        }
        return !!node;
    }
    remove(value) {
        const node = this.removeNode(value);
        if (node && !node.red) {
            this.removeFixup(node.parent, node.left ?? node.right);
        }
        return !!node;
    }
}
//# sourceMappingURL=data:application/json;base64,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