const rfc1924 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!#$%&()*+-;<=>?@^_`{|}~";
const Z85 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-:+=^!/*?&<>()[]{}@%$#";
export function encode(uint8, options) {
    const standard = options?.standard ?? "Adobe";
    let output = [], v, n = 0, difference = 0;
    if (uint8.length % 4 !== 0) {
        const tmp = uint8;
        difference = 4 - (tmp.length % 4);
        uint8 = new Uint8Array(tmp.length + difference);
        uint8.set(tmp);
    }
    const view = new DataView(uint8.buffer);
    for (let i = 0, len = uint8.length; i < len; i += 4) {
        v = view.getUint32(i);
        if ((standard === "Adobe" || standard === "btoa") &&
            v === 0 &&
            i < len - difference - 3) {
            output[n++] = "z";
            continue;
        }
        if (standard === "btoa" && v === 538976288) {
            output[n++] = "y";
            continue;
        }
        for (let j = 4; j >= 0; j--) {
            output[n + j] = String.fromCharCode((v % 85) + 33);
            v = Math.trunc(v / 85);
        }
        n += 5;
    }
    switch (standard) {
        case "Adobe":
            if (options?.delimiter) {
                return `<~${output.slice(0, output.length - difference).join("")}~>`;
            }
            break;
        case "btoa":
            if (options?.delimiter) {
                return `xbtoa Begin\n${output
                    .slice(0, output.length - difference)
                    .join("")}\nxbtoa End`;
            }
            break;
        case "RFC 1924":
            output = output.map((val) => rfc1924[val.charCodeAt(0) - 33]);
            break;
        case "Z85":
            output = output.map((val) => Z85[val.charCodeAt(0) - 33]);
            break;
    }
    return output.slice(0, output.length - difference).join("");
}
export function decode(ascii85, options) {
    const encoding = options?.standard ?? "Adobe";
    switch (encoding) {
        case "Adobe":
            ascii85 = ascii85.replaceAll(/(<~|~>)/g, "").replaceAll("z", "!!!!!");
            break;
        case "btoa":
            ascii85 = ascii85
                .replaceAll(/(xbtoa Begin|xbtoa End|\n)/g, "")
                .replaceAll("z", "!!!!!")
                .replaceAll("y", "+<VdL");
            break;
        case "RFC 1924":
            ascii85 = ascii85.replaceAll(/./g, (match) => String.fromCharCode(rfc1924.indexOf(match) + 33));
            break;
        case "Z85":
            ascii85 = ascii85.replaceAll(/./g, (match) => String.fromCharCode(Z85.indexOf(match) + 33));
            break;
    }
    ascii85 = ascii85.replaceAll(/[^!-u]/g, "");
    const len = ascii85.length, output = new Uint8Array(len + 4 - (len % 4));
    const view = new DataView(output.buffer);
    let v = 0, n = 0, max = 0;
    for (let i = 0; i < len;) {
        for (max += 5; i < max; i++) {
            v = v * 85 + (i < len ? ascii85.charCodeAt(i) : 117) - 33;
        }
        view.setUint32(n, v);
        v = 0;
        n += 4;
    }
    return output.slice(0, Math.trunc(len * 0.8));
}
//# sourceMappingURL=data:application/json;base64,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