import { assert } from "../../_util/assert.ts";
export const defaultReadOptions = {
    separator: ",",
    trimLeadingSpace: false,
};
export async function readRecord(startLine, reader, opt = defaultReadOptions) {
    const line = await reader.readLine();
    if (line === null)
        return null;
    if (line.length === 0) {
        return [];
    }
    return parseRecord(line, reader, opt, startLine, startLine + 1);
}
export async function parseRecord(line, reader, opt = defaultReadOptions, startLine, lineIndex = startLine) {
    if (opt.comment && line[0] === opt.comment) {
        return [];
    }
    assert(opt.separator != null);
    let fullLine = line;
    let quoteError = null;
    const quote = '"';
    const quoteLen = quote.length;
    const separatorLen = opt.separator.length;
    let recordBuffer = "";
    const fieldIndexes = [];
    parseField: for (;;) {
        if (opt.trimLeadingSpace) {
            line = line.trimStart();
        }
        if (line.length === 0 || !line.startsWith(quote)) {
            const i = line.indexOf(opt.separator);
            let field = line;
            if (i >= 0) {
                field = field.substring(0, i);
            }
            if (!opt.lazyQuotes) {
                const j = field.indexOf(quote);
                if (j >= 0) {
                    const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
                    quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
                    break parseField;
                }
            }
            recordBuffer += field;
            fieldIndexes.push(recordBuffer.length);
            if (i >= 0) {
                line = line.substring(i + separatorLen);
                continue parseField;
            }
            break parseField;
        }
        else {
            line = line.substring(quoteLen);
            for (;;) {
                const i = line.indexOf(quote);
                if (i >= 0) {
                    recordBuffer += line.substring(0, i);
                    line = line.substring(i + quoteLen);
                    if (line.startsWith(quote)) {
                        recordBuffer += quote;
                        line = line.substring(quoteLen);
                    }
                    else if (line.startsWith(opt.separator)) {
                        line = line.substring(separatorLen);
                        fieldIndexes.push(recordBuffer.length);
                        continue parseField;
                    }
                    else if (0 === line.length) {
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    else if (opt.lazyQuotes) {
                        recordBuffer += quote;
                    }
                    else {
                        const col = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                }
                else if (line.length > 0 || !(await reader.isEOF())) {
                    recordBuffer += line;
                    const r = await reader.readLine();
                    lineIndex++;
                    line = r ?? "";
                    fullLine = line;
                    if (r === null) {
                        if (!opt.lazyQuotes) {
                            const col = runeCount(fullLine);
                            quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                            break parseField;
                        }
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    recordBuffer += "\n";
                }
                else {
                    if (!opt.lazyQuotes) {
                        const col = runeCount(fullLine);
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                    fieldIndexes.push(recordBuffer.length);
                    break parseField;
                }
            }
        }
    }
    if (quoteError) {
        throw quoteError;
    }
    const result = [];
    let preIdx = 0;
    for (const i of fieldIndexes) {
        result.push(recordBuffer.slice(preIdx, i));
        preIdx = i;
    }
    return result;
}
function runeCount(s) {
    return Array.from(s).length;
}
export class ParseError extends SyntaxError {
    startLine;
    line;
    column;
    constructor(start, line, column, message) {
        super();
        this.startLine = start;
        this.column = column;
        this.line = line;
        if (message === ERR_FIELD_COUNT) {
            this.message = `record on line ${line}: ${message}`;
        }
        else if (start !== line) {
            this.message =
                `record on line ${start}; parse error on line ${line}, column ${column}: ${message}`;
        }
        else {
            this.message =
                `parse error on line ${line}, column ${column}: ${message}`;
        }
    }
}
export const ERR_BARE_QUOTE = 'bare " in non-quoted-field';
export const ERR_QUOTE = 'extraneous or missing " in quoted-field';
export const ERR_INVALID_DELIM = "Invalid Delimiter";
export const ERR_FIELD_COUNT = "wrong number of fields";
//# sourceMappingURL=data:application/json;base64,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