import { getLevelByName, getLevelName, LogLevels } from "./levels.ts";
export class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options) {
        this.msg = options.msg;
        this.#args = [...options.args];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [...this.#args];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
export class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}) {
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    _log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        }
        else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName,
        });
        this.#handlers.forEach((handler) => {
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data) {
        if (typeof data === "string") {
            return data;
        }
        else if (data === null ||
            typeof data === "number" ||
            typeof data === "bigint" ||
            typeof data === "boolean" ||
            typeof data === "undefined" ||
            typeof data === "symbol") {
            return String(data);
        }
        else if (data instanceof Error) {
            return data.stack;
        }
        else if (typeof data === "object") {
            return JSON.stringify(data);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this._log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this._log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this._log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this._log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this._log(LogLevels.CRITICAL, msg, ...args);
    }
}
//# sourceMappingURL=data:application/json;base64,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