import { equals, indexOf, lastIndexOf, startsWith } from "../bytes/mod.ts";
import { Buffer, BufReader, BufWriter } from "../io/buffer.ts";
import { copy } from "../streams/conversion.ts";
import { copyN } from "../io/util.ts";
import { MultiReader } from "../io/readers.ts";
import { extname } from "../path/mod.ts";
import { assert } from "../_util/assert.ts";
import { TextProtoReader } from "../textproto/mod.ts";
const { hasOwn } = Object;
export function isFormFile(x) {
    return hasOwn(x, "filename") && hasOwn(x, "type");
}
function randomBoundary() {
    let boundary = "--------------------------";
    for (let i = 0; i < 24; i++) {
        boundary += Math.floor(Math.random() * 16).toString(16);
    }
    return boundary;
}
const encoder = new TextEncoder();
export function matchAfterPrefix(buf, prefix, eof) {
    if (buf.length === prefix.length) {
        return eof ? 1 : 0;
    }
    const c = buf[prefix.length];
    if (c === " ".charCodeAt(0) ||
        c === "\t".charCodeAt(0) ||
        c === "\r".charCodeAt(0) ||
        c === "\n".charCodeAt(0) ||
        c === "-".charCodeAt(0)) {
        return 1;
    }
    return -1;
}
export function scanUntilBoundary(buf, dashBoundary, newLineDashBoundary, total, eof) {
    if (total === 0) {
        if (startsWith(buf, dashBoundary)) {
            switch (matchAfterPrefix(buf, dashBoundary, eof)) {
                case -1:
                    return dashBoundary.length;
                case 0:
                    return 0;
                case 1:
                    return null;
            }
        }
        if (startsWith(dashBoundary, buf)) {
            return 0;
        }
    }
    const i = indexOf(buf, newLineDashBoundary);
    if (i >= 0) {
        switch (matchAfterPrefix(buf.slice(i), newLineDashBoundary, eof)) {
            case -1:
                return i + newLineDashBoundary.length;
            case 0:
                return i;
            case 1:
                return i > 0 ? i : null;
        }
    }
    if (startsWith(newLineDashBoundary, buf)) {
        return 0;
    }
    const j = lastIndexOf(buf, newLineDashBoundary.slice(0, 1));
    if (j >= 0 && startsWith(newLineDashBoundary, buf.slice(j))) {
        return j;
    }
    return buf.length;
}
class PartReader {
    mr;
    headers;
    n = 0;
    total = 0;
    constructor(mr, headers) {
        this.mr = mr;
        this.headers = headers;
    }
    async read(p) {
        const br = this.mr.bufReader;
        let peekLength = 1;
        while (this.n === 0) {
            peekLength = Math.max(peekLength, br.buffered());
            const peekBuf = await br.peek(peekLength);
            if (peekBuf === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            const eof = peekBuf.length < peekLength;
            this.n = scanUntilBoundary(peekBuf, this.mr.dashBoundary, this.mr.newLineDashBoundary, this.total, eof);
            if (this.n === 0) {
                assert(eof === false);
                peekLength++;
            }
        }
        if (this.n === null) {
            return null;
        }
        const nread = Math.min(p.length, this.n);
        const buf = p.subarray(0, nread);
        const r = await br.readFull(buf);
        assert(r === buf);
        this.n -= nread;
        this.total += nread;
        return nread;
    }
    close() { }
    contentDisposition;
    contentDispositionParams;
    getContentDispositionParams() {
        if (this.contentDispositionParams)
            return this.contentDispositionParams;
        const cd = this.headers.get("content-disposition");
        const params = {};
        assert(cd != null, "content-disposition must be set");
        const comps = decodeURI(cd).split(";");
        this.contentDisposition = comps[0];
        comps
            .slice(1)
            .map((v) => v.trim())
            .map((kv) => {
            const [k, v] = kv.split("=");
            if (v) {
                const s = v.charAt(0);
                const e = v.charAt(v.length - 1);
                if ((s === e && s === '"') || s === "'") {
                    params[k] = v.substr(1, v.length - 2);
                }
                else {
                    params[k] = v;
                }
            }
        });
        return (this.contentDispositionParams = params);
    }
    get fileName() {
        return this.getContentDispositionParams()["filename"];
    }
    get formName() {
        const p = this.getContentDispositionParams();
        if (this.contentDisposition === "form-data") {
            return p["name"];
        }
        return "";
    }
}
function skipLWSPChar(u) {
    const ret = new Uint8Array(u.length);
    const sp = " ".charCodeAt(0);
    const ht = "\t".charCodeAt(0);
    let j = 0;
    for (let i = 0; i < u.length; i++) {
        if (u[i] === sp || u[i] === ht)
            continue;
        ret[j++] = u[i];
    }
    return ret.slice(0, j);
}
export class MultipartReader {
    boundary;
    newLine;
    newLineDashBoundary;
    dashBoundaryDash;
    dashBoundary;
    bufReader;
    constructor(reader, boundary) {
        this.boundary = boundary;
        this.newLine = encoder.encode("\r\n");
        this.newLineDashBoundary = encoder.encode(`\r\n--${boundary}`);
        this.dashBoundaryDash = encoder.encode(`--${this.boundary}--`);
        this.dashBoundary = encoder.encode(`--${this.boundary}`);
        this.bufReader = new BufReader(reader);
    }
    async readForm(maxMemoryOrOptions) {
        const options = typeof maxMemoryOrOptions === "number"
            ? { maxMemory: maxMemoryOrOptions }
            : maxMemoryOrOptions;
        let maxMemory = options?.maxMemory ?? 10 << 20;
        const fileMap = new Map();
        const valueMap = new Map();
        let maxValueBytes = maxMemory + (10 << 20);
        const buf = new Buffer(new Uint8Array(maxValueBytes));
        for (;;) {
            const p = await this.nextPart();
            if (p === null) {
                break;
            }
            if (p.formName === "") {
                continue;
            }
            buf.reset();
            if (!p.fileName) {
                const n = await copyN(p, buf, maxValueBytes);
                maxValueBytes -= n;
                if (maxValueBytes < 0) {
                    throw new RangeError("message too large");
                }
                const value = new TextDecoder().decode(buf.bytes());
                const mapVal = valueMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(value);
                }
                else {
                    valueMap.set(p.formName, [value]);
                }
                continue;
            }
            let formFile;
            const n = await copyN(p, buf, maxValueBytes);
            const contentType = p.headers.get("content-type");
            assert(contentType != null, "content-type must be set");
            if (n > maxMemory) {
                const ext = extname(p.fileName);
                const filepath = await Deno.makeTempFile({
                    dir: options?.dir ?? ".",
                    prefix: options?.prefix ?? "multipart-",
                    suffix: options?.suffix ?? ext,
                });
                const file = await Deno.open(filepath, { write: true });
                try {
                    const size = await copy(new MultiReader(buf, p), file);
                    file.close();
                    formFile = {
                        filename: p.fileName,
                        type: contentType,
                        tempfile: filepath,
                        size,
                    };
                }
                catch (e) {
                    await Deno.remove(filepath);
                    throw e;
                }
            }
            else {
                formFile = {
                    filename: p.fileName,
                    type: contentType,
                    content: buf.bytes(),
                    size: buf.length,
                };
                maxMemory -= n;
                maxValueBytes -= n;
            }
            if (formFile) {
                const mapVal = fileMap.get(p.formName);
                if (mapVal !== undefined) {
                    mapVal.push(formFile);
                }
                else {
                    fileMap.set(p.formName, [formFile]);
                }
            }
        }
        return multipartFormData(fileMap, valueMap);
    }
    currentPart;
    partsRead = 0;
    async nextPart() {
        if (this.currentPart) {
            this.currentPart.close();
        }
        if (equals(this.dashBoundary, encoder.encode("--"))) {
            throw new Error("boundary is empty");
        }
        let expectNewPart = false;
        for (;;) {
            const line = await this.bufReader.readSlice("\n".charCodeAt(0));
            if (line === null) {
                throw new Deno.errors.UnexpectedEof();
            }
            if (this.isBoundaryDelimiterLine(line)) {
                this.partsRead++;
                const r = new TextProtoReader(this.bufReader);
                const headers = await r.readMIMEHeader();
                if (headers === null) {
                    throw new Deno.errors.UnexpectedEof();
                }
                const np = new PartReader(this, headers);
                this.currentPart = np;
                return np;
            }
            if (this.isFinalBoundary(line)) {
                return null;
            }
            if (expectNewPart) {
                throw new Error(`expecting a new Part; got line ${line}`);
            }
            if (this.partsRead === 0) {
                continue;
            }
            if (equals(line, this.newLine)) {
                expectNewPart = true;
                continue;
            }
            throw new Error(`unexpected line in nextPart(): ${line}`);
        }
    }
    isFinalBoundary(line) {
        if (!startsWith(line, this.dashBoundaryDash)) {
            return false;
        }
        const rest = line.slice(this.dashBoundaryDash.length, line.length);
        return rest.length === 0 || equals(skipLWSPChar(rest), this.newLine);
    }
    isBoundaryDelimiterLine(line) {
        if (!startsWith(line, this.dashBoundary)) {
            return false;
        }
        const rest = line.slice(this.dashBoundary.length);
        return equals(skipLWSPChar(rest), this.newLine);
    }
}
function multipartFormData(fileMap, valueMap) {
    function files(key) {
        return fileMap.get(key);
    }
    function values(key) {
        return valueMap.get(key);
    }
    function* entries() {
        yield* fileMap;
        yield* valueMap;
    }
    async function removeAll() {
        const promises = [];
        for (const val of fileMap.values()) {
            for (const subVal of val) {
                if (!subVal.tempfile)
                    continue;
                promises.push(Deno.remove(subVal.tempfile));
            }
        }
        await Promise.all(promises);
    }
    return {
        files,
        values,
        entries,
        removeAll,
        [Symbol.iterator]() {
            return entries();
        },
    };
}
class PartWriter {
    writer;
    boundary;
    headers;
    closed = false;
    partHeader;
    headersWritten = false;
    constructor(writer, boundary, headers, isFirstBoundary) {
        this.writer = writer;
        this.boundary = boundary;
        this.headers = headers;
        let buf = "";
        if (isFirstBoundary) {
            buf += `--${boundary}\r\n`;
        }
        else {
            buf += `\r\n--${boundary}\r\n`;
        }
        for (const [key, value] of headers.entries()) {
            buf += `${key}: ${value}\r\n`;
        }
        buf += `\r\n`;
        this.partHeader = buf;
    }
    close() {
        this.closed = true;
    }
    async write(p) {
        if (this.closed) {
            throw new Error("part is closed");
        }
        if (!this.headersWritten) {
            await this.writer.write(encoder.encode(this.partHeader));
            this.headersWritten = true;
        }
        return this.writer.write(p);
    }
}
function checkBoundary(b) {
    if (b.length < 1 || b.length > 70) {
        throw new Error(`invalid boundary length: ${b.length}`);
    }
    const end = b.length - 1;
    for (let i = 0; i < end; i++) {
        const c = b.charAt(i);
        if (!c.match(/[a-zA-Z0-9'()+_,\-./:=?]/) || (c === " " && i !== end)) {
            throw new Error("invalid boundary character: " + c);
        }
    }
    return b;
}
export class MultipartWriter {
    writer;
    _boundary;
    get boundary() {
        return this._boundary;
    }
    lastPart;
    bufWriter;
    isClosed = false;
    constructor(writer, boundary) {
        this.writer = writer;
        if (boundary !== void 0) {
            this._boundary = checkBoundary(boundary);
        }
        else {
            this._boundary = randomBoundary();
        }
        this.bufWriter = new BufWriter(writer);
    }
    formDataContentType() {
        return `multipart/form-data; boundary=${this.boundary}`;
    }
    createPart(headers) {
        if (this.isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.lastPart) {
            this.lastPart.close();
        }
        const part = new PartWriter(this.writer, this.boundary, headers, !this.lastPart);
        this.lastPart = part;
        return part;
    }
    createFormFile(field, filename) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"; filename="${filename}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    createFormField(field) {
        const h = new Headers();
        h.set("Content-Disposition", `form-data; name="${field}"`);
        h.set("Content-Type", "application/octet-stream");
        return this.createPart(h);
    }
    async writeField(field, value) {
        const f = await this.createFormField(field);
        await f.write(encoder.encode(value));
    }
    async writeFile(field, filename, file) {
        const f = await this.createFormFile(field, filename);
        await copy(file, f);
    }
    flush() {
        return this.bufWriter.flush();
    }
    async close() {
        if (this.isClosed) {
            throw new Error("multipart: writer is closed");
        }
        if (this.lastPart) {
            this.lastPart.close();
            this.lastPart = void 0;
        }
        await this.writer.write(encoder.encode(`\r\n--${this.boundary}--\r\n`));
        await this.flush();
        this.isClosed = true;
    }
}
//# sourceMappingURL=data:application/json;base64,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