import { concat } from "../bytes/mod.ts";
const CHAR_SPACE = " ".charCodeAt(0);
const CHAR_TAB = "\t".charCodeAt(0);
const CHAR_COLON = ":".charCodeAt(0);
const WHITESPACES = [CHAR_SPACE, CHAR_TAB];
const decoder = new TextDecoder();
const invalidHeaderCharRegex = /[^\t\x20-\x7e\x80-\xff]/g;
function str(buf) {
    return !buf ? "" : decoder.decode(buf);
}
export class TextProtoReader {
    r;
    constructor(r) {
        this.r = r;
    }
    async readLine() {
        const s = await this.readLineSlice();
        return s === null ? null : str(s);
    }
    async readMIMEHeader() {
        const m = new Headers();
        let line;
        let buf = await this.r.peek(1);
        if (buf === null) {
            return null;
        }
        else if (WHITESPACES.includes(buf[0])) {
            line = (await this.readLineSlice());
        }
        buf = await this.r.peek(1);
        if (buf === null) {
            throw new Deno.errors.UnexpectedEof();
        }
        else if (WHITESPACES.includes(buf[0])) {
            throw new Deno.errors.InvalidData(`malformed MIME header initial line: ${str(line)}`);
        }
        while (true) {
            const kv = await this.readLineSlice();
            if (kv === null)
                throw new Deno.errors.UnexpectedEof();
            if (kv.byteLength === 0)
                return m;
            let i = kv.indexOf(CHAR_COLON);
            if (i < 0) {
                throw new Deno.errors.InvalidData(`malformed MIME header line: ${str(kv)}`);
            }
            const key = str(kv.subarray(0, i));
            if (key == "") {
                continue;
            }
            i++;
            while (i < kv.byteLength &&
                (WHITESPACES.includes(kv[i]))) {
                i++;
            }
            const value = str(kv.subarray(i)).replace(invalidHeaderCharRegex, encodeURI);
            try {
                m.append(key, value);
            }
            catch {
            }
        }
    }
    async readLineSlice() {
        let line = new Uint8Array(0);
        let r = null;
        do {
            r = await this.r.readLine();
            if (r !== null && this.skipSpace(r.line) !== 0) {
                line = concat(line, r.line);
            }
        } while (r !== null && r.more);
        return r === null ? null : line;
    }
    skipSpace(l) {
        let n = 0;
        for (const val of l) {
            if (!WHITESPACES.includes(val)) {
                n++;
            }
        }
        return n;
    }
}
//# sourceMappingURL=data:application/json;base64,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