import { BytesList } from "../bytes/bytes_list.ts";
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
export class LineStream extends TransformStream {
    #bufs = new BytesList();
    #prevHadCR = false;
    constructor() {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#mergeBufs(false));
            },
        });
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf(LF);
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            if (lfIndex === 0) {
                controller.enqueue(this.#mergeBufs(true));
                this.#handle(chunk.subarray(1), controller);
                return;
            }
        }
        if (lfIndex === -1) {
            if (chunk.at(-1) === CR) {
                this.#prevHadCR = true;
            }
            this.#bufs.add(chunk);
        }
        else {
            let crOrLfIndex = lfIndex;
            if (chunk[lfIndex - 1] === CR) {
                crOrLfIndex--;
            }
            this.#bufs.add(chunk.subarray(0, crOrLfIndex));
            controller.enqueue(this.#mergeBufs(false));
            this.#handle(chunk.subarray(lfIndex + 1), controller);
        }
    }
    #mergeBufs(prevHadCR) {
        const mergeBuf = this.#bufs.concat();
        this.#bufs = new BytesList();
        if (prevHadCR) {
            return mergeBuf.subarray(0, -1);
        }
        else {
            return mergeBuf;
        }
    }
}
export class TextLineStream extends TransformStream {
    #buf = "";
    #prevHadCR = false;
    #allowCR;
    constructor(options) {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#getBuf(this.#prevHadCR));
                if (this.#prevHadCR) {
                    controller.enqueue("");
                }
            },
        });
        this.#allowCR = options?.allowCR ?? false;
    }
    #handle(chunk, controller) {
        const lfIndex = chunk.indexOf("\n");
        const crIndex = chunk.indexOf("\r");
        if (this.#prevHadCR) {
            this.#prevHadCR = false;
            controller.enqueue(this.#getBuf(true));
            if (lfIndex === 0) {
                this.#handle(chunk.slice(1), controller);
                return;
            }
        }
        if (lfIndex === -1 && crIndex === -1) {
            this.#buf += chunk;
        }
        else if (lfIndex === -1 && crIndex !== -1) {
            if (crIndex === (chunk.length - 1)) {
                this.#buf += chunk;
                this.#prevHadCR = true;
            }
            else if (this.#allowCR) {
                this.#mergeHandle(chunk, crIndex, crIndex, controller);
            }
            else {
                this.#buf += chunk.slice(0, crIndex + 1);
                this.#handle(chunk.slice(crIndex + 1), controller);
            }
        }
        else if (lfIndex !== -1 && crIndex === -1) {
            this.#mergeHandle(chunk, lfIndex, lfIndex, controller);
        }
        else {
            if ((lfIndex - 1) === crIndex) {
                this.#mergeHandle(chunk, crIndex, lfIndex, controller);
            }
            else if (crIndex < lfIndex && this.#allowCR) {
                this.#mergeHandle(chunk, crIndex, crIndex, controller);
            }
            else {
                this.#mergeHandle(chunk, lfIndex, lfIndex, controller);
            }
        }
    }
    #mergeHandle(chunk, prevChunkEndIndex, newChunkStartIndex, controller) {
        this.#buf += chunk.slice(0, prevChunkEndIndex);
        controller.enqueue(this.#getBuf(false));
        this.#handle(chunk.slice(newChunkStartIndex + 1), controller);
    }
    #getBuf(prevHadCR) {
        const buf = this.#buf;
        this.#buf = "";
        if (prevHadCR) {
            return buf.slice(0, -1);
        }
        else {
            return buf;
        }
    }
}
export class DelimiterStream extends TransformStream {
    #bufs = new BytesList();
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLen;
    #delimLPS;
    constructor(delimiter) {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#bufs.concat());
            },
        });
        this.#delimiter = delimiter;
        this.#delimLen = delimiter.length;
        this.#delimLPS = createLPS(delimiter);
    }
    #handle(chunk, controller) {
        this.#bufs.add(chunk);
        let localIndex = 0;
        while (this.#inspectIndex < this.#bufs.size()) {
            if (chunk[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimLen) {
                    const matchEnd = this.#inspectIndex - this.#delimLen;
                    const readyBytes = this.#bufs.slice(0, matchEnd);
                    controller.enqueue(readyBytes);
                    this.#bufs.shift(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            }
            else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                }
                else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
export class TextDelimiterStream extends TransformStream {
    #buf = "";
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLPS;
    constructor(delimiter) {
        super({
            transform: (chunk, controller) => {
                this.#handle(chunk, controller);
            },
            flush: (controller) => {
                controller.enqueue(this.#buf);
            },
        });
        this.#delimiter = delimiter;
        this.#delimLPS = createLPS(new TextEncoder().encode(delimiter));
    }
    #handle(chunk, controller) {
        this.#buf += chunk;
        let localIndex = 0;
        while (this.#inspectIndex < this.#buf.length) {
            if (chunk[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimiter.length) {
                    const matchEnd = this.#inspectIndex - this.#delimiter.length;
                    const readyString = this.#buf.slice(0, matchEnd);
                    controller.enqueue(readyString);
                    this.#buf = this.#buf.slice(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            }
            else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                }
                else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i = 1;
    while (i < lps.length) {
        if (pat[i] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i] = prefixEnd;
            i++;
        }
        else if (prefixEnd === 0) {
            lps[i] = 0;
            i++;
        }
        else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
//# sourceMappingURL=data:application/json;base64,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