import { ascend } from "./_comparators.ts";
import { BSNode } from "./bs_node.ts";
export * from "./_comparators.ts";
export class BSTree {
    compare;
    root = null;
    _size = 0;
    constructor(compare = ascend) {
        this.compare = compare;
    }
    static from(collection, options) {
        let result;
        let unmappedValues = [];
        if (collection instanceof BSTree) {
            result = new BSTree(options?.compare ?? collection.compare);
            if (options?.compare || options?.map) {
                unmappedValues = collection;
            }
            else {
                const nodes = [];
                if (collection.root) {
                    result.root = BSNode.from(collection.root);
                    nodes.push(result.root);
                }
                while (nodes.length) {
                    const node = nodes.pop();
                    const left = node.left
                        ? BSNode.from(node.left)
                        : null;
                    const right = node.right
                        ? BSNode.from(node.right)
                        : null;
                    if (left) {
                        left.parent = node;
                        nodes.push(left);
                    }
                    if (right) {
                        right.parent = node;
                        nodes.push(right);
                    }
                }
            }
        }
        else {
            result = (options?.compare
                ? new BSTree(options.compare)
                : new BSTree());
            unmappedValues = collection;
        }
        const values = options?.map
            ? Array.from(unmappedValues, options.map, options.thisArg)
            : unmappedValues;
        for (const value of values)
            result.insert(value);
        return result;
    }
    get size() {
        return this._size;
    }
    findNode(value) {
        let node = this.root;
        while (node) {
            const order = this.compare(value, node.value);
            if (order === 0)
                break;
            const direction = order < 0 ? "left" : "right";
            node = node[direction];
        }
        return node;
    }
    rotateNode(node, direction) {
        const replacementDirection = direction === "left"
            ? "right"
            : "left";
        if (!node[replacementDirection]) {
            throw new TypeError(`cannot rotate ${direction} without ${replacementDirection} child`);
        }
        const replacement = node[replacementDirection];
        node[replacementDirection] = replacement[direction] ?? null;
        if (replacement[direction])
            replacement[direction].parent = node;
        replacement.parent = node.parent;
        if (node.parent) {
            const parentDirection = node === node.parent[direction]
                ? direction
                : replacementDirection;
            node.parent[parentDirection] = replacement;
        }
        else {
            this.root = replacement;
        }
        replacement[direction] = node;
        node.parent = replacement;
    }
    insertNode(Node, value) {
        if (!this.root) {
            this.root = new Node(null, value);
            this._size++;
            return this.root;
        }
        else {
            let node = this.root;
            while (true) {
                const order = this.compare(value, node.value);
                if (order === 0)
                    break;
                const direction = order < 0 ? "left" : "right";
                if (node[direction]) {
                    node = node[direction];
                }
                else {
                    node[direction] = new Node(node, value);
                    this._size++;
                    return node[direction];
                }
            }
        }
        return null;
    }
    removeNode(value) {
        let removeNode = this.findNode(value);
        if (removeNode) {
            const successorNode = !removeNode.left || !removeNode.right
                ? removeNode
                : removeNode.findSuccessorNode();
            const replacementNode = successorNode.left ??
                successorNode.right;
            if (replacementNode)
                replacementNode.parent = successorNode.parent;
            if (!successorNode.parent) {
                this.root = replacementNode;
            }
            else {
                successorNode.parent[successorNode.directionFromParent()] =
                    replacementNode;
            }
            if (successorNode !== removeNode) {
                removeNode.value = successorNode.value;
                removeNode = successorNode;
            }
            this._size--;
        }
        return removeNode;
    }
    insert(value) {
        return !!this.insertNode(BSNode, value);
    }
    remove(value) {
        return !!this.removeNode(value);
    }
    find(value) {
        return this.findNode(value)?.value ?? null;
    }
    min() {
        return this.root ? this.root.findMinNode().value : null;
    }
    max() {
        return this.root ? this.root.findMaxNode().value : null;
    }
    clear() {
        this.root = null;
        this._size = 0;
    }
    isEmpty() {
        return this.size === 0;
    }
    *lnrValues() {
        const nodes = [];
        let node = this.root;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.left;
            }
            else {
                node = nodes.pop();
                yield node.value;
                node = node.right;
            }
        }
    }
    *rnlValues() {
        const nodes = [];
        let node = this.root;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.right;
            }
            else {
                node = nodes.pop();
                yield node.value;
                node = node.left;
            }
        }
    }
    *nlrValues() {
        const nodes = [];
        if (this.root)
            nodes.push(this.root);
        while (nodes.length) {
            const node = nodes.pop();
            yield node.value;
            if (node.right)
                nodes.push(node.right);
            if (node.left)
                nodes.push(node.left);
        }
    }
    *lrnValues() {
        const nodes = [];
        let node = this.root;
        let lastNodeVisited = null;
        while (nodes.length || node) {
            if (node) {
                nodes.push(node);
                node = node.left;
            }
            else {
                const lastNode = nodes[nodes.length - 1];
                if (lastNode.right && lastNode.right !== lastNodeVisited) {
                    node = lastNode.right;
                }
                else {
                    yield lastNode.value;
                    lastNodeVisited = nodes.pop();
                }
            }
        }
    }
    *lvlValues() {
        const children = [];
        let cursor = this.root;
        while (cursor) {
            yield cursor.value;
            if (cursor.left)
                children.push(cursor.left);
            if (cursor.right)
                children.push(cursor.right);
            cursor = children.shift() ?? null;
        }
    }
    *[Symbol.iterator]() {
        yield* this.lnrValues();
    }
}
//# sourceMappingURL=data:application/json;base64,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