import { digestAlgorithms as wasmDigestAlgorithms, instantiateWasm, } from "../_wasm_crypto/mod.ts";
import { fnv } from "./_fnv/index.ts";
const webCrypto = ((crypto) => ({
    getRandomValues: crypto.getRandomValues?.bind(crypto),
    randomUUID: crypto.randomUUID?.bind(crypto),
    subtle: {
        decrypt: crypto.subtle?.decrypt?.bind(crypto.subtle),
        deriveBits: crypto.subtle?.deriveBits?.bind(crypto.subtle),
        deriveKey: crypto.subtle?.deriveKey?.bind(crypto.subtle),
        digest: crypto.subtle?.digest?.bind(crypto.subtle),
        encrypt: crypto.subtle?.encrypt?.bind(crypto.subtle),
        exportKey: crypto.subtle?.exportKey?.bind(crypto.subtle),
        generateKey: crypto.subtle?.generateKey?.bind(crypto.subtle),
        importKey: crypto.subtle?.importKey?.bind(crypto.subtle),
        sign: crypto.subtle?.sign?.bind(crypto.subtle),
        unwrapKey: crypto.subtle?.unwrapKey?.bind(crypto.subtle),
        verify: crypto.subtle?.verify?.bind(crypto.subtle),
        wrapKey: crypto.subtle?.wrapKey?.bind(crypto.subtle),
    },
}))(globalThis.crypto);
const bufferSourceBytes = (data) => {
    let bytes;
    if (data instanceof Uint8Array) {
        bytes = data;
    }
    else if (ArrayBuffer.isView(data)) {
        bytes = new Uint8Array(data.buffer, data.byteOffset, data.byteLength);
    }
    else if (data instanceof ArrayBuffer) {
        bytes = new Uint8Array(data);
    }
    return bytes;
};
const stdCrypto = ((x) => x)({
    ...webCrypto,
    subtle: {
        ...webCrypto.subtle,
        async digest(algorithm, data) {
            const { name, length } = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (FNVAlgorithms.includes(name)) {
                return fnv(name, bytes);
            }
            if (webCryptoDigestAlgorithms.includes(name) &&
                bytes) {
                return webCrypto.subtle.digest(algorithm, bytes);
            }
            else if (wasmDigestAlgorithms.includes(name)) {
                if (bytes) {
                    return stdCrypto.subtle.digestSync(algorithm, bytes);
                }
                else if (data[Symbol.iterator]) {
                    return stdCrypto.subtle.digestSync(algorithm, data);
                }
                else if (data[Symbol.asyncIterator]) {
                    const wasmCrypto = instantiateWasm();
                    const context = new wasmCrypto.DigestContext(name);
                    for await (const chunk of data) {
                        const chunkBytes = bufferSourceBytes(chunk);
                        if (!chunkBytes) {
                            throw new TypeError("data contained chunk of the wrong type");
                        }
                        context.update(chunkBytes);
                    }
                    return context.digestAndDrop(length).buffer;
                }
                else {
                    throw new TypeError("data must be a BufferSource or [Async]Iterable<BufferSource>");
                }
            }
            else if (webCrypto.subtle?.digest) {
                return webCrypto.subtle.digest(algorithm, data);
            }
            else {
                throw new TypeError(`unsupported digest algorithm: ${algorithm}`);
            }
        },
        digestSync(algorithm, data) {
            algorithm = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (FNVAlgorithms.includes(algorithm.name)) {
                return fnv(algorithm.name, bytes);
            }
            const wasmCrypto = instantiateWasm();
            if (bytes) {
                return wasmCrypto.digest(algorithm.name, bytes, algorithm.length)
                    .buffer;
            }
            else if (data[Symbol.iterator]) {
                const context = new wasmCrypto.DigestContext(algorithm.name);
                for (const chunk of data) {
                    const chunkBytes = bufferSourceBytes(chunk);
                    if (!chunkBytes) {
                        throw new TypeError("data contained chunk of the wrong type");
                    }
                    context.update(chunkBytes);
                }
                return context.digestAndDrop(algorithm.length).buffer;
            }
            else {
                throw new TypeError("data must be a BufferSource or Iterable<BufferSource>");
            }
        },
    },
});
const FNVAlgorithms = ["FNV32", "FNV32A", "FNV64", "FNV64A"];
const webCryptoDigestAlgorithms = [
    "SHA-384",
    "SHA-256",
    "SHA-512",
    "SHA-1",
];
const normalizeAlgorithm = (algorithm) => ((typeof algorithm === "string") ? { name: algorithm.toUpperCase() } : {
    ...algorithm,
    name: algorithm.name.toUpperCase(),
});
export { stdCrypto as crypto };
//# sourceMappingURL=data:application/json;base64,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