import { BufReader } from "../io/buffer.ts";
import { TextProtoReader } from "../textproto/mod.ts";
import { StringReader } from "../io/readers.ts";
import { assert } from "../_util/assert.ts";
export { NEWLINE, stringify, StringifyError } from "./csv_stringify.ts";
const INVALID_RUNE = ["\r", "\n", '"'];
export const ERR_BARE_QUOTE = 'bare " in non-quoted-field';
export const ERR_QUOTE = 'extraneous or missing " in quoted-field';
export const ERR_INVALID_DELIM = "Invalid Delimiter";
export const ERR_FIELD_COUNT = "wrong number of fields";
export class ParseError extends Error {
    startLine;
    line;
    column;
    constructor(start, line, column, message) {
        super();
        this.startLine = start;
        this.column = column;
        this.line = line;
        if (message === ERR_FIELD_COUNT) {
            this.message = `record on line ${line}: ${message}`;
        }
        else if (start !== line) {
            this.message =
                `record on line ${start}; parse error on line ${line}, column ${column}: ${message}`;
        }
        else {
            this.message =
                `parse error on line ${line}, column ${column}: ${message}`;
        }
    }
}
function chkOptions(opt) {
    if (!opt.separator) {
        opt.separator = ",";
    }
    if (!opt.trimLeadingSpace) {
        opt.trimLeadingSpace = false;
    }
    if (INVALID_RUNE.includes(opt.separator) ||
        (typeof opt.comment === "string" && INVALID_RUNE.includes(opt.comment)) ||
        opt.separator === opt.comment) {
        throw new Error(ERR_INVALID_DELIM);
    }
}
async function readRecord(startLine, reader, opt = { separator: ",", trimLeadingSpace: false }) {
    const tp = new TextProtoReader(reader);
    let line = await readLine(tp);
    let lineIndex = startLine + 1;
    if (line === null)
        return null;
    if (line.length === 0) {
        return [];
    }
    if (opt.comment && line[0] === opt.comment) {
        return [];
    }
    assert(opt.separator != null);
    let fullLine = line;
    let quoteError = null;
    const quote = '"';
    const quoteLen = quote.length;
    const separatorLen = opt.separator.length;
    let recordBuffer = "";
    const fieldIndexes = [];
    parseField: for (;;) {
        if (opt.trimLeadingSpace) {
            line = line.trimLeft();
        }
        if (line.length === 0 || !line.startsWith(quote)) {
            const i = line.indexOf(opt.separator);
            let field = line;
            if (i >= 0) {
                field = field.substring(0, i);
            }
            if (!opt.lazyQuotes) {
                const j = field.indexOf(quote);
                if (j >= 0) {
                    const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
                    quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
                    break parseField;
                }
            }
            recordBuffer += field;
            fieldIndexes.push(recordBuffer.length);
            if (i >= 0) {
                line = line.substring(i + separatorLen);
                continue parseField;
            }
            break parseField;
        }
        else {
            line = line.substring(quoteLen);
            for (;;) {
                const i = line.indexOf(quote);
                if (i >= 0) {
                    recordBuffer += line.substring(0, i);
                    line = line.substring(i + quoteLen);
                    if (line.startsWith(quote)) {
                        recordBuffer += quote;
                        line = line.substring(quoteLen);
                    }
                    else if (line.startsWith(opt.separator)) {
                        line = line.substring(separatorLen);
                        fieldIndexes.push(recordBuffer.length);
                        continue parseField;
                    }
                    else if (0 === line.length) {
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    else if (opt.lazyQuotes) {
                        recordBuffer += quote;
                    }
                    else {
                        const col = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                }
                else if (line.length > 0 || !(await isEOF(tp))) {
                    recordBuffer += line;
                    const r = await readLine(tp);
                    lineIndex++;
                    line = r ?? "";
                    fullLine = line;
                    if (r === null) {
                        if (!opt.lazyQuotes) {
                            const col = runeCount(fullLine);
                            quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                            break parseField;
                        }
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    recordBuffer += "\n";
                }
                else {
                    if (!opt.lazyQuotes) {
                        const col = runeCount(fullLine);
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                        break parseField;
                    }
                    fieldIndexes.push(recordBuffer.length);
                    break parseField;
                }
            }
        }
    }
    if (quoteError) {
        throw quoteError;
    }
    const result = [];
    let preIdx = 0;
    for (const i of fieldIndexes) {
        result.push(recordBuffer.slice(preIdx, i));
        preIdx = i;
    }
    return result;
}
async function isEOF(tp) {
    return (await tp.r.peek(0)) === null;
}
function runeCount(s) {
    return Array.from(s).length;
}
async function readLine(tp) {
    let line;
    const r = await tp.readLine();
    if (r === null)
        return null;
    line = r;
    if ((await isEOF(tp)) && line.length > 0 && line[line.length - 1] === "\r") {
        line = line.substring(0, line.length - 1);
    }
    if (line.length >= 2 &&
        line[line.length - 2] === "\r" &&
        line[line.length - 1] === "\n") {
        line = line.substring(0, line.length - 2);
        line = line + "\n";
    }
    return line;
}
export async function readMatrix(reader, opt = {
    separator: ",",
    trimLeadingSpace: false,
    lazyQuotes: false,
}) {
    const result = [];
    let _nbFields;
    let lineResult;
    let first = true;
    let lineIndex = 0;
    chkOptions(opt);
    for (;;) {
        const r = await readRecord(lineIndex, reader, opt);
        if (r === null)
            break;
        lineResult = r;
        lineIndex++;
        if (first) {
            first = false;
            if (opt.fieldsPerRecord !== undefined) {
                if (opt.fieldsPerRecord === 0) {
                    _nbFields = lineResult.length;
                }
                else {
                    _nbFields = opt.fieldsPerRecord;
                }
            }
        }
        if (lineResult.length > 0) {
            if (_nbFields && _nbFields !== lineResult.length) {
                throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
            }
            result.push(lineResult);
        }
    }
    return result;
}
export async function parse(input, opt = {
    skipFirstRow: false,
}) {
    let r;
    if (input instanceof BufReader) {
        r = await readMatrix(input, opt);
    }
    else {
        r = await readMatrix(new BufReader(new StringReader(input)), opt);
    }
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head.map((e) => {
                return {
                    name: e,
                };
            });
            i++;
        }
        if (opt.columns) {
            if (typeof opt.columns[0] !== "string") {
                headers = opt.columns;
            }
            else {
                const h = opt.columns;
                headers = h.map((e) => {
                    return {
                        name: e,
                    };
                });
            }
        }
        return r.map((e) => {
            if (e.length !== headers.length) {
                throw `Error number of fields line:${i}`;
            }
            i++;
            const out = {};
            for (let j = 0; j < e.length; j++) {
                const h = headers[j];
                if (h.parse) {
                    out[h.name] = h.parse(e[j]);
                }
                else {
                    out[h.name] = e[j];
                }
            }
            if (opt.parse) {
                return opt.parse(out);
            }
            return out;
        });
    }
    if (opt.parse) {
        return r.map((e) => {
            assert(opt.parse, "opt.parse must be set");
            return opt.parse(e);
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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