import { MultiReader } from "../io/readers.ts";
import { Buffer, PartialReadError } from "../io/buffer.ts";
import { assert } from "../_util/assert.ts";
import { readAll } from "../streams/conversion.ts";
const recordSize = 512;
const ustar = "ustar\u000000";
const initialChecksum = 8 * 32;
async function readBlock(reader, p) {
    let bytesRead = 0;
    while (bytesRead < p.length) {
        const rr = await reader.read(p.subarray(bytesRead));
        if (rr === null) {
            if (bytesRead === 0) {
                return null;
            }
            else {
                throw new PartialReadError();
            }
        }
        bytesRead += rr;
    }
    return bytesRead;
}
class FileReader {
    filePath;
    file;
    constructor(filePath) {
        this.filePath = filePath;
    }
    async read(p) {
        if (!this.file) {
            this.file = await Deno.open(this.filePath, { read: true });
        }
        const res = await Deno.read(this.file.rid, p);
        if (res === null) {
            Deno.close(this.file.rid);
            this.file = undefined;
        }
        return res;
    }
}
function trim(buffer) {
    const index = buffer.findIndex((v) => v === 0);
    if (index < 0)
        return buffer;
    return buffer.subarray(0, index);
}
function clean(length) {
    const buffer = new Uint8Array(length);
    buffer.fill(0, 0, length - 1);
    return buffer;
}
function pad(num, bytes, base = 8) {
    const numString = num.toString(base);
    return "000000000000".substr(numString.length + 12 - bytes) + numString;
}
var FileTypes;
(function (FileTypes) {
    FileTypes[FileTypes["file"] = 0] = "file";
    FileTypes[FileTypes["link"] = 1] = "link";
    FileTypes[FileTypes["symlink"] = 2] = "symlink";
    FileTypes[FileTypes["character-device"] = 3] = "character-device";
    FileTypes[FileTypes["block-device"] = 4] = "block-device";
    FileTypes[FileTypes["directory"] = 5] = "directory";
    FileTypes[FileTypes["fifo"] = 6] = "fifo";
    FileTypes[FileTypes["contiguous-file"] = 7] = "contiguous-file";
})(FileTypes || (FileTypes = {}));
const ustarStructure = [
    {
        field: "fileName",
        length: 100,
    },
    {
        field: "fileMode",
        length: 8,
    },
    {
        field: "uid",
        length: 8,
    },
    {
        field: "gid",
        length: 8,
    },
    {
        field: "fileSize",
        length: 12,
    },
    {
        field: "mtime",
        length: 12,
    },
    {
        field: "checksum",
        length: 8,
    },
    {
        field: "type",
        length: 1,
    },
    {
        field: "linkName",
        length: 100,
    },
    {
        field: "ustar",
        length: 8,
    },
    {
        field: "owner",
        length: 32,
    },
    {
        field: "group",
        length: 32,
    },
    {
        field: "majorNumber",
        length: 8,
    },
    {
        field: "minorNumber",
        length: 8,
    },
    {
        field: "fileNamePrefix",
        length: 155,
    },
    {
        field: "padding",
        length: 12,
    },
];
function formatHeader(data) {
    const encoder = new TextEncoder(), buffer = clean(512);
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const entry = encoder.encode(data[value.field] || "");
        buffer.set(entry, offset);
        offset += value.length;
    });
    return buffer;
}
function parseHeader(buffer) {
    const data = {};
    let offset = 0;
    ustarStructure.forEach(function (value) {
        const arr = buffer.subarray(offset, offset + value.length);
        data[value.field] = arr;
        offset += value.length;
    });
    return data;
}
export class Tar {
    data;
    constructor() {
        this.data = [];
    }
    async append(fn, opts) {
        if (typeof fn !== "string") {
            throw new Error("file name not specified");
        }
        let fileName = fn;
        let fileNamePrefix;
        if (fileName.length > 100) {
            let i = fileName.length;
            while (i >= 0) {
                i = fileName.lastIndexOf("/", i);
                if (i <= 155) {
                    fileNamePrefix = fileName.substr(0, i);
                    fileName = fileName.substr(i + 1);
                    break;
                }
                i--;
            }
            const errMsg = "ustar format does not allow a long file name (length of [file name" +
                "prefix] + / + [file name] must be shorter than 256 bytes)";
            if (i < 0 || fileName.length > 100) {
                throw new Error(errMsg);
            }
            else {
                assert(fileNamePrefix != null);
                if (fileNamePrefix.length > 155) {
                    throw new Error(errMsg);
                }
            }
        }
        opts = opts || {};
        let info;
        if (opts.filePath) {
            info = await Deno.stat(opts.filePath);
            if (info.isDirectory) {
                info.size = 0;
                opts.reader = new Buffer();
            }
        }
        const mode = opts.fileMode || (info && info.mode) ||
            parseInt("777", 8) & 0xfff, mtime = Math.floor(opts.mtime ?? (info?.mtime ?? new Date()).valueOf() / 1000), uid = opts.uid || 0, gid = opts.gid || 0;
        if (typeof opts.owner === "string" && opts.owner.length >= 32) {
            throw new Error("ustar format does not allow owner name length >= 32 bytes");
        }
        if (typeof opts.group === "string" && opts.group.length >= 32) {
            throw new Error("ustar format does not allow group name length >= 32 bytes");
        }
        const fileSize = info?.size ?? opts.contentSize;
        assert(fileSize != null, "fileSize must be set");
        const type = opts.type
            ? FileTypes[opts.type]
            : (info?.isDirectory ? FileTypes.directory : FileTypes.file);
        const tarData = {
            fileName,
            fileNamePrefix,
            fileMode: pad(mode, 7),
            uid: pad(uid, 7),
            gid: pad(gid, 7),
            fileSize: pad(fileSize, 11),
            mtime: pad(mtime, 11),
            checksum: "        ",
            type: type.toString(),
            ustar,
            owner: opts.owner || "",
            group: opts.group || "",
            filePath: opts.filePath,
            reader: opts.reader,
        };
        let checksum = 0;
        const encoder = new TextEncoder();
        Object.keys(tarData)
            .filter((key) => ["filePath", "reader"].indexOf(key) < 0)
            .forEach(function (key) {
            checksum += encoder
                .encode(tarData[key])
                .reduce((p, c) => p + c, 0);
        });
        tarData.checksum = pad(checksum, 6) + "\u0000 ";
        this.data.push(tarData);
    }
    getReader() {
        const readers = [];
        this.data.forEach((tarData) => {
            let { reader } = tarData;
            const { filePath } = tarData;
            const headerArr = formatHeader(tarData);
            readers.push(new Buffer(headerArr));
            if (!reader) {
                assert(filePath != null);
                reader = new FileReader(filePath);
            }
            readers.push(reader);
            assert(tarData.fileSize != null, "fileSize must be set");
            readers.push(new Buffer(clean(recordSize -
                (parseInt(tarData.fileSize, 8) % recordSize || recordSize))));
        });
        readers.push(new Buffer(clean(recordSize * 2)));
        return new MultiReader(...readers);
    }
}
class TarEntry {
    #header;
    #reader;
    #size;
    #read = 0;
    #consumed = false;
    #entrySize;
    constructor(meta, header, reader) {
        Object.assign(this, meta);
        this.#header = header;
        this.#reader = reader;
        this.#size = this.fileSize || 0;
        const blocks = Math.ceil(this.#size / recordSize);
        this.#entrySize = blocks * recordSize;
    }
    get consumed() {
        return this.#consumed;
    }
    async read(p) {
        const entryBytesLeft = this.#entrySize - this.#read;
        const bufSize = Math.min(p.length, entryBytesLeft);
        if (entryBytesLeft <= 0) {
            this.#consumed = true;
            return null;
        }
        const block = new Uint8Array(bufSize);
        const n = await readBlock(this.#reader, block);
        const bytesLeft = this.#size - this.#read;
        this.#read += n || 0;
        if (n === null || bytesLeft <= 0) {
            if (n === null)
                this.#consumed = true;
            return null;
        }
        const offset = bytesLeft < n ? bytesLeft : n;
        p.set(block.subarray(0, offset), 0);
        return offset < 0 ? n - Math.abs(offset) : offset;
    }
    async discard() {
        if (this.#consumed)
            return;
        this.#consumed = true;
        if (typeof this.#reader.seek === "function") {
            await this.#reader.seek(this.#entrySize - this.#read, Deno.SeekMode.Current);
            this.#read = this.#entrySize;
        }
        else {
            await readAll(this);
        }
    }
}
export class Untar {
    reader;
    block;
    #entry;
    constructor(reader) {
        this.reader = reader;
        this.block = new Uint8Array(recordSize);
    }
    #checksum(header) {
        let sum = initialChecksum;
        for (let i = 0; i < 512; i++) {
            if (i >= 148 && i < 156) {
                continue;
            }
            sum += header[i];
        }
        return sum;
    }
    async #getHeader() {
        await readBlock(this.reader, this.block);
        const header = parseHeader(this.block);
        const decoder = new TextDecoder();
        const checksum = this.#checksum(this.block);
        if (parseInt(decoder.decode(header.checksum), 8) !== checksum) {
            if (checksum === initialChecksum) {
                return null;
            }
            throw new Error("checksum error");
        }
        const magic = decoder.decode(header.ustar);
        if (magic.indexOf("ustar")) {
            throw new Error(`unsupported archive format: ${magic}`);
        }
        return header;
    }
    #getMetadata(header) {
        const decoder = new TextDecoder();
        const meta = {
            fileName: decoder.decode(trim(header.fileName)),
        };
        const fileNamePrefix = trim(header.fileNamePrefix);
        if (fileNamePrefix.byteLength > 0) {
            meta.fileName = decoder.decode(fileNamePrefix) + "/" + meta.fileName;
        }
        ["fileMode", "mtime", "uid", "gid"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = parseInt(decoder.decode(arr), 8);
            }
        });
        ["owner", "group", "type"].forEach((key) => {
            const arr = trim(header[key]);
            if (arr.byteLength > 0) {
                meta[key] = decoder.decode(arr);
            }
        });
        meta.fileSize = parseInt(decoder.decode(header.fileSize), 8);
        meta.type = FileTypes[parseInt(meta.type)] ?? meta.type;
        return meta;
    }
    async extract() {
        if (this.#entry && !this.#entry.consumed) {
            await this.#entry.discard();
        }
        const header = await this.#getHeader();
        if (header === null)
            return null;
        const meta = this.#getMetadata(header);
        this.#entry = new TarEntry(meta, header, this.reader);
        return this.#entry;
    }
    async *[Symbol.asyncIterator]() {
        while (true) {
            const entry = await this.extract();
            if (entry === null)
                return;
            yield entry;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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