function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i2 = 0; i2 <= b.length; i2++){
        matrix[i2] = [
            i2
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i1 = 1; i1 <= b.length; i1++){
        for(let j = 1; j <= a.length; j++){
            if (b.charAt(i1 - 1) == a.charAt(j - 1)) {
                matrix[i1][j] = matrix[i1 - 1][j - 1];
            } else {
                matrix[i1][j] = Math.min(matrix[i1 - 1][j - 1] + 1, Math.min(matrix[i1][j - 1] + 1, matrix[i1 - 1][j] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str1) {
    return str1.replace(/-([a-z])/g, (g)=>g[1].toUpperCase()
    );
}
function getOption(flags, name1) {
    while(name1[0] === "-"){
        name1 = name1.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name1)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option1, options) {
    const optionNames = options.map((option)=>[
            option.name,
            ...option.aliases ?? []
        ]
    ).flat().map((option)=>getFlag(option)
    );
    return didYouMean(" Did you mean option", getFlag(option1), optionNames);
}
function didYouMeanType(type1, types3) {
    return didYouMean(" Did you mean type", type1, types3);
}
function didYouMean(message, type2, types4) {
    const match = closest(type2, types4);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name2) {
    if (name2.startsWith("-")) {
        return name2;
    }
    if (name2.length > 1) {
        return `--${name2}`;
    }
    return `-${name2}`;
}
function isOption(option, name3) {
    return option.name === name3 || option.aliases && option.aliases.indexOf(name3) !== -1;
}
function closest(str2, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i1 = 0; i1 < arr.length; i1++){
        const dist = distance(str2, arr[i1]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i1;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOption extends FlagsError {
    constructor(option, options){
        super(`Unknown required option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownRequiredOption.prototype);
    }
}
class UnknownConflictingOption extends FlagsError {
    constructor(option, options){
        super(`Unknown conflicting option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownConflictingOption.prototype);
    }
}
class UnknownType extends FlagsError {
    constructor(type3, types5){
        super(`Unknown type "${type3}".${didYouMeanType(type3, types5)}`);
        Object.setPrototypeOf(this, UnknownType.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOption extends ValidationError {
    constructor(name4){
        super(`Option "${getFlag(name4).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOption.prototype);
    }
}
class UnknownOption extends ValidationError {
    constructor(option, options){
        super(`Unknown option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownOption.prototype);
    }
}
class MissingOptionValue extends ValidationError {
    constructor(option){
        super(`Missing value for option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
class InvalidOptionValue extends ValidationError {
    constructor(option, expected, value){
        super(`Option "${getFlag(option)}" must be of type "${expected}", but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValue.prototype);
    }
}
class OptionNotCombinable extends ValidationError {
    constructor(option){
        super(`Option "${getFlag(option)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinable.prototype);
    }
}
class ConflictingOption extends ValidationError {
    constructor(option, conflictingOption){
        super(`Option "${getFlag(option)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOption.prototype);
    }
}
class DependingOption extends ValidationError {
    constructor(option, dependingOption){
        super(`Option "${getFlag(option)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOption.prototype);
    }
}
class MissingRequiredOption extends ValidationError {
    constructor(option){
        super(`Missing required option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingRequiredOption.prototype);
    }
}
class RequiredArgumentFollowsOptionalArgument extends ValidationError {
    constructor(arg){
        super(`An required argument cannot follow an optional argument, but "${arg}"  is defined as required.`);
        Object.setPrototypeOf(this, RequiredArgumentFollowsOptionalArgument.prototype);
    }
}
class ArgumentFollowsVariadicArgument extends ValidationError {
    constructor(arg){
        super(`An argument cannot follow an variadic argument, but got "${arg}".`);
        Object.setPrototypeOf(this, ArgumentFollowsVariadicArgument.prototype);
    }
}
class NoArguments extends ValidationError {
    constructor(){
        super(`No arguments.`);
        Object.setPrototypeOf(this, NoArguments.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label , name: name5 , value: value1 , type: type4  }, expected){
        super(`${label} "${name5}" must be of type "${type4}", but got "${value1}".` + (expected ? ` Expected values: ${expected.map((value)=>`"${value}"`
        ).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
function normalize(args) {
    const normalized = [];
    let inLiteral = false;
    for (const arg of args){
        if (inLiteral) {
            normalized.push(arg);
        } else if (arg === "--") {
            inLiteral = true;
            normalized.push(arg);
        } else if (arg.length > 1 && arg[0] === "-") {
            const isLong = arg[1] === "-";
            const isDotted = !isLong && arg[2] === ".";
            if (arg.includes("=")) {
                const parts = arg.split("=");
                const flag = parts.shift();
                if (isLong) {
                    normalized.push(flag);
                } else {
                    normalizeShortFlags(flag);
                }
                normalized.push(parts.join("="));
            } else if (isLong || isDotted) {
                normalized.push(arg);
            } else {
                normalizeShortFlags(arg);
            }
        } else {
            normalized.push(arg);
        }
    }
    return normalized;
    function normalizeShortFlags(flag) {
        const flags = flag.slice(1).split("");
        if (isNaN(Number(flag[flag.length - 1]))) {
            flags.forEach((val)=>normalized.push(`-${val}`)
            );
        } else {
            normalized.push(`-${flags.shift()}`);
            normalized.push(flags.join(""));
        }
    }
}
var OptionType;
(function(OptionType1) {
    OptionType1["STRING"] = "string";
    OptionType1["NUMBER"] = "number";
    OptionType1["INTEGER"] = "integer";
    OptionType1["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {
}));
const __boolean = (type5)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type5.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type5.value)) {
        return false;
    }
    throw new InvalidTypeError(type5);
};
const number = (type6)=>{
    const value = Number(type6.value);
    if (Number.isFinite(value)) {
        return value;
    }
    throw new InvalidTypeError(type6);
};
const string = ({ value  })=>{
    return value;
};
function validateFlags(flags, values1, _knownFlaks, allowEmpty, optionNames = {
}) {
    const defaultValues = {
    };
    for (const option of flags){
        let name6;
        let defaultValue = undefined;
        if (option.name.startsWith("no-")) {
            const propName = option.name.replace(/^no-/, "");
            if (propName in values1) {
                continue;
            }
            const positiveOption = getOption(flags, propName);
            if (positiveOption) {
                continue;
            }
            name6 = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name6) {
            name6 = paramCaseToCamelCase(option.name);
        }
        if (!(name6 in optionNames)) {
            optionNames[name6] = option.name;
        }
        const hasDefaultValue = typeof values1[name6] === "undefined" && (typeof option.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            values1[name6] = getDefaultValue(option) ?? defaultValue;
            defaultValues[option.name] = true;
            if (typeof option.value === "function") {
                values1[name6] = option.value(values1[name6]);
            }
        }
    }
    const keys1 = Object.keys(values1);
    if (keys1.length === 0 && allowEmpty) {
        return;
    }
    const options = keys1.map((name7)=>({
            name: name7,
            option: getOption(flags, optionNames[name7])
        })
    );
    for (const { name: name1 , option: option1  } of options){
        if (!option1) {
            throw new UnknownOption(name1, flags);
        }
        if (option1.standalone) {
            if (keys1.length > 1) {
                if (options.every(({ option: opt  })=>opt && (option1 === opt || defaultValues[opt.name])
                )) {
                    return;
                }
                throw new OptionNotCombinable(option1.name);
            }
            return;
        }
        option1.conflicts?.forEach((flag)=>{
            if (isset(flag, values1)) {
                throw new ConflictingOption(option1.name, flag);
            }
        });
        option1.depends?.forEach((flag)=>{
            if (!isset(flag, values1) && !defaultValues[option1.name]) {
                throw new DependingOption(option1.name, flag);
            }
        });
        const isArray2 = (option1.args?.length || 0) > 1;
        option1.args?.forEach((arg, i)=>{
            if (arg.requiredValue && (typeof values1[name1] === "undefined" || isArray2 && typeof values1[name1][i] === "undefined")) {
                throw new MissingOptionValue(option1.name);
            }
        });
    }
    for (const option2 of flags){
        if (option2.required && !(paramCaseToCamelCase(option2.name) in values1)) {
            if ((!option2.conflicts || !option2.conflicts.find((flag)=>!!values1[flag]
            )) && !options.find((opt)=>opt.option?.conflicts?.find((flag)=>flag === option2.name
                )
            )) {
                throw new MissingRequiredOption(option2.name);
            }
        }
    }
    if (keys1.length === 0 && !allowEmpty) {
        throw new NoArguments();
    }
}
function isset(flag, values2) {
    const name = paramCaseToCamelCase(flag);
    return typeof values2[name] !== "undefined";
}
const integer = (type7)=>{
    const value = Number(type7.value);
    if (Number.isInteger(value)) {
        return value;
    }
    throw new InvalidTypeError(type7);
};
const Types = {
    [OptionType.STRING]: string,
    [OptionType.NUMBER]: number,
    [OptionType.INTEGER]: integer,
    [OptionType.BOOLEAN]: __boolean
};
function parseFlags(args2, opts = {
}) {
    !opts.flags && (opts.flags = []);
    const normalized = normalize(args2);
    let inLiteral = false;
    let negate1 = false;
    const flags = {
    };
    const optionNames = {
    };
    let literal1 = [];
    let unknown = [];
    let stopEarly = null;
    opts.flags.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOption(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOption(flag, opts.flags ?? []);
            }
        });
    });
    for(let i3 = 0; i3 < normalized.length; i3++){
        let option1;
        let args1;
        const current = normalized[i3];
        if (inLiteral) {
            literal1.push(current);
            continue;
        }
        if (current === "--") {
            inLiteral = true;
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        const next = ()=>normalized[i3 + 1]
        ;
        if (isFlag) {
            if (current[2] === "-" || current[1] === "-" && current.length === 3) {
                throw new UnknownOption(current, opts.flags);
            }
            negate1 = current.startsWith("--no-");
            option1 = getOption(opts.flags, current);
            if (!option1) {
                if (opts.flags.length) {
                    throw new UnknownOption(current, opts.flags);
                }
                option1 = {
                    name: current.replace(/^-+/, ""),
                    optionalValue: true,
                    type: OptionType.STRING
                };
            }
            const positiveName = option1.name.replace(/^no-?/, "");
            const propName = paramCaseToCamelCase(positiveName);
            if (typeof flags[propName] !== "undefined" && !option1.collect) {
                throw new DuplicateOption(current);
            }
            args1 = option1.args?.length ? option1.args : [
                {
                    type: option1.type,
                    requiredValue: option1.requiredValue,
                    optionalValue: option1.optionalValue,
                    variadic: option1.variadic,
                    list: option1.list,
                    separator: option1.separator
                }
            ];
            let argIndex = 0;
            let inOptionalArg = false;
            const previous = flags[propName];
            parseNext(option1, args1);
            if (typeof flags[propName] === "undefined") {
                if (typeof option1.default !== "undefined") {
                    flags[propName] = getDefaultValue(option1);
                } else if (args1[argIndex].requiredValue) {
                    throw new MissingOptionValue(option1.name);
                } else {
                    flags[propName] = true;
                }
            }
            if (option1.value) {
                flags[propName] = option1.value(flags[propName], previous);
            } else if (option1.collect) {
                const value = Array.isArray(previous) ? previous : [];
                value.push(flags[propName]);
                flags[propName] = value;
            }
            optionNames[propName] = option1.name;
            opts.option?.(option1, flags[propName]);
            function parseNext(option2, args) {
                const arg1 = args[argIndex];
                if (!arg1) {
                    const flag = next();
                    throw new UnknownOption(flag, opts.flags ?? []);
                }
                if (!arg1.type) {
                    arg1.type = OptionType.BOOLEAN;
                }
                if (option2.args?.length) {
                    if ((typeof arg1.optionalValue === "undefined" || arg1.optionalValue === false) && typeof arg1.requiredValue === "undefined") {
                        arg1.requiredValue = true;
                    }
                } else {
                    if (arg1.type !== OptionType.BOOLEAN && (typeof arg1.optionalValue === "undefined" || arg1.optionalValue === false) && typeof arg1.requiredValue === "undefined") {
                        arg1.requiredValue = true;
                    }
                }
                if (arg1.requiredValue) {
                    if (inOptionalArg) {
                        throw new RequiredArgumentFollowsOptionalArgument(option2.name);
                    }
                } else {
                    inOptionalArg = true;
                }
                if (negate1) {
                    flags[propName] = false;
                    return;
                }
                let result2;
                let increase = false;
                if (arg1.list && hasNext(arg1)) {
                    const parsed = next().split(arg1.separator || ",").map((nextValue)=>{
                        const value = parseValue(option2, arg1, nextValue);
                        if (typeof value === "undefined") {
                            throw new InvalidOptionValue(option2.name, arg1.type ?? "?", nextValue);
                        }
                        return value;
                    });
                    if (parsed?.length) {
                        result2 = parsed;
                    }
                } else {
                    if (hasNext(arg1)) {
                        result2 = parseValue(option2, arg1, next());
                    } else if (arg1.optionalValue && arg1.type === OptionType.BOOLEAN) {
                        result2 = true;
                    }
                }
                if (increase) {
                    i3++;
                    if (!arg1.variadic) {
                        argIndex++;
                    } else if (args[argIndex + 1]) {
                        throw new ArgumentFollowsVariadicArgument(next());
                    }
                }
                if (typeof result2 !== "undefined" && (args.length > 1 || arg1.variadic)) {
                    if (!flags[propName]) {
                        flags[propName] = [];
                    }
                    flags[propName].push(result2);
                    if (hasNext(arg1)) {
                        parseNext(option2, args);
                    }
                } else {
                    flags[propName] = result2;
                }
                function hasNext(arg) {
                    return !!(normalized[i3 + 1] && (arg.optionalValue || arg.requiredValue || arg.variadic) && (normalized[i3 + 1][0] !== "-" || arg.type === OptionType.NUMBER && !isNaN(Number(normalized[i3 + 1]))) && arg);
                }
                function parseValue(option, arg, value) {
                    const type8 = arg.type || OptionType.STRING;
                    const result3 = opts.parse ? opts.parse({
                        label: "Option",
                        type: type8,
                        name: `--${option.name}`,
                        value
                    }) : parseFlagValue(option, arg, value);
                    if (typeof result3 !== "undefined") {
                        increase = true;
                    }
                    return result3;
                }
            }
        } else {
            if (opts.stopEarly) {
                stopEarly = current;
                break;
            }
            unknown.push(current);
        }
    }
    if (stopEarly) {
        const stopEarlyArgIndex = args2.indexOf(stopEarly);
        if (stopEarlyArgIndex !== -1) {
            const doubleDashIndex = args2.indexOf("--");
            unknown = args2.slice(stopEarlyArgIndex, doubleDashIndex === -1 ? undefined : doubleDashIndex);
            if (doubleDashIndex !== -1) {
                literal1 = args2.slice(doubleDashIndex + 1);
            }
        }
    }
    if (opts.flags?.length) {
        validateFlags(opts.flags, flags, opts.knownFlaks, opts.allowEmpty, optionNames);
    }
    const result1 = Object.keys(flags).reduce((result3, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result2, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result2[subKey] = flags[key];
                } else {
                    result2[subKey] = result2[subKey] ?? {
                    };
                }
                return result2[subKey];
            }, result3);
        } else {
            result3[key] = flags[key];
        }
        return result3;
    }, {
    });
    return {
        flags: result1,
        unknown,
        literal: literal1
    };
}
function parseFlagValue(option, arg, value) {
    const type9 = arg.type || OptionType.STRING;
    const parseType = Types[type9];
    if (!parseType) {
        throw new UnknownType(type9, Object.keys(Types));
    }
    return parseType({
        label: "Option",
        type: type9,
        name: `--${option.name}`,
        value
    });
}
function getPermissions() {
    return hasPermissions([
        "env",
        "hrtime",
        "net",
        "plugin",
        "read",
        "run",
        "write", 
    ]);
}
function isUnstable() {
    return !!Deno.permissions;
}
function didYouMeanCommand(command1, commands1, excludes = []) {
    const commandNames = commands1.map((command)=>command.getName()
    ).filter((command)=>!excludes.includes(command)
    );
    return didYouMean(" Did you mean command", command1, commandNames);
}
async function hasPermission(permission) {
    try {
        return (await Deno.permissions?.query?.({
            name: permission
        }))?.state === "granted";
    } catch  {
        return false;
    }
}
async function hasPermissions(names) {
    const permissions = {
    };
    await Promise.all(names.map((name8)=>hasPermission(name8).then((hasPermission1)=>permissions[name8] = hasPermission1
        )
    ));
    return permissions;
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition
    };
}
function parseArgumentsDefinition(argsDefinition) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg of parts){
        if (hasVariadic) {
            throw new ArgumentFollowsVariadicArgument(arg);
        }
        const parts = arg.split(ARGUMENT_DETAILS_REGEX);
        const type10 = parts[2] || OptionType.STRING;
        const details = {
            optionalValue: arg[0] !== "<",
            name: parts[1],
            action: parts[3] || type10,
            variadic: false,
            list: type10 ? arg.indexOf(type10 + "[]") !== -1 : false,
            type: type10
        };
        if (!details.optionalValue && hasOptional) {
            throw new RequiredArgumentFollowsOptionalArgument(details.name);
        }
        if (arg[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        if (details.name) {
            argumentDetails.push(details);
        }
    }
    return argumentDetails;
}
const { Deno: Deno2  } = globalThis;
const noColor = typeof Deno2?.noColor === "boolean" ? Deno2.noColor : true;
let enabled = !noColor;
function setColorEnabled(value) {
    if (noColor) {
        return;
    }
    enabled = value;
}
function getColorEnabled() {
    return enabled;
}
function code(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run(str3, code1) {
    return enabled ? `${code1.open}${str3.replace(code1.regexp, code1.open)}${code1.close}` : str3;
}
function reset(str4) {
    return run(str4, code([
        0
    ], 0));
}
function bold(str5) {
    return run(str5, code([
        1
    ], 22));
}
function dim(str6) {
    return run(str6, code([
        2
    ], 22));
}
function italic(str7) {
    return run(str7, code([
        3
    ], 23));
}
function underline(str8) {
    return run(str8, code([
        4
    ], 24));
}
function inverse(str9) {
    return run(str9, code([
        7
    ], 27));
}
function hidden(str10) {
    return run(str10, code([
        8
    ], 28));
}
function strikethrough(str11) {
    return run(str11, code([
        9
    ], 29));
}
function black(str12) {
    return run(str12, code([
        30
    ], 39));
}
function red(str13) {
    return run(str13, code([
        31
    ], 39));
}
function green(str14) {
    return run(str14, code([
        32
    ], 39));
}
function yellow(str15) {
    return run(str15, code([
        33
    ], 39));
}
function blue(str16) {
    return run(str16, code([
        34
    ], 39));
}
function magenta(str17) {
    return run(str17, code([
        35
    ], 39));
}
function cyan(str18) {
    return run(str18, code([
        36
    ], 39));
}
function white(str19) {
    return run(str19, code([
        37
    ], 39));
}
function gray(str20) {
    return brightBlack(str20);
}
function brightBlack(str21) {
    return run(str21, code([
        90
    ], 39));
}
function brightRed(str22) {
    return run(str22, code([
        91
    ], 39));
}
function brightGreen(str23) {
    return run(str23, code([
        92
    ], 39));
}
function brightYellow(str24) {
    return run(str24, code([
        93
    ], 39));
}
function brightBlue(str25) {
    return run(str25, code([
        94
    ], 39));
}
function brightMagenta(str26) {
    return run(str26, code([
        95
    ], 39));
}
function brightCyan(str27) {
    return run(str27, code([
        96
    ], 39));
}
function brightWhite(str28) {
    return run(str28, code([
        97
    ], 39));
}
function bgBlack(str29) {
    return run(str29, code([
        40
    ], 49));
}
function bgRed(str30) {
    return run(str30, code([
        41
    ], 49));
}
function bgGreen(str31) {
    return run(str31, code([
        42
    ], 49));
}
function bgYellow(str32) {
    return run(str32, code([
        43
    ], 49));
}
function bgBlue(str33) {
    return run(str33, code([
        44
    ], 49));
}
function bgMagenta(str34) {
    return run(str34, code([
        45
    ], 49));
}
function bgCyan(str35) {
    return run(str35, code([
        46
    ], 49));
}
function bgWhite(str36) {
    return run(str36, code([
        47
    ], 49));
}
function bgBrightBlack(str37) {
    return run(str37, code([
        100
    ], 49));
}
function bgBrightRed(str38) {
    return run(str38, code([
        101
    ], 49));
}
function bgBrightGreen(str39) {
    return run(str39, code([
        102
    ], 49));
}
function bgBrightYellow(str40) {
    return run(str40, code([
        103
    ], 49));
}
function bgBrightBlue(str41) {
    return run(str41, code([
        104
    ], 49));
}
function bgBrightMagenta(str42) {
    return run(str42, code([
        105
    ], 49));
}
function bgBrightCyan(str43) {
    return run(str43, code([
        106
    ], 49));
}
function bgBrightWhite(str44) {
    return run(str44, code([
        107
    ], 49));
}
function clampAndTruncate(n, max1 = 255, min1 = 0) {
    return Math.trunc(Math.max(Math.min(n, max1), min1));
}
function rgb8(str45, color) {
    return run(str45, code([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str46, color) {
    return run(str46, code([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str47, color) {
    if (typeof color === "number") {
        return run(str47, code([
            38,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 39));
    }
    return run(str47, code([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 39));
}
function bgRgb24(str48, color) {
    if (typeof color === "number") {
        return run(str48, code([
            48,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 49));
    }
    return run(str48, code([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 49));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor(string2) {
    return string2.replace(ANSI_PATTERN, "");
}
const mod = {
    setColorEnabled: setColorEnabled,
    getColorEnabled: getColorEnabled,
    reset: reset,
    bold: bold,
    dim: dim,
    italic: italic,
    underline: underline,
    inverse: inverse,
    hidden: hidden,
    strikethrough: strikethrough,
    black: black,
    red: red,
    green: green,
    yellow: yellow,
    blue: blue,
    magenta: magenta,
    cyan: cyan,
    white: white,
    gray: gray,
    brightBlack: brightBlack,
    brightRed: brightRed,
    brightGreen: brightGreen,
    brightYellow: brightYellow,
    brightBlue: brightBlue,
    brightMagenta: brightMagenta,
    brightCyan: brightCyan,
    brightWhite: brightWhite,
    bgBlack: bgBlack,
    bgRed: bgRed,
    bgGreen: bgGreen,
    bgYellow: bgYellow,
    bgBlue: bgBlue,
    bgMagenta: bgMagenta,
    bgCyan: bgCyan,
    bgWhite: bgWhite,
    bgBrightBlack: bgBrightBlack,
    bgBrightRed: bgBrightRed,
    bgBrightGreen: bgBrightGreen,
    bgBrightYellow: bgBrightYellow,
    bgBrightBlue: bgBrightBlue,
    bgBrightMagenta: bgBrightMagenta,
    bgBrightCyan: bgBrightCyan,
    bgBrightWhite: bgBrightWhite,
    rgb8: rgb8,
    bgRgb8: bgRgb8,
    rgb24: rgb24,
    bgRgb24: bgRgb24,
    stripColor: stripColor
};
class CommandError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    constructor(message, { exitCode  } = {
    }){
        super(message);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 1;
    }
}
class DuplicateOptionName extends CommandError {
    constructor(name9){
        super(`Option with name "${getFlag(name9)}" already exists.`);
        Object.setPrototypeOf(this, DuplicateOptionName.prototype);
    }
}
class MissingCommandName extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandName.prototype);
    }
}
class DuplicateCommandName extends CommandError {
    constructor(name10){
        super(`Duplicate command name "${name10}".`);
        Object.setPrototypeOf(this, DuplicateCommandName.prototype);
    }
}
class DuplicateCommandAlias extends CommandError {
    constructor(alias){
        super(`Duplicate command alias "${alias}".`);
        Object.setPrototypeOf(this, DuplicateCommandAlias.prototype);
    }
}
class CommandNotFound extends CommandError {
    constructor(name11, commands2, excluded){
        super(`Unknown command "${name11}".${didYouMeanCommand(name11, commands2, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class DuplicateType extends CommandError {
    constructor(name12){
        super(`Type with name "${name12}" already exists.`);
        Object.setPrototypeOf(this, DuplicateType.prototype);
    }
}
class DuplicateCompletion extends CommandError {
    constructor(name13){
        super(`Completion with name "${name13}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletion.prototype);
    }
}
class DuplicateExample extends CommandError {
    constructor(name14){
        super(`Example with name "${name14}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExample.prototype);
    }
}
class DuplicateEnvironmentVariable extends CommandError {
    constructor(name15){
        super(`Environment variable with name "${name15}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvironmentVariable.prototype);
    }
}
class EnvironmentVariableSingleValue extends CommandError {
    constructor(name16){
        super(`An environment variable can only have one value, but "${name16}" has more than one.`);
        Object.setPrototypeOf(this, EnvironmentVariableSingleValue.prototype);
    }
}
class EnvironmentVariableOptionalValue extends CommandError {
    constructor(name17){
        super(`An environment variable cannot have an optional value, but "${name17}" is defined as optional.`);
        Object.setPrototypeOf(this, EnvironmentVariableOptionalValue.prototype);
    }
}
class EnvironmentVariableVariadicValue extends CommandError {
    constructor(name18){
        super(`An environment variable cannot have an variadic value, but "${name18}" is defined as variadic.`);
        Object.setPrototypeOf(this, EnvironmentVariableVariadicValue.prototype);
    }
}
class DefaultCommandNotFound extends CommandError {
    constructor(name19, commands3){
        super(`Default command "${name19}" not found.${didYouMeanCommand(name19, commands3)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFound.prototype);
    }
}
class CommandExecutableNotFound extends CommandError {
    constructor(name20, files){
        super(`Command executable not found: ${name20}:\n    - ${files.join("\\n    - ")}`);
        Object.setPrototypeOf(this, CommandExecutableNotFound.prototype);
    }
}
class UnknownCompletionCommand extends CommandError {
    constructor(name21, commands4){
        super(`Auto-completion failed. Unknown command "${name21}".${didYouMeanCommand(name21, commands4)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommand.prototype);
    }
}
class UnknownCommand extends ValidationError1 {
    constructor(name22, commands5, excluded){
        super(`Unknown command "${name22}".${didYouMeanCommand(name22, commands5, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class NoArgumentsAllowed extends ValidationError1 {
    constructor(name23){
        super(`No arguments allowed for command "${name23}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowed.prototype);
    }
}
class MissingArguments extends ValidationError1 {
    constructor(args){
        super("Missing argument(s): " + args.join(", "));
        Object.setPrototypeOf(this, MissingArguments.prototype);
    }
}
class MissingArgument extends ValidationError1 {
    constructor(arg){
        super(`Missing argument "${arg}".`);
        Object.setPrototypeOf(this, MissingArgument.prototype);
    }
}
class TooManyArguments extends ValidationError1 {
    constructor(args){
        super(`Too many arguments: ${args.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArguments.prototype);
    }
}
class Type {
}
class BooleanType extends Type {
    parse(type11) {
        return __boolean(type11);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class NumberType extends Type {
    parse(type12) {
        return number(type12);
    }
}
class StringType extends Type {
    parse(type13) {
        return string(type13);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    value;
    options = {
    };
    get length() {
        return this.toString().length;
    }
    static from(value) {
        const cell = new this(value);
        if (value instanceof Cell) {
            cell.options = {
                ...value.options
            };
        }
        return cell;
    }
    constructor(value){
        this.value = value;
    }
    toString() {
        return this.value.toString();
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        const cell = new Cell(value ?? this);
        cell.options = {
            ...this.options
        };
        return cell;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class Row extends Array {
    options = {
    };
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell
        ));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder()
        );
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
function consumeWords(length, content) {
    let consumed = "";
    const words1 = content.split(/ /g);
    for(let i4 = 0; i4 < words1.length; i4++){
        let word = words1[i4];
        const hasLineBreak = word.indexOf("\n") !== -1;
        if (hasLineBreak) {
            word = word.split("\n").shift();
        }
        if (consumed) {
            const nextLength = stripColor(word).length;
            const consumedLength = stripColor(consumed).length;
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i4 > 0 ? " " : "") + word;
        if (hasLineBreak) {
            break;
        }
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    return Math.max(...rows.map((row)=>(row[index] instanceof Cell && row[index].getColSpan() > 1 ? "" : row[index]?.toString() || "").split("\n").map((r)=>{
            const str49 = typeof maxWidth === "undefined" ? r : consumeWords(maxWidth, r);
            return stripColor(str49).length || 0;
        })
    ).flat());
}
class TableLayout {
    table;
    options;
    constructor(table, options){
        this.table = table;
        this.options = options;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const header = this.table.getHeader();
        const rows = this.spanRows(header ? [
            header,
            ...this.table
        ] : this.table.slice());
        const columns = Math.max(...rows.map((row)=>row.length
        ));
        for (const row1 of rows){
            const length = row1.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i5 = 0; i5 < diff; i5++){
                    row1.push(this.createCell(null, row1));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const minColWidth = Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth;
            const maxColWidth = Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth;
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding;
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    spanRows(_rows, rowIndex = 0, colIndex = 0, rowSpan = [], colSpan = 1) {
        const rows = _rows;
        if (rowIndex >= rows.length && rowSpan.every((span)=>span === 1
        )) {
            return rows;
        } else if (rows[rowIndex] && colIndex >= rows[rowIndex].length && colIndex >= rowSpan.length && colSpan === 1) {
            return this.spanRows(rows, ++rowIndex, 0, rowSpan, 1);
        }
        if (colSpan > 1) {
            colSpan--;
            rowSpan[colIndex] = rowSpan[colIndex - 1];
            rows[rowIndex].splice(colIndex - 1, 0, rows[rowIndex][colIndex - 1]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        if (colIndex === 0) {
            rows[rowIndex] = this.createRow(rows[rowIndex] || []);
        }
        if (rowSpan[colIndex] > 1) {
            rowSpan[colIndex]--;
            rows[rowIndex].splice(colIndex, 0, rows[rowIndex - 1][colIndex]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        rows[rowIndex][colIndex] = this.createCell(rows[rowIndex][colIndex] || null, rows[rowIndex]);
        colSpan = rows[rowIndex][colIndex].getColSpan();
        rowSpan[colIndex] = rows[rowIndex][colIndex].getRowSpan();
        return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row) {
        return Cell.from(cell ?? "").border(row.getBorder(), false).align(row.getAlign(), false);
    }
    renderRows(opts) {
        let result1 = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result1 += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result1.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result4 = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result4 += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result4 += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result4 += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result4 += this.options.chars.right;
            } else if (opts.hasBorder) {
                result4 += " ";
            }
        }
        result4 += "\n";
        if (isMultilineRow) {
            return result4 + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder) {
            result4 += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result4 += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result4;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result5 = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result5 += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result5 += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result5 += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result5 += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current , next  } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next);
        if (opts.hasBorder) {
            result5 += " ".repeat(opts.padding[colIndex]);
        }
        result5 += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result5 += " ".repeat(opts.padding[colIndex]);
        }
        return result5;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, stripColor(cell.toString()).length);
        let words2 = consumeWords(length, cell.toString());
        const breakWord = stripColor(words2).length > length;
        if (breakWord) {
            words2 = words2.slice(0, length);
        }
        const next = cell.toString().slice(words2.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - stripColor(words2).length;
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words2;
        } else if (align === "left") {
            current = words2 + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words2 + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words2;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result6 = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result6 += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result6.length ? " ".repeat(this.options.indent) + result6 + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1
        ;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1
        ;
        let result7 = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result7 += this.options.chars.left;
                } else {
                    result7 += " ";
                }
            } else if (b1Border && b2Border) {
                result7 += this.options.chars.leftMid;
            } else if (b1Border) {
                result7 += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result7 += this.options.chars.topLeft;
            } else {
                result7 += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result7 += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result7 += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result7 += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result7 += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result7 += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result7 += this.options.chars.rightMid;
                } else {
                    result7 += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result7 += this.options.chars.bottom;
                } else {
                    result7 += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result7 += this.options.chars.left;
                } else {
                    result7 += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result7 += this.options.chars.top;
                } else {
                    result7 += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result7 += this.options.chars.right;
                } else {
                    result7 += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result7 += this.options.chars.bottomRight;
            } else if (b1Border) {
                result7 += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result7 += this.options.chars.topRight;
            } else if (b2Border) {
                result7 += this.options.chars.topLeft;
            } else {
                result7 += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result7 += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result7 += this.options.chars.right;
                } else {
                    result7 += " ";
                }
                return result7;
            }
        } else if (b1Border && b2Border) {
            result7 += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result7 += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result7 += this.options.chars.top.repeat(length);
        } else {
            result7 += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result7 += this.options.chars.rightMid;
            } else if (b1Border) {
                result7 += this.options.chars.bottomRight;
            } else if (b2Border) {
                result7 += this.options.chars.topRight;
            } else {
                result7 += " ";
            }
        }
        return result7;
    }
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        }
    };
    headerRow;
    static from(rows) {
        const table = new this(...rows);
        if (rows instanceof Table) {
            table.options = {
                ...rows.options
            };
            table.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)
        ));
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()
        ));
        table.options = {
            ...this.options
        };
        table.headerRow = this.headerRow?.clone();
        return table;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.toString() + "\n"));
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder : false
            )
        );
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class HelpGenerator {
    cmd;
    indent = 2;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd, options = {
    }){
        this.cmd = cmd;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            ...options
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result8 = this.generateHeader() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples() + "\n";
        setColorEnabled(areColorsEnabled);
        return result8;
    }
    generateHeader() {
        const rows = [
            [
                bold("Usage:"),
                magenta(`${this.cmd.getPath()}${this.cmd.getArgsDefinition() ? " " + this.cmd.getArgsDefinition() : ""}`), 
            ], 
        ];
        const version1 = this.cmd.getVersion();
        if (version1) {
            rows.push([
                bold("Version:"),
                yellow(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).indent(this.indent).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                this.cmd.getDescription()
            ], 
        ]).indent(this.indent * 2).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options = this.cmd.getOptions(false);
        if (!options.length) {
            return "";
        }
        const hasTypeDefinitions = !!options.find((option)=>!!option.typeDefinition
        );
        if (hasTypeDefinitions) {
            return this.label("Options") + Table.from([
                ...options.map((option)=>[
                        option.flags.map((flag)=>blue(flag)
                        ).join(", "),
                        highlightArguments(option.typeDefinition || "", this.options.types),
                        red(bold("-")) + " " + option.description.split("\n").shift(),
                        this.generateHints(option), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).maxColWidth([
                60,
                60,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label("Options") + Table.from([
            ...options.map((option)=>[
                    option.flags.map((flag)=>blue(flag)
                    ).join(", "),
                    red(bold("-")) + " " + option.description.split("\n").shift(),
                    this.generateHints(option), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).maxColWidth([
            60,
            80,
            60
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands6 = this.cmd.getCommands(false);
        if (!commands6.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands6.find((command)=>!!command.getArgsDefinition()
        );
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands6.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name24)=>blue(name24)
                        ).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands6.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name25)=>blue(name25)
                    ).join(", "),
                    red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name26)=>blue(name26)
                    ).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    `${red(bold("-"))} ${envVar.description}`, 
                ]
            ), 
        ]).padding(2).indent(this.indent * 2).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim(bold(`${capitalize(example.name)}:`)),
                example.description, 
            ]
        )).padding(1).indent(this.indent * 2).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow(`required`));
        typeof option.default !== "undefined" && hints.push(bold(`Default: `) + inspect(option.default, this.options.colors));
        option.depends?.length && hints.push(yellow(bold(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red(bold(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type14 = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type14 instanceof Type) {
            const possibleValues = type14.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold(`Values: `) + possibleValues.map((value)=>inspect(value, this.options.colors)
                ).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + " ".repeat(this.indent) + bold(`${label}:`) + "\n\n";
    }
}
function capitalize(string3) {
    return (string3?.charAt(0).toUpperCase() + string3.slice(1)) ?? "";
}
function inspect(value, colors) {
    return Deno.inspect(value, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types6 = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition).map((arg)=>highlightArgumentDetails(arg, types6)
    ).join(" ");
}
function highlightArgumentDetails(arg, types7 = true) {
    let str50 = "";
    str50 += yellow(arg.optionalValue ? "[" : "<");
    let name27 = "";
    name27 += arg.name;
    if (arg.variadic) {
        name27 += "...";
    }
    name27 = magenta(name27);
    str50 += name27;
    if (types7) {
        str50 += yellow(":");
        str50 += red(arg.type);
    }
    if (arg.list) {
        str50 += green("[]");
    }
    str50 += yellow(arg.optionalValue ? "]" : ">");
    return str50;
}
class IntegerType extends Type {
    parse(type15) {
        return integer(type15);
    }
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    fn;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = true;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOption;
    _helpOption;
    _help;
    versionOption(flags, desc, opts) {
        this._versionOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        const result9 = splitArguments(nameAndArguments);
        const name28 = result9.flags.shift();
        const aliases = result9.flags;
        if (!name28) {
            throw new MissingCommandName();
        }
        if (this.getBaseCommand(name28, true)) {
            if (!override) {
                throw new DuplicateCommandName(name28);
            }
            this.removeCommand(name28);
        }
        let description1;
        let cmd;
        if (typeof cmdOrDescription === "string") {
            description1 = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd = cmdOrDescription.reset();
        } else {
            cmd = new Command();
        }
        cmd._name = name28;
        cmd._parent = this;
        if (description1) {
            cmd.description(description1);
        }
        if (result9.typeDefinition) {
            cmd.arguments(result9.typeDefinition);
        }
        aliases.forEach((alias)=>cmd.alias(alias)
        );
        this.commands.set(name28, cmd);
        this.select(name28);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAlias(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this.cmd = this;
        return this;
    }
    select(name29) {
        const cmd = this.getBaseCommand(name29, true);
        if (!cmd) {
            throw new CommandNotFound(name29, this.getBaseCommands(true));
        }
        this.cmd = cmd;
        return this;
    }
    name(name30) {
        this.cmd._name = name30;
        return this;
    }
    version(version2) {
        if (typeof version2 === "string") {
            this.cmd.ver = ()=>version2
            ;
        } else if (typeof version2 === "function") {
            this.cmd.ver = version2;
        }
        return this;
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help
            ;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd)=>HelpGenerator.generate(cmd, help)
            ;
        }
        return this;
    }
    description(description2) {
        this.cmd.desc = description2;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.fn = fn;
        return this;
    }
    allowEmpty(allowEmpty = true) {
        this.cmd._allowEmpty = allowEmpty;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name31) {
        this.cmd.defaultCommand = name31;
        return this;
    }
    globalType(name32, type16, options) {
        return this.type(name32, type16, {
            ...options,
            global: true
        });
    }
    type(name33, handler, options) {
        if (this.cmd.types.get(name33) && !options?.override) {
            throw new DuplicateType(name33);
        }
        this.cmd.types.set(name33, {
            ...options,
            name: name33,
            handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd, parent1)=>handler.complete?.(cmd, parent1) || []
            ;
            this.complete(name33, completeHandler, options);
        }
        return this;
    }
    globalComplete(name34, complete, options) {
        return this.complete(name34, complete, {
            ...options,
            global: true
        });
    }
    complete(name35, complete, options) {
        if (this.cmd.completions.has(name35) && !options?.override) {
            throw new DuplicateCompletion(name35);
        }
        this.cmd.completions.set(name35, {
            name: name35,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    shouldThrowErrors() {
        return this.cmd.throwOnError || !!this.cmd._parent?.shouldThrowErrors();
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts
            });
        }
        const result10 = splitArguments(flags);
        const args = result10.typeDefinition ? parseArgumentsDefinition(result10.typeDefinition) : [];
        const option = {
            ...opts,
            name: "",
            description: desc,
            args,
            flags: result10.flags,
            typeDefinition: result10.typeDefinition
        };
        if (option.separator) {
            for (const arg of args){
                if (arg.list) {
                    arg.separator = option.separator;
                }
            }
        }
        for (const part of option.flags){
            const arg = part.trim();
            const isLong = /^--/.test(arg);
            const name36 = isLong ? arg.slice(2) : arg.slice(1);
            if (this.cmd.getBaseOption(name36, true)) {
                if (opts?.override) {
                    this.removeOption(name36);
                } else {
                    throw new DuplicateOptionName(name36);
                }
            }
            if (!option.name && isLong) {
                option.name = name36;
            } else if (!option.aliases) {
                option.aliases = [
                    name36
                ];
            } else {
                option.aliases.push(name36);
            }
        }
        if (option.prepend) {
            this.cmd.options.unshift(option);
        } else {
            this.cmd.options.push(option);
        }
        return this;
    }
    example(name37, description3) {
        if (this.cmd.hasExample(name37)) {
            throw new DuplicateExample(name37);
        }
        this.cmd.examples.push({
            name: name37,
            description: description3
        });
        return this;
    }
    globalEnv(name38, description4, options) {
        return this.env(name38, description4, {
            ...options,
            global: true
        });
    }
    env(name39, description5, options) {
        const result11 = splitArguments(name39);
        if (!result11.typeDefinition) {
            result11.typeDefinition = "<value:boolean>";
        }
        if (result11.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true)
        )) {
            throw new DuplicateEnvironmentVariable(name39);
        }
        const details = parseArgumentsDefinition(result11.typeDefinition);
        if (details.length > 1) {
            throw new EnvironmentVariableSingleValue(name39);
        } else if (details.length && details[0].optionalValue) {
            throw new EnvironmentVariableOptionalValue(name39);
        } else if (details.length && details[0].variadic) {
            throw new EnvironmentVariableVariadicValue(name39);
        }
        this.cmd.envVars.push({
            name: result11.flags[0],
            names: result11.flags,
            description: description5,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    async parse(args = Deno.args, dry) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = args;
            const subCommand = args.length > 0 && this.getCommand(args[0], true);
            if (subCommand) {
                subCommand._globalParent = this;
                return await subCommand.parse(this.rawArgs.slice(1), dry);
            }
            const result12 = {
                options: {
                },
                args: this.rawArgs,
                cmd: this,
                literal: this.literalArgs
            };
            if (this.isExecutable) {
                if (!dry) {
                    await this.executeExecutable(this.rawArgs);
                }
                return result12;
            } else if (this._useRawArgs) {
                if (dry) {
                    return result12;
                }
                return await this.execute({
                }, ...this.rawArgs);
            } else {
                const { action , flags , unknown , literal: literal2  } = this.parseFlags(this.rawArgs);
                this.literalArgs = literal2;
                const params = this.parseArguments(unknown, flags);
                await this.validateEnvVars();
                if (dry || action) {
                    if (action) {
                        await action.call(this, flags, ...params);
                    }
                    return {
                        options: flags,
                        args: params,
                        cmd: this,
                        literal: this.literalArgs
                    };
                }
                return await this.execute(flags, ...params);
            }
        } catch (error1) {
            throw this.error(error1);
        }
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        if (!this._help) {
            this.help({
                hints: true,
                types: false
            });
        }
        if (this._versionOption !== false && (this._versionOption || this.ver)) {
            this.option(this._versionOption?.flags || "-V, --version", this._versionOption?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: function() {
                    this.showVersion();
                    Deno.exit(0);
                },
                ...this._versionOption?.opts ?? {
                }
            });
        }
        if (this._helpOption !== false) {
            this.option(this._helpOption?.flags || "-h, --help", this._helpOption?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: function() {
                    this.showHelp();
                    Deno.exit(0);
                },
                ...this._helpOption?.opts ?? {
                }
            });
        }
        return this;
    }
    async execute(options, ...args) {
        if (this.fn) {
            await this.fn(options, ...args);
        } else if (this.defaultCommand) {
            const cmd = this.getCommand(this.defaultCommand, true);
            if (!cmd) {
                throw new DefaultCommandNotFound(this.defaultCommand, this.getCommands());
            }
            cmd._globalParent = this;
            await cmd.execute(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeExecutable(args) {
        const permissions = await getPermissions();
        if (!permissions.read) {
            await Deno.permissions?.request({
                name: "read"
            });
        }
        if (!permissions.run) {
            await Deno.permissions?.request({
                name: "run"
            });
        }
        const [main1, ...names] = this.getPath().split(" ");
        names.unshift(main1.replace(/\.ts$/, ""));
        const executableName = names.join("-");
        const files = [];
        const parts = Deno.mainModule.replace(/^file:\/\//g, "").split("/");
        if (Deno.build.os === "windows" && parts[0] === "") {
            parts.shift();
        }
        parts.pop();
        const path4 = parts.join("/");
        files.push(path4 + "/" + executableName, path4 + "/" + executableName + ".ts");
        files.push(executableName, executableName + ".ts");
        const denoOpts = [];
        if (isUnstable()) {
            denoOpts.push("--unstable");
        }
        denoOpts.push("--allow-read", "--allow-run");
        Object.keys(permissions).forEach((name40)=>{
            if (name40 === "read" || name40 === "run") {
                return;
            }
            if (permissions[name40]) {
                denoOpts.push(`--allow-${name40}`);
            }
        });
        for (const file of files){
            try {
                Deno.lstatSync(file);
            } catch (error2) {
                if (error2 instanceof Deno.errors.NotFound) {
                    continue;
                }
                throw error2;
            }
            const cmd = [
                "deno",
                "run",
                ...denoOpts,
                file,
                ...args
            ];
            const process = Deno.run({
                cmd: cmd
            });
            const status = await process.status();
            if (!status.success) {
                Deno.exit(status.code);
            }
            return;
        }
        throw new CommandExecutableNotFound(executableName, files);
    }
    parseFlags(args) {
        try {
            let action;
            const result13 = parseFlags(args, {
                stopEarly: this._stopEarly,
                allowEmpty: this._allowEmpty,
                flags: this.getOptions(true),
                parse: (type17)=>this.parseType(type17)
                ,
                option: (option)=>{
                    if (!action && option.action) {
                        action = option.action;
                    }
                }
            });
            return {
                ...result13,
                action
            };
        } catch (error3) {
            if (error3 instanceof ValidationError) {
                throw new ValidationError1(error3.message);
            }
            throw error3;
        }
    }
    parseType(type1) {
        const typeSettings = this.getType(type1.type);
        if (!typeSettings) {
            throw new UnknownType(type1.type, this.getTypes().map((type18)=>type18.name
            ));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type1) : typeSettings.handler(type1);
    }
    async validateEnvVars() {
        if (!await hasPermission("env")) {
            return;
        }
        const envVars = this.getEnvVars(true);
        if (!envVars.length) {
            return;
        }
        envVars.forEach((env)=>{
            const name1 = env.names.find((name41)=>!!Deno.env.get(name41)
            );
            if (name1) {
                this.parseType({
                    label: "Environment variable",
                    type: env.type,
                    name: name1,
                    value: Deno.env.get(name1) ?? ""
                });
            }
        });
    }
    parseArguments(args, flags) {
        const params = [];
        args = args.slice(0);
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    throw new UnknownCommand(args[0], this.getCommands());
                } else {
                    throw new NoArgumentsAllowed(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required1 = this.getArguments().filter((expectedArg)=>!expectedArg.optionalValue
                ).map((expectedArg)=>expectedArg.name
                );
                if (required1.length) {
                    const flagNames = Object.keys(flags);
                    const hasStandaloneOption = !!flagNames.find((name42)=>this.getOption(name42, true)?.standalone
                    );
                    if (!hasStandaloneOption) {
                        throw new MissingArguments(required1);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optionalValue) {
                            break;
                        }
                        throw new MissingArgument(`Missing argument: ${expectedArg.name}`);
                    }
                    let arg;
                    if (expectedArg.variadic) {
                        arg = args.splice(0, args.length).map((value)=>this.parseType({
                                label: "Argument",
                                type: expectedArg.type,
                                name: expectedArg.name,
                                value
                            })
                        );
                    } else {
                        arg = this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value: args.shift()
                        });
                    }
                    if (arg) {
                        params.push(arg);
                    }
                }
                if (args.length) {
                    throw new TooManyArguments(args);
                }
            }
        }
        return params;
    }
    error(error4) {
        if (this.shouldThrowErrors() || !(error4 instanceof ValidationError1)) {
            return error4;
        }
        this.showHelp();
        Deno.stderr.writeSync(new TextEncoder().encode(red(`  ${bold("error")}: ${error4.message}\n`) + "\n"));
        Deno.exit(error4 instanceof ValidationError1 ? error4.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath() {
        return this._parent ? this._parent.getPath() + " " + this._name : this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name43) {
        return this.getArguments().find((arg)=>arg.name === name43
        );
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getShortDescription() {
        return this.getDescription().trim().split("\n").shift();
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getVersion()));
    }
    showHelp() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getHelp()));
    }
    getHelp() {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this);
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    hasOptions(hidden1) {
        return this.getOptions(hidden1).length > 0;
    }
    getOptions(hidden2) {
        return this.getGlobalOptions(hidden2).concat(this.getBaseOptions(hidden2));
    }
    getBaseOptions(hidden3) {
        if (!this.options.length) {
            return [];
        }
        return hidden3 ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden
        );
    }
    getGlobalOptions(hidden4) {
        const getOptions1 = (cmd, options = [], names = [])=>{
            if (cmd) {
                if (cmd.options.length) {
                    cmd.options.forEach((option)=>{
                        if (option.global && !this.options.find((opt)=>opt.name === option.name
                        ) && names.indexOf(option.name) === -1 && (hidden4 || !option.hidden)) {
                            names.push(option.name);
                            options.push(option);
                        }
                    });
                }
                return getOptions1(cmd._parent, options, names);
            }
            return options;
        };
        return getOptions1(this._parent);
    }
    hasOption(name44, hidden5) {
        return !!this.getOption(name44, hidden5);
    }
    getOption(name45, hidden6) {
        return this.getBaseOption(name45, hidden6) ?? this.getGlobalOption(name45, hidden6);
    }
    getBaseOption(name46, hidden7) {
        const option1 = this.options.find((option)=>option.name === name46
        );
        return option1 && (hidden7 || !option1.hidden) ? option1 : undefined;
    }
    getGlobalOption(name47, hidden8) {
        if (!this._parent) {
            return;
        }
        const option = this._parent.getBaseOption(name47, hidden8);
        if (!option || !option.global) {
            return this._parent.getGlobalOption(name47, hidden8);
        }
        return option;
    }
    removeOption(name48) {
        const index = this.options.findIndex((option)=>option.name === name48
        );
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden9) {
        return this.getCommands(hidden9).length > 0;
    }
    getCommands(hidden10) {
        return this.getGlobalCommands(hidden10).concat(this.getBaseCommands(hidden10));
    }
    getBaseCommands(hidden11) {
        const commands = Array.from(this.commands.values());
        return hidden11 ? commands : commands.filter((cmd)=>!cmd.isHidden
        );
    }
    getGlobalCommands(hidden12) {
        const getCommands = (cmd1, commands = [], names = [])=>{
            if (cmd1) {
                if (cmd1.commands.size) {
                    cmd1.commands.forEach((cmd)=>{
                        if (cmd.isGlobal && this !== cmd && !this.commands.has(cmd._name) && names.indexOf(cmd._name) === -1 && (hidden12 || !cmd.isHidden)) {
                            names.push(cmd._name);
                            commands.push(cmd);
                        }
                    });
                }
                return getCommands(cmd1._parent, commands, names);
            }
            return commands;
        };
        return getCommands(this._parent);
    }
    hasCommand(name49, hidden13) {
        return !!this.getCommand(name49, hidden13);
    }
    getCommand(name50, hidden14) {
        return this.getBaseCommand(name50, hidden14) ?? this.getGlobalCommand(name50, hidden14);
    }
    getBaseCommand(name51, hidden15) {
        for (const cmd of this.commands.values()){
            if (cmd._name === name51 || cmd.aliases.includes(name51)) {
                return cmd && (hidden15 || !cmd.isHidden) ? cmd : undefined;
            }
        }
    }
    getGlobalCommand(name52, hidden16) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseCommand(name52, hidden16);
        if (!cmd?.isGlobal) {
            return this._parent.getGlobalCommand(name52, hidden16);
        }
        return cmd;
    }
    removeCommand(name53) {
        const command = this.getBaseCommand(name53, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd, types = [], names = [])=>{
            if (cmd) {
                if (cmd.types.size) {
                    cmd.types.forEach((type19)=>{
                        if (type19.global && !this.types.has(type19.name) && names.indexOf(type19.name) === -1) {
                            names.push(type19.name);
                            types.push(type19);
                        }
                    });
                }
                return getTypes(cmd._parent, types, names);
            }
            return types;
        };
        return getTypes(this._parent);
    }
    getType(name54) {
        return this.getBaseType(name54) ?? this.getGlobalType(name54);
    }
    getBaseType(name55) {
        return this.types.get(name55);
    }
    getGlobalType(name56) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseType(name56);
        if (!cmd?.global) {
            return this._parent.getGlobalType(name56);
        }
        return cmd;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd, completions = [], names = [])=>{
            if (cmd) {
                if (cmd.completions.size) {
                    cmd.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name57) {
        return this.getBaseCompletion(name57) ?? this.getGlobalCompletion(name57);
    }
    getBaseCompletion(name58) {
        return this.completions.get(name58);
    }
    getGlobalCompletion(name59) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name59);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name59);
        }
        return completion;
    }
    hasEnvVars(hidden17) {
        return this.getEnvVars(hidden17).length > 0;
    }
    getEnvVars(hidden18) {
        return this.getGlobalEnvVars(hidden18).concat(this.getBaseEnvVars(hidden18));
    }
    getBaseEnvVars(hidden19) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden19 ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden
        );
    }
    getGlobalEnvVars(hidden20) {
        const getEnvVars = (cmd, envVars = [], names = [])=>{
            if (cmd) {
                if (cmd.envVars.length) {
                    cmd.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]
                        ) && names.indexOf(envVar.names[0]) === -1 && (hidden20 || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name60, hidden21) {
        return !!this.getEnvVar(name60, hidden21);
    }
    getEnvVar(name61, hidden22) {
        return this.getBaseEnvVar(name61, hidden22) ?? this.getGlobalEnvVar(name61, hidden22);
    }
    getBaseEnvVar(name62, hidden23) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name62) !== -1
        );
        return envVar && (hidden23 || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name63, hidden24) {
        if (!this._parent) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name63, hidden24);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name63, hidden24);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name64) {
        return !!this.getExample(name64);
    }
    getExample(name65) {
        return this.examples.find((example)=>example.name === name65
        );
    }
}
class BashCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new BashCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path5 = this.cmd.getPath();
        const version3 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path5}${version3}

_${replaceSpecialChars(path5)}() {
  local word cur prev
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()

  _${replaceSpecialChars(this.cmd.getName())}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.cmd.getName()} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  ${this.generateCompletions(this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars(path5)} -o bashdefault -o default ${path5}
`;
    }
    generateCompletions(command, path6 = "", index = 1) {
        path6 = (path6 ? path6 + " " : "") + command.getName();
        const commandCompletions = this.generateCommandCompletions(command, path6, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path6, index + 1)
        ).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path7, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName()
        );
        const completionsPath = ~path7.indexOf(" ") ? " " + path7.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command.getArguments(), completionsPath);
        return `  __${replaceSpecialChars(path7)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option)=>option.flags
        ).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options = command.getOptions(false);
        if (options.length) {
            opts += 'case "${prev}" in';
            for (const option of options){
                const flags = option.flags.map((flag)=>flag.trim()
                ).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(option.args, completionsPath, {
                    standalone: option.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(args, path8) {
        if (args.length) {
            return `_${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path8}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(args, path9, opts) {
        if (args.length) {
            return `opts=(); _${replaceSpecialChars(this.cmd.getName())}_complete ${args[0].action}${path9}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
}
function replaceSpecialChars(str51) {
    return str51.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim(italic("~/.bashrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(BashCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd1){
        super();
        this.description("Get completions for given action from given command.").arguments("<action:string> [command...:string]").action(async (_, action, commandNames)=>{
            let parent2;
            const completeCommand = commandNames?.reduce((cmd, name66)=>{
                parent2 = cmd;
                const childCmd = cmd.getCommand(name66, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommand(name66, cmd.getCommands());
                }
                return childCmd;
            }, cmd1 || this.getMainCommand()) ?? (cmd1 || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result14 = await completion?.complete(completeCommand, parent2) ?? [];
            if (result14?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result14.join("\n")));
            }
        }).reset();
    }
}
class FishCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new FishCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path10 = this.cmd.getPath();
        const version4 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path10}${version4}

function __fish_${replaceSpecialChars1(this.cmd.getName())}_using_command
  set cmds ${getCommandFnNames(this.cmd).join(" ")}
  set words (commandline -opc)
  set cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if [ "$cmd" = "$argv[1]" ]
    return 0
  end
  return 1
end

${this.generateCompletions(this.cmd).trim()}
`;
    }
    generateCompletions(command) {
        const parent3 = command.getParent();
        let result15 = ``;
        if (parent3) {
            result15 += "\n" + this.complete(parent3, {
                description: command.getShortDescription(),
                arguments: command.getName()
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result15 += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(commandArgs[0].action + " " + getCompletionsPath(command)) : undefined
            });
        }
        for (const option of command.getOptions(false)){
            result15 += "\n" + this.completeOption(command, option);
        }
        for (const subCommand of command.getCommands(false)){
            result15 += this.generateCompletions(subCommand);
        }
        return result15;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2
        )?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2
        )?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: option.description,
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(option.args[0].action + " " + getCompletionsPath(command)) : undefined
        });
    }
    complete(command, options) {
        const cmd = [
            "complete"
        ];
        cmd.push("-c", this.cmd.getName());
        cmd.push("-n", `'__fish_${replaceSpecialChars1(this.cmd.getName())}_using_command __${replaceSpecialChars1(command.getPath())}'`);
        options.shortOption && cmd.push("-s", options.shortOption);
        options.longOption && cmd.push("-l", options.longOption);
        options.standalone && cmd.push("-x");
        cmd.push("-k");
        cmd.push("-f");
        if (options.arguments) {
            options.required && cmd.push("-r");
            cmd.push("-a", options.arguments);
        }
        options.description && cmd.push("-d", `'${options.description}'`);
        return cmd.join(" ");
    }
    getCompletionCommand(cmd) {
        return `'(${this.cmd.getName()} completions complete ${cmd.trim()})'`;
    }
}
function getCommandFnNames(cmd, cmds = []) {
    cmds.push(`__${replaceSpecialChars1(cmd.getPath())}`);
    cmd.getCommands(false).forEach((command)=>{
        getCommandFnNames(command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars1(str52) {
    return str52.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim(italic("~/.config/fish/config.fish"))}:

    ${dim(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(FishCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class ZshCompletionsGenerator {
    cmd;
    actions = new Map();
    static generate(cmd) {
        return new ZshCompletionsGenerator(cmd).generate();
    }
    constructor(cmd){
        this.cmd = cmd;
    }
    generate() {
        const path11 = this.cmd.getPath();
        const name67 = this.cmd.getName();
        const version5 = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path11}${version5}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars2(name67)}_complete] )) ||
function __${replaceSpecialChars2(name67)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${name67} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.cmd).trim()}

# _${replaceSpecialChars2(path11)} "\${@}"

compdef _${replaceSpecialChars2(path11)} ${path11}

`;
    }
    generateCompletions(command, path12 = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path12 = (path12 ? path12 + " " : "") + command.getName();
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars2(path12)}] )) ||
function _${replaceSpecialChars2(path12)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path12) + this.generateSubCommandCompletions(command, path12) + this.generateArgumentCompletions(command, path12) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path12)
        ).join("");
    }
    generateCommandCompletions(command, path13) {
        const commands = command.getCommands(false);
        let completions = commands.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription()}'`
        ).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path13.split(" ").slice(1).join(" ");
            const arg = command.getArguments()[0];
            const action = this.addAction(arg, completionsPath);
            if (action && command.getCompletion(arg.action)) {
                completions += `\n    __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command1, path14) {
        if (command1.hasCommands(false)) {
            const actions = command1.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars2(path14 + " " + command.getName())} ;;`
            ).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path15) {
        this.actions.clear();
        const options = this.generateOptions(command, path15);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg)=>command.getCompletion(arg.action)
        ).length) {
            argsCommand += ` \\\n    '${++argIndex}: :_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args = [];
            for (const arg1 of command.getArguments().slice(1)){
                const completionsPath = path15.split(" ").slice(1).join(" ");
                const action = this.addAction(arg1, completionsPath);
                args.push(`${++argIndex}${arg1.optionalValue ? "::" : ":"}${action.name}`);
            }
            argsCommand += args.map((arg)=>`\\\n    '${arg}'`
            ).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*:: :->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path16) {
        const options = [];
        const cmdArgs = path16.split(" ");
        cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option)=>option.standalone ? option.flags : false
        ).flat().filter((flag)=>typeof flag === "string"
        );
        for (const option1 of command.getOptions(false)){
            options.push(this.generateOption(option1, completionsPath, excludedFlags));
        }
        return options;
    }
    generateOption(option, completionsPath, excludedOptions) {
        const flags = option.flags;
        let excludedFlags = option.conflicts?.length ? [
            ...excludedOptions,
            ...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")
            ), 
        ] : excludedOptions;
        excludedFlags = option.collect ? excludedFlags : [
            ...excludedFlags,
            ...flags, 
        ];
        let args = "";
        for (const arg of option.args){
            const action = this.addAction(arg, completionsPath);
            if (arg.variadic) {
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`;
            } else {
                args += `${arg.optionalValue ? "::" : ":"}${arg.name}:->${action.name}`;
            }
        }
        let description6 = option.description.trim().split("\n").shift();
        description6 = description6.replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        if (option.standalone) {
            return `'(- *)'{${collect}${flags}}'[${description6}]${args}'`;
        } else {
            const excluded = excludedFlags.length ? `'(${excludedFlags.join(" ")})'` : "";
            if (collect || flags.length > 1) {
                return `${excluded}{${collect}${flags}}'[${description6}]${args}'`;
            } else {
                return `${excluded}${flags}'[${description6}]${args}'`;
            }
        }
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name68, action])=>`${name68}) __${replaceSpecialChars2(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`
            );
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
}
function replaceSpecialChars2(str53) {
    return str53.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim(italic("~/.zshrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(ZshCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim(italic("~/.bashrc"))} or similar:

    ${dim(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).action(()=>this.showHelp()
        ).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd).hidden()).reset();
    }
}
class CommandType extends StringType {
    complete(_cmd, parent4) {
        return parent4?.getCommands(false).map((cmd)=>cmd.getName()
        ) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd){
        super();
        this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").action((_, name69)=>{
            if (!cmd) {
                cmd = name69 ? this.getGlobalParent()?.getBaseCommand(name69) : this.getGlobalParent();
            }
            if (!cmd) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommand(name69 ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases(), 
                ]);
            }
            cmd.showHelp();
            Deno.exit(0);
        });
    }
}
class EnumType extends Type {
    allowedValues;
    constructor(values3){
        super();
        this.allowedValues = values3;
    }
    parse(type20) {
        for (const value of this.allowedValues){
            if (value.toString() === type20.value) {
                return value;
            }
        }
        throw new InvalidTypeError(type20, this.allowedValues.slice());
    }
    values() {
        return this.allowedValues.slice();
    }
    complete() {
        return this.values();
    }
}
const osType = (()=>{
    const { Deno  } = globalThis;
    if (typeof Deno?.build?.os === "string") {
        return Deno.build.os;
    }
    const { navigator  } = globalThis;
    if (navigator?.appVersion?.includes?.("Win") ?? false) {
        return "windows";
    }
    return "linux";
})();
const isWindows = osType === "windows";
const CHAR_FORWARD_SLASH = 47;
function assertPath(path17) {
    if (typeof path17 !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path17)}`);
    }
}
function isPosixPathSeparator(code3) {
    return code3 === 47;
}
function isPathSeparator(code4) {
    return isPosixPathSeparator(code4) || code4 === 92;
}
function isWindowsDeviceRoot(code5) {
    return code5 >= 97 && code5 <= 122 || code5 >= 65 && code5 <= 90;
}
function normalizeString(path18, allowAboveRoot, separator, isPathSeparator1) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code6;
    for(let i6 = 0, len = path18.length; i6 <= len; ++i6){
        if (i6 < len) code6 = path18.charCodeAt(i6);
        else if (isPathSeparator1(code6)) break;
        else code6 = CHAR_FORWARD_SLASH;
        if (isPathSeparator1(code6)) {
            if (lastSlash === i6 - 1 || dots === 1) {
            } else if (lastSlash !== i6 - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i6;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i6;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path18.slice(lastSlash + 1, i6);
                else res = path18.slice(lastSlash + 1, i6);
                lastSegmentLength = i6 - lastSlash - 1;
            }
            lastSlash = i6;
            dots = 0;
        } else if (code6 === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function _format(sep6, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base1 = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base1;
    if (dir === pathObject.root) return dir + base1;
    return dir + sep6 + base1;
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string4) {
    return string4.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
class DenoStdInternalError extends Error {
    constructor(message){
        super(message);
        this.name = "DenoStdInternalError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new DenoStdInternalError(msg);
    }
}
const sep = "\\";
const delimiter = ";";
function resolve9(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i7 = pathSegments.length - 1; i7 >= -1; i7--){
        let path19;
        const { Deno  } = globalThis;
        if (i7 >= 0) {
            path19 = pathSegments[i7];
        } else if (!resolvedDevice) {
            if (typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path19 = Deno.cwd();
        } else {
            if (typeof Deno?.env?.get !== "function" || typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path19 = Deno.cwd();
            if (path19 === undefined || path19.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path19 = `${resolvedDevice}\\`;
            }
        }
        assertPath(path19);
        const len = path19.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute1 = false;
        const code7 = path19.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code7)) {
                isAbsolute1 = true;
                if (isPathSeparator(path19.charCodeAt(1))) {
                    let j = 2;
                    let last1 = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path19.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last1) {
                        const firstPart = path19.slice(last1, j);
                        last1 = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path19.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last1) {
                            last1 = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path19.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path19.slice(last1)}`;
                                rootEnd = j;
                            } else if (j !== last1) {
                                device = `\\\\${firstPart}\\${path19.slice(last1, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code7)) {
                if (path19.charCodeAt(1) === 58) {
                    device = path19.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path19.charCodeAt(2))) {
                            isAbsolute1 = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code7)) {
            rootEnd = 1;
            isAbsolute1 = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path19.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute1;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function normalize1(path20) {
    assertPath(path20);
    const len = path20.length;
    if (len === 0) return ".";
    let rootEnd = 0;
    let device;
    let isAbsolute2 = false;
    const code8 = path20.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code8)) {
            isAbsolute2 = true;
            if (isPathSeparator(path20.charCodeAt(1))) {
                let j = 2;
                let last2 = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path20.charCodeAt(j))) break;
                }
                if (j < len && j !== last2) {
                    const firstPart = path20.slice(last2, j);
                    last2 = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path20.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last2) {
                        last2 = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path20.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path20.slice(last2)}\\`;
                        } else if (j !== last2) {
                            device = `\\\\${firstPart}\\${path20.slice(last2, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code8)) {
            if (path20.charCodeAt(1) === 58) {
                device = path20.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path20.charCodeAt(2))) {
                        isAbsolute2 = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code8)) {
        return "\\";
    }
    let tail1;
    if (rootEnd < len) {
        tail1 = normalizeString(path20.slice(rootEnd), !isAbsolute2, "\\", isPathSeparator);
    } else {
        tail1 = "";
    }
    if (tail1.length === 0 && !isAbsolute2) tail1 = ".";
    if (tail1.length > 0 && isPathSeparator(path20.charCodeAt(len - 1))) {
        tail1 += "\\";
    }
    if (device === undefined) {
        if (isAbsolute2) {
            if (tail1.length > 0) return `\\${tail1}`;
            else return "\\";
        } else if (tail1.length > 0) {
            return tail1;
        } else {
            return "";
        }
    } else if (isAbsolute2) {
        if (tail1.length > 0) return `${device}\\${tail1}`;
        else return `${device}\\`;
    } else if (tail1.length > 0) {
        return device + tail1;
    } else {
        return device;
    }
}
function isAbsolute(path21) {
    assertPath(path21);
    const len = path21.length;
    if (len === 0) return false;
    const code9 = path21.charCodeAt(0);
    if (isPathSeparator(code9)) {
        return true;
    } else if (isWindowsDeviceRoot(code9)) {
        if (len > 2 && path21.charCodeAt(1) === 58) {
            if (isPathSeparator(path21.charCodeAt(2))) return true;
        }
    }
    return false;
}
function join(...paths) {
    const pathsCount = paths.length;
    if (pathsCount === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i8 = 0; i8 < pathsCount; ++i8){
        const path22 = paths[i8];
        assertPath(path22);
        if (path22.length > 0) {
            if (joined === undefined) joined = firstPart = path22;
            else joined += `\\${path22}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart != null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize1(joined);
}
function relative(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    const fromOrig = resolve9(from);
    const toOrig = resolve9(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i9 = 0;
    for(; i9 <= length; ++i9){
        if (i9 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i9) === 92) {
                    return toOrig.slice(toStart + i9 + 1);
                } else if (i9 === 2) {
                    return toOrig.slice(toStart + i9);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i9) === 92) {
                    lastCommonSep = i9;
                } else if (i9 === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i9);
        const toCode = to.charCodeAt(toStart + i9);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i9;
    }
    if (i9 !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i9 = fromStart + lastCommonSep + 1; i9 <= fromEnd; ++i9){
        if (i9 === fromEnd || from.charCodeAt(i9) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
function toNamespacedPath(path23) {
    if (typeof path23 !== "string") return path23;
    if (path23.length === 0) return "";
    const resolvedPath = resolve9(path23);
    if (resolvedPath.length >= 3) {
        if (resolvedPath.charCodeAt(0) === 92) {
            if (resolvedPath.charCodeAt(1) === 92) {
                const code10 = resolvedPath.charCodeAt(2);
                if (code10 !== 63 && code10 !== 46) {
                    return `\\\\?\\UNC\\${resolvedPath.slice(2)}`;
                }
            }
        } else if (isWindowsDeviceRoot(resolvedPath.charCodeAt(0))) {
            if (resolvedPath.charCodeAt(1) === 58 && resolvedPath.charCodeAt(2) === 92) {
                return `\\\\?\\${resolvedPath}`;
            }
        }
    }
    return path23;
}
function dirname(path24) {
    assertPath(path24);
    const len = path24.length;
    if (len === 0) return ".";
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code11 = path24.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code11)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path24.charCodeAt(1))) {
                let j = 2;
                let last3 = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path24.charCodeAt(j))) break;
                }
                if (j < len && j !== last3) {
                    last3 = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path24.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last3) {
                        last3 = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path24.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path24;
                        }
                        if (j !== last3) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code11)) {
            if (path24.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path24.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code11)) {
        return path24;
    }
    for(let i10 = len - 1; i10 >= offset; --i10){
        if (isPathSeparator(path24.charCodeAt(i10))) {
            if (!matchedSlash) {
                end = i10;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return path24.slice(0, end);
}
function basename(path25, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path25);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i11;
    if (path25.length >= 2) {
        const drive = path25.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path25.charCodeAt(1) === 58) start = 2;
        }
    }
    if (ext !== undefined && ext.length > 0 && ext.length <= path25.length) {
        if (ext.length === path25.length && ext === path25) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i11 = path25.length - 1; i11 >= start; --i11){
            const code12 = path25.charCodeAt(i11);
            if (isPathSeparator(code12)) {
                if (!matchedSlash) {
                    start = i11 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i11 + 1;
                }
                if (extIdx >= 0) {
                    if (code12 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i11;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path25.length;
        return path25.slice(start, end);
    } else {
        for(i11 = path25.length - 1; i11 >= start; --i11){
            if (isPathSeparator(path25.charCodeAt(i11))) {
                if (!matchedSlash) {
                    start = i11 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i11 + 1;
            }
        }
        if (end === -1) return "";
        return path25.slice(start, end);
    }
}
function extname(path26) {
    assertPath(path26);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path26.length >= 2 && path26.charCodeAt(1) === 58 && isWindowsDeviceRoot(path26.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i12 = path26.length - 1; i12 >= start; --i12){
        const code13 = path26.charCodeAt(i12);
        if (isPathSeparator(code13)) {
            if (!matchedSlash) {
                startPart = i12 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i12 + 1;
        }
        if (code13 === 46) {
            if (startDot === -1) startDot = i12;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path26.slice(startDot, end);
}
function format(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("\\", pathObject);
}
function parse(path27) {
    assertPath(path27);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    const len = path27.length;
    if (len === 0) return ret;
    let rootEnd = 0;
    let code14 = path27.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code14)) {
            rootEnd = 1;
            if (isPathSeparator(path27.charCodeAt(1))) {
                let j = 2;
                let last4 = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path27.charCodeAt(j))) break;
                }
                if (j < len && j !== last4) {
                    last4 = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path27.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last4) {
                        last4 = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path27.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            rootEnd = j;
                        } else if (j !== last4) {
                            rootEnd = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code14)) {
            if (path27.charCodeAt(1) === 58) {
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path27.charCodeAt(2))) {
                        if (len === 3) {
                            ret.root = ret.dir = path27;
                            return ret;
                        }
                        rootEnd = 3;
                    }
                } else {
                    ret.root = ret.dir = path27;
                    return ret;
                }
            }
        }
    } else if (isPathSeparator(code14)) {
        ret.root = ret.dir = path27;
        return ret;
    }
    if (rootEnd > 0) ret.root = path27.slice(0, rootEnd);
    let startDot = -1;
    let startPart = rootEnd;
    let end = -1;
    let matchedSlash = true;
    let i13 = path27.length - 1;
    let preDotState = 0;
    for(; i13 >= rootEnd; --i13){
        code14 = path27.charCodeAt(i13);
        if (isPathSeparator(code14)) {
            if (!matchedSlash) {
                startPart = i13 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i13 + 1;
        }
        if (code14 === 46) {
            if (startDot === -1) startDot = i13;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            ret.base = ret.name = path27.slice(startPart, end);
        }
    } else {
        ret.name = path27.slice(startPart, startDot);
        ret.base = path27.slice(startPart, end);
        ret.ext = path27.slice(startDot, end);
    }
    if (startPart > 0 && startPart !== rootEnd) {
        ret.dir = path27.slice(0, startPart - 1);
    } else ret.dir = ret.root;
    return ret;
}
function fromFileUrl(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    let path28 = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname != "") {
        path28 = `\\\\${url.hostname}${path28}`;
    }
    return path28;
}
function toFileUrl(path29) {
    if (!isAbsolute(path29)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path29.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname != null && hostname != "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
const mod1 = {
    sep: sep,
    delimiter: delimiter,
    resolve: resolve9,
    normalize: normalize1,
    isAbsolute: isAbsolute,
    join: join,
    relative: relative,
    toNamespacedPath: toNamespacedPath,
    dirname: dirname,
    basename: basename,
    extname: extname,
    format: format,
    parse: parse,
    fromFileUrl: fromFileUrl,
    toFileUrl: toFileUrl
};
const sep1 = "/";
const delimiter1 = ":";
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i14 = pathSegments.length - 1; i14 >= -1 && !resolvedAbsolute; i14--){
        let path30;
        if (i14 >= 0) path30 = pathSegments[i14];
        else {
            const { Deno  } = globalThis;
            if (typeof Deno?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path30 = Deno.cwd();
        }
        assertPath(path30);
        if (path30.length === 0) {
            continue;
        }
        resolvedPath = `${path30}/${resolvedPath}`;
        resolvedAbsolute = path30.charCodeAt(0) === CHAR_FORWARD_SLASH;
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function normalize2(path31) {
    assertPath(path31);
    if (path31.length === 0) return ".";
    const isAbsolute1 = path31.charCodeAt(0) === 47;
    const trailingSeparator = path31.charCodeAt(path31.length - 1) === 47;
    path31 = normalizeString(path31, !isAbsolute1, "/", isPosixPathSeparator);
    if (path31.length === 0 && !isAbsolute1) path31 = ".";
    if (path31.length > 0 && trailingSeparator) path31 += "/";
    if (isAbsolute1) return `/${path31}`;
    return path31;
}
function isAbsolute1(path32) {
    assertPath(path32);
    return path32.length > 0 && path32.charCodeAt(0) === 47;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i15 = 0, len = paths.length; i15 < len; ++i15){
        const path33 = paths[i15];
        assertPath(path33);
        if (path33.length > 0) {
            if (!joined) joined = path33;
            else joined += `/${path33}`;
        }
    }
    if (!joined) return ".";
    return normalize2(joined);
}
function relative1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 47) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 47) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i16 = 0;
    for(; i16 <= length; ++i16){
        if (i16 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i16) === 47) {
                    return to.slice(toStart + i16 + 1);
                } else if (i16 === 0) {
                    return to.slice(toStart + i16);
                }
            } else if (fromLen > length) {
                if (from.charCodeAt(fromStart + i16) === 47) {
                    lastCommonSep = i16;
                } else if (i16 === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i16);
        const toCode = to.charCodeAt(toStart + i16);
        if (fromCode !== toCode) break;
        else if (fromCode === 47) lastCommonSep = i16;
    }
    let out = "";
    for(i16 = fromStart + lastCommonSep + 1; i16 <= fromEnd; ++i16){
        if (i16 === fromEnd || from.charCodeAt(i16) === 47) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (to.charCodeAt(toStart) === 47) ++toStart;
        return to.slice(toStart);
    }
}
function toNamespacedPath1(path34) {
    return path34;
}
function dirname1(path35) {
    assertPath(path35);
    if (path35.length === 0) return ".";
    const hasRoot = path35.charCodeAt(0) === 47;
    let end = -1;
    let matchedSlash = true;
    for(let i17 = path35.length - 1; i17 >= 1; --i17){
        if (path35.charCodeAt(i17) === 47) {
            if (!matchedSlash) {
                end = i17;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) return hasRoot ? "/" : ".";
    if (hasRoot && end === 1) return "//";
    return path35.slice(0, end);
}
function basename1(path36, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path36);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i18;
    if (ext !== undefined && ext.length > 0 && ext.length <= path36.length) {
        if (ext.length === path36.length && ext === path36) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i18 = path36.length - 1; i18 >= 0; --i18){
            const code15 = path36.charCodeAt(i18);
            if (code15 === 47) {
                if (!matchedSlash) {
                    start = i18 + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i18 + 1;
                }
                if (extIdx >= 0) {
                    if (code15 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i18;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path36.length;
        return path36.slice(start, end);
    } else {
        for(i18 = path36.length - 1; i18 >= 0; --i18){
            if (path36.charCodeAt(i18) === 47) {
                if (!matchedSlash) {
                    start = i18 + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i18 + 1;
            }
        }
        if (end === -1) return "";
        return path36.slice(start, end);
    }
}
function extname1(path37) {
    assertPath(path37);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i19 = path37.length - 1; i19 >= 0; --i19){
        const code16 = path37.charCodeAt(i19);
        if (code16 === 47) {
            if (!matchedSlash) {
                startPart = i19 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i19 + 1;
        }
        if (code16 === 46) {
            if (startDot === -1) startDot = i19;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path37.slice(startDot, end);
}
function format1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("/", pathObject);
}
function parse1(path38) {
    assertPath(path38);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path38.length === 0) return ret;
    const isAbsolute2 = path38.charCodeAt(0) === 47;
    let start;
    if (isAbsolute2) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i20 = path38.length - 1;
    let preDotState = 0;
    for(; i20 >= start; --i20){
        const code17 = path38.charCodeAt(i20);
        if (code17 === 47) {
            if (!matchedSlash) {
                startPart = i20 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i20 + 1;
        }
        if (code17 === 46) {
            if (startDot === -1) startDot = i20;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute2) {
                ret.base = ret.name = path38.slice(1, end);
            } else {
                ret.base = ret.name = path38.slice(startPart, end);
            }
        }
    } else {
        if (startPart === 0 && isAbsolute2) {
            ret.name = path38.slice(1, startDot);
            ret.base = path38.slice(1, end);
        } else {
            ret.name = path38.slice(startPart, startDot);
            ret.base = path38.slice(startPart, end);
        }
        ret.ext = path38.slice(startDot, end);
    }
    if (startPart > 0) ret.dir = path38.slice(0, startPart - 1);
    else if (isAbsolute2) ret.dir = "/";
    return ret;
}
function fromFileUrl1(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function toFileUrl1(path39) {
    if (!isAbsolute1(path39)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path39.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
const mod2 = {
    sep: sep1,
    delimiter: delimiter1,
    resolve: resolve1,
    normalize: normalize2,
    isAbsolute: isAbsolute1,
    join: join1,
    relative: relative1,
    toNamespacedPath: toNamespacedPath1,
    dirname: dirname1,
    basename: basename1,
    extname: extname1,
    format: format1,
    parse: parse1,
    fromFileUrl: fromFileUrl1,
    toFileUrl: toFileUrl1
};
const SEP = isWindows ? "\\" : "/";
const SEP_PATTERN = isWindows ? /[\\/]+/ : /\/+/;
const path = isWindows ? mod1 : mod2;
const { join: join2 , normalize: normalize3  } = path;
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|", 
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function globToRegExp(glob, { extended =true , globstar: globstarOption = true , os =osType , caseInsensitive =false  } = {
}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep7 = os == "windows" ? "(?:\\\\|/)+" : "/+";
    const sepMaybe = os == "windows" ? "(?:\\\\|/)*" : "/*";
    const seps = os == "windows" ? [
        "\\",
        "/"
    ] : [
        "/"
    ];
    const globstar = os == "windows" ? "(?:[^\\\\/]*(?:\\\\|/|$)+)*" : "(?:[^/]*(?:/|$)+)*";
    const wildcard = os == "windows" ? "[^\\\\/]*" : "[^/]*";
    const escapePrefix = os == "windows" ? "`" : "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange1 = false;
        let inEscape = false;
        let endsWithSep = false;
        let i21 = j;
        for(; i21 < glob.length && !seps.includes(glob[i21]); i21++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange1 ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i21]) ? `\\${glob[i21]}` : glob[i21];
                continue;
            }
            if (glob[i21] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i21] == "[") {
                if (!inRange1) {
                    inRange1 = true;
                    segment += "[";
                    if (glob[i21 + 1] == "!") {
                        i21++;
                        segment += "^";
                    } else if (glob[i21 + 1] == "^") {
                        i21++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i21 + 1] == ":") {
                    let k = i21 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i21 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i21] == "]" && inRange1) {
                inRange1 = false;
                segment += "]";
                continue;
            }
            if (inRange1) {
                if (glob[i21] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i21];
                }
                continue;
            }
            if (glob[i21] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type21 = groupStack.pop();
                if (type21 == "!") {
                    segment += wildcard;
                } else if (type21 != "@") {
                    segment += type21;
                }
                continue;
            }
            if (glob[i21] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i21] == "+" && extended && glob[i21 + 1] == "(") {
                i21++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i21] == "@" && extended && glob[i21 + 1] == "(") {
                i21++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i21] == "?") {
                if (extended && glob[i21 + 1] == "(") {
                    i21++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i21] == "!" && extended && glob[i21 + 1] == "(") {
                i21++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i21] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i21] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i21] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i21] == "*") {
                if (extended && glob[i21 + 1] == "(") {
                    i21++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i21 - 1];
                    let numStars = 1;
                    while(glob[i21 + 1] == "*"){
                        i21++;
                        numStars++;
                    }
                    const nextChar = glob[i21 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i21]) ? `\\${glob[i21]}` : glob[i21];
        }
        if (groupStack.length > 0 || inRange1 || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i21)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i21 < glob.length ? sep7 : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i21]))i21++;
        if (!(i21 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i21;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob(str54) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str54 === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str54)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str54.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str54 = str54.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar =false  } = {
}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize3(glob);
    }
    const s = SEP_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize3(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended =false , globstar =false  } = {
}) {
    if (!globstar || globs.length == 0) {
        return join2(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path1 = glob;
        if (path1.length > 0) {
            if (!joined) joined = path1;
            else joined += `${SEP}${path1}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
const path1 = isWindows ? mod1 : mod2;
const posix = mod2;
const { basename: basename2 , delimiter: delimiter2 , dirname: dirname2 , extname: extname2 , format: format2 , fromFileUrl: fromFileUrl2 , isAbsolute: isAbsolute2 , join: join3 , normalize: normalize4 , parse: parse2 , relative: relative2 , resolve: resolve2 , sep: sep2 , toFileUrl: toFileUrl2 , toNamespacedPath: toNamespacedPath2 ,  } = path1;
function _createWalkEntrySync(path40) {
    path40 = normalize4(path40);
    const name70 = basename2(path40);
    const info1 = Deno.statSync(path40);
    return {
        path: path40,
        name: name70,
        isFile: info1.isFile,
        isDirectory: info1.isDirectory,
        isSymlink: info1.isSymlink
    };
}
function include1(path41, exts, match, skip) {
    if (exts && !exts.some((ext)=>path41.endsWith(ext)
    )) {
        return false;
    }
    if (match && !match.some((pattern1)=>!!path41.match(pattern1)
    )) {
        return false;
    }
    if (skip && skip.some((pattern2)=>!!path41.match(pattern2)
    )) {
        return false;
    }
    return true;
}
function wrapErrorWithRootPath(err, root1) {
    if (err instanceof Error && "root" in err) return err;
    const e = new Error();
    e.root = root1;
    e.message = err instanceof Error ? `${err.message} for path "${root1}"` : `[non-error thrown] for path "${root1}"`;
    e.stack = err instanceof Error ? err.stack : undefined;
    e.cause = err instanceof Error ? err.cause : undefined;
    return e;
}
function* walkSync(root2, { maxDepth =Infinity , includeFiles =true , includeDirs =true , followSymlinks =false , exts =undefined , match =undefined , skip =undefined  } = {
}) {
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include1(root2, exts, match, skip)) {
        yield _createWalkEntrySync(root2);
    }
    if (maxDepth < 1 || !include1(root2, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root2);
    } catch (err) {
        throw wrapErrorWithRootPath(err, normalize4(root2));
    }
    for (const entry of entries){
        assert(entry.name != null);
        let path42 = join3(root2, entry.name);
        let isFile = entry.isFile;
        if (entry.isSymlink) {
            if (followSymlinks) {
                path42 = Deno.realPathSync(path42);
                isFile = Deno.lstatSync(path42).isFile;
            } else {
                continue;
            }
        }
        if (isFile) {
            if (includeFiles && include1(path42, exts, match, skip)) {
                yield {
                    path: path42,
                    ...entry
                };
            }
        } else {
            yield* walkSync(path42, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                followSymlinks,
                exts,
                match,
                skip
            });
        }
    }
}
function split(path43) {
    const s = SEP_PATTERN.source;
    const segments = path43.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEP_PATTERN);
    const isAbsolute_ = isAbsolute2(path43);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path43.match(new RegExp(`${s}$`)),
        winRoot: isWindows && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error5) {
    if (!(error5 instanceof Deno.errors.NotFound)) {
        throw error5;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root: root3 = Deno.cwd() , exclude =[] , includeDirs =true , extended =false , globstar =false , caseInsensitive  } = {
}) {
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = resolve2(root3);
    const resolveFromRoot = (path44)=>resolve2(absRoot, path44)
    ;
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp(s, globOptions)
    );
    const shouldInclude1 = (path45)=>!excludePatterns.some((p)=>!!path45.match(p)
        )
    ;
    const { segments , isAbsolute: isGlobAbsolute , hasTrailingSep , winRoot  } = split(glob);
    let fixedRoot = isGlobAbsolute ? winRoot != undefined ? winRoot : "/" : absRoot;
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg != null);
        fixedRoot = joinGlobs([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = _createWalkEntrySync(fixedRoot);
    } catch (error2) {
        return throwUnlessNotFound(error2);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment == "..") {
            const parentPath = joinGlobs([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude1(parentPath)) {
                    return yield _createWalkEntrySync(parentPath);
                }
            } catch (error6) {
                throwUnlessNotFound(error6);
            }
            return;
        } else if (globSegment == "**") {
            return yield* walkSync(walkInfo.path, {
                skip: excludePatterns
            });
        }
        const globPattern = globToRegExp(globSegment, globOptions);
        for (const walkEntry of walkSync(walkInfo.path, {
            maxDepth: 1,
            skip: excludePatterns
        })){
            if (walkEntry.path != walkInfo.path && walkEntry.name.match(globPattern)) {
                yield walkEntry;
            }
        }
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory
        );
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory
        );
    }
    yield* currentMatches;
}
var LogLevels;
(function(LogLevels1) {
    LogLevels1[LogLevels1["NOTSET"] = 0] = "NOTSET";
    LogLevels1[LogLevels1["DEBUG"] = 10] = "DEBUG";
    LogLevels1[LogLevels1["INFO"] = 20] = "INFO";
    LogLevels1[LogLevels1["WARNING"] = 30] = "WARNING";
    LogLevels1[LogLevels1["ERROR"] = 40] = "ERROR";
    LogLevels1[LogLevels1["CRITICAL"] = 50] = "CRITICAL";
})(LogLevels || (LogLevels = {
}));
Object.keys(LogLevels).filter((key)=>isNaN(Number(key))
);
const byLevel = {
    [String(LogLevels.NOTSET)]: "NOTSET",
    [String(LogLevels.DEBUG)]: "DEBUG",
    [String(LogLevels.INFO)]: "INFO",
    [String(LogLevels.WARNING)]: "WARNING",
    [String(LogLevels.ERROR)]: "ERROR",
    [String(LogLevels.CRITICAL)]: "CRITICAL"
};
function getLevelByName(name71) {
    switch(name71){
        case "NOTSET":
            return LogLevels.NOTSET;
        case "DEBUG":
            return LogLevels.DEBUG;
        case "INFO":
            return LogLevels.INFO;
        case "WARNING":
            return LogLevels.WARNING;
        case "ERROR":
            return LogLevels.ERROR;
        case "CRITICAL":
            return LogLevels.CRITICAL;
        default:
            throw new Error(`no log level found for "${name71}"`);
    }
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`no level name found for level: ${level}`);
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {
    }){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    _log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data1) {
        if (typeof data1 === "string") {
            return data1;
        } else if (data1 === null || typeof data1 === "number" || typeof data1 === "bigint" || typeof data1 === "boolean" || typeof data1 === "undefined" || typeof data1 === "symbol") {
            return String(data1);
        } else if (data1 instanceof Error) {
            return data1.stack;
        } else if (typeof data1 === "object") {
            return JSON.stringify(data1);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this._log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this._log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this._log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this._log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this._log(LogLevels.CRITICAL, msg, ...args);
    }
}
const { Deno: Deno1  } = globalThis;
const noColor1 = typeof Deno1?.noColor === "boolean" ? Deno1.noColor : true;
let enabled1 = !noColor1;
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str55, code18) {
    return enabled1 ? `${code18.open}${str55.replace(code18.regexp, code18.open)}${code18.close}` : str55;
}
function bold1(str56) {
    return run1(str56, code1([
        1
    ], 22));
}
function dim1(str57) {
    return run1(str57, code1([
        2
    ], 22));
}
function underline1(str58) {
    return run1(str58, code1([
        4
    ], 24));
}
function red1(str59) {
    return run1(str59, code1([
        31
    ], 39));
}
function green1(str60) {
    return run1(str60, code1([
        32
    ], 39));
}
function yellow1(str61) {
    return run1(str61, code1([
        33
    ], 39));
}
function blue1(str62) {
    return run1(str62, code1([
        34
    ], 39));
}
function brightRed1(str63) {
    return run1(str63, code1([
        91
    ], 39));
}
const ANSI_PATTERN1 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor1(string5) {
    return string5.replace(ANSI_PATTERN1, "");
}
async function exists(filePath) {
    try {
        await Deno.lstat(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function existsSync(filePath) {
    try {
        Deno.lstatSync(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function concat(...buf) {
    let length = 0;
    for (const b of buf){
        length += b.length;
    }
    const output = new Uint8Array(length);
    let index = 0;
    for (const b1 of buf){
        output.set(b1, index);
        index += b1.length;
    }
    return output;
}
function copy(src, dst, off = 0) {
    off = Math.max(0, Math.min(off, dst.byteLength));
    const dstBytesAvailable = dst.byteLength - off;
    if (src.byteLength > dstBytesAvailable) {
        src = src.subarray(0, dstBytesAvailable);
    }
    dst.set(src, off);
    return src.byteLength;
}
const MIN_READ = 32 * 1024;
const MAX_SIZE = 2 ** 32 - 2;
class Buffer {
    #buf;
    #off = 0;
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
     #tryGrowByReslice(n) {
        const l = this.#buf.byteLength;
        if (n <= this.capacity - l) {
            this.#reslice(l + n);
            return l;
        }
        return -1;
    }
     #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    readSync(p) {
        if (this.empty()) {
            this.reset();
            if (p.byteLength === 0) {
                return 0;
            }
            return null;
        }
        const nread = copy(this.#buf.subarray(this.#off), p);
        this.#off += nread;
        return nread;
    }
    read(p) {
        const rr = this.readSync(p);
        return Promise.resolve(rr);
    }
    writeSync(p) {
        const m = this.#grow(p.byteLength);
        return copy(p, this.#buf, m);
    }
    write(p) {
        const n1 = this.writeSync(p);
        return Promise.resolve(n1);
    }
     #grow(n2) {
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i = this.#tryGrowByReslice(n2);
        if (i >= 0) {
            return i;
        }
        const c = this.capacity;
        if (n2 <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n2 > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n2, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n2, MAX_SIZE));
        return m;
    }
    grow(n3) {
        if (n3 < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n3);
        this.#reslice(m);
    }
    async readFrom(r) {
        let n4 = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = await r.read(buf);
            if (nread === null) {
                return n4;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n4 += nread;
        }
    }
    readFromSync(r) {
        let n5 = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = r.readSync(buf);
            if (nread === null) {
                return n5;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n5 += nread;
        }
    }
}
const DEFAULT_BUF_SIZE = 4096;
const MIN_BUF_SIZE = 16;
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
class BufferFullError extends Error {
    partial;
    name = "BufferFullError";
    constructor(partial){
        super("Buffer full");
        this.partial = partial;
    }
}
class PartialReadError extends Error {
    name = "PartialReadError";
    partial;
    constructor(){
        super("Encountered UnexpectedEof, data only partially read");
    }
}
class BufReader {
    #buf;
    #rd;
    #r = 0;
    #w = 0;
    #eof = false;
    static create(r, size1 = 4096) {
        return r instanceof BufReader ? r : new BufReader(r, size1);
    }
    constructor(rd, size2 = 4096){
        if (size2 < 16) {
            size2 = MIN_BUF_SIZE;
        }
        this.#reset(new Uint8Array(size2), rd);
    }
    size() {
        return this.#buf.byteLength;
    }
    buffered() {
        return this.#w - this.#r;
    }
    #fill = async ()=>{
        if (this.#r > 0) {
            this.#buf.copyWithin(0, this.#r, this.#w);
            this.#w -= this.#r;
            this.#r = 0;
        }
        if (this.#w >= this.#buf.byteLength) {
            throw Error("bufio: tried to fill full buffer");
        }
        for(let i22 = 100; i22 > 0; i22--){
            const rr = await this.#rd.read(this.#buf.subarray(this.#w));
            if (rr === null) {
                this.#eof = true;
                return;
            }
            assert(rr >= 0, "negative read");
            this.#w += rr;
            if (rr > 0) {
                return;
            }
        }
        throw new Error(`No progress after ${100} read() calls`);
    };
    reset(r) {
        this.#reset(this.#buf, r);
    }
    #reset = (buf, rd)=>{
        this.#buf = buf;
        this.#rd = rd;
        this.#eof = false;
    };
    async read(p) {
        let rr = p.byteLength;
        if (p.byteLength === 0) return rr;
        if (this.#r === this.#w) {
            if (p.byteLength >= this.#buf.byteLength) {
                const rr = await this.#rd.read(p);
                const nread = rr ?? 0;
                assert(nread >= 0, "negative read");
                return rr;
            }
            this.#r = 0;
            this.#w = 0;
            rr = await this.#rd.read(this.#buf);
            if (rr === 0 || rr === null) return rr;
            assert(rr >= 0, "negative read");
            this.#w += rr;
        }
        const copied = copy(this.#buf.subarray(this.#r, this.#w), p, 0);
        this.#r += copied;
        return copied;
    }
    async readFull(p) {
        let bytesRead = 0;
        while(bytesRead < p.length){
            try {
                const rr = await this.read(p.subarray(bytesRead));
                if (rr === null) {
                    if (bytesRead === 0) {
                        return null;
                    } else {
                        throw new PartialReadError();
                    }
                }
                bytesRead += rr;
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = p.subarray(0, bytesRead);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = p.subarray(0, bytesRead);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return p;
    }
    async readByte() {
        while(this.#r === this.#w){
            if (this.#eof) return null;
            await this.#fill();
        }
        const c = this.#buf[this.#r];
        this.#r++;
        return c;
    }
    async readString(delim) {
        if (delim.length !== 1) {
            throw new Error("Delimiter should be a single character");
        }
        const buffer = await this.readSlice(delim.charCodeAt(0));
        if (buffer === null) return null;
        return new TextDecoder().decode(buffer);
    }
    async readLine() {
        let line = null;
        try {
            line = await this.readSlice(LF);
        } catch (err) {
            if (err instanceof Deno.errors.BadResource) {
                throw err;
            }
            let partial;
            if (err instanceof PartialReadError) {
                partial = err.partial;
                assert(partial instanceof Uint8Array, "bufio: caught error from `readSlice()` without `partial` property");
            }
            if (!(err instanceof BufferFullError)) {
                throw err;
            }
            partial = err.partial;
            if (!this.#eof && partial && partial.byteLength > 0 && partial[partial.byteLength - 1] === CR) {
                assert(this.#r > 0, "bufio: tried to rewind past start of buffer");
                this.#r--;
                partial = partial.subarray(0, partial.byteLength - 1);
            }
            if (partial) {
                return {
                    line: partial,
                    more: !this.#eof
                };
            }
        }
        if (line === null) {
            return null;
        }
        if (line.byteLength === 0) {
            return {
                line,
                more: false
            };
        }
        if (line[line.byteLength - 1] == LF) {
            let drop1 = 1;
            if (line.byteLength > 1 && line[line.byteLength - 2] === CR) {
                drop1 = 2;
            }
            line = line.subarray(0, line.byteLength - drop1);
        }
        return {
            line,
            more: false
        };
    }
    async readSlice(delim) {
        let s = 0;
        let slice2;
        while(true){
            let i23 = this.#buf.subarray(this.#r + s, this.#w).indexOf(delim);
            if (i23 >= 0) {
                i23 += s;
                slice2 = this.#buf.subarray(this.#r, this.#r + i23 + 1);
                this.#r += i23 + 1;
                break;
            }
            if (this.#eof) {
                if (this.#r === this.#w) {
                    return null;
                }
                slice2 = this.#buf.subarray(this.#r, this.#w);
                this.#r = this.#w;
                break;
            }
            if (this.buffered() >= this.#buf.byteLength) {
                this.#r = this.#w;
                const oldbuf = this.#buf;
                const newbuf = this.#buf.slice(0);
                this.#buf = newbuf;
                throw new BufferFullError(oldbuf);
            }
            s = this.#w - this.#r;
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = slice2;
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = slice2;
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
        }
        return slice2;
    }
    async peek(n6) {
        if (n6 < 0) {
            throw Error("negative count");
        }
        let avail = this.#w - this.#r;
        while(avail < n6 && avail < this.#buf.byteLength && !this.#eof){
            try {
                await this.#fill();
            } catch (err) {
                if (err instanceof PartialReadError) {
                    err.partial = this.#buf.subarray(this.#r, this.#w);
                } else if (err instanceof Error) {
                    const e = new PartialReadError();
                    e.partial = this.#buf.subarray(this.#r, this.#w);
                    e.stack = err.stack;
                    e.message = err.message;
                    e.cause = err.cause;
                    throw err;
                }
                throw err;
            }
            avail = this.#w - this.#r;
        }
        if (avail === 0 && this.#eof) {
            return null;
        } else if (avail < n6 && this.#eof) {
            return this.#buf.subarray(this.#r, this.#r + avail);
        } else if (avail < n6) {
            throw new BufferFullError(this.#buf.subarray(this.#r, this.#w));
        }
        return this.#buf.subarray(this.#r, this.#r + n6);
    }
}
class AbstractBufBase {
    buf;
    usedBufferBytes = 0;
    err = null;
    constructor(buf){
        this.buf = buf;
    }
    size() {
        return this.buf.byteLength;
    }
    available() {
        return this.buf.byteLength - this.usedBufferBytes;
    }
    buffered() {
        return this.usedBufferBytes;
    }
}
class BufWriter extends AbstractBufBase {
    #writer;
    static create(writer, size3 = 4096) {
        return writer instanceof BufWriter ? writer : new BufWriter(writer, size3);
    }
    constructor(writer, size4 = 4096){
        if (size4 <= 0) {
            size4 = DEFAULT_BUF_SIZE;
        }
        const buf = new Uint8Array(size4);
        super(buf);
        this.#writer = writer;
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.#writer = w;
    }
    async flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            const p = this.buf.subarray(0, this.usedBufferBytes);
            let nwritten = 0;
            while(nwritten < p.length){
                nwritten += await this.#writer.write(p.subarray(nwritten));
            }
        } catch (e) {
            if (e instanceof Error) {
                this.err = e;
            }
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    async write(data2) {
        if (this.err !== null) throw this.err;
        if (data2.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data2.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = await this.#writer.write(data2);
                } catch (e) {
                    if (e instanceof Error) {
                        this.err = e;
                    }
                    throw e;
                }
            } else {
                numBytesWritten = copy(data2, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                await this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data2 = data2.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data2, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
class BufWriterSync extends AbstractBufBase {
    #writer;
    static create(writer, size5 = 4096) {
        return writer instanceof BufWriterSync ? writer : new BufWriterSync(writer, size5);
    }
    constructor(writer, size6 = 4096){
        if (size6 <= 0) {
            size6 = DEFAULT_BUF_SIZE;
        }
        const buf = new Uint8Array(size6);
        super(buf);
        this.#writer = writer;
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.#writer = w;
    }
    flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            const p = this.buf.subarray(0, this.usedBufferBytes);
            let nwritten = 0;
            while(nwritten < p.length){
                nwritten += this.#writer.writeSync(p.subarray(nwritten));
            }
        } catch (e) {
            if (e instanceof Error) {
                this.err = e;
            }
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    writeSync(data3) {
        if (this.err !== null) throw this.err;
        if (data3.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data3.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = this.#writer.writeSync(data3);
                } catch (e) {
                    if (e instanceof Error) {
                        this.err = e;
                    }
                    throw e;
                }
            } else {
                numBytesWritten = copy(data3, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data3 = data3.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data3, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
async function* readLines(reader, decoderOpts) {
    const bufReader = new BufReader(reader);
    let chunks = [];
    const decoder = new TextDecoder(decoderOpts?.encoding, decoderOpts);
    while(true){
        const res = await bufReader.readLine();
        if (!res) {
            if (chunks.length > 0) {
                yield decoder.decode(concat(...chunks));
            }
            break;
        }
        chunks.push(res.line);
        if (!res.more) {
            yield decoder.decode(concat(...chunks));
            chunks = [];
        }
    }
}
const DEFAULT_FORMATTER = "{levelName} {msg}";
class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {
    }){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{(\S+)}/g, (match, p1)=>{
            const value = logRecord[p1];
            if (value == null) {
                return match;
            }
            return String(value);
        });
    }
    log(_msg) {
    }
    async setup() {
    }
    async destroy() {
    }
}
class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue1(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow1(msg);
                break;
            case LogLevels.ERROR:
                msg = red1(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold1(red1(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
     #unloadCallback() {
        this.destroy();
    }
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    async setup() {
        this._file = await Deno.open(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback.bind(this));
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
        return Promise.resolve();
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state96 = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger(name) {
    if (!name) {
        const d = state96.loggers.get("default");
        assert(d != null, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result16 = state96.loggers.get(name);
    if (!result16) {
        const logger = new Logger(name, "NOTSET", {
            handlers: []
        });
        state96.loggers.set(name, logger);
        return logger;
    }
    return result16;
}
function debug(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
function info(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
function warning(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").warning(msg, ...args);
    }
    return getLogger("default").warning(msg, ...args);
}
function error(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
async function setup(config) {
    state96.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state96.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state96.handlers.clear();
    const handlers = state96.config.handlers || {
    };
    for(const handlerName1 in handlers){
        const handler = handlers[handlerName1];
        await handler.setup();
        state96.handlers.set(handlerName1, handler);
    }
    state96.loggers.clear();
    const loggers = state96.config.loggers || {
    };
    for(const loggerName in loggers){
        const loggerConfig = loggers[loggerName];
        const handlerNames = loggerConfig.handlers || [];
        const handlers = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state96.handlers.get(handlerName);
            if (handler) {
                handlers.push(handler);
            }
        });
        const levelName = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers
        });
        state96.loggers.set(loggerName, logger);
    }
}
await setup(DEFAULT_CONFIG);
function isSubdir(src, dest, sep8 = sep2) {
    if (src === dest) {
        return false;
    }
    const srcArray = src.split(sep8);
    const destArray = dest.split(sep8);
    return srcArray.every((current, i)=>destArray[i] === current
    );
}
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            await Deno.mkdir(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            Deno.mkdirSync(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function moveSync(src, dest, { overwrite =false  } = {
}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (overwrite) {
        if (existsSync(dest)) {
            Deno.removeSync(dest, {
                recursive: true
            });
        }
    } else {
        if (existsSync(dest)) {
            throw new Error("dest already exists.");
        }
    }
    Deno.renameSync(src, dest);
}
var EOL;
(function(EOL1) {
    EOL1["LF"] = "\n";
    EOL1["CRLF"] = "\r\n";
})(EOL || (EOL = {
}));
function glb(array1, value, compare) {
    compare = compare || ((a, b)=>a - b
    );
    if (array1.length === 0) {
        return -1;
    }
    if (array1.length === 1) {
        if (compare(value, array1[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array1.length - 1;
    const vLeft = array1[left], vRight = array1[right];
    if (compare(value, vRight) >= 0) {
        return right;
    }
    if (compare(value, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array1[center];
        const cmp = compare(value, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function* matchAll(text, regexp1) {
    let match;
    while((match = regexp1.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function* lineBreakPositions(text) {
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index;
    }
}
function indexToRowCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad1 = " ".repeat(lineWidth);
    const ls = lines(text);
    const result17 = [];
    for(let i24 = firstLine; i24 <= lastLine; ++i24){
        const numberStr = `${pad1}${i24 + 1}: `.slice(-(lineWidth + 2));
        const lineStr = ls[i24];
        result17.push({
            lineNumber: i24,
            content: numberStr + lineStr,
            rawLine: ls[i24]
        });
    }
    return {
        prefixWidth: lineWidth + 2,
        lines: result17
    };
}
function mappedString(source, pieces) {
    if (typeof source === "string") {
        const offsetInfo = [];
        let offset = 0;
        const resultList = pieces.filter((piece)=>typeof piece === "string" || piece.start !== piece.end
        ).map((piece)=>{
            if (typeof piece === "string") {
                offsetInfo.push({
                    fromSource: false,
                    length: piece.length,
                    offset
                });
                offset += piece.length;
                return piece;
            } else {
                const resultPiece = source.substring(piece.start, piece.end);
                offsetInfo.push({
                    fromSource: true,
                    length: resultPiece.length,
                    offset,
                    range: {
                        start: piece.start,
                        end: piece.end
                    }
                });
                offset += resultPiece.length;
                return resultPiece;
            }
        });
        const value = resultList.join("");
        const map2 = (targetOffset)=>{
            const ix = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            if (ix < 0) {
                return undefined;
            }
            const info2 = offsetInfo[ix];
            if (!info2.fromSource) {
                return undefined;
            }
            const localOffset = targetOffset - info2.offset;
            if (localOffset >= info2.length) {
                return undefined;
            }
            return info2.range.start + localOffset;
        };
        const mapClosest = (targetOffset)=>{
            if (offsetInfo.length === 0 || targetOffset < 0) {
                return undefined;
            }
            const firstIx = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            let ix = firstIx;
            let smallestSourceInfo = undefined;
            while(ix >= 0){
                const info3 = offsetInfo[ix];
                if (!info3.fromSource) {
                    ix--;
                    continue;
                }
                smallestSourceInfo = info3;
                if (ix === firstIx) {
                    const localOffset = targetOffset - info3.offset;
                    if (localOffset < info3.length) {
                        return info3.range.start + localOffset;
                    }
                }
                return info3.range.end - 1;
            }
            if (smallestSourceInfo === undefined) {
                return undefined;
            } else {
                return smallestSourceInfo.range.start;
            }
        };
        return {
            value,
            originalString: source,
            map: map2,
            mapClosest
        };
    } else {
        const { value , originalString , map: previousMap , mapClosest: previousMapClosest ,  } = source;
        const { value: resultValue , map: nextMap , mapClosest: nextMapClosest ,  } = mappedString(value, pieces);
        const composeMap = (offset)=>{
            const v = nextMap(offset);
            if (v === undefined) {
                return v;
            }
            return previousMap(v);
        };
        const composeMapClosest = (offset)=>{
            const v = nextMapClosest(offset);
            if (v === undefined) {
                return v;
            }
            return previousMapClosest(v);
        };
        return {
            value: resultValue,
            originalString,
            map: composeMap,
            mapClosest: composeMapClosest
        };
    }
}
function asMappedString(str64) {
    if (typeof str64 === "string") {
        return {
            value: str64,
            originalString: str64,
            map: (x)=>x
            ,
            mapClosest: (x)=>x
        };
    } else {
        return str64;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        throw new Error("strings must be non-empty");
    }
    let currentOffset = 0;
    const offsets = [];
    for (const s1 of strings){
        currentOffset += s1.value.length;
        offsets.push(currentOffset);
    }
    const value = "".concat(...strings.map((s)=>s.value
    ));
    return {
        value,
        originalString: strings[0].originalString,
        map (offset) {
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].map(offset - offsets[ix]);
        },
        mapClosest (offset) {
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].mapClosest(offset - offsets[ix]);
        }
    };
}
function mappedIndexToRowCol(eitherText) {
    const text = asMappedString(eitherText);
    const f = indexToRowCol(text.originalString);
    return function(offset) {
        const n1 = text.mapClosest(offset);
        if (n1 === undefined) {
            throw new Error("Internal Error: bad offset in mappedIndexRowCol");
        }
        return f(n1);
    };
}
function mappedNormalizeNewlines(eitherText) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const offset of lineBreakPositions(text.value)){
        if (text.value[offset] !== "\r") {
            continue;
        }
        chunks.push({
            start,
            end: offset
        });
        chunks.push({
            start: offset + 1,
            end: offset + 2
        });
        start = offset + 2;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexpAll(eitherText, re) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const match of matchAll(text.value, re)){
        chunks.push({
            start,
            end: match.index
        });
        start = match[0].length + match.index;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexp(eitherText, re) {
    const text = asMappedString(eitherText);
    const m = text.value.match(re);
    if (m) {
        return mappedString(text, [
            {
                start: 0,
                end: m.index
            },
            {
                start: m.index + m[0].length,
                end: text.value.length
            }, 
        ]);
    } else {
        return text;
    }
}
function schemaType(schema) {
    const t = schema.type;
    if (t) {
        return t;
    }
    if (schema.anyOf) {
        return "anyOf";
    }
    if (schema.oneOf) {
        return "oneOf";
    }
    if (schema.allOf) {
        return "allOf";
    }
    if (schema.enum) {
        return "enum";
    }
    return "any";
}
function schemaExhaustiveCompletions(schema) {
    switch(schemaType(schema)){
        case "anyOf":
            return schema.anyOf.every(schemaExhaustiveCompletions);
        case "oneOf":
            return schema.oneOf.every(schemaExhaustiveCompletions);
        case "allOf":
            return schema.allOf.every(schemaExhaustiveCompletions);
        case "array":
            return true;
        case "object":
            return true;
        default:
            return schema.exhaustiveCompletions || false;
    }
}
function walkSchema(schema, f) {
    f(schema);
    const t = schemaType(schema);
    switch(t){
        case "array":
            if (schema.items) {
                walkSchema(schema.items, f);
            }
            break;
        case "anyOf":
            for (const s of schema.anyOf){
                walkSchema(s, f);
            }
            break;
        case "oneOf":
            for (const s1 of schema.oneOf){
                walkSchema(s1, f);
            }
            break;
        case "allOf":
            for (const s2 of schema.allOf){
                walkSchema(s2, f);
            }
            break;
        case "object":
            if (schema.properties) {
                for (const key of Object.getOwnPropertyNames(schema.properties)){
                    const s = schema.properties[key];
                    walkSchema(s, f);
                }
            }
            if (schema.patternProperties) {
                for (const key of Object.getOwnPropertyNames(schema.patternProperties)){
                    const s = schema.patternProperties[key];
                    walkSchema(s, f);
                }
            }
            if (schema.additionalProperties) {
                walkSchema(schema.additionalProperties, f);
            }
            break;
    }
}
function normalizeSchema(schema1) {
    const result18 = JSON.parse(JSON.stringify(schema1));
    walkSchema(result18, (schema)=>{
        if (schema.completions) {
            delete schema.completions;
        }
        if (schema.exhaustiveCompletions) {
            delete schema.exhaustiveCompletions;
        }
        if (schema.documentation) {
            delete schema.documentation;
        }
    });
    return result18;
}
let ajv = undefined;
function setupAjv(_ajv) {
    ajv = _ajv;
}
function navigate(path46, annotation, returnKey = false, pathIndex = 0) {
    if (pathIndex >= path46.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components  } = annotation;
        const searchKey = path46[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i25 = lastKeyIndex; i25 >= 0; i25 -= 2){
            const key = components[i25].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path46.length - 1) {
                    return navigate(path46, components[i25], returnKey, pathIndex + 1);
                } else {
                    return navigate(path46, components[i25 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        throw new Error("Internal error: searchKey not found in mapping object");
    } else if (annotation.kind === "sequence" || annotation.kind === "block_sequence") {
        const searchKey = Number(path46[pathIndex]);
        return navigate(path46, annotation.components[searchKey], returnKey, pathIndex + 1);
    } else {
        throw new Error(`Internal error: unexpected kind ${annotation.kind}`);
    }
}
function navigateSchema(path47, schema, pathIndex = 0) {
    if (pathIndex >= path47.length - 1) {
        return schema;
    }
    const pathVal = path47[pathIndex];
    if (pathVal === "patternProperties") {
        const key = path47[pathIndex + 1];
        const subSchema = schema.patternProperties[key];
        return navigateSchema(path47, subSchema, pathIndex + 2);
    } else if (pathVal === "properties") {
        const key = path47[pathIndex + 1];
        const subSchema = schema.properties[key];
        return navigateSchema(path47, subSchema, pathIndex + 2);
    } else if (pathVal === "anyOf") {
        const key = Number(path47[pathIndex + 1]);
        const subSchema = schema.anyOf[key];
        return navigateSchema(path47, subSchema, pathIndex + 2);
    } else if (pathVal === "oneOf") {
        const key = Number(path47[pathIndex + 1]);
        const subSchema = schema.oneOf[key];
        return navigateSchema(path47, subSchema, pathIndex + 2);
    } else if (pathVal === "items") {
        const subSchema = schema.items;
        return navigateSchema(path47, subSchema, pathIndex + 1);
    } else {
        throw new Error(`Internal error: Failed to navigate schema path ${path47}`);
    }
}
function isProperPrefix(a, b) {
    return b.length > a.length && b.substring(0, a.length) === a;
}
function groupBy(lst, f) {
    const record = {
    };
    const result19 = [];
    for (const el of lst){
        const key = f(el);
        if (record[key] === undefined) {
            const lst = [];
            const entry = {
                key,
                values: lst
            };
            record[key] = lst;
            result19.push(entry);
        }
        record[key].push(el);
    }
    return result19;
}
function groupByEntries(entries) {
    const result20 = [];
    for (const { values: values4  } of entries){
        result20.push(...values4);
    }
    return result20;
}
function narrowOneOfError(oneOf1, suberrors) {
    const subschemaErrors = groupBy(suberrors.filter((error7)=>error7.schemaPath !== oneOf1.schemaPath
    ), (error8)=>error8.schemaPath.substring(0, error8.schemaPath.lastIndexOf("/"))
    );
    const onlyAdditionalProperties = subschemaErrors.filter(({ values: values5  })=>values5.every((v)=>v.keyword === "additionalProperties"
        )
    );
    if (onlyAdditionalProperties.length) {
        return onlyAdditionalProperties[0].values;
    }
    return [];
}
function localizeAndPruneErrors(annotation, validationErrors, source, schema) {
    const result21 = [];
    const locF = mappedIndexToRowCol(source);
    let errorsPerInstanceList = groupBy(validationErrors, (error9)=>error9.instancePath
    );
    do {
        const newErrors = [];
        errorsPerInstanceList = errorsPerInstanceList.filter(({ key: pathA  })=>errorsPerInstanceList.filter(({ key: pathB  })=>isProperPrefix(pathA, pathB)
            ).length === 0
        );
        for (let { key: instancePath , values: errors1  } of errorsPerInstanceList){
            let errorsPerSchemaList = groupBy(errors1, (error10)=>error10.schemaPath
            );
            errorsPerSchemaList = errorsPerSchemaList.filter(({ key: pathA  })=>errorsPerSchemaList.filter(({ key: pathB  })=>isProperPrefix(pathB, pathA)
                ).length === 0
            );
            for (const error1 of groupByEntries(errorsPerSchemaList)){
                if (error1.hasBeenTransformed) {
                    continue;
                }
                if (error1.keyword === "oneOf") {
                    error1.hasBeenTransformed = true;
                    newErrors.push(...narrowOneOfError(error1, errors1));
                } else if (error1.keyword === "additionalProperties") {
                    error1.hasBeenTransformed = true;
                    instancePath = `${instancePath}/${error1.params.additionalProperty}`;
                    newErrors.push({
                        ...error1,
                        instancePath,
                        keyword: "_custom_invalidProperty",
                        message: `property ${error1.params.additionalProperty} not allowed in object`,
                        params: {
                            ...error1.params,
                            originalError: error1
                        },
                        schemaPath: error1.schemaPath.slice(0, -21)
                    });
                }
            }
        }
        if (newErrors.length) {
            errorsPerInstanceList.push(...groupBy(newErrors, (error11)=>error11.instancePath
            ));
        } else {
            break;
        }
    }while (true)
    for (const { key: instancePath , values: allErrors  } of errorsPerInstanceList){
        const path48 = instancePath.split("/").slice(1);
        const errors2 = allErrors.filter(({ schemaPath: pathA  })=>!(allErrors.filter(({ schemaPath: pathB  })=>isProperPrefix(pathB, pathA)
            ).length > 0)
        );
        for (const error12 of errors2){
            const returnKey = error12.keyword === "_custom_invalidProperty";
            const violatingObject = navigate(path48, annotation, returnKey);
            const schemaPath = error12.schemaPath.split("/").slice(1);
            const start = locF(violatingObject.start);
            const end = locF(violatingObject.end);
            const locStr = start.line === end.line ? `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})` : `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
            let messageNoLocation;
            if (error12.keyword.startsWith("_custom_")) {
                messageNoLocation = error12.message;
            } else {
                const innerSchema = navigateSchema(schemaPath.map(decodeURIComponent), schema);
                if (instancePath === "") {
                    messageNoLocation = `(top-level error) ${error12.message}`;
                } else {
                    messageNoLocation = `Field ${instancePath} must ${innerSchema.description}`;
                }
            }
            const message = `${locStr}: ${messageNoLocation}`;
            result21.push({
                instancePath,
                violatingObject,
                message,
                messageNoLocation,
                source,
                start,
                end,
                error: error12
            });
        }
    }
    result21.sort((a, b)=>a.violatingObject.start - b.violatingObject.start
    );
    return result21;
}
class YAMLSchema {
    schema;
    validate;
    constructor(schema){
        this.schema = schema;
        this.validate = ajv.compile(normalizeSchema(schema));
    }
    validateParse(src, annotation) {
        let errors3 = [];
        if (!this.validate(annotation.result)) {
            errors3 = localizeAndPruneErrors(annotation, this.validate.errors, src, this.schema);
            return {
                result: annotation.result,
                errors: errors3
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result22, src, message, error13, log) {
        if (result22.errors.length) {
            const locF = mappedIndexToRowCol(src);
            const nLines = lines(src.originalString).length;
            error13(message);
            for (const err of result22.errors){
                log(err.message);
                let startO = err.violatingObject.start;
                let endO = err.violatingObject.end;
                while(src.mapClosest(startO) < src.originalString.length - 1 && src.originalString[src.mapClosest(startO)].match(/\s/)){
                    startO++;
                }
                while(src.mapClosest(endO) > src.mapClosest(startO) && src.originalString[src.mapClosest(endO)].match(/\s/)){
                    endO--;
                }
                const start = locF(startO);
                const end = locF(endO);
                const { prefixWidth , lines: lines1 ,  } = formatLineRange(src.originalString, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
                for (const { lineNumber , content , rawLine  } of lines1){
                    log(content);
                    if (lineNumber >= start.line && lineNumber <= end.line) {
                        const startColumn = lineNumber > start.line ? 0 : start.column;
                        const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                        log(" ".repeat(prefixWidth + startColumn) + "^".repeat(endColumn - startColumn + 1));
                    }
                }
            }
        }
        return result22;
    }
    validateParseWithErrors(src, annotation, message, error14, log) {
        const result23 = this.validateParse(src, annotation);
        this.reportErrorsInSource(result23, src, message, error14, log);
        return result23;
    }
}
class PromiseQueue {
    queue = new Array();
    running = false;
    enqueue(promise, clearPending = false) {
        return new Promise((resolve7, reject1)=>{
            if (clearPending) {
                this.queue.splice(0, this.queue.length);
            }
            this.queue.push({
                promise,
                resolve: resolve7,
                reject: reject1
            });
            this.dequeue();
        });
    }
    dequeue() {
        if (this.running) {
            return false;
        }
        const item = this.queue.shift();
        if (!item) {
            return false;
        }
        try {
            this.running = true;
            item.promise().then((value)=>{
                this.running = false;
                item.resolve(value);
                this.dequeue();
            }).catch((err)=>{
                this.running = false;
                item.reject(err);
                this.dequeue();
            });
        } catch (err) {
            this.running = false;
            item.reject(err);
            this.dequeue();
        }
        return true;
    }
}
const yamlValidators = {
};
const validatorQueues = {
};
function getSchemaName(schema) {
    const schemaName = schema["$id"];
    if (schemaName === undefined) {
        throw new Error("Expected schema to be named");
    }
    return schemaName;
}
function getValidator(schema) {
    const schemaName = getSchemaName(schema);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema);
    yamlValidators[schemaName] = validator;
    return validator;
}
async function withValidator(schema, fun) {
    const schemaName = getSchemaName(schema);
    if (validatorQueues[schemaName] === undefined) {
        validatorQueues[schemaName] = new PromiseQueue();
    }
    const queue = validatorQueues[schemaName];
    let result24;
    let error15;
    await queue.enqueue(async ()=>{
        const validator = getValidator(schema);
        try {
            result24 = fun(validator);
        } catch (e) {
            console.error("Error in validator queue", e);
            error15 = e;
        }
    });
    if (error15 !== undefined) {
        throw error15;
    }
    return result24;
}
function getenv(name, defaultValue) {
    const value = Deno.env.get(name);
    if (value === undefined) {
        if (defaultValue === undefined) {
            throw new Error(`Required environment variable ${name} not specified.`);
        } else {
            return defaultValue;
        }
    } else {
        return value;
    }
}
var State;
(function(State2) {
    State2[State2["PASSTHROUGH"] = 0] = "PASSTHROUGH";
    State2[State2["PERCENT"] = 1] = "PERCENT";
    State2[State2["POSITIONAL"] = 2] = "POSITIONAL";
    State2[State2["PRECISION"] = 3] = "PRECISION";
    State2[State2["WIDTH"] = 4] = "WIDTH";
})(State || (State = {
}));
var WorP;
(function(WorP1) {
    WorP1[WorP1["WIDTH"] = 0] = "WIDTH";
    WorP1[WorP1["PRECISION"] = 1] = "PRECISION";
})(WorP || (WorP = {
}));
var F;
(function(F1) {
    F1[F1["sign"] = 1] = "sign";
    F1[F1["mantissa"] = 2] = "mantissa";
    F1[F1["fractional"] = 3] = "fractional";
    F1[F1["esign"] = 4] = "esign";
    F1[F1["exponent"] = 5] = "exponent";
})(F || (F = {
}));
function appendLogOptions(cmd) {
    return cmd.option("--log <level>", "Path to log file", {
        global: true
    }).option("--log-level <level>", "Log level (info, warning, error, critical)", {
        global: true
    }).option("--log-format <format>", "Log format (plain, json-stream)", {
        global: true
    }).option("--quiet", "Suppress console output.", {
        global: true
    });
}
function logOptions(args) {
    const logOptions1 = {
    };
    logOptions1.log = args.l || args.log;
    logOptions1.level = args.ll || args["log-level"];
    logOptions1.quiet = args.q || args.quiet;
    logOptions1.format = parseFormat(args.lf || args["log-format"]);
    return logOptions1;
}
class StdErrOutputHandler extends BaseHandler {
    format(logRecord, prefix = true) {
        const options = {
            newline: true,
            colorize: true,
            ...logRecord.args[0]
        };
        let msg = super.format(logRecord);
        if (prefix && logRecord.level >= LogLevels.WARNING) {
            msg = `${logRecord.levelName}: ${msg}`;
        }
        switch(logRecord.level){
            case LogLevels.INFO:
            case LogLevels.DEBUG:
                msg = applyMsgOptions(msg, options);
                break;
            case LogLevels.WARNING:
                if (options.colorize) {
                    msg = yellow1(msg);
                }
                break;
            case LogLevels.ERROR:
                if (options.colorize) {
                    msg = brightRed1(msg);
                }
                break;
            case LogLevels.CRITICAL:
                if (options.colorize) {
                    msg = bold1(red1(msg));
                }
                break;
            default:
                break;
        }
        if (options.newline) {
            msg = msg + "\n";
        }
        return msg;
    }
    log(msg) {
        Deno.stderr.writeSync(new TextEncoder().encode(msg));
    }
}
class LogEventsHandler extends StdErrOutputHandler {
    constructor(levelName){
        super(levelName, {
            formatter: "{msg}"
        });
    }
    handle(logRecord) {
        LogEventsHandler.handlers_.forEach((handler)=>handler(logRecord, super.format(logRecord, false))
        );
    }
    static onLog(handler) {
        LogEventsHandler.handlers_.push(handler);
    }
    static handlers_ = new Array();
}
class LogFileHandler extends FileHandler {
    constructor(levelName, options){
        super(levelName, options);
        this.msgFormat = options.format;
    }
    msgFormat;
    format(logRecord) {
        if (logRecord.msg.startsWith("\r")) {
            return "";
        }
        if (this.msgFormat === undefined || this.msgFormat === "plain") {
            const options = {
                newline: true,
                ...logRecord.args[0],
                bold: false,
                dim: false,
                format: undefined
            };
            let msg = applyMsgOptions(logRecord.msg, options);
            if (options.newline) {
                msg = msg + "\n";
            }
            if (logRecord.level >= LogLevels.WARNING) {
                return `(${logRecord.levelName}) ${msg}`;
            } else {
                return msg;
            }
        } else {
            return JSON.stringify(logRecord, undefined, 0) + "\n";
        }
    }
    log(msg) {
        if (msg !== "") {
            msg = stripColor1(msg);
            this._buf.writeSync(this._encoder.encode(msg));
            this._buf.flush();
        }
    }
}
async function initializeLogger(logOptions2) {
    const handlers = {
    };
    const defaultHandlers = [];
    const file = logOptions2.log;
    const logLevel = logOptions2.level ? parseLevel(logOptions2.level) : "INFO";
    handlers["events"] = new LogEventsHandler(logLevel);
    defaultHandlers.push("events");
    if (!logOptions2.quiet) {
        handlers["console"] = new StdErrOutputHandler(logLevel, {
            formatter: "{msg}"
        });
        defaultHandlers.push("console");
    }
    if (file) {
        handlers["file"] = new LogFileHandler(logLevel, {
            filename: file,
            mode: "w",
            format: logOptions2.format
        });
        defaultHandlers.push("file");
    }
    await setup({
        handlers,
        loggers: {
            default: {
                level: "DEBUG",
                handlers: defaultHandlers
            }
        }
    });
}
async function cleanupLogger() {
    const logger = getLogger();
    for (const handler of logger.handlers){
        await handler.destroy();
    }
    logger.handlers = [];
}
function logError(e) {
    const isDebug = getenv("QUARTO_DEBUG", "false") === "true";
    if (isDebug) {
        error(e.stack);
    } else {
        if (e.message) {
            error(`${e.name}: ${e.message}`);
        }
    }
}
function warnOnce(msg) {
    if (!warnings[msg]) {
        warnings[msg] = true;
        warning(msg);
    }
}
const warnings = {
};
function applyMsgOptions(msg1, options) {
    if (options.indent) {
        const pad2 = " ".repeat(options.indent);
        msg1 = lines(msg1).map((msg)=>pad2 + msg
        ).join("\n");
    }
    if (options.bold) {
        msg1 = bold1(msg1);
    }
    if (options.dim) {
        msg1 = dim1(msg1);
    }
    if (options.format) {
        msg1 = options.format(msg1);
    }
    return msg1;
}
function parseFormat(format7) {
    if (format7) {
        format7 = format7.toLowerCase();
        switch(format7){
            case "plain":
            case "json-stream":
                return format7;
            default:
                return "plain";
        }
    } else {
        return "plain";
    }
}
function parseLevel(level) {
    const lvl = levelMap[level.toLowerCase()];
    if (lvl) {
        return lvl;
    } else {
        return "WARNING";
    }
}
const levelMap = {
    debug: "DEBUG",
    info: "INFO",
    warning: "WARNING",
    error: "ERROR",
    critical: "CRITICAL"
};
async function ensureValidCopy(src, dest, options) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
function ensureValidCopySync(src, dest, options) {
    let destStat;
    try {
        destStat = Deno.lstatSync(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    await Deno.copyFile(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copyFileSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    Deno.copyFileSync(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = Deno.statSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    const originSrcFilePath = await Deno.readLink(src);
    const type22 = getFileInfoType(await Deno.lstat(src));
    if (isWindows) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type22 === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copySymlinkSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    const originSrcFilePath = Deno.readLinkSync(src);
    const type23 = getFileInfoType(Deno.lstatSync(src));
    if (isWindows) {
        Deno.symlinkSync(originSrcFilePath, dest, {
            type: type23 === "dir" ? "dir" : "file"
        });
    } else {
        Deno.symlinkSync(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = Deno.lstatSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for await (const entry of Deno.readDir(src)){
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            await copySymLink(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            await copyDir(srcPath, destPath, options);
        } else if (entry.isFile) {
            await copyFile(srcPath, destPath, options);
        }
    }
}
function copyDirSync(src, dest, options) {
    const destStat = ensureValidCopySync(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        ensureDirSync(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = Deno.statSync(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for (const entry of Deno.readDirSync(src)){
        assert(entry.name != null, "file.name must be set");
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            copySymlinkSync(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            copyDirSync(srcPath, destPath, options);
        } else if (entry.isFile) {
            copyFileSync(srcPath, destPath, options);
        }
    }
}
async function copy1(src, dest, options = {
}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options);
    }
}
function copySync(src, dest, options = {
}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = Deno.lstatSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        copySymlinkSync(src, dest, options);
    } else if (srcStat.isDirectory) {
        copyDirSync(src, dest, options);
    } else if (srcStat.isFile) {
        copyFileSync(src, dest, options);
    }
}
var global$1 = typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {
};
var freeGlobal = typeof global$1 == "object" && global$1 && global$1.Object === Object && global$1;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = freeGlobal || freeSelf || Function("return this")();
var Symbol1 = root.Symbol;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = Symbol1 ? Symbol1.toStringTag : void 0;
function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag), tag = value[symToStringTag];
    try {
        value[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {
    }
    var result2 = nativeObjectToString.call(value);
    if (unmasked) {
        if (isOwn) {
            value[symToStringTag] = tag;
        } else {
            delete value[symToStringTag];
        }
    }
    return result2;
}
var objectProto$1 = Object.prototype;
var nativeObjectToString$1 = objectProto$1.toString;
function objectToString(value) {
    return nativeObjectToString$1.call(value);
}
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag$1 = Symbol1 ? Symbol1.toStringTag : void 0;
function baseGetTag(value) {
    if (value == null) {
        return value === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag$1 && symToStringTag$1 in Object(value) ? getRawTag(value) : objectToString(value);
}
function isObjectLike(value) {
    return value != null && typeof value == "object";
}
var symbolTag = "[object Symbol]";
function isSymbol(value) {
    return typeof value == "symbol" || isObjectLike(value) && baseGetTag(value) == symbolTag;
}
var NAN = 0 / 0;
function baseToNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (isSymbol(value)) {
        return NAN;
    }
    return +value;
}
function arrayMap(array2, iteratee2) {
    var index = -1, length = array2 == null ? 0 : array2.length, result2 = Array(length);
    while(++index < length){
        result2[index] = iteratee2(array2[index], index, array2);
    }
    return result2;
}
var isArray = Array.isArray;
var INFINITY = 1 / 0;
var symbolProto = Symbol1 ? Symbol1.prototype : void 0, symbolToString = symbolProto ? symbolProto.toString : void 0;
function baseToString(value) {
    if (typeof value == "string") {
        return value;
    }
    if (isArray(value)) {
        return arrayMap(value, baseToString) + "";
    }
    if (isSymbol(value)) {
        return symbolToString ? symbolToString.call(value) : "";
    }
    var result2 = value + "";
    return result2 == "0" && 1 / value == -INFINITY ? "-0" : result2;
}
function createMathOperation(operator, defaultValue) {
    return function(value, other) {
        var result2;
        if (value === void 0 && other === void 0) {
            return defaultValue;
        }
        if (value !== void 0) {
            result2 = value;
        }
        if (other !== void 0) {
            if (result2 === void 0) {
                return other;
            }
            if (typeof value == "string" || typeof other == "string") {
                value = baseToString(value);
                other = baseToString(other);
            } else {
                value = baseToNumber(value);
                other = baseToNumber(other);
            }
            result2 = operator(value, other);
        }
        return result2;
    };
}
var add = createMathOperation(function(augend, addend) {
    return augend + addend;
}, 0);
var reWhitespace = /\s/;
function trimmedEndIndex(string2) {
    var index = string2.length;
    while(index-- && reWhitespace.test(string2.charAt(index))){
    }
    return index;
}
var reTrimStart = /^\s+/;
function baseTrim(string2) {
    return string2 ? string2.slice(0, trimmedEndIndex(string2) + 1).replace(reTrimStart, "") : string2;
}
function isObject(value) {
    var type24 = typeof value;
    return value != null && (type24 == "object" || type24 == "function");
}
var NAN$1 = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (isSymbol(value)) {
        return NAN$1;
    }
    if (isObject(value)) {
        var other = typeof value.valueOf == "function" ? value.valueOf() : value;
        value = isObject(other) ? other + "" : other;
    }
    if (typeof value != "string") {
        return value === 0 ? value : +value;
    }
    value = baseTrim(value);
    var isBinary1 = reIsBinary.test(value);
    return isBinary1 || reIsOctal.test(value) ? freeParseInt(value.slice(2), isBinary1 ? 2 : 8) : reIsBadHex.test(value) ? NAN$1 : +value;
}
var INFINITY$1 = 1 / 0, MAX_INTEGER = 179769313486231570000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
function toFinite(value) {
    if (!value) {
        return value === 0 ? value : 0;
    }
    value = toNumber(value);
    if (value === INFINITY$1 || value === -INFINITY$1) {
        var sign = value < 0 ? -1 : 1;
        return sign * MAX_INTEGER;
    }
    return value === value ? value : 0;
}
function toInteger(value) {
    var result2 = toFinite(value), remainder = result2 % 1;
    return result2 === result2 ? remainder ? result2 - remainder : result2 : 0;
}
var FUNC_ERROR_TEXT = "Expected a function";
function after(n3, func2) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    n3 = toInteger(n3);
    return function() {
        if (--n3 < 1) {
            return func2.apply(this, arguments);
        }
    };
}
function identity(value) {
    return value;
}
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value) {
    if (!isObject(value)) {
        return false;
    }
    var tag = baseGetTag(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
}
var coreJsData = root["__core-js_shared__"];
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func2) {
    return !!maskSrcKey && maskSrcKey in func2;
}
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func2) {
    if (func2 != null) {
        try {
            return funcToString.call(func2);
        } catch (e) {
        }
        try {
            return func2 + "";
        } catch (e1) {
        }
    }
    return "";
}
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto$1 = Function.prototype, objectProto$2 = Object.prototype;
var funcToString$1 = funcProto$1.toString;
var hasOwnProperty$1 = objectProto$2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString$1.call(hasOwnProperty$1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value) {
    if (!isObject(value) || isMasked(value)) {
        return false;
    }
    var pattern3 = isFunction(value) ? reIsNative : reIsHostCtor;
    return pattern3.test(toSource(value));
}
function getValue(object2, key) {
    return object2 == null ? void 0 : object2[key];
}
function getNative(object2, key) {
    var value = getValue(object2, key);
    return baseIsNative(value) ? value : void 0;
}
var WeakMap = getNative(root, "WeakMap");
var metaMap = WeakMap && new WeakMap();
var baseSetData = !metaMap ? identity : function(func2, data4) {
    metaMap.set(func2, data4);
    return func2;
};
var objectCreate = Object.create;
var baseCreate = function() {
    function object2() {
    }
    return function(proto1) {
        if (!isObject(proto1)) {
            return {
            };
        }
        if (objectCreate) {
            return objectCreate(proto1);
        }
        object2.prototype = proto1;
        var result2 = new object2();
        object2.prototype = void 0;
        return result2;
    };
}();
function createCtor(Ctor) {
    return function() {
        var args = arguments;
        switch(args.length){
            case 0:
                return new Ctor();
            case 1:
                return new Ctor(args[0]);
            case 2:
                return new Ctor(args[0], args[1]);
            case 3:
                return new Ctor(args[0], args[1], args[2]);
            case 4:
                return new Ctor(args[0], args[1], args[2], args[3]);
            case 5:
                return new Ctor(args[0], args[1], args[2], args[3], args[4]);
            case 6:
                return new Ctor(args[0], args[1], args[2], args[3], args[4], args[5]);
            case 7:
                return new Ctor(args[0], args[1], args[2], args[3], args[4], args[5], args[6]);
        }
        var thisBinding = baseCreate(Ctor.prototype), result2 = Ctor.apply(thisBinding, args);
        return isObject(result2) ? result2 : thisBinding;
    };
}
var WRAP_BIND_FLAG = 1;
function createBind(func2, bitmask, thisArg) {
    var isBind = bitmask & WRAP_BIND_FLAG, Ctor = createCtor(func2);
    function wrapper() {
        var fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        return fn.apply(isBind ? thisArg : this, arguments);
    }
    return wrapper;
}
function apply(func2, thisArg, args) {
    switch(args.length){
        case 0:
            return func2.call(thisArg);
        case 1:
            return func2.call(thisArg, args[0]);
        case 2:
            return func2.call(thisArg, args[0], args[1]);
        case 3:
            return func2.call(thisArg, args[0], args[1], args[2]);
    }
    return func2.apply(thisArg, args);
}
var nativeMax = Math.max;
function composeArgs(args, partials, holders, isCurried) {
    var argsIndex = -1, argsLength = args.length, holdersLength = holders.length, leftIndex = -1, leftLength = partials.length, rangeLength = nativeMax(argsLength - holdersLength, 0), result2 = Array(leftLength + rangeLength), isUncurried = !isCurried;
    while(++leftIndex < leftLength){
        result2[leftIndex] = partials[leftIndex];
    }
    while(++argsIndex < holdersLength){
        if (isUncurried || argsIndex < argsLength) {
            result2[holders[argsIndex]] = args[argsIndex];
        }
    }
    while(rangeLength--){
        result2[leftIndex++] = args[argsIndex++];
    }
    return result2;
}
var nativeMax$1 = Math.max;
function composeArgsRight(args, partials, holders, isCurried) {
    var argsIndex = -1, argsLength = args.length, holdersIndex = -1, holdersLength = holders.length, rightIndex = -1, rightLength = partials.length, rangeLength = nativeMax$1(argsLength - holdersLength, 0), result2 = Array(rangeLength + rightLength), isUncurried = !isCurried;
    while(++argsIndex < rangeLength){
        result2[argsIndex] = args[argsIndex];
    }
    var offset = argsIndex;
    while(++rightIndex < rightLength){
        result2[offset + rightIndex] = partials[rightIndex];
    }
    while(++holdersIndex < holdersLength){
        if (isUncurried || argsIndex < argsLength) {
            result2[offset + holders[holdersIndex]] = args[argsIndex++];
        }
    }
    return result2;
}
function countHolders(array2, placeholder) {
    var length = array2.length, result2 = 0;
    while(length--){
        if (array2[length] === placeholder) {
            ++result2;
        }
    }
    return result2;
}
function baseLodash() {
}
var MAX_ARRAY_LENGTH = 4294967295;
function LazyWrapper(value) {
    this.__wrapped__ = value;
    this.__actions__ = [];
    this.__dir__ = 1;
    this.__filtered__ = false;
    this.__iteratees__ = [];
    this.__takeCount__ = MAX_ARRAY_LENGTH;
    this.__views__ = [];
}
LazyWrapper.prototype = baseCreate(baseLodash.prototype);
LazyWrapper.prototype.constructor = LazyWrapper;
function noop() {
}
var getData = !metaMap ? noop : function(func2) {
    return metaMap.get(func2);
};
var realNames = {
};
var objectProto$3 = Object.prototype;
var hasOwnProperty$2 = objectProto$3.hasOwnProperty;
function getFuncName(func2) {
    var result2 = func2.name + "", array2 = realNames[result2], length = hasOwnProperty$2.call(realNames, result2) ? array2.length : 0;
    while(length--){
        var data5 = array2[length], otherFunc = data5.func;
        if (otherFunc == null || otherFunc == func2) {
            return data5.name;
        }
    }
    return result2;
}
function LodashWrapper(value, chainAll) {
    this.__wrapped__ = value;
    this.__actions__ = [];
    this.__chain__ = !!chainAll;
    this.__index__ = 0;
    this.__values__ = void 0;
}
LodashWrapper.prototype = baseCreate(baseLodash.prototype);
LodashWrapper.prototype.constructor = LodashWrapper;
function copyArray(source, array2) {
    var index = -1, length = source.length;
    array2 || (array2 = Array(length));
    while(++index < length){
        array2[index] = source[index];
    }
    return array2;
}
function wrapperClone(wrapper) {
    if (wrapper instanceof LazyWrapper) {
        return wrapper.clone();
    }
    var result2 = new LodashWrapper(wrapper.__wrapped__, wrapper.__chain__);
    result2.__actions__ = copyArray(wrapper.__actions__);
    result2.__index__ = wrapper.__index__;
    result2.__values__ = wrapper.__values__;
    return result2;
}
var objectProto$4 = Object.prototype;
var hasOwnProperty$3 = objectProto$4.hasOwnProperty;
function lodash(value) {
    if (isObjectLike(value) && !isArray(value) && !(value instanceof LazyWrapper)) {
        if (value instanceof LodashWrapper) {
            return value;
        }
        if (hasOwnProperty$3.call(value, "__wrapped__")) {
            return wrapperClone(value);
        }
    }
    return new LodashWrapper(value);
}
lodash.prototype = baseLodash.prototype;
lodash.prototype.constructor = lodash;
function isLaziable(func2) {
    var funcName = getFuncName(func2), other = lodash[funcName];
    if (typeof other != "function" || !(funcName in LazyWrapper.prototype)) {
        return false;
    }
    if (func2 === other) {
        return true;
    }
    var data6 = getData(other);
    return !!data6 && func2 === data6[0];
}
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func2) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func2.apply(void 0, arguments);
    };
}
var setData = shortOut(baseSetData);
var reWrapDetails = /\{\n\/\* \[wrapped with (.+)\] \*/, reSplitDetails = /,? & /;
function getWrapDetails(source) {
    var match = source.match(reWrapDetails);
    return match ? match[1].split(reSplitDetails) : [];
}
var reWrapComment = /\{(?:\n\/\* \[wrapped with .+\] \*\/)?\n?/;
function insertWrapDetails(source, details) {
    var length = details.length;
    if (!length) {
        return source;
    }
    var lastIndex = length - 1;
    details[lastIndex] = (length > 1 ? "& " : "") + details[lastIndex];
    details = details.join(length > 2 ? ", " : " ");
    return source.replace(reWrapComment, "{\n/* [wrapped with " + details + "] */\n");
}
function constant(value) {
    return function() {
        return value;
    };
}
var defineProperty = function() {
    try {
        var func2 = getNative(Object, "defineProperty");
        func2({
        }, "", {
        });
        return func2;
    } catch (e) {
    }
}();
var baseSetToString = !defineProperty ? identity : function(func2, string2) {
    return defineProperty(func2, "toString", {
        configurable: true,
        enumerable: false,
        value: constant(string2),
        writable: true
    });
};
var setToString = shortOut(baseSetToString);
function arrayEach(array2, iteratee2) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (iteratee2(array2[index], index, array2) === false) {
            break;
        }
    }
    return array2;
}
function baseFindIndex(array2, predicate, fromIndex, fromRight) {
    var length = array2.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array2[index], index, array2)) {
            return index;
        }
    }
    return -1;
}
function baseIsNaN(value) {
    return value !== value;
}
function strictIndexOf(array2, value, fromIndex) {
    var index = fromIndex - 1, length = array2.length;
    while(++index < length){
        if (array2[index] === value) {
            return index;
        }
    }
    return -1;
}
function baseIndexOf(array2, value, fromIndex) {
    return value === value ? strictIndexOf(array2, value, fromIndex) : baseFindIndex(array2, baseIsNaN, fromIndex);
}
function arrayIncludes(array2, value) {
    var length = array2 == null ? 0 : array2.length;
    return !!length && baseIndexOf(array2, value, 0) > -1;
}
var WRAP_BIND_FLAG$1 = 1, WRAP_BIND_KEY_FLAG = 2, WRAP_CURRY_FLAG = 8, WRAP_CURRY_RIGHT_FLAG = 16, WRAP_PARTIAL_FLAG = 32, WRAP_PARTIAL_RIGHT_FLAG = 64, WRAP_ARY_FLAG = 128, WRAP_REARG_FLAG = 256, WRAP_FLIP_FLAG = 512;
var wrapFlags = [
    [
        "ary",
        WRAP_ARY_FLAG
    ],
    [
        "bind",
        WRAP_BIND_FLAG$1
    ],
    [
        "bindKey",
        WRAP_BIND_KEY_FLAG
    ],
    [
        "curry",
        WRAP_CURRY_FLAG
    ],
    [
        "curryRight",
        WRAP_CURRY_RIGHT_FLAG
    ],
    [
        "flip",
        WRAP_FLIP_FLAG
    ],
    [
        "partial",
        WRAP_PARTIAL_FLAG
    ],
    [
        "partialRight",
        WRAP_PARTIAL_RIGHT_FLAG
    ],
    [
        "rearg",
        WRAP_REARG_FLAG
    ]
];
function updateWrapDetails(details, bitmask) {
    arrayEach(wrapFlags, function(pair) {
        var value = "_." + pair[0];
        if (bitmask & pair[1] && !arrayIncludes(details, value)) {
            details.push(value);
        }
    });
    return details.sort();
}
function setWrapToString(wrapper, reference, bitmask) {
    var source = reference + "";
    return setToString(wrapper, insertWrapDetails(source, updateWrapDetails(getWrapDetails(source), bitmask)));
}
var WRAP_BIND_FLAG$2 = 1, WRAP_BIND_KEY_FLAG$1 = 2, WRAP_CURRY_BOUND_FLAG = 4, WRAP_CURRY_FLAG$1 = 8, WRAP_PARTIAL_FLAG$1 = 32, WRAP_PARTIAL_RIGHT_FLAG$1 = 64;
function createRecurry(func2, bitmask, wrapFunc, placeholder, thisArg, partials, holders, argPos, ary2, arity) {
    var isCurry = bitmask & WRAP_CURRY_FLAG$1, newHolders = isCurry ? holders : void 0, newHoldersRight = isCurry ? void 0 : holders, newPartials = isCurry ? partials : void 0, newPartialsRight = isCurry ? void 0 : partials;
    bitmask |= isCurry ? WRAP_PARTIAL_FLAG$1 : WRAP_PARTIAL_RIGHT_FLAG$1;
    bitmask &= ~(isCurry ? WRAP_PARTIAL_RIGHT_FLAG$1 : WRAP_PARTIAL_FLAG$1);
    if (!(bitmask & WRAP_CURRY_BOUND_FLAG)) {
        bitmask &= ~(WRAP_BIND_FLAG$2 | WRAP_BIND_KEY_FLAG$1);
    }
    var newData = [
        func2,
        bitmask,
        thisArg,
        newPartials,
        newHolders,
        newPartialsRight,
        newHoldersRight,
        argPos,
        ary2,
        arity
    ];
    var result2 = wrapFunc.apply(void 0, newData);
    if (isLaziable(func2)) {
        setData(result2, newData);
    }
    result2.placeholder = placeholder;
    return setWrapToString(result2, func2, bitmask);
}
function getHolder(func2) {
    var object2 = func2;
    return object2.placeholder;
}
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value, length) {
    var type25 = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type25 == "number" || type25 != "symbol" && reIsUint.test(value)) && value > -1 && value % 1 == 0 && value < length;
}
var nativeMin = Math.min;
function reorder(array2, indexes) {
    var arrLength = array2.length, length = nativeMin(indexes.length, arrLength), oldArray = copyArray(array2);
    while(length--){
        var index = indexes[length];
        array2[length] = isIndex(index, arrLength) ? oldArray[index] : void 0;
    }
    return array2;
}
var PLACEHOLDER = "__lodash_placeholder__";
function replaceHolders(array2, placeholder) {
    var index = -1, length = array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value = array2[index];
        if (value === placeholder || value === PLACEHOLDER) {
            array2[index] = PLACEHOLDER;
            result2[resIndex++] = index;
        }
    }
    return result2;
}
var WRAP_BIND_FLAG$3 = 1, WRAP_BIND_KEY_FLAG$2 = 2, WRAP_CURRY_FLAG$2 = 8, WRAP_CURRY_RIGHT_FLAG$1 = 16, WRAP_ARY_FLAG$1 = 128, WRAP_FLIP_FLAG$1 = 512;
function createHybrid(func2, bitmask, thisArg, partials, holders, partialsRight, holdersRight, argPos, ary2, arity) {
    var isAry = bitmask & WRAP_ARY_FLAG$1, isBind = bitmask & WRAP_BIND_FLAG$3, isBindKey = bitmask & WRAP_BIND_KEY_FLAG$2, isCurried = bitmask & (WRAP_CURRY_FLAG$2 | WRAP_CURRY_RIGHT_FLAG$1), isFlip = bitmask & WRAP_FLIP_FLAG$1, Ctor = isBindKey ? void 0 : createCtor(func2);
    function wrapper() {
        var length = arguments.length, args = Array(length), index = length;
        while(index--){
            args[index] = arguments[index];
        }
        if (isCurried) {
            var placeholder = getHolder(wrapper), holdersCount = countHolders(args, placeholder);
        }
        if (partials) {
            args = composeArgs(args, partials, holders, isCurried);
        }
        if (partialsRight) {
            args = composeArgsRight(args, partialsRight, holdersRight, isCurried);
        }
        length -= holdersCount;
        if (isCurried && length < arity) {
            var newHolders = replaceHolders(args, placeholder);
            return createRecurry(func2, bitmask, createHybrid, wrapper.placeholder, thisArg, args, newHolders, argPos, ary2, arity - length);
        }
        var thisBinding = isBind ? thisArg : this, fn = isBindKey ? thisBinding[func2] : func2;
        length = args.length;
        if (argPos) {
            args = reorder(args, argPos);
        } else if (isFlip && length > 1) {
            args.reverse();
        }
        if (isAry && ary2 < length) {
            args.length = ary2;
        }
        if (this && this !== root && this instanceof wrapper) {
            fn = Ctor || createCtor(fn);
        }
        return fn.apply(thisBinding, args);
    }
    return wrapper;
}
function createCurry(func2, bitmask, arity) {
    var Ctor = createCtor(func2);
    function wrapper() {
        var length = arguments.length, args = Array(length), index = length, placeholder = getHolder(wrapper);
        while(index--){
            args[index] = arguments[index];
        }
        var holders = length < 3 && args[0] !== placeholder && args[length - 1] !== placeholder ? [] : replaceHolders(args, placeholder);
        length -= holders.length;
        if (length < arity) {
            return createRecurry(func2, bitmask, createHybrid, wrapper.placeholder, void 0, args, holders, void 0, void 0, arity - length);
        }
        var fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        return apply(fn, this, args);
    }
    return wrapper;
}
var WRAP_BIND_FLAG$4 = 1;
function createPartial(func2, bitmask, thisArg, partials) {
    var isBind = bitmask & WRAP_BIND_FLAG$4, Ctor = createCtor(func2);
    function wrapper() {
        var argsIndex = -1, argsLength = arguments.length, leftIndex = -1, leftLength = partials.length, args = Array(leftLength + argsLength), fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        while(++leftIndex < leftLength){
            args[leftIndex] = partials[leftIndex];
        }
        while(argsLength--){
            args[leftIndex++] = arguments[++argsIndex];
        }
        return apply(fn, isBind ? thisArg : this, args);
    }
    return wrapper;
}
var PLACEHOLDER$1 = "__lodash_placeholder__";
var WRAP_BIND_FLAG$5 = 1, WRAP_BIND_KEY_FLAG$3 = 2, WRAP_CURRY_BOUND_FLAG$1 = 4, WRAP_CURRY_FLAG$3 = 8, WRAP_ARY_FLAG$2 = 128, WRAP_REARG_FLAG$1 = 256;
var nativeMin$1 = Math.min;
function mergeData(data7, source) {
    var bitmask = data7[1], srcBitmask = source[1], newBitmask = bitmask | srcBitmask, isCommon = newBitmask < (WRAP_BIND_FLAG$5 | WRAP_BIND_KEY_FLAG$3 | WRAP_ARY_FLAG$2);
    var isCombo = srcBitmask == WRAP_ARY_FLAG$2 && bitmask == WRAP_CURRY_FLAG$3 || srcBitmask == WRAP_ARY_FLAG$2 && bitmask == WRAP_REARG_FLAG$1 && data7[7].length <= source[8] || srcBitmask == (WRAP_ARY_FLAG$2 | WRAP_REARG_FLAG$1) && source[7].length <= source[8] && bitmask == WRAP_CURRY_FLAG$3;
    if (!(isCommon || isCombo)) {
        return data7;
    }
    if (srcBitmask & WRAP_BIND_FLAG$5) {
        data7[2] = source[2];
        newBitmask |= bitmask & WRAP_BIND_FLAG$5 ? 0 : WRAP_CURRY_BOUND_FLAG$1;
    }
    var value = source[3];
    if (value) {
        var partials = data7[3];
        data7[3] = partials ? composeArgs(partials, value, source[4]) : value;
        data7[4] = partials ? replaceHolders(data7[3], PLACEHOLDER$1) : source[4];
    }
    value = source[5];
    if (value) {
        partials = data7[5];
        data7[5] = partials ? composeArgsRight(partials, value, source[6]) : value;
        data7[6] = partials ? replaceHolders(data7[5], PLACEHOLDER$1) : source[6];
    }
    value = source[7];
    if (value) {
        data7[7] = value;
    }
    if (srcBitmask & WRAP_ARY_FLAG$2) {
        data7[8] = data7[8] == null ? source[8] : nativeMin$1(data7[8], source[8]);
    }
    if (data7[9] == null) {
        data7[9] = source[9];
    }
    data7[0] = source[0];
    data7[1] = newBitmask;
    return data7;
}
var FUNC_ERROR_TEXT$1 = "Expected a function";
var WRAP_BIND_FLAG$6 = 1, WRAP_BIND_KEY_FLAG$4 = 2, WRAP_CURRY_FLAG$4 = 8, WRAP_CURRY_RIGHT_FLAG$2 = 16, WRAP_PARTIAL_FLAG$2 = 32, WRAP_PARTIAL_RIGHT_FLAG$2 = 64;
var nativeMax$2 = Math.max;
function createWrap(func2, bitmask, thisArg, partials, holders, argPos, ary2, arity) {
    var isBindKey = bitmask & WRAP_BIND_KEY_FLAG$4;
    if (!isBindKey && typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$1);
    }
    var length = partials ? partials.length : 0;
    if (!length) {
        bitmask &= ~(WRAP_PARTIAL_FLAG$2 | WRAP_PARTIAL_RIGHT_FLAG$2);
        partials = holders = void 0;
    }
    ary2 = ary2 === void 0 ? ary2 : nativeMax$2(toInteger(ary2), 0);
    arity = arity === void 0 ? arity : toInteger(arity);
    length -= holders ? holders.length : 0;
    if (bitmask & WRAP_PARTIAL_RIGHT_FLAG$2) {
        var partialsRight = partials, holdersRight = holders;
        partials = holders = void 0;
    }
    var data8 = isBindKey ? void 0 : getData(func2);
    var newData = [
        func2,
        bitmask,
        thisArg,
        partials,
        holders,
        partialsRight,
        holdersRight,
        argPos,
        ary2,
        arity
    ];
    if (data8) {
        mergeData(newData, data8);
    }
    func2 = newData[0];
    bitmask = newData[1];
    thisArg = newData[2];
    partials = newData[3];
    holders = newData[4];
    arity = newData[9] = newData[9] === void 0 ? isBindKey ? 0 : func2.length : nativeMax$2(newData[9] - length, 0);
    if (!arity && bitmask & (WRAP_CURRY_FLAG$4 | WRAP_CURRY_RIGHT_FLAG$2)) {
        bitmask &= ~(WRAP_CURRY_FLAG$4 | WRAP_CURRY_RIGHT_FLAG$2);
    }
    if (!bitmask || bitmask == WRAP_BIND_FLAG$6) {
        var result2 = createBind(func2, bitmask, thisArg);
    } else if (bitmask == WRAP_CURRY_FLAG$4 || bitmask == WRAP_CURRY_RIGHT_FLAG$2) {
        result2 = createCurry(func2, bitmask, arity);
    } else if ((bitmask == WRAP_PARTIAL_FLAG$2 || bitmask == (WRAP_BIND_FLAG$6 | WRAP_PARTIAL_FLAG$2)) && !holders.length) {
        result2 = createPartial(func2, bitmask, thisArg, partials);
    } else {
        result2 = createHybrid.apply(void 0, newData);
    }
    var setter = data8 ? baseSetData : setData;
    return setWrapToString(setter(result2, newData), func2, bitmask);
}
var WRAP_ARY_FLAG$3 = 128;
function ary(func2, n4, guard) {
    n4 = guard ? void 0 : n4;
    n4 = func2 && n4 == null ? func2.length : n4;
    return createWrap(func2, WRAP_ARY_FLAG$3, void 0, void 0, void 0, void 0, n4);
}
function baseAssignValue(object2, key, value) {
    if (key == "__proto__" && defineProperty) {
        defineProperty(object2, key, {
            configurable: true,
            enumerable: true,
            value,
            writable: true
        });
    } else {
        object2[key] = value;
    }
}
function eq(value, other) {
    return value === other || value !== value && other !== other;
}
var objectProto$5 = Object.prototype;
var hasOwnProperty$4 = objectProto$5.hasOwnProperty;
function assignValue(object2, key, value) {
    var objValue = object2[key];
    if (!(hasOwnProperty$4.call(object2, key) && eq(objValue, value)) || value === void 0 && !(key in object2)) {
        baseAssignValue(object2, key, value);
    }
}
function copyObject(source, props, object2, customizer) {
    var isNew = !object2;
    object2 || (object2 = {
    });
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object2[key], source[key], key, object2, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            baseAssignValue(object2, key, newValue);
        } else {
            assignValue(object2, key, newValue);
        }
    }
    return object2;
}
var nativeMax$3 = Math.max;
function overRest(func2, start, transform2) {
    start = nativeMax$3(start === void 0 ? func2.length - 1 : start, 0);
    return function() {
        var args = arguments, index = -1, length = nativeMax$3(args.length - start, 0), array2 = Array(length);
        while(++index < length){
            array2[index] = args[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args[index];
        }
        otherArgs[start] = transform2(array2);
        return apply(func2, this, otherArgs);
    };
}
function baseRest(func2, start) {
    return setToString(overRest(func2, start, identity), func2 + "");
}
var MAX_SAFE_INTEGER$1 = 9007199254740991;
function isLength(value) {
    return typeof value == "number" && value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER$1;
}
function isArrayLike(value) {
    return value != null && isLength(value.length) && !isFunction(value);
}
function isIterateeCall(value, index, object2) {
    if (!isObject(object2)) {
        return false;
    }
    var type26 = typeof index;
    if (type26 == "number" ? isArrayLike(object2) && isIndex(index, object2.length) : type26 == "string" && index in object2) {
        return eq(object2[index], value);
    }
    return false;
}
function createAssigner(assigner) {
    return baseRest(function(object2, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && isIterateeCall(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object2 = Object(object2);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object2, source, index, customizer);
            }
        }
        return object2;
    });
}
var objectProto$6 = Object.prototype;
function isPrototype(value) {
    var Ctor = value && value.constructor, proto2 = typeof Ctor == "function" && Ctor.prototype || objectProto$6;
    return value === proto2;
}
function baseTimes(n5, iteratee2) {
    var index = -1, result2 = Array(n5);
    while(++index < n5){
        result2[index] = iteratee2(index);
    }
    return result2;
}
var argsTag = "[object Arguments]";
function baseIsArguments(value) {
    return isObjectLike(value) && baseGetTag(value) == argsTag;
}
var objectProto$7 = Object.prototype;
var hasOwnProperty$5 = objectProto$7.hasOwnProperty;
var propertyIsEnumerable = objectProto$7.propertyIsEnumerable;
var isArguments = baseIsArguments(function() {
    return arguments;
}()) ? baseIsArguments : function(value) {
    return isObjectLike(value) && hasOwnProperty$5.call(value, "callee") && !propertyIsEnumerable.call(value, "callee");
};
function stubFalse() {
    return false;
}
var freeExports = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule = freeExports && typeof module == "object" && module && !module.nodeType && module;
var moduleExports = freeModule && freeModule.exports === freeExports;
var Buffer1 = moduleExports ? root.Buffer : void 0;
var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
var isBuffer = nativeIsBuffer || stubFalse;
var argsTag$1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag$1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {
};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag$1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag$1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value) {
    return isObjectLike(value) && isLength(value.length) && !!typedArrayTags[baseGetTag(value)];
}
function baseUnary(func2) {
    return function(value) {
        return func2(value);
    };
}
var freeExports$1 = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule$1 = freeExports$1 && typeof module == "object" && module && !module.nodeType && module;
var moduleExports$1 = freeModule$1 && freeModule$1.exports === freeExports$1;
var freeProcess = moduleExports$1 && freeGlobal.process;
var nodeUtil = function() {
    try {
        var types = freeModule$1 && freeModule$1.require && freeModule$1.require("util").types;
        if (types) {
            return types;
        }
        return freeProcess && freeProcess.binding && freeProcess.binding("util");
    } catch (e) {
    }
}();
var nodeIsTypedArray = nodeUtil && nodeUtil.isTypedArray;
var isTypedArray = nodeIsTypedArray ? baseUnary(nodeIsTypedArray) : baseIsTypedArray;
var objectProto$8 = Object.prototype;
var hasOwnProperty$6 = objectProto$8.hasOwnProperty;
function arrayLikeKeys(value, inherited) {
    var isArr = isArray(value), isArg = !isArr && isArguments(value), isBuff = !isArr && !isArg && isBuffer(value), isType = !isArr && !isArg && !isBuff && isTypedArray(value), skipIndexes = isArr || isArg || isBuff || isType, result2 = skipIndexes ? baseTimes(value.length, String) : [], length = result2.length;
    for(var key in value){
        if ((inherited || hasOwnProperty$6.call(value, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || isIndex(key, length)))) {
            result2.push(key);
        }
    }
    return result2;
}
function overArg(func2, transform2) {
    return function(arg) {
        return func2(transform2(arg));
    };
}
var nativeKeys = overArg(Object.keys, Object);
var objectProto$9 = Object.prototype;
var hasOwnProperty$7 = objectProto$9.hasOwnProperty;
function baseKeys(object2) {
    if (!isPrototype(object2)) {
        return nativeKeys(object2);
    }
    var result2 = [];
    for(var key in Object(object2)){
        if (hasOwnProperty$7.call(object2, key) && key != "constructor") {
            result2.push(key);
        }
    }
    return result2;
}
function keys(object2) {
    return isArrayLike(object2) ? arrayLikeKeys(object2) : baseKeys(object2);
}
var objectProto$a = Object.prototype;
var hasOwnProperty$8 = objectProto$a.hasOwnProperty;
var assign = createAssigner(function(object2, source) {
    if (isPrototype(source) || isArrayLike(source)) {
        copyObject(source, keys(source), object2);
        return;
    }
    for(var key in source){
        if (hasOwnProperty$8.call(source, key)) {
            assignValue(object2, key, source[key]);
        }
    }
});
function nativeKeysIn(object2) {
    var result2 = [];
    if (object2 != null) {
        for(var key in Object(object2)){
            result2.push(key);
        }
    }
    return result2;
}
var objectProto$b = Object.prototype;
var hasOwnProperty$9 = objectProto$b.hasOwnProperty;
function baseKeysIn(object2) {
    if (!isObject(object2)) {
        return nativeKeysIn(object2);
    }
    var isProto = isPrototype(object2), result2 = [];
    for(var key in object2){
        if (!(key == "constructor" && (isProto || !hasOwnProperty$9.call(object2, key)))) {
            result2.push(key);
        }
    }
    return result2;
}
function keysIn(object2) {
    return isArrayLike(object2) ? arrayLikeKeys(object2, true) : baseKeysIn(object2);
}
var assignIn = createAssigner(function(object2, source) {
    copyObject(source, keysIn(source), object2);
});
var assignInWith = createAssigner(function(object2, source, srcIndex, customizer) {
    copyObject(source, keysIn(source), object2, customizer);
});
var assignWith = createAssigner(function(object2, source, srcIndex, customizer) {
    copyObject(source, keys(source), object2, customizer);
});
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value, object2) {
    if (isArray(value)) {
        return false;
    }
    var type27 = typeof value;
    if (type27 == "number" || type27 == "symbol" || type27 == "boolean" || value == null || isSymbol(value)) {
        return true;
    }
    return reIsPlainProp.test(value) || !reIsDeepProp.test(value) || object2 != null && value in Object(object2);
}
var nativeCreate = getNative(Object, "create");
function hashClear() {
    this.__data__ = nativeCreate ? nativeCreate(null) : {
    };
    this.size = 0;
}
function hashDelete(key) {
    var result2 = this.has(key) && delete this.__data__[key];
    this.size -= result2 ? 1 : 0;
    return result2;
}
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto$c = Object.prototype;
var hasOwnProperty$a = objectProto$c.hasOwnProperty;
function hashGet(key) {
    var data9 = this.__data__;
    if (nativeCreate) {
        var result2 = data9[key];
        return result2 === HASH_UNDEFINED ? void 0 : result2;
    }
    return hasOwnProperty$a.call(data9, key) ? data9[key] : void 0;
}
var objectProto$d = Object.prototype;
var hasOwnProperty$b = objectProto$d.hasOwnProperty;
function hashHas(key) {
    var data10 = this.__data__;
    return nativeCreate ? data10[key] !== void 0 : hasOwnProperty$b.call(data10, key);
}
var HASH_UNDEFINED$1 = "__lodash_hash_undefined__";
function hashSet(key, value) {
    var data11 = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data11[key] = nativeCreate && value === void 0 ? HASH_UNDEFINED$1 : value;
    return this;
}
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = hashClear;
Hash.prototype["delete"] = hashDelete;
Hash.prototype.get = hashGet;
Hash.prototype.has = hashHas;
Hash.prototype.set = hashSet;
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
function assocIndexOf(array2, key) {
    var length = array2.length;
    while(length--){
        if (eq(array2[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data12 = this.__data__, index = assocIndexOf(data12, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data12.length - 1;
    if (index == lastIndex) {
        data12.pop();
    } else {
        splice.call(data12, index, 1);
    }
    --this.size;
    return true;
}
function listCacheGet(key) {
    var data13 = this.__data__, index = assocIndexOf(data13, key);
    return index < 0 ? void 0 : data13[index][1];
}
function listCacheHas(key) {
    return assocIndexOf(this.__data__, key) > -1;
}
function listCacheSet(key, value) {
    var data14 = this.__data__, index = assocIndexOf(data14, key);
    if (index < 0) {
        ++this.size;
        data14.push([
            key,
            value
        ]);
    } else {
        data14[index][1] = value;
    }
    return this;
}
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = listCacheClear;
ListCache.prototype["delete"] = listCacheDelete;
ListCache.prototype.get = listCacheGet;
ListCache.prototype.has = listCacheHas;
ListCache.prototype.set = listCacheSet;
var Map1 = getNative(root, "Map");
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new Hash(),
        map: new (Map1 || ListCache)(),
        string: new Hash()
    };
}
function isKeyable(value) {
    var type28 = typeof value;
    return type28 == "string" || type28 == "number" || type28 == "symbol" || type28 == "boolean" ? value !== "__proto__" : value === null;
}
function getMapData(map2, key) {
    var data15 = map2.__data__;
    return isKeyable(key) ? data15[typeof key == "string" ? "string" : "hash"] : data15.map;
}
function mapCacheDelete(key) {
    var result2 = getMapData(this, key)["delete"](key);
    this.size -= result2 ? 1 : 0;
    return result2;
}
function mapCacheGet(key) {
    return getMapData(this, key).get(key);
}
function mapCacheHas(key) {
    return getMapData(this, key).has(key);
}
function mapCacheSet(key, value) {
    var data16 = getMapData(this, key), size2 = data16.size;
    data16.set(key, value);
    this.size += data16.size == size2 ? 0 : 1;
    return this;
}
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = mapCacheClear;
MapCache.prototype["delete"] = mapCacheDelete;
MapCache.prototype.get = mapCacheGet;
MapCache.prototype.has = mapCacheHas;
MapCache.prototype.set = mapCacheSet;
var FUNC_ERROR_TEXT$2 = "Expected a function";
function memoize(func2, resolver) {
    if (typeof func2 != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$2);
    }
    var memoized = function() {
        var args = arguments, key = resolver ? resolver.apply(this, args) : args[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result2 = func2.apply(this, args);
        memoized.cache = cache.set(key, result2) || cache;
        return result2;
    };
    memoized.cache = new (memoize.Cache || MapCache)();
    return memoized;
}
memoize.Cache = MapCache;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func2) {
    var result2 = memoize(func2, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result2.cache;
    return result2;
}
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = memoizeCapped(function(string2) {
    var result2 = [];
    if (string2.charCodeAt(0) === 46) {
        result2.push("");
    }
    string2.replace(rePropName, function(match, number2, quote, subString) {
        result2.push(quote ? subString.replace(reEscapeChar, "$1") : number2 || match);
    });
    return result2;
});
function toString(value) {
    return value == null ? "" : baseToString(value);
}
function castPath(value, object2) {
    if (isArray(value)) {
        return value;
    }
    return isKey(value, object2) ? [
        value
    ] : stringToPath(toString(value));
}
var INFINITY$2 = 1 / 0;
function toKey(value) {
    if (typeof value == "string" || isSymbol(value)) {
        return value;
    }
    var result2 = value + "";
    return result2 == "0" && 1 / value == -INFINITY$2 ? "-0" : result2;
}
function baseGet(object2, path49) {
    path49 = castPath(path49, object2);
    var index = 0, length = path49.length;
    while(object2 != null && index < length){
        object2 = object2[toKey(path49[index++])];
    }
    return index && index == length ? object2 : void 0;
}
function get(object2, path50, defaultValue) {
    var result2 = object2 == null ? void 0 : baseGet(object2, path50);
    return result2 === void 0 ? defaultValue : result2;
}
function baseAt(object2, paths) {
    var index = -1, length = paths.length, result2 = Array(length), skip = object2 == null;
    while(++index < length){
        result2[index] = skip ? void 0 : get(object2, paths[index]);
    }
    return result2;
}
function arrayPush(array2, values2) {
    var index = -1, length = values2.length, offset = array2.length;
    while(++index < length){
        array2[offset + index] = values2[index];
    }
    return array2;
}
var spreadableSymbol = Symbol1 ? Symbol1.isConcatSpreadable : void 0;
function isFlattenable(value) {
    return isArray(value) || isArguments(value) || !!(spreadableSymbol && value && value[spreadableSymbol]);
}
function baseFlatten(array2, depth, predicate, isStrict, result2) {
    var index = -1, length = array2.length;
    predicate || (predicate = isFlattenable);
    result2 || (result2 = []);
    while(++index < length){
        var value = array2[index];
        if (depth > 0 && predicate(value)) {
            if (depth > 1) {
                baseFlatten(value, depth - 1, predicate, isStrict, result2);
            } else {
                arrayPush(result2, value);
            }
        } else if (!isStrict) {
            result2[result2.length] = value;
        }
    }
    return result2;
}
function flatten(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseFlatten(array2, 1) : [];
}
function flatRest(func2) {
    return setToString(overRest(func2, void 0, flatten), func2 + "");
}
var at = flatRest(baseAt);
var getPrototype = overArg(Object.getPrototypeOf, Object);
var objectTag$1 = "[object Object]";
var funcProto$2 = Function.prototype, objectProto$e = Object.prototype;
var funcToString$2 = funcProto$2.toString;
var hasOwnProperty$c = objectProto$e.hasOwnProperty;
var objectCtorString = funcToString$2.call(Object);
function isPlainObject(value) {
    if (!isObjectLike(value) || baseGetTag(value) != objectTag$1) {
        return false;
    }
    var proto3 = getPrototype(value);
    if (proto3 === null) {
        return true;
    }
    var Ctor = hasOwnProperty$c.call(proto3, "constructor") && proto3.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString$2.call(Ctor) == objectCtorString;
}
var domExcTag = "[object DOMException]", errorTag$1 = "[object Error]";
function isError(value) {
    if (!isObjectLike(value)) {
        return false;
    }
    var tag = baseGetTag(value);
    return tag == errorTag$1 || tag == domExcTag || typeof value.message == "string" && typeof value.name == "string" && !isPlainObject(value);
}
var attempt = baseRest(function(func2, args) {
    try {
        return apply(func2, void 0, args);
    } catch (e) {
        return isError(e) ? e : new Error(e);
    }
});
var FUNC_ERROR_TEXT$3 = "Expected a function";
function before(n6, func2) {
    var result2;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$3);
    }
    n6 = toInteger(n6);
    return function() {
        if (--n6 > 0) {
            result2 = func2.apply(this, arguments);
        }
        if (n6 <= 1) {
            func2 = void 0;
        }
        return result2;
    };
}
var WRAP_BIND_FLAG$7 = 1, WRAP_PARTIAL_FLAG$3 = 32;
var bind = baseRest(function(func2, thisArg, partials) {
    var bitmask = WRAP_BIND_FLAG$7;
    if (partials.length) {
        var holders = replaceHolders(partials, getHolder(bind));
        bitmask |= WRAP_PARTIAL_FLAG$3;
    }
    return createWrap(func2, bitmask, thisArg, partials, holders);
});
bind.placeholder = {
};
var bindAll = flatRest(function(object2, methodNames1) {
    arrayEach(methodNames1, function(key) {
        key = toKey(key);
        baseAssignValue(object2, key, bind(object2[key], object2));
    });
    return object2;
});
var WRAP_BIND_FLAG$8 = 1, WRAP_BIND_KEY_FLAG$5 = 2, WRAP_PARTIAL_FLAG$4 = 32;
var bindKey = baseRest(function(object2, key, partials) {
    var bitmask = WRAP_BIND_FLAG$8 | WRAP_BIND_KEY_FLAG$5;
    if (partials.length) {
        var holders = replaceHolders(partials, getHolder(bindKey));
        bitmask |= WRAP_PARTIAL_FLAG$4;
    }
    return createWrap(key, bitmask, object2, partials, holders);
});
bindKey.placeholder = {
};
function baseSlice(array2, start, end) {
    var index = -1, length = array2.length;
    if (start < 0) {
        start = -start > length ? 0 : length + start;
    }
    end = end > length ? length : end;
    if (end < 0) {
        end += length;
    }
    length = start > end ? 0 : end - start >>> 0;
    start >>>= 0;
    var result2 = Array(length);
    while(++index < length){
        result2[index] = array2[index + start];
    }
    return result2;
}
function castSlice(array2, start, end) {
    var length = array2.length;
    end = end === void 0 ? length : end;
    return !start && end >= length ? array2 : baseSlice(array2, start, end);
}
var rsAstralRange = "\\ud800-\\udfff", rsComboMarksRange = "\\u0300-\\u036f", reComboHalfMarksRange = "\\ufe20-\\ufe2f", rsComboSymbolsRange = "\\u20d0-\\u20ff", rsComboRange = rsComboMarksRange + reComboHalfMarksRange + rsComboSymbolsRange, rsVarRange = "\\ufe0e\\ufe0f";
var rsZWJ = "\\u200d";
var reHasUnicode = RegExp("[" + rsZWJ + rsAstralRange + rsComboRange + rsVarRange + "]");
function hasUnicode(string2) {
    return reHasUnicode.test(string2);
}
function asciiToArray(string2) {
    return string2.split("");
}
var rsAstralRange$1 = "\\ud800-\\udfff", rsComboMarksRange$1 = "\\u0300-\\u036f", reComboHalfMarksRange$1 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$1 = "\\u20d0-\\u20ff", rsComboRange$1 = rsComboMarksRange$1 + reComboHalfMarksRange$1 + rsComboSymbolsRange$1, rsVarRange$1 = "\\ufe0e\\ufe0f";
var rsAstral = "[" + rsAstralRange$1 + "]", rsCombo = "[" + rsComboRange$1 + "]", rsFitz = "\\ud83c[\\udffb-\\udfff]", rsModifier = "(?:" + rsCombo + "|" + rsFitz + ")", rsNonAstral = "[^" + rsAstralRange$1 + "]", rsRegional = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsZWJ$1 = "\\u200d";
var reOptMod = rsModifier + "?", rsOptVar = "[" + rsVarRange$1 + "]?", rsOptJoin = "(?:" + rsZWJ$1 + "(?:" + [
    rsNonAstral,
    rsRegional,
    rsSurrPair
].join("|") + ")" + rsOptVar + reOptMod + ")*", rsSeq = rsOptVar + reOptMod + rsOptJoin, rsSymbol = "(?:" + [
    rsNonAstral + rsCombo + "?",
    rsCombo,
    rsRegional,
    rsSurrPair,
    rsAstral
].join("|") + ")";
var reUnicode = RegExp(rsFitz + "(?=" + rsFitz + ")|" + rsSymbol + rsSeq, "g");
function unicodeToArray(string2) {
    return string2.match(reUnicode) || [];
}
function stringToArray(string2) {
    return hasUnicode(string2) ? unicodeToArray(string2) : asciiToArray(string2);
}
function createCaseFirst(methodName) {
    return function(string2) {
        string2 = toString(string2);
        var strSymbols = hasUnicode(string2) ? stringToArray(string2) : void 0;
        var chr = strSymbols ? strSymbols[0] : string2.charAt(0);
        var trailing = strSymbols ? castSlice(strSymbols, 1).join("") : string2.slice(1);
        return chr[methodName]() + trailing;
    };
}
var upperFirst = createCaseFirst("toUpperCase");
function capitalize1(string2) {
    return upperFirst(toString(string2).toLowerCase());
}
function arrayReduce(array2, iteratee2, accumulator, initAccum) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    if (initAccum && length) {
        accumulator = array2[++index];
    }
    while(++index < length){
        accumulator = iteratee2(accumulator, array2[index], index, array2);
    }
    return accumulator;
}
function basePropertyOf(object2) {
    return function(key) {
        return object2 == null ? void 0 : object2[key];
    };
}
var deburredLetters = {
    À: "A",
    Á: "A",
    Â: "A",
    Ã: "A",
    Ä: "A",
    Å: "A",
    à: "a",
    á: "a",
    â: "a",
    ã: "a",
    ä: "a",
    å: "a",
    Ç: "C",
    ç: "c",
    Ð: "D",
    ð: "d",
    È: "E",
    É: "E",
    Ê: "E",
    Ë: "E",
    è: "e",
    é: "e",
    ê: "e",
    ë: "e",
    Ì: "I",
    Í: "I",
    Î: "I",
    Ï: "I",
    ì: "i",
    í: "i",
    î: "i",
    ï: "i",
    Ñ: "N",
    ñ: "n",
    Ò: "O",
    Ó: "O",
    Ô: "O",
    Õ: "O",
    Ö: "O",
    Ø: "O",
    ò: "o",
    ó: "o",
    ô: "o",
    õ: "o",
    ö: "o",
    ø: "o",
    Ù: "U",
    Ú: "U",
    Û: "U",
    Ü: "U",
    ù: "u",
    ú: "u",
    û: "u",
    ü: "u",
    Ý: "Y",
    ý: "y",
    ÿ: "y",
    Æ: "Ae",
    æ: "ae",
    Þ: "Th",
    þ: "th",
    ß: "ss",
    Ā: "A",
    Ă: "A",
    Ą: "A",
    ā: "a",
    ă: "a",
    ą: "a",
    Ć: "C",
    Ĉ: "C",
    Ċ: "C",
    Č: "C",
    ć: "c",
    ĉ: "c",
    ċ: "c",
    č: "c",
    Ď: "D",
    Đ: "D",
    ď: "d",
    đ: "d",
    Ē: "E",
    Ĕ: "E",
    Ė: "E",
    Ę: "E",
    Ě: "E",
    ē: "e",
    ĕ: "e",
    ė: "e",
    ę: "e",
    ě: "e",
    Ĝ: "G",
    Ğ: "G",
    Ġ: "G",
    Ģ: "G",
    ĝ: "g",
    ğ: "g",
    ġ: "g",
    ģ: "g",
    Ĥ: "H",
    Ħ: "H",
    ĥ: "h",
    ħ: "h",
    Ĩ: "I",
    Ī: "I",
    Ĭ: "I",
    Į: "I",
    İ: "I",
    ĩ: "i",
    ī: "i",
    ĭ: "i",
    į: "i",
    ı: "i",
    Ĵ: "J",
    ĵ: "j",
    Ķ: "K",
    ķ: "k",
    ĸ: "k",
    Ĺ: "L",
    Ļ: "L",
    Ľ: "L",
    Ŀ: "L",
    Ł: "L",
    ĺ: "l",
    ļ: "l",
    ľ: "l",
    ŀ: "l",
    ł: "l",
    Ń: "N",
    Ņ: "N",
    Ň: "N",
    Ŋ: "N",
    ń: "n",
    ņ: "n",
    ň: "n",
    ŋ: "n",
    Ō: "O",
    Ŏ: "O",
    Ő: "O",
    ō: "o",
    ŏ: "o",
    ő: "o",
    Ŕ: "R",
    Ŗ: "R",
    Ř: "R",
    ŕ: "r",
    ŗ: "r",
    ř: "r",
    Ś: "S",
    Ŝ: "S",
    Ş: "S",
    Š: "S",
    ś: "s",
    ŝ: "s",
    ş: "s",
    š: "s",
    Ţ: "T",
    Ť: "T",
    Ŧ: "T",
    ţ: "t",
    ť: "t",
    ŧ: "t",
    Ũ: "U",
    Ū: "U",
    Ŭ: "U",
    Ů: "U",
    Ű: "U",
    Ų: "U",
    ũ: "u",
    ū: "u",
    ŭ: "u",
    ů: "u",
    ű: "u",
    ų: "u",
    Ŵ: "W",
    ŵ: "w",
    Ŷ: "Y",
    ŷ: "y",
    Ÿ: "Y",
    Ź: "Z",
    Ż: "Z",
    Ž: "Z",
    ź: "z",
    ż: "z",
    ž: "z",
    Ĳ: "IJ",
    ĳ: "ij",
    Œ: "Oe",
    œ: "oe",
    ŉ: "'n",
    ſ: "s"
};
var deburrLetter = basePropertyOf(deburredLetters);
var reLatin = /[\xc0-\xd6\xd8-\xf6\xf8-\xff\u0100-\u017f]/g;
var rsComboMarksRange$2 = "\\u0300-\\u036f", reComboHalfMarksRange$2 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$2 = "\\u20d0-\\u20ff", rsComboRange$2 = rsComboMarksRange$2 + reComboHalfMarksRange$2 + rsComboSymbolsRange$2;
var rsCombo$1 = "[" + rsComboRange$2 + "]";
var reComboMark = RegExp(rsCombo$1, "g");
function deburr(string2) {
    string2 = toString(string2);
    return string2 && string2.replace(reLatin, deburrLetter).replace(reComboMark, "");
}
var reAsciiWord = /[^\x00-\x2f\x3a-\x40\x5b-\x60\x7b-\x7f]+/g;
function asciiWords(string2) {
    return string2.match(reAsciiWord) || [];
}
var reHasUnicodeWord = /[a-z][A-Z]|[A-Z]{2}[a-z]|[0-9][a-zA-Z]|[a-zA-Z][0-9]|[^a-zA-Z0-9 ]/;
function hasUnicodeWord(string2) {
    return reHasUnicodeWord.test(string2);
}
var rsAstralRange$2 = "\\ud800-\\udfff", rsComboMarksRange$3 = "\\u0300-\\u036f", reComboHalfMarksRange$3 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$3 = "\\u20d0-\\u20ff", rsComboRange$3 = rsComboMarksRange$3 + reComboHalfMarksRange$3 + rsComboSymbolsRange$3, rsDingbatRange = "\\u2700-\\u27bf", rsLowerRange = "a-z\\xdf-\\xf6\\xf8-\\xff", rsMathOpRange = "\\xac\\xb1\\xd7\\xf7", rsNonCharRange = "\\x00-\\x2f\\x3a-\\x40\\x5b-\\x60\\x7b-\\xbf", rsPunctuationRange = "\\u2000-\\u206f", rsSpaceRange = " \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000", rsUpperRange = "A-Z\\xc0-\\xd6\\xd8-\\xde", rsVarRange$2 = "\\ufe0e\\ufe0f", rsBreakRange = rsMathOpRange + rsNonCharRange + rsPunctuationRange + rsSpaceRange;
var rsApos = "['\u2019]", rsBreak = "[" + rsBreakRange + "]", rsCombo$2 = "[" + rsComboRange$3 + "]", rsDigits = "\\d+", rsDingbat = "[" + rsDingbatRange + "]", rsLower = "[" + rsLowerRange + "]", rsMisc = "[^" + rsAstralRange$2 + rsBreakRange + rsDigits + rsDingbatRange + rsLowerRange + rsUpperRange + "]", rsFitz$1 = "\\ud83c[\\udffb-\\udfff]", rsModifier$1 = "(?:" + rsCombo$2 + "|" + rsFitz$1 + ")", rsNonAstral$1 = "[^" + rsAstralRange$2 + "]", rsRegional$1 = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair$1 = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsUpper = "[" + rsUpperRange + "]", rsZWJ$2 = "\\u200d";
var rsMiscLower = "(?:" + rsLower + "|" + rsMisc + ")", rsMiscUpper = "(?:" + rsUpper + "|" + rsMisc + ")", rsOptContrLower = "(?:" + rsApos + "(?:d|ll|m|re|s|t|ve))?", rsOptContrUpper = "(?:" + rsApos + "(?:D|LL|M|RE|S|T|VE))?", reOptMod$1 = rsModifier$1 + "?", rsOptVar$1 = "[" + rsVarRange$2 + "]?", rsOptJoin$1 = "(?:" + rsZWJ$2 + "(?:" + [
    rsNonAstral$1,
    rsRegional$1,
    rsSurrPair$1
].join("|") + ")" + rsOptVar$1 + reOptMod$1 + ")*", rsOrdLower = "\\d*(?:1st|2nd|3rd|(?![123])\\dth)(?=\\b|[A-Z_])", rsOrdUpper = "\\d*(?:1ST|2ND|3RD|(?![123])\\dTH)(?=\\b|[a-z_])", rsSeq$1 = rsOptVar$1 + reOptMod$1 + rsOptJoin$1, rsEmoji = "(?:" + [
    rsDingbat,
    rsRegional$1,
    rsSurrPair$1
].join("|") + ")" + rsSeq$1;
var reUnicodeWord = RegExp([
    rsUpper + "?" + rsLower + "+" + rsOptContrLower + "(?=" + [
        rsBreak,
        rsUpper,
        "$"
    ].join("|") + ")",
    rsMiscUpper + "+" + rsOptContrUpper + "(?=" + [
        rsBreak,
        rsUpper + rsMiscLower,
        "$"
    ].join("|") + ")",
    rsUpper + "?" + rsMiscLower + "+" + rsOptContrLower,
    rsUpper + "+" + rsOptContrUpper,
    rsOrdUpper,
    rsOrdLower,
    rsDigits,
    rsEmoji
].join("|"), "g");
function unicodeWords(string2) {
    return string2.match(reUnicodeWord) || [];
}
function words(string2, pattern4, guard) {
    string2 = toString(string2);
    pattern4 = guard ? void 0 : pattern4;
    if (pattern4 === void 0) {
        return hasUnicodeWord(string2) ? unicodeWords(string2) : asciiWords(string2);
    }
    return string2.match(pattern4) || [];
}
var rsApos$1 = "['\u2019]";
var reApos = RegExp(rsApos$1, "g");
function createCompounder(callback) {
    return function(string2) {
        return arrayReduce(words(deburr(string2).replace(reApos, "")), callback, "");
    };
}
var camelCase = createCompounder(function(result2, word, index) {
    word = word.toLowerCase();
    return result2 + (index ? capitalize1(word) : word);
});
function castArray() {
    if (!arguments.length) {
        return [];
    }
    var value = arguments[0];
    return isArray(value) ? value : [
        value
    ];
}
var nativeIsFinite = root.isFinite, nativeMin$2 = Math.min;
function createRound(methodName) {
    var func2 = Math[methodName];
    return function(number2, precision) {
        number2 = toNumber(number2);
        precision = precision == null ? 0 : nativeMin$2(toInteger(precision), 292);
        if (precision && nativeIsFinite(number2)) {
            var pair = (toString(number2) + "e").split("e"), value = func2(pair[0] + "e" + (+pair[1] + precision));
            pair = (toString(value) + "e").split("e");
            return +(pair[0] + "e" + (+pair[1] - precision));
        }
        return func2(number2);
    };
}
var ceil = createRound("ceil");
function chain(value) {
    var result2 = lodash(value);
    result2.__chain__ = true;
    return result2;
}
var nativeCeil = Math.ceil, nativeMax$4 = Math.max;
function chunk(array2, size2, guard) {
    if (guard ? isIterateeCall(array2, size2, guard) : size2 === void 0) {
        size2 = 1;
    } else {
        size2 = nativeMax$4(toInteger(size2), 0);
    }
    var length = array2 == null ? 0 : array2.length;
    if (!length || size2 < 1) {
        return [];
    }
    var index = 0, resIndex = 0, result2 = Array(nativeCeil(length / size2));
    while(index < length){
        result2[resIndex++] = baseSlice(array2, index, index += size2);
    }
    return result2;
}
function baseClamp(number2, lower, upper) {
    if (number2 === number2) {
        if (upper !== void 0) {
            number2 = number2 <= upper ? number2 : upper;
        }
        if (lower !== void 0) {
            number2 = number2 >= lower ? number2 : lower;
        }
    }
    return number2;
}
function clamp(number2, lower, upper) {
    if (upper === void 0) {
        upper = lower;
        lower = void 0;
    }
    if (upper !== void 0) {
        upper = toNumber(upper);
        upper = upper === upper ? upper : 0;
    }
    if (lower !== void 0) {
        lower = toNumber(lower);
        lower = lower === lower ? lower : 0;
    }
    return baseClamp(toNumber(number2), lower, upper);
}
function stackClear() {
    this.__data__ = new ListCache();
    this.size = 0;
}
function stackDelete(key) {
    var data17 = this.__data__, result2 = data17["delete"](key);
    this.size = data17.size;
    return result2;
}
function stackGet(key) {
    return this.__data__.get(key);
}
function stackHas(key) {
    return this.__data__.has(key);
}
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value) {
    var data18 = this.__data__;
    if (data18 instanceof ListCache) {
        var pairs1 = data18.__data__;
        if (!Map1 || pairs1.length < LARGE_ARRAY_SIZE - 1) {
            pairs1.push([
                key,
                value
            ]);
            this.size = ++data18.size;
            return this;
        }
        data18 = this.__data__ = new MapCache(pairs1);
    }
    data18.set(key, value);
    this.size = data18.size;
    return this;
}
function Stack(entries) {
    var data19 = this.__data__ = new ListCache(entries);
    this.size = data19.size;
}
Stack.prototype.clear = stackClear;
Stack.prototype["delete"] = stackDelete;
Stack.prototype.get = stackGet;
Stack.prototype.has = stackHas;
Stack.prototype.set = stackSet;
function baseAssign(object2, source) {
    return object2 && copyObject(source, keys(source), object2);
}
function baseAssignIn(object2, source) {
    return object2 && copyObject(source, keysIn(source), object2);
}
var freeExports$2 = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule$2 = freeExports$2 && typeof module == "object" && module && !module.nodeType && module;
var moduleExports$2 = freeModule$2 && freeModule$2.exports === freeExports$2;
var Buffer$1 = moduleExports$2 ? root.Buffer : void 0, allocUnsafe = Buffer$1 ? Buffer$1.allocUnsafe : void 0;
function cloneBuffer(buffer, isDeep) {
    if (isDeep) {
        return buffer.slice();
    }
    var length = buffer.length, result2 = allocUnsafe ? allocUnsafe(length) : new buffer.constructor(length);
    buffer.copy(result2);
    return result2;
}
function arrayFilter(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value = array2[index];
        if (predicate(value, index, array2)) {
            result2[resIndex++] = value;
        }
    }
    return result2;
}
function stubArray() {
    return [];
}
var objectProto$f = Object.prototype;
var propertyIsEnumerable$1 = objectProto$f.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? stubArray : function(object2) {
    if (object2 == null) {
        return [];
    }
    object2 = Object(object2);
    return arrayFilter(nativeGetSymbols(object2), function(symbol) {
        return propertyIsEnumerable$1.call(object2, symbol);
    });
};
function copySymbols(source, object2) {
    return copyObject(source, getSymbols(source), object2);
}
var nativeGetSymbols$1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols$1 ? stubArray : function(object2) {
    var result2 = [];
    while(object2){
        arrayPush(result2, getSymbols(object2));
        object2 = getPrototype(object2);
    }
    return result2;
};
function copySymbolsIn(source, object2) {
    return copyObject(source, getSymbolsIn(source), object2);
}
function baseGetAllKeys(object2, keysFunc, symbolsFunc) {
    var result2 = keysFunc(object2);
    return isArray(object2) ? result2 : arrayPush(result2, symbolsFunc(object2));
}
function getAllKeys(object2) {
    return baseGetAllKeys(object2, keys, getSymbols);
}
function getAllKeysIn(object2) {
    return baseGetAllKeys(object2, keysIn, getSymbolsIn);
}
var DataView1 = getNative(root, "DataView");
var Promise1 = getNative(root, "Promise");
var Set1 = getNative(root, "Set");
var mapTag$1 = "[object Map]", objectTag$2 = "[object Object]", promiseTag = "[object Promise]", setTag$1 = "[object Set]", weakMapTag$1 = "[object WeakMap]";
var dataViewTag$1 = "[object DataView]";
var dataViewCtorString = toSource(DataView1), mapCtorString = toSource(Map1), promiseCtorString = toSource(Promise1), setCtorString = toSource(Set1), weakMapCtorString = toSource(WeakMap);
var getTag = baseGetTag;
if (DataView1 && getTag(new DataView1(new ArrayBuffer(1))) != dataViewTag$1 || Map1 && getTag(new Map1()) != mapTag$1 || Promise1 && getTag(Promise1.resolve()) != promiseTag || Set1 && getTag(new Set1()) != setTag$1 || WeakMap && getTag(new WeakMap()) != weakMapTag$1) {
    getTag = function(value) {
        var result2 = baseGetTag(value), Ctor = result2 == objectTag$2 ? value.constructor : void 0, ctorString = Ctor ? toSource(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag$1;
                case mapCtorString:
                    return mapTag$1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag$1;
                case weakMapCtorString:
                    return weakMapTag$1;
            }
        }
        return result2;
    };
}
var getTag$1 = getTag;
var objectProto$g = Object.prototype;
var hasOwnProperty$d = objectProto$g.hasOwnProperty;
function initCloneArray(array2) {
    var length = array2.length, result2 = new array2.constructor(length);
    if (length && typeof array2[0] == "string" && hasOwnProperty$d.call(array2, "index")) {
        result2.index = array2.index;
        result2.input = array2.input;
    }
    return result2;
}
var Uint8Array1 = root.Uint8Array;
function cloneArrayBuffer(arrayBuffer) {
    var result2 = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new Uint8Array1(result2).set(new Uint8Array1(arrayBuffer));
    return result2;
}
function cloneDataView(dataView, isDeep) {
    var buffer = isDeep ? cloneArrayBuffer(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}
var reFlags = /\w*$/;
function cloneRegExp(regexp2) {
    var result2 = new regexp2.constructor(regexp2.source, reFlags.exec(regexp2));
    result2.lastIndex = regexp2.lastIndex;
    return result2;
}
var symbolProto$1 = Symbol1 ? Symbol1.prototype : void 0, symbolValueOf = symbolProto$1 ? symbolProto$1.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {
    };
}
function cloneTypedArray(typedArray, isDeep) {
    var buffer = isDeep ? cloneArrayBuffer(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}
var boolTag$1 = "[object Boolean]", dateTag$1 = "[object Date]", mapTag$2 = "[object Map]", numberTag$1 = "[object Number]", regexpTag$1 = "[object RegExp]", setTag$2 = "[object Set]", stringTag$1 = "[object String]", symbolTag$1 = "[object Symbol]";
var arrayBufferTag$1 = "[object ArrayBuffer]", dataViewTag$2 = "[object DataView]", float32Tag$1 = "[object Float32Array]", float64Tag$1 = "[object Float64Array]", int8Tag$1 = "[object Int8Array]", int16Tag$1 = "[object Int16Array]", int32Tag$1 = "[object Int32Array]", uint8Tag$1 = "[object Uint8Array]", uint8ClampedTag$1 = "[object Uint8ClampedArray]", uint16Tag$1 = "[object Uint16Array]", uint32Tag$1 = "[object Uint32Array]";
function initCloneByTag(object2, tag, isDeep) {
    var Ctor = object2.constructor;
    switch(tag){
        case arrayBufferTag$1:
            return cloneArrayBuffer(object2);
        case boolTag$1:
        case dateTag$1:
            return new Ctor(+object2);
        case dataViewTag$2:
            return cloneDataView(object2, isDeep);
        case float32Tag$1:
        case float64Tag$1:
        case int8Tag$1:
        case int16Tag$1:
        case int32Tag$1:
        case uint8Tag$1:
        case uint8ClampedTag$1:
        case uint16Tag$1:
        case uint32Tag$1:
            return cloneTypedArray(object2, isDeep);
        case mapTag$2:
            return new Ctor();
        case numberTag$1:
        case stringTag$1:
            return new Ctor(object2);
        case regexpTag$1:
            return cloneRegExp(object2);
        case setTag$2:
            return new Ctor();
        case symbolTag$1:
            return cloneSymbol(object2);
    }
}
function initCloneObject(object2) {
    return typeof object2.constructor == "function" && !isPrototype(object2) ? baseCreate(getPrototype(object2)) : {
    };
}
var mapTag$3 = "[object Map]";
function baseIsMap(value) {
    return isObjectLike(value) && getTag$1(value) == mapTag$3;
}
var nodeIsMap = nodeUtil && nodeUtil.isMap;
var isMap = nodeIsMap ? baseUnary(nodeIsMap) : baseIsMap;
var setTag$3 = "[object Set]";
function baseIsSet(value) {
    return isObjectLike(value) && getTag$1(value) == setTag$3;
}
var nodeIsSet = nodeUtil && nodeUtil.isSet;
var isSet = nodeIsSet ? baseUnary(nodeIsSet) : baseIsSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag$2 = "[object Arguments]", arrayTag$1 = "[object Array]", boolTag$2 = "[object Boolean]", dateTag$2 = "[object Date]", errorTag$2 = "[object Error]", funcTag$2 = "[object Function]", genTag$1 = "[object GeneratorFunction]", mapTag$4 = "[object Map]", numberTag$2 = "[object Number]", objectTag$3 = "[object Object]", regexpTag$2 = "[object RegExp]", setTag$4 = "[object Set]", stringTag$2 = "[object String]", symbolTag$2 = "[object Symbol]", weakMapTag$2 = "[object WeakMap]";
var arrayBufferTag$2 = "[object ArrayBuffer]", dataViewTag$3 = "[object DataView]", float32Tag$2 = "[object Float32Array]", float64Tag$2 = "[object Float64Array]", int8Tag$2 = "[object Int8Array]", int16Tag$2 = "[object Int16Array]", int32Tag$2 = "[object Int32Array]", uint8Tag$2 = "[object Uint8Array]", uint8ClampedTag$2 = "[object Uint8ClampedArray]", uint16Tag$2 = "[object Uint16Array]", uint32Tag$2 = "[object Uint32Array]";
var cloneableTags = {
};
cloneableTags[argsTag$2] = cloneableTags[arrayTag$1] = cloneableTags[arrayBufferTag$2] = cloneableTags[dataViewTag$3] = cloneableTags[boolTag$2] = cloneableTags[dateTag$2] = cloneableTags[float32Tag$2] = cloneableTags[float64Tag$2] = cloneableTags[int8Tag$2] = cloneableTags[int16Tag$2] = cloneableTags[int32Tag$2] = cloneableTags[mapTag$4] = cloneableTags[numberTag$2] = cloneableTags[objectTag$3] = cloneableTags[regexpTag$2] = cloneableTags[setTag$4] = cloneableTags[stringTag$2] = cloneableTags[symbolTag$2] = cloneableTags[uint8Tag$2] = cloneableTags[uint8ClampedTag$2] = cloneableTags[uint16Tag$2] = cloneableTags[uint32Tag$2] = true;
cloneableTags[errorTag$2] = cloneableTags[funcTag$2] = cloneableTags[weakMapTag$2] = false;
function baseClone(value, bitmask, customizer, key, object2, stack) {
    var result2, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result2 = object2 ? customizer(value, key, object2, stack) : customizer(value);
    }
    if (result2 !== void 0) {
        return result2;
    }
    if (!isObject(value)) {
        return value;
    }
    var isArr = isArray(value);
    if (isArr) {
        result2 = initCloneArray(value);
        if (!isDeep) {
            return copyArray(value, result2);
        }
    } else {
        var tag = getTag$1(value), isFunc = tag == funcTag$2 || tag == genTag$1;
        if (isBuffer(value)) {
            return cloneBuffer(value, isDeep);
        }
        if (tag == objectTag$3 || tag == argsTag$2 || isFunc && !object2) {
            result2 = isFlat || isFunc ? {
            } : initCloneObject(value);
            if (!isDeep) {
                return isFlat ? copySymbolsIn(value, baseAssignIn(result2, value)) : copySymbols(value, baseAssign(result2, value));
            }
        } else {
            if (!cloneableTags[tag]) {
                return object2 ? value : {
                };
            }
            result2 = initCloneByTag(value, tag, isDeep);
        }
    }
    stack || (stack = new Stack());
    var stacked = stack.get(value);
    if (stacked) {
        return stacked;
    }
    stack.set(value, result2);
    if (isSet(value)) {
        value.forEach(function(subValue) {
            result2.add(baseClone(subValue, bitmask, customizer, subValue, value, stack));
        });
    } else if (isMap(value)) {
        value.forEach(function(subValue, key2) {
            result2.set(key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? getAllKeysIn : getAllKeys : isFlat ? keysIn : keys;
    var props = isArr ? void 0 : keysFunc(value);
    arrayEach(props || value, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value[key2];
        }
        assignValue(result2, key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
    });
    return result2;
}
var CLONE_SYMBOLS_FLAG$1 = 4;
function clone(value) {
    return baseClone(value, CLONE_SYMBOLS_FLAG$1);
}
var CLONE_DEEP_FLAG$1 = 1, CLONE_SYMBOLS_FLAG$2 = 4;
function cloneDeep(value) {
    return baseClone(value, CLONE_DEEP_FLAG$1 | CLONE_SYMBOLS_FLAG$2);
}
var CLONE_DEEP_FLAG$2 = 1, CLONE_SYMBOLS_FLAG$3 = 4;
function cloneDeepWith(value, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseClone(value, CLONE_DEEP_FLAG$2 | CLONE_SYMBOLS_FLAG$3, customizer);
}
var CLONE_SYMBOLS_FLAG$4 = 4;
function cloneWith(value, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseClone(value, CLONE_SYMBOLS_FLAG$4, customizer);
}
function wrapperCommit() {
    return new LodashWrapper(this.value(), this.__chain__);
}
function compact(array2) {
    var index = -1, length = array2 == null ? 0 : array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value = array2[index];
        if (value) {
            result2[resIndex++] = value;
        }
    }
    return result2;
}
function concat1() {
    var length = arguments.length;
    if (!length) {
        return [];
    }
    var args = Array(length - 1), array2 = arguments[0], index = length;
    while(index--){
        args[index - 1] = arguments[index];
    }
    return arrayPush(isArray(array2) ? copyArray(array2) : [
        array2
    ], baseFlatten(args, 1));
}
var HASH_UNDEFINED$2 = "__lodash_hash_undefined__";
function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED$2);
    return this;
}
function setCacheHas(value) {
    return this.__data__.has(value);
}
function SetCache(values2) {
    var index = -1, length = values2 == null ? 0 : values2.length;
    this.__data__ = new MapCache();
    while(++index < length){
        this.add(values2[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = setCacheAdd;
SetCache.prototype.has = setCacheHas;
function arraySome(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (predicate(array2[index], index, array2)) {
            return true;
        }
    }
    return false;
}
function cacheHas(cache, key) {
    return cache.has(key);
}
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array2, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array2.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array2);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array2;
    }
    var index = -1, result2 = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new SetCache() : void 0;
    stack.set(array2, other);
    stack.set(other, array2);
    while(++index < arrLength){
        var arrValue = array2[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array2, stack) : customizer(arrValue, othValue, index, array2, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result2 = false;
            break;
        }
        if (seen) {
            if (!arraySome(other, function(othValue2, othIndex) {
                if (!cacheHas(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result2 = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result2 = false;
            break;
        }
    }
    stack["delete"](array2);
    stack["delete"](other);
    return result2;
}
function mapToArray(map2) {
    var index = -1, result2 = Array(map2.size);
    map2.forEach(function(value, key) {
        result2[++index] = [
            key,
            value
        ];
    });
    return result2;
}
function setToArray(set2) {
    var index = -1, result2 = Array(set2.size);
    set2.forEach(function(value) {
        result2[++index] = value;
    });
    return result2;
}
var COMPARE_PARTIAL_FLAG$1 = 1, COMPARE_UNORDERED_FLAG$1 = 2;
var boolTag$3 = "[object Boolean]", dateTag$3 = "[object Date]", errorTag$3 = "[object Error]", mapTag$5 = "[object Map]", numberTag$3 = "[object Number]", regexpTag$3 = "[object RegExp]", setTag$5 = "[object Set]", stringTag$3 = "[object String]", symbolTag$3 = "[object Symbol]";
var arrayBufferTag$3 = "[object ArrayBuffer]", dataViewTag$4 = "[object DataView]";
var symbolProto$2 = Symbol1 ? Symbol1.prototype : void 0, symbolValueOf$1 = symbolProto$2 ? symbolProto$2.valueOf : void 0;
function equalByTag(object2, other, tag, bitmask, customizer, equalFunc, stack) {
    switch(tag){
        case dataViewTag$4:
            if (object2.byteLength != other.byteLength || object2.byteOffset != other.byteOffset) {
                return false;
            }
            object2 = object2.buffer;
            other = other.buffer;
        case arrayBufferTag$3:
            if (object2.byteLength != other.byteLength || !equalFunc(new Uint8Array1(object2), new Uint8Array1(other))) {
                return false;
            }
            return true;
        case boolTag$3:
        case dateTag$3:
        case numberTag$3:
            return eq(+object2, +other);
        case errorTag$3:
            return object2.name == other.name && object2.message == other.message;
        case regexpTag$3:
        case stringTag$3:
            return object2 == other + "";
        case mapTag$5:
            var convert = mapToArray;
        case setTag$5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG$1;
            convert || (convert = setToArray);
            if (object2.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object2);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG$1;
            stack.set(object2, other);
            var result2 = equalArrays(convert(object2), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object2);
            return result2;
        case symbolTag$3:
            if (symbolValueOf$1) {
                return symbolValueOf$1.call(object2) == symbolValueOf$1.call(other);
            }
    }
    return false;
}
var COMPARE_PARTIAL_FLAG$2 = 1;
var objectProto$h = Object.prototype;
var hasOwnProperty$e = objectProto$h.hasOwnProperty;
function equalObjects(object2, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG$2, objProps = getAllKeys(object2), objLength = objProps.length, othProps = getAllKeys(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty$e.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object2);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object2;
    }
    var result2 = true;
    stack.set(object2, other);
    stack.set(other, object2);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object2[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object2, stack) : customizer(objValue, othValue, key, object2, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result2 = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result2 && !skipCtor) {
        var objCtor = object2.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object2 && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result2 = false;
        }
    }
    stack["delete"](object2);
    stack["delete"](other);
    return result2;
}
var COMPARE_PARTIAL_FLAG$3 = 1;
var argsTag$3 = "[object Arguments]", arrayTag$2 = "[object Array]", objectTag$4 = "[object Object]";
var objectProto$i = Object.prototype;
var hasOwnProperty$f = objectProto$i.hasOwnProperty;
function baseIsEqualDeep(object2, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = isArray(object2), othIsArr = isArray(other), objTag = objIsArr ? arrayTag$2 : getTag$1(object2), othTag = othIsArr ? arrayTag$2 : getTag$1(other);
    objTag = objTag == argsTag$3 ? objectTag$4 : objTag;
    othTag = othTag == argsTag$3 ? objectTag$4 : othTag;
    var objIsObj = objTag == objectTag$4, othIsObj = othTag == objectTag$4, isSameTag = objTag == othTag;
    if (isSameTag && isBuffer(object2)) {
        if (!isBuffer(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new Stack());
        return objIsArr || isTypedArray(object2) ? equalArrays(object2, other, bitmask, customizer, equalFunc, stack) : equalByTag(object2, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG$3)) {
        var objIsWrapped = objIsObj && hasOwnProperty$f.call(object2, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty$f.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object2.value() : object2, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new Stack());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new Stack());
    return equalObjects(object2, other, bitmask, customizer, equalFunc, stack);
}
function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
        return true;
    }
    if (value == null || other == null || !isObjectLike(value) && !isObjectLike(other)) {
        return value !== value && other !== other;
    }
    return baseIsEqualDeep(value, other, bitmask, customizer, baseIsEqual, stack);
}
var COMPARE_PARTIAL_FLAG$4 = 1, COMPARE_UNORDERED_FLAG$2 = 2;
function baseIsMatch(object2, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object2 == null) {
        return !length;
    }
    object2 = Object(object2);
    while(index--){
        var data20 = matchData[index];
        if (noCustomizer && data20[2] ? data20[1] !== object2[data20[0]] : !(data20[0] in object2)) {
            return false;
        }
    }
    while(++index < length){
        data20 = matchData[index];
        var key = data20[0], objValue = object2[key], srcValue = data20[1];
        if (noCustomizer && data20[2]) {
            if (objValue === void 0 && !(key in object2)) {
                return false;
            }
        } else {
            var stack = new Stack();
            if (customizer) {
                var result2 = customizer(objValue, srcValue, key, object2, source, stack);
            }
            if (!(result2 === void 0 ? baseIsEqual(srcValue, objValue, COMPARE_PARTIAL_FLAG$4 | COMPARE_UNORDERED_FLAG$2, customizer, stack) : result2)) {
                return false;
            }
        }
    }
    return true;
}
function isStrictComparable(value) {
    return value === value && !isObject(value);
}
function getMatchData(object2) {
    var result2 = keys(object2), length = result2.length;
    while(length--){
        var key = result2[length], value = object2[key];
        result2[length] = [
            key,
            value,
            isStrictComparable(value)
        ];
    }
    return result2;
}
function matchesStrictComparable(key, srcValue) {
    return function(object2) {
        if (object2 == null) {
            return false;
        }
        return object2[key] === srcValue && (srcValue !== void 0 || key in Object(object2));
    };
}
function baseMatches(source) {
    var matchData = getMatchData(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return matchesStrictComparable(matchData[0][0], matchData[0][1]);
    }
    return function(object2) {
        return object2 === source || baseIsMatch(object2, source, matchData);
    };
}
function baseHasIn(object2, key) {
    return object2 != null && key in Object(object2);
}
function hasPath(object2, path51, hasFunc) {
    path51 = castPath(path51, object2);
    var index = -1, length = path51.length, result2 = false;
    while(++index < length){
        var key = toKey(path51[index]);
        if (!(result2 = object2 != null && hasFunc(object2, key))) {
            break;
        }
        object2 = object2[key];
    }
    if (result2 || ++index != length) {
        return result2;
    }
    length = object2 == null ? 0 : object2.length;
    return !!length && isLength(length) && isIndex(key, length) && (isArray(object2) || isArguments(object2));
}
function hasIn(object2, path52) {
    return object2 != null && hasPath(object2, path52, baseHasIn);
}
var COMPARE_PARTIAL_FLAG$5 = 1, COMPARE_UNORDERED_FLAG$3 = 2;
function baseMatchesProperty(path53, srcValue) {
    if (isKey(path53) && isStrictComparable(srcValue)) {
        return matchesStrictComparable(toKey(path53), srcValue);
    }
    return function(object2) {
        var objValue = get(object2, path53);
        return objValue === void 0 && objValue === srcValue ? hasIn(object2, path53) : baseIsEqual(srcValue, objValue, COMPARE_PARTIAL_FLAG$5 | COMPARE_UNORDERED_FLAG$3);
    };
}
function baseProperty(key) {
    return function(object2) {
        return object2 == null ? void 0 : object2[key];
    };
}
function basePropertyDeep(path54) {
    return function(object2) {
        return baseGet(object2, path54);
    };
}
function property(path55) {
    return isKey(path55) ? baseProperty(toKey(path55)) : basePropertyDeep(path55);
}
function baseIteratee(value) {
    if (typeof value == "function") {
        return value;
    }
    if (value == null) {
        return identity;
    }
    if (typeof value == "object") {
        return isArray(value) ? baseMatchesProperty(value[0], value[1]) : baseMatches(value);
    }
    return property(value);
}
var FUNC_ERROR_TEXT$4 = "Expected a function";
function cond(pairs2) {
    var length = pairs2 == null ? 0 : pairs2.length, toIteratee = baseIteratee;
    pairs2 = !length ? [] : arrayMap(pairs2, function(pair) {
        if (typeof pair[1] != "function") {
            throw new TypeError(FUNC_ERROR_TEXT$4);
        }
        return [
            toIteratee(pair[0]),
            pair[1]
        ];
    });
    return baseRest(function(args) {
        var index = -1;
        while(++index < length){
            var pair = pairs2[index];
            if (apply(pair[0], this, args)) {
                return apply(pair[1], this, args);
            }
        }
    });
}
function baseConformsTo(object2, source, props) {
    var length = props.length;
    if (object2 == null) {
        return !length;
    }
    object2 = Object(object2);
    while(length--){
        var key = props[length], predicate = source[key], value = object2[key];
        if (value === void 0 && !(key in object2) || !predicate(value)) {
            return false;
        }
    }
    return true;
}
function baseConforms(source) {
    var props = keys(source);
    return function(object2) {
        return baseConformsTo(object2, source, props);
    };
}
var CLONE_DEEP_FLAG$3 = 1;
function conforms(source) {
    return baseConforms(baseClone(source, CLONE_DEEP_FLAG$3));
}
function conformsTo(object2, source) {
    return source == null || baseConformsTo(object2, source, keys(source));
}
function arrayAggregator(array2, setter, iteratee2, accumulator) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        var value = array2[index];
        setter(accumulator, value, iteratee2(value), array2);
    }
    return accumulator;
}
function createBaseFor(fromRight) {
    return function(object2, iteratee2, keysFunc) {
        var index = -1, iterable = Object(object2), props = keysFunc(object2), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee2(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object2;
    };
}
var baseFor = createBaseFor();
function baseForOwn(object2, iteratee2) {
    return object2 && baseFor(object2, iteratee2, keys);
}
function createBaseEach(eachFunc, fromRight) {
    return function(collection2, iteratee2) {
        if (collection2 == null) {
            return collection2;
        }
        if (!isArrayLike(collection2)) {
            return eachFunc(collection2, iteratee2);
        }
        var length = collection2.length, index = fromRight ? length : -1, iterable = Object(collection2);
        while(fromRight ? index-- : ++index < length){
            if (iteratee2(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection2;
    };
}
var baseEach = createBaseEach(baseForOwn);
function baseAggregator(collection2, setter, iteratee2, accumulator) {
    baseEach(collection2, function(value, key, collection3) {
        setter(accumulator, value, iteratee2(value), collection3);
    });
    return accumulator;
}
function createAggregator(setter, initializer) {
    return function(collection2, iteratee2) {
        var func2 = isArray(collection2) ? arrayAggregator : baseAggregator, accumulator = initializer ? initializer() : {
        };
        return func2(collection2, setter, baseIteratee(iteratee2), accumulator);
    };
}
var objectProto$j = Object.prototype;
var hasOwnProperty$g = objectProto$j.hasOwnProperty;
var countBy = createAggregator(function(result2, value, key) {
    if (hasOwnProperty$g.call(result2, key)) {
        ++result2[key];
    } else {
        baseAssignValue(result2, key, 1);
    }
});
function create(prototype, properties1) {
    var result2 = baseCreate(prototype);
    return properties1 == null ? result2 : baseAssign(result2, properties1);
}
var WRAP_CURRY_FLAG$5 = 8;
function curry(func2, arity, guard) {
    arity = guard ? void 0 : arity;
    var result2 = createWrap(func2, WRAP_CURRY_FLAG$5, void 0, void 0, void 0, void 0, void 0, arity);
    result2.placeholder = curry.placeholder;
    return result2;
}
curry.placeholder = {
};
var WRAP_CURRY_RIGHT_FLAG$3 = 16;
function curryRight(func2, arity, guard) {
    arity = guard ? void 0 : arity;
    var result2 = createWrap(func2, WRAP_CURRY_RIGHT_FLAG$3, void 0, void 0, void 0, void 0, void 0, arity);
    result2.placeholder = curryRight.placeholder;
    return result2;
}
curryRight.placeholder = {
};
var now = function() {
    return root.Date.now();
};
var FUNC_ERROR_TEXT$5 = "Expected a function";
var nativeMax$5 = Math.max, nativeMin$3 = Math.min;
function debounce(func2, wait, options) {
    var lastArgs, lastThis, maxWait, result2, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$5);
    }
    wait = toNumber(wait) || 0;
    if (isObject(options)) {
        leading = !!options.leading;
        maxing = "maxWait" in options;
        maxWait = maxing ? nativeMax$5(toNumber(options.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    function invokeFunc(time) {
        var args = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result2 = func2.apply(thisArg, args);
        return result2;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result2;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin$3(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = now();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result2;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result2 : trailingEdge(now());
    }
    function debounced() {
        var time = now(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result2;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
function defaultTo(value, defaultValue) {
    return value == null || value !== value ? defaultValue : value;
}
var objectProto$k = Object.prototype;
var hasOwnProperty$h = objectProto$k.hasOwnProperty;
var defaults = baseRest(function(object2, sources) {
    object2 = Object(object2);
    var index = -1;
    var length = sources.length;
    var guard = length > 2 ? sources[2] : void 0;
    if (guard && isIterateeCall(sources[0], sources[1], guard)) {
        length = 1;
    }
    while(++index < length){
        var source = sources[index];
        var props = keysIn(source);
        var propsIndex = -1;
        var propsLength = props.length;
        while(++propsIndex < propsLength){
            var key = props[propsIndex];
            var value = object2[key];
            if (value === void 0 || eq(value, objectProto$k[key]) && !hasOwnProperty$h.call(object2, key)) {
                object2[key] = source[key];
            }
        }
    }
    return object2;
});
function assignMergeValue(object2, key, value) {
    if (value !== void 0 && !eq(object2[key], value) || value === void 0 && !(key in object2)) {
        baseAssignValue(object2, key, value);
    }
}
function isArrayLikeObject(value) {
    return isObjectLike(value) && isArrayLike(value);
}
function safeGet(object2, key) {
    if (key === "constructor" && typeof object2[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object2[key];
}
function toPlainObject(value) {
    return copyObject(value, keysIn(value));
}
function baseMergeDeep(object2, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = safeGet(object2, key), srcValue = safeGet(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        assignMergeValue(object2, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object2, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = isArray(srcValue), isBuff = !isArr && isBuffer(srcValue), isTyped = !isArr && !isBuff && isTypedArray(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (isArray(objValue)) {
                newValue = objValue;
            } else if (isArrayLikeObject(objValue)) {
                newValue = copyArray(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = cloneBuffer(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = cloneTypedArray(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (isPlainObject(srcValue) || isArguments(srcValue)) {
            newValue = objValue;
            if (isArguments(objValue)) {
                newValue = toPlainObject(objValue);
            } else if (!isObject(objValue) || isFunction(objValue)) {
                newValue = initCloneObject(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    assignMergeValue(object2, key, newValue);
}
function baseMerge(object2, source, srcIndex, customizer, stack) {
    if (object2 === source) {
        return;
    }
    baseFor(source, function(srcValue, key) {
        stack || (stack = new Stack());
        if (isObject(srcValue)) {
            baseMergeDeep(object2, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(safeGet(object2, key), srcValue, key + "", object2, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            assignMergeValue(object2, key, newValue);
        }
    }, keysIn);
}
function customDefaultsMerge(objValue, srcValue, key, object2, source, stack) {
    if (isObject(objValue) && isObject(srcValue)) {
        stack.set(srcValue, objValue);
        baseMerge(objValue, srcValue, void 0, customDefaultsMerge, stack);
        stack["delete"](srcValue);
    }
    return objValue;
}
var mergeWith = createAssigner(function(object2, source, srcIndex, customizer) {
    baseMerge(object2, source, srcIndex, customizer);
});
var defaultsDeep = baseRest(function(args) {
    args.push(void 0, customDefaultsMerge);
    return apply(mergeWith, void 0, args);
});
var FUNC_ERROR_TEXT$6 = "Expected a function";
function baseDelay(func2, wait, args) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$6);
    }
    return setTimeout(function() {
        func2.apply(void 0, args);
    }, wait);
}
var defer = baseRest(function(func2, args) {
    return baseDelay(func2, 1, args);
});
var delay = baseRest(function(func2, wait, args) {
    return baseDelay(func2, toNumber(wait) || 0, args);
});
function arrayIncludesWith(array2, value, comparator) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (comparator(value, array2[index])) {
            return true;
        }
    }
    return false;
}
var LARGE_ARRAY_SIZE$1 = 200;
function baseDifference(array2, values2, iteratee2, comparator) {
    var index = -1, includes2 = arrayIncludes, isCommon = true, length = array2.length, result2 = [], valuesLength = values2.length;
    if (!length) {
        return result2;
    }
    if (iteratee2) {
        values2 = arrayMap(values2, baseUnary(iteratee2));
    }
    if (comparator) {
        includes2 = arrayIncludesWith;
        isCommon = false;
    } else if (values2.length >= LARGE_ARRAY_SIZE$1) {
        includes2 = cacheHas;
        isCommon = false;
        values2 = new SetCache(values2);
    }
    outer: while(++index < length){
        var value = array2[index], computed = iteratee2 == null ? value : iteratee2(value);
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values2[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result2.push(value);
        } else if (!includes2(values2, computed, comparator)) {
            result2.push(value);
        }
    }
    return result2;
}
var difference = baseRest(function(array2, values2) {
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true)) : [];
});
function last(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? array2[length - 1] : void 0;
}
var differenceBy = baseRest(function(array2, values2) {
    var iteratee2 = last(values2);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true), baseIteratee(iteratee2)) : [];
});
var differenceWith = baseRest(function(array2, values2) {
    var comparator = last(values2);
    if (isArrayLikeObject(comparator)) {
        comparator = void 0;
    }
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true), void 0, comparator) : [];
});
var divide = createMathOperation(function(dividend, divisor) {
    return dividend / divisor;
}, 1);
function drop(array2, n7, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n7 = guard || n7 === void 0 ? 1 : toInteger(n7);
    return baseSlice(array2, n7 < 0 ? 0 : n7, length);
}
function dropRight(array2, n8, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n8 = guard || n8 === void 0 ? 1 : toInteger(n8);
    n8 = length - n8;
    return baseSlice(array2, 0, n8 < 0 ? 0 : n8);
}
function baseWhile(array2, predicate, isDrop, fromRight) {
    var length = array2.length, index = fromRight ? length : -1;
    while((fromRight ? index-- : ++index < length) && predicate(array2[index], index, array2)){
    }
    return isDrop ? baseSlice(array2, fromRight ? 0 : index, fromRight ? index + 1 : length) : baseSlice(array2, fromRight ? index + 1 : 0, fromRight ? length : index);
}
function dropRightWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), true, true) : [];
}
function dropWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), true) : [];
}
function castFunction(value) {
    return typeof value == "function" ? value : identity;
}
function forEach(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayEach : baseEach;
    return func2(collection2, castFunction(iteratee2));
}
function arrayEachRight(array2, iteratee2) {
    var length = array2 == null ? 0 : array2.length;
    while(length--){
        if (iteratee2(array2[length], length, array2) === false) {
            break;
        }
    }
    return array2;
}
var baseForRight = createBaseFor(true);
function baseForOwnRight(object2, iteratee2) {
    return object2 && baseForRight(object2, iteratee2, keys);
}
var baseEachRight = createBaseEach(baseForOwnRight, true);
function forEachRight(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayEachRight : baseEachRight;
    return func2(collection2, castFunction(iteratee2));
}
function endsWith(string2, target, position) {
    string2 = toString(string2);
    target = baseToString(target);
    var length = string2.length;
    position = position === void 0 ? length : baseClamp(toInteger(position), 0, length);
    var end = position;
    position -= target.length;
    return position >= 0 && string2.slice(position, end) == target;
}
function baseToPairs(object2, props) {
    return arrayMap(props, function(key) {
        return [
            key,
            object2[key]
        ];
    });
}
function setToPairs(set2) {
    var index = -1, result2 = Array(set2.size);
    set2.forEach(function(value) {
        result2[++index] = [
            value,
            value
        ];
    });
    return result2;
}
var mapTag$6 = "[object Map]", setTag$6 = "[object Set]";
function createToPairs(keysFunc) {
    return function(object2) {
        var tag = getTag$1(object2);
        if (tag == mapTag$6) {
            return mapToArray(object2);
        }
        if (tag == setTag$6) {
            return setToPairs(object2);
        }
        return baseToPairs(object2, keysFunc(object2));
    };
}
var toPairs = createToPairs(keys);
var toPairsIn = createToPairs(keysIn);
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = basePropertyOf(htmlEscapes);
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string2) {
    string2 = toString(string2);
    return string2 && reHasUnescapedHtml.test(string2) ? string2.replace(reUnescapedHtml, escapeHtmlChar) : string2;
}
var reRegExpChar$1 = /[\\^$.*+?()[\]{}|]/g, reHasRegExpChar = RegExp(reRegExpChar$1.source);
function escapeRegExp(string2) {
    string2 = toString(string2);
    return string2 && reHasRegExpChar.test(string2) ? string2.replace(reRegExpChar$1, "\\$&") : string2;
}
function arrayEvery(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (!predicate(array2[index], index, array2)) {
            return false;
        }
    }
    return true;
}
function baseEvery(collection2, predicate) {
    var result2 = true;
    baseEach(collection2, function(value, index, collection3) {
        result2 = !!predicate(value, index, collection3);
        return result2;
    });
    return result2;
}
function every(collection2, predicate, guard) {
    var func2 = isArray(collection2) ? arrayEvery : baseEvery;
    if (guard && isIterateeCall(collection2, predicate, guard)) {
        predicate = void 0;
    }
    return func2(collection2, baseIteratee(predicate));
}
var MAX_ARRAY_LENGTH$1 = 4294967295;
function toLength(value) {
    return value ? baseClamp(toInteger(value), 0, MAX_ARRAY_LENGTH$1) : 0;
}
function baseFill(array2, value, start, end) {
    var length = array2.length;
    start = toInteger(start);
    if (start < 0) {
        start = -start > length ? 0 : length + start;
    }
    end = end === void 0 || end > length ? length : toInteger(end);
    if (end < 0) {
        end += length;
    }
    end = start > end ? 0 : toLength(end);
    while(start < end){
        array2[start++] = value;
    }
    return array2;
}
function fill(array2, value, start, end) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    if (start && typeof start != "number" && isIterateeCall(array2, value, start)) {
        start = 0;
        end = length;
    }
    return baseFill(array2, value, start, end);
}
function baseFilter(collection2, predicate) {
    var result2 = [];
    baseEach(collection2, function(value, index, collection3) {
        if (predicate(value, index, collection3)) {
            result2.push(value);
        }
    });
    return result2;
}
function filter(collection2, predicate) {
    var func2 = isArray(collection2) ? arrayFilter : baseFilter;
    return func2(collection2, baseIteratee(predicate));
}
function createFind(findIndexFunc) {
    return function(collection2, predicate, fromIndex) {
        var iterable = Object(collection2);
        if (!isArrayLike(collection2)) {
            var iteratee2 = baseIteratee(predicate);
            collection2 = keys(collection2);
            predicate = function(key) {
                return iteratee2(iterable[key], key, iterable);
            };
        }
        var index = findIndexFunc(collection2, predicate, fromIndex);
        return index > -1 ? iterable[iteratee2 ? collection2[index] : index] : void 0;
    };
}
var nativeMax$6 = Math.max;
function findIndex(array2, predicate, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = fromIndex == null ? 0 : toInteger(fromIndex);
    if (index < 0) {
        index = nativeMax$6(length + index, 0);
    }
    return baseFindIndex(array2, baseIteratee(predicate), index);
}
var find = createFind(findIndex);
function baseFindKey(collection2, predicate, eachFunc) {
    var result2;
    eachFunc(collection2, function(value, key, collection3) {
        if (predicate(value, key, collection3)) {
            result2 = key;
            return false;
        }
    });
    return result2;
}
function findKey(object2, predicate) {
    return baseFindKey(object2, baseIteratee(predicate), baseForOwn);
}
var nativeMax$7 = Math.max, nativeMin$4 = Math.min;
function findLastIndex(array2, predicate, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = length - 1;
    if (fromIndex !== void 0) {
        index = toInteger(fromIndex);
        index = fromIndex < 0 ? nativeMax$7(length + index, 0) : nativeMin$4(index, length - 1);
    }
    return baseFindIndex(array2, baseIteratee(predicate), index, true);
}
var findLast = createFind(findLastIndex);
function findLastKey(object2, predicate) {
    return baseFindKey(object2, baseIteratee(predicate), baseForOwnRight);
}
function head4(array2) {
    return array2 && array2.length ? array2[0] : void 0;
}
function baseMap(collection2, iteratee2) {
    var index = -1, result2 = isArrayLike(collection2) ? Array(collection2.length) : [];
    baseEach(collection2, function(value, key, collection3) {
        result2[++index] = iteratee2(value, key, collection3);
    });
    return result2;
}
function map(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayMap : baseMap;
    return func2(collection2, baseIteratee(iteratee2));
}
function flatMap(collection2, iteratee2) {
    return baseFlatten(map(collection2, iteratee2), 1);
}
var INFINITY$3 = 1 / 0;
function flatMapDeep(collection2, iteratee2) {
    return baseFlatten(map(collection2, iteratee2), INFINITY$3);
}
function flatMapDepth(collection2, iteratee2, depth) {
    depth = depth === void 0 ? 1 : toInteger(depth);
    return baseFlatten(map(collection2, iteratee2), depth);
}
var INFINITY$4 = 1 / 0;
function flattenDeep(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseFlatten(array2, INFINITY$4) : [];
}
function flattenDepth(array2, depth) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    depth = depth === void 0 ? 1 : toInteger(depth);
    return baseFlatten(array2, depth);
}
var WRAP_FLIP_FLAG$2 = 512;
function flip(func2) {
    return createWrap(func2, WRAP_FLIP_FLAG$2);
}
var floor = createRound("floor");
var FUNC_ERROR_TEXT$7 = "Expected a function";
var WRAP_CURRY_FLAG$6 = 8, WRAP_PARTIAL_FLAG$5 = 32, WRAP_ARY_FLAG$4 = 128, WRAP_REARG_FLAG$2 = 256;
function createFlow(fromRight) {
    return flatRest(function(funcs) {
        var length = funcs.length, index = length, prereq = LodashWrapper.prototype.thru;
        if (fromRight) {
            funcs.reverse();
        }
        while(index--){
            var func2 = funcs[index];
            if (typeof func2 != "function") {
                throw new TypeError(FUNC_ERROR_TEXT$7);
            }
            if (prereq && !wrapper && getFuncName(func2) == "wrapper") {
                var wrapper = new LodashWrapper([], true);
            }
        }
        index = wrapper ? index : length;
        while(++index < length){
            func2 = funcs[index];
            var funcName = getFuncName(func2), data21 = funcName == "wrapper" ? getData(func2) : void 0;
            if (data21 && isLaziable(data21[0]) && data21[1] == (WRAP_ARY_FLAG$4 | WRAP_CURRY_FLAG$6 | WRAP_PARTIAL_FLAG$5 | WRAP_REARG_FLAG$2) && !data21[4].length && data21[9] == 1) {
                wrapper = wrapper[getFuncName(data21[0])].apply(wrapper, data21[3]);
            } else {
                wrapper = func2.length == 1 && isLaziable(func2) ? wrapper[funcName]() : wrapper.thru(func2);
            }
        }
        return function() {
            var args = arguments, value = args[0];
            if (wrapper && args.length == 1 && isArray(value)) {
                return wrapper.plant(value).value();
            }
            var index2 = 0, result2 = length ? funcs[index2].apply(this, args) : value;
            while(++index2 < length){
                result2 = funcs[index2].call(this, result2);
            }
            return result2;
        };
    });
}
var flow = createFlow();
var flowRight = createFlow(true);
function forIn(object2, iteratee2) {
    return object2 == null ? object2 : baseFor(object2, castFunction(iteratee2), keysIn);
}
function forInRight(object2, iteratee2) {
    return object2 == null ? object2 : baseForRight(object2, castFunction(iteratee2), keysIn);
}
function forOwn(object2, iteratee2) {
    return object2 && baseForOwn(object2, castFunction(iteratee2));
}
function forOwnRight(object2, iteratee2) {
    return object2 && baseForOwnRight(object2, castFunction(iteratee2));
}
function fromPairs(pairs3) {
    var index = -1, length = pairs3 == null ? 0 : pairs3.length, result2 = {
    };
    while(++index < length){
        var pair = pairs3[index];
        result2[pair[0]] = pair[1];
    }
    return result2;
}
function baseFunctions(object2, props) {
    return arrayFilter(props, function(key) {
        return isFunction(object2[key]);
    });
}
function functions(object2) {
    return object2 == null ? [] : baseFunctions(object2, keys(object2));
}
function functionsIn(object2) {
    return object2 == null ? [] : baseFunctions(object2, keysIn(object2));
}
var objectProto$l = Object.prototype;
var hasOwnProperty$i = objectProto$l.hasOwnProperty;
var groupBy1 = createAggregator(function(result2, value, key) {
    if (hasOwnProperty$i.call(result2, key)) {
        result2[key].push(value);
    } else {
        baseAssignValue(result2, key, [
            value
        ]);
    }
});
function baseGt(value, other) {
    return value > other;
}
function createRelationalOperation(operator) {
    return function(value, other) {
        if (!(typeof value == "string" && typeof other == "string")) {
            value = toNumber(value);
            other = toNumber(other);
        }
        return operator(value, other);
    };
}
var gt = createRelationalOperation(baseGt);
var gte = createRelationalOperation(function(value, other) {
    return value >= other;
});
var objectProto$m = Object.prototype;
var hasOwnProperty$j = objectProto$m.hasOwnProperty;
function baseHas(object2, key) {
    return object2 != null && hasOwnProperty$j.call(object2, key);
}
function has(object2, path56) {
    return object2 != null && hasPath(object2, path56, baseHas);
}
var nativeMax$8 = Math.max, nativeMin$5 = Math.min;
function baseInRange(number2, start, end) {
    return number2 >= nativeMin$5(start, end) && number2 < nativeMax$8(start, end);
}
function inRange(number2, start, end) {
    start = toFinite(start);
    if (end === void 0) {
        end = start;
        start = 0;
    } else {
        end = toFinite(end);
    }
    number2 = toNumber(number2);
    return baseInRange(number2, start, end);
}
var stringTag$4 = "[object String]";
function isString(value) {
    return typeof value == "string" || !isArray(value) && isObjectLike(value) && baseGetTag(value) == stringTag$4;
}
function baseValues(object2, props) {
    return arrayMap(props, function(key) {
        return object2[key];
    });
}
function values(object2) {
    return object2 == null ? [] : baseValues(object2, keys(object2));
}
var nativeMax$9 = Math.max;
function includes(collection2, value, fromIndex, guard) {
    collection2 = isArrayLike(collection2) ? collection2 : values(collection2);
    fromIndex = fromIndex && !guard ? toInteger(fromIndex) : 0;
    var length = collection2.length;
    if (fromIndex < 0) {
        fromIndex = nativeMax$9(length + fromIndex, 0);
    }
    return isString(collection2) ? fromIndex <= length && collection2.indexOf(value, fromIndex) > -1 : !!length && baseIndexOf(collection2, value, fromIndex) > -1;
}
var nativeMax$a = Math.max;
function indexOf(array2, value, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = fromIndex == null ? 0 : toInteger(fromIndex);
    if (index < 0) {
        index = nativeMax$a(length + index, 0);
    }
    return baseIndexOf(array2, value, index);
}
function initial(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSlice(array2, 0, -1) : [];
}
var nativeMin$6 = Math.min;
function baseIntersection(arrays, iteratee2, comparator) {
    var includes2 = comparator ? arrayIncludesWith : arrayIncludes, length = arrays[0].length, othLength = arrays.length, othIndex = othLength, caches = Array(othLength), maxLength = Infinity, result2 = [];
    while(othIndex--){
        var array2 = arrays[othIndex];
        if (othIndex && iteratee2) {
            array2 = arrayMap(array2, baseUnary(iteratee2));
        }
        maxLength = nativeMin$6(array2.length, maxLength);
        caches[othIndex] = !comparator && (iteratee2 || length >= 120 && array2.length >= 120) ? new SetCache(othIndex && array2) : void 0;
    }
    array2 = arrays[0];
    var index = -1, seen = caches[0];
    outer: while(++index < length && result2.length < maxLength){
        var value = array2[index], computed = iteratee2 ? iteratee2(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (!(seen ? cacheHas(seen, computed) : includes2(result2, computed, comparator))) {
            othIndex = othLength;
            while(--othIndex){
                var cache = caches[othIndex];
                if (!(cache ? cacheHas(cache, computed) : includes2(arrays[othIndex], computed, comparator))) {
                    continue outer;
                }
            }
            if (seen) {
                seen.push(computed);
            }
            result2.push(value);
        }
    }
    return result2;
}
function castArrayLikeObject(value) {
    return isArrayLikeObject(value) ? value : [];
}
var intersection = baseRest(function(arrays) {
    var mapped = arrayMap(arrays, castArrayLikeObject);
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped) : [];
});
var intersectionBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays), mapped = arrayMap(arrays, castArrayLikeObject);
    if (iteratee2 === last(mapped)) {
        iteratee2 = void 0;
    } else {
        mapped.pop();
    }
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped, baseIteratee(iteratee2)) : [];
});
var intersectionWith = baseRest(function(arrays) {
    var comparator = last(arrays), mapped = arrayMap(arrays, castArrayLikeObject);
    comparator = typeof comparator == "function" ? comparator : void 0;
    if (comparator) {
        mapped.pop();
    }
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped, void 0, comparator) : [];
});
function baseInverter(object2, setter, iteratee2, accumulator) {
    baseForOwn(object2, function(value, key, object3) {
        setter(accumulator, iteratee2(value), key, object3);
    });
    return accumulator;
}
function createInverter(setter, toIteratee) {
    return function(object2, iteratee2) {
        return baseInverter(object2, setter, toIteratee(iteratee2), {
        });
    };
}
var objectProto$n = Object.prototype;
var nativeObjectToString$2 = objectProto$n.toString;
var invert = createInverter(function(result2, value, key) {
    if (value != null && typeof value.toString != "function") {
        value = nativeObjectToString$2.call(value);
    }
    result2[value] = key;
}, constant(identity));
var objectProto$o = Object.prototype;
var hasOwnProperty$k = objectProto$o.hasOwnProperty;
var nativeObjectToString$3 = objectProto$o.toString;
var invertBy = createInverter(function(result2, value, key) {
    if (value != null && typeof value.toString != "function") {
        value = nativeObjectToString$3.call(value);
    }
    if (hasOwnProperty$k.call(result2, value)) {
        result2[value].push(key);
    } else {
        result2[value] = [
            key
        ];
    }
}, baseIteratee);
function parent(object2, path57) {
    return path57.length < 2 ? object2 : baseGet(object2, baseSlice(path57, 0, -1));
}
function baseInvoke(object2, path58, args) {
    path58 = castPath(path58, object2);
    object2 = parent(object2, path58);
    var func2 = object2 == null ? object2 : object2[toKey(last(path58))];
    return func2 == null ? void 0 : apply(func2, object2, args);
}
var invoke = baseRest(baseInvoke);
var invokeMap = baseRest(function(collection2, path59, args) {
    var index = -1, isFunc = typeof path59 == "function", result2 = isArrayLike(collection2) ? Array(collection2.length) : [];
    baseEach(collection2, function(value) {
        result2[++index] = isFunc ? apply(path59, value, args) : baseInvoke(value, path59, args);
    });
    return result2;
});
var arrayBufferTag$4 = "[object ArrayBuffer]";
function baseIsArrayBuffer(value) {
    return isObjectLike(value) && baseGetTag(value) == arrayBufferTag$4;
}
var nodeIsArrayBuffer = nodeUtil && nodeUtil.isArrayBuffer;
var isArrayBuffer = nodeIsArrayBuffer ? baseUnary(nodeIsArrayBuffer) : baseIsArrayBuffer;
var boolTag$4 = "[object Boolean]";
function isBoolean(value) {
    return value === true || value === false || isObjectLike(value) && baseGetTag(value) == boolTag$4;
}
var dateTag$4 = "[object Date]";
function baseIsDate(value) {
    return isObjectLike(value) && baseGetTag(value) == dateTag$4;
}
var nodeIsDate = nodeUtil && nodeUtil.isDate;
var isDate = nodeIsDate ? baseUnary(nodeIsDate) : baseIsDate;
function isElement(value) {
    return isObjectLike(value) && value.nodeType === 1 && !isPlainObject(value);
}
var mapTag$7 = "[object Map]", setTag$7 = "[object Set]";
var objectProto$p = Object.prototype;
var hasOwnProperty$l = objectProto$p.hasOwnProperty;
function isEmpty(value) {
    if (value == null) {
        return true;
    }
    if (isArrayLike(value) && (isArray(value) || typeof value == "string" || typeof value.splice == "function" || isBuffer(value) || isTypedArray(value) || isArguments(value))) {
        return !value.length;
    }
    var tag = getTag$1(value);
    if (tag == mapTag$7 || tag == setTag$7) {
        return !value.size;
    }
    if (isPrototype(value)) {
        return !baseKeys(value).length;
    }
    for(var key in value){
        if (hasOwnProperty$l.call(value, key)) {
            return false;
        }
    }
    return true;
}
function isEqual(value, other) {
    return baseIsEqual(value, other);
}
function isEqualWith(value, other, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    var result2 = customizer ? customizer(value, other) : void 0;
    return result2 === void 0 ? baseIsEqual(value, other, void 0, customizer) : !!result2;
}
var nativeIsFinite$1 = root.isFinite;
function isFinite(value) {
    return typeof value == "number" && nativeIsFinite$1(value);
}
function isInteger(value) {
    return typeof value == "number" && value == toInteger(value);
}
function isMatch(object2, source) {
    return object2 === source || baseIsMatch(object2, source, getMatchData(source));
}
function isMatchWith(object2, source, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseIsMatch(object2, source, getMatchData(source), customizer);
}
var numberTag$4 = "[object Number]";
function isNumber(value) {
    return typeof value == "number" || isObjectLike(value) && baseGetTag(value) == numberTag$4;
}
function isNaN1(value) {
    return isNumber(value) && value != +value;
}
var isMaskable = coreJsData ? isFunction : stubFalse;
var CORE_ERROR_TEXT = "Unsupported core-js use. Try https://npms.io/search?q=ponyfill.";
function isNative(value) {
    if (isMaskable(value)) {
        throw new Error(CORE_ERROR_TEXT);
    }
    return baseIsNative(value);
}
function isNil(value) {
    return value == null;
}
function isNull(value) {
    return value === null;
}
var regexpTag$4 = "[object RegExp]";
function baseIsRegExp(value) {
    return isObjectLike(value) && baseGetTag(value) == regexpTag$4;
}
var nodeIsRegExp = nodeUtil && nodeUtil.isRegExp;
var isRegExp = nodeIsRegExp ? baseUnary(nodeIsRegExp) : baseIsRegExp;
var MAX_SAFE_INTEGER$2 = 9007199254740991;
function isSafeInteger(value) {
    return isInteger(value) && value >= -MAX_SAFE_INTEGER$2 && value <= MAX_SAFE_INTEGER$2;
}
function isUndefined(value) {
    return value === void 0;
}
var weakMapTag$3 = "[object WeakMap]";
function isWeakMap(value) {
    return isObjectLike(value) && getTag$1(value) == weakMapTag$3;
}
var weakSetTag = "[object WeakSet]";
function isWeakSet(value) {
    return isObjectLike(value) && baseGetTag(value) == weakSetTag;
}
var CLONE_DEEP_FLAG$4 = 1;
function iteratee(func2) {
    return baseIteratee(typeof func2 == "function" ? func2 : baseClone(func2, CLONE_DEEP_FLAG$4));
}
var arrayProto$1 = Array.prototype;
var nativeJoin = arrayProto$1.join;
function join4(array2, separator) {
    return array2 == null ? "" : nativeJoin.call(array2, separator);
}
var kebabCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? "-" : "") + word.toLowerCase();
});
var keyBy = createAggregator(function(result2, value, key) {
    baseAssignValue(result2, key, value);
});
function strictLastIndexOf(array2, value, fromIndex) {
    var index = fromIndex + 1;
    while(index--){
        if (array2[index] === value) {
            return index;
        }
    }
    return index;
}
var nativeMax$b = Math.max, nativeMin$7 = Math.min;
function lastIndexOf(array2, value, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = length;
    if (fromIndex !== void 0) {
        index = toInteger(fromIndex);
        index = index < 0 ? nativeMax$b(length + index, 0) : nativeMin$7(index, length - 1);
    }
    return value === value ? strictLastIndexOf(array2, value, index) : baseFindIndex(array2, baseIsNaN, index, true);
}
var lowerCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + word.toLowerCase();
});
var lowerFirst = createCaseFirst("toLowerCase");
function baseLt(value, other) {
    return value < other;
}
var lt = createRelationalOperation(baseLt);
var lte = createRelationalOperation(function(value, other) {
    return value <= other;
});
function mapKeys(object2, iteratee2) {
    var result2 = {
    };
    iteratee2 = baseIteratee(iteratee2);
    baseForOwn(object2, function(value, key, object3) {
        baseAssignValue(result2, iteratee2(value, key, object3), value);
    });
    return result2;
}
function mapValues(object2, iteratee2) {
    var result2 = {
    };
    iteratee2 = baseIteratee(iteratee2);
    baseForOwn(object2, function(value, key, object3) {
        baseAssignValue(result2, key, iteratee2(value, key, object3));
    });
    return result2;
}
var CLONE_DEEP_FLAG$5 = 1;
function matches(source) {
    return baseMatches(baseClone(source, CLONE_DEEP_FLAG$5));
}
var CLONE_DEEP_FLAG$6 = 1;
function matchesProperty(path60, srcValue) {
    return baseMatchesProperty(path60, baseClone(srcValue, CLONE_DEEP_FLAG$6));
}
function baseExtremum(array2, iteratee2, comparator) {
    var index = -1, length = array2.length;
    while(++index < length){
        var value = array2[index], current = iteratee2(value);
        if (current != null && (computed === void 0 ? current === current && !isSymbol(current) : comparator(current, computed))) {
            var computed = current, result2 = value;
        }
    }
    return result2;
}
function max(array2) {
    return array2 && array2.length ? baseExtremum(array2, identity, baseGt) : void 0;
}
function maxBy(array2, iteratee2) {
    return array2 && array2.length ? baseExtremum(array2, baseIteratee(iteratee2), baseGt) : void 0;
}
function baseSum(array2, iteratee2) {
    var result2, index = -1, length = array2.length;
    while(++index < length){
        var current = iteratee2(array2[index]);
        if (current !== void 0) {
            result2 = result2 === void 0 ? current : result2 + current;
        }
    }
    return result2;
}
var NAN$2 = 0 / 0;
function baseMean(array2, iteratee2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSum(array2, iteratee2) / length : NAN$2;
}
function mean(array2) {
    return baseMean(array2, identity);
}
function meanBy(array2, iteratee2) {
    return baseMean(array2, baseIteratee(iteratee2));
}
var merge = createAssigner(function(object2, source, srcIndex) {
    baseMerge(object2, source, srcIndex);
});
var method = baseRest(function(path61, args) {
    return function(object2) {
        return baseInvoke(object2, path61, args);
    };
});
var methodOf = baseRest(function(object2, args) {
    return function(path62) {
        return baseInvoke(object2, path62, args);
    };
});
function min(array2) {
    return array2 && array2.length ? baseExtremum(array2, identity, baseLt) : void 0;
}
function minBy(array2, iteratee2) {
    return array2 && array2.length ? baseExtremum(array2, baseIteratee(iteratee2), baseLt) : void 0;
}
function mixin(object2, source, options) {
    var props = keys(source), methodNames2 = baseFunctions(source, props);
    var chain2 = !(isObject(options) && "chain" in options) || !!options.chain, isFunc = isFunction(object2);
    arrayEach(methodNames2, function(methodName) {
        var func2 = source[methodName];
        object2[methodName] = func2;
        if (isFunc) {
            object2.prototype[methodName] = function() {
                var chainAll = this.__chain__;
                if (chain2 || chainAll) {
                    var result2 = object2(this.__wrapped__), actions = result2.__actions__ = copyArray(this.__actions__);
                    actions.push({
                        func: func2,
                        args: arguments,
                        thisArg: object2
                    });
                    result2.__chain__ = chainAll;
                    return result2;
                }
                return func2.apply(object2, arrayPush([
                    this.value()
                ], arguments));
            };
        }
    });
    return object2;
}
var multiply = createMathOperation(function(multiplier, multiplicand) {
    return multiplier * multiplicand;
}, 1);
var FUNC_ERROR_TEXT$8 = "Expected a function";
function negate(predicate) {
    if (typeof predicate != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$8);
    }
    return function() {
        var args = arguments;
        switch(args.length){
            case 0:
                return !predicate.call(this);
            case 1:
                return !predicate.call(this, args[0]);
            case 2:
                return !predicate.call(this, args[0], args[1]);
            case 3:
                return !predicate.call(this, args[0], args[1], args[2]);
        }
        return !predicate.apply(this, args);
    };
}
function iteratorToArray(iterator) {
    var data22, result2 = [];
    while(!(data22 = iterator.next()).done){
        result2.push(data22.value);
    }
    return result2;
}
var mapTag$8 = "[object Map]", setTag$8 = "[object Set]";
var symIterator = Symbol1 ? Symbol1.iterator : void 0;
function toArray(value) {
    if (!value) {
        return [];
    }
    if (isArrayLike(value)) {
        return isString(value) ? stringToArray(value) : copyArray(value);
    }
    if (symIterator && value[symIterator]) {
        return iteratorToArray(value[symIterator]());
    }
    var tag = getTag$1(value), func2 = tag == mapTag$8 ? mapToArray : tag == setTag$8 ? setToArray : values;
    return func2(value);
}
function wrapperNext() {
    if (this.__values__ === void 0) {
        this.__values__ = toArray(this.value());
    }
    var done = this.__index__ >= this.__values__.length, value = done ? void 0 : this.__values__[this.__index__++];
    return {
        done,
        value
    };
}
function baseNth(array2, n9) {
    var length = array2.length;
    if (!length) {
        return;
    }
    n9 += n9 < 0 ? length : 0;
    return isIndex(n9, length) ? array2[n9] : void 0;
}
function nth(array2, n10) {
    return array2 && array2.length ? baseNth(array2, toInteger(n10)) : void 0;
}
function nthArg(n11) {
    n11 = toInteger(n11);
    return baseRest(function(args) {
        return baseNth(args, n11);
    });
}
function baseUnset(object2, path63) {
    path63 = castPath(path63, object2);
    object2 = parent(object2, path63);
    return object2 == null || delete object2[toKey(last(path63))];
}
function customOmitClone(value) {
    return isPlainObject(value) ? void 0 : value;
}
var CLONE_DEEP_FLAG$7 = 1, CLONE_FLAT_FLAG$1 = 2, CLONE_SYMBOLS_FLAG$5 = 4;
var omit = flatRest(function(object2, paths) {
    var result2 = {
    };
    if (object2 == null) {
        return result2;
    }
    var isDeep = false;
    paths = arrayMap(paths, function(path64) {
        path64 = castPath(path64, object2);
        isDeep || (isDeep = path64.length > 1);
        return path64;
    });
    copyObject(object2, getAllKeysIn(object2), result2);
    if (isDeep) {
        result2 = baseClone(result2, CLONE_DEEP_FLAG$7 | CLONE_FLAT_FLAG$1 | CLONE_SYMBOLS_FLAG$5, customOmitClone);
    }
    var length = paths.length;
    while(length--){
        baseUnset(result2, paths[length]);
    }
    return result2;
});
function baseSet(object2, path65, value, customizer) {
    if (!isObject(object2)) {
        return object2;
    }
    path65 = castPath(path65, object2);
    var index = -1, length = path65.length, lastIndex = length - 1, nested = object2;
    while(nested != null && ++index < length){
        var key = toKey(path65[index]), newValue = value;
        if (key === "__proto__" || key === "constructor" || key === "prototype") {
            return object2;
        }
        if (index != lastIndex) {
            var objValue = nested[key];
            newValue = customizer ? customizer(objValue, key, nested) : void 0;
            if (newValue === void 0) {
                newValue = isObject(objValue) ? objValue : isIndex(path65[index + 1]) ? [] : {
                };
            }
        }
        assignValue(nested, key, newValue);
        nested = nested[key];
    }
    return object2;
}
function basePickBy(object2, paths, predicate) {
    var index = -1, length = paths.length, result2 = {
    };
    while(++index < length){
        var path66 = paths[index], value = baseGet(object2, path66);
        if (predicate(value, path66)) {
            baseSet(result2, castPath(path66, object2), value);
        }
    }
    return result2;
}
function pickBy(object2, predicate) {
    if (object2 == null) {
        return {
        };
    }
    var props = arrayMap(getAllKeysIn(object2), function(prop) {
        return [
            prop
        ];
    });
    predicate = baseIteratee(predicate);
    return basePickBy(object2, props, function(value, path67) {
        return predicate(value, path67[0]);
    });
}
function omitBy(object2, predicate) {
    return pickBy(object2, negate(baseIteratee(predicate)));
}
function once(func2) {
    return before(2, func2);
}
function baseSortBy(array2, comparer) {
    var length = array2.length;
    array2.sort(comparer);
    while(length--){
        array2[length] = array2[length].value;
    }
    return array2;
}
function compareAscending(value, other) {
    if (value !== other) {
        var valIsDefined = value !== void 0, valIsNull = value === null, valIsReflexive = value === value, valIsSymbol = isSymbol(value);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = isSymbol(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
function compareMultiple(object2, other, orders) {
    var index = -1, objCriteria = object2.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result2 = compareAscending(objCriteria[index], othCriteria[index]);
        if (result2) {
            if (index >= ordersLength) {
                return result2;
            }
            var order = orders[index];
            return result2 * (order == "desc" ? -1 : 1);
        }
    }
    return object2.index - other.index;
}
function baseOrderBy(collection2, iteratees, orders) {
    if (iteratees.length) {
        iteratees = arrayMap(iteratees, function(iteratee2) {
            if (isArray(iteratee2)) {
                return function(value) {
                    return baseGet(value, iteratee2.length === 1 ? iteratee2[0] : iteratee2);
                };
            }
            return iteratee2;
        });
    } else {
        iteratees = [
            identity
        ];
    }
    var index = -1;
    iteratees = arrayMap(iteratees, baseUnary(baseIteratee));
    var result2 = baseMap(collection2, function(value, key, collection3) {
        var criteria = arrayMap(iteratees, function(iteratee2) {
            return iteratee2(value);
        });
        return {
            criteria,
            index: ++index,
            value
        };
    });
    return baseSortBy(result2, function(object2, other) {
        return compareMultiple(object2, other, orders);
    });
}
function orderBy(collection2, iteratees, orders, guard) {
    if (collection2 == null) {
        return [];
    }
    if (!isArray(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!isArray(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return baseOrderBy(collection2, iteratees, orders);
}
function createOver(arrayFunc) {
    return flatRest(function(iteratees) {
        iteratees = arrayMap(iteratees, baseUnary(baseIteratee));
        return baseRest(function(args) {
            var thisArg = this;
            return arrayFunc(iteratees, function(iteratee2) {
                return apply(iteratee2, thisArg, args);
            });
        });
    });
}
var over = createOver(arrayMap);
var castRest = baseRest;
var nativeMin$8 = Math.min;
var overArgs = castRest(function(func2, transforms) {
    transforms = transforms.length == 1 && isArray(transforms[0]) ? arrayMap(transforms[0], baseUnary(baseIteratee)) : arrayMap(baseFlatten(transforms, 1), baseUnary(baseIteratee));
    var funcsLength = transforms.length;
    return baseRest(function(args) {
        var index = -1, length = nativeMin$8(args.length, funcsLength);
        while(++index < length){
            args[index] = transforms[index].call(this, args[index]);
        }
        return apply(func2, this, args);
    });
});
var overEvery = createOver(arrayEvery);
var overSome = createOver(arraySome);
var MAX_SAFE_INTEGER$3 = 9007199254740991;
var nativeFloor = Math.floor;
function baseRepeat(string2, n12) {
    var result2 = "";
    if (!string2 || n12 < 1 || n12 > MAX_SAFE_INTEGER$3) {
        return result2;
    }
    do {
        if (n12 % 2) {
            result2 += string2;
        }
        n12 = nativeFloor(n12 / 2);
        if (n12) {
            string2 += string2;
        }
    }while (n12)
    return result2;
}
var asciiSize = baseProperty("length");
var rsAstralRange$3 = "\\ud800-\\udfff", rsComboMarksRange$4 = "\\u0300-\\u036f", reComboHalfMarksRange$4 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$4 = "\\u20d0-\\u20ff", rsComboRange$4 = rsComboMarksRange$4 + reComboHalfMarksRange$4 + rsComboSymbolsRange$4, rsVarRange$3 = "\\ufe0e\\ufe0f";
var rsAstral$1 = "[" + rsAstralRange$3 + "]", rsCombo$3 = "[" + rsComboRange$4 + "]", rsFitz$2 = "\\ud83c[\\udffb-\\udfff]", rsModifier$2 = "(?:" + rsCombo$3 + "|" + rsFitz$2 + ")", rsNonAstral$2 = "[^" + rsAstralRange$3 + "]", rsRegional$2 = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair$2 = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsZWJ$3 = "\\u200d";
var reOptMod$2 = rsModifier$2 + "?", rsOptVar$2 = "[" + rsVarRange$3 + "]?", rsOptJoin$2 = "(?:" + rsZWJ$3 + "(?:" + [
    rsNonAstral$2,
    rsRegional$2,
    rsSurrPair$2
].join("|") + ")" + rsOptVar$2 + reOptMod$2 + ")*", rsSeq$2 = rsOptVar$2 + reOptMod$2 + rsOptJoin$2, rsSymbol$1 = "(?:" + [
    rsNonAstral$2 + rsCombo$3 + "?",
    rsCombo$3,
    rsRegional$2,
    rsSurrPair$2,
    rsAstral$1
].join("|") + ")";
var reUnicode$1 = RegExp(rsFitz$2 + "(?=" + rsFitz$2 + ")|" + rsSymbol$1 + rsSeq$2, "g");
function unicodeSize(string2) {
    var result2 = reUnicode$1.lastIndex = 0;
    while(reUnicode$1.test(string2)){
        ++result2;
    }
    return result2;
}
function stringSize(string2) {
    return hasUnicode(string2) ? unicodeSize(string2) : asciiSize(string2);
}
var nativeCeil$1 = Math.ceil;
function createPadding(length, chars) {
    chars = chars === void 0 ? " " : baseToString(chars);
    var charsLength = chars.length;
    if (charsLength < 2) {
        return charsLength ? baseRepeat(chars, length) : chars;
    }
    var result2 = baseRepeat(chars, nativeCeil$1(length / stringSize(chars)));
    return hasUnicode(chars) ? castSlice(stringToArray(result2), 0, length).join("") : result2.slice(0, length);
}
var nativeCeil$2 = Math.ceil, nativeFloor$1 = Math.floor;
function pad(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    if (!length || strLength >= length) {
        return string2;
    }
    var mid = (length - strLength) / 2;
    return createPadding(nativeFloor$1(mid), chars) + string2 + createPadding(nativeCeil$2(mid), chars);
}
function padEnd(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    return length && strLength < length ? string2 + createPadding(length - strLength, chars) : string2;
}
function padStart(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    return length && strLength < length ? createPadding(length - strLength, chars) + string2 : string2;
}
var reTrimStart$1 = /^\s+/;
var nativeParseInt = root.parseInt;
function parseInt$1(string2, radix, guard) {
    if (guard || radix == null) {
        radix = 0;
    } else if (radix) {
        radix = +radix;
    }
    return nativeParseInt(toString(string2).replace(reTrimStart$1, ""), radix || 0);
}
var WRAP_PARTIAL_FLAG$6 = 32;
var partial1 = baseRest(function(func2, partials) {
    var holders = replaceHolders(partials, getHolder(partial1));
    return createWrap(func2, WRAP_PARTIAL_FLAG$6, void 0, partials, holders);
});
partial1.placeholder = {
};
var WRAP_PARTIAL_RIGHT_FLAG$3 = 64;
var partialRight = baseRest(function(func2, partials) {
    var holders = replaceHolders(partials, getHolder(partialRight));
    return createWrap(func2, WRAP_PARTIAL_RIGHT_FLAG$3, void 0, partials, holders);
});
partialRight.placeholder = {
};
var partition = createAggregator(function(result2, value, key) {
    result2[key ? 0 : 1].push(value);
}, function() {
    return [
        [],
        []
    ];
});
function basePick(object2, paths) {
    return basePickBy(object2, paths, function(value, path68) {
        return hasIn(object2, path68);
    });
}
var pick = flatRest(function(object2, paths) {
    return object2 == null ? {
    } : basePick(object2, paths);
});
function wrapperPlant(value) {
    var result2, parent2 = this;
    while(parent2 instanceof baseLodash){
        var clone2 = wrapperClone(parent2);
        clone2.__index__ = 0;
        clone2.__values__ = void 0;
        if (result2) {
            previous.__wrapped__ = clone2;
        } else {
            result2 = clone2;
        }
        var previous = clone2;
        parent2 = parent2.__wrapped__;
    }
    previous.__wrapped__ = value;
    return result2;
}
function propertyOf(object2) {
    return function(path69) {
        return object2 == null ? void 0 : baseGet(object2, path69);
    };
}
function baseIndexOfWith(array2, value, fromIndex, comparator) {
    var index = fromIndex - 1, length = array2.length;
    while(++index < length){
        if (comparator(array2[index], value)) {
            return index;
        }
    }
    return -1;
}
var arrayProto$2 = Array.prototype;
var splice$1 = arrayProto$2.splice;
function basePullAll(array2, values2, iteratee2, comparator) {
    var indexOf2 = comparator ? baseIndexOfWith : baseIndexOf, index = -1, length = values2.length, seen = array2;
    if (array2 === values2) {
        values2 = copyArray(values2);
    }
    if (iteratee2) {
        seen = arrayMap(array2, baseUnary(iteratee2));
    }
    while(++index < length){
        var fromIndex = 0, value = values2[index], computed = iteratee2 ? iteratee2(value) : value;
        while((fromIndex = indexOf2(seen, computed, fromIndex, comparator)) > -1){
            if (seen !== array2) {
                splice$1.call(seen, fromIndex, 1);
            }
            splice$1.call(array2, fromIndex, 1);
        }
    }
    return array2;
}
function pullAll(array2, values2) {
    return array2 && array2.length && values2 && values2.length ? basePullAll(array2, values2) : array2;
}
var pull = baseRest(pullAll);
function pullAllBy(array2, values2, iteratee2) {
    return array2 && array2.length && values2 && values2.length ? basePullAll(array2, values2, baseIteratee(iteratee2)) : array2;
}
function pullAllWith(array2, values2, comparator) {
    return array2 && array2.length && values2 && values2.length ? basePullAll(array2, values2, void 0, comparator) : array2;
}
var arrayProto$3 = Array.prototype;
var splice$2 = arrayProto$3.splice;
function basePullAt(array2, indexes) {
    var length = array2 ? indexes.length : 0, lastIndex = length - 1;
    while(length--){
        var index = indexes[length];
        if (length == lastIndex || index !== previous) {
            var previous = index;
            if (isIndex(index)) {
                splice$2.call(array2, index, 1);
            } else {
                baseUnset(array2, index);
            }
        }
    }
    return array2;
}
var pullAt = flatRest(function(array2, indexes) {
    var length = array2 == null ? 0 : array2.length, result2 = baseAt(array2, indexes);
    basePullAt(array2, arrayMap(indexes, function(index) {
        return isIndex(index, length) ? +index : index;
    }).sort(compareAscending));
    return result2;
});
var nativeFloor$2 = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor$2(nativeRandom() * (upper - lower + 1));
}
var freeParseFloat = parseFloat;
var nativeMin$9 = Math.min, nativeRandom$1 = Math.random;
function random(lower, upper, floating) {
    if (floating && typeof floating != "boolean" && isIterateeCall(lower, upper, floating)) {
        upper = floating = void 0;
    }
    if (floating === void 0) {
        if (typeof upper == "boolean") {
            floating = upper;
            upper = void 0;
        } else if (typeof lower == "boolean") {
            floating = lower;
            lower = void 0;
        }
    }
    if (lower === void 0 && upper === void 0) {
        lower = 0;
        upper = 1;
    } else {
        lower = toFinite(lower);
        if (upper === void 0) {
            upper = lower;
            lower = 0;
        } else {
            upper = toFinite(upper);
        }
    }
    if (lower > upper) {
        var temp = lower;
        lower = upper;
        upper = temp;
    }
    if (floating || lower % 1 || upper % 1) {
        var rand = nativeRandom$1();
        return nativeMin$9(lower + rand * (upper - lower + freeParseFloat("1e-" + ((rand + "").length - 1))), upper);
    }
    return baseRandom(lower, upper);
}
var nativeCeil$3 = Math.ceil, nativeMax$c = Math.max;
function baseRange(start, end, step, fromRight) {
    var index = -1, length = nativeMax$c(nativeCeil$3((end - start) / (step || 1)), 0), result2 = Array(length);
    while(length--){
        result2[fromRight ? length : ++index] = start;
        start += step;
    }
    return result2;
}
function createRange(fromRight) {
    return function(start, end, step) {
        if (step && typeof step != "number" && isIterateeCall(start, end, step)) {
            end = step = void 0;
        }
        start = toFinite(start);
        if (end === void 0) {
            end = start;
            start = 0;
        } else {
            end = toFinite(end);
        }
        step = step === void 0 ? start < end ? 1 : -1 : toFinite(step);
        return baseRange(start, end, step, fromRight);
    };
}
var range = createRange();
var rangeRight = createRange(true);
var WRAP_REARG_FLAG$3 = 256;
var rearg = flatRest(function(func2, indexes) {
    return createWrap(func2, WRAP_REARG_FLAG$3, void 0, void 0, void 0, indexes);
});
function baseReduce(collection2, iteratee2, accumulator, initAccum, eachFunc) {
    eachFunc(collection2, function(value, index, collection3) {
        accumulator = initAccum ? (initAccum = false, value) : iteratee2(accumulator, value, index, collection3);
    });
    return accumulator;
}
function reduce(collection2, iteratee2, accumulator) {
    var func2 = isArray(collection2) ? arrayReduce : baseReduce, initAccum = arguments.length < 3;
    return func2(collection2, baseIteratee(iteratee2), accumulator, initAccum, baseEach);
}
function arrayReduceRight(array2, iteratee2, accumulator, initAccum) {
    var length = array2 == null ? 0 : array2.length;
    if (initAccum && length) {
        accumulator = array2[--length];
    }
    while(length--){
        accumulator = iteratee2(accumulator, array2[length], length, array2);
    }
    return accumulator;
}
function reduceRight(collection2, iteratee2, accumulator) {
    var func2 = isArray(collection2) ? arrayReduceRight : baseReduce, initAccum = arguments.length < 3;
    return func2(collection2, baseIteratee(iteratee2), accumulator, initAccum, baseEachRight);
}
function reject(collection2, predicate) {
    var func2 = isArray(collection2) ? arrayFilter : baseFilter;
    return func2(collection2, negate(baseIteratee(predicate)));
}
function remove(array2, predicate) {
    var result2 = [];
    if (!(array2 && array2.length)) {
        return result2;
    }
    var index = -1, indexes = [], length = array2.length;
    predicate = baseIteratee(predicate);
    while(++index < length){
        var value = array2[index];
        if (predicate(value, index, array2)) {
            result2.push(value);
            indexes.push(index);
        }
    }
    basePullAt(array2, indexes);
    return result2;
}
function repeat(string2, n13, guard) {
    if (guard ? isIterateeCall(string2, n13, guard) : n13 === void 0) {
        n13 = 1;
    } else {
        n13 = toInteger(n13);
    }
    return baseRepeat(toString(string2), n13);
}
function replace() {
    var args = arguments, string2 = toString(args[0]);
    return args.length < 3 ? string2 : string2.replace(args[1], args[2]);
}
var FUNC_ERROR_TEXT$9 = "Expected a function";
function rest(func2, start) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$9);
    }
    start = start === void 0 ? start : toInteger(start);
    return baseRest(func2, start);
}
function result36(object2, path70, defaultValue) {
    path70 = castPath(path70, object2);
    var index = -1, length = path70.length;
    if (!length) {
        length = 1;
        object2 = void 0;
    }
    while(++index < length){
        var value = object2 == null ? void 0 : object2[toKey(path70[index])];
        if (value === void 0) {
            index = length;
            value = defaultValue;
        }
        object2 = isFunction(value) ? value.call(object2) : value;
    }
    return object2;
}
var arrayProto$4 = Array.prototype;
var nativeReverse = arrayProto$4.reverse;
function reverse(array2) {
    return array2 == null ? array2 : nativeReverse.call(array2);
}
var round = createRound("round");
function arraySample(array2) {
    var length = array2.length;
    return length ? array2[baseRandom(0, length - 1)] : void 0;
}
function baseSample(collection2) {
    return arraySample(values(collection2));
}
function sample(collection2) {
    var func2 = isArray(collection2) ? arraySample : baseSample;
    return func2(collection2);
}
function shuffleSelf(array2, size2) {
    var index = -1, length = array2.length, lastIndex = length - 1;
    size2 = size2 === void 0 ? length : size2;
    while(++index < size2){
        var rand = baseRandom(index, lastIndex), value = array2[rand];
        array2[rand] = array2[index];
        array2[index] = value;
    }
    array2.length = size2;
    return array2;
}
function arraySampleSize(array2, n14) {
    return shuffleSelf(copyArray(array2), baseClamp(n14, 0, array2.length));
}
function baseSampleSize(collection2, n15) {
    var array2 = values(collection2);
    return shuffleSelf(array2, baseClamp(n15, 0, array2.length));
}
function sampleSize(collection2, n16, guard) {
    if (guard ? isIterateeCall(collection2, n16, guard) : n16 === void 0) {
        n16 = 1;
    } else {
        n16 = toInteger(n16);
    }
    var func2 = isArray(collection2) ? arraySampleSize : baseSampleSize;
    return func2(collection2, n16);
}
function set(object2, path71, value) {
    return object2 == null ? object2 : baseSet(object2, path71, value);
}
function setWith(object2, path72, value, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return object2 == null ? object2 : baseSet(object2, path72, value, customizer);
}
function arrayShuffle(array2) {
    return shuffleSelf(copyArray(array2));
}
function baseShuffle(collection2) {
    return shuffleSelf(values(collection2));
}
function shuffle(collection2) {
    var func2 = isArray(collection2) ? arrayShuffle : baseShuffle;
    return func2(collection2);
}
var mapTag$9 = "[object Map]", setTag$9 = "[object Set]";
function size(collection2) {
    if (collection2 == null) {
        return 0;
    }
    if (isArrayLike(collection2)) {
        return isString(collection2) ? stringSize(collection2) : collection2.length;
    }
    var tag = getTag$1(collection2);
    if (tag == mapTag$9 || tag == setTag$9) {
        return collection2.size;
    }
    return baseKeys(collection2).length;
}
function slice(array2, start, end) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    if (end && typeof end != "number" && isIterateeCall(array2, start, end)) {
        start = 0;
        end = length;
    } else {
        start = start == null ? 0 : toInteger(start);
        end = end === void 0 ? length : toInteger(end);
    }
    return baseSlice(array2, start, end);
}
var snakeCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? "_" : "") + word.toLowerCase();
});
function baseSome(collection2, predicate) {
    var result2;
    baseEach(collection2, function(value, index, collection3) {
        result2 = predicate(value, index, collection3);
        return !result2;
    });
    return !!result2;
}
function some(collection2, predicate, guard) {
    var func2 = isArray(collection2) ? arraySome : baseSome;
    if (guard && isIterateeCall(collection2, predicate, guard)) {
        predicate = void 0;
    }
    return func2(collection2, baseIteratee(predicate));
}
var sortBy = baseRest(function(collection2, iteratees) {
    if (collection2 == null) {
        return [];
    }
    var length = iteratees.length;
    if (length > 1 && isIterateeCall(collection2, iteratees[0], iteratees[1])) {
        iteratees = [];
    } else if (length > 2 && isIterateeCall(iteratees[0], iteratees[1], iteratees[2])) {
        iteratees = [
            iteratees[0]
        ];
    }
    return baseOrderBy(collection2, baseFlatten(iteratees, 1), []);
});
var MAX_ARRAY_LENGTH$2 = 4294967295, MAX_ARRAY_INDEX = MAX_ARRAY_LENGTH$2 - 1;
var nativeFloor$3 = Math.floor, nativeMin$a = Math.min;
function baseSortedIndexBy(array2, value, iteratee2, retHighest) {
    var low = 0, high = array2 == null ? 0 : array2.length;
    if (high === 0) {
        return 0;
    }
    value = iteratee2(value);
    var valIsNaN = value !== value, valIsNull = value === null, valIsSymbol = isSymbol(value), valIsUndefined = value === void 0;
    while(low < high){
        var mid = nativeFloor$3((low + high) / 2), computed = iteratee2(array2[mid]), othIsDefined = computed !== void 0, othIsNull = computed === null, othIsReflexive = computed === computed, othIsSymbol = isSymbol(computed);
        if (valIsNaN) {
            var setLow = retHighest || othIsReflexive;
        } else if (valIsUndefined) {
            setLow = othIsReflexive && (retHighest || othIsDefined);
        } else if (valIsNull) {
            setLow = othIsReflexive && othIsDefined && (retHighest || !othIsNull);
        } else if (valIsSymbol) {
            setLow = othIsReflexive && othIsDefined && !othIsNull && (retHighest || !othIsSymbol);
        } else if (othIsNull || othIsSymbol) {
            setLow = false;
        } else {
            setLow = retHighest ? computed <= value : computed < value;
        }
        if (setLow) {
            low = mid + 1;
        } else {
            high = mid;
        }
    }
    return nativeMin$a(high, MAX_ARRAY_INDEX);
}
var MAX_ARRAY_LENGTH$3 = 4294967295, HALF_MAX_ARRAY_LENGTH = MAX_ARRAY_LENGTH$3 >>> 1;
function baseSortedIndex(array2, value, retHighest) {
    var low = 0, high = array2 == null ? low : array2.length;
    if (typeof value == "number" && value === value && high <= HALF_MAX_ARRAY_LENGTH) {
        while(low < high){
            var mid = low + high >>> 1, computed = array2[mid];
            if (computed !== null && !isSymbol(computed) && (retHighest ? computed <= value : computed < value)) {
                low = mid + 1;
            } else {
                high = mid;
            }
        }
        return high;
    }
    return baseSortedIndexBy(array2, value, identity, retHighest);
}
function sortedIndex(array2, value) {
    return baseSortedIndex(array2, value);
}
function sortedIndexBy(array2, value, iteratee2) {
    return baseSortedIndexBy(array2, value, baseIteratee(iteratee2));
}
function sortedIndexOf(array2, value) {
    var length = array2 == null ? 0 : array2.length;
    if (length) {
        var index = baseSortedIndex(array2, value);
        if (index < length && eq(array2[index], value)) {
            return index;
        }
    }
    return -1;
}
function sortedLastIndex(array2, value) {
    return baseSortedIndex(array2, value, true);
}
function sortedLastIndexBy(array2, value, iteratee2) {
    return baseSortedIndexBy(array2, value, baseIteratee(iteratee2), true);
}
function sortedLastIndexOf(array2, value) {
    var length = array2 == null ? 0 : array2.length;
    if (length) {
        var index = baseSortedIndex(array2, value, true) - 1;
        if (eq(array2[index], value)) {
            return index;
        }
    }
    return -1;
}
function baseSortedUniq(array2, iteratee2) {
    var index = -1, length = array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value = array2[index], computed = iteratee2 ? iteratee2(value) : value;
        if (!index || !eq(computed, seen)) {
            var seen = computed;
            result2[resIndex++] = value === 0 ? 0 : value;
        }
    }
    return result2;
}
function sortedUniq(array2) {
    return array2 && array2.length ? baseSortedUniq(array2) : [];
}
function sortedUniqBy(array2, iteratee2) {
    return array2 && array2.length ? baseSortedUniq(array2, baseIteratee(iteratee2)) : [];
}
var MAX_ARRAY_LENGTH$4 = 4294967295;
function split1(string2, separator, limit) {
    if (limit && typeof limit != "number" && isIterateeCall(string2, separator, limit)) {
        separator = limit = void 0;
    }
    limit = limit === void 0 ? MAX_ARRAY_LENGTH$4 : limit >>> 0;
    if (!limit) {
        return [];
    }
    string2 = toString(string2);
    if (string2 && (typeof separator == "string" || separator != null && !isRegExp(separator))) {
        separator = baseToString(separator);
        if (!separator && hasUnicode(string2)) {
            return castSlice(stringToArray(string2), 0, limit);
        }
    }
    return string2.split(separator, limit);
}
var FUNC_ERROR_TEXT$a = "Expected a function";
var nativeMax$d = Math.max;
function spread(func2, start) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$a);
    }
    start = start == null ? 0 : nativeMax$d(toInteger(start), 0);
    return baseRest(function(args) {
        var array2 = args[start], otherArgs = castSlice(args, 0, start);
        if (array2) {
            arrayPush(otherArgs, array2);
        }
        return apply(func2, this, otherArgs);
    });
}
var startCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + upperFirst(word);
});
function startsWith(string2, target, position) {
    string2 = toString(string2);
    position = position == null ? 0 : baseClamp(toInteger(position), 0, string2.length);
    target = baseToString(target);
    return string2.slice(position, position + target.length) == target;
}
function stubObject() {
    return {
    };
}
function stubString() {
    return "";
}
function stubTrue() {
    return true;
}
var subtract = createMathOperation(function(minuend, subtrahend) {
    return minuend - subtrahend;
}, 0);
function sum(array2) {
    return array2 && array2.length ? baseSum(array2, identity) : 0;
}
function sumBy(array2, iteratee2) {
    return array2 && array2.length ? baseSum(array2, baseIteratee(iteratee2)) : 0;
}
function tail(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSlice(array2, 1, length) : [];
}
function take(array2, n17, guard) {
    if (!(array2 && array2.length)) {
        return [];
    }
    n17 = guard || n17 === void 0 ? 1 : toInteger(n17);
    return baseSlice(array2, 0, n17 < 0 ? 0 : n17);
}
function takeRight(array2, n18, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n18 = guard || n18 === void 0 ? 1 : toInteger(n18);
    n18 = length - n18;
    return baseSlice(array2, n18 < 0 ? 0 : n18, length);
}
function takeRightWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), false, true) : [];
}
function takeWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate)) : [];
}
function tap(value, interceptor) {
    interceptor(value);
    return value;
}
var objectProto$q = Object.prototype;
var hasOwnProperty$m = objectProto$q.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object2) {
    if (objValue === void 0 || eq(objValue, objectProto$q[key]) && !hasOwnProperty$m.call(object2, key)) {
        return srcValue;
    }
    return objValue;
}
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var reInterpolate = /<%=([\s\S]+?)%>/g;
var reEscape = /<%-([\s\S]+?)%>/g;
var reEvaluate = /<%([\s\S]+?)%>/g;
var templateSettings = {
    escape: reEscape,
    evaluate: reEvaluate,
    interpolate: reInterpolate,
    variable: "",
    imports: {
        _: {
            escape
        }
    }
};
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto$r = Object.prototype;
var hasOwnProperty$n = objectProto$r.hasOwnProperty;
function template(string2, options, guard) {
    var settings = templateSettings.imports._.templateSettings || templateSettings;
    if (guard && isIterateeCall(string2, options, guard)) {
        options = void 0;
    }
    string2 = toString(string2);
    options = assignInWith({
    }, options, settings, customDefaultsAssignIn);
    var imports = assignInWith({
    }, options.imports, settings.imports, customDefaultsAssignIn), importsKeys = keys(imports), importsValues = baseValues(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === reInterpolate ? reEsTemplate : reNoMatch).source + "|" + (options.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty$n.call(options, "sourceURL") ? "//# sourceURL=" + (options.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string2.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string2.slice(index, offset).replace(reUnescapedString, escapeStringChar);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty$n.call(options, "variable") && options.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result2 = attempt(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result2.source = source;
    if (isError(result2)) {
        throw result2;
    }
    return result2;
}
var FUNC_ERROR_TEXT$b = "Expected a function";
function throttle(func2, wait, options) {
    var leading = true, trailing = true;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$b);
    }
    if (isObject(options)) {
        leading = "leading" in options ? !!options.leading : leading;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    return debounce(func2, wait, {
        leading,
        maxWait: wait,
        trailing
    });
}
function thru(value, interceptor) {
    return interceptor(value);
}
var MAX_SAFE_INTEGER$4 = 9007199254740991;
var MAX_ARRAY_LENGTH$5 = 4294967295;
var nativeMin$b = Math.min;
function times(n19, iteratee2) {
    n19 = toInteger(n19);
    if (n19 < 1 || n19 > MAX_SAFE_INTEGER$4) {
        return [];
    }
    var index = MAX_ARRAY_LENGTH$5, length = nativeMin$b(n19, MAX_ARRAY_LENGTH$5);
    iteratee2 = castFunction(iteratee2);
    n19 -= MAX_ARRAY_LENGTH$5;
    var result2 = baseTimes(length, iteratee2);
    while(++index < n19){
        iteratee2(index);
    }
    return result2;
}
function wrapperToIterator() {
    return this;
}
function baseWrapperValue(value, actions) {
    var result2 = value;
    if (result2 instanceof LazyWrapper) {
        result2 = result2.value();
    }
    return arrayReduce(actions, function(result3, action) {
        return action.func.apply(action.thisArg, arrayPush([
            result3
        ], action.args));
    }, result2);
}
function wrapperValue() {
    return baseWrapperValue(this.__wrapped__, this.__actions__);
}
function toLower(value) {
    return toString(value).toLowerCase();
}
function toPath(value) {
    if (isArray(value)) {
        return arrayMap(value, toKey);
    }
    return isSymbol(value) ? [
        value
    ] : copyArray(stringToPath(toString(value)));
}
var MAX_SAFE_INTEGER$5 = 9007199254740991;
function toSafeInteger(value) {
    return value ? baseClamp(toInteger(value), -MAX_SAFE_INTEGER$5, MAX_SAFE_INTEGER$5) : value === 0 ? value : 0;
}
function toUpper(value) {
    return toString(value).toUpperCase();
}
function transform(object2, iteratee2, accumulator) {
    var isArr = isArray(object2), isArrLike = isArr || isBuffer(object2) || isTypedArray(object2);
    iteratee2 = baseIteratee(iteratee2);
    if (accumulator == null) {
        var Ctor = object2 && object2.constructor;
        if (isArrLike) {
            accumulator = isArr ? new Ctor() : [];
        } else if (isObject(object2)) {
            accumulator = isFunction(Ctor) ? baseCreate(getPrototype(object2)) : {
            };
        } else {
            accumulator = {
            };
        }
    }
    (isArrLike ? arrayEach : baseForOwn)(object2, function(value, index, object3) {
        return iteratee2(accumulator, value, index, object3);
    });
    return accumulator;
}
function charsEndIndex(strSymbols, chrSymbols) {
    var index = strSymbols.length;
    while((index--) && baseIndexOf(chrSymbols, strSymbols[index], 0) > -1){
    }
    return index;
}
function charsStartIndex(strSymbols, chrSymbols) {
    var index = -1, length = strSymbols.length;
    while(++index < length && baseIndexOf(chrSymbols, strSymbols[index], 0) > -1){
    }
    return index;
}
function trim(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return baseTrim(string2);
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), chrSymbols = stringToArray(chars), start = charsStartIndex(strSymbols, chrSymbols), end = charsEndIndex(strSymbols, chrSymbols) + 1;
    return castSlice(strSymbols, start, end).join("");
}
function trimEnd(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return string2.slice(0, trimmedEndIndex(string2) + 1);
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), end = charsEndIndex(strSymbols, stringToArray(chars)) + 1;
    return castSlice(strSymbols, 0, end).join("");
}
var reTrimStart$2 = /^\s+/;
function trimStart(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return string2.replace(reTrimStart$2, "");
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), start = charsStartIndex(strSymbols, stringToArray(chars));
    return castSlice(strSymbols, start).join("");
}
var DEFAULT_TRUNC_LENGTH = 30, DEFAULT_TRUNC_OMISSION = "...";
var reFlags$1 = /\w*$/;
function truncate(string2, options) {
    var length = DEFAULT_TRUNC_LENGTH, omission = DEFAULT_TRUNC_OMISSION;
    if (isObject(options)) {
        var separator = "separator" in options ? options.separator : separator;
        length = "length" in options ? toInteger(options.length) : length;
        omission = "omission" in options ? baseToString(options.omission) : omission;
    }
    string2 = toString(string2);
    var strLength = string2.length;
    if (hasUnicode(string2)) {
        var strSymbols = stringToArray(string2);
        strLength = strSymbols.length;
    }
    if (length >= strLength) {
        return string2;
    }
    var end = length - stringSize(omission);
    if (end < 1) {
        return omission;
    }
    var result2 = strSymbols ? castSlice(strSymbols, 0, end).join("") : string2.slice(0, end);
    if (separator === void 0) {
        return result2 + omission;
    }
    if (strSymbols) {
        end += result2.length - end;
    }
    if (isRegExp(separator)) {
        if (string2.slice(end).search(separator)) {
            var match, substring = result2;
            if (!separator.global) {
                separator = RegExp(separator.source, toString(reFlags$1.exec(separator)) + "g");
            }
            separator.lastIndex = 0;
            while(match = separator.exec(substring)){
                var newEnd = match.index;
            }
            result2 = result2.slice(0, newEnd === void 0 ? end : newEnd);
        }
    } else if (string2.indexOf(baseToString(separator), end) != end) {
        var index = result2.lastIndexOf(separator);
        if (index > -1) {
            result2 = result2.slice(0, index);
        }
    }
    return result2 + omission;
}
function unary(func2) {
    return ary(func2, 1);
}
var htmlUnescapes = {
    "&amp;": "&",
    "&lt;": "<",
    "&gt;": ">",
    "&quot;": '"',
    "&#39;": "'"
};
var unescapeHtmlChar = basePropertyOf(htmlUnescapes);
var reEscapedHtml = /&(?:amp|lt|gt|quot|#39);/g, reHasEscapedHtml = RegExp(reEscapedHtml.source);
function unescape1(string2) {
    string2 = toString(string2);
    return string2 && reHasEscapedHtml.test(string2) ? string2.replace(reEscapedHtml, unescapeHtmlChar) : string2;
}
var INFINITY$5 = 1 / 0;
var createSet = !(Set1 && 1 / setToArray(new Set1([
    ,
    -0
]))[1] == INFINITY$5) ? noop : function(values2) {
    return new Set1(values2);
};
var LARGE_ARRAY_SIZE$2 = 200;
function baseUniq(array2, iteratee2, comparator) {
    var index = -1, includes2 = arrayIncludes, length = array2.length, isCommon = true, result2 = [], seen = result2;
    if (comparator) {
        isCommon = false;
        includes2 = arrayIncludesWith;
    } else if (length >= LARGE_ARRAY_SIZE$2) {
        var set2 = iteratee2 ? null : createSet(array2);
        if (set2) {
            return setToArray(set2);
        }
        isCommon = false;
        includes2 = cacheHas;
        seen = new SetCache();
    } else {
        seen = iteratee2 ? [] : result2;
    }
    outer: while(++index < length){
        var value = array2[index], computed = iteratee2 ? iteratee2(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee2) {
                seen.push(computed);
            }
            result2.push(value);
        } else if (!includes2(seen, computed, comparator)) {
            if (seen !== result2) {
                seen.push(computed);
            }
            result2.push(value);
        }
    }
    return result2;
}
var union = baseRest(function(arrays) {
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true));
});
var unionBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true), baseIteratee(iteratee2));
});
var unionWith = baseRest(function(arrays) {
    var comparator = last(arrays);
    comparator = typeof comparator == "function" ? comparator : void 0;
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true), void 0, comparator);
});
function uniq(array2) {
    return array2 && array2.length ? baseUniq(array2) : [];
}
function uniqBy(array2, iteratee2) {
    return array2 && array2.length ? baseUniq(array2, baseIteratee(iteratee2)) : [];
}
function uniqWith(array2, comparator) {
    comparator = typeof comparator == "function" ? comparator : void 0;
    return array2 && array2.length ? baseUniq(array2, void 0, comparator) : [];
}
var idCounter = 0;
function uniqueId(prefix) {
    var id1 = ++idCounter;
    return toString(prefix) + id1;
}
function unset(object2, path73) {
    return object2 == null ? true : baseUnset(object2, path73);
}
var nativeMax$e = Math.max;
function unzip(array2) {
    if (!(array2 && array2.length)) {
        return [];
    }
    var length = 0;
    array2 = arrayFilter(array2, function(group) {
        if (isArrayLikeObject(group)) {
            length = nativeMax$e(group.length, length);
            return true;
        }
    });
    return baseTimes(length, function(index) {
        return arrayMap(array2, baseProperty(index));
    });
}
function unzipWith(array2, iteratee2) {
    if (!(array2 && array2.length)) {
        return [];
    }
    var result2 = unzip(array2);
    if (iteratee2 == null) {
        return result2;
    }
    return arrayMap(result2, function(group) {
        return apply(iteratee2, void 0, group);
    });
}
function baseUpdate(object2, path74, updater, customizer) {
    return baseSet(object2, path74, updater(baseGet(object2, path74)), customizer);
}
function update(object2, path75, updater) {
    return object2 == null ? object2 : baseUpdate(object2, path75, castFunction(updater));
}
function updateWith(object2, path76, updater, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return object2 == null ? object2 : baseUpdate(object2, path76, castFunction(updater), customizer);
}
var upperCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + word.toUpperCase();
});
function valuesIn(object2) {
    return object2 == null ? [] : baseValues(object2, keysIn(object2));
}
var without = baseRest(function(array2, values2) {
    return isArrayLikeObject(array2) ? baseDifference(array2, values2) : [];
});
function wrap(value, wrapper) {
    return partial1(castFunction(wrapper), value);
}
var wrapperAt = flatRest(function(paths) {
    var length = paths.length, start = length ? paths[0] : 0, value = this.__wrapped__, interceptor = function(object2) {
        return baseAt(object2, paths);
    };
    if (length > 1 || this.__actions__.length || !(value instanceof LazyWrapper) || !isIndex(start)) {
        return this.thru(interceptor);
    }
    value = value.slice(start, +start + (length ? 1 : 0));
    value.__actions__.push({
        func: thru,
        args: [
            interceptor
        ],
        thisArg: void 0
    });
    return new LodashWrapper(value, this.__chain__).thru(function(array2) {
        if (length && !array2.length) {
            array2.push(void 0);
        }
        return array2;
    });
});
function wrapperChain() {
    return chain(this);
}
function wrapperReverse() {
    var value = this.__wrapped__;
    if (value instanceof LazyWrapper) {
        var wrapped = value;
        if (this.__actions__.length) {
            wrapped = new LazyWrapper(this);
        }
        wrapped = wrapped.reverse();
        wrapped.__actions__.push({
            func: thru,
            args: [
                reverse
            ],
            thisArg: void 0
        });
        return new LodashWrapper(wrapped, this.__chain__);
    }
    return this.thru(reverse);
}
function baseXor(arrays, iteratee2, comparator) {
    var length = arrays.length;
    if (length < 2) {
        return length ? baseUniq(arrays[0]) : [];
    }
    var index = -1, result2 = Array(length);
    while(++index < length){
        var array2 = arrays[index], othIndex = -1;
        while(++othIndex < length){
            if (othIndex != index) {
                result2[index] = baseDifference(result2[index] || array2, arrays[othIndex], iteratee2, comparator);
            }
        }
    }
    return baseUniq(baseFlatten(result2, 1), iteratee2, comparator);
}
var xor = baseRest(function(arrays) {
    return baseXor(arrayFilter(arrays, isArrayLikeObject));
});
var xorBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return baseXor(arrayFilter(arrays, isArrayLikeObject), baseIteratee(iteratee2));
});
var xorWith = baseRest(function(arrays) {
    var comparator = last(arrays);
    comparator = typeof comparator == "function" ? comparator : void 0;
    return baseXor(arrayFilter(arrays, isArrayLikeObject), void 0, comparator);
});
var zip = baseRest(unzip);
function baseZipObject(props, values2, assignFunc) {
    var index = -1, length = props.length, valsLength = values2.length, result2 = {
    };
    while(++index < length){
        var value = index < valsLength ? values2[index] : void 0;
        assignFunc(result2, props[index], value);
    }
    return result2;
}
function zipObject(props, values2) {
    return baseZipObject(props || [], values2 || [], assignValue);
}
function zipObjectDeep(props, values2) {
    return baseZipObject(props || [], values2 || [], baseSet);
}
var zipWith = baseRest(function(arrays) {
    var length = arrays.length, iteratee2 = length > 1 ? arrays[length - 1] : void 0;
    iteratee2 = typeof iteratee2 == "function" ? (arrays.pop(), iteratee2) : void 0;
    return unzipWith(arrays, iteratee2);
});
var array = {
    chunk,
    compact,
    concat: concat1,
    difference,
    differenceBy,
    differenceWith,
    drop,
    dropRight,
    dropRightWhile,
    dropWhile,
    fill,
    findIndex,
    findLastIndex,
    first: head4,
    flatten,
    flattenDeep,
    flattenDepth,
    fromPairs,
    head: head4,
    indexOf,
    initial,
    intersection,
    intersectionBy,
    intersectionWith,
    join: join4,
    last,
    lastIndexOf,
    nth,
    pull,
    pullAll,
    pullAllBy,
    pullAllWith,
    pullAt,
    remove,
    reverse,
    slice,
    sortedIndex,
    sortedIndexBy,
    sortedIndexOf,
    sortedLastIndex,
    sortedLastIndexBy,
    sortedLastIndexOf,
    sortedUniq,
    sortedUniqBy,
    tail,
    take,
    takeRight,
    takeRightWhile,
    takeWhile,
    union,
    unionBy,
    unionWith,
    uniq,
    uniqBy,
    uniqWith,
    unzip,
    unzipWith,
    without,
    xor,
    xorBy,
    xorWith,
    zip,
    zipObject,
    zipObjectDeep,
    zipWith
};
var collection = {
    countBy,
    each: forEach,
    eachRight: forEachRight,
    every,
    filter,
    find,
    findLast,
    flatMap,
    flatMapDeep,
    flatMapDepth,
    forEach,
    forEachRight,
    groupBy: groupBy1,
    includes,
    invokeMap,
    keyBy,
    map,
    orderBy,
    partition,
    reduce,
    reduceRight,
    reject,
    sample,
    sampleSize,
    shuffle,
    size,
    some,
    sortBy
};
var date = {
    now
};
var func = {
    after,
    ary,
    before,
    bind,
    bindKey,
    curry,
    curryRight,
    debounce,
    defer,
    delay,
    flip,
    memoize,
    negate,
    once,
    overArgs,
    partial: partial1,
    partialRight,
    rearg,
    rest,
    spread,
    throttle,
    unary,
    wrap
};
var lang = {
    castArray,
    clone,
    cloneDeep,
    cloneDeepWith,
    cloneWith,
    conformsTo,
    eq,
    gt,
    gte,
    isArguments,
    isArray,
    isArrayBuffer,
    isArrayLike,
    isArrayLikeObject,
    isBoolean,
    isBuffer,
    isDate,
    isElement,
    isEmpty,
    isEqual,
    isEqualWith,
    isError,
    isFinite,
    isFunction,
    isInteger,
    isLength,
    isMap,
    isMatch,
    isMatchWith,
    isNaN: isNaN1,
    isNative,
    isNil,
    isNull,
    isNumber,
    isObject,
    isObjectLike,
    isPlainObject,
    isRegExp,
    isSafeInteger,
    isSet,
    isString,
    isSymbol,
    isTypedArray,
    isUndefined,
    isWeakMap,
    isWeakSet,
    lt,
    lte,
    toArray,
    toFinite,
    toInteger,
    toLength,
    toNumber,
    toPlainObject,
    toSafeInteger,
    toString
};
var math = {
    add,
    ceil,
    divide,
    floor,
    max,
    maxBy,
    mean,
    meanBy,
    min,
    minBy,
    multiply,
    round,
    subtract,
    sum,
    sumBy
};
var number1 = {
    clamp,
    inRange,
    random
};
var object = {
    assign,
    assignIn,
    assignInWith,
    assignWith,
    at,
    create,
    defaults,
    defaultsDeep,
    entries: toPairs,
    entriesIn: toPairsIn,
    extend: assignIn,
    extendWith: assignInWith,
    findKey,
    findLastKey,
    forIn,
    forInRight,
    forOwn,
    forOwnRight,
    functions,
    functionsIn,
    get,
    has,
    hasIn,
    invert,
    invertBy,
    invoke,
    keys,
    keysIn,
    mapKeys,
    mapValues,
    merge,
    mergeWith,
    omit,
    omitBy,
    pick,
    pickBy,
    result: result36,
    set,
    setWith,
    toPairs,
    toPairsIn,
    transform,
    unset,
    update,
    updateWith,
    values,
    valuesIn
};
var seq = {
    at: wrapperAt,
    chain,
    commit: wrapperCommit,
    lodash,
    next: wrapperNext,
    plant: wrapperPlant,
    reverse: wrapperReverse,
    tap,
    thru,
    toIterator: wrapperToIterator,
    toJSON: wrapperValue,
    value: wrapperValue,
    valueOf: wrapperValue,
    wrapperChain
};
var string1 = {
    camelCase,
    capitalize: capitalize1,
    deburr,
    endsWith,
    escape,
    escapeRegExp,
    kebabCase,
    lowerCase,
    lowerFirst,
    pad,
    padEnd,
    padStart,
    parseInt: parseInt$1,
    repeat,
    replace,
    snakeCase,
    split: split1,
    startCase,
    startsWith,
    template,
    templateSettings,
    toLower,
    toUpper,
    trim,
    trimEnd,
    trimStart,
    truncate,
    unescape: unescape1,
    upperCase,
    upperFirst,
    words
};
var util = {
    attempt,
    bindAll,
    cond,
    conforms,
    constant,
    defaultTo,
    flow,
    flowRight,
    identity,
    iteratee,
    matches,
    matchesProperty,
    method,
    methodOf,
    mixin,
    noop,
    nthArg,
    over,
    overEvery,
    overSome,
    property,
    propertyOf,
    range,
    rangeRight,
    stubArray,
    stubFalse,
    stubObject,
    stubString,
    stubTrue,
    times,
    toPath,
    uniqueId
};
function lazyClone() {
    var result2 = new LazyWrapper(this.__wrapped__);
    result2.__actions__ = copyArray(this.__actions__);
    result2.__dir__ = this.__dir__;
    result2.__filtered__ = this.__filtered__;
    result2.__iteratees__ = copyArray(this.__iteratees__);
    result2.__takeCount__ = this.__takeCount__;
    result2.__views__ = copyArray(this.__views__);
    return result2;
}
function lazyReverse() {
    if (this.__filtered__) {
        var result2 = new LazyWrapper(this);
        result2.__dir__ = -1;
        result2.__filtered__ = true;
    } else {
        result2 = this.clone();
        result2.__dir__ *= -1;
    }
    return result2;
}
var nativeMax$f = Math.max, nativeMin$c = Math.min;
function getView(start, end, transforms) {
    var index = -1, length = transforms.length;
    while(++index < length){
        var data23 = transforms[index], size2 = data23.size;
        switch(data23.type){
            case "drop":
                start += size2;
                break;
            case "dropRight":
                end -= size2;
                break;
            case "take":
                end = nativeMin$c(end, start + size2);
                break;
            case "takeRight":
                start = nativeMax$f(start, end - size2);
                break;
        }
    }
    return {
        start,
        end
    };
}
var LAZY_FILTER_FLAG = 1, LAZY_MAP_FLAG = 2;
var nativeMin$d = Math.min;
function lazyValue() {
    var array2 = this.__wrapped__.value(), dir = this.__dir__, isArr = isArray(array2), isRight = dir < 0, arrLength = isArr ? array2.length : 0, view = getView(0, arrLength, this.__views__), start = view.start, end = view.end, length = end - start, index = isRight ? end : start - 1, iteratees = this.__iteratees__, iterLength = iteratees.length, resIndex = 0, takeCount = nativeMin$d(length, this.__takeCount__);
    if (!isArr || !isRight && arrLength == length && takeCount == length) {
        return baseWrapperValue(array2, this.__actions__);
    }
    var result2 = [];
    outer: while((length--) && resIndex < takeCount){
        index += dir;
        var iterIndex = -1, value = array2[index];
        while(++iterIndex < iterLength){
            var data24 = iteratees[iterIndex], iteratee2 = data24.iteratee, type29 = data24.type, computed = iteratee2(value);
            if (type29 == LAZY_MAP_FLAG) {
                value = computed;
            } else if (!computed) {
                if (type29 == LAZY_FILTER_FLAG) {
                    continue outer;
                } else {
                    break outer;
                }
            }
        }
        result2[resIndex++] = value;
    }
    return result2;
}
var VERSION = "4.17.21";
var WRAP_BIND_KEY_FLAG$6 = 2;
var LAZY_FILTER_FLAG$1 = 1, LAZY_WHILE_FLAG = 3;
var MAX_ARRAY_LENGTH$6 = 4294967295;
var arrayProto$5 = Array.prototype, objectProto$s = Object.prototype;
var hasOwnProperty$o = objectProto$s.hasOwnProperty;
var symIterator$1 = Symbol1 ? Symbol1.iterator : void 0;
var nativeMax$g = Math.max, nativeMin$e = Math.min;
var mixin$1 = function(func2) {
    return function(object2, source, options) {
        if (options == null) {
            var isObj = isObject(source), props = isObj && keys(source), methodNames3 = props && props.length && baseFunctions(source, props);
            if (!(methodNames3 ? methodNames3.length : isObj)) {
                options = source;
                source = object2;
                object2 = this;
            }
        }
        return func2(object2, source, options);
    };
}(mixin);
lodash.after = func.after;
lodash.ary = func.ary;
lodash.assign = object.assign;
lodash.assignIn = object.assignIn;
lodash.assignInWith = object.assignInWith;
lodash.assignWith = object.assignWith;
lodash.at = object.at;
lodash.before = func.before;
lodash.bind = func.bind;
lodash.bindAll = util.bindAll;
lodash.bindKey = func.bindKey;
lodash.castArray = lang.castArray;
lodash.chain = seq.chain;
lodash.chunk = array.chunk;
lodash.compact = array.compact;
lodash.concat = array.concat;
lodash.cond = util.cond;
lodash.conforms = util.conforms;
lodash.constant = util.constant;
lodash.countBy = collection.countBy;
lodash.create = object.create;
lodash.curry = func.curry;
lodash.curryRight = func.curryRight;
lodash.debounce = func.debounce;
lodash.defaults = object.defaults;
lodash.defaultsDeep = object.defaultsDeep;
lodash.defer = func.defer;
lodash.delay = func.delay;
lodash.difference = array.difference;
lodash.differenceBy = array.differenceBy;
lodash.differenceWith = array.differenceWith;
lodash.drop = array.drop;
lodash.dropRight = array.dropRight;
lodash.dropRightWhile = array.dropRightWhile;
lodash.dropWhile = array.dropWhile;
lodash.fill = array.fill;
lodash.filter = collection.filter;
lodash.flatMap = collection.flatMap;
lodash.flatMapDeep = collection.flatMapDeep;
lodash.flatMapDepth = collection.flatMapDepth;
lodash.flatten = array.flatten;
lodash.flattenDeep = array.flattenDeep;
lodash.flattenDepth = array.flattenDepth;
lodash.flip = func.flip;
lodash.flow = util.flow;
lodash.flowRight = util.flowRight;
lodash.fromPairs = array.fromPairs;
lodash.functions = object.functions;
lodash.functionsIn = object.functionsIn;
lodash.groupBy = collection.groupBy;
lodash.initial = array.initial;
lodash.intersection = array.intersection;
lodash.intersectionBy = array.intersectionBy;
lodash.intersectionWith = array.intersectionWith;
lodash.invert = object.invert;
lodash.invertBy = object.invertBy;
lodash.invokeMap = collection.invokeMap;
lodash.iteratee = util.iteratee;
lodash.keyBy = collection.keyBy;
lodash.keys = keys;
lodash.keysIn = object.keysIn;
lodash.map = collection.map;
lodash.mapKeys = object.mapKeys;
lodash.mapValues = object.mapValues;
lodash.matches = util.matches;
lodash.matchesProperty = util.matchesProperty;
lodash.memoize = func.memoize;
lodash.merge = object.merge;
lodash.mergeWith = object.mergeWith;
lodash.method = util.method;
lodash.methodOf = util.methodOf;
lodash.mixin = mixin$1;
lodash.negate = negate;
lodash.nthArg = util.nthArg;
lodash.omit = object.omit;
lodash.omitBy = object.omitBy;
lodash.once = func.once;
lodash.orderBy = collection.orderBy;
lodash.over = util.over;
lodash.overArgs = func.overArgs;
lodash.overEvery = util.overEvery;
lodash.overSome = util.overSome;
lodash.partial = func.partial;
lodash.partialRight = func.partialRight;
lodash.partition = collection.partition;
lodash.pick = object.pick;
lodash.pickBy = object.pickBy;
lodash.property = util.property;
lodash.propertyOf = util.propertyOf;
lodash.pull = array.pull;
lodash.pullAll = array.pullAll;
lodash.pullAllBy = array.pullAllBy;
lodash.pullAllWith = array.pullAllWith;
lodash.pullAt = array.pullAt;
lodash.range = util.range;
lodash.rangeRight = util.rangeRight;
lodash.rearg = func.rearg;
lodash.reject = collection.reject;
lodash.remove = array.remove;
lodash.rest = func.rest;
lodash.reverse = array.reverse;
lodash.sampleSize = collection.sampleSize;
lodash.set = object.set;
lodash.setWith = object.setWith;
lodash.shuffle = collection.shuffle;
lodash.slice = array.slice;
lodash.sortBy = collection.sortBy;
lodash.sortedUniq = array.sortedUniq;
lodash.sortedUniqBy = array.sortedUniqBy;
lodash.split = string1.split;
lodash.spread = func.spread;
lodash.tail = array.tail;
lodash.take = array.take;
lodash.takeRight = array.takeRight;
lodash.takeRightWhile = array.takeRightWhile;
lodash.takeWhile = array.takeWhile;
lodash.tap = seq.tap;
lodash.throttle = func.throttle;
lodash.thru = thru;
lodash.toArray = lang.toArray;
lodash.toPairs = object.toPairs;
lodash.toPairsIn = object.toPairsIn;
lodash.toPath = util.toPath;
lodash.toPlainObject = lang.toPlainObject;
lodash.transform = object.transform;
lodash.unary = func.unary;
lodash.union = array.union;
lodash.unionBy = array.unionBy;
lodash.unionWith = array.unionWith;
lodash.uniq = array.uniq;
lodash.uniqBy = array.uniqBy;
lodash.uniqWith = array.uniqWith;
lodash.unset = object.unset;
lodash.unzip = array.unzip;
lodash.unzipWith = array.unzipWith;
lodash.update = object.update;
lodash.updateWith = object.updateWith;
lodash.values = object.values;
lodash.valuesIn = object.valuesIn;
lodash.without = array.without;
lodash.words = string1.words;
lodash.wrap = func.wrap;
lodash.xor = array.xor;
lodash.xorBy = array.xorBy;
lodash.xorWith = array.xorWith;
lodash.zip = array.zip;
lodash.zipObject = array.zipObject;
lodash.zipObjectDeep = array.zipObjectDeep;
lodash.zipWith = array.zipWith;
lodash.entries = object.toPairs;
lodash.entriesIn = object.toPairsIn;
lodash.extend = object.assignIn;
lodash.extendWith = object.assignInWith;
mixin$1(lodash, lodash);
lodash.add = math.add;
lodash.attempt = util.attempt;
lodash.camelCase = string1.camelCase;
lodash.capitalize = string1.capitalize;
lodash.ceil = math.ceil;
lodash.clamp = number1.clamp;
lodash.clone = lang.clone;
lodash.cloneDeep = lang.cloneDeep;
lodash.cloneDeepWith = lang.cloneDeepWith;
lodash.cloneWith = lang.cloneWith;
lodash.conformsTo = lang.conformsTo;
lodash.deburr = string1.deburr;
lodash.defaultTo = util.defaultTo;
lodash.divide = math.divide;
lodash.endsWith = string1.endsWith;
lodash.eq = lang.eq;
lodash.escape = string1.escape;
lodash.escapeRegExp = string1.escapeRegExp;
lodash.every = collection.every;
lodash.find = collection.find;
lodash.findIndex = array.findIndex;
lodash.findKey = object.findKey;
lodash.findLast = collection.findLast;
lodash.findLastIndex = array.findLastIndex;
lodash.findLastKey = object.findLastKey;
lodash.floor = math.floor;
lodash.forEach = collection.forEach;
lodash.forEachRight = collection.forEachRight;
lodash.forIn = object.forIn;
lodash.forInRight = object.forInRight;
lodash.forOwn = object.forOwn;
lodash.forOwnRight = object.forOwnRight;
lodash.get = object.get;
lodash.gt = lang.gt;
lodash.gte = lang.gte;
lodash.has = object.has;
lodash.hasIn = object.hasIn;
lodash.head = array.head;
lodash.identity = identity;
lodash.includes = collection.includes;
lodash.indexOf = array.indexOf;
lodash.inRange = number1.inRange;
lodash.invoke = object.invoke;
lodash.isArguments = lang.isArguments;
lodash.isArray = isArray;
lodash.isArrayBuffer = lang.isArrayBuffer;
lodash.isArrayLike = lang.isArrayLike;
lodash.isArrayLikeObject = lang.isArrayLikeObject;
lodash.isBoolean = lang.isBoolean;
lodash.isBuffer = lang.isBuffer;
lodash.isDate = lang.isDate;
lodash.isElement = lang.isElement;
lodash.isEmpty = lang.isEmpty;
lodash.isEqual = lang.isEqual;
lodash.isEqualWith = lang.isEqualWith;
lodash.isError = lang.isError;
lodash.isFinite = lang.isFinite;
lodash.isFunction = lang.isFunction;
lodash.isInteger = lang.isInteger;
lodash.isLength = lang.isLength;
lodash.isMap = lang.isMap;
lodash.isMatch = lang.isMatch;
lodash.isMatchWith = lang.isMatchWith;
lodash.isNaN = lang.isNaN;
lodash.isNative = lang.isNative;
lodash.isNil = lang.isNil;
lodash.isNull = lang.isNull;
lodash.isNumber = lang.isNumber;
lodash.isObject = isObject;
lodash.isObjectLike = lang.isObjectLike;
lodash.isPlainObject = lang.isPlainObject;
lodash.isRegExp = lang.isRegExp;
lodash.isSafeInteger = lang.isSafeInteger;
lodash.isSet = lang.isSet;
lodash.isString = lang.isString;
lodash.isSymbol = lang.isSymbol;
lodash.isTypedArray = lang.isTypedArray;
lodash.isUndefined = lang.isUndefined;
lodash.isWeakMap = lang.isWeakMap;
lodash.isWeakSet = lang.isWeakSet;
lodash.join = array.join;
lodash.kebabCase = string1.kebabCase;
lodash.last = last;
lodash.lastIndexOf = array.lastIndexOf;
lodash.lowerCase = string1.lowerCase;
lodash.lowerFirst = string1.lowerFirst;
lodash.lt = lang.lt;
lodash.lte = lang.lte;
lodash.max = math.max;
lodash.maxBy = math.maxBy;
lodash.mean = math.mean;
lodash.meanBy = math.meanBy;
lodash.min = math.min;
lodash.minBy = math.minBy;
lodash.stubArray = util.stubArray;
lodash.stubFalse = util.stubFalse;
lodash.stubObject = util.stubObject;
lodash.stubString = util.stubString;
lodash.stubTrue = util.stubTrue;
lodash.multiply = math.multiply;
lodash.nth = array.nth;
lodash.noop = util.noop;
lodash.now = date.now;
lodash.pad = string1.pad;
lodash.padEnd = string1.padEnd;
lodash.padStart = string1.padStart;
lodash.parseInt = string1.parseInt;
lodash.random = number1.random;
lodash.reduce = collection.reduce;
lodash.reduceRight = collection.reduceRight;
lodash.repeat = string1.repeat;
lodash.replace = string1.replace;
lodash.result = object.result;
lodash.round = math.round;
lodash.sample = collection.sample;
lodash.size = collection.size;
lodash.snakeCase = string1.snakeCase;
lodash.some = collection.some;
lodash.sortedIndex = array.sortedIndex;
lodash.sortedIndexBy = array.sortedIndexBy;
lodash.sortedIndexOf = array.sortedIndexOf;
lodash.sortedLastIndex = array.sortedLastIndex;
lodash.sortedLastIndexBy = array.sortedLastIndexBy;
lodash.sortedLastIndexOf = array.sortedLastIndexOf;
lodash.startCase = string1.startCase;
lodash.startsWith = string1.startsWith;
lodash.subtract = math.subtract;
lodash.sum = math.sum;
lodash.sumBy = math.sumBy;
lodash.template = string1.template;
lodash.times = util.times;
lodash.toFinite = lang.toFinite;
lodash.toInteger = toInteger;
lodash.toLength = lang.toLength;
lodash.toLower = string1.toLower;
lodash.toNumber = lang.toNumber;
lodash.toSafeInteger = lang.toSafeInteger;
lodash.toString = lang.toString;
lodash.toUpper = string1.toUpper;
lodash.trim = string1.trim;
lodash.trimEnd = string1.trimEnd;
lodash.trimStart = string1.trimStart;
lodash.truncate = string1.truncate;
lodash.unescape = string1.unescape;
lodash.uniqueId = util.uniqueId;
lodash.upperCase = string1.upperCase;
lodash.upperFirst = string1.upperFirst;
lodash.each = collection.forEach;
lodash.eachRight = collection.forEachRight;
lodash.first = array.head;
mixin$1(lodash, function() {
    var source = {
    };
    baseForOwn(lodash, function(func2, methodName) {
        if (!hasOwnProperty$o.call(lodash.prototype, methodName)) {
            source[methodName] = func2;
        }
    });
    return source;
}(), {
    chain: false
});
lodash.VERSION = VERSION;
(lodash.templateSettings = string1.templateSettings).imports._ = lodash;
arrayEach([
    "bind",
    "bindKey",
    "curry",
    "curryRight",
    "partial",
    "partialRight"
], function(methodName) {
    lodash[methodName].placeholder = lodash;
});
arrayEach([
    "drop",
    "take"
], function(methodName, index) {
    LazyWrapper.prototype[methodName] = function(n20) {
        n20 = n20 === void 0 ? 1 : nativeMax$g(toInteger(n20), 0);
        var result2 = this.__filtered__ && !index ? new LazyWrapper(this) : this.clone();
        if (result2.__filtered__) {
            result2.__takeCount__ = nativeMin$e(n20, result2.__takeCount__);
        } else {
            result2.__views__.push({
                size: nativeMin$e(n20, MAX_ARRAY_LENGTH$6),
                type: methodName + (result2.__dir__ < 0 ? "Right" : "")
            });
        }
        return result2;
    };
    LazyWrapper.prototype[methodName + "Right"] = function(n21) {
        return this.reverse()[methodName](n21).reverse();
    };
});
arrayEach([
    "filter",
    "map",
    "takeWhile"
], function(methodName, index) {
    var type30 = index + 1, isFilter = type30 == LAZY_FILTER_FLAG$1 || type30 == LAZY_WHILE_FLAG;
    LazyWrapper.prototype[methodName] = function(iteratee2) {
        var result2 = this.clone();
        result2.__iteratees__.push({
            iteratee: baseIteratee(iteratee2),
            type: type30
        });
        result2.__filtered__ = result2.__filtered__ || isFilter;
        return result2;
    };
});
arrayEach([
    "head",
    "last"
], function(methodName, index) {
    var takeName = "take" + (index ? "Right" : "");
    LazyWrapper.prototype[methodName] = function() {
        return this[takeName](1).value()[0];
    };
});
arrayEach([
    "initial",
    "tail"
], function(methodName, index) {
    var dropName = "drop" + (index ? "" : "Right");
    LazyWrapper.prototype[methodName] = function() {
        return this.__filtered__ ? new LazyWrapper(this) : this[dropName](1);
    };
});
LazyWrapper.prototype.compact = function() {
    return this.filter(identity);
};
LazyWrapper.prototype.find = function(predicate) {
    return this.filter(predicate).head();
};
LazyWrapper.prototype.findLast = function(predicate) {
    return this.reverse().find(predicate);
};
LazyWrapper.prototype.invokeMap = baseRest(function(path77, args) {
    if (typeof path77 == "function") {
        return new LazyWrapper(this);
    }
    return this.map(function(value) {
        return baseInvoke(value, path77, args);
    });
});
LazyWrapper.prototype.reject = function(predicate) {
    return this.filter(negate(baseIteratee(predicate)));
};
LazyWrapper.prototype.slice = function(start, end) {
    start = toInteger(start);
    var result2 = this;
    if (result2.__filtered__ && (start > 0 || end < 0)) {
        return new LazyWrapper(result2);
    }
    if (start < 0) {
        result2 = result2.takeRight(-start);
    } else if (start) {
        result2 = result2.drop(start);
    }
    if (end !== void 0) {
        end = toInteger(end);
        result2 = end < 0 ? result2.dropRight(-end) : result2.take(end - start);
    }
    return result2;
};
LazyWrapper.prototype.takeRightWhile = function(predicate) {
    return this.reverse().takeWhile(predicate).reverse();
};
LazyWrapper.prototype.toArray = function() {
    return this.take(MAX_ARRAY_LENGTH$6);
};
baseForOwn(LazyWrapper.prototype, function(func2, methodName) {
    var checkIteratee = /^(?:filter|find|map|reject)|While$/.test(methodName), isTaker = /^(?:head|last)$/.test(methodName), lodashFunc = lodash[isTaker ? "take" + (methodName == "last" ? "Right" : "") : methodName], retUnwrapped = isTaker || /^find/.test(methodName);
    if (!lodashFunc) {
        return;
    }
    lodash.prototype[methodName] = function() {
        var value = this.__wrapped__, args = isTaker ? [
            1
        ] : arguments, isLazy = value instanceof LazyWrapper, iteratee2 = args[0], useLazy = isLazy || isArray(value);
        var interceptor = function(value2) {
            var result3 = lodashFunc.apply(lodash, arrayPush([
                value2
            ], args));
            return isTaker && chainAll ? result3[0] : result3;
        };
        if (useLazy && checkIteratee && typeof iteratee2 == "function" && iteratee2.length != 1) {
            isLazy = useLazy = false;
        }
        var chainAll = this.__chain__, isHybrid = !!this.__actions__.length, isUnwrapped = retUnwrapped && !chainAll, onlyLazy = isLazy && !isHybrid;
        if (!retUnwrapped && useLazy) {
            value = onlyLazy ? value : new LazyWrapper(this);
            var result2 = func2.apply(value, args);
            result2.__actions__.push({
                func: thru,
                args: [
                    interceptor
                ],
                thisArg: void 0
            });
            return new LodashWrapper(result2, chainAll);
        }
        if (isUnwrapped && onlyLazy) {
            return func2.apply(this, args);
        }
        result2 = this.thru(interceptor);
        return isUnwrapped ? isTaker ? result2.value()[0] : result2.value() : result2;
    };
});
arrayEach([
    "pop",
    "push",
    "shift",
    "sort",
    "splice",
    "unshift"
], function(methodName) {
    var func2 = arrayProto$5[methodName], chainName = /^(?:push|sort|unshift)$/.test(methodName) ? "tap" : "thru", retUnwrapped = /^(?:pop|shift)$/.test(methodName);
    lodash.prototype[methodName] = function() {
        var args = arguments;
        if (retUnwrapped && !this.__chain__) {
            var value = this.value();
            return func2.apply(isArray(value) ? value : [], args);
        }
        return this[chainName](function(value2) {
            return func2.apply(isArray(value2) ? value2 : [], args);
        });
    };
});
baseForOwn(LazyWrapper.prototype, function(func2, methodName) {
    var lodashFunc = lodash[methodName];
    if (lodashFunc) {
        var key = lodashFunc.name + "";
        if (!hasOwnProperty$o.call(realNames, key)) {
            realNames[key] = [];
        }
        realNames[key].push({
            name: methodName,
            func: lodashFunc
        });
    }
});
realNames[createHybrid(void 0, WRAP_BIND_KEY_FLAG$6).name] = [
    {
        name: "wrapper",
        func: void 0
    }
];
LazyWrapper.prototype.clone = lazyClone;
LazyWrapper.prototype.reverse = lazyReverse;
LazyWrapper.prototype.value = lazyValue;
lodash.prototype.at = seq.at;
lodash.prototype.chain = seq.wrapperChain;
lodash.prototype.commit = seq.commit;
lodash.prototype.next = seq.next;
lodash.prototype.plant = seq.plant;
lodash.prototype.reverse = seq.reverse;
lodash.prototype.toJSON = lodash.prototype.valueOf = lodash.prototype.value = seq.value;
lodash.prototype.first = lodash.prototype.head;
if (symIterator$1) {
    lodash.prototype[symIterator$1] = seq.toIterator;
}
const ld = lodash;
function deferred() {
    let methods;
    let state1 = "pending";
    const promise = new Promise((resolve8, reject2)=>{
        methods = {
            async resolve (value) {
                await value;
                state1 = "fulfilled";
                resolve8(value);
            },
            reject (reason) {
                state1 = "rejected";
                reject2(reason);
            }
        };
    });
    Object.defineProperty(promise, "state", {
        get: ()=>state1
    });
    return Object.assign(promise, methods);
}
class MuxAsyncIterator {
    iteratorCount = 0;
    yields = [];
    throws = [];
    signal = deferred();
    add(iterable) {
        ++this.iteratorCount;
        this.callIteratorNext(iterable[Symbol.asyncIterator]());
    }
    async callIteratorNext(iterator) {
        try {
            const { value , done  } = await iterator.next();
            if (done) {
                --this.iteratorCount;
            } else {
                this.yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.throws.push(e);
        }
        this.signal.resolve();
    }
    async *iterate() {
        while(this.iteratorCount > 0){
            await this.signal;
            for(let i26 = 0; i26 < this.yields.length; i26++){
                const { iterator , value  } = this.yields[i26];
                yield value;
                this.callIteratorNext(iterator);
            }
            if (this.throws.length) {
                for (const e of this.throws){
                    throw e;
                }
                this.throws.length = 0;
            }
            this.yields.length = 0;
            this.signal = deferred();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
function pooledMap(poolLimit, array2, iteratorFn) {
    const res = new TransformStream({
        async transform (p, controller) {
            controller.enqueue(await p);
        }
    });
    (async ()=>{
        const writer = res.writable.getWriter();
        const executing = [];
        try {
            for await (const item of array2){
                const p = Promise.resolve().then(()=>iteratorFn(item)
                );
                p.then((v)=>writer.write(Promise.resolve(v))
                ).catch(()=>{
                });
                const e = p.then(()=>executing.splice(executing.indexOf(e), 1)
                );
                executing.push(e);
                if (executing.length >= poolLimit) {
                    await Promise.race(executing);
                }
            }
            await Promise.all(executing);
            writer.close();
        } catch  {
            const errors4 = [];
            for (const result25 of (await Promise.allSettled(executing))){
                if (result25.status == "rejected") {
                    errors4.push(result25.reason);
                }
            }
            writer.write(Promise.reject(new AggregateError(errors4, "Threw while mapping."))).catch(()=>{
            });
        }
    })();
    return res.readable[Symbol.asyncIterator]();
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
function readAllSync(r) {
    const buf = new Buffer();
    buf.readFromSync(r);
    return buf.bytes();
}
async function writeAll(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += await w.write(arr.subarray(nwritten));
    }
}
function writeAllSync(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += w.writeSync(arr.subarray(nwritten));
    }
}
async function* iterateReader(r, options) {
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result26 = await r.read(b);
        if (result26 === null) {
            break;
        }
        yield b.subarray(0, result26);
    }
}
"\r".charCodeAt(0);
"\n".charCodeAt(0);
async function execProcess(options, stdin, mergeOutput, stderrFilter) {
    try {
        const process = Deno.run({
            ...options,
            stdin: stdin !== undefined ? "piped" : options.stdin,
            stdout: typeof options.stdout === "number" ? options.stdout : "piped",
            stderr: typeof options.stderr === "number" ? options.stderr : "piped"
        });
        if (stdin !== undefined) {
            if (!process.stdin) {
                throw new Error("Process stdin not available");
            }
            const buffer = new TextEncoder().encode(stdin);
            let offset = 0;
            while(offset < buffer.length){
                const end = Math.min(offset + 4096, buffer.length);
                const window = buffer.subarray(offset, end);
                const written = await process.stdin.write(window);
                offset += written;
            }
            process.stdin.close();
        }
        let stdoutText = "";
        let stderrText = "";
        if (mergeOutput) {
            const multiplexIterator = new MuxAsyncIterator();
            const addStream = (stream, filter2)=>{
                if (stream !== null) {
                    const streamIter = filter2 ? filteredAsyncIterator(iterateReader(stream), filter2) : iterateReader(stream);
                    multiplexIterator.add(streamIter);
                }
            };
            addStream(process.stdout);
            addStream(process.stderr, stderrFilter);
            const allOutput = await processOutput(multiplexIterator, mergeOutput === "stderr>stdout" ? options.stdout : options.stderr);
            if (mergeOutput === "stderr>stdout") {
                stdoutText = allOutput;
            } else {
                stderrText = allOutput;
            }
            const closeStream = (stream)=>{
                if (stream) {
                    stream.close();
                }
            };
            closeStream(process.stdout);
            closeStream(process.stderr);
        } else {
            if (process.stdout !== null) {
                stdoutText = await processOutput(iterateReader(process.stdout), options.stdout);
                process.stdout.close();
            }
            if (process.stderr != null) {
                const iterator = stderrFilter ? filteredAsyncIterator(iterateReader(process.stderr), stderrFilter) : iterateReader(process.stderr);
                stderrText = await processOutput(iterator, options.stderr);
                process.stderr.close();
            }
        }
        const status = await process.status();
        process.close();
        return {
            success: status.success,
            code: status.code,
            stdout: stdoutText,
            stderr: stderrText
        };
    } catch (e) {
        throw new Error(`Error executing '${options.cmd[0]}': ${e.message}`);
    }
}
function processSuccessResult() {
    return {
        success: true,
        code: 0
    };
}
function filteredAsyncIterator(iterator, filter3) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    return pooledMap(1, iterator, (data25)=>{
        return Promise.resolve(encoder.encode(filter3(decoder.decode(data25))));
    });
}
async function processOutput(iterator, output) {
    const decoder = new TextDecoder();
    let outputText = "";
    for await (const chunk1 of iterator){
        if (output === "inherit" || output === undefined) {
            info(decoder.decode(chunk1), {
                newline: false
            });
        }
        const text = decoder.decode(chunk1);
        outputText += text;
    }
    return outputText;
}
const kSkipHidden = /[/\\][\.]/;
function removeIfExists(file) {
    if (existsSync(file)) {
        Deno.removeSync(file, {
            recursive: true
        });
    }
}
function removeIfEmptyDir(dir) {
    if (existsSync(dir)) {
        let empty1 = true;
        for (const _entry of Deno.readDirSync(dir)){
            empty1 = false;
            break;
        }
        if (empty1) {
            Deno.removeSync(dir, {
                recursive: true
            });
        }
    }
}
function isModifiedAfter(file, otherFile) {
    if (!existsSync(file)) {
        throw new Error(`${file} does not exist`);
    }
    if (!existsSync(otherFile)) {
        return true;
    }
    const fileInfo = Deno.statSync(file);
    const otherfileInfo = Deno.statSync(otherFile);
    if (fileInfo.mtime === null || otherfileInfo.mtime === null) {
        return true;
    } else {
        return fileInfo.mtime > otherfileInfo.mtime;
    }
}
function dirAndStem(file) {
    return [
        dirname2(file),
        basename2(file, extname2(file)), 
    ];
}
function expandPath(path78) {
    if (path78 === "~") {
        return getenv("HOME", "~");
    } else {
        return path78.replace(/^~\//, getenv("HOME", "~") + "/");
    }
}
function safeExistsSync(path79) {
    try {
        return existsSync(path79);
    } catch  {
        return false;
    }
}
async function which(cmd) {
    const args = Deno.build.os === "windows" ? [
        "CMD",
        "/C",
        "where",
        cmd
    ] : [
        "which",
        cmd
    ];
    const result27 = await execProcess({
        cmd: args,
        stderr: "piped",
        stdout: "piped"
    });
    if (result27.code === 0) {
        return result27.stdout?.trim();
    } else {
        return undefined;
    }
}
function resolvePathGlobs(root4, globs, exclude) {
    const includeGlobs = [];
    const excludeGlobs = [];
    const asFullGlob = (glob)=>{
        if (glob.startsWith("\\!")) {
            glob = glob.slice(1);
        }
        if (glob.endsWith("/")) {
            glob = glob + "**/*";
        }
        if (!glob.startsWith("/")) {
            return "**/" + glob;
        } else {
            return join3(root4, glob.slice(1));
        }
    };
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            for (const file of expandGlobSync(glob, {
                root: root4,
                exclude,
                includeDirs: true,
                extended: true,
                globstar: true
            })){
                expanded.push(file.path);
            }
        }
        return lodash.uniq(expanded);
    };
    for (const glob1 of globs){
        if (glob1.startsWith("!")) {
            excludeGlobs.push(asFullGlob(glob1.slice(1)));
        } else {
            includeGlobs.push(asFullGlob(glob1));
        }
    }
    const includeFiles = expandGlobs(includeGlobs);
    const excludeFiles = expandGlobs(excludeGlobs);
    return {
        include: includeFiles,
        exclude: excludeFiles
    };
}
function pathWithForwardSlashes(path80) {
    return path80.replace(/\\/g, "/");
}
function copyMinimal(srcDir, destDir, skip, filter4) {
    skip = skip || [];
    const srcFiles = [];
    for (const walk1 of walkSync(srcDir, {
        includeDirs: false,
        followSymlinks: false,
        skip
    })){
        const srcFile = walk1.path;
        if (filter4 && !filter4(srcFile)) {
            continue;
        }
        srcFiles.push(srcFile);
    }
    for (const srcFile of srcFiles){
        if (!existsSync(srcFile)) {
            continue;
        }
        const destFile = join3(destDir, relative2(srcDir, srcFile));
        copyFileIfNewer(srcFile, destFile);
    }
}
function copyFileIfNewer(srcFile, destFile) {
    ensureDirSync(dirname2(destFile));
    if (existsSync(destFile)) {
        const srcInfo = Deno.statSync(srcFile);
        const destInfo = Deno.statSync(destFile);
        if (!srcInfo.mtime || !destInfo.mtime || destInfo.mtime < srcInfo.mtime) {
            copySync(srcFile, destFile, {
                overwrite: true,
                preserveTimestamps: true
            });
        }
    } else {
        copySync(srcFile, destFile, {
            overwrite: true,
            preserveTimestamps: true
        });
    }
}
const kLocalDevelopment = "99.9.9";
const quartoConfig = {
    binPath: ()=>getenv("QUARTO_BIN_PATH")
    ,
    sharePath: ()=>getenv("QUARTO_SHARE_PATH")
    ,
    isDebug: ()=>getenv("QUARTO_DEBUG", "false") === "true"
    ,
    version: ()=>{
        const versionPath = join3(getenv("QUARTO_SHARE_PATH"), "version");
        if (existsSync(versionPath)) {
            return Deno.readTextFileSync(versionPath);
        } else {
            return kLocalDevelopment;
        }
    }
};
const kHKeyCurrentUser = "HKCU";
const kHKeyLocalMachine = "HKLM";
async function registryReadString(root5, key, value) {
    if (Array.isArray(root5)) {
        for (const r of root5){
            const val = registryReadString(r, key, value);
            if (val !== undefined) {
                return val;
            }
        }
        return undefined;
    }
    const kTypeString = "REG_SZ";
    const cmd = [
        "reg",
        "query",
        `${root5}\\${key}`,
        "/v",
        value, 
    ];
    const result28 = await execProcess({
        cmd,
        stdout: "piped"
    });
    if (result28.success && result28.stdout) {
        const typePos = result28.stdout?.indexOf(kTypeString);
        if (typePos !== -1) {
            return result28.stdout.substring(typePos + kTypeString.length).trim();
        }
    }
    return undefined;
}
function resourcePath(resource) {
    const sharePath = quartoConfig.sharePath();
    if (resource) {
        return join3(sharePath, resource);
    } else {
        return sharePath;
    }
}
function formatResourcePath(format8, resource) {
    return join3(resourcePath("formats"), format8, resource);
}
function binaryPath(binary1) {
    return join3(quartoConfig.binPath(), binary1);
}
function pandocBinaryPath() {
    const quartoPandoc = Deno.env.get("QUARTO_PANDOC");
    if (quartoPandoc) {
        if (!existsSync(quartoPandoc)) {
            warnOnce("Specified QUARTO_PANDOC does not exist, using built in Pandoc");
        }
        if (Deno.statSync(quartoPandoc).isFile) {
            return quartoPandoc;
        } else {
            return join3(quartoPandoc, "pandoc");
        }
    }
    return binaryPath("pandoc");
}
async function rBinaryPath(binary2) {
    const rHome = Deno.env.get("R_HOME");
    if (rHome) {
        return join3(rHome, "bin", binary2);
    }
    const path81 = await which(binary2);
    if (path81) {
        return path81;
    }
    if (Deno.build.os === "windows") {
        const version6 = await registryReadString([
            kHKeyLocalMachine,
            kHKeyCurrentUser
        ], "Software\\R-core\\R", "Current Version");
        if (version6) {
            const installPath = await registryReadString([
                kHKeyLocalMachine,
                kHKeyCurrentUser
            ], `Software\\R-core\\R\\${version6}`, "InstallPath");
            if (installPath) {
                return join3(installPath, "bin", binary2);
            }
        }
        const progFiles = Deno.env.get("programfiles");
        if (progFiles) {
            for (const entry of Deno.readDirSync(progFiles)){
                if (entry.isDirectory && entry.name === "R") {
                    for (const walk2 of walkSync(join3(progFiles, "R"))){
                        if (walk2.isDirectory && walk2.name === "bin") {
                            return join3(walk2.path, binary2);
                        }
                    }
                }
            }
        }
    }
    return binary2;
}
function projectTypeResourcePath(projectType1) {
    return resourcePath(join3("projects", projectType1));
}
const kDarkSuffix = "dark";
const kLightSuffix = "light";
function textHighlightThemePath(theme, style) {
    const names = [
        `${theme}-${style === "dark" ? kDarkSuffix : kLightSuffix}`,
        theme, 
    ];
    const themePath = names.map((name)=>{
        return resourcePath(join3("pandoc", "highlight-styles", `${name}.theme`));
    }).find((path82)=>existsSync(path82)
    );
    return themePath;
}
class YAMLError extends Error {
    mark;
    constructor(message = "(unknown reason)", mark = ""){
        super(`${message} ${mark}`);
        this.mark = mark;
        this.name = this.constructor.name;
    }
    toString(_compact) {
        return `${this.name}: ${this.message} ${this.mark}`;
    }
}
function isBoolean1(value) {
    return typeof value === "boolean" || value instanceof Boolean;
}
function isObject1(value) {
    return value !== null && typeof value === "object";
}
function repeat1(str65, count) {
    let result29 = "";
    for(let cycle = 0; cycle < count; cycle++){
        result29 += str65;
    }
    return result29;
}
function isNegativeZero(i27) {
    return i27 === 0 && Number.NEGATIVE_INFINITY === 1 / i27;
}
class Mark {
    name;
    buffer;
    position;
    line;
    column;
    constructor(name, buffer, position, line, column){
        this.name = name;
        this.buffer = buffer;
        this.position = position;
        this.line = line;
        this.column = column;
    }
    getSnippet(indent = 4, maxLength = 75) {
        if (!this.buffer) return null;
        let head1 = "";
        let start = this.position;
        while(start > 0 && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(start - 1)) === -1){
            start -= 1;
            if (this.position - start > maxLength / 2 - 1) {
                head1 = " ... ";
                start += 5;
                break;
            }
        }
        let tail2 = "";
        let end = this.position;
        while(end < this.buffer.length && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(end)) === -1){
            end += 1;
            if (end - this.position > maxLength / 2 - 1) {
                tail2 = " ... ";
                end -= 5;
                break;
            }
        }
        const snippet = this.buffer.slice(start, end);
        return `${repeat1(" ", indent)}${head1}${snippet}${tail2}\n${repeat1(" ", indent + this.position - start + head1.length)}^`;
    }
    toString(compact1) {
        let snippet, where = "";
        if (this.name) {
            where += `in "${this.name}" `;
        }
        where += `at line ${this.line + 1}, column ${this.column + 1}`;
        if (!compact1) {
            snippet = this.getSnippet();
            if (snippet) {
                where += `:\n${snippet}`;
            }
        }
        return where;
    }
}
function compileList(schema, name, result30) {
    const exclude = [];
    for (const includedSchema of schema.include){
        result30 = compileList(includedSchema, name, result30);
    }
    for (const currentType of schema[name]){
        for(let previousIndex = 0; previousIndex < result30.length; previousIndex++){
            const previousType = result30[previousIndex];
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind) {
                exclude.push(previousIndex);
            }
        }
        result30.push(currentType);
    }
    return result30.filter((_type, index)=>!exclude.includes(index)
    );
}
function compileMap(...typesList) {
    const result31 = {
        fallback: {
        },
        mapping: {
        },
        scalar: {
        },
        sequence: {
        }
    };
    for (const types of typesList){
        for (const type31 of types){
            if (type31.kind !== null) {
                result31[type31.kind][type31.tag] = result31["fallback"][type31.tag] = type31;
            }
        }
    }
    return result31;
}
class Schema {
    static SCHEMA_DEFAULT;
    implicit;
    explicit;
    include;
    compiledImplicit;
    compiledExplicit;
    compiledTypeMap;
    constructor(definition){
        this.explicit = definition.explicit || [];
        this.implicit = definition.implicit || [];
        this.include = definition.include || [];
        for (const type32 of this.implicit){
            if (type32.loadKind && type32.loadKind !== "scalar") {
                throw new YAMLError("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
            }
        }
        this.compiledImplicit = compileList(this, "implicit", []);
        this.compiledExplicit = compileList(this, "explicit", []);
        this.compiledTypeMap = compileMap(this.compiledImplicit, this.compiledExplicit);
    }
    extend(definition) {
        return new Schema({
            implicit: [
                ...new Set([
                    ...this.implicit,
                    ...definition?.implicit ?? []
                ]), 
            ],
            explicit: [
                ...new Set([
                    ...this.explicit,
                    ...definition?.explicit ?? []
                ]), 
            ],
            include: [
                ...new Set([
                    ...this.include,
                    ...definition?.include ?? []
                ])
            ]
        });
    }
    static create() {
    }
}
const DEFAULT_RESOLVE = ()=>true
;
const DEFAULT_CONSTRUCT = (data26)=>data26
;
function checkTagFormat(tag) {
    return tag;
}
class Type1 {
    tag;
    kind = null;
    instanceOf;
    predicate;
    represent;
    defaultStyle;
    styleAliases;
    loadKind;
    constructor(tag, options){
        this.tag = checkTagFormat(tag);
        if (options) {
            this.kind = options.kind;
            this.resolve = options.resolve || DEFAULT_RESOLVE;
            this.construct = options.construct || DEFAULT_CONSTRUCT;
            this.instanceOf = options.instanceOf;
            this.predicate = options.predicate;
            this.represent = options.represent;
            this.defaultStyle = options.defaultStyle;
            this.styleAliases = options.styleAliases;
        }
    }
    resolve = ()=>true
    ;
    construct = (data27)=>data27
    ;
}
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data28) {
    if (data28 === null) return false;
    let code19;
    let bitlen = 0;
    const max2 = data28.length;
    const map3 = BASE64_MAP;
    for(let idx = 0; idx < max2; idx++){
        code19 = map3.indexOf(data28.charAt(idx));
        if (code19 > 64) continue;
        if (code19 < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data29) {
    const input = data29.replace(/[\r\n=]/g, "");
    const max3 = input.length;
    const map4 = BASE64_MAP;
    const result32 = [];
    let bits = 0;
    for(let idx = 0; idx < max3; idx++){
        if (idx % 4 === 0 && idx) {
            result32.push(bits >> 16 & 255);
            result32.push(bits >> 8 & 255);
            result32.push(bits & 255);
        }
        bits = bits << 6 | map4.indexOf(input.charAt(idx));
    }
    const tailbits = max3 % 4 * 6;
    if (tailbits === 0) {
        result32.push(bits >> 16 & 255);
        result32.push(bits >> 8 & 255);
        result32.push(bits & 255);
    } else if (tailbits === 18) {
        result32.push(bits >> 10 & 255);
        result32.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result32.push(bits >> 4 & 255);
    }
    return new Buffer(new Uint8Array(result32));
}
function representYamlBinary(object1) {
    const max4 = object1.length;
    const map5 = BASE64_MAP;
    let result33 = "";
    let bits = 0;
    for(let idx = 0; idx < max4; idx++){
        if (idx % 3 === 0 && idx) {
            result33 += map5[bits >> 18 & 63];
            result33 += map5[bits >> 12 & 63];
            result33 += map5[bits >> 6 & 63];
            result33 += map5[bits & 63];
        }
        bits = (bits << 8) + object1[idx];
    }
    const tail3 = max4 % 3;
    if (tail3 === 0) {
        result33 += map5[bits >> 18 & 63];
        result33 += map5[bits >> 12 & 63];
        result33 += map5[bits >> 6 & 63];
        result33 += map5[bits & 63];
    } else if (tail3 === 2) {
        result33 += map5[bits >> 10 & 63];
        result33 += map5[bits >> 4 & 63];
        result33 += map5[bits << 2 & 63];
        result33 += map5[64];
    } else if (tail3 === 1) {
        result33 += map5[bits >> 2 & 63];
        result33 += map5[bits << 4 & 63];
        result33 += map5[64];
        result33 += map5[64];
    }
    return result33;
}
function isBinary(obj) {
    const buf = new Buffer();
    try {
        if (0 > buf.readFromSync(obj)) return true;
        return false;
    } catch  {
        return false;
    } finally{
        buf.reset();
    }
}
const binary = new Type1("tag:yaml.org,2002:binary", {
    construct: constructYamlBinary,
    kind: "scalar",
    predicate: isBinary,
    represent: representYamlBinary,
    resolve: resolveYamlBinary
});
function resolveYamlBoolean(data30) {
    const max5 = data30.length;
    return max5 === 4 && (data30 === "true" || data30 === "True" || data30 === "TRUE") || max5 === 5 && (data30 === "false" || data30 === "False" || data30 === "FALSE");
}
function constructYamlBoolean(data31) {
    return data31 === "true" || data31 === "True" || data31 === "TRUE";
}
const bool = new Type1("tag:yaml.org,2002:bool", {
    construct: constructYamlBoolean,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isBoolean1,
    represent: {
        lowercase (object2) {
            return object2 ? "true" : "false";
        },
        uppercase (object3) {
            return object3 ? "TRUE" : "FALSE";
        },
        camelcase (object4) {
            return object4 ? "True" : "False";
        }
    },
    resolve: resolveYamlBoolean
});
const YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + "|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + "|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + "|[-+]?\\.(?:inf|Inf|INF)" + "|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data32) {
    if (!YAML_FLOAT_PATTERN.test(data32) || data32[data32.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data33) {
    let value = data33.replace(/_/g, "").toLowerCase();
    const sign = value[0] === "-" ? -1 : 1;
    const digits = [];
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    }
    if (value === ".nan") {
        return NaN;
    }
    if (value.indexOf(":") >= 0) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseFloat(v));
        });
        let valueNb = 0;
        let base2 = 1;
        digits.forEach((d)=>{
            valueNb += d * base2;
            base2 *= 60;
        });
        return sign * valueNb;
    }
    return sign * parseFloat(value);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object5, style) {
    if (isNaN(object5)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object5) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object5) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (isNegativeZero(object5)) {
        return "-0.0";
    }
    const res = object5.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object6) {
    return Object.prototype.toString.call(object6) === "[object Number]" && (object6 % 1 !== 0 || isNegativeZero(object6));
}
const __float = new Type1("tag:yaml.org,2002:float", {
    construct: constructYamlFloat,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isFloat,
    represent: representYamlFloat,
    resolve: resolveYamlFloat
});
function reconstructFunction(code20) {
    const func1 = new Function(`return ${code20}`)();
    if (!(func1 instanceof Function)) {
        throw new TypeError(`Expected function but got ${typeof func1}: ${code20}`);
    }
    return func1;
}
new Type1("tag:yaml.org,2002:js/function", {
    kind: "scalar",
    resolve (data34) {
        if (data34 === null) {
            return false;
        }
        try {
            reconstructFunction(`${data34}`);
            return true;
        } catch (_err) {
            return false;
        }
    },
    construct (data35) {
        return reconstructFunction(data35);
    },
    predicate (object7) {
        return object7 instanceof Function;
    },
    represent (object8) {
        return object8.toString();
    }
});
function isHexCode(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode(c) {
    return 48 <= c && c <= 55;
}
function isDecCode(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger(data36) {
    const max6 = data36.length;
    let index = 0;
    let hasDigits = false;
    if (!max6) return false;
    let ch = data36[index];
    if (ch === "-" || ch === "+") {
        ch = data36[++index];
    }
    if (ch === "0") {
        if (index + 1 === max6) return true;
        ch = data36[++index];
        if (ch === "b") {
            index++;
            for(; index < max6; index++){
                ch = data36[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max6; index++){
                ch = data36[index];
                if (ch === "_") continue;
                if (!isHexCode(data36.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for(; index < max6; index++){
            ch = data36[index];
            if (ch === "_") continue;
            if (!isOctCode(data36.charCodeAt(index))) return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_") return false;
    for(; index < max6; index++){
        ch = data36[index];
        if (ch === "_") continue;
        if (ch === ":") break;
        if (!isDecCode(data36.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    if (ch !== ":") return true;
    return /^(:[0-5]?[0-9])+$/.test(data36.slice(index));
}
function constructYamlInteger(data37) {
    let value = data37;
    const digits = [];
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value, 16);
        return sign * parseInt(value, 8);
    }
    if (value.indexOf(":") !== -1) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base3 = 1;
        digits.forEach((d)=>{
            valueInt += d * base3;
            base3 *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value, 10);
}
function isInteger1(object9) {
    return Object.prototype.toString.call(object9) === "[object Number]" && object9 % 1 === 0 && !isNegativeZero(object9);
}
const __int = new Type1("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger1,
    represent: {
        binary (obj) {
            return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
        },
        octal (obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal (obj) {
            return obj.toString(10);
        },
        hexadecimal (obj) {
            return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        }
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ],
        octal: [
            8,
            "oct"
        ]
    }
});
const map1 = new Type1("tag:yaml.org,2002:map", {
    construct (data38) {
        return data38 !== null ? data38 : {
        };
    },
    kind: "mapping"
});
function resolveYamlMerge(data39) {
    return data39 === "<<" || data39 === null;
}
const merge1 = new Type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
function resolveYamlNull(data40) {
    const max7 = data40.length;
    return max7 === 1 && data40 === "~" || max7 === 4 && (data40 === "null" || data40 === "Null" || data40 === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull1(object10) {
    return object10 === null;
}
const nil = new Type1("tag:yaml.org,2002:null", {
    construct: constructYamlNull,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isNull1,
    represent: {
        canonical () {
            return "~";
        },
        lowercase () {
            return "null";
        },
        uppercase () {
            return "NULL";
        },
        camelcase () {
            return "Null";
        }
    },
    resolve: resolveYamlNull
});
const { hasOwn  } = Object;
const _toString = Object.prototype.toString;
function resolveYamlOmap(data41) {
    const objectKeys = [];
    let pairKey = "";
    let pairHasKey = false;
    for (const pair of data41){
        pairHasKey = false;
        if (_toString.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (hasOwn(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data42) {
    return data42 !== null ? data42 : [];
}
const omap = new Type1("tag:yaml.org,2002:omap", {
    construct: constructYamlOmap,
    kind: "sequence",
    resolve: resolveYamlOmap
});
const _toString1 = Object.prototype.toString;
function resolveYamlPairs(data43) {
    const result34 = Array.from({
        length: data43.length
    });
    for(let index = 0; index < data43.length; index++){
        const pair = data43[index];
        if (_toString1.call(pair) !== "[object Object]") return false;
        const keys2 = Object.keys(pair);
        if (keys2.length !== 1) return false;
        result34[index] = [
            keys2[0],
            pair[keys2[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data44) {
    if (data44 === null) return [];
    const result35 = Array.from({
        length: data44.length
    });
    for(let index = 0; index < data44.length; index += 1){
        const pair = data44[index];
        const keys3 = Object.keys(pair);
        result35[index] = [
            keys3[0],
            pair[keys3[0]]
        ];
    }
    return result35;
}
const pairs = new Type1("tag:yaml.org,2002:pairs", {
    construct: constructYamlPairs,
    kind: "sequence",
    resolve: resolveYamlPairs
});
const REGEXP = /^\/(?<regexp>[\s\S]+)\/(?<modifiers>[gismuy]*)$/;
const regexp = new Type1("tag:yaml.org,2002:js/regexp", {
    kind: "scalar",
    resolve (data45) {
        if (data45 === null || !data45.length) {
            return false;
        }
        const regexp1 = `${data45}`;
        if (regexp1.charAt(0) === "/") {
            if (!REGEXP.test(data45)) {
                return false;
            }
            const modifiers = [
                ...regexp1.match(REGEXP)?.groups?.modifiers ?? ""
            ];
            if (new Set(modifiers).size < modifiers.length) {
                return false;
            }
        }
        return true;
    },
    construct (data46) {
        const { regexp: regexp2 = `${data46}` , modifiers =""  } = `${data46}`.match(REGEXP)?.groups ?? {
        };
        return new RegExp(regexp2, modifiers);
    },
    predicate (object11) {
        return object11 instanceof RegExp;
    },
    represent (object12) {
        return object12.toString();
    }
});
const seq1 = new Type1("tag:yaml.org,2002:seq", {
    construct (data47) {
        return data47 !== null ? data47 : [];
    },
    kind: "sequence"
});
const { hasOwn: hasOwn1  } = Object;
function resolveYamlSet(data48) {
    if (data48 === null) return true;
    for(const key in data48){
        if (hasOwn1(data48, key)) {
            if (data48[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data49) {
    return data49 !== null ? data49 : {
    };
}
const set1 = new Type1("tag:yaml.org,2002:set", {
    construct: constructYamlSet,
    kind: "mapping",
    resolve: resolveYamlSet
});
const str = new Type1("tag:yaml.org,2002:str", {
    construct (data50) {
        return data50 !== null ? data50 : "";
    },
    kind: "scalar"
});
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9])" + "-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9]?)" + "-([0-9][0-9]?)" + "(?:[Tt]|[ \\t]+)" + "([0-9][0-9]?)" + ":([0-9][0-9])" + ":([0-9][0-9])" + "(?:\\.([0-9]*))?" + "(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + "(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data51) {
    if (data51 === null) return false;
    if (YAML_DATE_REGEXP.exec(data51) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data51) !== null) return true;
    return false;
}
function constructYamlTimestamp(data52) {
    let match = YAML_DATE_REGEXP.exec(data52);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data52);
    if (match === null) throw new Error("Date resolve error");
    const year = +match[1];
    const month = +match[2] - 1;
    const day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    const hour = +match[4];
    const minute = +match[5];
    const second = +match[6];
    let fraction = 0;
    if (match[7]) {
        let partFraction = match[7].slice(0, 3);
        while(partFraction.length < 3){
            partFraction += "0";
        }
        fraction = +partFraction;
    }
    let delta = null;
    if (match[9]) {
        const tzHour = +match[10];
        const tzMinute = +(match[11] || 0);
        delta = (tzHour * 60 + tzMinute) * 60000;
        if (match[9] === "-") delta = -delta;
    }
    const date1 = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date1.setTime(date1.getTime() - delta);
    return date1;
}
function representYamlTimestamp(date2) {
    return date2.toISOString();
}
const timestamp = new Type1("tag:yaml.org,2002:timestamp", {
    construct: constructYamlTimestamp,
    instanceOf: Date,
    kind: "scalar",
    represent: representYamlTimestamp,
    resolve: resolveYamlTimestamp
});
const undefinedType = new Type1("tag:yaml.org,2002:js/undefined", {
    kind: "scalar",
    resolve () {
        return true;
    },
    construct () {
        return undefined;
    },
    predicate (object13) {
        return typeof object13 === "undefined";
    },
    represent () {
        return "";
    }
});
const failsafe = new Schema({
    explicit: [
        str,
        seq1,
        map1
    ]
});
const json1 = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ]
});
const core = new Schema({
    include: [
        json1
    ]
});
const def = new Schema({
    explicit: [
        binary,
        omap,
        pairs,
        set1
    ],
    implicit: [
        timestamp,
        merge1
    ],
    include: [
        core
    ]
});
new Schema({
    explicit: [
        regexp,
        undefinedType
    ],
    include: [
        def
    ]
});
class State1 {
    schema;
    constructor(schema = def){
        this.schema = schema;
    }
}
class LoaderState extends State1 {
    input;
    documents = [];
    length;
    lineIndent = 0;
    lineStart = 0;
    position = 0;
    line = 0;
    filename;
    onWarning;
    legacy;
    json;
    listener;
    implicitTypes;
    typeMap;
    version;
    checkLineBreaks;
    tagMap;
    anchorMap;
    tag;
    anchor;
    kind;
    result = "";
    constructor(input, { filename , schema , onWarning , legacy =false , json =false , listener =null  }){
        super(schema);
        this.input = input;
        this.filename = filename;
        this.onWarning = onWarning;
        this.legacy = legacy;
        this.json = json;
        this.listener = listener;
        this.implicitTypes = this.schema.compiledImplicit;
        this.typeMap = this.schema.compiledTypeMap;
        this.length = input.length;
    }
}
const { hasOwn: hasOwn2  } = Object;
const CONTEXT_BLOCK_IN = 3;
const CONTEXT_BLOCK_OUT = 4;
const CHOMPING_STRIP = 2;
const CHOMPING_KEEP = 3;
const PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
const PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
const PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
const PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
const PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function isEOL(c) {
    return c === 10 || c === 13;
}
function isWhiteSpace(c) {
    return c === 9 || c === 32;
}
function isWsOrEol(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function isFlowIndicator(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    const lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 48 ? "\x00" : c === 97 ? "\x07" : c === 98 ? "\x08" : c === 116 ? "\x09" : c === 9 ? "\x09" : c === 110 ? "\x0A" : c === 118 ? "\x0B" : c === 102 ? "\x0C" : c === 114 ? "\x0D" : c === 101 ? "\x1B" : c === 32 ? " " : c === 34 ? "\x22" : c === 47 ? "/" : c === 92 ? "\x5C" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
const simpleEscapeCheck = Array.from({
    length: 256
});
const simpleEscapeMap = Array.from({
    length: 256
});
for(let i73 = 0; i73 < 256; i73++){
    simpleEscapeCheck[i73] = simpleEscapeSequence(i73) ? 1 : 0;
    simpleEscapeMap[i73] = simpleEscapeSequence(i73);
}
function generateError(state2, message) {
    return new YAMLError(message, new Mark(state2.filename, state2.input, state2.position, state2.line, state2.position - state2.lineStart));
}
function throwError(state3, message) {
    throw generateError(state3, message);
}
function throwWarning(state4, message) {
    if (state4.onWarning) {
        state4.onWarning.call(null, generateError(state4, message));
    }
}
const directiveHandlers = {
    YAML (state5, _name, ...args) {
        if (state5.version !== null) {
            return throwError(state5, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            return throwError(state5, "YAML directive accepts exactly one argument");
        }
        const match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            return throwError(state5, "ill-formed argument of the YAML directive");
        }
        const major = parseInt(match[1], 10);
        const minor = parseInt(match[2], 10);
        if (major !== 1) {
            return throwError(state5, "unacceptable YAML version of the document");
        }
        state5.version = args[0];
        state5.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            return throwWarning(state5, "unsupported YAML version of the document");
        }
    },
    TAG (state6, _name, ...args) {
        if (args.length !== 2) {
            return throwError(state6, "TAG directive accepts exactly two arguments");
        }
        const handle = args[0];
        const prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            return throwError(state6, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (state6.tagMap && hasOwn2(state6.tagMap, handle)) {
            return throwError(state6, `there is a previously declared suffix for "${handle}" tag handle`);
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            return throwError(state6, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        if (typeof state6.tagMap === "undefined") {
            state6.tagMap = {
            };
        }
        state6.tagMap[handle] = prefix;
    }
};
function captureSegment(state7, start, end, checkJson) {
    let result;
    if (start < end) {
        result = state7.input.slice(start, end);
        if (checkJson) {
            for(let position = 0, length = result.length; position < length; position++){
                const character = result.charCodeAt(position);
                if (!(character === 9 || 32 <= character && character <= 1114111)) {
                    return throwError(state7, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(result)) {
            return throwError(state7, "the stream contains non-printable characters");
        }
        state7.result += result;
    }
}
function mergeMappings(state8, destination, source, overridableKeys) {
    if (!isObject1(source)) {
        return throwError(state8, "cannot merge mappings; the provided source object is unacceptable");
    }
    const keys4 = Object.keys(source);
    for(let i1 = 0, len1 = keys4.length; i1 < len1; i1++){
        const key = keys4[i1];
        if (!hasOwn2(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state9, result, overridableKeys, keyTag, keyNode, valueNode, startLine, startPos) {
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(let index = 0, quantity = keyNode.length; index < quantity; index++){
            if (Array.isArray(keyNode[index])) {
                return throwError(state9, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (result === null) {
        result = {
        };
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(let index = 0, quantity = valueNode.length; index < quantity; index++){
                mergeMappings(state9, result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings(state9, result, valueNode, overridableKeys);
        }
    } else {
        if (!state9.json && !hasOwn2(overridableKeys, keyNode) && hasOwn2(result, keyNode)) {
            state9.line = startLine || state9.line;
            state9.position = startPos || state9.position;
            return throwError(state9, "duplicated mapping key");
        }
        result[keyNode] = valueNode;
        delete overridableKeys[keyNode];
    }
    return result;
}
function readLineBreak(state10) {
    const ch = state10.input.charCodeAt(state10.position);
    if (ch === 10) {
        state10.position++;
    } else if (ch === 13) {
        state10.position++;
        if (state10.input.charCodeAt(state10.position) === 10) {
            state10.position++;
        }
    } else {
        return throwError(state10, "a line break is expected");
    }
    state10.line += 1;
    state10.lineStart = state10.position;
}
function skipSeparationSpace(state11, allowComments, checkIndent) {
    let lineBreaks = 0, ch = state11.input.charCodeAt(state11.position);
    while(ch !== 0){
        while(isWhiteSpace(ch)){
            ch = state11.input.charCodeAt(++state11.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state11.input.charCodeAt(++state11.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (isEOL(ch)) {
            readLineBreak(state11);
            ch = state11.input.charCodeAt(state11.position);
            lineBreaks++;
            state11.lineIndent = 0;
            while(ch === 32){
                state11.lineIndent++;
                ch = state11.input.charCodeAt(++state11.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state11.lineIndent < checkIndent) {
        throwWarning(state11, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state12) {
    let _position = state12.position;
    let ch = state12.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state12.input.charCodeAt(_position + 1) && ch === state12.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state12.input.charCodeAt(_position);
        if (ch === 0 || isWsOrEol(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state13, count) {
    if (count === 1) {
        state13.result += " ";
    } else if (count > 1) {
        state13.result += repeat1("\n", count - 1);
    }
}
function readPlainScalar(state14, nodeIndent, withinFlowCollection) {
    const kind = state14.kind;
    const result = state14.result;
    let ch = state14.input.charCodeAt(state14.position);
    if (isWsOrEol(ch) || isFlowIndicator(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    let following;
    if (ch === 63 || ch === 45) {
        following = state14.input.charCodeAt(state14.position + 1);
        if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
            return false;
        }
    }
    state14.kind = "scalar";
    state14.result = "";
    let captureEnd, captureStart = captureEnd = state14.position;
    let hasPendingContent = false;
    let line = 0;
    while(ch !== 0){
        if (ch === 58) {
            following = state14.input.charCodeAt(state14.position + 1);
            if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
                break;
            }
        } else if (ch === 35) {
            const preceding = state14.input.charCodeAt(state14.position - 1);
            if (isWsOrEol(preceding)) {
                break;
            }
        } else if (state14.position === state14.lineStart && testDocumentSeparator(state14) || withinFlowCollection && isFlowIndicator(ch)) {
            break;
        } else if (isEOL(ch)) {
            line = state14.line;
            const lineStart = state14.lineStart;
            const lineIndent = state14.lineIndent;
            skipSeparationSpace(state14, false, -1);
            if (state14.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state14.input.charCodeAt(state14.position);
                continue;
            } else {
                state14.position = captureEnd;
                state14.line = line;
                state14.lineStart = lineStart;
                state14.lineIndent = lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state14, captureStart, captureEnd, false);
            writeFoldedLines(state14, state14.line - line);
            captureStart = captureEnd = state14.position;
            hasPendingContent = false;
        }
        if (!isWhiteSpace(ch)) {
            captureEnd = state14.position + 1;
        }
        ch = state14.input.charCodeAt(++state14.position);
    }
    captureSegment(state14, captureStart, captureEnd, false);
    if (state14.result) {
        return true;
    }
    state14.kind = kind;
    state14.result = result;
    return false;
}
function readSingleQuotedScalar(state15, nodeIndent) {
    let ch, captureStart, captureEnd;
    ch = state15.input.charCodeAt(state15.position);
    if (ch !== 39) {
        return false;
    }
    state15.kind = "scalar";
    state15.result = "";
    state15.position++;
    captureStart = captureEnd = state15.position;
    while((ch = state15.input.charCodeAt(state15.position)) !== 0){
        if (ch === 39) {
            captureSegment(state15, captureStart, state15.position, true);
            ch = state15.input.charCodeAt(++state15.position);
            if (ch === 39) {
                captureStart = state15.position;
                state15.position++;
                captureEnd = state15.position;
            } else {
                return true;
            }
        } else if (isEOL(ch)) {
            captureSegment(state15, captureStart, captureEnd, true);
            writeFoldedLines(state15, skipSeparationSpace(state15, false, nodeIndent));
            captureStart = captureEnd = state15.position;
        } else if (state15.position === state15.lineStart && testDocumentSeparator(state15)) {
            return throwError(state15, "unexpected end of the document within a single quoted scalar");
        } else {
            state15.position++;
            captureEnd = state15.position;
        }
    }
    return throwError(state15, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state16, nodeIndent) {
    let ch = state16.input.charCodeAt(state16.position);
    if (ch !== 34) {
        return false;
    }
    state16.kind = "scalar";
    state16.result = "";
    state16.position++;
    let captureEnd, captureStart = captureEnd = state16.position;
    let tmp;
    while((ch = state16.input.charCodeAt(state16.position)) !== 0){
        if (ch === 34) {
            captureSegment(state16, captureStart, state16.position, true);
            state16.position++;
            return true;
        }
        if (ch === 92) {
            captureSegment(state16, captureStart, state16.position, true);
            ch = state16.input.charCodeAt(++state16.position);
            if (isEOL(ch)) {
                skipSeparationSpace(state16, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state16.result += simpleEscapeMap[ch];
                state16.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                let hexLength = tmp;
                let hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state16.input.charCodeAt(++state16.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        return throwError(state16, "expected hexadecimal character");
                    }
                }
                state16.result += charFromCodepoint(hexResult);
                state16.position++;
            } else {
                return throwError(state16, "unknown escape sequence");
            }
            captureStart = captureEnd = state16.position;
        } else if (isEOL(ch)) {
            captureSegment(state16, captureStart, captureEnd, true);
            writeFoldedLines(state16, skipSeparationSpace(state16, false, nodeIndent));
            captureStart = captureEnd = state16.position;
        } else if (state16.position === state16.lineStart && testDocumentSeparator(state16)) {
            return throwError(state16, "unexpected end of the document within a double quoted scalar");
        } else {
            state16.position++;
            captureEnd = state16.position;
        }
    }
    return throwError(state16, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state17, nodeIndent) {
    let ch = state17.input.charCodeAt(state17.position);
    let terminator;
    let isMapping = true;
    let result = {
    };
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        result = [];
    } else if (ch === 123) {
        terminator = 125;
    } else {
        return false;
    }
    if (state17.anchor !== null && typeof state17.anchor != "undefined" && typeof state17.anchorMap != "undefined") {
        state17.anchorMap[state17.anchor] = result;
    }
    ch = state17.input.charCodeAt(++state17.position);
    const tag = state17.tag, anchor = state17.anchor;
    let readNext = true;
    let valueNode, keyNode, keyTag = keyNode = valueNode = null, isExplicitPair, isPair = isExplicitPair = false;
    let following = 0, line = 0;
    const overridableKeys = {
    };
    while(ch !== 0){
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if (ch === terminator) {
            state17.position++;
            state17.tag = tag;
            state17.anchor = anchor;
            state17.kind = isMapping ? "mapping" : "sequence";
            state17.result = result;
            return true;
        }
        if (!readNext) {
            return throwError(state17, "missed comma between flow collection entries");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state17.input.charCodeAt(state17.position + 1);
            if (isWsOrEol(following)) {
                isPair = isExplicitPair = true;
                state17.position++;
                skipSeparationSpace(state17, true, nodeIndent);
            }
        }
        line = state17.line;
        composeNode(state17, nodeIndent, 1, false, true);
        keyTag = state17.tag || null;
        keyNode = state17.result;
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if ((isExplicitPair || state17.line === line) && ch === 58) {
            isPair = true;
            ch = state17.input.charCodeAt(++state17.position);
            skipSeparationSpace(state17, true, nodeIndent);
            composeNode(state17, nodeIndent, 1, false, true);
            valueNode = state17.result;
        }
        if (isMapping) {
            storeMappingPair(state17, result, overridableKeys, keyTag, keyNode, valueNode);
        } else if (isPair) {
            result.push(storeMappingPair(state17, null, overridableKeys, keyTag, keyNode, valueNode));
        } else {
            result.push(keyNode);
        }
        skipSeparationSpace(state17, true, nodeIndent);
        ch = state17.input.charCodeAt(state17.position);
        if (ch === 44) {
            readNext = true;
            ch = state17.input.charCodeAt(++state17.position);
        } else {
            readNext = false;
        }
    }
    return throwError(state17, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state18, nodeIndent) {
    let chomping = 1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false;
    let ch = state18.input.charCodeAt(state18.position);
    let folding = false;
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state18.kind = "scalar";
    state18.result = "";
    let tmp = 0;
    while(ch !== 0){
        ch = state18.input.charCodeAt(++state18.position);
        if (ch === 43 || ch === 45) {
            if (1 === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                return throwError(state18, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                return throwError(state18, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                return throwError(state18, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (isWhiteSpace(ch)) {
        do {
            ch = state18.input.charCodeAt(++state18.position);
        }while (isWhiteSpace(ch))
        if (ch === 35) {
            do {
                ch = state18.input.charCodeAt(++state18.position);
            }while (!isEOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state18);
        state18.lineIndent = 0;
        ch = state18.input.charCodeAt(state18.position);
        while((!detectedIndent || state18.lineIndent < textIndent) && ch === 32){
            state18.lineIndent++;
            ch = state18.input.charCodeAt(++state18.position);
        }
        if (!detectedIndent && state18.lineIndent > textIndent) {
            textIndent = state18.lineIndent;
        }
        if (isEOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state18.lineIndent < textIndent) {
            if (chomping === 3) {
                state18.result += repeat1("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === 1) {
                if (didReadContent) {
                    state18.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (isWhiteSpace(ch)) {
                atMoreIndented = true;
                state18.result += repeat1("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state18.result += repeat1("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state18.result += " ";
                }
            } else {
                state18.result += repeat1("\n", emptyLines);
            }
        } else {
            state18.result += repeat1("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        const captureStart = state18.position;
        while(!isEOL(ch) && ch !== 0){
            ch = state18.input.charCodeAt(++state18.position);
        }
        captureSegment(state18, captureStart, state18.position, false);
    }
    return true;
}
function readBlockSequence(state19, nodeIndent) {
    let line, following, detected = false, ch;
    const tag = state19.tag, anchor = state19.anchor, result = [];
    if (state19.anchor !== null && typeof state19.anchor !== "undefined" && typeof state19.anchorMap !== "undefined") {
        state19.anchorMap[state19.anchor] = result;
    }
    ch = state19.input.charCodeAt(state19.position);
    while(ch !== 0){
        if (ch !== 45) {
            break;
        }
        following = state19.input.charCodeAt(state19.position + 1);
        if (!isWsOrEol(following)) {
            break;
        }
        detected = true;
        state19.position++;
        if (skipSeparationSpace(state19, true, -1)) {
            if (state19.lineIndent <= nodeIndent) {
                result.push(null);
                ch = state19.input.charCodeAt(state19.position);
                continue;
            }
        }
        line = state19.line;
        composeNode(state19, nodeIndent, 3, false, true);
        result.push(state19.result);
        skipSeparationSpace(state19, true, -1);
        ch = state19.input.charCodeAt(state19.position);
        if ((state19.line === line || state19.lineIndent > nodeIndent) && ch !== 0) {
            return throwError(state19, "bad indentation of a sequence entry");
        } else if (state19.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state19.tag = tag;
        state19.anchor = anchor;
        state19.kind = "sequence";
        state19.result = result;
        return true;
    }
    return false;
}
function readBlockMapping(state20, nodeIndent, flowIndent) {
    const tag = state20.tag, anchor = state20.anchor, result = {
    }, overridableKeys = {
    };
    let following, allowCompact = false, line, pos, keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state20.anchor !== null && typeof state20.anchor !== "undefined" && typeof state20.anchorMap !== "undefined") {
        state20.anchorMap[state20.anchor] = result;
    }
    ch = state20.input.charCodeAt(state20.position);
    while(ch !== 0){
        following = state20.input.charCodeAt(state20.position + 1);
        line = state20.line;
        pos = state20.position;
        if ((ch === 63 || ch === 58) && isWsOrEol(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                return throwError(state20, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state20.position += 1;
            ch = following;
        } else if (composeNode(state20, flowIndent, 2, false, true)) {
            if (state20.line === line) {
                ch = state20.input.charCodeAt(state20.position);
                while(isWhiteSpace(ch)){
                    ch = state20.input.charCodeAt(++state20.position);
                }
                if (ch === 58) {
                    ch = state20.input.charCodeAt(++state20.position);
                    if (!isWsOrEol(ch)) {
                        return throwError(state20, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state20.tag;
                    keyNode = state20.result;
                } else if (detected) {
                    return throwError(state20, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state20.tag = tag;
                    state20.anchor = anchor;
                    return true;
                }
            } else if (detected) {
                return throwError(state20, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state20.tag = tag;
                state20.anchor = anchor;
                return true;
            }
        } else {
            break;
        }
        if (state20.line === line || state20.lineIndent > nodeIndent) {
            if (composeNode(state20, nodeIndent, 4, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state20.result;
                } else {
                    valueNode = state20.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, valueNode, line, pos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state20, true, -1);
            ch = state20.input.charCodeAt(state20.position);
        }
        if (state20.lineIndent > nodeIndent && ch !== 0) {
            return throwError(state20, "bad indentation of a mapping entry");
        } else if (state20.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state20, result, overridableKeys, keyTag, keyNode, null);
    }
    if (detected) {
        state20.tag = tag;
        state20.anchor = anchor;
        state20.kind = "mapping";
        state20.result = result;
    }
    return detected;
}
function readTagProperty(state21) {
    let position, isVerbatim = false, isNamed = false, tagHandle = "", tagName, ch;
    ch = state21.input.charCodeAt(state21.position);
    if (ch !== 33) return false;
    if (state21.tag !== null) {
        return throwError(state21, "duplication of a tag property");
    }
    ch = state21.input.charCodeAt(++state21.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state21.input.charCodeAt(++state21.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state21.input.charCodeAt(++state21.position);
    } else {
        tagHandle = "!";
    }
    position = state21.position;
    if (isVerbatim) {
        do {
            ch = state21.input.charCodeAt(++state21.position);
        }while (ch !== 0 && ch !== 62)
        if (state21.position < state21.length) {
            tagName = state21.input.slice(position, state21.position);
            ch = state21.input.charCodeAt(++state21.position);
        } else {
            return throwError(state21, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !isWsOrEol(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state21.input.slice(position - 1, state21.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        return throwError(state21, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    position = state21.position + 1;
                } else {
                    return throwError(state21, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state21.input.charCodeAt(++state21.position);
        }
        tagName = state21.input.slice(position, state21.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            return throwError(state21, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        return throwError(state21, `tag name cannot contain such characters: ${tagName}`);
    }
    if (isVerbatim) {
        state21.tag = tagName;
    } else if (typeof state21.tagMap !== "undefined" && hasOwn2(state21.tagMap, tagHandle)) {
        state21.tag = state21.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state21.tag = `!${tagName}`;
    } else if (tagHandle === "!!") {
        state21.tag = `tag:yaml.org,2002:${tagName}`;
    } else {
        return throwError(state21, `undeclared tag handle "${tagHandle}"`);
    }
    return true;
}
function readAnchorProperty(state22) {
    let ch = state22.input.charCodeAt(state22.position);
    if (ch !== 38) return false;
    if (state22.anchor !== null) {
        return throwError(state22, "duplication of an anchor property");
    }
    ch = state22.input.charCodeAt(++state22.position);
    const position = state22.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state22.input.charCodeAt(++state22.position);
    }
    if (state22.position === position) {
        return throwError(state22, "name of an anchor node must contain at least one character");
    }
    state22.anchor = state22.input.slice(position, state22.position);
    return true;
}
function readAlias(state23) {
    let ch = state23.input.charCodeAt(state23.position);
    if (ch !== 42) return false;
    ch = state23.input.charCodeAt(++state23.position);
    const _position = state23.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state23.input.charCodeAt(++state23.position);
    }
    if (state23.position === _position) {
        return throwError(state23, "name of an alias node must contain at least one character");
    }
    const alias = state23.input.slice(_position, state23.position);
    if (typeof state23.anchorMap !== "undefined" && !hasOwn2(state23.anchorMap, alias)) {
        return throwError(state23, `unidentified alias "${alias}"`);
    }
    if (typeof state23.anchorMap !== "undefined") {
        state23.result = state23.anchorMap[alias];
    }
    skipSeparationSpace(state23, true, -1);
    return true;
}
function composeNode(state24, parentIndent, nodeContext, allowToSeek, allowCompact) {
    let allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, type33, flowIndent, blockIndent;
    if (state24.listener && state24.listener !== null) {
        state24.listener("open", state24);
    }
    state24.tag = null;
    state24.anchor = null;
    state24.kind = null;
    state24.result = null;
    const allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state24, true, -1)) {
            atNewLine = true;
            if (state24.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state24.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state24.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state24) || readAnchorProperty(state24)){
            if (skipSeparationSpace(state24, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state24.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state24.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state24.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || 4 === nodeContext) {
        const cond1 = 1 === nodeContext || 2 === nodeContext;
        flowIndent = cond1 ? parentIndent : parentIndent + 1;
        blockIndent = state24.position - state24.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state24, blockIndent) || readBlockMapping(state24, blockIndent, flowIndent)) || readFlowCollection(state24, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state24, flowIndent) || readSingleQuotedScalar(state24, flowIndent) || readDoubleQuotedScalar(state24, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state24)) {
                    hasContent = true;
                    if (state24.tag !== null || state24.anchor !== null) {
                        return throwError(state24, "alias node should not have Any properties");
                    }
                } else if (readPlainScalar(state24, flowIndent, 1 === nodeContext)) {
                    hasContent = true;
                    if (state24.tag === null) {
                        state24.tag = "?";
                    }
                }
                if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                    state24.anchorMap[state24.anchor] = state24.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state24, blockIndent);
        }
    }
    if (state24.tag !== null && state24.tag !== "!") {
        if (state24.tag === "?") {
            for(let typeIndex = 0, typeQuantity = state24.implicitTypes.length; typeIndex < typeQuantity; typeIndex++){
                type33 = state24.implicitTypes[typeIndex];
                if (type33.resolve(state24.result)) {
                    state24.result = type33.construct(state24.result);
                    state24.tag = type33.tag;
                    if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                        state24.anchorMap[state24.anchor] = state24.result;
                    }
                    break;
                }
            }
        } else if (hasOwn2(state24.typeMap[state24.kind || "fallback"], state24.tag)) {
            type33 = state24.typeMap[state24.kind || "fallback"][state24.tag];
            if (state24.result !== null && type33.kind !== state24.kind) {
                return throwError(state24, `unacceptable node kind for !<${state24.tag}> tag; it should be "${type33.kind}", not "${state24.kind}"`);
            }
            if (!type33.resolve(state24.result)) {
                return throwError(state24, `cannot resolve a node with !<${state24.tag}> explicit tag`);
            } else {
                state24.result = type33.construct(state24.result);
                if (state24.anchor !== null && typeof state24.anchorMap !== "undefined") {
                    state24.anchorMap[state24.anchor] = state24.result;
                }
            }
        } else {
            return throwError(state24, `unknown tag !<${state24.tag}>`);
        }
    }
    if (state24.listener && state24.listener !== null) {
        state24.listener("close", state24);
    }
    return state24.tag !== null || state24.anchor !== null || hasContent;
}
function readDocument(state25) {
    const documentStart = state25.position;
    let position, directiveName, directiveArgs, hasDirectives = false, ch;
    state25.version = null;
    state25.checkLineBreaks = state25.legacy;
    state25.tagMap = {
    };
    state25.anchorMap = {
    };
    while((ch = state25.input.charCodeAt(state25.position)) !== 0){
        skipSeparationSpace(state25, true, -1);
        ch = state25.input.charCodeAt(state25.position);
        if (state25.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state25.input.charCodeAt(++state25.position);
        position = state25.position;
        while(ch !== 0 && !isWsOrEol(ch)){
            ch = state25.input.charCodeAt(++state25.position);
        }
        directiveName = state25.input.slice(position, state25.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            return throwError(state25, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(isWhiteSpace(ch)){
                ch = state25.input.charCodeAt(++state25.position);
            }
            if (ch === 35) {
                do {
                    ch = state25.input.charCodeAt(++state25.position);
                }while (ch !== 0 && !isEOL(ch))
                break;
            }
            if (isEOL(ch)) break;
            position = state25.position;
            while(ch !== 0 && !isWsOrEol(ch)){
                ch = state25.input.charCodeAt(++state25.position);
            }
            directiveArgs.push(state25.input.slice(position, state25.position));
        }
        if (ch !== 0) readLineBreak(state25);
        if (hasOwn2(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state25, directiveName, ...directiveArgs);
        } else {
            throwWarning(state25, `unknown document directive "${directiveName}"`);
        }
    }
    skipSeparationSpace(state25, true, -1);
    if (state25.lineIndent === 0 && state25.input.charCodeAt(state25.position) === 45 && state25.input.charCodeAt(state25.position + 1) === 45 && state25.input.charCodeAt(state25.position + 2) === 45) {
        state25.position += 3;
        skipSeparationSpace(state25, true, -1);
    } else if (hasDirectives) {
        return throwError(state25, "directives end mark is expected");
    }
    composeNode(state25, state25.lineIndent - 1, 4, false, true);
    skipSeparationSpace(state25, true, -1);
    if (state25.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state25.input.slice(documentStart, state25.position))) {
        throwWarning(state25, "non-ASCII line breaks are interpreted as content");
    }
    state25.documents.push(state25.result);
    if (state25.position === state25.lineStart && testDocumentSeparator(state25)) {
        if (state25.input.charCodeAt(state25.position) === 46) {
            state25.position += 3;
            skipSeparationSpace(state25, true, -1);
        }
        return;
    }
    if (state25.position < state25.length - 1) {
        return throwError(state25, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments(input, options) {
    input = String(input);
    options = options || {
    };
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    const state26 = new LoaderState(input, options);
    state26.input += "\0";
    while(state26.input.charCodeAt(state26.position) === 32){
        state26.lineIndent += 1;
        state26.position += 1;
    }
    while(state26.position < state26.length - 1){
        readDocument(state26);
    }
    return state26.documents;
}
function load(input, options) {
    const documents = loadDocuments(input, options);
    if (documents.length === 0) {
        return;
    }
    if (documents.length === 1) {
        return documents[0];
    }
    throw new YAMLError("expected a single document in the stream, but found more");
}
function parse3(content, options) {
    return load(content, options);
}
const { hasOwn: hasOwn3  } = Object;
function compileStyleMap(schema, map6) {
    if (typeof map6 === "undefined" || map6 === null) return {
    };
    let type34;
    const result = {
    };
    const keys5 = Object.keys(map6);
    let tag, style;
    for(let index = 0, length = keys5.length; index < length; index += 1){
        tag = keys5[index];
        style = String(map6[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = `tag:yaml.org,2002:${tag.slice(2)}`;
        }
        type34 = schema.compiledTypeMap.fallback[tag];
        if (type34 && typeof type34.styleAliases !== "undefined" && hasOwn3(type34.styleAliases, style)) {
            style = type34.styleAliases[style];
        }
        result[tag] = style;
    }
    return result;
}
class DumperState extends State1 {
    indent;
    noArrayIndent;
    skipInvalid;
    flowLevel;
    sortKeys;
    lineWidth;
    noRefs;
    noCompatMode;
    condenseFlow;
    implicitTypes;
    explicitTypes;
    tag = null;
    result = "";
    duplicates = [];
    usedDuplicates = [];
    styleMap;
    dump;
    constructor({ schema , indent =2 , noArrayIndent =false , skipInvalid =false , flowLevel =-1 , styles =null , sortKeys =false , lineWidth =80 , noRefs =false , noCompatMode =false , condenseFlow =false  }){
        super(schema);
        this.indent = Math.max(1, indent);
        this.noArrayIndent = noArrayIndent;
        this.skipInvalid = skipInvalid;
        this.flowLevel = flowLevel;
        this.styleMap = compileStyleMap(this.schema, styles);
        this.sortKeys = sortKeys;
        this.lineWidth = lineWidth;
        this.noRefs = noRefs;
        this.noCompatMode = noCompatMode;
        this.condenseFlow = condenseFlow;
        this.implicitTypes = this.schema.compiledImplicit;
        this.explicitTypes = this.schema.compiledExplicit;
    }
}
const _toString2 = Object.prototype.toString;
const { hasOwn: hasOwn4  } = Object;
const ESCAPE_SEQUENCES = {
};
ESCAPE_SEQUENCES[0] = "\\0";
ESCAPE_SEQUENCES[7] = "\\a";
ESCAPE_SEQUENCES[8] = "\\b";
ESCAPE_SEQUENCES[9] = "\\t";
ESCAPE_SEQUENCES[10] = "\\n";
ESCAPE_SEQUENCES[11] = "\\v";
ESCAPE_SEQUENCES[12] = "\\f";
ESCAPE_SEQUENCES[13] = "\\r";
ESCAPE_SEQUENCES[27] = "\\e";
ESCAPE_SEQUENCES[34] = '\\"';
ESCAPE_SEQUENCES[92] = "\\\\";
ESCAPE_SEQUENCES[133] = "\\N";
ESCAPE_SEQUENCES[160] = "\\_";
ESCAPE_SEQUENCES[8232] = "\\L";
ESCAPE_SEQUENCES[8233] = "\\P";
const DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF", 
];
function encodeHex(character) {
    const string6 = character.toString(16).toUpperCase();
    let handle;
    let length;
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new YAMLError("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return `\\${handle}${repeat1("0", length - string6.length)}${string6}`;
}
function indentString(string7, spaces) {
    const ind = repeat1(" ", spaces), length = string7.length;
    let position = 0, next = -1, result = "", line;
    while(position < length){
        next = string7.indexOf("\n", position);
        if (next === -1) {
            line = string7.slice(position);
            position = length;
        } else {
            line = string7.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result += ind;
        result += line;
    }
    return result;
}
function generateNextLine(state27, level) {
    return `\n${repeat1(" ", state27.indent * level)}`;
}
function testImplicitResolving(state28, str66) {
    let type35;
    for(let index = 0, length = state28.implicitTypes.length; index < length; index += 1){
        type35 = state28.implicitTypes[index];
        if (type35.resolve(str66)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === 32 || c === 9;
}
function isPrintable(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== 65279 || 65536 <= c && c <= 1114111;
}
function isPlainSafe(c) {
    return isPrintable(c) && c !== 65279 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 58 && c !== 35;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== 65279 && !isWhitespace(c) && c !== 45 && c !== 63 && c !== 58 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 35 && c !== 38 && c !== 42 && c !== 33 && c !== 124 && c !== 62 && c !== 39 && c !== 34 && c !== 37 && c !== 64 && c !== 96;
}
function needIndentIndicator(string8) {
    const leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string8);
}
const STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string9, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType) {
    const shouldTrackWidth = lineWidth !== -1;
    let hasLineBreak = false, hasFoldableLine = false, previousLineBreak = -1, plain = isPlainSafeFirst(string9.charCodeAt(0)) && !isWhitespace(string9.charCodeAt(string9.length - 1));
    let __char, i28;
    if (singleLineOnly) {
        for(i28 = 0; i28 < string9.length; i28++){
            __char = string9.charCodeAt(i28);
            if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
    } else {
        for(i28 = 0; i28 < string9.length; i28++){
            __char = string9.charCodeAt(i28);
            if (__char === 10) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i28 - previousLineBreak - 1 > lineWidth && string9[previousLineBreak + 1] !== " ";
                    previousLineBreak = i28;
                }
            } else if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i28 - previousLineBreak - 1 > lineWidth && string9[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        return plain && !testAmbiguousType(string9) ? 1 : 2;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string9)) {
        return 5;
    }
    return hasFoldableLine ? 4 : 3;
}
function foldLine(line, width) {
    if (line === "" || line[0] === " ") return line;
    const breakRe = / [^ ]/g;
    let match;
    let start = 0, end, curr = 0, next = 0;
    let result = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result += `\n${line.slice(start, end)}`;
            start = end + 1;
        }
        curr = next;
    }
    result += "\n";
    if (line.length - start > width && curr > start) {
        result += `${line.slice(start, curr)}\n${line.slice(curr + 1)}`;
    } else {
        result += line.slice(start);
    }
    return result.slice(1);
}
function dropEndingNewline(string10) {
    return string10[string10.length - 1] === "\n" ? string10.slice(0, -1) : string10;
}
function foldString(string11, width) {
    const lineRe = /(\n+)([^\n]*)/g;
    let result = (()=>{
        let nextLF = string11.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string11.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string11.slice(0, nextLF), width);
    })();
    let prevMoreIndented = string11[0] === "\n" || string11[0] === " ";
    let moreIndented;
    let match;
    while(match = lineRe.exec(string11)){
        const prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine(line, width);
        prevMoreIndented = moreIndented;
    }
    return result;
}
function escapeString(string12) {
    let result = "";
    let __char, nextChar;
    let escapeSeq;
    for(let i29 = 0; i29 < string12.length; i29++){
        __char = string12.charCodeAt(i29);
        if (__char >= 55296 && __char <= 56319) {
            nextChar = string12.charCodeAt(i29 + 1);
            if (nextChar >= 56320 && nextChar <= 57343) {
                result += encodeHex((__char - 55296) * 1024 + nextChar - 56320 + 65536);
                i29++;
                continue;
            }
        }
        escapeSeq = ESCAPE_SEQUENCES[__char];
        result += !escapeSeq && isPrintable(__char) ? string12[i29] : escapeSeq || encodeHex(__char);
    }
    return result;
}
function blockHeader(string13, indentPerLevel) {
    const indentIndicator = needIndentIndicator(string13) ? String(indentPerLevel) : "";
    const clip = string13[string13.length - 1] === "\n";
    const keep = clip && (string13[string13.length - 2] === "\n" || string13 === "\n");
    const chomp = keep ? "+" : clip ? "" : "-";
    return `${indentIndicator}${chomp}\n`;
}
function writeScalar(state29, string14, level, iskey) {
    state29.dump = (()=>{
        if (string14.length === 0) {
            return "''";
        }
        if (!state29.noCompatMode && DEPRECATED_BOOLEANS_SYNTAX.indexOf(string14) !== -1) {
            return `'${string14}'`;
        }
        const indent = state29.indent * Math.max(1, level);
        const lineWidth = state29.lineWidth === -1 ? -1 : Math.max(Math.min(state29.lineWidth, 40), state29.lineWidth - indent);
        const singleLineOnly = iskey || state29.flowLevel > -1 && level >= state29.flowLevel;
        function testAmbiguity(str67) {
            return testImplicitResolving(state29, str67);
        }
        switch(chooseScalarStyle(string14, singleLineOnly, state29.indent, lineWidth, testAmbiguity)){
            case STYLE_PLAIN:
                return string14;
            case STYLE_SINGLE:
                return `'${string14.replace(/'/g, "''")}'`;
            case STYLE_LITERAL:
                return `|${blockHeader(string14, state29.indent)}${dropEndingNewline(indentString(string14, indent))}`;
            case STYLE_FOLDED:
                return `>${blockHeader(string14, state29.indent)}${dropEndingNewline(indentString(foldString(string14, lineWidth), indent))}`;
            case STYLE_DOUBLE:
                return `"${escapeString(string14)}"`;
            default:
                throw new YAMLError("impossible error: invalid scalar style");
        }
    })();
}
function writeFlowSequence(state30, level, object14) {
    let _result = "";
    const _tag = state30.tag;
    for(let index = 0, length = object14.length; index < length; index += 1){
        if (writeNode(state30, level, object14[index], false, false)) {
            if (index !== 0) _result += `,${!state30.condenseFlow ? " " : ""}`;
            _result += state30.dump;
        }
    }
    state30.tag = _tag;
    state30.dump = `[${_result}]`;
}
function writeBlockSequence(state31, level, object15, compact2 = false) {
    let _result = "";
    const _tag = state31.tag;
    for(let index = 0, length = object15.length; index < length; index += 1){
        if (writeNode(state31, level + 1, object15[index], true, true)) {
            if (!compact2 || index !== 0) {
                _result += generateNextLine(state31, level);
            }
            if (state31.dump && 10 === state31.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state31.dump;
        }
    }
    state31.tag = _tag;
    state31.dump = _result || "[]";
}
function writeFlowMapping(state32, level, object16) {
    let _result = "";
    const _tag = state32.tag, objectKeyList = Object.keys(object16);
    let pairBuffer, objectKey, objectValue;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = state32.condenseFlow ? '"' : "";
        if (index !== 0) pairBuffer += ", ";
        objectKey = objectKeyList[index];
        objectValue = object16[objectKey];
        if (!writeNode(state32, level, objectKey, false, false)) {
            continue;
        }
        if (state32.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += `${state32.dump}${state32.condenseFlow ? '"' : ""}:${state32.condenseFlow ? "" : " "}`;
        if (!writeNode(state32, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state32.dump;
        _result += pairBuffer;
    }
    state32.tag = _tag;
    state32.dump = `{${_result}}`;
}
function writeBlockMapping(state33, level, object17, compact3 = false) {
    const _tag = state33.tag, objectKeyList = Object.keys(object17);
    let _result = "";
    if (state33.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state33.sortKeys === "function") {
        objectKeyList.sort(state33.sortKeys);
    } else if (state33.sortKeys) {
        throw new YAMLError("sortKeys must be a boolean or a function");
    }
    let pairBuffer = "", objectKey, objectValue, explicitPair;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact3 || index !== 0) {
            pairBuffer += generateNextLine(state33, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object17[objectKey];
        if (!writeNode(state33, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state33.tag !== null && state33.tag !== "?" || state33.dump && state33.dump.length > 1024;
        if (explicitPair) {
            if (state33.dump && 10 === state33.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state33.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state33, level);
        }
        if (!writeNode(state33, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state33.dump && 10 === state33.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state33.dump;
        _result += pairBuffer;
    }
    state33.tag = _tag;
    state33.dump = _result || "{}";
}
function detectType(state34, object18, explicit = false) {
    const typeList = explicit ? state34.explicitTypes : state34.implicitTypes;
    let type36;
    let style;
    let _result;
    for(let index = 0, length = typeList.length; index < length; index += 1){
        type36 = typeList[index];
        if ((type36.instanceOf || type36.predicate) && (!type36.instanceOf || typeof object18 === "object" && object18 instanceof type36.instanceOf) && (!type36.predicate || type36.predicate(object18))) {
            state34.tag = explicit ? type36.tag : "?";
            if (type36.represent) {
                style = state34.styleMap[type36.tag] || type36.defaultStyle;
                if (_toString2.call(type36.represent) === "[object Function]") {
                    _result = type36.represent(object18, style);
                } else if (hasOwn4(type36.represent, style)) {
                    _result = type36.represent[style](object18, style);
                } else {
                    throw new YAMLError(`!<${type36.tag}> tag resolver accepts not "${style}" style`);
                }
                state34.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state35, level, object19, block, compact4, iskey = false) {
    state35.tag = null;
    state35.dump = object19;
    if (!detectType(state35, object19, false)) {
        detectType(state35, object19, true);
    }
    const type37 = _toString2.call(state35.dump);
    if (block) {
        block = state35.flowLevel < 0 || state35.flowLevel > level;
    }
    const objectOrArray = type37 === "[object Object]" || type37 === "[object Array]";
    let duplicateIndex = -1;
    let duplicate = false;
    if (objectOrArray) {
        duplicateIndex = state35.duplicates.indexOf(object19);
        duplicate = duplicateIndex !== -1;
    }
    if (state35.tag !== null && state35.tag !== "?" || duplicate || state35.indent !== 2 && level > 0) {
        compact4 = false;
    }
    if (duplicate && state35.usedDuplicates[duplicateIndex]) {
        state35.dump = `*ref_${duplicateIndex}`;
    } else {
        if (objectOrArray && duplicate && !state35.usedDuplicates[duplicateIndex]) {
            state35.usedDuplicates[duplicateIndex] = true;
        }
        if (type37 === "[object Object]") {
            if (block && Object.keys(state35.dump).length !== 0) {
                writeBlockMapping(state35, level, state35.dump, compact4);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex}${state35.dump}`;
                }
            } else {
                writeFlowMapping(state35, level, state35.dump);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex} ${state35.dump}`;
                }
            }
        } else if (type37 === "[object Array]") {
            const arrayLevel = state35.noArrayIndent && level > 0 ? level - 1 : level;
            if (block && state35.dump.length !== 0) {
                writeBlockSequence(state35, arrayLevel, state35.dump, compact4);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex}${state35.dump}`;
                }
            } else {
                writeFlowSequence(state35, arrayLevel, state35.dump);
                if (duplicate) {
                    state35.dump = `&ref_${duplicateIndex} ${state35.dump}`;
                }
            }
        } else if (type37 === "[object String]") {
            if (state35.tag !== "?") {
                writeScalar(state35, state35.dump, level, iskey);
            }
        } else {
            if (state35.skipInvalid) return false;
            throw new YAMLError(`unacceptable kind of an object to dump ${type37}`);
        }
        if (state35.tag !== null && state35.tag !== "?") {
            state35.dump = `!<${state35.tag}> ${state35.dump}`;
        }
    }
    return true;
}
function inspectNode(object20, objects, duplicatesIndexes) {
    if (object20 !== null && typeof object20 === "object") {
        const index = objects.indexOf(object20);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object20);
            if (Array.isArray(object20)) {
                for(let idx = 0, length = object20.length; idx < length; idx += 1){
                    inspectNode(object20[idx], objects, duplicatesIndexes);
                }
            } else {
                const objectKeyList = Object.keys(object20);
                for(let idx = 0, length = objectKeyList.length; idx < length; idx += 1){
                    inspectNode(object20[objectKeyList[idx]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function getDuplicateReferences(object21, state36) {
    const objects = [], duplicatesIndexes = [];
    inspectNode(object21, objects, duplicatesIndexes);
    const length = duplicatesIndexes.length;
    for(let index = 0; index < length; index += 1){
        state36.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state36.usedDuplicates = Array.from({
        length
    });
}
function dump1(input, options) {
    options = options || {
    };
    const state37 = new DumperState(options);
    if (!state37.noRefs) getDuplicateReferences(input, state37);
    if (writeNode(state37, 0, input, true, true)) return `${state37.dump}\n`;
    return "";
}
function stringify(obj, options) {
    return dump1(obj, options);
}
function readAnnotatedYamlFromMappedString(yml) {
    return readAnnotatedYamlFromString(yml.value);
}
function readAnnotatedYamlFromString(yml) {
    const stack = [];
    const results = [];
    function listener(what, state38) {
        const { result , position , kind  } = state38;
        if (what === "close") {
            const { position: openPosition  } = stack.pop();
            if (results.length > 0) {
                const last5 = results[results.length - 1];
                if (last5.start === openPosition && last5.end === position) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last6 = results[results.length - 1];
                if (last6.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position);
            const leftTrim = rawRange.length - rawRange.trimLeft().length;
            const rightTrim = rawRange.length - rawRange.trimRight().length;
            results.push({
                start: openPosition + leftTrim,
                end: position - rightTrim,
                result: result,
                components,
                kind
            });
        } else {
            stack.push({
                position
            });
        }
    }
    parse3(yml, {
        listener
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: []
        };
    }
    if (results.length !== 1) {
        throw new Error(`Internal Error - expected a single result, got ${results.length} instead`);
    }
    return results[0];
}
const BooleanSchema = {
    "type": "boolean",
    "description": "be a boolean value",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const NumberSchema = {
    "type": "number",
    "description": "be a number"
};
const IntegerSchema = {
    "type": "integer",
    "description": "be an integral number"
};
function numericSchema(obj) {
    return Object.assign({
        description: "be a number"
    }, obj);
}
const StringSchema = {
    "type": "string",
    "description": "be a string"
};
const NullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function enumSchema(...args) {
    if (args.length === 0) {
        throw new Error("Internal Error: Empty enum schema not supported.");
    }
    return {
        "enum": args,
        "description": args.length > 1 ? `be one of: ${args.map((x)=>"'" + x + "'"
        ).join(", ")}` : `be '${args[0]}'`,
        "completions": args,
        "exhaustiveCompletions": true
    };
}
function regexSchema(arg, description7) {
    return {
        "type": "string",
        "pattern": arg,
        description: description7
    };
}
function oneOfSchema(...args) {
    return {
        "oneOf": args,
        "description": `be exactly one of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function anyOfSchema(...args) {
    return {
        "anyOf": args,
        "description": `be at least one of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function allOfSchema(...args) {
    return {
        "allOf": args,
        "description": `be all of: ${args.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function objectSchema(params = {
}) {
    let { properties: properties2 , patternProperties: patternProperties1 , required: required2 , additionalProperties , description: description8 , baseSchema , exhaustive , completions: completionsParam ,  } = params;
    required2 = required2 || [];
    properties2 = properties2 || {
    };
    patternProperties1 = patternProperties1 || {
    };
    const hasDescription = description8 !== undefined;
    description8 = description8 || "be an object";
    let result = undefined;
    const completions = [];
    const uniqueValues = (lst)=>{
        const obj = {
        };
        for (const c of lst){
            obj[c.value] = c;
        }
        return Object.getOwnPropertyNames(obj).map((k)=>obj[k]
        );
    };
    for (const k of Object.getOwnPropertyNames(completionsParam || properties2)){
        const valueS = properties2[k];
        completions.push({
            type: "key",
            display: k,
            value: `${k}: `,
            description: completionsParam?.[k] || "",
            suggest_on_accept: valueS && schemaExhaustiveCompletions(valueS)
        });
    }
    if (baseSchema) {
        if (baseSchema.type !== "object") {
            throw new Error("Internal Error: can only extend other object Schema");
        }
        result = Object.assign({
        }, baseSchema);
        if (result.$id) {
            delete result.$id;
        }
        if (exhaustive && baseSchema.exhaustiveCompletions) {
            result.exhaustiveCompletions = true;
        }
        if (hasDescription) {
            result.description = description8;
        }
        result.properties = Object.assign({
        }, result.properties, properties2);
        result.patternProperties = Object.assign({
        }, result.patternProperties, patternProperties1);
        if (required2) {
            result.required = (result.required ?? []).slice();
            result.required.push(...required2);
        }
        if (result.completions && result.completions.length || completions.length) {
            result.completions = (result.completions || []).slice();
            result.completions.push(...completions);
            result.completions = uniqueValues(result.completions);
        }
        if (additionalProperties !== undefined) {
            if (result.additionalProperties === false) {
                throw new Error("Internal Error: don't know how to subclass object schema with additionalProperties === false");
            }
            if (result.additionalProperties) {
                result.additionalProperties = allOfSchema(result.additionalProperties, additionalProperties);
            } else {
                result.additionalProperties = additionalProperties;
            }
        }
    } else {
        result = {
            "type": "object",
            description: description8
        };
        if (exhaustive) {
            result.exhaustiveCompletions = true;
        }
        if (properties2) {
            result.properties = properties2;
        }
        if (patternProperties1) {
            result.patternProperties = patternProperties1;
        }
        if (required2 && required2.length > 0) {
            result.required = required2;
        }
        if (completions.length) {
            result.completions = completions;
        }
        if (additionalProperties !== undefined) {
            result.additionalProperties = additionalProperties;
        }
    }
    return result;
}
function arraySchema(items1) {
    if (items1) {
        return {
            "type": "array",
            "description": `be an array of values, where each element should ` + items1.description,
            items: items1
        };
    } else {
        return {
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema, doc) {
    const result = Object.assign({
    }, schema);
    result.documentation = doc;
    return result;
}
function describeSchema(schema, description9) {
    const result = Object.assign({
    }, schema);
    result.description = `be ${description9}`;
    return result;
}
function completeSchema(schema, ...completions) {
    const result = Object.assign({
    }, schema);
    const prevCompletions = (schema.completions ?? []).slice();
    prevCompletions.push(...completions);
    result.completions = prevCompletions;
    return result;
}
function completeSchemaOverwrite(schema, ...completions) {
    const result = Object.assign({
    }, schema);
    result.completions = completions;
    return result;
}
function idSchema(schema, id2) {
    const result = Object.assign({
    }, schema);
    result["$id"] = id2;
    return result;
}
function refSchema($ref, description10) {
    return {
        $ref,
        description: description10
    };
}
function valueSchema(val) {
    return {
        "enum": [
            val
        ]
    };
}
function setBaseSchemaProperties(yaml, schema) {
    if (yaml.additionalCompletions) schema = completeSchema(schema, yaml.additionalCompletions);
    if (yaml.completions) schema = completeSchemaOverwrite(schema, yaml.completions);
    if (yaml.id) schema = idSchema(schema, yaml.id);
    if (yaml.hidden === false) {
        schema = completeSchemaOverwrite(schema);
    }
    if (yaml.description) {
        if (typeof yaml.description === "string") {
            schema = documentSchema(schema, yaml.description);
        } else if (typeof yaml.description === "object") {
            schema = documentSchema(schema, yaml.description.short);
        }
    }
    return schema;
}
function convertFromNull(yaml, _dict) {
    return setBaseSchemaProperties(yaml, NullSchema);
}
function convertFromString(yaml, _dict) {
    return setBaseSchemaProperties(yaml, StringSchema);
}
function convertFromPath(yaml, _dict) {
    return setBaseSchemaProperties(yaml, StringSchema);
}
function convertFromNumber(yaml, _dict) {
    return setBaseSchemaProperties(yaml, NumberSchema);
}
function convertFromBoolean(yaml, _dict) {
    return setBaseSchemaProperties(yaml, BooleanSchema);
}
function convertFromRef(yaml, _dict) {
    return refSchema(yaml.ref, yaml.description || "");
}
function convertFromMaybeArrayOf(yaml, dict) {
    const schema = convertFromYaml(yaml.maybeArrayOf, dict);
    return oneOfSchema(schema, arraySchema(schema));
}
function convertFromArrayOf(yaml, dict) {
    const schema = yaml.arrayOf;
    if (schema.schema) {
        let result = arraySchema(convertFromYaml(schema.schema, dict));
        return setBaseSchemaProperties(schema, result);
    } else {
        return arraySchema(convertFromYaml(schema, dict));
    }
}
function convertFromOneOf(yaml, dict) {
    const schema = yaml.oneOf;
    if (schema.schemas) {
        let inner = schema.schemas.map((x)=>convertFromYaml(x, dict)
        );
        let result = oneOfSchema(...inner);
        return setBaseSchemaProperties(schema, result);
    } else {
        return oneOfSchema(...schema.map((x)=>convertFromYaml(x, dict)
        ));
    }
}
function convertFromAnyOf(yaml, dict) {
    const schema = yaml.anyOf;
    if (schema.schemas) {
        let result = anyOfSchema(...schema.schemas.map((x)=>convertFromYaml(x, dict)
        ));
        return setBaseSchemaProperties(schema, result);
    } else {
        return anyOfSchema(...schema.map((x)=>convertFromYaml(x, dict)
        ));
    }
}
function convertFromEnum(yaml, _dict) {
    const schema = yaml["enum"];
    if (schema.hasOwnProperty("values")) {
        let result = enumSchema(...schema.values);
        return setBaseSchemaProperties(schema, result);
    } else {
        return enumSchema(...schema);
    }
}
function convertFromObject(yaml, dict) {
    const schema = yaml["object"];
    const params = {
    };
    if (schema.properties) {
        params.properties = Object.fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value, dict)
            ]
        ));
    }
    if (schema.patternProperties) {
        params.patternProperties = Object.fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value, dict)
            ]
        ));
    }
    if (schema.additionalProperties !== undefined) {
        if (schema.additionalProperties === false) {
            params.additionalProperties = false;
        } else {
            params.additionalProperties = convertFromYaml(schema.additionalProperties, dict);
        }
    }
    if (schema["super"]) {
        params.baseSchema = convertFromYaml(schema["super"], dict);
    }
    if (schema["required"] === "all") {
        params.required = Object.keys(schema.properties || {
        });
    } else if (schema["required"]) {
        params.required = schema["required"];
    }
    if (schema["completions"]) {
        params.completions = schema["completions"];
    }
    return setBaseSchemaProperties(schema, objectSchema(params));
}
function lookup(yaml, dict) {
    if (dict[yaml.resolveRef] === undefined) {
        throw new Error(`lookup of key ${yaml.resolveRef} failed`);
    }
    return dict[yaml.resolveRef];
}
function convertFromYaml(yaml, dict) {
    dict = dict ?? {
    };
    const literalValues = [
        [
            "object",
            objectSchema()
        ],
        [
            "path",
            StringSchema
        ],
        [
            "string",
            StringSchema
        ],
        [
            "number",
            NumberSchema
        ],
        [
            "boolean",
            BooleanSchema
        ],
        [
            null,
            NullSchema
        ], 
    ];
    for (const [testVal, result] of literalValues){
        if (yaml === testVal) {
            return result;
        }
    }
    if (typeof yaml !== "object") {
        return valueSchema(yaml);
    }
    const schemaObjectKeyFunctions = [
        {
            key: "anyOf",
            value: convertFromAnyOf
        },
        {
            key: "boolean",
            value: convertFromBoolean
        },
        {
            key: "arrayOf",
            value: convertFromArrayOf
        },
        {
            key: "enum",
            value: convertFromEnum
        },
        {
            key: "maybeArrayOf",
            value: convertFromMaybeArrayOf
        },
        {
            key: "null",
            value: convertFromNull
        },
        {
            key: "number",
            value: convertFromNumber
        },
        {
            key: "object",
            value: convertFromObject
        },
        {
            key: "oneOf",
            value: convertFromOneOf
        },
        {
            key: "path",
            value: convertFromPath
        },
        {
            key: "ref",
            value: convertFromRef
        },
        {
            key: "resolveRef",
            value: lookup
        },
        {
            key: "string",
            value: convertFromString
        }, 
    ];
    for (const { key: objectKey , value: fun  } of schemaObjectKeyFunctions){
        try {
            if (yaml[objectKey] !== undefined) {
                return fun(yaml, dict);
            }
        } catch (e) {
            error({
                yaml
            });
            throw e;
        }
    }
    error(JSON.stringify(yaml, null, 2));
    throw new Error("Internal Error: Cannot convert object; this should have failed validation.");
}
var fastDeepEqual = function equal(a, b) {
    if (a === b) return true;
    if (a && b && typeof a == "object" && typeof b == "object") {
        if (a.constructor !== b.constructor) return false;
        var length, i30, keys6;
        if (Array.isArray(a)) {
            length = a.length;
            if (length != b.length) return false;
            for(i30 = length; (i30--) !== 0;)if (!equal(a[i30], b[i30])) return false;
            return true;
        }
        if (a.constructor === RegExp) return a.source === b.source && a.flags === b.flags;
        if (a.valueOf !== Object.prototype.valueOf) return a.valueOf() === b.valueOf();
        if (a.toString !== Object.prototype.toString) return a.toString() === b.toString();
        keys6 = Object.keys(a);
        length = keys6.length;
        if (length !== Object.keys(b).length) return false;
        for(i30 = length; (i30--) !== 0;)if (!Object.prototype.hasOwnProperty.call(b, keys6[i30])) return false;
        for(i30 = length; (i30--) !== 0;){
            var key = keys6[i30];
            if (!equal(a[key], b[key])) return false;
        }
        return true;
    }
    return a !== a && b !== b;
};
function createCommonjsModule(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path83, base4) {
            return commonjsRequire(path83, base4 === void 0 || base4 === null ? module.path : base4);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var jsonSchemaTraverse = createCommonjsModule(function(module) {
    var traverse = module.exports = function(schema, opts, cb) {
        if (typeof opts == "function") {
            cb = opts;
            opts = {
            };
        }
        cb = opts.cb || cb;
        var pre = typeof cb == "function" ? cb : cb.pre || function() {
        };
        var post = cb.post || function() {
        };
        _traverse(opts, pre, post, schema, "", schema);
    };
    traverse.keywords = {
        additionalItems: true,
        items: true,
        contains: true,
        additionalProperties: true,
        propertyNames: true,
        not: true,
        if: true,
        then: true,
        else: true
    };
    traverse.arrayKeywords = {
        items: true,
        allOf: true,
        anyOf: true,
        oneOf: true
    };
    traverse.propsKeywords = {
        $defs: true,
        definitions: true,
        properties: true,
        patternProperties: true,
        dependencies: true
    };
    traverse.skipKeywords = {
        default: true,
        enum: true,
        const: true,
        required: true,
        maximum: true,
        minimum: true,
        exclusiveMaximum: true,
        exclusiveMinimum: true,
        multipleOf: true,
        maxLength: true,
        minLength: true,
        pattern: true,
        format: true,
        maxItems: true,
        minItems: true,
        uniqueItems: true,
        maxProperties: true,
        minProperties: true
    };
    function _traverse(opts, pre, post, schema, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex) {
        if (schema && typeof schema == "object" && !Array.isArray(schema)) {
            pre(schema, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex);
            for(var key in schema){
                var sch = schema[key];
                if (Array.isArray(sch)) {
                    if (key in traverse.arrayKeywords) {
                        for(var i31 = 0; i31 < sch.length; i31++)_traverse(opts, pre, post, sch[i31], jsonPtr + "/" + key + "/" + i31, rootSchema, jsonPtr, key, schema, i31);
                    }
                } else if (key in traverse.propsKeywords) {
                    if (sch && typeof sch == "object") {
                        for(var prop in sch)_traverse(opts, pre, post, sch[prop], jsonPtr + "/" + key + "/" + escapeJsonPtr(prop), rootSchema, jsonPtr, key, schema, prop);
                    }
                } else if (key in traverse.keywords || opts.allKeys && !(key in traverse.skipKeywords)) {
                    _traverse(opts, pre, post, sch, jsonPtr + "/" + key, rootSchema, jsonPtr, key, schema);
                }
            }
            post(schema, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex);
        }
    }
    function escapeJsonPtr(str68) {
        return str68.replace(/~/g, "~0").replace(/\//g, "~1");
    }
});
jsonSchemaTraverse.arrayKeywords;
jsonSchemaTraverse.keywords;
jsonSchemaTraverse.propsKeywords;
jsonSchemaTraverse.skipKeywords;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {
};
function getDefaultExportFromCjs(x) {
    return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, "default") ? x["default"] : x;
}
function createCommonjsModule1(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path84, base5) {
            return commonjsRequire1(path84, base5 === void 0 || base5 === null ? module.path : base5);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire1() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var uri_all = createCommonjsModule1(function(module, exports) {
    (function(global2, factory3) {
        factory3(exports);
    })(commonjsGlobal, function(exports2) {
        function merge2() {
            for(var _len = arguments.length, sets = Array(_len), _key = 0; _key < _len; _key++){
                sets[_key] = arguments[_key];
            }
            if (sets.length > 1) {
                sets[0] = sets[0].slice(0, -1);
                var xl = sets.length - 1;
                for(var x = 1; x < xl; ++x){
                    sets[x] = sets[x].slice(1, -1);
                }
                sets[xl] = sets[xl].slice(1);
                return sets.join("");
            } else {
                return sets[0];
            }
        }
        function subexp(str69) {
            return "(?:" + str69 + ")";
        }
        function typeOf(o) {
            return o === void 0 ? "undefined" : o === null ? "null" : Object.prototype.toString.call(o).split(" ").pop().split("]").shift().toLowerCase();
        }
        function toUpperCase(str70) {
            return str70.toUpperCase();
        }
        function toArray1(obj) {
            return obj !== void 0 && obj !== null ? obj instanceof Array ? obj : typeof obj.length !== "number" || obj.split || obj.setInterval || obj.call ? [
                obj
            ] : Array.prototype.slice.call(obj) : [];
        }
        function assign1(target, source) {
            var obj = target;
            if (source) {
                for(var key in source){
                    obj[key] = source[key];
                }
            }
            return obj;
        }
        function buildExps(isIRI) {
            var ALPHA$$ = "[A-Za-z]", DIGIT$$ = "[0-9]", HEXDIG$$2 = merge2(DIGIT$$, "[A-Fa-f]"), PCT_ENCODED$2 = subexp(subexp("%[EFef]" + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2) + "|" + subexp("%[89A-Fa-f]" + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2) + "|" + subexp("%" + HEXDIG$$2 + HEXDIG$$2)), GEN_DELIMS$$ = "[\\:\\/\\?\\#\\[\\]\\@]", SUB_DELIMS$$ = "[\\!\\$\\&\\'\\(\\)\\*\\+\\,\\;\\=]", RESERVED$$ = merge2(GEN_DELIMS$$, SUB_DELIMS$$), UCSCHAR$$ = isIRI ? "[\\xA0-\\u200D\\u2010-\\u2029\\u202F-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]" : "[]", IPRIVATE$$ = isIRI ? "[\\uE000-\\uF8FF]" : "[]", UNRESERVED$$2 = merge2(ALPHA$$, DIGIT$$, "[\\-\\.\\_\\~]", UCSCHAR$$), SCHEME$ = subexp(ALPHA$$ + merge2(ALPHA$$, DIGIT$$, "[\\+\\-\\.]") + "*"), USERINFO$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge2(UNRESERVED$$2, SUB_DELIMS$$, "[\\:]")) + "*"), DEC_OCTET_RELAXED$ = subexp(subexp("25[0-5]") + "|" + subexp("2[0-4]" + DIGIT$$) + "|" + subexp("1" + DIGIT$$ + DIGIT$$) + "|" + subexp("0?[1-9]" + DIGIT$$) + "|0?0?" + DIGIT$$), IPV4ADDRESS$ = subexp(DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$), H16$ = subexp(HEXDIG$$2 + "{1,4}"), LS32$ = subexp(subexp(H16$ + "\\:" + H16$) + "|" + IPV4ADDRESS$), IPV6ADDRESS1$ = subexp(subexp(H16$ + "\\:") + "{6}" + LS32$), IPV6ADDRESS2$ = subexp("\\:\\:" + subexp(H16$ + "\\:") + "{5}" + LS32$), IPV6ADDRESS3$ = subexp(subexp(H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{4}" + LS32$), IPV6ADDRESS4$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,1}" + H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{3}" + LS32$), IPV6ADDRESS5$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,2}" + H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{2}" + LS32$), IPV6ADDRESS6$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,3}" + H16$) + "?\\:\\:" + H16$ + "\\:" + LS32$), IPV6ADDRESS7$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,4}" + H16$) + "?\\:\\:" + LS32$), IPV6ADDRESS8$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,5}" + H16$) + "?\\:\\:" + H16$), IPV6ADDRESS9$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,6}" + H16$) + "?\\:\\:"), IPV6ADDRESS$ = subexp([
                IPV6ADDRESS1$,
                IPV6ADDRESS2$,
                IPV6ADDRESS3$,
                IPV6ADDRESS4$,
                IPV6ADDRESS5$,
                IPV6ADDRESS6$,
                IPV6ADDRESS7$,
                IPV6ADDRESS8$,
                IPV6ADDRESS9$
            ].join("|")), ZONEID$ = subexp(subexp(UNRESERVED$$2 + "|" + PCT_ENCODED$2) + "+"), IPVFUTURE$ = subexp("[vV]" + HEXDIG$$2 + "+\\." + merge2(UNRESERVED$$2, SUB_DELIMS$$, "[\\:]") + "+"), REG_NAME$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge2(UNRESERVED$$2, SUB_DELIMS$$)) + "*"), PCHAR$ = subexp(PCT_ENCODED$2 + "|" + merge2(UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@]")), SEGMENT_NZ_NC$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge2(UNRESERVED$$2, SUB_DELIMS$$, "[\\@]")) + "+"), QUERY$ = subexp(subexp(PCHAR$ + "|" + merge2("[\\/\\?]", IPRIVATE$$)) + "*");
            return {
                NOT_SCHEME: new RegExp(merge2("[^]", ALPHA$$, DIGIT$$, "[\\+\\-\\.]"), "g"),
                NOT_USERINFO: new RegExp(merge2("[^\\%\\:]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_HOST: new RegExp(merge2("[^\\%\\[\\]\\:]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_PATH: new RegExp(merge2("[^\\%\\/\\:\\@]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_PATH_NOSCHEME: new RegExp(merge2("[^\\%\\/\\@]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_QUERY: new RegExp(merge2("[^\\%]", UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@\\/\\?]", IPRIVATE$$), "g"),
                NOT_FRAGMENT: new RegExp(merge2("[^\\%]", UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@\\/\\?]"), "g"),
                ESCAPE: new RegExp(merge2("[^]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                UNRESERVED: new RegExp(UNRESERVED$$2, "g"),
                OTHER_CHARS: new RegExp(merge2("[^\\%]", UNRESERVED$$2, RESERVED$$), "g"),
                PCT_ENCODED: new RegExp(PCT_ENCODED$2, "g"),
                IPV4ADDRESS: new RegExp("^(" + IPV4ADDRESS$ + ")$"),
                IPV6ADDRESS: new RegExp("^\\[?(" + IPV6ADDRESS$ + ")" + subexp(subexp("\\%25|\\%(?!" + HEXDIG$$2 + "{2})") + "(" + ZONEID$ + ")") + "?\\]?$")
            };
        }
        var URI_PROTOCOL = buildExps(false);
        var IRI_PROTOCOL = buildExps(true);
        var slicedToArray = function() {
            function sliceIterator(arr, i32) {
                var _arr = [];
                var _n = true;
                var _d = false;
                var _e = void 0;
                try {
                    for(var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true){
                        _arr.push(_s.value);
                        if (i32 && _arr.length === i32) break;
                    }
                } catch (err) {
                    _d = true;
                    _e = err;
                } finally{
                    try {
                        if (!_n && _i["return"]) _i["return"]();
                    } finally{
                        if (_d) throw _e;
                    }
                }
                return _arr;
            }
            return function(arr, i33) {
                if (Array.isArray(arr)) {
                    return arr;
                } else if (Symbol.iterator in Object(arr)) {
                    return sliceIterator(arr, i33);
                } else {
                    throw new TypeError("Invalid attempt to destructure non-iterable instance");
                }
            };
        }();
        var toConsumableArray = function(arr) {
            if (Array.isArray(arr)) {
                for(var i34 = 0, arr2 = Array(arr.length); i34 < arr.length; i34++)arr2[i34] = arr[i34];
                return arr2;
            } else {
                return Array.from(arr);
            }
        };
        var maxInt = 2147483647;
        var base6 = 36;
        var tMin = 1;
        var tMax = 26;
        var skew = 38;
        var damp = 700;
        var initialBias = 72;
        var initialN = 128;
        var delimiter6 = "-";
        var regexPunycode = /^xn--/;
        var regexNonASCII = /[^\0-\x7E]/;
        var regexSeparators = /[\x2E\u3002\uFF0E\uFF61]/g;
        var errors5 = {
            overflow: "Overflow: input needs wider integers to process",
            "not-basic": "Illegal input >= 0x80 (not a basic code point)",
            "invalid-input": "Invalid input"
        };
        var baseMinusTMin = base6 - tMin;
        var floor1 = Math.floor;
        var stringFromCharCode = String.fromCharCode;
        function error$1(type) {
            throw new RangeError(errors5[type]);
        }
        function map7(array3, fn) {
            var result = [];
            var length = array3.length;
            while(length--){
                result[length] = fn(array3[length]);
            }
            return result;
        }
        function mapDomain(string15, fn) {
            var parts = string15.split("@");
            var result = "";
            if (parts.length > 1) {
                result = parts[0] + "@";
                string15 = parts[1];
            }
            string15 = string15.replace(regexSeparators, ".");
            var labels = string15.split(".");
            var encoded = map7(labels, fn).join(".");
            return result + encoded;
        }
        function ucs2decode(string16) {
            var output = [];
            var counter = 0;
            var length = string16.length;
            while(counter < length){
                var value = string16.charCodeAt(counter++);
                if (value >= 55296 && value <= 56319 && counter < length) {
                    var extra = string16.charCodeAt(counter++);
                    if ((extra & 64512) == 56320) {
                        output.push(((value & 1023) << 10) + (extra & 1023) + 65536);
                    } else {
                        output.push(value);
                        counter--;
                    }
                } else {
                    output.push(value);
                }
            }
            return output;
        }
        var ucs2encode = function ucs2encode2(array4) {
            return String.fromCodePoint.apply(String, toConsumableArray(array4));
        };
        var basicToDigit = function basicToDigit2(codePoint) {
            if (codePoint - 48 < 10) {
                return codePoint - 22;
            }
            if (codePoint - 65 < 26) {
                return codePoint - 65;
            }
            if (codePoint - 97 < 26) {
                return codePoint - 97;
            }
            return base6;
        };
        var digitToBasic = function digitToBasic2(digit, flag) {
            return digit + 22 + 75 * (digit < 26) - ((flag != 0) << 5);
        };
        var adapt = function adapt2(delta, numPoints, firstTime) {
            var k = 0;
            delta = firstTime ? floor1(delta / damp) : delta >> 1;
            delta += floor1(delta / numPoints);
            for(; delta > baseMinusTMin * tMax >> 1; k += base6){
                delta = floor1(delta / baseMinusTMin);
            }
            return floor1(k + (baseMinusTMin + 1) * delta / (delta + skew));
        };
        var decode1 = function decode2(input) {
            var output = [];
            var inputLength = input.length;
            var i35 = 0;
            var n22 = initialN;
            var bias = initialBias;
            var basic = input.lastIndexOf(delimiter6);
            if (basic < 0) {
                basic = 0;
            }
            for(var j = 0; j < basic; ++j){
                if (input.charCodeAt(j) >= 128) {
                    error$1("not-basic");
                }
                output.push(input.charCodeAt(j));
            }
            for(var index = basic > 0 ? basic + 1 : 0; index < inputLength;){
                var oldi = i35;
                for(var w = 1, k = base6;; k += base6){
                    if (index >= inputLength) {
                        error$1("invalid-input");
                    }
                    var digit = basicToDigit(input.charCodeAt(index++));
                    if (digit >= base6 || digit > floor1((maxInt - i35) / w)) {
                        error$1("overflow");
                    }
                    i35 += digit * w;
                    var t = k <= bias ? tMin : k >= bias + tMax ? tMax : k - bias;
                    if (digit < t) {
                        break;
                    }
                    var baseMinusT = base6 - t;
                    if (w > floor1(maxInt / baseMinusT)) {
                        error$1("overflow");
                    }
                    w *= baseMinusT;
                }
                var out = output.length + 1;
                bias = adapt(i35 - oldi, out, oldi == 0);
                if (floor1(i35 / out) > maxInt - n22) {
                    error$1("overflow");
                }
                n22 += floor1(i35 / out);
                i35 %= out;
                output.splice(i35++, 0, n22);
            }
            return String.fromCodePoint.apply(String, output);
        };
        var encode2 = function encode2(input) {
            var output = [];
            input = ucs2decode(input);
            var inputLength = input.length;
            var n23 = initialN;
            var delta = 0;
            var bias = initialBias;
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = void 0;
            try {
                for(var _iterator = input[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true){
                    var _currentValue2 = _step.value;
                    if (_currentValue2 < 128) {
                        output.push(stringFromCharCode(_currentValue2));
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally{
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally{
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }
            var basicLength = output.length;
            var handledCPCount = basicLength;
            if (basicLength) {
                output.push(delimiter6);
            }
            while(handledCPCount < inputLength){
                var m = maxInt;
                var _iteratorNormalCompletion2 = true;
                var _didIteratorError2 = false;
                var _iteratorError2 = void 0;
                try {
                    for(var _iterator2 = input[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true){
                        var currentValue = _step2.value;
                        if (currentValue >= n23 && currentValue < m) {
                            m = currentValue;
                        }
                    }
                } catch (err) {
                    _didIteratorError2 = true;
                    _iteratorError2 = err;
                } finally{
                    try {
                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                            _iterator2.return();
                        }
                    } finally{
                        if (_didIteratorError2) {
                            throw _iteratorError2;
                        }
                    }
                }
                var handledCPCountPlusOne = handledCPCount + 1;
                if (m - n23 > floor1((maxInt - delta) / handledCPCountPlusOne)) {
                    error$1("overflow");
                }
                delta += (m - n23) * handledCPCountPlusOne;
                n23 = m;
                var _iteratorNormalCompletion3 = true;
                var _didIteratorError3 = false;
                var _iteratorError3 = void 0;
                try {
                    for(var _iterator3 = input[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true){
                        var _currentValue = _step3.value;
                        if (_currentValue < n23 && ++delta > maxInt) {
                            error$1("overflow");
                        }
                        if (_currentValue == n23) {
                            var q = delta;
                            for(var k = base6;; k += base6){
                                var t = k <= bias ? tMin : k >= bias + tMax ? tMax : k - bias;
                                if (q < t) {
                                    break;
                                }
                                var qMinusT = q - t;
                                var baseMinusT = base6 - t;
                                output.push(stringFromCharCode(digitToBasic(t + qMinusT % baseMinusT, 0)));
                                q = floor1(qMinusT / baseMinusT);
                            }
                            output.push(stringFromCharCode(digitToBasic(q, 0)));
                            bias = adapt(delta, handledCPCountPlusOne, handledCPCount == basicLength);
                            delta = 0;
                            ++handledCPCount;
                        }
                    }
                } catch (err1) {
                    _didIteratorError3 = true;
                    _iteratorError3 = err1;
                } finally{
                    try {
                        if (!_iteratorNormalCompletion3 && _iterator3.return) {
                            _iterator3.return();
                        }
                    } finally{
                        if (_didIteratorError3) {
                            throw _iteratorError3;
                        }
                    }
                }
                ++delta;
                ++n23;
            }
            return output.join("");
        };
        var toUnicode = function toUnicode2(input) {
            return mapDomain(input, function(string17) {
                return regexPunycode.test(string17) ? decode1(string17.slice(4).toLowerCase()) : string17;
            });
        };
        var toASCII = function toASCII2(input) {
            return mapDomain(input, function(string18) {
                return regexNonASCII.test(string18) ? "xn--" + encode2(string18) : string18;
            });
        };
        var punycode = {
            version: "2.1.0",
            ucs2: {
                decode: ucs2decode,
                encode: ucs2encode
            },
            decode: decode1,
            encode: encode2,
            toASCII,
            toUnicode
        };
        var SCHEMES2 = {
        };
        function pctEncChar2(chr) {
            var c = chr.charCodeAt(0);
            var e = void 0;
            if (c < 16) e = "%0" + c.toString(16).toUpperCase();
            else if (c < 128) e = "%" + c.toString(16).toUpperCase();
            else if (c < 2048) e = "%" + (c >> 6 | 192).toString(16).toUpperCase() + "%" + (c & 63 | 128).toString(16).toUpperCase();
            else e = "%" + (c >> 12 | 224).toString(16).toUpperCase() + "%" + (c >> 6 & 63 | 128).toString(16).toUpperCase() + "%" + (c & 63 | 128).toString(16).toUpperCase();
            return e;
        }
        function pctDecChars2(str71) {
            var newStr = "";
            var i36 = 0;
            var il = str71.length;
            while(i36 < il){
                var c = parseInt(str71.substr(i36 + 1, 2), 16);
                if (c < 128) {
                    newStr += String.fromCharCode(c);
                    i36 += 3;
                } else if (c >= 194 && c < 224) {
                    if (il - i36 >= 6) {
                        var c2 = parseInt(str71.substr(i36 + 4, 2), 16);
                        newStr += String.fromCharCode((c & 31) << 6 | c2 & 63);
                    } else {
                        newStr += str71.substr(i36, 6);
                    }
                    i36 += 6;
                } else if (c >= 224) {
                    if (il - i36 >= 9) {
                        var _c = parseInt(str71.substr(i36 + 4, 2), 16);
                        var c3 = parseInt(str71.substr(i36 + 7, 2), 16);
                        newStr += String.fromCharCode((c & 15) << 12 | (_c & 63) << 6 | c3 & 63);
                    } else {
                        newStr += str71.substr(i36, 9);
                    }
                    i36 += 9;
                } else {
                    newStr += str71.substr(i36, 3);
                    i36 += 3;
                }
            }
            return newStr;
        }
        function _normalizeComponentEncoding(components, protocol) {
            function decodeUnreserved2(str72) {
                var decStr = pctDecChars2(str72);
                return !decStr.match(protocol.UNRESERVED) ? str72 : decStr;
            }
            if (components.scheme) components.scheme = String(components.scheme).replace(protocol.PCT_ENCODED, decodeUnreserved2).toLowerCase().replace(protocol.NOT_SCHEME, "");
            if (components.userinfo !== void 0) components.userinfo = String(components.userinfo).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_USERINFO, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.host !== void 0) components.host = String(components.host).replace(protocol.PCT_ENCODED, decodeUnreserved2).toLowerCase().replace(protocol.NOT_HOST, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.path !== void 0) components.path = String(components.path).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(components.scheme ? protocol.NOT_PATH : protocol.NOT_PATH_NOSCHEME, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.query !== void 0) components.query = String(components.query).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_QUERY, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.fragment !== void 0) components.fragment = String(components.fragment).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_FRAGMENT, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            return components;
        }
        function _stripLeadingZeros(str73) {
            return str73.replace(/^0*(.*)/, "$1") || "0";
        }
        function _normalizeIPv4(host, protocol) {
            var matches1 = host.match(protocol.IPV4ADDRESS) || [];
            var _matches = slicedToArray(matches1, 2), address = _matches[1];
            if (address) {
                return address.split(".").map(_stripLeadingZeros).join(".");
            } else {
                return host;
            }
        }
        function _normalizeIPv6(host, protocol) {
            var matches2 = host.match(protocol.IPV6ADDRESS) || [];
            var _matches2 = slicedToArray(matches2, 3), address = _matches2[1], zone = _matches2[2];
            if (address) {
                var _address$toLowerCase$ = address.toLowerCase().split("::").reverse(), _address$toLowerCase$2 = slicedToArray(_address$toLowerCase$, 2), last7 = _address$toLowerCase$2[0], first = _address$toLowerCase$2[1];
                var firstFields = first ? first.split(":").map(_stripLeadingZeros) : [];
                var lastFields = last7.split(":").map(_stripLeadingZeros);
                var isLastFieldIPv4Address = protocol.IPV4ADDRESS.test(lastFields[lastFields.length - 1]);
                var fieldCount = isLastFieldIPv4Address ? 7 : 8;
                var lastFieldsStart = lastFields.length - fieldCount;
                var fields = Array(fieldCount);
                for(var x = 0; x < fieldCount; ++x){
                    fields[x] = firstFields[x] || lastFields[lastFieldsStart + x] || "";
                }
                if (isLastFieldIPv4Address) {
                    fields[fieldCount - 1] = _normalizeIPv4(fields[fieldCount - 1], protocol);
                }
                var allZeroFields = fields.reduce(function(acc, field, index) {
                    if (!field || field === "0") {
                        var lastLongest = acc[acc.length - 1];
                        if (lastLongest && lastLongest.index + lastLongest.length === index) {
                            lastLongest.length++;
                        } else {
                            acc.push({
                                index,
                                length: 1
                            });
                        }
                    }
                    return acc;
                }, []);
                var longestZeroFields = allZeroFields.sort(function(a, b) {
                    return b.length - a.length;
                })[0];
                var newHost = void 0;
                if (longestZeroFields && longestZeroFields.length > 1) {
                    var newFirst = fields.slice(0, longestZeroFields.index);
                    var newLast = fields.slice(longestZeroFields.index + longestZeroFields.length);
                    newHost = newFirst.join(":") + "::" + newLast.join(":");
                } else {
                    newHost = fields.join(":");
                }
                if (zone) {
                    newHost += "%" + zone;
                }
                return newHost;
            } else {
                return host;
            }
        }
        var URI_PARSE = /^(?:([^:\/?#]+):)?(?:\/\/((?:([^\/?#@]*)@)?(\[[^\/?#\]]+\]|[^\/?#:]*)(?:\:(\d*))?))?([^?#]*)(?:\?([^#]*))?(?:#((?:.|\n|\r)*))?/i;
        var NO_MATCH_IS_UNDEFINED = "".match(/(){0}/)[1] === void 0;
        function parse21(uriString) {
            var options = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : {
            };
            var components = {
            };
            var protocol = options.iri !== false ? IRI_PROTOCOL : URI_PROTOCOL;
            if (options.reference === "suffix") uriString = (options.scheme ? options.scheme + ":" : "") + "//" + uriString;
            var matches3 = uriString.match(URI_PARSE);
            if (matches3) {
                if (NO_MATCH_IS_UNDEFINED) {
                    components.scheme = matches3[1];
                    components.userinfo = matches3[3];
                    components.host = matches3[4];
                    components.port = parseInt(matches3[5], 10);
                    components.path = matches3[6] || "";
                    components.query = matches3[7];
                    components.fragment = matches3[8];
                    if (isNaN(components.port)) {
                        components.port = matches3[5];
                    }
                } else {
                    components.scheme = matches3[1] || void 0;
                    components.userinfo = uriString.indexOf("@") !== -1 ? matches3[3] : void 0;
                    components.host = uriString.indexOf("//") !== -1 ? matches3[4] : void 0;
                    components.port = parseInt(matches3[5], 10);
                    components.path = matches3[6] || "";
                    components.query = uriString.indexOf("?") !== -1 ? matches3[7] : void 0;
                    components.fragment = uriString.indexOf("#") !== -1 ? matches3[8] : void 0;
                    if (isNaN(components.port)) {
                        components.port = uriString.match(/\/\/(?:.|\n)*\:(?:\/|\?|\#|$)/) ? matches3[4] : void 0;
                    }
                }
                if (components.host) {
                    components.host = _normalizeIPv6(_normalizeIPv4(components.host, protocol), protocol);
                }
                if (components.scheme === void 0 && components.userinfo === void 0 && components.host === void 0 && components.port === void 0 && !components.path && components.query === void 0) {
                    components.reference = "same-document";
                } else if (components.scheme === void 0) {
                    components.reference = "relative";
                } else if (components.fragment === void 0) {
                    components.reference = "absolute";
                } else {
                    components.reference = "uri";
                }
                if (options.reference && options.reference !== "suffix" && options.reference !== components.reference) {
                    components.error = components.error || "URI is not a " + options.reference + " reference.";
                }
                var schemeHandler = SCHEMES2[(options.scheme || components.scheme || "").toLowerCase()];
                if (!options.unicodeSupport && (!schemeHandler || !schemeHandler.unicodeSupport)) {
                    if (components.host && (options.domainHost || schemeHandler && schemeHandler.domainHost)) {
                        try {
                            components.host = punycode.toASCII(components.host.replace(protocol.PCT_ENCODED, pctDecChars2).toLowerCase());
                        } catch (e) {
                            components.error = components.error || "Host's domain name can not be converted to ASCII via punycode: " + e;
                        }
                    }
                    _normalizeComponentEncoding(components, URI_PROTOCOL);
                } else {
                    _normalizeComponentEncoding(components, protocol);
                }
                if (schemeHandler && schemeHandler.parse) {
                    schemeHandler.parse(components, options);
                }
            } else {
                components.error = components.error || "URI can not be parsed.";
            }
            return components;
        }
        function _recomposeAuthority(components, options) {
            var protocol = options.iri !== false ? IRI_PROTOCOL : URI_PROTOCOL;
            var uriTokens = [];
            if (components.userinfo !== void 0) {
                uriTokens.push(components.userinfo);
                uriTokens.push("@");
            }
            if (components.host !== void 0) {
                uriTokens.push(_normalizeIPv6(_normalizeIPv4(String(components.host), protocol), protocol).replace(protocol.IPV6ADDRESS, function(_, $1, $2) {
                    return "[" + $1 + ($2 ? "%25" + $2 : "") + "]";
                }));
            }
            if (typeof components.port === "number" || typeof components.port === "string") {
                uriTokens.push(":");
                uriTokens.push(String(components.port));
            }
            return uriTokens.length ? uriTokens.join("") : void 0;
        }
        var RDS1 = /^\.\.?\//;
        var RDS2 = /^\/\.(\/|$)/;
        var RDS3 = /^\/\.\.(\/|$)/;
        var RDS5 = /^\/?(?:.|\n)*?(?=\/|$)/;
        function removeDotSegments2(input) {
            var output = [];
            while(input.length){
                if (input.match(RDS1)) {
                    input = input.replace(RDS1, "");
                } else if (input.match(RDS2)) {
                    input = input.replace(RDS2, "/");
                } else if (input.match(RDS3)) {
                    input = input.replace(RDS3, "/");
                    output.pop();
                } else if (input === "." || input === "..") {
                    input = "";
                } else {
                    var im = input.match(RDS5);
                    if (im) {
                        var s = im[0];
                        input = input.slice(s.length);
                        output.push(s);
                    } else {
                        throw new Error("Unexpected dot segment condition");
                    }
                }
            }
            return output.join("");
        }
        function serialize2(components) {
            var options = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : {
            };
            var protocol = options.iri ? IRI_PROTOCOL : URI_PROTOCOL;
            var uriTokens = [];
            var schemeHandler = SCHEMES2[(options.scheme || components.scheme || "").toLowerCase()];
            if (schemeHandler && schemeHandler.serialize) schemeHandler.serialize(components, options);
            if (components.host) {
                if (protocol.IPV6ADDRESS.test(components.host)) ;
                else if (options.domainHost || schemeHandler && schemeHandler.domainHost) {
                    try {
                        components.host = !options.iri ? punycode.toASCII(components.host.replace(protocol.PCT_ENCODED, pctDecChars2).toLowerCase()) : punycode.toUnicode(components.host);
                    } catch (e) {
                        components.error = components.error || "Host's domain name can not be converted to " + (!options.iri ? "ASCII" : "Unicode") + " via punycode: " + e;
                    }
                }
            }
            _normalizeComponentEncoding(components, protocol);
            if (options.reference !== "suffix" && components.scheme) {
                uriTokens.push(components.scheme);
                uriTokens.push(":");
            }
            var authority = _recomposeAuthority(components, options);
            if (authority !== void 0) {
                if (options.reference !== "suffix") {
                    uriTokens.push("//");
                }
                uriTokens.push(authority);
                if (components.path && components.path.charAt(0) !== "/") {
                    uriTokens.push("/");
                }
            }
            if (components.path !== void 0) {
                var s = components.path;
                if (!options.absolutePath && (!schemeHandler || !schemeHandler.absolutePath)) {
                    s = removeDotSegments2(s);
                }
                if (authority === void 0) {
                    s = s.replace(/^\/\//, "/%2F");
                }
                uriTokens.push(s);
            }
            if (components.query !== void 0) {
                uriTokens.push("?");
                uriTokens.push(components.query);
            }
            if (components.fragment !== void 0) {
                uriTokens.push("#");
                uriTokens.push(components.fragment);
            }
            return uriTokens.join("");
        }
        function resolveComponents2(base2, relative6) {
            var options = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {
            };
            var skipNormalization = arguments[3];
            var target = {
            };
            if (!skipNormalization) {
                base2 = parse21(serialize2(base2, options), options);
                relative6 = parse21(serialize2(relative6, options), options);
            }
            options = options || {
            };
            if (!options.tolerant && relative6.scheme) {
                target.scheme = relative6.scheme;
                target.userinfo = relative6.userinfo;
                target.host = relative6.host;
                target.port = relative6.port;
                target.path = removeDotSegments2(relative6.path || "");
                target.query = relative6.query;
            } else {
                if (relative6.userinfo !== void 0 || relative6.host !== void 0 || relative6.port !== void 0) {
                    target.userinfo = relative6.userinfo;
                    target.host = relative6.host;
                    target.port = relative6.port;
                    target.path = removeDotSegments2(relative6.path || "");
                    target.query = relative6.query;
                } else {
                    if (!relative6.path) {
                        target.path = base2.path;
                        if (relative6.query !== void 0) {
                            target.query = relative6.query;
                        } else {
                            target.query = base2.query;
                        }
                    } else {
                        if (relative6.path.charAt(0) === "/") {
                            target.path = removeDotSegments2(relative6.path);
                        } else {
                            if ((base2.userinfo !== void 0 || base2.host !== void 0 || base2.port !== void 0) && !base2.path) {
                                target.path = "/" + relative6.path;
                            } else if (!base2.path) {
                                target.path = relative6.path;
                            } else {
                                target.path = base2.path.slice(0, base2.path.lastIndexOf("/") + 1) + relative6.path;
                            }
                            target.path = removeDotSegments2(target.path);
                        }
                        target.query = relative6.query;
                    }
                    target.userinfo = base2.userinfo;
                    target.host = base2.host;
                    target.port = base2.port;
                }
                target.scheme = base2.scheme;
            }
            target.fragment = relative6.fragment;
            return target;
        }
        function resolve21(baseURI, relativeURI, options) {
            var schemelessOptions = assign1({
                scheme: "null"
            }, options);
            return serialize2(resolveComponents2(parse21(baseURI, schemelessOptions), parse21(relativeURI, schemelessOptions), schemelessOptions, true), schemelessOptions);
        }
        function normalize21(uri, options) {
            if (typeof uri === "string") {
                uri = serialize2(parse21(uri, options), options);
            } else if (typeOf(uri) === "object") {
                uri = parse21(serialize2(uri, options), options);
            }
            return uri;
        }
        function equal2(uriA, uriB, options) {
            if (typeof uriA === "string") {
                uriA = serialize2(parse21(uriA, options), options);
            } else if (typeOf(uriA) === "object") {
                uriA = serialize2(uriA, options);
            }
            if (typeof uriB === "string") {
                uriB = serialize2(parse21(uriB, options), options);
            } else if (typeOf(uriB) === "object") {
                uriB = serialize2(uriB, options);
            }
            return uriA === uriB;
        }
        function escapeComponent2(str74, options) {
            return str74 && str74.toString().replace(!options || !options.iri ? URI_PROTOCOL.ESCAPE : IRI_PROTOCOL.ESCAPE, pctEncChar2);
        }
        function unescapeComponent2(str75, options) {
            return str75 && str75.toString().replace(!options || !options.iri ? URI_PROTOCOL.PCT_ENCODED : IRI_PROTOCOL.PCT_ENCODED, pctDecChars2);
        }
        var handler = {
            scheme: "http",
            domainHost: true,
            parse: function parse3(components, options) {
                if (!components.host) {
                    components.error = components.error || "HTTP URIs must have a host.";
                }
                return components;
            },
            serialize: function serialize3(components, options) {
                var secure = String(components.scheme).toLowerCase() === "https";
                if (components.port === (secure ? 443 : 80) || components.port === "") {
                    components.port = void 0;
                }
                if (!components.path) {
                    components.path = "/";
                }
                return components;
            }
        };
        var handler$1 = {
            scheme: "https",
            domainHost: handler.domainHost,
            parse: handler.parse,
            serialize: handler.serialize
        };
        function isSecure(wsComponents) {
            return typeof wsComponents.secure === "boolean" ? wsComponents.secure : String(wsComponents.scheme).toLowerCase() === "wss";
        }
        var handler$2 = {
            scheme: "ws",
            domainHost: true,
            parse: function parse3(components, options) {
                var wsComponents = components;
                wsComponents.secure = isSecure(wsComponents);
                wsComponents.resourceName = (wsComponents.path || "/") + (wsComponents.query ? "?" + wsComponents.query : "");
                wsComponents.path = void 0;
                wsComponents.query = void 0;
                return wsComponents;
            },
            serialize: function serialize3(wsComponents, options) {
                if (wsComponents.port === (isSecure(wsComponents) ? 443 : 80) || wsComponents.port === "") {
                    wsComponents.port = void 0;
                }
                if (typeof wsComponents.secure === "boolean") {
                    wsComponents.scheme = wsComponents.secure ? "wss" : "ws";
                    wsComponents.secure = void 0;
                }
                if (wsComponents.resourceName) {
                    var _wsComponents$resourc = wsComponents.resourceName.split("?"), _wsComponents$resourc2 = slicedToArray(_wsComponents$resourc, 2), path85 = _wsComponents$resourc2[0], query = _wsComponents$resourc2[1];
                    wsComponents.path = path85 && path85 !== "/" ? path85 : void 0;
                    wsComponents.query = query;
                    wsComponents.resourceName = void 0;
                }
                wsComponents.fragment = void 0;
                return wsComponents;
            }
        };
        var handler$3 = {
            scheme: "wss",
            domainHost: handler$2.domainHost,
            parse: handler$2.parse,
            serialize: handler$2.serialize
        };
        var O = {
        };
        var UNRESERVED$$ = "[A-Za-z0-9\\-\\.\\_\\~\\xA0-\\u200D\\u2010-\\u2029\\u202F-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]";
        var HEXDIG$$ = "[0-9A-Fa-f]";
        var PCT_ENCODED$ = subexp(subexp("%[EFef]" + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$) + "|" + subexp("%[89A-Fa-f]" + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$) + "|" + subexp("%" + HEXDIG$$ + HEXDIG$$));
        var ATEXT$$ = "[A-Za-z0-9\\!\\$\\%\\'\\*\\+\\-\\^\\_\\`\\{\\|\\}\\~]";
        var QTEXT$$ = "[\\!\\$\\%\\'\\(\\)\\*\\+\\,\\-\\.0-9\\<\\>A-Z\\x5E-\\x7E]";
        var VCHAR$$ = merge2(QTEXT$$, '[\\"\\\\]');
        var SOME_DELIMS$$ = "[\\!\\$\\'\\(\\)\\*\\+\\,\\;\\:\\@]";
        var UNRESERVED = new RegExp(UNRESERVED$$, "g");
        var PCT_ENCODED = new RegExp(PCT_ENCODED$, "g");
        var NOT_LOCAL_PART = new RegExp(merge2("[^]", ATEXT$$, "[\\.]", '[\\"]', VCHAR$$), "g");
        var NOT_HFNAME = new RegExp(merge2("[^]", UNRESERVED$$, SOME_DELIMS$$), "g");
        var NOT_HFVALUE = NOT_HFNAME;
        function decodeUnreserved(str76) {
            var decStr = pctDecChars2(str76);
            return !decStr.match(UNRESERVED) ? str76 : decStr;
        }
        var handler$4 = {
            scheme: "mailto",
            parse: function parse$$1(components, options) {
                var mailtoComponents = components;
                var to = mailtoComponents.to = mailtoComponents.path ? mailtoComponents.path.split(",") : [];
                mailtoComponents.path = void 0;
                if (mailtoComponents.query) {
                    var unknownHeaders = false;
                    var headers = {
                    };
                    var hfields = mailtoComponents.query.split("&");
                    for(var x = 0, xl = hfields.length; x < xl; ++x){
                        var hfield = hfields[x].split("=");
                        switch(hfield[0]){
                            case "to":
                                var toAddrs = hfield[1].split(",");
                                for(var _x = 0, _xl = toAddrs.length; _x < _xl; ++_x){
                                    to.push(toAddrs[_x]);
                                }
                                break;
                            case "subject":
                                mailtoComponents.subject = unescapeComponent2(hfield[1], options);
                                break;
                            case "body":
                                mailtoComponents.body = unescapeComponent2(hfield[1], options);
                                break;
                            default:
                                unknownHeaders = true;
                                headers[unescapeComponent2(hfield[0], options)] = unescapeComponent2(hfield[1], options);
                                break;
                        }
                    }
                    if (unknownHeaders) mailtoComponents.headers = headers;
                }
                mailtoComponents.query = void 0;
                for(var _x2 = 0, _xl2 = to.length; _x2 < _xl2; ++_x2){
                    var addr = to[_x2].split("@");
                    addr[0] = unescapeComponent2(addr[0]);
                    if (!options.unicodeSupport) {
                        try {
                            addr[1] = punycode.toASCII(unescapeComponent2(addr[1], options).toLowerCase());
                        } catch (e) {
                            mailtoComponents.error = mailtoComponents.error || "Email address's domain name can not be converted to ASCII via punycode: " + e;
                        }
                    } else {
                        addr[1] = unescapeComponent2(addr[1], options).toLowerCase();
                    }
                    to[_x2] = addr.join("@");
                }
                return mailtoComponents;
            },
            serialize: function serialize$$1(mailtoComponents, options) {
                var components = mailtoComponents;
                var to = toArray1(mailtoComponents.to);
                if (to) {
                    for(var x = 0, xl = to.length; x < xl; ++x){
                        var toAddr = String(to[x]);
                        var atIdx = toAddr.lastIndexOf("@");
                        var localPart = toAddr.slice(0, atIdx).replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_LOCAL_PART, pctEncChar2);
                        var domain = toAddr.slice(atIdx + 1);
                        try {
                            domain = !options.iri ? punycode.toASCII(unescapeComponent2(domain, options).toLowerCase()) : punycode.toUnicode(domain);
                        } catch (e) {
                            components.error = components.error || "Email address's domain name can not be converted to " + (!options.iri ? "ASCII" : "Unicode") + " via punycode: " + e;
                        }
                        to[x] = localPart + "@" + domain;
                    }
                    components.path = to.join(",");
                }
                var headers = mailtoComponents.headers = mailtoComponents.headers || {
                };
                if (mailtoComponents.subject) headers["subject"] = mailtoComponents.subject;
                if (mailtoComponents.body) headers["body"] = mailtoComponents.body;
                var fields = [];
                for(var name in headers){
                    if (headers[name] !== O[name]) {
                        fields.push(name.replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_HFNAME, pctEncChar2) + "=" + headers[name].replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_HFVALUE, pctEncChar2));
                    }
                }
                if (fields.length) {
                    components.query = fields.join("&");
                }
                return components;
            }
        };
        var URN_PARSE = /^([^\:]+)\:(.*)/;
        var handler$5 = {
            scheme: "urn",
            parse: function parse$$1(components, options) {
                var matches4 = components.path && components.path.match(URN_PARSE);
                var urnComponents = components;
                if (matches4) {
                    var scheme = options.scheme || urnComponents.scheme || "urn";
                    var nid = matches4[1].toLowerCase();
                    var nss = matches4[2];
                    var urnScheme = scheme + ":" + (options.nid || nid);
                    var schemeHandler = SCHEMES2[urnScheme];
                    urnComponents.nid = nid;
                    urnComponents.nss = nss;
                    urnComponents.path = void 0;
                    if (schemeHandler) {
                        urnComponents = schemeHandler.parse(urnComponents, options);
                    }
                } else {
                    urnComponents.error = urnComponents.error || "URN can not be parsed.";
                }
                return urnComponents;
            },
            serialize: function serialize$$1(urnComponents, options) {
                var scheme = options.scheme || urnComponents.scheme || "urn";
                var nid = urnComponents.nid;
                var urnScheme = scheme + ":" + (options.nid || nid);
                var schemeHandler = SCHEMES2[urnScheme];
                if (schemeHandler) {
                    urnComponents = schemeHandler.serialize(urnComponents, options);
                }
                var uriComponents = urnComponents;
                var nss = urnComponents.nss;
                uriComponents.path = (nid || options.nid) + ":" + nss;
                return uriComponents;
            }
        };
        var UUID = /^[0-9A-Fa-f]{8}(?:\-[0-9A-Fa-f]{4}){3}\-[0-9A-Fa-f]{12}$/;
        var handler$6 = {
            scheme: "urn:uuid",
            parse: function parse3(urnComponents, options) {
                var uuidComponents = urnComponents;
                uuidComponents.uuid = uuidComponents.nss;
                uuidComponents.nss = void 0;
                if (!options.tolerant && (!uuidComponents.uuid || !uuidComponents.uuid.match(UUID))) {
                    uuidComponents.error = uuidComponents.error || "UUID is not valid.";
                }
                return uuidComponents;
            },
            serialize: function serialize3(uuidComponents, options) {
                var urnComponents = uuidComponents;
                urnComponents.nss = (uuidComponents.uuid || "").toLowerCase();
                return urnComponents;
            }
        };
        SCHEMES2[handler.scheme] = handler;
        SCHEMES2[handler$1.scheme] = handler$1;
        SCHEMES2[handler$2.scheme] = handler$2;
        SCHEMES2[handler$3.scheme] = handler$3;
        SCHEMES2[handler$4.scheme] = handler$4;
        SCHEMES2[handler$5.scheme] = handler$5;
        SCHEMES2[handler$6.scheme] = handler$6;
        exports2.SCHEMES = SCHEMES2;
        exports2.pctEncChar = pctEncChar2;
        exports2.pctDecChars = pctDecChars2;
        exports2.parse = parse21;
        exports2.removeDotSegments = removeDotSegments2;
        exports2.serialize = serialize2;
        exports2.resolveComponents = resolveComponents2;
        exports2.resolve = resolve21;
        exports2.normalize = normalize21;
        exports2.equal = equal2;
        exports2.escapeComponent = escapeComponent2;
        exports2.unescapeComponent = unescapeComponent2;
        Object.defineProperty(exports2, "__esModule", {
            value: true
        });
    });
});
var __pika_web_default_export_for_treeshaking__ = getDefaultExportFromCjs(uri_all);
uri_all.SCHEMES;
uri_all.equal;
uri_all.escapeComponent;
uri_all.normalize;
uri_all.parse;
uri_all.pctDecChars;
uri_all.pctEncChar;
uri_all.removeDotSegments;
uri_all.resolve;
uri_all.resolveComponents;
uri_all.serialize;
uri_all.unescapeComponent;
function getDefaultExportFromCjs1(x) {
    return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, "default") ? x["default"] : x;
}
function createCommonjsModule2(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path86, base7) {
            return commonjsRequire2(path86, base7 === void 0 || base7 === null ? module.path : base7);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire2() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var code2 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.regexpCode = exports.getProperty = exports.safeStringify = exports.stringify = exports.strConcat = exports.addCodeArg = exports.str = exports._ = exports.nil = exports._Code = exports.Name = exports.IDENTIFIER = exports._CodeOrName = void 0;
    class _CodeOrName {
    }
    exports._CodeOrName = _CodeOrName;
    exports.IDENTIFIER = /^[a-z$_][a-z$_0-9]*$/i;
    class Name2 extends _CodeOrName {
        constructor(s){
            super();
            if (!exports.IDENTIFIER.test(s)) throw new Error("CodeGen: name must be a valid identifier");
            this.str = s;
        }
        toString() {
            return this.str;
        }
        emptyStr() {
            return false;
        }
        get names() {
            return {
                [this.str]: 1
            };
        }
    }
    exports.Name = Name2;
    class _Code extends _CodeOrName {
        constructor(code21){
            super();
            this._items = typeof code21 === "string" ? [
                code21
            ] : code21;
        }
        toString() {
            return this.str;
        }
        emptyStr() {
            if (this._items.length > 1) return false;
            const item = this._items[0];
            return item === "" || item === '""';
        }
        get str() {
            var _a;
            return (_a = this._str) !== null && _a !== void 0 ? _a : this._str = this._items.reduce((s, c)=>`${s}${c}`
            , "");
        }
        get names() {
            var _a;
            return (_a = this._names) !== null && _a !== void 0 ? _a : this._names = this._items.reduce((names, c)=>{
                if (c instanceof Name2) names[c.str] = (names[c.str] || 0) + 1;
                return names;
            }, {
            });
        }
    }
    exports._Code = _Code;
    exports.nil = new _Code("");
    function _2(strs, ...args) {
        const code22 = [
            strs[0]
        ];
        let i37 = 0;
        while(i37 < args.length){
            addCodeArg(code22, args[i37]);
            code22.push(strs[++i37]);
        }
        return new _Code(code22);
    }
    exports._ = _2;
    const plus = new _Code("+");
    function str2(strs, ...args) {
        const expr = [
            safeStringify(strs[0])
        ];
        let i38 = 0;
        while(i38 < args.length){
            expr.push(plus);
            addCodeArg(expr, args[i38]);
            expr.push(plus, safeStringify(strs[++i38]));
        }
        optimize(expr);
        return new _Code(expr);
    }
    exports.str = str2;
    function addCodeArg(code23, arg) {
        if (arg instanceof _Code) code23.push(...arg._items);
        else if (arg instanceof Name2) code23.push(arg);
        else code23.push(interpolate(arg));
    }
    exports.addCodeArg = addCodeArg;
    function optimize(expr) {
        let i39 = 1;
        while(i39 < expr.length - 1){
            if (expr[i39] === plus) {
                const res = mergeExprItems(expr[i39 - 1], expr[i39 + 1]);
                if (res !== void 0) {
                    expr.splice(i39 - 1, 3, res);
                    continue;
                }
                expr[i39++] = "+";
            }
            i39++;
        }
    }
    function mergeExprItems(a, b) {
        if (b === '""') return a;
        if (a === '""') return b;
        if (typeof a == "string") {
            if (b instanceof Name2 || a[a.length - 1] !== '"') return;
            if (typeof b != "string") return `${a.slice(0, -1)}${b}"`;
            if (b[0] === '"') return a.slice(0, -1) + b.slice(1);
            return;
        }
        if (typeof b == "string" && b[0] === '"' && !(a instanceof Name2)) return `"${a}${b.slice(1)}`;
        return;
    }
    function strConcat(c1, c2) {
        return c2.emptyStr() ? c1 : c1.emptyStr() ? c2 : str2`${c1}${c2}`;
    }
    exports.strConcat = strConcat;
    function interpolate(x) {
        return typeof x == "number" || typeof x == "boolean" || x === null ? x : safeStringify(Array.isArray(x) ? x.join(",") : x);
    }
    function stringify2(x) {
        return new _Code(safeStringify(x));
    }
    exports.stringify = stringify2;
    function safeStringify(x) {
        return JSON.stringify(x).replace(/\u2028/g, "\\u2028").replace(/\u2029/g, "\\u2029");
    }
    exports.safeStringify = safeStringify;
    function getProperty(key) {
        return typeof key == "string" && exports.IDENTIFIER.test(key) ? new _Code(`.${key}`) : _2`[${key}]`;
    }
    exports.getProperty = getProperty;
    function regexpCode(rx) {
        return new _Code(rx.toString());
    }
    exports.regexpCode = regexpCode;
});
var scope = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.ValueScope = exports.ValueScopeName = exports.Scope = exports.varKinds = exports.UsedValueState = void 0;
    class ValueError extends Error {
        constructor(name){
            super(`CodeGen: "code" for ${name} not defined`);
            this.value = name.value;
        }
    }
    var UsedValueState;
    (function(UsedValueState2) {
        UsedValueState2[UsedValueState2["Started"] = 0] = "Started";
        UsedValueState2[UsedValueState2["Completed"] = 1] = "Completed";
    })(UsedValueState = exports.UsedValueState || (exports.UsedValueState = {
    }));
    exports.varKinds = {
        const: new code2.Name("const"),
        let: new code2.Name("let"),
        var: new code2.Name("var")
    };
    class Scope1 {
        constructor({ prefixes , parent: parent5  } = {
        }){
            this._names = {
            };
            this._prefixes = prefixes;
            this._parent = parent5;
        }
        toName(nameOrPrefix) {
            return nameOrPrefix instanceof code2.Name ? nameOrPrefix : this.name(nameOrPrefix);
        }
        name(prefix) {
            return new code2.Name(this._newName(prefix));
        }
        _newName(prefix) {
            const ng = this._names[prefix] || this._nameGroup(prefix);
            return `${prefix}${ng.index++}`;
        }
        _nameGroup(prefix) {
            var _a, _b;
            if (((_b = (_a = this._parent) === null || _a === void 0 ? void 0 : _a._prefixes) === null || _b === void 0 ? void 0 : _b.has(prefix)) || this._prefixes && !this._prefixes.has(prefix)) {
                throw new Error(`CodeGen: prefix "${prefix}" is not allowed in this scope`);
            }
            return this._names[prefix] = {
                prefix,
                index: 0
            };
        }
    }
    exports.Scope = Scope1;
    class ValueScopeName extends code2.Name {
        constructor(prefix, nameStr){
            super(nameStr);
            this.prefix = prefix;
        }
        setValue(value, { property: property1 , itemIndex  }) {
            this.value = value;
            this.scopePath = code2._`.${new code2.Name(property1)}[${itemIndex}]`;
        }
    }
    exports.ValueScopeName = ValueScopeName;
    const line = code2._`\n`;
    class ValueScope extends Scope1 {
        constructor(opts){
            super(opts);
            this._values = {
            };
            this._scope = opts.scope;
            this.opts = {
                ...opts,
                _n: opts.lines ? line : code2.nil
            };
        }
        get() {
            return this._scope;
        }
        name(prefix) {
            return new ValueScopeName(prefix, this._newName(prefix));
        }
        value(nameOrPrefix, value) {
            var _a;
            if (value.ref === void 0) throw new Error("CodeGen: ref must be passed in value");
            const name = this.toName(nameOrPrefix);
            const { prefix  } = name;
            const valueKey = (_a = value.key) !== null && _a !== void 0 ? _a : value.ref;
            let vs = this._values[prefix];
            if (vs) {
                const _name = vs.get(valueKey);
                if (_name) return _name;
            } else {
                vs = this._values[prefix] = new Map();
            }
            vs.set(valueKey, name);
            const s = this._scope[prefix] || (this._scope[prefix] = []);
            const itemIndex = s.length;
            s[itemIndex] = value.ref;
            name.setValue(value, {
                property: prefix,
                itemIndex
            });
            return name;
        }
        getValue(prefix, keyOrRef) {
            const vs = this._values[prefix];
            if (!vs) return;
            return vs.get(keyOrRef);
        }
        scopeRefs(scopeName, values6 = this._values) {
            return this._reduceValues(values6, (name)=>{
                if (name.scopePath === void 0) throw new Error(`CodeGen: name "${name}" has no value`);
                return code2._`${scopeName}${name.scopePath}`;
            });
        }
        scopeCode(values7 = this._values, usedValues, getCode) {
            return this._reduceValues(values7, (name)=>{
                if (name.value === void 0) throw new Error(`CodeGen: name "${name}" has no value`);
                return name.value.code;
            }, usedValues, getCode);
        }
        _reduceValues(values8, valueCode, usedValues = {
        }, getCode) {
            let code$12 = code2.nil;
            for(const prefix in values8){
                const vs = values8[prefix];
                if (!vs) continue;
                const nameSet = usedValues[prefix] = usedValues[prefix] || new Map();
                vs.forEach((name)=>{
                    if (nameSet.has(name)) return;
                    nameSet.set(name, UsedValueState.Started);
                    let c = valueCode(name);
                    if (c) {
                        const def1 = this.opts.es5 ? exports.varKinds.var : exports.varKinds.const;
                        code$12 = code2._`${code$12}${def1} ${name} = ${c};${this.opts._n}`;
                    } else if (c = getCode === null || getCode === void 0 ? void 0 : getCode(name)) {
                        code$12 = code2._`${code$12}${c}${this.opts._n}`;
                    } else {
                        throw new ValueError(name);
                    }
                    nameSet.set(name, UsedValueState.Completed);
                });
            }
            return code$12;
        }
    }
    exports.ValueScope = ValueScope;
});
var codegen = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.or = exports.and = exports.not = exports.CodeGen = exports.operators = exports.varKinds = exports.ValueScopeName = exports.ValueScope = exports.Scope = exports.Name = exports.regexpCode = exports.stringify = exports.getProperty = exports.nil = exports.strConcat = exports.str = exports._ = void 0;
    var code_2 = code2;
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return code_2._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return code_2.str;
        }
    });
    Object.defineProperty(exports, "strConcat", {
        enumerable: true,
        get: function() {
            return code_2.strConcat;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return code_2.nil;
        }
    });
    Object.defineProperty(exports, "getProperty", {
        enumerable: true,
        get: function() {
            return code_2.getProperty;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return code_2.stringify;
        }
    });
    Object.defineProperty(exports, "regexpCode", {
        enumerable: true,
        get: function() {
            return code_2.regexpCode;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return code_2.Name;
        }
    });
    var scope_2 = scope;
    Object.defineProperty(exports, "Scope", {
        enumerable: true,
        get: function() {
            return scope_2.Scope;
        }
    });
    Object.defineProperty(exports, "ValueScope", {
        enumerable: true,
        get: function() {
            return scope_2.ValueScope;
        }
    });
    Object.defineProperty(exports, "ValueScopeName", {
        enumerable: true,
        get: function() {
            return scope_2.ValueScopeName;
        }
    });
    Object.defineProperty(exports, "varKinds", {
        enumerable: true,
        get: function() {
            return scope_2.varKinds;
        }
    });
    exports.operators = {
        GT: new code2._Code(">"),
        GTE: new code2._Code(">="),
        LT: new code2._Code("<"),
        LTE: new code2._Code("<="),
        EQ: new code2._Code("==="),
        NEQ: new code2._Code("!=="),
        NOT: new code2._Code("!"),
        OR: new code2._Code("||"),
        AND: new code2._Code("&&"),
        ADD: new code2._Code("+")
    };
    class Node2 {
        optimizeNodes() {
            return this;
        }
        optimizeNames(_names, _constants) {
            return this;
        }
    }
    class Def extends Node2 {
        constructor(varKind, name, rhs){
            super();
            this.varKind = varKind;
            this.name = name;
            this.rhs = rhs;
        }
        render({ es5 , _n  }) {
            const varKind = es5 ? scope.varKinds.var : this.varKind;
            const rhs = this.rhs === void 0 ? "" : ` = ${this.rhs}`;
            return `${varKind} ${this.name}${rhs};` + _n;
        }
        optimizeNames(names, constants) {
            if (!names[this.name.str]) return;
            if (this.rhs) this.rhs = optimizeExpr(this.rhs, names, constants);
            return this;
        }
        get names() {
            return this.rhs instanceof code2._CodeOrName ? this.rhs.names : {
            };
        }
    }
    class Assign extends Node2 {
        constructor(lhs, rhs, sideEffects){
            super();
            this.lhs = lhs;
            this.rhs = rhs;
            this.sideEffects = sideEffects;
        }
        render({ _n  }) {
            return `${this.lhs} = ${this.rhs};` + _n;
        }
        optimizeNames(names, constants) {
            if (this.lhs instanceof code2.Name && !names[this.lhs.str] && !this.sideEffects) return;
            this.rhs = optimizeExpr(this.rhs, names, constants);
            return this;
        }
        get names() {
            const names = this.lhs instanceof code2.Name ? {
            } : {
                ...this.lhs.names
            };
            return addExprNames(names, this.rhs);
        }
    }
    class AssignOp extends Assign {
        constructor(lhs, op, rhs, sideEffects){
            super(lhs, rhs, sideEffects);
            this.op = op;
        }
        render({ _n  }) {
            return `${this.lhs} ${this.op}= ${this.rhs};` + _n;
        }
    }
    class Label extends Node2 {
        constructor(label){
            super();
            this.label = label;
            this.names = {
            };
        }
        render({ _n  }) {
            return `${this.label}:` + _n;
        }
    }
    class Break extends Node2 {
        constructor(label){
            super();
            this.label = label;
            this.names = {
            };
        }
        render({ _n  }) {
            const label = this.label ? ` ${this.label}` : "";
            return `break${label};` + _n;
        }
    }
    class Throw extends Node2 {
        constructor(error16){
            super();
            this.error = error16;
        }
        render({ _n  }) {
            return `throw ${this.error};` + _n;
        }
        get names() {
            return this.error.names;
        }
    }
    class AnyCode extends Node2 {
        constructor(code24){
            super();
            this.code = code24;
        }
        render({ _n  }) {
            return `${this.code};` + _n;
        }
        optimizeNodes() {
            return `${this.code}` ? this : void 0;
        }
        optimizeNames(names, constants) {
            this.code = optimizeExpr(this.code, names, constants);
            return this;
        }
        get names() {
            return this.code instanceof code2._CodeOrName ? this.code.names : {
            };
        }
    }
    class ParentNode extends Node2 {
        constructor(nodes = []){
            super();
            this.nodes = nodes;
        }
        render(opts) {
            return this.nodes.reduce((code25, n24)=>code25 + n24.render(opts)
            , "");
        }
        optimizeNodes() {
            const { nodes  } = this;
            let i40 = nodes.length;
            while(i40--){
                const n25 = nodes[i40].optimizeNodes();
                if (Array.isArray(n25)) nodes.splice(i40, 1, ...n25);
                else if (n25) nodes[i40] = n25;
                else nodes.splice(i40, 1);
            }
            return nodes.length > 0 ? this : void 0;
        }
        optimizeNames(names, constants) {
            const { nodes  } = this;
            let i41 = nodes.length;
            while(i41--){
                const n26 = nodes[i41];
                if (n26.optimizeNames(names, constants)) continue;
                subtractNames(names, n26.names);
                nodes.splice(i41, 1);
            }
            return nodes.length > 0 ? this : void 0;
        }
        get names() {
            return this.nodes.reduce((names, n27)=>addNames(names, n27.names)
            , {
            });
        }
    }
    class BlockNode extends ParentNode {
        render(opts) {
            return "{" + opts._n + super.render(opts) + "}" + opts._n;
        }
    }
    class Root extends ParentNode {
    }
    class Else extends BlockNode {
    }
    Else.kind = "else";
    class If extends BlockNode {
        constructor(condition, nodes){
            super(nodes);
            this.condition = condition;
        }
        render(opts) {
            let code26 = `if(${this.condition})` + super.render(opts);
            if (this.else) code26 += "else " + this.else.render(opts);
            return code26;
        }
        optimizeNodes() {
            super.optimizeNodes();
            const cond2 = this.condition;
            if (cond2 === true) return this.nodes;
            let e = this.else;
            if (e) {
                const ns = e.optimizeNodes();
                e = this.else = Array.isArray(ns) ? new Else(ns) : ns;
            }
            if (e) {
                if (cond2 === false) return e instanceof If ? e : e.nodes;
                if (this.nodes.length) return this;
                return new If(not2(cond2), e instanceof If ? [
                    e
                ] : e.nodes);
            }
            if (cond2 === false || !this.nodes.length) return void 0;
            return this;
        }
        optimizeNames(names, constants) {
            var _a;
            this.else = (_a = this.else) === null || _a === void 0 ? void 0 : _a.optimizeNames(names, constants);
            if (!(super.optimizeNames(names, constants) || this.else)) return;
            this.condition = optimizeExpr(this.condition, names, constants);
            return this;
        }
        get names() {
            const names = super.names;
            addExprNames(names, this.condition);
            if (this.else) addNames(names, this.else.names);
            return names;
        }
    }
    If.kind = "if";
    class For extends BlockNode {
    }
    For.kind = "for";
    class ForLoop extends For {
        constructor(iteration){
            super();
            this.iteration = iteration;
        }
        render(opts) {
            return `for(${this.iteration})` + super.render(opts);
        }
        optimizeNames(names, constants) {
            if (!super.optimizeNames(names, constants)) return;
            this.iteration = optimizeExpr(this.iteration, names, constants);
            return this;
        }
        get names() {
            return addNames(super.names, this.iteration.names);
        }
    }
    class ForRange extends For {
        constructor(varKind, name, from, to){
            super();
            this.varKind = varKind;
            this.name = name;
            this.from = from;
            this.to = to;
        }
        render(opts) {
            const varKind = opts.es5 ? scope.varKinds.var : this.varKind;
            const { name , from , to  } = this;
            return `for(${varKind} ${name}=${from}; ${name}<${to}; ${name}++)` + super.render(opts);
        }
        get names() {
            const names = addExprNames(super.names, this.from);
            return addExprNames(names, this.to);
        }
    }
    class ForIter extends For {
        constructor(loop, varKind, name, iterable){
            super();
            this.loop = loop;
            this.varKind = varKind;
            this.name = name;
            this.iterable = iterable;
        }
        render(opts) {
            return `for(${this.varKind} ${this.name} ${this.loop} ${this.iterable})` + super.render(opts);
        }
        optimizeNames(names, constants) {
            if (!super.optimizeNames(names, constants)) return;
            this.iterable = optimizeExpr(this.iterable, names, constants);
            return this;
        }
        get names() {
            return addNames(super.names, this.iterable.names);
        }
    }
    class Func extends BlockNode {
        constructor(name, args, async){
            super();
            this.name = name;
            this.args = args;
            this.async = async;
        }
        render(opts) {
            const _async = this.async ? "async " : "";
            return `${_async}function ${this.name}(${this.args})` + super.render(opts);
        }
    }
    Func.kind = "func";
    class Return extends ParentNode {
        render(opts) {
            return "return " + super.render(opts);
        }
    }
    Return.kind = "return";
    class Try extends BlockNode {
        render(opts) {
            let code27 = "try" + super.render(opts);
            if (this.catch) code27 += this.catch.render(opts);
            if (this.finally) code27 += this.finally.render(opts);
            return code27;
        }
        optimizeNodes() {
            var _a, _b;
            super.optimizeNodes();
            (_a = this.catch) === null || _a === void 0 ? void 0 : _a.optimizeNodes();
            (_b = this.finally) === null || _b === void 0 ? void 0 : _b.optimizeNodes();
            return this;
        }
        optimizeNames(names, constants) {
            var _a, _b;
            super.optimizeNames(names, constants);
            (_a = this.catch) === null || _a === void 0 ? void 0 : _a.optimizeNames(names, constants);
            (_b = this.finally) === null || _b === void 0 ? void 0 : _b.optimizeNames(names, constants);
            return this;
        }
        get names() {
            const names = super.names;
            if (this.catch) addNames(names, this.catch.names);
            if (this.finally) addNames(names, this.finally.names);
            return names;
        }
    }
    class Catch extends BlockNode {
        constructor(error17){
            super();
            this.error = error17;
        }
        render(opts) {
            return `catch(${this.error})` + super.render(opts);
        }
    }
    Catch.kind = "catch";
    class Finally extends BlockNode {
        render(opts) {
            return "finally" + super.render(opts);
        }
    }
    Finally.kind = "finally";
    class CodeGen2 {
        constructor(extScope, opts = {
        }){
            this._values = {
            };
            this._blockStarts = [];
            this._constants = {
            };
            this.opts = {
                ...opts,
                _n: opts.lines ? "\n" : ""
            };
            this._extScope = extScope;
            this._scope = new scope.Scope({
                parent: extScope
            });
            this._nodes = [
                new Root()
            ];
        }
        toString() {
            return this._root.render(this.opts);
        }
        name(prefix) {
            return this._scope.name(prefix);
        }
        scopeName(prefix) {
            return this._extScope.name(prefix);
        }
        scopeValue(prefixOrName, value) {
            const name = this._extScope.value(prefixOrName, value);
            const vs = this._values[name.prefix] || (this._values[name.prefix] = new Set());
            vs.add(name);
            return name;
        }
        getScopeValue(prefix, keyOrRef) {
            return this._extScope.getValue(prefix, keyOrRef);
        }
        scopeRefs(scopeName) {
            return this._extScope.scopeRefs(scopeName, this._values);
        }
        scopeCode() {
            return this._extScope.scopeCode(this._values);
        }
        _def(varKind, nameOrPrefix, rhs, constant1) {
            const name = this._scope.toName(nameOrPrefix);
            if (rhs !== void 0 && constant1) this._constants[name.str] = rhs;
            this._leafNode(new Def(varKind, name, rhs));
            return name;
        }
        const(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.const, nameOrPrefix, rhs, _constant);
        }
        let(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.let, nameOrPrefix, rhs, _constant);
        }
        var(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.var, nameOrPrefix, rhs, _constant);
        }
        assign(lhs, rhs, sideEffects) {
            return this._leafNode(new Assign(lhs, rhs, sideEffects));
        }
        add(lhs, rhs) {
            return this._leafNode(new AssignOp(lhs, exports.operators.ADD, rhs));
        }
        code(c) {
            if (typeof c == "function") c();
            else if (c !== code2.nil) this._leafNode(new AnyCode(c));
            return this;
        }
        object(...keyValues) {
            const code$12 = [
                "{"
            ];
            for (const [key, value] of keyValues){
                if (code$12.length > 1) code$12.push(",");
                code$12.push(key);
                if (key !== value || this.opts.es5) {
                    code$12.push(":");
                    (0, code2.addCodeArg)(code$12, value);
                }
            }
            code$12.push("}");
            return new code2._Code(code$12);
        }
        if(condition, thenBody, elseBody) {
            this._blockNode(new If(condition));
            if (thenBody && elseBody) {
                this.code(thenBody).else().code(elseBody).endIf();
            } else if (thenBody) {
                this.code(thenBody).endIf();
            } else if (elseBody) {
                throw new Error('CodeGen: "else" body without "then" body');
            }
            return this;
        }
        elseIf(condition) {
            return this._elseNode(new If(condition));
        }
        else() {
            return this._elseNode(new Else());
        }
        endIf() {
            return this._endBlockNode(If, Else);
        }
        _for(node, forBody) {
            this._blockNode(node);
            if (forBody) this.code(forBody).endFor();
            return this;
        }
        for(iteration, forBody) {
            return this._for(new ForLoop(iteration), forBody);
        }
        forRange(nameOrPrefix, from, to, forBody, varKind = this.opts.es5 ? scope.varKinds.var : scope.varKinds.let) {
            const name = this._scope.toName(nameOrPrefix);
            return this._for(new ForRange(varKind, name, from, to), ()=>forBody(name)
            );
        }
        forOf(nameOrPrefix, iterable, forBody, varKind = scope.varKinds.const) {
            const name = this._scope.toName(nameOrPrefix);
            if (this.opts.es5) {
                const arr = iterable instanceof code2.Name ? iterable : this.var("_arr", iterable);
                return this.forRange("_i", 0, code2._`${arr}.length`, (i42)=>{
                    this.var(name, code2._`${arr}[${i42}]`);
                    forBody(name);
                });
            }
            return this._for(new ForIter("of", varKind, name, iterable), ()=>forBody(name)
            );
        }
        forIn(nameOrPrefix, obj, forBody, varKind = this.opts.es5 ? scope.varKinds.var : scope.varKinds.const) {
            if (this.opts.ownProperties) {
                return this.forOf(nameOrPrefix, code2._`Object.keys(${obj})`, forBody);
            }
            const name = this._scope.toName(nameOrPrefix);
            return this._for(new ForIter("in", varKind, name, obj), ()=>forBody(name)
            );
        }
        endFor() {
            return this._endBlockNode(For);
        }
        label(label) {
            return this._leafNode(new Label(label));
        }
        break(label) {
            return this._leafNode(new Break(label));
        }
        return(value) {
            const node = new Return();
            this._blockNode(node);
            this.code(value);
            if (node.nodes.length !== 1) throw new Error('CodeGen: "return" should have one node');
            return this._endBlockNode(Return);
        }
        try(tryBody, catchCode, finallyCode) {
            if (!catchCode && !finallyCode) throw new Error('CodeGen: "try" without "catch" and "finally"');
            const node = new Try();
            this._blockNode(node);
            this.code(tryBody);
            if (catchCode) {
                const error18 = this.name("e");
                this._currNode = node.catch = new Catch(error18);
                catchCode(error18);
            }
            if (finallyCode) {
                this._currNode = node.finally = new Finally();
                this.code(finallyCode);
            }
            return this._endBlockNode(Catch, Finally);
        }
        throw(error19) {
            return this._leafNode(new Throw(error19));
        }
        block(body, nodeCount) {
            this._blockStarts.push(this._nodes.length);
            if (body) this.code(body).endBlock(nodeCount);
            return this;
        }
        endBlock(nodeCount) {
            const len2 = this._blockStarts.pop();
            if (len2 === void 0) throw new Error("CodeGen: not in self-balancing block");
            const toClose = this._nodes.length - len2;
            if (toClose < 0 || nodeCount !== void 0 && toClose !== nodeCount) {
                throw new Error(`CodeGen: wrong number of nodes: ${toClose} vs ${nodeCount} expected`);
            }
            this._nodes.length = len2;
            return this;
        }
        func(name, args = code2.nil, async, funcBody) {
            this._blockNode(new Func(name, args, async));
            if (funcBody) this.code(funcBody).endFunc();
            return this;
        }
        endFunc() {
            return this._endBlockNode(Func);
        }
        optimize(n28 = 1) {
            while(n28-- > 0){
                this._root.optimizeNodes();
                this._root.optimizeNames(this._root.names, this._constants);
            }
        }
        _leafNode(node) {
            this._currNode.nodes.push(node);
            return this;
        }
        _blockNode(node) {
            this._currNode.nodes.push(node);
            this._nodes.push(node);
        }
        _endBlockNode(N1, N2) {
            const n29 = this._currNode;
            if (n29 instanceof N1 || N2 && n29 instanceof N2) {
                this._nodes.pop();
                return this;
            }
            throw new Error(`CodeGen: not in block "${N2 ? `${N1.kind}/${N2.kind}` : N1.kind}"`);
        }
        _elseNode(node) {
            const n30 = this._currNode;
            if (!(n30 instanceof If)) {
                throw new Error('CodeGen: "else" without "if"');
            }
            this._currNode = n30.else = node;
            return this;
        }
        get _root() {
            return this._nodes[0];
        }
        get _currNode() {
            const ns = this._nodes;
            return ns[ns.length - 1];
        }
        set _currNode(node) {
            const ns = this._nodes;
            ns[ns.length - 1] = node;
        }
    }
    exports.CodeGen = CodeGen2;
    function addNames(names, from) {
        for(const n in from)names[n] = (names[n] || 0) + (from[n] || 0);
        return names;
    }
    function addExprNames(names, from) {
        return from instanceof code2._CodeOrName ? addNames(names, from.names) : names;
    }
    function optimizeExpr(expr, names, constants) {
        if (expr instanceof code2.Name) return replaceName(expr);
        if (!canOptimize(expr)) return expr;
        return new code2._Code(expr._items.reduce((items2, c)=>{
            if (c instanceof code2.Name) c = replaceName(c);
            if (c instanceof code2._Code) items2.push(...c._items);
            else items2.push(c);
            return items2;
        }, []));
        function replaceName(n31) {
            const c = constants[n31.str];
            if (c === void 0 || names[n31.str] !== 1) return n31;
            delete names[n31.str];
            return c;
        }
        function canOptimize(e) {
            return e instanceof code2._Code && e._items.some((c)=>c instanceof code2.Name && names[c.str] === 1 && constants[c.str] !== void 0
            );
        }
    }
    function subtractNames(names, from) {
        for(const n in from)names[n] = (names[n] || 0) - (from[n] || 0);
    }
    function not2(x) {
        return typeof x == "boolean" || typeof x == "number" || x === null ? !x : code2._`!${par(x)}`;
    }
    exports.not = not2;
    const andCode = mappend(exports.operators.AND);
    function and(...args) {
        return args.reduce(andCode);
    }
    exports.and = and;
    const orCode = mappend(exports.operators.OR);
    function or(...args) {
        return args.reduce(orCode);
    }
    exports.or = or;
    function mappend(op) {
        return (x, y)=>x === code2.nil ? y : y === code2.nil ? x : code2._`${par(x)} ${op} ${par(y)}`
        ;
    }
    function par(x) {
        return x instanceof code2.Name ? x : code2._`(${x})`;
    }
});
var util1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.checkStrictMode = exports.getErrorPath = exports.Type = exports.useFunc = exports.setEvaluated = exports.evaluatedPropsToName = exports.mergeEvaluated = exports.eachItem = exports.unescapeJsonPointer = exports.escapeJsonPointer = exports.escapeFragment = exports.unescapeFragment = exports.schemaRefOrVal = exports.schemaHasRulesButRef = exports.schemaHasRules = exports.checkUnknownRules = exports.alwaysValidSchema = exports.toHash = void 0;
    function toHash(arr) {
        const hash = {
        };
        for (const item of arr)hash[item] = true;
        return hash;
    }
    exports.toHash = toHash;
    function alwaysValidSchema(it, schema) {
        if (typeof schema == "boolean") return schema;
        if (Object.keys(schema).length === 0) return true;
        checkUnknownRules(it, schema);
        return !schemaHasRules(schema, it.self.RULES.all);
    }
    exports.alwaysValidSchema = alwaysValidSchema;
    function checkUnknownRules(it, schema = it.schema) {
        const { opts , self  } = it;
        if (!opts.strictSchema) return;
        if (typeof schema === "boolean") return;
        const rules2 = self.RULES.keywords;
        for(const key in schema){
            if (!rules2[key]) checkStrictMode(it, `unknown keyword: "${key}"`);
        }
    }
    exports.checkUnknownRules = checkUnknownRules;
    function schemaHasRules(schema, rules2) {
        if (typeof schema == "boolean") return !schema;
        for(const key in schema)if (rules2[key]) return true;
        return false;
    }
    exports.schemaHasRules = schemaHasRules;
    function schemaHasRulesButRef(schema, RULES) {
        if (typeof schema == "boolean") return !schema;
        for(const key in schema)if (key !== "$ref" && RULES.all[key]) return true;
        return false;
    }
    exports.schemaHasRulesButRef = schemaHasRulesButRef;
    function schemaRefOrVal({ topSchemaRef , schemaPath  }, schema, keyword2, $data) {
        if (!$data) {
            if (typeof schema == "number" || typeof schema == "boolean") return schema;
            if (typeof schema == "string") return codegen._`${schema}`;
        }
        return codegen._`${topSchemaRef}${schemaPath}${(0, codegen.getProperty)(keyword2)}`;
    }
    exports.schemaRefOrVal = schemaRefOrVal;
    function unescapeFragment(str2) {
        return unescapeJsonPointer(decodeURIComponent(str2));
    }
    exports.unescapeFragment = unescapeFragment;
    function escapeFragment(str2) {
        return encodeURIComponent(escapeJsonPointer(str2));
    }
    exports.escapeFragment = escapeFragment;
    function escapeJsonPointer(str2) {
        if (typeof str2 == "number") return `${str2}`;
        return str2.replace(/~/g, "~0").replace(/\//g, "~1");
    }
    exports.escapeJsonPointer = escapeJsonPointer;
    function unescapeJsonPointer(str2) {
        return str2.replace(/~1/g, "/").replace(/~0/g, "~");
    }
    exports.unescapeJsonPointer = unescapeJsonPointer;
    function eachItem(xs, f) {
        if (Array.isArray(xs)) {
            for (const x of xs)f(x);
        } else {
            f(xs);
        }
    }
    exports.eachItem = eachItem;
    function makeMergeEvaluated({ mergeNames , mergeToName , mergeValues , resultToName  }) {
        return (gen, from, to, toName)=>{
            const res = to === void 0 ? from : to instanceof codegen.Name ? (from instanceof codegen.Name ? mergeNames(gen, from, to) : mergeToName(gen, from, to), to) : from instanceof codegen.Name ? (mergeToName(gen, to, from), from) : mergeValues(from, to);
            return toName === codegen.Name && !(res instanceof codegen.Name) ? resultToName(gen, res) : res;
        };
    }
    exports.mergeEvaluated = {
        props: makeMergeEvaluated({
            mergeNames: (gen, from, to)=>gen.if(codegen._`${to} !== true && ${from} !== undefined`, ()=>{
                    gen.if(codegen._`${from} === true`, ()=>gen.assign(to, true)
                    , ()=>gen.assign(to, codegen._`${to} || {}`).code(codegen._`Object.assign(${to}, ${from})`)
                    );
                })
            ,
            mergeToName: (gen, from, to)=>gen.if(codegen._`${to} !== true`, ()=>{
                    if (from === true) {
                        gen.assign(to, true);
                    } else {
                        gen.assign(to, codegen._`${to} || {}`);
                        setEvaluated(gen, to, from);
                    }
                })
            ,
            mergeValues: (from, to)=>from === true ? true : {
                    ...from,
                    ...to
                }
            ,
            resultToName: evaluatedPropsToName
        }),
        items: makeMergeEvaluated({
            mergeNames: (gen, from, to)=>gen.if(codegen._`${to} !== true && ${from} !== undefined`, ()=>gen.assign(to, codegen._`${from} === true ? true : ${to} > ${from} ? ${to} : ${from}`)
                )
            ,
            mergeToName: (gen, from, to)=>gen.if(codegen._`${to} !== true`, ()=>gen.assign(to, from === true ? true : codegen._`${to} > ${from} ? ${to} : ${from}`)
                )
            ,
            mergeValues: (from, to)=>from === true ? true : Math.max(from, to)
            ,
            resultToName: (gen, items2)=>gen.var("items", items2)
        })
    };
    function evaluatedPropsToName(gen, ps) {
        if (ps === true) return gen.var("props", true);
        const props = gen.var("props", codegen._`{}`);
        if (ps !== void 0) setEvaluated(gen, props, ps);
        return props;
    }
    exports.evaluatedPropsToName = evaluatedPropsToName;
    function setEvaluated(gen, props, ps) {
        Object.keys(ps).forEach((p)=>gen.assign(codegen._`${props}${(0, codegen.getProperty)(p)}`, true)
        );
    }
    exports.setEvaluated = setEvaluated;
    const snippets = {
    };
    function useFunc(gen, f) {
        return gen.scopeValue("func", {
            ref: f,
            code: snippets[f.code] || (snippets[f.code] = new code2._Code(f.code))
        });
    }
    exports.useFunc = useFunc;
    var Type2;
    (function(Type2) {
        Type2[Type2["Num"] = 0] = "Num";
        Type2[Type2["Str"] = 1] = "Str";
    })(Type2 = exports.Type || (exports.Type = {
    }));
    function getErrorPath(dataProp, dataPropType, jsPropertySyntax) {
        if (dataProp instanceof codegen.Name) {
            const isNumber2 = dataPropType === Type2.Num;
            return jsPropertySyntax ? isNumber2 ? codegen._`"[" + ${dataProp} + "]"` : codegen._`"['" + ${dataProp} + "']"` : isNumber2 ? codegen._`"/" + ${dataProp}` : codegen._`"/" + ${dataProp}.replace(/~/g, "~0").replace(/\\//g, "~1")`;
        }
        return jsPropertySyntax ? (0, codegen.getProperty)(dataProp).toString() : "/" + escapeJsonPointer(dataProp);
    }
    exports.getErrorPath = getErrorPath;
    function checkStrictMode(it, msg, mode = it.opts.strictSchema) {
        if (!mode) return;
        msg = `strict mode: ${msg}`;
        if (mode === true) throw new Error(msg);
        it.self.logger.warn(msg);
    }
    exports.checkStrictMode = checkStrictMode;
});
var names_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const names = {
        data: new codegen.Name("data"),
        valCxt: new codegen.Name("valCxt"),
        instancePath: new codegen.Name("instancePath"),
        parentData: new codegen.Name("parentData"),
        parentDataProperty: new codegen.Name("parentDataProperty"),
        rootData: new codegen.Name("rootData"),
        dynamicAnchors: new codegen.Name("dynamicAnchors"),
        vErrors: new codegen.Name("vErrors"),
        errors: new codegen.Name("errors"),
        this: new codegen.Name("this"),
        self: new codegen.Name("self"),
        scope: new codegen.Name("scope"),
        json: new codegen.Name("json"),
        jsonPos: new codegen.Name("jsonPos"),
        jsonLen: new codegen.Name("jsonLen"),
        jsonPart: new codegen.Name("jsonPart")
    };
    exports.default = names;
});
var errors = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.extendErrors = exports.resetErrorsCount = exports.reportExtraError = exports.reportError = exports.keyword$DataError = exports.keywordError = void 0;
    exports.keywordError = {
        message: ({ keyword: keyword2  })=>codegen.str`must pass "${keyword2}" keyword validation`
    };
    exports.keyword$DataError = {
        message: ({ keyword: keyword2 , schemaType: schemaType1  })=>schemaType1 ? codegen.str`"${keyword2}" keyword must be ${schemaType1} ($data)` : codegen.str`"${keyword2}" keyword is invalid ($data)`
    };
    function reportError(cxt, error20 = exports.keywordError, errorPaths, overrideAllErrors) {
        const { it  } = cxt;
        const { gen , compositeRule , allErrors  } = it;
        const errObj = errorObjectCode(cxt, error20, errorPaths);
        if (overrideAllErrors !== null && overrideAllErrors !== void 0 ? overrideAllErrors : compositeRule || allErrors) {
            addError(gen, errObj);
        } else {
            returnErrors(it, codegen._`[${errObj}]`);
        }
    }
    exports.reportError = reportError;
    function reportExtraError(cxt, error21 = exports.keywordError, errorPaths) {
        const { it  } = cxt;
        const { gen , compositeRule , allErrors  } = it;
        const errObj = errorObjectCode(cxt, error21, errorPaths);
        addError(gen, errObj);
        if (!(compositeRule || allErrors)) {
            returnErrors(it, names_1.default.vErrors);
        }
    }
    exports.reportExtraError = reportExtraError;
    function resetErrorsCount(gen, errsCount) {
        gen.assign(names_1.default.errors, errsCount);
        gen.if(codegen._`${names_1.default.vErrors} !== null`, ()=>gen.if(errsCount, ()=>gen.assign(codegen._`${names_1.default.vErrors}.length`, errsCount)
            , ()=>gen.assign(names_1.default.vErrors, null)
            )
        );
    }
    exports.resetErrorsCount = resetErrorsCount;
    function extendErrors({ gen , keyword: keyword2 , schemaValue , data: data53 , errsCount , it  }) {
        if (errsCount === void 0) throw new Error("ajv implementation error");
        const err = gen.name("err");
        gen.forRange("i", errsCount, names_1.default.errors, (i43)=>{
            gen.const(err, codegen._`${names_1.default.vErrors}[${i43}]`);
            gen.if(codegen._`${err}.instancePath === undefined`, ()=>gen.assign(codegen._`${err}.instancePath`, (0, codegen.strConcat)(names_1.default.instancePath, it.errorPath))
            );
            gen.assign(codegen._`${err}.schemaPath`, codegen.str`${it.errSchemaPath}/${keyword2}`);
            if (it.opts.verbose) {
                gen.assign(codegen._`${err}.schema`, schemaValue);
                gen.assign(codegen._`${err}.data`, data53);
            }
        });
    }
    exports.extendErrors = extendErrors;
    function addError(gen, errObj) {
        const err = gen.const("err", errObj);
        gen.if(codegen._`${names_1.default.vErrors} === null`, ()=>gen.assign(names_1.default.vErrors, codegen._`[${err}]`)
        , codegen._`${names_1.default.vErrors}.push(${err})`);
        gen.code(codegen._`${names_1.default.errors}++`);
    }
    function returnErrors(it, errs) {
        const { gen , validateName , schemaEnv  } = it;
        if (schemaEnv.$async) {
            gen.throw(codegen._`new ${it.ValidationError}(${errs})`);
        } else {
            gen.assign(codegen._`${validateName}.errors`, errs);
            gen.return(false);
        }
    }
    const E = {
        keyword: new codegen.Name("keyword"),
        schemaPath: new codegen.Name("schemaPath"),
        params: new codegen.Name("params"),
        propertyName: new codegen.Name("propertyName"),
        message: new codegen.Name("message"),
        schema: new codegen.Name("schema"),
        parentSchema: new codegen.Name("parentSchema")
    };
    function errorObjectCode(cxt, error22, errorPaths) {
        const { createErrors  } = cxt.it;
        if (createErrors === false) return codegen._`{}`;
        return errorObject(cxt, error22, errorPaths);
    }
    function errorObject(cxt, error23, errorPaths = {
    }) {
        const { gen , it  } = cxt;
        const keyValues = [
            errorInstancePath(it, errorPaths),
            errorSchemaPath(cxt, errorPaths)
        ];
        extraErrorProps(cxt, error23, keyValues);
        return gen.object(...keyValues);
    }
    function errorInstancePath({ errorPath  }, { instancePath  }) {
        const instPath = instancePath ? codegen.str`${errorPath}${(0, util1.getErrorPath)(instancePath, util1.Type.Str)}` : errorPath;
        return [
            names_1.default.instancePath,
            (0, codegen.strConcat)(names_1.default.instancePath, instPath)
        ];
    }
    function errorSchemaPath({ keyword: keyword2 , it: { errSchemaPath  }  }, { schemaPath , parentSchema  }) {
        let schPath = parentSchema ? errSchemaPath : codegen.str`${errSchemaPath}/${keyword2}`;
        if (schemaPath) {
            schPath = codegen.str`${schPath}${(0, util1.getErrorPath)(schemaPath, util1.Type.Str)}`;
        }
        return [
            E.schemaPath,
            schPath
        ];
    }
    function extraErrorProps(cxt, { params , message  }, keyValues) {
        const { keyword: keyword2 , data: data54 , schemaValue , it  } = cxt;
        const { opts , propertyName , topSchemaRef , schemaPath  } = it;
        keyValues.push([
            E.keyword,
            keyword2
        ], [
            E.params,
            typeof params == "function" ? params(cxt) : params || codegen._`{}`
        ]);
        if (opts.messages) {
            keyValues.push([
                E.message,
                typeof message == "function" ? message(cxt) : message
            ]);
        }
        if (opts.verbose) {
            keyValues.push([
                E.schema,
                schemaValue
            ], [
                E.parentSchema,
                codegen._`${topSchemaRef}${schemaPath}`
            ], [
                names_1.default.data,
                data54
            ]);
        }
        if (propertyName) keyValues.push([
            E.propertyName,
            propertyName
        ]);
    }
});
var boolSchema = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.boolOrEmptySchema = exports.topBoolOrEmptySchema = void 0;
    const boolError = {
        message: "boolean schema is false"
    };
    function topBoolOrEmptySchema(it) {
        const { gen , schema , validateName  } = it;
        if (schema === false) {
            falseSchemaError(it, false);
        } else if (typeof schema == "object" && schema.$async === true) {
            gen.return(names_1.default.data);
        } else {
            gen.assign(codegen._`${validateName}.errors`, null);
            gen.return(true);
        }
    }
    exports.topBoolOrEmptySchema = topBoolOrEmptySchema;
    function boolOrEmptySchema(it, valid) {
        const { gen , schema  } = it;
        if (schema === false) {
            gen.var(valid, false);
            falseSchemaError(it);
        } else {
            gen.var(valid, true);
        }
    }
    exports.boolOrEmptySchema = boolOrEmptySchema;
    function falseSchemaError(it, overrideAllErrors) {
        const { gen , data: data55  } = it;
        const cxt = {
            gen,
            keyword: "false schema",
            data: data55,
            schema: false,
            schemaCode: false,
            schemaValue: false,
            params: {
            },
            it
        };
        (0, errors.reportError)(cxt, boolError, void 0, overrideAllErrors);
    }
});
var rules = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getRules = exports.isJSONType = void 0;
    const _jsonTypes = [
        "string",
        "number",
        "integer",
        "boolean",
        "null",
        "object",
        "array"
    ];
    const jsonTypes = new Set(_jsonTypes);
    function isJSONType(x) {
        return typeof x == "string" && jsonTypes.has(x);
    }
    exports.isJSONType = isJSONType;
    function getRules() {
        const groups = {
            number: {
                type: "number",
                rules: []
            },
            string: {
                type: "string",
                rules: []
            },
            array: {
                type: "array",
                rules: []
            },
            object: {
                type: "object",
                rules: []
            }
        };
        return {
            types: {
                ...groups,
                integer: true,
                boolean: true,
                null: true
            },
            rules: [
                {
                    rules: []
                },
                groups.number,
                groups.string,
                groups.array,
                groups.object
            ],
            post: {
                rules: []
            },
            all: {
            },
            keywords: {
            }
        };
    }
    exports.getRules = getRules;
});
var applicability = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.shouldUseRule = exports.shouldUseGroup = exports.schemaHasRulesForType = void 0;
    function schemaHasRulesForType({ schema , self  }, type2) {
        const group = self.RULES.types[type2];
        return group && group !== true && shouldUseGroup(schema, group);
    }
    exports.schemaHasRulesForType = schemaHasRulesForType;
    function shouldUseGroup(schema, group) {
        return group.rules.some((rule)=>shouldUseRule(schema, rule)
        );
    }
    exports.shouldUseGroup = shouldUseGroup;
    function shouldUseRule(schema, rule) {
        var _a;
        return schema[rule.keyword] !== void 0 || ((_a = rule.definition.implements) === null || _a === void 0 ? void 0 : _a.some((kwd)=>schema[kwd] !== void 0
        ));
    }
    exports.shouldUseRule = shouldUseRule;
});
var dataType = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.reportTypeError = exports.checkDataTypes = exports.checkDataType = exports.coerceAndCheckDataType = exports.getJSONTypes = exports.getSchemaTypes = exports.DataType = void 0;
    var DataType;
    (function(DataType2) {
        DataType2[DataType2["Correct"] = 0] = "Correct";
        DataType2[DataType2["Wrong"] = 1] = "Wrong";
    })(DataType = exports.DataType || (exports.DataType = {
    }));
    function getSchemaTypes(schema) {
        const types2 = getJSONTypes(schema.type);
        const hasNull = types2.includes("null");
        if (hasNull) {
            if (schema.nullable === false) throw new Error("type: null contradicts nullable: false");
        } else {
            if (!types2.length && schema.nullable !== void 0) {
                throw new Error('"nullable" cannot be used without "type"');
            }
            if (schema.nullable === true) types2.push("null");
        }
        return types2;
    }
    exports.getSchemaTypes = getSchemaTypes;
    function getJSONTypes(ts) {
        const types2 = Array.isArray(ts) ? ts : ts ? [
            ts
        ] : [];
        if (types2.every(rules.isJSONType)) return types2;
        throw new Error("type must be JSONType or JSONType[]: " + types2.join(","));
    }
    exports.getJSONTypes = getJSONTypes;
    function coerceAndCheckDataType(it, types2) {
        const { gen , data: data56 , opts  } = it;
        const coerceTo = coerceToTypes(types2, opts.coerceTypes);
        const checkTypes = types2.length > 0 && !(coerceTo.length === 0 && types2.length === 1 && (0, applicability.schemaHasRulesForType)(it, types2[0]));
        if (checkTypes) {
            const wrongType = checkDataTypes(types2, data56, opts.strictNumbers, DataType.Wrong);
            gen.if(wrongType, ()=>{
                if (coerceTo.length) coerceData(it, types2, coerceTo);
                else reportTypeError(it);
            });
        }
        return checkTypes;
    }
    exports.coerceAndCheckDataType = coerceAndCheckDataType;
    const COERCIBLE = new Set([
        "string",
        "number",
        "integer",
        "boolean",
        "null"
    ]);
    function coerceToTypes(types2, coerceTypes) {
        return coerceTypes ? types2.filter((t)=>COERCIBLE.has(t) || coerceTypes === "array" && t === "array"
        ) : [];
    }
    function coerceData(it, types2, coerceTo) {
        const { gen , data: data57 , opts  } = it;
        const dataType2 = gen.let("dataType", codegen._`typeof ${data57}`);
        const coerced = gen.let("coerced", codegen._`undefined`);
        if (opts.coerceTypes === "array") {
            gen.if(codegen._`${dataType2} == 'object' && Array.isArray(${data57}) && ${data57}.length == 1`, ()=>gen.assign(data57, codegen._`${data57}[0]`).assign(dataType2, codegen._`typeof ${data57}`).if(checkDataTypes(types2, data57, opts.strictNumbers), ()=>gen.assign(coerced, data57)
                )
            );
        }
        gen.if(codegen._`${coerced} !== undefined`);
        for (const t1 of coerceTo){
            if (COERCIBLE.has(t1) || t1 === "array" && opts.coerceTypes === "array") {
                coerceSpecificType(t1);
            }
        }
        gen.else();
        reportTypeError(it);
        gen.endIf();
        gen.if(codegen._`${coerced} !== undefined`, ()=>{
            gen.assign(data57, coerced);
            assignParentData(it, coerced);
        });
        function coerceSpecificType(t) {
            switch(t){
                case "string":
                    gen.elseIf(codegen._`${dataType2} == "number" || ${dataType2} == "boolean"`).assign(coerced, codegen._`"" + ${data57}`).elseIf(codegen._`${data57} === null`).assign(coerced, codegen._`""`);
                    return;
                case "number":
                    gen.elseIf(codegen._`${dataType2} == "boolean" || ${data57} === null
              || (${dataType2} == "string" && ${data57} && ${data57} == +${data57})`).assign(coerced, codegen._`+${data57}`);
                    return;
                case "integer":
                    gen.elseIf(codegen._`${dataType2} === "boolean" || ${data57} === null
              || (${dataType2} === "string" && ${data57} && ${data57} == +${data57} && !(${data57} % 1))`).assign(coerced, codegen._`+${data57}`);
                    return;
                case "boolean":
                    gen.elseIf(codegen._`${data57} === "false" || ${data57} === 0 || ${data57} === null`).assign(coerced, false).elseIf(codegen._`${data57} === "true" || ${data57} === 1`).assign(coerced, true);
                    return;
                case "null":
                    gen.elseIf(codegen._`${data57} === "" || ${data57} === 0 || ${data57} === false`);
                    gen.assign(coerced, null);
                    return;
                case "array":
                    gen.elseIf(codegen._`${dataType2} === "string" || ${dataType2} === "number"
              || ${dataType2} === "boolean" || ${data57} === null`).assign(coerced, codegen._`[${data57}]`);
            }
        }
    }
    function assignParentData({ gen , parentData , parentDataProperty  }, expr) {
        gen.if(codegen._`${parentData} !== undefined`, ()=>gen.assign(codegen._`${parentData}[${parentDataProperty}]`, expr)
        );
    }
    function checkDataType(dataType2, data58, strictNums, correct = DataType.Correct) {
        const EQ = correct === DataType.Correct ? codegen.operators.EQ : codegen.operators.NEQ;
        let cond3;
        switch(dataType2){
            case "null":
                return codegen._`${data58} ${EQ} null`;
            case "array":
                cond3 = codegen._`Array.isArray(${data58})`;
                break;
            case "object":
                cond3 = codegen._`${data58} && typeof ${data58} == "object" && !Array.isArray(${data58})`;
                break;
            case "integer":
                cond3 = numCond(codegen._`!(${data58} % 1) && !isNaN(${data58})`);
                break;
            case "number":
                cond3 = numCond();
                break;
            default:
                return codegen._`typeof ${data58} ${EQ} ${dataType2}`;
        }
        return correct === DataType.Correct ? cond3 : (0, codegen.not)(cond3);
        function numCond(_cond = codegen.nil) {
            return (0, codegen.and)(codegen._`typeof ${data58} == "number"`, _cond, strictNums ? codegen._`isFinite(${data58})` : codegen.nil);
        }
    }
    exports.checkDataType = checkDataType;
    function checkDataTypes(dataTypes, data59, strictNums, correct) {
        if (dataTypes.length === 1) {
            return checkDataType(dataTypes[0], data59, strictNums, correct);
        }
        let cond4;
        const types2 = (0, util1.toHash)(dataTypes);
        if (types2.array && types2.object) {
            const notObj = codegen._`typeof ${data59} != "object"`;
            cond4 = types2.null ? notObj : codegen._`!${data59} || ${notObj}`;
            delete types2.null;
            delete types2.array;
            delete types2.object;
        } else {
            cond4 = codegen.nil;
        }
        if (types2.number) delete types2.integer;
        for(const t in types2)cond4 = (0, codegen.and)(cond4, checkDataType(t, data59, strictNums, correct));
        return cond4;
    }
    exports.checkDataTypes = checkDataTypes;
    const typeError = {
        message: ({ schema  })=>`must be ${schema}`
        ,
        params: ({ schema , schemaValue  })=>typeof schema == "string" ? codegen._`{type: ${schema}}` : codegen._`{type: ${schemaValue}}`
    };
    function reportTypeError(it) {
        const cxt = getTypeErrorContext(it);
        (0, errors.reportError)(cxt, typeError);
    }
    exports.reportTypeError = reportTypeError;
    function getTypeErrorContext(it) {
        const { gen , data: data60 , schema  } = it;
        const schemaCode = (0, util1.schemaRefOrVal)(it, schema, "type");
        return {
            gen,
            keyword: "type",
            data: data60,
            schema: schema.type,
            schemaCode,
            schemaValue: schemaCode,
            parentSchema: schema,
            params: {
            },
            it
        };
    }
});
var defaults1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.assignDefaults = void 0;
    function assignDefaults(it, ty) {
        const { properties: properties2 , items: items2  } = it.schema;
        if (ty === "object" && properties2) {
            for(const key in properties2){
                assignDefault(it, key, properties2[key].default);
            }
        } else if (ty === "array" && Array.isArray(items2)) {
            items2.forEach((sch, i44)=>assignDefault(it, i44, sch.default)
            );
        }
    }
    exports.assignDefaults = assignDefaults;
    function assignDefault(it, prop, defaultValue) {
        const { gen , compositeRule , data: data61 , opts  } = it;
        if (defaultValue === void 0) return;
        const childData = codegen._`${data61}${(0, codegen.getProperty)(prop)}`;
        if (compositeRule) {
            (0, util1.checkStrictMode)(it, `default is ignored for: ${childData}`);
            return;
        }
        let condition = codegen._`${childData} === undefined`;
        if (opts.useDefaults === "empty") {
            condition = codegen._`${condition} || ${childData} === null || ${childData} === ""`;
        }
        gen.if(condition, codegen._`${childData} = ${(0, codegen.stringify)(defaultValue)}`);
    }
});
var code$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateUnion = exports.validateArray = exports.usePattern = exports.callValidateCode = exports.schemaProperties = exports.allSchemaProperties = exports.noPropertyInData = exports.propertyInData = exports.isOwnProperty = exports.hasPropFunc = exports.reportMissingProp = exports.checkMissingProp = exports.checkReportMissingProp = void 0;
    const util_2 = util1;
    function checkReportMissingProp(cxt, prop) {
        const { gen , data: data62 , it  } = cxt;
        gen.if(noPropertyInData(gen, data62, prop, it.opts.ownProperties), ()=>{
            cxt.setParams({
                missingProperty: codegen._`${prop}`
            }, true);
            cxt.error();
        });
    }
    exports.checkReportMissingProp = checkReportMissingProp;
    function checkMissingProp({ gen , data: data63 , it: { opts  }  }, properties2, missing) {
        return (0, codegen.or)(...properties2.map((prop)=>(0, codegen.and)(noPropertyInData(gen, data63, prop, opts.ownProperties), codegen._`${missing} = ${prop}`)
        ));
    }
    exports.checkMissingProp = checkMissingProp;
    function reportMissingProp(cxt, missing) {
        cxt.setParams({
            missingProperty: missing
        }, true);
        cxt.error();
    }
    exports.reportMissingProp = reportMissingProp;
    function hasPropFunc(gen) {
        return gen.scopeValue("func", {
            ref: Object.prototype.hasOwnProperty,
            code: codegen._`Object.prototype.hasOwnProperty`
        });
    }
    exports.hasPropFunc = hasPropFunc;
    function isOwnProperty(gen, data64, property2) {
        return codegen._`${hasPropFunc(gen)}.call(${data64}, ${property2})`;
    }
    exports.isOwnProperty = isOwnProperty;
    function propertyInData(gen, data65, property3, ownProperties) {
        const cond5 = codegen._`${data65}${(0, codegen.getProperty)(property3)} !== undefined`;
        return ownProperties ? codegen._`${cond5} && ${isOwnProperty(gen, data65, property3)}` : cond5;
    }
    exports.propertyInData = propertyInData;
    function noPropertyInData(gen, data66, property4, ownProperties) {
        const cond6 = codegen._`${data66}${(0, codegen.getProperty)(property4)} === undefined`;
        return ownProperties ? (0, codegen.or)(cond6, (0, codegen.not)(isOwnProperty(gen, data66, property4))) : cond6;
    }
    exports.noPropertyInData = noPropertyInData;
    function allSchemaProperties(schemaMap) {
        return schemaMap ? Object.keys(schemaMap).filter((p)=>p !== "__proto__"
        ) : [];
    }
    exports.allSchemaProperties = allSchemaProperties;
    function schemaProperties(it, schemaMap) {
        return allSchemaProperties(schemaMap).filter((p)=>!(0, util1.alwaysValidSchema)(it, schemaMap[p])
        );
    }
    exports.schemaProperties = schemaProperties;
    function callValidateCode({ schemaCode , data: data67 , it: { gen , topSchemaRef , schemaPath , errorPath  } , it  }, func1, context, passSchema) {
        const dataAndSchema = passSchema ? codegen._`${schemaCode}, ${data67}, ${topSchemaRef}${schemaPath}` : data67;
        const valCxt = [
            [
                names_1.default.instancePath,
                (0, codegen.strConcat)(names_1.default.instancePath, errorPath)
            ],
            [
                names_1.default.parentData,
                it.parentData
            ],
            [
                names_1.default.parentDataProperty,
                it.parentDataProperty
            ],
            [
                names_1.default.rootData,
                names_1.default.rootData
            ]
        ];
        if (it.opts.dynamicRef) valCxt.push([
            names_1.default.dynamicAnchors,
            names_1.default.dynamicAnchors
        ]);
        const args = codegen._`${dataAndSchema}, ${gen.object(...valCxt)}`;
        return context !== codegen.nil ? codegen._`${func1}.call(${context}, ${args})` : codegen._`${func1}(${args})`;
    }
    exports.callValidateCode = callValidateCode;
    const newRegExp = codegen._`new RegExp`;
    function usePattern({ gen , it: { opts  }  }, pattern2) {
        const u = opts.unicodeRegExp ? "u" : "";
        const { regExp  } = opts.code;
        const rx = regExp(pattern2, u);
        return gen.scopeValue("pattern", {
            key: rx.toString(),
            ref: rx,
            code: codegen._`${regExp.code === "new RegExp" ? newRegExp : (0, util_2.useFunc)(gen, regExp)}(${pattern2}, ${u})`
        });
    }
    exports.usePattern = usePattern;
    function validateArray(cxt) {
        const { gen , data: data68 , keyword: keyword2 , it  } = cxt;
        const valid = gen.name("valid");
        if (it.allErrors) {
            const validArr = gen.let("valid", true);
            validateItems(()=>gen.assign(validArr, false)
            );
            return validArr;
        }
        gen.var(valid, true);
        validateItems(()=>gen.break()
        );
        return valid;
        function validateItems(notValid) {
            const len3 = gen.const("len", codegen._`${data68}.length`);
            gen.forRange("i", 0, len3, (i45)=>{
                cxt.subschema({
                    keyword: keyword2,
                    dataProp: i45,
                    dataPropType: util1.Type.Num
                }, valid);
                gen.if((0, codegen.not)(valid), notValid);
            });
        }
    }
    exports.validateArray = validateArray;
    function validateUnion(cxt) {
        const { gen , schema , keyword: keyword2 , it  } = cxt;
        if (!Array.isArray(schema)) throw new Error("ajv implementation error");
        const alwaysValid = schema.some((sch)=>(0, util1.alwaysValidSchema)(it, sch)
        );
        if (alwaysValid && !it.opts.unevaluated) return;
        const valid = gen.let("valid", false);
        const schValid = gen.name("_valid");
        gen.block(()=>schema.forEach((_sch, i46)=>{
                const schCxt = cxt.subschema({
                    keyword: keyword2,
                    schemaProp: i46,
                    compositeRule: true
                }, schValid);
                gen.assign(valid, codegen._`${valid} || ${schValid}`);
                const merged = cxt.mergeValidEvaluated(schCxt, schValid);
                if (!merged) gen.if((0, codegen.not)(valid));
            })
        );
        cxt.result(valid, ()=>cxt.reset()
        , ()=>cxt.error(true)
        );
    }
    exports.validateUnion = validateUnion;
});
var keyword = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateKeywordUsage = exports.validSchemaType = exports.funcKeywordCode = exports.macroKeywordCode = void 0;
    function macroKeywordCode(cxt, def2) {
        const { gen , keyword: keyword2 , schema , parentSchema , it  } = cxt;
        const macroSchema = def2.macro.call(it.self, schema, parentSchema, it);
        const schemaRef = useKeyword(gen, keyword2, macroSchema);
        if (it.opts.validateSchema !== false) it.self.validateSchema(macroSchema, true);
        const valid = gen.name("valid");
        cxt.subschema({
            schema: macroSchema,
            schemaPath: codegen.nil,
            errSchemaPath: `${it.errSchemaPath}/${keyword2}`,
            topSchemaRef: schemaRef,
            compositeRule: true
        }, valid);
        cxt.pass(valid, ()=>cxt.error(true)
        );
    }
    exports.macroKeywordCode = macroKeywordCode;
    function funcKeywordCode(cxt, def3) {
        var _a;
        const { gen , keyword: keyword2 , schema , parentSchema , $data , it  } = cxt;
        checkAsyncKeyword(it, def3);
        const validate2 = !$data && def3.compile ? def3.compile.call(it.self, schema, parentSchema, it) : def3.validate;
        const validateRef = useKeyword(gen, keyword2, validate2);
        const valid = gen.let("valid");
        cxt.block$data(valid, validateKeyword);
        cxt.ok((_a = def3.valid) !== null && _a !== void 0 ? _a : valid);
        function validateKeyword() {
            if (def3.errors === false) {
                assignValid();
                if (def3.modifying) modifyData(cxt);
                reportErrs(()=>cxt.error()
                );
            } else {
                const ruleErrs = def3.async ? validateAsync() : validateSync();
                if (def3.modifying) modifyData(cxt);
                reportErrs(()=>addErrs(cxt, ruleErrs)
                );
            }
        }
        function validateAsync() {
            const ruleErrs = gen.let("ruleErrs", null);
            gen.try(()=>assignValid(codegen._`await `)
            , (e)=>gen.assign(valid, false).if(codegen._`${e} instanceof ${it.ValidationError}`, ()=>gen.assign(ruleErrs, codegen._`${e}.errors`)
                , ()=>gen.throw(e)
                )
            );
            return ruleErrs;
        }
        function validateSync() {
            const validateErrs = codegen._`${validateRef}.errors`;
            gen.assign(validateErrs, null);
            assignValid(codegen.nil);
            return validateErrs;
        }
        function assignValid(_await = def3.async ? codegen._`await ` : codegen.nil) {
            const passCxt = it.opts.passContext ? names_1.default.this : names_1.default.self;
            const passSchema = !("compile" in def3 && !$data || def3.schema === false);
            gen.assign(valid, codegen._`${_await}${(0, code$1.callValidateCode)(cxt, validateRef, passCxt, passSchema)}`, def3.modifying);
        }
        function reportErrs(errors2) {
            var _a2;
            gen.if((0, codegen.not)((_a2 = def3.valid) !== null && _a2 !== void 0 ? _a2 : valid), errors2);
        }
    }
    exports.funcKeywordCode = funcKeywordCode;
    function modifyData(cxt) {
        const { gen , data: data69 , it  } = cxt;
        gen.if(it.parentData, ()=>gen.assign(data69, codegen._`${it.parentData}[${it.parentDataProperty}]`)
        );
    }
    function addErrs(cxt, errs) {
        const { gen  } = cxt;
        gen.if(codegen._`Array.isArray(${errs})`, ()=>{
            gen.assign(names_1.default.vErrors, codegen._`${names_1.default.vErrors} === null ? ${errs} : ${names_1.default.vErrors}.concat(${errs})`).assign(names_1.default.errors, codegen._`${names_1.default.vErrors}.length`);
            (0, errors.extendErrors)(cxt);
        }, ()=>cxt.error()
        );
    }
    function checkAsyncKeyword({ schemaEnv  }, def4) {
        if (def4.async && !schemaEnv.$async) throw new Error("async keyword in sync schema");
    }
    function useKeyword(gen, keyword2, result) {
        if (result === void 0) throw new Error(`keyword "${keyword2}" failed to compile`);
        return gen.scopeValue("keyword", typeof result == "function" ? {
            ref: result
        } : {
            ref: result,
            code: (0, codegen.stringify)(result)
        });
    }
    function validSchemaType(schema, schemaType2, allowUndefined = false) {
        return !schemaType2.length || schemaType2.some((st)=>st === "array" ? Array.isArray(schema) : st === "object" ? schema && typeof schema == "object" && !Array.isArray(schema) : typeof schema == st || allowUndefined && typeof schema == "undefined"
        );
    }
    exports.validSchemaType = validSchemaType;
    function validateKeywordUsage({ schema , opts , self , errSchemaPath  }, def5, keyword2) {
        if (Array.isArray(def5.keyword) ? !def5.keyword.includes(keyword2) : def5.keyword !== keyword2) {
            throw new Error("ajv implementation error");
        }
        const deps = def5.dependencies;
        if (deps === null || deps === void 0 ? void 0 : deps.some((kwd)=>!Object.prototype.hasOwnProperty.call(schema, kwd)
        )) {
            throw new Error(`parent schema must have dependencies of ${keyword2}: ${deps.join(",")}`);
        }
        if (def5.validateSchema) {
            const valid = def5.validateSchema(schema[keyword2]);
            if (!valid) {
                const msg = `keyword "${keyword2}" value is invalid at path "${errSchemaPath}": ` + self.errorsText(def5.validateSchema.errors);
                if (opts.validateSchema === "log") self.logger.error(msg);
                else throw new Error(msg);
            }
        }
    }
    exports.validateKeywordUsage = validateKeywordUsage;
});
var subschema = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.extendSubschemaMode = exports.extendSubschemaData = exports.getSubschema = void 0;
    function getSubschema(it, { keyword: keyword2 , schemaProp , schema , schemaPath , errSchemaPath , topSchemaRef  }) {
        if (keyword2 !== void 0 && schema !== void 0) {
            throw new Error('both "keyword" and "schema" passed, only one allowed');
        }
        if (keyword2 !== void 0) {
            const sch = it.schema[keyword2];
            return schemaProp === void 0 ? {
                schema: sch,
                schemaPath: codegen._`${it.schemaPath}${(0, codegen.getProperty)(keyword2)}`,
                errSchemaPath: `${it.errSchemaPath}/${keyword2}`
            } : {
                schema: sch[schemaProp],
                schemaPath: codegen._`${it.schemaPath}${(0, codegen.getProperty)(keyword2)}${(0, codegen.getProperty)(schemaProp)}`,
                errSchemaPath: `${it.errSchemaPath}/${keyword2}/${(0, util1.escapeFragment)(schemaProp)}`
            };
        }
        if (schema !== void 0) {
            if (schemaPath === void 0 || errSchemaPath === void 0 || topSchemaRef === void 0) {
                throw new Error('"schemaPath", "errSchemaPath" and "topSchemaRef" are required with "schema"');
            }
            return {
                schema,
                schemaPath,
                topSchemaRef,
                errSchemaPath
            };
        }
        throw new Error('either "keyword" or "schema" must be passed');
    }
    exports.getSubschema = getSubschema;
    function extendSubschemaData(subschema2, it, { dataProp , dataPropType: dpType , data: data70 , dataTypes , propertyName  }) {
        if (data70 !== void 0 && dataProp !== void 0) {
            throw new Error('both "data" and "dataProp" passed, only one allowed');
        }
        const { gen  } = it;
        if (dataProp !== void 0) {
            const { errorPath , dataPathArr , opts  } = it;
            const nextData = gen.let("data", codegen._`${it.data}${(0, codegen.getProperty)(dataProp)}`, true);
            dataContextProps(nextData);
            subschema2.errorPath = codegen.str`${errorPath}${(0, util1.getErrorPath)(dataProp, dpType, opts.jsPropertySyntax)}`;
            subschema2.parentDataProperty = codegen._`${dataProp}`;
            subschema2.dataPathArr = [
                ...dataPathArr,
                subschema2.parentDataProperty
            ];
        }
        if (data70 !== void 0) {
            const nextData = data70 instanceof codegen.Name ? data70 : gen.let("data", data70, true);
            dataContextProps(nextData);
            if (propertyName !== void 0) subschema2.propertyName = propertyName;
        }
        if (dataTypes) subschema2.dataTypes = dataTypes;
        function dataContextProps(_nextData) {
            subschema2.data = _nextData;
            subschema2.dataLevel = it.dataLevel + 1;
            subschema2.dataTypes = [];
            it.definedProperties = new Set();
            subschema2.parentData = it.data;
            subschema2.dataNames = [
                ...it.dataNames,
                _nextData
            ];
        }
    }
    exports.extendSubschemaData = extendSubschemaData;
    function extendSubschemaMode(subschema2, { jtdDiscriminator , jtdMetadata , compositeRule , createErrors , allErrors  }) {
        if (compositeRule !== void 0) subschema2.compositeRule = compositeRule;
        if (createErrors !== void 0) subschema2.createErrors = createErrors;
        if (allErrors !== void 0) subschema2.allErrors = allErrors;
        subschema2.jtdDiscriminator = jtdDiscriminator;
        subschema2.jtdMetadata = jtdMetadata;
    }
    exports.extendSubschemaMode = extendSubschemaMode;
});
var resolve3 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getSchemaRefs = exports.resolveUrl = exports.normalizeId = exports._getFullPath = exports.getFullPath = exports.inlineRef = void 0;
    const SIMPLE_INLINED = new Set([
        "type",
        "format",
        "pattern",
        "maxLength",
        "minLength",
        "maxProperties",
        "minProperties",
        "maxItems",
        "minItems",
        "maximum",
        "minimum",
        "uniqueItems",
        "multipleOf",
        "required",
        "enum",
        "const"
    ]);
    function inlineRef(schema, limit = true) {
        if (typeof schema == "boolean") return true;
        if (limit === true) return !hasRef(schema);
        if (!limit) return false;
        return countKeys(schema) <= limit;
    }
    exports.inlineRef = inlineRef;
    const REF_KEYWORDS = new Set([
        "$ref",
        "$recursiveRef",
        "$recursiveAnchor",
        "$dynamicRef",
        "$dynamicAnchor"
    ]);
    function hasRef(schema) {
        for(const key in schema){
            if (REF_KEYWORDS.has(key)) return true;
            const sch = schema[key];
            if (Array.isArray(sch) && sch.some(hasRef)) return true;
            if (typeof sch == "object" && hasRef(sch)) return true;
        }
        return false;
    }
    function countKeys(schema) {
        let count = 0;
        for(const key in schema){
            if (key === "$ref") return Infinity;
            count++;
            if (SIMPLE_INLINED.has(key)) continue;
            if (typeof schema[key] == "object") {
                (0, util1.eachItem)(schema[key], (sch)=>count += countKeys(sch)
                );
            }
            if (count === Infinity) return Infinity;
        }
        return count;
    }
    function getFullPath(id2 = "", normalize9) {
        if (normalize9 !== false) id2 = normalizeId(id2);
        const p = __pika_web_default_export_for_treeshaking__.parse(id2);
        return _getFullPath(p);
    }
    exports.getFullPath = getFullPath;
    function _getFullPath(p) {
        return __pika_web_default_export_for_treeshaking__.serialize(p).split("#")[0] + "#";
    }
    exports._getFullPath = _getFullPath;
    const TRAILING_SLASH_HASH = /#\/?$/;
    function normalizeId(id2) {
        return id2 ? id2.replace(TRAILING_SLASH_HASH, "") : "";
    }
    exports.normalizeId = normalizeId;
    function resolveUrl(baseId, id2) {
        id2 = normalizeId(id2);
        return __pika_web_default_export_for_treeshaking__.resolve(baseId, id2);
    }
    exports.resolveUrl = resolveUrl;
    const ANCHOR = /^[a-z_][-a-z0-9._]*$/i;
    function getSchemaRefs(schema, baseId) {
        if (typeof schema == "boolean") return {
        };
        const { schemaId  } = this.opts;
        const schId = normalizeId(schema[schemaId] || baseId);
        const baseIds = {
            "": schId
        };
        const pathPrefix = getFullPath(schId, false);
        const localRefs = {
        };
        const schemaRefs = new Set();
        jsonSchemaTraverse(schema, {
            allKeys: true
        }, (sch, jsonPtr, _2, parentJsonPtr)=>{
            if (parentJsonPtr === void 0) return;
            const fullPath = pathPrefix + jsonPtr;
            let baseId2 = baseIds[parentJsonPtr];
            if (typeof sch[schemaId] == "string") baseId2 = addRef.call(this, sch[schemaId]);
            addAnchor.call(this, sch.$anchor);
            addAnchor.call(this, sch.$dynamicAnchor);
            baseIds[jsonPtr] = baseId2;
            function addRef(ref2) {
                ref2 = normalizeId(baseId2 ? __pika_web_default_export_for_treeshaking__.resolve(baseId2, ref2) : ref2);
                if (schemaRefs.has(ref2)) throw ambiguos(ref2);
                schemaRefs.add(ref2);
                let schOrRef = this.refs[ref2];
                if (typeof schOrRef == "string") schOrRef = this.refs[schOrRef];
                if (typeof schOrRef == "object") {
                    checkAmbiguosRef(sch, schOrRef.schema, ref2);
                } else if (ref2 !== normalizeId(fullPath)) {
                    if (ref2[0] === "#") {
                        checkAmbiguosRef(sch, localRefs[ref2], ref2);
                        localRefs[ref2] = sch;
                    } else {
                        this.refs[ref2] = fullPath;
                    }
                }
                return ref2;
            }
            function addAnchor(anchor) {
                if (typeof anchor == "string") {
                    if (!ANCHOR.test(anchor)) throw new Error(`invalid anchor "${anchor}"`);
                    addRef.call(this, `#${anchor}`);
                }
            }
        });
        return localRefs;
        function checkAmbiguosRef(sch1, sch2, ref2) {
            if (sch2 !== void 0 && !fastDeepEqual(sch1, sch2)) throw ambiguos(ref2);
        }
        function ambiguos(ref2) {
            return new Error(`reference "${ref2}" resolves to more than one schema`);
        }
    }
    exports.getSchemaRefs = getSchemaRefs;
});
var validate1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getData = exports.KeywordCxt = exports.validateFunctionCode = void 0;
    const dataType_2 = dataType;
    function validateFunctionCode(it) {
        if (isSchemaObj(it)) {
            checkKeywords(it);
            if (schemaCxtHasRules(it)) {
                topSchemaObjCode(it);
                return;
            }
        }
        validateFunction(it, ()=>(0, boolSchema.topBoolOrEmptySchema)(it)
        );
    }
    exports.validateFunctionCode = validateFunctionCode;
    function validateFunction({ gen , validateName , schema , schemaEnv , opts  }, body) {
        if (opts.code.es5) {
            gen.func(validateName, codegen._`${names_1.default.data}, ${names_1.default.valCxt}`, schemaEnv.$async, ()=>{
                gen.code(codegen._`"use strict"; ${funcSourceUrl(schema, opts)}`);
                destructureValCxtES5(gen, opts);
                gen.code(body);
            });
        } else {
            gen.func(validateName, codegen._`${names_1.default.data}, ${destructureValCxt(opts)}`, schemaEnv.$async, ()=>gen.code(funcSourceUrl(schema, opts)).code(body)
            );
        }
    }
    function destructureValCxt(opts) {
        return codegen._`{${names_1.default.instancePath}="", ${names_1.default.parentData}, ${names_1.default.parentDataProperty}, ${names_1.default.rootData}=${names_1.default.data}${opts.dynamicRef ? codegen._`, ${names_1.default.dynamicAnchors}={}` : codegen.nil}}={}`;
    }
    function destructureValCxtES5(gen, opts) {
        gen.if(names_1.default.valCxt, ()=>{
            gen.var(names_1.default.instancePath, codegen._`${names_1.default.valCxt}.${names_1.default.instancePath}`);
            gen.var(names_1.default.parentData, codegen._`${names_1.default.valCxt}.${names_1.default.parentData}`);
            gen.var(names_1.default.parentDataProperty, codegen._`${names_1.default.valCxt}.${names_1.default.parentDataProperty}`);
            gen.var(names_1.default.rootData, codegen._`${names_1.default.valCxt}.${names_1.default.rootData}`);
            if (opts.dynamicRef) gen.var(names_1.default.dynamicAnchors, codegen._`${names_1.default.valCxt}.${names_1.default.dynamicAnchors}`);
        }, ()=>{
            gen.var(names_1.default.instancePath, codegen._`""`);
            gen.var(names_1.default.parentData, codegen._`undefined`);
            gen.var(names_1.default.parentDataProperty, codegen._`undefined`);
            gen.var(names_1.default.rootData, names_1.default.data);
            if (opts.dynamicRef) gen.var(names_1.default.dynamicAnchors, codegen._`{}`);
        });
    }
    function topSchemaObjCode(it) {
        const { schema , opts , gen  } = it;
        validateFunction(it, ()=>{
            if (opts.$comment && schema.$comment) commentKeyword(it);
            checkNoDefault(it);
            gen.let(names_1.default.vErrors, null);
            gen.let(names_1.default.errors, 0);
            if (opts.unevaluated) resetEvaluated(it);
            typeAndKeywords(it);
            returnResults(it);
        });
        return;
    }
    function resetEvaluated(it) {
        const { gen , validateName  } = it;
        it.evaluated = gen.const("evaluated", codegen._`${validateName}.evaluated`);
        gen.if(codegen._`${it.evaluated}.dynamicProps`, ()=>gen.assign(codegen._`${it.evaluated}.props`, codegen._`undefined`)
        );
        gen.if(codegen._`${it.evaluated}.dynamicItems`, ()=>gen.assign(codegen._`${it.evaluated}.items`, codegen._`undefined`)
        );
    }
    function funcSourceUrl(schema, opts) {
        const schId = typeof schema == "object" && schema[opts.schemaId];
        return schId && (opts.code.source || opts.code.process) ? codegen._`/*# sourceURL=${schId} */` : codegen.nil;
    }
    function subschemaCode(it, valid) {
        if (isSchemaObj(it)) {
            checkKeywords(it);
            if (schemaCxtHasRules(it)) {
                subSchemaObjCode(it, valid);
                return;
            }
        }
        (0, boolSchema.boolOrEmptySchema)(it, valid);
    }
    function schemaCxtHasRules({ schema , self  }) {
        if (typeof schema == "boolean") return !schema;
        for(const key in schema)if (self.RULES.all[key]) return true;
        return false;
    }
    function isSchemaObj(it) {
        return typeof it.schema != "boolean";
    }
    function subSchemaObjCode(it, valid) {
        const { schema , gen , opts  } = it;
        if (opts.$comment && schema.$comment) commentKeyword(it);
        updateContext(it);
        checkAsyncSchema(it);
        const errsCount = gen.const("_errs", names_1.default.errors);
        typeAndKeywords(it, errsCount);
        gen.var(valid, codegen._`${errsCount} === ${names_1.default.errors}`);
    }
    function checkKeywords(it) {
        (0, util1.checkUnknownRules)(it);
        checkRefsAndKeywords(it);
    }
    function typeAndKeywords(it, errsCount) {
        if (it.opts.jtd) return schemaKeywords(it, [], false, errsCount);
        const types2 = (0, dataType.getSchemaTypes)(it.schema);
        const checkedTypes = (0, dataType.coerceAndCheckDataType)(it, types2);
        schemaKeywords(it, types2, !checkedTypes, errsCount);
    }
    function checkRefsAndKeywords(it) {
        const { schema , errSchemaPath , opts , self  } = it;
        if (schema.$ref && opts.ignoreKeywordsWithRef && (0, util1.schemaHasRulesButRef)(schema, self.RULES)) {
            self.logger.warn(`$ref: keywords ignored in schema at path "${errSchemaPath}"`);
        }
    }
    function checkNoDefault(it) {
        const { schema , opts  } = it;
        if (schema.default !== void 0 && opts.useDefaults && opts.strictSchema) {
            (0, util1.checkStrictMode)(it, "default is ignored in the schema root");
        }
    }
    function updateContext(it) {
        const schId = it.schema[it.opts.schemaId];
        if (schId) it.baseId = (0, resolve3.resolveUrl)(it.baseId, schId);
    }
    function checkAsyncSchema(it) {
        if (it.schema.$async && !it.schemaEnv.$async) throw new Error("async schema in sync schema");
    }
    function commentKeyword({ gen , schemaEnv , schema , errSchemaPath , opts  }) {
        const msg = schema.$comment;
        if (opts.$comment === true) {
            gen.code(codegen._`${names_1.default.self}.logger.log(${msg})`);
        } else if (typeof opts.$comment == "function") {
            const schemaPath = codegen.str`${errSchemaPath}/$comment`;
            const rootName = gen.scopeValue("root", {
                ref: schemaEnv.root
            });
            gen.code(codegen._`${names_1.default.self}.opts.$comment(${msg}, ${schemaPath}, ${rootName}.schema)`);
        }
    }
    function returnResults(it) {
        const { gen , schemaEnv , validateName , ValidationError: ValidationError3 , opts  } = it;
        if (schemaEnv.$async) {
            gen.if(codegen._`${names_1.default.errors} === 0`, ()=>gen.return(names_1.default.data)
            , ()=>gen.throw(codegen._`new ${ValidationError3}(${names_1.default.vErrors})`)
            );
        } else {
            gen.assign(codegen._`${validateName}.errors`, names_1.default.vErrors);
            if (opts.unevaluated) assignEvaluated(it);
            gen.return(codegen._`${names_1.default.errors} === 0`);
        }
    }
    function assignEvaluated({ gen , evaluated , props , items: items2  }) {
        if (props instanceof codegen.Name) gen.assign(codegen._`${evaluated}.props`, props);
        if (items2 instanceof codegen.Name) gen.assign(codegen._`${evaluated}.items`, items2);
    }
    function schemaKeywords(it, types2, typeErrors, errsCount) {
        const { gen , schema , data: data71 , allErrors , opts , self  } = it;
        const { RULES  } = self;
        if (schema.$ref && (opts.ignoreKeywordsWithRef || !(0, util1.schemaHasRulesButRef)(schema, RULES))) {
            gen.block(()=>keywordCode(it, "$ref", RULES.all.$ref.definition)
            );
            return;
        }
        if (!opts.jtd) checkStrictTypes(it, types2);
        gen.block(()=>{
            for (const group of RULES.rules)groupKeywords(group);
            groupKeywords(RULES.post);
        });
        function groupKeywords(group) {
            if (!(0, applicability.shouldUseGroup)(schema, group)) return;
            if (group.type) {
                gen.if((0, dataType_2.checkDataType)(group.type, data71, opts.strictNumbers));
                iterateKeywords(it, group);
                if (types2.length === 1 && types2[0] === group.type && typeErrors) {
                    gen.else();
                    (0, dataType_2.reportTypeError)(it);
                }
                gen.endIf();
            } else {
                iterateKeywords(it, group);
            }
            if (!allErrors) gen.if(codegen._`${names_1.default.errors} === ${errsCount || 0}`);
        }
    }
    function iterateKeywords(it, group) {
        const { gen , schema , opts: { useDefaults  }  } = it;
        if (useDefaults) (0, defaults1.assignDefaults)(it, group.type);
        gen.block(()=>{
            for (const rule of group.rules){
                if ((0, applicability.shouldUseRule)(schema, rule)) {
                    keywordCode(it, rule.keyword, rule.definition, group.type);
                }
            }
        });
    }
    function checkStrictTypes(it, types2) {
        if (it.schemaEnv.meta || !it.opts.strictTypes) return;
        checkContextTypes(it, types2);
        if (!it.opts.allowUnionTypes) checkMultipleTypes(it, types2);
        checkKeywordTypes(it, it.dataTypes);
    }
    function checkContextTypes(it, types2) {
        if (!types2.length) return;
        if (!it.dataTypes.length) {
            it.dataTypes = types2;
            return;
        }
        types2.forEach((t)=>{
            if (!includesType(it.dataTypes, t)) {
                strictTypesError(it, `type "${t}" not allowed by context "${it.dataTypes.join(",")}"`);
            }
        });
        it.dataTypes = it.dataTypes.filter((t)=>includesType(types2, t)
        );
    }
    function checkMultipleTypes(it, ts) {
        if (ts.length > 1 && !(ts.length === 2 && ts.includes("null"))) {
            strictTypesError(it, "use allowUnionTypes to allow union type keyword");
        }
    }
    function checkKeywordTypes(it, ts) {
        const rules2 = it.self.RULES.all;
        for(const keyword2 in rules2){
            const rule = rules2[keyword2];
            if (typeof rule == "object" && (0, applicability.shouldUseRule)(it.schema, rule)) {
                const { type: type2  } = rule.definition;
                if (type2.length && !type2.some((t)=>hasApplicableType(ts, t)
                )) {
                    strictTypesError(it, `missing type "${type2.join(",")}" for keyword "${keyword2}"`);
                }
            }
        }
    }
    function hasApplicableType(schTs, kwdT) {
        return schTs.includes(kwdT) || kwdT === "number" && schTs.includes("integer");
    }
    function includesType(ts, t) {
        return ts.includes(t) || t === "integer" && ts.includes("number");
    }
    function strictTypesError(it, msg) {
        const schemaPath = it.schemaEnv.baseId + it.errSchemaPath;
        msg += ` at "${schemaPath}" (strictTypes)`;
        (0, util1.checkStrictMode)(it, msg, it.opts.strictTypes);
    }
    class KeywordCxt2 {
        constructor(it, def6, keyword$1){
            (0, keyword.validateKeywordUsage)(it, def6, keyword$1);
            this.gen = it.gen;
            this.allErrors = it.allErrors;
            this.keyword = keyword$1;
            this.data = it.data;
            this.schema = it.schema[keyword$1];
            this.$data = def6.$data && it.opts.$data && this.schema && this.schema.$data;
            this.schemaValue = (0, util1.schemaRefOrVal)(it, this.schema, keyword$1, this.$data);
            this.schemaType = def6.schemaType;
            this.parentSchema = it.schema;
            this.params = {
            };
            this.it = it;
            this.def = def6;
            if (this.$data) {
                this.schemaCode = it.gen.const("vSchema", getData1(this.$data, it));
            } else {
                this.schemaCode = this.schemaValue;
                if (!(0, keyword.validSchemaType)(this.schema, def6.schemaType, def6.allowUndefined)) {
                    throw new Error(`${keyword$1} value must be ${JSON.stringify(def6.schemaType)}`);
                }
            }
            if ("code" in def6 ? def6.trackErrors : def6.errors !== false) {
                this.errsCount = it.gen.const("_errs", names_1.default.errors);
            }
        }
        result(condition, successAction, failAction) {
            this.failResult((0, codegen.not)(condition), successAction, failAction);
        }
        failResult(condition, successAction, failAction) {
            this.gen.if(condition);
            if (failAction) failAction();
            else this.error();
            if (successAction) {
                this.gen.else();
                successAction();
                if (this.allErrors) this.gen.endIf();
            } else {
                if (this.allErrors) this.gen.endIf();
                else this.gen.else();
            }
        }
        pass(condition, failAction) {
            this.failResult((0, codegen.not)(condition), void 0, failAction);
        }
        fail(condition) {
            if (condition === void 0) {
                this.error();
                if (!this.allErrors) this.gen.if(false);
                return;
            }
            this.gen.if(condition);
            this.error();
            if (this.allErrors) this.gen.endIf();
            else this.gen.else();
        }
        fail$data(condition) {
            if (!this.$data) return this.fail(condition);
            const { schemaCode  } = this;
            this.fail(codegen._`${schemaCode} !== undefined && (${(0, codegen.or)(this.invalid$data(), condition)})`);
        }
        error(append, errorParams, errorPaths) {
            if (errorParams) {
                this.setParams(errorParams);
                this._error(append, errorPaths);
                this.setParams({
                });
                return;
            }
            this._error(append, errorPaths);
        }
        _error(append, errorPaths) {
            (append ? errors.reportExtraError : errors.reportError)(this, this.def.error, errorPaths);
        }
        $dataError() {
            (0, errors.reportError)(this, this.def.$dataError || errors.keyword$DataError);
        }
        reset() {
            if (this.errsCount === void 0) throw new Error('add "trackErrors" to keyword definition');
            (0, errors.resetErrorsCount)(this.gen, this.errsCount);
        }
        ok(cond7) {
            if (!this.allErrors) this.gen.if(cond7);
        }
        setParams(obj, assign2) {
            if (assign2) Object.assign(this.params, obj);
            else this.params = obj;
        }
        block$data(valid, codeBlock, $dataValid = codegen.nil) {
            this.gen.block(()=>{
                this.check$data(valid, $dataValid);
                codeBlock();
            });
        }
        check$data(valid = codegen.nil, $dataValid = codegen.nil) {
            if (!this.$data) return;
            const { gen , schemaCode , schemaType: schemaType3 , def: def7  } = this;
            gen.if((0, codegen.or)(codegen._`${schemaCode} === undefined`, $dataValid));
            if (valid !== codegen.nil) gen.assign(valid, true);
            if (schemaType3.length || def7.validateSchema) {
                gen.elseIf(this.invalid$data());
                this.$dataError();
                if (valid !== codegen.nil) gen.assign(valid, false);
            }
            gen.else();
        }
        invalid$data() {
            const { gen , schemaCode , schemaType: schemaType4 , def: def8 , it  } = this;
            return (0, codegen.or)(wrong$DataType(), invalid$DataSchema());
            function wrong$DataType() {
                if (schemaType4.length) {
                    if (!(schemaCode instanceof codegen.Name)) throw new Error("ajv implementation error");
                    const st = Array.isArray(schemaType4) ? schemaType4 : [
                        schemaType4
                    ];
                    return codegen._`${(0, dataType_2.checkDataTypes)(st, schemaCode, it.opts.strictNumbers, dataType_2.DataType.Wrong)}`;
                }
                return codegen.nil;
            }
            function invalid$DataSchema() {
                if (def8.validateSchema) {
                    const validateSchemaRef = gen.scopeValue("validate$data", {
                        ref: def8.validateSchema
                    });
                    return codegen._`!${validateSchemaRef}(${schemaCode})`;
                }
                return codegen.nil;
            }
        }
        subschema(appl, valid) {
            const subschema$1 = (0, subschema.getSubschema)(this.it, appl);
            (0, subschema.extendSubschemaData)(subschema$1, this.it, appl);
            (0, subschema.extendSubschemaMode)(subschema$1, appl);
            const nextContext = {
                ...this.it,
                ...subschema$1,
                items: void 0,
                props: void 0
            };
            subschemaCode(nextContext, valid);
            return nextContext;
        }
        mergeEvaluated(schemaCxt, toName) {
            const { it , gen  } = this;
            if (!it.opts.unevaluated) return;
            if (it.props !== true && schemaCxt.props !== void 0) {
                it.props = util1.mergeEvaluated.props(gen, schemaCxt.props, it.props, toName);
            }
            if (it.items !== true && schemaCxt.items !== void 0) {
                it.items = util1.mergeEvaluated.items(gen, schemaCxt.items, it.items, toName);
            }
        }
        mergeValidEvaluated(schemaCxt, valid) {
            const { it , gen  } = this;
            if (it.opts.unevaluated && (it.props !== true || it.items !== true)) {
                gen.if(valid, ()=>this.mergeEvaluated(schemaCxt, codegen.Name)
                );
                return true;
            }
        }
    }
    exports.KeywordCxt = KeywordCxt2;
    function keywordCode(it, keyword$1, def9, ruleType) {
        const cxt = new KeywordCxt2(it, def9, keyword$1);
        if ("code" in def9) {
            def9.code(cxt, ruleType);
        } else if (cxt.$data && def9.validate) {
            (0, keyword.funcKeywordCode)(cxt, def9);
        } else if ("macro" in def9) {
            (0, keyword.macroKeywordCode)(cxt, def9);
        } else if (def9.compile || def9.validate) {
            (0, keyword.funcKeywordCode)(cxt, def9);
        }
    }
    const JSON_POINTER = /^\/(?:[^~]|~0|~1)*$/;
    const RELATIVE_JSON_POINTER = /^([0-9]+)(#|\/(?:[^~]|~0|~1)*)?$/;
    function getData1($data, { dataLevel , dataNames , dataPathArr  }) {
        let jsonPointer;
        let data72;
        if ($data === "") return names_1.default.rootData;
        if ($data[0] === "/") {
            if (!JSON_POINTER.test($data)) throw new Error(`Invalid JSON-pointer: ${$data}`);
            jsonPointer = $data;
            data72 = names_1.default.rootData;
        } else {
            const matches5 = RELATIVE_JSON_POINTER.exec($data);
            if (!matches5) throw new Error(`Invalid JSON-pointer: ${$data}`);
            const up = +matches5[1];
            jsonPointer = matches5[2];
            if (jsonPointer === "#") {
                if (up >= dataLevel) throw new Error(errorMsg("property/index", up));
                return dataPathArr[dataLevel - up];
            }
            if (up > dataLevel) throw new Error(errorMsg("data", up));
            data72 = dataNames[dataLevel - up];
            if (!jsonPointer) return data72;
        }
        let expr = data72;
        const segments = jsonPointer.split("/");
        for (const segment of segments){
            if (segment) {
                data72 = codegen._`${data72}${(0, codegen.getProperty)((0, util1.unescapeJsonPointer)(segment))}`;
                expr = codegen._`${expr} && ${data72}`;
            }
        }
        return expr;
        function errorMsg(pointerType, up) {
            return `Cannot access ${pointerType} ${up} levels up, current level is ${dataLevel}`;
        }
    }
    exports.getData = getData1;
});
var validation_error = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    class ValidationError4 extends Error {
        constructor(errors2){
            super("validation failed");
            this.errors = errors2;
            this.ajv = this.validation = true;
        }
    }
    exports.default = ValidationError4;
});
var ref_error = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    class MissingRefError extends Error {
        constructor(baseId, ref2, msg){
            super(msg || `can't resolve reference ${ref2} from id ${baseId}`);
            this.missingRef = (0, resolve3.resolveUrl)(baseId, ref2);
            this.missingSchema = (0, resolve3.normalizeId)((0, resolve3.getFullPath)(this.missingRef));
        }
    }
    exports.default = MissingRefError;
});
var compile = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.resolveSchema = exports.getCompilingSchema = exports.resolveRef = exports.compileSchema = exports.SchemaEnv = void 0;
    class SchemaEnv {
        constructor(env){
            var _a;
            this.refs = {
            };
            this.dynamicAnchors = {
            };
            let schema;
            if (typeof env.schema == "object") schema = env.schema;
            this.schema = env.schema;
            this.schemaId = env.schemaId;
            this.root = env.root || this;
            this.baseId = (_a = env.baseId) !== null && _a !== void 0 ? _a : (0, resolve3.normalizeId)(schema === null || schema === void 0 ? void 0 : schema[env.schemaId || "$id"]);
            this.schemaPath = env.schemaPath;
            this.localRefs = env.localRefs;
            this.meta = env.meta;
            this.$async = schema === null || schema === void 0 ? void 0 : schema.$async;
            this.refs = {
            };
        }
    }
    exports.SchemaEnv = SchemaEnv;
    function compileSchema(sch) {
        const _sch = getCompilingSchema.call(this, sch);
        if (_sch) return _sch;
        const rootId = (0, resolve3.getFullPath)(sch.root.baseId);
        const { es5 , lines: lines1  } = this.opts.code;
        const { ownProperties  } = this.opts;
        const gen = new codegen.CodeGen(this.scope, {
            es5,
            lines: lines1,
            ownProperties
        });
        let _ValidationError;
        if (sch.$async) {
            _ValidationError = gen.scopeValue("Error", {
                ref: validation_error.default,
                code: codegen._`require("ajv/dist/runtime/validation_error").default`
            });
        }
        const validateName = gen.scopeName("validate");
        sch.validateName = validateName;
        const schemaCxt = {
            gen,
            allErrors: this.opts.allErrors,
            data: names_1.default.data,
            parentData: names_1.default.parentData,
            parentDataProperty: names_1.default.parentDataProperty,
            dataNames: [
                names_1.default.data
            ],
            dataPathArr: [
                codegen.nil
            ],
            dataLevel: 0,
            dataTypes: [],
            definedProperties: new Set(),
            topSchemaRef: gen.scopeValue("schema", this.opts.code.source === true ? {
                ref: sch.schema,
                code: (0, codegen.stringify)(sch.schema)
            } : {
                ref: sch.schema
            }),
            validateName,
            ValidationError: _ValidationError,
            schema: sch.schema,
            schemaEnv: sch,
            rootId,
            baseId: sch.baseId || rootId,
            schemaPath: codegen.nil,
            errSchemaPath: sch.schemaPath || (this.opts.jtd ? "" : "#"),
            errorPath: codegen._`""`,
            opts: this.opts,
            self: this
        };
        let sourceCode;
        try {
            this._compilations.add(sch);
            (0, validate1.validateFunctionCode)(schemaCxt);
            gen.optimize(this.opts.code.optimize);
            const validateCode = gen.toString();
            sourceCode = `${gen.scopeRefs(names_1.default.scope)}return ${validateCode}`;
            if (this.opts.code.process) sourceCode = this.opts.code.process(sourceCode, sch);
            const makeValidate = new Function(`${names_1.default.self}`, `${names_1.default.scope}`, sourceCode);
            const validate$1 = makeValidate(this, this.scope.get());
            this.scope.value(validateName, {
                ref: validate$1
            });
            validate$1.errors = null;
            validate$1.schema = sch.schema;
            validate$1.schemaEnv = sch;
            if (sch.$async) validate$1.$async = true;
            if (this.opts.code.source === true) {
                validate$1.source = {
                    validateName,
                    validateCode,
                    scopeValues: gen._values
                };
            }
            if (this.opts.unevaluated) {
                const { props , items: items2  } = schemaCxt;
                validate$1.evaluated = {
                    props: props instanceof codegen.Name ? void 0 : props,
                    items: items2 instanceof codegen.Name ? void 0 : items2,
                    dynamicProps: props instanceof codegen.Name,
                    dynamicItems: items2 instanceof codegen.Name
                };
                if (validate$1.source) validate$1.source.evaluated = (0, codegen.stringify)(validate$1.evaluated);
            }
            sch.validate = validate$1;
            return sch;
        } catch (e) {
            delete sch.validate;
            delete sch.validateName;
            if (sourceCode) this.logger.error("Error compiling schema, function code:", sourceCode);
            throw e;
        } finally{
            this._compilations.delete(sch);
        }
    }
    exports.compileSchema = compileSchema;
    function resolveRef(root6, baseId, ref2) {
        var _a;
        ref2 = (0, resolve3.resolveUrl)(baseId, ref2);
        const schOrFunc = root6.refs[ref2];
        if (schOrFunc) return schOrFunc;
        let _sch = resolve$1.call(this, root6, ref2);
        if (_sch === void 0) {
            const schema = (_a = root6.localRefs) === null || _a === void 0 ? void 0 : _a[ref2];
            const { schemaId  } = this.opts;
            if (schema) _sch = new SchemaEnv({
                schema,
                schemaId,
                root: root6,
                baseId
            });
        }
        if (_sch === void 0) return;
        return root6.refs[ref2] = inlineOrCompile.call(this, _sch);
    }
    exports.resolveRef = resolveRef;
    function inlineOrCompile(sch) {
        if ((0, resolve3.inlineRef)(sch.schema, this.opts.inlineRefs)) return sch.schema;
        return sch.validate ? sch : compileSchema.call(this, sch);
    }
    function getCompilingSchema(schEnv) {
        for (const sch of this._compilations){
            if (sameSchemaEnv(sch, schEnv)) return sch;
        }
    }
    exports.getCompilingSchema = getCompilingSchema;
    function sameSchemaEnv(s1, s2) {
        return s1.schema === s2.schema && s1.root === s2.root && s1.baseId === s2.baseId;
    }
    function resolve$1(root7, ref2) {
        let sch;
        while(typeof (sch = this.refs[ref2]) == "string")ref2 = sch;
        return sch || this.schemas[ref2] || resolveSchema.call(this, root7, ref2);
    }
    function resolveSchema(root8, ref2) {
        const p = __pika_web_default_export_for_treeshaking__.parse(ref2);
        const refPath = (0, resolve3._getFullPath)(p);
        let baseId = (0, resolve3.getFullPath)(root8.baseId);
        if (Object.keys(root8.schema).length > 0 && refPath === baseId) {
            return getJsonPointer.call(this, p, root8);
        }
        const id2 = (0, resolve3.normalizeId)(refPath);
        const schOrRef = this.refs[id2] || this.schemas[id2];
        if (typeof schOrRef == "string") {
            const sch = resolveSchema.call(this, root8, schOrRef);
            if (typeof (sch === null || sch === void 0 ? void 0 : sch.schema) !== "object") return;
            return getJsonPointer.call(this, p, sch);
        }
        if (typeof (schOrRef === null || schOrRef === void 0 ? void 0 : schOrRef.schema) !== "object") return;
        if (!schOrRef.validate) compileSchema.call(this, schOrRef);
        if (id2 === (0, resolve3.normalizeId)(ref2)) {
            const { schema  } = schOrRef;
            const { schemaId  } = this.opts;
            const schId = schema[schemaId];
            if (schId) baseId = (0, resolve3.resolveUrl)(baseId, schId);
            return new SchemaEnv({
                schema,
                schemaId,
                root: root8,
                baseId
            });
        }
        return getJsonPointer.call(this, p, schOrRef);
    }
    exports.resolveSchema = resolveSchema;
    const PREVENT_SCOPE_CHANGE = new Set([
        "properties",
        "patternProperties",
        "enum",
        "dependencies",
        "definitions"
    ]);
    function getJsonPointer(parsedRef, { baseId , schema , root: root9  }) {
        var _a;
        if (((_a = parsedRef.fragment) === null || _a === void 0 ? void 0 : _a[0]) !== "/") return;
        for (const part of parsedRef.fragment.slice(1).split("/")){
            if (typeof schema === "boolean") return;
            const partSchema = schema[(0, util1.unescapeFragment)(part)];
            if (partSchema === void 0) return;
            schema = partSchema;
            const schId = typeof schema === "object" && schema[this.opts.schemaId];
            if (!PREVENT_SCOPE_CHANGE.has(part) && schId) {
                baseId = (0, resolve3.resolveUrl)(baseId, schId);
            }
        }
        let env;
        if (typeof schema != "boolean" && schema.$ref && !(0, util1.schemaHasRulesButRef)(schema, this.RULES)) {
            const $ref = (0, resolve3.resolveUrl)(baseId, schema.$ref);
            env = resolveSchema.call(this, root9, $ref);
        }
        const { schemaId  } = this.opts;
        env = env || new SchemaEnv({
            schema,
            schemaId,
            root: root9,
            baseId
        });
        if (env.schema !== env.root.schema) return env;
        return void 0;
    }
});
const $id = "https://raw.githubusercontent.com/ajv-validator/ajv/master/lib/refs/data.json#";
const description = "Meta-schema for $data reference (JSON AnySchema extension proposal)";
const type40 = "object";
const required = [
    "$data"
];
const properties = {
    $data: {
        type: "string",
        anyOf: [
            {
                format: "relative-json-pointer"
            },
            {
                format: "json-pointer"
            }
        ]
    }
};
var $dataRefSchema = {
    $id,
    description,
    type: type40,
    required,
    properties,
    additionalProperties: false
};
var core1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.CodeGen = exports.Name = exports.nil = exports.stringify = exports.str = exports._ = exports.KeywordCxt = void 0;
    Object.defineProperty(exports, "KeywordCxt", {
        enumerable: true,
        get: function() {
            return validate1.KeywordCxt;
        }
    });
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return codegen._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return codegen.str;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return codegen.stringify;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return codegen.nil;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return codegen.Name;
        }
    });
    Object.defineProperty(exports, "CodeGen", {
        enumerable: true,
        get: function() {
            return codegen.CodeGen;
        }
    });
    const codegen_2 = codegen;
    const defaultRegExp = (str2, flags)=>new RegExp(str2, flags)
    ;
    defaultRegExp.code = "new RegExp";
    const META_IGNORE_OPTIONS = [
        "removeAdditional",
        "useDefaults",
        "coerceTypes"
    ];
    const EXT_SCOPE_NAMES = new Set([
        "validate",
        "serialize",
        "parse",
        "wrapper",
        "root",
        "schema",
        "keyword",
        "pattern",
        "formats",
        "validate$data",
        "func",
        "obj",
        "Error"
    ]);
    const removedOptions = {
        errorDataPath: "",
        format: "`validateFormats: false` can be used instead.",
        nullable: '"nullable" keyword is supported by default.',
        jsonPointers: "Deprecated jsPropertySyntax can be used instead.",
        extendRefs: "Deprecated ignoreKeywordsWithRef can be used instead.",
        missingRefs: "Pass empty schema with $id that should be ignored to ajv.addSchema.",
        processCode: "Use option `code: {process: (code, schemaEnv: object) => string}`",
        sourceCode: "Use option `code: {source: true}`",
        strictDefaults: "It is default now, see option `strict`.",
        strictKeywords: "It is default now, see option `strict`.",
        uniqueItems: '"uniqueItems" keyword is always validated.',
        unknownFormats: "Disable strict mode or pass `true` to `ajv.addFormat` (or `formats` option).",
        cache: "Map is used as cache, schema object as key.",
        serialize: "Map is used as cache, schema object as key.",
        ajvErrors: "It is default now."
    };
    const deprecatedOptions = {
        ignoreKeywordsWithRef: "",
        jsPropertySyntax: "",
        unicode: '"minLength"/"maxLength" account for unicode characters by default.'
    };
    function requiredOptions(o) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z;
        const s = o.strict;
        const _optz = (_a = o.code) === null || _a === void 0 ? void 0 : _a.optimize;
        const optimize = _optz === true || _optz === void 0 ? 1 : _optz || 0;
        const regExp = (_c = (_b = o.code) === null || _b === void 0 ? void 0 : _b.regExp) !== null && _c !== void 0 ? _c : defaultRegExp;
        return {
            strictSchema: (_e = (_d = o.strictSchema) !== null && _d !== void 0 ? _d : s) !== null && _e !== void 0 ? _e : true,
            strictNumbers: (_g = (_f = o.strictNumbers) !== null && _f !== void 0 ? _f : s) !== null && _g !== void 0 ? _g : true,
            strictTypes: (_j = (_h = o.strictTypes) !== null && _h !== void 0 ? _h : s) !== null && _j !== void 0 ? _j : "log",
            strictTuples: (_l = (_k = o.strictTuples) !== null && _k !== void 0 ? _k : s) !== null && _l !== void 0 ? _l : "log",
            strictRequired: (_o = (_m = o.strictRequired) !== null && _m !== void 0 ? _m : s) !== null && _o !== void 0 ? _o : false,
            code: o.code ? {
                ...o.code,
                optimize,
                regExp
            } : {
                optimize,
                regExp
            },
            loopRequired: (_p = o.loopRequired) !== null && _p !== void 0 ? _p : 200,
            loopEnum: (_q = o.loopEnum) !== null && _q !== void 0 ? _q : 200,
            meta: (_r = o.meta) !== null && _r !== void 0 ? _r : true,
            messages: (_s = o.messages) !== null && _s !== void 0 ? _s : true,
            inlineRefs: (_t = o.inlineRefs) !== null && _t !== void 0 ? _t : true,
            schemaId: (_u = o.schemaId) !== null && _u !== void 0 ? _u : "$id",
            addUsedSchema: (_v = o.addUsedSchema) !== null && _v !== void 0 ? _v : true,
            validateSchema: (_w = o.validateSchema) !== null && _w !== void 0 ? _w : true,
            validateFormats: (_x = o.validateFormats) !== null && _x !== void 0 ? _x : true,
            unicodeRegExp: (_y = o.unicodeRegExp) !== null && _y !== void 0 ? _y : true,
            int32range: (_z = o.int32range) !== null && _z !== void 0 ? _z : true
        };
    }
    class Ajv {
        constructor(opts = {
        }){
            this.schemas = {
            };
            this.refs = {
            };
            this.formats = {
            };
            this._compilations = new Set();
            this._loading = {
            };
            this._cache = new Map();
            opts = this.opts = {
                ...opts,
                ...requiredOptions(opts)
            };
            const { es5 , lines: lines2  } = this.opts.code;
            this.scope = new codegen_2.ValueScope({
                scope: {
                },
                prefixes: EXT_SCOPE_NAMES,
                es5,
                lines: lines2
            });
            this.logger = getLogger1(opts.logger);
            const formatOpt = opts.validateFormats;
            opts.validateFormats = false;
            this.RULES = (0, rules.getRules)();
            checkOptions.call(this, removedOptions, opts, "NOT SUPPORTED");
            checkOptions.call(this, deprecatedOptions, opts, "DEPRECATED", "warn");
            this._metaOpts = getMetaSchemaOptions.call(this);
            if (opts.formats) addInitialFormats.call(this);
            this._addVocabularies();
            this._addDefaultMetaSchema();
            if (opts.keywords) addInitialKeywords.call(this, opts.keywords);
            if (typeof opts.meta == "object") this.addMetaSchema(opts.meta);
            addInitialSchemas.call(this);
            opts.validateFormats = formatOpt;
        }
        _addVocabularies() {
            this.addKeyword("$async");
        }
        _addDefaultMetaSchema() {
            const { $data , meta , schemaId  } = this.opts;
            let _dataRefSchema = $dataRefSchema;
            if (schemaId === "id") {
                _dataRefSchema = {
                    ...$dataRefSchema
                };
                _dataRefSchema.id = _dataRefSchema.$id;
                delete _dataRefSchema.$id;
            }
            if (meta && $data) this.addMetaSchema(_dataRefSchema, _dataRefSchema[schemaId], false);
        }
        defaultMeta() {
            const { meta , schemaId  } = this.opts;
            return this.opts.defaultMeta = typeof meta == "object" ? meta[schemaId] || meta : void 0;
        }
        validate(schemaKeyRef, data73) {
            let v;
            if (typeof schemaKeyRef == "string") {
                v = this.getSchema(schemaKeyRef);
                if (!v) throw new Error(`no schema with key or ref "${schemaKeyRef}"`);
            } else {
                v = this.compile(schemaKeyRef);
            }
            const valid = v(data73);
            if (!("$async" in v)) this.errors = v.errors;
            return valid;
        }
        compile(schema, _meta) {
            const sch = this._addSchema(schema, _meta);
            return sch.validate || this._compileSchemaEnv(sch);
        }
        compileAsync(schema, meta) {
            if (typeof this.opts.loadSchema != "function") {
                throw new Error("options.loadSchema should be a function");
            }
            const { loadSchema  } = this.opts;
            return runCompileAsync.call(this, schema, meta);
            async function runCompileAsync(_schema, _meta) {
                await loadMetaSchema.call(this, _schema.$schema);
                const sch = this._addSchema(_schema, _meta);
                return sch.validate || _compileAsync.call(this, sch);
            }
            async function loadMetaSchema($ref) {
                if ($ref && !this.getSchema($ref)) {
                    await runCompileAsync.call(this, {
                        $ref
                    }, true);
                }
            }
            async function _compileAsync(sch) {
                try {
                    return this._compileSchemaEnv(sch);
                } catch (e) {
                    if (!(e instanceof ref_error.default)) throw e;
                    checkLoaded.call(this, e);
                    await loadMissingSchema.call(this, e.missingSchema);
                    return _compileAsync.call(this, sch);
                }
            }
            function checkLoaded({ missingSchema: ref2 , missingRef  }) {
                if (this.refs[ref2]) {
                    throw new Error(`AnySchema ${ref2} is loaded but ${missingRef} cannot be resolved`);
                }
            }
            async function loadMissingSchema(ref2) {
                const _schema = await _loadSchema.call(this, ref2);
                if (!this.refs[ref2]) await loadMetaSchema.call(this, _schema.$schema);
                if (!this.refs[ref2]) this.addSchema(_schema, ref2, meta);
            }
            async function _loadSchema(ref2) {
                const p = this._loading[ref2];
                if (p) return p;
                try {
                    return await (this._loading[ref2] = loadSchema(ref2));
                } finally{
                    delete this._loading[ref2];
                }
            }
        }
        addSchema(schema, key, _meta, _validateSchema = this.opts.validateSchema) {
            if (Array.isArray(schema)) {
                for (const sch of schema)this.addSchema(sch, void 0, _meta, _validateSchema);
                return this;
            }
            let id2;
            if (typeof schema === "object") {
                const { schemaId  } = this.opts;
                id2 = schema[schemaId];
                if (id2 !== void 0 && typeof id2 != "string") {
                    throw new Error(`schema ${schemaId} must be string`);
                }
            }
            key = (0, resolve3.normalizeId)(key || id2);
            this._checkUnique(key);
            this.schemas[key] = this._addSchema(schema, _meta, key, _validateSchema, true);
            return this;
        }
        addMetaSchema(schema, key, _validateSchema = this.opts.validateSchema) {
            this.addSchema(schema, key, true, _validateSchema);
            return this;
        }
        validateSchema(schema, throwOrLogError) {
            if (typeof schema == "boolean") return true;
            let $schema2;
            $schema2 = schema.$schema;
            if ($schema2 !== void 0 && typeof $schema2 != "string") {
                throw new Error("$schema must be a string");
            }
            $schema2 = $schema2 || this.opts.defaultMeta || this.defaultMeta();
            if (!$schema2) {
                this.logger.warn("meta-schema not available");
                this.errors = null;
                return true;
            }
            const valid = this.validate($schema2, schema);
            if (!valid && throwOrLogError) {
                const message = "schema is invalid: " + this.errorsText();
                if (this.opts.validateSchema === "log") this.logger.error(message);
                else throw new Error(message);
            }
            return valid;
        }
        getSchema(keyRef) {
            let sch;
            while(typeof (sch = getSchEnv.call(this, keyRef)) == "string")keyRef = sch;
            if (sch === void 0) {
                const { schemaId  } = this.opts;
                const root10 = new compile.SchemaEnv({
                    schema: {
                    },
                    schemaId
                });
                sch = compile.resolveSchema.call(this, root10, keyRef);
                if (!sch) return;
                this.refs[keyRef] = sch;
            }
            return sch.validate || this._compileSchemaEnv(sch);
        }
        removeSchema(schemaKeyRef) {
            if (schemaKeyRef instanceof RegExp) {
                this._removeAllSchemas(this.schemas, schemaKeyRef);
                this._removeAllSchemas(this.refs, schemaKeyRef);
                return this;
            }
            switch(typeof schemaKeyRef){
                case "undefined":
                    this._removeAllSchemas(this.schemas);
                    this._removeAllSchemas(this.refs);
                    this._cache.clear();
                    return this;
                case "string":
                    {
                        const sch = getSchEnv.call(this, schemaKeyRef);
                        if (typeof sch == "object") this._cache.delete(sch.schema);
                        delete this.schemas[schemaKeyRef];
                        delete this.refs[schemaKeyRef];
                        return this;
                    }
                case "object":
                    {
                        const cacheKey = schemaKeyRef;
                        this._cache.delete(cacheKey);
                        let id2 = schemaKeyRef[this.opts.schemaId];
                        if (id2) {
                            id2 = (0, resolve3.normalizeId)(id2);
                            delete this.schemas[id2];
                            delete this.refs[id2];
                        }
                        return this;
                    }
                default:
                    throw new Error("ajv.removeSchema: invalid parameter");
            }
        }
        addVocabulary(definitions2) {
            for (const def10 of definitions2)this.addKeyword(def10);
            return this;
        }
        addKeyword(kwdOrDef, def11) {
            let keyword2;
            if (typeof kwdOrDef == "string") {
                keyword2 = kwdOrDef;
                if (typeof def11 == "object") {
                    this.logger.warn("these parameters are deprecated, see docs for addKeyword");
                    def11.keyword = keyword2;
                }
            } else if (typeof kwdOrDef == "object" && def11 === void 0) {
                def11 = kwdOrDef;
                keyword2 = def11.keyword;
                if (Array.isArray(keyword2) && !keyword2.length) {
                    throw new Error("addKeywords: keyword must be string or non-empty array");
                }
            } else {
                throw new Error("invalid addKeywords parameters");
            }
            checkKeyword.call(this, keyword2, def11);
            if (!def11) {
                (0, util1.eachItem)(keyword2, (kwd)=>addRule.call(this, kwd)
                );
                return this;
            }
            keywordMetaschema.call(this, def11);
            const definition = {
                ...def11,
                type: (0, dataType.getJSONTypes)(def11.type),
                schemaType: (0, dataType.getJSONTypes)(def11.schemaType)
            };
            (0, util1.eachItem)(keyword2, definition.type.length === 0 ? (k)=>addRule.call(this, k, definition)
             : (k)=>definition.type.forEach((t)=>addRule.call(this, k, definition, t)
                )
            );
            return this;
        }
        getKeyword(keyword2) {
            const rule = this.RULES.all[keyword2];
            return typeof rule == "object" ? rule.definition : !!rule;
        }
        removeKeyword(keyword2) {
            const { RULES  } = this;
            delete RULES.keywords[keyword2];
            delete RULES.all[keyword2];
            for (const group of RULES.rules){
                const i47 = group.rules.findIndex((rule)=>rule.keyword === keyword2
                );
                if (i47 >= 0) group.rules.splice(i47, 1);
            }
            return this;
        }
        addFormat(name, format21) {
            if (typeof format21 == "string") format21 = new RegExp(format21);
            this.formats[name] = format21;
            return this;
        }
        errorsText(errors2 = this.errors, { separator =", " , dataVar ="data"  } = {
        }) {
            if (!errors2 || errors2.length === 0) return "No errors";
            return errors2.map((e)=>`${dataVar}${e.instancePath} ${e.message}`
            ).reduce((text, msg)=>text + separator + msg
            );
        }
        $dataMetaSchema(metaSchema, keywordsJsonPointers) {
            const rules2 = this.RULES.all;
            metaSchema = JSON.parse(JSON.stringify(metaSchema));
            for (const jsonPointer of keywordsJsonPointers){
                const segments = jsonPointer.split("/").slice(1);
                let keywords1 = metaSchema;
                for (const seg of segments)keywords1 = keywords1[seg];
                for(const key in rules2){
                    const rule = rules2[key];
                    if (typeof rule != "object") continue;
                    const { $data  } = rule.definition;
                    const schema = keywords1[key];
                    if ($data && schema) keywords1[key] = schemaOrData(schema);
                }
            }
            return metaSchema;
        }
        _removeAllSchemas(schemas, regex) {
            for(const keyRef in schemas){
                const sch = schemas[keyRef];
                if (!regex || regex.test(keyRef)) {
                    if (typeof sch == "string") {
                        delete schemas[keyRef];
                    } else if (sch && !sch.meta) {
                        this._cache.delete(sch.schema);
                        delete schemas[keyRef];
                    }
                }
            }
        }
        _addSchema(schema, meta, baseId, validateSchema = this.opts.validateSchema, addSchema = this.opts.addUsedSchema) {
            let id2;
            const { schemaId  } = this.opts;
            if (typeof schema == "object") {
                id2 = schema[schemaId];
            } else {
                if (this.opts.jtd) throw new Error("schema must be object");
                else if (typeof schema != "boolean") throw new Error("schema must be object or boolean");
            }
            let sch = this._cache.get(schema);
            if (sch !== void 0) return sch;
            baseId = (0, resolve3.normalizeId)(id2 || baseId);
            const localRefs = resolve3.getSchemaRefs.call(this, schema, baseId);
            sch = new compile.SchemaEnv({
                schema,
                schemaId,
                meta,
                baseId,
                localRefs
            });
            this._cache.set(sch.schema, sch);
            if (addSchema && !baseId.startsWith("#")) {
                if (baseId) this._checkUnique(baseId);
                this.refs[baseId] = sch;
            }
            if (validateSchema) this.validateSchema(schema, true);
            return sch;
        }
        _checkUnique(id2) {
            if (this.schemas[id2] || this.refs[id2]) {
                throw new Error(`schema with key or id "${id2}" already exists`);
            }
        }
        _compileSchemaEnv(sch) {
            if (sch.meta) this._compileMetaSchema(sch);
            else compile.compileSchema.call(this, sch);
            if (!sch.validate) throw new Error("ajv implementation error");
            return sch.validate;
        }
        _compileMetaSchema(sch) {
            const currentOpts = this.opts;
            this.opts = this._metaOpts;
            try {
                compile.compileSchema.call(this, sch);
            } finally{
                this.opts = currentOpts;
            }
        }
    }
    exports.default = Ajv;
    Ajv.ValidationError = validation_error.default;
    Ajv.MissingRefError = ref_error.default;
    function checkOptions(checkOpts, options, msg, log = "error") {
        for(const key in checkOpts){
            const opt = key;
            if (opt in options) this.logger[log](`${msg}: option ${key}. ${checkOpts[opt]}`);
        }
    }
    function getSchEnv(keyRef) {
        keyRef = (0, resolve3.normalizeId)(keyRef);
        return this.schemas[keyRef] || this.refs[keyRef];
    }
    function addInitialSchemas() {
        const optsSchemas = this.opts.schemas;
        if (!optsSchemas) return;
        if (Array.isArray(optsSchemas)) this.addSchema(optsSchemas);
        else for(const key in optsSchemas)this.addSchema(optsSchemas[key], key);
    }
    function addInitialFormats() {
        for(const name in this.opts.formats){
            const format22 = this.opts.formats[name];
            if (format22) this.addFormat(name, format22);
        }
    }
    function addInitialKeywords(defs) {
        if (Array.isArray(defs)) {
            this.addVocabulary(defs);
            return;
        }
        this.logger.warn("keywords option as map is deprecated, pass array");
        for(const keyword2 in defs){
            const def12 = defs[keyword2];
            if (!def12.keyword) def12.keyword = keyword2;
            this.addKeyword(def12);
        }
    }
    function getMetaSchemaOptions() {
        const metaOpts = {
            ...this.opts
        };
        for (const opt of META_IGNORE_OPTIONS)delete metaOpts[opt];
        return metaOpts;
    }
    const noLogs = {
        log () {
        },
        warn () {
        },
        error () {
        }
    };
    function getLogger1(logger) {
        if (logger === false) return noLogs;
        if (logger === void 0) return console;
        if (logger.log && logger.warn && logger.error) return logger;
        throw new Error("logger must implement log, warn and error methods");
    }
    const KEYWORD_NAME = /^[a-z_$][a-z0-9_$:-]*$/i;
    function checkKeyword(keyword2, def13) {
        const { RULES  } = this;
        (0, util1.eachItem)(keyword2, (kwd)=>{
            if (RULES.keywords[kwd]) throw new Error(`Keyword ${kwd} is already defined`);
            if (!KEYWORD_NAME.test(kwd)) throw new Error(`Keyword ${kwd} has invalid name`);
        });
        if (!def13) return;
        if (def13.$data && !("code" in def13 || "validate" in def13)) {
            throw new Error('$data keyword must have "code" or "validate" function');
        }
    }
    function addRule(keyword2, definition, dataType$1) {
        var _a;
        const post = definition === null || definition === void 0 ? void 0 : definition.post;
        if (dataType$1 && post) throw new Error('keyword with "post" flag cannot have "type"');
        const { RULES  } = this;
        let ruleGroup = post ? RULES.post : RULES.rules.find(({ type: t  })=>t === dataType$1
        );
        if (!ruleGroup) {
            ruleGroup = {
                type: dataType$1,
                rules: []
            };
            RULES.rules.push(ruleGroup);
        }
        RULES.keywords[keyword2] = true;
        if (!definition) return;
        const rule = {
            keyword: keyword2,
            definition: {
                ...definition,
                type: (0, dataType.getJSONTypes)(definition.type),
                schemaType: (0, dataType.getJSONTypes)(definition.schemaType)
            }
        };
        if (definition.before) addBeforeRule.call(this, ruleGroup, rule, definition.before);
        else ruleGroup.rules.push(rule);
        RULES.all[keyword2] = rule;
        (_a = definition.implements) === null || _a === void 0 ? void 0 : _a.forEach((kwd)=>this.addKeyword(kwd)
        );
    }
    function addBeforeRule(ruleGroup, rule, before1) {
        const i48 = ruleGroup.rules.findIndex((_rule)=>_rule.keyword === before1
        );
        if (i48 >= 0) {
            ruleGroup.rules.splice(i48, 0, rule);
        } else {
            ruleGroup.rules.push(rule);
            this.logger.warn(`rule ${before1} is not defined`);
        }
    }
    function keywordMetaschema(def14) {
        let { metaSchema  } = def14;
        if (metaSchema === void 0) return;
        if (def14.$data && this.opts.$data) metaSchema = schemaOrData(metaSchema);
        def14.validateSchema = this.compile(metaSchema, true);
    }
    const $dataRef = {
        $ref: "https://raw.githubusercontent.com/ajv-validator/ajv/master/lib/refs/data.json#"
    };
    function schemaOrData(schema) {
        return {
            anyOf: [
                schema,
                $dataRef
            ]
        };
    }
});
var id = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def15 = {
        keyword: "id",
        code () {
            throw new Error('NOT SUPPORTED: keyword "id", use "$id" for schema ID');
        }
    };
    exports.default = def15;
});
var ref = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.callRef = exports.getValidate = void 0;
    const def16 = {
        keyword: "$ref",
        schemaType: "string",
        code (cxt) {
            const { gen , schema: $ref , it  } = cxt;
            const { baseId , schemaEnv: env , validateName , opts , self  } = it;
            const { root: root11  } = env;
            if (($ref === "#" || $ref === "#/") && baseId === root11.baseId) return callRootRef();
            const schOrEnv = compile.resolveRef.call(self, root11, baseId, $ref);
            if (schOrEnv === void 0) throw new ref_error.default(baseId, $ref);
            if (schOrEnv instanceof compile.SchemaEnv) return callValidate(schOrEnv);
            return inlineRefSchema(schOrEnv);
            function callRootRef() {
                if (env === root11) return callRef(cxt, validateName, env, env.$async);
                const rootName = gen.scopeValue("root", {
                    ref: root11
                });
                return callRef(cxt, codegen._`${rootName}.validate`, root11, root11.$async);
            }
            function callValidate(sch) {
                const v = getValidate(cxt, sch);
                callRef(cxt, v, sch, sch.$async);
            }
            function inlineRefSchema(sch) {
                const schName = gen.scopeValue("schema", opts.code.source === true ? {
                    ref: sch,
                    code: (0, codegen.stringify)(sch)
                } : {
                    ref: sch
                });
                const valid = gen.name("valid");
                const schCxt = cxt.subschema({
                    schema: sch,
                    dataTypes: [],
                    schemaPath: codegen.nil,
                    topSchemaRef: schName,
                    errSchemaPath: $ref
                }, valid);
                cxt.mergeEvaluated(schCxt);
                cxt.ok(valid);
            }
        }
    };
    function getValidate(cxt, sch) {
        const { gen  } = cxt;
        return sch.validate ? gen.scopeValue("validate", {
            ref: sch.validate
        }) : codegen._`${gen.scopeValue("wrapper", {
            ref: sch
        })}.validate`;
    }
    exports.getValidate = getValidate;
    function callRef(cxt, v, sch, $async) {
        const { gen , it  } = cxt;
        const { allErrors , schemaEnv: env , opts  } = it;
        const passCxt = opts.passContext ? names_1.default.this : codegen.nil;
        if ($async) callAsyncRef();
        else callSyncRef();
        function callAsyncRef() {
            if (!env.$async) throw new Error("async schema referenced by sync schema");
            const valid = gen.let("valid");
            gen.try(()=>{
                gen.code(codegen._`await ${(0, code$1.callValidateCode)(cxt, v, passCxt)}`);
                addEvaluatedFrom(v);
                if (!allErrors) gen.assign(valid, true);
            }, (e)=>{
                gen.if(codegen._`!(${e} instanceof ${it.ValidationError})`, ()=>gen.throw(e)
                );
                addErrorsFrom(e);
                if (!allErrors) gen.assign(valid, false);
            });
            cxt.ok(valid);
        }
        function callSyncRef() {
            cxt.result((0, code$1.callValidateCode)(cxt, v, passCxt), ()=>addEvaluatedFrom(v)
            , ()=>addErrorsFrom(v)
            );
        }
        function addErrorsFrom(source) {
            const errs = codegen._`${source}.errors`;
            gen.assign(names_1.default.vErrors, codegen._`${names_1.default.vErrors} === null ? ${errs} : ${names_1.default.vErrors}.concat(${errs})`);
            gen.assign(names_1.default.errors, codegen._`${names_1.default.vErrors}.length`);
        }
        function addEvaluatedFrom(source) {
            var _a;
            if (!it.opts.unevaluated) return;
            const schEvaluated = (_a = sch === null || sch === void 0 ? void 0 : sch.validate) === null || _a === void 0 ? void 0 : _a.evaluated;
            if (it.props !== true) {
                if (schEvaluated && !schEvaluated.dynamicProps) {
                    if (schEvaluated.props !== void 0) {
                        it.props = util1.mergeEvaluated.props(gen, schEvaluated.props, it.props);
                    }
                } else {
                    const props = gen.var("props", codegen._`${source}.evaluated.props`);
                    it.props = util1.mergeEvaluated.props(gen, props, it.props, codegen.Name);
                }
            }
            if (it.items !== true) {
                if (schEvaluated && !schEvaluated.dynamicItems) {
                    if (schEvaluated.items !== void 0) {
                        it.items = util1.mergeEvaluated.items(gen, schEvaluated.items, it.items);
                    }
                } else {
                    const items2 = gen.var("items", codegen._`${source}.evaluated.items`);
                    it.items = util1.mergeEvaluated.items(gen, items2, it.items, codegen.Name);
                }
            }
        }
    }
    exports.callRef = callRef;
    exports.default = def16;
});
var core_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const core2 = [
        "$schema",
        "$id",
        "$defs",
        "$vocabulary",
        {
            keyword: "$comment"
        },
        "definitions",
        id.default,
        ref.default
    ];
    exports.default = core2;
});
var limitNumber = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const ops = codegen.operators;
    const KWDs = {
        maximum: {
            okStr: "<=",
            ok: ops.LTE,
            fail: ops.GT
        },
        minimum: {
            okStr: ">=",
            ok: ops.GTE,
            fail: ops.LT
        },
        exclusiveMaximum: {
            okStr: "<",
            ok: ops.LT,
            fail: ops.GTE
        },
        exclusiveMinimum: {
            okStr: ">",
            ok: ops.GT,
            fail: ops.LTE
        }
    };
    const error24 = {
        message: ({ keyword: keyword2 , schemaCode  })=>codegen.str`must be ${KWDs[keyword2].okStr} ${schemaCode}`
        ,
        params: ({ keyword: keyword2 , schemaCode  })=>codegen._`{comparison: ${KWDs[keyword2].okStr}, limit: ${schemaCode}}`
    };
    const def17 = {
        keyword: Object.keys(KWDs),
        type: "number",
        schemaType: "number",
        $data: true,
        error: error24,
        code (cxt) {
            const { keyword: keyword2 , data: data74 , schemaCode  } = cxt;
            cxt.fail$data(codegen._`${data74} ${KWDs[keyword2].fail} ${schemaCode} || isNaN(${data74})`);
        }
    };
    exports.default = def17;
});
var multipleOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error25 = {
        message: ({ schemaCode  })=>codegen.str`must be multiple of ${schemaCode}`
        ,
        params: ({ schemaCode  })=>codegen._`{multipleOf: ${schemaCode}}`
    };
    const def18 = {
        keyword: "multipleOf",
        type: "number",
        schemaType: "number",
        $data: true,
        error: error25,
        code (cxt) {
            const { gen , data: data75 , schemaCode , it  } = cxt;
            const prec = it.opts.multipleOfPrecision;
            const res = gen.let("res");
            const invalid = prec ? codegen._`Math.abs(Math.round(${res}) - ${res}) > 1e-${prec}` : codegen._`${res} !== parseInt(${res})`;
            cxt.fail$data(codegen._`(${schemaCode} === 0 || (${res} = ${data75}/${schemaCode}, ${invalid}))`);
        }
    };
    exports.default = def18;
});
var ucs2length_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    function ucs2length(str2) {
        const len4 = str2.length;
        let length = 0;
        let pos = 0;
        let value;
        while(pos < len4){
            length++;
            value = str2.charCodeAt(pos++);
            if (value >= 55296 && value <= 56319 && pos < len4) {
                value = str2.charCodeAt(pos);
                if ((value & 64512) === 56320) pos++;
            }
        }
        return length;
    }
    exports.default = ucs2length;
    ucs2length.code = 'require("ajv/dist/runtime/ucs2length").default';
});
var limitLength = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error26 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxLength" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} characters`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def19 = {
        keyword: [
            "maxLength",
            "minLength"
        ],
        type: "string",
        schemaType: "number",
        $data: true,
        error: error26,
        code (cxt) {
            const { keyword: keyword2 , data: data76 , schemaCode , it  } = cxt;
            const op = keyword2 === "maxLength" ? codegen.operators.GT : codegen.operators.LT;
            const len5 = it.opts.unicode === false ? codegen._`${data76}.length` : codegen._`${(0, util1.useFunc)(cxt.gen, ucs2length_1.default)}(${data76})`;
            cxt.fail$data(codegen._`${len5} ${op} ${schemaCode}`);
        }
    };
    exports.default = def19;
});
var pattern = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error27 = {
        message: ({ schemaCode  })=>codegen.str`must match pattern "${schemaCode}"`
        ,
        params: ({ schemaCode  })=>codegen._`{pattern: ${schemaCode}}`
    };
    const def20 = {
        keyword: "pattern",
        type: "string",
        schemaType: "string",
        $data: true,
        error: error27,
        code (cxt) {
            const { data: data77 , $data , schema , schemaCode , it  } = cxt;
            const u = it.opts.unicodeRegExp ? "u" : "";
            const regExp = $data ? codegen._`(new RegExp(${schemaCode}, ${u}))` : (0, code$1.usePattern)(cxt, schema);
            cxt.fail$data(codegen._`!${regExp}.test(${data77})`);
        }
    };
    exports.default = def20;
});
var limitProperties = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error28 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxProperties" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} items`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def21 = {
        keyword: [
            "maxProperties",
            "minProperties"
        ],
        type: "object",
        schemaType: "number",
        $data: true,
        error: error28,
        code (cxt) {
            const { keyword: keyword2 , data: data78 , schemaCode  } = cxt;
            const op = keyword2 === "maxProperties" ? codegen.operators.GT : codegen.operators.LT;
            cxt.fail$data(codegen._`Object.keys(${data78}).length ${op} ${schemaCode}`);
        }
    };
    exports.default = def21;
});
var required$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error29 = {
        message: ({ params: { missingProperty  }  })=>codegen.str`must have required property '${missingProperty}'`
        ,
        params: ({ params: { missingProperty  }  })=>codegen._`{missingProperty: ${missingProperty}}`
    };
    const def22 = {
        keyword: "required",
        type: "object",
        schemaType: "array",
        $data: true,
        error: error29,
        code (cxt) {
            const { gen , schema , schemaCode , data: data79 , $data , it  } = cxt;
            const { opts  } = it;
            if (!$data && schema.length === 0) return;
            const useLoop = schema.length >= opts.loopRequired;
            if (it.allErrors) allErrorsMode();
            else exitOnErrorMode();
            if (opts.strictRequired) {
                const props = cxt.parentSchema.properties;
                const { definedProperties  } = cxt.it;
                for (const requiredKey of schema){
                    if ((props === null || props === void 0 ? void 0 : props[requiredKey]) === void 0 && !definedProperties.has(requiredKey)) {
                        const schemaPath = it.schemaEnv.baseId + it.errSchemaPath;
                        const msg = `required property "${requiredKey}" is not defined at "${schemaPath}" (strictRequired)`;
                        (0, util1.checkStrictMode)(it, msg, it.opts.strictRequired);
                    }
                }
            }
            function allErrorsMode() {
                if (useLoop || $data) {
                    cxt.block$data(codegen.nil, loopAllRequired);
                } else {
                    for (const prop of schema){
                        (0, code$1.checkReportMissingProp)(cxt, prop);
                    }
                }
            }
            function exitOnErrorMode() {
                const missing = gen.let("missing");
                if (useLoop || $data) {
                    const valid = gen.let("valid", true);
                    cxt.block$data(valid, ()=>loopUntilMissing(missing, valid)
                    );
                    cxt.ok(valid);
                } else {
                    gen.if((0, code$1.checkMissingProp)(cxt, schema, missing));
                    (0, code$1.reportMissingProp)(cxt, missing);
                    gen.else();
                }
            }
            function loopAllRequired() {
                gen.forOf("prop", schemaCode, (prop)=>{
                    cxt.setParams({
                        missingProperty: prop
                    });
                    gen.if((0, code$1.noPropertyInData)(gen, data79, prop, opts.ownProperties), ()=>cxt.error()
                    );
                });
            }
            function loopUntilMissing(missing, valid) {
                cxt.setParams({
                    missingProperty: missing
                });
                gen.forOf(missing, schemaCode, ()=>{
                    gen.assign(valid, (0, code$1.propertyInData)(gen, data79, missing, opts.ownProperties));
                    gen.if((0, codegen.not)(valid), ()=>{
                        cxt.error();
                        gen.break();
                    });
                }, codegen.nil);
            }
        }
    };
    exports.default = def22;
});
var limitItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error30 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxItems" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} items`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def23 = {
        keyword: [
            "maxItems",
            "minItems"
        ],
        type: "array",
        schemaType: "number",
        $data: true,
        error: error30,
        code (cxt) {
            const { keyword: keyword2 , data: data80 , schemaCode  } = cxt;
            const op = keyword2 === "maxItems" ? codegen.operators.GT : codegen.operators.LT;
            cxt.fail$data(codegen._`${data80}.length ${op} ${schemaCode}`);
        }
    };
    exports.default = def23;
});
var equal_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    fastDeepEqual.code = 'require("ajv/dist/runtime/equal").default';
    exports.default = fastDeepEqual;
});
var uniqueItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error31 = {
        message: ({ params: { i: i49 , j  }  })=>codegen.str`must NOT have duplicate items (items ## ${j} and ${i49} are identical)`
        ,
        params: ({ params: { i: i50 , j  }  })=>codegen._`{i: ${i50}, j: ${j}}`
    };
    const def24 = {
        keyword: "uniqueItems",
        type: "array",
        schemaType: "boolean",
        $data: true,
        error: error31,
        code (cxt) {
            const { gen , data: data81 , $data , schema , parentSchema , schemaCode , it  } = cxt;
            if (!$data && !schema) return;
            const valid = gen.let("valid");
            const itemTypes = parentSchema.items ? (0, dataType.getSchemaTypes)(parentSchema.items) : [];
            cxt.block$data(valid, validateUniqueItems, codegen._`${schemaCode} === false`);
            cxt.ok(valid);
            function validateUniqueItems() {
                const i51 = gen.let("i", codegen._`${data81}.length`);
                const j = gen.let("j");
                cxt.setParams({
                    i: i51,
                    j
                });
                gen.assign(valid, true);
                gen.if(codegen._`${i51} > 1`, ()=>(canOptimize() ? loopN : loopN2)(i51, j)
                );
            }
            function canOptimize() {
                return itemTypes.length > 0 && !itemTypes.some((t)=>t === "object" || t === "array"
                );
            }
            function loopN(i52, j) {
                const item = gen.name("item");
                const wrongType = (0, dataType.checkDataTypes)(itemTypes, item, it.opts.strictNumbers, dataType.DataType.Wrong);
                const indices = gen.const("indices", codegen._`{}`);
                gen.for(codegen._`;${i52}--;`, ()=>{
                    gen.let(item, codegen._`${data81}[${i52}]`);
                    gen.if(wrongType, codegen._`continue`);
                    if (itemTypes.length > 1) gen.if(codegen._`typeof ${item} == "string"`, codegen._`${item} += "_"`);
                    gen.if(codegen._`typeof ${indices}[${item}] == "number"`, ()=>{
                        gen.assign(j, codegen._`${indices}[${item}]`);
                        cxt.error();
                        gen.assign(valid, false).break();
                    }).code(codegen._`${indices}[${item}] = ${i52}`);
                });
            }
            function loopN2(i53, j) {
                const eql = (0, util1.useFunc)(gen, equal_1.default);
                const outer = gen.name("outer");
                gen.label(outer).for(codegen._`;${i53}--;`, ()=>gen.for(codegen._`${j} = ${i53}; ${j}--;`, ()=>gen.if(codegen._`${eql}(${data81}[${i53}], ${data81}[${j}])`, ()=>{
                            cxt.error();
                            gen.assign(valid, false).break(outer);
                        })
                    )
                );
            }
        }
    };
    exports.default = def24;
});
var _const = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error32 = {
        message: "must be equal to constant",
        params: ({ schemaCode  })=>codegen._`{allowedValue: ${schemaCode}}`
    };
    const def25 = {
        keyword: "const",
        $data: true,
        error: error32,
        code (cxt) {
            const { gen , data: data82 , $data , schemaCode , schema  } = cxt;
            if ($data || schema && typeof schema == "object") {
                cxt.fail$data(codegen._`!${(0, util1.useFunc)(gen, equal_1.default)}(${data82}, ${schemaCode})`);
            } else {
                cxt.fail(codegen._`${schema} !== ${data82}`);
            }
        }
    };
    exports.default = def25;
});
var _enum = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error33 = {
        message: "must be equal to one of the allowed values",
        params: ({ schemaCode  })=>codegen._`{allowedValues: ${schemaCode}}`
    };
    const def26 = {
        keyword: "enum",
        schemaType: "array",
        $data: true,
        error: error33,
        code (cxt) {
            const { gen , data: data83 , $data , schema , schemaCode , it  } = cxt;
            if (!$data && schema.length === 0) throw new Error("enum must have non-empty array");
            const useLoop = schema.length >= it.opts.loopEnum;
            const eql = (0, util1.useFunc)(gen, equal_1.default);
            let valid;
            if (useLoop || $data) {
                valid = gen.let("valid");
                cxt.block$data(valid, loopEnum);
            } else {
                if (!Array.isArray(schema)) throw new Error("ajv implementation error");
                const vSchema = gen.const("vSchema", schemaCode);
                valid = (0, codegen.or)(...schema.map((_x, i54)=>equalCode(vSchema, i54)
                ));
            }
            cxt.pass(valid);
            function loopEnum() {
                gen.assign(valid, false);
                gen.forOf("v", schemaCode, (v)=>gen.if(codegen._`${eql}(${data83}, ${v})`, ()=>gen.assign(valid, true).break()
                    )
                );
            }
            function equalCode(vSchema, i55) {
                const sch = schema[i55];
                return typeof sch === "object" && sch !== null ? codegen._`${eql}(${data83}, ${vSchema}[${i55}])` : codegen._`${data83} === ${sch}`;
            }
        }
    };
    exports.default = def26;
});
var validation_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const validation = [
        limitNumber.default,
        multipleOf.default,
        limitLength.default,
        pattern.default,
        limitProperties.default,
        required$1.default,
        limitItems.default,
        uniqueItems.default,
        {
            keyword: "type",
            schemaType: [
                "string",
                "array"
            ]
        },
        {
            keyword: "nullable",
            schemaType: "boolean"
        },
        _const.default,
        _enum.default
    ];
    exports.default = validation;
});
var additionalItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateAdditionalItems = void 0;
    const error34 = {
        message: ({ params: { len: len6  }  })=>codegen.str`must NOT have more than ${len6} items`
        ,
        params: ({ params: { len: len7  }  })=>codegen._`{limit: ${len7}}`
    };
    const def27 = {
        keyword: "additionalItems",
        type: "array",
        schemaType: [
            "boolean",
            "object"
        ],
        before: "uniqueItems",
        error: error34,
        code (cxt) {
            const { parentSchema , it  } = cxt;
            const { items: items2  } = parentSchema;
            if (!Array.isArray(items2)) {
                (0, util1.checkStrictMode)(it, '"additionalItems" is ignored when "items" is not an array of schemas');
                return;
            }
            validateAdditionalItems(cxt, items2);
        }
    };
    function validateAdditionalItems(cxt, items2) {
        const { gen , schema , data: data84 , keyword: keyword2 , it  } = cxt;
        it.items = true;
        const len8 = gen.const("len", codegen._`${data84}.length`);
        if (schema === false) {
            cxt.setParams({
                len: items2.length
            });
            cxt.pass(codegen._`${len8} <= ${items2.length}`);
        } else if (typeof schema == "object" && !(0, util1.alwaysValidSchema)(it, schema)) {
            const valid = gen.var("valid", codegen._`${len8} <= ${items2.length}`);
            gen.if((0, codegen.not)(valid), ()=>validateItems(valid)
            );
            cxt.ok(valid);
        }
        function validateItems(valid) {
            gen.forRange("i", items2.length, len8, (i56)=>{
                cxt.subschema({
                    keyword: keyword2,
                    dataProp: i56,
                    dataPropType: util1.Type.Num
                }, valid);
                if (!it.allErrors) gen.if((0, codegen.not)(valid), ()=>gen.break()
                );
            });
        }
    }
    exports.validateAdditionalItems = validateAdditionalItems;
    exports.default = def27;
});
var items = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateTuple = void 0;
    const def28 = {
        keyword: "items",
        type: "array",
        schemaType: [
            "object",
            "array",
            "boolean"
        ],
        before: "uniqueItems",
        code (cxt) {
            const { schema , it  } = cxt;
            if (Array.isArray(schema)) return validateTuple(cxt, "additionalItems", schema);
            it.items = true;
            if ((0, util1.alwaysValidSchema)(it, schema)) return;
            cxt.ok((0, code$1.validateArray)(cxt));
        }
    };
    function validateTuple(cxt, extraItems, schArr = cxt.schema) {
        const { gen , parentSchema , data: data85 , keyword: keyword2 , it  } = cxt;
        checkStrictTuple(parentSchema);
        if (it.opts.unevaluated && schArr.length && it.items !== true) {
            it.items = util1.mergeEvaluated.items(gen, schArr.length, it.items);
        }
        const valid = gen.name("valid");
        const len9 = gen.const("len", codegen._`${data85}.length`);
        schArr.forEach((sch, i57)=>{
            if ((0, util1.alwaysValidSchema)(it, sch)) return;
            gen.if(codegen._`${len9} > ${i57}`, ()=>cxt.subschema({
                    keyword: keyword2,
                    schemaProp: i57,
                    dataProp: i57
                }, valid)
            );
            cxt.ok(valid);
        });
        function checkStrictTuple(sch) {
            const { opts , errSchemaPath  } = it;
            const l = schArr.length;
            const fullTuple = l === sch.minItems && (l === sch.maxItems || sch[extraItems] === false);
            if (opts.strictTuples && !fullTuple) {
                const msg = `"${keyword2}" is ${l}-tuple, but minItems or maxItems/${extraItems} are not specified or different at path "${errSchemaPath}"`;
                (0, util1.checkStrictMode)(it, msg, opts.strictTuples);
            }
        }
    }
    exports.validateTuple = validateTuple;
    exports.default = def28;
});
var prefixItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def29 = {
        keyword: "prefixItems",
        type: "array",
        schemaType: [
            "array"
        ],
        before: "uniqueItems",
        code: (cxt)=>(0, items.validateTuple)(cxt, "items")
    };
    exports.default = def29;
});
var items2020 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error35 = {
        message: ({ params: { len: len10  }  })=>codegen.str`must NOT have more than ${len10} items`
        ,
        params: ({ params: { len: len11  }  })=>codegen._`{limit: ${len11}}`
    };
    const def30 = {
        keyword: "items",
        type: "array",
        schemaType: [
            "object",
            "boolean"
        ],
        before: "uniqueItems",
        error: error35,
        code (cxt) {
            const { schema , parentSchema , it  } = cxt;
            const { prefixItems: prefixItems2  } = parentSchema;
            it.items = true;
            if ((0, util1.alwaysValidSchema)(it, schema)) return;
            if (prefixItems2) (0, additionalItems.validateAdditionalItems)(cxt, prefixItems2);
            else cxt.ok((0, code$1.validateArray)(cxt));
        }
    };
    exports.default = def30;
});
var contains = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error36 = {
        message: ({ params: { min: min2 , max: max8  }  })=>max8 === void 0 ? codegen.str`must contain at least ${min2} valid item(s)` : codegen.str`must contain at least ${min2} and no more than ${max8} valid item(s)`
        ,
        params: ({ params: { min: min3 , max: max9  }  })=>max9 === void 0 ? codegen._`{minContains: ${min3}}` : codegen._`{minContains: ${min3}, maxContains: ${max9}}`
    };
    const def31 = {
        keyword: "contains",
        type: "array",
        schemaType: [
            "object",
            "boolean"
        ],
        before: "uniqueItems",
        trackErrors: true,
        error: error36,
        code (cxt) {
            const { gen , schema , parentSchema , data: data86 , it  } = cxt;
            let min4;
            let max10;
            const { minContains , maxContains  } = parentSchema;
            if (it.opts.next) {
                min4 = minContains === void 0 ? 1 : minContains;
                max10 = maxContains;
            } else {
                min4 = 1;
            }
            const len12 = gen.const("len", codegen._`${data86}.length`);
            cxt.setParams({
                min: min4,
                max: max10
            });
            if (max10 === void 0 && min4 === 0) {
                (0, util1.checkStrictMode)(it, `"minContains" == 0 without "maxContains": "contains" keyword ignored`);
                return;
            }
            if (max10 !== void 0 && min4 > max10) {
                (0, util1.checkStrictMode)(it, `"minContains" > "maxContains" is always invalid`);
                cxt.fail();
                return;
            }
            if ((0, util1.alwaysValidSchema)(it, schema)) {
                let cond8 = codegen._`${len12} >= ${min4}`;
                if (max10 !== void 0) cond8 = codegen._`${cond8} && ${len12} <= ${max10}`;
                cxt.pass(cond8);
                return;
            }
            it.items = true;
            const valid = gen.name("valid");
            if (max10 === void 0 && min4 === 1) {
                validateItems(valid, ()=>gen.if(valid, ()=>gen.break()
                    )
                );
            } else if (min4 === 0) {
                gen.let(valid, true);
                if (max10 !== void 0) gen.if(codegen._`${data86}.length > 0`, validateItemsWithCount);
            } else {
                gen.let(valid, false);
                validateItemsWithCount();
            }
            cxt.result(valid, ()=>cxt.reset()
            );
            function validateItemsWithCount() {
                const schValid = gen.name("_valid");
                const count = gen.let("count", 0);
                validateItems(schValid, ()=>gen.if(schValid, ()=>checkLimits(count)
                    )
                );
            }
            function validateItems(_valid, block) {
                gen.forRange("i", 0, len12, (i58)=>{
                    cxt.subschema({
                        keyword: "contains",
                        dataProp: i58,
                        dataPropType: util1.Type.Num,
                        compositeRule: true
                    }, _valid);
                    block();
                });
            }
            function checkLimits(count) {
                gen.code(codegen._`${count}++`);
                if (max10 === void 0) {
                    gen.if(codegen._`${count} >= ${min4}`, ()=>gen.assign(valid, true).break()
                    );
                } else {
                    gen.if(codegen._`${count} > ${max10}`, ()=>gen.assign(valid, false).break()
                    );
                    if (min4 === 1) gen.assign(valid, true);
                    else gen.if(codegen._`${count} >= ${min4}`, ()=>gen.assign(valid, true)
                    );
                }
            }
        }
    };
    exports.default = def31;
});
var dependencies = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateSchemaDeps = exports.validatePropertyDeps = exports.error = void 0;
    exports.error = {
        message: ({ params: { property: property5 , depsCount , deps  }  })=>{
            const property_ies = depsCount === 1 ? "property" : "properties";
            return codegen.str`must have ${property_ies} ${deps} when property ${property5} is present`;
        },
        params: ({ params: { property: property6 , depsCount , deps , missingProperty  }  })=>codegen._`{property: ${property6},
    missingProperty: ${missingProperty},
    depsCount: ${depsCount},
    deps: ${deps}}`
    };
    const def32 = {
        keyword: "dependencies",
        type: "object",
        schemaType: "object",
        error: exports.error,
        code (cxt) {
            const [propDeps, schDeps] = splitDependencies(cxt);
            validatePropertyDeps(cxt, propDeps);
            validateSchemaDeps(cxt, schDeps);
        }
    };
    function splitDependencies({ schema  }) {
        const propertyDeps = {
        };
        const schemaDeps = {
        };
        for(const key in schema){
            if (key === "__proto__") continue;
            const deps = Array.isArray(schema[key]) ? propertyDeps : schemaDeps;
            deps[key] = schema[key];
        }
        return [
            propertyDeps,
            schemaDeps
        ];
    }
    function validatePropertyDeps(cxt, propertyDeps = cxt.schema) {
        const { gen , data: data87 , it  } = cxt;
        if (Object.keys(propertyDeps).length === 0) return;
        const missing = gen.let("missing");
        for(const prop in propertyDeps){
            const deps = propertyDeps[prop];
            if (deps.length === 0) continue;
            const hasProperty = (0, code$1.propertyInData)(gen, data87, prop, it.opts.ownProperties);
            cxt.setParams({
                property: prop,
                depsCount: deps.length,
                deps: deps.join(", ")
            });
            if (it.allErrors) {
                gen.if(hasProperty, ()=>{
                    for (const depProp of deps){
                        (0, code$1.checkReportMissingProp)(cxt, depProp);
                    }
                });
            } else {
                gen.if(codegen._`${hasProperty} && (${(0, code$1.checkMissingProp)(cxt, deps, missing)})`);
                (0, code$1.reportMissingProp)(cxt, missing);
                gen.else();
            }
        }
    }
    exports.validatePropertyDeps = validatePropertyDeps;
    function validateSchemaDeps(cxt, schemaDeps = cxt.schema) {
        const { gen , data: data88 , keyword: keyword2 , it  } = cxt;
        const valid = gen.name("valid");
        for(const prop in schemaDeps){
            if ((0, util1.alwaysValidSchema)(it, schemaDeps[prop])) continue;
            gen.if((0, code$1.propertyInData)(gen, data88, prop, it.opts.ownProperties), ()=>{
                const schCxt = cxt.subschema({
                    keyword: keyword2,
                    schemaProp: prop
                }, valid);
                cxt.mergeValidEvaluated(schCxt, valid);
            }, ()=>gen.var(valid, true)
            );
            cxt.ok(valid);
        }
    }
    exports.validateSchemaDeps = validateSchemaDeps;
    exports.default = def32;
});
var propertyNames = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error37 = {
        message: "property name must be valid",
        params: ({ params  })=>codegen._`{propertyName: ${params.propertyName}}`
    };
    const def33 = {
        keyword: "propertyNames",
        type: "object",
        schemaType: [
            "object",
            "boolean"
        ],
        error: error37,
        code (cxt) {
            const { gen , schema , data: data89 , it  } = cxt;
            if ((0, util1.alwaysValidSchema)(it, schema)) return;
            const valid = gen.name("valid");
            gen.forIn("key", data89, (key)=>{
                cxt.setParams({
                    propertyName: key
                });
                cxt.subschema({
                    keyword: "propertyNames",
                    data: key,
                    dataTypes: [
                        "string"
                    ],
                    propertyName: key,
                    compositeRule: true
                }, valid);
                gen.if((0, codegen.not)(valid), ()=>{
                    cxt.error(true);
                    if (!it.allErrors) gen.break();
                });
            });
            cxt.ok(valid);
        }
    };
    exports.default = def33;
});
var additionalProperties$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error38 = {
        message: "must NOT have additional properties",
        params: ({ params  })=>codegen._`{additionalProperty: ${params.additionalProperty}}`
    };
    const def34 = {
        keyword: "additionalProperties",
        type: [
            "object"
        ],
        schemaType: [
            "boolean",
            "object"
        ],
        allowUndefined: true,
        trackErrors: true,
        error: error38,
        code (cxt) {
            const { gen , schema , parentSchema , data: data90 , errsCount , it  } = cxt;
            if (!errsCount) throw new Error("ajv implementation error");
            const { allErrors , opts  } = it;
            it.props = true;
            if (opts.removeAdditional !== "all" && (0, util1.alwaysValidSchema)(it, schema)) return;
            const props = (0, code$1.allSchemaProperties)(parentSchema.properties);
            const patProps = (0, code$1.allSchemaProperties)(parentSchema.patternProperties);
            checkAdditionalProperties();
            cxt.ok(codegen._`${errsCount} === ${names_1.default.errors}`);
            function checkAdditionalProperties() {
                gen.forIn("key", data90, (key)=>{
                    if (!props.length && !patProps.length) additionalPropertyCode(key);
                    else gen.if(isAdditional(key), ()=>additionalPropertyCode(key)
                    );
                });
            }
            function isAdditional(key) {
                let definedProp;
                if (props.length > 8) {
                    const propsSchema = (0, util1.schemaRefOrVal)(it, parentSchema.properties, "properties");
                    definedProp = (0, code$1.isOwnProperty)(gen, propsSchema, key);
                } else if (props.length) {
                    definedProp = (0, codegen.or)(...props.map((p)=>codegen._`${key} === ${p}`
                    ));
                } else {
                    definedProp = codegen.nil;
                }
                if (patProps.length) {
                    definedProp = (0, codegen.or)(definedProp, ...patProps.map((p)=>codegen._`${(0, code$1.usePattern)(cxt, p)}.test(${key})`
                    ));
                }
                return (0, codegen.not)(definedProp);
            }
            function deleteAdditional(key) {
                gen.code(codegen._`delete ${data90}[${key}]`);
            }
            function additionalPropertyCode(key) {
                if (opts.removeAdditional === "all" || opts.removeAdditional && schema === false) {
                    deleteAdditional(key);
                    return;
                }
                if (schema === false) {
                    cxt.setParams({
                        additionalProperty: key
                    });
                    cxt.error();
                    if (!allErrors) gen.break();
                    return;
                }
                if (typeof schema == "object" && !(0, util1.alwaysValidSchema)(it, schema)) {
                    const valid = gen.name("valid");
                    if (opts.removeAdditional === "failing") {
                        applyAdditionalSchema(key, valid, false);
                        gen.if((0, codegen.not)(valid), ()=>{
                            cxt.reset();
                            deleteAdditional(key);
                        });
                    } else {
                        applyAdditionalSchema(key, valid);
                        if (!allErrors) gen.if((0, codegen.not)(valid), ()=>gen.break()
                        );
                    }
                }
            }
            function applyAdditionalSchema(key, valid, errors2) {
                const subschema2 = {
                    keyword: "additionalProperties",
                    dataProp: key,
                    dataPropType: util1.Type.Str
                };
                if (errors2 === false) {
                    Object.assign(subschema2, {
                        compositeRule: true,
                        createErrors: false,
                        allErrors: false
                    });
                }
                cxt.subschema(subschema2, valid);
            }
        }
    };
    exports.default = def34;
});
var properties$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def35 = {
        keyword: "properties",
        type: "object",
        schemaType: "object",
        code (cxt) {
            const { gen , schema , parentSchema , data: data91 , it  } = cxt;
            if (it.opts.removeAdditional === "all" && parentSchema.additionalProperties === void 0) {
                additionalProperties$1.default.code(new validate1.KeywordCxt(it, additionalProperties$1.default, "additionalProperties"));
            }
            const allProps = (0, code$1.allSchemaProperties)(schema);
            for (const prop2 of allProps){
                it.definedProperties.add(prop2);
            }
            if (it.opts.unevaluated && allProps.length && it.props !== true) {
                it.props = util1.mergeEvaluated.props(gen, (0, util1.toHash)(allProps), it.props);
            }
            const properties2 = allProps.filter((p)=>!(0, util1.alwaysValidSchema)(it, schema[p])
            );
            if (properties2.length === 0) return;
            const valid = gen.name("valid");
            for (const prop1 of properties2){
                if (hasDefault(prop1)) {
                    applyPropertySchema(prop1);
                } else {
                    gen.if((0, code$1.propertyInData)(gen, data91, prop1, it.opts.ownProperties));
                    applyPropertySchema(prop1);
                    if (!it.allErrors) gen.else().var(valid, true);
                    gen.endIf();
                }
                cxt.it.definedProperties.add(prop1);
                cxt.ok(valid);
            }
            function hasDefault(prop) {
                return it.opts.useDefaults && !it.compositeRule && schema[prop].default !== void 0;
            }
            function applyPropertySchema(prop) {
                cxt.subschema({
                    keyword: "properties",
                    schemaProp: prop,
                    dataProp: prop
                }, valid);
            }
        }
    };
    exports.default = def35;
});
var patternProperties = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const util_2 = util1;
    const def36 = {
        keyword: "patternProperties",
        type: "object",
        schemaType: "object",
        code (cxt) {
            const { gen , schema , data: data92 , parentSchema , it  } = cxt;
            const { opts  } = it;
            const patterns = (0, code$1.allSchemaProperties)(schema);
            const alwaysValidPatterns = patterns.filter((p)=>(0, util1.alwaysValidSchema)(it, schema[p])
            );
            if (patterns.length === 0 || alwaysValidPatterns.length === patterns.length && (!it.opts.unevaluated || it.props === true)) {
                return;
            }
            const checkProperties = opts.strictSchema && !opts.allowMatchingProperties && parentSchema.properties;
            const valid = gen.name("valid");
            if (it.props !== true && !(it.props instanceof codegen.Name)) {
                it.props = (0, util_2.evaluatedPropsToName)(gen, it.props);
            }
            const { props  } = it;
            validatePatternProperties();
            function validatePatternProperties() {
                for (const pat of patterns){
                    if (checkProperties) checkMatchingProperties(pat);
                    if (it.allErrors) {
                        validateProperties(pat);
                    } else {
                        gen.var(valid, true);
                        validateProperties(pat);
                        gen.if(valid);
                    }
                }
            }
            function checkMatchingProperties(pat) {
                for(const prop in checkProperties){
                    if (new RegExp(pat).test(prop)) {
                        (0, util1.checkStrictMode)(it, `property ${prop} matches pattern ${pat} (use allowMatchingProperties)`);
                    }
                }
            }
            function validateProperties(pat) {
                gen.forIn("key", data92, (key)=>{
                    gen.if(codegen._`${(0, code$1.usePattern)(cxt, pat)}.test(${key})`, ()=>{
                        const alwaysValid = alwaysValidPatterns.includes(pat);
                        if (!alwaysValid) {
                            cxt.subschema({
                                keyword: "patternProperties",
                                schemaProp: pat,
                                dataProp: key,
                                dataPropType: util_2.Type.Str
                            }, valid);
                        }
                        if (it.opts.unevaluated && props !== true) {
                            gen.assign(codegen._`${props}[${key}]`, true);
                        } else if (!alwaysValid && !it.allErrors) {
                            gen.if((0, codegen.not)(valid), ()=>gen.break()
                            );
                        }
                    });
                });
            }
        }
    };
    exports.default = def36;
});
var not = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def37 = {
        keyword: "not",
        schemaType: [
            "object",
            "boolean"
        ],
        trackErrors: true,
        code (cxt) {
            const { gen , schema , it  } = cxt;
            if ((0, util1.alwaysValidSchema)(it, schema)) {
                cxt.fail();
                return;
            }
            const valid = gen.name("valid");
            cxt.subschema({
                keyword: "not",
                compositeRule: true,
                createErrors: false,
                allErrors: false
            }, valid);
            cxt.failResult(valid, ()=>cxt.reset()
            , ()=>cxt.error()
            );
        },
        error: {
            message: "must NOT be valid"
        }
    };
    exports.default = def37;
});
var anyOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def38 = {
        keyword: "anyOf",
        schemaType: "array",
        trackErrors: true,
        code: code$1.validateUnion,
        error: {
            message: "must match a schema in anyOf"
        }
    };
    exports.default = def38;
});
var oneOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error39 = {
        message: "must match exactly one schema in oneOf",
        params: ({ params  })=>codegen._`{passingSchemas: ${params.passing}}`
    };
    const def39 = {
        keyword: "oneOf",
        schemaType: "array",
        trackErrors: true,
        error: error39,
        code (cxt) {
            const { gen , schema , parentSchema , it  } = cxt;
            if (!Array.isArray(schema)) throw new Error("ajv implementation error");
            if (it.opts.discriminator && parentSchema.discriminator) return;
            const schArr = schema;
            const valid = gen.let("valid", false);
            const passing = gen.let("passing", null);
            const schValid = gen.name("_valid");
            cxt.setParams({
                passing
            });
            gen.block(validateOneOf);
            cxt.result(valid, ()=>cxt.reset()
            , ()=>cxt.error(true)
            );
            function validateOneOf() {
                schArr.forEach((sch, i59)=>{
                    let schCxt;
                    if ((0, util1.alwaysValidSchema)(it, sch)) {
                        gen.var(schValid, true);
                    } else {
                        schCxt = cxt.subschema({
                            keyword: "oneOf",
                            schemaProp: i59,
                            compositeRule: true
                        }, schValid);
                    }
                    if (i59 > 0) {
                        gen.if(codegen._`${schValid} && ${valid}`).assign(valid, false).assign(passing, codegen._`[${passing}, ${i59}]`).else();
                    }
                    gen.if(schValid, ()=>{
                        gen.assign(valid, true);
                        gen.assign(passing, i59);
                        if (schCxt) cxt.mergeEvaluated(schCxt, codegen.Name);
                    });
                });
            }
        }
    };
    exports.default = def39;
});
var allOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def40 = {
        keyword: "allOf",
        schemaType: "array",
        code (cxt) {
            const { gen , schema , it  } = cxt;
            if (!Array.isArray(schema)) throw new Error("ajv implementation error");
            const valid = gen.name("valid");
            schema.forEach((sch, i60)=>{
                if ((0, util1.alwaysValidSchema)(it, sch)) return;
                const schCxt = cxt.subschema({
                    keyword: "allOf",
                    schemaProp: i60
                }, valid);
                cxt.ok(valid);
                cxt.mergeEvaluated(schCxt);
            });
        }
    };
    exports.default = def40;
});
var _if = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error40 = {
        message: ({ params  })=>codegen.str`must match "${params.ifClause}" schema`
        ,
        params: ({ params  })=>codegen._`{failingKeyword: ${params.ifClause}}`
    };
    const def41 = {
        keyword: "if",
        schemaType: [
            "object",
            "boolean"
        ],
        trackErrors: true,
        error: error40,
        code (cxt) {
            const { gen , parentSchema , it  } = cxt;
            if (parentSchema.then === void 0 && parentSchema.else === void 0) {
                (0, util1.checkStrictMode)(it, '"if" without "then" and "else" is ignored');
            }
            const hasThen = hasSchema(it, "then");
            const hasElse = hasSchema(it, "else");
            if (!hasThen && !hasElse) return;
            const valid = gen.let("valid", true);
            const schValid = gen.name("_valid");
            validateIf();
            cxt.reset();
            if (hasThen && hasElse) {
                const ifClause = gen.let("ifClause");
                cxt.setParams({
                    ifClause
                });
                gen.if(schValid, validateClause("then", ifClause), validateClause("else", ifClause));
            } else if (hasThen) {
                gen.if(schValid, validateClause("then"));
            } else {
                gen.if((0, codegen.not)(schValid), validateClause("else"));
            }
            cxt.pass(valid, ()=>cxt.error(true)
            );
            function validateIf() {
                const schCxt = cxt.subschema({
                    keyword: "if",
                    compositeRule: true,
                    createErrors: false,
                    allErrors: false
                }, schValid);
                cxt.mergeEvaluated(schCxt);
            }
            function validateClause(keyword2, ifClause) {
                return ()=>{
                    const schCxt = cxt.subschema({
                        keyword: keyword2
                    }, schValid);
                    gen.assign(valid, schValid);
                    cxt.mergeValidEvaluated(schCxt, valid);
                    if (ifClause) gen.assign(ifClause, codegen._`${keyword2}`);
                    else cxt.setParams({
                        ifClause: keyword2
                    });
                };
            }
        }
    };
    function hasSchema(it, keyword2) {
        const schema = it.schema[keyword2];
        return schema !== void 0 && !(0, util1.alwaysValidSchema)(it, schema);
    }
    exports.default = def41;
});
var thenElse = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def42 = {
        keyword: [
            "then",
            "else"
        ],
        schemaType: [
            "object",
            "boolean"
        ],
        code ({ keyword: keyword2 , parentSchema , it  }) {
            if (parentSchema.if === void 0) (0, util1.checkStrictMode)(it, `"${keyword2}" without "if" is ignored`);
        }
    };
    exports.default = def42;
});
var applicator = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    function getApplicator(draft2020 = false) {
        const applicator2 = [
            not.default,
            anyOf.default,
            oneOf.default,
            allOf.default,
            _if.default,
            thenElse.default,
            propertyNames.default,
            additionalProperties$1.default,
            dependencies.default,
            properties$1.default,
            patternProperties.default
        ];
        if (draft2020) applicator2.push(prefixItems.default, items2020.default);
        else applicator2.push(additionalItems.default, items.default);
        applicator2.push(contains.default);
        return applicator2;
    }
    exports.default = getApplicator;
});
var format3 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error41 = {
        message: ({ schemaCode  })=>codegen.str`must match format "${schemaCode}"`
        ,
        params: ({ schemaCode  })=>codegen._`{format: ${schemaCode}}`
    };
    const def43 = {
        keyword: "format",
        type: [
            "number",
            "string"
        ],
        schemaType: "string",
        $data: true,
        error: error41,
        code (cxt, ruleType) {
            const { gen , data: data93 , $data , schema , schemaCode , it  } = cxt;
            const { opts , errSchemaPath , schemaEnv , self  } = it;
            if (!opts.validateFormats) return;
            if ($data) validate$DataFormat();
            else validateFormat();
            function validate$DataFormat() {
                const fmts = gen.scopeValue("formats", {
                    ref: self.formats,
                    code: opts.code.formats
                });
                const fDef = gen.const("fDef", codegen._`${fmts}[${schemaCode}]`);
                const fType = gen.let("fType");
                const format23 = gen.let("format");
                gen.if(codegen._`typeof ${fDef} == "object" && !(${fDef} instanceof RegExp)`, ()=>gen.assign(fType, codegen._`${fDef}.type || "string"`).assign(format23, codegen._`${fDef}.validate`)
                , ()=>gen.assign(fType, codegen._`"string"`).assign(format23, fDef)
                );
                cxt.fail$data((0, codegen.or)(unknownFmt(), invalidFmt()));
                function unknownFmt() {
                    if (opts.strictSchema === false) return codegen.nil;
                    return codegen._`${schemaCode} && !${format23}`;
                }
                function invalidFmt() {
                    const callFormat = schemaEnv.$async ? codegen._`(${fDef}.async ? await ${format23}(${data93}) : ${format23}(${data93}))` : codegen._`${format23}(${data93})`;
                    const validData = codegen._`(typeof ${format23} == "function" ? ${callFormat} : ${format23}.test(${data93}))`;
                    return codegen._`${format23} && ${format23} !== true && ${fType} === ${ruleType} && !${validData}`;
                }
            }
            function validateFormat() {
                const formatDef = self.formats[schema];
                if (!formatDef) {
                    unknownFormat();
                    return;
                }
                if (formatDef === true) return;
                const [fmtType, format24, fmtRef] = getFormat(formatDef);
                if (fmtType === ruleType) cxt.pass(validCondition());
                function unknownFormat() {
                    if (opts.strictSchema === false) {
                        self.logger.warn(unknownMsg());
                        return;
                    }
                    throw new Error(unknownMsg());
                    function unknownMsg() {
                        return `unknown format "${schema}" ignored in schema at path "${errSchemaPath}"`;
                    }
                }
                function getFormat(fmtDef) {
                    const code28 = fmtDef instanceof RegExp ? (0, codegen.regexpCode)(fmtDef) : opts.code.formats ? codegen._`${opts.code.formats}${(0, codegen.getProperty)(schema)}` : void 0;
                    const fmt = gen.scopeValue("formats", {
                        key: schema,
                        ref: fmtDef,
                        code: code28
                    });
                    if (typeof fmtDef == "object" && !(fmtDef instanceof RegExp)) {
                        return [
                            fmtDef.type || "string",
                            fmtDef.validate,
                            codegen._`${fmt}.validate`
                        ];
                    }
                    return [
                        "string",
                        fmtDef,
                        fmt
                    ];
                }
                function validCondition() {
                    if (typeof formatDef == "object" && !(formatDef instanceof RegExp) && formatDef.async) {
                        if (!schemaEnv.$async) throw new Error("async format in sync schema");
                        return codegen._`await ${fmtRef}(${data93})`;
                    }
                    return typeof format24 == "function" ? codegen._`${fmtRef}(${data93})` : codegen._`${fmtRef}.test(${data93})`;
                }
            }
        }
    };
    exports.default = def43;
});
var format_2 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const format$1 = [
        format3.default
    ];
    exports.default = format$1;
});
var metadata = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.contentVocabulary = exports.metadataVocabulary = void 0;
    exports.metadataVocabulary = [
        "title",
        "description",
        "default",
        "deprecated",
        "readOnly",
        "writeOnly",
        "examples"
    ];
    exports.contentVocabulary = [
        "contentMediaType",
        "contentEncoding",
        "contentSchema"
    ];
});
var draft7 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const draft7Vocabularies = [
        core_1.default,
        validation_1.default,
        (0, applicator.default)(),
        format_2.default,
        metadata.metadataVocabulary,
        metadata.contentVocabulary
    ];
    exports.default = draft7Vocabularies;
});
var types8 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.DiscrError = void 0;
    (function(DiscrError) {
        DiscrError["Tag"] = "tag";
        DiscrError["Mapping"] = "mapping";
    })(exports.DiscrError || (exports.DiscrError = {
    }));
});
var discriminator = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error42 = {
        message: ({ params: { discrError , tagName  }  })=>discrError === types8.DiscrError.Tag ? `tag "${tagName}" must be string` : `value of tag "${tagName}" must be in oneOf`
        ,
        params: ({ params: { discrError , tag , tagName  }  })=>codegen._`{error: ${discrError}, tag: ${tagName}, tagValue: ${tag}}`
    };
    const def44 = {
        keyword: "discriminator",
        type: "object",
        schemaType: "object",
        error: error42,
        code (cxt) {
            const { gen , data: data94 , schema , parentSchema , it  } = cxt;
            const { oneOf: oneOf2  } = parentSchema;
            if (!it.opts.discriminator) {
                throw new Error("discriminator: requires discriminator option");
            }
            const tagName = schema.propertyName;
            if (typeof tagName != "string") throw new Error("discriminator: requires propertyName");
            if (schema.mapping) throw new Error("discriminator: mapping is not supported");
            if (!oneOf2) throw new Error("discriminator: requires oneOf keyword");
            const valid = gen.let("valid", false);
            const tag = gen.const("tag", codegen._`${data94}${(0, codegen.getProperty)(tagName)}`);
            gen.if(codegen._`typeof ${tag} == "string"`, ()=>validateMapping()
            , ()=>cxt.error(false, {
                    discrError: types8.DiscrError.Tag,
                    tag,
                    tagName
                })
            );
            cxt.ok(valid);
            function validateMapping() {
                const mapping = getMapping();
                gen.if(false);
                for(const tagValue in mapping){
                    gen.elseIf(codegen._`${tag} === ${tagValue}`);
                    gen.assign(valid, applyTagSchema(mapping[tagValue]));
                }
                gen.else();
                cxt.error(false, {
                    discrError: types8.DiscrError.Mapping,
                    tag,
                    tagName
                });
                gen.endIf();
            }
            function applyTagSchema(schemaProp) {
                const _valid = gen.name("valid");
                const schCxt = cxt.subschema({
                    keyword: "oneOf",
                    schemaProp
                }, _valid);
                cxt.mergeEvaluated(schCxt, codegen.Name);
                return _valid;
            }
            function getMapping() {
                var _a;
                const oneOfMapping = {
                };
                const topRequired = hasRequired(parentSchema);
                let tagRequired = true;
                for(let i1 = 0; i1 < oneOf2.length; i1++){
                    const sch = oneOf2[i1];
                    const propSch = (_a = sch.properties) === null || _a === void 0 ? void 0 : _a[tagName];
                    if (typeof propSch != "object") {
                        throw new Error(`discriminator: oneOf schemas must have "properties/${tagName}"`);
                    }
                    tagRequired = tagRequired && (topRequired || hasRequired(sch));
                    addMappings(propSch, i1);
                }
                if (!tagRequired) throw new Error(`discriminator: "${tagName}" must be required`);
                return oneOfMapping;
                function hasRequired({ required: required2  }) {
                    return Array.isArray(required2) && required2.includes(tagName);
                }
                function addMappings(sch, i61) {
                    if (sch.const) {
                        addMapping(sch.const, i61);
                    } else if (sch.enum) {
                        for (const tagValue of sch.enum){
                            addMapping(tagValue, i61);
                        }
                    } else {
                        throw new Error(`discriminator: "properties/${tagName}" must have "const" or "enum"`);
                    }
                }
                function addMapping(tagValue, i62) {
                    if (typeof tagValue != "string" || tagValue in oneOfMapping) {
                        throw new Error(`discriminator: "${tagName}" values must be unique strings`);
                    }
                    oneOfMapping[tagValue] = i62;
                }
            }
        }
    };
    exports.default = def44;
});
const $schema = "http://json-schema.org/draft-07/schema#";
const $id$1 = "http://json-schema.org/draft-07/schema#";
const title = "Core schema meta-schema";
const definitions = {
    schemaArray: {
        type: "array",
        minItems: 1,
        items: {
            $ref: "#"
        }
    },
    nonNegativeInteger: {
        type: "integer",
        minimum: 0
    },
    nonNegativeIntegerDefault0: {
        allOf: [
            {
                $ref: "#/definitions/nonNegativeInteger"
            },
            {
                default: 0
            }
        ]
    },
    simpleTypes: {
        enum: [
            "array",
            "boolean",
            "integer",
            "null",
            "number",
            "object",
            "string"
        ]
    },
    stringArray: {
        type: "array",
        items: {
            type: "string"
        },
        uniqueItems: true,
        default: []
    }
};
const type$1 = [
    "object",
    "boolean"
];
const properties$2 = {
    $id: {
        type: "string",
        format: "uri-reference"
    },
    $schema: {
        type: "string",
        format: "uri"
    },
    $ref: {
        type: "string",
        format: "uri-reference"
    },
    $comment: {
        type: "string"
    },
    title: {
        type: "string"
    },
    description: {
        type: "string"
    },
    default: true,
    readOnly: {
        type: "boolean",
        default: false
    },
    examples: {
        type: "array",
        items: true
    },
    multipleOf: {
        type: "number",
        exclusiveMinimum: 0
    },
    maximum: {
        type: "number"
    },
    exclusiveMaximum: {
        type: "number"
    },
    minimum: {
        type: "number"
    },
    exclusiveMinimum: {
        type: "number"
    },
    maxLength: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minLength: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    pattern: {
        type: "string",
        format: "regex"
    },
    additionalItems: {
        $ref: "#"
    },
    items: {
        anyOf: [
            {
                $ref: "#"
            },
            {
                $ref: "#/definitions/schemaArray"
            }
        ],
        default: true
    },
    maxItems: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minItems: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    uniqueItems: {
        type: "boolean",
        default: false
    },
    contains: {
        $ref: "#"
    },
    maxProperties: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minProperties: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    required: {
        $ref: "#/definitions/stringArray"
    },
    additionalProperties: {
        $ref: "#"
    },
    definitions: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        default: {
        }
    },
    properties: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        default: {
        }
    },
    patternProperties: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        propertyNames: {
            format: "regex"
        },
        default: {
        }
    },
    dependencies: {
        type: "object",
        additionalProperties: {
            anyOf: [
                {
                    $ref: "#"
                },
                {
                    $ref: "#/definitions/stringArray"
                }
            ]
        }
    },
    propertyNames: {
        $ref: "#"
    },
    const: true,
    enum: {
        type: "array",
        items: true,
        minItems: 1,
        uniqueItems: true
    },
    type: {
        anyOf: [
            {
                $ref: "#/definitions/simpleTypes"
            },
            {
                type: "array",
                items: {
                    $ref: "#/definitions/simpleTypes"
                },
                minItems: 1,
                uniqueItems: true
            }
        ]
    },
    format: {
        type: "string"
    },
    contentMediaType: {
        type: "string"
    },
    contentEncoding: {
        type: "string"
    },
    if: {
        $ref: "#"
    },
    then: {
        $ref: "#"
    },
    else: {
        $ref: "#"
    },
    allOf: {
        $ref: "#/definitions/schemaArray"
    },
    anyOf: {
        $ref: "#/definitions/schemaArray"
    },
    oneOf: {
        $ref: "#/definitions/schemaArray"
    },
    not: {
        $ref: "#"
    }
};
var draft7MetaSchema = {
    $schema,
    $id: $id$1,
    title,
    definitions,
    type: type$1,
    properties: properties$2,
    default: true
};
var ajv1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.CodeGen = exports.Name = exports.nil = exports.stringify = exports.str = exports._ = exports.KeywordCxt = void 0;
    const META_SUPPORT_DATA = [
        "/properties"
    ];
    const META_SCHEMA_ID = "http://json-schema.org/draft-07/schema";
    class Ajv extends core1.default {
        _addVocabularies() {
            super._addVocabularies();
            draft7.default.forEach((v)=>this.addVocabulary(v)
            );
            if (this.opts.discriminator) this.addKeyword(discriminator.default);
        }
        _addDefaultMetaSchema() {
            super._addDefaultMetaSchema();
            if (!this.opts.meta) return;
            const metaSchema = this.opts.$data ? this.$dataMetaSchema(draft7MetaSchema, META_SUPPORT_DATA) : draft7MetaSchema;
            this.addMetaSchema(metaSchema, META_SCHEMA_ID, false);
            this.refs["http://json-schema.org/schema"] = META_SCHEMA_ID;
        }
        defaultMeta() {
            return this.opts.defaultMeta = super.defaultMeta() || (this.getSchema(META_SCHEMA_ID) ? META_SCHEMA_ID : void 0);
        }
    }
    module.exports = exports = Ajv;
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.default = Ajv;
    Object.defineProperty(exports, "KeywordCxt", {
        enumerable: true,
        get: function() {
            return validate1.KeywordCxt;
        }
    });
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return codegen._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return codegen.str;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return codegen.stringify;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return codegen.nil;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return codegen.Name;
        }
    });
    Object.defineProperty(exports, "CodeGen", {
        enumerable: true,
        get: function() {
            return codegen.CodeGen;
        }
    });
});
var __pika_web_default_export_for_treeshaking__1 = getDefaultExportFromCjs1(ajv1);
ajv1.CodeGen;
ajv1.KeywordCxt;
ajv1.Name;
ajv1._;
ajv1.nil;
ajv1.str;
ajv1.stringify;
function Diff() {
}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {
        };
        var callback = options.callback;
        if (typeof options === "function") {
            callback = options;
            options = {
            };
        }
        this.options = options;
        var self = this;
        function done(value) {
            if (callback) {
                setTimeout(function() {
                    callback(void 0, value);
                }, 0);
                return true;
            } else {
                return value;
            }
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath1 = bestPath[diagonalPath - 1], removePath1 = bestPath[diagonalPath + 1], _oldPos = (removePath1 ? removePath1.newPos : 0) - diagonalPath;
                if (addPath1) {
                    bestPath[diagonalPath - 1] = void 0;
                }
                var canAdd = addPath1 && addPath1.newPos + 1 < newLen, canRemove = removePath1 && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = void 0;
                    continue;
                }
                if (!canAdd || canRemove && addPath1.newPos < removePath1.newPos) {
                    basePath = clonePath(removePath1);
                    self.pushComponent(basePath.components, void 0, true);
                } else {
                    basePath = addPath1;
                    basePath.newPos++;
                    self.pushComponent(basePath.components, true, void 0);
                }
                _oldPos = self.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self, basePath.components, newString, oldString, self.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        if (callback) {
            (function exec1() {
                setTimeout(function() {
                    if (editLength > maxEditLength) {
                        return callback();
                    }
                    if (!execEditLength()) {
                        exec1();
                    }
                }, 0);
            })();
        } else {
            while(editLength <= maxEditLength){
                var ret = execEditLength();
                if (ret) {
                    return ret;
                }
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last8 = components[components.length - 1];
        if (last8 && last8.added === added && last8.removed === removed) {
            components[components.length - 1] = {
                count: last8.count + 1,
                added,
                removed
            };
        } else {
            components.push({
                count: 1,
                added,
                removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array5) {
        var ret = [];
        for(var i63 = 0; i63 < array5.length; i63++){
            if (array5[i63]) {
                ret.push(array5[i63]);
            }
        }
        return ret;
    },
    castInput: function castInput(value) {
        return value;
    },
    tokenize: function tokenize(value) {
        return value.split("");
    },
    join: function join(chars) {
        return chars.join("");
    }
};
function buildValues(diff2, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value = newString.slice(newPos, newPos + component.count);
                value = value.map(function(value2, i) {
                    var oldValue = oldString[oldPos + i];
                    return oldValue.length > value2.length ? oldValue : value2;
                });
                component.value = diff2.join(value);
            } else {
                component.value = diff2.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff2.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === "string" && (lastComponent.added || lastComponent.removed) && diff2.equals("", lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path87) {
    return {
        newPos: path87.newPos,
        components: path87.components.slice(0)
    };
}
new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace1 = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace1.test(left) && !reWhitespace1.test(right);
};
wordDiff.tokenize = function(value) {
    var tokens = value.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i64 = 0; i64 < tokens.length - 1; i64++){
        if (!tokens[i64 + 1] && tokens[i64 + 2] && extendedWordChars.test(tokens[i64]) && extendedWordChars.test(tokens[i64 + 2])) {
            tokens[i64] += tokens[i64 + 2];
            tokens.splice(i64 + 1, 2);
            i64--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value) {
    var retLines = [], linesAndNewlines = value.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i65 = 0; i65 < linesAndNewlines.length; i65++){
        var line = linesAndNewlines[i65];
        if (i65 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line;
        } else {
            if (this.options.ignoreWhitespace) {
                line = line.trim();
            }
            retLines.push(line);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value) {
    return value.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value) {
    return value.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj2) {
            return typeof obj2;
        };
    } else {
        _typeof = function(obj2) {
            return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === "undefined" ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value === "string" ? value : JSON.stringify(canonicalize(value, null, null, stringifyReplacer), stringifyReplacer, "  ");
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, "$1"), right.replace(/,([\r\n])/g, "$1"));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i66;
    for(i66 = 0; i66 < stack.length; i66 += 1){
        if (stack[i66] === obj) {
            return replacementStack[i66];
        }
    }
    var canonicalizedObj;
    if (objectPrototypeToString.call(obj) === "[object Array]") {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i66 = 0; i66 < obj.length; i66 += 1){
            canonicalizedObj[i66] = canonicalize(obj[i66], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === "object" && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {
        };
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i66 = 0; i66 < sortedKeys.length; i66 += 1){
            _key = sortedKeys[i66];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value) {
    return value.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value) {
    return value;
};
function matchAll1(str77, regex) {
    let match;
    regex = new RegExp(regex);
    const result = [];
    while((match = regex.exec(str77)) != null){
        result.push(match);
    }
    return result;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result;
    } else {
        const matches6 = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r of matches6){
            const stringEnd = r.index + 1;
            result.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result;
    }
}
function mappedDiff(source, target) {
    const sourceLineRanges = rangedLines(source.value).map((x)=>x.range
    );
    let sourceCursor = 0;
    const resultChunks = [];
    for (const action of diffLines(source.value, target)){
        if (action.removed) {
            sourceCursor += action.count;
        } else if (action.added) {
            resultChunks.push(action.value);
        } else {
            const start = sourceLineRanges[sourceCursor].start;
            const nextCursor = sourceCursor + action.count;
            const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
            sourceCursor = nextCursor;
            resultChunks.push({
                start,
                end
            });
        }
    }
    return mappedString(source, resultChunks);
}
const kMetadataFormat = "format";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig-width";
const kFigHeight = "fig-height";
const kFigFormat = "fig-format";
const kFigDpi = "fig-dpi";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kKeepHidden = "keep-hidden";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepYaml = "keep-yaml";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kMergeIncludes = "merge-includes";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug, 
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepYaml,
    kKeepSource,
    kKeepHidden,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter, 
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kCalloutTipCaption = "callout-tip-caption";
const kCalloutNoteCaption = "callout-note-caption";
const kCalloutWarningCaption = "callout-warning-caption";
const kCalloutImportantCaption = "callout-important-caption";
const kCalloutDangerCaption = "callout-danger-caption";
const kSectionTitleFootnotes = "section-title-footnotes";
const kCodeSummary = "code-summary";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kCopyButtonTooltip = "copy-button-tooltip";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutDangerCaption,
    kSectionTitleFootnotes,
    kCodeSummary,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kSearchNoResultsText,
    kCopyButtonTooltip,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle, 
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kAtxHeaders = "atx-headers";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kStandalone = "standalone";
const kSelfContained = "self-contained";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kMathjax = "mathjax";
const kKatex = "katex";
const kMathml = "mathml";
const kGladtex = "gladtex";
const kWebtex = "webtex";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDoi = "doi";
const kAbstract = "abstract";
const kDescription = "description";
const kTocTitle = "toc-title";
const kLang = "lang";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kCss = "css";
const kBibliography = "bibliography";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kFigResponsive = "fig-responsive";
const kCapLoc = "cap-location";
const kFigCapLoc = "fig-cap-location";
const kTblCapLoc = "tbl-cap-location";
const kCapTop = "top";
const kCapBottom = "bottom";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    "resource-path",
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    "wrap",
    "columns",
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    "toc-depth",
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "strip-empty-paragraphs",
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    "default-image-extension",
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kAtxHeaders,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace", 
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
objectSchema();
const dependencyFileSchema = objectSchema({
    properties: {
        name: StringSchema,
        path: StringSchema,
        attribs: objectSchema()
    },
    required: [
        "name",
        "path"
    ]
});
objectSchema({
    properties: {
        name: StringSchema,
        version: StringSchema,
        meta: objectSchema(),
        links: arraySchema(objectSchema({
            properties: {
                rel: StringSchema,
                href: StringSchema
            },
            description: "be an object with 'rel' and 'href' keys with string values"
        })),
        scripts: arraySchema(dependencyFileSchema),
        stylesheets: arraySchema(dependencyFileSchema),
        resources: arraySchema(dependencyFileSchema)
    },
    required: [
        "name"
    ]
});
objectSchema({
    properties: {
        header: StringSchema,
        before: StringSchema,
        after: StringSchema
    }
});
const sassLayerSchema = objectSchema({
    properties: {
        use: arraySchema(StringSchema),
        defaults: StringSchema,
        functions: StringSchema,
        mixins: StringSchema,
        rules: StringSchema
    },
    required: [
        "defaults",
        "functions",
        "mixins",
        "rules"
    ]
});
objectSchema({
    properties: {
        key: StringSchema,
        dependency: StringSchema,
        user: sassLayerSchema,
        quarto: sassLayerSchema,
        framework: sassLayerSchema,
        loadPath: StringSchema,
        dark: objectSchema({
            properties: {
                user: sassLayerSchema,
                quarto: sassLayerSchema,
                framework: sassLayerSchema,
                "default": BooleanSchema
            }
        }),
        attribs: objectSchema({
            additionalProperties: StringSchema
        })
    },
    required: [
        "key",
        "dependency"
    ]
});
objectSchema({
    properties: {
        [kKeepTex]: BooleanSchema,
        [kKeepYaml]: BooleanSchema,
        [kKeepSource]: BooleanSchema,
        [kKeepHidden]: BooleanSchema,
        [kPreferHtml]: BooleanSchema,
        [kOutputDivs]: BooleanSchema,
        [kVariant]: StringSchema,
        [kOutputExt]: StringSchema,
        [kPageWidth]: NumberSchema,
        [kFigAlign]: enumSchema("left", "right", "center", "default"),
        [kCodeFold]: oneOfSchema(enumSchema("none", "show", "hide"), BooleanSchema),
        [kCodeSummary]: StringSchema,
        [kCodeOverflow]: enumSchema("wrap", "scroll"),
        [kCodeLink]: BooleanSchema,
        [kCodeTools]: oneOfSchema(BooleanSchema, objectSchema({
            properties: {
                source: BooleanSchema,
                toggle: BooleanSchema,
                caption: StringSchema
            }
        })),
        [kMergeIncludes]: BooleanSchema,
        [kSelfContainedMath]: BooleanSchema,
        [kLatexAutoMk]: BooleanSchema,
        [kLatexAutoInstall]: BooleanSchema,
        [kLatexMinRuns]: NumberSchema,
        [kLatexMaxRuns]: NumberSchema,
        [kLatexClean]: BooleanSchema,
        [kLatexMakeIndex]: StringSchema,
        [kLatexMakeIndexOpts]: arraySchema(StringSchema),
        [kLatexTlmgrOpts]: arraySchema(StringSchema),
        [kLatexOutputDir]: oneOfSchema(StringSchema, NullSchema),
        [kLinkExternalIcon]: oneOfSchema(StringSchema, BooleanSchema),
        [kLinkExternalNewwindow]: BooleanSchema
    }
});
const formatExecuteOptionsSchema = objectSchema({
    properties: {
        "fig-width": NumberSchema,
        "fig-height": NumberSchema,
        "fig-format": enumSchema("retina", "png", "jpeg", "svg", "pdf"),
        "fig-dpi": NumberSchema,
        [kCache]: oneOfSchema(BooleanSchema, NullSchema, enumSchema("refresh")),
        [kFreeze]: oneOfSchema(BooleanSchema, enumSchema("auto")),
        [kExecuteEnabled]: oneOfSchema(BooleanSchema, NullSchema),
        [kExecuteIpynb]: oneOfSchema(BooleanSchema, NullSchema),
        [kExecuteDaemon]: oneOfSchema(NumberSchema, BooleanSchema, NullSchema),
        [kExecuteDaemonRestart]: BooleanSchema,
        [kExecuteDebug]: BooleanSchema,
        [kEngine]: StringSchema,
        [kEval]: oneOfSchema(BooleanSchema, NullSchema),
        [kError]: BooleanSchema,
        [kEcho]: oneOfSchema(BooleanSchema, enumSchema("fenced")),
        [kOutput]: oneOfSchema(BooleanSchema, enumSchema("all", "asis")),
        [kWarning]: BooleanSchema,
        [kInclude]: BooleanSchema,
        [kKeepMd]: BooleanSchema,
        [kKeepIpynb]: BooleanSchema
    }
});
objectSchema({
    properties: {
        "from": StringSchema,
        "to": StringSchema,
        "writer": StringSchema,
        [kTemplate]: StringSchema,
        [kOutputFile]: StringSchema,
        "standalone": BooleanSchema,
        [kSelfContained]: BooleanSchema,
        [kVariables]: objectSchema(),
        [kAtxHeaders]: BooleanSchema,
        [kMarkdownHeadings]: BooleanSchema,
        [kIncludeBeforeBody]: arraySchema(StringSchema),
        [kIncludeAfterBody]: arraySchema(StringSchema),
        [kIncludeInHeader]: arraySchema(StringSchema),
        [kCiteproc]: BooleanSchema,
        [kCiteMethod]: StringSchema,
        [kFilters]: arraySchema(StringSchema),
        [kPdfEngine]: StringSchema,
        [kPdfEngineOpts]: arraySchema(StringSchema),
        [kPdfEngineOpt]: StringSchema,
        [kEPubCoverImage]: StringSchema,
        [kCss]: oneOfSchema(StringSchema, arraySchema(StringSchema)),
        [kToc]: BooleanSchema,
        [kTableOfContents]: BooleanSchema,
        [kListings]: BooleanSchema,
        [kNumberSections]: BooleanSchema,
        [kNumberOffset]: arraySchema(NumberSchema),
        [kHighlightStyle]: StringSchema,
        [kSectionDivs]: BooleanSchema,
        [kHtmlMathMethod]: oneOfSchema(StringSchema, objectSchema({
            properties: {
                method: StringSchema,
                url: StringSchema
            },
            required: [
                "method",
                "url"
            ]
        })),
        [kTopLevelDivision]: StringSchema,
        [kShiftHeadingLevelBy]: NumberSchema,
        [kTitlePrefix]: StringSchema
    }
});
objectSchema({
    properties: {
        to: StringSchema,
        output: StringSchema,
        [kSelfContained]: BooleanSchema,
        pdfEngine: StringSchema,
        pdfEngineOpts: arraySchema(StringSchema),
        makeIndexOpts: arraySchema(StringSchema),
        tlmgrOpts: arraySchema(StringSchema),
        natbib: BooleanSchema,
        biblatex: BooleanSchema,
        [kToc]: BooleanSchema,
        [kTocTitle]: StringSchema,
        [kListings]: BooleanSchema,
        [kNumberSections]: BooleanSchema,
        [kNumberOffset]: arraySchema(NumberSchema),
        [kTopLevelDivision]: StringSchema,
        [kShiftHeadingLevelBy]: StringSchema,
        [kIncludeInHeader]: StringSchema,
        [kIncludeBeforeBody]: StringSchema,
        [kIncludeAfterBody]: StringSchema,
        [kMathjax]: BooleanSchema,
        [kKatex]: BooleanSchema,
        [kMathml]: BooleanSchema,
        [kGladtex]: BooleanSchema,
        [kWebtex]: BooleanSchema
    }
});
objectSchema({
    properties: {
        pdfEngine: StringSchema,
        pdfEngineOpts: arraySchema(StringSchema),
        bibEngine: enumSchema("natbib", "biblatex"),
        indexEngine: StringSchema,
        indexEngineOpts: arraySchema(StringSchema),
        tlmgrOpts: arraySchema(StringSchema)
    },
    required: [
        "pdfEngine"
    ]
});
const kRegExBeginYAML = /^---[ \t]*$/;
oneOfSchema(formatExecuteOptionsSchema, enumSchema("default"));
let formatAliases = undefined;
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        return parse3(decoder.decode(yml), {
            schema: json1
        });
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function getFormatAliases() {
    if (formatAliases !== undefined) {
        return formatAliases;
    }
    formatAliases = readYaml(join3(resourcePath(), "schema/format-aliases.yml")).aliases;
    return formatAliases;
}
function getExpandedFormatAliases() {
    const aliases = getFormatAliases();
    const result = {
    };
    for (const [key, value] of Object.entries(aliases)){
        result[key] = expandFormatAliases(value);
    }
    return result;
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg, validationErrors){
        super(msg);
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
function readAndValidateYamlFromFile(file, schema, errorMessage) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    const contents = asMappedString(Deno.readTextFileSync(file));
    return readAndValidateYamlFromMappedString(contents, schema, errorMessage);
}
function expandFormatAliases(lst) {
    const aliases = getFormatAliases();
    const result = [];
    lst = lst.slice();
    for(let i67 = 0; i67 < lst.length; ++i67){
        const el = lst[i67];
        if (el.startsWith("$")) {
            const v = aliases[el.slice(1)];
            if (v === undefined) {
                throw new Error(`Internal Error: ${el} doesn't have an entry in the aliases map`);
            }
            lst.push(...v);
        } else {
            result.push(el);
        }
    }
    return result;
}
async function getFormatSchema(format9) {
    const aliases = getExpandedFormatAliases();
    aliases["pandoc-all"];
    const entries = [];
    try {
        entries.push(...await readAndValidateYamlFromFile(resourcePath("schema/format-pandoc.yml"), {
            $id: "good"
        }, "schema entry file validation failed."));
    } catch (e) {
        error("\n");
        error(e);
        if (e instanceof ValidationError2) {
            for (const err of e.validationErrors){
                error(err.message);
                error(err.error.message);
                error("\n");
            }
        }
        throw e;
    }
    const useEntry = (entry)=>{
        if (entry.disabled) {
            return expandFormatAliases(entry.disabled).indexOf(format9) === -1;
        } else if (entry.enabled) {
            return expandFormatAliases(entry.enabled).indexOf(format9) !== -1;
        }
        return true;
    };
    const properties3 = {
    };
    for (const entry1 of entries){
        if (!useEntry(entry1)) {
            continue;
        }
        let schema = convertFromYaml(entry1.schema);
        if (entry1.description) {
            if (typeof entry1.description === "string") {
                schema = documentSchema(schema, entry1.description);
            } else if (typeof entry1.description === "object") {
                schema = documentSchema(schema, entry1.description.short);
            }
        }
        properties3[entry1.name] = schema;
    }
    return oneOfSchema(objectSchema({
        properties: properties3
    }), enumSchema("default"));
}
function loadDefaultSchemaDefinitions() {
    return loadSchemaDefinitions(resourcePath("schema/definitions.yml"));
}
function readYamlFromString(yml) {
    return parse3(yml, {
        schema: json1
    });
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            const yamlBlock = removeYamlDelimiters(match[2]);
            if (!yamlBlock.startsWith("\n\n") && !yamlBlock.startsWith("\n---") && yamlBlock.trim().length > 0) {
                parse3(yamlBlock, {
                    json: true,
                    schema: json1
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata1 = parse3(yaml, {
            json: true,
            schema: json1
        });
        return metadata1 || {
        };
    } else {
        return {
        };
    }
}
let ajvInit = false;
async function ensureAjv() {
    if (!ajvInit) {
        setupAjv(new __pika_web_default_export_for_treeshaking__1({
            allErrors: true
        }));
        ajvInit = true;
        await loadDefaultSchemaDefinitions();
    }
}
async function readAndValidateYamlFromMappedString(mappedYaml, schema, errorMessage) {
    await ensureAjv();
    const result = await withValidator(schema, (validator)=>{
        const annotation = readAnnotatedYamlFromMappedString(mappedYaml);
        const validateYaml = !(annotation.result?.["validate-yaml"] === false);
        const yaml = annotation.result;
        if (validateYaml) {
            const valResult = validator.validateParse(mappedYaml, annotation);
            if (valResult.errors.length) {
                validator.reportErrorsInSource({
                    result: yaml,
                    errors: valResult.errors
                }, mappedYaml, errorMessage, error, info);
            }
            return {
                yaml: yaml,
                yamlValidationErrors: valResult.errors
            };
        } else {
            return {
                yaml: yaml,
                yamlValidationErrors: []
            };
        }
    });
    if (result.yamlValidationErrors.length > 0) {
        throw new ValidationError2(errorMessage, result.yamlValidationErrors);
    }
    return result.yaml;
}
const schemaCache = {
};
const schemaCacheNormalized = {
};
function cacheSchemaFunction(name, maker) {
    const getter = async (normalized)=>{
        if (normalized) {
            if (schemaCacheNormalized[name]) {
                return schemaCacheNormalized[name];
            }
            const schema = await getter();
            schemaCacheNormalized[name] = normalizeSchema(schema);
            return schemaCacheNormalized[name];
        } else {
            if (schemaCache[name]) {
                return schemaCache[name];
            }
            const schema = await maker();
            schemaCache[name] = schema;
            return schema;
        }
    };
    return getter;
}
const pandocOutputFormats = [
    "asciidoc",
    "asciidoctor",
    "beamer",
    "biblatex",
    "bibtex",
    "commonmark",
    "commonmark_x",
    "context",
    "csljson",
    "docbook",
    "docbook4",
    "docbook5",
    "docx",
    "dokuwiki",
    "dzslides",
    "epub",
    "epub2",
    "epub3",
    "fb2",
    "gfm",
    "haddock",
    "html",
    "html4",
    "html5",
    "icml",
    "ipynb",
    "jats",
    "jats_archiving",
    "jats_articleauthoring",
    "jats_publishing",
    "jira",
    "json",
    "latex",
    "man",
    "markdown",
    "markdown_github",
    "markdown_mmd",
    "markdown_phpextra",
    "markdown_strict",
    "mediawiki",
    "ms",
    "muse",
    "native",
    "odt",
    "opendocument",
    "opml",
    "org",
    "pdf",
    "plain",
    "pptx",
    "revealjs",
    "rst",
    "rtf",
    "s5",
    "slideous",
    "slidy",
    "tei",
    "texinfo",
    "textile",
    "xwiki",
    "zimwiki", 
];
async function makeFrontMatterFormatSchema() {
    const formatSchemaDescriptorList = await Promise.all(pandocOutputFormats.map(async (x)=>{
        return {
            regex: `^${x}(\\+.+)?$`,
            schema: await getFormatSchema(x),
            name: x
        };
    }));
    const formatSchemas = formatSchemaDescriptorList.map(({ regex , schema  })=>[
            regex,
            schema
        ]
    );
    const plusFormatStringSchemas = formatSchemaDescriptorList.map(({ regex , name  })=>completeSchema(regexSchema(regex, `be '${name}'`), {
            type: "value",
            display: name,
            suggest_on_accept: true,
            value: name,
            description: name
        })
    );
    const completionsObject = Object.fromEntries(formatSchemaDescriptorList.map(({ name  })=>[
            name,
            name
        ]
    ));
    return oneOfSchema(describeSchema(oneOfSchema(...plusFormatStringSchemas), "the name of a pandoc-supported output format"), regexSchema("^hugo(\\+.+)?$", "be 'hugo'"), objectSchema({
        patternProperties: Object.fromEntries(formatSchemas),
        completions: completionsObject,
        additionalProperties: false
    }));
}
cacheSchemaFunction("front-matter-format", makeFrontMatterFormatSchema);
async function makeFrontMatterSchema() {
    return idSchema(oneOfSchema(NullSchema, objectSchema({
        properties: {
            title: StringSchema
        },
        description: "be a Quarto YAML front matter object"
    })), "front-matter");
}
const getFrontMatterSchema = cacheSchemaFunction("front-matter", makeFrontMatterSchema);
async function readAndValidateYamlFromMarkdown(eitherMarkdown) {
    let markdown = asMappedString(eitherMarkdown);
    if (!markdown.value) {
        return {
        };
    }
    markdown = mappedNormalizeNewlines(markdown);
    markdown = skipRegexpAll(markdown, kRegxHTMLComment);
    markdown = skipRegexpAll(markdown, kRegexFencedCode);
    const yaml = [];
    kRegExYAML.lastIndex = 0;
    for (const match of matchAll(markdown.value, kRegExYAML)){
        const yamlBlock = removeYamlDelimitersMapped(match[2]);
        const yamlBlockValue = yamlBlock.value;
        if (!yamlBlockValue.startsWith("\n\n") && !yamlBlockValue.startsWith("\n---") && yamlBlockValue.trim().length > 0) {
            parse3(yamlBlockValue, {
                json: true,
                schema: json1
            });
            yaml.push(yamlBlock);
        }
    }
    kRegExYAML.lastIndex = 0;
    if (yaml.length === 0) {
        return {
        };
    }
    const mappedYaml = mappedConcat(yaml);
    const metadata2 = parse3(mappedYaml.value, {
        json: true,
        schema: json1
    });
    if (metadata2?.["validate-yaml"] === false) {
        return readAndValidateYamlFromMappedString(mappedYaml, await getFrontMatterSchema(), "YAML front matter validation failed");
    }
    return metadata2;
}
async function readYamlFromMarkdownFile(file) {
    const markdown = Deno.readTextFileSync(file);
    const result = await readAndValidateYamlFromMarkdown(markdown);
    return result;
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line, index)=>index > 0 && line.match(kRegExEndYAML)
        );
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
function removeYamlDelimitersMapped(eitherYaml) {
    let yaml = asMappedString(eitherYaml);
    yaml = skipRegexp(yaml, /^---/);
    yaml = skipRegexp(yaml, /---\s*$/);
    return yaml;
}
async function loadSchemaDefinitions(file) {
    let yaml = readYaml(file);
    await ensureAjv();
    await Promise.all(Object.entries(yaml).map(async ([name, yamlSchema])=>{
        const schema = normalizeSchema(idSchema(convertFromYaml(yamlSchema), name));
        await withValidator(schema, (_validator)=>{
            return;
        });
    }));
}
function generate() {
    return crypto.randomUUID();
}
function mergeConfigs(config, ...configs) {
    config = ld.cloneDeep(config);
    configs = ld.cloneDeep(configs);
    return lodash.mergeWith(config, ...configs, (objValue, srcValue)=>{
        if (lodash.isArray(objValue) || lodash.isArray(srcValue)) {
            if (!objValue) {
                return srcValue;
            } else if (!srcValue) {
                return objValue;
            } else {
                if (!lodash.isArray(objValue)) {
                    objValue = [
                        objValue
                    ];
                }
                if (!lodash.isArray(srcValue)) {
                    srcValue = [
                        srcValue
                    ];
                }
            }
            const combined = objValue.concat(srcValue);
            return lodash.uniqBy(combined, (value)=>{
                if (typeof value === "function") {
                    return generate();
                } else {
                    return JSON.stringify(value);
                }
            });
        }
    });
}
function camelToKebab(camel) {
    const kebab = [];
    for(let i68 = 0; i68 < camel.length; i68++){
        const ch = camel.charAt(i68);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i68 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i69 = 0; i69 < kebab.length; i69++){
        const ch = kebab.charAt(i69);
        if (ch === "-") {
            camel.push(kebab.charAt(++i69).toUpperCase());
        } else if (i69 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
async function includedMetadata(dir, baseMetadata, schema) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join3(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join3(dir, file))
        );
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (await exists(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeConfigs({
        }, ...filesMetadata),
        files: yamlFiles
    };
}
function formatFromMetadata(baseFormat, to, debug1) {
    const typedFormat = {
        render: {
        },
        execute: {
        },
        pandoc: {
        },
        language: {
        },
        metadata: {
        }
    };
    let format10 = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format10 = metadataAsFormat({
            });
        } else if (configFormat instanceof Object) {
            format10 = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeConfigs(baseFormat, format10);
    if (debug1) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
    }
    return mergedFormat;
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key);
}
function metadataAsFormat(metadata3) {
    const typedFormat = {
        render: {
        },
        execute: {
        },
        pandoc: {
        },
        language: {
        },
        metadata: {
        }
    };
    const format11 = typedFormat;
    Object.keys(metadata3).forEach((key)=>{
        if ([
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata, 
        ].includes(key)) {
            if (typeof metadata3[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format11[key] = format11[key] || {
                    };
                    format11[kExecuteDefaults][kExecuteEnabled] = metadata3[key];
                }
            } else {
                format11[key] = {
                    ...format11[key],
                    ...metadata3[key]
                };
            }
        } else {
            if (kRenderDefaultsKeys.includes(key)) {
                format11.render[key] = metadata3[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format11.execute[key] = metadata3[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format11.pandoc[key] = metadata3[key];
            } else if (kLanguageDefaultsKeys.includes(key)) {
                format11.language[key] = metadata3[key];
            } else {
                format11.metadata[key] = metadata3[key];
            }
        }
    });
    return typedFormat;
}
function metadataGetDeep(metadata4, property7) {
    let values9 = [];
    lodash.each(metadata4, (value, key)=>{
        if (key === property7) {
            values9.push(value);
        } else if (lodash.isObject(value)) {
            values9 = values9.concat(metadataGetDeep(value, property7));
        }
    });
    return values9;
}
const kStdOut = "-";
function parseRenderFlags(args) {
    const flags = {
    };
    const argsStack = [
        ...args
    ];
    let arg = argsStack.shift();
    while(arg !== undefined){
        switch(arg){
            case "-t":
            case "--to":
                arg = argsStack.shift();
                if (arg && !arg.startsWith("-")) {
                    flags.to = arg;
                }
                break;
            case "-o":
            case "--output":
                arg = argsStack.shift();
                if (!arg || arg.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    flags.output = arg;
                }
                break;
            case "--output-dir":
                arg = argsStack.shift();
                flags.outputDir = arg;
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg = argsStack.shift();
                break;
            case "--pdf-engine":
                arg = argsStack.shift();
                flags.pdfEngine = arg;
                break;
            case "--pdf-engine-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg);
                }
                break;
            case "--latex-makeindex-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg);
                }
                break;
            case "--latex-tlmgr-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg);
                }
                break;
            case "--natbib":
                arg = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg);
                break;
            case "--top-level-division":
                arg = argsStack.shift();
                flags[kTopLevelDivision] = arg;
                break;
            case "--shift-heading-level-by":
                arg = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include = arg.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include] = includeFlags[include] || [];
                    arg = argsStack.shift();
                    includeFlags[include].push(arg);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg = argsStack.shift();
                break;
            case "--execute-params":
                arg = argsStack.shift();
                flags.paramsFile = arg;
                break;
            case "--execute-dir":
                arg = argsStack.shift();
                flags.executeDir = arg;
                break;
            case "--execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = parseInt(arg, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--cache":
                arg = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--debug":
                flags.debug = true;
                arg = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg = argsStack.shift();
                if (arg) {
                    const param = parseMetadataFlagValue(arg);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {
                            };
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg = argsStack.shift();
                if (arg) {
                    const metadata5 = parseMetadataFlagValue(arg);
                    if (metadata5) {
                        if (metadata5.value !== undefined) {
                            if (isQuartoMetadata(metadata5.name)) {
                                flags.metadata = flags.metadata || {
                                };
                                flags.metadata[metadata5.name] = metadata5.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {
                                };
                                flags.pandocMetadata[metadata5.name] = metadata5.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg = argsStack.shift();
                if (arg) {
                    if (existsSync(arg)) {
                        const metadata6 = readYamlFromMarkdownFile(arg);
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata6
                        };
                    }
                }
                break;
            case "--reference-location":
                arg = argsStack.shift();
                if (arg) {
                    flags[kReferenceLocation] = arg;
                }
                break;
            default:
                arg = argsStack.shift();
                break;
        }
    }
    return flags;
}
function havePandocArg(pandocArgs, arg) {
    return pandocArgs.indexOf(arg) !== -1;
}
function replacePandocArg(pandocArgs, arg, value) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value;
    } else {
        newArgs.push(arg);
        newArgs.push(value);
    }
    return newArgs;
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args, arg, index)=>{
        args.push(arg);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg === "-o" || arg === "--output")) {
            args.push(kStdOut);
        }
        return args;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    let removeNext = false;
    return pandocArgs.reduce((args, arg)=>{
        if (!removeArgs.has(arg)) {
            if (!removeNext) {
                args.push(arg);
            }
            removeNext = false;
        } else {
            removeNext = removeArgs.get(arg);
        }
        return args;
    }, new Array());
}
function removePandocToArg(args) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args, removeArgs);
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args = [];
    for(let i70 = 0; i70 < pandocArgs.length; i70++){
        const arg = pandocArgs[i70];
        if (arg === "--metadata" || arg === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i70 + 1] || "");
            if (flagValue !== undefined && isQuartoMetadata(flagValue.name)) {
                i70++;
            } else {
                args.push(arg);
            }
        } else {
            args.push(arg);
        }
    }
    return args;
}
function parseMetadataFlagValue(arg) {
    const match = arg.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {
        };
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value) {
    if (value) {
        const numbers = value.split(/,/).map((number2)=>parseInt(number2.trim(), 10)
        ).filter((number3)=>!isNaN(number3)
        );
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
const importMeta = {
    url: "https://deno.land/x/deno_dom@v0.1.20-alpha/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder('utf-8');
const encodeString = typeof cachedTextEncoder.encodeInto === 'function' ? function(arg, view) {
    return cachedTextEncoder.encodeInto(arg, view);
} : function(arg, view) {
    const buf = cachedTextEncoder.encode(arg);
    view.set(buf);
    return {
        read: arg.length,
        written: buf.length
    };
};
function passStringToWasm0(arg, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len13 = arg.length;
    let ptr = malloc(len13);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len13; offset++){
        const code29 = arg.charCodeAt(offset);
        if (code29 > 127) break;
        mem[ptr + offset] = code29;
    }
    if (offset !== len13) {
        if (offset !== 0) {
            arg = arg.slice(offset);
        }
        ptr = realloc(ptr, len13, len13 = offset + arg.length * 3);
        const view = getUint8Memory0().subarray(ptr + offset, ptr + len13);
        const ret = encodeString(arg, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder('utf-8', {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len14) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len14));
}
function parse4(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load1(module, imports) {
    if (typeof Response === 'function' && module instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === 'function') {
            try {
                return await WebAssembly.instantiateStreaming(module, imports);
            } catch (e) {
                if (module.headers.get('Content-Type') != 'application/wasm') {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module
            };
        } else {
            return instance;
        }
    }
}
async function init(input) {
    if (typeof input === 'undefined') {
        input = new URL('deno-wasm_bg.wasm', importMeta.url);
    }
    const imports = {
    };
    imports.env = {
        now () {
        }
    };
    input = Uint8Array.from(atob("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