export { parse } from "./_toml/parser.ts";
function joinKeys(keys) {
    return keys
        .map((str) => {
        return str.match(/[^A-Za-z0-9_-]/) ? JSON.stringify(str) : str;
    })
        .join(".");
}
var ArrayType;
(function (ArrayType) {
    ArrayType[ArrayType["ONLY_PRIMITIVE"] = 0] = "ONLY_PRIMITIVE";
    ArrayType[ArrayType["ONLY_OBJECT_EXCLUDING_ARRAY"] = 1] = "ONLY_OBJECT_EXCLUDING_ARRAY";
    ArrayType[ArrayType["MIXED"] = 2] = "MIXED";
})(ArrayType || (ArrayType = {}));
class Dumper {
    maxPad = 0;
    srcObject;
    output = [];
    #arrayTypeCache = new Map();
    constructor(srcObjc) {
        this.srcObject = srcObjc;
    }
    dump(fmtOptions = {}) {
        this.output = this.#printObject(this.srcObject);
        this.output = this.#format(fmtOptions);
        return this.output;
    }
    #printObject(obj, keys = []) {
        const out = [];
        const props = Object.keys(obj);
        const inlineProps = [];
        const multilineProps = [];
        for (const prop of props) {
            if (this.#isSimplySerializable(obj[prop])) {
                inlineProps.push(prop);
            }
            else {
                multilineProps.push(prop);
            }
        }
        const sortedProps = inlineProps.concat(multilineProps);
        for (let i = 0; i < sortedProps.length; i++) {
            const prop = sortedProps[i];
            const value = obj[prop];
            if (value instanceof Date) {
                out.push(this.#dateDeclaration([prop], value));
            }
            else if (typeof value === "string" || value instanceof RegExp) {
                out.push(this.#strDeclaration([prop], value.toString()));
            }
            else if (typeof value === "number") {
                out.push(this.#numberDeclaration([prop], value));
            }
            else if (typeof value === "boolean") {
                out.push(this.#boolDeclaration([prop], value));
            }
            else if (value instanceof Array) {
                const arrayType = this.#getTypeOfArray(value);
                if (arrayType === ArrayType.ONLY_PRIMITIVE) {
                    out.push(this.#arrayDeclaration([prop], value));
                }
                else if (arrayType === ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY) {
                    for (let i = 0; i < value.length; i++) {
                        out.push("");
                        out.push(this.#headerGroup([...keys, prop]));
                        out.push(...this.#printObject(value[i], [...keys, prop]));
                    }
                }
                else {
                    const str = value.map((x) => this.#printAsInlineValue(x)).join(",");
                    out.push(`${this.#declaration([prop])}[${str}]`);
                }
            }
            else if (typeof value === "object") {
                out.push("");
                out.push(this.#header([...keys, prop]));
                if (value) {
                    const toParse = value;
                    out.push(...this.#printObject(toParse, [...keys, prop]));
                }
            }
        }
        out.push("");
        return out;
    }
    #isPrimitive(value) {
        return value instanceof Date ||
            value instanceof RegExp ||
            ["string", "number", "boolean"].includes(typeof value);
    }
    #getTypeOfArray(arr) {
        if (this.#arrayTypeCache.has(arr)) {
            return this.#arrayTypeCache.get(arr);
        }
        const type = this.#doGetTypeOfArray(arr);
        this.#arrayTypeCache.set(arr, type);
        return type;
    }
    #doGetTypeOfArray(arr) {
        if (!arr.length) {
            return ArrayType.ONLY_PRIMITIVE;
        }
        const onlyPrimitive = this.#isPrimitive(arr[0]);
        if (arr[0] instanceof Array) {
            return ArrayType.MIXED;
        }
        for (let i = 1; i < arr.length; i++) {
            if (onlyPrimitive !== this.#isPrimitive(arr[i]) || arr[i] instanceof Array) {
                return ArrayType.MIXED;
            }
        }
        return onlyPrimitive
            ? ArrayType.ONLY_PRIMITIVE
            : ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY;
    }
    #printAsInlineValue(value) {
        if (value instanceof Date) {
            return `"${this.#printDate(value)}"`;
        }
        else if (typeof value === "string" || value instanceof RegExp) {
            return JSON.stringify(value.toString());
        }
        else if (typeof value === "number") {
            return value;
        }
        else if (typeof value === "boolean") {
            return value.toString();
        }
        else if (value instanceof Array) {
            const str = value.map((x) => this.#printAsInlineValue(x)).join(",");
            return `[${str}]`;
        }
        else if (typeof value === "object") {
            if (!value) {
                throw new Error("should never reach");
            }
            const str = Object.keys(value).map((key) => {
                return `${key} = ${this.#printAsInlineValue(value[key])}`;
            }).join(",");
            return `{${str}}`;
        }
        throw new Error("should never reach");
    }
    #isSimplySerializable(value) {
        return (typeof value === "string" ||
            typeof value === "number" ||
            typeof value === "boolean" ||
            value instanceof RegExp ||
            value instanceof Date ||
            (value instanceof Array &&
                this.#getTypeOfArray(value) !== ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY));
    }
    #header(keys) {
        return `[${joinKeys(keys)}]`;
    }
    #headerGroup(keys) {
        return `[[${joinKeys(keys)}]]`;
    }
    #declaration(keys) {
        const title = joinKeys(keys);
        if (title.length > this.maxPad) {
            this.maxPad = title.length;
        }
        return `${title} = `;
    }
    #arrayDeclaration(keys, value) {
        return `${this.#declaration(keys)}${JSON.stringify(value)}`;
    }
    #strDeclaration(keys, value) {
        return `${this.#declaration(keys)}${JSON.stringify(value)}`;
    }
    #numberDeclaration(keys, value) {
        switch (value) {
            case Infinity:
                return `${this.#declaration(keys)}inf`;
            case -Infinity:
                return `${this.#declaration(keys)}-inf`;
            default:
                return `${this.#declaration(keys)}${value}`;
        }
    }
    #boolDeclaration(keys, value) {
        return `${this.#declaration(keys)}${value}`;
    }
    #printDate(value) {
        function dtPad(v, lPad = 2) {
            return v.padStart(lPad, "0");
        }
        const m = dtPad((value.getUTCMonth() + 1).toString());
        const d = dtPad(value.getUTCDate().toString());
        const h = dtPad(value.getUTCHours().toString());
        const min = dtPad(value.getUTCMinutes().toString());
        const s = dtPad(value.getUTCSeconds().toString());
        const ms = dtPad(value.getUTCMilliseconds().toString(), 3);
        const fData = `${value.getUTCFullYear()}-${m}-${d}T${h}:${min}:${s}.${ms}`;
        return fData;
    }
    #dateDeclaration(keys, value) {
        return `${this.#declaration(keys)}${this.#printDate(value)}`;
    }
    #format(options = {}) {
        const { keyAlignment = false } = options;
        const rDeclaration = /^(\".*\"|[^=]*)\s=/;
        const out = [];
        for (let i = 0; i < this.output.length; i++) {
            const l = this.output[i];
            if (l[0] === "[" && l[1] !== "[") {
                if (this.output[i + 1] === "") {
                    i += 1;
                    continue;
                }
                out.push(l);
            }
            else {
                if (keyAlignment) {
                    const m = rDeclaration.exec(l);
                    if (m) {
                        out.push(l.replace(m[1], m[1].padEnd(this.maxPad)));
                    }
                    else {
                        out.push(l);
                    }
                }
                else {
                    out.push(l);
                }
            }
        }
        const cleanedOutput = [];
        for (let i = 0; i < out.length; i++) {
            const l = out[i];
            if (!(l === "" && out[i + 1] === "")) {
                cleanedOutput.push(l);
            }
        }
        return cleanedOutput;
    }
}
export function stringify(srcObj, fmtOptions) {
    return new Dumper(srcObj).dump(fmtOptions).join("\n");
}
//# sourceMappingURL=data:application/json;base64,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