function deferred() {
    let methods;
    const promise = new Promise((resolve, reject)=>{
        methods = {
            resolve,
            reject
        };
    });
    return Object.assign(promise, methods);
}
class MuxAsyncIterator {
    iteratorCount = 0;
    yields = [];
    throws = [];
    signal = deferred();
    add(iterator) {
        ++this.iteratorCount;
        this.callIteratorNext(iterator);
    }
    async callIteratorNext(iterator) {
        try {
            const { value , done  } = await iterator.next();
            if (done) {
                --this.iteratorCount;
            } else {
                this.yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.throws.push(e);
        }
        this.signal.resolve();
    }
    async *iterate() {
        while(this.iteratorCount > 0){
            await this.signal;
            for(let i = 0; i < this.yields.length; i++){
                const { iterator , value  } = this.yields[i];
                yield value;
                this.callIteratorNext(iterator);
            }
            if (this.throws.length) {
                for (const e of this.throws){
                    throw e;
                }
                this.throws.length = 0;
            }
            this.yields.length = 0;
            this.signal = deferred();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
function signal(...signos) {
    const mux = new MuxAsyncIterator();
    if (signos.length < 1) {
        throw new Error("No signals are given. You need to specify at least one signal to create a signal stream.");
    }
    const streams = signos.map(Deno.signal);
    streams.forEach((stream)=>{
        mux.add(stream);
    });
    const dispose = ()=>{
        streams.forEach((stream)=>{
            stream.dispose();
        });
    };
    return Object.assign(mux, {
        dispose
    });
}
function onSignal(signo, callback) {
    const sig = signal(signo);
    (async ()=>{
        for await (const _ of sig){
            callback();
        }
    })();
    return sig;
}
function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i = 0; i <= b.length; i++){
        matrix[i] = [
            i
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i1 = 1; i1 <= b.length; i1++){
        for(let j1 = 1; j1 <= a.length; j1++){
            if (b.charAt(i1 - 1) == a.charAt(j1 - 1)) {
                matrix[i1][j1] = matrix[i1 - 1][j1 - 1];
            } else {
                matrix[i1][j1] = Math.min(matrix[i1 - 1][j1 - 1] + 1, Math.min(matrix[i1][j1 - 1] + 1, matrix[i1 - 1][j1] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str) {
    return str.replace(/-([a-z])/g, (g)=>g[1].toUpperCase()
    );
}
function getOption(flags, name) {
    while(name[0] === "-"){
        name = name.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option, options) {
    const optionNames = options.map((option1)=>[
            option1.name,
            ...option1.aliases ?? []
        ]
    ).flat().map((option1)=>getFlag(option1)
    );
    return didYouMean(" Did you mean option", getFlag(option), optionNames);
}
function didYouMeanType(type, types) {
    return didYouMean(" Did you mean type", type, types);
}
function didYouMean(message, type, types) {
    const match = closest(type, types);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name) {
    if (name.startsWith("-")) {
        return name;
    }
    if (name.length > 1) {
        return `--${name}`;
    }
    return `-${name}`;
}
function isOption(option, name) {
    return option.name === name || option.aliases && option.aliases.indexOf(name) !== -1;
}
function closest(str, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i = 0; i < arr.length; i++){
        const dist = distance(str, arr[i]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message7){
        super(message7);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOption extends FlagsError {
    constructor(option9, options3){
        super(`Unknown required option "${getFlag(option9)}".${didYouMeanOption(option9, options3)}`);
        Object.setPrototypeOf(this, UnknownRequiredOption.prototype);
    }
}
class UnknownConflictingOption extends FlagsError {
    constructor(option1, options1){
        super(`Unknown conflicting option "${getFlag(option1)}".${didYouMeanOption(option1, options1)}`);
        Object.setPrototypeOf(this, UnknownConflictingOption.prototype);
    }
}
class UnknownType extends FlagsError {
    constructor(type2, types1){
        super(`Unknown type "${type2}".${didYouMeanType(type2, types1)}`);
        Object.setPrototypeOf(this, UnknownType.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message1){
        super(message1);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOption extends ValidationError {
    constructor(name17){
        super(`Option "${getFlag(name17).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOption.prototype);
    }
}
class UnknownOption extends ValidationError {
    constructor(option2, options2){
        super(`Unknown option "${getFlag(option2)}".${didYouMeanOption(option2, options2)}`);
        Object.setPrototypeOf(this, UnknownOption.prototype);
    }
}
class MissingOptionValue extends ValidationError {
    constructor(option3){
        super(`Missing value for option "${getFlag(option3)}".`);
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
class InvalidOptionValue extends ValidationError {
    constructor(option4, expected, value2){
        super(`Option "${getFlag(option4)}" must be of type "${expected}", but got "${value2}".`);
        Object.setPrototypeOf(this, InvalidOptionValue.prototype);
    }
}
class OptionNotCombinable extends ValidationError {
    constructor(option5){
        super(`Option "${getFlag(option5)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinable.prototype);
    }
}
class ConflictingOption extends ValidationError {
    constructor(option6, conflictingOption){
        super(`Option "${getFlag(option6)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOption.prototype);
    }
}
class DependingOption extends ValidationError {
    constructor(option7, dependingOption){
        super(`Option "${getFlag(option7)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOption.prototype);
    }
}
class MissingRequiredOption extends ValidationError {
    constructor(option8){
        super(`Missing required option "${getFlag(option8)}".`);
        Object.setPrototypeOf(this, MissingRequiredOption.prototype);
    }
}
class RequiredArgumentFollowsOptionalArgument extends ValidationError {
    constructor(arg3){
        super(`An required argument cannot follow an optional argument, but "${arg3}"  is defined as required.`);
        Object.setPrototypeOf(this, RequiredArgumentFollowsOptionalArgument.prototype);
    }
}
class ArgumentFollowsVariadicArgument extends ValidationError {
    constructor(arg1){
        super(`An argument cannot follow an variadic argument, but got "${arg1}".`);
        Object.setPrototypeOf(this, ArgumentFollowsVariadicArgument.prototype);
    }
}
class NoArguments extends ValidationError {
    constructor(){
        super(`No arguments.`);
        Object.setPrototypeOf(this, NoArguments.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label: label1 , name: name1 , value: value1 , type: type1  }, expected1){
        super(`${label1} "${name1}" must be of type "${type1}", but got "${value1}".` + (expected1 ? ` Expected values: ${expected1.map((value2)=>`"${value2}"`
        ).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValue.prototype);
    }
}
function normalize(args) {
    const normalized = [];
    let inLiteral = false;
    for (const arg2 of args){
        if (inLiteral) {
            normalized.push(arg2);
        } else if (arg2 === "--") {
            inLiteral = true;
            normalized.push(arg2);
        } else if (arg2.length > 1 && arg2[0] === "-") {
            const isLong = arg2[1] === "-";
            const isDotted = !isLong && arg2[2] === ".";
            if (arg2.includes("=")) {
                const parts = arg2.split("=");
                const flag = parts.shift();
                if (isLong) {
                    normalized.push(flag);
                } else {
                    normalizeShortFlags(flag);
                }
                normalized.push(parts.join("="));
            } else if (isLong || isDotted) {
                normalized.push(arg2);
            } else {
                normalizeShortFlags(arg2);
            }
        } else {
            normalized.push(arg2);
        }
    }
    return normalized;
    function normalizeShortFlags(flag) {
        const flags = flag.slice(1).split("");
        if (isNaN(Number(flag[flag.length - 1]))) {
            flags.forEach((val)=>normalized.push(`-${val}`)
            );
        } else {
            normalized.push(`-${flags.shift()}`);
            normalized.push(flags.join(""));
        }
    }
}
var OptionType;
(function(OptionType1) {
    OptionType1["STRING"] = "string";
    OptionType1["NUMBER"] = "number";
    OptionType1["INTEGER"] = "integer";
    OptionType1["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {
}));
const __boolean = (type2)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type2.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type2.value)) {
        return false;
    }
    throw new InvalidTypeError(type2);
};
const number = (type2)=>{
    const value2 = Number(type2.value);
    if (Number.isFinite(value2)) {
        return value2;
    }
    throw new InvalidTypeError(type2);
};
const string = ({ value: value2  })=>{
    return value2;
};
function validateFlags(flags, values, _knownFlaks, allowEmpty, optionNames = {
}) {
    const defaultValues = {
    };
    for (const option9 of flags){
        let name2;
        let defaultValue = undefined;
        if (option9.name.startsWith("no-")) {
            const propName = option9.name.replace(/^no-/, "");
            if (propName in values) {
                continue;
            }
            const positiveOption = getOption(flags, propName);
            if (positiveOption) {
                continue;
            }
            name2 = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name2) {
            name2 = paramCaseToCamelCase(option9.name);
        }
        if (!(name2 in optionNames)) {
            optionNames[name2] = option9.name;
        }
        const hasDefaultValue = typeof values[name2] === "undefined" && (typeof option9.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            values[name2] = getDefaultValue(option9) ?? defaultValue;
            defaultValues[option9.name] = true;
            if (typeof option9.value === "function") {
                values[name2] = option9.value(values[name2]);
            }
        }
    }
    const keys = Object.keys(values);
    if (keys.length === 0 && allowEmpty) {
        return;
    }
    const options3 = keys.map((name2)=>({
            name: name2,
            option: getOption(flags, optionNames[name2])
        })
    );
    for (const { name: name2 , option: option10  } of options3){
        if (!option10) {
            throw new UnknownOption(name2, flags);
        }
        if (option10.standalone) {
            if (keys.length > 1) {
                if (options3.every(({ option: opt  })=>opt && (option10 === opt || defaultValues[opt.name])
                )) {
                    return;
                }
                throw new OptionNotCombinable(option10.name);
            }
            return;
        }
        option10.conflicts?.forEach((flag)=>{
            if (isset(flag, values)) {
                throw new ConflictingOption(option10.name, flag);
            }
        });
        option10.depends?.forEach((flag)=>{
            if (!isset(flag, values) && !defaultValues[option10.name]) {
                throw new DependingOption(option10.name, flag);
            }
        });
        const isArray = (option10.args?.length || 0) > 1;
        option10.args?.forEach((arg2, i)=>{
            if (arg2.requiredValue && (typeof values[name2] === "undefined" || isArray && typeof values[name2][i] === "undefined")) {
                throw new MissingOptionValue(option10.name);
            }
        });
    }
    for (const option11 of flags){
        if (option11.required && !(paramCaseToCamelCase(option11.name) in values)) {
            if ((!option11.conflicts || !option11.conflicts.find((flag)=>!!values[flag]
            )) && !options3.find((opt)=>opt.option?.conflicts?.find((flag)=>flag === option11.name
                )
            )) {
                throw new MissingRequiredOption(option11.name);
            }
        }
    }
    if (keys.length === 0 && !allowEmpty) {
        throw new NoArguments();
    }
}
function isset(flag, values) {
    const name2 = paramCaseToCamelCase(flag);
    return typeof values[name2] !== "undefined";
}
const integer = (type2)=>{
    const value2 = Number(type2.value);
    if (Number.isInteger(value2)) {
        return value2;
    }
    throw new InvalidTypeError(type2);
};
const Types = {
    [OptionType.STRING]: string,
    [OptionType.NUMBER]: number,
    [OptionType.INTEGER]: integer,
    [OptionType.BOOLEAN]: __boolean
};
function parseFlags(args, opts = {
}) {
    !opts.flags && (opts.flags = []);
    const normalized = normalize(args);
    let inLiteral = false;
    let negate = false;
    const flags = {
    };
    const optionNames = {
    };
    let literal = [];
    let unknown = [];
    let stopEarly = null;
    opts.flags.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOption(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOption(flag, opts.flags ?? []);
            }
        });
    });
    for(let i = 0; i < normalized.length; i++){
        let option9;
        let args1;
        const current = normalized[i];
        if (inLiteral) {
            literal.push(current);
            continue;
        }
        if (current === "--") {
            inLiteral = true;
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        const next = ()=>normalized[i + 1]
        ;
        if (isFlag) {
            if (current[2] === "-" || current[1] === "-" && current.length === 3) {
                throw new UnknownOption(current, opts.flags);
            }
            negate = current.startsWith("--no-");
            option9 = getOption(opts.flags, current);
            if (!option9) {
                if (opts.flags.length) {
                    throw new UnknownOption(current, opts.flags);
                }
                option9 = {
                    name: current.replace(/^-+/, ""),
                    optionalValue: true,
                    type: OptionType.STRING
                };
            }
            const positiveName = option9.name.replace(/^no-?/, "");
            const propName = paramCaseToCamelCase(positiveName);
            if (typeof flags[propName] !== "undefined" && !option9.collect) {
                throw new DuplicateOption(current);
            }
            args1 = option9.args?.length ? option9.args : [
                {
                    type: option9.type,
                    requiredValue: option9.requiredValue,
                    optionalValue: option9.optionalValue,
                    variadic: option9.variadic,
                    list: option9.list,
                    separator: option9.separator
                }
            ];
            let argIndex = 0;
            let inOptionalArg = false;
            const previous = flags[propName];
            parseNext(option9, args1);
            if (typeof flags[propName] === "undefined") {
                if (typeof option9.default !== "undefined") {
                    flags[propName] = getDefaultValue(option9);
                } else if (args1[argIndex].requiredValue) {
                    throw new MissingOptionValue(option9.name);
                } else {
                    flags[propName] = true;
                }
            }
            if (option9.value) {
                flags[propName] = option9.value(flags[propName], previous);
            } else if (option9.collect) {
                const value2 = Array.isArray(previous) ? previous : [];
                value2.push(flags[propName]);
                flags[propName] = value2;
            }
            optionNames[propName] = option9.name;
            opts.option?.(option9, flags[propName]);
            function parseNext(option10, args2) {
                const arg2 = args2[argIndex];
                if (!arg2) {
                    const flag = next();
                    throw new UnknownOption(flag, opts.flags ?? []);
                }
                if (!arg2.type) {
                    arg2.type = OptionType.BOOLEAN;
                }
                if (option10.args?.length) {
                    if ((typeof arg2.optionalValue === "undefined" || arg2.optionalValue === false) && typeof arg2.requiredValue === "undefined") {
                        arg2.requiredValue = true;
                    }
                } else {
                    if (arg2.type !== OptionType.BOOLEAN && (typeof arg2.optionalValue === "undefined" || arg2.optionalValue === false) && typeof arg2.requiredValue === "undefined") {
                        arg2.requiredValue = true;
                    }
                }
                if (arg2.requiredValue) {
                    if (inOptionalArg) {
                        throw new RequiredArgumentFollowsOptionalArgument(option10.name);
                    }
                } else {
                    inOptionalArg = true;
                }
                if (negate) {
                    flags[propName] = false;
                    return;
                }
                let result;
                let increase = false;
                if (arg2.list && hasNext(arg2)) {
                    const parsed = next().split(arg2.separator || ",").map((nextValue)=>{
                        const value2 = parseValue(option10, arg2, nextValue);
                        if (typeof value2 === "undefined") {
                            throw new InvalidOptionValue(option10.name, arg2.type ?? "?", nextValue);
                        }
                        return value2;
                    });
                    if (parsed?.length) {
                        result = parsed;
                    }
                } else {
                    if (hasNext(arg2)) {
                        result = parseValue(option10, arg2, next());
                    } else if (arg2.optionalValue && arg2.type === OptionType.BOOLEAN) {
                        result = true;
                    }
                }
                if (increase) {
                    i++;
                    if (!arg2.variadic) {
                        argIndex++;
                    } else if (args2[argIndex + 1]) {
                        throw new ArgumentFollowsVariadicArgument(next());
                    }
                }
                if (typeof result !== "undefined" && (args2.length > 1 || arg2.variadic)) {
                    if (!flags[propName]) {
                        flags[propName] = [];
                    }
                    flags[propName].push(result);
                    if (hasNext(arg2)) {
                        parseNext(option10, args2);
                    }
                } else {
                    flags[propName] = result;
                }
                function hasNext(arg3) {
                    return !!(normalized[i + 1] && (arg3.optionalValue || arg3.requiredValue || arg3.variadic) && (normalized[i + 1][0] !== "-" || arg3.type === OptionType.NUMBER && !isNaN(Number(normalized[i + 1]))) && arg3);
                }
                function parseValue(option11, arg3, value2) {
                    const type2 = arg3.type || OptionType.STRING;
                    const result1 = opts.parse ? opts.parse({
                        label: "Option",
                        type: type2,
                        name: `--${option11.name}`,
                        value: value2
                    }) : parseFlagValue(option11, arg3, value2);
                    if (typeof result1 !== "undefined") {
                        increase = true;
                    }
                    return result1;
                }
            }
        } else {
            if (opts.stopEarly) {
                stopEarly = current;
                break;
            }
            unknown.push(current);
        }
    }
    if (stopEarly) {
        const stopEarlyArgIndex = args.indexOf(stopEarly);
        if (stopEarlyArgIndex !== -1) {
            const doubleDashIndex = args.indexOf("--");
            unknown = args.slice(stopEarlyArgIndex, doubleDashIndex === -1 ? undefined : doubleDashIndex);
            if (doubleDashIndex !== -1) {
                literal = args.slice(doubleDashIndex + 1);
            }
        }
    }
    if (opts.flags?.length) {
        validateFlags(opts.flags, flags, opts.knownFlaks, opts.allowEmpty, optionNames);
    }
    const result = Object.keys(flags).reduce((result1, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result2, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result2[subKey] = flags[key];
                } else {
                    result2[subKey] = result2[subKey] ?? {
                    };
                }
                return result2[subKey];
            }, result1);
        } else {
            result1[key] = flags[key];
        }
        return result1;
    }, {
    });
    return {
        flags: result,
        unknown,
        literal
    };
}
function parseFlagValue(option9, arg2, value2) {
    const type2 = arg2.type || OptionType.STRING;
    const parseType = Types[type2];
    if (!parseType) {
        throw new UnknownType(type2, Object.keys(Types));
    }
    return parseType({
        label: "Option",
        type: type2,
        name: `--${option9.name}`,
        value: value2
    });
}
function getPermissions() {
    return hasPermissions([
        "env",
        "hrtime",
        "net",
        "plugin",
        "read",
        "run",
        "write", 
    ]);
}
function isUnstable() {
    return !!Deno.permissions;
}
function didYouMeanCommand(command, commands, excludes = []) {
    const commandNames = commands.map((command1)=>command1.getName()
    ).filter((command1)=>!excludes.includes(command1)
    );
    return didYouMean(" Did you mean command", command, commandNames);
}
async function hasPermission(permission) {
    try {
        return (await Deno.permissions?.query?.({
            name: permission
        }))?.state === "granted";
    } catch  {
        return false;
    }
}
async function hasPermissions(names) {
    const permissions = {
    };
    await Promise.all(names.map((name2)=>hasPermission(name2).then((hasPermission1)=>permissions[name2] = hasPermission1
        )
    ));
    return permissions;
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition
    };
}
function parseArgumentsDefinition(argsDefinition) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg2 of parts){
        if (hasVariadic) {
            throw new ArgumentFollowsVariadicArgument(arg2);
        }
        const parts1 = arg2.split(ARGUMENT_DETAILS_REGEX);
        const type2 = parts1[2] || OptionType.STRING;
        const details = {
            optionalValue: arg2[0] !== "<",
            name: parts1[1],
            action: parts1[3] || type2,
            variadic: false,
            list: type2 ? arg2.indexOf(type2 + "[]") !== -1 : false,
            type: type2
        };
        if (!details.optionalValue && hasOptional) {
            throw new RequiredArgumentFollowsOptionalArgument(details.name);
        }
        if (arg2[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        if (details.name) {
            argumentDetails.push(details);
        }
    }
    return argumentDetails;
}
const { Deno: Deno1  } = globalThis;
const noColor = typeof Deno1?.noColor === "boolean" ? Deno1.noColor : true;
let enabled = !noColor;
function setColorEnabled(value2) {
    if (noColor) {
        return;
    }
    enabled = value2;
}
function getColorEnabled() {
    return enabled;
}
function code4(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run3(str, code1) {
    return enabled ? `${code1.open}${str.replace(code1.regexp, code1.open)}${code1.close}` : str;
}
function reset(str) {
    return run3(str, code4([
        0
    ], 0));
}
function bold(str) {
    return run3(str, code4([
        1
    ], 22));
}
function dim(str) {
    return run3(str, code4([
        2
    ], 22));
}
function italic(str) {
    return run3(str, code4([
        3
    ], 23));
}
function underline(str) {
    return run3(str, code4([
        4
    ], 24));
}
function inverse(str) {
    return run3(str, code4([
        7
    ], 27));
}
function hidden1(str) {
    return run3(str, code4([
        8
    ], 28));
}
function strikethrough(str) {
    return run3(str, code4([
        9
    ], 29));
}
function black(str) {
    return run3(str, code4([
        30
    ], 39));
}
function red(str) {
    return run3(str, code4([
        31
    ], 39));
}
function green(str) {
    return run3(str, code4([
        32
    ], 39));
}
function yellow(str) {
    return run3(str, code4([
        33
    ], 39));
}
function blue(str) {
    return run3(str, code4([
        34
    ], 39));
}
function magenta(str) {
    return run3(str, code4([
        35
    ], 39));
}
function cyan(str) {
    return run3(str, code4([
        36
    ], 39));
}
function white(str) {
    return run3(str, code4([
        37
    ], 39));
}
function gray(str) {
    return brightBlack(str);
}
function brightBlack(str) {
    return run3(str, code4([
        90
    ], 39));
}
function brightRed(str) {
    return run3(str, code4([
        91
    ], 39));
}
function brightGreen(str) {
    return run3(str, code4([
        92
    ], 39));
}
function brightYellow(str) {
    return run3(str, code4([
        93
    ], 39));
}
function brightBlue(str) {
    return run3(str, code4([
        94
    ], 39));
}
function brightMagenta(str) {
    return run3(str, code4([
        95
    ], 39));
}
function brightCyan(str) {
    return run3(str, code4([
        96
    ], 39));
}
function brightWhite(str) {
    return run3(str, code4([
        97
    ], 39));
}
function bgBlack(str) {
    return run3(str, code4([
        40
    ], 49));
}
function bgRed(str) {
    return run3(str, code4([
        41
    ], 49));
}
function bgGreen(str) {
    return run3(str, code4([
        42
    ], 49));
}
function bgYellow(str) {
    return run3(str, code4([
        43
    ], 49));
}
function bgBlue(str) {
    return run3(str, code4([
        44
    ], 49));
}
function bgMagenta(str) {
    return run3(str, code4([
        45
    ], 49));
}
function bgCyan(str) {
    return run3(str, code4([
        46
    ], 49));
}
function bgWhite(str) {
    return run3(str, code4([
        47
    ], 49));
}
function bgBrightBlack(str) {
    return run3(str, code4([
        100
    ], 49));
}
function bgBrightRed(str) {
    return run3(str, code4([
        101
    ], 49));
}
function bgBrightGreen(str) {
    return run3(str, code4([
        102
    ], 49));
}
function bgBrightYellow(str) {
    return run3(str, code4([
        103
    ], 49));
}
function bgBrightBlue(str) {
    return run3(str, code4([
        104
    ], 49));
}
function bgBrightMagenta(str) {
    return run3(str, code4([
        105
    ], 49));
}
function bgBrightCyan(str) {
    return run3(str, code4([
        106
    ], 49));
}
function bgBrightWhite(str) {
    return run3(str, code4([
        107
    ], 49));
}
function clampAndTruncate(n, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n, max), min));
}
function rgb8(str, color) {
    return run3(str, code4([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str, color) {
    return run3(str, code4([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str, color) {
    if (typeof color === "number") {
        return run3(str, code4([
            38,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 39));
    }
    return run3(str, code4([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 39));
}
function bgRgb24(str, color) {
    if (typeof color === "number") {
        return run3(str, code4([
            48,
            2,
            color >> 16 & 255,
            color >> 8 & 255,
            color & 255
        ], 49));
    }
    return run3(str, code4([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b), 
    ], 49));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor(string1) {
    return string1.replace(ANSI_PATTERN, "");
}
const mod = function() {
    return {
        setColorEnabled: setColorEnabled,
        getColorEnabled: getColorEnabled,
        reset: reset,
        bold: bold,
        dim: dim,
        italic: italic,
        underline: underline,
        inverse: inverse,
        hidden: hidden1,
        strikethrough: strikethrough,
        black: black,
        red: red,
        green: green,
        yellow: yellow,
        blue: blue,
        magenta: magenta,
        cyan: cyan,
        white: white,
        gray: gray,
        brightBlack: brightBlack,
        brightRed: brightRed,
        brightGreen: brightGreen,
        brightYellow: brightYellow,
        brightBlue: brightBlue,
        brightMagenta: brightMagenta,
        brightCyan: brightCyan,
        brightWhite: brightWhite,
        bgBlack: bgBlack,
        bgRed: bgRed,
        bgGreen: bgGreen,
        bgYellow: bgYellow,
        bgBlue: bgBlue,
        bgMagenta: bgMagenta,
        bgCyan: bgCyan,
        bgWhite: bgWhite,
        bgBrightBlack: bgBrightBlack,
        bgBrightRed: bgBrightRed,
        bgBrightGreen: bgBrightGreen,
        bgBrightYellow: bgBrightYellow,
        bgBrightBlue: bgBrightBlue,
        bgBrightMagenta: bgBrightMagenta,
        bgBrightCyan: bgBrightCyan,
        bgBrightWhite: bgBrightWhite,
        rgb8: rgb8,
        bgRgb8: bgRgb8,
        rgb24: rgb24,
        bgRgb24: bgRgb24,
        stripColor: stripColor
    };
}();
class CommandError extends Error {
    constructor(message2){
        super(message2);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    constructor(message3, { exitCode  } = {
    }){
        super(message3);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 1;
    }
}
class DuplicateOptionName extends CommandError {
    constructor(name2){
        super(`Option with name "${getFlag(name2)}" already exists.`);
        Object.setPrototypeOf(this, DuplicateOptionName.prototype);
    }
}
class MissingCommandName extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandName.prototype);
    }
}
class DuplicateCommandName extends CommandError {
    constructor(name3){
        super(`Duplicate command name "${name3}".`);
        Object.setPrototypeOf(this, DuplicateCommandName.prototype);
    }
}
class DuplicateCommandAlias extends CommandError {
    constructor(alias1){
        super(`Duplicate command alias "${alias1}".`);
        Object.setPrototypeOf(this, DuplicateCommandAlias.prototype);
    }
}
class CommandNotFound extends CommandError {
    constructor(name4, commands4, excluded){
        super(`Unknown command "${name4}".${didYouMeanCommand(name4, commands4, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class DuplicateType extends CommandError {
    constructor(name5){
        super(`Type with name "${name5}" already exists.`);
        Object.setPrototypeOf(this, DuplicateType.prototype);
    }
}
class DuplicateCompletion extends CommandError {
    constructor(name6){
        super(`Completion with name "${name6}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletion.prototype);
    }
}
class DuplicateExample extends CommandError {
    constructor(name7){
        super(`Example with name "${name7}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExample.prototype);
    }
}
class DuplicateEnvironmentVariable extends CommandError {
    constructor(name8){
        super(`Environment variable with name "${name8}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvironmentVariable.prototype);
    }
}
class EnvironmentVariableSingleValue extends CommandError {
    constructor(name9){
        super(`An environment variable can only have one value, but "${name9}" has more than one.`);
        Object.setPrototypeOf(this, EnvironmentVariableSingleValue.prototype);
    }
}
class EnvironmentVariableOptionalValue extends CommandError {
    constructor(name10){
        super(`An environment variable cannot have an optional value, but "${name10}" is defined as optional.`);
        Object.setPrototypeOf(this, EnvironmentVariableOptionalValue.prototype);
    }
}
class EnvironmentVariableVariadicValue extends CommandError {
    constructor(name11){
        super(`An environment variable cannot have an variadic value, but "${name11}" is defined as variadic.`);
        Object.setPrototypeOf(this, EnvironmentVariableVariadicValue.prototype);
    }
}
class DefaultCommandNotFound extends CommandError {
    constructor(name12, commands1){
        super(`Default command "${name12}" not found.${didYouMeanCommand(name12, commands1)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFound.prototype);
    }
}
class CommandExecutableNotFound extends CommandError {
    constructor(name13, files){
        super(`Command executable not found: ${name13}:\n    - ${files.join("\\n    - ")}`);
        Object.setPrototypeOf(this, CommandExecutableNotFound.prototype);
    }
}
class UnknownCompletionCommand extends CommandError {
    constructor(name14, commands2){
        super(`Auto-completion failed. Unknown command "${name14}".${didYouMeanCommand(name14, commands2)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommand.prototype);
    }
}
class UnknownCommand extends ValidationError1 {
    constructor(name15, commands3, excluded1){
        super(`Unknown command "${name15}".${didYouMeanCommand(name15, commands3, excluded1)}`);
        Object.setPrototypeOf(this, UnknownCommand.prototype);
    }
}
class NoArgumentsAllowed extends ValidationError1 {
    constructor(name16){
        super(`No arguments allowed for command "${name16}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowed.prototype);
    }
}
class MissingArguments extends ValidationError1 {
    constructor(args2){
        super("Missing argument(s): " + args2.join(", "));
        Object.setPrototypeOf(this, MissingArguments.prototype);
    }
}
class MissingArgument extends ValidationError1 {
    constructor(arg2){
        super(`Missing argument "${arg2}".`);
        Object.setPrototypeOf(this, MissingArgument.prototype);
    }
}
class TooManyArguments extends ValidationError1 {
    constructor(args1){
        super(`Too many arguments: ${args1.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArguments.prototype);
    }
}
class Type {
}
class BooleanType extends Type {
    parse(type) {
        return __boolean(type);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class NumberType extends Type {
    parse(type) {
        return number(type);
    }
}
class StringType extends Type {
    parse(type) {
        return string(type);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    value;
    options = {
    };
    get length() {
        return this.toString().length;
    }
    static from(value) {
        const cell = new this(value);
        if (value instanceof Cell) {
            cell.options = {
                ...value.options
            };
        }
        return cell;
    }
    constructor(value3){
        this.value = value3;
    }
    toString() {
        return this.value.toString();
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        const cell = new Cell(value ?? this);
        cell.options = {
            ...this.options
        };
        return cell;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class Row extends Array {
    options = {
    };
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell
        ));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder()
        );
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
function consumeWords(length, content) {
    let consumed = "";
    const words = content.split(/ /g);
    for(let i = 0; i < words.length; i++){
        let word = words[i];
        const hasLineBreak = word.indexOf("\n") !== -1;
        if (hasLineBreak) {
            word = word.split("\n").shift();
        }
        if (consumed) {
            const nextLength = stripColor(word).length;
            const consumedLength = stripColor(consumed).length;
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i > 0 ? " " : "") + word;
        if (hasLineBreak) {
            break;
        }
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    return Math.max(...rows.map((row)=>(row[index] instanceof Cell && row[index].getColSpan() > 1 ? "" : row[index]?.toString() || "").split("\n").map((r)=>{
            const str = typeof maxWidth === "undefined" ? r : consumeWords(maxWidth, r);
            return stripColor(str).length || 0;
        })
    ).flat());
}
class TableLayout {
    table;
    options;
    constructor(table, options4){
        this.table = table;
        this.options = options4;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const header = this.table.getHeader();
        const rows = this.spanRows(header ? [
            header,
            ...this.table
        ] : this.table.slice());
        const columns = Math.max(...rows.map((row)=>row.length
        ));
        for (const row of rows){
            const length = row.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i = 0; i < diff; i++){
                    row.push(this.createCell(null, row));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const minColWidth = Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth;
            const maxColWidth = Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth;
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding;
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    spanRows(_rows, rowIndex = 0, colIndex = 0, rowSpan = [], colSpan = 1) {
        const rows = _rows;
        if (rowIndex >= rows.length && rowSpan.every((span)=>span === 1
        )) {
            return rows;
        } else if (rows[rowIndex] && colIndex >= rows[rowIndex].length && colIndex >= rowSpan.length && colSpan === 1) {
            return this.spanRows(rows, ++rowIndex, 0, rowSpan, 1);
        }
        if (colSpan > 1) {
            colSpan--;
            rowSpan[colIndex] = rowSpan[colIndex - 1];
            rows[rowIndex].splice(colIndex - 1, 0, rows[rowIndex][colIndex - 1]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        if (colIndex === 0) {
            rows[rowIndex] = this.createRow(rows[rowIndex] || []);
        }
        if (rowSpan[colIndex] > 1) {
            rowSpan[colIndex]--;
            rows[rowIndex].splice(colIndex, 0, rows[rowIndex - 1][colIndex]);
            return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
        }
        rows[rowIndex][colIndex] = this.createCell(rows[rowIndex][colIndex] || null, rows[rowIndex]);
        colSpan = rows[rowIndex][colIndex].getColSpan();
        rowSpan[colIndex] = rows[rowIndex][colIndex].getRowSpan();
        return this.spanRows(rows, rowIndex, ++colIndex, rowSpan, colSpan);
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row) {
        return Cell.from(cell ?? "").border(row.getBorder(), false).align(row.getAlign(), false);
    }
    renderRows(opts) {
        let result = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result += this.options.chars.right;
            } else if (opts.hasBorder) {
                result += " ";
            }
        }
        result += "\n";
        if (isMultilineRow) {
            return result + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder) {
            result += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current , next  } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next);
        if (opts.hasBorder) {
            result += " ".repeat(opts.padding[colIndex]);
        }
        result += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result += " ".repeat(opts.padding[colIndex]);
        }
        return result;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, stripColor(cell.toString()).length);
        let words = consumeWords(length, cell.toString());
        const breakWord = stripColor(words).length > length;
        if (breakWord) {
            words = words.slice(0, length);
        }
        const next = cell.toString().slice(words.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - stripColor(words).length;
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words;
        } else if (align === "left") {
            current = words + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result.length ? " ".repeat(this.options.indent) + result + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1
        ;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1
        ;
        let result = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result += this.options.chars.left;
                } else {
                    result += " ";
                }
            } else if (b1Border && b2Border) {
                result += this.options.chars.leftMid;
            } else if (b1Border) {
                result += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result += this.options.chars.topLeft;
            } else {
                result += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result += this.options.chars.rightMid;
                } else {
                    result += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result += this.options.chars.bottom;
                } else {
                    result += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result += this.options.chars.left;
                } else {
                    result += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result += this.options.chars.top;
                } else {
                    result += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result += this.options.chars.right;
                } else {
                    result += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result += this.options.chars.bottomRight;
            } else if (b1Border) {
                result += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result += this.options.chars.topRight;
            } else if (b2Border) {
                result += this.options.chars.topLeft;
            } else {
                result += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result += this.options.chars.right;
                } else {
                    result += " ";
                }
                return result;
            }
        } else if (b1Border && b2Border) {
            result += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result += this.options.chars.top.repeat(length);
        } else {
            result += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result += this.options.chars.rightMid;
            } else if (b1Border) {
                result += this.options.chars.bottomRight;
            } else if (b2Border) {
                result += this.options.chars.topRight;
            } else {
                result += " ";
            }
        }
        return result;
    }
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        }
    };
    headerRow;
    static from(rows) {
        const table1 = new this(...rows);
        if (rows instanceof Table) {
            table1.options = {
                ...rows.options
            };
            table1.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table1;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)
        ));
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table1 = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()
        ));
        table1.options = {
            ...this.options
        };
        table1.headerRow = this.headerRow?.clone();
        return table1;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.toString() + "\n"));
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder : false
            )
        );
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class HelpGenerator {
    cmd;
    indent = 2;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd1, options5 = {
    }){
        this.cmd = cmd1;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            ...options5
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result = this.generateHeader() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples() + "\n";
        setColorEnabled(areColorsEnabled);
        return result;
    }
    generateHeader() {
        const rows = [
            [
                bold("Usage:"),
                magenta(`${this.cmd.getPath()}${this.cmd.getArgsDefinition() ? " " + this.cmd.getArgsDefinition() : ""}`), 
            ], 
        ];
        const version = this.cmd.getVersion();
        if (version) {
            rows.push([
                bold("Version:"),
                yellow(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).indent(this.indent).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                this.cmd.getDescription()
            ], 
        ]).indent(this.indent * 2).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options6 = this.cmd.getOptions(false);
        if (!options6.length) {
            return "";
        }
        const hasTypeDefinitions = !!options6.find((option9)=>!!option9.typeDefinition
        );
        if (hasTypeDefinitions) {
            return this.label("Options") + Table.from([
                ...options6.map((option9)=>[
                        option9.flags.map((flag)=>blue(flag)
                        ).join(", "),
                        highlightArguments(option9.typeDefinition || "", this.options.types),
                        red(bold("-")) + " " + option9.description.split("\n").shift(),
                        this.generateHints(option9), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).maxColWidth([
                60,
                60,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label("Options") + Table.from([
            ...options6.map((option9)=>[
                    option9.flags.map((flag)=>blue(flag)
                    ).join(", "),
                    red(bold("-")) + " " + option9.description.split("\n").shift(),
                    this.generateHints(option9), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).maxColWidth([
            60,
            80,
            60
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands4 = this.cmd.getCommands(false);
        if (!commands4.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands4.find((command)=>!!command.getArgsDefinition()
        );
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands4.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name17)=>blue(name17)
                        ).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                    ]
                ), 
            ]).padding([
                2,
                2,
                2
            ]).indent(this.indent * 2).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands4.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name17)=>blue(name17)
                    ).join(", "),
                    red(bold("-")) + " " + command.getDescription().split("\n").shift(), 
                ]
            ), 
        ]).padding([
            2,
            2
        ]).indent(this.indent * 2).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name17)=>blue(name17)
                    ).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    `${red(bold("-"))} ${envVar.description}`, 
                ]
            ), 
        ]).padding(2).indent(this.indent * 2).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim(bold(`${capitalize(example.name)}:`)),
                example.description, 
            ]
        )).padding(1).indent(this.indent * 2).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow(`required`));
        typeof option.default !== "undefined" && hints.push(bold(`Default: `) + inspect(option.default, this.options.colors));
        option.depends?.length && hints.push(yellow(bold(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red(bold(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type3 = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type3 instanceof Type) {
            const possibleValues = type3.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold(`Values: `) + possibleValues.map((value4)=>inspect(value4, this.options.colors)
                ).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + " ".repeat(this.indent) + bold(`${label}:`) + "\n\n";
    }
}
function capitalize(string1) {
    return (string1?.charAt(0).toUpperCase() + string1.slice(1)) ?? "";
}
function inspect(value4, colors) {
    return Deno.inspect(value4, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types1 = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition).map((arg3)=>highlightArgumentDetails(arg3, types1)
    ).join(" ");
}
function highlightArgumentDetails(arg3, types1 = true) {
    let str = "";
    str += yellow(arg3.optionalValue ? "[" : "<");
    let name17 = "";
    name17 += arg3.name;
    if (arg3.variadic) {
        name17 += "...";
    }
    name17 = magenta(name17);
    str += name17;
    if (types1) {
        str += yellow(":");
        str += red(arg3.type);
    }
    if (arg3.list) {
        str += green("[]");
    }
    str += yellow(arg3.optionalValue ? "]" : ">");
    return str;
}
class IntegerType extends Type {
    parse(type) {
        return integer(type);
    }
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    fn;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = true;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOption;
    _helpOption;
    _help;
    versionOption(flags, desc, opts) {
        this._versionOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOption = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        const result = splitArguments(nameAndArguments);
        const name17 = result.flags.shift();
        const aliases = result.flags;
        if (!name17) {
            throw new MissingCommandName();
        }
        if (this.getBaseCommand(name17, true)) {
            if (!override) {
                throw new DuplicateCommandName(name17);
            }
            this.removeCommand(name17);
        }
        let description;
        let cmd2;
        if (typeof cmdOrDescription === "string") {
            description = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd2 = cmdOrDescription.reset();
        } else {
            cmd2 = new Command();
        }
        cmd2._name = name17;
        cmd2._parent = this;
        if (description) {
            cmd2.description(description);
        }
        if (result.typeDefinition) {
            cmd2.arguments(result.typeDefinition);
        }
        aliases.forEach((alias1)=>cmd2.alias(alias1)
        );
        this.commands.set(name17, cmd2);
        this.select(name17);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAlias(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this.cmd = this;
        return this;
    }
    select(name) {
        const cmd2 = this.getBaseCommand(name, true);
        if (!cmd2) {
            throw new CommandNotFound(name, this.getBaseCommands(true));
        }
        this.cmd = cmd2;
        return this;
    }
    name(name) {
        this.cmd._name = name;
        return this;
    }
    version(version) {
        if (typeof version === "string") {
            this.cmd.ver = ()=>version
            ;
        } else if (typeof version === "function") {
            this.cmd.ver = version;
        }
        return this;
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help
            ;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd2)=>HelpGenerator.generate(cmd2, help)
            ;
        }
        return this;
    }
    description(description) {
        this.cmd.desc = description;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.fn = fn;
        return this;
    }
    allowEmpty(allowEmpty = true) {
        this.cmd._allowEmpty = allowEmpty;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name) {
        this.cmd.defaultCommand = name;
        return this;
    }
    globalType(name, type, options) {
        return this.type(name, type, {
            ...options,
            global: true
        });
    }
    type(name, handler, options) {
        if (this.cmd.types.get(name) && !options?.override) {
            throw new DuplicateType(name);
        }
        this.cmd.types.set(name, {
            ...options,
            name,
            handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd2, parent)=>handler.complete?.(cmd2, parent) || []
            ;
            this.complete(name, completeHandler, options);
        }
        return this;
    }
    globalComplete(name, complete, options) {
        return this.complete(name, complete, {
            ...options,
            global: true
        });
    }
    complete(name, complete, options) {
        if (this.cmd.completions.has(name) && !options?.override) {
            throw new DuplicateCompletion(name);
        }
        this.cmd.completions.set(name, {
            name,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    shouldThrowErrors() {
        return this.cmd.throwOnError || !!this.cmd._parent?.shouldThrowErrors();
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts
            });
        }
        const result = splitArguments(flags);
        const args3 = result.typeDefinition ? parseArgumentsDefinition(result.typeDefinition) : [];
        const option10 = {
            ...opts,
            name: "",
            description: desc,
            args: args3,
            flags: result.flags,
            typeDefinition: result.typeDefinition
        };
        if (option10.separator) {
            for (const arg3 of args3){
                if (arg3.list) {
                    arg3.separator = option10.separator;
                }
            }
        }
        for (const part of option10.flags){
            const arg3 = part.trim();
            const isLong = /^--/.test(arg3);
            const name18 = isLong ? arg3.slice(2) : arg3.slice(1);
            if (this.cmd.getBaseOption(name18, true)) {
                if (opts?.override) {
                    this.removeOption(name18);
                } else {
                    throw new DuplicateOptionName(name18);
                }
            }
            if (!option10.name && isLong) {
                option10.name = name18;
            } else if (!option10.aliases) {
                option10.aliases = [
                    name18
                ];
            } else {
                option10.aliases.push(name18);
            }
        }
        if (option10.prepend) {
            this.cmd.options.unshift(option10);
        } else {
            this.cmd.options.push(option10);
        }
        return this;
    }
    example(name, description) {
        if (this.cmd.hasExample(name)) {
            throw new DuplicateExample(name);
        }
        this.cmd.examples.push({
            name,
            description
        });
        return this;
    }
    globalEnv(name, description, options) {
        return this.env(name, description, {
            ...options,
            global: true
        });
    }
    env(name, description, options) {
        const result = splitArguments(name);
        if (!result.typeDefinition) {
            result.typeDefinition = "<value:boolean>";
        }
        if (result.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true)
        )) {
            throw new DuplicateEnvironmentVariable(name);
        }
        const details = parseArgumentsDefinition(result.typeDefinition);
        if (details.length > 1) {
            throw new EnvironmentVariableSingleValue(name);
        } else if (details.length && details[0].optionalValue) {
            throw new EnvironmentVariableOptionalValue(name);
        } else if (details.length && details[0].variadic) {
            throw new EnvironmentVariableVariadicValue(name);
        }
        this.cmd.envVars.push({
            name: result.flags[0],
            names: result.flags,
            description,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    async parse(args = Deno.args, dry) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = args;
            const subCommand = args.length > 0 && this.getCommand(args[0], true);
            if (subCommand) {
                subCommand._globalParent = this;
                return await subCommand.parse(this.rawArgs.slice(1), dry);
            }
            const result = {
                options: {
                },
                args: this.rawArgs,
                cmd: this,
                literal: this.literalArgs
            };
            if (this.isExecutable) {
                if (!dry) {
                    await this.executeExecutable(this.rawArgs);
                }
                return result;
            } else if (this._useRawArgs) {
                if (dry) {
                    return result;
                }
                return await this.execute({
                }, ...this.rawArgs);
            } else {
                const { action , flags , unknown , literal  } = this.parseFlags(this.rawArgs);
                this.literalArgs = literal;
                const params = this.parseArguments(unknown, flags);
                await this.validateEnvVars();
                if (dry || action) {
                    if (action) {
                        await action.call(this, flags, ...params);
                    }
                    return {
                        options: flags,
                        args: params,
                        cmd: this,
                        literal: this.literalArgs
                    };
                }
                return await this.execute(flags, ...params);
            }
        } catch (error) {
            throw this.error(error);
        }
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        if (!this._help) {
            this.help({
                hints: true,
                types: false
            });
        }
        if (this._versionOption !== false && (this._versionOption || this.ver)) {
            this.option(this._versionOption?.flags || "-V, --version", this._versionOption?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: function() {
                    this.showVersion();
                    Deno.exit(0);
                },
                ...this._versionOption?.opts ?? {
                }
            });
        }
        if (this._helpOption !== false) {
            this.option(this._helpOption?.flags || "-h, --help", this._helpOption?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: function() {
                    this.showHelp();
                    Deno.exit(0);
                },
                ...this._helpOption?.opts ?? {
                }
            });
        }
        return this;
    }
    async execute(options, ...args) {
        if (this.fn) {
            await this.fn(options, ...args);
        } else if (this.defaultCommand) {
            const cmd2 = this.getCommand(this.defaultCommand, true);
            if (!cmd2) {
                throw new DefaultCommandNotFound(this.defaultCommand, this.getCommands());
            }
            cmd2._globalParent = this;
            await cmd2.execute(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeExecutable(args) {
        const permissions = await getPermissions();
        if (!permissions.read) {
            await Deno.permissions?.request({
                name: "read"
            });
        }
        if (!permissions.run) {
            await Deno.permissions?.request({
                name: "run"
            });
        }
        const [main, ...names] = this.getPath().split(" ");
        names.unshift(main.replace(/\.ts$/, ""));
        const executableName = names.join("-");
        const files1 = [];
        const parts = Deno.mainModule.replace(/^file:\/\//g, "").split("/");
        if (Deno.build.os === "windows" && parts[0] === "") {
            parts.shift();
        }
        parts.pop();
        const path = parts.join("/");
        files1.push(path + "/" + executableName, path + "/" + executableName + ".ts");
        files1.push(executableName, executableName + ".ts");
        const denoOpts = [];
        if (isUnstable()) {
            denoOpts.push("--unstable");
        }
        denoOpts.push("--allow-read", "--allow-run");
        Object.keys(permissions).forEach((name18)=>{
            if (name18 === "read" || name18 === "run") {
                return;
            }
            if (permissions[name18]) {
                denoOpts.push(`--allow-${name18}`);
            }
        });
        for (const file of files1){
            try {
                Deno.lstatSync(file);
            } catch (error) {
                if (error instanceof Deno.errors.NotFound) {
                    continue;
                }
                throw error;
            }
            const cmd2 = [
                "deno",
                "run",
                ...denoOpts,
                file,
                ...args
            ];
            const process = Deno.run({
                cmd: cmd2
            });
            const status = await process.status();
            if (!status.success) {
                Deno.exit(status.code);
            }
            return;
        }
        throw new CommandExecutableNotFound(executableName, files1);
    }
    parseFlags(args) {
        try {
            let action;
            const result = parseFlags(args, {
                stopEarly: this._stopEarly,
                allowEmpty: this._allowEmpty,
                flags: this.getOptions(true),
                parse: (type3)=>this.parseType(type3)
                ,
                option: (option10)=>{
                    if (!action && option10.action) {
                        action = option10.action;
                    }
                }
            });
            return {
                ...result,
                action
            };
        } catch (error) {
            if (error instanceof ValidationError) {
                throw new ValidationError1(error.message);
            }
            throw error;
        }
    }
    parseType(type) {
        const typeSettings = this.getType(type.type);
        if (!typeSettings) {
            throw new UnknownType(type.type, this.getTypes().map((type3)=>type3.name
            ));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type) : typeSettings.handler(type);
    }
    async validateEnvVars() {
        if (!await hasPermission("env")) {
            return;
        }
        const envVars = this.getEnvVars(true);
        if (!envVars.length) {
            return;
        }
        envVars.forEach((env)=>{
            const name18 = env.names.find((name19)=>!!Deno.env.get(name19)
            );
            if (name18) {
                this.parseType({
                    label: "Environment variable",
                    type: env.type,
                    name: name18,
                    value: Deno.env.get(name18) ?? ""
                });
            }
        });
    }
    parseArguments(args, flags) {
        const params = [];
        args = args.slice(0);
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    throw new UnknownCommand(args[0], this.getCommands());
                } else {
                    throw new NoArgumentsAllowed(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required = this.getArguments().filter((expectedArg)=>!expectedArg.optionalValue
                ).map((expectedArg)=>expectedArg.name
                );
                if (required.length) {
                    const flagNames = Object.keys(flags);
                    const hasStandaloneOption = !!flagNames.find((name18)=>this.getOption(name18, true)?.standalone
                    );
                    if (!hasStandaloneOption) {
                        throw new MissingArguments(required);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optionalValue) {
                            break;
                        }
                        throw new MissingArgument(`Missing argument: ${expectedArg.name}`);
                    }
                    let arg3;
                    if (expectedArg.variadic) {
                        arg3 = args.splice(0, args.length).map((value4)=>this.parseType({
                                label: "Argument",
                                type: expectedArg.type,
                                name: expectedArg.name,
                                value: value4
                            })
                        );
                    } else {
                        arg3 = this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value: args.shift()
                        });
                    }
                    if (arg3) {
                        params.push(arg3);
                    }
                }
                if (args.length) {
                    throw new TooManyArguments(args);
                }
            }
        }
        return params;
    }
    error(error) {
        if (this.shouldThrowErrors() || !(error instanceof ValidationError1)) {
            return error;
        }
        this.showHelp();
        Deno.stderr.writeSync(new TextEncoder().encode(red(`  ${bold("error")}: ${error.message}\n`) + "\n"));
        Deno.exit(error instanceof ValidationError1 ? error.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath() {
        return this._parent ? this._parent.getPath() + " " + this._name : this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name) {
        return this.getArguments().find((arg3)=>arg3.name === name
        );
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getShortDescription() {
        return this.getDescription().trim().split("\n").shift();
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getVersion()));
    }
    showHelp() {
        Deno.stdout.writeSync(new TextEncoder().encode(this.getHelp()));
    }
    getHelp() {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this);
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    hasOptions(hidden) {
        return this.getOptions(hidden).length > 0;
    }
    getOptions(hidden) {
        return this.getGlobalOptions(hidden).concat(this.getBaseOptions(hidden));
    }
    getBaseOptions(hidden) {
        if (!this.options.length) {
            return [];
        }
        return hidden ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden
        );
    }
    getGlobalOptions(hidden) {
        const getOptions = (cmd2, options6 = [], names = [])=>{
            if (cmd2) {
                if (cmd2.options.length) {
                    cmd2.options.forEach((option10)=>{
                        if (option10.global && !this.options.find((opt)=>opt.name === option10.name
                        ) && names.indexOf(option10.name) === -1 && (hidden || !option10.hidden)) {
                            names.push(option10.name);
                            options6.push(option10);
                        }
                    });
                }
                return getOptions(cmd2._parent, options6, names);
            }
            return options6;
        };
        return getOptions(this._parent);
    }
    hasOption(name, hidden) {
        return !!this.getOption(name, hidden);
    }
    getOption(name, hidden) {
        return this.getBaseOption(name, hidden) ?? this.getGlobalOption(name, hidden);
    }
    getBaseOption(name, hidden) {
        const option10 = this.options.find((option11)=>option11.name === name
        );
        return option10 && (hidden || !option10.hidden) ? option10 : undefined;
    }
    getGlobalOption(name, hidden) {
        if (!this._parent) {
            return;
        }
        const option10 = this._parent.getBaseOption(name, hidden);
        if (!option10 || !option10.global) {
            return this._parent.getGlobalOption(name, hidden);
        }
        return option10;
    }
    removeOption(name) {
        const index = this.options.findIndex((option10)=>option10.name === name
        );
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden) {
        return this.getCommands(hidden).length > 0;
    }
    getCommands(hidden) {
        return this.getGlobalCommands(hidden).concat(this.getBaseCommands(hidden));
    }
    getBaseCommands(hidden) {
        const commands5 = Array.from(this.commands.values());
        return hidden ? commands5 : commands5.filter((cmd2)=>!cmd2.isHidden
        );
    }
    getGlobalCommands(hidden) {
        const getCommands = (cmd2, commands5 = [], names = [])=>{
            if (cmd2) {
                if (cmd2.commands.size) {
                    cmd2.commands.forEach((cmd3)=>{
                        if (cmd3.isGlobal && this !== cmd3 && !this.commands.has(cmd3._name) && names.indexOf(cmd3._name) === -1 && (hidden || !cmd3.isHidden)) {
                            names.push(cmd3._name);
                            commands5.push(cmd3);
                        }
                    });
                }
                return getCommands(cmd2._parent, commands5, names);
            }
            return commands5;
        };
        return getCommands(this._parent);
    }
    hasCommand(name, hidden) {
        return !!this.getCommand(name, hidden);
    }
    getCommand(name, hidden) {
        return this.getBaseCommand(name, hidden) ?? this.getGlobalCommand(name, hidden);
    }
    getBaseCommand(name, hidden) {
        for (const cmd2 of this.commands.values()){
            if (cmd2._name === name || cmd2.aliases.includes(name)) {
                return cmd2 && (hidden || !cmd2.isHidden) ? cmd2 : undefined;
            }
        }
    }
    getGlobalCommand(name, hidden) {
        if (!this._parent) {
            return;
        }
        const cmd2 = this._parent.getBaseCommand(name, hidden);
        if (!cmd2?.isGlobal) {
            return this._parent.getGlobalCommand(name, hidden);
        }
        return cmd2;
    }
    removeCommand(name) {
        const command = this.getBaseCommand(name, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd2, types2 = [], names = [])=>{
            if (cmd2) {
                if (cmd2.types.size) {
                    cmd2.types.forEach((type3)=>{
                        if (type3.global && !this.types.has(type3.name) && names.indexOf(type3.name) === -1) {
                            names.push(type3.name);
                            types2.push(type3);
                        }
                    });
                }
                return getTypes(cmd2._parent, types2, names);
            }
            return types2;
        };
        return getTypes(this._parent);
    }
    getType(name) {
        return this.getBaseType(name) ?? this.getGlobalType(name);
    }
    getBaseType(name) {
        return this.types.get(name);
    }
    getGlobalType(name) {
        if (!this._parent) {
            return;
        }
        const cmd2 = this._parent.getBaseType(name);
        if (!cmd2?.global) {
            return this._parent.getGlobalType(name);
        }
        return cmd2;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd2, completions = [], names = [])=>{
            if (cmd2) {
                if (cmd2.completions.size) {
                    cmd2.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd2._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name) {
        return this.getBaseCompletion(name) ?? this.getGlobalCompletion(name);
    }
    getBaseCompletion(name) {
        return this.completions.get(name);
    }
    getGlobalCompletion(name) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name);
        }
        return completion;
    }
    hasEnvVars(hidden) {
        return this.getEnvVars(hidden).length > 0;
    }
    getEnvVars(hidden) {
        return this.getGlobalEnvVars(hidden).concat(this.getBaseEnvVars(hidden));
    }
    getBaseEnvVars(hidden) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden
        );
    }
    getGlobalEnvVars(hidden) {
        const getEnvVars = (cmd2, envVars = [], names = [])=>{
            if (cmd2) {
                if (cmd2.envVars.length) {
                    cmd2.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]
                        ) && names.indexOf(envVar.names[0]) === -1 && (hidden || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd2._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name, hidden) {
        return !!this.getEnvVar(name, hidden);
    }
    getEnvVar(name, hidden) {
        return this.getBaseEnvVar(name, hidden) ?? this.getGlobalEnvVar(name, hidden);
    }
    getBaseEnvVar(name, hidden) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name) !== -1
        );
        return envVar && (hidden || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name, hidden) {
        if (!this._parent) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name, hidden);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name, hidden);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name) {
        return !!this.getExample(name);
    }
    getExample(name) {
        return this.examples.find((example)=>example.name === name
        );
    }
}
class BashCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new BashCompletionsGenerator(cmd).generate();
    }
    constructor(cmd2){
        this.cmd = cmd2;
    }
    generate() {
        const path = this.cmd.getPath();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path}${version}

_${replaceSpecialChars1(path)}() {
  local word cur prev
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()

  _${replaceSpecialChars1(this.cmd.getName())}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.cmd.getName()} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  ${this.generateCompletions(this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars1(path)} -o bashdefault -o default ${path}
`;
    }
    generateCompletions(command, path = "", index = 1) {
        path = (path ? path + " " : "") + command.getName();
        const commandCompletions = this.generateCommandCompletions(command, path, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path, index + 1)
        ).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName()
        );
        const completionsPath = ~path.indexOf(" ") ? " " + path.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command.getArguments(), completionsPath);
        return `  __${replaceSpecialChars1(path)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option10)=>option10.flags
        ).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options6 = command.getOptions(false);
        if (options6.length) {
            opts += 'case "${prev}" in';
            for (const option10 of options6){
                const flags = option10.flags.map((flag)=>flag.trim()
                ).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(option10.args, completionsPath, {
                    standalone: option10.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(args, path) {
        if (args.length) {
            return `_${replaceSpecialChars1(this.cmd.getName())}_complete ${args[0].action}${path}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(args, path, opts) {
        if (args.length) {
            return `opts=(); _${replaceSpecialChars1(this.cmd.getName())}_complete ${args[0].action}${path}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
}
function replaceSpecialChars1(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd3){
        super();
        this.#cmd = cmd3;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim(italic("~/.bashrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(BashCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd4){
        super();
        this.description("Get completions for given action from given command.").arguments("<action:string> [command...:string]").action(async (_, action, commandNames)=>{
            let parent;
            const completeCommand = commandNames?.reduce((cmd5, name18)=>{
                parent = cmd5;
                const childCmd = cmd5.getCommand(name18, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommand(name18, cmd5.getCommands());
                }
                return childCmd;
            }, cmd4 || this.getMainCommand()) ?? (cmd4 || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result = await completion?.complete(completeCommand, parent) ?? [];
            if (result?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result.join("\n")));
            }
        }).reset();
    }
}
class FishCompletionsGenerator {
    cmd;
    static generate(cmd) {
        return new FishCompletionsGenerator(cmd).generate();
    }
    constructor(cmd5){
        this.cmd = cmd5;
    }
    generate() {
        const path = this.cmd.getPath();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path}${version}

function __fish_${replaceSpecialChars2(this.cmd.getName())}_using_command
  set cmds ${getCommandFnNames(this.cmd).join(" ")}
  set words (commandline -opc)
  set cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if [ "$cmd" = "$argv[1]" ]
    return 0
  end
  return 1
end

${this.generateCompletions(this.cmd).trim()}
`;
    }
    generateCompletions(command) {
        const parent = command.getParent();
        let result = ``;
        if (parent) {
            result += "\n" + this.complete(parent, {
                description: command.getShortDescription(),
                arguments: command.getName()
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(commandArgs[0].action + " " + getCompletionsPath(command)) : undefined
            });
        }
        for (const option10 of command.getOptions(false)){
            result += "\n" + this.completeOption(command, option10);
        }
        for (const subCommand of command.getCommands(false)){
            result += this.generateCompletions(subCommand);
        }
        return result;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2
        )?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2
        )?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: option.description,
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(option.args[0].action + " " + getCompletionsPath(command)) : undefined
        });
    }
    complete(command, options) {
        const cmd6 = [
            "complete"
        ];
        cmd6.push("-c", this.cmd.getName());
        cmd6.push("-n", `'__fish_${replaceSpecialChars2(this.cmd.getName())}_using_command __${replaceSpecialChars2(command.getPath())}'`);
        options.shortOption && cmd6.push("-s", options.shortOption);
        options.longOption && cmd6.push("-l", options.longOption);
        options.standalone && cmd6.push("-x");
        cmd6.push("-k");
        cmd6.push("-f");
        if (options.arguments) {
            options.required && cmd6.push("-r");
            cmd6.push("-a", options.arguments);
        }
        options.description && cmd6.push("-d", `'${options.description}'`);
        return cmd6.join(" ");
    }
    getCompletionCommand(cmd) {
        return `'(${this.cmd.getName()} completions complete ${cmd.trim()})'`;
    }
}
function getCommandFnNames(cmd6, cmds = []) {
    cmds.push(`__${replaceSpecialChars2(cmd6.getPath())}`);
    cmd6.getCommands(false).forEach((command)=>{
        getCommandFnNames(command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars2(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd6){
        super();
        this.#cmd = cmd6;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim(italic("~/.config/fish/config.fish"))}:

    ${dim(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(FishCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class ZshCompletionsGenerator {
    cmd;
    actions = new Map();
    static generate(cmd) {
        return new ZshCompletionsGenerator(cmd).generate();
    }
    constructor(cmd7){
        this.cmd = cmd7;
    }
    generate() {
        const path = this.cmd.getPath();
        const name18 = this.cmd.getName();
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path}${version}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars3(name18)}_complete] )) ||
function __${replaceSpecialChars3(name18)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${name18} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.cmd).trim()}

# _${replaceSpecialChars3(path)} "\${@}"

compdef _${replaceSpecialChars3(path)} ${path}

`;
    }
    generateCompletions(command, path = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path = (path ? path + " " : "") + command.getName();
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars3(path)}] )) ||
function _${replaceSpecialChars3(path)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path) + this.generateSubCommandCompletions(command, path) + this.generateArgumentCompletions(command, path) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command
        ).map((subCommand)=>this.generateCompletions(subCommand, path)
        ).join("");
    }
    generateCommandCompletions(command, path) {
        const commands5 = command.getCommands(false);
        let completions = commands5.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription()}'`
        ).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path.split(" ").slice(1).join(" ");
            const arg3 = command.getArguments()[0];
            const action = this.addAction(arg3, completionsPath);
            if (action && command.getCompletion(arg3.action)) {
                completions += `\n    __${replaceSpecialChars3(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command, path) {
        if (command.hasCommands(false)) {
            const actions = command.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars3(path + " " + command.getName())} ;;`
            ).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path) {
        this.actions.clear();
        const options6 = this.generateOptions(command, path);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options6.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg3)=>command.getCompletion(arg3.action)
        ).length) {
            argsCommand += ` \\\n    '${++argIndex}: :_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args3 = [];
            for (const arg3 of command.getArguments().slice(1)){
                const completionsPath = path.split(" ").slice(1).join(" ");
                const action = this.addAction(arg3, completionsPath);
                args3.push(`${++argIndex}${arg3.optionalValue ? "::" : ":"}${action.name}`);
            }
            argsCommand += args3.map((arg4)=>`\\\n    '${arg4}'`
            ).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*:: :->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path) {
        const options6 = [];
        const cmdArgs = path.split(" ");
        const _baseName = cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option10)=>option10.standalone ? option10.flags : false
        ).flat().filter((flag)=>typeof flag === "string"
        );
        for (const option10 of command.getOptions(false)){
            options6.push(this.generateOption(option10, completionsPath, excludedFlags));
        }
        return options6;
    }
    generateOption(option, completionsPath, excludedOptions) {
        const flags = option.flags;
        let excludedFlags = option.conflicts?.length ? [
            ...excludedOptions,
            ...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")
            ), 
        ] : excludedOptions;
        excludedFlags = option.collect ? excludedFlags : [
            ...excludedFlags,
            ...flags, 
        ];
        let args3 = "";
        for (const arg3 of option.args){
            const action = this.addAction(arg3, completionsPath);
            if (arg3.variadic) {
                args3 += `${arg3.optionalValue ? "::" : ":"}${arg3.name}:->${action.name}`;
            } else {
                args3 += `${arg3.optionalValue ? "::" : ":"}${arg3.name}:->${action.name}`;
            }
        }
        let description = option.description.trim().split("\n").shift();
        description = description.replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        if (option.standalone) {
            return `'(- *)'{${collect}${flags}}'[${description}]${args3}'`;
        } else {
            const excluded2 = excludedFlags.length ? `'(${excludedFlags.join(" ")})'` : "";
            if (collect || flags.length > 1) {
                return `${excluded2}{${collect}${flags}}'[${description}]${args3}'`;
            } else {
                return `${excluded2}${flags}'[${description}]${args3}'`;
            }
        }
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name18, action])=>`${name18}) __${replaceSpecialChars3(this.cmd.getName())}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`
            );
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
}
function replaceSpecialChars3(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd8){
        super();
        this.#cmd = cmd8;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim(italic("~/.zshrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).action(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            Deno.stdout.writeSync(new TextEncoder().encode(ZshCompletionsGenerator.generate(baseCmd)));
        });
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd9){
        super();
        this.#cmd = cmd9;
        this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim(italic("~/.bashrc"))} or similar:

    ${dim(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).action(()=>this.showHelp()
        ).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd).hidden()).reset();
    }
}
class CommandType extends StringType {
    complete(_cmd, parent) {
        return parent?.getCommands(false).map((cmd10)=>cmd10.getName()
        ) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd10){
        super();
        this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").action((_, name18)=>{
            if (!cmd10) {
                cmd10 = name18 ? this.getGlobalParent()?.getBaseCommand(name18) : this.getGlobalParent();
            }
            if (!cmd10) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommand(name18 ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases(), 
                ]);
            }
            cmd10.showHelp();
            Deno.exit(0);
        });
    }
}
class ActionListType extends StringType {
    cmd;
    constructor(cmd11){
        super();
        this.cmd = cmd11;
    }
    complete() {
        return this.cmd.getCompletions().map((type3)=>type3.name
        ).filter((value4, index, self)=>self.indexOf(value4) === index
        );
    }
}
class ChildCommandType extends StringType {
    #cmd;
    constructor(cmd12){
        super();
        this.#cmd = cmd12;
    }
    complete(cmd) {
        return (this.#cmd ?? cmd)?.getCommands(false).map((cmd13)=>cmd13.getName()
        ) || [];
    }
}
class EnumType extends Type {
    allowedValues;
    constructor(values1){
        super();
        this.allowedValues = values1;
    }
    parse(type) {
        for (const value4 of this.allowedValues){
            if (value4.toString() === type.value) {
                return value4;
            }
        }
        throw new InvalidTypeError(type, this.allowedValues.slice());
    }
    values() {
        return this.allowedValues.slice();
    }
    complete() {
        return this.values();
    }
}
const osType = (()=>{
    if (globalThis.Deno != null) {
        return Deno.build.os;
    }
    const navigator = globalThis.navigator;
    if (navigator?.appVersion?.includes?.("Win") ?? false) {
        return "windows";
    }
    return "linux";
})();
const isWindows = osType === "windows";
const CHAR_FORWARD_SLASH = 47;
function assertPath(path) {
    if (typeof path !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path)}`);
    }
}
function isPosixPathSeparator(code1) {
    return code1 === 47;
}
function isPathSeparator(code1) {
    return isPosixPathSeparator(code1) || code1 === 92;
}
function isWindowsDeviceRoot(code1) {
    return code1 >= 97 && code1 <= 122 || code1 >= 65 && code1 <= 90;
}
function normalizeString(path, allowAboveRoot, separator, isPathSeparator1) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code1;
    for(let i = 0, len = path.length; i <= len; ++i){
        if (i < len) code1 = path.charCodeAt(i);
        else if (isPathSeparator1(code1)) break;
        else code1 = CHAR_FORWARD_SLASH;
        if (isPathSeparator1(code1)) {
            if (lastSlash === i - 1 || dots === 1) {
            } else if (lastSlash !== i - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path.slice(lastSlash + 1, i);
                else res = path.slice(lastSlash + 1, i);
                lastSegmentLength = i - lastSlash - 1;
            }
            lastSlash = i;
            dots = 0;
        } else if (code1 === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function _format(sep, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base;
    if (dir === pathObject.root) return dir + base;
    return dir + sep + base;
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string1) {
    return string1.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
class DenoStdInternalError extends Error {
    constructor(message4){
        super(message4);
        this.name = "DenoStdInternalError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new DenoStdInternalError(msg);
    }
}
const sep = "\\";
const delimiter = ";";
function resolve3(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i = pathSegments.length - 1; i >= -1; i--){
        let path;
        if (i >= 0) {
            path = pathSegments[i];
        } else if (!resolvedDevice) {
            if (globalThis.Deno == null) {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path = Deno.cwd();
        } else {
            if (globalThis.Deno == null) {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno.env.get(`=${resolvedDevice}`) || Deno.cwd();
            if (path === undefined || path.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path = `${resolvedDevice}\\`;
            }
        }
        assertPath(path);
        const len = path.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute = false;
        const code1 = path.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code1)) {
                isAbsolute = true;
                if (isPathSeparator(path.charCodeAt(1))) {
                    let j = 2;
                    let last = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        const firstPart = path.slice(last, j);
                        last = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last) {
                            last = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path.slice(last)}`;
                                rootEnd = j;
                            } else if (j !== last) {
                                device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code1)) {
                if (path.charCodeAt(1) === 58) {
                    device = path.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path.charCodeAt(2))) {
                            isAbsolute = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code1)) {
            rootEnd = 1;
            isAbsolute = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function normalize1(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return ".";
    let rootEnd = 0;
    let device;
    let isAbsolute = false;
    const code1 = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code1)) {
            isAbsolute = true;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    const firstPart = path.slice(last, j);
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path.slice(last)}\\`;
                        } else if (j !== last) {
                            device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code1)) {
            if (path.charCodeAt(1) === 58) {
                device = path.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        isAbsolute = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code1)) {
        return "\\";
    }
    let tail;
    if (rootEnd < len) {
        tail = normalizeString(path.slice(rootEnd), !isAbsolute, "\\", isPathSeparator);
    } else {
        tail = "";
    }
    if (tail.length === 0 && !isAbsolute) tail = ".";
    if (tail.length > 0 && isPathSeparator(path.charCodeAt(len - 1))) {
        tail += "\\";
    }
    if (device === undefined) {
        if (isAbsolute) {
            if (tail.length > 0) return `\\${tail}`;
            else return "\\";
        } else if (tail.length > 0) {
            return tail;
        } else {
            return "";
        }
    } else if (isAbsolute) {
        if (tail.length > 0) return `${device}\\${tail}`;
        else return `${device}\\`;
    } else if (tail.length > 0) {
        return device + tail;
    } else {
        return device;
    }
}
function isAbsolute(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return false;
    const code1 = path.charCodeAt(0);
    if (isPathSeparator(code1)) {
        return true;
    } else if (isWindowsDeviceRoot(code1)) {
        if (len > 2 && path.charCodeAt(1) === 58) {
            if (isPathSeparator(path.charCodeAt(2))) return true;
        }
    }
    return false;
}
function join(...paths) {
    const pathsCount = paths.length;
    if (pathsCount === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i = 0; i < pathsCount; ++i){
        const path = paths[i];
        assertPath(path);
        if (path.length > 0) {
            if (joined === undefined) joined = firstPart = path;
            else joined += `\\${path}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart != null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize1(joined);
}
function relative(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    const fromOrig = resolve3(from);
    const toOrig = resolve3(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i = 0;
    for(; i <= length; ++i){
        if (i === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i) === 92) {
                    return toOrig.slice(toStart + i + 1);
                } else if (i === 2) {
                    return toOrig.slice(toStart + i);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i) === 92) {
                    lastCommonSep = i;
                } else if (i === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i);
        const toCode = to.charCodeAt(toStart + i);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i;
    }
    if (i !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i = fromStart + lastCommonSep + 1; i <= fromEnd; ++i){
        if (i === fromEnd || from.charCodeAt(i) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
function toNamespacedPath(path) {
    if (typeof path !== "string") return path;
    if (path.length === 0) return "";
    const resolvedPath = resolve3(path);
    if (resolvedPath.length >= 3) {
        if (resolvedPath.charCodeAt(0) === 92) {
            if (resolvedPath.charCodeAt(1) === 92) {
                const code1 = resolvedPath.charCodeAt(2);
                if (code1 !== 63 && code1 !== 46) {
                    return `\\\\?\\UNC\\${resolvedPath.slice(2)}`;
                }
            }
        } else if (isWindowsDeviceRoot(resolvedPath.charCodeAt(0))) {
            if (resolvedPath.charCodeAt(1) === 58 && resolvedPath.charCodeAt(2) === 92) {
                return `\\\\?\\${resolvedPath}`;
            }
        }
    }
    return path;
}
function dirname(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return ".";
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code1 = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code1)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path;
                        }
                        if (j !== last) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code1)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code1)) {
        return path;
    }
    for(let i = len - 1; i >= offset; --i){
        if (isPathSeparator(path.charCodeAt(i))) {
            if (!matchedSlash) {
                end = i;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return path.slice(0, end);
}
function basename(path, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i;
    if (path.length >= 2) {
        const drive = path.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path.charCodeAt(1) === 58) start = 2;
        }
    }
    if (ext !== undefined && ext.length > 0 && ext.length <= path.length) {
        if (ext.length === path.length && ext === path) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i = path.length - 1; i >= start; --i){
            const code1 = path.charCodeAt(i);
            if (isPathSeparator(code1)) {
                if (!matchedSlash) {
                    start = i + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i + 1;
                }
                if (extIdx >= 0) {
                    if (code1 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path.length;
        return path.slice(start, end);
    } else {
        for(i = path.length - 1; i >= start; --i){
            if (isPathSeparator(path.charCodeAt(i))) {
                if (!matchedSlash) {
                    start = i + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i + 1;
            }
        }
        if (end === -1) return "";
        return path.slice(start, end);
    }
}
function extname(path) {
    assertPath(path);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path.length >= 2 && path.charCodeAt(1) === 58 && isWindowsDeviceRoot(path.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i = path.length - 1; i >= start; --i){
        const code1 = path.charCodeAt(i);
        if (isPathSeparator(code1)) {
            if (!matchedSlash) {
                startPart = i + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i + 1;
        }
        if (code1 === 46) {
            if (startDot === -1) startDot = i;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format3(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("\\", pathObject);
}
function parse(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    const len = path.length;
    if (len === 0) return ret;
    let rootEnd = 0;
    let code1 = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code1)) {
            rootEnd = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            rootEnd = j;
                        } else if (j !== last) {
                            rootEnd = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code1)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        if (len === 3) {
                            ret.root = ret.dir = path;
                            return ret;
                        }
                        rootEnd = 3;
                    }
                } else {
                    ret.root = ret.dir = path;
                    return ret;
                }
            }
        }
    } else if (isPathSeparator(code1)) {
        ret.root = ret.dir = path;
        return ret;
    }
    if (rootEnd > 0) ret.root = path.slice(0, rootEnd);
    let startDot = -1;
    let startPart = rootEnd;
    let end = -1;
    let matchedSlash = true;
    let i = path.length - 1;
    let preDotState = 0;
    for(; i >= rootEnd; --i){
        code1 = path.charCodeAt(i);
        if (isPathSeparator(code1)) {
            if (!matchedSlash) {
                startPart = i + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i + 1;
        }
        if (code1 === 46) {
            if (startDot === -1) startDot = i;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            ret.base = ret.name = path.slice(startPart, end);
        }
    } else {
        ret.name = path.slice(startPart, startDot);
        ret.base = path.slice(startPart, end);
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0 && startPart !== rootEnd) {
        ret.dir = path.slice(0, startPart - 1);
    } else ret.dir = ret.root;
    return ret;
}
function fromFileUrl(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    let path = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname != "") {
        path = `\\\\${url.hostname}${path}`;
    }
    return path;
}
function toFileUrl(path) {
    if (!isAbsolute(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname != null && hostname != "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
const mod1 = function() {
    return {
        sep: sep,
        delimiter: delimiter,
        resolve: resolve3,
        normalize: normalize1,
        isAbsolute: isAbsolute,
        join: join,
        relative: relative,
        toNamespacedPath: toNamespacedPath,
        dirname: dirname,
        basename: basename,
        extname: extname,
        format: format3,
        parse: parse,
        fromFileUrl: fromFileUrl,
        toFileUrl: toFileUrl
    };
}();
const sep1 = "/";
const delimiter1 = ":";
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i = pathSegments.length - 1; i >= -1 && !resolvedAbsolute; i--){
        let path;
        if (i >= 0) path = pathSegments[i];
        else {
            if (globalThis.Deno == null) {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno.cwd();
        }
        assertPath(path);
        if (path.length === 0) {
            continue;
        }
        resolvedPath = `${path}/${resolvedPath}`;
        resolvedAbsolute = path.charCodeAt(0) === CHAR_FORWARD_SLASH;
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function normalize2(path) {
    assertPath(path);
    if (path.length === 0) return ".";
    const isAbsolute1 = path.charCodeAt(0) === 47;
    const trailingSeparator = path.charCodeAt(path.length - 1) === 47;
    path = normalizeString(path, !isAbsolute1, "/", isPosixPathSeparator);
    if (path.length === 0 && !isAbsolute1) path = ".";
    if (path.length > 0 && trailingSeparator) path += "/";
    if (isAbsolute1) return `/${path}`;
    return path;
}
function isAbsolute1(path) {
    assertPath(path);
    return path.length > 0 && path.charCodeAt(0) === 47;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i = 0, len = paths.length; i < len; ++i){
        const path = paths[i];
        assertPath(path);
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `/${path}`;
        }
    }
    if (!joined) return ".";
    return normalize2(joined);
}
function relative1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 47) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 47) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i = 0;
    for(; i <= length; ++i){
        if (i === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i) === 47) {
                    return to.slice(toStart + i + 1);
                } else if (i === 0) {
                    return to.slice(toStart + i);
                }
            } else if (fromLen > length) {
                if (from.charCodeAt(fromStart + i) === 47) {
                    lastCommonSep = i;
                } else if (i === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i);
        const toCode = to.charCodeAt(toStart + i);
        if (fromCode !== toCode) break;
        else if (fromCode === 47) lastCommonSep = i;
    }
    let out = "";
    for(i = fromStart + lastCommonSep + 1; i <= fromEnd; ++i){
        if (i === fromEnd || from.charCodeAt(i) === 47) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (to.charCodeAt(toStart) === 47) ++toStart;
        return to.slice(toStart);
    }
}
function toNamespacedPath1(path) {
    return path;
}
function dirname1(path) {
    assertPath(path);
    if (path.length === 0) return ".";
    const hasRoot = path.charCodeAt(0) === 47;
    let end = -1;
    let matchedSlash = true;
    for(let i = path.length - 1; i >= 1; --i){
        if (path.charCodeAt(i) === 47) {
            if (!matchedSlash) {
                end = i;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) return hasRoot ? "/" : ".";
    if (hasRoot && end === 1) return "//";
    return path.slice(0, end);
}
function basename1(path, ext = "") {
    if (ext !== undefined && typeof ext !== "string") {
        throw new TypeError('"ext" argument must be a string');
    }
    assertPath(path);
    let start = 0;
    let end = -1;
    let matchedSlash = true;
    let i;
    if (ext !== undefined && ext.length > 0 && ext.length <= path.length) {
        if (ext.length === path.length && ext === path) return "";
        let extIdx = ext.length - 1;
        let firstNonSlashEnd = -1;
        for(i = path.length - 1; i >= 0; --i){
            const code1 = path.charCodeAt(i);
            if (code1 === 47) {
                if (!matchedSlash) {
                    start = i + 1;
                    break;
                }
            } else {
                if (firstNonSlashEnd === -1) {
                    matchedSlash = false;
                    firstNonSlashEnd = i + 1;
                }
                if (extIdx >= 0) {
                    if (code1 === ext.charCodeAt(extIdx)) {
                        if (--extIdx === -1) {
                            end = i;
                        }
                    } else {
                        extIdx = -1;
                        end = firstNonSlashEnd;
                    }
                }
            }
        }
        if (start === end) end = firstNonSlashEnd;
        else if (end === -1) end = path.length;
        return path.slice(start, end);
    } else {
        for(i = path.length - 1; i >= 0; --i){
            if (path.charCodeAt(i) === 47) {
                if (!matchedSlash) {
                    start = i + 1;
                    break;
                }
            } else if (end === -1) {
                matchedSlash = false;
                end = i + 1;
            }
        }
        if (end === -1) return "";
        return path.slice(start, end);
    }
}
function extname1(path) {
    assertPath(path);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i = path.length - 1; i >= 0; --i){
        const code1 = path.charCodeAt(i);
        if (code1 === 47) {
            if (!matchedSlash) {
                startPart = i + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i + 1;
        }
        if (code1 === 46) {
            if (startDot === -1) startDot = i;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
    return _format("/", pathObject);
}
function parse1(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path.length === 0) return ret;
    const isAbsolute2 = path.charCodeAt(0) === 47;
    let start;
    if (isAbsolute2) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i = path.length - 1;
    let preDotState = 0;
    for(; i >= start; --i){
        const code1 = path.charCodeAt(i);
        if (code1 === 47) {
            if (!matchedSlash) {
                startPart = i + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i + 1;
        }
        if (code1 === 46) {
            if (startDot === -1) startDot = i;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute2) {
                ret.base = ret.name = path.slice(1, end);
            } else {
                ret.base = ret.name = path.slice(startPart, end);
            }
        }
    } else {
        if (startPart === 0 && isAbsolute2) {
            ret.name = path.slice(1, startDot);
            ret.base = path.slice(1, end);
        } else {
            ret.name = path.slice(startPart, startDot);
            ret.base = path.slice(startPart, end);
        }
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0) ret.dir = path.slice(0, startPart - 1);
    else if (isAbsolute2) ret.dir = "/";
    return ret;
}
function fromFileUrl1(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol != "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function toFileUrl1(path) {
    if (!isAbsolute1(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
const mod2 = function() {
    return {
        sep: sep1,
        delimiter: delimiter1,
        resolve: resolve1,
        normalize: normalize2,
        isAbsolute: isAbsolute1,
        join: join1,
        relative: relative1,
        toNamespacedPath: toNamespacedPath1,
        dirname: dirname1,
        basename: basename1,
        extname: extname1,
        format: format1,
        parse: parse1,
        fromFileUrl: fromFileUrl1,
        toFileUrl: toFileUrl1
    };
}();
const SEP = isWindows ? "\\" : "/";
const SEP_PATTERN = isWindows ? /[\\/]+/ : /\/+/;
const path = isWindows ? mod1 : mod2;
const { join: join2 , normalize: normalize3  } = path;
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|"
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function globToRegExp(glob, { extended =true , globstar: globstarOption = true , os =osType , caseInsensitive =false  } = {
}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep2 = os == "windows" ? "(?:\\\\|/)+" : "/+";
    const sepMaybe = os == "windows" ? "(?:\\\\|/)*" : "/*";
    const seps = os == "windows" ? [
        "\\",
        "/"
    ] : [
        "/"
    ];
    const globstar = os == "windows" ? "(?:[^\\\\/]*(?:\\\\|/|$)+)*" : "(?:[^/]*(?:/|$)+)*";
    const wildcard = os == "windows" ? "[^\\\\/]*" : "[^/]*";
    const escapePrefix = os == "windows" ? "`" : "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i = j;
        for(; i < glob.length && !seps.includes(glob[i]); i++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i]) ? `\\${glob[i]}` : glob[i];
                continue;
            }
            if (glob[i] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i + 1] == "!") {
                        i++;
                        segment += "^";
                    } else if (glob[i + 1] == "^") {
                        i++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i + 1] == ":") {
                    let k = i + 1;
                    let value4 = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value4 += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i = k + 2;
                        if (value4 == "alnum") segment += "\\dA-Za-z";
                        else if (value4 == "alpha") segment += "A-Za-z";
                        else if (value4 == "ascii") segment += "\x00-\x7F";
                        else if (value4 == "blank") segment += "\t ";
                        else if (value4 == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value4 == "digit") segment += "\\d";
                        else if (value4 == "graph") segment += "\x21-\x7E";
                        else if (value4 == "lower") segment += "a-z";
                        else if (value4 == "print") segment += "\x20-\x7E";
                        else if (value4 == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value4 == "space") segment += "\\s\v";
                        else if (value4 == "upper") segment += "A-Z";
                        else if (value4 == "word") segment += "\\w";
                        else if (value4 == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i];
                }
                continue;
            }
            if (glob[i] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type3 = groupStack.pop();
                if (type3 == "!") {
                    segment += wildcard;
                } else if (type3 != "@") {
                    segment += type3;
                }
                continue;
            }
            if (glob[i] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i] == "+" && extended && glob[i + 1] == "(") {
                i++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i] == "@" && extended && glob[i + 1] == "(") {
                i++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i] == "?") {
                if (extended && glob[i + 1] == "(") {
                    i++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i] == "!" && extended && glob[i + 1] == "(") {
                i++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i] == "*") {
                if (extended && glob[i + 1] == "(") {
                    i++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i - 1];
                    let numStars = 1;
                    while(glob[i + 1] == "*"){
                        i++;
                        numStars++;
                    }
                    const nextChar = glob[i + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i]) ? `\\${glob[i]}` : glob[i];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i < glob.length ? sep2 : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i]))i++;
        if (!(i > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar =false  } = {
}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize3(glob);
    }
    const s = SEP_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize3(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended =false , globstar =false  } = {
}) {
    if (!globstar || globs.length == 0) {
        return join2(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path1 = glob;
        if (path1.length > 0) {
            if (!joined) joined = path1;
            else joined += `${SEP}${path1}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
const path1 = isWindows ? mod1 : mod2;
const posix = mod2;
const { basename: basename2 , delimiter: delimiter2 , dirname: dirname2 , extname: extname2 , format: format2 , fromFileUrl: fromFileUrl2 , isAbsolute: isAbsolute2 , join: join3 , normalize: normalize4 , parse: parse2 , relative: relative2 , resolve: resolve2 , sep: sep2 , toFileUrl: toFileUrl2 , toNamespacedPath: toNamespacedPath2 ,  } = path1;
function _createWalkEntrySync(path2) {
    path2 = normalize4(path2);
    const name18 = basename2(path2);
    const info = Deno.statSync(path2);
    return {
        path: path2,
        name: name18,
        isFile: info.isFile,
        isDirectory: info.isDirectory,
        isSymlink: info.isSymlink
    };
}
function include1(path2, exts, match, skip) {
    if (exts && !exts.some((ext)=>path2.endsWith(ext)
    )) {
        return false;
    }
    if (match && !match.some((pattern)=>!!path2.match(pattern)
    )) {
        return false;
    }
    if (skip && skip.some((pattern)=>!!path2.match(pattern)
    )) {
        return false;
    }
    return true;
}
function wrapErrorWithRootPath(err, root) {
    if (err.root) return err;
    err.root = root;
    err.message = `${err.message} for path "${root}"`;
    return err;
}
function* walkSync(root, { maxDepth =Infinity , includeFiles =true , includeDirs =true , followSymlinks =false , exts =undefined , match =undefined , skip =undefined  } = {
}) {
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include1(root, exts, match, skip)) {
        yield _createWalkEntrySync(root);
    }
    if (maxDepth < 1 || !include1(root, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root);
    } catch (err) {
        throw wrapErrorWithRootPath(err, normalize4(root));
    }
    for (const entry of entries){
        assert(entry.name != null);
        let path2 = join3(root, entry.name);
        if (entry.isSymlink) {
            if (followSymlinks) {
                path2 = Deno.realPathSync(path2);
            } else {
                continue;
            }
        }
        if (entry.isFile) {
            if (includeFiles && include1(path2, exts, match, skip)) {
                yield {
                    path: path2,
                    ...entry
                };
            }
        } else {
            yield* walkSync(path2, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                followSymlinks,
                exts,
                match,
                skip
            });
        }
    }
}
function split(path2) {
    const s = SEP_PATTERN.source;
    const segments = path2.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEP_PATTERN);
    const isAbsolute_ = isAbsolute2(path2);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path2.match(new RegExp(`${s}$`)),
        winRoot: isWindows && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error) {
    if (!(error instanceof Deno.errors.NotFound)) {
        throw error;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root =Deno.cwd() , exclude =[] , includeDirs =true , extended =false , globstar =false , caseInsensitive  } = {
}) {
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = isAbsolute2(root) ? normalize4(root) : joinGlobs([
        Deno.cwd(),
        root
    ], globOptions);
    const resolveFromRoot = (path2)=>isAbsolute2(path2) ? normalize4(path2) : joinGlobs([
            absRoot,
            path2
        ], globOptions)
    ;
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp(s, globOptions)
    );
    const shouldInclude = (path2)=>!excludePatterns.some((p)=>!!path2.match(p)
        )
    ;
    const { segments , hasTrailingSep , winRoot  } = split(resolveFromRoot(glob));
    let fixedRoot = winRoot != undefined ? winRoot : "/";
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg != null);
        fixedRoot = joinGlobs([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = _createWalkEntrySync(fixedRoot);
    } catch (error) {
        return throwUnlessNotFound(error);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment == "..") {
            const parentPath = joinGlobs([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude(parentPath)) {
                    return yield _createWalkEntrySync(parentPath);
                }
            } catch (error) {
                throwUnlessNotFound(error);
            }
            return;
        } else if (globSegment == "**") {
            return yield* walkSync(walkInfo.path, {
                includeFiles: false,
                skip: excludePatterns
            });
        }
        yield* walkSync(walkInfo.path, {
            maxDepth: 1,
            match: [
                globToRegExp(joinGlobs([
                    walkInfo.path,
                    globSegment
                ], globOptions), globOptions), 
            ],
            skip: excludePatterns
        });
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory
        );
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory
        );
    }
    yield* currentMatches;
}
var LogLevels;
(function(LogLevels1) {
    LogLevels1[LogLevels1["NOTSET"] = 0] = "NOTSET";
    LogLevels1[LogLevels1["DEBUG"] = 10] = "DEBUG";
    LogLevels1[LogLevels1["INFO"] = 20] = "INFO";
    LogLevels1[LogLevels1["WARNING"] = 30] = "WARNING";
    LogLevels1[LogLevels1["ERROR"] = 40] = "ERROR";
    LogLevels1[LogLevels1["CRITICAL"] = 50] = "CRITICAL";
})(LogLevels || (LogLevels = {
}));
const byLevel = {
    [String(LogLevels.NOTSET)]: "NOTSET",
    [String(LogLevels.DEBUG)]: "DEBUG",
    [String(LogLevels.INFO)]: "INFO",
    [String(LogLevels.WARNING)]: "WARNING",
    [String(LogLevels.ERROR)]: "ERROR",
    [String(LogLevels.CRITICAL)]: "CRITICAL"
};
function getLevelByName(name18) {
    switch(name18){
        case "NOTSET":
            return LogLevels.NOTSET;
        case "DEBUG":
            return LogLevels.DEBUG;
        case "INFO":
            return LogLevels.INFO;
        case "WARNING":
            return LogLevels.WARNING;
        case "ERROR":
            return LogLevels.ERROR;
        case "CRITICAL":
            return LogLevels.CRITICAL;
        default:
            throw new Error(`no log level found for "${name18}"`);
    }
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`no level name found for level: ${level}`);
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options6){
        this.msg = options6.msg;
        this.#args = [
            ...options6.args
        ];
        this.level = options6.level;
        this.loggerName = options6.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options6.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName1, options7 = {
    }){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName1);
        this.#handlers = options7.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    _log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data) {
        if (typeof data === "string") {
            return data;
        } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
            return String(data);
        } else if (data instanceof Error) {
            return data.stack;
        } else if (typeof data === "object") {
            return JSON.stringify(data);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this._log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this._log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this._log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this._log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this._log(LogLevels.CRITICAL, msg, ...args);
    }
}
const noColor1 = globalThis.Deno?.noColor ?? true;
let enabled1 = !noColor1;
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str, code2) {
    return enabled1 ? `${code2.open}${str.replace(code2.regexp, code2.open)}${code2.close}` : str;
}
function bold1(str) {
    return run1(str, code1([
        1
    ], 22));
}
function dim1(str) {
    return run1(str, code1([
        2
    ], 22));
}
function underline1(str) {
    return run1(str, code1([
        4
    ], 24));
}
function red1(str) {
    return run1(str, code1([
        31
    ], 39));
}
function green1(str) {
    return run1(str, code1([
        32
    ], 39));
}
function yellow1(str) {
    return run1(str, code1([
        33
    ], 39));
}
function blue1(str) {
    return run1(str, code1([
        34
    ], 39));
}
function brightRed1(str) {
    return run1(str, code1([
        91
    ], 39));
}
const ANSI_PATTERN1 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))", 
].join("|"), "g");
function stripColor1(string1) {
    return string1.replace(ANSI_PATTERN1, "");
}
async function exists(filePath) {
    try {
        await Deno.lstat(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function existsSync(filePath) {
    try {
        Deno.lstatSync(filePath);
        return true;
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return false;
        }
        throw err;
    }
}
function concat(...buf) {
    let length = 0;
    for (const b of buf){
        length += b.length;
    }
    const output = new Uint8Array(length);
    let index = 0;
    for (const b1 of buf){
        output.set(b1, index);
        index += b1.length;
    }
    return output;
}
function copy(src, dst, off = 0) {
    off = Math.max(0, Math.min(off, dst.byteLength));
    const dstBytesAvailable = dst.byteLength - off;
    if (src.byteLength > dstBytesAvailable) {
        src = src.subarray(0, dstBytesAvailable);
    }
    dst.set(src, off);
    return src.byteLength;
}
class BytesList {
    len = 0;
    chunks = [];
    constructor(){
    }
    size() {
        return this.len;
    }
    add(value, start = 0, end = value.byteLength) {
        if (value.byteLength === 0 || end - start === 0) {
            return;
        }
        checkRange(start, end, value.byteLength);
        this.chunks.push({
            value,
            end,
            start,
            offset: this.len
        });
        this.len += end - start;
    }
    shift(n) {
        if (n === 0) {
            return;
        }
        if (this.len <= n) {
            this.chunks = [];
            this.len = 0;
            return;
        }
        const idx = this.getChunkIndex(n);
        this.chunks.splice(0, idx);
        const [chunk] = this.chunks;
        if (chunk) {
            const diff = n - chunk.offset;
            chunk.start += diff;
        }
        let offset = 0;
        for (const chunk1 of this.chunks){
            chunk1.offset = offset;
            offset += chunk1.end - chunk1.start;
        }
        this.len = offset;
    }
    getChunkIndex(pos) {
        let max = this.chunks.length;
        let min = 0;
        while(true){
            const i = min + Math.floor((max - min) / 2);
            if (i < 0 || this.chunks.length <= i) {
                return -1;
            }
            const { offset , start , end  } = this.chunks[i];
            const len = end - start;
            if (offset <= pos && pos < offset + len) {
                return i;
            } else if (offset + len <= pos) {
                min = i + 1;
            } else {
                max = i - 1;
            }
        }
    }
    get(i) {
        if (i < 0 || this.len <= i) {
            throw new Error("out of range");
        }
        const idx = this.getChunkIndex(i);
        const { value: value4 , offset , start  } = this.chunks[idx];
        return value4[start + i - offset];
    }
    *iterator(start = 0) {
        const startIdx = this.getChunkIndex(start);
        if (startIdx < 0) return;
        const first = this.chunks[startIdx];
        let firstOffset = start - first.offset;
        for(let i = startIdx; i < this.chunks.length; i++){
            const chunk = this.chunks[i];
            for(let j = chunk.start + firstOffset; j < chunk.end; j++){
                yield chunk.value[j];
            }
            firstOffset = 0;
        }
    }
    slice(start, end = this.len) {
        if (end === start) {
            return new Uint8Array();
        }
        checkRange(start, end, this.len);
        const result = new Uint8Array(end - start);
        const startIdx = this.getChunkIndex(start);
        const endIdx = this.getChunkIndex(end - 1);
        let written = 0;
        for(let i = startIdx; i < endIdx; i++){
            const chunk = this.chunks[i];
            const len = chunk.end - chunk.start;
            result.set(chunk.value.subarray(chunk.start, chunk.end), written);
            written += len;
        }
        const last = this.chunks[endIdx];
        const rest = end - start - written;
        result.set(last.value.subarray(last.start, last.start + rest), written);
        return result;
    }
    concat() {
        const result = new Uint8Array(this.len);
        let sum = 0;
        for (const { value: value4 , start , end  } of this.chunks){
            result.set(value4.subarray(start, end), sum);
            sum += end - start;
        }
        return result;
    }
}
function checkRange(start, end, len) {
    if (start < 0 || len < start || end < 0 || len < end || end < start) {
        throw new Error("invalid range");
    }
}
const MIN_READ = 32 * 1024;
const MAX_SIZE = 2 ** 32 - 2;
class Buffer {
    #buf;
    #off = 0;
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice = (n)=>{
        const l = this.#buf.byteLength;
        if (n <= this.capacity - l) {
            this.#reslice(l + n);
            return l;
        }
        return -1;
    };
    #reslice = (len)=>{
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    };
    readSync(p) {
        if (this.empty()) {
            this.reset();
            if (p.byteLength === 0) {
                return 0;
            }
            return null;
        }
        const nread = copy(this.#buf.subarray(this.#off), p);
        this.#off += nread;
        return nread;
    }
    read(p) {
        const rr = this.readSync(p);
        return Promise.resolve(rr);
    }
    writeSync(p) {
        const m = this.#grow(p.byteLength);
        return copy(p, this.#buf, m);
    }
    write(p) {
        const n = this.writeSync(p);
        return Promise.resolve(n);
    }
    #grow = (n)=>{
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i = this.#tryGrowByReslice(n);
        if (i >= 0) {
            return i;
        }
        const c = this.capacity;
        if (n <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n, MAX_SIZE));
        return m;
    };
    grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
    async readFrom(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = await r.read(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
    readFromSync(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = r.readSync(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
}
var DiffType;
(function(DiffType1) {
    DiffType1["removed"] = "removed";
    DiffType1["common"] = "common";
    DiffType1["added"] = "added";
})(DiffType || (DiffType = {
}));
class AssertionError extends Error {
    constructor(message5){
        super(message5);
        this.name = "AssertionError";
    }
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
function readAllSync(r) {
    const buf = new Buffer();
    buf.readFromSync(r);
    return buf.bytes();
}
async function writeAll(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += await w.write(arr.subarray(nwritten));
    }
}
function writeAllSync(w, arr) {
    let nwritten = 0;
    while(nwritten < arr.length){
        nwritten += w.writeSync(arr.subarray(nwritten));
    }
}
async function* iter(r, options8) {
    const bufSize = options8?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result = await r.read(b);
        if (result === null) {
            break;
        }
        yield b.subarray(0, result);
    }
}
const DEFAULT_BUF_SIZE = 4096;
const MIN_BUF_SIZE = 16;
const CR = "\r".charCodeAt(0);
const LF = "\n".charCodeAt(0);
class BufferFullError extends Error {
    partial;
    name = "BufferFullError";
    constructor(partial1){
        super("Buffer full");
        this.partial = partial1;
    }
}
class PartialReadError extends Error {
    name = "PartialReadError";
    partial;
    constructor(){
        super("Encountered UnexpectedEof, data only partially read");
    }
}
class BufReader {
    buf;
    rd;
    r = 0;
    w = 0;
    eof = false;
    static create(r, size = 4096) {
        return r instanceof BufReader ? r : new BufReader(r, size);
    }
    constructor(rd1, size1 = 4096){
        if (size1 < 16) {
            size1 = MIN_BUF_SIZE;
        }
        this._reset(new Uint8Array(size1), rd1);
    }
    size() {
        return this.buf.byteLength;
    }
    buffered() {
        return this.w - this.r;
    }
    async _fill() {
        if (this.r > 0) {
            this.buf.copyWithin(0, this.r, this.w);
            this.w -= this.r;
            this.r = 0;
        }
        if (this.w >= this.buf.byteLength) {
            throw Error("bufio: tried to fill full buffer");
        }
        for(let i = 100; i > 0; i--){
            const rr = await this.rd.read(this.buf.subarray(this.w));
            if (rr === null) {
                this.eof = true;
                return;
            }
            assert(rr >= 0, "negative read");
            this.w += rr;
            if (rr > 0) {
                return;
            }
        }
        throw new Error(`No progress after ${100} read() calls`);
    }
    reset(r) {
        this._reset(this.buf, r);
    }
    _reset(buf, rd) {
        this.buf = buf;
        this.rd = rd;
        this.eof = false;
    }
    async read(p) {
        let rr = p.byteLength;
        if (p.byteLength === 0) return rr;
        if (this.r === this.w) {
            if (p.byteLength >= this.buf.byteLength) {
                const rr1 = await this.rd.read(p);
                const nread = rr1 ?? 0;
                assert(nread >= 0, "negative read");
                return rr1;
            }
            this.r = 0;
            this.w = 0;
            rr = await this.rd.read(this.buf);
            if (rr === 0 || rr === null) return rr;
            assert(rr >= 0, "negative read");
            this.w += rr;
        }
        const copied = copy(this.buf.subarray(this.r, this.w), p, 0);
        this.r += copied;
        return copied;
    }
    async readFull(p) {
        let bytesRead = 0;
        while(bytesRead < p.length){
            try {
                const rr = await this.read(p.subarray(bytesRead));
                if (rr === null) {
                    if (bytesRead === 0) {
                        return null;
                    } else {
                        throw new PartialReadError();
                    }
                }
                bytesRead += rr;
            } catch (err) {
                err.partial = p.subarray(0, bytesRead);
                throw err;
            }
        }
        return p;
    }
    async readByte() {
        while(this.r === this.w){
            if (this.eof) return null;
            await this._fill();
        }
        const c = this.buf[this.r];
        this.r++;
        return c;
    }
    async readString(delim) {
        if (delim.length !== 1) {
            throw new Error("Delimiter should be a single character");
        }
        const buffer = await this.readSlice(delim.charCodeAt(0));
        if (buffer === null) return null;
        return new TextDecoder().decode(buffer);
    }
    async readLine() {
        let line;
        try {
            line = await this.readSlice(LF);
        } catch (err) {
            let { partial: partial2  } = err;
            assert(partial2 instanceof Uint8Array, "bufio: caught error from `readSlice()` without `partial` property");
            if (!(err instanceof BufferFullError)) {
                throw err;
            }
            if (!this.eof && partial2.byteLength > 0 && partial2[partial2.byteLength - 1] === CR) {
                assert(this.r > 0, "bufio: tried to rewind past start of buffer");
                this.r--;
                partial2 = partial2.subarray(0, partial2.byteLength - 1);
            }
            return {
                line: partial2,
                more: !this.eof
            };
        }
        if (line === null) {
            return null;
        }
        if (line.byteLength === 0) {
            return {
                line,
                more: false
            };
        }
        if (line[line.byteLength - 1] == LF) {
            let drop = 1;
            if (line.byteLength > 1 && line[line.byteLength - 2] === CR) {
                drop = 2;
            }
            line = line.subarray(0, line.byteLength - drop);
        }
        return {
            line,
            more: false
        };
    }
    async readSlice(delim) {
        let s = 0;
        let slice;
        while(true){
            let i = this.buf.subarray(this.r + s, this.w).indexOf(delim);
            if (i >= 0) {
                i += s;
                slice = this.buf.subarray(this.r, this.r + i + 1);
                this.r += i + 1;
                break;
            }
            if (this.eof) {
                if (this.r === this.w) {
                    return null;
                }
                slice = this.buf.subarray(this.r, this.w);
                this.r = this.w;
                break;
            }
            if (this.buffered() >= this.buf.byteLength) {
                this.r = this.w;
                const oldbuf = this.buf;
                const newbuf = this.buf.slice(0);
                this.buf = newbuf;
                throw new BufferFullError(oldbuf);
            }
            s = this.w - this.r;
            try {
                await this._fill();
            } catch (err) {
                err.partial = slice;
                throw err;
            }
        }
        return slice;
    }
    async peek(n) {
        if (n < 0) {
            throw Error("negative count");
        }
        let avail = this.w - this.r;
        while(avail < n && avail < this.buf.byteLength && !this.eof){
            try {
                await this._fill();
            } catch (err) {
                err.partial = this.buf.subarray(this.r, this.w);
                throw err;
            }
            avail = this.w - this.r;
        }
        if (avail === 0 && this.eof) {
            return null;
        } else if (avail < n && this.eof) {
            return this.buf.subarray(this.r, this.r + avail);
        } else if (avail < n) {
            throw new BufferFullError(this.buf.subarray(this.r, this.w));
        }
        return this.buf.subarray(this.r, this.r + n);
    }
}
class AbstractBufBase {
    buf;
    usedBufferBytes = 0;
    err = null;
    size() {
        return this.buf.byteLength;
    }
    available() {
        return this.buf.byteLength - this.usedBufferBytes;
    }
    buffered() {
        return this.usedBufferBytes;
    }
}
class BufWriter extends AbstractBufBase {
    writer;
    static create(writer, size = 4096) {
        return writer instanceof BufWriter ? writer : new BufWriter(writer, size);
    }
    constructor(writer1, size2 = 4096){
        super();
        this.writer = writer1;
        if (size2 <= 0) {
            size2 = DEFAULT_BUF_SIZE;
        }
        this.buf = new Uint8Array(size2);
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.writer = w;
    }
    async flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            await writeAll(this.writer, this.buf.subarray(0, this.usedBufferBytes));
        } catch (e) {
            this.err = e;
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    async write(data) {
        if (this.err !== null) throw this.err;
        if (data.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = await this.writer.write(data);
                } catch (e) {
                    this.err = e;
                    throw e;
                }
            } else {
                numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                await this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data = data.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
class BufWriterSync extends AbstractBufBase {
    writer;
    static create(writer, size = 4096) {
        return writer instanceof BufWriterSync ? writer : new BufWriterSync(writer, size);
    }
    constructor(writer2, size3 = 4096){
        super();
        this.writer = writer2;
        if (size3 <= 0) {
            size3 = DEFAULT_BUF_SIZE;
        }
        this.buf = new Uint8Array(size3);
    }
    reset(w) {
        this.err = null;
        this.usedBufferBytes = 0;
        this.writer = w;
    }
    flush() {
        if (this.err !== null) throw this.err;
        if (this.usedBufferBytes === 0) return;
        try {
            writeAllSync(this.writer, this.buf.subarray(0, this.usedBufferBytes));
        } catch (e) {
            this.err = e;
            throw e;
        }
        this.buf = new Uint8Array(this.buf.length);
        this.usedBufferBytes = 0;
    }
    writeSync(data) {
        if (this.err !== null) throw this.err;
        if (data.length === 0) return 0;
        let totalBytesWritten = 0;
        let numBytesWritten = 0;
        while(data.byteLength > this.available()){
            if (this.buffered() === 0) {
                try {
                    numBytesWritten = this.writer.writeSync(data);
                } catch (e) {
                    this.err = e;
                    throw e;
                }
            } else {
                numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
                this.usedBufferBytes += numBytesWritten;
                this.flush();
            }
            totalBytesWritten += numBytesWritten;
            data = data.subarray(numBytesWritten);
        }
        numBytesWritten = copy(data, this.buf, this.usedBufferBytes);
        this.usedBufferBytes += numBytesWritten;
        totalBytesWritten += numBytesWritten;
        return totalBytesWritten;
    }
}
function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i = 1;
    while(i < lps.length){
        if (pat[i] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i] = prefixEnd;
            i++;
        } else if (prefixEnd === 0) {
            lps[i] = 0;
            i++;
        } else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
async function* readDelim(reader, delim) {
    const delimLen = delim.length;
    const delimLPS = createLPS(delim);
    const chunks = new BytesList();
    const bufSize = Math.max(1024, delimLen + 1);
    let inspectIndex = 0;
    let matchIndex = 0;
    while(true){
        const inspectArr = new Uint8Array(bufSize);
        const result = await reader.read(inspectArr);
        if (result === null) {
            yield chunks.concat();
            return;
        } else if (result < 0) {
            return;
        }
        chunks.add(inspectArr, 0, result);
        let localIndex = 0;
        while(inspectIndex < chunks.size()){
            if (inspectArr[localIndex] === delim[matchIndex]) {
                inspectIndex++;
                localIndex++;
                matchIndex++;
                if (matchIndex === delimLen) {
                    const matchEnd = inspectIndex - delimLen;
                    const readyBytes = chunks.slice(0, matchEnd);
                    yield readyBytes;
                    chunks.shift(inspectIndex);
                    inspectIndex = 0;
                    matchIndex = 0;
                }
            } else {
                if (matchIndex === 0) {
                    inspectIndex++;
                    localIndex++;
                } else {
                    matchIndex = delimLPS[matchIndex - 1];
                }
            }
        }
    }
}
async function* readStringDelim(reader, delim) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    for await (const chunk of readDelim(reader, encoder.encode(delim))){
        yield decoder.decode(chunk);
    }
}
async function* readLines(reader) {
    for await (let chunk of readStringDelim(reader, "\n")){
        if (chunk.endsWith("\r")) {
            chunk = chunk.slice(0, -1);
        }
        yield chunk;
    }
}
const DEFAULT_FORMATTER = "{levelName} {msg}";
class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName2, options8 = {
    }){
        this.level = getLevelByName(levelName2);
        this.levelName = levelName2;
        this.formatter = options8.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{(\S+)}/g, (match, p1)=>{
            const value4 = logRecord[p1];
            if (value4 == null) {
                return match;
            }
            return String(value4);
        });
    }
    log(_msg) {
    }
    async setup() {
    }
    async destroy() {
    }
}
class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue1(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow1(msg);
                break;
            case LogLevels.ERROR:
                msg = red1(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold1(red1(msg));
                break;
            default: break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = ()=>this.destroy()
    ;
    constructor(levelName3, options9){
        super(levelName3, options9);
        this._filename = options9.filename;
        this._mode = options9.mode ? options9.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    async setup() {
        this._file = await Deno.open(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
        return Promise.resolve();
    }
}
class RotatingFileHandler extends FileHandler {
    #maxBytes;
    #maxBackupCount;
    #currentFileSize = 0;
    constructor(levelName4, options10){
        super(levelName4, options10);
        this.#maxBytes = options10.maxBytes;
        this.#maxBackupCount = options10.maxBackupCount;
    }
    async setup() {
        if (this.#maxBytes < 1) {
            this.destroy();
            throw new Error("maxBytes cannot be less than 1");
        }
        if (this.#maxBackupCount < 1) {
            this.destroy();
            throw new Error("maxBackupCount cannot be less than 1");
        }
        await super.setup();
        if (this._mode === "w") {
            for(let i = 1; i <= this.#maxBackupCount; i++){
                if (await exists(this._filename + "." + i)) {
                    await Deno.remove(this._filename + "." + i);
                }
            }
        } else if (this._mode === "x") {
            for(let i = 1; i <= this.#maxBackupCount; i++){
                if (await exists(this._filename + "." + i)) {
                    this.destroy();
                    throw new Deno.errors.AlreadyExists("Backup log file " + this._filename + "." + i + " already exists");
                }
            }
        } else {
            this.#currentFileSize = (await Deno.stat(this._filename)).size;
        }
    }
    log(msg) {
        const msgByteLength = this._encoder.encode(msg).byteLength + 1;
        if (this.#currentFileSize + msgByteLength > this.#maxBytes) {
            this.rotateLogFiles();
            this.#currentFileSize = 0;
        }
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
        this.#currentFileSize += msgByteLength;
    }
    rotateLogFiles() {
        this._buf.flush();
        Deno.close(this._file.rid);
        for(let i = this.#maxBackupCount - 1; i >= 0; i--){
            const source = this._filename + (i === 0 ? "" : "." + i);
            const dest = this._filename + "." + (i + 1);
            if (existsSync(source)) {
                Deno.renameSync(source, dest);
            }
        }
        this._file = Deno.openSync(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state1 = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger1(name18) {
    if (!name18) {
        const d = state1.loggers.get("default");
        assert(d != null, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result = state1.loggers.get(name18);
    if (!result) {
        const logger = new Logger(name18, "NOTSET", {
            handlers: []
        });
        state1.loggers.set(name18, logger);
        return logger;
    }
    return result;
}
function debug(msg, ...args3) {
    if (msg instanceof Function) {
        return getLogger1("default").debug(msg, ...args3);
    }
    return getLogger1("default").debug(msg, ...args3);
}
function info(msg, ...args3) {
    if (msg instanceof Function) {
        return getLogger1("default").info(msg, ...args3);
    }
    return getLogger1("default").info(msg, ...args3);
}
function warning(msg, ...args3) {
    if (msg instanceof Function) {
        return getLogger1("default").warning(msg, ...args3);
    }
    return getLogger1("default").warning(msg, ...args3);
}
function error3(msg, ...args3) {
    if (msg instanceof Function) {
        return getLogger1("default").error(msg, ...args3);
    }
    return getLogger1("default").error(msg, ...args3);
}
async function setup(config) {
    state1.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state1.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state1.handlers.clear();
    const handlers = state1.config.handlers || {
    };
    for(const handlerName in handlers){
        const handler = handlers[handlerName];
        await handler.setup();
        state1.handlers.set(handlerName, handler);
    }
    state1.loggers.clear();
    const loggers = state1.config.loggers || {
    };
    for(const loggerName1 in loggers){
        const loggerConfig = loggers[loggerName1];
        const handlerNames = loggerConfig.handlers || [];
        const handlers1 = [];
        handlerNames.forEach((handlerName1)=>{
            const handler = state1.handlers.get(handlerName1);
            if (handler) {
                handlers1.push(handler);
            }
        });
        const levelName5 = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName1, levelName5, {
            handlers: handlers1
        });
        state1.loggers.set(loggerName1, logger);
    }
}
await setup(DEFAULT_CONFIG);
function isSubdir(src, dest, sep3 = sep2) {
    if (src === dest) {
        return false;
    }
    const srcArray = src.split(sep3);
    const destArray = dest.split(sep3);
    return srcArray.every((current, i)=>destArray[i] === current
    );
}
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            await Deno.mkdir(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            Deno.mkdirSync(dir, {
                recursive: true
            });
            return;
        }
        throw err;
    }
}
function moveSync(src, dest, { overwrite =false  } = {
}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (overwrite) {
        if (existsSync(dest)) {
            Deno.removeSync(dest, {
                recursive: true
            });
        }
    } else {
        if (existsSync(dest)) {
            throw new Error("dest already exists.");
        }
    }
    Deno.renameSync(src, dest);
}
async function ensureValidCopy(src, dest, options11) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options11.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options11.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
function ensureValidCopySync(src, dest, options11) {
    let destStat;
    try {
        destStat = Deno.lstatSync(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options11.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options11.overwrite) {
        throw new Error(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options11) {
    await ensureValidCopy(src, dest, options11);
    await Deno.copyFile(src, dest);
    if (options11.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copyFileSync(src, dest, options11) {
    ensureValidCopySync(src, dest, options11);
    Deno.copyFileSync(src, dest);
    if (options11.preserveTimestamps) {
        const statInfo = Deno.statSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options11) {
    await ensureValidCopy(src, dest, options11);
    const originSrcFilePath = await Deno.readLink(src);
    const type3 = getFileInfoType(await Deno.lstat(src));
    if (isWindows) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type3 === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options11.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copySymlinkSync(src, dest, options11) {
    ensureValidCopySync(src, dest, options11);
    const originSrcFilePath = Deno.readLinkSync(src);
    const type3 = getFileInfoType(Deno.lstatSync(src));
    if (isWindows) {
        Deno.symlinkSync(originSrcFilePath, dest, {
            type: type3 === "dir" ? "dir" : "file"
        });
    } else {
        Deno.symlinkSync(originSrcFilePath, dest);
    }
    if (options11.preserveTimestamps) {
        const statInfo = Deno.lstatSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options11) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options11,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options11.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for await (const entry of Deno.readDir(src)){
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            await copySymLink(srcPath, destPath, options11);
        } else if (entry.isDirectory) {
            await copyDir(srcPath, destPath, options11);
        } else if (entry.isFile) {
            await copyFile(srcPath, destPath, options11);
        }
    }
}
function copyDirSync(src, dest, options11) {
    const destStat = ensureValidCopySync(src, dest, {
        ...options11,
        isFolder: true
    });
    if (!destStat) {
        ensureDirSync(dest);
    }
    if (options11.preserveTimestamps) {
        const srcStatInfo = Deno.statSync(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    for (const entry of Deno.readDirSync(src)){
        assert(entry.name != null, "file.name must be set");
        const srcPath = join3(src, entry.name);
        const destPath = join3(dest, basename2(srcPath));
        if (entry.isSymlink) {
            copySymlinkSync(srcPath, destPath, options11);
        } else if (entry.isDirectory) {
            copyDirSync(srcPath, destPath, options11);
        } else if (entry.isFile) {
            copyFileSync(srcPath, destPath, options11);
        }
    }
}
async function copy1(src, dest, options11 = {
}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options11);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options11);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options11);
    }
}
function copySync(src, dest, options11 = {
}) {
    src = resolve2(src);
    dest = resolve2(dest);
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = Deno.lstatSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        copySymlinkSync(src, dest, options11);
    } else if (srcStat.isDirectory) {
        copyDirSync(src, dest, options11);
    } else if (srcStat.isFile) {
        copyFileSync(src, dest, options11);
    }
}
var EOL;
(function(EOL1) {
    EOL1["LF"] = "\n";
    EOL1["CRLF"] = "\r\n";
})(EOL || (EOL = {
}));
class YAMLError extends Error {
    mark;
    constructor(message6 = "(unknown reason)", mark = ""){
        super(`${message6} ${mark}`);
        this.mark = mark;
        this.name = this.constructor.name;
    }
    toString(_compact) {
        return `${this.name}: ${this.message} ${this.mark}`;
    }
}
function isBoolean(value4) {
    return typeof value4 === "boolean" || value4 instanceof Boolean;
}
function isObject(value4) {
    return value4 !== null && typeof value4 === "object";
}
function repeat(str, count) {
    let result = "";
    for(let cycle = 0; cycle < count; cycle++){
        result += str;
    }
    return result;
}
function isNegativeZero(i) {
    return i === 0 && Number.NEGATIVE_INFINITY === 1 / i;
}
class Mark {
    name;
    buffer;
    position;
    line;
    column;
    constructor(name18, buffer, position1, line1, column){
        this.name = name18;
        this.buffer = buffer;
        this.position = position1;
        this.line = line1;
        this.column = column;
    }
    getSnippet(indent = 4, maxLength = 75) {
        if (!this.buffer) return null;
        let head = "";
        let start = this.position;
        while(start > 0 && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(start - 1)) === -1){
            start -= 1;
            if (this.position - start > maxLength / 2 - 1) {
                head = " ... ";
                start += 5;
                break;
            }
        }
        let tail = "";
        let end = this.position;
        while(end < this.buffer.length && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(end)) === -1){
            end += 1;
            if (end - this.position > maxLength / 2 - 1) {
                tail = " ... ";
                end -= 5;
                break;
            }
        }
        const snippet = this.buffer.slice(start, end);
        return `${repeat(" ", indent)}${head}${snippet}${tail}\n${repeat(" ", indent + this.position - start + head.length)}^`;
    }
    toString(compact) {
        let snippet, where = "";
        if (this.name) {
            where += `in "${this.name}" `;
        }
        where += `at line ${this.line + 1}, column ${this.column + 1}`;
        if (!compact) {
            snippet = this.getSnippet();
            if (snippet) {
                where += `:\n${snippet}`;
            }
        }
        return where;
    }
}
function compileList(schema, name19, result) {
    const exclude = [];
    for (const includedSchema of schema.include){
        result = compileList(includedSchema, name19, result);
    }
    for (const currentType of schema[name19]){
        for(let previousIndex = 0; previousIndex < result.length; previousIndex++){
            const previousType = result[previousIndex];
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind) {
                exclude.push(previousIndex);
            }
        }
        result.push(currentType);
    }
    return result.filter((_type, index)=>!exclude.includes(index)
    );
}
function compileMap(...typesList) {
    const result = {
        fallback: {
        },
        mapping: {
        },
        scalar: {
        },
        sequence: {
        }
    };
    for (const types2 of typesList){
        for (const type3 of types2){
            if (type3.kind !== null) {
                result[type3.kind][type3.tag] = result["fallback"][type3.tag] = type3;
            }
        }
    }
    return result;
}
class Schema {
    static SCHEMA_DEFAULT;
    implicit;
    explicit;
    include;
    compiledImplicit;
    compiledExplicit;
    compiledTypeMap;
    constructor(definition1){
        this.explicit = definition1.explicit || [];
        this.implicit = definition1.implicit || [];
        this.include = definition1.include || [];
        for (const type3 of this.implicit){
            if (type3.loadKind && type3.loadKind !== "scalar") {
                throw new YAMLError("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
            }
        }
        this.compiledImplicit = compileList(this, "implicit", []);
        this.compiledExplicit = compileList(this, "explicit", []);
        this.compiledTypeMap = compileMap(this.compiledImplicit, this.compiledExplicit);
    }
    extend(definition) {
        return new Schema({
            implicit: [
                ...new Set([
                    ...this.implicit,
                    ...definition?.implicit ?? []
                ]), 
            ],
            explicit: [
                ...new Set([
                    ...this.explicit,
                    ...definition?.explicit ?? []
                ]), 
            ],
            include: [
                ...new Set([
                    ...this.include,
                    ...definition?.include ?? []
                ])
            ]
        });
    }
    static create() {
    }
}
const DEFAULT_RESOLVE = ()=>true
;
const DEFAULT_CONSTRUCT = (data)=>data
;
function checkTagFormat(tag) {
    return tag;
}
class Type1 {
    tag;
    kind = null;
    instanceOf;
    predicate;
    represent;
    defaultStyle;
    styleAliases;
    loadKind;
    constructor(tag1, options11){
        this.tag = checkTagFormat(tag1);
        if (options11) {
            this.kind = options11.kind;
            this.resolve = options11.resolve || DEFAULT_RESOLVE;
            this.construct = options11.construct || DEFAULT_CONSTRUCT;
            this.instanceOf = options11.instanceOf;
            this.predicate = options11.predicate;
            this.represent = options11.represent;
            this.defaultStyle = options11.defaultStyle;
            this.styleAliases = options11.styleAliases;
        }
    }
    resolve = ()=>true
    ;
    construct = (data)=>data
    ;
}
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data) {
    if (data === null) return false;
    let code2;
    let bitlen = 0;
    const max = data.length;
    const map = BASE64_MAP;
    for(let idx = 0; idx < max; idx++){
        code2 = map.indexOf(data.charAt(idx));
        if (code2 > 64) continue;
        if (code2 < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data) {
    const input = data.replace(/[\r\n=]/g, "");
    const max = input.length;
    const map = BASE64_MAP;
    const result = [];
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result.push(bits >> 16 & 255);
            result.push(bits >> 8 & 255);
            result.push(bits & 255);
        }
        bits = bits << 6 | map.indexOf(input.charAt(idx));
    }
    const tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result.push(bits >> 16 & 255);
        result.push(bits >> 8 & 255);
        result.push(bits & 255);
    } else if (tailbits === 18) {
        result.push(bits >> 10 & 255);
        result.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result.push(bits >> 4 & 255);
    }
    return new Buffer(new Uint8Array(result));
}
function representYamlBinary(object) {
    const max = object.length;
    const map = BASE64_MAP;
    let result = "";
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result += map[bits >> 18 & 63];
            result += map[bits >> 12 & 63];
            result += map[bits >> 6 & 63];
            result += map[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    const tail = max % 3;
    if (tail === 0) {
        result += map[bits >> 18 & 63];
        result += map[bits >> 12 & 63];
        result += map[bits >> 6 & 63];
        result += map[bits & 63];
    } else if (tail === 2) {
        result += map[bits >> 10 & 63];
        result += map[bits >> 4 & 63];
        result += map[bits << 2 & 63];
        result += map[64];
    } else if (tail === 1) {
        result += map[bits >> 2 & 63];
        result += map[bits << 4 & 63];
        result += map[64];
        result += map[64];
    }
    return result;
}
function isBinary(obj) {
    const buf = new Buffer();
    try {
        if (0 > buf.readFromSync(obj)) return true;
        return false;
    } catch  {
        return false;
    } finally{
        buf.reset();
    }
}
const binary = new Type1("tag:yaml.org,2002:binary", {
    construct: constructYamlBinary,
    kind: "scalar",
    predicate: isBinary,
    represent: representYamlBinary,
    resolve: resolveYamlBinary
});
function resolveYamlBoolean(data) {
    const max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
const bool = new Type1("tag:yaml.org,2002:bool", {
    construct: constructYamlBoolean,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isBoolean,
    represent: {
        lowercase (object) {
            return object ? "true" : "false";
        },
        uppercase (object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase (object) {
            return object ? "True" : "False";
        }
    },
    resolve: resolveYamlBoolean
});
const YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + "|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + "|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + "|[-+]?\\.(?:inf|Inf|INF)" + "|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data) {
    if (!YAML_FLOAT_PATTERN.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data) {
    let value4 = data.replace(/_/g, "").toLowerCase();
    const sign = value4[0] === "-" ? -1 : 1;
    const digits = [];
    if ("+-".indexOf(value4[0]) >= 0) {
        value4 = value4.slice(1);
    }
    if (value4 === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    }
    if (value4 === ".nan") {
        return NaN;
    }
    if (value4.indexOf(":") >= 0) {
        value4.split(":").forEach((v)=>{
            digits.unshift(parseFloat(v));
        });
        let valueNb = 0;
        let base = 1;
        digits.forEach((d)=>{
            valueNb += d * base;
            base *= 60;
        });
        return sign * valueNb;
    }
    return sign * parseFloat(value4);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (isNegativeZero(object)) {
        return "-0.0";
    }
    const res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || isNegativeZero(object));
}
const __float = new Type1("tag:yaml.org,2002:float", {
    construct: constructYamlFloat,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isFloat,
    represent: representYamlFloat,
    resolve: resolveYamlFloat
});
function reconstructFunction(code2) {
    const func = new Function(`return ${code2}`)();
    if (!(func instanceof Function)) {
        throw new TypeError(`Expected function but got ${typeof func}: ${code2}`);
    }
    return func;
}
const func = new Type1("tag:yaml.org,2002:js/function", {
    kind: "scalar",
    resolve (data) {
        if (data === null) {
            return false;
        }
        try {
            reconstructFunction(`${data}`);
            return true;
        } catch (_err) {
            return false;
        }
    },
    construct (data) {
        return reconstructFunction(data);
    },
    predicate (object) {
        return object instanceof Function;
    },
    represent (object) {
        return object.toString();
    }
});
function isHexCode(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode(c) {
    return 48 <= c && c <= 55;
}
function isDecCode(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger(data) {
    const max = data.length;
    let index = 0;
    let hasDigits = false;
    if (!max) return false;
    let ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for(; index < max; index++){
            ch = data[index];
            if (ch === "_") continue;
            if (!isOctCode(data.charCodeAt(index))) return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (ch === ":") break;
        if (!isDecCode(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    if (ch !== ":") return true;
    return /^(:[0-5]?[0-9])+$/.test(data.slice(index));
}
function constructYamlInteger(data) {
    let value4 = data;
    const digits = [];
    if (value4.indexOf("_") !== -1) {
        value4 = value4.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value4[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value4 = value4.slice(1);
        ch = value4[0];
    }
    if (value4 === "0") return 0;
    if (ch === "0") {
        if (value4[1] === "b") return sign * parseInt(value4.slice(2), 2);
        if (value4[1] === "x") return sign * parseInt(value4, 16);
        return sign * parseInt(value4, 8);
    }
    if (value4.indexOf(":") !== -1) {
        value4.split(":").forEach((v)=>{
            digits.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base = 1;
        digits.forEach((d)=>{
            valueInt += d * base;
            base *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value4, 10);
}
function isInteger(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !isNegativeZero(object);
}
const __int = new Type1("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger,
    represent: {
        binary (obj) {
            return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
        },
        octal (obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal (obj) {
            return obj.toString(10);
        },
        hexadecimal (obj) {
            return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        }
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ],
        octal: [
            8,
            "oct"
        ]
    }
});
const map = new Type1("tag:yaml.org,2002:map", {
    construct (data) {
        return data !== null ? data : {
        };
    },
    kind: "mapping"
});
function resolveYamlMerge(data) {
    return data === "<<" || data === null;
}
const merge = new Type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
function resolveYamlNull(data) {
    const max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull(object) {
    return object === null;
}
const nil = new Type1("tag:yaml.org,2002:null", {
    construct: constructYamlNull,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isNull,
    represent: {
        canonical () {
            return "~";
        },
        lowercase () {
            return "null";
        },
        uppercase () {
            return "NULL";
        },
        camelcase () {
            return "Null";
        }
    },
    resolve: resolveYamlNull
});
const _hasOwnProperty = Object.prototype.hasOwnProperty;
const _toString = Object.prototype.toString;
function resolveYamlOmap(data) {
    const objectKeys = [];
    let pairKey = "";
    let pairHasKey = false;
    for (const pair of data){
        pairHasKey = false;
        if (_toString.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (_hasOwnProperty.call(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data) {
    return data !== null ? data : [];
}
const omap = new Type1("tag:yaml.org,2002:omap", {
    construct: constructYamlOmap,
    kind: "sequence",
    resolve: resolveYamlOmap
});
const _toString1 = Object.prototype.toString;
function resolveYamlPairs(data) {
    const result = new Array(data.length);
    for(let index = 0; index < data.length; index++){
        const pair = data[index];
        if (_toString1.call(pair) !== "[object Object]") return false;
        const keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data) {
    if (data === null) return [];
    const result = new Array(data.length);
    for(let index = 0; index < data.length; index += 1){
        const pair = data[index];
        const keys = Object.keys(pair);
        result[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result;
}
const pairs = new Type1("tag:yaml.org,2002:pairs", {
    construct: constructYamlPairs,
    kind: "sequence",
    resolve: resolveYamlPairs
});
const REGEXP = /^\/(?<regexp>[\s\S]+)\/(?<modifiers>[gismuy]*)$/;
const regexp = new Type1("tag:yaml.org,2002:js/regexp", {
    kind: "scalar",
    resolve (data) {
        if (data === null || !data.length) {
            return false;
        }
        const regexp1 = `${data}`;
        if (regexp1.charAt(0) === "/") {
            if (!REGEXP.test(data)) {
                return false;
            }
            const modifiers = [
                ...regexp1.match(REGEXP)?.groups?.modifiers ?? ""
            ];
            if (new Set(modifiers).size < modifiers.length) {
                return false;
            }
        }
        return true;
    },
    construct (data) {
        const { regexp: regexp1 = `${data}` , modifiers =""  } = `${data}`.match(REGEXP)?.groups ?? {
        };
        return new RegExp(regexp1, modifiers);
    },
    predicate (object) {
        return object instanceof RegExp;
    },
    represent (object) {
        return object.toString();
    }
});
const seq = new Type1("tag:yaml.org,2002:seq", {
    construct (data) {
        return data !== null ? data : [];
    },
    kind: "sequence"
});
const _hasOwnProperty1 = Object.prototype.hasOwnProperty;
function resolveYamlSet(data) {
    if (data === null) return true;
    for(const key in data){
        if (_hasOwnProperty1.call(data, key)) {
            if (data[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data) {
    return data !== null ? data : {
    };
}
const set = new Type1("tag:yaml.org,2002:set", {
    construct: constructYamlSet,
    kind: "mapping",
    resolve: resolveYamlSet
});
const str = new Type1("tag:yaml.org,2002:str", {
    construct (data) {
        return data !== null ? data : "";
    },
    kind: "scalar"
});
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9])" + "-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9]?)" + "-([0-9][0-9]?)" + "(?:[Tt]|[ \\t]+)" + "([0-9][0-9]?)" + ":([0-9][0-9])" + ":([0-9][0-9])" + "(?:\\.([0-9]*))?" + "(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + "(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp(data) {
    let match = YAML_DATE_REGEXP.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
    if (match === null) throw new Error("Date resolve error");
    const year = +match[1];
    const month = +match[2] - 1;
    const day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    const hour = +match[4];
    const minute = +match[5];
    const second = +match[6];
    let fraction = 0;
    if (match[7]) {
        let partFraction = match[7].slice(0, 3);
        while(partFraction.length < 3){
            partFraction += "0";
        }
        fraction = +partFraction;
    }
    let delta = null;
    if (match[9]) {
        const tzHour = +match[10];
        const tzMinute = +(match[11] || 0);
        delta = (tzHour * 60 + tzMinute) * 60000;
        if (match[9] === "-") delta = -delta;
    }
    const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp(date) {
    return date.toISOString();
}
const timestamp = new Type1("tag:yaml.org,2002:timestamp", {
    construct: constructYamlTimestamp,
    instanceOf: Date,
    kind: "scalar",
    represent: representYamlTimestamp,
    resolve: resolveYamlTimestamp
});
const undefinedType = new Type1("tag:yaml.org,2002:js/undefined", {
    kind: "scalar",
    resolve () {
        return true;
    },
    construct () {
        return undefined;
    },
    predicate (object) {
        return typeof object === "undefined";
    },
    represent () {
        return "";
    }
});
const failsafe = new Schema({
    explicit: [
        str,
        seq,
        map
    ]
});
const json1 = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ]
});
const core = new Schema({
    include: [
        json1
    ]
});
const def1 = new Schema({
    explicit: [
        binary,
        omap,
        pairs,
        set
    ],
    implicit: [
        timestamp,
        merge
    ],
    include: [
        core
    ]
});
const extended = new Schema({
    explicit: [
        func,
        regexp,
        undefinedType
    ],
    include: [
        def1
    ]
});
class State {
    schema;
    constructor(schema3 = def1){
        this.schema = schema3;
    }
}
class LoaderState extends State {
    input;
    documents = [];
    length;
    lineIndent = 0;
    lineStart = 0;
    position = 0;
    line = 0;
    filename;
    onWarning;
    legacy;
    json;
    listener;
    implicitTypes;
    typeMap;
    version;
    checkLineBreaks;
    tagMap;
    anchorMap;
    tag;
    anchor;
    kind;
    result = "";
    constructor(input, { filename , schema: schema1 , onWarning , legacy =false , json: json2 = false , listener: listener1 = null  }){
        super(schema1);
        this.input = input;
        this.filename = filename;
        this.onWarning = onWarning;
        this.legacy = legacy;
        this.json = json2;
        this.listener = listener1;
        this.implicitTypes = this.schema.compiledImplicit;
        this.typeMap = this.schema.compiledTypeMap;
        this.length = input.length;
    }
}
const _hasOwnProperty2 = Object.prototype.hasOwnProperty;
const CONTEXT_BLOCK_IN = 3;
const CONTEXT_BLOCK_OUT = 4;
const CHOMPING_STRIP = 2;
const CHOMPING_KEEP = 3;
const PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
const PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
const PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
const PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
const PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function isEOL(c) {
    return c === 10 || c === 13;
}
function isWhiteSpace(c) {
    return c === 9 || c === 32;
}
function isWsOrEol(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function isFlowIndicator(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    const lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 48 ? "\x00" : c === 97 ? "\x07" : c === 98 ? "\x08" : c === 116 ? "\x09" : c === 9 ? "\x09" : c === 110 ? "\x0A" : c === 118 ? "\x0B" : c === 102 ? "\x0C" : c === 114 ? "\x0D" : c === 101 ? "\x1B" : c === 32 ? " " : c === 34 ? "\x22" : c === 47 ? "/" : c === 92 ? "\x5C" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
const simpleEscapeCheck = new Array(256);
const simpleEscapeMap = new Array(256);
for(let i1 = 0; i1 < 256; i1++){
    simpleEscapeCheck[i1] = simpleEscapeSequence(i1) ? 1 : 0;
    simpleEscapeMap[i1] = simpleEscapeSequence(i1);
}
function generateError(state1, message7) {
    return new YAMLError(message7, new Mark(state1.filename, state1.input, state1.position, state1.line, state1.position - state1.lineStart));
}
function throwError(state1, message7) {
    throw generateError(state1, message7);
}
function throwWarning(state1, message7) {
    if (state1.onWarning) {
        state1.onWarning.call(null, generateError(state1, message7));
    }
}
const directiveHandlers = {
    YAML (state, _name, ...args) {
        if (state.version !== null) {
            return throwError(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            return throwError(state, "YAML directive accepts exactly one argument");
        }
        const match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            return throwError(state, "ill-formed argument of the YAML directive");
        }
        const major = parseInt(match[1], 10);
        const minor = parseInt(match[2], 10);
        if (major !== 1) {
            return throwError(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            return throwWarning(state, "unsupported YAML version of the document");
        }
    },
    TAG (state, _name, ...args) {
        if (args.length !== 2) {
            return throwError(state, "TAG directive accepts exactly two arguments");
        }
        const handle = args[0];
        const prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            return throwError(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (_hasOwnProperty2.call(state.tagMap, handle)) {
            return throwError(state, `there is a previously declared suffix for "${handle}" tag handle`);
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            return throwError(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        if (typeof state.tagMap === "undefined") {
            state.tagMap = {
            };
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment(state2, start, end, checkJson) {
    let result;
    if (start < end) {
        result = state2.input.slice(start, end);
        if (checkJson) {
            for(let position2 = 0, length = result.length; position2 < length; position2++){
                const character = result.charCodeAt(position2);
                if (!(character === 9 || 32 <= character && character <= 1114111)) {
                    return throwError(state2, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(result)) {
            return throwError(state2, "the stream contains non-printable characters");
        }
        state2.result += result;
    }
}
function mergeMappings(state2, destination, source, overridableKeys) {
    if (!isObject(source)) {
        return throwError(state2, "cannot merge mappings; the provided source object is unacceptable");
    }
    const keys = Object.keys(source);
    for(let i1 = 0, len = keys.length; i1 < len; i1++){
        const key = keys[i1];
        if (!_hasOwnProperty2.call(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, valueNode, startLine, startPos) {
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(let index = 0, quantity = keyNode.length; index < quantity; index++){
            if (Array.isArray(keyNode[index])) {
                return throwError(state2, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (result === null) {
        result = {
        };
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(let index = 0, quantity = valueNode.length; index < quantity; index++){
                mergeMappings(state2, result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings(state2, result, valueNode, overridableKeys);
        }
    } else {
        if (!state2.json && !_hasOwnProperty2.call(overridableKeys, keyNode) && _hasOwnProperty2.call(result, keyNode)) {
            state2.line = startLine || state2.line;
            state2.position = startPos || state2.position;
            return throwError(state2, "duplicated mapping key");
        }
        result[keyNode] = valueNode;
        delete overridableKeys[keyNode];
    }
    return result;
}
function readLineBreak(state2) {
    const ch = state2.input.charCodeAt(state2.position);
    if (ch === 10) {
        state2.position++;
    } else if (ch === 13) {
        state2.position++;
        if (state2.input.charCodeAt(state2.position) === 10) {
            state2.position++;
        }
    } else {
        return throwError(state2, "a line break is expected");
    }
    state2.line += 1;
    state2.lineStart = state2.position;
}
function skipSeparationSpace(state2, allowComments, checkIndent) {
    let lineBreaks = 0, ch = state2.input.charCodeAt(state2.position);
    while(ch !== 0){
        while(isWhiteSpace(ch)){
            ch = state2.input.charCodeAt(++state2.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state2.input.charCodeAt(++state2.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (isEOL(ch)) {
            readLineBreak(state2);
            ch = state2.input.charCodeAt(state2.position);
            lineBreaks++;
            state2.lineIndent = 0;
            while(ch === 32){
                state2.lineIndent++;
                ch = state2.input.charCodeAt(++state2.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state2.lineIndent < checkIndent) {
        throwWarning(state2, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state2) {
    let _position = state2.position;
    let ch = state2.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state2.input.charCodeAt(_position + 1) && ch === state2.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state2.input.charCodeAt(_position);
        if (ch === 0 || isWsOrEol(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state2, count) {
    if (count === 1) {
        state2.result += " ";
    } else if (count > 1) {
        state2.result += repeat("\n", count - 1);
    }
}
function readPlainScalar(state2, nodeIndent, withinFlowCollection) {
    const kind = state2.kind;
    const result = state2.result;
    let ch = state2.input.charCodeAt(state2.position);
    if (isWsOrEol(ch) || isFlowIndicator(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    let following;
    if (ch === 63 || ch === 45) {
        following = state2.input.charCodeAt(state2.position + 1);
        if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
            return false;
        }
    }
    state2.kind = "scalar";
    state2.result = "";
    let captureEnd, captureStart = captureEnd = state2.position;
    let hasPendingContent = false;
    let line2 = 0;
    while(ch !== 0){
        if (ch === 58) {
            following = state2.input.charCodeAt(state2.position + 1);
            if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
                break;
            }
        } else if (ch === 35) {
            const preceding = state2.input.charCodeAt(state2.position - 1);
            if (isWsOrEol(preceding)) {
                break;
            }
        } else if (state2.position === state2.lineStart && testDocumentSeparator(state2) || withinFlowCollection && isFlowIndicator(ch)) {
            break;
        } else if (isEOL(ch)) {
            line2 = state2.line;
            const lineStart = state2.lineStart;
            const lineIndent = state2.lineIndent;
            skipSeparationSpace(state2, false, -1);
            if (state2.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state2.input.charCodeAt(state2.position);
                continue;
            } else {
                state2.position = captureEnd;
                state2.line = line2;
                state2.lineStart = lineStart;
                state2.lineIndent = lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state2, captureStart, captureEnd, false);
            writeFoldedLines(state2, state2.line - line2);
            captureStart = captureEnd = state2.position;
            hasPendingContent = false;
        }
        if (!isWhiteSpace(ch)) {
            captureEnd = state2.position + 1;
        }
        ch = state2.input.charCodeAt(++state2.position);
    }
    captureSegment(state2, captureStart, captureEnd, false);
    if (state2.result) {
        return true;
    }
    state2.kind = kind;
    state2.result = result;
    return false;
}
function readSingleQuotedScalar(state2, nodeIndent) {
    let ch, captureStart, captureEnd;
    ch = state2.input.charCodeAt(state2.position);
    if (ch !== 39) {
        return false;
    }
    state2.kind = "scalar";
    state2.result = "";
    state2.position++;
    captureStart = captureEnd = state2.position;
    while((ch = state2.input.charCodeAt(state2.position)) !== 0){
        if (ch === 39) {
            captureSegment(state2, captureStart, state2.position, true);
            ch = state2.input.charCodeAt(++state2.position);
            if (ch === 39) {
                captureStart = state2.position;
                state2.position++;
                captureEnd = state2.position;
            } else {
                return true;
            }
        } else if (isEOL(ch)) {
            captureSegment(state2, captureStart, captureEnd, true);
            writeFoldedLines(state2, skipSeparationSpace(state2, false, nodeIndent));
            captureStart = captureEnd = state2.position;
        } else if (state2.position === state2.lineStart && testDocumentSeparator(state2)) {
            return throwError(state2, "unexpected end of the document within a single quoted scalar");
        } else {
            state2.position++;
            captureEnd = state2.position;
        }
    }
    return throwError(state2, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state2, nodeIndent) {
    let ch = state2.input.charCodeAt(state2.position);
    if (ch !== 34) {
        return false;
    }
    state2.kind = "scalar";
    state2.result = "";
    state2.position++;
    let captureEnd, captureStart = captureEnd = state2.position;
    let tmp;
    while((ch = state2.input.charCodeAt(state2.position)) !== 0){
        if (ch === 34) {
            captureSegment(state2, captureStart, state2.position, true);
            state2.position++;
            return true;
        }
        if (ch === 92) {
            captureSegment(state2, captureStart, state2.position, true);
            ch = state2.input.charCodeAt(++state2.position);
            if (isEOL(ch)) {
                skipSeparationSpace(state2, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state2.result += simpleEscapeMap[ch];
                state2.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                let hexLength = tmp;
                let hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state2.input.charCodeAt(++state2.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        return throwError(state2, "expected hexadecimal character");
                    }
                }
                state2.result += charFromCodepoint(hexResult);
                state2.position++;
            } else {
                return throwError(state2, "unknown escape sequence");
            }
            captureStart = captureEnd = state2.position;
        } else if (isEOL(ch)) {
            captureSegment(state2, captureStart, captureEnd, true);
            writeFoldedLines(state2, skipSeparationSpace(state2, false, nodeIndent));
            captureStart = captureEnd = state2.position;
        } else if (state2.position === state2.lineStart && testDocumentSeparator(state2)) {
            return throwError(state2, "unexpected end of the document within a double quoted scalar");
        } else {
            state2.position++;
            captureEnd = state2.position;
        }
    }
    return throwError(state2, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state2, nodeIndent) {
    let ch = state2.input.charCodeAt(state2.position);
    let terminator;
    let isMapping = true;
    let result = {
    };
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        result = [];
    } else if (ch === 123) {
        terminator = 125;
    } else {
        return false;
    }
    if (state2.anchor !== null && typeof state2.anchor != "undefined" && typeof state2.anchorMap != "undefined") {
        state2.anchorMap[state2.anchor] = result;
    }
    ch = state2.input.charCodeAt(++state2.position);
    const tag2 = state2.tag, anchor = state2.anchor;
    let readNext = true;
    let valueNode, keyNode, keyTag = keyNode = valueNode = null, isExplicitPair, isPair = isExplicitPair = false;
    let following = 0, line2 = 0;
    const overridableKeys = {
    };
    while(ch !== 0){
        skipSeparationSpace(state2, true, nodeIndent);
        ch = state2.input.charCodeAt(state2.position);
        if (ch === terminator) {
            state2.position++;
            state2.tag = tag2;
            state2.anchor = anchor;
            state2.kind = isMapping ? "mapping" : "sequence";
            state2.result = result;
            return true;
        }
        if (!readNext) {
            return throwError(state2, "missed comma between flow collection entries");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state2.input.charCodeAt(state2.position + 1);
            if (isWsOrEol(following)) {
                isPair = isExplicitPair = true;
                state2.position++;
                skipSeparationSpace(state2, true, nodeIndent);
            }
        }
        line2 = state2.line;
        composeNode(state2, nodeIndent, 1, false, true);
        keyTag = state2.tag || null;
        keyNode = state2.result;
        skipSeparationSpace(state2, true, nodeIndent);
        ch = state2.input.charCodeAt(state2.position);
        if ((isExplicitPair || state2.line === line2) && ch === 58) {
            isPair = true;
            ch = state2.input.charCodeAt(++state2.position);
            skipSeparationSpace(state2, true, nodeIndent);
            composeNode(state2, nodeIndent, 1, false, true);
            valueNode = state2.result;
        }
        if (isMapping) {
            storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, valueNode);
        } else if (isPair) {
            result.push(storeMappingPair(state2, null, overridableKeys, keyTag, keyNode, valueNode));
        } else {
            result.push(keyNode);
        }
        skipSeparationSpace(state2, true, nodeIndent);
        ch = state2.input.charCodeAt(state2.position);
        if (ch === 44) {
            readNext = true;
            ch = state2.input.charCodeAt(++state2.position);
        } else {
            readNext = false;
        }
    }
    return throwError(state2, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state2, nodeIndent) {
    let chomping = 1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false;
    let ch = state2.input.charCodeAt(state2.position);
    let folding = false;
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state2.kind = "scalar";
    state2.result = "";
    let tmp = 0;
    while(ch !== 0){
        ch = state2.input.charCodeAt(++state2.position);
        if (ch === 43 || ch === 45) {
            if (1 === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                return throwError(state2, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                return throwError(state2, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                return throwError(state2, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (isWhiteSpace(ch)) {
        do {
            ch = state2.input.charCodeAt(++state2.position);
        }while (isWhiteSpace(ch))
        if (ch === 35) {
            do {
                ch = state2.input.charCodeAt(++state2.position);
            }while (!isEOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state2);
        state2.lineIndent = 0;
        ch = state2.input.charCodeAt(state2.position);
        while((!detectedIndent || state2.lineIndent < textIndent) && ch === 32){
            state2.lineIndent++;
            ch = state2.input.charCodeAt(++state2.position);
        }
        if (!detectedIndent && state2.lineIndent > textIndent) {
            textIndent = state2.lineIndent;
        }
        if (isEOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state2.lineIndent < textIndent) {
            if (chomping === 3) {
                state2.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === 1) {
                if (didReadContent) {
                    state2.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (isWhiteSpace(ch)) {
                atMoreIndented = true;
                state2.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state2.result += repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state2.result += " ";
                }
            } else {
                state2.result += repeat("\n", emptyLines);
            }
        } else {
            state2.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        const captureStart = state2.position;
        while(!isEOL(ch) && ch !== 0){
            ch = state2.input.charCodeAt(++state2.position);
        }
        captureSegment(state2, captureStart, state2.position, false);
    }
    return true;
}
function readBlockSequence(state2, nodeIndent) {
    let line2, following, detected = false, ch;
    const tag2 = state2.tag, anchor = state2.anchor, result = [];
    if (state2.anchor !== null && typeof state2.anchor !== "undefined" && typeof state2.anchorMap !== "undefined") {
        state2.anchorMap[state2.anchor] = result;
    }
    ch = state2.input.charCodeAt(state2.position);
    while(ch !== 0){
        if (ch !== 45) {
            break;
        }
        following = state2.input.charCodeAt(state2.position + 1);
        if (!isWsOrEol(following)) {
            break;
        }
        detected = true;
        state2.position++;
        if (skipSeparationSpace(state2, true, -1)) {
            if (state2.lineIndent <= nodeIndent) {
                result.push(null);
                ch = state2.input.charCodeAt(state2.position);
                continue;
            }
        }
        line2 = state2.line;
        composeNode(state2, nodeIndent, 3, false, true);
        result.push(state2.result);
        skipSeparationSpace(state2, true, -1);
        ch = state2.input.charCodeAt(state2.position);
        if ((state2.line === line2 || state2.lineIndent > nodeIndent) && ch !== 0) {
            return throwError(state2, "bad indentation of a sequence entry");
        } else if (state2.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state2.tag = tag2;
        state2.anchor = anchor;
        state2.kind = "sequence";
        state2.result = result;
        return true;
    }
    return false;
}
function readBlockMapping(state2, nodeIndent, flowIndent) {
    const tag2 = state2.tag, anchor = state2.anchor, result = {
    }, overridableKeys = {
    };
    let following, allowCompact = false, line2, pos, keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state2.anchor !== null && typeof state2.anchor !== "undefined" && typeof state2.anchorMap !== "undefined") {
        state2.anchorMap[state2.anchor] = result;
    }
    ch = state2.input.charCodeAt(state2.position);
    while(ch !== 0){
        following = state2.input.charCodeAt(state2.position + 1);
        line2 = state2.line;
        pos = state2.position;
        if ((ch === 63 || ch === 58) && isWsOrEol(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, null);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                return throwError(state2, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state2.position += 1;
            ch = following;
        } else if (composeNode(state2, flowIndent, 2, false, true)) {
            if (state2.line === line2) {
                ch = state2.input.charCodeAt(state2.position);
                while(isWhiteSpace(ch)){
                    ch = state2.input.charCodeAt(++state2.position);
                }
                if (ch === 58) {
                    ch = state2.input.charCodeAt(++state2.position);
                    if (!isWsOrEol(ch)) {
                        return throwError(state2, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, null);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state2.tag;
                    keyNode = state2.result;
                } else if (detected) {
                    return throwError(state2, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state2.tag = tag2;
                    state2.anchor = anchor;
                    return true;
                }
            } else if (detected) {
                return throwError(state2, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state2.tag = tag2;
                state2.anchor = anchor;
                return true;
            }
        } else {
            break;
        }
        if (state2.line === line2 || state2.lineIndent > nodeIndent) {
            if (composeNode(state2, nodeIndent, 4, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state2.result;
                } else {
                    valueNode = state2.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, valueNode, line2, pos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state2, true, -1);
            ch = state2.input.charCodeAt(state2.position);
        }
        if (state2.lineIndent > nodeIndent && ch !== 0) {
            return throwError(state2, "bad indentation of a mapping entry");
        } else if (state2.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state2, result, overridableKeys, keyTag, keyNode, null);
    }
    if (detected) {
        state2.tag = tag2;
        state2.anchor = anchor;
        state2.kind = "mapping";
        state2.result = result;
    }
    return detected;
}
function readTagProperty(state2) {
    let position2, isVerbatim = false, isNamed = false, tagHandle = "", tagName, ch;
    ch = state2.input.charCodeAt(state2.position);
    if (ch !== 33) return false;
    if (state2.tag !== null) {
        return throwError(state2, "duplication of a tag property");
    }
    ch = state2.input.charCodeAt(++state2.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state2.input.charCodeAt(++state2.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state2.input.charCodeAt(++state2.position);
    } else {
        tagHandle = "!";
    }
    position2 = state2.position;
    if (isVerbatim) {
        do {
            ch = state2.input.charCodeAt(++state2.position);
        }while (ch !== 0 && ch !== 62)
        if (state2.position < state2.length) {
            tagName = state2.input.slice(position2, state2.position);
            ch = state2.input.charCodeAt(++state2.position);
        } else {
            return throwError(state2, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !isWsOrEol(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state2.input.slice(position2 - 1, state2.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        return throwError(state2, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    position2 = state2.position + 1;
                } else {
                    return throwError(state2, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state2.input.charCodeAt(++state2.position);
        }
        tagName = state2.input.slice(position2, state2.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            return throwError(state2, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        return throwError(state2, `tag name cannot contain such characters: ${tagName}`);
    }
    if (isVerbatim) {
        state2.tag = tagName;
    } else if (typeof state2.tagMap !== "undefined" && _hasOwnProperty2.call(state2.tagMap, tagHandle)) {
        state2.tag = state2.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state2.tag = `!${tagName}`;
    } else if (tagHandle === "!!") {
        state2.tag = `tag:yaml.org,2002:${tagName}`;
    } else {
        return throwError(state2, `undeclared tag handle "${tagHandle}"`);
    }
    return true;
}
function readAnchorProperty(state2) {
    let ch = state2.input.charCodeAt(state2.position);
    if (ch !== 38) return false;
    if (state2.anchor !== null) {
        return throwError(state2, "duplication of an anchor property");
    }
    ch = state2.input.charCodeAt(++state2.position);
    const position2 = state2.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state2.input.charCodeAt(++state2.position);
    }
    if (state2.position === position2) {
        return throwError(state2, "name of an anchor node must contain at least one character");
    }
    state2.anchor = state2.input.slice(position2, state2.position);
    return true;
}
function readAlias(state2) {
    let ch = state2.input.charCodeAt(state2.position);
    if (ch !== 42) return false;
    ch = state2.input.charCodeAt(++state2.position);
    const _position = state2.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state2.input.charCodeAt(++state2.position);
    }
    if (state2.position === _position) {
        return throwError(state2, "name of an alias node must contain at least one character");
    }
    const alias2 = state2.input.slice(_position, state2.position);
    if (typeof state2.anchorMap !== "undefined" && !Object.prototype.hasOwnProperty.call(state2.anchorMap, alias2)) {
        return throwError(state2, `unidentified alias "${alias2}"`);
    }
    if (typeof state2.anchorMap !== "undefined") {
        state2.result = state2.anchorMap[alias2];
    }
    skipSeparationSpace(state2, true, -1);
    return true;
}
function composeNode(state2, parentIndent, nodeContext, allowToSeek, allowCompact) {
    let allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, type4, flowIndent, blockIndent;
    if (state2.listener && state2.listener !== null) {
        state2.listener("open", state2);
    }
    state2.tag = null;
    state2.anchor = null;
    state2.kind = null;
    state2.result = null;
    const allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state2, true, -1)) {
            atNewLine = true;
            if (state2.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state2.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state2.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state2) || readAnchorProperty(state2)){
            if (skipSeparationSpace(state2, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state2.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state2.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state2.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || 4 === nodeContext) {
        const cond = 1 === nodeContext || 2 === nodeContext;
        flowIndent = cond ? parentIndent : parentIndent + 1;
        blockIndent = state2.position - state2.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state2, blockIndent) || readBlockMapping(state2, blockIndent, flowIndent)) || readFlowCollection(state2, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state2, flowIndent) || readSingleQuotedScalar(state2, flowIndent) || readDoubleQuotedScalar(state2, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state2)) {
                    hasContent = true;
                    if (state2.tag !== null || state2.anchor !== null) {
                        return throwError(state2, "alias node should not have Any properties");
                    }
                } else if (readPlainScalar(state2, flowIndent, 1 === nodeContext)) {
                    hasContent = true;
                    if (state2.tag === null) {
                        state2.tag = "?";
                    }
                }
                if (state2.anchor !== null && typeof state2.anchorMap !== "undefined") {
                    state2.anchorMap[state2.anchor] = state2.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state2, blockIndent);
        }
    }
    if (state2.tag !== null && state2.tag !== "!") {
        if (state2.tag === "?") {
            for(let typeIndex = 0, typeQuantity = state2.implicitTypes.length; typeIndex < typeQuantity; typeIndex++){
                type4 = state2.implicitTypes[typeIndex];
                if (type4.resolve(state2.result)) {
                    state2.result = type4.construct(state2.result);
                    state2.tag = type4.tag;
                    if (state2.anchor !== null && typeof state2.anchorMap !== "undefined") {
                        state2.anchorMap[state2.anchor] = state2.result;
                    }
                    break;
                }
            }
        } else if (_hasOwnProperty2.call(state2.typeMap[state2.kind || "fallback"], state2.tag)) {
            type4 = state2.typeMap[state2.kind || "fallback"][state2.tag];
            if (state2.result !== null && type4.kind !== state2.kind) {
                return throwError(state2, `unacceptable node kind for !<${state2.tag}> tag; it should be "${type4.kind}", not "${state2.kind}"`);
            }
            if (!type4.resolve(state2.result)) {
                return throwError(state2, `cannot resolve a node with !<${state2.tag}> explicit tag`);
            } else {
                state2.result = type4.construct(state2.result);
                if (state2.anchor !== null && typeof state2.anchorMap !== "undefined") {
                    state2.anchorMap[state2.anchor] = state2.result;
                }
            }
        } else {
            return throwError(state2, `unknown tag !<${state2.tag}>`);
        }
    }
    if (state2.listener && state2.listener !== null) {
        state2.listener("close", state2);
    }
    return state2.tag !== null || state2.anchor !== null || hasContent;
}
function readDocument(state2) {
    const documentStart = state2.position;
    let position2, directiveName, directiveArgs, hasDirectives = false, ch;
    state2.version = null;
    state2.checkLineBreaks = state2.legacy;
    state2.tagMap = {
    };
    state2.anchorMap = {
    };
    while((ch = state2.input.charCodeAt(state2.position)) !== 0){
        skipSeparationSpace(state2, true, -1);
        ch = state2.input.charCodeAt(state2.position);
        if (state2.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state2.input.charCodeAt(++state2.position);
        position2 = state2.position;
        while(ch !== 0 && !isWsOrEol(ch)){
            ch = state2.input.charCodeAt(++state2.position);
        }
        directiveName = state2.input.slice(position2, state2.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            return throwError(state2, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(isWhiteSpace(ch)){
                ch = state2.input.charCodeAt(++state2.position);
            }
            if (ch === 35) {
                do {
                    ch = state2.input.charCodeAt(++state2.position);
                }while (ch !== 0 && !isEOL(ch))
                break;
            }
            if (isEOL(ch)) break;
            position2 = state2.position;
            while(ch !== 0 && !isWsOrEol(ch)){
                ch = state2.input.charCodeAt(++state2.position);
            }
            directiveArgs.push(state2.input.slice(position2, state2.position));
        }
        if (ch !== 0) readLineBreak(state2);
        if (_hasOwnProperty2.call(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state2, directiveName, ...directiveArgs);
        } else {
            throwWarning(state2, `unknown document directive "${directiveName}"`);
        }
    }
    skipSeparationSpace(state2, true, -1);
    if (state2.lineIndent === 0 && state2.input.charCodeAt(state2.position) === 45 && state2.input.charCodeAt(state2.position + 1) === 45 && state2.input.charCodeAt(state2.position + 2) === 45) {
        state2.position += 3;
        skipSeparationSpace(state2, true, -1);
    } else if (hasDirectives) {
        return throwError(state2, "directives end mark is expected");
    }
    composeNode(state2, state2.lineIndent - 1, 4, false, true);
    skipSeparationSpace(state2, true, -1);
    if (state2.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state2.input.slice(documentStart, state2.position))) {
        throwWarning(state2, "non-ASCII line breaks are interpreted as content");
    }
    state2.documents.push(state2.result);
    if (state2.position === state2.lineStart && testDocumentSeparator(state2)) {
        if (state2.input.charCodeAt(state2.position) === 46) {
            state2.position += 3;
            skipSeparationSpace(state2, true, -1);
        }
        return;
    }
    if (state2.position < state2.length - 1) {
        return throwError(state2, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments(input1, options12) {
    input1 = String(input1);
    options12 = options12 || {
    };
    if (input1.length !== 0) {
        if (input1.charCodeAt(input1.length - 1) !== 10 && input1.charCodeAt(input1.length - 1) !== 13) {
            input1 += "\n";
        }
        if (input1.charCodeAt(0) === 65279) {
            input1 = input1.slice(1);
        }
    }
    const state2 = new LoaderState(input1, options12);
    state2.input += "\0";
    while(state2.input.charCodeAt(state2.position) === 32){
        state2.lineIndent += 1;
        state2.position += 1;
    }
    while(state2.position < state2.length - 1){
        readDocument(state2);
    }
    return state2.documents;
}
function load(input1, options12) {
    const documents = loadDocuments(input1, options12);
    if (documents.length === 0) {
        return;
    }
    if (documents.length === 1) {
        return documents[0];
    }
    throw new YAMLError("expected a single document in the stream, but found more");
}
function parse3(content, options12) {
    return load(content, options12);
}
const _hasOwnProperty3 = Object.prototype.hasOwnProperty;
function compileStyleMap(schema2, map1) {
    if (typeof map1 === "undefined" || map1 === null) return {
    };
    let type4;
    const result = {
    };
    const keys = Object.keys(map1);
    let tag2, style;
    for(let index = 0, length = keys.length; index < length; index += 1){
        tag2 = keys[index];
        style = String(map1[tag2]);
        if (tag2.slice(0, 2) === "!!") {
            tag2 = `tag:yaml.org,2002:${tag2.slice(2)}`;
        }
        type4 = schema2.compiledTypeMap.fallback[tag2];
        if (type4 && typeof type4.styleAliases !== "undefined" && _hasOwnProperty3.call(type4.styleAliases, style)) {
            style = type4.styleAliases[style];
        }
        result[tag2] = style;
    }
    return result;
}
class DumperState extends State {
    indent;
    noArrayIndent;
    skipInvalid;
    flowLevel;
    sortKeys;
    lineWidth;
    noRefs;
    noCompatMode;
    condenseFlow;
    implicitTypes;
    explicitTypes;
    tag = null;
    result = "";
    duplicates = [];
    usedDuplicates = [];
    styleMap;
    dump;
    constructor({ schema: schema2 , indent: indent1 = 2 , noArrayIndent =false , skipInvalid =false , flowLevel =-1 , styles =null , sortKeys =false , lineWidth =80 , noRefs =false , noCompatMode =false , condenseFlow =false  }){
        super(schema2);
        this.indent = Math.max(1, indent1);
        this.noArrayIndent = noArrayIndent;
        this.skipInvalid = skipInvalid;
        this.flowLevel = flowLevel;
        this.styleMap = compileStyleMap(this.schema, styles);
        this.sortKeys = sortKeys;
        this.lineWidth = lineWidth;
        this.noRefs = noRefs;
        this.noCompatMode = noCompatMode;
        this.condenseFlow = condenseFlow;
        this.implicitTypes = this.schema.compiledImplicit;
        this.explicitTypes = this.schema.compiledExplicit;
    }
}
const _toString2 = Object.prototype.toString;
const _hasOwnProperty4 = Object.prototype.hasOwnProperty;
const ESCAPE_SEQUENCES = {
};
ESCAPE_SEQUENCES[0] = "\\0";
ESCAPE_SEQUENCES[7] = "\\a";
ESCAPE_SEQUENCES[8] = "\\b";
ESCAPE_SEQUENCES[9] = "\\t";
ESCAPE_SEQUENCES[10] = "\\n";
ESCAPE_SEQUENCES[11] = "\\v";
ESCAPE_SEQUENCES[12] = "\\f";
ESCAPE_SEQUENCES[13] = "\\r";
ESCAPE_SEQUENCES[27] = "\\e";
ESCAPE_SEQUENCES[34] = '\\"';
ESCAPE_SEQUENCES[92] = "\\\\";
ESCAPE_SEQUENCES[133] = "\\N";
ESCAPE_SEQUENCES[160] = "\\_";
ESCAPE_SEQUENCES[8232] = "\\L";
ESCAPE_SEQUENCES[8233] = "\\P";
const DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF", 
];
function encodeHex(character) {
    const string1 = character.toString(16).toUpperCase();
    let handle;
    let length;
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new YAMLError("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return `\\${handle}${repeat("0", length - string1.length)}${string1}`;
}
function indentString(string1, spaces) {
    const ind = repeat(" ", spaces), length = string1.length;
    let position2 = 0, next = -1, result = "", line2;
    while(position2 < length){
        next = string1.indexOf("\n", position2);
        if (next === -1) {
            line2 = string1.slice(position2);
            position2 = length;
        } else {
            line2 = string1.slice(position2, next + 1);
            position2 = next + 1;
        }
        if (line2.length && line2 !== "\n") result += ind;
        result += line2;
    }
    return result;
}
function generateNextLine(state2, level) {
    return `\n${repeat(" ", state2.indent * level)}`;
}
function testImplicitResolving(state2, str1) {
    let type4;
    for(let index = 0, length = state2.implicitTypes.length; index < length; index += 1){
        type4 = state2.implicitTypes[index];
        if (type4.resolve(str1)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === 32 || c === 9;
}
function isPrintable(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== 65279 || 65536 <= c && c <= 1114111;
}
function isPlainSafe(c) {
    return isPrintable(c) && c !== 65279 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 58 && c !== 35;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== 65279 && !isWhitespace(c) && c !== 45 && c !== 63 && c !== 58 && c !== 44 && c !== 91 && c !== 93 && c !== 123 && c !== 125 && c !== 35 && c !== 38 && c !== 42 && c !== 33 && c !== 124 && c !== 62 && c !== 39 && c !== 34 && c !== 37 && c !== 64 && c !== 96;
}
function needIndentIndicator(string1) {
    const leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string1);
}
const STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string1, singleLineOnly, indentPerLevel, lineWidth1, testAmbiguousType) {
    const shouldTrackWidth = lineWidth1 !== -1;
    let hasLineBreak = false, hasFoldableLine = false, previousLineBreak = -1, plain = isPlainSafeFirst(string1.charCodeAt(0)) && !isWhitespace(string1.charCodeAt(string1.length - 1));
    let __char, i1;
    if (singleLineOnly) {
        for(i1 = 0; i1 < string1.length; i1++){
            __char = string1.charCodeAt(i1);
            if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
    } else {
        for(i1 = 0; i1 < string1.length; i1++){
            __char = string1.charCodeAt(i1);
            if (__char === 10) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth1 && string1[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth1 && string1[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        return plain && !testAmbiguousType(string1) ? 1 : 2;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string1)) {
        return 5;
    }
    return hasFoldableLine ? 4 : 3;
}
function foldLine(line2, width) {
    if (line2 === "" || line2[0] === " ") return line2;
    const breakRe = / [^ ]/g;
    let match;
    let start = 0, end, curr = 0, next = 0;
    let result = "";
    while(match = breakRe.exec(line2)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result += `\n${line2.slice(start, end)}`;
            start = end + 1;
        }
        curr = next;
    }
    result += "\n";
    if (line2.length - start > width && curr > start) {
        result += `${line2.slice(start, curr)}\n${line2.slice(curr + 1)}`;
    } else {
        result += line2.slice(start);
    }
    return result.slice(1);
}
function dropEndingNewline(string1) {
    return string1[string1.length - 1] === "\n" ? string1.slice(0, -1) : string1;
}
function foldString(string1, width) {
    const lineRe = /(\n+)([^\n]*)/g;
    let result = (()=>{
        let nextLF = string1.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string1.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string1.slice(0, nextLF), width);
    })();
    let prevMoreIndented = string1[0] === "\n" || string1[0] === " ";
    let moreIndented;
    let match;
    while(match = lineRe.exec(string1)){
        const prefix = match[1], line2 = match[2];
        moreIndented = line2[0] === " ";
        result += prefix + (!prevMoreIndented && !moreIndented && line2 !== "" ? "\n" : "") + foldLine(line2, width);
        prevMoreIndented = moreIndented;
    }
    return result;
}
function escapeString(string1) {
    let result = "";
    let __char, nextChar;
    let escapeSeq;
    for(let i1 = 0; i1 < string1.length; i1++){
        __char = string1.charCodeAt(i1);
        if (__char >= 55296 && __char <= 56319) {
            nextChar = string1.charCodeAt(i1 + 1);
            if (nextChar >= 56320 && nextChar <= 57343) {
                result += encodeHex((__char - 55296) * 1024 + nextChar - 56320 + 65536);
                i1++;
                continue;
            }
        }
        escapeSeq = ESCAPE_SEQUENCES[__char];
        result += !escapeSeq && isPrintable(__char) ? string1[i1] : escapeSeq || encodeHex(__char);
    }
    return result;
}
function blockHeader(string1, indentPerLevel) {
    const indentIndicator = needIndentIndicator(string1) ? String(indentPerLevel) : "";
    const clip = string1[string1.length - 1] === "\n";
    const keep = clip && (string1[string1.length - 2] === "\n" || string1 === "\n");
    const chomp = keep ? "+" : clip ? "" : "-";
    return `${indentIndicator}${chomp}\n`;
}
function writeScalar(state2, string1, level, iskey) {
    state2.dump = (()=>{
        if (string1.length === 0) {
            return "''";
        }
        if (!state2.noCompatMode && DEPRECATED_BOOLEANS_SYNTAX.indexOf(string1) !== -1) {
            return `'${string1}'`;
        }
        const indent1 = state2.indent * Math.max(1, level);
        const lineWidth1 = state2.lineWidth === -1 ? -1 : Math.max(Math.min(state2.lineWidth, 40), state2.lineWidth - indent1);
        const singleLineOnly = iskey || state2.flowLevel > -1 && level >= state2.flowLevel;
        function testAmbiguity(str1) {
            return testImplicitResolving(state2, str1);
        }
        switch(chooseScalarStyle(string1, singleLineOnly, state2.indent, lineWidth1, testAmbiguity)){
            case STYLE_PLAIN:
                return string1;
            case STYLE_SINGLE:
                return `'${string1.replace(/'/g, "''")}'`;
            case STYLE_LITERAL:
                return `|${blockHeader(string1, state2.indent)}${dropEndingNewline(indentString(string1, indent1))}`;
            case STYLE_FOLDED:
                return `>${blockHeader(string1, state2.indent)}${dropEndingNewline(indentString(foldString(string1, lineWidth1), indent1))}`;
            case STYLE_DOUBLE:
                return `"${escapeString(string1)}"`;
            default:
                throw new YAMLError("impossible error: invalid scalar style");
        }
    })();
}
function writeFlowSequence(state2, level, object) {
    let _result = "";
    const _tag = state2.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state2, level, object[index], false, false)) {
            if (index !== 0) _result += `,${!state2.condenseFlow ? " " : ""}`;
            _result += state2.dump;
        }
    }
    state2.tag = _tag;
    state2.dump = `[${_result}]`;
}
function writeBlockSequence(state2, level, object, compact = false) {
    let _result = "";
    const _tag = state2.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state2, level + 1, object[index], true, true)) {
            if (!compact || index !== 0) {
                _result += generateNextLine(state2, level);
            }
            if (state2.dump && 10 === state2.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state2.dump;
        }
    }
    state2.tag = _tag;
    state2.dump = _result || "[]";
}
function writeFlowMapping(state2, level, object) {
    let _result = "";
    const _tag = state2.tag, objectKeyList = Object.keys(object);
    let pairBuffer, objectKey, objectValue;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = state2.condenseFlow ? '"' : "";
        if (index !== 0) pairBuffer += ", ";
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state2, level, objectKey, false, false)) {
            continue;
        }
        if (state2.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += `${state2.dump}${state2.condenseFlow ? '"' : ""}:${state2.condenseFlow ? "" : " "}`;
        if (!writeNode(state2, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state2.dump;
        _result += pairBuffer;
    }
    state2.tag = _tag;
    state2.dump = `{${_result}}`;
}
function writeBlockMapping(state2, level, object, compact = false) {
    const _tag = state2.tag, objectKeyList = Object.keys(object);
    let _result = "";
    if (state2.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state2.sortKeys === "function") {
        objectKeyList.sort(state2.sortKeys);
    } else if (state2.sortKeys) {
        throw new YAMLError("sortKeys must be a boolean or a function");
    }
    let pairBuffer = "", objectKey, objectValue, explicitPair;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || index !== 0) {
            pairBuffer += generateNextLine(state2, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state2, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state2.tag !== null && state2.tag !== "?" || state2.dump && state2.dump.length > 1024;
        if (explicitPair) {
            if (state2.dump && 10 === state2.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state2.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state2, level);
        }
        if (!writeNode(state2, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state2.dump && 10 === state2.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state2.dump;
        _result += pairBuffer;
    }
    state2.tag = _tag;
    state2.dump = _result || "{}";
}
function detectType(state2, object, explicit = false) {
    const typeList = explicit ? state2.explicitTypes : state2.implicitTypes;
    let type4;
    let style;
    let _result;
    for(let index = 0, length = typeList.length; index < length; index += 1){
        type4 = typeList[index];
        if ((type4.instanceOf || type4.predicate) && (!type4.instanceOf || typeof object === "object" && object instanceof type4.instanceOf) && (!type4.predicate || type4.predicate(object))) {
            state2.tag = explicit ? type4.tag : "?";
            if (type4.represent) {
                style = state2.styleMap[type4.tag] || type4.defaultStyle;
                if (_toString2.call(type4.represent) === "[object Function]") {
                    _result = type4.represent(object, style);
                } else if (_hasOwnProperty4.call(type4.represent, style)) {
                    _result = type4.represent[style](object, style);
                } else {
                    throw new YAMLError(`!<${type4.tag}> tag resolver accepts not "${style}" style`);
                }
                state2.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state2, level, object, block, compact, iskey = false) {
    state2.tag = null;
    state2.dump = object;
    if (!detectType(state2, object, false)) {
        detectType(state2, object, true);
    }
    const type4 = _toString2.call(state2.dump);
    if (block) {
        block = state2.flowLevel < 0 || state2.flowLevel > level;
    }
    const objectOrArray = type4 === "[object Object]" || type4 === "[object Array]";
    let duplicateIndex = -1;
    let duplicate = false;
    if (objectOrArray) {
        duplicateIndex = state2.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state2.tag !== null && state2.tag !== "?" || duplicate || state2.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state2.usedDuplicates[duplicateIndex]) {
        state2.dump = `*ref_${duplicateIndex}`;
    } else {
        if (objectOrArray && duplicate && !state2.usedDuplicates[duplicateIndex]) {
            state2.usedDuplicates[duplicateIndex] = true;
        }
        if (type4 === "[object Object]") {
            if (block && Object.keys(state2.dump).length !== 0) {
                writeBlockMapping(state2, level, state2.dump, compact);
                if (duplicate) {
                    state2.dump = `&ref_${duplicateIndex}${state2.dump}`;
                }
            } else {
                writeFlowMapping(state2, level, state2.dump);
                if (duplicate) {
                    state2.dump = `&ref_${duplicateIndex} ${state2.dump}`;
                }
            }
        } else if (type4 === "[object Array]") {
            const arrayLevel = state2.noArrayIndent && level > 0 ? level - 1 : level;
            if (block && state2.dump.length !== 0) {
                writeBlockSequence(state2, arrayLevel, state2.dump, compact);
                if (duplicate) {
                    state2.dump = `&ref_${duplicateIndex}${state2.dump}`;
                }
            } else {
                writeFlowSequence(state2, arrayLevel, state2.dump);
                if (duplicate) {
                    state2.dump = `&ref_${duplicateIndex} ${state2.dump}`;
                }
            }
        } else if (type4 === "[object String]") {
            if (state2.tag !== "?") {
                writeScalar(state2, state2.dump, level, iskey);
            }
        } else {
            if (state2.skipInvalid) return false;
            throw new YAMLError(`unacceptable kind of an object to dump ${type4}`);
        }
        if (state2.tag !== null && state2.tag !== "?") {
            state2.dump = `!<${state2.tag}> ${state2.dump}`;
        }
    }
    return true;
}
function inspectNode(object, objects, duplicatesIndexes) {
    if (object !== null && typeof object === "object") {
        const index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(let idx = 0, length = object.length; idx < length; idx += 1){
                    inspectNode(object[idx], objects, duplicatesIndexes);
                }
            } else {
                const objectKeyList = Object.keys(object);
                for(let idx = 0, length = objectKeyList.length; idx < length; idx += 1){
                    inspectNode(object[objectKeyList[idx]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function getDuplicateReferences(object, state2) {
    const objects = [], duplicatesIndexes = [];
    inspectNode(object, objects, duplicatesIndexes);
    const length = duplicatesIndexes.length;
    for(let index = 0; index < length; index += 1){
        state2.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state2.usedDuplicates = new Array(length);
}
function dump(input1, options12) {
    options12 = options12 || {
    };
    const state2 = new DumperState(options12);
    if (!state2.noRefs) getDuplicateReferences(input1, state2);
    if (writeNode(state2, 0, input1, true, true)) return `${state2.dump}\n`;
    return "";
}
function stringify(obj, options12) {
    return dump(obj, options12);
}
var State1;
(function(State2) {
    State2[State2["PASSTHROUGH"] = 0] = "PASSTHROUGH";
    State2[State2["PERCENT"] = 1] = "PERCENT";
    State2[State2["POSITIONAL"] = 2] = "POSITIONAL";
    State2[State2["PRECISION"] = 3] = "PRECISION";
    State2[State2["WIDTH"] = 4] = "WIDTH";
})(State1 || (State1 = {
}));
var WorP;
(function(WorP1) {
    WorP1[WorP1["WIDTH"] = 0] = "WIDTH";
    WorP1[WorP1["PRECISION"] = 1] = "PRECISION";
})(WorP || (WorP = {
}));
class Flags {
    plus;
    dash;
    sharp;
    space;
    zero;
    lessthan;
    width = -1;
    precision = -1;
}
const min = Math.min;
const UNICODE_REPLACEMENT_CHARACTER = "\ufffd";
const FLOAT_REGEXP = /(-?)(\d)\.?(\d*)e([+-])(\d+)/;
var F;
(function(F1) {
    F1[F1["sign"] = 1] = "sign";
    F1[F1["mantissa"] = 2] = "mantissa";
    F1[F1["fractional"] = 3] = "fractional";
    F1[F1["esign"] = 4] = "esign";
    F1[F1["exponent"] = 5] = "exponent";
})(F || (F = {
}));
class Printf {
    format;
    args;
    i;
    state = State1.PASSTHROUGH;
    verb = "";
    buf = "";
    argNum = 0;
    flags = new Flags();
    haveSeen;
    tmpError;
    constructor(format4, ...args3){
        this.format = format4;
        this.args = args3;
        this.haveSeen = new Array(args3.length);
        this.i = 0;
    }
    doPrintf() {
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PASSTHROUGH:
                    if (c === "%") {
                        this.state = State1.PERCENT;
                    } else {
                        this.buf += c;
                    }
                    break;
                case State1.PERCENT:
                    if (c === "%") {
                        this.buf += c;
                        this.state = State1.PASSTHROUGH;
                    } else {
                        this.handleFormat();
                    }
                    break;
                default:
                    throw Error("Should be unreachable, certainly a bug in the lib.");
            }
        }
        let extras = false;
        let err = "%!(EXTRA";
        for(let i2 = 0; i2 !== this.haveSeen.length; ++i2){
            if (!this.haveSeen[i2]) {
                extras = true;
                err += ` '${Deno.inspect(this.args[i2])}'`;
            }
        }
        err += ")";
        if (extras) {
            this.buf += err;
        }
        return this.buf;
    }
    handleFormat() {
        this.flags = new Flags();
        const flags = this.flags;
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PERCENT:
                    switch(c){
                        case "[":
                            this.handlePositional();
                            this.state = State1.POSITIONAL;
                            break;
                        case "+":
                            flags.plus = true;
                            break;
                        case "<":
                            flags.lessthan = true;
                            break;
                        case "-":
                            flags.dash = true;
                            flags.zero = false;
                            break;
                        case "#":
                            flags.sharp = true;
                            break;
                        case " ":
                            flags.space = true;
                            break;
                        case "0":
                            flags.zero = !flags.dash;
                            break;
                        default:
                            if ("1" <= c && c <= "9" || c === "." || c === "*") {
                                if (c === ".") {
                                    this.flags.precision = 0;
                                    this.state = State1.PRECISION;
                                    this.i++;
                                } else {
                                    this.state = State1.WIDTH;
                                }
                                this.handleWidthAndPrecision(flags);
                            } else {
                                this.handleVerb();
                                return;
                            }
                    }
                    break;
                case State1.POSITIONAL:
                    if (c === "*") {
                        const worp = this.flags.precision === -1 ? WorP.WIDTH : WorP.PRECISION;
                        this.handleWidthOrPrecisionRef(worp);
                        this.state = State1.PERCENT;
                        break;
                    } else {
                        this.handleVerb();
                        return;
                    }
                default:
                    throw new Error(`Should not be here ${this.state}, library bug!`);
            }
        }
    }
    handleWidthOrPrecisionRef(wOrP) {
        if (this.argNum >= this.args.length) {
            return;
        }
        const arg4 = this.args[this.argNum];
        this.haveSeen[this.argNum] = true;
        if (typeof arg4 === "number") {
            switch(wOrP){
                case WorP.WIDTH:
                    this.flags.width = arg4;
                    break;
                default:
                    this.flags.precision = arg4;
            }
        } else {
            const tmp = wOrP === WorP.WIDTH ? "WIDTH" : "PREC";
            this.tmpError = `%!(BAD ${tmp} '${this.args[this.argNum]}')`;
        }
        this.argNum++;
    }
    handleWidthAndPrecision(flags) {
        const fmt = this.format;
        for(; this.i !== this.format.length; ++this.i){
            const c = fmt[this.i];
            switch(this.state){
                case State1.WIDTH:
                    switch(c){
                        case ".":
                            this.flags.precision = 0;
                            this.state = State1.PRECISION;
                            break;
                        case "*":
                            this.handleWidthOrPrecisionRef(WorP.WIDTH);
                            break;
                        default:
                            {
                                const val = parseInt(c);
                                if (isNaN(val)) {
                                    this.i--;
                                    this.state = State1.PERCENT;
                                    return;
                                }
                                flags.width = flags.width == -1 ? 0 : flags.width;
                                flags.width *= 10;
                                flags.width += val;
                            }
                    }
                    break;
                case State1.PRECISION:
                    {
                        if (c === "*") {
                            this.handleWidthOrPrecisionRef(WorP.PRECISION);
                            break;
                        }
                        const val = parseInt(c);
                        if (isNaN(val)) {
                            this.i--;
                            this.state = State1.PERCENT;
                            return;
                        }
                        flags.precision *= 10;
                        flags.precision += val;
                        break;
                    }
                default:
                    throw new Error("can't be here. bug.");
            }
        }
    }
    handlePositional() {
        if (this.format[this.i] !== "[") {
            throw new Error("Can't happen? Bug.");
        }
        let positional = 0;
        const format5 = this.format;
        this.i++;
        let err = false;
        for(; this.i !== this.format.length; ++this.i){
            if (format5[this.i] === "]") {
                break;
            }
            positional *= 10;
            const val = parseInt(format5[this.i]);
            if (isNaN(val)) {
                this.tmpError = "%!(BAD INDEX)";
                err = true;
            }
            positional += val;
        }
        if (positional - 1 >= this.args.length) {
            this.tmpError = "%!(BAD INDEX)";
            err = true;
        }
        this.argNum = err ? this.argNum : positional - 1;
        return;
    }
    handleLessThan() {
        const arg4 = this.args[this.argNum];
        if ((arg4 || {
        }).constructor.name !== "Array") {
            throw new Error(`arg ${arg4} is not an array. Todo better error handling`);
        }
        let str1 = "[ ";
        for(let i2 = 0; i2 !== arg4.length; ++i2){
            if (i2 !== 0) str1 += ", ";
            str1 += this._handleVerb(arg4[i2]);
        }
        return str1 + " ]";
    }
    handleVerb() {
        const verb = this.format[this.i];
        this.verb = verb;
        if (this.tmpError) {
            this.buf += this.tmpError;
            this.tmpError = undefined;
            if (this.argNum < this.haveSeen.length) {
                this.haveSeen[this.argNum] = true;
            }
        } else if (this.args.length <= this.argNum) {
            this.buf += `%!(MISSING '${verb}')`;
        } else {
            const arg4 = this.args[this.argNum];
            this.haveSeen[this.argNum] = true;
            if (this.flags.lessthan) {
                this.buf += this.handleLessThan();
            } else {
                this.buf += this._handleVerb(arg4);
            }
        }
        this.argNum++;
        this.state = State1.PASSTHROUGH;
    }
    _handleVerb(arg) {
        switch(this.verb){
            case "t":
                return this.pad(arg.toString());
            case "b":
                return this.fmtNumber(arg, 2);
            case "c":
                return this.fmtNumberCodePoint(arg);
            case "d":
                return this.fmtNumber(arg, 10);
            case "o":
                return this.fmtNumber(arg, 8);
            case "x":
                return this.fmtHex(arg);
            case "X":
                return this.fmtHex(arg, true);
            case "e":
                return this.fmtFloatE(arg);
            case "E":
                return this.fmtFloatE(arg, true);
            case "f":
            case "F":
                return this.fmtFloatF(arg);
            case "g":
                return this.fmtFloatG(arg);
            case "G":
                return this.fmtFloatG(arg, true);
            case "s":
                return this.fmtString(arg);
            case "T":
                return this.fmtString(typeof arg);
            case "v":
                return this.fmtV(arg);
            case "j":
                return this.fmtJ(arg);
            default:
                return `%!(BAD VERB '${this.verb}')`;
        }
    }
    pad(s) {
        const padding = this.flags.zero ? "0" : " ";
        if (this.flags.dash) {
            return s.padEnd(this.flags.width, padding);
        }
        return s.padStart(this.flags.width, padding);
    }
    padNum(nStr, neg) {
        let sign;
        if (neg) {
            sign = "-";
        } else if (this.flags.plus || this.flags.space) {
            sign = this.flags.plus ? "+" : " ";
        } else {
            sign = "";
        }
        const zero = this.flags.zero;
        if (!zero) {
            nStr = sign + nStr;
        }
        const pad = zero ? "0" : " ";
        const len = zero ? this.flags.width - sign.length : this.flags.width;
        if (this.flags.dash) {
            nStr = nStr.padEnd(len, pad);
        } else {
            nStr = nStr.padStart(len, pad);
        }
        if (zero) {
            nStr = sign + nStr;
        }
        return nStr;
    }
    fmtNumber(n, radix, upcase = false) {
        let num = Math.abs(n).toString(radix);
        const prec = this.flags.precision;
        if (prec !== -1) {
            this.flags.zero = false;
            num = n === 0 && prec === 0 ? "" : num;
            while(num.length < prec){
                num = "0" + num;
            }
        }
        let prefix = "";
        if (this.flags.sharp) {
            switch(radix){
                case 2:
                    prefix += "0b";
                    break;
                case 8:
                    prefix += num.startsWith("0") ? "" : "0";
                    break;
                case 16:
                    prefix += "0x";
                    break;
                default:
                    throw new Error("cannot handle base: " + radix);
            }
        }
        num = num.length === 0 ? num : prefix + num;
        if (upcase) {
            num = num.toUpperCase();
        }
        return this.padNum(num, n < 0);
    }
    fmtNumberCodePoint(n) {
        let s = "";
        try {
            s = String.fromCodePoint(n);
        } catch  {
            s = UNICODE_REPLACEMENT_CHARACTER;
        }
        return this.pad(s);
    }
    fmtFloatSpecial(n) {
        if (isNaN(n)) {
            this.flags.zero = false;
            return this.padNum("NaN", false);
        }
        if (n === Number.POSITIVE_INFINITY) {
            this.flags.zero = false;
            this.flags.plus = true;
            return this.padNum("Inf", false);
        }
        if (n === Number.NEGATIVE_INFINITY) {
            this.flags.zero = false;
            return this.padNum("Inf", true);
        }
        return "";
    }
    roundFractionToPrecision(fractional, precision) {
        if (fractional.length > precision) {
            fractional = "1" + fractional;
            let tmp = parseInt(fractional.substr(0, precision + 2)) / 10;
            tmp = Math.round(tmp);
            fractional = Math.floor(tmp).toString();
            fractional = fractional.substr(1);
        } else {
            while(fractional.length < precision){
                fractional += "0";
            }
        }
        return fractional;
    }
    fmtFloatE(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen, bug");
        }
        let fractional = m[F.fractional];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        fractional = this.roundFractionToPrecision(fractional, precision);
        let e = m[F.exponent];
        e = e.length == 1 ? "0" + e : e;
        const val = `${m[F.mantissa]}.${fractional}${upcase ? "E" : "e"}${m[F.esign]}${e}`;
        return this.padNum(val, n < 0);
    }
    fmtFloatF(n) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        function expandNumber(n) {
            if (Number.isSafeInteger(n)) {
                return n.toString() + ".";
            }
            const t = n.toExponential().split("e");
            let m = t[0].replace(".", "");
            const e = parseInt(t[1]);
            if (e < 0) {
                let nStr = "0.";
                for(let i2 = 0; i2 !== Math.abs(e) - 1; ++i2){
                    nStr += "0";
                }
                return nStr += m;
            } else {
                const splIdx = e + 1;
                while(m.length < splIdx){
                    m += "0";
                }
                return m.substr(0, splIdx) + "." + m.substr(splIdx);
            }
        }
        const val = expandNumber(Math.abs(n));
        const arr = val.split(".");
        const dig = arr[0];
        let fractional = arr[1];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        fractional = this.roundFractionToPrecision(fractional, precision);
        return this.padNum(`${dig}.${fractional}`, n < 0);
    }
    fmtFloatG(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        let P = this.flags.precision !== -1 ? this.flags.precision : 6;
        P = P === 0 ? 1 : P;
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen");
        }
        const X = parseInt(m[F.exponent]) * (m[F.esign] === "-" ? -1 : 1);
        let nStr = "";
        if (P > X && X >= -4) {
            this.flags.precision = P - (X + 1);
            nStr = this.fmtFloatF(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*$/, "");
            }
        } else {
            this.flags.precision = P - 1;
            nStr = this.fmtFloatE(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*e/, upcase ? "E" : "e");
            }
        }
        return nStr;
    }
    fmtString(s) {
        if (this.flags.precision !== -1) {
            s = s.substr(0, this.flags.precision);
        }
        return this.pad(s);
    }
    fmtHex(val, upper = false) {
        switch(typeof val){
            case "number":
                return this.fmtNumber(val, 16, upper);
            case "string":
                {
                    const sharp = this.flags.sharp && val.length !== 0;
                    let hex = sharp ? "0x" : "";
                    const prec = this.flags.precision;
                    const end = prec !== -1 ? min(prec, val.length) : val.length;
                    for(let i2 = 0; i2 !== end; ++i2){
                        if (i2 !== 0 && this.flags.space) {
                            hex += sharp ? " 0x" : " ";
                        }
                        const c = (val.charCodeAt(i2) & 255).toString(16);
                        hex += c.length === 1 ? `0${c}` : c;
                    }
                    if (upper) {
                        hex = hex.toUpperCase();
                    }
                    return this.pad(hex);
                }
            default:
                throw new Error("currently only number and string are implemented for hex");
        }
    }
    fmtV(val) {
        if (this.flags.sharp) {
            const options12 = this.flags.precision !== -1 ? {
                depth: this.flags.precision
            } : {
            };
            return this.pad(Deno.inspect(val, options12));
        } else {
            const p = this.flags.precision;
            return p === -1 ? val.toString() : val.toString().substr(0, p);
        }
    }
    fmtJ(val) {
        return JSON.stringify(val);
    }
}
function glb(array, value4, compare) {
    compare = compare || ((a, b)=>a - b
    );
    if (array.length === 0) {
        return -1;
    }
    if (array.length === 1) {
        if (compare(value4, array[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array.length - 1;
    const vLeft = array[left], vRight = array[right];
    if (compare(value4, vRight) >= 0) {
        return right;
    }
    if (compare(value4, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array[center];
        const cmp = compare(value4, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function* matchAll(text, regexp1) {
    let match;
    while((match = regexp1.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function* lineBreakPositions(text) {
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index;
    }
}
function indexToRowCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth1 = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad = " ".repeat(lineWidth1);
    const ls = lines(text);
    const result = [];
    for(let i2 = firstLine; i2 <= lastLine; ++i2){
        const numberStr = `${pad}${i2 + 1}: `.slice(-(lineWidth1 + 2));
        const lineStr = ls[i2];
        result.push({
            lineNumber: i2,
            content: numberStr + lineStr,
            rawLine: ls[i2]
        });
    }
    return {
        prefixWidth: lineWidth1 + 2,
        lines: result
    };
}
function schemaType(schema3) {
    const t = schema3.type;
    if (t) {
        return t;
    }
    if (schema3.anyOf) {
        return "anyOf";
    }
    if (schema3.oneOf) {
        return "oneOf";
    }
    if (schema3.allOf) {
        return "allOf";
    }
    if (schema3.enum) {
        return "enum";
    }
    return "any";
}
function schemaExhaustiveCompletions(schema3) {
    switch(schemaType(schema3)){
        case "anyOf":
            return schema3.anyOf.every(schemaExhaustiveCompletions);
        case "oneOf":
            return schema3.oneOf.every(schemaExhaustiveCompletions);
        case "allOf":
            return schema3.allOf.every(schemaExhaustiveCompletions);
        case "array":
            return true;
        case "object":
            return true;
        default:
            return schema3.exhaustiveCompletions || false;
    }
}
function walkSchema(schema3, f) {
    f(schema3);
    switch(schemaType(schema3)){
        case "array":
            if (schema3.items) {
                walkSchema(schema3.items, f);
            }
            break;
        case "anyOf":
            for (const s of schema3.anyOf){
                walkSchema(s, f);
            }
            break;
        case "oneOf":
            for (const s1 of schema3.oneOf){
                walkSchema(s1, f);
            }
            break;
        case "allOf":
            for (const s2 of schema3.allOf){
                walkSchema(s2, f);
            }
            break;
        case "object":
            if (schema3.properties) {
                for (const key of Object.getOwnPropertyNames(schema3.properties)){
                    const s3 = schema3.properties[key];
                    walkSchema(s3, f);
                }
            }
            if (schema3.additionalProperties) {
                walkSchema(schema3.additionalProperties, f);
            }
            break;
    }
}
function normalizeSchema(schema3) {
    const result = JSON.parse(JSON.stringify(schema3));
    walkSchema(result, (schema4)=>{
        if (schema4.completions) {
            delete schema4.completions;
        }
        if (schema4.exhaustiveCompletions) {
            delete schema4.exhaustiveCompletions;
        }
        if (schema4.documentation) {
            delete schema4.documentation;
        }
    });
    return result;
}
const BooleanSchema = {
    "type": "boolean",
    "description": "be a boolean value",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const NumberSchema = {
    "type": "number",
    "description": "be a number"
};
const IntegerSchema = {
    "type": "integer",
    "description": "be an integral number"
};
function numericSchema(obj) {
    return Object.assign({
        description: "be a number"
    }, obj);
}
const StringSchema = {
    "type": "string",
    "description": "be a string"
};
const NullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function enumSchema(...args4) {
    if (args4.length === 0) {
        throw new Error("Internal Error: Empty enum schema not supported.");
    }
    return {
        "enum": args4,
        "description": args4.length > 1 ? `be one of: ${args4.map((x)=>"'" + x + "'"
        ).join(", ")}` : `be '${args4[0]}'`,
        "completions": args4,
        "exhaustiveCompletions": true
    };
}
function oneOfSchema(...args4) {
    return {
        "oneOf": args4,
        "description": `be exactly one of: ${args4.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function anyOfSchema(...args4) {
    return {
        "anyOf": args4,
        "description": `be at least one of: ${args4.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function allOfSchema(...args4) {
    return {
        "allOf": args4,
        "description": `be all of: ${args4.map((x)=>x.description.slice(3)
        ).join(", ")}`
    };
}
function objectSchema(params = {
}) {
    let { properties , patternProperties , required , additionalProperties , description , baseSchema , exhaustive , completions: completionsParam ,  } = params;
    required = required || [];
    properties = properties || {
    };
    patternProperties = patternProperties || {
    };
    const hasDescription = description !== undefined;
    description = description || "be an object";
    let result = undefined;
    const completions = [];
    const uniqueValues = (lst)=>{
        const obj = {
        };
        for (const c of lst){
            obj[c.value] = c;
        }
        return Object.getOwnPropertyNames(obj).map((k)=>obj[k]
        );
    };
    for (const k of Object.getOwnPropertyNames(completionsParam || properties)){
        const valueS = properties[k];
        completions.push({
            type: "key",
            display: k,
            value: `${k}: `,
            description: completionsParam?.[k] || "",
            suggest_on_accept: valueS && schemaExhaustiveCompletions(valueS)
        });
    }
    if (baseSchema) {
        if (baseSchema.type !== "object") {
            throw new Error("Internal Error: can only extend other object Schema");
        }
        result = Object.assign({
        }, baseSchema);
        if (exhaustive && baseSchema.exhaustiveCompletions) {
            result.exhaustiveCompletions = true;
        }
        if (hasDescription) {
            result.description = description;
        }
        result.properties = Object.assign({
        }, result.properties, properties);
        result.patternProperties = Object.assign({
        }, result.patternProperties, patternProperties);
        if (required) {
            result.required = (result.required ?? []).slice();
            result.required.push(...required);
        }
        if (result.completions && result.completions.length || completions.length) {
            result.completions = (result.completions || []).slice();
            result.completions.push(...completions);
            result.completions = uniqueValues(result.completions);
        }
        if (additionalProperties !== undefined) {
            if (result.additionalProperties === false) {
                throw new Error("Internal Error: don't know how to subclass object schema with additionalProperties === false");
            }
            if (result.additionalProperties) {
                result.additionalProperties = allOfSchema(result.additionalProperties, additionalProperties);
            } else {
                result.additionalProperties = additionalProperties;
            }
        }
    } else {
        result = {
            "type": "object",
            description
        };
        if (exhaustive) {
            result.exhaustiveCompletions = true;
        }
        if (properties) {
            result.properties = properties;
        }
        if (patternProperties) {
            result.patternProperties = patternProperties;
        }
        if (required && required.length > 0) {
            result.required = required;
        }
        if (completions.length) {
            result.completions = completions;
        }
        if (additionalProperties !== undefined) {
            result.additionalProperties = additionalProperties;
        }
    }
    return result;
}
function arraySchema(items) {
    if (items) {
        return {
            "type": "array",
            "description": `be an array of values, where each element should ` + items.description,
            items
        };
    } else {
        return {
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema3, doc) {
    const result = Object.assign({
    }, schema3);
    result.documentation = doc;
    return result;
}
function idSchema(schema3, id) {
    const result = Object.assign({
    }, schema3);
    result["$id"] = id;
    return result;
}
function refSchema($ref, description) {
    return {
        $ref,
        description
    };
}
const kMetadataFormat = "format";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig.width";
const kFigHeight = "fig.height";
const kFigFormat = "fig.format";
const kFigDpi = "fig.dpi";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kKeepHidden = "keep-hidden";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepYaml = "keep-yaml";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kMergeIncludes = "merge-includes";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug, 
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepYaml,
    kKeepSource,
    kKeepHidden,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter, 
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kCalloutTipCaption = "callout-tip-caption";
const kCalloutNoteCaption = "callout-note-caption";
const kCalloutWarningCaption = "callout-warning-caption";
const kCalloutImportantCaption = "callout-important-caption";
const kCalloutDangerCaption = "callout-danger-caption";
const kSectionTitleFootnotes = "section-title-footnotes";
const kCodeSummary = "code-summary";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kCopyButtonTooltip = "copy-button-tooltip";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutDangerCaption,
    kSectionTitleFootnotes,
    kCodeSummary,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kSearchNoResultsText,
    kCopyButtonTooltip,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle, 
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kAtxHeaders = "atx-headers";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kStandalone = "standalone";
const kSelfContained = "self-contained";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDoi = "doi";
const kAbstract = "abstract";
const kDescription = "description";
const kTocTitle = "toc-title";
const kLang = "lang";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kCss = "css";
const kBibliography = "bibliography";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kFigResponsive = "fig-responsive";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    "resource-path",
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    "wrap",
    "columns",
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    "toc-depth",
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "strip-empty-paragraphs",
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    "default-image-extension",
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kAtxHeaders,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace", 
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellCapLoc = "caption-location";
const kCellFigCapLoc = "fig-cap-location";
const kCellTblCapLoc = "tbl-cap-location";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
const formatExecuteOptionsSchema = objectSchema({
    properties: {
        "fig-width": NumberSchema,
        "fig-height": NumberSchema,
        "fig-format": enumSchema("retina", "png", "jpeg", "svg", "pdf"),
        "fig-dpi": NumberSchema,
        [kCache]: oneOfSchema(BooleanSchema, NullSchema, enumSchema("refresh")),
        [kFreeze]: oneOfSchema(BooleanSchema, enumSchema("auto")),
        [kExecuteEnabled]: oneOfSchema(BooleanSchema, NullSchema),
        [kExecuteIpynb]: oneOfSchema(BooleanSchema, NullSchema),
        [kExecuteDaemon]: oneOfSchema(NumberSchema, BooleanSchema, NullSchema),
        [kExecuteDaemonRestart]: BooleanSchema,
        [kExecuteDebug]: BooleanSchema,
        [kEngine]: StringSchema,
        [kEval]: oneOfSchema(BooleanSchema, NullSchema),
        [kError]: BooleanSchema,
        [kEcho]: oneOfSchema(BooleanSchema, enumSchema("fenced")),
        [kOutput]: oneOfSchema(BooleanSchema, enumSchema("all", "asis")),
        [kWarning]: BooleanSchema,
        [kInclude]: BooleanSchema,
        [kKeepMd]: BooleanSchema,
        [kKeepIpynb]: BooleanSchema
    }
});
const htmlOptionsSchema = oneOfSchema(formatExecuteOptionsSchema, enumSchema("default"));
const htmlFormatSchema = objectSchema({
    properties: {
        "html": htmlOptionsSchema
    },
    description: "be an HTML format object"
});
const frontMatterFormatSchema = oneOfSchema(enumSchema("html", "revealjs", "pdf"), htmlFormatSchema);
const frontMatterSchema = idSchema(oneOfSchema(NullSchema, objectSchema({
    properties: {
        title: StringSchema
    },
    description: "be a Quarto YAML front matter object"
})), "front-matter");
function Diff() {
}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options12 = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {
        };
        var callback = options12.callback;
        if (typeof options12 === "function") {
            callback = options12;
            options12 = {
            };
        }
        this.options = options12;
        var self = this;
        function done(value4) {
            if (callback) {
                setTimeout(function() {
                    callback(void 0, value4);
                }, 0);
                return true;
            } else {
                return value4;
            }
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath = bestPath[diagonalPath - 1], removePath = bestPath[diagonalPath + 1], _oldPos = (removePath ? removePath.newPos : 0) - diagonalPath;
                if (addPath) {
                    bestPath[diagonalPath - 1] = void 0;
                }
                var canAdd = addPath && addPath.newPos + 1 < newLen, canRemove = removePath && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = void 0;
                    continue;
                }
                if (!canAdd || canRemove && addPath.newPos < removePath.newPos) {
                    basePath = clonePath(removePath);
                    self.pushComponent(basePath.components, void 0, true);
                } else {
                    basePath = addPath;
                    basePath.newPos++;
                    self.pushComponent(basePath.components, true, void 0);
                }
                _oldPos = self.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self, basePath.components, newString, oldString, self.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        if (callback) {
            (function exec() {
                setTimeout(function() {
                    if (editLength > maxEditLength) {
                        return callback();
                    }
                    if (!execEditLength()) {
                        exec();
                    }
                }, 0);
            })();
        } else {
            while(editLength <= maxEditLength){
                var ret = execEditLength();
                if (ret) {
                    return ret;
                }
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last = components[components.length - 1];
        if (last && last.added === added && last.removed === removed) {
            components[components.length - 1] = {
                count: last.count + 1,
                added,
                removed
            };
        } else {
            components.push({
                count: 1,
                added,
                removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array) {
        var ret = [];
        for(var i2 = 0; i2 < array.length; i2++){
            if (array[i2]) {
                ret.push(array[i2]);
            }
        }
        return ret;
    },
    castInput: function castInput(value4) {
        return value4;
    },
    tokenize: function tokenize(value4) {
        return value4.split("");
    },
    join: function join4(chars) {
        return chars.join("");
    }
};
function buildValues(diff2, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value4 = newString.slice(newPos, newPos + component.count);
                value4 = value4.map(function(value21, i2) {
                    var oldValue = oldString[oldPos + i2];
                    return oldValue.length > value21.length ? oldValue : value21;
                });
                component.value = diff2.join(value4);
            } else {
                component.value = diff2.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff2.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === "string" && (lastComponent.added || lastComponent.removed) && diff2.equals("", lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path2) {
    return {
        newPos: path2.newPos,
        components: path2.components.slice(0)
    };
}
var characterDiff = new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace.test(left) && !reWhitespace.test(right);
};
wordDiff.tokenize = function(value5) {
    var tokens = value5.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i2 = 0; i2 < tokens.length - 1; i2++){
        if (!tokens[i2 + 1] && tokens[i2 + 2] && extendedWordChars.test(tokens[i2]) && extendedWordChars.test(tokens[i2 + 2])) {
            tokens[i2] += tokens[i2 + 2];
            tokens.splice(i2 + 1, 2);
            i2--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value5) {
    var retLines = [], linesAndNewlines = value5.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i2 = 0; i2 < linesAndNewlines.length; i2++){
        var line2 = linesAndNewlines[i2];
        if (i2 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line2;
        } else {
            if (this.options.ignoreWhitespace) {
                line2 = line2.trim();
            }
            retLines.push(line2);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value5) {
    return value5.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value5) {
    return value5.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj2) {
            return typeof obj2;
        };
    } else {
        _typeof = function(obj2) {
            return obj2 && typeof Symbol === "function" && obj2.constructor === Symbol && obj2 !== Symbol.prototype ? "symbol" : typeof obj2;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value5) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === "undefined" ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value5 === "string" ? value5 : JSON.stringify(canonicalize(value5, null, null, stringifyReplacer), stringifyReplacer, "  ");
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, "$1"), right.replace(/,([\r\n])/g, "$1"));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i2;
    for(i2 = 0; i2 < stack.length; i2 += 1){
        if (stack[i2] === obj) {
            return replacementStack[i2];
        }
    }
    var canonicalizedObj;
    if (objectPrototypeToString.call(obj) === "[object Array]") {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i2 = 0; i2 < obj.length; i2 += 1){
            canonicalizedObj[i2] = canonicalize(obj[i2], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === "object" && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {
        };
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i2 = 0; i2 < sortedKeys.length; i2 += 1){
            _key = sortedKeys[i2];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value5) {
    return value5.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value5) {
    return value5;
};
function matchAll1(str1, regex) {
    let match;
    regex = new RegExp(regex);
    const result = [];
    while((match = regex.exec(str1)) != null){
        result.push(match);
    }
    return result;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result;
    } else {
        const matches = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r of matches){
            const stringEnd = r.index + 1;
            result.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result;
    }
}
function mappedString(source, pieces) {
    ;
    if (typeof source === "string") {
        const offsetInfo = [];
        let offset = 0;
        const resultList = pieces.filter((piece)=>typeof piece === "string" || piece.start !== piece.end
        ).map((piece)=>{
            if (typeof piece === "string") {
                offsetInfo.push({
                    fromSource: false,
                    length: piece.length,
                    offset
                });
                offset += piece.length;
                return piece;
            } else {
                const resultPiece = source.substring(piece.start, piece.end);
                offsetInfo.push({
                    fromSource: true,
                    length: resultPiece.length,
                    offset,
                    range: {
                        start: piece.start,
                        end: piece.end
                    }
                });
                offset += resultPiece.length;
                return resultPiece;
            }
        });
        const value5 = resultList.join("");
        const map1 = (targetOffset)=>{
            const ix = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            if (ix < 0) {
                return undefined;
            }
            const info1 = offsetInfo[ix];
            if (!info1.fromSource) {
                return undefined;
            }
            const localOffset = targetOffset - info1.offset;
            if (localOffset >= info1.length) {
                return undefined;
            }
            return info1.range.start + localOffset;
        };
        const mapClosest = (targetOffset)=>{
            if (offsetInfo.length === 0 || targetOffset < 0) {
                return undefined;
            }
            const firstIx = glb(offsetInfo, {
                offset: targetOffset
            }, (a, b)=>a.offset - b.offset
            );
            let ix = firstIx;
            let smallestSourceInfo = undefined;
            while(ix >= 0){
                const info1 = offsetInfo[ix];
                if (!info1.fromSource) {
                    ix--;
                    continue;
                }
                smallestSourceInfo = info1;
                if (ix === firstIx) {
                    const localOffset = targetOffset - info1.offset;
                    if (localOffset < info1.length) {
                        return info1.range.start + localOffset;
                    }
                }
                return info1.range.end - 1;
            }
            if (smallestSourceInfo === undefined) {
                return undefined;
            } else {
                return smallestSourceInfo.range.start;
            }
        };
        return {
            value: value5,
            originalString: source,
            map: map1,
            mapClosest
        };
    } else {
        const { value: value5 , originalString , map: previousMap , mapClosest: previousMapClosest ,  } = source;
        const { value: resultValue , map: nextMap , mapClosest: nextMapClosest ,  } = mappedString(value5, pieces);
        const composeMap = (offset)=>{
            const v = nextMap(offset);
            if (v === undefined) {
                return v;
            }
            return previousMap(v);
        };
        const composeMapClosest = (offset)=>{
            const v = nextMapClosest(offset);
            if (v === undefined) {
                return v;
            }
            return previousMapClosest(v);
        };
        return {
            value: resultValue,
            originalString,
            map: composeMap,
            mapClosest: composeMapClosest
        };
    }
}
function asMappedString(str1) {
    if (typeof str1 === "string") {
        return {
            value: str1,
            originalString: str1,
            map: (x)=>x
            ,
            mapClosest: (x)=>x
        };
    } else {
        return str1;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        throw new Error("strings must be non-empty");
    }
    let currentOffset = 0;
    const offsets = [];
    for (const s of strings){
        currentOffset += s.value.length;
        offsets.push(currentOffset);
    }
    const value5 = "".concat(...strings.map((s1)=>s1.value
    ));
    return {
        value: value5,
        originalString: strings[0].originalString,
        map (offset) {
            if (offset < 0 || offset >= value5.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].map(offset - offsets[ix]);
        },
        mapClosest (offset) {
            if (offset < 0 || offset >= value5.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            return strings[ix].mapClosest(offset - offsets[ix]);
        }
    };
}
function mappedIndexToRowCol(eitherText) {
    const text = asMappedString(eitherText);
    const f = indexToRowCol(text.originalString);
    return function(offset) {
        const n = text.mapClosest(offset);
        if (n === undefined) {
            throw new Error("Internal Error: bad offset in mappedIndexRowCol");
        }
        return f(n);
    };
}
function mappedNormalizeNewlines(eitherText) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const offset of lineBreakPositions(text.value)){
        if (text.value[offset] !== "\r") {
            continue;
        }
        chunks.push({
            start,
            end: offset
        });
        chunks.push({
            start: offset + 1,
            end: offset + 2
        });
        start = offset + 2;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexpAll(eitherText, re) {
    const text = asMappedString(eitherText);
    let start = 0;
    const chunks = [];
    for (const match of matchAll(text.value, re)){
        chunks.push({
            start,
            end: match.index
        });
        start = match[0].length + match.index;
    }
    if (start !== text.value.length) {
        chunks.push({
            start,
            end: text.value.length
        });
    }
    return mappedString(text, chunks);
}
function skipRegexp(eitherText, re) {
    const text = asMappedString(eitherText);
    const m = text.value.match(re);
    if (m) {
        return mappedString(text, [
            {
                start: 0,
                end: m.index
            },
            {
                start: m.index + m[0].length,
                end: text.value.length
            }, 
        ]);
    } else {
        return text;
    }
}
function mappedDiff(source, target) {
    const sourceLineRanges = rangedLines(source.value).map((x)=>x.range
    );
    let sourceCursor = 0;
    const resultChunks = [];
    for (const action of diffLines(source.value, target)){
        if (action.removed) {
            sourceCursor += action.count;
        } else if (action.added) {
            resultChunks.push(action.value);
        } else {
            const start = sourceLineRanges[sourceCursor].start;
            const nextCursor = sourceCursor + action.count;
            const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
            sourceCursor = nextCursor;
            resultChunks.push({
                start,
                end
            });
        }
    }
    return mappedString(source, resultChunks);
}
function readAnnotatedYamlFromMappedString(yml) {
    return readAnnotatedYamlFromString(yml.value);
}
function readAnnotatedYamlFromString(yml) {
    const stack = [];
    const results = [];
    function listener1(what, state2) {
        const { result , position: position2 , kind  } = state2;
        if (what === "close") {
            const { position: openPosition  } = stack.pop();
            if (results.length > 0) {
                const last = results[results.length - 1];
                if (last.start === openPosition && last.end === position2) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last = results[results.length - 1];
                if (last.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position2);
            const leftTrim = rawRange.length - rawRange.trimLeft().length;
            const rightTrim = rawRange.length - rawRange.trimRight().length;
            results.push({
                start: openPosition + leftTrim,
                end: position2 - rightTrim,
                result: result,
                components,
                kind
            });
        } else {
            stack.push({
                position: position2
            });
        }
    }
    parse3(yml, {
        listener: listener1
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: []
        };
    }
    if (results.length !== 1) {
        throw new Error(`Internal Error - expected a single result, got ${results.length} instead`);
    }
    return results[0];
}
var fastDeepEqual = function equal(a, b) {
    if (a === b) return true;
    if (a && b && typeof a == "object" && typeof b == "object") {
        if (a.constructor !== b.constructor) return false;
        var length, i2, keys;
        if (Array.isArray(a)) {
            length = a.length;
            if (length != b.length) return false;
            for(i2 = length; (i2--) !== 0;)if (!equal(a[i2], b[i2])) return false;
            return true;
        }
        if (a.constructor === RegExp) return a.source === b.source && a.flags === b.flags;
        if (a.valueOf !== Object.prototype.valueOf) return a.valueOf() === b.valueOf();
        if (a.toString !== Object.prototype.toString) return a.toString() === b.toString();
        keys = Object.keys(a);
        length = keys.length;
        if (length !== Object.keys(b).length) return false;
        for(i2 = length; (i2--) !== 0;)if (!Object.prototype.hasOwnProperty.call(b, keys[i2])) return false;
        for(i2 = length; (i2--) !== 0;){
            var key = keys[i2];
            if (!equal(a[key], b[key])) return false;
        }
        return true;
    }
    return a !== a && b !== b;
};
function createCommonjsModule(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path2, base) {
            return commonjsRequire1(path2, base === void 0 || base === null ? module.path : base);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire1() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var jsonSchemaTraverse = createCommonjsModule(function(module) {
    var traverse = module.exports = function(schema3, opts, cb) {
        if (typeof opts == "function") {
            cb = opts;
            opts = {
            };
        }
        cb = opts.cb || cb;
        var pre = typeof cb == "function" ? cb : cb.pre || function() {
        };
        var post = cb.post || function() {
        };
        _traverse(opts, pre, post, schema3, "", schema3);
    };
    traverse.keywords = {
        additionalItems: true,
        items: true,
        contains: true,
        additionalProperties: true,
        propertyNames: true,
        not: true,
        if: true,
        then: true,
        else: true
    };
    traverse.arrayKeywords = {
        items: true,
        allOf: true,
        anyOf: true,
        oneOf: true
    };
    traverse.propsKeywords = {
        $defs: true,
        definitions: true,
        properties: true,
        patternProperties: true,
        dependencies: true
    };
    traverse.skipKeywords = {
        default: true,
        enum: true,
        const: true,
        required: true,
        maximum: true,
        minimum: true,
        exclusiveMaximum: true,
        exclusiveMinimum: true,
        multipleOf: true,
        maxLength: true,
        minLength: true,
        pattern: true,
        format: true,
        maxItems: true,
        minItems: true,
        uniqueItems: true,
        maxProperties: true,
        minProperties: true
    };
    function _traverse(opts, pre, post, schema3, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex) {
        if (schema3 && typeof schema3 == "object" && !Array.isArray(schema3)) {
            pre(schema3, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex);
            for(var key in schema3){
                var sch = schema3[key];
                if (Array.isArray(sch)) {
                    if (key in traverse.arrayKeywords) {
                        for(var i3 = 0; i3 < sch.length; i3++)_traverse(opts, pre, post, sch[i3], jsonPtr + "/" + key + "/" + i3, rootSchema, jsonPtr, key, schema3, i3);
                    }
                } else if (key in traverse.propsKeywords) {
                    if (sch && typeof sch == "object") {
                        for(var prop in sch)_traverse(opts, pre, post, sch[prop], jsonPtr + "/" + key + "/" + escapeJsonPtr(prop), rootSchema, jsonPtr, key, schema3, prop);
                    }
                } else if (key in traverse.keywords || opts.allKeys && !(key in traverse.skipKeywords)) {
                    _traverse(opts, pre, post, sch, jsonPtr + "/" + key, rootSchema, jsonPtr, key, schema3);
                }
            }
            post(schema3, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex);
        }
    }
    function escapeJsonPtr(str1) {
        return str1.replace(/~/g, "~0").replace(/\//g, "~1");
    }
});
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {
};
function getDefaultExportFromCjs(x) {
    return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, "default") ? x["default"] : x;
}
function createCommonjsModule1(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path2, base) {
            return commonjsRequire2(path2, base === void 0 || base === null ? module.path : base);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire2() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var uri_all = createCommonjsModule1(function(module, exports) {
    (function(global2, factory) {
        factory(exports);
    })(commonjsGlobal, function(exports2) {
        function merge1() {
            for(var _len = arguments.length, sets = Array(_len), _key = 0; _key < _len; _key++){
                sets[_key] = arguments[_key];
            }
            if (sets.length > 1) {
                sets[0] = sets[0].slice(0, -1);
                var xl = sets.length - 1;
                for(var x = 1; x < xl; ++x){
                    sets[x] = sets[x].slice(1, -1);
                }
                sets[xl] = sets[xl].slice(1);
                return sets.join("");
            } else {
                return sets[0];
            }
        }
        function subexp(str1) {
            return "(?:" + str1 + ")";
        }
        function typeOf(o) {
            return o === void 0 ? "undefined" : o === null ? "null" : Object.prototype.toString.call(o).split(" ").pop().split("]").shift().toLowerCase();
        }
        function toUpperCase(str1) {
            return str1.toUpperCase();
        }
        function toArray(obj) {
            return obj !== void 0 && obj !== null ? obj instanceof Array ? obj : typeof obj.length !== "number" || obj.split || obj.setInterval || obj.call ? [
                obj
            ] : Array.prototype.slice.call(obj) : [];
        }
        function assign(target, source) {
            var obj = target;
            if (source) {
                for(var key in source){
                    obj[key] = source[key];
                }
            }
            return obj;
        }
        function buildExps(isIRI) {
            var ALPHA$$ = "[A-Za-z]", DIGIT$$ = "[0-9]", HEXDIG$$2 = merge1(DIGIT$$, "[A-Fa-f]"), PCT_ENCODED$2 = subexp(subexp("%[EFef]" + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2) + "|" + subexp("%[89A-Fa-f]" + HEXDIG$$2 + "%" + HEXDIG$$2 + HEXDIG$$2) + "|" + subexp("%" + HEXDIG$$2 + HEXDIG$$2)), GEN_DELIMS$$ = "[\\:\\/\\?\\#\\[\\]\\@]", SUB_DELIMS$$ = "[\\!\\$\\&\\'\\(\\)\\*\\+\\,\\;\\=]", RESERVED$$ = merge1(GEN_DELIMS$$, SUB_DELIMS$$), UCSCHAR$$ = isIRI ? "[\\xA0-\\u200D\\u2010-\\u2029\\u202F-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]" : "[]", IPRIVATE$$ = isIRI ? "[\\uE000-\\uF8FF]" : "[]", UNRESERVED$$2 = merge1(ALPHA$$, DIGIT$$, "[\\-\\.\\_\\~]", UCSCHAR$$), SCHEME$ = subexp(ALPHA$$ + merge1(ALPHA$$, DIGIT$$, "[\\+\\-\\.]") + "*"), USERINFO$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge1(UNRESERVED$$2, SUB_DELIMS$$, "[\\:]")) + "*"), DEC_OCTET_RELAXED$ = subexp(subexp("25[0-5]") + "|" + subexp("2[0-4]" + DIGIT$$) + "|" + subexp("1" + DIGIT$$ + DIGIT$$) + "|" + subexp("0?[1-9]" + DIGIT$$) + "|0?0?" + DIGIT$$), IPV4ADDRESS$ = subexp(DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$ + "\\." + DEC_OCTET_RELAXED$), H16$ = subexp(HEXDIG$$2 + "{1,4}"), LS32$ = subexp(subexp(H16$ + "\\:" + H16$) + "|" + IPV4ADDRESS$), IPV6ADDRESS1$ = subexp(subexp(H16$ + "\\:") + "{6}" + LS32$), IPV6ADDRESS2$ = subexp("\\:\\:" + subexp(H16$ + "\\:") + "{5}" + LS32$), IPV6ADDRESS3$ = subexp(subexp(H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{4}" + LS32$), IPV6ADDRESS4$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,1}" + H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{3}" + LS32$), IPV6ADDRESS5$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,2}" + H16$) + "?\\:\\:" + subexp(H16$ + "\\:") + "{2}" + LS32$), IPV6ADDRESS6$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,3}" + H16$) + "?\\:\\:" + H16$ + "\\:" + LS32$), IPV6ADDRESS7$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,4}" + H16$) + "?\\:\\:" + LS32$), IPV6ADDRESS8$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,5}" + H16$) + "?\\:\\:" + H16$), IPV6ADDRESS9$ = subexp(subexp(subexp(H16$ + "\\:") + "{0,6}" + H16$) + "?\\:\\:"), IPV6ADDRESS$ = subexp([
                IPV6ADDRESS1$,
                IPV6ADDRESS2$,
                IPV6ADDRESS3$,
                IPV6ADDRESS4$,
                IPV6ADDRESS5$,
                IPV6ADDRESS6$,
                IPV6ADDRESS7$,
                IPV6ADDRESS8$,
                IPV6ADDRESS9$
            ].join("|")), ZONEID$ = subexp(subexp(UNRESERVED$$2 + "|" + PCT_ENCODED$2) + "+"), IPVFUTURE$ = subexp("[vV]" + HEXDIG$$2 + "+\\." + merge1(UNRESERVED$$2, SUB_DELIMS$$, "[\\:]") + "+"), REG_NAME$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge1(UNRESERVED$$2, SUB_DELIMS$$)) + "*"), PCHAR$ = subexp(PCT_ENCODED$2 + "|" + merge1(UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@]")), SEGMENT_NZ_NC$ = subexp(subexp(PCT_ENCODED$2 + "|" + merge1(UNRESERVED$$2, SUB_DELIMS$$, "[\\@]")) + "+"), QUERY$ = subexp(subexp(PCHAR$ + "|" + merge1("[\\/\\?]", IPRIVATE$$)) + "*");
            return {
                NOT_SCHEME: new RegExp(merge1("[^]", ALPHA$$, DIGIT$$, "[\\+\\-\\.]"), "g"),
                NOT_USERINFO: new RegExp(merge1("[^\\%\\:]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_HOST: new RegExp(merge1("[^\\%\\[\\]\\:]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_PATH: new RegExp(merge1("[^\\%\\/\\:\\@]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_PATH_NOSCHEME: new RegExp(merge1("[^\\%\\/\\@]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                NOT_QUERY: new RegExp(merge1("[^\\%]", UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@\\/\\?]", IPRIVATE$$), "g"),
                NOT_FRAGMENT: new RegExp(merge1("[^\\%]", UNRESERVED$$2, SUB_DELIMS$$, "[\\:\\@\\/\\?]"), "g"),
                ESCAPE: new RegExp(merge1("[^]", UNRESERVED$$2, SUB_DELIMS$$), "g"),
                UNRESERVED: new RegExp(UNRESERVED$$2, "g"),
                OTHER_CHARS: new RegExp(merge1("[^\\%]", UNRESERVED$$2, RESERVED$$), "g"),
                PCT_ENCODED: new RegExp(PCT_ENCODED$2, "g"),
                IPV4ADDRESS: new RegExp("^(" + IPV4ADDRESS$ + ")$"),
                IPV6ADDRESS: new RegExp("^\\[?(" + IPV6ADDRESS$ + ")" + subexp(subexp("\\%25|\\%(?!" + HEXDIG$$2 + "{2})") + "(" + ZONEID$ + ")") + "?\\]?$")
            };
        }
        var URI_PROTOCOL = buildExps(false);
        var IRI_PROTOCOL = buildExps(true);
        var slicedToArray = function() {
            function sliceIterator(arr, i4) {
                var _arr = [];
                var _n = true;
                var _d = false;
                var _e = void 0;
                try {
                    for(var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true){
                        _arr.push(_s.value);
                        if (i4 && _arr.length === i4) break;
                    }
                } catch (err) {
                    _d = true;
                    _e = err;
                } finally{
                    try {
                        if (!_n && _i["return"]) _i["return"]();
                    } finally{
                        if (_d) throw _e;
                    }
                }
                return _arr;
            }
            return function(arr, i4) {
                if (Array.isArray(arr)) {
                    return arr;
                } else if (Symbol.iterator in Object(arr)) {
                    return sliceIterator(arr, i4);
                } else {
                    throw new TypeError("Invalid attempt to destructure non-iterable instance");
                }
            };
        }();
        var toConsumableArray = function(arr) {
            if (Array.isArray(arr)) {
                for(var i4 = 0, arr2 = Array(arr.length); i4 < arr.length; i4++)arr2[i4] = arr[i4];
                return arr2;
            } else {
                return Array.from(arr);
            }
        };
        var maxInt = 2147483647;
        var base = 36;
        var tMin = 1;
        var tMax = 26;
        var skew = 38;
        var damp = 700;
        var initialBias = 72;
        var initialN = 128;
        var delimiter3 = "-";
        var regexPunycode = /^xn--/;
        var regexNonASCII = /[^\0-\x7E]/;
        var regexSeparators = /[\x2E\u3002\uFF0E\uFF61]/g;
        var errors = {
            overflow: "Overflow: input needs wider integers to process",
            "not-basic": "Illegal input >= 0x80 (not a basic code point)",
            "invalid-input": "Invalid input"
        };
        var baseMinusTMin = base - tMin;
        var floor = Math.floor;
        var stringFromCharCode = String.fromCharCode;
        function error$1(type4) {
            throw new RangeError(errors[type4]);
        }
        function map1(array, fn) {
            var result = [];
            var length = array.length;
            while(length--){
                result[length] = fn(array[length]);
            }
            return result;
        }
        function mapDomain(string1, fn) {
            var parts = string1.split("@");
            var result = "";
            if (parts.length > 1) {
                result = parts[0] + "@";
                string1 = parts[1];
            }
            string1 = string1.replace(regexSeparators, ".");
            var labels = string1.split(".");
            var encoded = map1(labels, fn).join(".");
            return result + encoded;
        }
        function ucs2decode(string1) {
            var output = [];
            var counter = 0;
            var length = string1.length;
            while(counter < length){
                var value5 = string1.charCodeAt(counter++);
                if (value5 >= 55296 && value5 <= 56319 && counter < length) {
                    var extra = string1.charCodeAt(counter++);
                    if ((extra & 64512) == 56320) {
                        output.push(((value5 & 1023) << 10) + (extra & 1023) + 65536);
                    } else {
                        output.push(value5);
                        counter--;
                    }
                } else {
                    output.push(value5);
                }
            }
            return output;
        }
        var ucs2encode = function ucs2encode2(array) {
            return String.fromCodePoint.apply(String, toConsumableArray(array));
        };
        var basicToDigit = function basicToDigit2(codePoint) {
            if (codePoint - 48 < 10) {
                return codePoint - 22;
            }
            if (codePoint - 65 < 26) {
                return codePoint - 65;
            }
            if (codePoint - 97 < 26) {
                return codePoint - 97;
            }
            return base;
        };
        var digitToBasic = function digitToBasic2(digit, flag) {
            return digit + 22 + 75 * (digit < 26) - ((flag != 0) << 5);
        };
        var adapt = function adapt2(delta, numPoints, firstTime) {
            var k = 0;
            delta = firstTime ? floor(delta / damp) : delta >> 1;
            delta += floor(delta / numPoints);
            for(; delta > baseMinusTMin * tMax >> 1; k += base){
                delta = floor(delta / baseMinusTMin);
            }
            return floor(k + (baseMinusTMin + 1) * delta / (delta + skew));
        };
        var decode = function decode2(input1) {
            var output = [];
            var inputLength = input1.length;
            var i5 = 0;
            var n = initialN;
            var bias = initialBias;
            var basic = input1.lastIndexOf(delimiter3);
            if (basic < 0) {
                basic = 0;
            }
            for(var j = 0; j < basic; ++j){
                if (input1.charCodeAt(j) >= 128) {
                    error$1("not-basic");
                }
                output.push(input1.charCodeAt(j));
            }
            for(var index = basic > 0 ? basic + 1 : 0; index < inputLength;){
                var oldi = i5;
                for(var w = 1, k = base;; k += base){
                    if (index >= inputLength) {
                        error$1("invalid-input");
                    }
                    var digit = basicToDigit(input1.charCodeAt(index++));
                    if (digit >= base || digit > floor((maxInt - i5) / w)) {
                        error$1("overflow");
                    }
                    i5 += digit * w;
                    var t = k <= bias ? tMin : k >= bias + tMax ? tMax : k - bias;
                    if (digit < t) {
                        break;
                    }
                    var baseMinusT = base - t;
                    if (w > floor(maxInt / baseMinusT)) {
                        error$1("overflow");
                    }
                    w *= baseMinusT;
                }
                var out = output.length + 1;
                bias = adapt(i5 - oldi, out, oldi == 0);
                if (floor(i5 / out) > maxInt - n) {
                    error$1("overflow");
                }
                n += floor(i5 / out);
                i5 %= out;
                output.splice(i5++, 0, n);
            }
            return String.fromCodePoint.apply(String, output);
        };
        var encode = function encode2(input1) {
            var output = [];
            input1 = ucs2decode(input1);
            var inputLength = input1.length;
            var n = initialN;
            var delta = 0;
            var bias = initialBias;
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = void 0;
            try {
                for(var _iterator = input1[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true){
                    var _currentValue2 = _step.value;
                    if (_currentValue2 < 128) {
                        output.push(stringFromCharCode(_currentValue2));
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally{
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally{
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }
            var basicLength = output.length;
            var handledCPCount = basicLength;
            if (basicLength) {
                output.push(delimiter3);
            }
            while(handledCPCount < inputLength){
                var m = maxInt;
                var _iteratorNormalCompletion2 = true;
                var _didIteratorError2 = false;
                var _iteratorError2 = void 0;
                try {
                    for(var _iterator2 = input1[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true){
                        var currentValue = _step2.value;
                        if (currentValue >= n && currentValue < m) {
                            m = currentValue;
                        }
                    }
                } catch (err) {
                    _didIteratorError2 = true;
                    _iteratorError2 = err;
                } finally{
                    try {
                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                            _iterator2.return();
                        }
                    } finally{
                        if (_didIteratorError2) {
                            throw _iteratorError2;
                        }
                    }
                }
                var handledCPCountPlusOne = handledCPCount + 1;
                if (m - n > floor((maxInt - delta) / handledCPCountPlusOne)) {
                    error$1("overflow");
                }
                delta += (m - n) * handledCPCountPlusOne;
                n = m;
                var _iteratorNormalCompletion3 = true;
                var _didIteratorError3 = false;
                var _iteratorError3 = void 0;
                try {
                    for(var _iterator3 = input1[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true){
                        var _currentValue = _step3.value;
                        if (_currentValue < n && ++delta > maxInt) {
                            error$1("overflow");
                        }
                        if (_currentValue == n) {
                            var q = delta;
                            for(var k = base;; k += base){
                                var t = k <= bias ? tMin : k >= bias + tMax ? tMax : k - bias;
                                if (q < t) {
                                    break;
                                }
                                var qMinusT = q - t;
                                var baseMinusT = base - t;
                                output.push(stringFromCharCode(digitToBasic(t + qMinusT % baseMinusT, 0)));
                                q = floor(qMinusT / baseMinusT);
                            }
                            output.push(stringFromCharCode(digitToBasic(q, 0)));
                            bias = adapt(delta, handledCPCountPlusOne, handledCPCount == basicLength);
                            delta = 0;
                            ++handledCPCount;
                        }
                    }
                } catch (err) {
                    _didIteratorError3 = true;
                    _iteratorError3 = err;
                } finally{
                    try {
                        if (!_iteratorNormalCompletion3 && _iterator3.return) {
                            _iterator3.return();
                        }
                    } finally{
                        if (_didIteratorError3) {
                            throw _iteratorError3;
                        }
                    }
                }
                ++delta;
                ++n;
            }
            return output.join("");
        };
        var toUnicode = function toUnicode2(input1) {
            return mapDomain(input1, function(string1) {
                return regexPunycode.test(string1) ? decode(string1.slice(4).toLowerCase()) : string1;
            });
        };
        var toASCII = function toASCII2(input1) {
            return mapDomain(input1, function(string1) {
                return regexNonASCII.test(string1) ? "xn--" + encode(string1) : string1;
            });
        };
        var punycode = {
            version: "2.1.0",
            ucs2: {
                decode: ucs2decode,
                encode: ucs2encode
            },
            decode,
            encode,
            toASCII,
            toUnicode
        };
        var SCHEMES2 = {
        };
        function pctEncChar2(chr) {
            var c = chr.charCodeAt(0);
            var e = void 0;
            if (c < 16) e = "%0" + c.toString(16).toUpperCase();
            else if (c < 128) e = "%" + c.toString(16).toUpperCase();
            else if (c < 2048) e = "%" + (c >> 6 | 192).toString(16).toUpperCase() + "%" + (c & 63 | 128).toString(16).toUpperCase();
            else e = "%" + (c >> 12 | 224).toString(16).toUpperCase() + "%" + (c >> 6 & 63 | 128).toString(16).toUpperCase() + "%" + (c & 63 | 128).toString(16).toUpperCase();
            return e;
        }
        function pctDecChars2(str1) {
            var newStr = "";
            var i5 = 0;
            var il = str1.length;
            while(i5 < il){
                var c = parseInt(str1.substr(i5 + 1, 2), 16);
                if (c < 128) {
                    newStr += String.fromCharCode(c);
                    i5 += 3;
                } else if (c >= 194 && c < 224) {
                    if (il - i5 >= 6) {
                        var c2 = parseInt(str1.substr(i5 + 4, 2), 16);
                        newStr += String.fromCharCode((c & 31) << 6 | c2 & 63);
                    } else {
                        newStr += str1.substr(i5, 6);
                    }
                    i5 += 6;
                } else if (c >= 224) {
                    if (il - i5 >= 9) {
                        var _c = parseInt(str1.substr(i5 + 4, 2), 16);
                        var c3 = parseInt(str1.substr(i5 + 7, 2), 16);
                        newStr += String.fromCharCode((c & 15) << 12 | (_c & 63) << 6 | c3 & 63);
                    } else {
                        newStr += str1.substr(i5, 9);
                    }
                    i5 += 9;
                } else {
                    newStr += str1.substr(i5, 3);
                    i5 += 3;
                }
            }
            return newStr;
        }
        function _normalizeComponentEncoding(components, protocol) {
            function decodeUnreserved2(str1) {
                var decStr = pctDecChars2(str1);
                return !decStr.match(protocol.UNRESERVED) ? str1 : decStr;
            }
            if (components.scheme) components.scheme = String(components.scheme).replace(protocol.PCT_ENCODED, decodeUnreserved2).toLowerCase().replace(protocol.NOT_SCHEME, "");
            if (components.userinfo !== void 0) components.userinfo = String(components.userinfo).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_USERINFO, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.host !== void 0) components.host = String(components.host).replace(protocol.PCT_ENCODED, decodeUnreserved2).toLowerCase().replace(protocol.NOT_HOST, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.path !== void 0) components.path = String(components.path).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(components.scheme ? protocol.NOT_PATH : protocol.NOT_PATH_NOSCHEME, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.query !== void 0) components.query = String(components.query).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_QUERY, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            if (components.fragment !== void 0) components.fragment = String(components.fragment).replace(protocol.PCT_ENCODED, decodeUnreserved2).replace(protocol.NOT_FRAGMENT, pctEncChar2).replace(protocol.PCT_ENCODED, toUpperCase);
            return components;
        }
        function _stripLeadingZeros(str1) {
            return str1.replace(/^0*(.*)/, "$1") || "0";
        }
        function _normalizeIPv4(host, protocol) {
            var matches = host.match(protocol.IPV4ADDRESS) || [];
            var _matches = slicedToArray(matches, 2), address = _matches[1];
            if (address) {
                return address.split(".").map(_stripLeadingZeros).join(".");
            } else {
                return host;
            }
        }
        function _normalizeIPv6(host, protocol) {
            var matches = host.match(protocol.IPV6ADDRESS) || [];
            var _matches2 = slicedToArray(matches, 3), address = _matches2[1], zone = _matches2[2];
            if (address) {
                var _address$toLowerCase$ = address.toLowerCase().split("::").reverse(), _address$toLowerCase$2 = slicedToArray(_address$toLowerCase$, 2), last = _address$toLowerCase$2[0], first = _address$toLowerCase$2[1];
                var firstFields = first ? first.split(":").map(_stripLeadingZeros) : [];
                var lastFields = last.split(":").map(_stripLeadingZeros);
                var isLastFieldIPv4Address = protocol.IPV4ADDRESS.test(lastFields[lastFields.length - 1]);
                var fieldCount = isLastFieldIPv4Address ? 7 : 8;
                var lastFieldsStart = lastFields.length - fieldCount;
                var fields = Array(fieldCount);
                for(var x = 0; x < fieldCount; ++x){
                    fields[x] = firstFields[x] || lastFields[lastFieldsStart + x] || "";
                }
                if (isLastFieldIPv4Address) {
                    fields[fieldCount - 1] = _normalizeIPv4(fields[fieldCount - 1], protocol);
                }
                var allZeroFields = fields.reduce(function(acc, field, index) {
                    if (!field || field === "0") {
                        var lastLongest = acc[acc.length - 1];
                        if (lastLongest && lastLongest.index + lastLongest.length === index) {
                            lastLongest.length++;
                        } else {
                            acc.push({
                                index,
                                length: 1
                            });
                        }
                    }
                    return acc;
                }, []);
                var longestZeroFields = allZeroFields.sort(function(a, b) {
                    return b.length - a.length;
                })[0];
                var newHost = void 0;
                if (longestZeroFields && longestZeroFields.length > 1) {
                    var newFirst = fields.slice(0, longestZeroFields.index);
                    var newLast = fields.slice(longestZeroFields.index + longestZeroFields.length);
                    newHost = newFirst.join(":") + "::" + newLast.join(":");
                } else {
                    newHost = fields.join(":");
                }
                if (zone) {
                    newHost += "%" + zone;
                }
                return newHost;
            } else {
                return host;
            }
        }
        var URI_PARSE = /^(?:([^:\/?#]+):)?(?:\/\/((?:([^\/?#@]*)@)?(\[[^\/?#\]]+\]|[^\/?#:]*)(?:\:(\d*))?))?([^?#]*)(?:\?([^#]*))?(?:#((?:.|\n|\r)*))?/i;
        var NO_MATCH_IS_UNDEFINED = "".match(/(){0}/)[1] === void 0;
        function parse21(uriString) {
            var options12 = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : {
            };
            var components = {
            };
            var protocol = options12.iri !== false ? IRI_PROTOCOL : URI_PROTOCOL;
            if (options12.reference === "suffix") uriString = (options12.scheme ? options12.scheme + ":" : "") + "//" + uriString;
            var matches = uriString.match(URI_PARSE);
            if (matches) {
                if (NO_MATCH_IS_UNDEFINED) {
                    components.scheme = matches[1];
                    components.userinfo = matches[3];
                    components.host = matches[4];
                    components.port = parseInt(matches[5], 10);
                    components.path = matches[6] || "";
                    components.query = matches[7];
                    components.fragment = matches[8];
                    if (isNaN(components.port)) {
                        components.port = matches[5];
                    }
                } else {
                    components.scheme = matches[1] || void 0;
                    components.userinfo = uriString.indexOf("@") !== -1 ? matches[3] : void 0;
                    components.host = uriString.indexOf("//") !== -1 ? matches[4] : void 0;
                    components.port = parseInt(matches[5], 10);
                    components.path = matches[6] || "";
                    components.query = uriString.indexOf("?") !== -1 ? matches[7] : void 0;
                    components.fragment = uriString.indexOf("#") !== -1 ? matches[8] : void 0;
                    if (isNaN(components.port)) {
                        components.port = uriString.match(/\/\/(?:.|\n)*\:(?:\/|\?|\#|$)/) ? matches[4] : void 0;
                    }
                }
                if (components.host) {
                    components.host = _normalizeIPv6(_normalizeIPv4(components.host, protocol), protocol);
                }
                if (components.scheme === void 0 && components.userinfo === void 0 && components.host === void 0 && components.port === void 0 && !components.path && components.query === void 0) {
                    components.reference = "same-document";
                } else if (components.scheme === void 0) {
                    components.reference = "relative";
                } else if (components.fragment === void 0) {
                    components.reference = "absolute";
                } else {
                    components.reference = "uri";
                }
                if (options12.reference && options12.reference !== "suffix" && options12.reference !== components.reference) {
                    components.error = components.error || "URI is not a " + options12.reference + " reference.";
                }
                var schemeHandler = SCHEMES2[(options12.scheme || components.scheme || "").toLowerCase()];
                if (!options12.unicodeSupport && (!schemeHandler || !schemeHandler.unicodeSupport)) {
                    if (components.host && (options12.domainHost || schemeHandler && schemeHandler.domainHost)) {
                        try {
                            components.host = punycode.toASCII(components.host.replace(protocol.PCT_ENCODED, pctDecChars2).toLowerCase());
                        } catch (e) {
                            components.error = components.error || "Host's domain name can not be converted to ASCII via punycode: " + e;
                        }
                    }
                    _normalizeComponentEncoding(components, URI_PROTOCOL);
                } else {
                    _normalizeComponentEncoding(components, protocol);
                }
                if (schemeHandler && schemeHandler.parse) {
                    schemeHandler.parse(components, options12);
                }
            } else {
                components.error = components.error || "URI can not be parsed.";
            }
            return components;
        }
        function _recomposeAuthority(components, options12) {
            var protocol = options12.iri !== false ? IRI_PROTOCOL : URI_PROTOCOL;
            var uriTokens = [];
            if (components.userinfo !== void 0) {
                uriTokens.push(components.userinfo);
                uriTokens.push("@");
            }
            if (components.host !== void 0) {
                uriTokens.push(_normalizeIPv6(_normalizeIPv4(String(components.host), protocol), protocol).replace(protocol.IPV6ADDRESS, function(_, $1, $2) {
                    return "[" + $1 + ($2 ? "%25" + $2 : "") + "]";
                }));
            }
            if (typeof components.port === "number" || typeof components.port === "string") {
                uriTokens.push(":");
                uriTokens.push(String(components.port));
            }
            return uriTokens.length ? uriTokens.join("") : void 0;
        }
        var RDS1 = /^\.\.?\//;
        var RDS2 = /^\/\.(\/|$)/;
        var RDS3 = /^\/\.\.(\/|$)/;
        var RDS5 = /^\/?(?:.|\n)*?(?=\/|$)/;
        function removeDotSegments2(input1) {
            var output = [];
            while(input1.length){
                if (input1.match(RDS1)) {
                    input1 = input1.replace(RDS1, "");
                } else if (input1.match(RDS2)) {
                    input1 = input1.replace(RDS2, "/");
                } else if (input1.match(RDS3)) {
                    input1 = input1.replace(RDS3, "/");
                    output.pop();
                } else if (input1 === "." || input1 === "..") {
                    input1 = "";
                } else {
                    var im = input1.match(RDS5);
                    if (im) {
                        var s = im[0];
                        input1 = input1.slice(s.length);
                        output.push(s);
                    } else {
                        throw new Error("Unexpected dot segment condition");
                    }
                }
            }
            return output.join("");
        }
        function serialize2(components) {
            var options12 = arguments.length > 1 && arguments[1] !== void 0 ? arguments[1] : {
            };
            var protocol = options12.iri ? IRI_PROTOCOL : URI_PROTOCOL;
            var uriTokens = [];
            var schemeHandler = SCHEMES2[(options12.scheme || components.scheme || "").toLowerCase()];
            if (schemeHandler && schemeHandler.serialize) schemeHandler.serialize(components, options12);
            if (components.host) {
                if (protocol.IPV6ADDRESS.test(components.host)) ;
                else if (options12.domainHost || schemeHandler && schemeHandler.domainHost) {
                    try {
                        components.host = !options12.iri ? punycode.toASCII(components.host.replace(protocol.PCT_ENCODED, pctDecChars2).toLowerCase()) : punycode.toUnicode(components.host);
                    } catch (e) {
                        components.error = components.error || "Host's domain name can not be converted to " + (!options12.iri ? "ASCII" : "Unicode") + " via punycode: " + e;
                    }
                }
            }
            _normalizeComponentEncoding(components, protocol);
            if (options12.reference !== "suffix" && components.scheme) {
                uriTokens.push(components.scheme);
                uriTokens.push(":");
            }
            var authority = _recomposeAuthority(components, options12);
            if (authority !== void 0) {
                if (options12.reference !== "suffix") {
                    uriTokens.push("//");
                }
                uriTokens.push(authority);
                if (components.path && components.path.charAt(0) !== "/") {
                    uriTokens.push("/");
                }
            }
            if (components.path !== void 0) {
                var s = components.path;
                if (!options12.absolutePath && (!schemeHandler || !schemeHandler.absolutePath)) {
                    s = removeDotSegments2(s);
                }
                if (authority === void 0) {
                    s = s.replace(/^\/\//, "/%2F");
                }
                uriTokens.push(s);
            }
            if (components.query !== void 0) {
                uriTokens.push("?");
                uriTokens.push(components.query);
            }
            if (components.fragment !== void 0) {
                uriTokens.push("#");
                uriTokens.push(components.fragment);
            }
            return uriTokens.join("");
        }
        function resolveComponents2(base2, relative3) {
            var options12 = arguments.length > 2 && arguments[2] !== void 0 ? arguments[2] : {
            };
            var skipNormalization = arguments[3];
            var target = {
            };
            if (!skipNormalization) {
                base2 = parse21(serialize2(base2, options12), options12);
                relative3 = parse21(serialize2(relative3, options12), options12);
            }
            options12 = options12 || {
            };
            if (!options12.tolerant && relative3.scheme) {
                target.scheme = relative3.scheme;
                target.userinfo = relative3.userinfo;
                target.host = relative3.host;
                target.port = relative3.port;
                target.path = removeDotSegments2(relative3.path || "");
                target.query = relative3.query;
            } else {
                if (relative3.userinfo !== void 0 || relative3.host !== void 0 || relative3.port !== void 0) {
                    target.userinfo = relative3.userinfo;
                    target.host = relative3.host;
                    target.port = relative3.port;
                    target.path = removeDotSegments2(relative3.path || "");
                    target.query = relative3.query;
                } else {
                    if (!relative3.path) {
                        target.path = base2.path;
                        if (relative3.query !== void 0) {
                            target.query = relative3.query;
                        } else {
                            target.query = base2.query;
                        }
                    } else {
                        if (relative3.path.charAt(0) === "/") {
                            target.path = removeDotSegments2(relative3.path);
                        } else {
                            if ((base2.userinfo !== void 0 || base2.host !== void 0 || base2.port !== void 0) && !base2.path) {
                                target.path = "/" + relative3.path;
                            } else if (!base2.path) {
                                target.path = relative3.path;
                            } else {
                                target.path = base2.path.slice(0, base2.path.lastIndexOf("/") + 1) + relative3.path;
                            }
                            target.path = removeDotSegments2(target.path);
                        }
                        target.query = relative3.query;
                    }
                    target.userinfo = base2.userinfo;
                    target.host = base2.host;
                    target.port = base2.port;
                }
                target.scheme = base2.scheme;
            }
            target.fragment = relative3.fragment;
            return target;
        }
        function resolve21(baseURI, relativeURI, options12) {
            var schemelessOptions = assign({
                scheme: "null"
            }, options12);
            return serialize2(resolveComponents2(parse21(baseURI, schemelessOptions), parse21(relativeURI, schemelessOptions), schemelessOptions, true), schemelessOptions);
        }
        function normalize21(uri, options12) {
            if (typeof uri === "string") {
                uri = serialize2(parse21(uri, options12), options12);
            } else if (typeOf(uri) === "object") {
                uri = parse21(serialize2(uri, options12), options12);
            }
            return uri;
        }
        function equal2(uriA, uriB, options12) {
            if (typeof uriA === "string") {
                uriA = serialize2(parse21(uriA, options12), options12);
            } else if (typeOf(uriA) === "object") {
                uriA = serialize2(uriA, options12);
            }
            if (typeof uriB === "string") {
                uriB = serialize2(parse21(uriB, options12), options12);
            } else if (typeOf(uriB) === "object") {
                uriB = serialize2(uriB, options12);
            }
            return uriA === uriB;
        }
        function escapeComponent2(str1, options12) {
            return str1 && str1.toString().replace(!options12 || !options12.iri ? URI_PROTOCOL.ESCAPE : IRI_PROTOCOL.ESCAPE, pctEncChar2);
        }
        function unescapeComponent2(str1, options12) {
            return str1 && str1.toString().replace(!options12 || !options12.iri ? URI_PROTOCOL.PCT_ENCODED : IRI_PROTOCOL.PCT_ENCODED, pctDecChars2);
        }
        var handler = {
            scheme: "http",
            domainHost: true,
            parse: function parse31(components, options12) {
                if (!components.host) {
                    components.error = components.error || "HTTP URIs must have a host.";
                }
                return components;
            },
            serialize: function serialize3(components, options12) {
                var secure = String(components.scheme).toLowerCase() === "https";
                if (components.port === (secure ? 443 : 80) || components.port === "") {
                    components.port = void 0;
                }
                if (!components.path) {
                    components.path = "/";
                }
                return components;
            }
        };
        var handler$1 = {
            scheme: "https",
            domainHost: handler.domainHost,
            parse: handler.parse,
            serialize: handler.serialize
        };
        function isSecure(wsComponents) {
            return typeof wsComponents.secure === "boolean" ? wsComponents.secure : String(wsComponents.scheme).toLowerCase() === "wss";
        }
        var handler$2 = {
            scheme: "ws",
            domainHost: true,
            parse: function parse31(components, options12) {
                var wsComponents = components;
                wsComponents.secure = isSecure(wsComponents);
                wsComponents.resourceName = (wsComponents.path || "/") + (wsComponents.query ? "?" + wsComponents.query : "");
                wsComponents.path = void 0;
                wsComponents.query = void 0;
                return wsComponents;
            },
            serialize: function serialize3(wsComponents, options12) {
                if (wsComponents.port === (isSecure(wsComponents) ? 443 : 80) || wsComponents.port === "") {
                    wsComponents.port = void 0;
                }
                if (typeof wsComponents.secure === "boolean") {
                    wsComponents.scheme = wsComponents.secure ? "wss" : "ws";
                    wsComponents.secure = void 0;
                }
                if (wsComponents.resourceName) {
                    var _wsComponents$resourc = wsComponents.resourceName.split("?"), _wsComponents$resourc2 = slicedToArray(_wsComponents$resourc, 2), path2 = _wsComponents$resourc2[0], query = _wsComponents$resourc2[1];
                    wsComponents.path = path2 && path2 !== "/" ? path2 : void 0;
                    wsComponents.query = query;
                    wsComponents.resourceName = void 0;
                }
                wsComponents.fragment = void 0;
                return wsComponents;
            }
        };
        var handler$3 = {
            scheme: "wss",
            domainHost: handler$2.domainHost,
            parse: handler$2.parse,
            serialize: handler$2.serialize
        };
        var O = {
        };
        var UNRESERVED$$ = "[A-Za-z0-9\\-\\.\\_\\~\\xA0-\\u200D\\u2010-\\u2029\\u202F-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]";
        var HEXDIG$$ = "[0-9A-Fa-f]";
        var PCT_ENCODED$ = subexp(subexp("%[EFef]" + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$) + "|" + subexp("%[89A-Fa-f]" + HEXDIG$$ + "%" + HEXDIG$$ + HEXDIG$$) + "|" + subexp("%" + HEXDIG$$ + HEXDIG$$));
        var ATEXT$$ = "[A-Za-z0-9\\!\\$\\%\\'\\*\\+\\-\\^\\_\\`\\{\\|\\}\\~]";
        var QTEXT$$ = "[\\!\\$\\%\\'\\(\\)\\*\\+\\,\\-\\.0-9\\<\\>A-Z\\x5E-\\x7E]";
        var VCHAR$$ = merge1(QTEXT$$, '[\\"\\\\]');
        var SOME_DELIMS$$ = "[\\!\\$\\'\\(\\)\\*\\+\\,\\;\\:\\@]";
        var UNRESERVED = new RegExp(UNRESERVED$$, "g");
        var PCT_ENCODED = new RegExp(PCT_ENCODED$, "g");
        var NOT_LOCAL_PART = new RegExp(merge1("[^]", ATEXT$$, "[\\.]", '[\\"]', VCHAR$$), "g");
        var NOT_HFNAME = new RegExp(merge1("[^]", UNRESERVED$$, SOME_DELIMS$$), "g");
        var NOT_HFVALUE = NOT_HFNAME;
        function decodeUnreserved(str1) {
            var decStr = pctDecChars2(str1);
            return !decStr.match(UNRESERVED) ? str1 : decStr;
        }
        var handler$4 = {
            scheme: "mailto",
            parse: function parse$$1(components, options12) {
                var mailtoComponents = components;
                var to = mailtoComponents.to = mailtoComponents.path ? mailtoComponents.path.split(",") : [];
                mailtoComponents.path = void 0;
                if (mailtoComponents.query) {
                    var unknownHeaders = false;
                    var headers = {
                    };
                    var hfields = mailtoComponents.query.split("&");
                    for(var x = 0, xl = hfields.length; x < xl; ++x){
                        var hfield = hfields[x].split("=");
                        switch(hfield[0]){
                            case "to":
                                var toAddrs = hfield[1].split(",");
                                for(var _x = 0, _xl = toAddrs.length; _x < _xl; ++_x){
                                    to.push(toAddrs[_x]);
                                }
                                break;
                            case "subject":
                                mailtoComponents.subject = unescapeComponent2(hfield[1], options12);
                                break;
                            case "body":
                                mailtoComponents.body = unescapeComponent2(hfield[1], options12);
                                break;
                            default:
                                unknownHeaders = true;
                                headers[unescapeComponent2(hfield[0], options12)] = unescapeComponent2(hfield[1], options12);
                                break;
                        }
                    }
                    if (unknownHeaders) mailtoComponents.headers = headers;
                }
                mailtoComponents.query = void 0;
                for(var _x2 = 0, _xl2 = to.length; _x2 < _xl2; ++_x2){
                    var addr = to[_x2].split("@");
                    addr[0] = unescapeComponent2(addr[0]);
                    if (!options12.unicodeSupport) {
                        try {
                            addr[1] = punycode.toASCII(unescapeComponent2(addr[1], options12).toLowerCase());
                        } catch (e) {
                            mailtoComponents.error = mailtoComponents.error || "Email address's domain name can not be converted to ASCII via punycode: " + e;
                        }
                    } else {
                        addr[1] = unescapeComponent2(addr[1], options12).toLowerCase();
                    }
                    to[_x2] = addr.join("@");
                }
                return mailtoComponents;
            },
            serialize: function serialize$$1(mailtoComponents, options12) {
                var components = mailtoComponents;
                var to = toArray(mailtoComponents.to);
                if (to) {
                    for(var x = 0, xl = to.length; x < xl; ++x){
                        var toAddr = String(to[x]);
                        var atIdx = toAddr.lastIndexOf("@");
                        var localPart = toAddr.slice(0, atIdx).replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_LOCAL_PART, pctEncChar2);
                        var domain = toAddr.slice(atIdx + 1);
                        try {
                            domain = !options12.iri ? punycode.toASCII(unescapeComponent2(domain, options12).toLowerCase()) : punycode.toUnicode(domain);
                        } catch (e) {
                            components.error = components.error || "Email address's domain name can not be converted to " + (!options12.iri ? "ASCII" : "Unicode") + " via punycode: " + e;
                        }
                        to[x] = localPart + "@" + domain;
                    }
                    components.path = to.join(",");
                }
                var headers = mailtoComponents.headers = mailtoComponents.headers || {
                };
                if (mailtoComponents.subject) headers["subject"] = mailtoComponents.subject;
                if (mailtoComponents.body) headers["body"] = mailtoComponents.body;
                var fields = [];
                for(var name19 in headers){
                    if (headers[name19] !== O[name19]) {
                        fields.push(name19.replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_HFNAME, pctEncChar2) + "=" + headers[name19].replace(PCT_ENCODED, decodeUnreserved).replace(PCT_ENCODED, toUpperCase).replace(NOT_HFVALUE, pctEncChar2));
                    }
                }
                if (fields.length) {
                    components.query = fields.join("&");
                }
                return components;
            }
        };
        var URN_PARSE = /^([^\:]+)\:(.*)/;
        var handler$5 = {
            scheme: "urn",
            parse: function parse$$1(components, options12) {
                var matches = components.path && components.path.match(URN_PARSE);
                var urnComponents = components;
                if (matches) {
                    var scheme = options12.scheme || urnComponents.scheme || "urn";
                    var nid = matches[1].toLowerCase();
                    var nss = matches[2];
                    var urnScheme = scheme + ":" + (options12.nid || nid);
                    var schemeHandler = SCHEMES2[urnScheme];
                    urnComponents.nid = nid;
                    urnComponents.nss = nss;
                    urnComponents.path = void 0;
                    if (schemeHandler) {
                        urnComponents = schemeHandler.parse(urnComponents, options12);
                    }
                } else {
                    urnComponents.error = urnComponents.error || "URN can not be parsed.";
                }
                return urnComponents;
            },
            serialize: function serialize$$1(urnComponents, options12) {
                var scheme = options12.scheme || urnComponents.scheme || "urn";
                var nid = urnComponents.nid;
                var urnScheme = scheme + ":" + (options12.nid || nid);
                var schemeHandler = SCHEMES2[urnScheme];
                if (schemeHandler) {
                    urnComponents = schemeHandler.serialize(urnComponents, options12);
                }
                var uriComponents = urnComponents;
                var nss = urnComponents.nss;
                uriComponents.path = (nid || options12.nid) + ":" + nss;
                return uriComponents;
            }
        };
        var UUID = /^[0-9A-Fa-f]{8}(?:\-[0-9A-Fa-f]{4}){3}\-[0-9A-Fa-f]{12}$/;
        var handler$6 = {
            scheme: "urn:uuid",
            parse: function parse31(urnComponents, options12) {
                var uuidComponents = urnComponents;
                uuidComponents.uuid = uuidComponents.nss;
                uuidComponents.nss = void 0;
                if (!options12.tolerant && (!uuidComponents.uuid || !uuidComponents.uuid.match(UUID))) {
                    uuidComponents.error = uuidComponents.error || "UUID is not valid.";
                }
                return uuidComponents;
            },
            serialize: function serialize3(uuidComponents, options12) {
                var urnComponents = uuidComponents;
                urnComponents.nss = (uuidComponents.uuid || "").toLowerCase();
                return urnComponents;
            }
        };
        SCHEMES2[handler.scheme] = handler;
        SCHEMES2[handler$1.scheme] = handler$1;
        SCHEMES2[handler$2.scheme] = handler$2;
        SCHEMES2[handler$3.scheme] = handler$3;
        SCHEMES2[handler$4.scheme] = handler$4;
        SCHEMES2[handler$5.scheme] = handler$5;
        SCHEMES2[handler$6.scheme] = handler$6;
        exports2.SCHEMES = SCHEMES2;
        exports2.pctEncChar = pctEncChar2;
        exports2.pctDecChars = pctDecChars2;
        exports2.parse = parse21;
        exports2.removeDotSegments = removeDotSegments2;
        exports2.serialize = serialize2;
        exports2.resolveComponents = resolveComponents2;
        exports2.resolve = resolve21;
        exports2.normalize = normalize21;
        exports2.equal = equal2;
        exports2.escapeComponent = escapeComponent2;
        exports2.unescapeComponent = unescapeComponent2;
        Object.defineProperty(exports2, "__esModule", {
            value: true
        });
    });
});
var __pika_web_default_export_for_treeshaking__ = getDefaultExportFromCjs(uri_all);
function getDefaultExportFromCjs1(x) {
    return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, "default") ? x["default"] : x;
}
function createCommonjsModule2(fn, basedir, module) {
    return module = {
        path: basedir,
        exports: {
        },
        require: function(path3, base) {
            return commonjsRequire3(path3, base === void 0 || base === null ? module.path : base);
        }
    }, fn(module, module.exports), module.exports;
}
function commonjsRequire3() {
    throw new Error("Dynamic requires are not currently supported by @rollup/plugin-commonjs");
}
var code2 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.regexpCode = exports.getProperty = exports.safeStringify = exports.stringify = exports.strConcat = exports.addCodeArg = exports.str = exports._ = exports.nil = exports._Code = exports.Name = exports.IDENTIFIER = exports._CodeOrName = void 0;
    class _CodeOrName {
    }
    exports._CodeOrName = _CodeOrName;
    exports.IDENTIFIER = /^[a-z$_][a-z$_0-9]*$/i;
    class Name2 extends _CodeOrName {
        constructor(s){
            super();
            if (!exports.IDENTIFIER.test(s)) throw new Error("CodeGen: name must be a valid identifier");
            this.str = s;
        }
        toString() {
            return this.str;
        }
        emptyStr() {
            return false;
        }
        get names() {
            return {
                [this.str]: 1
            };
        }
    }
    exports.Name = Name2;
    class _Code extends _CodeOrName {
        constructor(code21){
            super();
            this._items = typeof code21 === "string" ? [
                code21
            ] : code21;
        }
        toString() {
            return this.str;
        }
        emptyStr() {
            if (this._items.length > 1) return false;
            const item = this._items[0];
            return item === "" || item === '""';
        }
        get str() {
            var _a;
            return (_a = this._str) !== null && _a !== void 0 ? _a : this._str = this._items.reduce((s1, c)=>`${s1}${c}`
            , "");
        }
        get names() {
            var _a;
            return (_a = this._names) !== null && _a !== void 0 ? _a : this._names = this._items.reduce((names, c)=>{
                if (c instanceof Name2) names[c.str] = (names[c.str] || 0) + 1;
                return names;
            }, {
            });
        }
    }
    exports._Code = _Code;
    exports.nil = new _Code("");
    function _2(strs, ...args4) {
        const code22 = [
            strs[0]
        ];
        let i5 = 0;
        while(i5 < args4.length){
            addCodeArg(code22, args4[i5]);
            code22.push(strs[++i5]);
        }
        return new _Code(code22);
    }
    exports._ = _2;
    const plus = new _Code("+");
    function str2(strs, ...args4) {
        const expr = [
            safeStringify(strs[0])
        ];
        let i5 = 0;
        while(i5 < args4.length){
            expr.push(plus);
            addCodeArg(expr, args4[i5]);
            expr.push(plus, safeStringify(strs[++i5]));
        }
        optimize(expr);
        return new _Code(expr);
    }
    exports.str = str2;
    function addCodeArg(code22, arg4) {
        if (arg4 instanceof _Code) code22.push(...arg4._items);
        else if (arg4 instanceof Name2) code22.push(arg4);
        else code22.push(interpolate(arg4));
    }
    exports.addCodeArg = addCodeArg;
    function optimize(expr) {
        let i5 = 1;
        while(i5 < expr.length - 1){
            if (expr[i5] === plus) {
                const res = mergeExprItems(expr[i5 - 1], expr[i5 + 1]);
                if (res !== void 0) {
                    expr.splice(i5 - 1, 3, res);
                    continue;
                }
                expr[i5++] = "+";
            }
            i5++;
        }
    }
    function mergeExprItems(a, b) {
        if (b === '""') return a;
        if (a === '""') return b;
        if (typeof a == "string") {
            if (b instanceof Name2 || a[a.length - 1] !== '"') return;
            if (typeof b != "string") return `${a.slice(0, -1)}${b}"`;
            if (b[0] === '"') return a.slice(0, -1) + b.slice(1);
            return;
        }
        if (typeof b == "string" && b[0] === '"' && !(a instanceof Name2)) return `"${a}${b.slice(1)}`;
        return;
    }
    function strConcat(c1, c2) {
        return c2.emptyStr() ? c1 : c1.emptyStr() ? c2 : str2`${c1}${c2}`;
    }
    exports.strConcat = strConcat;
    function interpolate(x) {
        return typeof x == "number" || typeof x == "boolean" || x === null ? x : safeStringify(Array.isArray(x) ? x.join(",") : x);
    }
    function stringify2(x) {
        return new _Code(safeStringify(x));
    }
    exports.stringify = stringify2;
    function safeStringify(x) {
        return JSON.stringify(x).replace(/\u2028/g, "\\u2028").replace(/\u2029/g, "\\u2029");
    }
    exports.safeStringify = safeStringify;
    function getProperty(key) {
        return typeof key == "string" && exports.IDENTIFIER.test(key) ? new _Code(`.${key}`) : _2`[${key}]`;
    }
    exports.getProperty = getProperty;
    function regexpCode(rx) {
        return new _Code(rx.toString());
    }
    exports.regexpCode = regexpCode;
});
var scope = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.ValueScope = exports.ValueScopeName = exports.Scope = exports.varKinds = exports.UsedValueState = void 0;
    class ValueError extends Error {
        constructor(name19){
            super(`CodeGen: "code" for ${name19} not defined`);
            this.value = name19.value;
        }
    }
    var UsedValueState;
    (function(UsedValueState2) {
        UsedValueState2[UsedValueState2["Started"] = 0] = "Started";
        UsedValueState2[UsedValueState2["Completed"] = 1] = "Completed";
    })(UsedValueState = exports.UsedValueState || (exports.UsedValueState = {
    }));
    exports.varKinds = {
        const: new code2.Name("const"),
        let: new code2.Name("let"),
        var: new code2.Name("var")
    };
    class Scope {
        constructor({ prefixes , parent  } = {
        }){
            this._names = {
            };
            this._prefixes = prefixes;
            this._parent = parent;
        }
        toName(nameOrPrefix) {
            return nameOrPrefix instanceof code2.Name ? nameOrPrefix : this.name(nameOrPrefix);
        }
        name(prefix) {
            return new code2.Name(this._newName(prefix));
        }
        _newName(prefix) {
            const ng = this._names[prefix] || this._nameGroup(prefix);
            return `${prefix}${ng.index++}`;
        }
        _nameGroup(prefix) {
            var _a, _b;
            if (((_b = (_a = this._parent) === null || _a === void 0 ? void 0 : _a._prefixes) === null || _b === void 0 ? void 0 : _b.has(prefix)) || this._prefixes && !this._prefixes.has(prefix)) {
                throw new Error(`CodeGen: prefix "${prefix}" is not allowed in this scope`);
            }
            return this._names[prefix] = {
                prefix,
                index: 0
            };
        }
    }
    exports.Scope = Scope;
    class ValueScopeName extends code2.Name {
        constructor(prefix1, nameStr){
            super(nameStr);
            this.prefix = prefix1;
        }
        setValue(value, { property , itemIndex  }) {
            this.value = value;
            this.scopePath = code2._`.${new code2.Name(property)}[${itemIndex}]`;
        }
    }
    exports.ValueScopeName = ValueScopeName;
    const line3 = code2._`\n`;
    class ValueScope extends Scope {
        constructor(opts){
            super(opts);
            this._values = {
            };
            this._scope = opts.scope;
            this.opts = {
                ...opts,
                _n: opts.lines ? line3 : code2.nil
            };
        }
        get() {
            return this._scope;
        }
        name(prefix) {
            return new ValueScopeName(prefix, this._newName(prefix));
        }
        value(nameOrPrefix, value) {
            var _a;
            if (value.ref === void 0) throw new Error("CodeGen: ref must be passed in value");
            const name20 = this.toName(nameOrPrefix);
            const { prefix: prefix2  } = name20;
            const valueKey = (_a = value.key) !== null && _a !== void 0 ? _a : value.ref;
            let vs = this._values[prefix2];
            if (vs) {
                const _name = vs.get(valueKey);
                if (_name) return _name;
            } else {
                vs = this._values[prefix2] = new Map();
            }
            vs.set(valueKey, name20);
            const s = this._scope[prefix2] || (this._scope[prefix2] = []);
            const itemIndex = s.length;
            s[itemIndex] = value.ref;
            name20.setValue(value, {
                property: prefix2,
                itemIndex
            });
            return name20;
        }
        getValue(prefix, keyOrRef) {
            const vs = this._values[prefix];
            if (!vs) return;
            return vs.get(keyOrRef);
        }
        scopeRefs(scopeName, values = this._values) {
            return this._reduceValues(values, (name20)=>{
                if (name20.scopePath === void 0) throw new Error(`CodeGen: name "${name20}" has no value`);
                return code2._`${scopeName}${name20.scopePath}`;
            });
        }
        scopeCode(values = this._values, usedValues, getCode) {
            return this._reduceValues(values, (name20)=>{
                if (name20.value === void 0) throw new Error(`CodeGen: name "${name20}" has no value`);
                return name20.value.code;
            }, usedValues, getCode);
        }
        _reduceValues(values, valueCode, usedValues = {
        }, getCode) {
            let code$12 = code2.nil;
            for(const prefix2 in values){
                const vs = values[prefix2];
                if (!vs) continue;
                const nameSet = usedValues[prefix2] = usedValues[prefix2] || new Map();
                vs.forEach((name20)=>{
                    if (nameSet.has(name20)) return;
                    nameSet.set(name20, UsedValueState.Started);
                    let c = valueCode(name20);
                    if (c) {
                        const def1 = this.opts.es5 ? exports.varKinds.var : exports.varKinds.const;
                        code$12 = code2._`${code$12}${def1} ${name20} = ${c};${this.opts._n}`;
                    } else if (c = getCode === null || getCode === void 0 ? void 0 : getCode(name20)) {
                        code$12 = code2._`${code$12}${c}${this.opts._n}`;
                    } else {
                        throw new ValueError(name20);
                    }
                    nameSet.set(name20, UsedValueState.Completed);
                });
            }
            return code$12;
        }
    }
    exports.ValueScope = ValueScope;
});
var codegen = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.or = exports.and = exports.not = exports.CodeGen = exports.operators = exports.varKinds = exports.ValueScopeName = exports.ValueScope = exports.Scope = exports.Name = exports.regexpCode = exports.stringify = exports.getProperty = exports.nil = exports.strConcat = exports.str = exports._ = void 0;
    var code_2 = code2;
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return code_2._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return code_2.str;
        }
    });
    Object.defineProperty(exports, "strConcat", {
        enumerable: true,
        get: function() {
            return code_2.strConcat;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return code_2.nil;
        }
    });
    Object.defineProperty(exports, "getProperty", {
        enumerable: true,
        get: function() {
            return code_2.getProperty;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return code_2.stringify;
        }
    });
    Object.defineProperty(exports, "regexpCode", {
        enumerable: true,
        get: function() {
            return code_2.regexpCode;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return code_2.Name;
        }
    });
    var scope_2 = scope;
    Object.defineProperty(exports, "Scope", {
        enumerable: true,
        get: function() {
            return scope_2.Scope;
        }
    });
    Object.defineProperty(exports, "ValueScope", {
        enumerable: true,
        get: function() {
            return scope_2.ValueScope;
        }
    });
    Object.defineProperty(exports, "ValueScopeName", {
        enumerable: true,
        get: function() {
            return scope_2.ValueScopeName;
        }
    });
    Object.defineProperty(exports, "varKinds", {
        enumerable: true,
        get: function() {
            return scope_2.varKinds;
        }
    });
    exports.operators = {
        GT: new code2._Code(">"),
        GTE: new code2._Code(">="),
        LT: new code2._Code("<"),
        LTE: new code2._Code("<="),
        EQ: new code2._Code("==="),
        NEQ: new code2._Code("!=="),
        NOT: new code2._Code("!"),
        OR: new code2._Code("||"),
        AND: new code2._Code("&&"),
        ADD: new code2._Code("+")
    };
    class Node1 {
        optimizeNodes() {
            return this;
        }
        optimizeNames(_names, _constants) {
            return this;
        }
    }
    class Def extends Node1 {
        constructor(varKind3, name19, rhs3){
            super();
            this.varKind = varKind3;
            this.name = name19;
            this.rhs = rhs3;
        }
        render({ es5 , _n  }) {
            const varKind1 = es5 ? scope.varKinds.var : this.varKind;
            const rhs1 = this.rhs === void 0 ? "" : ` = ${this.rhs}`;
            return `${varKind1} ${this.name}${rhs1};` + _n;
        }
        optimizeNames(names, constants) {
            if (!names[this.name.str]) return;
            if (this.rhs) this.rhs = optimizeExpr(this.rhs, names, constants);
            return this;
        }
        get names() {
            return this.rhs instanceof code2._CodeOrName ? this.rhs.names : {
            };
        }
    }
    class Assign extends Node1 {
        constructor(lhs2, rhs1, sideEffects2){
            super();
            this.lhs = lhs2;
            this.rhs = rhs1;
            this.sideEffects = sideEffects2;
        }
        render({ _n  }) {
            return `${this.lhs} = ${this.rhs};` + _n;
        }
        optimizeNames(names, constants) {
            if (this.lhs instanceof code2.Name && !names[this.lhs.str] && !this.sideEffects) return;
            this.rhs = optimizeExpr(this.rhs, names, constants);
            return this;
        }
        get names() {
            const names = this.lhs instanceof code2.Name ? {
            } : {
                ...this.lhs.names
            };
            return addExprNames(names, this.rhs);
        }
    }
    class AssignOp extends Assign {
        constructor(lhs1, op, rhs2, sideEffects1){
            super(lhs1, rhs2, sideEffects1);
            this.op = op;
        }
        render({ _n  }) {
            return `${this.lhs} ${this.op}= ${this.rhs};` + _n;
        }
    }
    class Label extends Node1 {
        constructor(label2){
            super();
            this.label = label2;
            this.names = {
            };
        }
        render({ _n  }) {
            return `${this.label}:` + _n;
        }
    }
    class Break extends Node1 {
        constructor(label3){
            super();
            this.label = label3;
            this.names = {
            };
        }
        render({ _n  }) {
            const label4 = this.label ? ` ${this.label}` : "";
            return `break${label4};` + _n;
        }
    }
    class Throw extends Node1 {
        constructor(error1){
            super();
            this.error = error1;
        }
        render({ _n  }) {
            return `throw ${this.error};` + _n;
        }
        get names() {
            return this.error.names;
        }
    }
    class AnyCode extends Node1 {
        constructor(code21){
            super();
            this.code = code21;
        }
        render({ _n  }) {
            return `${this.code};` + _n;
        }
        optimizeNodes() {
            return `${this.code}` ? this : void 0;
        }
        optimizeNames(names, constants) {
            this.code = optimizeExpr(this.code, names, constants);
            return this;
        }
        get names() {
            return this.code instanceof code2._CodeOrName ? this.code.names : {
            };
        }
    }
    class ParentNode extends Node1 {
        constructor(nodes = []){
            super();
            this.nodes = nodes;
        }
        render(opts) {
            return this.nodes.reduce((code22, n)=>code22 + n.render(opts)
            , "");
        }
        optimizeNodes() {
            const { nodes: nodes1  } = this;
            let i5 = nodes1.length;
            while(i5--){
                const n = nodes1[i5].optimizeNodes();
                if (Array.isArray(n)) nodes1.splice(i5, 1, ...n);
                else if (n) nodes1[i5] = n;
                else nodes1.splice(i5, 1);
            }
            return nodes1.length > 0 ? this : void 0;
        }
        optimizeNames(names, constants) {
            const { nodes: nodes1  } = this;
            let i5 = nodes1.length;
            while(i5--){
                const n = nodes1[i5];
                if (n.optimizeNames(names, constants)) continue;
                subtractNames(names, n.names);
                nodes1.splice(i5, 1);
            }
            return nodes1.length > 0 ? this : void 0;
        }
        get names() {
            return this.nodes.reduce((names, n)=>addNames(names, n.names)
            , {
            });
        }
    }
    class BlockNode extends ParentNode {
        render(opts) {
            return "{" + opts._n + super.render(opts) + "}" + opts._n;
        }
    }
    class Root extends ParentNode {
    }
    class Else extends BlockNode {
    }
    Else.kind = "else";
    class If extends BlockNode {
        constructor(condition1, nodes1){
            super(nodes1);
            this.condition = condition1;
        }
        render(opts) {
            let code22 = `if(${this.condition})` + super.render(opts);
            if (this.else) code22 += "else " + this.else.render(opts);
            return code22;
        }
        optimizeNodes() {
            super.optimizeNodes();
            const cond = this.condition;
            if (cond === true) return this.nodes;
            let e = this.else;
            if (e) {
                const ns = e.optimizeNodes();
                e = this.else = Array.isArray(ns) ? new Else(ns) : ns;
            }
            if (e) {
                if (cond === false) return e instanceof If ? e : e.nodes;
                if (this.nodes.length) return this;
                return new If(not2(cond), e instanceof If ? [
                    e
                ] : e.nodes);
            }
            if (cond === false || !this.nodes.length) return void 0;
            return this;
        }
        optimizeNames(names, constants) {
            var _a;
            this.else = (_a = this.else) === null || _a === void 0 ? void 0 : _a.optimizeNames(names, constants);
            if (!(super.optimizeNames(names, constants) || this.else)) return;
            this.condition = optimizeExpr(this.condition, names, constants);
            return this;
        }
        get names() {
            const names = super.names;
            addExprNames(names, this.condition);
            if (this.else) addNames(names, this.else.names);
            return names;
        }
    }
    If.kind = "if";
    class For extends BlockNode {
    }
    For.kind = "for";
    class ForLoop extends For {
        constructor(iteration1){
            super();
            this.iteration = iteration1;
        }
        render(opts) {
            return `for(${this.iteration})` + super.render(opts);
        }
        optimizeNames(names, constants) {
            if (!super.optimizeNames(names, constants)) return;
            this.iteration = optimizeExpr(this.iteration, names, constants);
            return this;
        }
        get names() {
            return addNames(super.names, this.iteration.names);
        }
    }
    class ForRange extends For {
        constructor(varKind1, name20, from1, to1){
            super();
            this.varKind = varKind1;
            this.name = name20;
            this.from = from1;
            this.to = to1;
        }
        render(opts) {
            const varKind2 = opts.es5 ? scope.varKinds.var : this.varKind;
            const { name: name21 , from: from1 , to: to1  } = this;
            return `for(${varKind2} ${name21}=${from1}; ${name21}<${to1}; ${name21}++)` + super.render(opts);
        }
        get names() {
            const names = addExprNames(super.names, this.from);
            return addExprNames(names, this.to);
        }
    }
    class ForIter extends For {
        constructor(loop, varKind2, name21, iterable1){
            super();
            this.loop = loop;
            this.varKind = varKind2;
            this.name = name21;
            this.iterable = iterable1;
        }
        render(opts) {
            return `for(${this.varKind} ${this.name} ${this.loop} ${this.iterable})` + super.render(opts);
        }
        optimizeNames(names, constants) {
            if (!super.optimizeNames(names, constants)) return;
            this.iterable = optimizeExpr(this.iterable, names, constants);
            return this;
        }
        get names() {
            return addNames(super.names, this.iterable.names);
        }
    }
    class Func extends BlockNode {
        constructor(name22, args4, async1){
            super();
            this.name = name22;
            this.args = args4;
            this.async = async1;
        }
        render(opts) {
            const _async = this.async ? "async " : "";
            return `${_async}function ${this.name}(${this.args})` + super.render(opts);
        }
    }
    Func.kind = "func";
    class Return extends ParentNode {
        render(opts) {
            return "return " + super.render(opts);
        }
    }
    Return.kind = "return";
    class Try extends BlockNode {
        render(opts) {
            let code22 = "try" + super.render(opts);
            if (this.catch) code22 += this.catch.render(opts);
            if (this.finally) code22 += this.finally.render(opts);
            return code22;
        }
        optimizeNodes() {
            var _a, _b;
            super.optimizeNodes();
            (_a = this.catch) === null || _a === void 0 ? void 0 : _a.optimizeNodes();
            (_b = this.finally) === null || _b === void 0 ? void 0 : _b.optimizeNodes();
            return this;
        }
        optimizeNames(names, constants) {
            var _a, _b;
            super.optimizeNames(names, constants);
            (_a = this.catch) === null || _a === void 0 ? void 0 : _a.optimizeNames(names, constants);
            (_b = this.finally) === null || _b === void 0 ? void 0 : _b.optimizeNames(names, constants);
            return this;
        }
        get names() {
            const names = super.names;
            if (this.catch) addNames(names, this.catch.names);
            if (this.finally) addNames(names, this.finally.names);
            return names;
        }
    }
    class Catch extends BlockNode {
        constructor(error2){
            super();
            this.error = error2;
        }
        render(opts) {
            return `catch(${this.error})` + super.render(opts);
        }
    }
    Catch.kind = "catch";
    class Finally extends BlockNode {
        render(opts) {
            return "finally" + super.render(opts);
        }
    }
    Finally.kind = "finally";
    class CodeGen2 {
        constructor(extScope, opts = {
        }){
            this._values = {
            };
            this._blockStarts = [];
            this._constants = {
            };
            this.opts = {
                ...opts,
                _n: opts.lines ? "\n" : ""
            };
            this._extScope = extScope;
            this._scope = new scope.Scope({
                parent: extScope
            });
            this._nodes = [
                new Root()
            ];
        }
        toString() {
            return this._root.render(this.opts);
        }
        name(prefix) {
            return this._scope.name(prefix);
        }
        scopeName(prefix) {
            return this._extScope.name(prefix);
        }
        scopeValue(prefixOrName, value) {
            const name23 = this._extScope.value(prefixOrName, value);
            const vs = this._values[name23.prefix] || (this._values[name23.prefix] = new Set());
            vs.add(name23);
            return name23;
        }
        getScopeValue(prefix, keyOrRef) {
            return this._extScope.getValue(prefix, keyOrRef);
        }
        scopeRefs(scopeName) {
            return this._extScope.scopeRefs(scopeName, this._values);
        }
        scopeCode() {
            return this._extScope.scopeCode(this._values);
        }
        _def(varKind, nameOrPrefix, rhs, constant) {
            const name23 = this._scope.toName(nameOrPrefix);
            if (rhs !== void 0 && constant) this._constants[name23.str] = rhs;
            this._leafNode(new Def(varKind, name23, rhs));
            return name23;
        }
        const(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.const, nameOrPrefix, rhs, _constant);
        }
        let(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.let, nameOrPrefix, rhs, _constant);
        }
        var(nameOrPrefix, rhs, _constant) {
            return this._def(scope.varKinds.var, nameOrPrefix, rhs, _constant);
        }
        assign(lhs, rhs, sideEffects) {
            return this._leafNode(new Assign(lhs, rhs, sideEffects));
        }
        add(lhs, rhs) {
            return this._leafNode(new AssignOp(lhs, exports.operators.ADD, rhs));
        }
        code(c) {
            if (typeof c == "function") c();
            else if (c !== code2.nil) this._leafNode(new AnyCode(c));
            return this;
        }
        object(...keyValues) {
            const code$12 = [
                "{"
            ];
            for (const [key, value6] of keyValues){
                if (code$12.length > 1) code$12.push(",");
                code$12.push(key);
                if (key !== value6 || this.opts.es5) {
                    code$12.push(":");
                    (0, code2.addCodeArg)(code$12, value6);
                }
            }
            code$12.push("}");
            return new code2._Code(code$12);
        }
        if(condition, thenBody, elseBody) {
            this._blockNode(new If(condition));
            if (thenBody && elseBody) {
                this.code(thenBody).else().code(elseBody).endIf();
            } else if (thenBody) {
                this.code(thenBody).endIf();
            } else if (elseBody) {
                throw new Error('CodeGen: "else" body without "then" body');
            }
            return this;
        }
        elseIf(condition) {
            return this._elseNode(new If(condition));
        }
        else() {
            return this._elseNode(new Else());
        }
        endIf() {
            return this._endBlockNode(If, Else);
        }
        _for(node, forBody) {
            this._blockNode(node);
            if (forBody) this.code(forBody).endFor();
            return this;
        }
        for(iteration, forBody) {
            return this._for(new ForLoop(iteration), forBody);
        }
        forRange(nameOrPrefix, from, to, forBody, varKind = this.opts.es5 ? scope.varKinds.var : scope.varKinds.let) {
            const name23 = this._scope.toName(nameOrPrefix);
            return this._for(new ForRange(varKind, name23, from, to), ()=>forBody(name23)
            );
        }
        forOf(nameOrPrefix, iterable, forBody, varKind = scope.varKinds.const) {
            const name23 = this._scope.toName(nameOrPrefix);
            if (this.opts.es5) {
                const arr = iterable instanceof code2.Name ? iterable : this.var("_arr", iterable);
                return this.forRange("_i", 0, code2._`${arr}.length`, (i5)=>{
                    this.var(name23, code2._`${arr}[${i5}]`);
                    forBody(name23);
                });
            }
            return this._for(new ForIter("of", varKind, name23, iterable), ()=>forBody(name23)
            );
        }
        forIn(nameOrPrefix, obj, forBody, varKind = this.opts.es5 ? scope.varKinds.var : scope.varKinds.const) {
            if (this.opts.ownProperties) {
                return this.forOf(nameOrPrefix, code2._`Object.keys(${obj})`, forBody);
            }
            const name23 = this._scope.toName(nameOrPrefix);
            return this._for(new ForIter("in", varKind, name23, obj), ()=>forBody(name23)
            );
        }
        endFor() {
            return this._endBlockNode(For);
        }
        label(label) {
            return this._leafNode(new Label(label));
        }
        break(label) {
            return this._leafNode(new Break(label));
        }
        return(value) {
            const node = new Return();
            this._blockNode(node);
            this.code(value);
            if (node.nodes.length !== 1) throw new Error('CodeGen: "return" should have one node');
            return this._endBlockNode(Return);
        }
        try(tryBody, catchCode, finallyCode) {
            if (!catchCode && !finallyCode) throw new Error('CodeGen: "try" without "catch" and "finally"');
            const node = new Try();
            this._blockNode(node);
            this.code(tryBody);
            if (catchCode) {
                const error3 = this.name("e");
                this._currNode = node.catch = new Catch(error3);
                catchCode(error3);
            }
            if (finallyCode) {
                this._currNode = node.finally = new Finally();
                this.code(finallyCode);
            }
            return this._endBlockNode(Catch, Finally);
        }
        throw(error) {
            return this._leafNode(new Throw(error));
        }
        block(body, nodeCount) {
            this._blockStarts.push(this._nodes.length);
            if (body) this.code(body).endBlock(nodeCount);
            return this;
        }
        endBlock(nodeCount) {
            const len = this._blockStarts.pop();
            if (len === void 0) throw new Error("CodeGen: not in self-balancing block");
            const toClose = this._nodes.length - len;
            if (toClose < 0 || nodeCount !== void 0 && toClose !== nodeCount) {
                throw new Error(`CodeGen: wrong number of nodes: ${toClose} vs ${nodeCount} expected`);
            }
            this._nodes.length = len;
            return this;
        }
        func(name, args = code2.nil, async, funcBody) {
            this._blockNode(new Func(name, args, async));
            if (funcBody) this.code(funcBody).endFunc();
            return this;
        }
        endFunc() {
            return this._endBlockNode(Func);
        }
        optimize(n = 1) {
            while(n-- > 0){
                this._root.optimizeNodes();
                this._root.optimizeNames(this._root.names, this._constants);
            }
        }
        _leafNode(node) {
            this._currNode.nodes.push(node);
            return this;
        }
        _blockNode(node) {
            this._currNode.nodes.push(node);
            this._nodes.push(node);
        }
        _endBlockNode(N1, N2) {
            const n = this._currNode;
            if (n instanceof N1 || N2 && n instanceof N2) {
                this._nodes.pop();
                return this;
            }
            throw new Error(`CodeGen: not in block "${N2 ? `${N1.kind}/${N2.kind}` : N1.kind}"`);
        }
        _elseNode(node) {
            const n = this._currNode;
            if (!(n instanceof If)) {
                throw new Error('CodeGen: "else" without "if"');
            }
            this._currNode = n.else = node;
            return this;
        }
        get _root() {
            return this._nodes[0];
        }
        get _currNode() {
            const ns = this._nodes;
            return ns[ns.length - 1];
        }
        set _currNode(node) {
            const ns = this._nodes;
            ns[ns.length - 1] = node;
        }
    }
    exports.CodeGen = CodeGen2;
    function addNames(names, from2) {
        for(const n in from2)names[n] = (names[n] || 0) + (from2[n] || 0);
        return names;
    }
    function addExprNames(names, from2) {
        return from2 instanceof code2._CodeOrName ? addNames(names, from2.names) : names;
    }
    function optimizeExpr(expr, names, constants) {
        if (expr instanceof code2.Name) return replaceName(expr);
        if (!canOptimize(expr)) return expr;
        return new code2._Code(expr._items.reduce((items2, c)=>{
            if (c instanceof code2.Name) c = replaceName(c);
            if (c instanceof code2._Code) items2.push(...c._items);
            else items2.push(c);
            return items2;
        }, []));
        function replaceName(n) {
            const c = constants[n.str];
            if (c === void 0 || names[n.str] !== 1) return n;
            delete names[n.str];
            return c;
        }
        function canOptimize(e) {
            return e instanceof code2._Code && e._items.some((c)=>c instanceof code2.Name && names[c.str] === 1 && constants[c.str] !== void 0
            );
        }
    }
    function subtractNames(names, from2) {
        for(const n in from2)names[n] = (names[n] || 0) - (from2[n] || 0);
    }
    function not2(x) {
        return typeof x == "boolean" || typeof x == "number" || x === null ? !x : code2._`!${par(x)}`;
    }
    exports.not = not2;
    const andCode = mappend(exports.operators.AND);
    function and(...args5) {
        return args5.reduce(andCode);
    }
    exports.and = and;
    const orCode = mappend(exports.operators.OR);
    function or(...args5) {
        return args5.reduce(orCode);
    }
    exports.or = or;
    function mappend(op1) {
        return (x, y)=>x === code2.nil ? y : y === code2.nil ? x : code2._`${par(x)} ${op1} ${par(y)}`
        ;
    }
    function par(x) {
        return x instanceof code2.Name ? x : code2._`(${x})`;
    }
});
var util = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.checkStrictMode = exports.getErrorPath = exports.Type = exports.useFunc = exports.setEvaluated = exports.evaluatedPropsToName = exports.mergeEvaluated = exports.eachItem = exports.unescapeJsonPointer = exports.escapeJsonPointer = exports.escapeFragment = exports.unescapeFragment = exports.schemaRefOrVal = exports.schemaHasRulesButRef = exports.schemaHasRules = exports.checkUnknownRules = exports.alwaysValidSchema = exports.toHash = void 0;
    function toHash(arr) {
        const hash = {
        };
        for (const item of arr)hash[item] = true;
        return hash;
    }
    exports.toHash = toHash;
    function alwaysValidSchema(it, schema3) {
        if (typeof schema3 == "boolean") return schema3;
        if (Object.keys(schema3).length === 0) return true;
        checkUnknownRules(it, schema3);
        return !schemaHasRules(schema3, it.self.RULES.all);
    }
    exports.alwaysValidSchema = alwaysValidSchema;
    function checkUnknownRules(it, schema3 = it.schema) {
        const { opts , self  } = it;
        if (!opts.strictSchema) return;
        if (typeof schema3 === "boolean") return;
        const rules2 = self.RULES.keywords;
        for(const key in schema3){
            if (!rules2[key]) checkStrictMode(it, `unknown keyword: "${key}"`);
        }
    }
    exports.checkUnknownRules = checkUnknownRules;
    function schemaHasRules(schema3, rules2) {
        if (typeof schema3 == "boolean") return !schema3;
        for(const key in schema3)if (rules2[key]) return true;
        return false;
    }
    exports.schemaHasRules = schemaHasRules;
    function schemaHasRulesButRef(schema3, RULES) {
        if (typeof schema3 == "boolean") return !schema3;
        for(const key in schema3)if (key !== "$ref" && RULES.all[key]) return true;
        return false;
    }
    exports.schemaHasRulesButRef = schemaHasRulesButRef;
    function schemaRefOrVal({ topSchemaRef , schemaPath  }, schema3, keyword2, $data) {
        if (!$data) {
            if (typeof schema3 == "number" || typeof schema3 == "boolean") return schema3;
            if (typeof schema3 == "string") return codegen._`${schema3}`;
        }
        return codegen._`${topSchemaRef}${schemaPath}${(0, codegen.getProperty)(keyword2)}`;
    }
    exports.schemaRefOrVal = schemaRefOrVal;
    function unescapeFragment(str2) {
        return unescapeJsonPointer(decodeURIComponent(str2));
    }
    exports.unescapeFragment = unescapeFragment;
    function escapeFragment(str2) {
        return encodeURIComponent(escapeJsonPointer(str2));
    }
    exports.escapeFragment = escapeFragment;
    function escapeJsonPointer(str2) {
        if (typeof str2 == "number") return `${str2}`;
        return str2.replace(/~/g, "~0").replace(/\//g, "~1");
    }
    exports.escapeJsonPointer = escapeJsonPointer;
    function unescapeJsonPointer(str2) {
        return str2.replace(/~1/g, "/").replace(/~0/g, "~");
    }
    exports.unescapeJsonPointer = unescapeJsonPointer;
    function eachItem(xs, f) {
        if (Array.isArray(xs)) {
            for (const x of xs)f(x);
        } else {
            f(xs);
        }
    }
    exports.eachItem = eachItem;
    function makeMergeEvaluated({ mergeNames , mergeToName , mergeValues , resultToName  }) {
        return (gen, from, to, toName)=>{
            const res = to === void 0 ? from : to instanceof codegen.Name ? (from instanceof codegen.Name ? mergeNames(gen, from, to) : mergeToName(gen, from, to), to) : from instanceof codegen.Name ? (mergeToName(gen, to, from), from) : mergeValues(from, to);
            return toName === codegen.Name && !(res instanceof codegen.Name) ? resultToName(gen, res) : res;
        };
    }
    exports.mergeEvaluated = {
        props: makeMergeEvaluated({
            mergeNames: (gen, from, to)=>gen.if(codegen._`${to} !== true && ${from} !== undefined`, ()=>{
                    gen.if(codegen._`${from} === true`, ()=>gen.assign(to, true)
                    , ()=>gen.assign(to, codegen._`${to} || {}`).code(codegen._`Object.assign(${to}, ${from})`)
                    );
                })
            ,
            mergeToName: (gen, from, to)=>gen.if(codegen._`${to} !== true`, ()=>{
                    if (from === true) {
                        gen.assign(to, true);
                    } else {
                        gen.assign(to, codegen._`${to} || {}`);
                        setEvaluated(gen, to, from);
                    }
                })
            ,
            mergeValues: (from, to)=>from === true ? true : {
                    ...from,
                    ...to
                }
            ,
            resultToName: evaluatedPropsToName
        }),
        items: makeMergeEvaluated({
            mergeNames: (gen, from, to)=>gen.if(codegen._`${to} !== true && ${from} !== undefined`, ()=>gen.assign(to, codegen._`${from} === true ? true : ${to} > ${from} ? ${to} : ${from}`)
                )
            ,
            mergeToName: (gen, from, to)=>gen.if(codegen._`${to} !== true`, ()=>gen.assign(to, from === true ? true : codegen._`${to} > ${from} ? ${to} : ${from}`)
                )
            ,
            mergeValues: (from, to)=>from === true ? true : Math.max(from, to)
            ,
            resultToName: (gen, items2)=>gen.var("items", items2)
        })
    };
    function evaluatedPropsToName(gen, ps) {
        if (ps === true) return gen.var("props", true);
        const props = gen.var("props", codegen._`{}`);
        if (ps !== void 0) setEvaluated(gen, props, ps);
        return props;
    }
    exports.evaluatedPropsToName = evaluatedPropsToName;
    function setEvaluated(gen, props, ps) {
        Object.keys(ps).forEach((p)=>gen.assign(codegen._`${props}${(0, codegen.getProperty)(p)}`, true)
        );
    }
    exports.setEvaluated = setEvaluated;
    const snippets = {
    };
    function useFunc(gen, f) {
        return gen.scopeValue("func", {
            ref: f,
            code: snippets[f.code] || (snippets[f.code] = new code2._Code(f.code))
        });
    }
    exports.useFunc = useFunc;
    var Type2;
    (function(Type21) {
        Type21[Type21["Num"] = 0] = "Num";
        Type21[Type21["Str"] = 1] = "Str";
    })(Type2 = exports.Type || (exports.Type = {
    }));
    function getErrorPath(dataProp, dataPropType, jsPropertySyntax) {
        if (dataProp instanceof codegen.Name) {
            const isNumber = dataPropType === Type2.Num;
            return jsPropertySyntax ? isNumber ? codegen._`"[" + ${dataProp} + "]"` : codegen._`"['" + ${dataProp} + "']"` : isNumber ? codegen._`"/" + ${dataProp}` : codegen._`"/" + ${dataProp}.replace(/~/g, "~0").replace(/\\//g, "~1")`;
        }
        return jsPropertySyntax ? (0, codegen.getProperty)(dataProp).toString() : "/" + escapeJsonPointer(dataProp);
    }
    exports.getErrorPath = getErrorPath;
    function checkStrictMode(it, msg, mode = it.opts.strictSchema) {
        if (!mode) return;
        msg = `strict mode: ${msg}`;
        if (mode === true) throw new Error(msg);
        it.self.logger.warn(msg);
    }
    exports.checkStrictMode = checkStrictMode;
});
var names_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const names = {
        data: new codegen.Name("data"),
        valCxt: new codegen.Name("valCxt"),
        instancePath: new codegen.Name("instancePath"),
        parentData: new codegen.Name("parentData"),
        parentDataProperty: new codegen.Name("parentDataProperty"),
        rootData: new codegen.Name("rootData"),
        dynamicAnchors: new codegen.Name("dynamicAnchors"),
        vErrors: new codegen.Name("vErrors"),
        errors: new codegen.Name("errors"),
        this: new codegen.Name("this"),
        self: new codegen.Name("self"),
        scope: new codegen.Name("scope"),
        json: new codegen.Name("json"),
        jsonPos: new codegen.Name("jsonPos"),
        jsonLen: new codegen.Name("jsonLen"),
        jsonPart: new codegen.Name("jsonPart")
    };
    exports.default = names;
});
var errors = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.extendErrors = exports.resetErrorsCount = exports.reportExtraError = exports.reportError = exports.keyword$DataError = exports.keywordError = void 0;
    exports.keywordError = {
        message: ({ keyword: keyword2  })=>codegen.str`must pass "${keyword2}" keyword validation`
    };
    exports.keyword$DataError = {
        message: ({ keyword: keyword2 , schemaType: schemaType1  })=>schemaType1 ? codegen.str`"${keyword2}" keyword must be ${schemaType1} ($data)` : codegen.str`"${keyword2}" keyword is invalid ($data)`
    };
    function reportError(cxt, error1 = exports.keywordError, errorPaths, overrideAllErrors) {
        const { it  } = cxt;
        const { gen , compositeRule , allErrors  } = it;
        const errObj = errorObjectCode(cxt, error1, errorPaths);
        if (overrideAllErrors !== null && overrideAllErrors !== void 0 ? overrideAllErrors : compositeRule || allErrors) {
            addError(gen, errObj);
        } else {
            returnErrors(it, codegen._`[${errObj}]`);
        }
    }
    exports.reportError = reportError;
    function reportExtraError(cxt, error1 = exports.keywordError, errorPaths) {
        const { it  } = cxt;
        const { gen , compositeRule , allErrors  } = it;
        const errObj = errorObjectCode(cxt, error1, errorPaths);
        addError(gen, errObj);
        if (!(compositeRule || allErrors)) {
            returnErrors(it, names_1.default.vErrors);
        }
    }
    exports.reportExtraError = reportExtraError;
    function resetErrorsCount(gen, errsCount) {
        gen.assign(names_1.default.errors, errsCount);
        gen.if(codegen._`${names_1.default.vErrors} !== null`, ()=>gen.if(errsCount, ()=>gen.assign(codegen._`${names_1.default.vErrors}.length`, errsCount)
            , ()=>gen.assign(names_1.default.vErrors, null)
            )
        );
    }
    exports.resetErrorsCount = resetErrorsCount;
    function extendErrors({ gen , keyword: keyword2 , schemaValue , data , errsCount , it  }) {
        if (errsCount === void 0) throw new Error("ajv implementation error");
        const err = gen.name("err");
        gen.forRange("i", errsCount, names_1.default.errors, (i5)=>{
            gen.const(err, codegen._`${names_1.default.vErrors}[${i5}]`);
            gen.if(codegen._`${err}.instancePath === undefined`, ()=>gen.assign(codegen._`${err}.instancePath`, (0, codegen.strConcat)(names_1.default.instancePath, it.errorPath))
            );
            gen.assign(codegen._`${err}.schemaPath`, codegen.str`${it.errSchemaPath}/${keyword2}`);
            if (it.opts.verbose) {
                gen.assign(codegen._`${err}.schema`, schemaValue);
                gen.assign(codegen._`${err}.data`, data);
            }
        });
    }
    exports.extendErrors = extendErrors;
    function addError(gen, errObj) {
        const err = gen.const("err", errObj);
        gen.if(codegen._`${names_1.default.vErrors} === null`, ()=>gen.assign(names_1.default.vErrors, codegen._`[${err}]`)
        , codegen._`${names_1.default.vErrors}.push(${err})`);
        gen.code(codegen._`${names_1.default.errors}++`);
    }
    function returnErrors(it, errs) {
        const { gen , validateName , schemaEnv  } = it;
        if (schemaEnv.$async) {
            gen.throw(codegen._`new ${it.ValidationError}(${errs})`);
        } else {
            gen.assign(codegen._`${validateName}.errors`, errs);
            gen.return(false);
        }
    }
    const E = {
        keyword: new codegen.Name("keyword"),
        schemaPath: new codegen.Name("schemaPath"),
        params: new codegen.Name("params"),
        propertyName: new codegen.Name("propertyName"),
        message: new codegen.Name("message"),
        schema: new codegen.Name("schema"),
        parentSchema: new codegen.Name("parentSchema")
    };
    function errorObjectCode(cxt, error1, errorPaths) {
        const { createErrors  } = cxt.it;
        if (createErrors === false) return codegen._`{}`;
        return errorObject(cxt, error1, errorPaths);
    }
    function errorObject(cxt, error1, errorPaths = {
    }) {
        const { gen , it  } = cxt;
        const keyValues = [
            errorInstancePath(it, errorPaths),
            errorSchemaPath(cxt, errorPaths)
        ];
        extraErrorProps(cxt, error1, keyValues);
        return gen.object(...keyValues);
    }
    function errorInstancePath({ errorPath  }, { instancePath  }) {
        const instPath = instancePath ? codegen.str`${errorPath}${(0, util.getErrorPath)(instancePath, util.Type.Str)}` : errorPath;
        return [
            names_1.default.instancePath,
            (0, codegen.strConcat)(names_1.default.instancePath, instPath)
        ];
    }
    function errorSchemaPath({ keyword: keyword2 , it: { errSchemaPath  }  }, { schemaPath , parentSchema  }) {
        let schPath = parentSchema ? errSchemaPath : codegen.str`${errSchemaPath}/${keyword2}`;
        if (schemaPath) {
            schPath = codegen.str`${schPath}${(0, util.getErrorPath)(schemaPath, util.Type.Str)}`;
        }
        return [
            E.schemaPath,
            schPath
        ];
    }
    function extraErrorProps(cxt, { params , message: message7  }, keyValues) {
        const { keyword: keyword2 , data , schemaValue , it  } = cxt;
        const { opts , propertyName , topSchemaRef , schemaPath  } = it;
        keyValues.push([
            E.keyword,
            keyword2
        ], [
            E.params,
            typeof params == "function" ? params(cxt) : params || codegen._`{}`
        ]);
        if (opts.messages) {
            keyValues.push([
                E.message,
                typeof message7 == "function" ? message7(cxt) : message7
            ]);
        }
        if (opts.verbose) {
            keyValues.push([
                E.schema,
                schemaValue
            ], [
                E.parentSchema,
                codegen._`${topSchemaRef}${schemaPath}`
            ], [
                names_1.default.data,
                data
            ]);
        }
        if (propertyName) keyValues.push([
            E.propertyName,
            propertyName
        ]);
    }
});
var boolSchema = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.boolOrEmptySchema = exports.topBoolOrEmptySchema = void 0;
    const boolError = {
        message: "boolean schema is false"
    };
    function topBoolOrEmptySchema(it) {
        const { gen , schema: schema3 , validateName  } = it;
        if (schema3 === false) {
            falseSchemaError(it, false);
        } else if (typeof schema3 == "object" && schema3.$async === true) {
            gen.return(names_1.default.data);
        } else {
            gen.assign(codegen._`${validateName}.errors`, null);
            gen.return(true);
        }
    }
    exports.topBoolOrEmptySchema = topBoolOrEmptySchema;
    function boolOrEmptySchema(it, valid) {
        const { gen , schema: schema3  } = it;
        if (schema3 === false) {
            gen.var(valid, false);
            falseSchemaError(it);
        } else {
            gen.var(valid, true);
        }
    }
    exports.boolOrEmptySchema = boolOrEmptySchema;
    function falseSchemaError(it, overrideAllErrors) {
        const { gen , data  } = it;
        const cxt = {
            gen,
            keyword: "false schema",
            data,
            schema: false,
            schemaCode: false,
            schemaValue: false,
            params: {
            },
            it
        };
        (0, errors.reportError)(cxt, boolError, void 0, overrideAllErrors);
    }
});
var rules = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getRules = exports.isJSONType = void 0;
    const _jsonTypes = [
        "string",
        "number",
        "integer",
        "boolean",
        "null",
        "object",
        "array"
    ];
    const jsonTypes = new Set(_jsonTypes);
    function isJSONType(x) {
        return typeof x == "string" && jsonTypes.has(x);
    }
    exports.isJSONType = isJSONType;
    function getRules() {
        const groups = {
            number: {
                type: "number",
                rules: []
            },
            string: {
                type: "string",
                rules: []
            },
            array: {
                type: "array",
                rules: []
            },
            object: {
                type: "object",
                rules: []
            }
        };
        return {
            types: {
                ...groups,
                integer: true,
                boolean: true,
                null: true
            },
            rules: [
                {
                    rules: []
                },
                groups.number,
                groups.string,
                groups.array,
                groups.object
            ],
            post: {
                rules: []
            },
            all: {
            },
            keywords: {
            }
        };
    }
    exports.getRules = getRules;
});
var applicability = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.shouldUseRule = exports.shouldUseGroup = exports.schemaHasRulesForType = void 0;
    function schemaHasRulesForType({ schema: schema3 , self  }, type21) {
        const group = self.RULES.types[type21];
        return group && group !== true && shouldUseGroup(schema3, group);
    }
    exports.schemaHasRulesForType = schemaHasRulesForType;
    function shouldUseGroup(schema3, group) {
        return group.rules.some((rule)=>shouldUseRule(schema3, rule)
        );
    }
    exports.shouldUseGroup = shouldUseGroup;
    function shouldUseRule(schema3, rule) {
        var _a;
        return schema3[rule.keyword] !== void 0 || ((_a = rule.definition.implements) === null || _a === void 0 ? void 0 : _a.some((kwd)=>schema3[kwd] !== void 0
        ));
    }
    exports.shouldUseRule = shouldUseRule;
});
var dataType = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.reportTypeError = exports.checkDataTypes = exports.checkDataType = exports.coerceAndCheckDataType = exports.getJSONTypes = exports.getSchemaTypes = exports.DataType = void 0;
    var DataType;
    (function(DataType2) {
        DataType2[DataType2["Correct"] = 0] = "Correct";
        DataType2[DataType2["Wrong"] = 1] = "Wrong";
    })(DataType = exports.DataType || (exports.DataType = {
    }));
    function getSchemaTypes(schema3) {
        const types2 = getJSONTypes(schema3.type);
        const hasNull = types2.includes("null");
        if (hasNull) {
            if (schema3.nullable === false) throw new Error("type: null contradicts nullable: false");
        } else {
            if (!types2.length && schema3.nullable !== void 0) {
                throw new Error('"nullable" cannot be used without "type"');
            }
            if (schema3.nullable === true) types2.push("null");
        }
        return types2;
    }
    exports.getSchemaTypes = getSchemaTypes;
    function getJSONTypes(ts) {
        const types2 = Array.isArray(ts) ? ts : ts ? [
            ts
        ] : [];
        if (types2.every(rules.isJSONType)) return types2;
        throw new Error("type must be JSONType or JSONType[]: " + types2.join(","));
    }
    exports.getJSONTypes = getJSONTypes;
    function coerceAndCheckDataType(it, types2) {
        const { gen , data , opts  } = it;
        const coerceTo = coerceToTypes(types2, opts.coerceTypes);
        const checkTypes = types2.length > 0 && !(coerceTo.length === 0 && types2.length === 1 && (0, applicability.schemaHasRulesForType)(it, types2[0]));
        if (checkTypes) {
            const wrongType = checkDataTypes(types2, data, opts.strictNumbers, DataType.Wrong);
            gen.if(wrongType, ()=>{
                if (coerceTo.length) coerceData(it, types2, coerceTo);
                else reportTypeError(it);
            });
        }
        return checkTypes;
    }
    exports.coerceAndCheckDataType = coerceAndCheckDataType;
    const COERCIBLE = new Set([
        "string",
        "number",
        "integer",
        "boolean",
        "null"
    ]);
    function coerceToTypes(types2, coerceTypes) {
        return coerceTypes ? types2.filter((t)=>COERCIBLE.has(t) || coerceTypes === "array" && t === "array"
        ) : [];
    }
    function coerceData(it, types2, coerceTo) {
        const { gen , data , opts  } = it;
        const dataType2 = gen.let("dataType", codegen._`typeof ${data}`);
        const coerced = gen.let("coerced", codegen._`undefined`);
        if (opts.coerceTypes === "array") {
            gen.if(codegen._`${dataType2} == 'object' && Array.isArray(${data}) && ${data}.length == 1`, ()=>gen.assign(data, codegen._`${data}[0]`).assign(dataType2, codegen._`typeof ${data}`).if(checkDataTypes(types2, data, opts.strictNumbers), ()=>gen.assign(coerced, data)
                )
            );
        }
        gen.if(codegen._`${coerced} !== undefined`);
        for (const t of coerceTo){
            if (COERCIBLE.has(t) || t === "array" && opts.coerceTypes === "array") {
                coerceSpecificType(t);
            }
        }
        gen.else();
        reportTypeError(it);
        gen.endIf();
        gen.if(codegen._`${coerced} !== undefined`, ()=>{
            gen.assign(data, coerced);
            assignParentData(it, coerced);
        });
        function coerceSpecificType(t1) {
            switch(t1){
                case "string":
                    gen.elseIf(codegen._`${dataType2} == "number" || ${dataType2} == "boolean"`).assign(coerced, codegen._`"" + ${data}`).elseIf(codegen._`${data} === null`).assign(coerced, codegen._`""`);
                    return;
                case "number":
                    gen.elseIf(codegen._`${dataType2} == "boolean" || ${data} === null
              || (${dataType2} == "string" && ${data} && ${data} == +${data})`).assign(coerced, codegen._`+${data}`);
                    return;
                case "integer":
                    gen.elseIf(codegen._`${dataType2} === "boolean" || ${data} === null
              || (${dataType2} === "string" && ${data} && ${data} == +${data} && !(${data} % 1))`).assign(coerced, codegen._`+${data}`);
                    return;
                case "boolean":
                    gen.elseIf(codegen._`${data} === "false" || ${data} === 0 || ${data} === null`).assign(coerced, false).elseIf(codegen._`${data} === "true" || ${data} === 1`).assign(coerced, true);
                    return;
                case "null":
                    gen.elseIf(codegen._`${data} === "" || ${data} === 0 || ${data} === false`);
                    gen.assign(coerced, null);
                    return;
                case "array":
                    gen.elseIf(codegen._`${dataType2} === "string" || ${dataType2} === "number"
              || ${dataType2} === "boolean" || ${data} === null`).assign(coerced, codegen._`[${data}]`);
            }
        }
    }
    function assignParentData({ gen , parentData , parentDataProperty  }, expr) {
        gen.if(codegen._`${parentData} !== undefined`, ()=>gen.assign(codegen._`${parentData}[${parentDataProperty}]`, expr)
        );
    }
    function checkDataType(dataType2, data, strictNums, correct = DataType.Correct) {
        const EQ = correct === DataType.Correct ? codegen.operators.EQ : codegen.operators.NEQ;
        let cond;
        switch(dataType2){
            case "null":
                return codegen._`${data} ${EQ} null`;
            case "array":
                cond = codegen._`Array.isArray(${data})`;
                break;
            case "object":
                cond = codegen._`${data} && typeof ${data} == "object" && !Array.isArray(${data})`;
                break;
            case "integer":
                cond = numCond(codegen._`!(${data} % 1) && !isNaN(${data})`);
                break;
            case "number":
                cond = numCond();
                break;
            default:
                return codegen._`typeof ${data} ${EQ} ${dataType2}`;
        }
        return correct === DataType.Correct ? cond : (0, codegen.not)(cond);
        function numCond(_cond = codegen.nil) {
            return (0, codegen.and)(codegen._`typeof ${data} == "number"`, _cond, strictNums ? codegen._`isFinite(${data})` : codegen.nil);
        }
    }
    exports.checkDataType = checkDataType;
    function checkDataTypes(dataTypes, data, strictNums, correct) {
        if (dataTypes.length === 1) {
            return checkDataType(dataTypes[0], data, strictNums, correct);
        }
        let cond;
        const types2 = (0, util.toHash)(dataTypes);
        if (types2.array && types2.object) {
            const notObj = codegen._`typeof ${data} != "object"`;
            cond = types2.null ? notObj : codegen._`!${data} || ${notObj}`;
            delete types2.null;
            delete types2.array;
            delete types2.object;
        } else {
            cond = codegen.nil;
        }
        if (types2.number) delete types2.integer;
        for(const t in types2)cond = (0, codegen.and)(cond, checkDataType(t, data, strictNums, correct));
        return cond;
    }
    exports.checkDataTypes = checkDataTypes;
    const typeError = {
        message: ({ schema: schema3  })=>`must be ${schema3}`
        ,
        params: ({ schema: schema3 , schemaValue  })=>typeof schema3 == "string" ? codegen._`{type: ${schema3}}` : codegen._`{type: ${schemaValue}}`
    };
    function reportTypeError(it) {
        const cxt = getTypeErrorContext(it);
        (0, errors.reportError)(cxt, typeError);
    }
    exports.reportTypeError = reportTypeError;
    function getTypeErrorContext(it) {
        const { gen , data , schema: schema3  } = it;
        const schemaCode = (0, util.schemaRefOrVal)(it, schema3, "type");
        return {
            gen,
            keyword: "type",
            data,
            schema: schema3.type,
            schemaCode,
            schemaValue: schemaCode,
            parentSchema: schema3,
            params: {
            },
            it
        };
    }
});
var defaults = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.assignDefaults = void 0;
    function assignDefaults(it, ty) {
        const { properties: properties2 , items: items2  } = it.schema;
        if (ty === "object" && properties2) {
            for(const key in properties2){
                assignDefault(it, key, properties2[key].default);
            }
        } else if (ty === "array" && Array.isArray(items2)) {
            items2.forEach((sch, i5)=>assignDefault(it, i5, sch.default)
            );
        }
    }
    exports.assignDefaults = assignDefaults;
    function assignDefault(it, prop, defaultValue) {
        const { gen , compositeRule , data , opts  } = it;
        if (defaultValue === void 0) return;
        const childData = codegen._`${data}${(0, codegen.getProperty)(prop)}`;
        if (compositeRule) {
            (0, util.checkStrictMode)(it, `default is ignored for: ${childData}`);
            return;
        }
        let condition = codegen._`${childData} === undefined`;
        if (opts.useDefaults === "empty") {
            condition = codegen._`${condition} || ${childData} === null || ${childData} === ""`;
        }
        gen.if(condition, codegen._`${childData} = ${(0, codegen.stringify)(defaultValue)}`);
    }
});
var code$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateUnion = exports.validateArray = exports.usePattern = exports.callValidateCode = exports.schemaProperties = exports.allSchemaProperties = exports.noPropertyInData = exports.propertyInData = exports.isOwnProperty = exports.hasPropFunc = exports.reportMissingProp = exports.checkMissingProp = exports.checkReportMissingProp = void 0;
    const util_2 = util;
    function checkReportMissingProp(cxt, prop) {
        const { gen , data , it  } = cxt;
        gen.if(noPropertyInData(gen, data, prop, it.opts.ownProperties), ()=>{
            cxt.setParams({
                missingProperty: codegen._`${prop}`
            }, true);
            cxt.error();
        });
    }
    exports.checkReportMissingProp = checkReportMissingProp;
    function checkMissingProp({ gen , data , it: { opts  }  }, properties2, missing) {
        return (0, codegen.or)(...properties2.map((prop)=>(0, codegen.and)(noPropertyInData(gen, data, prop, opts.ownProperties), codegen._`${missing} = ${prop}`)
        ));
    }
    exports.checkMissingProp = checkMissingProp;
    function reportMissingProp(cxt, missing) {
        cxt.setParams({
            missingProperty: missing
        }, true);
        cxt.error();
    }
    exports.reportMissingProp = reportMissingProp;
    function hasPropFunc(gen) {
        return gen.scopeValue("func", {
            ref: Object.prototype.hasOwnProperty,
            code: codegen._`Object.prototype.hasOwnProperty`
        });
    }
    exports.hasPropFunc = hasPropFunc;
    function isOwnProperty(gen, data, property) {
        return codegen._`${hasPropFunc(gen)}.call(${data}, ${property})`;
    }
    exports.isOwnProperty = isOwnProperty;
    function propertyInData(gen, data, property, ownProperties) {
        const cond = codegen._`${data}${(0, codegen.getProperty)(property)} !== undefined`;
        return ownProperties ? codegen._`${cond} && ${isOwnProperty(gen, data, property)}` : cond;
    }
    exports.propertyInData = propertyInData;
    function noPropertyInData(gen, data, property, ownProperties) {
        const cond = codegen._`${data}${(0, codegen.getProperty)(property)} === undefined`;
        return ownProperties ? (0, codegen.or)(cond, (0, codegen.not)(isOwnProperty(gen, data, property))) : cond;
    }
    exports.noPropertyInData = noPropertyInData;
    function allSchemaProperties(schemaMap) {
        return schemaMap ? Object.keys(schemaMap).filter((p)=>p !== "__proto__"
        ) : [];
    }
    exports.allSchemaProperties = allSchemaProperties;
    function schemaProperties(it, schemaMap) {
        return allSchemaProperties(schemaMap).filter((p)=>!(0, util.alwaysValidSchema)(it, schemaMap[p])
        );
    }
    exports.schemaProperties = schemaProperties;
    function callValidateCode({ schemaCode , data , it: { gen , topSchemaRef , schemaPath , errorPath  } , it  }, func1, context, passSchema) {
        const dataAndSchema = passSchema ? codegen._`${schemaCode}, ${data}, ${topSchemaRef}${schemaPath}` : data;
        const valCxt = [
            [
                names_1.default.instancePath,
                (0, codegen.strConcat)(names_1.default.instancePath, errorPath)
            ],
            [
                names_1.default.parentData,
                it.parentData
            ],
            [
                names_1.default.parentDataProperty,
                it.parentDataProperty
            ],
            [
                names_1.default.rootData,
                names_1.default.rootData
            ]
        ];
        if (it.opts.dynamicRef) valCxt.push([
            names_1.default.dynamicAnchors,
            names_1.default.dynamicAnchors
        ]);
        const args4 = codegen._`${dataAndSchema}, ${gen.object(...valCxt)}`;
        return context !== codegen.nil ? codegen._`${func1}.call(${context}, ${args4})` : codegen._`${func1}(${args4})`;
    }
    exports.callValidateCode = callValidateCode;
    const newRegExp = codegen._`new RegExp`;
    function usePattern({ gen , it: { opts  }  }, pattern2) {
        const u = opts.unicodeRegExp ? "u" : "";
        const { regExp  } = opts.code;
        const rx = regExp(pattern2, u);
        return gen.scopeValue("pattern", {
            key: rx.toString(),
            ref: rx,
            code: codegen._`${regExp.code === "new RegExp" ? newRegExp : (0, util_2.useFunc)(gen, regExp)}(${pattern2}, ${u})`
        });
    }
    exports.usePattern = usePattern;
    function validateArray(cxt) {
        const { gen , data , keyword: keyword2 , it  } = cxt;
        const valid = gen.name("valid");
        if (it.allErrors) {
            const validArr = gen.let("valid", true);
            validateItems(()=>gen.assign(validArr, false)
            );
            return validArr;
        }
        gen.var(valid, true);
        validateItems(()=>gen.break()
        );
        return valid;
        function validateItems(notValid) {
            const len = gen.const("len", codegen._`${data}.length`);
            gen.forRange("i", 0, len, (i5)=>{
                cxt.subschema({
                    keyword: keyword2,
                    dataProp: i5,
                    dataPropType: util.Type.Num
                }, valid);
                gen.if((0, codegen.not)(valid), notValid);
            });
        }
    }
    exports.validateArray = validateArray;
    function validateUnion(cxt) {
        const { gen , schema: schema3 , keyword: keyword2 , it  } = cxt;
        if (!Array.isArray(schema3)) throw new Error("ajv implementation error");
        const alwaysValid = schema3.some((sch)=>(0, util.alwaysValidSchema)(it, sch)
        );
        if (alwaysValid && !it.opts.unevaluated) return;
        const valid = gen.let("valid", false);
        const schValid = gen.name("_valid");
        gen.block(()=>schema3.forEach((_sch, i5)=>{
                const schCxt = cxt.subschema({
                    keyword: keyword2,
                    schemaProp: i5,
                    compositeRule: true
                }, schValid);
                gen.assign(valid, codegen._`${valid} || ${schValid}`);
                const merged = cxt.mergeValidEvaluated(schCxt, schValid);
                if (!merged) gen.if((0, codegen.not)(valid));
            })
        );
        cxt.result(valid, ()=>cxt.reset()
        , ()=>cxt.error(true)
        );
    }
    exports.validateUnion = validateUnion;
});
var keyword1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateKeywordUsage = exports.validSchemaType = exports.funcKeywordCode = exports.macroKeywordCode = void 0;
    function macroKeywordCode(cxt, def1) {
        const { gen , keyword: keyword2 , schema: schema3 , parentSchema , it  } = cxt;
        const macroSchema = def1.macro.call(it.self, schema3, parentSchema, it);
        const schemaRef = useKeyword(gen, keyword2, macroSchema);
        if (it.opts.validateSchema !== false) it.self.validateSchema(macroSchema, true);
        const valid = gen.name("valid");
        cxt.subschema({
            schema: macroSchema,
            schemaPath: codegen.nil,
            errSchemaPath: `${it.errSchemaPath}/${keyword2}`,
            topSchemaRef: schemaRef,
            compositeRule: true
        }, valid);
        cxt.pass(valid, ()=>cxt.error(true)
        );
    }
    exports.macroKeywordCode = macroKeywordCode;
    function funcKeywordCode(cxt, def1) {
        var _a;
        const { gen , keyword: keyword2 , schema: schema3 , parentSchema , $data , it  } = cxt;
        checkAsyncKeyword(it, def1);
        const validate2 = !$data && def1.compile ? def1.compile.call(it.self, schema3, parentSchema, it) : def1.validate;
        const validateRef = useKeyword(gen, keyword2, validate2);
        const valid = gen.let("valid");
        cxt.block$data(valid, validateKeyword);
        cxt.ok((_a = def1.valid) !== null && _a !== void 0 ? _a : valid);
        function validateKeyword() {
            if (def1.errors === false) {
                assignValid();
                if (def1.modifying) modifyData(cxt);
                reportErrs(()=>cxt.error()
                );
            } else {
                const ruleErrs = def1.async ? validateAsync() : validateSync();
                if (def1.modifying) modifyData(cxt);
                reportErrs(()=>addErrs(cxt, ruleErrs)
                );
            }
        }
        function validateAsync() {
            const ruleErrs = gen.let("ruleErrs", null);
            gen.try(()=>assignValid(codegen._`await `)
            , (e)=>gen.assign(valid, false).if(codegen._`${e} instanceof ${it.ValidationError}`, ()=>gen.assign(ruleErrs, codegen._`${e}.errors`)
                , ()=>gen.throw(e)
                )
            );
            return ruleErrs;
        }
        function validateSync() {
            const validateErrs = codegen._`${validateRef}.errors`;
            gen.assign(validateErrs, null);
            assignValid(codegen.nil);
            return validateErrs;
        }
        function assignValid(_await = def1.async ? codegen._`await ` : codegen.nil) {
            const passCxt = it.opts.passContext ? names_1.default.this : names_1.default.self;
            const passSchema = !("compile" in def1 && !$data || def1.schema === false);
            gen.assign(valid, codegen._`${_await}${(0, code$1.callValidateCode)(cxt, validateRef, passCxt, passSchema)}`, def1.modifying);
        }
        function reportErrs(errors2) {
            var _a2;
            gen.if((0, codegen.not)((_a2 = def1.valid) !== null && _a2 !== void 0 ? _a2 : valid), errors2);
        }
    }
    exports.funcKeywordCode = funcKeywordCode;
    function modifyData(cxt) {
        const { gen , data , it  } = cxt;
        gen.if(it.parentData, ()=>gen.assign(data, codegen._`${it.parentData}[${it.parentDataProperty}]`)
        );
    }
    function addErrs(cxt, errs) {
        const { gen  } = cxt;
        gen.if(codegen._`Array.isArray(${errs})`, ()=>{
            gen.assign(names_1.default.vErrors, codegen._`${names_1.default.vErrors} === null ? ${errs} : ${names_1.default.vErrors}.concat(${errs})`).assign(names_1.default.errors, codegen._`${names_1.default.vErrors}.length`);
            (0, errors.extendErrors)(cxt);
        }, ()=>cxt.error()
        );
    }
    function checkAsyncKeyword({ schemaEnv  }, def1) {
        if (def1.async && !schemaEnv.$async) throw new Error("async keyword in sync schema");
    }
    function useKeyword(gen, keyword2, result) {
        if (result === void 0) throw new Error(`keyword "${keyword2}" failed to compile`);
        return gen.scopeValue("keyword", typeof result == "function" ? {
            ref: result
        } : {
            ref: result,
            code: (0, codegen.stringify)(result)
        });
    }
    function validSchemaType(schema3, schemaType1, allowUndefined = false) {
        return !schemaType1.length || schemaType1.some((st)=>st === "array" ? Array.isArray(schema3) : st === "object" ? schema3 && typeof schema3 == "object" && !Array.isArray(schema3) : typeof schema3 == st || allowUndefined && typeof schema3 == "undefined"
        );
    }
    exports.validSchemaType = validSchemaType;
    function validateKeywordUsage({ schema: schema3 , opts , self , errSchemaPath  }, def1, keyword2) {
        if (Array.isArray(def1.keyword) ? !def1.keyword.includes(keyword2) : def1.keyword !== keyword2) {
            throw new Error("ajv implementation error");
        }
        const deps = def1.dependencies;
        if (deps === null || deps === void 0 ? void 0 : deps.some((kwd)=>!Object.prototype.hasOwnProperty.call(schema3, kwd)
        )) {
            throw new Error(`parent schema must have dependencies of ${keyword2}: ${deps.join(",")}`);
        }
        if (def1.validateSchema) {
            const valid = def1.validateSchema(schema3[keyword2]);
            if (!valid) {
                const msg = `keyword "${keyword2}" value is invalid at path "${errSchemaPath}": ` + self.errorsText(def1.validateSchema.errors);
                if (opts.validateSchema === "log") self.logger.error(msg);
                else throw new Error(msg);
            }
        }
    }
    exports.validateKeywordUsage = validateKeywordUsage;
});
var subschema = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.extendSubschemaMode = exports.extendSubschemaData = exports.getSubschema = void 0;
    function getSubschema(it, { keyword: keyword2 , schemaProp , schema: schema3 , schemaPath , errSchemaPath , topSchemaRef  }) {
        if (keyword2 !== void 0 && schema3 !== void 0) {
            throw new Error('both "keyword" and "schema" passed, only one allowed');
        }
        if (keyword2 !== void 0) {
            const sch = it.schema[keyword2];
            return schemaProp === void 0 ? {
                schema: sch,
                schemaPath: codegen._`${it.schemaPath}${(0, codegen.getProperty)(keyword2)}`,
                errSchemaPath: `${it.errSchemaPath}/${keyword2}`
            } : {
                schema: sch[schemaProp],
                schemaPath: codegen._`${it.schemaPath}${(0, codegen.getProperty)(keyword2)}${(0, codegen.getProperty)(schemaProp)}`,
                errSchemaPath: `${it.errSchemaPath}/${keyword2}/${(0, util.escapeFragment)(schemaProp)}`
            };
        }
        if (schema3 !== void 0) {
            if (schemaPath === void 0 || errSchemaPath === void 0 || topSchemaRef === void 0) {
                throw new Error('"schemaPath", "errSchemaPath" and "topSchemaRef" are required with "schema"');
            }
            return {
                schema: schema3,
                schemaPath,
                topSchemaRef,
                errSchemaPath
            };
        }
        throw new Error('either "keyword" or "schema" must be passed');
    }
    exports.getSubschema = getSubschema;
    function extendSubschemaData(subschema2, it, { dataProp , dataPropType: dpType , data , dataTypes , propertyName  }) {
        if (data !== void 0 && dataProp !== void 0) {
            throw new Error('both "data" and "dataProp" passed, only one allowed');
        }
        const { gen  } = it;
        if (dataProp !== void 0) {
            const { errorPath , dataPathArr , opts  } = it;
            const nextData = gen.let("data", codegen._`${it.data}${(0, codegen.getProperty)(dataProp)}`, true);
            dataContextProps(nextData);
            subschema2.errorPath = codegen.str`${errorPath}${(0, util.getErrorPath)(dataProp, dpType, opts.jsPropertySyntax)}`;
            subschema2.parentDataProperty = codegen._`${dataProp}`;
            subschema2.dataPathArr = [
                ...dataPathArr,
                subschema2.parentDataProperty
            ];
        }
        if (data !== void 0) {
            const nextData = data instanceof codegen.Name ? data : gen.let("data", data, true);
            dataContextProps(nextData);
            if (propertyName !== void 0) subschema2.propertyName = propertyName;
        }
        if (dataTypes) subschema2.dataTypes = dataTypes;
        function dataContextProps(_nextData) {
            subschema2.data = _nextData;
            subschema2.dataLevel = it.dataLevel + 1;
            subschema2.dataTypes = [];
            it.definedProperties = new Set();
            subschema2.parentData = it.data;
            subschema2.dataNames = [
                ...it.dataNames,
                _nextData
            ];
        }
    }
    exports.extendSubschemaData = extendSubschemaData;
    function extendSubschemaMode(subschema2, { jtdDiscriminator , jtdMetadata , compositeRule , createErrors , allErrors  }) {
        if (compositeRule !== void 0) subschema2.compositeRule = compositeRule;
        if (createErrors !== void 0) subschema2.createErrors = createErrors;
        if (allErrors !== void 0) subschema2.allErrors = allErrors;
        subschema2.jtdDiscriminator = jtdDiscriminator;
        subschema2.jtdMetadata = jtdMetadata;
    }
    exports.extendSubschemaMode = extendSubschemaMode;
});
var resolve4 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getSchemaRefs = exports.resolveUrl = exports.normalizeId = exports._getFullPath = exports.getFullPath = exports.inlineRef = void 0;
    const SIMPLE_INLINED = new Set([
        "type",
        "format",
        "pattern",
        "maxLength",
        "minLength",
        "maxProperties",
        "minProperties",
        "maxItems",
        "minItems",
        "maximum",
        "minimum",
        "uniqueItems",
        "multipleOf",
        "required",
        "enum",
        "const"
    ]);
    function inlineRef(schema3, limit = true) {
        if (typeof schema3 == "boolean") return true;
        if (limit === true) return !hasRef(schema3);
        if (!limit) return false;
        return countKeys(schema3) <= limit;
    }
    exports.inlineRef = inlineRef;
    const REF_KEYWORDS = new Set([
        "$ref",
        "$recursiveRef",
        "$recursiveAnchor",
        "$dynamicRef",
        "$dynamicAnchor"
    ]);
    function hasRef(schema3) {
        for(const key in schema3){
            if (REF_KEYWORDS.has(key)) return true;
            const sch = schema3[key];
            if (Array.isArray(sch) && sch.some(hasRef)) return true;
            if (typeof sch == "object" && hasRef(sch)) return true;
        }
        return false;
    }
    function countKeys(schema3) {
        let count = 0;
        for(const key in schema3){
            if (key === "$ref") return Infinity;
            count++;
            if (SIMPLE_INLINED.has(key)) continue;
            if (typeof schema3[key] == "object") {
                (0, util.eachItem)(schema3[key], (sch)=>count += countKeys(sch)
                );
            }
            if (count === Infinity) return Infinity;
        }
        return count;
    }
    function getFullPath(id2 = "", normalize5) {
        if (normalize5 !== false) id2 = normalizeId(id2);
        const p = __pika_web_default_export_for_treeshaking__.parse(id2);
        return _getFullPath(p);
    }
    exports.getFullPath = getFullPath;
    function _getFullPath(p) {
        return __pika_web_default_export_for_treeshaking__.serialize(p).split("#")[0] + "#";
    }
    exports._getFullPath = _getFullPath;
    const TRAILING_SLASH_HASH = /#\/?$/;
    function normalizeId(id2) {
        return id2 ? id2.replace(TRAILING_SLASH_HASH, "") : "";
    }
    exports.normalizeId = normalizeId;
    function resolveUrl(baseId, id2) {
        id2 = normalizeId(id2);
        return __pika_web_default_export_for_treeshaking__.resolve(baseId, id2);
    }
    exports.resolveUrl = resolveUrl;
    const ANCHOR = /^[a-z_][-a-z0-9._]*$/i;
    function getSchemaRefs(schema3, baseId) {
        if (typeof schema3 == "boolean") return {
        };
        const { schemaId  } = this.opts;
        const schId = normalizeId(schema3[schemaId] || baseId);
        const baseIds = {
            "": schId
        };
        const pathPrefix = getFullPath(schId, false);
        const localRefs = {
        };
        const schemaRefs = new Set();
        jsonSchemaTraverse(schema3, {
            allKeys: true
        }, (sch, jsonPtr, _2, parentJsonPtr)=>{
            if (parentJsonPtr === void 0) return;
            const fullPath = pathPrefix + jsonPtr;
            let baseId2 = baseIds[parentJsonPtr];
            if (typeof sch[schemaId] == "string") baseId2 = addRef.call(this, sch[schemaId]);
            addAnchor.call(this, sch.$anchor);
            addAnchor.call(this, sch.$dynamicAnchor);
            baseIds[jsonPtr] = baseId2;
            function addRef(ref2) {
                ref2 = normalizeId(baseId2 ? __pika_web_default_export_for_treeshaking__.resolve(baseId2, ref2) : ref2);
                if (schemaRefs.has(ref2)) throw ambiguos(ref2);
                schemaRefs.add(ref2);
                let schOrRef = this.refs[ref2];
                if (typeof schOrRef == "string") schOrRef = this.refs[schOrRef];
                if (typeof schOrRef == "object") {
                    checkAmbiguosRef(sch, schOrRef.schema, ref2);
                } else if (ref2 !== normalizeId(fullPath)) {
                    if (ref2[0] === "#") {
                        checkAmbiguosRef(sch, localRefs[ref2], ref2);
                        localRefs[ref2] = sch;
                    } else {
                        this.refs[ref2] = fullPath;
                    }
                }
                return ref2;
            }
            function addAnchor(anchor) {
                if (typeof anchor == "string") {
                    if (!ANCHOR.test(anchor)) throw new Error(`invalid anchor "${anchor}"`);
                    addRef.call(this, `#${anchor}`);
                }
            }
        });
        return localRefs;
        function checkAmbiguosRef(sch1, sch2, ref2) {
            if (sch2 !== void 0 && !fastDeepEqual(sch1, sch2)) throw ambiguos(ref2);
        }
        function ambiguos(ref2) {
            return new Error(`reference "${ref2}" resolves to more than one schema`);
        }
    }
    exports.getSchemaRefs = getSchemaRefs;
});
var validate1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.getData = exports.KeywordCxt = exports.validateFunctionCode = void 0;
    const dataType_2 = dataType;
    function validateFunctionCode(it) {
        if (isSchemaObj(it)) {
            checkKeywords(it);
            if (schemaCxtHasRules(it)) {
                topSchemaObjCode(it);
                return;
            }
        }
        validateFunction(it, ()=>(0, boolSchema.topBoolOrEmptySchema)(it)
        );
    }
    exports.validateFunctionCode = validateFunctionCode;
    function validateFunction({ gen , validateName , schema: schema3 , schemaEnv , opts  }, body) {
        if (opts.code.es5) {
            gen.func(validateName, codegen._`${names_1.default.data}, ${names_1.default.valCxt}`, schemaEnv.$async, ()=>{
                gen.code(codegen._`"use strict"; ${funcSourceUrl(schema3, opts)}`);
                destructureValCxtES5(gen, opts);
                gen.code(body);
            });
        } else {
            gen.func(validateName, codegen._`${names_1.default.data}, ${destructureValCxt(opts)}`, schemaEnv.$async, ()=>gen.code(funcSourceUrl(schema3, opts)).code(body)
            );
        }
    }
    function destructureValCxt(opts) {
        return codegen._`{${names_1.default.instancePath}="", ${names_1.default.parentData}, ${names_1.default.parentDataProperty}, ${names_1.default.rootData}=${names_1.default.data}${opts.dynamicRef ? codegen._`, ${names_1.default.dynamicAnchors}={}` : codegen.nil}}={}`;
    }
    function destructureValCxtES5(gen, opts) {
        gen.if(names_1.default.valCxt, ()=>{
            gen.var(names_1.default.instancePath, codegen._`${names_1.default.valCxt}.${names_1.default.instancePath}`);
            gen.var(names_1.default.parentData, codegen._`${names_1.default.valCxt}.${names_1.default.parentData}`);
            gen.var(names_1.default.parentDataProperty, codegen._`${names_1.default.valCxt}.${names_1.default.parentDataProperty}`);
            gen.var(names_1.default.rootData, codegen._`${names_1.default.valCxt}.${names_1.default.rootData}`);
            if (opts.dynamicRef) gen.var(names_1.default.dynamicAnchors, codegen._`${names_1.default.valCxt}.${names_1.default.dynamicAnchors}`);
        }, ()=>{
            gen.var(names_1.default.instancePath, codegen._`""`);
            gen.var(names_1.default.parentData, codegen._`undefined`);
            gen.var(names_1.default.parentDataProperty, codegen._`undefined`);
            gen.var(names_1.default.rootData, names_1.default.data);
            if (opts.dynamicRef) gen.var(names_1.default.dynamicAnchors, codegen._`{}`);
        });
    }
    function topSchemaObjCode(it) {
        const { schema: schema3 , opts , gen  } = it;
        validateFunction(it, ()=>{
            if (opts.$comment && schema3.$comment) commentKeyword(it);
            checkNoDefault(it);
            gen.let(names_1.default.vErrors, null);
            gen.let(names_1.default.errors, 0);
            if (opts.unevaluated) resetEvaluated(it);
            typeAndKeywords(it);
            returnResults(it);
        });
        return;
    }
    function resetEvaluated(it) {
        const { gen , validateName  } = it;
        it.evaluated = gen.const("evaluated", codegen._`${validateName}.evaluated`);
        gen.if(codegen._`${it.evaluated}.dynamicProps`, ()=>gen.assign(codegen._`${it.evaluated}.props`, codegen._`undefined`)
        );
        gen.if(codegen._`${it.evaluated}.dynamicItems`, ()=>gen.assign(codegen._`${it.evaluated}.items`, codegen._`undefined`)
        );
    }
    function funcSourceUrl(schema3, opts) {
        const schId = typeof schema3 == "object" && schema3[opts.schemaId];
        return schId && (opts.code.source || opts.code.process) ? codegen._`/*# sourceURL=${schId} */` : codegen.nil;
    }
    function subschemaCode(it, valid) {
        if (isSchemaObj(it)) {
            checkKeywords(it);
            if (schemaCxtHasRules(it)) {
                subSchemaObjCode(it, valid);
                return;
            }
        }
        (0, boolSchema.boolOrEmptySchema)(it, valid);
    }
    function schemaCxtHasRules({ schema: schema3 , self  }) {
        if (typeof schema3 == "boolean") return !schema3;
        for(const key in schema3)if (self.RULES.all[key]) return true;
        return false;
    }
    function isSchemaObj(it) {
        return typeof it.schema != "boolean";
    }
    function subSchemaObjCode(it, valid) {
        const { schema: schema3 , gen , opts  } = it;
        if (opts.$comment && schema3.$comment) commentKeyword(it);
        updateContext(it);
        checkAsyncSchema(it);
        const errsCount = gen.const("_errs", names_1.default.errors);
        typeAndKeywords(it, errsCount);
        gen.var(valid, codegen._`${errsCount} === ${names_1.default.errors}`);
    }
    function checkKeywords(it) {
        (0, util.checkUnknownRules)(it);
        checkRefsAndKeywords(it);
    }
    function typeAndKeywords(it, errsCount) {
        if (it.opts.jtd) return schemaKeywords(it, [], false, errsCount);
        const types2 = (0, dataType.getSchemaTypes)(it.schema);
        const checkedTypes = (0, dataType.coerceAndCheckDataType)(it, types2);
        schemaKeywords(it, types2, !checkedTypes, errsCount);
    }
    function checkRefsAndKeywords(it) {
        const { schema: schema3 , errSchemaPath , opts , self  } = it;
        if (schema3.$ref && opts.ignoreKeywordsWithRef && (0, util.schemaHasRulesButRef)(schema3, self.RULES)) {
            self.logger.warn(`$ref: keywords ignored in schema at path "${errSchemaPath}"`);
        }
    }
    function checkNoDefault(it) {
        const { schema: schema3 , opts  } = it;
        if (schema3.default !== void 0 && opts.useDefaults && opts.strictSchema) {
            (0, util.checkStrictMode)(it, "default is ignored in the schema root");
        }
    }
    function updateContext(it) {
        const schId = it.schema[it.opts.schemaId];
        if (schId) it.baseId = (0, resolve4.resolveUrl)(it.baseId, schId);
    }
    function checkAsyncSchema(it) {
        if (it.schema.$async && !it.schemaEnv.$async) throw new Error("async schema in sync schema");
    }
    function commentKeyword({ gen , schemaEnv , schema: schema3 , errSchemaPath , opts  }) {
        const msg = schema3.$comment;
        if (opts.$comment === true) {
            gen.code(codegen._`${names_1.default.self}.logger.log(${msg})`);
        } else if (typeof opts.$comment == "function") {
            const schemaPath = codegen.str`${errSchemaPath}/$comment`;
            const rootName = gen.scopeValue("root", {
                ref: schemaEnv.root
            });
            gen.code(codegen._`${names_1.default.self}.opts.$comment(${msg}, ${schemaPath}, ${rootName}.schema)`);
        }
    }
    function returnResults(it) {
        const { gen , schemaEnv , validateName , ValidationError: ValidationError2 , opts  } = it;
        if (schemaEnv.$async) {
            gen.if(codegen._`${names_1.default.errors} === 0`, ()=>gen.return(names_1.default.data)
            , ()=>gen.throw(codegen._`new ${ValidationError2}(${names_1.default.vErrors})`)
            );
        } else {
            gen.assign(codegen._`${validateName}.errors`, names_1.default.vErrors);
            if (opts.unevaluated) assignEvaluated(it);
            gen.return(codegen._`${names_1.default.errors} === 0`);
        }
    }
    function assignEvaluated({ gen , evaluated , props , items: items2  }) {
        if (props instanceof codegen.Name) gen.assign(codegen._`${evaluated}.props`, props);
        if (items2 instanceof codegen.Name) gen.assign(codegen._`${evaluated}.items`, items2);
    }
    function schemaKeywords(it, types2, typeErrors, errsCount) {
        const { gen , schema: schema3 , data , allErrors , opts , self  } = it;
        const { RULES  } = self;
        if (schema3.$ref && (opts.ignoreKeywordsWithRef || !(0, util.schemaHasRulesButRef)(schema3, RULES))) {
            gen.block(()=>keywordCode(it, "$ref", RULES.all.$ref.definition)
            );
            return;
        }
        if (!opts.jtd) checkStrictTypes(it, types2);
        gen.block(()=>{
            for (const group of RULES.rules)groupKeywords(group);
            groupKeywords(RULES.post);
        });
        function groupKeywords(group) {
            if (!(0, applicability.shouldUseGroup)(schema3, group)) return;
            if (group.type) {
                gen.if((0, dataType_2.checkDataType)(group.type, data, opts.strictNumbers));
                iterateKeywords(it, group);
                if (types2.length === 1 && types2[0] === group.type && typeErrors) {
                    gen.else();
                    (0, dataType_2.reportTypeError)(it);
                }
                gen.endIf();
            } else {
                iterateKeywords(it, group);
            }
            if (!allErrors) gen.if(codegen._`${names_1.default.errors} === ${errsCount || 0}`);
        }
    }
    function iterateKeywords(it, group) {
        const { gen , schema: schema3 , opts: { useDefaults  }  } = it;
        if (useDefaults) (0, defaults.assignDefaults)(it, group.type);
        gen.block(()=>{
            for (const rule of group.rules){
                if ((0, applicability.shouldUseRule)(schema3, rule)) {
                    keywordCode(it, rule.keyword, rule.definition, group.type);
                }
            }
        });
    }
    function checkStrictTypes(it, types2) {
        if (it.schemaEnv.meta || !it.opts.strictTypes) return;
        checkContextTypes(it, types2);
        if (!it.opts.allowUnionTypes) checkMultipleTypes(it, types2);
        checkKeywordTypes(it, it.dataTypes);
    }
    function checkContextTypes(it, types2) {
        if (!types2.length) return;
        if (!it.dataTypes.length) {
            it.dataTypes = types2;
            return;
        }
        types2.forEach((t)=>{
            if (!includesType(it.dataTypes, t)) {
                strictTypesError(it, `type "${t}" not allowed by context "${it.dataTypes.join(",")}"`);
            }
        });
        it.dataTypes = it.dataTypes.filter((t)=>includesType(types2, t)
        );
    }
    function checkMultipleTypes(it, ts) {
        if (ts.length > 1 && !(ts.length === 2 && ts.includes("null"))) {
            strictTypesError(it, "use allowUnionTypes to allow union type keyword");
        }
    }
    function checkKeywordTypes(it, ts) {
        const rules2 = it.self.RULES.all;
        for(const keyword2 in rules2){
            const rule = rules2[keyword2];
            if (typeof rule == "object" && (0, applicability.shouldUseRule)(it.schema, rule)) {
                const { type: type21  } = rule.definition;
                if (type21.length && !type21.some((t)=>hasApplicableType(ts, t)
                )) {
                    strictTypesError(it, `missing type "${type21.join(",")}" for keyword "${keyword2}"`);
                }
            }
        }
    }
    function hasApplicableType(schTs, kwdT) {
        return schTs.includes(kwdT) || kwdT === "number" && schTs.includes("integer");
    }
    function includesType(ts, t) {
        return ts.includes(t) || t === "integer" && ts.includes("number");
    }
    function strictTypesError(it, msg) {
        const schemaPath = it.schemaEnv.baseId + it.errSchemaPath;
        msg += ` at "${schemaPath}" (strictTypes)`;
        (0, util.checkStrictMode)(it, msg, it.opts.strictTypes);
    }
    class KeywordCxt2 {
        constructor(it, def1, keyword$1){
            (0, keyword1.validateKeywordUsage)(it, def1, keyword$1);
            this.gen = it.gen;
            this.allErrors = it.allErrors;
            this.keyword = keyword$1;
            this.data = it.data;
            this.schema = it.schema[keyword$1];
            this.$data = def1.$data && it.opts.$data && this.schema && this.schema.$data;
            this.schemaValue = (0, util.schemaRefOrVal)(it, this.schema, keyword$1, this.$data);
            this.schemaType = def1.schemaType;
            this.parentSchema = it.schema;
            this.params = {
            };
            this.it = it;
            this.def = def1;
            if (this.$data) {
                this.schemaCode = it.gen.const("vSchema", getData(this.$data, it));
            } else {
                this.schemaCode = this.schemaValue;
                if (!(0, keyword1.validSchemaType)(this.schema, def1.schemaType, def1.allowUndefined)) {
                    throw new Error(`${keyword$1} value must be ${JSON.stringify(def1.schemaType)}`);
                }
            }
            if ("code" in def1 ? def1.trackErrors : def1.errors !== false) {
                this.errsCount = it.gen.const("_errs", names_1.default.errors);
            }
        }
        result(condition, successAction, failAction) {
            this.failResult((0, codegen.not)(condition), successAction, failAction);
        }
        failResult(condition, successAction, failAction) {
            this.gen.if(condition);
            if (failAction) failAction();
            else this.error();
            if (successAction) {
                this.gen.else();
                successAction();
                if (this.allErrors) this.gen.endIf();
            } else {
                if (this.allErrors) this.gen.endIf();
                else this.gen.else();
            }
        }
        pass(condition, failAction) {
            this.failResult((0, codegen.not)(condition), void 0, failAction);
        }
        fail(condition) {
            if (condition === void 0) {
                this.error();
                if (!this.allErrors) this.gen.if(false);
                return;
            }
            this.gen.if(condition);
            this.error();
            if (this.allErrors) this.gen.endIf();
            else this.gen.else();
        }
        fail$data(condition) {
            if (!this.$data) return this.fail(condition);
            const { schemaCode  } = this;
            this.fail(codegen._`${schemaCode} !== undefined && (${(0, codegen.or)(this.invalid$data(), condition)})`);
        }
        error(append, errorParams, errorPaths) {
            if (errorParams) {
                this.setParams(errorParams);
                this._error(append, errorPaths);
                this.setParams({
                });
                return;
            }
            this._error(append, errorPaths);
        }
        _error(append, errorPaths) {
            (append ? errors.reportExtraError : errors.reportError)(this, this.def.error, errorPaths);
        }
        $dataError() {
            (0, errors.reportError)(this, this.def.$dataError || errors.keyword$DataError);
        }
        reset() {
            if (this.errsCount === void 0) throw new Error('add "trackErrors" to keyword definition');
            (0, errors.resetErrorsCount)(this.gen, this.errsCount);
        }
        ok(cond) {
            if (!this.allErrors) this.gen.if(cond);
        }
        setParams(obj, assign) {
            if (assign) Object.assign(this.params, obj);
            else this.params = obj;
        }
        block$data(valid, codeBlock, $dataValid = codegen.nil) {
            this.gen.block(()=>{
                this.check$data(valid, $dataValid);
                codeBlock();
            });
        }
        check$data(valid = codegen.nil, $dataValid = codegen.nil) {
            if (!this.$data) return;
            const { gen , schemaCode , schemaType: schemaType1 , def: def2  } = this;
            gen.if((0, codegen.or)(codegen._`${schemaCode} === undefined`, $dataValid));
            if (valid !== codegen.nil) gen.assign(valid, true);
            if (schemaType1.length || def2.validateSchema) {
                gen.elseIf(this.invalid$data());
                this.$dataError();
                if (valid !== codegen.nil) gen.assign(valid, false);
            }
            gen.else();
        }
        invalid$data() {
            const { gen , schemaCode , schemaType: schemaType1 , def: def2 , it: it1  } = this;
            return (0, codegen.or)(wrong$DataType(), invalid$DataSchema());
            function wrong$DataType() {
                if (schemaType1.length) {
                    if (!(schemaCode instanceof codegen.Name)) throw new Error("ajv implementation error");
                    const st = Array.isArray(schemaType1) ? schemaType1 : [
                        schemaType1
                    ];
                    return codegen._`${(0, dataType_2.checkDataTypes)(st, schemaCode, it1.opts.strictNumbers, dataType_2.DataType.Wrong)}`;
                }
                return codegen.nil;
            }
            function invalid$DataSchema() {
                if (def2.validateSchema) {
                    const validateSchemaRef = gen.scopeValue("validate$data", {
                        ref: def2.validateSchema
                    });
                    return codegen._`!${validateSchemaRef}(${schemaCode})`;
                }
                return codegen.nil;
            }
        }
        subschema(appl, valid) {
            const subschema$1 = (0, subschema.getSubschema)(this.it, appl);
            (0, subschema.extendSubschemaData)(subschema$1, this.it, appl);
            (0, subschema.extendSubschemaMode)(subschema$1, appl);
            const nextContext = {
                ...this.it,
                ...subschema$1,
                items: void 0,
                props: void 0
            };
            subschemaCode(nextContext, valid);
            return nextContext;
        }
        mergeEvaluated(schemaCxt, toName) {
            const { it: it1 , gen  } = this;
            if (!it1.opts.unevaluated) return;
            if (it1.props !== true && schemaCxt.props !== void 0) {
                it1.props = util.mergeEvaluated.props(gen, schemaCxt.props, it1.props, toName);
            }
            if (it1.items !== true && schemaCxt.items !== void 0) {
                it1.items = util.mergeEvaluated.items(gen, schemaCxt.items, it1.items, toName);
            }
        }
        mergeValidEvaluated(schemaCxt, valid) {
            const { it: it1 , gen  } = this;
            if (it1.opts.unevaluated && (it1.props !== true || it1.items !== true)) {
                gen.if(valid, ()=>this.mergeEvaluated(schemaCxt, codegen.Name)
                );
                return true;
            }
        }
    }
    exports.KeywordCxt = KeywordCxt2;
    function keywordCode(it1, keyword$11, def2, ruleType) {
        const cxt = new KeywordCxt2(it1, def2, keyword$11);
        if ("code" in def2) {
            def2.code(cxt, ruleType);
        } else if (cxt.$data && def2.validate) {
            (0, keyword1.funcKeywordCode)(cxt, def2);
        } else if ("macro" in def2) {
            (0, keyword1.macroKeywordCode)(cxt, def2);
        } else if (def2.compile || def2.validate) {
            (0, keyword1.funcKeywordCode)(cxt, def2);
        }
    }
    const JSON_POINTER = /^\/(?:[^~]|~0|~1)*$/;
    const RELATIVE_JSON_POINTER = /^([0-9]+)(#|\/(?:[^~]|~0|~1)*)?$/;
    function getData($data, { dataLevel , dataNames , dataPathArr  }) {
        let jsonPointer;
        let data;
        if ($data === "") return names_1.default.rootData;
        if ($data[0] === "/") {
            if (!JSON_POINTER.test($data)) throw new Error(`Invalid JSON-pointer: ${$data}`);
            jsonPointer = $data;
            data = names_1.default.rootData;
        } else {
            const matches = RELATIVE_JSON_POINTER.exec($data);
            if (!matches) throw new Error(`Invalid JSON-pointer: ${$data}`);
            const up = +matches[1];
            jsonPointer = matches[2];
            if (jsonPointer === "#") {
                if (up >= dataLevel) throw new Error(errorMsg("property/index", up));
                return dataPathArr[dataLevel - up];
            }
            if (up > dataLevel) throw new Error(errorMsg("data", up));
            data = dataNames[dataLevel - up];
            if (!jsonPointer) return data;
        }
        let expr = data;
        const segments = jsonPointer.split("/");
        for (const segment of segments){
            if (segment) {
                data = codegen._`${data}${(0, codegen.getProperty)((0, util.unescapeJsonPointer)(segment))}`;
                expr = codegen._`${expr} && ${data}`;
            }
        }
        return expr;
        function errorMsg(pointerType, up) {
            return `Cannot access ${pointerType} ${up} levels up, current level is ${dataLevel}`;
        }
    }
    exports.getData = getData;
});
var validation_error = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    class ValidationError2 extends Error {
        constructor(errors2){
            super("validation failed");
            this.errors = errors2;
            this.ajv = this.validation = true;
        }
    }
    exports.default = ValidationError2;
});
var ref_error = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    class MissingRefError extends Error {
        constructor(baseId, ref2, msg){
            super(msg || `can't resolve reference ${ref2} from id ${baseId}`);
            this.missingRef = (0, resolve4.resolveUrl)(baseId, ref2);
            this.missingSchema = (0, resolve4.normalizeId)((0, resolve4.getFullPath)(this.missingRef));
        }
    }
    exports.default = MissingRefError;
});
var compile = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.resolveSchema = exports.getCompilingSchema = exports.resolveRef = exports.compileSchema = exports.SchemaEnv = void 0;
    class SchemaEnv {
        constructor(env){
            var _a;
            this.refs = {
            };
            this.dynamicAnchors = {
            };
            let schema3;
            if (typeof env.schema == "object") schema3 = env.schema;
            this.schema = env.schema;
            this.schemaId = env.schemaId;
            this.root = env.root || this;
            this.baseId = (_a = env.baseId) !== null && _a !== void 0 ? _a : (0, resolve4.normalizeId)(schema3 === null || schema3 === void 0 ? void 0 : schema3[env.schemaId || "$id"]);
            this.schemaPath = env.schemaPath;
            this.localRefs = env.localRefs;
            this.meta = env.meta;
            this.$async = schema3 === null || schema3 === void 0 ? void 0 : schema3.$async;
            this.refs = {
            };
        }
    }
    exports.SchemaEnv = SchemaEnv;
    function compileSchema(sch) {
        const _sch = getCompilingSchema.call(this, sch);
        if (_sch) return _sch;
        const rootId = (0, resolve4.getFullPath)(sch.root.baseId);
        const { es5 , lines: lines1  } = this.opts.code;
        const { ownProperties  } = this.opts;
        const gen = new codegen.CodeGen(this.scope, {
            es5,
            lines: lines1,
            ownProperties
        });
        let _ValidationError;
        if (sch.$async) {
            _ValidationError = gen.scopeValue("Error", {
                ref: validation_error.default,
                code: codegen._`require("ajv/dist/runtime/validation_error").default`
            });
        }
        const validateName = gen.scopeName("validate");
        sch.validateName = validateName;
        const schemaCxt = {
            gen,
            allErrors: this.opts.allErrors,
            data: names_1.default.data,
            parentData: names_1.default.parentData,
            parentDataProperty: names_1.default.parentDataProperty,
            dataNames: [
                names_1.default.data
            ],
            dataPathArr: [
                codegen.nil
            ],
            dataLevel: 0,
            dataTypes: [],
            definedProperties: new Set(),
            topSchemaRef: gen.scopeValue("schema", this.opts.code.source === true ? {
                ref: sch.schema,
                code: (0, codegen.stringify)(sch.schema)
            } : {
                ref: sch.schema
            }),
            validateName,
            ValidationError: _ValidationError,
            schema: sch.schema,
            schemaEnv: sch,
            rootId,
            baseId: sch.baseId || rootId,
            schemaPath: codegen.nil,
            errSchemaPath: sch.schemaPath || (this.opts.jtd ? "" : "#"),
            errorPath: codegen._`""`,
            opts: this.opts,
            self: this
        };
        let sourceCode;
        try {
            this._compilations.add(sch);
            (0, validate1.validateFunctionCode)(schemaCxt);
            gen.optimize(this.opts.code.optimize);
            const validateCode = gen.toString();
            sourceCode = `${gen.scopeRefs(names_1.default.scope)}return ${validateCode}`;
            if (this.opts.code.process) sourceCode = this.opts.code.process(sourceCode, sch);
            const makeValidate = new Function(`${names_1.default.self}`, `${names_1.default.scope}`, sourceCode);
            const validate$1 = makeValidate(this, this.scope.get());
            this.scope.value(validateName, {
                ref: validate$1
            });
            validate$1.errors = null;
            validate$1.schema = sch.schema;
            validate$1.schemaEnv = sch;
            if (sch.$async) validate$1.$async = true;
            if (this.opts.code.source === true) {
                validate$1.source = {
                    validateName,
                    validateCode,
                    scopeValues: gen._values
                };
            }
            if (this.opts.unevaluated) {
                const { props , items: items2  } = schemaCxt;
                validate$1.evaluated = {
                    props: props instanceof codegen.Name ? void 0 : props,
                    items: items2 instanceof codegen.Name ? void 0 : items2,
                    dynamicProps: props instanceof codegen.Name,
                    dynamicItems: items2 instanceof codegen.Name
                };
                if (validate$1.source) validate$1.source.evaluated = (0, codegen.stringify)(validate$1.evaluated);
            }
            sch.validate = validate$1;
            return sch;
        } catch (e) {
            delete sch.validate;
            delete sch.validateName;
            if (sourceCode) this.logger.error("Error compiling schema, function code:", sourceCode);
            throw e;
        } finally{
            this._compilations.delete(sch);
        }
    }
    exports.compileSchema = compileSchema;
    function resolveRef(root, baseId, ref2) {
        var _a1;
        ref2 = (0, resolve4.resolveUrl)(baseId, ref2);
        const schOrFunc = root.refs[ref2];
        if (schOrFunc) return schOrFunc;
        let _sch = resolve$1.call(this, root, ref2);
        if (_sch === void 0) {
            const schema4 = (_a1 = root.localRefs) === null || _a1 === void 0 ? void 0 : _a1[ref2];
            const { schemaId  } = this.opts;
            if (schema4) _sch = new SchemaEnv({
                schema: schema4,
                schemaId,
                root,
                baseId
            });
        }
        if (_sch === void 0) return;
        return root.refs[ref2] = inlineOrCompile.call(this, _sch);
    }
    exports.resolveRef = resolveRef;
    function inlineOrCompile(sch) {
        if ((0, resolve4.inlineRef)(sch.schema, this.opts.inlineRefs)) return sch.schema;
        return sch.validate ? sch : compileSchema.call(this, sch);
    }
    function getCompilingSchema(schEnv) {
        for (const sch of this._compilations){
            if (sameSchemaEnv(sch, schEnv)) return sch;
        }
    }
    exports.getCompilingSchema = getCompilingSchema;
    function sameSchemaEnv(s1, s2) {
        return s1.schema === s2.schema && s1.root === s2.root && s1.baseId === s2.baseId;
    }
    function resolve$1(root, ref2) {
        let sch;
        while(typeof (sch = this.refs[ref2]) == "string")ref2 = sch;
        return sch || this.schemas[ref2] || resolveSchema.call(this, root, ref2);
    }
    function resolveSchema(root, ref2) {
        const p = __pika_web_default_export_for_treeshaking__.parse(ref2);
        const refPath = (0, resolve4._getFullPath)(p);
        let baseId = (0, resolve4.getFullPath)(root.baseId);
        if (Object.keys(root.schema).length > 0 && refPath === baseId) {
            return getJsonPointer.call(this, p, root);
        }
        const id2 = (0, resolve4.normalizeId)(refPath);
        const schOrRef = this.refs[id2] || this.schemas[id2];
        if (typeof schOrRef == "string") {
            const sch = resolveSchema.call(this, root, schOrRef);
            if (typeof (sch === null || sch === void 0 ? void 0 : sch.schema) !== "object") return;
            return getJsonPointer.call(this, p, sch);
        }
        if (typeof (schOrRef === null || schOrRef === void 0 ? void 0 : schOrRef.schema) !== "object") return;
        if (!schOrRef.validate) compileSchema.call(this, schOrRef);
        if (id2 === (0, resolve4.normalizeId)(ref2)) {
            const { schema: schema4  } = schOrRef;
            const { schemaId  } = this.opts;
            const schId = schema4[schemaId];
            if (schId) baseId = (0, resolve4.resolveUrl)(baseId, schId);
            return new SchemaEnv({
                schema: schema4,
                schemaId,
                root,
                baseId
            });
        }
        return getJsonPointer.call(this, p, schOrRef);
    }
    exports.resolveSchema = resolveSchema;
    const PREVENT_SCOPE_CHANGE = new Set([
        "properties",
        "patternProperties",
        "enum",
        "dependencies",
        "definitions"
    ]);
    function getJsonPointer(parsedRef, { baseId , schema: schema4 , root  }) {
        var _a1;
        if (((_a1 = parsedRef.fragment) === null || _a1 === void 0 ? void 0 : _a1[0]) !== "/") return;
        for (const part of parsedRef.fragment.slice(1).split("/")){
            if (typeof schema4 === "boolean") return;
            const partSchema = schema4[(0, util.unescapeFragment)(part)];
            if (partSchema === void 0) return;
            schema4 = partSchema;
            const schId = typeof schema4 === "object" && schema4[this.opts.schemaId];
            if (!PREVENT_SCOPE_CHANGE.has(part) && schId) {
                baseId = (0, resolve4.resolveUrl)(baseId, schId);
            }
        }
        let env1;
        if (typeof schema4 != "boolean" && schema4.$ref && !(0, util.schemaHasRulesButRef)(schema4, this.RULES)) {
            const $ref = (0, resolve4.resolveUrl)(baseId, schema4.$ref);
            env1 = resolveSchema.call(this, root, $ref);
        }
        const { schemaId  } = this.opts;
        env1 = env1 || new SchemaEnv({
            schema: schema4,
            schemaId,
            root,
            baseId
        });
        if (env1.schema !== env1.root.schema) return env1;
        return void 0;
    }
});
const $id = "https://raw.githubusercontent.com/ajv-validator/ajv/master/lib/refs/data.json#";
const description = "Meta-schema for $data reference (JSON AnySchema extension proposal)";
const type4 = "object";
const required = [
    "$data"
];
const properties = {
    $data: {
        type: "string",
        anyOf: [
            {
                format: "relative-json-pointer"
            },
            {
                format: "json-pointer"
            }
        ]
    }
};
var $dataRefSchema = {
    $id,
    description,
    type: type4,
    required,
    properties,
    additionalProperties: false
};
var core1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.CodeGen = exports.Name = exports.nil = exports.stringify = exports.str = exports._ = exports.KeywordCxt = void 0;
    Object.defineProperty(exports, "KeywordCxt", {
        enumerable: true,
        get: function() {
            return validate1.KeywordCxt;
        }
    });
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return codegen._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return codegen.str;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return codegen.stringify;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return codegen.nil;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return codegen.Name;
        }
    });
    Object.defineProperty(exports, "CodeGen", {
        enumerable: true,
        get: function() {
            return codegen.CodeGen;
        }
    });
    const codegen_2 = codegen;
    const defaultRegExp = (str2, flags)=>new RegExp(str2, flags)
    ;
    defaultRegExp.code = "new RegExp";
    const META_IGNORE_OPTIONS = [
        "removeAdditional",
        "useDefaults",
        "coerceTypes"
    ];
    const EXT_SCOPE_NAMES = new Set([
        "validate",
        "serialize",
        "parse",
        "wrapper",
        "root",
        "schema",
        "keyword",
        "pattern",
        "formats",
        "validate$data",
        "func",
        "obj",
        "Error"
    ]);
    const removedOptions = {
        errorDataPath: "",
        format: "`validateFormats: false` can be used instead.",
        nullable: '"nullable" keyword is supported by default.',
        jsonPointers: "Deprecated jsPropertySyntax can be used instead.",
        extendRefs: "Deprecated ignoreKeywordsWithRef can be used instead.",
        missingRefs: "Pass empty schema with $id that should be ignored to ajv.addSchema.",
        processCode: "Use option `code: {process: (code, schemaEnv: object) => string}`",
        sourceCode: "Use option `code: {source: true}`",
        strictDefaults: "It is default now, see option `strict`.",
        strictKeywords: "It is default now, see option `strict`.",
        uniqueItems: '"uniqueItems" keyword is always validated.',
        unknownFormats: "Disable strict mode or pass `true` to `ajv.addFormat` (or `formats` option).",
        cache: "Map is used as cache, schema object as key.",
        serialize: "Map is used as cache, schema object as key.",
        ajvErrors: "It is default now."
    };
    const deprecatedOptions = {
        ignoreKeywordsWithRef: "",
        jsPropertySyntax: "",
        unicode: '"minLength"/"maxLength" account for unicode characters by default.'
    };
    const MAX_EXPRESSION = 200;
    function requiredOptions(o) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z;
        const s = o.strict;
        const _optz = (_a = o.code) === null || _a === void 0 ? void 0 : _a.optimize;
        const optimize = _optz === true || _optz === void 0 ? 1 : _optz || 0;
        const regExp = (_c = (_b = o.code) === null || _b === void 0 ? void 0 : _b.regExp) !== null && _c !== void 0 ? _c : defaultRegExp;
        return {
            strictSchema: (_e = (_d = o.strictSchema) !== null && _d !== void 0 ? _d : s) !== null && _e !== void 0 ? _e : true,
            strictNumbers: (_g = (_f = o.strictNumbers) !== null && _f !== void 0 ? _f : s) !== null && _g !== void 0 ? _g : true,
            strictTypes: (_j = (_h = o.strictTypes) !== null && _h !== void 0 ? _h : s) !== null && _j !== void 0 ? _j : "log",
            strictTuples: (_l = (_k = o.strictTuples) !== null && _k !== void 0 ? _k : s) !== null && _l !== void 0 ? _l : "log",
            strictRequired: (_o = (_m = o.strictRequired) !== null && _m !== void 0 ? _m : s) !== null && _o !== void 0 ? _o : false,
            code: o.code ? {
                ...o.code,
                optimize,
                regExp
            } : {
                optimize,
                regExp
            },
            loopRequired: (_p = o.loopRequired) !== null && _p !== void 0 ? _p : 200,
            loopEnum: (_q = o.loopEnum) !== null && _q !== void 0 ? _q : 200,
            meta: (_r = o.meta) !== null && _r !== void 0 ? _r : true,
            messages: (_s = o.messages) !== null && _s !== void 0 ? _s : true,
            inlineRefs: (_t = o.inlineRefs) !== null && _t !== void 0 ? _t : true,
            schemaId: (_u = o.schemaId) !== null && _u !== void 0 ? _u : "$id",
            addUsedSchema: (_v = o.addUsedSchema) !== null && _v !== void 0 ? _v : true,
            validateSchema: (_w = o.validateSchema) !== null && _w !== void 0 ? _w : true,
            validateFormats: (_x = o.validateFormats) !== null && _x !== void 0 ? _x : true,
            unicodeRegExp: (_y = o.unicodeRegExp) !== null && _y !== void 0 ? _y : true,
            int32range: (_z = o.int32range) !== null && _z !== void 0 ? _z : true
        };
    }
    class Ajv {
        constructor(opts = {
        }){
            this.schemas = {
            };
            this.refs = {
            };
            this.formats = {
            };
            this._compilations = new Set();
            this._loading = {
            };
            this._cache = new Map();
            opts = this.opts = {
                ...opts,
                ...requiredOptions(opts)
            };
            const { es5 , lines: lines1  } = this.opts.code;
            this.scope = new codegen_2.ValueScope({
                scope: {
                },
                prefixes: EXT_SCOPE_NAMES,
                es5,
                lines: lines1
            });
            this.logger = getLogger2(opts.logger);
            const formatOpt = opts.validateFormats;
            opts.validateFormats = false;
            this.RULES = (0, rules.getRules)();
            checkOptions.call(this, removedOptions, opts, "NOT SUPPORTED");
            checkOptions.call(this, deprecatedOptions, opts, "DEPRECATED", "warn");
            this._metaOpts = getMetaSchemaOptions.call(this);
            if (opts.formats) addInitialFormats.call(this);
            this._addVocabularies();
            this._addDefaultMetaSchema();
            if (opts.keywords) addInitialKeywords.call(this, opts.keywords);
            if (typeof opts.meta == "object") this.addMetaSchema(opts.meta);
            addInitialSchemas.call(this);
            opts.validateFormats = formatOpt;
        }
        _addVocabularies() {
            this.addKeyword("$async");
        }
        _addDefaultMetaSchema() {
            const { $data , meta , schemaId  } = this.opts;
            let _dataRefSchema = $dataRefSchema;
            if (schemaId === "id") {
                _dataRefSchema = {
                    ...$dataRefSchema
                };
                _dataRefSchema.id = _dataRefSchema.$id;
                delete _dataRefSchema.$id;
            }
            if (meta && $data) this.addMetaSchema(_dataRefSchema, _dataRefSchema[schemaId], false);
        }
        defaultMeta() {
            const { meta , schemaId  } = this.opts;
            return this.opts.defaultMeta = typeof meta == "object" ? meta[schemaId] || meta : void 0;
        }
        validate(schemaKeyRef, data) {
            let v;
            if (typeof schemaKeyRef == "string") {
                v = this.getSchema(schemaKeyRef);
                if (!v) throw new Error(`no schema with key or ref "${schemaKeyRef}"`);
            } else {
                v = this.compile(schemaKeyRef);
            }
            const valid = v(data);
            if (!("$async" in v)) this.errors = v.errors;
            return valid;
        }
        compile(schema, _meta) {
            const sch = this._addSchema(schema, _meta);
            return sch.validate || this._compileSchemaEnv(sch);
        }
        compileAsync(schema, meta) {
            if (typeof this.opts.loadSchema != "function") {
                throw new Error("options.loadSchema should be a function");
            }
            const { loadSchema  } = this.opts;
            return runCompileAsync.call(this, schema, meta);
            async function runCompileAsync(_schema, _meta) {
                await loadMetaSchema.call(this, _schema.$schema);
                const sch = this._addSchema(_schema, _meta);
                return sch.validate || _compileAsync.call(this, sch);
            }
            async function loadMetaSchema($ref) {
                if ($ref && !this.getSchema($ref)) {
                    await runCompileAsync.call(this, {
                        $ref
                    }, true);
                }
            }
            async function _compileAsync(sch) {
                try {
                    return this._compileSchemaEnv(sch);
                } catch (e) {
                    if (!(e instanceof ref_error.default)) throw e;
                    checkLoaded.call(this, e);
                    await loadMissingSchema.call(this, e.missingSchema);
                    return _compileAsync.call(this, sch);
                }
            }
            function checkLoaded({ missingSchema: ref2 , missingRef  }) {
                if (this.refs[ref2]) {
                    throw new Error(`AnySchema ${ref2} is loaded but ${missingRef} cannot be resolved`);
                }
            }
            async function loadMissingSchema(ref2) {
                const _schema = await _loadSchema.call(this, ref2);
                if (!this.refs[ref2]) await loadMetaSchema.call(this, _schema.$schema);
                if (!this.refs[ref2]) this.addSchema(_schema, ref2, meta);
            }
            async function _loadSchema(ref2) {
                const p = this._loading[ref2];
                if (p) return p;
                try {
                    return await (this._loading[ref2] = loadSchema(ref2));
                } finally{
                    delete this._loading[ref2];
                }
            }
        }
        addSchema(schema, key, _meta, _validateSchema = this.opts.validateSchema) {
            if (Array.isArray(schema)) {
                for (const sch of schema)this.addSchema(sch, void 0, _meta, _validateSchema);
                return this;
            }
            let id2;
            if (typeof schema === "object") {
                const { schemaId  } = this.opts;
                id2 = schema[schemaId];
                if (id2 !== void 0 && typeof id2 != "string") {
                    throw new Error(`schema ${schemaId} must be string`);
                }
            }
            key = (0, resolve4.normalizeId)(key || id2);
            this._checkUnique(key);
            this.schemas[key] = this._addSchema(schema, _meta, key, _validateSchema, true);
            return this;
        }
        addMetaSchema(schema, key, _validateSchema = this.opts.validateSchema) {
            this.addSchema(schema, key, true, _validateSchema);
            return this;
        }
        validateSchema(schema, throwOrLogError) {
            if (typeof schema == "boolean") return true;
            let $schema2;
            $schema2 = schema.$schema;
            if ($schema2 !== void 0 && typeof $schema2 != "string") {
                throw new Error("$schema must be a string");
            }
            $schema2 = $schema2 || this.opts.defaultMeta || this.defaultMeta();
            if (!$schema2) {
                this.logger.warn("meta-schema not available");
                this.errors = null;
                return true;
            }
            const valid = this.validate($schema2, schema);
            if (!valid && throwOrLogError) {
                const message7 = "schema is invalid: " + this.errorsText();
                if (this.opts.validateSchema === "log") this.logger.error(message7);
                else throw new Error(message7);
            }
            return valid;
        }
        getSchema(keyRef) {
            let sch;
            while(typeof (sch = getSchEnv.call(this, keyRef)) == "string")keyRef = sch;
            if (sch === void 0) {
                const { schemaId  } = this.opts;
                const root = new compile.SchemaEnv({
                    schema: {
                    },
                    schemaId
                });
                sch = compile.resolveSchema.call(this, root, keyRef);
                if (!sch) return;
                this.refs[keyRef] = sch;
            }
            return sch.validate || this._compileSchemaEnv(sch);
        }
        removeSchema(schemaKeyRef) {
            if (schemaKeyRef instanceof RegExp) {
                this._removeAllSchemas(this.schemas, schemaKeyRef);
                this._removeAllSchemas(this.refs, schemaKeyRef);
                return this;
            }
            switch(typeof schemaKeyRef){
                case "undefined":
                    this._removeAllSchemas(this.schemas);
                    this._removeAllSchemas(this.refs);
                    this._cache.clear();
                    return this;
                case "string":
                    {
                        const sch = getSchEnv.call(this, schemaKeyRef);
                        if (typeof sch == "object") this._cache.delete(sch.schema);
                        delete this.schemas[schemaKeyRef];
                        delete this.refs[schemaKeyRef];
                        return this;
                    }
                case "object":
                    {
                        const cacheKey = schemaKeyRef;
                        this._cache.delete(cacheKey);
                        let id2 = schemaKeyRef[this.opts.schemaId];
                        if (id2) {
                            id2 = (0, resolve4.normalizeId)(id2);
                            delete this.schemas[id2];
                            delete this.refs[id2];
                        }
                        return this;
                    }
                default:
                    throw new Error("ajv.removeSchema: invalid parameter");
            }
        }
        addVocabulary(definitions2) {
            for (const def1 of definitions2)this.addKeyword(def1);
            return this;
        }
        addKeyword(kwdOrDef, def) {
            let keyword2;
            if (typeof kwdOrDef == "string") {
                keyword2 = kwdOrDef;
                if (typeof def == "object") {
                    this.logger.warn("these parameters are deprecated, see docs for addKeyword");
                    def.keyword = keyword2;
                }
            } else if (typeof kwdOrDef == "object" && def === void 0) {
                def = kwdOrDef;
                keyword2 = def.keyword;
                if (Array.isArray(keyword2) && !keyword2.length) {
                    throw new Error("addKeywords: keyword must be string or non-empty array");
                }
            } else {
                throw new Error("invalid addKeywords parameters");
            }
            checkKeyword.call(this, keyword2, def);
            if (!def) {
                (0, util.eachItem)(keyword2, (kwd)=>addRule.call(this, kwd)
                );
                return this;
            }
            keywordMetaschema.call(this, def);
            const definition2 = {
                ...def,
                type: (0, dataType.getJSONTypes)(def.type),
                schemaType: (0, dataType.getJSONTypes)(def.schemaType)
            };
            (0, util.eachItem)(keyword2, definition2.type.length === 0 ? (k)=>addRule.call(this, k, definition2)
             : (k)=>definition2.type.forEach((t)=>addRule.call(this, k, definition2, t)
                )
            );
            return this;
        }
        getKeyword(keyword2) {
            const rule = this.RULES.all[keyword2];
            return typeof rule == "object" ? rule.definition : !!rule;
        }
        removeKeyword(keyword2) {
            const { RULES  } = this;
            delete RULES.keywords[keyword2];
            delete RULES.all[keyword2];
            for (const group of RULES.rules){
                const i5 = group.rules.findIndex((rule)=>rule.keyword === keyword2
                );
                if (i5 >= 0) group.rules.splice(i5, 1);
            }
            return this;
        }
        addFormat(name, format2) {
            if (typeof format2 == "string") format2 = new RegExp(format2);
            this.formats[name] = format2;
            return this;
        }
        errorsText(errors2 = this.errors, { separator =", " , dataVar ="data"  } = {
        }) {
            if (!errors2 || errors2.length === 0) return "No errors";
            return errors2.map((e)=>`${dataVar}${e.instancePath} ${e.message}`
            ).reduce((text, msg)=>text + separator + msg
            );
        }
        $dataMetaSchema(metaSchema, keywordsJsonPointers) {
            const rules2 = this.RULES.all;
            metaSchema = JSON.parse(JSON.stringify(metaSchema));
            for (const jsonPointer of keywordsJsonPointers){
                const segments = jsonPointer.split("/").slice(1);
                let keywords = metaSchema;
                for (const seg of segments)keywords = keywords[seg];
                for(const key in rules2){
                    const rule = rules2[key];
                    if (typeof rule != "object") continue;
                    const { $data  } = rule.definition;
                    const schema4 = keywords[key];
                    if ($data && schema4) keywords[key] = schemaOrData(schema4);
                }
            }
            return metaSchema;
        }
        _removeAllSchemas(schemas, regex) {
            for(const keyRef in schemas){
                const sch = schemas[keyRef];
                if (!regex || regex.test(keyRef)) {
                    if (typeof sch == "string") {
                        delete schemas[keyRef];
                    } else if (sch && !sch.meta) {
                        this._cache.delete(sch.schema);
                        delete schemas[keyRef];
                    }
                }
            }
        }
        _addSchema(schema, meta, baseId, validateSchema = this.opts.validateSchema, addSchema = this.opts.addUsedSchema) {
            let id2;
            const { schemaId  } = this.opts;
            if (typeof schema == "object") {
                id2 = schema[schemaId];
            } else {
                if (this.opts.jtd) throw new Error("schema must be object");
                else if (typeof schema != "boolean") throw new Error("schema must be object or boolean");
            }
            let sch = this._cache.get(schema);
            if (sch !== void 0) return sch;
            baseId = (0, resolve4.normalizeId)(id2 || baseId);
            const localRefs = resolve4.getSchemaRefs.call(this, schema, baseId);
            sch = new compile.SchemaEnv({
                schema,
                schemaId,
                meta,
                baseId,
                localRefs
            });
            this._cache.set(sch.schema, sch);
            if (addSchema && !baseId.startsWith("#")) {
                if (baseId) this._checkUnique(baseId);
                this.refs[baseId] = sch;
            }
            if (validateSchema) this.validateSchema(schema, true);
            return sch;
        }
        _checkUnique(id2) {
            if (this.schemas[id2] || this.refs[id2]) {
                throw new Error(`schema with key or id "${id2}" already exists`);
            }
        }
        _compileSchemaEnv(sch) {
            if (sch.meta) this._compileMetaSchema(sch);
            else compile.compileSchema.call(this, sch);
            if (!sch.validate) throw new Error("ajv implementation error");
            return sch.validate;
        }
        _compileMetaSchema(sch) {
            const currentOpts = this.opts;
            this.opts = this._metaOpts;
            try {
                compile.compileSchema.call(this, sch);
            } finally{
                this.opts = currentOpts;
            }
        }
    }
    exports.default = Ajv;
    Ajv.ValidationError = validation_error.default;
    Ajv.MissingRefError = ref_error.default;
    function checkOptions(checkOpts, options12, msg, log = "error") {
        for(const key in checkOpts){
            const opt = key;
            if (opt in options12) this.logger[log](`${msg}: option ${key}. ${checkOpts[opt]}`);
        }
    }
    function getSchEnv(keyRef) {
        keyRef = (0, resolve4.normalizeId)(keyRef);
        return this.schemas[keyRef] || this.refs[keyRef];
    }
    function addInitialSchemas() {
        const optsSchemas = this.opts.schemas;
        if (!optsSchemas) return;
        if (Array.isArray(optsSchemas)) this.addSchema(optsSchemas);
        else for(const key in optsSchemas)this.addSchema(optsSchemas[key], key);
    }
    function addInitialFormats() {
        for(const name19 in this.opts.formats){
            const format22 = this.opts.formats[name19];
            if (format22) this.addFormat(name19, format22);
        }
    }
    function addInitialKeywords(defs) {
        if (Array.isArray(defs)) {
            this.addVocabulary(defs);
            return;
        }
        this.logger.warn("keywords option as map is deprecated, pass array");
        for(const keyword2 in defs){
            const def2 = defs[keyword2];
            if (!def2.keyword) def2.keyword = keyword2;
            this.addKeyword(def2);
        }
    }
    function getMetaSchemaOptions() {
        const metaOpts = {
            ...this.opts
        };
        for (const opt of META_IGNORE_OPTIONS)delete metaOpts[opt];
        return metaOpts;
    }
    const noLogs = {
        log () {
        },
        warn () {
        },
        error () {
        }
    };
    function getLogger2(logger) {
        if (logger === false) return noLogs;
        if (logger === void 0) return console;
        if (logger.log && logger.warn && logger.error) return logger;
        throw new Error("logger must implement log, warn and error methods");
    }
    const KEYWORD_NAME = /^[a-z_$][a-z0-9_$:-]*$/i;
    function checkKeyword(keyword2, def2) {
        const { RULES  } = this;
        (0, util.eachItem)(keyword2, (kwd)=>{
            if (RULES.keywords[kwd]) throw new Error(`Keyword ${kwd} is already defined`);
            if (!KEYWORD_NAME.test(kwd)) throw new Error(`Keyword ${kwd} has invalid name`);
        });
        if (!def2) return;
        if (def2.$data && !("code" in def2 || "validate" in def2)) {
            throw new Error('$data keyword must have "code" or "validate" function');
        }
    }
    function addRule(keyword2, definition2, dataType$1) {
        var _a;
        const post = definition2 === null || definition2 === void 0 ? void 0 : definition2.post;
        if (dataType$1 && post) throw new Error('keyword with "post" flag cannot have "type"');
        const { RULES  } = this;
        let ruleGroup = post ? RULES.post : RULES.rules.find(({ type: t  })=>t === dataType$1
        );
        if (!ruleGroup) {
            ruleGroup = {
                type: dataType$1,
                rules: []
            };
            RULES.rules.push(ruleGroup);
        }
        RULES.keywords[keyword2] = true;
        if (!definition2) return;
        const rule = {
            keyword: keyword2,
            definition: {
                ...definition2,
                type: (0, dataType.getJSONTypes)(definition2.type),
                schemaType: (0, dataType.getJSONTypes)(definition2.schemaType)
            }
        };
        if (definition2.before) addBeforeRule.call(this, ruleGroup, rule, definition2.before);
        else ruleGroup.rules.push(rule);
        RULES.all[keyword2] = rule;
        (_a = definition2.implements) === null || _a === void 0 ? void 0 : _a.forEach((kwd)=>this.addKeyword(kwd)
        );
    }
    function addBeforeRule(ruleGroup, rule, before) {
        const i5 = ruleGroup.rules.findIndex((_rule)=>_rule.keyword === before
        );
        if (i5 >= 0) {
            ruleGroup.rules.splice(i5, 0, rule);
        } else {
            ruleGroup.rules.push(rule);
            this.logger.warn(`rule ${before} is not defined`);
        }
    }
    function keywordMetaschema(def2) {
        let { metaSchema  } = def2;
        if (metaSchema === void 0) return;
        if (def2.$data && this.opts.$data) metaSchema = schemaOrData(metaSchema);
        def2.validateSchema = this.compile(metaSchema, true);
    }
    const $dataRef = {
        $ref: "https://raw.githubusercontent.com/ajv-validator/ajv/master/lib/refs/data.json#"
    };
    function schemaOrData(schema4) {
        return {
            anyOf: [
                schema4,
                $dataRef
            ]
        };
    }
});
var id1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "id",
        code () {
            throw new Error('NOT SUPPORTED: keyword "id", use "$id" for schema ID');
        }
    };
    exports.default = def2;
});
var ref = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.callRef = exports.getValidate = void 0;
    const def2 = {
        keyword: "$ref",
        schemaType: "string",
        code (cxt) {
            const { gen , schema: $ref , it  } = cxt;
            const { baseId , schemaEnv: env , validateName , opts , self  } = it;
            const { root  } = env;
            if (($ref === "#" || $ref === "#/") && baseId === root.baseId) return callRootRef();
            const schOrEnv = compile.resolveRef.call(self, root, baseId, $ref);
            if (schOrEnv === void 0) throw new ref_error.default(baseId, $ref);
            if (schOrEnv instanceof compile.SchemaEnv) return callValidate(schOrEnv);
            return inlineRefSchema(schOrEnv);
            function callRootRef() {
                if (env === root) return callRef(cxt, validateName, env, env.$async);
                const rootName = gen.scopeValue("root", {
                    ref: root
                });
                return callRef(cxt, codegen._`${rootName}.validate`, root, root.$async);
            }
            function callValidate(sch) {
                const v = getValidate(cxt, sch);
                callRef(cxt, v, sch, sch.$async);
            }
            function inlineRefSchema(sch) {
                const schName = gen.scopeValue("schema", opts.code.source === true ? {
                    ref: sch,
                    code: (0, codegen.stringify)(sch)
                } : {
                    ref: sch
                });
                const valid = gen.name("valid");
                const schCxt = cxt.subschema({
                    schema: sch,
                    dataTypes: [],
                    schemaPath: codegen.nil,
                    topSchemaRef: schName,
                    errSchemaPath: $ref
                }, valid);
                cxt.mergeEvaluated(schCxt);
                cxt.ok(valid);
            }
        }
    };
    function getValidate(cxt, sch) {
        const { gen  } = cxt;
        return sch.validate ? gen.scopeValue("validate", {
            ref: sch.validate
        }) : codegen._`${gen.scopeValue("wrapper", {
            ref: sch
        })}.validate`;
    }
    exports.getValidate = getValidate;
    function callRef(cxt, v, sch, $async) {
        const { gen , it  } = cxt;
        const { allErrors , schemaEnv: env , opts  } = it;
        const passCxt = opts.passContext ? names_1.default.this : codegen.nil;
        if ($async) callAsyncRef();
        else callSyncRef();
        function callAsyncRef() {
            if (!env.$async) throw new Error("async schema referenced by sync schema");
            const valid = gen.let("valid");
            gen.try(()=>{
                gen.code(codegen._`await ${(0, code$1.callValidateCode)(cxt, v, passCxt)}`);
                addEvaluatedFrom(v);
                if (!allErrors) gen.assign(valid, true);
            }, (e)=>{
                gen.if(codegen._`!(${e} instanceof ${it.ValidationError})`, ()=>gen.throw(e)
                );
                addErrorsFrom(e);
                if (!allErrors) gen.assign(valid, false);
            });
            cxt.ok(valid);
        }
        function callSyncRef() {
            cxt.result((0, code$1.callValidateCode)(cxt, v, passCxt), ()=>addEvaluatedFrom(v)
            , ()=>addErrorsFrom(v)
            );
        }
        function addErrorsFrom(source) {
            const errs = codegen._`${source}.errors`;
            gen.assign(names_1.default.vErrors, codegen._`${names_1.default.vErrors} === null ? ${errs} : ${names_1.default.vErrors}.concat(${errs})`);
            gen.assign(names_1.default.errors, codegen._`${names_1.default.vErrors}.length`);
        }
        function addEvaluatedFrom(source) {
            var _a;
            if (!it.opts.unevaluated) return;
            const schEvaluated = (_a = sch === null || sch === void 0 ? void 0 : sch.validate) === null || _a === void 0 ? void 0 : _a.evaluated;
            if (it.props !== true) {
                if (schEvaluated && !schEvaluated.dynamicProps) {
                    if (schEvaluated.props !== void 0) {
                        it.props = util.mergeEvaluated.props(gen, schEvaluated.props, it.props);
                    }
                } else {
                    const props = gen.var("props", codegen._`${source}.evaluated.props`);
                    it.props = util.mergeEvaluated.props(gen, props, it.props, codegen.Name);
                }
            }
            if (it.items !== true) {
                if (schEvaluated && !schEvaluated.dynamicItems) {
                    if (schEvaluated.items !== void 0) {
                        it.items = util.mergeEvaluated.items(gen, schEvaluated.items, it.items);
                    }
                } else {
                    const items2 = gen.var("items", codegen._`${source}.evaluated.items`);
                    it.items = util.mergeEvaluated.items(gen, items2, it.items, codegen.Name);
                }
            }
        }
    }
    exports.callRef = callRef;
    exports.default = def2;
});
var core_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const core2 = [
        "$schema",
        "$id",
        "$defs",
        "$vocabulary",
        {
            keyword: "$comment"
        },
        "definitions",
        id1.default,
        ref.default
    ];
    exports.default = core2;
});
var limitNumber = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const ops = codegen.operators;
    const KWDs = {
        maximum: {
            okStr: "<=",
            ok: ops.LTE,
            fail: ops.GT
        },
        minimum: {
            okStr: ">=",
            ok: ops.GTE,
            fail: ops.LT
        },
        exclusiveMaximum: {
            okStr: "<",
            ok: ops.LT,
            fail: ops.GTE
        },
        exclusiveMinimum: {
            okStr: ">",
            ok: ops.GT,
            fail: ops.LTE
        }
    };
    const error1 = {
        message: ({ keyword: keyword2 , schemaCode  })=>codegen.str`must be ${KWDs[keyword2].okStr} ${schemaCode}`
        ,
        params: ({ keyword: keyword2 , schemaCode  })=>codegen._`{comparison: ${KWDs[keyword2].okStr}, limit: ${schemaCode}}`
    };
    const def2 = {
        keyword: Object.keys(KWDs),
        type: "number",
        schemaType: "number",
        $data: true,
        error: error1,
        code (cxt) {
            const { keyword: keyword2 , data , schemaCode  } = cxt;
            cxt.fail$data(codegen._`${data} ${KWDs[keyword2].fail} ${schemaCode} || isNaN(${data})`);
        }
    };
    exports.default = def2;
});
var multipleOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ schemaCode  })=>codegen.str`must be multiple of ${schemaCode}`
        ,
        params: ({ schemaCode  })=>codegen._`{multipleOf: ${schemaCode}}`
    };
    const def2 = {
        keyword: "multipleOf",
        type: "number",
        schemaType: "number",
        $data: true,
        error: error1,
        code (cxt) {
            const { gen , data , schemaCode , it  } = cxt;
            const prec = it.opts.multipleOfPrecision;
            const res = gen.let("res");
            const invalid = prec ? codegen._`Math.abs(Math.round(${res}) - ${res}) > 1e-${prec}` : codegen._`${res} !== parseInt(${res})`;
            cxt.fail$data(codegen._`(${schemaCode} === 0 || (${res} = ${data}/${schemaCode}, ${invalid}))`);
        }
    };
    exports.default = def2;
});
var ucs2length_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    function ucs2length(str2) {
        const len = str2.length;
        let length = 0;
        let pos = 0;
        let value6;
        while(pos < len){
            length++;
            value6 = str2.charCodeAt(pos++);
            if (value6 >= 55296 && value6 <= 56319 && pos < len) {
                value6 = str2.charCodeAt(pos);
                if ((value6 & 64512) === 56320) pos++;
            }
        }
        return length;
    }
    exports.default = ucs2length;
    ucs2length.code = 'require("ajv/dist/runtime/ucs2length").default';
});
var limitLength = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxLength" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} characters`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def2 = {
        keyword: [
            "maxLength",
            "minLength"
        ],
        type: "string",
        schemaType: "number",
        $data: true,
        error: error1,
        code (cxt) {
            const { keyword: keyword2 , data , schemaCode , it  } = cxt;
            const op = keyword2 === "maxLength" ? codegen.operators.GT : codegen.operators.LT;
            const len = it.opts.unicode === false ? codegen._`${data}.length` : codegen._`${(0, util.useFunc)(cxt.gen, ucs2length_1.default)}(${data})`;
            cxt.fail$data(codegen._`${len} ${op} ${schemaCode}`);
        }
    };
    exports.default = def2;
});
var pattern = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ schemaCode  })=>codegen.str`must match pattern "${schemaCode}"`
        ,
        params: ({ schemaCode  })=>codegen._`{pattern: ${schemaCode}}`
    };
    const def2 = {
        keyword: "pattern",
        type: "string",
        schemaType: "string",
        $data: true,
        error: error1,
        code (cxt) {
            const { data , $data , schema: schema4 , schemaCode , it  } = cxt;
            const u = it.opts.unicodeRegExp ? "u" : "";
            const regExp = $data ? codegen._`(new RegExp(${schemaCode}, ${u}))` : (0, code$1.usePattern)(cxt, schema4);
            cxt.fail$data(codegen._`!${regExp}.test(${data})`);
        }
    };
    exports.default = def2;
});
var limitProperties = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxProperties" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} items`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def2 = {
        keyword: [
            "maxProperties",
            "minProperties"
        ],
        type: "object",
        schemaType: "number",
        $data: true,
        error: error1,
        code (cxt) {
            const { keyword: keyword2 , data , schemaCode  } = cxt;
            const op = keyword2 === "maxProperties" ? codegen.operators.GT : codegen.operators.LT;
            cxt.fail$data(codegen._`Object.keys(${data}).length ${op} ${schemaCode}`);
        }
    };
    exports.default = def2;
});
var required$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params: { missingProperty  }  })=>codegen.str`must have required property '${missingProperty}'`
        ,
        params: ({ params: { missingProperty  }  })=>codegen._`{missingProperty: ${missingProperty}}`
    };
    const def2 = {
        keyword: "required",
        type: "object",
        schemaType: "array",
        $data: true,
        error: error1,
        code (cxt) {
            const { gen , schema: schema4 , schemaCode , data , $data , it  } = cxt;
            const { opts  } = it;
            if (!$data && schema4.length === 0) return;
            const useLoop = schema4.length >= opts.loopRequired;
            if (it.allErrors) allErrorsMode();
            else exitOnErrorMode();
            if (opts.strictRequired) {
                const props = cxt.parentSchema.properties;
                const { definedProperties  } = cxt.it;
                for (const requiredKey of schema4){
                    if ((props === null || props === void 0 ? void 0 : props[requiredKey]) === void 0 && !definedProperties.has(requiredKey)) {
                        const schemaPath = it.schemaEnv.baseId + it.errSchemaPath;
                        const msg = `required property "${requiredKey}" is not defined at "${schemaPath}" (strictRequired)`;
                        (0, util.checkStrictMode)(it, msg, it.opts.strictRequired);
                    }
                }
            }
            function allErrorsMode() {
                if (useLoop || $data) {
                    cxt.block$data(codegen.nil, loopAllRequired);
                } else {
                    for (const prop of schema4){
                        (0, code$1.checkReportMissingProp)(cxt, prop);
                    }
                }
            }
            function exitOnErrorMode() {
                const missing = gen.let("missing");
                if (useLoop || $data) {
                    const valid = gen.let("valid", true);
                    cxt.block$data(valid, ()=>loopUntilMissing(missing, valid)
                    );
                    cxt.ok(valid);
                } else {
                    gen.if((0, code$1.checkMissingProp)(cxt, schema4, missing));
                    (0, code$1.reportMissingProp)(cxt, missing);
                    gen.else();
                }
            }
            function loopAllRequired() {
                gen.forOf("prop", schemaCode, (prop)=>{
                    cxt.setParams({
                        missingProperty: prop
                    });
                    gen.if((0, code$1.noPropertyInData)(gen, data, prop, opts.ownProperties), ()=>cxt.error()
                    );
                });
            }
            function loopUntilMissing(missing, valid) {
                cxt.setParams({
                    missingProperty: missing
                });
                gen.forOf(missing, schemaCode, ()=>{
                    gen.assign(valid, (0, code$1.propertyInData)(gen, data, missing, opts.ownProperties));
                    gen.if((0, codegen.not)(valid), ()=>{
                        cxt.error();
                        gen.break();
                    });
                }, codegen.nil);
            }
        }
    };
    exports.default = def2;
});
var limitItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message ({ keyword: keyword2 , schemaCode  }) {
            const comp = keyword2 === "maxItems" ? "more" : "fewer";
            return codegen.str`must NOT have ${comp} than ${schemaCode} items`;
        },
        params: ({ schemaCode  })=>codegen._`{limit: ${schemaCode}}`
    };
    const def2 = {
        keyword: [
            "maxItems",
            "minItems"
        ],
        type: "array",
        schemaType: "number",
        $data: true,
        error: error1,
        code (cxt) {
            const { keyword: keyword2 , data , schemaCode  } = cxt;
            const op = keyword2 === "maxItems" ? codegen.operators.GT : codegen.operators.LT;
            cxt.fail$data(codegen._`${data}.length ${op} ${schemaCode}`);
        }
    };
    exports.default = def2;
});
var equal_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    fastDeepEqual.code = 'require("ajv/dist/runtime/equal").default';
    exports.default = fastDeepEqual;
});
var uniqueItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params: { i: i5 , j  }  })=>codegen.str`must NOT have duplicate items (items ## ${j} and ${i5} are identical)`
        ,
        params: ({ params: { i: i5 , j  }  })=>codegen._`{i: ${i5}, j: ${j}}`
    };
    const def2 = {
        keyword: "uniqueItems",
        type: "array",
        schemaType: "boolean",
        $data: true,
        error: error1,
        code (cxt) {
            const { gen , data , $data , schema: schema4 , parentSchema , schemaCode , it  } = cxt;
            if (!$data && !schema4) return;
            const valid = gen.let("valid");
            const itemTypes = parentSchema.items ? (0, dataType.getSchemaTypes)(parentSchema.items) : [];
            cxt.block$data(valid, validateUniqueItems, codegen._`${schemaCode} === false`);
            cxt.ok(valid);
            function validateUniqueItems() {
                const i5 = gen.let("i", codegen._`${data}.length`);
                const j = gen.let("j");
                cxt.setParams({
                    i: i5,
                    j
                });
                gen.assign(valid, true);
                gen.if(codegen._`${i5} > 1`, ()=>(canOptimize() ? loopN : loopN2)(i5, j)
                );
            }
            function canOptimize() {
                return itemTypes.length > 0 && !itemTypes.some((t)=>t === "object" || t === "array"
                );
            }
            function loopN(i5, j) {
                const item = gen.name("item");
                const wrongType = (0, dataType.checkDataTypes)(itemTypes, item, it.opts.strictNumbers, dataType.DataType.Wrong);
                const indices = gen.const("indices", codegen._`{}`);
                gen.for(codegen._`;${i5}--;`, ()=>{
                    gen.let(item, codegen._`${data}[${i5}]`);
                    gen.if(wrongType, codegen._`continue`);
                    if (itemTypes.length > 1) gen.if(codegen._`typeof ${item} == "string"`, codegen._`${item} += "_"`);
                    gen.if(codegen._`typeof ${indices}[${item}] == "number"`, ()=>{
                        gen.assign(j, codegen._`${indices}[${item}]`);
                        cxt.error();
                        gen.assign(valid, false).break();
                    }).code(codegen._`${indices}[${item}] = ${i5}`);
                });
            }
            function loopN2(i5, j) {
                const eql = (0, util.useFunc)(gen, equal_1.default);
                const outer = gen.name("outer");
                gen.label(outer).for(codegen._`;${i5}--;`, ()=>gen.for(codegen._`${j} = ${i5}; ${j}--;`, ()=>gen.if(codegen._`${eql}(${data}[${i5}], ${data}[${j}])`, ()=>{
                            cxt.error();
                            gen.assign(valid, false).break(outer);
                        })
                    )
                );
            }
        }
    };
    exports.default = def2;
});
var _const = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: "must be equal to constant",
        params: ({ schemaCode  })=>codegen._`{allowedValue: ${schemaCode}}`
    };
    const def2 = {
        keyword: "const",
        $data: true,
        error: error1,
        code (cxt) {
            const { gen , data , $data , schemaCode , schema: schema4  } = cxt;
            if ($data || schema4 && typeof schema4 == "object") {
                cxt.fail$data(codegen._`!${(0, util.useFunc)(gen, equal_1.default)}(${data}, ${schemaCode})`);
            } else {
                cxt.fail(codegen._`${schema4} !== ${data}`);
            }
        }
    };
    exports.default = def2;
});
var _enum = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: "must be equal to one of the allowed values",
        params: ({ schemaCode  })=>codegen._`{allowedValues: ${schemaCode}}`
    };
    const def2 = {
        keyword: "enum",
        schemaType: "array",
        $data: true,
        error: error1,
        code (cxt) {
            const { gen , data , $data , schema: schema4 , schemaCode , it  } = cxt;
            if (!$data && schema4.length === 0) throw new Error("enum must have non-empty array");
            const useLoop = schema4.length >= it.opts.loopEnum;
            const eql = (0, util.useFunc)(gen, equal_1.default);
            let valid;
            if (useLoop || $data) {
                valid = gen.let("valid");
                cxt.block$data(valid, loopEnum);
            } else {
                if (!Array.isArray(schema4)) throw new Error("ajv implementation error");
                const vSchema = gen.const("vSchema", schemaCode);
                valid = (0, codegen.or)(...schema4.map((_x, i5)=>equalCode(vSchema, i5)
                ));
            }
            cxt.pass(valid);
            function loopEnum() {
                gen.assign(valid, false);
                gen.forOf("v", schemaCode, (v)=>gen.if(codegen._`${eql}(${data}, ${v})`, ()=>gen.assign(valid, true).break()
                    )
                );
            }
            function equalCode(vSchema, i5) {
                const sch = schema4[i5];
                return typeof sch === "object" && sch !== null ? codegen._`${eql}(${data}, ${vSchema}[${i5}])` : codegen._`${data} === ${sch}`;
            }
        }
    };
    exports.default = def2;
});
var validation_1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const validation = [
        limitNumber.default,
        multipleOf.default,
        limitLength.default,
        pattern.default,
        limitProperties.default,
        required$1.default,
        limitItems.default,
        uniqueItems.default,
        {
            keyword: "type",
            schemaType: [
                "string",
                "array"
            ]
        },
        {
            keyword: "nullable",
            schemaType: "boolean"
        },
        _const.default,
        _enum.default
    ];
    exports.default = validation;
});
var additionalItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateAdditionalItems = void 0;
    const error1 = {
        message: ({ params: { len  }  })=>codegen.str`must NOT have more than ${len} items`
        ,
        params: ({ params: { len  }  })=>codegen._`{limit: ${len}}`
    };
    const def2 = {
        keyword: "additionalItems",
        type: "array",
        schemaType: [
            "boolean",
            "object"
        ],
        before: "uniqueItems",
        error: error1,
        code (cxt) {
            const { parentSchema , it  } = cxt;
            const { items: items2  } = parentSchema;
            if (!Array.isArray(items2)) {
                (0, util.checkStrictMode)(it, '"additionalItems" is ignored when "items" is not an array of schemas');
                return;
            }
            validateAdditionalItems(cxt, items2);
        }
    };
    function validateAdditionalItems(cxt, items2) {
        const { gen , schema: schema4 , data , keyword: keyword2 , it  } = cxt;
        it.items = true;
        const len = gen.const("len", codegen._`${data}.length`);
        if (schema4 === false) {
            cxt.setParams({
                len: items2.length
            });
            cxt.pass(codegen._`${len} <= ${items2.length}`);
        } else if (typeof schema4 == "object" && !(0, util.alwaysValidSchema)(it, schema4)) {
            const valid = gen.var("valid", codegen._`${len} <= ${items2.length}`);
            gen.if((0, codegen.not)(valid), ()=>validateItems(valid)
            );
            cxt.ok(valid);
        }
        function validateItems(valid) {
            gen.forRange("i", items2.length, len, (i5)=>{
                cxt.subschema({
                    keyword: keyword2,
                    dataProp: i5,
                    dataPropType: util.Type.Num
                }, valid);
                if (!it.allErrors) gen.if((0, codegen.not)(valid), ()=>gen.break()
                );
            });
        }
    }
    exports.validateAdditionalItems = validateAdditionalItems;
    exports.default = def2;
});
var items = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateTuple = void 0;
    const def2 = {
        keyword: "items",
        type: "array",
        schemaType: [
            "object",
            "array",
            "boolean"
        ],
        before: "uniqueItems",
        code (cxt) {
            const { schema: schema4 , it  } = cxt;
            if (Array.isArray(schema4)) return validateTuple(cxt, "additionalItems", schema4);
            it.items = true;
            if ((0, util.alwaysValidSchema)(it, schema4)) return;
            cxt.ok((0, code$1.validateArray)(cxt));
        }
    };
    function validateTuple(cxt, extraItems, schArr = cxt.schema) {
        const { gen , parentSchema , data , keyword: keyword2 , it  } = cxt;
        checkStrictTuple(parentSchema);
        if (it.opts.unevaluated && schArr.length && it.items !== true) {
            it.items = util.mergeEvaluated.items(gen, schArr.length, it.items);
        }
        const valid = gen.name("valid");
        const len = gen.const("len", codegen._`${data}.length`);
        schArr.forEach((sch, i5)=>{
            if ((0, util.alwaysValidSchema)(it, sch)) return;
            gen.if(codegen._`${len} > ${i5}`, ()=>cxt.subschema({
                    keyword: keyword2,
                    schemaProp: i5,
                    dataProp: i5
                }, valid)
            );
            cxt.ok(valid);
        });
        function checkStrictTuple(sch) {
            const { opts , errSchemaPath  } = it;
            const l = schArr.length;
            const fullTuple = l === sch.minItems && (l === sch.maxItems || sch[extraItems] === false);
            if (opts.strictTuples && !fullTuple) {
                const msg = `"${keyword2}" is ${l}-tuple, but minItems or maxItems/${extraItems} are not specified or different at path "${errSchemaPath}"`;
                (0, util.checkStrictMode)(it, msg, opts.strictTuples);
            }
        }
    }
    exports.validateTuple = validateTuple;
    exports.default = def2;
});
var prefixItems = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "prefixItems",
        type: "array",
        schemaType: [
            "array"
        ],
        before: "uniqueItems",
        code: (cxt)=>(0, items.validateTuple)(cxt, "items")
    };
    exports.default = def2;
});
var items2020 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params: { len  }  })=>codegen.str`must NOT have more than ${len} items`
        ,
        params: ({ params: { len  }  })=>codegen._`{limit: ${len}}`
    };
    const def2 = {
        keyword: "items",
        type: "array",
        schemaType: [
            "object",
            "boolean"
        ],
        before: "uniqueItems",
        error: error1,
        code (cxt) {
            const { schema: schema4 , parentSchema , it  } = cxt;
            const { prefixItems: prefixItems2  } = parentSchema;
            it.items = true;
            if ((0, util.alwaysValidSchema)(it, schema4)) return;
            if (prefixItems2) (0, additionalItems.validateAdditionalItems)(cxt, prefixItems2);
            else cxt.ok((0, code$1.validateArray)(cxt));
        }
    };
    exports.default = def2;
});
var contains = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params: { min: min1 , max  }  })=>max === void 0 ? codegen.str`must contain at least ${min1} valid item(s)` : codegen.str`must contain at least ${min1} and no more than ${max} valid item(s)`
        ,
        params: ({ params: { min: min1 , max  }  })=>max === void 0 ? codegen._`{minContains: ${min1}}` : codegen._`{minContains: ${min1}, maxContains: ${max}}`
    };
    const def2 = {
        keyword: "contains",
        type: "array",
        schemaType: [
            "object",
            "boolean"
        ],
        before: "uniqueItems",
        trackErrors: true,
        error: error1,
        code (cxt) {
            const { gen , schema: schema4 , parentSchema , data , it  } = cxt;
            let min1;
            let max;
            const { minContains , maxContains  } = parentSchema;
            if (it.opts.next) {
                min1 = minContains === void 0 ? 1 : minContains;
                max = maxContains;
            } else {
                min1 = 1;
            }
            const len = gen.const("len", codegen._`${data}.length`);
            cxt.setParams({
                min: min1,
                max
            });
            if (max === void 0 && min1 === 0) {
                (0, util.checkStrictMode)(it, `"minContains" == 0 without "maxContains": "contains" keyword ignored`);
                return;
            }
            if (max !== void 0 && min1 > max) {
                (0, util.checkStrictMode)(it, `"minContains" > "maxContains" is always invalid`);
                cxt.fail();
                return;
            }
            if ((0, util.alwaysValidSchema)(it, schema4)) {
                let cond = codegen._`${len} >= ${min1}`;
                if (max !== void 0) cond = codegen._`${cond} && ${len} <= ${max}`;
                cxt.pass(cond);
                return;
            }
            it.items = true;
            const valid = gen.name("valid");
            if (max === void 0 && min1 === 1) {
                validateItems(valid, ()=>gen.if(valid, ()=>gen.break()
                    )
                );
            } else if (min1 === 0) {
                gen.let(valid, true);
                if (max !== void 0) gen.if(codegen._`${data}.length > 0`, validateItemsWithCount);
            } else {
                gen.let(valid, false);
                validateItemsWithCount();
            }
            cxt.result(valid, ()=>cxt.reset()
            );
            function validateItemsWithCount() {
                const schValid = gen.name("_valid");
                const count = gen.let("count", 0);
                validateItems(schValid, ()=>gen.if(schValid, ()=>checkLimits(count)
                    )
                );
            }
            function validateItems(_valid, block) {
                gen.forRange("i", 0, len, (i5)=>{
                    cxt.subschema({
                        keyword: "contains",
                        dataProp: i5,
                        dataPropType: util.Type.Num,
                        compositeRule: true
                    }, _valid);
                    block();
                });
            }
            function checkLimits(count) {
                gen.code(codegen._`${count}++`);
                if (max === void 0) {
                    gen.if(codegen._`${count} >= ${min1}`, ()=>gen.assign(valid, true).break()
                    );
                } else {
                    gen.if(codegen._`${count} > ${max}`, ()=>gen.assign(valid, false).break()
                    );
                    if (min1 === 1) gen.assign(valid, true);
                    else gen.if(codegen._`${count} >= ${min1}`, ()=>gen.assign(valid, true)
                    );
                }
            }
        }
    };
    exports.default = def2;
});
var dependencies = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.validateSchemaDeps = exports.validatePropertyDeps = exports.error = void 0;
    exports.error = {
        message: ({ params: { property , depsCount , deps  }  })=>{
            const property_ies = depsCount === 1 ? "property" : "properties";
            return codegen.str`must have ${property_ies} ${deps} when property ${property} is present`;
        },
        params: ({ params: { property , depsCount , deps , missingProperty  }  })=>codegen._`{property: ${property},
    missingProperty: ${missingProperty},
    depsCount: ${depsCount},
    deps: ${deps}}`
    };
    const def2 = {
        keyword: "dependencies",
        type: "object",
        schemaType: "object",
        error: exports.error,
        code (cxt) {
            const [propDeps, schDeps] = splitDependencies(cxt);
            validatePropertyDeps(cxt, propDeps);
            validateSchemaDeps(cxt, schDeps);
        }
    };
    function splitDependencies({ schema: schema4  }) {
        const propertyDeps = {
        };
        const schemaDeps = {
        };
        for(const key in schema4){
            if (key === "__proto__") continue;
            const deps = Array.isArray(schema4[key]) ? propertyDeps : schemaDeps;
            deps[key] = schema4[key];
        }
        return [
            propertyDeps,
            schemaDeps
        ];
    }
    function validatePropertyDeps(cxt, propertyDeps = cxt.schema) {
        const { gen , data , it  } = cxt;
        if (Object.keys(propertyDeps).length === 0) return;
        const missing = gen.let("missing");
        for(const prop in propertyDeps){
            const deps = propertyDeps[prop];
            if (deps.length === 0) continue;
            const hasProperty = (0, code$1.propertyInData)(gen, data, prop, it.opts.ownProperties);
            cxt.setParams({
                property: prop,
                depsCount: deps.length,
                deps: deps.join(", ")
            });
            if (it.allErrors) {
                gen.if(hasProperty, ()=>{
                    for (const depProp of deps){
                        (0, code$1.checkReportMissingProp)(cxt, depProp);
                    }
                });
            } else {
                gen.if(codegen._`${hasProperty} && (${(0, code$1.checkMissingProp)(cxt, deps, missing)})`);
                (0, code$1.reportMissingProp)(cxt, missing);
                gen.else();
            }
        }
    }
    exports.validatePropertyDeps = validatePropertyDeps;
    function validateSchemaDeps(cxt, schemaDeps = cxt.schema) {
        const { gen , data , keyword: keyword2 , it  } = cxt;
        const valid = gen.name("valid");
        for(const prop in schemaDeps){
            if ((0, util.alwaysValidSchema)(it, schemaDeps[prop])) continue;
            gen.if((0, code$1.propertyInData)(gen, data, prop, it.opts.ownProperties), ()=>{
                const schCxt = cxt.subschema({
                    keyword: keyword2,
                    schemaProp: prop
                }, valid);
                cxt.mergeValidEvaluated(schCxt, valid);
            }, ()=>gen.var(valid, true)
            );
            cxt.ok(valid);
        }
    }
    exports.validateSchemaDeps = validateSchemaDeps;
    exports.default = def2;
});
var propertyNames = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: "property name must be valid",
        params: ({ params  })=>codegen._`{propertyName: ${params.propertyName}}`
    };
    const def2 = {
        keyword: "propertyNames",
        type: "object",
        schemaType: [
            "object",
            "boolean"
        ],
        error: error1,
        code (cxt) {
            const { gen , schema: schema4 , data , it  } = cxt;
            if ((0, util.alwaysValidSchema)(it, schema4)) return;
            const valid = gen.name("valid");
            gen.forIn("key", data, (key)=>{
                cxt.setParams({
                    propertyName: key
                });
                cxt.subschema({
                    keyword: "propertyNames",
                    data: key,
                    dataTypes: [
                        "string"
                    ],
                    propertyName: key,
                    compositeRule: true
                }, valid);
                gen.if((0, codegen.not)(valid), ()=>{
                    cxt.error(true);
                    if (!it.allErrors) gen.break();
                });
            });
            cxt.ok(valid);
        }
    };
    exports.default = def2;
});
var additionalProperties$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: "must NOT have additional properties",
        params: ({ params  })=>codegen._`{additionalProperty: ${params.additionalProperty}}`
    };
    const def2 = {
        keyword: "additionalProperties",
        type: [
            "object"
        ],
        schemaType: [
            "boolean",
            "object"
        ],
        allowUndefined: true,
        trackErrors: true,
        error: error1,
        code (cxt) {
            const { gen , schema: schema4 , parentSchema , data , errsCount , it  } = cxt;
            if (!errsCount) throw new Error("ajv implementation error");
            const { allErrors , opts  } = it;
            it.props = true;
            if (opts.removeAdditional !== "all" && (0, util.alwaysValidSchema)(it, schema4)) return;
            const props = (0, code$1.allSchemaProperties)(parentSchema.properties);
            const patProps = (0, code$1.allSchemaProperties)(parentSchema.patternProperties);
            checkAdditionalProperties();
            cxt.ok(codegen._`${errsCount} === ${names_1.default.errors}`);
            function checkAdditionalProperties() {
                gen.forIn("key", data, (key)=>{
                    if (!props.length && !patProps.length) additionalPropertyCode(key);
                    else gen.if(isAdditional(key), ()=>additionalPropertyCode(key)
                    );
                });
            }
            function isAdditional(key) {
                let definedProp;
                if (props.length > 8) {
                    const propsSchema = (0, util.schemaRefOrVal)(it, parentSchema.properties, "properties");
                    definedProp = (0, code$1.isOwnProperty)(gen, propsSchema, key);
                } else if (props.length) {
                    definedProp = (0, codegen.or)(...props.map((p)=>codegen._`${key} === ${p}`
                    ));
                } else {
                    definedProp = codegen.nil;
                }
                if (patProps.length) {
                    definedProp = (0, codegen.or)(definedProp, ...patProps.map((p)=>codegen._`${(0, code$1.usePattern)(cxt, p)}.test(${key})`
                    ));
                }
                return (0, codegen.not)(definedProp);
            }
            function deleteAdditional(key) {
                gen.code(codegen._`delete ${data}[${key}]`);
            }
            function additionalPropertyCode(key) {
                if (opts.removeAdditional === "all" || opts.removeAdditional && schema4 === false) {
                    deleteAdditional(key);
                    return;
                }
                if (schema4 === false) {
                    cxt.setParams({
                        additionalProperty: key
                    });
                    cxt.error();
                    if (!allErrors) gen.break();
                    return;
                }
                if (typeof schema4 == "object" && !(0, util.alwaysValidSchema)(it, schema4)) {
                    const valid = gen.name("valid");
                    if (opts.removeAdditional === "failing") {
                        applyAdditionalSchema(key, valid, false);
                        gen.if((0, codegen.not)(valid), ()=>{
                            cxt.reset();
                            deleteAdditional(key);
                        });
                    } else {
                        applyAdditionalSchema(key, valid);
                        if (!allErrors) gen.if((0, codegen.not)(valid), ()=>gen.break()
                        );
                    }
                }
            }
            function applyAdditionalSchema(key, valid, errors2) {
                const subschema2 = {
                    keyword: "additionalProperties",
                    dataProp: key,
                    dataPropType: util.Type.Str
                };
                if (errors2 === false) {
                    Object.assign(subschema2, {
                        compositeRule: true,
                        createErrors: false,
                        allErrors: false
                    });
                }
                cxt.subschema(subschema2, valid);
            }
        }
    };
    exports.default = def2;
});
var properties$1 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "properties",
        type: "object",
        schemaType: "object",
        code (cxt) {
            const { gen , schema: schema4 , parentSchema , data , it  } = cxt;
            if (it.opts.removeAdditional === "all" && parentSchema.additionalProperties === void 0) {
                additionalProperties$1.default.code(new validate1.KeywordCxt(it, additionalProperties$1.default, "additionalProperties"));
            }
            const allProps = (0, code$1.allSchemaProperties)(schema4);
            for (const prop of allProps){
                it.definedProperties.add(prop);
            }
            if (it.opts.unevaluated && allProps.length && it.props !== true) {
                it.props = util.mergeEvaluated.props(gen, (0, util.toHash)(allProps), it.props);
            }
            const properties2 = allProps.filter((p)=>!(0, util.alwaysValidSchema)(it, schema4[p])
            );
            if (properties2.length === 0) return;
            const valid = gen.name("valid");
            for (const prop1 of properties2){
                if (hasDefault(prop1)) {
                    applyPropertySchema(prop1);
                } else {
                    gen.if((0, code$1.propertyInData)(gen, data, prop1, it.opts.ownProperties));
                    applyPropertySchema(prop1);
                    if (!it.allErrors) gen.else().var(valid, true);
                    gen.endIf();
                }
                cxt.it.definedProperties.add(prop1);
                cxt.ok(valid);
            }
            function hasDefault(prop2) {
                return it.opts.useDefaults && !it.compositeRule && schema4[prop2].default !== void 0;
            }
            function applyPropertySchema(prop2) {
                cxt.subschema({
                    keyword: "properties",
                    schemaProp: prop2,
                    dataProp: prop2
                }, valid);
            }
        }
    };
    exports.default = def2;
});
var patternProperties = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const util_2 = util;
    const def2 = {
        keyword: "patternProperties",
        type: "object",
        schemaType: "object",
        code (cxt) {
            const { gen , schema: schema4 , data , parentSchema , it  } = cxt;
            const { opts  } = it;
            const patterns = (0, code$1.allSchemaProperties)(schema4);
            const alwaysValidPatterns = patterns.filter((p)=>(0, util.alwaysValidSchema)(it, schema4[p])
            );
            if (patterns.length === 0 || alwaysValidPatterns.length === patterns.length && (!it.opts.unevaluated || it.props === true)) {
                return;
            }
            const checkProperties = opts.strictSchema && !opts.allowMatchingProperties && parentSchema.properties;
            const valid = gen.name("valid");
            if (it.props !== true && !(it.props instanceof codegen.Name)) {
                it.props = (0, util_2.evaluatedPropsToName)(gen, it.props);
            }
            const { props  } = it;
            validatePatternProperties();
            function validatePatternProperties() {
                for (const pat of patterns){
                    if (checkProperties) checkMatchingProperties(pat);
                    if (it.allErrors) {
                        validateProperties(pat);
                    } else {
                        gen.var(valid, true);
                        validateProperties(pat);
                        gen.if(valid);
                    }
                }
            }
            function checkMatchingProperties(pat) {
                for(const prop in checkProperties){
                    if (new RegExp(pat).test(prop)) {
                        (0, util.checkStrictMode)(it, `property ${prop} matches pattern ${pat} (use allowMatchingProperties)`);
                    }
                }
            }
            function validateProperties(pat) {
                gen.forIn("key", data, (key)=>{
                    gen.if(codegen._`${(0, code$1.usePattern)(cxt, pat)}.test(${key})`, ()=>{
                        const alwaysValid = alwaysValidPatterns.includes(pat);
                        if (!alwaysValid) {
                            cxt.subschema({
                                keyword: "patternProperties",
                                schemaProp: pat,
                                dataProp: key,
                                dataPropType: util_2.Type.Str
                            }, valid);
                        }
                        if (it.opts.unevaluated && props !== true) {
                            gen.assign(codegen._`${props}[${key}]`, true);
                        } else if (!alwaysValid && !it.allErrors) {
                            gen.if((0, codegen.not)(valid), ()=>gen.break()
                            );
                        }
                    });
                });
            }
        }
    };
    exports.default = def2;
});
var not = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "not",
        schemaType: [
            "object",
            "boolean"
        ],
        trackErrors: true,
        code (cxt) {
            const { gen , schema: schema4 , it  } = cxt;
            if ((0, util.alwaysValidSchema)(it, schema4)) {
                cxt.fail();
                return;
            }
            const valid = gen.name("valid");
            cxt.subschema({
                keyword: "not",
                compositeRule: true,
                createErrors: false,
                allErrors: false
            }, valid);
            cxt.failResult(valid, ()=>cxt.reset()
            , ()=>cxt.error()
            );
        },
        error: {
            message: "must NOT be valid"
        }
    };
    exports.default = def2;
});
var anyOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "anyOf",
        schemaType: "array",
        trackErrors: true,
        code: code$1.validateUnion,
        error: {
            message: "must match a schema in anyOf"
        }
    };
    exports.default = def2;
});
var oneOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: "must match exactly one schema in oneOf",
        params: ({ params  })=>codegen._`{passingSchemas: ${params.passing}}`
    };
    const def2 = {
        keyword: "oneOf",
        schemaType: "array",
        trackErrors: true,
        error: error1,
        code (cxt) {
            const { gen , schema: schema4 , parentSchema , it  } = cxt;
            if (!Array.isArray(schema4)) throw new Error("ajv implementation error");
            if (it.opts.discriminator && parentSchema.discriminator) return;
            const schArr = schema4;
            const valid = gen.let("valid", false);
            const passing = gen.let("passing", null);
            const schValid = gen.name("_valid");
            cxt.setParams({
                passing
            });
            gen.block(validateOneOf);
            cxt.result(valid, ()=>cxt.reset()
            , ()=>cxt.error(true)
            );
            function validateOneOf() {
                schArr.forEach((sch, i5)=>{
                    let schCxt;
                    if ((0, util.alwaysValidSchema)(it, sch)) {
                        gen.var(schValid, true);
                    } else {
                        schCxt = cxt.subschema({
                            keyword: "oneOf",
                            schemaProp: i5,
                            compositeRule: true
                        }, schValid);
                    }
                    if (i5 > 0) {
                        gen.if(codegen._`${schValid} && ${valid}`).assign(valid, false).assign(passing, codegen._`[${passing}, ${i5}]`).else();
                    }
                    gen.if(schValid, ()=>{
                        gen.assign(valid, true);
                        gen.assign(passing, i5);
                        if (schCxt) cxt.mergeEvaluated(schCxt, codegen.Name);
                    });
                });
            }
        }
    };
    exports.default = def2;
});
var allOf = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: "allOf",
        schemaType: "array",
        code (cxt) {
            const { gen , schema: schema4 , it  } = cxt;
            if (!Array.isArray(schema4)) throw new Error("ajv implementation error");
            const valid = gen.name("valid");
            schema4.forEach((sch, i5)=>{
                if ((0, util.alwaysValidSchema)(it, sch)) return;
                const schCxt = cxt.subschema({
                    keyword: "allOf",
                    schemaProp: i5
                }, valid);
                cxt.ok(valid);
                cxt.mergeEvaluated(schCxt);
            });
        }
    };
    exports.default = def2;
});
var _if = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params  })=>codegen.str`must match "${params.ifClause}" schema`
        ,
        params: ({ params  })=>codegen._`{failingKeyword: ${params.ifClause}}`
    };
    const def2 = {
        keyword: "if",
        schemaType: [
            "object",
            "boolean"
        ],
        trackErrors: true,
        error: error1,
        code (cxt) {
            const { gen , parentSchema , it  } = cxt;
            if (parentSchema.then === void 0 && parentSchema.else === void 0) {
                (0, util.checkStrictMode)(it, '"if" without "then" and "else" is ignored');
            }
            const hasThen = hasSchema(it, "then");
            const hasElse = hasSchema(it, "else");
            if (!hasThen && !hasElse) return;
            const valid = gen.let("valid", true);
            const schValid = gen.name("_valid");
            validateIf();
            cxt.reset();
            if (hasThen && hasElse) {
                const ifClause = gen.let("ifClause");
                cxt.setParams({
                    ifClause
                });
                gen.if(schValid, validateClause("then", ifClause), validateClause("else", ifClause));
            } else if (hasThen) {
                gen.if(schValid, validateClause("then"));
            } else {
                gen.if((0, codegen.not)(schValid), validateClause("else"));
            }
            cxt.pass(valid, ()=>cxt.error(true)
            );
            function validateIf() {
                const schCxt = cxt.subschema({
                    keyword: "if",
                    compositeRule: true,
                    createErrors: false,
                    allErrors: false
                }, schValid);
                cxt.mergeEvaluated(schCxt);
            }
            function validateClause(keyword2, ifClause) {
                return ()=>{
                    const schCxt = cxt.subschema({
                        keyword: keyword2
                    }, schValid);
                    gen.assign(valid, schValid);
                    cxt.mergeValidEvaluated(schCxt, valid);
                    if (ifClause) gen.assign(ifClause, codegen._`${keyword2}`);
                    else cxt.setParams({
                        ifClause: keyword2
                    });
                };
            }
        }
    };
    function hasSchema(it, keyword2) {
        const schema4 = it.schema[keyword2];
        return schema4 !== void 0 && !(0, util.alwaysValidSchema)(it, schema4);
    }
    exports.default = def2;
});
var thenElse = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const def2 = {
        keyword: [
            "then",
            "else"
        ],
        schemaType: [
            "object",
            "boolean"
        ],
        code ({ keyword: keyword2 , parentSchema , it  }) {
            if (parentSchema.if === void 0) (0, util.checkStrictMode)(it, `"${keyword2}" without "if" is ignored`);
        }
    };
    exports.default = def2;
});
var applicator = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    function getApplicator(draft2020 = false) {
        const applicator2 = [
            not.default,
            anyOf.default,
            oneOf.default,
            allOf.default,
            _if.default,
            thenElse.default,
            propertyNames.default,
            additionalProperties$1.default,
            dependencies.default,
            properties$1.default,
            patternProperties.default
        ];
        if (draft2020) applicator2.push(prefixItems.default, items2020.default);
        else applicator2.push(additionalItems.default, items.default);
        applicator2.push(contains.default);
        return applicator2;
    }
    exports.default = getApplicator;
});
var format5 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ schemaCode  })=>codegen.str`must match format "${schemaCode}"`
        ,
        params: ({ schemaCode  })=>codegen._`{format: ${schemaCode}}`
    };
    const def2 = {
        keyword: "format",
        type: [
            "number",
            "string"
        ],
        schemaType: "string",
        $data: true,
        error: error1,
        code (cxt, ruleType) {
            const { gen , data , $data , schema: schema4 , schemaCode , it  } = cxt;
            const { opts , errSchemaPath , schemaEnv , self  } = it;
            if (!opts.validateFormats) return;
            if ($data) validate$DataFormat();
            else validateFormat();
            function validate$DataFormat() {
                const fmts = gen.scopeValue("formats", {
                    ref: self.formats,
                    code: opts.code.formats
                });
                const fDef = gen.const("fDef", codegen._`${fmts}[${schemaCode}]`);
                const fType = gen.let("fType");
                const format22 = gen.let("format");
                gen.if(codegen._`typeof ${fDef} == "object" && !(${fDef} instanceof RegExp)`, ()=>gen.assign(fType, codegen._`${fDef}.type || "string"`).assign(format22, codegen._`${fDef}.validate`)
                , ()=>gen.assign(fType, codegen._`"string"`).assign(format22, fDef)
                );
                cxt.fail$data((0, codegen.or)(unknownFmt(), invalidFmt()));
                function unknownFmt() {
                    if (opts.strictSchema === false) return codegen.nil;
                    return codegen._`${schemaCode} && !${format22}`;
                }
                function invalidFmt() {
                    const callFormat = schemaEnv.$async ? codegen._`(${fDef}.async ? await ${format22}(${data}) : ${format22}(${data}))` : codegen._`${format22}(${data})`;
                    const validData = codegen._`(typeof ${format22} == "function" ? ${callFormat} : ${format22}.test(${data}))`;
                    return codegen._`${format22} && ${format22} !== true && ${fType} === ${ruleType} && !${validData}`;
                }
            }
            function validateFormat() {
                const formatDef = self.formats[schema4];
                if (!formatDef) {
                    unknownFormat();
                    return;
                }
                if (formatDef === true) return;
                const [fmtType, format22, fmtRef] = getFormat(formatDef);
                if (fmtType === ruleType) cxt.pass(validCondition());
                function unknownFormat() {
                    if (opts.strictSchema === false) {
                        self.logger.warn(unknownMsg());
                        return;
                    }
                    throw new Error(unknownMsg());
                    function unknownMsg() {
                        return `unknown format "${schema4}" ignored in schema at path "${errSchemaPath}"`;
                    }
                }
                function getFormat(fmtDef) {
                    const code21 = fmtDef instanceof RegExp ? (0, codegen.regexpCode)(fmtDef) : opts.code.formats ? codegen._`${opts.code.formats}${(0, codegen.getProperty)(schema4)}` : void 0;
                    const fmt = gen.scopeValue("formats", {
                        key: schema4,
                        ref: fmtDef,
                        code: code21
                    });
                    if (typeof fmtDef == "object" && !(fmtDef instanceof RegExp)) {
                        return [
                            fmtDef.type || "string",
                            fmtDef.validate,
                            codegen._`${fmt}.validate`
                        ];
                    }
                    return [
                        "string",
                        fmtDef,
                        fmt
                    ];
                }
                function validCondition() {
                    if (typeof formatDef == "object" && !(formatDef instanceof RegExp) && formatDef.async) {
                        if (!schemaEnv.$async) throw new Error("async format in sync schema");
                        return codegen._`await ${fmtRef}(${data})`;
                    }
                    return typeof format22 == "function" ? codegen._`${fmtRef}(${data})` : codegen._`${fmtRef}.test(${data})`;
                }
            }
        }
    };
    exports.default = def2;
});
var format_2 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const format$1 = [
        format5.default
    ];
    exports.default = format$1;
});
var metadata = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.contentVocabulary = exports.metadataVocabulary = void 0;
    exports.metadataVocabulary = [
        "title",
        "description",
        "default",
        "deprecated",
        "readOnly",
        "writeOnly",
        "examples"
    ];
    exports.contentVocabulary = [
        "contentMediaType",
        "contentEncoding",
        "contentSchema"
    ];
});
var draft7 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const draft7Vocabularies = [
        core_1.default,
        validation_1.default,
        (0, applicator.default)(),
        format_2.default,
        metadata.metadataVocabulary,
        metadata.contentVocabulary
    ];
    exports.default = draft7Vocabularies;
});
var types2 = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.DiscrError = void 0;
    (function(DiscrError) {
        DiscrError["Tag"] = "tag";
        DiscrError["Mapping"] = "mapping";
    })(exports.DiscrError || (exports.DiscrError = {
    }));
});
var discriminator = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    const error1 = {
        message: ({ params: { discrError , tagName  }  })=>discrError === types2.DiscrError.Tag ? `tag "${tagName}" must be string` : `value of tag "${tagName}" must be in oneOf`
        ,
        params: ({ params: { discrError , tag: tag2 , tagName  }  })=>codegen._`{error: ${discrError}, tag: ${tagName}, tagValue: ${tag2}}`
    };
    const def2 = {
        keyword: "discriminator",
        type: "object",
        schemaType: "object",
        error: error1,
        code (cxt) {
            const { gen , data , schema: schema4 , parentSchema , it  } = cxt;
            const { oneOf: oneOf2  } = parentSchema;
            if (!it.opts.discriminator) {
                throw new Error("discriminator: requires discriminator option");
            }
            const tagName = schema4.propertyName;
            if (typeof tagName != "string") throw new Error("discriminator: requires propertyName");
            if (schema4.mapping) throw new Error("discriminator: mapping is not supported");
            if (!oneOf2) throw new Error("discriminator: requires oneOf keyword");
            const valid = gen.let("valid", false);
            const tag2 = gen.const("tag", codegen._`${data}${(0, codegen.getProperty)(tagName)}`);
            gen.if(codegen._`typeof ${tag2} == "string"`, ()=>validateMapping()
            , ()=>cxt.error(false, {
                    discrError: types2.DiscrError.Tag,
                    tag: tag2,
                    tagName
                })
            );
            cxt.ok(valid);
            function validateMapping() {
                const mapping = getMapping();
                gen.if(false);
                for(const tagValue in mapping){
                    gen.elseIf(codegen._`${tag2} === ${tagValue}`);
                    gen.assign(valid, applyTagSchema(mapping[tagValue]));
                }
                gen.else();
                cxt.error(false, {
                    discrError: types2.DiscrError.Mapping,
                    tag: tag2,
                    tagName
                });
                gen.endIf();
            }
            function applyTagSchema(schemaProp) {
                const _valid = gen.name("valid");
                const schCxt = cxt.subschema({
                    keyword: "oneOf",
                    schemaProp
                }, _valid);
                cxt.mergeEvaluated(schCxt, codegen.Name);
                return _valid;
            }
            function getMapping() {
                var _a;
                const oneOfMapping = {
                };
                const topRequired = hasRequired(parentSchema);
                let tagRequired = true;
                for(let i5 = 0; i5 < oneOf2.length; i5++){
                    const sch = oneOf2[i5];
                    const propSch = (_a = sch.properties) === null || _a === void 0 ? void 0 : _a[tagName];
                    if (typeof propSch != "object") {
                        throw new Error(`discriminator: oneOf schemas must have "properties/${tagName}"`);
                    }
                    tagRequired = tagRequired && (topRequired || hasRequired(sch));
                    addMappings(propSch, i5);
                }
                if (!tagRequired) throw new Error(`discriminator: "${tagName}" must be required`);
                return oneOfMapping;
                function hasRequired({ required: required2  }) {
                    return Array.isArray(required2) && required2.includes(tagName);
                }
                function addMappings(sch, i6) {
                    if (sch.const) {
                        addMapping(sch.const, i6);
                    } else if (sch.enum) {
                        for (const tagValue of sch.enum){
                            addMapping(tagValue, i6);
                        }
                    } else {
                        throw new Error(`discriminator: "properties/${tagName}" must have "const" or "enum"`);
                    }
                }
                function addMapping(tagValue, i6) {
                    if (typeof tagValue != "string" || tagValue in oneOfMapping) {
                        throw new Error(`discriminator: "${tagName}" values must be unique strings`);
                    }
                    oneOfMapping[tagValue] = i6;
                }
            }
        }
    };
    exports.default = def2;
});
const $schema = "http://json-schema.org/draft-07/schema#";
const $id$1 = "http://json-schema.org/draft-07/schema#";
const title = "Core schema meta-schema";
const definitions = {
    schemaArray: {
        type: "array",
        minItems: 1,
        items: {
            $ref: "#"
        }
    },
    nonNegativeInteger: {
        type: "integer",
        minimum: 0
    },
    nonNegativeIntegerDefault0: {
        allOf: [
            {
                $ref: "#/definitions/nonNegativeInteger"
            },
            {
                default: 0
            }
        ]
    },
    simpleTypes: {
        enum: [
            "array",
            "boolean",
            "integer",
            "null",
            "number",
            "object",
            "string"
        ]
    },
    stringArray: {
        type: "array",
        items: {
            type: "string"
        },
        uniqueItems: true,
        default: []
    }
};
const type$1 = [
    "object",
    "boolean"
];
const properties$2 = {
    $id: {
        type: "string",
        format: "uri-reference"
    },
    $schema: {
        type: "string",
        format: "uri"
    },
    $ref: {
        type: "string",
        format: "uri-reference"
    },
    $comment: {
        type: "string"
    },
    title: {
        type: "string"
    },
    description: {
        type: "string"
    },
    default: true,
    readOnly: {
        type: "boolean",
        default: false
    },
    examples: {
        type: "array",
        items: true
    },
    multipleOf: {
        type: "number",
        exclusiveMinimum: 0
    },
    maximum: {
        type: "number"
    },
    exclusiveMaximum: {
        type: "number"
    },
    minimum: {
        type: "number"
    },
    exclusiveMinimum: {
        type: "number"
    },
    maxLength: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minLength: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    pattern: {
        type: "string",
        format: "regex"
    },
    additionalItems: {
        $ref: "#"
    },
    items: {
        anyOf: [
            {
                $ref: "#"
            },
            {
                $ref: "#/definitions/schemaArray"
            }
        ],
        default: true
    },
    maxItems: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minItems: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    uniqueItems: {
        type: "boolean",
        default: false
    },
    contains: {
        $ref: "#"
    },
    maxProperties: {
        $ref: "#/definitions/nonNegativeInteger"
    },
    minProperties: {
        $ref: "#/definitions/nonNegativeIntegerDefault0"
    },
    required: {
        $ref: "#/definitions/stringArray"
    },
    additionalProperties: {
        $ref: "#"
    },
    definitions: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        default: {
        }
    },
    properties: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        default: {
        }
    },
    patternProperties: {
        type: "object",
        additionalProperties: {
            $ref: "#"
        },
        propertyNames: {
            format: "regex"
        },
        default: {
        }
    },
    dependencies: {
        type: "object",
        additionalProperties: {
            anyOf: [
                {
                    $ref: "#"
                },
                {
                    $ref: "#/definitions/stringArray"
                }
            ]
        }
    },
    propertyNames: {
        $ref: "#"
    },
    const: true,
    enum: {
        type: "array",
        items: true,
        minItems: 1,
        uniqueItems: true
    },
    type: {
        anyOf: [
            {
                $ref: "#/definitions/simpleTypes"
            },
            {
                type: "array",
                items: {
                    $ref: "#/definitions/simpleTypes"
                },
                minItems: 1,
                uniqueItems: true
            }
        ]
    },
    format: {
        type: "string"
    },
    contentMediaType: {
        type: "string"
    },
    contentEncoding: {
        type: "string"
    },
    if: {
        $ref: "#"
    },
    then: {
        $ref: "#"
    },
    else: {
        $ref: "#"
    },
    allOf: {
        $ref: "#/definitions/schemaArray"
    },
    anyOf: {
        $ref: "#/definitions/schemaArray"
    },
    oneOf: {
        $ref: "#/definitions/schemaArray"
    },
    not: {
        $ref: "#"
    }
};
var draft7MetaSchema = {
    $schema,
    $id: $id$1,
    title,
    definitions,
    type: type$1,
    properties: properties$2,
    default: true
};
var ajv = createCommonjsModule2(function(module, exports) {
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.CodeGen = exports.Name = exports.nil = exports.stringify = exports.str = exports._ = exports.KeywordCxt = void 0;
    const META_SUPPORT_DATA = [
        "/properties"
    ];
    const META_SCHEMA_ID = "http://json-schema.org/draft-07/schema";
    class Ajv extends core1.default {
        _addVocabularies() {
            super._addVocabularies();
            draft7.default.forEach((v)=>this.addVocabulary(v)
            );
            if (this.opts.discriminator) this.addKeyword(discriminator.default);
        }
        _addDefaultMetaSchema() {
            super._addDefaultMetaSchema();
            if (!this.opts.meta) return;
            const metaSchema = this.opts.$data ? this.$dataMetaSchema(draft7MetaSchema, META_SUPPORT_DATA) : draft7MetaSchema;
            this.addMetaSchema(metaSchema, META_SCHEMA_ID, false);
            this.refs["http://json-schema.org/schema"] = META_SCHEMA_ID;
        }
        defaultMeta() {
            return this.opts.defaultMeta = super.defaultMeta() || (this.getSchema(META_SCHEMA_ID) ? META_SCHEMA_ID : void 0);
        }
    }
    module.exports = exports = Ajv;
    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.default = Ajv;
    Object.defineProperty(exports, "KeywordCxt", {
        enumerable: true,
        get: function() {
            return validate1.KeywordCxt;
        }
    });
    Object.defineProperty(exports, "_", {
        enumerable: true,
        get: function() {
            return codegen._;
        }
    });
    Object.defineProperty(exports, "str", {
        enumerable: true,
        get: function() {
            return codegen.str;
        }
    });
    Object.defineProperty(exports, "stringify", {
        enumerable: true,
        get: function() {
            return codegen.stringify;
        }
    });
    Object.defineProperty(exports, "nil", {
        enumerable: true,
        get: function() {
            return codegen.nil;
        }
    });
    Object.defineProperty(exports, "Name", {
        enumerable: true,
        get: function() {
            return codegen.Name;
        }
    });
    Object.defineProperty(exports, "CodeGen", {
        enumerable: true,
        get: function() {
            return codegen.CodeGen;
        }
    });
});
var __pika_web_default_export_for_treeshaking__1 = getDefaultExportFromCjs1(ajv);
let ajv1 = undefined;
function setupAjv(_ajv) {
    ajv1 = _ajv;
}
function navigate(path3, annotation, returnKey = false, pathIndex = 0) {
    if (pathIndex >= path3.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components  } = annotation;
        const searchKey = path3[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i5 = lastKeyIndex; i5 >= 0; i5 -= 2){
            const key = components[i5].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path3.length - 1) {
                    return navigate(path3, components[i5], returnKey, pathIndex + 1);
                } else {
                    return navigate(path3, components[i5 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        throw new Error("Internal error: searchKey not found in mapping object");
    } else if (annotation.kind === "sequence" || annotation.kind === "block_sequence") {
        const searchKey = Number(path3[pathIndex]);
        return navigate(path3, annotation.components[searchKey], returnKey, pathIndex + 1);
    } else {
        throw new Error(`Internal error: unexpected kind ${annotation.kind}`);
    }
}
function navigateSchema(path3, schema4, pathIndex = 0) {
    if (pathIndex >= path3.length - 1) {
        return schema4;
    }
    const pathVal = path3[pathIndex];
    if (pathVal === "properties") {
        const key = path3[pathIndex + 1];
        const subSchema = schema4.properties[key];
        return navigateSchema(path3, subSchema, pathIndex + 2);
    } else if (pathVal === "anyOf") {
        const key = Number(path3[pathIndex + 1]);
        const subSchema = schema4.anyOf[key];
        return navigateSchema(path3, subSchema, pathIndex + 2);
    } else if (pathVal === "oneOf") {
        const key = Number(path3[pathIndex + 1]);
        const subSchema = schema4.oneOf[key];
        return navigateSchema(path3, subSchema, pathIndex + 2);
    } else if (pathVal === "items") {
        const subSchema = schema4.items;
        return navigateSchema(path3, subSchema, pathIndex + 1);
    } else {
        throw new Error(`Internal error: Failed to navigate schema path ${path3}`);
    }
}
function isProperPrefix(a, b) {
    return b.length > a.length && b.substring(0, a.length) === a;
}
function groupBy(lst, f) {
    const record = {
    };
    const result = [];
    for (const el of lst){
        const key = f(el);
        if (record[key] === undefined) {
            const lst1 = [];
            const entry = {
                key,
                values: lst1
            };
            record[key] = lst1;
            result.push(entry);
        }
        record[key].push(el);
    }
    return result;
}
function groupByEntries(entries) {
    const result = [];
    for (const { values: values2  } of entries){
        result.push(...values2);
    }
    return result;
}
function narrowOneOfError(oneOf1, suberrors) {
    const subschemaErrors = groupBy(suberrors.filter((error1)=>error1.schemaPath !== oneOf1.schemaPath
    ), (error1)=>error1.schemaPath.substring(0, error1.schemaPath.lastIndexOf("/"))
    );
    const onlyAdditionalProperties = subschemaErrors.filter(({ values: values2  })=>values2.every((v)=>v.keyword === "additionalProperties"
        )
    );
    if (onlyAdditionalProperties.length) {
        return onlyAdditionalProperties[0].values;
    }
    return [];
}
function localizeAndPruneErrors(annotation, validationErrors, source, schema4) {
    const result = [];
    const locF = mappedIndexToRowCol(source);
    let errorsPerInstanceList = groupBy(validationErrors, (error1)=>error1.instancePath
    );
    do {
        const newErrors = [];
        errorsPerInstanceList = errorsPerInstanceList.filter(({ key: pathA  })=>errorsPerInstanceList.filter(({ key: pathB  })=>isProperPrefix(pathA, pathB)
            ).length === 0
        );
        for (let { key: instancePath , values: errors1  } of errorsPerInstanceList){
            let errorsPerSchemaList = groupBy(errors1, (error1)=>error1.schemaPath
            );
            errorsPerSchemaList = errorsPerSchemaList.filter(({ key: pathA  })=>errorsPerSchemaList.filter(({ key: pathB  })=>isProperPrefix(pathB, pathA)
                ).length === 0
            );
            for (const error1 of groupByEntries(errorsPerSchemaList)){
                if (error1.hasBeenTransformed) {
                    continue;
                }
                if (error1.keyword === "oneOf") {
                    error1.hasBeenTransformed = true;
                    newErrors.push(...narrowOneOfError(error1, errors1));
                } else if (error1.keyword === "additionalProperties") {
                    error1.hasBeenTransformed = true;
                    instancePath = `${instancePath}/${error1.params.additionalProperty}`;
                    newErrors.push({
                        ...error1,
                        instancePath,
                        keyword: "_custom_invalidProperty",
                        message: `property ${error1.params.additionalProperty} not allowed in object`,
                        params: {
                            ...error1.params,
                            originalError: error1
                        },
                        schemaPath: error1.schemaPath.slice(0, -21)
                    });
                }
            }
        }
        if (newErrors.length) {
            errorsPerInstanceList.push(...groupBy(newErrors, (error1)=>error1.instancePath
            ));
        } else {
            break;
        }
    }while (true)
    for (const { key: instancePath , values: allErrors  } of errorsPerInstanceList){
        const path3 = instancePath.split("/").slice(1);
        const errors1 = allErrors.filter(({ schemaPath: pathA  })=>!(allErrors.filter(({ schemaPath: pathB  })=>isProperPrefix(pathB, pathA)
            ).length > 0)
        );
        for (const error1 of errors1){
            const returnKey = error1.keyword === "_custom_invalidProperty";
            const violatingObject = navigate(path3, annotation, returnKey);
            const schemaPath = error1.schemaPath.split("/").slice(1);
            const start = locF(violatingObject.start);
            const end = locF(violatingObject.end);
            const locStr = start.line === end.line ? `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})` : `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
            let messageNoLocation;
            if (error1.keyword.startsWith("_custom_")) {
                messageNoLocation = error1.message;
            } else {
                const innerSchema = navigateSchema(schemaPath, schema4);
                if (instancePath === "") {
                    messageNoLocation = `(top-level error) ${error1.message}`;
                } else {
                    messageNoLocation = `Field ${instancePath} must ${innerSchema.description}`;
                }
            }
            const message7 = `${locStr}: ${messageNoLocation}`;
            result.push({
                instancePath,
                violatingObject,
                message: message7,
                messageNoLocation,
                source,
                start,
                end,
                error: error1
            });
        }
    }
    result.sort((a, b)=>a.violatingObject.start - b.violatingObject.start
    );
    return result;
}
class YAMLSchema {
    schema;
    validate;
    constructor(schema4){
        this.schema = schema4;
        this.validate = ajv1.compile(normalizeSchema(schema4));
    }
    validateParse(src, annotation) {
        let errors1 = [];
        if (!this.validate(annotation.result)) {
            errors1 = localizeAndPruneErrors(annotation, this.validate.errors, src, this.schema);
            return {
                result: annotation.result,
                errors: errors1
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result, src, message, error, log) {
        if (result.errors.length) {
            const locF = mappedIndexToRowCol(src);
            const nLines = lines(src.originalString).length;
            error(message);
            for (const err of result.errors){
                log(err.message);
                let startO = err.violatingObject.start;
                let endO = err.violatingObject.end;
                while(src.mapClosest(startO) < src.originalString.length - 1 && src.originalString[src.mapClosest(startO)].match(/\s/)){
                    startO++;
                }
                while(src.mapClosest(endO) > src.mapClosest(startO) && src.originalString[src.mapClosest(endO)].match(/\s/)){
                    endO--;
                }
                const start = locF(startO);
                const end = locF(endO);
                const { prefixWidth , lines: lines1 ,  } = formatLineRange(src.originalString, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
                for (const { lineNumber , content , rawLine  } of lines1){
                    log(content);
                    if (lineNumber >= start.line && lineNumber <= end.line) {
                        const startColumn = lineNumber > start.line ? 0 : start.column;
                        const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                        log(" ".repeat(prefixWidth + startColumn) + "^".repeat(endColumn - startColumn + 1));
                    }
                }
            }
        }
        return result;
    }
    validateParseWithErrors(src, annotation, message, error, log) {
        const result = this.validateParse(src, annotation);
        this.reportErrorsInSource(result, src, message, error, log);
        return result;
    }
}
let ajvInit = false;
function ensureAjv() {
    if (!ajvInit) {
        setupAjv(new __pika_web_default_export_for_treeshaking__1({
            allErrors: true
        }));
        ajvInit = true;
    }
}
class PromiseQueue {
    queue = new Array();
    running = false;
    enqueue(promise, clearPending = false) {
        return new Promise((resolve5, reject)=>{
            if (clearPending) {
                this.queue.splice(0, this.queue.length);
            }
            this.queue.push({
                promise,
                resolve: resolve5,
                reject
            });
            this.dequeue();
        });
    }
    dequeue() {
        if (this.running) {
            return false;
        }
        const item = this.queue.shift();
        if (!item) {
            return false;
        }
        try {
            this.running = true;
            item.promise().then((value6)=>{
                this.running = false;
                item.resolve(value6);
                this.dequeue();
            }).catch((err)=>{
                this.running = false;
                item.reject(err);
                this.dequeue();
            });
        } catch (err) {
            this.running = false;
            item.reject(err);
            this.dequeue();
        }
        return true;
    }
}
const yamlValidators = {
};
const validatorQueues = {
};
function getSchemaName(schema5) {
    const schemaName = schema5["$id"];
    if (schemaName === undefined) {
        throw new Error("Expected schema to be named");
    }
    return schemaName;
}
function getValidator(schema5) {
    const schemaName = getSchemaName(schema5);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema5);
    yamlValidators[schemaName] = validator;
    return validator;
}
async function withValidator(schema5, fun) {
    const schemaName = getSchemaName(schema5);
    if (validatorQueues[schemaName] === undefined) {
        validatorQueues[schemaName] = new PromiseQueue();
    }
    const queue = validatorQueues[schemaName];
    let result;
    let error1;
    await queue.enqueue(async ()=>{
        const validator = getValidator(schema5);
        try {
            result = fun(validator);
        } catch (e) {
            console.error("Error in validator queue", e);
            error1 = e;
        }
    });
    if (error1 !== undefined) {
        throw error1;
    }
    return result;
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg1, validationErrors){
        super(msg1);
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
function readAndValidateYamlFromFile(file, schema5, errorMessage) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    const contents = asMappedString(Deno.readTextFileSync(file));
    return readAndValidateYamlFromMappedString(contents, schema5, errorMessage);
}
async function readAndValidateYamlFromMappedString(mappedYaml, schema5, errorMessage) {
    ensureAjv();
    const result = await withValidator(schema5, (validator)=>{
        const annotation = readAnnotatedYamlFromMappedString(mappedYaml);
        const validateYaml = !(annotation.result?.["validate-yaml"] === false);
        const yaml = annotation.result;
        if (validateYaml) {
            const valResult = validator.validateParse(mappedYaml, annotation);
            if (valResult.errors.length) {
                validator.reportErrorsInSource({
                    result: yaml,
                    errors: valResult.errors
                }, mappedYaml, errorMessage, error3, info);
            }
            return {
                yaml: yaml,
                yamlValidationErrors: valResult.errors
            };
        } else {
            return {
                yaml: yaml,
                yamlValidationErrors: []
            };
        }
    });
    if (result.yamlValidationErrors.length > 0) {
        throw new ValidationError2(errorMessage, result.yamlValidationErrors);
    }
    return result.yaml;
}
const kRegExBeginYAML = /^---[ \t]*$/;
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        return parse3(decoder.decode(yml), {
            schema: json1
        });
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function readYamlFromString(yml) {
    return parse3(yml, {
        schema: json1
    });
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            const yamlBlock = removeYamlDelimiters(match[2]);
            if (!yamlBlock.startsWith("\n\n") && !yamlBlock.startsWith("\n---") && yamlBlock.trim().length > 0) {
                parse3(yamlBlock, {
                    json: true,
                    schema: json1
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata1 = parse3(yaml, {
            json: true,
            schema: json1
        });
        return metadata1 || {
        };
    } else {
        return {
        };
    }
}
async function readAndValidateYamlFromMarkdown(eitherMarkdown) {
    let markdown = asMappedString(eitherMarkdown);
    if (!markdown.value) {
        return {
        };
    }
    markdown = mappedNormalizeNewlines(markdown);
    markdown = skipRegexpAll(markdown, kRegxHTMLComment);
    markdown = skipRegexpAll(markdown, kRegexFencedCode);
    const yaml = [];
    kRegExYAML.lastIndex = 0;
    for (const match of matchAll(markdown.value, kRegExYAML)){
        const yamlBlock = removeYamlDelimitersMapped(match[2]);
        const yamlBlockValue = yamlBlock.value;
        if (!yamlBlockValue.startsWith("\n\n") && !yamlBlockValue.startsWith("\n---") && yamlBlockValue.trim().length > 0) {
            parse3(yamlBlockValue, {
                json: true,
                schema: json1
            });
            yaml.push(yamlBlock);
        }
    }
    kRegExYAML.lastIndex = 0;
    if (yaml.length === 0) {
        return {
        };
    }
    const mappedYaml = mappedConcat(yaml);
    const metadata1 = parse3(mappedYaml.value, {
        json: true,
        schema: json1
    });
    if (metadata1?.["validate-yaml"] === false) {
        return readAndValidateYamlFromMappedString(mappedYaml, frontMatterSchema, "YAML front matter validation failed");
    }
    return metadata1;
}
async function readYamlFromMarkdownFile(file) {
    const markdown = Deno.readTextFileSync(file);
    const result = await readAndValidateYamlFromMarkdown(markdown);
    return result;
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line3, index)=>index > 0 && line3.match(kRegExEndYAML)
        );
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
function removeYamlDelimitersMapped(eitherYaml) {
    let yaml = asMappedString(eitherYaml);
    yaml = skipRegexp(yaml, /^---/);
    yaml = skipRegexp(yaml, /---\s*$/);
    return yaml;
}
var global$1 = typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {
};
var freeGlobal = typeof global$1 == "object" && global$1 && global$1.Object === Object && global$1;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = freeGlobal || freeSelf || Function("return this")();
var Symbol1 = root.Symbol;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = Symbol1 ? Symbol1.toStringTag : void 0;
function getRawTag(value6) {
    var isOwn = hasOwnProperty.call(value6, symToStringTag), tag2 = value6[symToStringTag];
    try {
        value6[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {
    }
    var result2 = nativeObjectToString.call(value6);
    if (unmasked) {
        if (isOwn) {
            value6[symToStringTag] = tag2;
        } else {
            delete value6[symToStringTag];
        }
    }
    return result2;
}
var objectProto$1 = Object.prototype;
var nativeObjectToString$1 = objectProto$1.toString;
function objectToString(value6) {
    return nativeObjectToString$1.call(value6);
}
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag$1 = Symbol1 ? Symbol1.toStringTag : void 0;
function baseGetTag(value6) {
    if (value6 == null) {
        return value6 === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag$1 && symToStringTag$1 in Object(value6) ? getRawTag(value6) : objectToString(value6);
}
function isObjectLike(value6) {
    return value6 != null && typeof value6 == "object";
}
var symbolTag = "[object Symbol]";
function isSymbol(value6) {
    return typeof value6 == "symbol" || isObjectLike(value6) && baseGetTag(value6) == symbolTag;
}
var NAN = 0 / 0;
function baseToNumber(value6) {
    if (typeof value6 == "number") {
        return value6;
    }
    if (isSymbol(value6)) {
        return NAN;
    }
    return +value6;
}
function arrayMap(array2, iteratee2) {
    var index = -1, length = array2 == null ? 0 : array2.length, result2 = Array(length);
    while(++index < length){
        result2[index] = iteratee2(array2[index], index, array2);
    }
    return result2;
}
var isArray = Array.isArray;
var INFINITY = 1 / 0;
var symbolProto = Symbol1 ? Symbol1.prototype : void 0, symbolToString = symbolProto ? symbolProto.toString : void 0;
function baseToString(value6) {
    if (typeof value6 == "string") {
        return value6;
    }
    if (isArray(value6)) {
        return arrayMap(value6, baseToString) + "";
    }
    if (isSymbol(value6)) {
        return symbolToString ? symbolToString.call(value6) : "";
    }
    var result2 = value6 + "";
    return result2 == "0" && 1 / value6 == -INFINITY ? "-0" : result2;
}
function createMathOperation(operator, defaultValue) {
    return function(value6, other) {
        var result2;
        if (value6 === void 0 && other === void 0) {
            return defaultValue;
        }
        if (value6 !== void 0) {
            result2 = value6;
        }
        if (other !== void 0) {
            if (result2 === void 0) {
                return other;
            }
            if (typeof value6 == "string" || typeof other == "string") {
                value6 = baseToString(value6);
                other = baseToString(other);
            } else {
                value6 = baseToNumber(value6);
                other = baseToNumber(other);
            }
            result2 = operator(value6, other);
        }
        return result2;
    };
}
var add = createMathOperation(function(augend, addend) {
    return augend + addend;
}, 0);
var reWhitespace1 = /\s/;
function trimmedEndIndex(string2) {
    var index = string2.length;
    while(index-- && reWhitespace1.test(string2.charAt(index))){
    }
    return index;
}
var reTrimStart = /^\s+/;
function baseTrim(string2) {
    return string2 ? string2.slice(0, trimmedEndIndex(string2) + 1).replace(reTrimStart, "") : string2;
}
function isObject1(value6) {
    var type5 = typeof value6;
    return value6 != null && (type5 == "object" || type5 == "function");
}
var NAN$1 = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value6) {
    if (typeof value6 == "number") {
        return value6;
    }
    if (isSymbol(value6)) {
        return NAN$1;
    }
    if (isObject1(value6)) {
        var other = typeof value6.valueOf == "function" ? value6.valueOf() : value6;
        value6 = isObject1(other) ? other + "" : other;
    }
    if (typeof value6 != "string") {
        return value6 === 0 ? value6 : +value6;
    }
    value6 = baseTrim(value6);
    var isBinary1 = reIsBinary.test(value6);
    return isBinary1 || reIsOctal.test(value6) ? freeParseInt(value6.slice(2), isBinary1 ? 2 : 8) : reIsBadHex.test(value6) ? NAN$1 : +value6;
}
var INFINITY$1 = 1 / 0, MAX_INTEGER = 179769313486231570000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000;
function toFinite(value6) {
    if (!value6) {
        return value6 === 0 ? value6 : 0;
    }
    value6 = toNumber(value6);
    if (value6 === INFINITY$1 || value6 === -INFINITY$1) {
        var sign = value6 < 0 ? -1 : 1;
        return sign * MAX_INTEGER;
    }
    return value6 === value6 ? value6 : 0;
}
function toInteger(value6) {
    var result2 = toFinite(value6), remainder = result2 % 1;
    return result2 === result2 ? remainder ? result2 - remainder : result2 : 0;
}
var FUNC_ERROR_TEXT = "Expected a function";
function after(n, func2) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    n = toInteger(n);
    return function() {
        if (--n < 1) {
            return func2.apply(this, arguments);
        }
    };
}
function identity(value6) {
    return value6;
}
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value6) {
    if (!isObject1(value6)) {
        return false;
    }
    var tag2 = baseGetTag(value6);
    return tag2 == funcTag || tag2 == genTag || tag2 == asyncTag || tag2 == proxyTag;
}
var coreJsData = root["__core-js_shared__"];
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func2) {
    return !!maskSrcKey && maskSrcKey in func2;
}
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func2) {
    if (func2 != null) {
        try {
            return funcToString.call(func2);
        } catch (e) {
        }
        try {
            return func2 + "";
        } catch (e) {
        }
    }
    return "";
}
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto$1 = Function.prototype, objectProto$2 = Object.prototype;
var funcToString$1 = funcProto$1.toString;
var hasOwnProperty$1 = objectProto$2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString$1.call(hasOwnProperty$1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value6) {
    if (!isObject1(value6) || isMasked(value6)) {
        return false;
    }
    var pattern1 = isFunction(value6) ? reIsNative : reIsHostCtor;
    return pattern1.test(toSource(value6));
}
function getValue(object2, key) {
    return object2 == null ? void 0 : object2[key];
}
function getNative(object2, key) {
    var value6 = getValue(object2, key);
    return baseIsNative(value6) ? value6 : void 0;
}
var WeakMap1 = getNative(root, "WeakMap");
var metaMap = WeakMap1 && new WeakMap1();
var baseSetData = !metaMap ? identity : function(func2, data) {
    metaMap.set(func2, data);
    return func2;
};
var objectCreate = Object.create;
var baseCreate = function() {
    function object2() {
    }
    return function(proto) {
        if (!isObject1(proto)) {
            return {
            };
        }
        if (objectCreate) {
            return objectCreate(proto);
        }
        object2.prototype = proto;
        var result2 = new object2();
        object2.prototype = void 0;
        return result2;
    };
}();
function createCtor(Ctor) {
    return function() {
        var args4 = arguments;
        switch(args4.length){
            case 0:
                return new Ctor();
            case 1:
                return new Ctor(args4[0]);
            case 2:
                return new Ctor(args4[0], args4[1]);
            case 3:
                return new Ctor(args4[0], args4[1], args4[2]);
            case 4:
                return new Ctor(args4[0], args4[1], args4[2], args4[3]);
            case 5:
                return new Ctor(args4[0], args4[1], args4[2], args4[3], args4[4]);
            case 6:
                return new Ctor(args4[0], args4[1], args4[2], args4[3], args4[4], args4[5]);
            case 7:
                return new Ctor(args4[0], args4[1], args4[2], args4[3], args4[4], args4[5], args4[6]);
        }
        var thisBinding = baseCreate(Ctor.prototype), result2 = Ctor.apply(thisBinding, args4);
        return isObject1(result2) ? result2 : thisBinding;
    };
}
var WRAP_BIND_FLAG = 1;
function createBind(func2, bitmask, thisArg) {
    var isBind = bitmask & WRAP_BIND_FLAG, Ctor = createCtor(func2);
    function wrapper() {
        var fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        return fn.apply(isBind ? thisArg : this, arguments);
    }
    return wrapper;
}
function apply(func2, thisArg, args4) {
    switch(args4.length){
        case 0:
            return func2.call(thisArg);
        case 1:
            return func2.call(thisArg, args4[0]);
        case 2:
            return func2.call(thisArg, args4[0], args4[1]);
        case 3:
            return func2.call(thisArg, args4[0], args4[1], args4[2]);
    }
    return func2.apply(thisArg, args4);
}
var nativeMax = Math.max;
function composeArgs(args4, partials, holders, isCurried) {
    var argsIndex = -1, argsLength = args4.length, holdersLength = holders.length, leftIndex = -1, leftLength = partials.length, rangeLength = nativeMax(argsLength - holdersLength, 0), result2 = Array(leftLength + rangeLength), isUncurried = !isCurried;
    while(++leftIndex < leftLength){
        result2[leftIndex] = partials[leftIndex];
    }
    while(++argsIndex < holdersLength){
        if (isUncurried || argsIndex < argsLength) {
            result2[holders[argsIndex]] = args4[argsIndex];
        }
    }
    while(rangeLength--){
        result2[leftIndex++] = args4[argsIndex++];
    }
    return result2;
}
var nativeMax$1 = Math.max;
function composeArgsRight(args4, partials, holders, isCurried) {
    var argsIndex = -1, argsLength = args4.length, holdersIndex = -1, holdersLength = holders.length, rightIndex = -1, rightLength = partials.length, rangeLength = nativeMax$1(argsLength - holdersLength, 0), result2 = Array(rangeLength + rightLength), isUncurried = !isCurried;
    while(++argsIndex < rangeLength){
        result2[argsIndex] = args4[argsIndex];
    }
    var offset = argsIndex;
    while(++rightIndex < rightLength){
        result2[offset + rightIndex] = partials[rightIndex];
    }
    while(++holdersIndex < holdersLength){
        if (isUncurried || argsIndex < argsLength) {
            result2[offset + holders[holdersIndex]] = args4[argsIndex++];
        }
    }
    return result2;
}
function countHolders(array2, placeholder) {
    var length = array2.length, result2 = 0;
    while(length--){
        if (array2[length] === placeholder) {
            ++result2;
        }
    }
    return result2;
}
function baseLodash() {
}
var MAX_ARRAY_LENGTH = 4294967295;
function LazyWrapper(value6) {
    this.__wrapped__ = value6;
    this.__actions__ = [];
    this.__dir__ = 1;
    this.__filtered__ = false;
    this.__iteratees__ = [];
    this.__takeCount__ = MAX_ARRAY_LENGTH;
    this.__views__ = [];
}
LazyWrapper.prototype = baseCreate(baseLodash.prototype);
LazyWrapper.prototype.constructor = LazyWrapper;
function noop() {
}
var getData = !metaMap ? noop : function(func2) {
    return metaMap.get(func2);
};
var realNames = {
};
var objectProto$3 = Object.prototype;
var hasOwnProperty$2 = objectProto$3.hasOwnProperty;
function getFuncName(func2) {
    var result2 = func2.name + "", array2 = realNames[result2], length = hasOwnProperty$2.call(realNames, result2) ? array2.length : 0;
    while(length--){
        var data = array2[length], otherFunc = data.func;
        if (otherFunc == null || otherFunc == func2) {
            return data.name;
        }
    }
    return result2;
}
function LodashWrapper(value6, chainAll) {
    this.__wrapped__ = value6;
    this.__actions__ = [];
    this.__chain__ = !!chainAll;
    this.__index__ = 0;
    this.__values__ = void 0;
}
LodashWrapper.prototype = baseCreate(baseLodash.prototype);
LodashWrapper.prototype.constructor = LodashWrapper;
function copyArray(source, array2) {
    var index = -1, length = source.length;
    array2 || (array2 = Array(length));
    while(++index < length){
        array2[index] = source[index];
    }
    return array2;
}
function wrapperClone(wrapper) {
    if (wrapper instanceof LazyWrapper) {
        return wrapper.clone();
    }
    var result2 = new LodashWrapper(wrapper.__wrapped__, wrapper.__chain__);
    result2.__actions__ = copyArray(wrapper.__actions__);
    result2.__index__ = wrapper.__index__;
    result2.__values__ = wrapper.__values__;
    return result2;
}
var objectProto$4 = Object.prototype;
var hasOwnProperty$3 = objectProto$4.hasOwnProperty;
function lodash(value6) {
    if (isObjectLike(value6) && !isArray(value6) && !(value6 instanceof LazyWrapper)) {
        if (value6 instanceof LodashWrapper) {
            return value6;
        }
        if (hasOwnProperty$3.call(value6, "__wrapped__")) {
            return wrapperClone(value6);
        }
    }
    return new LodashWrapper(value6);
}
lodash.prototype = baseLodash.prototype;
lodash.prototype.constructor = lodash;
function isLaziable(func2) {
    var funcName = getFuncName(func2), other = lodash[funcName];
    if (typeof other != "function" || !(funcName in LazyWrapper.prototype)) {
        return false;
    }
    if (func2 === other) {
        return true;
    }
    var data = getData(other);
    return !!data && func2 === data[0];
}
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func2) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func2.apply(void 0, arguments);
    };
}
var setData = shortOut(baseSetData);
var reWrapDetails = /\{\n\/\* \[wrapped with (.+)\] \*/, reSplitDetails = /,? & /;
function getWrapDetails(source) {
    var match = source.match(reWrapDetails);
    return match ? match[1].split(reSplitDetails) : [];
}
var reWrapComment = /\{(?:\n\/\* \[wrapped with .+\] \*\/)?\n?/;
function insertWrapDetails(source, details) {
    var length = details.length;
    if (!length) {
        return source;
    }
    var lastIndex = length - 1;
    details[lastIndex] = (length > 1 ? "& " : "") + details[lastIndex];
    details = details.join(length > 2 ? ", " : " ");
    return source.replace(reWrapComment, "{\n/* [wrapped with " + details + "] */\n");
}
function constant(value6) {
    return function() {
        return value6;
    };
}
var defineProperty = function() {
    try {
        var func2 = getNative(Object, "defineProperty");
        func2({
        }, "", {
        });
        return func2;
    } catch (e) {
    }
}();
var baseSetToString = !defineProperty ? identity : function(func2, string2) {
    return defineProperty(func2, "toString", {
        configurable: true,
        enumerable: false,
        value: constant(string2),
        writable: true
    });
};
var setToString = shortOut(baseSetToString);
function arrayEach(array2, iteratee2) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (iteratee2(array2[index], index, array2) === false) {
            break;
        }
    }
    return array2;
}
function baseFindIndex(array2, predicate, fromIndex, fromRight) {
    var length = array2.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array2[index], index, array2)) {
            return index;
        }
    }
    return -1;
}
function baseIsNaN(value6) {
    return value6 !== value6;
}
function strictIndexOf(array2, value6, fromIndex) {
    var index = fromIndex - 1, length = array2.length;
    while(++index < length){
        if (array2[index] === value6) {
            return index;
        }
    }
    return -1;
}
function baseIndexOf(array2, value6, fromIndex) {
    return value6 === value6 ? strictIndexOf(array2, value6, fromIndex) : baseFindIndex(array2, baseIsNaN, fromIndex);
}
function arrayIncludes(array2, value6) {
    var length = array2 == null ? 0 : array2.length;
    return !!length && baseIndexOf(array2, value6, 0) > -1;
}
var WRAP_BIND_FLAG$1 = 1, WRAP_BIND_KEY_FLAG = 2, WRAP_CURRY_FLAG = 8, WRAP_CURRY_RIGHT_FLAG = 16, WRAP_PARTIAL_FLAG = 32, WRAP_PARTIAL_RIGHT_FLAG = 64, WRAP_ARY_FLAG = 128, WRAP_REARG_FLAG = 256, WRAP_FLIP_FLAG = 512;
var wrapFlags = [
    [
        "ary",
        WRAP_ARY_FLAG
    ],
    [
        "bind",
        WRAP_BIND_FLAG$1
    ],
    [
        "bindKey",
        WRAP_BIND_KEY_FLAG
    ],
    [
        "curry",
        WRAP_CURRY_FLAG
    ],
    [
        "curryRight",
        WRAP_CURRY_RIGHT_FLAG
    ],
    [
        "flip",
        WRAP_FLIP_FLAG
    ],
    [
        "partial",
        WRAP_PARTIAL_FLAG
    ],
    [
        "partialRight",
        WRAP_PARTIAL_RIGHT_FLAG
    ],
    [
        "rearg",
        WRAP_REARG_FLAG
    ]
];
function updateWrapDetails(details, bitmask) {
    arrayEach(wrapFlags, function(pair) {
        var value6 = "_." + pair[0];
        if (bitmask & pair[1] && !arrayIncludes(details, value6)) {
            details.push(value6);
        }
    });
    return details.sort();
}
function setWrapToString(wrapper, reference, bitmask) {
    var source = reference + "";
    return setToString(wrapper, insertWrapDetails(source, updateWrapDetails(getWrapDetails(source), bitmask)));
}
var WRAP_BIND_FLAG$2 = 1, WRAP_BIND_KEY_FLAG$1 = 2, WRAP_CURRY_BOUND_FLAG = 4, WRAP_CURRY_FLAG$1 = 8, WRAP_PARTIAL_FLAG$1 = 32, WRAP_PARTIAL_RIGHT_FLAG$1 = 64;
function createRecurry(func2, bitmask, wrapFunc, placeholder, thisArg, partials, holders, argPos, ary2, arity) {
    var isCurry = bitmask & WRAP_CURRY_FLAG$1, newHolders = isCurry ? holders : void 0, newHoldersRight = isCurry ? void 0 : holders, newPartials = isCurry ? partials : void 0, newPartialsRight = isCurry ? void 0 : partials;
    bitmask |= isCurry ? WRAP_PARTIAL_FLAG$1 : WRAP_PARTIAL_RIGHT_FLAG$1;
    bitmask &= ~(isCurry ? WRAP_PARTIAL_RIGHT_FLAG$1 : WRAP_PARTIAL_FLAG$1);
    if (!(bitmask & WRAP_CURRY_BOUND_FLAG)) {
        bitmask &= ~(WRAP_BIND_FLAG$2 | WRAP_BIND_KEY_FLAG$1);
    }
    var newData = [
        func2,
        bitmask,
        thisArg,
        newPartials,
        newHolders,
        newPartialsRight,
        newHoldersRight,
        argPos,
        ary2,
        arity
    ];
    var result2 = wrapFunc.apply(void 0, newData);
    if (isLaziable(func2)) {
        setData(result2, newData);
    }
    result2.placeholder = placeholder;
    return setWrapToString(result2, func2, bitmask);
}
function getHolder(func2) {
    var object2 = func2;
    return object2.placeholder;
}
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value6, length) {
    var type5 = typeof value6;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type5 == "number" || type5 != "symbol" && reIsUint.test(value6)) && value6 > -1 && value6 % 1 == 0 && value6 < length;
}
var nativeMin = Math.min;
function reorder(array2, indexes) {
    var arrLength = array2.length, length = nativeMin(indexes.length, arrLength), oldArray = copyArray(array2);
    while(length--){
        var index = indexes[length];
        array2[length] = isIndex(index, arrLength) ? oldArray[index] : void 0;
    }
    return array2;
}
var PLACEHOLDER = "__lodash_placeholder__";
function replaceHolders(array2, placeholder) {
    var index = -1, length = array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value6 = array2[index];
        if (value6 === placeholder || value6 === PLACEHOLDER) {
            array2[index] = PLACEHOLDER;
            result2[resIndex++] = index;
        }
    }
    return result2;
}
var WRAP_BIND_FLAG$3 = 1, WRAP_BIND_KEY_FLAG$2 = 2, WRAP_CURRY_FLAG$2 = 8, WRAP_CURRY_RIGHT_FLAG$1 = 16, WRAP_ARY_FLAG$1 = 128, WRAP_FLIP_FLAG$1 = 512;
function createHybrid(func2, bitmask, thisArg, partials, holders, partialsRight, holdersRight, argPos, ary2, arity) {
    var isAry = bitmask & WRAP_ARY_FLAG$1, isBind = bitmask & WRAP_BIND_FLAG$3, isBindKey = bitmask & WRAP_BIND_KEY_FLAG$2, isCurried = bitmask & (WRAP_CURRY_FLAG$2 | WRAP_CURRY_RIGHT_FLAG$1), isFlip = bitmask & WRAP_FLIP_FLAG$1, Ctor = isBindKey ? void 0 : createCtor(func2);
    function wrapper() {
        var length = arguments.length, args4 = Array(length), index = length;
        while(index--){
            args4[index] = arguments[index];
        }
        if (isCurried) {
            var placeholder = getHolder(wrapper), holdersCount = countHolders(args4, placeholder);
        }
        if (partials) {
            args4 = composeArgs(args4, partials, holders, isCurried);
        }
        if (partialsRight) {
            args4 = composeArgsRight(args4, partialsRight, holdersRight, isCurried);
        }
        length -= holdersCount;
        if (isCurried && length < arity) {
            var newHolders = replaceHolders(args4, placeholder);
            return createRecurry(func2, bitmask, createHybrid, wrapper.placeholder, thisArg, args4, newHolders, argPos, ary2, arity - length);
        }
        var thisBinding = isBind ? thisArg : this, fn = isBindKey ? thisBinding[func2] : func2;
        length = args4.length;
        if (argPos) {
            args4 = reorder(args4, argPos);
        } else if (isFlip && length > 1) {
            args4.reverse();
        }
        if (isAry && ary2 < length) {
            args4.length = ary2;
        }
        if (this && this !== root && this instanceof wrapper) {
            fn = Ctor || createCtor(fn);
        }
        return fn.apply(thisBinding, args4);
    }
    return wrapper;
}
function createCurry(func2, bitmask, arity) {
    var Ctor = createCtor(func2);
    function wrapper() {
        var length = arguments.length, args4 = Array(length), index = length, placeholder = getHolder(wrapper);
        while(index--){
            args4[index] = arguments[index];
        }
        var holders = length < 3 && args4[0] !== placeholder && args4[length - 1] !== placeholder ? [] : replaceHolders(args4, placeholder);
        length -= holders.length;
        if (length < arity) {
            return createRecurry(func2, bitmask, createHybrid, wrapper.placeholder, void 0, args4, holders, void 0, void 0, arity - length);
        }
        var fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        return apply(fn, this, args4);
    }
    return wrapper;
}
var WRAP_BIND_FLAG$4 = 1;
function createPartial(func2, bitmask, thisArg, partials) {
    var isBind = bitmask & WRAP_BIND_FLAG$4, Ctor = createCtor(func2);
    function wrapper() {
        var argsIndex = -1, argsLength = arguments.length, leftIndex = -1, leftLength = partials.length, args4 = Array(leftLength + argsLength), fn = this && this !== root && this instanceof wrapper ? Ctor : func2;
        while(++leftIndex < leftLength){
            args4[leftIndex] = partials[leftIndex];
        }
        while(argsLength--){
            args4[leftIndex++] = arguments[++argsIndex];
        }
        return apply(fn, isBind ? thisArg : this, args4);
    }
    return wrapper;
}
var PLACEHOLDER$1 = "__lodash_placeholder__";
var WRAP_BIND_FLAG$5 = 1, WRAP_BIND_KEY_FLAG$3 = 2, WRAP_CURRY_BOUND_FLAG$1 = 4, WRAP_CURRY_FLAG$3 = 8, WRAP_ARY_FLAG$2 = 128, WRAP_REARG_FLAG$1 = 256;
var nativeMin$1 = Math.min;
function mergeData(data, source) {
    var bitmask = data[1], srcBitmask = source[1], newBitmask = bitmask | srcBitmask, isCommon = newBitmask < (WRAP_BIND_FLAG$5 | WRAP_BIND_KEY_FLAG$3 | WRAP_ARY_FLAG$2);
    var isCombo = srcBitmask == WRAP_ARY_FLAG$2 && bitmask == WRAP_CURRY_FLAG$3 || srcBitmask == WRAP_ARY_FLAG$2 && bitmask == WRAP_REARG_FLAG$1 && data[7].length <= source[8] || srcBitmask == (WRAP_ARY_FLAG$2 | WRAP_REARG_FLAG$1) && source[7].length <= source[8] && bitmask == WRAP_CURRY_FLAG$3;
    if (!(isCommon || isCombo)) {
        return data;
    }
    if (srcBitmask & WRAP_BIND_FLAG$5) {
        data[2] = source[2];
        newBitmask |= bitmask & WRAP_BIND_FLAG$5 ? 0 : WRAP_CURRY_BOUND_FLAG$1;
    }
    var value7 = source[3];
    if (value7) {
        var partials = data[3];
        data[3] = partials ? composeArgs(partials, value7, source[4]) : value7;
        data[4] = partials ? replaceHolders(data[3], PLACEHOLDER$1) : source[4];
    }
    value7 = source[5];
    if (value7) {
        partials = data[5];
        data[5] = partials ? composeArgsRight(partials, value7, source[6]) : value7;
        data[6] = partials ? replaceHolders(data[5], PLACEHOLDER$1) : source[6];
    }
    value7 = source[7];
    if (value7) {
        data[7] = value7;
    }
    if (srcBitmask & WRAP_ARY_FLAG$2) {
        data[8] = data[8] == null ? source[8] : nativeMin$1(data[8], source[8]);
    }
    if (data[9] == null) {
        data[9] = source[9];
    }
    data[0] = source[0];
    data[1] = newBitmask;
    return data;
}
var FUNC_ERROR_TEXT$1 = "Expected a function";
var WRAP_BIND_FLAG$6 = 1, WRAP_BIND_KEY_FLAG$4 = 2, WRAP_CURRY_FLAG$4 = 8, WRAP_CURRY_RIGHT_FLAG$2 = 16, WRAP_PARTIAL_FLAG$2 = 32, WRAP_PARTIAL_RIGHT_FLAG$2 = 64;
var nativeMax$2 = Math.max;
function createWrap(func2, bitmask, thisArg, partials, holders, argPos, ary2, arity) {
    var isBindKey = bitmask & WRAP_BIND_KEY_FLAG$4;
    if (!isBindKey && typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$1);
    }
    var length = partials ? partials.length : 0;
    if (!length) {
        bitmask &= ~(WRAP_PARTIAL_FLAG$2 | WRAP_PARTIAL_RIGHT_FLAG$2);
        partials = holders = void 0;
    }
    ary2 = ary2 === void 0 ? ary2 : nativeMax$2(toInteger(ary2), 0);
    arity = arity === void 0 ? arity : toInteger(arity);
    length -= holders ? holders.length : 0;
    if (bitmask & WRAP_PARTIAL_RIGHT_FLAG$2) {
        var partialsRight = partials, holdersRight = holders;
        partials = holders = void 0;
    }
    var data = isBindKey ? void 0 : getData(func2);
    var newData = [
        func2,
        bitmask,
        thisArg,
        partials,
        holders,
        partialsRight,
        holdersRight,
        argPos,
        ary2,
        arity
    ];
    if (data) {
        mergeData(newData, data);
    }
    func2 = newData[0];
    bitmask = newData[1];
    thisArg = newData[2];
    partials = newData[3];
    holders = newData[4];
    arity = newData[9] = newData[9] === void 0 ? isBindKey ? 0 : func2.length : nativeMax$2(newData[9] - length, 0);
    if (!arity && bitmask & (WRAP_CURRY_FLAG$4 | WRAP_CURRY_RIGHT_FLAG$2)) {
        bitmask &= ~(WRAP_CURRY_FLAG$4 | WRAP_CURRY_RIGHT_FLAG$2);
    }
    if (!bitmask || bitmask == WRAP_BIND_FLAG$6) {
        var result2 = createBind(func2, bitmask, thisArg);
    } else if (bitmask == WRAP_CURRY_FLAG$4 || bitmask == WRAP_CURRY_RIGHT_FLAG$2) {
        result2 = createCurry(func2, bitmask, arity);
    } else if ((bitmask == WRAP_PARTIAL_FLAG$2 || bitmask == (WRAP_BIND_FLAG$6 | WRAP_PARTIAL_FLAG$2)) && !holders.length) {
        result2 = createPartial(func2, bitmask, thisArg, partials);
    } else {
        result2 = createHybrid.apply(void 0, newData);
    }
    var setter = data ? baseSetData : setData;
    return setWrapToString(setter(result2, newData), func2, bitmask);
}
var WRAP_ARY_FLAG$3 = 128;
function ary(func2, n, guard) {
    n = guard ? void 0 : n;
    n = func2 && n == null ? func2.length : n;
    return createWrap(func2, WRAP_ARY_FLAG$3, void 0, void 0, void 0, void 0, n);
}
function baseAssignValue(object2, key, value7) {
    if (key == "__proto__" && defineProperty) {
        defineProperty(object2, key, {
            configurable: true,
            enumerable: true,
            value: value7,
            writable: true
        });
    } else {
        object2[key] = value7;
    }
}
function eq(value7, other) {
    return value7 === other || value7 !== value7 && other !== other;
}
var objectProto$5 = Object.prototype;
var hasOwnProperty$4 = objectProto$5.hasOwnProperty;
function assignValue(object2, key, value7) {
    var objValue = object2[key];
    if (!(hasOwnProperty$4.call(object2, key) && eq(objValue, value7)) || value7 === void 0 && !(key in object2)) {
        baseAssignValue(object2, key, value7);
    }
}
function copyObject(source, props, object2, customizer) {
    var isNew = !object2;
    object2 || (object2 = {
    });
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object2[key], source[key], key, object2, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            baseAssignValue(object2, key, newValue);
        } else {
            assignValue(object2, key, newValue);
        }
    }
    return object2;
}
var nativeMax$3 = Math.max;
function overRest(func2, start, transform2) {
    start = nativeMax$3(start === void 0 ? func2.length - 1 : start, 0);
    return function() {
        var args4 = arguments, index = -1, length = nativeMax$3(args4.length - start, 0), array2 = Array(length);
        while(++index < length){
            array2[index] = args4[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args4[index];
        }
        otherArgs[start] = transform2(array2);
        return apply(func2, this, otherArgs);
    };
}
function baseRest(func2, start) {
    return setToString(overRest(func2, start, identity), func2 + "");
}
var MAX_SAFE_INTEGER$1 = 9007199254740991;
function isLength(value7) {
    return typeof value7 == "number" && value7 > -1 && value7 % 1 == 0 && value7 <= MAX_SAFE_INTEGER$1;
}
function isArrayLike(value7) {
    return value7 != null && isLength(value7.length) && !isFunction(value7);
}
function isIterateeCall(value7, index, object2) {
    if (!isObject1(object2)) {
        return false;
    }
    var type5 = typeof index;
    if (type5 == "number" ? isArrayLike(object2) && isIndex(index, object2.length) : type5 == "string" && index in object2) {
        return eq(object2[index], value7);
    }
    return false;
}
function createAssigner(assigner) {
    return baseRest(function(object2, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && isIterateeCall(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object2 = Object(object2);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object2, source, index, customizer);
            }
        }
        return object2;
    });
}
var objectProto$6 = Object.prototype;
function isPrototype(value7) {
    var Ctor = value7 && value7.constructor, proto = typeof Ctor == "function" && Ctor.prototype || objectProto$6;
    return value7 === proto;
}
function baseTimes(n, iteratee2) {
    var index = -1, result2 = Array(n);
    while(++index < n){
        result2[index] = iteratee2(index);
    }
    return result2;
}
var argsTag = "[object Arguments]";
function baseIsArguments(value7) {
    return isObjectLike(value7) && baseGetTag(value7) == argsTag;
}
var objectProto$7 = Object.prototype;
var hasOwnProperty$5 = objectProto$7.hasOwnProperty;
var propertyIsEnumerable = objectProto$7.propertyIsEnumerable;
var isArguments = baseIsArguments(function() {
    return arguments;
}()) ? baseIsArguments : function(value7) {
    return isObjectLike(value7) && hasOwnProperty$5.call(value7, "callee") && !propertyIsEnumerable.call(value7, "callee");
};
function stubFalse() {
    return false;
}
var freeExports = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule = freeExports && typeof module == "object" && module && !module.nodeType && module;
var moduleExports = freeModule && freeModule.exports === freeExports;
var Buffer1 = moduleExports ? root.Buffer : void 0;
var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
var isBuffer = nativeIsBuffer || stubFalse;
var argsTag$1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag$1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {
};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag$1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag$1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value7) {
    return isObjectLike(value7) && isLength(value7.length) && !!typedArrayTags[baseGetTag(value7)];
}
function baseUnary(func2) {
    return function(value7) {
        return func2(value7);
    };
}
var freeExports$1 = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule$1 = freeExports$1 && typeof module == "object" && module && !module.nodeType && module;
var moduleExports$1 = freeModule$1 && freeModule$1.exports === freeExports$1;
var freeProcess = moduleExports$1 && freeGlobal.process;
var nodeUtil = function() {
    try {
        var types3 = freeModule$1 && freeModule$1.require && freeModule$1.require("util").types;
        if (types3) {
            return types3;
        }
        return freeProcess && freeProcess.binding && freeProcess.binding("util");
    } catch (e) {
    }
}();
var nodeIsTypedArray = nodeUtil && nodeUtil.isTypedArray;
var isTypedArray = nodeIsTypedArray ? baseUnary(nodeIsTypedArray) : baseIsTypedArray;
var objectProto$8 = Object.prototype;
var hasOwnProperty$6 = objectProto$8.hasOwnProperty;
function arrayLikeKeys(value7, inherited) {
    var isArr = isArray(value7), isArg = !isArr && isArguments(value7), isBuff = !isArr && !isArg && isBuffer(value7), isType = !isArr && !isArg && !isBuff && isTypedArray(value7), skipIndexes = isArr || isArg || isBuff || isType, result2 = skipIndexes ? baseTimes(value7.length, String) : [], length = result2.length;
    for(var key in value7){
        if ((inherited || hasOwnProperty$6.call(value7, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || isIndex(key, length)))) {
            result2.push(key);
        }
    }
    return result2;
}
function overArg(func2, transform2) {
    return function(arg4) {
        return func2(transform2(arg4));
    };
}
var nativeKeys = overArg(Object.keys, Object);
var objectProto$9 = Object.prototype;
var hasOwnProperty$7 = objectProto$9.hasOwnProperty;
function baseKeys(object2) {
    if (!isPrototype(object2)) {
        return nativeKeys(object2);
    }
    var result2 = [];
    for(var key in Object(object2)){
        if (hasOwnProperty$7.call(object2, key) && key != "constructor") {
            result2.push(key);
        }
    }
    return result2;
}
function keys1(object2) {
    return isArrayLike(object2) ? arrayLikeKeys(object2) : baseKeys(object2);
}
var objectProto$a = Object.prototype;
var hasOwnProperty$8 = objectProto$a.hasOwnProperty;
var assign = createAssigner(function(object2, source) {
    if (isPrototype(source) || isArrayLike(source)) {
        copyObject(source, keys1(source), object2);
        return;
    }
    for(var key in source){
        if (hasOwnProperty$8.call(source, key)) {
            assignValue(object2, key, source[key]);
        }
    }
});
function nativeKeysIn(object2) {
    var result2 = [];
    if (object2 != null) {
        for(var key in Object(object2)){
            result2.push(key);
        }
    }
    return result2;
}
var objectProto$b = Object.prototype;
var hasOwnProperty$9 = objectProto$b.hasOwnProperty;
function baseKeysIn(object2) {
    if (!isObject1(object2)) {
        return nativeKeysIn(object2);
    }
    var isProto = isPrototype(object2), result2 = [];
    for(var key in object2){
        if (!(key == "constructor" && (isProto || !hasOwnProperty$9.call(object2, key)))) {
            result2.push(key);
        }
    }
    return result2;
}
function keysIn(object2) {
    return isArrayLike(object2) ? arrayLikeKeys(object2, true) : baseKeysIn(object2);
}
var assignIn = createAssigner(function(object2, source) {
    copyObject(source, keysIn(source), object2);
});
var assignInWith = createAssigner(function(object2, source, srcIndex, customizer) {
    copyObject(source, keysIn(source), object2, customizer);
});
var assignWith = createAssigner(function(object2, source, srcIndex, customizer) {
    copyObject(source, keys1(source), object2, customizer);
});
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value7, object2) {
    if (isArray(value7)) {
        return false;
    }
    var type5 = typeof value7;
    if (type5 == "number" || type5 == "symbol" || type5 == "boolean" || value7 == null || isSymbol(value7)) {
        return true;
    }
    return reIsPlainProp.test(value7) || !reIsDeepProp.test(value7) || object2 != null && value7 in Object(object2);
}
var nativeCreate = getNative(Object, "create");
function hashClear() {
    this.__data__ = nativeCreate ? nativeCreate(null) : {
    };
    this.size = 0;
}
function hashDelete(key) {
    var result2 = this.has(key) && delete this.__data__[key];
    this.size -= result2 ? 1 : 0;
    return result2;
}
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto$c = Object.prototype;
var hasOwnProperty$a = objectProto$c.hasOwnProperty;
function hashGet(key) {
    var data = this.__data__;
    if (nativeCreate) {
        var result2 = data[key];
        return result2 === HASH_UNDEFINED ? void 0 : result2;
    }
    return hasOwnProperty$a.call(data, key) ? data[key] : void 0;
}
var objectProto$d = Object.prototype;
var hasOwnProperty$b = objectProto$d.hasOwnProperty;
function hashHas(key) {
    var data = this.__data__;
    return nativeCreate ? data[key] !== void 0 : hasOwnProperty$b.call(data, key);
}
var HASH_UNDEFINED$1 = "__lodash_hash_undefined__";
function hashSet(key, value7) {
    var data = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data[key] = nativeCreate && value7 === void 0 ? HASH_UNDEFINED$1 : value7;
    return this;
}
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = hashClear;
Hash.prototype["delete"] = hashDelete;
Hash.prototype.get = hashGet;
Hash.prototype.has = hashHas;
Hash.prototype.set = hashSet;
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
function assocIndexOf(array2, key) {
    var length = array2.length;
    while(length--){
        if (eq(array2[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data = this.__data__, index = assocIndexOf(data, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data.length - 1;
    if (index == lastIndex) {
        data.pop();
    } else {
        splice.call(data, index, 1);
    }
    --this.size;
    return true;
}
function listCacheGet(key) {
    var data = this.__data__, index = assocIndexOf(data, key);
    return index < 0 ? void 0 : data[index][1];
}
function listCacheHas(key) {
    return assocIndexOf(this.__data__, key) > -1;
}
function listCacheSet(key, value7) {
    var data = this.__data__, index = assocIndexOf(data, key);
    if (index < 0) {
        ++this.size;
        data.push([
            key,
            value7
        ]);
    } else {
        data[index][1] = value7;
    }
    return this;
}
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = listCacheClear;
ListCache.prototype["delete"] = listCacheDelete;
ListCache.prototype.get = listCacheGet;
ListCache.prototype.has = listCacheHas;
ListCache.prototype.set = listCacheSet;
var Map1 = getNative(root, "Map");
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new Hash(),
        map: new (Map1 || ListCache)(),
        string: new Hash()
    };
}
function isKeyable(value7) {
    var type5 = typeof value7;
    return type5 == "string" || type5 == "number" || type5 == "symbol" || type5 == "boolean" ? value7 !== "__proto__" : value7 === null;
}
function getMapData(map2, key) {
    var data = map2.__data__;
    return isKeyable(key) ? data[typeof key == "string" ? "string" : "hash"] : data.map;
}
function mapCacheDelete(key) {
    var result2 = getMapData(this, key)["delete"](key);
    this.size -= result2 ? 1 : 0;
    return result2;
}
function mapCacheGet(key) {
    return getMapData(this, key).get(key);
}
function mapCacheHas(key) {
    return getMapData(this, key).has(key);
}
function mapCacheSet(key, value7) {
    var data = getMapData(this, key), size21 = data.size;
    data.set(key, value7);
    this.size += data.size == size21 ? 0 : 1;
    return this;
}
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = mapCacheClear;
MapCache.prototype["delete"] = mapCacheDelete;
MapCache.prototype.get = mapCacheGet;
MapCache.prototype.has = mapCacheHas;
MapCache.prototype.set = mapCacheSet;
var FUNC_ERROR_TEXT$2 = "Expected a function";
function memoize(func2, resolver) {
    if (typeof func2 != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$2);
    }
    var memoized = function() {
        var args4 = arguments, key = resolver ? resolver.apply(this, args4) : args4[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result2 = func2.apply(this, args4);
        memoized.cache = cache.set(key, result2) || cache;
        return result2;
    };
    memoized.cache = new (memoize.Cache || MapCache)();
    return memoized;
}
memoize.Cache = MapCache;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func2) {
    var result2 = memoize(func2, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result2.cache;
    return result2;
}
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = memoizeCapped(function(string2) {
    var result2 = [];
    if (string2.charCodeAt(0) === 46) {
        result2.push("");
    }
    string2.replace(rePropName, function(match, number2, quote, subString) {
        result2.push(quote ? subString.replace(reEscapeChar, "$1") : number2 || match);
    });
    return result2;
});
function toString(value7) {
    return value7 == null ? "" : baseToString(value7);
}
function castPath(value7, object2) {
    if (isArray(value7)) {
        return value7;
    }
    return isKey(value7, object2) ? [
        value7
    ] : stringToPath(toString(value7));
}
var INFINITY$2 = 1 / 0;
function toKey(value7) {
    if (typeof value7 == "string" || isSymbol(value7)) {
        return value7;
    }
    var result2 = value7 + "";
    return result2 == "0" && 1 / value7 == -INFINITY$2 ? "-0" : result2;
}
function baseGet(object2, path3) {
    path3 = castPath(path3, object2);
    var index = 0, length = path3.length;
    while(object2 != null && index < length){
        object2 = object2[toKey(path3[index++])];
    }
    return index && index == length ? object2 : void 0;
}
function get(object2, path3, defaultValue) {
    var result2 = object2 == null ? void 0 : baseGet(object2, path3);
    return result2 === void 0 ? defaultValue : result2;
}
function baseAt(object2, paths) {
    var index = -1, length = paths.length, result2 = Array(length), skip = object2 == null;
    while(++index < length){
        result2[index] = skip ? void 0 : get(object2, paths[index]);
    }
    return result2;
}
function arrayPush(array2, values2) {
    var index = -1, length = values2.length, offset = array2.length;
    while(++index < length){
        array2[offset + index] = values2[index];
    }
    return array2;
}
var spreadableSymbol = Symbol1 ? Symbol1.isConcatSpreadable : void 0;
function isFlattenable(value7) {
    return isArray(value7) || isArguments(value7) || !!(spreadableSymbol && value7 && value7[spreadableSymbol]);
}
function baseFlatten(array2, depth, predicate, isStrict, result2) {
    var index = -1, length = array2.length;
    predicate || (predicate = isFlattenable);
    result2 || (result2 = []);
    while(++index < length){
        var value7 = array2[index];
        if (depth > 0 && predicate(value7)) {
            if (depth > 1) {
                baseFlatten(value7, depth - 1, predicate, isStrict, result2);
            } else {
                arrayPush(result2, value7);
            }
        } else if (!isStrict) {
            result2[result2.length] = value7;
        }
    }
    return result2;
}
function flatten(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseFlatten(array2, 1) : [];
}
function flatRest(func2) {
    return setToString(overRest(func2, void 0, flatten), func2 + "");
}
var at = flatRest(baseAt);
var getPrototype = overArg(Object.getPrototypeOf, Object);
var objectTag$1 = "[object Object]";
var funcProto$2 = Function.prototype, objectProto$e = Object.prototype;
var funcToString$2 = funcProto$2.toString;
var hasOwnProperty$c = objectProto$e.hasOwnProperty;
var objectCtorString = funcToString$2.call(Object);
function isPlainObject(value8) {
    if (!isObjectLike(value8) || baseGetTag(value8) != objectTag$1) {
        return false;
    }
    var proto = getPrototype(value8);
    if (proto === null) {
        return true;
    }
    var Ctor = hasOwnProperty$c.call(proto, "constructor") && proto.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString$2.call(Ctor) == objectCtorString;
}
var domExcTag = "[object DOMException]", errorTag$1 = "[object Error]";
function isError(value8) {
    if (!isObjectLike(value8)) {
        return false;
    }
    var tag2 = baseGetTag(value8);
    return tag2 == errorTag$1 || tag2 == domExcTag || typeof value8.message == "string" && typeof value8.name == "string" && !isPlainObject(value8);
}
var attempt = baseRest(function(func2, args4) {
    try {
        return apply(func2, void 0, args4);
    } catch (e) {
        return isError(e) ? e : new Error(e);
    }
});
var FUNC_ERROR_TEXT$3 = "Expected a function";
function before(n, func2) {
    var result2;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$3);
    }
    n = toInteger(n);
    return function() {
        if (--n > 0) {
            result2 = func2.apply(this, arguments);
        }
        if (n <= 1) {
            func2 = void 0;
        }
        return result2;
    };
}
var WRAP_BIND_FLAG$7 = 1, WRAP_PARTIAL_FLAG$3 = 32;
var bind = baseRest(function(func2, thisArg, partials) {
    var bitmask = WRAP_BIND_FLAG$7;
    if (partials.length) {
        var holders = replaceHolders(partials, getHolder(bind));
        bitmask |= WRAP_PARTIAL_FLAG$3;
    }
    return createWrap(func2, bitmask, thisArg, partials, holders);
});
bind.placeholder = {
};
var bindAll = flatRest(function(object2, methodNames) {
    arrayEach(methodNames, function(key) {
        key = toKey(key);
        baseAssignValue(object2, key, bind(object2[key], object2));
    });
    return object2;
});
var WRAP_BIND_FLAG$8 = 1, WRAP_BIND_KEY_FLAG$5 = 2, WRAP_PARTIAL_FLAG$4 = 32;
var bindKey = baseRest(function(object2, key, partials) {
    var bitmask = WRAP_BIND_FLAG$8 | WRAP_BIND_KEY_FLAG$5;
    if (partials.length) {
        var holders = replaceHolders(partials, getHolder(bindKey));
        bitmask |= WRAP_PARTIAL_FLAG$4;
    }
    return createWrap(key, bitmask, object2, partials, holders);
});
bindKey.placeholder = {
};
function baseSlice(array2, start, end) {
    var index = -1, length = array2.length;
    if (start < 0) {
        start = -start > length ? 0 : length + start;
    }
    end = end > length ? length : end;
    if (end < 0) {
        end += length;
    }
    length = start > end ? 0 : end - start >>> 0;
    start >>>= 0;
    var result2 = Array(length);
    while(++index < length){
        result2[index] = array2[index + start];
    }
    return result2;
}
function castSlice(array2, start, end) {
    var length = array2.length;
    end = end === void 0 ? length : end;
    return !start && end >= length ? array2 : baseSlice(array2, start, end);
}
var rsAstralRange = "\\ud800-\\udfff", rsComboMarksRange = "\\u0300-\\u036f", reComboHalfMarksRange = "\\ufe20-\\ufe2f", rsComboSymbolsRange = "\\u20d0-\\u20ff", rsComboRange = rsComboMarksRange + reComboHalfMarksRange + rsComboSymbolsRange, rsVarRange = "\\ufe0e\\ufe0f";
var rsZWJ = "\\u200d";
var reHasUnicode = RegExp("[" + rsZWJ + rsAstralRange + rsComboRange + rsVarRange + "]");
function hasUnicode(string2) {
    return reHasUnicode.test(string2);
}
function asciiToArray(string2) {
    return string2.split("");
}
var rsAstralRange$1 = "\\ud800-\\udfff", rsComboMarksRange$1 = "\\u0300-\\u036f", reComboHalfMarksRange$1 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$1 = "\\u20d0-\\u20ff", rsComboRange$1 = rsComboMarksRange$1 + reComboHalfMarksRange$1 + rsComboSymbolsRange$1, rsVarRange$1 = "\\ufe0e\\ufe0f";
var rsAstral = "[" + rsAstralRange$1 + "]", rsCombo = "[" + rsComboRange$1 + "]", rsFitz = "\\ud83c[\\udffb-\\udfff]", rsModifier = "(?:" + rsCombo + "|" + rsFitz + ")", rsNonAstral = "[^" + rsAstralRange$1 + "]", rsRegional = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsZWJ$1 = "\\u200d";
var reOptMod = rsModifier + "?", rsOptVar = "[" + rsVarRange$1 + "]?", rsOptJoin = "(?:" + rsZWJ$1 + "(?:" + [
    rsNonAstral,
    rsRegional,
    rsSurrPair
].join("|") + ")" + rsOptVar + reOptMod + ")*", rsSeq = rsOptVar + reOptMod + rsOptJoin, rsSymbol = "(?:" + [
    rsNonAstral + rsCombo + "?",
    rsCombo,
    rsRegional,
    rsSurrPair,
    rsAstral
].join("|") + ")";
var reUnicode = RegExp(rsFitz + "(?=" + rsFitz + ")|" + rsSymbol + rsSeq, "g");
function unicodeToArray(string2) {
    return string2.match(reUnicode) || [];
}
function stringToArray(string2) {
    return hasUnicode(string2) ? unicodeToArray(string2) : asciiToArray(string2);
}
function createCaseFirst(methodName) {
    return function(string2) {
        string2 = toString(string2);
        var strSymbols = hasUnicode(string2) ? stringToArray(string2) : void 0;
        var chr = strSymbols ? strSymbols[0] : string2.charAt(0);
        var trailing = strSymbols ? castSlice(strSymbols, 1).join("") : string2.slice(1);
        return chr[methodName]() + trailing;
    };
}
var upperFirst = createCaseFirst("toUpperCase");
function capitalize1(string2) {
    return upperFirst(toString(string2).toLowerCase());
}
function arrayReduce(array2, iteratee2, accumulator, initAccum) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    if (initAccum && length) {
        accumulator = array2[++index];
    }
    while(++index < length){
        accumulator = iteratee2(accumulator, array2[index], index, array2);
    }
    return accumulator;
}
function basePropertyOf(object2) {
    return function(key) {
        return object2 == null ? void 0 : object2[key];
    };
}
var deburredLetters = {
    À: "A",
    Á: "A",
    Â: "A",
    Ã: "A",
    Ä: "A",
    Å: "A",
    à: "a",
    á: "a",
    â: "a",
    ã: "a",
    ä: "a",
    å: "a",
    Ç: "C",
    ç: "c",
    Ð: "D",
    ð: "d",
    È: "E",
    É: "E",
    Ê: "E",
    Ë: "E",
    è: "e",
    é: "e",
    ê: "e",
    ë: "e",
    Ì: "I",
    Í: "I",
    Î: "I",
    Ï: "I",
    ì: "i",
    í: "i",
    î: "i",
    ï: "i",
    Ñ: "N",
    ñ: "n",
    Ò: "O",
    Ó: "O",
    Ô: "O",
    Õ: "O",
    Ö: "O",
    Ø: "O",
    ò: "o",
    ó: "o",
    ô: "o",
    õ: "o",
    ö: "o",
    ø: "o",
    Ù: "U",
    Ú: "U",
    Û: "U",
    Ü: "U",
    ù: "u",
    ú: "u",
    û: "u",
    ü: "u",
    Ý: "Y",
    ý: "y",
    ÿ: "y",
    Æ: "Ae",
    æ: "ae",
    Þ: "Th",
    þ: "th",
    ß: "ss",
    Ā: "A",
    Ă: "A",
    Ą: "A",
    ā: "a",
    ă: "a",
    ą: "a",
    Ć: "C",
    Ĉ: "C",
    Ċ: "C",
    Č: "C",
    ć: "c",
    ĉ: "c",
    ċ: "c",
    č: "c",
    Ď: "D",
    Đ: "D",
    ď: "d",
    đ: "d",
    Ē: "E",
    Ĕ: "E",
    Ė: "E",
    Ę: "E",
    Ě: "E",
    ē: "e",
    ĕ: "e",
    ė: "e",
    ę: "e",
    ě: "e",
    Ĝ: "G",
    Ğ: "G",
    Ġ: "G",
    Ģ: "G",
    ĝ: "g",
    ğ: "g",
    ġ: "g",
    ģ: "g",
    Ĥ: "H",
    Ħ: "H",
    ĥ: "h",
    ħ: "h",
    Ĩ: "I",
    Ī: "I",
    Ĭ: "I",
    Į: "I",
    İ: "I",
    ĩ: "i",
    ī: "i",
    ĭ: "i",
    į: "i",
    ı: "i",
    Ĵ: "J",
    ĵ: "j",
    Ķ: "K",
    ķ: "k",
    ĸ: "k",
    Ĺ: "L",
    Ļ: "L",
    Ľ: "L",
    Ŀ: "L",
    Ł: "L",
    ĺ: "l",
    ļ: "l",
    ľ: "l",
    ŀ: "l",
    ł: "l",
    Ń: "N",
    Ņ: "N",
    Ň: "N",
    Ŋ: "N",
    ń: "n",
    ņ: "n",
    ň: "n",
    ŋ: "n",
    Ō: "O",
    Ŏ: "O",
    Ő: "O",
    ō: "o",
    ŏ: "o",
    ő: "o",
    Ŕ: "R",
    Ŗ: "R",
    Ř: "R",
    ŕ: "r",
    ŗ: "r",
    ř: "r",
    Ś: "S",
    Ŝ: "S",
    Ş: "S",
    Š: "S",
    ś: "s",
    ŝ: "s",
    ş: "s",
    š: "s",
    Ţ: "T",
    Ť: "T",
    Ŧ: "T",
    ţ: "t",
    ť: "t",
    ŧ: "t",
    Ũ: "U",
    Ū: "U",
    Ŭ: "U",
    Ů: "U",
    Ű: "U",
    Ų: "U",
    ũ: "u",
    ū: "u",
    ŭ: "u",
    ů: "u",
    ű: "u",
    ų: "u",
    Ŵ: "W",
    ŵ: "w",
    Ŷ: "Y",
    ŷ: "y",
    Ÿ: "Y",
    Ź: "Z",
    Ż: "Z",
    Ž: "Z",
    ź: "z",
    ż: "z",
    ž: "z",
    Ĳ: "IJ",
    ĳ: "ij",
    Œ: "Oe",
    œ: "oe",
    ŉ: "'n",
    ſ: "s"
};
var deburrLetter = basePropertyOf(deburredLetters);
var reLatin = /[\xc0-\xd6\xd8-\xf6\xf8-\xff\u0100-\u017f]/g;
var rsComboMarksRange$2 = "\\u0300-\\u036f", reComboHalfMarksRange$2 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$2 = "\\u20d0-\\u20ff", rsComboRange$2 = rsComboMarksRange$2 + reComboHalfMarksRange$2 + rsComboSymbolsRange$2;
var rsCombo$1 = "[" + rsComboRange$2 + "]";
var reComboMark = RegExp(rsCombo$1, "g");
function deburr(string2) {
    string2 = toString(string2);
    return string2 && string2.replace(reLatin, deburrLetter).replace(reComboMark, "");
}
var reAsciiWord = /[^\x00-\x2f\x3a-\x40\x5b-\x60\x7b-\x7f]+/g;
function asciiWords(string2) {
    return string2.match(reAsciiWord) || [];
}
var reHasUnicodeWord = /[a-z][A-Z]|[A-Z]{2}[a-z]|[0-9][a-zA-Z]|[a-zA-Z][0-9]|[^a-zA-Z0-9 ]/;
function hasUnicodeWord(string2) {
    return reHasUnicodeWord.test(string2);
}
var rsAstralRange$2 = "\\ud800-\\udfff", rsComboMarksRange$3 = "\\u0300-\\u036f", reComboHalfMarksRange$3 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$3 = "\\u20d0-\\u20ff", rsComboRange$3 = rsComboMarksRange$3 + reComboHalfMarksRange$3 + rsComboSymbolsRange$3, rsDingbatRange = "\\u2700-\\u27bf", rsLowerRange = "a-z\\xdf-\\xf6\\xf8-\\xff", rsMathOpRange = "\\xac\\xb1\\xd7\\xf7", rsNonCharRange = "\\x00-\\x2f\\x3a-\\x40\\x5b-\\x60\\x7b-\\xbf", rsPunctuationRange = "\\u2000-\\u206f", rsSpaceRange = " \\t\\x0b\\f\\xa0\\ufeff\\n\\r\\u2028\\u2029\\u1680\\u180e\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u202f\\u205f\\u3000", rsUpperRange = "A-Z\\xc0-\\xd6\\xd8-\\xde", rsVarRange$2 = "\\ufe0e\\ufe0f", rsBreakRange = rsMathOpRange + rsNonCharRange + rsPunctuationRange + rsSpaceRange;
var rsApos = "['\u2019]", rsBreak = "[" + rsBreakRange + "]", rsCombo$2 = "[" + rsComboRange$3 + "]", rsDigits = "\\d+", rsDingbat = "[" + rsDingbatRange + "]", rsLower = "[" + rsLowerRange + "]", rsMisc = "[^" + rsAstralRange$2 + rsBreakRange + rsDigits + rsDingbatRange + rsLowerRange + rsUpperRange + "]", rsFitz$1 = "\\ud83c[\\udffb-\\udfff]", rsModifier$1 = "(?:" + rsCombo$2 + "|" + rsFitz$1 + ")", rsNonAstral$1 = "[^" + rsAstralRange$2 + "]", rsRegional$1 = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair$1 = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsUpper = "[" + rsUpperRange + "]", rsZWJ$2 = "\\u200d";
var rsMiscLower = "(?:" + rsLower + "|" + rsMisc + ")", rsMiscUpper = "(?:" + rsUpper + "|" + rsMisc + ")", rsOptContrLower = "(?:" + rsApos + "(?:d|ll|m|re|s|t|ve))?", rsOptContrUpper = "(?:" + rsApos + "(?:D|LL|M|RE|S|T|VE))?", reOptMod$1 = rsModifier$1 + "?", rsOptVar$1 = "[" + rsVarRange$2 + "]?", rsOptJoin$1 = "(?:" + rsZWJ$2 + "(?:" + [
    rsNonAstral$1,
    rsRegional$1,
    rsSurrPair$1
].join("|") + ")" + rsOptVar$1 + reOptMod$1 + ")*", rsOrdLower = "\\d*(?:1st|2nd|3rd|(?![123])\\dth)(?=\\b|[A-Z_])", rsOrdUpper = "\\d*(?:1ST|2ND|3RD|(?![123])\\dTH)(?=\\b|[a-z_])", rsSeq$1 = rsOptVar$1 + reOptMod$1 + rsOptJoin$1, rsEmoji = "(?:" + [
    rsDingbat,
    rsRegional$1,
    rsSurrPair$1
].join("|") + ")" + rsSeq$1;
var reUnicodeWord = RegExp([
    rsUpper + "?" + rsLower + "+" + rsOptContrLower + "(?=" + [
        rsBreak,
        rsUpper,
        "$"
    ].join("|") + ")",
    rsMiscUpper + "+" + rsOptContrUpper + "(?=" + [
        rsBreak,
        rsUpper + rsMiscLower,
        "$"
    ].join("|") + ")",
    rsUpper + "?" + rsMiscLower + "+" + rsOptContrLower,
    rsUpper + "+" + rsOptContrUpper,
    rsOrdUpper,
    rsOrdLower,
    rsDigits,
    rsEmoji
].join("|"), "g");
function unicodeWords(string2) {
    return string2.match(reUnicodeWord) || [];
}
function words(string2, pattern1, guard) {
    string2 = toString(string2);
    pattern1 = guard ? void 0 : pattern1;
    if (pattern1 === void 0) {
        return hasUnicodeWord(string2) ? unicodeWords(string2) : asciiWords(string2);
    }
    return string2.match(pattern1) || [];
}
var rsApos$1 = "['\u2019]";
var reApos = RegExp(rsApos$1, "g");
function createCompounder(callback) {
    return function(string2) {
        return arrayReduce(words(deburr(string2).replace(reApos, "")), callback, "");
    };
}
var camelCase = createCompounder(function(result2, word, index) {
    word = word.toLowerCase();
    return result2 + (index ? capitalize1(word) : word);
});
function castArray() {
    if (!arguments.length) {
        return [];
    }
    var value8 = arguments[0];
    return isArray(value8) ? value8 : [
        value8
    ];
}
var nativeIsFinite = root.isFinite, nativeMin$2 = Math.min;
function createRound(methodName) {
    var func2 = Math[methodName];
    return function(number2, precision) {
        number2 = toNumber(number2);
        precision = precision == null ? 0 : nativeMin$2(toInteger(precision), 292);
        if (precision && nativeIsFinite(number2)) {
            var pair = (toString(number2) + "e").split("e"), value8 = func2(pair[0] + "e" + (+pair[1] + precision));
            pair = (toString(value8) + "e").split("e");
            return +(pair[0] + "e" + (+pair[1] - precision));
        }
        return func2(number2);
    };
}
var ceil = createRound("ceil");
function chain(value9) {
    var result2 = lodash(value9);
    result2.__chain__ = true;
    return result2;
}
var nativeCeil = Math.ceil, nativeMax$4 = Math.max;
function chunk1(array2, size21, guard) {
    if (guard ? isIterateeCall(array2, size21, guard) : size21 === void 0) {
        size21 = 1;
    } else {
        size21 = nativeMax$4(toInteger(size21), 0);
    }
    var length = array2 == null ? 0 : array2.length;
    if (!length || size21 < 1) {
        return [];
    }
    var index = 0, resIndex = 0, result2 = Array(nativeCeil(length / size21));
    while(index < length){
        result2[resIndex++] = baseSlice(array2, index, index += size21);
    }
    return result2;
}
function baseClamp(number2, lower, upper) {
    if (number2 === number2) {
        if (upper !== void 0) {
            number2 = number2 <= upper ? number2 : upper;
        }
        if (lower !== void 0) {
            number2 = number2 >= lower ? number2 : lower;
        }
    }
    return number2;
}
function clamp(number2, lower, upper) {
    if (upper === void 0) {
        upper = lower;
        lower = void 0;
    }
    if (upper !== void 0) {
        upper = toNumber(upper);
        upper = upper === upper ? upper : 0;
    }
    if (lower !== void 0) {
        lower = toNumber(lower);
        lower = lower === lower ? lower : 0;
    }
    return baseClamp(toNumber(number2), lower, upper);
}
function stackClear() {
    this.__data__ = new ListCache();
    this.size = 0;
}
function stackDelete(key) {
    var data = this.__data__, result2 = data["delete"](key);
    this.size = data.size;
    return result2;
}
function stackGet(key) {
    return this.__data__.get(key);
}
function stackHas(key) {
    return this.__data__.has(key);
}
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value9) {
    var data = this.__data__;
    if (data instanceof ListCache) {
        var pairs1 = data.__data__;
        if (!Map1 || pairs1.length < LARGE_ARRAY_SIZE - 1) {
            pairs1.push([
                key,
                value9
            ]);
            this.size = ++data.size;
            return this;
        }
        data = this.__data__ = new MapCache(pairs1);
    }
    data.set(key, value9);
    this.size = data.size;
    return this;
}
function Stack(entries) {
    var data = this.__data__ = new ListCache(entries);
    this.size = data.size;
}
Stack.prototype.clear = stackClear;
Stack.prototype["delete"] = stackDelete;
Stack.prototype.get = stackGet;
Stack.prototype.has = stackHas;
Stack.prototype.set = stackSet;
function baseAssign(object2, source) {
    return object2 && copyObject(source, keys1(source), object2);
}
function baseAssignIn(object2, source) {
    return object2 && copyObject(source, keysIn(source), object2);
}
var freeExports$2 = typeof exports == "object" && exports && !exports.nodeType && exports;
var freeModule$2 = freeExports$2 && typeof module == "object" && module && !module.nodeType && module;
var moduleExports$2 = freeModule$2 && freeModule$2.exports === freeExports$2;
var Buffer$1 = moduleExports$2 ? root.Buffer : void 0, allocUnsafe = Buffer$1 ? Buffer$1.allocUnsafe : void 0;
function cloneBuffer(buffer1, isDeep) {
    if (isDeep) {
        return buffer1.slice();
    }
    var length = buffer1.length, result2 = allocUnsafe ? allocUnsafe(length) : new buffer1.constructor(length);
    buffer1.copy(result2);
    return result2;
}
function arrayFilter(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value9 = array2[index];
        if (predicate(value9, index, array2)) {
            result2[resIndex++] = value9;
        }
    }
    return result2;
}
function stubArray() {
    return [];
}
var objectProto$f = Object.prototype;
var propertyIsEnumerable$1 = objectProto$f.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? stubArray : function(object2) {
    if (object2 == null) {
        return [];
    }
    object2 = Object(object2);
    return arrayFilter(nativeGetSymbols(object2), function(symbol) {
        return propertyIsEnumerable$1.call(object2, symbol);
    });
};
function copySymbols(source, object2) {
    return copyObject(source, getSymbols(source), object2);
}
var nativeGetSymbols$1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols$1 ? stubArray : function(object2) {
    var result2 = [];
    while(object2){
        arrayPush(result2, getSymbols(object2));
        object2 = getPrototype(object2);
    }
    return result2;
};
function copySymbolsIn(source, object2) {
    return copyObject(source, getSymbolsIn(source), object2);
}
function baseGetAllKeys(object2, keysFunc, symbolsFunc) {
    var result2 = keysFunc(object2);
    return isArray(object2) ? result2 : arrayPush(result2, symbolsFunc(object2));
}
function getAllKeys(object2) {
    return baseGetAllKeys(object2, keys1, getSymbols);
}
function getAllKeysIn(object2) {
    return baseGetAllKeys(object2, keysIn, getSymbolsIn);
}
var DataView1 = getNative(root, "DataView");
var Promise1 = getNative(root, "Promise");
var Set1 = getNative(root, "Set");
var mapTag$1 = "[object Map]", objectTag$2 = "[object Object]", promiseTag = "[object Promise]", setTag$1 = "[object Set]", weakMapTag$1 = "[object WeakMap]";
var dataViewTag$1 = "[object DataView]";
var dataViewCtorString = toSource(DataView1), mapCtorString = toSource(Map1), promiseCtorString = toSource(Promise1), setCtorString = toSource(Set1), weakMapCtorString = toSource(WeakMap1);
var getTag = baseGetTag;
if (DataView1 && getTag(new DataView1(new ArrayBuffer(1))) != dataViewTag$1 || Map1 && getTag(new Map1()) != mapTag$1 || Promise1 && getTag(Promise1.resolve()) != promiseTag || Set1 && getTag(new Set1()) != setTag$1 || WeakMap1 && getTag(new WeakMap1()) != weakMapTag$1) {
    getTag = function(value10) {
        var result2 = baseGetTag(value10), Ctor = result2 == objectTag$2 ? value10.constructor : void 0, ctorString = Ctor ? toSource(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag$1;
                case mapCtorString:
                    return mapTag$1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag$1;
                case weakMapCtorString:
                    return weakMapTag$1;
            }
        }
        return result2;
    };
}
var getTag$1 = getTag;
var objectProto$g = Object.prototype;
var hasOwnProperty$d = objectProto$g.hasOwnProperty;
function initCloneArray(array2) {
    var length = array2.length, result2 = new array2.constructor(length);
    if (length && typeof array2[0] == "string" && hasOwnProperty$d.call(array2, "index")) {
        result2.index = array2.index;
        result2.input = array2.input;
    }
    return result2;
}
var Uint8Array1 = root.Uint8Array;
function cloneArrayBuffer(arrayBuffer) {
    var result2 = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new Uint8Array1(result2).set(new Uint8Array1(arrayBuffer));
    return result2;
}
function cloneDataView(dataView, isDeep) {
    var buffer1 = isDeep ? cloneArrayBuffer(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer1, dataView.byteOffset, dataView.byteLength);
}
var reFlags = /\w*$/;
function cloneRegExp(regexp1) {
    var result2 = new regexp1.constructor(regexp1.source, reFlags.exec(regexp1));
    result2.lastIndex = regexp1.lastIndex;
    return result2;
}
var symbolProto$1 = Symbol1 ? Symbol1.prototype : void 0, symbolValueOf = symbolProto$1 ? symbolProto$1.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {
    };
}
function cloneTypedArray(typedArray, isDeep) {
    var buffer1 = isDeep ? cloneArrayBuffer(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer1, typedArray.byteOffset, typedArray.length);
}
var boolTag$1 = "[object Boolean]", dateTag$1 = "[object Date]", mapTag$2 = "[object Map]", numberTag$1 = "[object Number]", regexpTag$1 = "[object RegExp]", setTag$2 = "[object Set]", stringTag$1 = "[object String]", symbolTag$1 = "[object Symbol]";
var arrayBufferTag$1 = "[object ArrayBuffer]", dataViewTag$2 = "[object DataView]", float32Tag$1 = "[object Float32Array]", float64Tag$1 = "[object Float64Array]", int8Tag$1 = "[object Int8Array]", int16Tag$1 = "[object Int16Array]", int32Tag$1 = "[object Int32Array]", uint8Tag$1 = "[object Uint8Array]", uint8ClampedTag$1 = "[object Uint8ClampedArray]", uint16Tag$1 = "[object Uint16Array]", uint32Tag$1 = "[object Uint32Array]";
function initCloneByTag(object2, tag2, isDeep) {
    var Ctor = object2.constructor;
    switch(tag2){
        case arrayBufferTag$1:
            return cloneArrayBuffer(object2);
        case boolTag$1:
        case dateTag$1:
            return new Ctor(+object2);
        case dataViewTag$2:
            return cloneDataView(object2, isDeep);
        case float32Tag$1:
        case float64Tag$1:
        case int8Tag$1:
        case int16Tag$1:
        case int32Tag$1:
        case uint8Tag$1:
        case uint8ClampedTag$1:
        case uint16Tag$1:
        case uint32Tag$1:
            return cloneTypedArray(object2, isDeep);
        case mapTag$2:
            return new Ctor();
        case numberTag$1:
        case stringTag$1:
            return new Ctor(object2);
        case regexpTag$1:
            return cloneRegExp(object2);
        case setTag$2:
            return new Ctor();
        case symbolTag$1:
            return cloneSymbol(object2);
    }
}
function initCloneObject(object2) {
    return typeof object2.constructor == "function" && !isPrototype(object2) ? baseCreate(getPrototype(object2)) : {
    };
}
var mapTag$3 = "[object Map]";
function baseIsMap(value10) {
    return isObjectLike(value10) && getTag$1(value10) == mapTag$3;
}
var nodeIsMap = nodeUtil && nodeUtil.isMap;
var isMap = nodeIsMap ? baseUnary(nodeIsMap) : baseIsMap;
var setTag$3 = "[object Set]";
function baseIsSet(value10) {
    return isObjectLike(value10) && getTag$1(value10) == setTag$3;
}
var nodeIsSet = nodeUtil && nodeUtil.isSet;
var isSet = nodeIsSet ? baseUnary(nodeIsSet) : baseIsSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag$2 = "[object Arguments]", arrayTag$1 = "[object Array]", boolTag$2 = "[object Boolean]", dateTag$2 = "[object Date]", errorTag$2 = "[object Error]", funcTag$2 = "[object Function]", genTag$1 = "[object GeneratorFunction]", mapTag$4 = "[object Map]", numberTag$2 = "[object Number]", objectTag$3 = "[object Object]", regexpTag$2 = "[object RegExp]", setTag$4 = "[object Set]", stringTag$2 = "[object String]", symbolTag$2 = "[object Symbol]", weakMapTag$2 = "[object WeakMap]";
var arrayBufferTag$2 = "[object ArrayBuffer]", dataViewTag$3 = "[object DataView]", float32Tag$2 = "[object Float32Array]", float64Tag$2 = "[object Float64Array]", int8Tag$2 = "[object Int8Array]", int16Tag$2 = "[object Int16Array]", int32Tag$2 = "[object Int32Array]", uint8Tag$2 = "[object Uint8Array]", uint8ClampedTag$2 = "[object Uint8ClampedArray]", uint16Tag$2 = "[object Uint16Array]", uint32Tag$2 = "[object Uint32Array]";
var cloneableTags = {
};
cloneableTags[argsTag$2] = cloneableTags[arrayTag$1] = cloneableTags[arrayBufferTag$2] = cloneableTags[dataViewTag$3] = cloneableTags[boolTag$2] = cloneableTags[dateTag$2] = cloneableTags[float32Tag$2] = cloneableTags[float64Tag$2] = cloneableTags[int8Tag$2] = cloneableTags[int16Tag$2] = cloneableTags[int32Tag$2] = cloneableTags[mapTag$4] = cloneableTags[numberTag$2] = cloneableTags[objectTag$3] = cloneableTags[regexpTag$2] = cloneableTags[setTag$4] = cloneableTags[stringTag$2] = cloneableTags[symbolTag$2] = cloneableTags[uint8Tag$2] = cloneableTags[uint8ClampedTag$2] = cloneableTags[uint16Tag$2] = cloneableTags[uint32Tag$2] = true;
cloneableTags[errorTag$2] = cloneableTags[funcTag$2] = cloneableTags[weakMapTag$2] = false;
function baseClone(value10, bitmask, customizer, key, object2, stack) {
    var result2, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result2 = object2 ? customizer(value10, key, object2, stack) : customizer(value10);
    }
    if (result2 !== void 0) {
        return result2;
    }
    if (!isObject1(value10)) {
        return value10;
    }
    var isArr = isArray(value10);
    if (isArr) {
        result2 = initCloneArray(value10);
        if (!isDeep) {
            return copyArray(value10, result2);
        }
    } else {
        var tag2 = getTag$1(value10), isFunc = tag2 == funcTag$2 || tag2 == genTag$1;
        if (isBuffer(value10)) {
            return cloneBuffer(value10, isDeep);
        }
        if (tag2 == objectTag$3 || tag2 == argsTag$2 || isFunc && !object2) {
            result2 = isFlat || isFunc ? {
            } : initCloneObject(value10);
            if (!isDeep) {
                return isFlat ? copySymbolsIn(value10, baseAssignIn(result2, value10)) : copySymbols(value10, baseAssign(result2, value10));
            }
        } else {
            if (!cloneableTags[tag2]) {
                return object2 ? value10 : {
                };
            }
            result2 = initCloneByTag(value10, tag2, isDeep);
        }
    }
    stack || (stack = new Stack());
    var stacked = stack.get(value10);
    if (stacked) {
        return stacked;
    }
    stack.set(value10, result2);
    if (isSet(value10)) {
        value10.forEach(function(subValue) {
            result2.add(baseClone(subValue, bitmask, customizer, subValue, value10, stack));
        });
    } else if (isMap(value10)) {
        value10.forEach(function(subValue, key2) {
            result2.set(key2, baseClone(subValue, bitmask, customizer, key2, value10, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? getAllKeysIn : getAllKeys : isFlat ? keysIn : keys1;
    var props = isArr ? void 0 : keysFunc(value10);
    arrayEach(props || value10, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value10[key2];
        }
        assignValue(result2, key2, baseClone(subValue, bitmask, customizer, key2, value10, stack));
    });
    return result2;
}
var CLONE_SYMBOLS_FLAG$1 = 4;
function clone(value10) {
    return baseClone(value10, CLONE_SYMBOLS_FLAG$1);
}
var CLONE_DEEP_FLAG$1 = 1, CLONE_SYMBOLS_FLAG$2 = 4;
function cloneDeep(value10) {
    return baseClone(value10, CLONE_DEEP_FLAG$1 | CLONE_SYMBOLS_FLAG$2);
}
var CLONE_DEEP_FLAG$2 = 1, CLONE_SYMBOLS_FLAG$3 = 4;
function cloneDeepWith(value10, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseClone(value10, CLONE_DEEP_FLAG$2 | CLONE_SYMBOLS_FLAG$3, customizer);
}
var CLONE_SYMBOLS_FLAG$4 = 4;
function cloneWith(value10, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseClone(value10, CLONE_SYMBOLS_FLAG$4, customizer);
}
function wrapperCommit() {
    return new LodashWrapper(this.value(), this.__chain__);
}
function compact(array2) {
    var index = -1, length = array2 == null ? 0 : array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value10 = array2[index];
        if (value10) {
            result2[resIndex++] = value10;
        }
    }
    return result2;
}
function concat1() {
    var length = arguments.length;
    if (!length) {
        return [];
    }
    var args4 = Array(length - 1), array2 = arguments[0], index = length;
    while(index--){
        args4[index - 1] = arguments[index];
    }
    return arrayPush(isArray(array2) ? copyArray(array2) : [
        array2
    ], baseFlatten(args4, 1));
}
var HASH_UNDEFINED$2 = "__lodash_hash_undefined__";
function setCacheAdd(value11) {
    this.__data__.set(value11, HASH_UNDEFINED$2);
    return this;
}
function setCacheHas(value11) {
    return this.__data__.has(value11);
}
function SetCache(values2) {
    var index = -1, length = values2 == null ? 0 : values2.length;
    this.__data__ = new MapCache();
    while(++index < length){
        this.add(values2[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = setCacheAdd;
SetCache.prototype.has = setCacheHas;
function arraySome(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (predicate(array2[index], index, array2)) {
            return true;
        }
    }
    return false;
}
function cacheHas(cache, key) {
    return cache.has(key);
}
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array2, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array2.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array2);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array2;
    }
    var index = -1, result2 = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new SetCache() : void 0;
    stack.set(array2, other);
    stack.set(other, array2);
    while(++index < arrLength){
        var arrValue = array2[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array2, stack) : customizer(arrValue, othValue, index, array2, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result2 = false;
            break;
        }
        if (seen) {
            if (!arraySome(other, function(othValue2, othIndex) {
                if (!cacheHas(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result2 = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result2 = false;
            break;
        }
    }
    stack["delete"](array2);
    stack["delete"](other);
    return result2;
}
function mapToArray(map2) {
    var index = -1, result2 = Array(map2.size);
    map2.forEach(function(value11, key) {
        result2[++index] = [
            key,
            value11
        ];
    });
    return result2;
}
function setToArray(set2) {
    var index = -1, result2 = Array(set2.size);
    set2.forEach(function(value11) {
        result2[++index] = value11;
    });
    return result2;
}
var COMPARE_PARTIAL_FLAG$1 = 1, COMPARE_UNORDERED_FLAG$1 = 2;
var boolTag$3 = "[object Boolean]", dateTag$3 = "[object Date]", errorTag$3 = "[object Error]", mapTag$5 = "[object Map]", numberTag$3 = "[object Number]", regexpTag$3 = "[object RegExp]", setTag$5 = "[object Set]", stringTag$3 = "[object String]", symbolTag$3 = "[object Symbol]";
var arrayBufferTag$3 = "[object ArrayBuffer]", dataViewTag$4 = "[object DataView]";
var symbolProto$2 = Symbol1 ? Symbol1.prototype : void 0, symbolValueOf$1 = symbolProto$2 ? symbolProto$2.valueOf : void 0;
function equalByTag(object2, other, tag3, bitmask, customizer, equalFunc, stack) {
    switch(tag3){
        case dataViewTag$4:
            if (object2.byteLength != other.byteLength || object2.byteOffset != other.byteOffset) {
                return false;
            }
            object2 = object2.buffer;
            other = other.buffer;
        case arrayBufferTag$3:
            if (object2.byteLength != other.byteLength || !equalFunc(new Uint8Array1(object2), new Uint8Array1(other))) {
                return false;
            }
            return true;
        case boolTag$3:
        case dateTag$3:
        case numberTag$3:
            return eq(+object2, +other);
        case errorTag$3:
            return object2.name == other.name && object2.message == other.message;
        case regexpTag$3:
        case stringTag$3:
            return object2 == other + "";
        case mapTag$5:
            var convert = mapToArray;
        case setTag$5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG$1;
            convert || (convert = setToArray);
            if (object2.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object2);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG$1;
            stack.set(object2, other);
            var result2 = equalArrays(convert(object2), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object2);
            return result2;
        case symbolTag$3:
            if (symbolValueOf$1) {
                return symbolValueOf$1.call(object2) == symbolValueOf$1.call(other);
            }
    }
    return false;
}
var COMPARE_PARTIAL_FLAG$2 = 1;
var objectProto$h = Object.prototype;
var hasOwnProperty$e = objectProto$h.hasOwnProperty;
function equalObjects(object2, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG$2, objProps = getAllKeys(object2), objLength = objProps.length, othProps = getAllKeys(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty$e.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object2);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object2;
    }
    var result2 = true;
    stack.set(object2, other);
    stack.set(other, object2);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object2[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object2, stack) : customizer(objValue, othValue, key, object2, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result2 = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result2 && !skipCtor) {
        var objCtor = object2.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object2 && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result2 = false;
        }
    }
    stack["delete"](object2);
    stack["delete"](other);
    return result2;
}
var COMPARE_PARTIAL_FLAG$3 = 1;
var argsTag$3 = "[object Arguments]", arrayTag$2 = "[object Array]", objectTag$4 = "[object Object]";
var objectProto$i = Object.prototype;
var hasOwnProperty$f = objectProto$i.hasOwnProperty;
function baseIsEqualDeep(object2, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = isArray(object2), othIsArr = isArray(other), objTag = objIsArr ? arrayTag$2 : getTag$1(object2), othTag = othIsArr ? arrayTag$2 : getTag$1(other);
    objTag = objTag == argsTag$3 ? objectTag$4 : objTag;
    othTag = othTag == argsTag$3 ? objectTag$4 : othTag;
    var objIsObj = objTag == objectTag$4, othIsObj = othTag == objectTag$4, isSameTag = objTag == othTag;
    if (isSameTag && isBuffer(object2)) {
        if (!isBuffer(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new Stack());
        return objIsArr || isTypedArray(object2) ? equalArrays(object2, other, bitmask, customizer, equalFunc, stack) : equalByTag(object2, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG$3)) {
        var objIsWrapped = objIsObj && hasOwnProperty$f.call(object2, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty$f.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object2.value() : object2, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new Stack());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new Stack());
    return equalObjects(object2, other, bitmask, customizer, equalFunc, stack);
}
function baseIsEqual(value11, other, bitmask, customizer, stack) {
    if (value11 === other) {
        return true;
    }
    if (value11 == null || other == null || !isObjectLike(value11) && !isObjectLike(other)) {
        return value11 !== value11 && other !== other;
    }
    return baseIsEqualDeep(value11, other, bitmask, customizer, baseIsEqual, stack);
}
var COMPARE_PARTIAL_FLAG$4 = 1, COMPARE_UNORDERED_FLAG$2 = 2;
function baseIsMatch(object2, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object2 == null) {
        return !length;
    }
    object2 = Object(object2);
    while(index--){
        var data = matchData[index];
        if (noCustomizer && data[2] ? data[1] !== object2[data[0]] : !(data[0] in object2)) {
            return false;
        }
    }
    while(++index < length){
        data = matchData[index];
        var key = data[0], objValue = object2[key], srcValue = data[1];
        if (noCustomizer && data[2]) {
            if (objValue === void 0 && !(key in object2)) {
                return false;
            }
        } else {
            var stack = new Stack();
            if (customizer) {
                var result2 = customizer(objValue, srcValue, key, object2, source, stack);
            }
            if (!(result2 === void 0 ? baseIsEqual(srcValue, objValue, COMPARE_PARTIAL_FLAG$4 | COMPARE_UNORDERED_FLAG$2, customizer, stack) : result2)) {
                return false;
            }
        }
    }
    return true;
}
function isStrictComparable(value11) {
    return value11 === value11 && !isObject1(value11);
}
function getMatchData(object2) {
    var result2 = keys1(object2), length = result2.length;
    while(length--){
        var key = result2[length], value11 = object2[key];
        result2[length] = [
            key,
            value11,
            isStrictComparable(value11)
        ];
    }
    return result2;
}
function matchesStrictComparable(key, srcValue) {
    return function(object2) {
        if (object2 == null) {
            return false;
        }
        return object2[key] === srcValue && (srcValue !== void 0 || key in Object(object2));
    };
}
function baseMatches(source) {
    var matchData = getMatchData(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return matchesStrictComparable(matchData[0][0], matchData[0][1]);
    }
    return function(object2) {
        return object2 === source || baseIsMatch(object2, source, matchData);
    };
}
function baseHasIn(object2, key) {
    return object2 != null && key in Object(object2);
}
function hasPath(object2, path3, hasFunc) {
    path3 = castPath(path3, object2);
    var index = -1, length = path3.length, result2 = false;
    while(++index < length){
        var key = toKey(path3[index]);
        if (!(result2 = object2 != null && hasFunc(object2, key))) {
            break;
        }
        object2 = object2[key];
    }
    if (result2 || ++index != length) {
        return result2;
    }
    length = object2 == null ? 0 : object2.length;
    return !!length && isLength(length) && isIndex(key, length) && (isArray(object2) || isArguments(object2));
}
function hasIn(object2, path3) {
    return object2 != null && hasPath(object2, path3, baseHasIn);
}
var COMPARE_PARTIAL_FLAG$5 = 1, COMPARE_UNORDERED_FLAG$3 = 2;
function baseMatchesProperty(path3, srcValue) {
    if (isKey(path3) && isStrictComparable(srcValue)) {
        return matchesStrictComparable(toKey(path3), srcValue);
    }
    return function(object2) {
        var objValue = get(object2, path3);
        return objValue === void 0 && objValue === srcValue ? hasIn(object2, path3) : baseIsEqual(srcValue, objValue, COMPARE_PARTIAL_FLAG$5 | COMPARE_UNORDERED_FLAG$3);
    };
}
function baseProperty(key) {
    return function(object2) {
        return object2 == null ? void 0 : object2[key];
    };
}
function basePropertyDeep(path3) {
    return function(object2) {
        return baseGet(object2, path3);
    };
}
function property(path3) {
    return isKey(path3) ? baseProperty(toKey(path3)) : basePropertyDeep(path3);
}
function baseIteratee(value12) {
    if (typeof value12 == "function") {
        return value12;
    }
    if (value12 == null) {
        return identity;
    }
    if (typeof value12 == "object") {
        return isArray(value12) ? baseMatchesProperty(value12[0], value12[1]) : baseMatches(value12);
    }
    return property(value12);
}
var FUNC_ERROR_TEXT$4 = "Expected a function";
function cond(pairs2) {
    var length = pairs2 == null ? 0 : pairs2.length, toIteratee = baseIteratee;
    pairs2 = !length ? [] : arrayMap(pairs2, function(pair) {
        if (typeof pair[1] != "function") {
            throw new TypeError(FUNC_ERROR_TEXT$4);
        }
        return [
            toIteratee(pair[0]),
            pair[1]
        ];
    });
    return baseRest(function(args4) {
        var index = -1;
        while(++index < length){
            var pair = pairs2[index];
            if (apply(pair[0], this, args4)) {
                return apply(pair[1], this, args4);
            }
        }
    });
}
function baseConformsTo(object2, source, props) {
    var length = props.length;
    if (object2 == null) {
        return !length;
    }
    object2 = Object(object2);
    while(length--){
        var key = props[length], predicate = source[key], value12 = object2[key];
        if (value12 === void 0 && !(key in object2) || !predicate(value12)) {
            return false;
        }
    }
    return true;
}
function baseConforms(source) {
    var props = keys1(source);
    return function(object2) {
        return baseConformsTo(object2, source, props);
    };
}
var CLONE_DEEP_FLAG$3 = 1;
function conforms(source) {
    return baseConforms(baseClone(source, CLONE_DEEP_FLAG$3));
}
function conformsTo(object2, source) {
    return source == null || baseConformsTo(object2, source, keys1(source));
}
function arrayAggregator(array2, setter, iteratee2, accumulator) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        var value13 = array2[index];
        setter(accumulator, value13, iteratee2(value13), array2);
    }
    return accumulator;
}
function createBaseFor(fromRight) {
    return function(object2, iteratee2, keysFunc) {
        var index = -1, iterable = Object(object2), props = keysFunc(object2), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee2(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object2;
    };
}
var baseFor = createBaseFor();
function baseForOwn(object2, iteratee2) {
    return object2 && baseFor(object2, iteratee2, keys1);
}
function createBaseEach(eachFunc, fromRight) {
    return function(collection2, iteratee2) {
        if (collection2 == null) {
            return collection2;
        }
        if (!isArrayLike(collection2)) {
            return eachFunc(collection2, iteratee2);
        }
        var length = collection2.length, index = fromRight ? length : -1, iterable = Object(collection2);
        while(fromRight ? index-- : ++index < length){
            if (iteratee2(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection2;
    };
}
var baseEach = createBaseEach(baseForOwn);
function baseAggregator(collection2, setter, iteratee2, accumulator) {
    baseEach(collection2, function(value14, key, collection3) {
        setter(accumulator, value14, iteratee2(value14), collection3);
    });
    return accumulator;
}
function createAggregator(setter, initializer) {
    return function(collection2, iteratee2) {
        var func2 = isArray(collection2) ? arrayAggregator : baseAggregator, accumulator = initializer ? initializer() : {
        };
        return func2(collection2, setter, baseIteratee(iteratee2), accumulator);
    };
}
var objectProto$j = Object.prototype;
var hasOwnProperty$g = objectProto$j.hasOwnProperty;
var countBy = createAggregator(function(result2, value14, key) {
    if (hasOwnProperty$g.call(result2, key)) {
        ++result2[key];
    } else {
        baseAssignValue(result2, key, 1);
    }
});
function create(prototype, properties1) {
    var result2 = baseCreate(prototype);
    return properties1 == null ? result2 : baseAssign(result2, properties1);
}
var WRAP_CURRY_FLAG$5 = 8;
function curry(func2, arity, guard) {
    arity = guard ? void 0 : arity;
    var result2 = createWrap(func2, WRAP_CURRY_FLAG$5, void 0, void 0, void 0, void 0, void 0, arity);
    result2.placeholder = curry.placeholder;
    return result2;
}
curry.placeholder = {
};
var WRAP_CURRY_RIGHT_FLAG$3 = 16;
function curryRight(func2, arity, guard) {
    arity = guard ? void 0 : arity;
    var result2 = createWrap(func2, WRAP_CURRY_RIGHT_FLAG$3, void 0, void 0, void 0, void 0, void 0, arity);
    result2.placeholder = curryRight.placeholder;
    return result2;
}
curryRight.placeholder = {
};
var now = function() {
    return root.Date.now();
};
var FUNC_ERROR_TEXT$5 = "Expected a function";
var nativeMax$5 = Math.max, nativeMin$3 = Math.min;
function debounce(func2, wait, options12) {
    var lastArgs, lastThis, maxWait, result2, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$5);
    }
    wait = toNumber(wait) || 0;
    if (isObject1(options12)) {
        leading = !!options12.leading;
        maxing = "maxWait" in options12;
        maxWait = maxing ? nativeMax$5(toNumber(options12.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options12 ? !!options12.trailing : trailing;
    }
    function invokeFunc(time) {
        var args4 = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result2 = func2.apply(thisArg, args4);
        return result2;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result2;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin$3(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = now();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result2;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result2 : trailingEdge(now());
    }
    function debounced() {
        var time = now(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result2;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
function defaultTo(value14, defaultValue) {
    return value14 == null || value14 !== value14 ? defaultValue : value14;
}
var objectProto$k = Object.prototype;
var hasOwnProperty$h = objectProto$k.hasOwnProperty;
var defaults1 = baseRest(function(object2, sources) {
    object2 = Object(object2);
    var index = -1;
    var length = sources.length;
    var guard = length > 2 ? sources[2] : void 0;
    if (guard && isIterateeCall(sources[0], sources[1], guard)) {
        length = 1;
    }
    while(++index < length){
        var source = sources[index];
        var props = keysIn(source);
        var propsIndex = -1;
        var propsLength = props.length;
        while(++propsIndex < propsLength){
            var key = props[propsIndex];
            var value14 = object2[key];
            if (value14 === void 0 || eq(value14, objectProto$k[key]) && !hasOwnProperty$h.call(object2, key)) {
                object2[key] = source[key];
            }
        }
    }
    return object2;
});
function assignMergeValue(object2, key, value15) {
    if (value15 !== void 0 && !eq(object2[key], value15) || value15 === void 0 && !(key in object2)) {
        baseAssignValue(object2, key, value15);
    }
}
function isArrayLikeObject(value15) {
    return isObjectLike(value15) && isArrayLike(value15);
}
function safeGet(object2, key) {
    if (key === "constructor" && typeof object2[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object2[key];
}
function toPlainObject(value15) {
    return copyObject(value15, keysIn(value15));
}
function baseMergeDeep(object2, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = safeGet(object2, key), srcValue = safeGet(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        assignMergeValue(object2, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object2, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = isArray(srcValue), isBuff = !isArr && isBuffer(srcValue), isTyped = !isArr && !isBuff && isTypedArray(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (isArray(objValue)) {
                newValue = objValue;
            } else if (isArrayLikeObject(objValue)) {
                newValue = copyArray(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = cloneBuffer(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = cloneTypedArray(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (isPlainObject(srcValue) || isArguments(srcValue)) {
            newValue = objValue;
            if (isArguments(objValue)) {
                newValue = toPlainObject(objValue);
            } else if (!isObject1(objValue) || isFunction(objValue)) {
                newValue = initCloneObject(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    assignMergeValue(object2, key, newValue);
}
function baseMerge(object2, source, srcIndex, customizer, stack) {
    if (object2 === source) {
        return;
    }
    baseFor(source, function(srcValue, key) {
        stack || (stack = new Stack());
        if (isObject1(srcValue)) {
            baseMergeDeep(object2, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(safeGet(object2, key), srcValue, key + "", object2, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            assignMergeValue(object2, key, newValue);
        }
    }, keysIn);
}
function customDefaultsMerge(objValue, srcValue, key, object2, source, stack) {
    if (isObject1(objValue) && isObject1(srcValue)) {
        stack.set(srcValue, objValue);
        baseMerge(objValue, srcValue, void 0, customDefaultsMerge, stack);
        stack["delete"](srcValue);
    }
    return objValue;
}
var mergeWith = createAssigner(function(object2, source, srcIndex, customizer) {
    baseMerge(object2, source, srcIndex, customizer);
});
var defaultsDeep = baseRest(function(args4) {
    args4.push(void 0, customDefaultsMerge);
    return apply(mergeWith, void 0, args4);
});
var FUNC_ERROR_TEXT$6 = "Expected a function";
function baseDelay(func2, wait, args4) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$6);
    }
    return setTimeout(function() {
        func2.apply(void 0, args4);
    }, wait);
}
var defer = baseRest(function(func2, args4) {
    return baseDelay(func2, 1, args4);
});
var delay = baseRest(function(func2, wait, args4) {
    return baseDelay(func2, toNumber(wait) || 0, args4);
});
function arrayIncludesWith(array2, value15, comparator) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (comparator(value15, array2[index])) {
            return true;
        }
    }
    return false;
}
var LARGE_ARRAY_SIZE$1 = 200;
function baseDifference(array2, values2, iteratee2, comparator) {
    var index = -1, includes2 = arrayIncludes, isCommon = true, length = array2.length, result2 = [], valuesLength = values2.length;
    if (!length) {
        return result2;
    }
    if (iteratee2) {
        values2 = arrayMap(values2, baseUnary(iteratee2));
    }
    if (comparator) {
        includes2 = arrayIncludesWith;
        isCommon = false;
    } else if (values2.length >= LARGE_ARRAY_SIZE$1) {
        includes2 = cacheHas;
        isCommon = false;
        values2 = new SetCache(values2);
    }
    outer: while(++index < length){
        var value15 = array2[index], computed = iteratee2 == null ? value15 : iteratee2(value15);
        value15 = comparator || value15 !== 0 ? value15 : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values2[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result2.push(value15);
        } else if (!includes2(values2, computed, comparator)) {
            result2.push(value15);
        }
    }
    return result2;
}
var difference = baseRest(function(array2, values2) {
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true)) : [];
});
function last(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? array2[length - 1] : void 0;
}
var differenceBy = baseRest(function(array2, values2) {
    var iteratee2 = last(values2);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true), baseIteratee(iteratee2)) : [];
});
var differenceWith = baseRest(function(array2, values2) {
    var comparator = last(values2);
    if (isArrayLikeObject(comparator)) {
        comparator = void 0;
    }
    return isArrayLikeObject(array2) ? baseDifference(array2, baseFlatten(values2, 1, isArrayLikeObject, true), void 0, comparator) : [];
});
var divide = createMathOperation(function(dividend, divisor) {
    return dividend / divisor;
}, 1);
function drop(array2, n, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n = guard || n === void 0 ? 1 : toInteger(n);
    return baseSlice(array2, n < 0 ? 0 : n, length);
}
function dropRight(array2, n, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n = guard || n === void 0 ? 1 : toInteger(n);
    n = length - n;
    return baseSlice(array2, 0, n < 0 ? 0 : n);
}
function baseWhile(array2, predicate, isDrop, fromRight) {
    var length = array2.length, index = fromRight ? length : -1;
    while((fromRight ? index-- : ++index < length) && predicate(array2[index], index, array2)){
    }
    return isDrop ? baseSlice(array2, fromRight ? 0 : index, fromRight ? index + 1 : length) : baseSlice(array2, fromRight ? index + 1 : 0, fromRight ? length : index);
}
function dropRightWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), true, true) : [];
}
function dropWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), true) : [];
}
function castFunction(value16) {
    return typeof value16 == "function" ? value16 : identity;
}
function forEach(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayEach : baseEach;
    return func2(collection2, castFunction(iteratee2));
}
function arrayEachRight(array2, iteratee2) {
    var length = array2 == null ? 0 : array2.length;
    while(length--){
        if (iteratee2(array2[length], length, array2) === false) {
            break;
        }
    }
    return array2;
}
var baseForRight = createBaseFor(true);
function baseForOwnRight(object2, iteratee2) {
    return object2 && baseForRight(object2, iteratee2, keys1);
}
var baseEachRight = createBaseEach(baseForOwnRight, true);
function forEachRight(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayEachRight : baseEachRight;
    return func2(collection2, castFunction(iteratee2));
}
function endsWith(string2, target, position2) {
    string2 = toString(string2);
    target = baseToString(target);
    var length = string2.length;
    position2 = position2 === void 0 ? length : baseClamp(toInteger(position2), 0, length);
    var end = position2;
    position2 -= target.length;
    return position2 >= 0 && string2.slice(position2, end) == target;
}
function baseToPairs(object2, props) {
    return arrayMap(props, function(key) {
        return [
            key,
            object2[key]
        ];
    });
}
function setToPairs(set2) {
    var index = -1, result2 = Array(set2.size);
    set2.forEach(function(value16) {
        result2[++index] = [
            value16,
            value16
        ];
    });
    return result2;
}
var mapTag$6 = "[object Map]", setTag$6 = "[object Set]";
function createToPairs(keysFunc) {
    return function(object2) {
        var tag3 = getTag$1(object2);
        if (tag3 == mapTag$6) {
            return mapToArray(object2);
        }
        if (tag3 == setTag$6) {
            return setToPairs(object2);
        }
        return baseToPairs(object2, keysFunc(object2));
    };
}
var toPairs = createToPairs(keys1);
var toPairsIn = createToPairs(keysIn);
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = basePropertyOf(htmlEscapes);
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string2) {
    string2 = toString(string2);
    return string2 && reHasUnescapedHtml.test(string2) ? string2.replace(reUnescapedHtml, escapeHtmlChar) : string2;
}
var reRegExpChar$1 = /[\\^$.*+?()[\]{}|]/g, reHasRegExpChar = RegExp(reRegExpChar$1.source);
function escapeRegExp(string2) {
    string2 = toString(string2);
    return string2 && reHasRegExpChar.test(string2) ? string2.replace(reRegExpChar$1, "\\$&") : string2;
}
function arrayEvery(array2, predicate) {
    var index = -1, length = array2 == null ? 0 : array2.length;
    while(++index < length){
        if (!predicate(array2[index], index, array2)) {
            return false;
        }
    }
    return true;
}
function baseEvery(collection2, predicate) {
    var result2 = true;
    baseEach(collection2, function(value16, index, collection3) {
        result2 = !!predicate(value16, index, collection3);
        return result2;
    });
    return result2;
}
function every(collection2, predicate, guard) {
    var func2 = isArray(collection2) ? arrayEvery : baseEvery;
    if (guard && isIterateeCall(collection2, predicate, guard)) {
        predicate = void 0;
    }
    return func2(collection2, baseIteratee(predicate));
}
var MAX_ARRAY_LENGTH$1 = 4294967295;
function toLength(value16) {
    return value16 ? baseClamp(toInteger(value16), 0, MAX_ARRAY_LENGTH$1) : 0;
}
function baseFill(array2, value16, start, end) {
    var length = array2.length;
    start = toInteger(start);
    if (start < 0) {
        start = -start > length ? 0 : length + start;
    }
    end = end === void 0 || end > length ? length : toInteger(end);
    if (end < 0) {
        end += length;
    }
    end = start > end ? 0 : toLength(end);
    while(start < end){
        array2[start++] = value16;
    }
    return array2;
}
function fill(array2, value16, start, end) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    if (start && typeof start != "number" && isIterateeCall(array2, value16, start)) {
        start = 0;
        end = length;
    }
    return baseFill(array2, value16, start, end);
}
function baseFilter(collection2, predicate) {
    var result2 = [];
    baseEach(collection2, function(value16, index, collection3) {
        if (predicate(value16, index, collection3)) {
            result2.push(value16);
        }
    });
    return result2;
}
function filter(collection2, predicate) {
    var func2 = isArray(collection2) ? arrayFilter : baseFilter;
    return func2(collection2, baseIteratee(predicate));
}
function createFind(findIndexFunc) {
    return function(collection2, predicate, fromIndex) {
        var iterable = Object(collection2);
        if (!isArrayLike(collection2)) {
            var iteratee2 = baseIteratee(predicate);
            collection2 = keys1(collection2);
            predicate = function(key) {
                return iteratee2(iterable[key], key, iterable);
            };
        }
        var index = findIndexFunc(collection2, predicate, fromIndex);
        return index > -1 ? iterable[iteratee2 ? collection2[index] : index] : void 0;
    };
}
var nativeMax$6 = Math.max;
function findIndex(array2, predicate, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = fromIndex == null ? 0 : toInteger(fromIndex);
    if (index < 0) {
        index = nativeMax$6(length + index, 0);
    }
    return baseFindIndex(array2, baseIteratee(predicate), index);
}
var find = createFind(findIndex);
function baseFindKey(collection2, predicate, eachFunc) {
    var result2;
    eachFunc(collection2, function(value16, key, collection3) {
        if (predicate(value16, key, collection3)) {
            result2 = key;
            return false;
        }
    });
    return result2;
}
function findKey(object2, predicate) {
    return baseFindKey(object2, baseIteratee(predicate), baseForOwn);
}
var nativeMax$7 = Math.max, nativeMin$4 = Math.min;
function findLastIndex(array2, predicate, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = length - 1;
    if (fromIndex !== void 0) {
        index = toInteger(fromIndex);
        index = fromIndex < 0 ? nativeMax$7(length + index, 0) : nativeMin$4(index, length - 1);
    }
    return baseFindIndex(array2, baseIteratee(predicate), index, true);
}
var findLast = createFind(findLastIndex);
function findLastKey(object2, predicate) {
    return baseFindKey(object2, baseIteratee(predicate), baseForOwnRight);
}
function head1(array2) {
    return array2 && array2.length ? array2[0] : void 0;
}
function baseMap(collection2, iteratee2) {
    var index = -1, result2 = isArrayLike(collection2) ? Array(collection2.length) : [];
    baseEach(collection2, function(value16, key, collection3) {
        result2[++index] = iteratee2(value16, key, collection3);
    });
    return result2;
}
function map1(collection2, iteratee2) {
    var func2 = isArray(collection2) ? arrayMap : baseMap;
    return func2(collection2, baseIteratee(iteratee2));
}
function flatMap(collection2, iteratee2) {
    return baseFlatten(map1(collection2, iteratee2), 1);
}
var INFINITY$3 = 1 / 0;
function flatMapDeep(collection2, iteratee2) {
    return baseFlatten(map1(collection2, iteratee2), INFINITY$3);
}
function flatMapDepth(collection2, iteratee2, depth) {
    depth = depth === void 0 ? 1 : toInteger(depth);
    return baseFlatten(map1(collection2, iteratee2), depth);
}
var INFINITY$4 = 1 / 0;
function flattenDeep(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseFlatten(array2, INFINITY$4) : [];
}
function flattenDepth(array2, depth) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    depth = depth === void 0 ? 1 : toInteger(depth);
    return baseFlatten(array2, depth);
}
var WRAP_FLIP_FLAG$2 = 512;
function flip(func2) {
    return createWrap(func2, WRAP_FLIP_FLAG$2);
}
var floor = createRound("floor");
var FUNC_ERROR_TEXT$7 = "Expected a function";
var WRAP_CURRY_FLAG$6 = 8, WRAP_PARTIAL_FLAG$5 = 32, WRAP_ARY_FLAG$4 = 128, WRAP_REARG_FLAG$2 = 256;
function createFlow(fromRight) {
    return flatRest(function(funcs) {
        var length = funcs.length, index = length, prereq = LodashWrapper.prototype.thru;
        if (fromRight) {
            funcs.reverse();
        }
        while(index--){
            var func2 = funcs[index];
            if (typeof func2 != "function") {
                throw new TypeError(FUNC_ERROR_TEXT$7);
            }
            if (prereq && !wrapper && getFuncName(func2) == "wrapper") {
                var wrapper = new LodashWrapper([], true);
            }
        }
        index = wrapper ? index : length;
        while(++index < length){
            func2 = funcs[index];
            var funcName = getFuncName(func2), data = funcName == "wrapper" ? getData(func2) : void 0;
            if (data && isLaziable(data[0]) && data[1] == (WRAP_ARY_FLAG$4 | WRAP_CURRY_FLAG$6 | WRAP_PARTIAL_FLAG$5 | WRAP_REARG_FLAG$2) && !data[4].length && data[9] == 1) {
                wrapper = wrapper[getFuncName(data[0])].apply(wrapper, data[3]);
            } else {
                wrapper = func2.length == 1 && isLaziable(func2) ? wrapper[funcName]() : wrapper.thru(func2);
            }
        }
        return function() {
            var args4 = arguments, value16 = args4[0];
            if (wrapper && args4.length == 1 && isArray(value16)) {
                return wrapper.plant(value16).value();
            }
            var index2 = 0, result2 = length ? funcs[index2].apply(this, args4) : value16;
            while(++index2 < length){
                result2 = funcs[index2].call(this, result2);
            }
            return result2;
        };
    });
}
var flow = createFlow();
var flowRight = createFlow(true);
function forIn(object2, iteratee2) {
    return object2 == null ? object2 : baseFor(object2, castFunction(iteratee2), keysIn);
}
function forInRight(object2, iteratee2) {
    return object2 == null ? object2 : baseForRight(object2, castFunction(iteratee2), keysIn);
}
function forOwn(object2, iteratee2) {
    return object2 && baseForOwn(object2, castFunction(iteratee2));
}
function forOwnRight(object2, iteratee2) {
    return object2 && baseForOwnRight(object2, castFunction(iteratee2));
}
function fromPairs(pairs2) {
    var index = -1, length = pairs2 == null ? 0 : pairs2.length, result2 = {
    };
    while(++index < length){
        var pair = pairs2[index];
        result2[pair[0]] = pair[1];
    }
    return result2;
}
function baseFunctions(object2, props) {
    return arrayFilter(props, function(key) {
        return isFunction(object2[key]);
    });
}
function functions(object2) {
    return object2 == null ? [] : baseFunctions(object2, keys1(object2));
}
function functionsIn(object2) {
    return object2 == null ? [] : baseFunctions(object2, keysIn(object2));
}
var objectProto$l = Object.prototype;
var hasOwnProperty$i = objectProto$l.hasOwnProperty;
var groupBy1 = createAggregator(function(result2, value16, key) {
    if (hasOwnProperty$i.call(result2, key)) {
        result2[key].push(value16);
    } else {
        baseAssignValue(result2, key, [
            value16
        ]);
    }
});
function baseGt(value16, other) {
    return value16 > other;
}
function createRelationalOperation(operator) {
    return function(value16, other) {
        if (!(typeof value16 == "string" && typeof other == "string")) {
            value16 = toNumber(value16);
            other = toNumber(other);
        }
        return operator(value16, other);
    };
}
var gt = createRelationalOperation(baseGt);
var gte = createRelationalOperation(function(value16, other) {
    return value16 >= other;
});
var objectProto$m = Object.prototype;
var hasOwnProperty$j = objectProto$m.hasOwnProperty;
function baseHas(object2, key) {
    return object2 != null && hasOwnProperty$j.call(object2, key);
}
function has(object2, path3) {
    return object2 != null && hasPath(object2, path3, baseHas);
}
var nativeMax$8 = Math.max, nativeMin$5 = Math.min;
function baseInRange(number2, start, end) {
    return number2 >= nativeMin$5(start, end) && number2 < nativeMax$8(start, end);
}
function inRange(number2, start, end) {
    start = toFinite(start);
    if (end === void 0) {
        end = start;
        start = 0;
    } else {
        end = toFinite(end);
    }
    number2 = toNumber(number2);
    return baseInRange(number2, start, end);
}
var stringTag$4 = "[object String]";
function isString(value16) {
    return typeof value16 == "string" || !isArray(value16) && isObjectLike(value16) && baseGetTag(value16) == stringTag$4;
}
function baseValues(object2, props) {
    return arrayMap(props, function(key) {
        return object2[key];
    });
}
function values2(object2) {
    return object2 == null ? [] : baseValues(object2, keys1(object2));
}
var nativeMax$9 = Math.max;
function includes(collection2, value16, fromIndex, guard) {
    collection2 = isArrayLike(collection2) ? collection2 : values2(collection2);
    fromIndex = fromIndex && !guard ? toInteger(fromIndex) : 0;
    var length = collection2.length;
    if (fromIndex < 0) {
        fromIndex = nativeMax$9(length + fromIndex, 0);
    }
    return isString(collection2) ? fromIndex <= length && collection2.indexOf(value16, fromIndex) > -1 : !!length && baseIndexOf(collection2, value16, fromIndex) > -1;
}
var nativeMax$a = Math.max;
function indexOf(array2, value16, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = fromIndex == null ? 0 : toInteger(fromIndex);
    if (index < 0) {
        index = nativeMax$a(length + index, 0);
    }
    return baseIndexOf(array2, value16, index);
}
function initial(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSlice(array2, 0, -1) : [];
}
var nativeMin$6 = Math.min;
function baseIntersection(arrays, iteratee2, comparator) {
    var includes2 = comparator ? arrayIncludesWith : arrayIncludes, length = arrays[0].length, othLength = arrays.length, othIndex = othLength, caches = Array(othLength), maxLength = Infinity, result2 = [];
    while(othIndex--){
        var array2 = arrays[othIndex];
        if (othIndex && iteratee2) {
            array2 = arrayMap(array2, baseUnary(iteratee2));
        }
        maxLength = nativeMin$6(array2.length, maxLength);
        caches[othIndex] = !comparator && (iteratee2 || length >= 120 && array2.length >= 120) ? new SetCache(othIndex && array2) : void 0;
    }
    array2 = arrays[0];
    var index = -1, seen = caches[0];
    outer: while(++index < length && result2.length < maxLength){
        var value16 = array2[index], computed = iteratee2 ? iteratee2(value16) : value16;
        value16 = comparator || value16 !== 0 ? value16 : 0;
        if (!(seen ? cacheHas(seen, computed) : includes2(result2, computed, comparator))) {
            othIndex = othLength;
            while(--othIndex){
                var cache = caches[othIndex];
                if (!(cache ? cacheHas(cache, computed) : includes2(arrays[othIndex], computed, comparator))) {
                    continue outer;
                }
            }
            if (seen) {
                seen.push(computed);
            }
            result2.push(value16);
        }
    }
    return result2;
}
function castArrayLikeObject(value17) {
    return isArrayLikeObject(value17) ? value17 : [];
}
var intersection = baseRest(function(arrays) {
    var mapped = arrayMap(arrays, castArrayLikeObject);
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped) : [];
});
var intersectionBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays), mapped = arrayMap(arrays, castArrayLikeObject);
    if (iteratee2 === last(mapped)) {
        iteratee2 = void 0;
    } else {
        mapped.pop();
    }
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped, baseIteratee(iteratee2)) : [];
});
var intersectionWith = baseRest(function(arrays) {
    var comparator = last(arrays), mapped = arrayMap(arrays, castArrayLikeObject);
    comparator = typeof comparator == "function" ? comparator : void 0;
    if (comparator) {
        mapped.pop();
    }
    return mapped.length && mapped[0] === arrays[0] ? baseIntersection(mapped, void 0, comparator) : [];
});
function baseInverter(object2, setter, iteratee2, accumulator) {
    baseForOwn(object2, function(value17, key, object3) {
        setter(accumulator, iteratee2(value17), key, object3);
    });
    return accumulator;
}
function createInverter(setter, toIteratee) {
    return function(object2, iteratee2) {
        return baseInverter(object2, setter, toIteratee(iteratee2), {
        });
    };
}
var objectProto$n = Object.prototype;
var nativeObjectToString$2 = objectProto$n.toString;
var invert = createInverter(function(result2, value17, key) {
    if (value17 != null && typeof value17.toString != "function") {
        value17 = nativeObjectToString$2.call(value17);
    }
    result2[value17] = key;
}, constant(identity));
var objectProto$o = Object.prototype;
var hasOwnProperty$k = objectProto$o.hasOwnProperty;
var nativeObjectToString$3 = objectProto$o.toString;
var invertBy = createInverter(function(result2, value17, key) {
    if (value17 != null && typeof value17.toString != "function") {
        value17 = nativeObjectToString$3.call(value17);
    }
    if (hasOwnProperty$k.call(result2, value17)) {
        result2[value17].push(key);
    } else {
        result2[value17] = [
            key
        ];
    }
}, baseIteratee);
function parent(object2, path3) {
    return path3.length < 2 ? object2 : baseGet(object2, baseSlice(path3, 0, -1));
}
function baseInvoke(object2, path3, args4) {
    path3 = castPath(path3, object2);
    object2 = parent(object2, path3);
    var func2 = object2 == null ? object2 : object2[toKey(last(path3))];
    return func2 == null ? void 0 : apply(func2, object2, args4);
}
var invoke = baseRest(baseInvoke);
var invokeMap = baseRest(function(collection2, path3, args4) {
    var index = -1, isFunc = typeof path3 == "function", result2 = isArrayLike(collection2) ? Array(collection2.length) : [];
    baseEach(collection2, function(value17) {
        result2[++index] = isFunc ? apply(path3, value17, args4) : baseInvoke(value17, path3, args4);
    });
    return result2;
});
var arrayBufferTag$4 = "[object ArrayBuffer]";
function baseIsArrayBuffer(value17) {
    return isObjectLike(value17) && baseGetTag(value17) == arrayBufferTag$4;
}
var nodeIsArrayBuffer = nodeUtil && nodeUtil.isArrayBuffer;
var isArrayBuffer = nodeIsArrayBuffer ? baseUnary(nodeIsArrayBuffer) : baseIsArrayBuffer;
var boolTag$4 = "[object Boolean]";
function isBoolean1(value17) {
    return value17 === true || value17 === false || isObjectLike(value17) && baseGetTag(value17) == boolTag$4;
}
var dateTag$4 = "[object Date]";
function baseIsDate(value17) {
    return isObjectLike(value17) && baseGetTag(value17) == dateTag$4;
}
var nodeIsDate = nodeUtil && nodeUtil.isDate;
var isDate = nodeIsDate ? baseUnary(nodeIsDate) : baseIsDate;
function isElement(value17) {
    return isObjectLike(value17) && value17.nodeType === 1 && !isPlainObject(value17);
}
var mapTag$7 = "[object Map]", setTag$7 = "[object Set]";
var objectProto$p = Object.prototype;
var hasOwnProperty$l = objectProto$p.hasOwnProperty;
function isEmpty(value17) {
    if (value17 == null) {
        return true;
    }
    if (isArrayLike(value17) && (isArray(value17) || typeof value17 == "string" || typeof value17.splice == "function" || isBuffer(value17) || isTypedArray(value17) || isArguments(value17))) {
        return !value17.length;
    }
    var tag3 = getTag$1(value17);
    if (tag3 == mapTag$7 || tag3 == setTag$7) {
        return !value17.size;
    }
    if (isPrototype(value17)) {
        return !baseKeys(value17).length;
    }
    for(var key in value17){
        if (hasOwnProperty$l.call(value17, key)) {
            return false;
        }
    }
    return true;
}
function isEqual(value17, other) {
    return baseIsEqual(value17, other);
}
function isEqualWith(value17, other, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    var result2 = customizer ? customizer(value17, other) : void 0;
    return result2 === void 0 ? baseIsEqual(value17, other, void 0, customizer) : !!result2;
}
var nativeIsFinite$1 = root.isFinite;
function isFinite(value17) {
    return typeof value17 == "number" && nativeIsFinite$1(value17);
}
function isInteger1(value17) {
    return typeof value17 == "number" && value17 == toInteger(value17);
}
function isMatch(object2, source) {
    return object2 === source || baseIsMatch(object2, source, getMatchData(source));
}
function isMatchWith(object2, source, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return baseIsMatch(object2, source, getMatchData(source), customizer);
}
var numberTag$4 = "[object Number]";
function isNumber(value17) {
    return typeof value17 == "number" || isObjectLike(value17) && baseGetTag(value17) == numberTag$4;
}
function isNaN1(value17) {
    return isNumber(value17) && value17 != +value17;
}
var isMaskable = coreJsData ? isFunction : stubFalse;
var CORE_ERROR_TEXT = "Unsupported core-js use. Try https://npms.io/search?q=ponyfill.";
function isNative(value17) {
    if (isMaskable(value17)) {
        throw new Error(CORE_ERROR_TEXT);
    }
    return baseIsNative(value17);
}
function isNil(value17) {
    return value17 == null;
}
function isNull1(value17) {
    return value17 === null;
}
var regexpTag$4 = "[object RegExp]";
function baseIsRegExp(value17) {
    return isObjectLike(value17) && baseGetTag(value17) == regexpTag$4;
}
var nodeIsRegExp = nodeUtil && nodeUtil.isRegExp;
var isRegExp = nodeIsRegExp ? baseUnary(nodeIsRegExp) : baseIsRegExp;
var MAX_SAFE_INTEGER$2 = 9007199254740991;
function isSafeInteger(value17) {
    return isInteger1(value17) && value17 >= -MAX_SAFE_INTEGER$2 && value17 <= MAX_SAFE_INTEGER$2;
}
function isUndefined(value17) {
    return value17 === void 0;
}
var weakMapTag$3 = "[object WeakMap]";
function isWeakMap(value17) {
    return isObjectLike(value17) && getTag$1(value17) == weakMapTag$3;
}
var weakSetTag = "[object WeakSet]";
function isWeakSet(value17) {
    return isObjectLike(value17) && baseGetTag(value17) == weakSetTag;
}
var CLONE_DEEP_FLAG$4 = 1;
function iteratee(func2) {
    return baseIteratee(typeof func2 == "function" ? func2 : baseClone(func2, CLONE_DEEP_FLAG$4));
}
var arrayProto$1 = Array.prototype;
var nativeJoin = arrayProto$1.join;
function join4(array2, separator) {
    return array2 == null ? "" : nativeJoin.call(array2, separator);
}
var kebabCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? "-" : "") + word.toLowerCase();
});
var keyBy = createAggregator(function(result2, value17, key) {
    baseAssignValue(result2, key, value17);
});
function strictLastIndexOf(array2, value17, fromIndex) {
    var index = fromIndex + 1;
    while(index--){
        if (array2[index] === value17) {
            return index;
        }
    }
    return index;
}
var nativeMax$b = Math.max, nativeMin$7 = Math.min;
function lastIndexOf(array2, value17, fromIndex) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return -1;
    }
    var index = length;
    if (fromIndex !== void 0) {
        index = toInteger(fromIndex);
        index = index < 0 ? nativeMax$b(length + index, 0) : nativeMin$7(index, length - 1);
    }
    return value17 === value17 ? strictLastIndexOf(array2, value17, index) : baseFindIndex(array2, baseIsNaN, index, true);
}
var lowerCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + word.toLowerCase();
});
var lowerFirst = createCaseFirst("toLowerCase");
function baseLt(value17, other) {
    return value17 < other;
}
var lt = createRelationalOperation(baseLt);
var lte = createRelationalOperation(function(value17, other) {
    return value17 <= other;
});
function mapKeys(object2, iteratee2) {
    var result2 = {
    };
    iteratee2 = baseIteratee(iteratee2);
    baseForOwn(object2, function(value17, key, object3) {
        baseAssignValue(result2, iteratee2(value17, key, object3), value17);
    });
    return result2;
}
function mapValues(object2, iteratee2) {
    var result2 = {
    };
    iteratee2 = baseIteratee(iteratee2);
    baseForOwn(object2, function(value17, key, object3) {
        baseAssignValue(result2, key, iteratee2(value17, key, object3));
    });
    return result2;
}
var CLONE_DEEP_FLAG$5 = 1;
function matches(source) {
    return baseMatches(baseClone(source, CLONE_DEEP_FLAG$5));
}
var CLONE_DEEP_FLAG$6 = 1;
function matchesProperty(path3, srcValue) {
    return baseMatchesProperty(path3, baseClone(srcValue, CLONE_DEEP_FLAG$6));
}
function baseExtremum(array2, iteratee2, comparator) {
    var index = -1, length = array2.length;
    while(++index < length){
        var value17 = array2[index], current = iteratee2(value17);
        if (current != null && (computed === void 0 ? current === current && !isSymbol(current) : comparator(current, computed))) {
            var computed = current, result2 = value17;
        }
    }
    return result2;
}
function max(array2) {
    return array2 && array2.length ? baseExtremum(array2, identity, baseGt) : void 0;
}
function maxBy(array2, iteratee2) {
    return array2 && array2.length ? baseExtremum(array2, baseIteratee(iteratee2), baseGt) : void 0;
}
function baseSum(array2, iteratee2) {
    var result2, index = -1, length = array2.length;
    while(++index < length){
        var current = iteratee2(array2[index]);
        if (current !== void 0) {
            result2 = result2 === void 0 ? current : result2 + current;
        }
    }
    return result2;
}
var NAN$2 = 0 / 0;
function baseMean(array2, iteratee2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSum(array2, iteratee2) / length : NAN$2;
}
function mean(array2) {
    return baseMean(array2, identity);
}
function meanBy(array2, iteratee2) {
    return baseMean(array2, baseIteratee(iteratee2));
}
var merge1 = createAssigner(function(object2, source, srcIndex) {
    baseMerge(object2, source, srcIndex);
});
var method1 = baseRest(function(path3, args4) {
    return function(object2) {
        return baseInvoke(object2, path3, args4);
    };
});
var methodOf = baseRest(function(object2, args4) {
    return function(path3) {
        return baseInvoke(object2, path3, args4);
    };
});
function min1(array2) {
    return array2 && array2.length ? baseExtremum(array2, identity, baseLt) : void 0;
}
function minBy(array2, iteratee2) {
    return array2 && array2.length ? baseExtremum(array2, baseIteratee(iteratee2), baseLt) : void 0;
}
function mixin(object2, source, options12) {
    var props = keys1(source), methodNames = baseFunctions(source, props);
    var chain2 = !(isObject1(options12) && "chain" in options12) || !!options12.chain, isFunc = isFunction(object2);
    arrayEach(methodNames, function(methodName) {
        var func2 = source[methodName];
        object2[methodName] = func2;
        if (isFunc) {
            object2.prototype[methodName] = function() {
                var chainAll = this.__chain__;
                if (chain2 || chainAll) {
                    var result2 = object2(this.__wrapped__), actions = result2.__actions__ = copyArray(this.__actions__);
                    actions.push({
                        func: func2,
                        args: arguments,
                        thisArg: object2
                    });
                    result2.__chain__ = chainAll;
                    return result2;
                }
                return func2.apply(object2, arrayPush([
                    this.value()
                ], arguments));
            };
        }
    });
    return object2;
}
var multiply = createMathOperation(function(multiplier, multiplicand) {
    return multiplier * multiplicand;
}, 1);
var FUNC_ERROR_TEXT$8 = "Expected a function";
function negate(predicate) {
    if (typeof predicate != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$8);
    }
    return function() {
        var args4 = arguments;
        switch(args4.length){
            case 0:
                return !predicate.call(this);
            case 1:
                return !predicate.call(this, args4[0]);
            case 2:
                return !predicate.call(this, args4[0], args4[1]);
            case 3:
                return !predicate.call(this, args4[0], args4[1], args4[2]);
        }
        return !predicate.apply(this, args4);
    };
}
function iteratorToArray(iterator) {
    var data, result2 = [];
    while(!(data = iterator.next()).done){
        result2.push(data.value);
    }
    return result2;
}
var mapTag$8 = "[object Map]", setTag$8 = "[object Set]";
var symIterator = Symbol1 ? Symbol1.iterator : void 0;
function toArray(value18) {
    if (!value18) {
        return [];
    }
    if (isArrayLike(value18)) {
        return isString(value18) ? stringToArray(value18) : copyArray(value18);
    }
    if (symIterator && value18[symIterator]) {
        return iteratorToArray(value18[symIterator]());
    }
    var tag3 = getTag$1(value18), func2 = tag3 == mapTag$8 ? mapToArray : tag3 == setTag$8 ? setToArray : values2;
    return func2(value18);
}
function wrapperNext() {
    if (this.__values__ === void 0) {
        this.__values__ = toArray(this.value());
    }
    var done = this.__index__ >= this.__values__.length, value18 = done ? void 0 : this.__values__[this.__index__++];
    return {
        done,
        value: value18
    };
}
function baseNth(array2, n) {
    var length = array2.length;
    if (!length) {
        return;
    }
    n += n < 0 ? length : 0;
    return isIndex(n, length) ? array2[n] : void 0;
}
function nth(array2, n) {
    return array2 && array2.length ? baseNth(array2, toInteger(n)) : void 0;
}
function nthArg(n) {
    n = toInteger(n);
    return baseRest(function(args4) {
        return baseNth(args4, n);
    });
}
function baseUnset(object2, path3) {
    path3 = castPath(path3, object2);
    object2 = parent(object2, path3);
    return object2 == null || delete object2[toKey(last(path3))];
}
function customOmitClone(value18) {
    return isPlainObject(value18) ? void 0 : value18;
}
var CLONE_DEEP_FLAG$7 = 1, CLONE_FLAT_FLAG$1 = 2, CLONE_SYMBOLS_FLAG$5 = 4;
var omit = flatRest(function(object2, paths) {
    var result2 = {
    };
    if (object2 == null) {
        return result2;
    }
    var isDeep = false;
    paths = arrayMap(paths, function(path3) {
        path3 = castPath(path3, object2);
        isDeep || (isDeep = path3.length > 1);
        return path3;
    });
    copyObject(object2, getAllKeysIn(object2), result2);
    if (isDeep) {
        result2 = baseClone(result2, CLONE_DEEP_FLAG$7 | CLONE_FLAT_FLAG$1 | CLONE_SYMBOLS_FLAG$5, customOmitClone);
    }
    var length = paths.length;
    while(length--){
        baseUnset(result2, paths[length]);
    }
    return result2;
});
function baseSet(object2, path3, value18, customizer) {
    if (!isObject1(object2)) {
        return object2;
    }
    path3 = castPath(path3, object2);
    var index = -1, length = path3.length, lastIndex = length - 1, nested = object2;
    while(nested != null && ++index < length){
        var key = toKey(path3[index]), newValue = value18;
        if (key === "__proto__" || key === "constructor" || key === "prototype") {
            return object2;
        }
        if (index != lastIndex) {
            var objValue = nested[key];
            newValue = customizer ? customizer(objValue, key, nested) : void 0;
            if (newValue === void 0) {
                newValue = isObject1(objValue) ? objValue : isIndex(path3[index + 1]) ? [] : {
                };
            }
        }
        assignValue(nested, key, newValue);
        nested = nested[key];
    }
    return object2;
}
function basePickBy(object2, paths, predicate) {
    var index = -1, length = paths.length, result2 = {
    };
    while(++index < length){
        var path3 = paths[index], value18 = baseGet(object2, path3);
        if (predicate(value18, path3)) {
            baseSet(result2, castPath(path3, object2), value18);
        }
    }
    return result2;
}
function pickBy(object2, predicate) {
    if (object2 == null) {
        return {
        };
    }
    var props = arrayMap(getAllKeysIn(object2), function(prop) {
        return [
            prop
        ];
    });
    predicate = baseIteratee(predicate);
    return basePickBy(object2, props, function(value19, path4) {
        return predicate(value19, path4[0]);
    });
}
function omitBy(object2, predicate) {
    return pickBy(object2, negate(baseIteratee(predicate)));
}
function once(func2) {
    return before(2, func2);
}
function baseSortBy(array2, comparer) {
    var length = array2.length;
    array2.sort(comparer);
    while(length--){
        array2[length] = array2[length].value;
    }
    return array2;
}
function compareAscending(value19, other) {
    if (value19 !== other) {
        var valIsDefined = value19 !== void 0, valIsNull = value19 === null, valIsReflexive = value19 === value19, valIsSymbol = isSymbol(value19);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = isSymbol(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value19 > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value19 < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
function compareMultiple(object2, other, orders) {
    var index = -1, objCriteria = object2.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result2 = compareAscending(objCriteria[index], othCriteria[index]);
        if (result2) {
            if (index >= ordersLength) {
                return result2;
            }
            var order = orders[index];
            return result2 * (order == "desc" ? -1 : 1);
        }
    }
    return object2.index - other.index;
}
function baseOrderBy(collection2, iteratees, orders) {
    if (iteratees.length) {
        iteratees = arrayMap(iteratees, function(iteratee2) {
            if (isArray(iteratee2)) {
                return function(value19) {
                    return baseGet(value19, iteratee2.length === 1 ? iteratee2[0] : iteratee2);
                };
            }
            return iteratee2;
        });
    } else {
        iteratees = [
            identity
        ];
    }
    var index = -1;
    iteratees = arrayMap(iteratees, baseUnary(baseIteratee));
    var result2 = baseMap(collection2, function(value19, key, collection3) {
        var criteria = arrayMap(iteratees, function(iteratee2) {
            return iteratee2(value19);
        });
        return {
            criteria,
            index: ++index,
            value: value19
        };
    });
    return baseSortBy(result2, function(object2, other) {
        return compareMultiple(object2, other, orders);
    });
}
function orderBy(collection2, iteratees, orders, guard) {
    if (collection2 == null) {
        return [];
    }
    if (!isArray(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!isArray(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return baseOrderBy(collection2, iteratees, orders);
}
function createOver(arrayFunc) {
    return flatRest(function(iteratees) {
        iteratees = arrayMap(iteratees, baseUnary(baseIteratee));
        return baseRest(function(args4) {
            var thisArg = this;
            return arrayFunc(iteratees, function(iteratee2) {
                return apply(iteratee2, thisArg, args4);
            });
        });
    });
}
var over = createOver(arrayMap);
var castRest = baseRest;
var nativeMin$8 = Math.min;
var overArgs = castRest(function(func2, transforms) {
    transforms = transforms.length == 1 && isArray(transforms[0]) ? arrayMap(transforms[0], baseUnary(baseIteratee)) : arrayMap(baseFlatten(transforms, 1), baseUnary(baseIteratee));
    var funcsLength = transforms.length;
    return baseRest(function(args4) {
        var index = -1, length = nativeMin$8(args4.length, funcsLength);
        while(++index < length){
            args4[index] = transforms[index].call(this, args4[index]);
        }
        return apply(func2, this, args4);
    });
});
var overEvery = createOver(arrayEvery);
var overSome = createOver(arraySome);
var MAX_SAFE_INTEGER$3 = 9007199254740991;
var nativeFloor = Math.floor;
function baseRepeat(string2, n) {
    var result2 = "";
    if (!string2 || n < 1 || n > MAX_SAFE_INTEGER$3) {
        return result2;
    }
    do {
        if (n % 2) {
            result2 += string2;
        }
        n = nativeFloor(n / 2);
        if (n) {
            string2 += string2;
        }
    }while (n)
    return result2;
}
var asciiSize = baseProperty("length");
var rsAstralRange$3 = "\\ud800-\\udfff", rsComboMarksRange$4 = "\\u0300-\\u036f", reComboHalfMarksRange$4 = "\\ufe20-\\ufe2f", rsComboSymbolsRange$4 = "\\u20d0-\\u20ff", rsComboRange$4 = rsComboMarksRange$4 + reComboHalfMarksRange$4 + rsComboSymbolsRange$4, rsVarRange$3 = "\\ufe0e\\ufe0f";
var rsAstral$1 = "[" + rsAstralRange$3 + "]", rsCombo$3 = "[" + rsComboRange$4 + "]", rsFitz$2 = "\\ud83c[\\udffb-\\udfff]", rsModifier$2 = "(?:" + rsCombo$3 + "|" + rsFitz$2 + ")", rsNonAstral$2 = "[^" + rsAstralRange$3 + "]", rsRegional$2 = "(?:\\ud83c[\\udde6-\\uddff]){2}", rsSurrPair$2 = "[\\ud800-\\udbff][\\udc00-\\udfff]", rsZWJ$3 = "\\u200d";
var reOptMod$2 = rsModifier$2 + "?", rsOptVar$2 = "[" + rsVarRange$3 + "]?", rsOptJoin$2 = "(?:" + rsZWJ$3 + "(?:" + [
    rsNonAstral$2,
    rsRegional$2,
    rsSurrPair$2
].join("|") + ")" + rsOptVar$2 + reOptMod$2 + ")*", rsSeq$2 = rsOptVar$2 + reOptMod$2 + rsOptJoin$2, rsSymbol$1 = "(?:" + [
    rsNonAstral$2 + rsCombo$3 + "?",
    rsCombo$3,
    rsRegional$2,
    rsSurrPair$2,
    rsAstral$1
].join("|") + ")";
var reUnicode$1 = RegExp(rsFitz$2 + "(?=" + rsFitz$2 + ")|" + rsSymbol$1 + rsSeq$2, "g");
function unicodeSize(string2) {
    var result2 = reUnicode$1.lastIndex = 0;
    while(reUnicode$1.test(string2)){
        ++result2;
    }
    return result2;
}
function stringSize(string2) {
    return hasUnicode(string2) ? unicodeSize(string2) : asciiSize(string2);
}
var nativeCeil$1 = Math.ceil;
function createPadding(length, chars) {
    chars = chars === void 0 ? " " : baseToString(chars);
    var charsLength = chars.length;
    if (charsLength < 2) {
        return charsLength ? baseRepeat(chars, length) : chars;
    }
    var result2 = baseRepeat(chars, nativeCeil$1(length / stringSize(chars)));
    return hasUnicode(chars) ? castSlice(stringToArray(result2), 0, length).join("") : result2.slice(0, length);
}
var nativeCeil$2 = Math.ceil, nativeFloor$1 = Math.floor;
function pad(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    if (!length || strLength >= length) {
        return string2;
    }
    var mid = (length - strLength) / 2;
    return createPadding(nativeFloor$1(mid), chars) + string2 + createPadding(nativeCeil$2(mid), chars);
}
function padEnd(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    return length && strLength < length ? string2 + createPadding(length - strLength, chars) : string2;
}
function padStart(string2, length, chars) {
    string2 = toString(string2);
    length = toInteger(length);
    var strLength = length ? stringSize(string2) : 0;
    return length && strLength < length ? createPadding(length - strLength, chars) + string2 : string2;
}
var reTrimStart$1 = /^\s+/;
var nativeParseInt = root.parseInt;
function parseInt$1(string2, radix, guard) {
    if (guard || radix == null) {
        radix = 0;
    } else if (radix) {
        radix = +radix;
    }
    return nativeParseInt(toString(string2).replace(reTrimStart$1, ""), radix || 0);
}
var WRAP_PARTIAL_FLAG$6 = 32;
var partial2 = baseRest(function(func2, partials) {
    var holders = replaceHolders(partials, getHolder(partial2));
    return createWrap(func2, WRAP_PARTIAL_FLAG$6, void 0, partials, holders);
});
partial2.placeholder = {
};
var WRAP_PARTIAL_RIGHT_FLAG$3 = 64;
var partialRight = baseRest(function(func2, partials) {
    var holders = replaceHolders(partials, getHolder(partialRight));
    return createWrap(func2, WRAP_PARTIAL_RIGHT_FLAG$3, void 0, partials, holders);
});
partialRight.placeholder = {
};
var partition = createAggregator(function(result2, value19, key) {
    result2[key ? 0 : 1].push(value19);
}, function() {
    return [
        [],
        []
    ];
});
function basePick(object2, paths) {
    return basePickBy(object2, paths, function(value19, path4) {
        return hasIn(object2, path4);
    });
}
var pick = flatRest(function(object2, paths) {
    return object2 == null ? {
    } : basePick(object2, paths);
});
function wrapperPlant(value19) {
    var result2, parent2 = this;
    while(parent2 instanceof baseLodash){
        var clone2 = wrapperClone(parent2);
        clone2.__index__ = 0;
        clone2.__values__ = void 0;
        if (result2) {
            previous.__wrapped__ = clone2;
        } else {
            result2 = clone2;
        }
        var previous = clone2;
        parent2 = parent2.__wrapped__;
    }
    previous.__wrapped__ = value19;
    return result2;
}
function propertyOf(object2) {
    return function(path4) {
        return object2 == null ? void 0 : baseGet(object2, path4);
    };
}
function baseIndexOfWith(array2, value19, fromIndex, comparator) {
    var index = fromIndex - 1, length = array2.length;
    while(++index < length){
        if (comparator(array2[index], value19)) {
            return index;
        }
    }
    return -1;
}
var arrayProto$2 = Array.prototype;
var splice$1 = arrayProto$2.splice;
function basePullAll(array2, values21, iteratee2, comparator) {
    var indexOf2 = comparator ? baseIndexOfWith : baseIndexOf, index = -1, length = values21.length, seen = array2;
    if (array2 === values21) {
        values21 = copyArray(values21);
    }
    if (iteratee2) {
        seen = arrayMap(array2, baseUnary(iteratee2));
    }
    while(++index < length){
        var fromIndex = 0, value19 = values21[index], computed = iteratee2 ? iteratee2(value19) : value19;
        while((fromIndex = indexOf2(seen, computed, fromIndex, comparator)) > -1){
            if (seen !== array2) {
                splice$1.call(seen, fromIndex, 1);
            }
            splice$1.call(array2, fromIndex, 1);
        }
    }
    return array2;
}
function pullAll(array2, values21) {
    return array2 && array2.length && values21 && values21.length ? basePullAll(array2, values21) : array2;
}
var pull = baseRest(pullAll);
function pullAllBy(array2, values21, iteratee2) {
    return array2 && array2.length && values21 && values21.length ? basePullAll(array2, values21, baseIteratee(iteratee2)) : array2;
}
function pullAllWith(array2, values21, comparator) {
    return array2 && array2.length && values21 && values21.length ? basePullAll(array2, values21, void 0, comparator) : array2;
}
var arrayProto$3 = Array.prototype;
var splice$2 = arrayProto$3.splice;
function basePullAt(array2, indexes) {
    var length = array2 ? indexes.length : 0, lastIndex = length - 1;
    while(length--){
        var index = indexes[length];
        if (length == lastIndex || index !== previous) {
            var previous = index;
            if (isIndex(index)) {
                splice$2.call(array2, index, 1);
            } else {
                baseUnset(array2, index);
            }
        }
    }
    return array2;
}
var pullAt = flatRest(function(array2, indexes) {
    var length = array2 == null ? 0 : array2.length, result2 = baseAt(array2, indexes);
    basePullAt(array2, arrayMap(indexes, function(index) {
        return isIndex(index, length) ? +index : index;
    }).sort(compareAscending));
    return result2;
});
var nativeFloor$2 = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor$2(nativeRandom() * (upper - lower + 1));
}
var freeParseFloat = parseFloat;
var nativeMin$9 = Math.min, nativeRandom$1 = Math.random;
function random1(lower, upper, floating) {
    if (floating && typeof floating != "boolean" && isIterateeCall(lower, upper, floating)) {
        upper = floating = void 0;
    }
    if (floating === void 0) {
        if (typeof upper == "boolean") {
            floating = upper;
            upper = void 0;
        } else if (typeof lower == "boolean") {
            floating = lower;
            lower = void 0;
        }
    }
    if (lower === void 0 && upper === void 0) {
        lower = 0;
        upper = 1;
    } else {
        lower = toFinite(lower);
        if (upper === void 0) {
            upper = lower;
            lower = 0;
        } else {
            upper = toFinite(upper);
        }
    }
    if (lower > upper) {
        var temp = lower;
        lower = upper;
        upper = temp;
    }
    if (floating || lower % 1 || upper % 1) {
        var rand = nativeRandom$1();
        return nativeMin$9(lower + rand * (upper - lower + freeParseFloat("1e-" + ((rand + "").length - 1))), upper);
    }
    return baseRandom(lower, upper);
}
var nativeCeil$3 = Math.ceil, nativeMax$c = Math.max;
function baseRange(start, end, step, fromRight) {
    var index = -1, length = nativeMax$c(nativeCeil$3((end - start) / (step || 1)), 0), result2 = Array(length);
    while(length--){
        result2[fromRight ? length : ++index] = start;
        start += step;
    }
    return result2;
}
function createRange(fromRight) {
    return function(start, end, step) {
        if (step && typeof step != "number" && isIterateeCall(start, end, step)) {
            end = step = void 0;
        }
        start = toFinite(start);
        if (end === void 0) {
            end = start;
            start = 0;
        } else {
            end = toFinite(end);
        }
        step = step === void 0 ? start < end ? 1 : -1 : toFinite(step);
        return baseRange(start, end, step, fromRight);
    };
}
var range = createRange();
var rangeRight = createRange(true);
var WRAP_REARG_FLAG$3 = 256;
var rearg = flatRest(function(func2, indexes) {
    return createWrap(func2, WRAP_REARG_FLAG$3, void 0, void 0, void 0, indexes);
});
function baseReduce(collection2, iteratee2, accumulator, initAccum, eachFunc) {
    eachFunc(collection2, function(value20, index, collection3) {
        accumulator = initAccum ? (initAccum = false, value20) : iteratee2(accumulator, value20, index, collection3);
    });
    return accumulator;
}
function reduce(collection2, iteratee2, accumulator) {
    var func2 = isArray(collection2) ? arrayReduce : baseReduce, initAccum = arguments.length < 3;
    return func2(collection2, baseIteratee(iteratee2), accumulator, initAccum, baseEach);
}
function arrayReduceRight(array2, iteratee2, accumulator, initAccum) {
    var length = array2 == null ? 0 : array2.length;
    if (initAccum && length) {
        accumulator = array2[--length];
    }
    while(length--){
        accumulator = iteratee2(accumulator, array2[length], length, array2);
    }
    return accumulator;
}
function reduceRight(collection2, iteratee2, accumulator) {
    var func2 = isArray(collection2) ? arrayReduceRight : baseReduce, initAccum = arguments.length < 3;
    return func2(collection2, baseIteratee(iteratee2), accumulator, initAccum, baseEachRight);
}
function reject(collection2, predicate) {
    var func2 = isArray(collection2) ? arrayFilter : baseFilter;
    return func2(collection2, negate(baseIteratee(predicate)));
}
function remove(array2, predicate) {
    var result2 = [];
    if (!(array2 && array2.length)) {
        return result2;
    }
    var index = -1, indexes = [], length = array2.length;
    predicate = baseIteratee(predicate);
    while(++index < length){
        var value20 = array2[index];
        if (predicate(value20, index, array2)) {
            result2.push(value20);
            indexes.push(index);
        }
    }
    basePullAt(array2, indexes);
    return result2;
}
function repeat1(string2, n, guard) {
    if (guard ? isIterateeCall(string2, n, guard) : n === void 0) {
        n = 1;
    } else {
        n = toInteger(n);
    }
    return baseRepeat(toString(string2), n);
}
function replace() {
    var args4 = arguments, string2 = toString(args4[0]);
    return args4.length < 3 ? string2 : string2.replace(args4[1], args4[2]);
}
var FUNC_ERROR_TEXT$9 = "Expected a function";
function rest(func2, start) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$9);
    }
    start = start === void 0 ? start : toInteger(start);
    return baseRest(func2, start);
}
function result1(object2, path4, defaultValue) {
    path4 = castPath(path4, object2);
    var index = -1, length = path4.length;
    if (!length) {
        length = 1;
        object2 = void 0;
    }
    while(++index < length){
        var value21 = object2 == null ? void 0 : object2[toKey(path4[index])];
        if (value21 === void 0) {
            index = length;
            value21 = defaultValue;
        }
        object2 = isFunction(value21) ? value21.call(object2) : value21;
    }
    return object2;
}
var arrayProto$4 = Array.prototype;
var nativeReverse = arrayProto$4.reverse;
function reverse(array2) {
    return array2 == null ? array2 : nativeReverse.call(array2);
}
var round = createRound("round");
function arraySample(array2) {
    var length = array2.length;
    return length ? array2[baseRandom(0, length - 1)] : void 0;
}
function baseSample(collection2) {
    return arraySample(values2(collection2));
}
function sample(collection2) {
    var func2 = isArray(collection2) ? arraySample : baseSample;
    return func2(collection2);
}
function shuffleSelf(array2, size21) {
    var index = -1, length = array2.length, lastIndex = length - 1;
    size21 = size21 === void 0 ? length : size21;
    while(++index < size21){
        var rand = baseRandom(index, lastIndex), value22 = array2[rand];
        array2[rand] = array2[index];
        array2[index] = value22;
    }
    array2.length = size21;
    return array2;
}
function arraySampleSize(array2, n) {
    return shuffleSelf(copyArray(array2), baseClamp(n, 0, array2.length));
}
function baseSampleSize(collection2, n) {
    var array2 = values2(collection2);
    return shuffleSelf(array2, baseClamp(n, 0, array2.length));
}
function sampleSize(collection2, n, guard) {
    if (guard ? isIterateeCall(collection2, n, guard) : n === void 0) {
        n = 1;
    } else {
        n = toInteger(n);
    }
    var func2 = isArray(collection2) ? arraySampleSize : baseSampleSize;
    return func2(collection2, n);
}
function set1(object2, path4, value23) {
    return object2 == null ? object2 : baseSet(object2, path4, value23);
}
function setWith(object2, path4, value23, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return object2 == null ? object2 : baseSet(object2, path4, value23, customizer);
}
function arrayShuffle(array2) {
    return shuffleSelf(copyArray(array2));
}
function baseShuffle(collection2) {
    return shuffleSelf(values2(collection2));
}
function shuffle(collection2) {
    var func2 = isArray(collection2) ? arrayShuffle : baseShuffle;
    return func2(collection2);
}
var mapTag$9 = "[object Map]", setTag$9 = "[object Set]";
function size4(collection2) {
    if (collection2 == null) {
        return 0;
    }
    if (isArrayLike(collection2)) {
        return isString(collection2) ? stringSize(collection2) : collection2.length;
    }
    var tag3 = getTag$1(collection2);
    if (tag3 == mapTag$9 || tag3 == setTag$9) {
        return collection2.size;
    }
    return baseKeys(collection2).length;
}
function slice(array2, start, end) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    if (end && typeof end != "number" && isIterateeCall(array2, start, end)) {
        start = 0;
        end = length;
    } else {
        start = start == null ? 0 : toInteger(start);
        end = end === void 0 ? length : toInteger(end);
    }
    return baseSlice(array2, start, end);
}
var snakeCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? "_" : "") + word.toLowerCase();
});
function baseSome(collection2, predicate) {
    var result2;
    baseEach(collection2, function(value23, index, collection3) {
        result2 = predicate(value23, index, collection3);
        return !result2;
    });
    return !!result2;
}
function some(collection2, predicate, guard) {
    var func2 = isArray(collection2) ? arraySome : baseSome;
    if (guard && isIterateeCall(collection2, predicate, guard)) {
        predicate = void 0;
    }
    return func2(collection2, baseIteratee(predicate));
}
var sortBy = baseRest(function(collection2, iteratees) {
    if (collection2 == null) {
        return [];
    }
    var length = iteratees.length;
    if (length > 1 && isIterateeCall(collection2, iteratees[0], iteratees[1])) {
        iteratees = [];
    } else if (length > 2 && isIterateeCall(iteratees[0], iteratees[1], iteratees[2])) {
        iteratees = [
            iteratees[0]
        ];
    }
    return baseOrderBy(collection2, baseFlatten(iteratees, 1), []);
});
var MAX_ARRAY_LENGTH$2 = 4294967295, MAX_ARRAY_INDEX = MAX_ARRAY_LENGTH$2 - 1;
var nativeFloor$3 = Math.floor, nativeMin$a = Math.min;
function baseSortedIndexBy(array2, value23, iteratee2, retHighest) {
    var low = 0, high = array2 == null ? 0 : array2.length;
    if (high === 0) {
        return 0;
    }
    value23 = iteratee2(value23);
    var valIsNaN = value23 !== value23, valIsNull = value23 === null, valIsSymbol = isSymbol(value23), valIsUndefined = value23 === void 0;
    while(low < high){
        var mid = nativeFloor$3((low + high) / 2), computed = iteratee2(array2[mid]), othIsDefined = computed !== void 0, othIsNull = computed === null, othIsReflexive = computed === computed, othIsSymbol = isSymbol(computed);
        if (valIsNaN) {
            var setLow = retHighest || othIsReflexive;
        } else if (valIsUndefined) {
            setLow = othIsReflexive && (retHighest || othIsDefined);
        } else if (valIsNull) {
            setLow = othIsReflexive && othIsDefined && (retHighest || !othIsNull);
        } else if (valIsSymbol) {
            setLow = othIsReflexive && othIsDefined && !othIsNull && (retHighest || !othIsSymbol);
        } else if (othIsNull || othIsSymbol) {
            setLow = false;
        } else {
            setLow = retHighest ? computed <= value23 : computed < value23;
        }
        if (setLow) {
            low = mid + 1;
        } else {
            high = mid;
        }
    }
    return nativeMin$a(high, MAX_ARRAY_INDEX);
}
var MAX_ARRAY_LENGTH$3 = 4294967295, HALF_MAX_ARRAY_LENGTH = MAX_ARRAY_LENGTH$3 >>> 1;
function baseSortedIndex(array2, value23, retHighest) {
    var low = 0, high = array2 == null ? low : array2.length;
    if (typeof value23 == "number" && value23 === value23 && high <= HALF_MAX_ARRAY_LENGTH) {
        while(low < high){
            var mid = low + high >>> 1, computed = array2[mid];
            if (computed !== null && !isSymbol(computed) && (retHighest ? computed <= value23 : computed < value23)) {
                low = mid + 1;
            } else {
                high = mid;
            }
        }
        return high;
    }
    return baseSortedIndexBy(array2, value23, identity, retHighest);
}
function sortedIndex(array2, value23) {
    return baseSortedIndex(array2, value23);
}
function sortedIndexBy(array2, value23, iteratee2) {
    return baseSortedIndexBy(array2, value23, baseIteratee(iteratee2));
}
function sortedIndexOf(array2, value23) {
    var length = array2 == null ? 0 : array2.length;
    if (length) {
        var index = baseSortedIndex(array2, value23);
        if (index < length && eq(array2[index], value23)) {
            return index;
        }
    }
    return -1;
}
function sortedLastIndex(array2, value23) {
    return baseSortedIndex(array2, value23, true);
}
function sortedLastIndexBy(array2, value23, iteratee2) {
    return baseSortedIndexBy(array2, value23, baseIteratee(iteratee2), true);
}
function sortedLastIndexOf(array2, value23) {
    var length = array2 == null ? 0 : array2.length;
    if (length) {
        var index = baseSortedIndex(array2, value23, true) - 1;
        if (eq(array2[index], value23)) {
            return index;
        }
    }
    return -1;
}
function baseSortedUniq(array2, iteratee2) {
    var index = -1, length = array2.length, resIndex = 0, result2 = [];
    while(++index < length){
        var value23 = array2[index], computed = iteratee2 ? iteratee2(value23) : value23;
        if (!index || !eq(computed, seen)) {
            var seen = computed;
            result2[resIndex++] = value23 === 0 ? 0 : value23;
        }
    }
    return result2;
}
function sortedUniq(array2) {
    return array2 && array2.length ? baseSortedUniq(array2) : [];
}
function sortedUniqBy(array2, iteratee2) {
    return array2 && array2.length ? baseSortedUniq(array2, baseIteratee(iteratee2)) : [];
}
var MAX_ARRAY_LENGTH$4 = 4294967295;
function split1(string2, separator, limit) {
    if (limit && typeof limit != "number" && isIterateeCall(string2, separator, limit)) {
        separator = limit = void 0;
    }
    limit = limit === void 0 ? MAX_ARRAY_LENGTH$4 : limit >>> 0;
    if (!limit) {
        return [];
    }
    string2 = toString(string2);
    if (string2 && (typeof separator == "string" || separator != null && !isRegExp(separator))) {
        separator = baseToString(separator);
        if (!separator && hasUnicode(string2)) {
            return castSlice(stringToArray(string2), 0, limit);
        }
    }
    return string2.split(separator, limit);
}
var FUNC_ERROR_TEXT$a = "Expected a function";
var nativeMax$d = Math.max;
function spread(func2, start) {
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$a);
    }
    start = start == null ? 0 : nativeMax$d(toInteger(start), 0);
    return baseRest(function(args4) {
        var array2 = args4[start], otherArgs = castSlice(args4, 0, start);
        if (array2) {
            arrayPush(otherArgs, array2);
        }
        return apply(func2, this, otherArgs);
    });
}
var startCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + upperFirst(word);
});
function startsWith(string2, target, position2) {
    string2 = toString(string2);
    position2 = position2 == null ? 0 : baseClamp(toInteger(position2), 0, string2.length);
    target = baseToString(target);
    return string2.slice(position2, position2 + target.length) == target;
}
function stubObject() {
    return {
    };
}
function stubString() {
    return "";
}
function stubTrue() {
    return true;
}
var subtract = createMathOperation(function(minuend, subtrahend) {
    return minuend - subtrahend;
}, 0);
function sum(array2) {
    return array2 && array2.length ? baseSum(array2, identity) : 0;
}
function sumBy(array2, iteratee2) {
    return array2 && array2.length ? baseSum(array2, baseIteratee(iteratee2)) : 0;
}
function tail(array2) {
    var length = array2 == null ? 0 : array2.length;
    return length ? baseSlice(array2, 1, length) : [];
}
function take(array2, n, guard) {
    if (!(array2 && array2.length)) {
        return [];
    }
    n = guard || n === void 0 ? 1 : toInteger(n);
    return baseSlice(array2, 0, n < 0 ? 0 : n);
}
function takeRight(array2, n, guard) {
    var length = array2 == null ? 0 : array2.length;
    if (!length) {
        return [];
    }
    n = guard || n === void 0 ? 1 : toInteger(n);
    n = length - n;
    return baseSlice(array2, n < 0 ? 0 : n, length);
}
function takeRightWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate), false, true) : [];
}
function takeWhile(array2, predicate) {
    return array2 && array2.length ? baseWhile(array2, baseIteratee(predicate)) : [];
}
function tap(value24, interceptor) {
    interceptor(value24);
    return value24;
}
var objectProto$q = Object.prototype;
var hasOwnProperty$m = objectProto$q.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object2) {
    if (objValue === void 0 || eq(objValue, objectProto$q[key]) && !hasOwnProperty$m.call(object2, key)) {
        return srcValue;
    }
    return objValue;
}
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var reInterpolate = /<%=([\s\S]+?)%>/g;
var reEscape = /<%-([\s\S]+?)%>/g;
var reEvaluate = /<%([\s\S]+?)%>/g;
var templateSettings = {
    escape: reEscape,
    evaluate: reEvaluate,
    interpolate: reInterpolate,
    variable: "",
    imports: {
        _: {
            escape
        }
    }
};
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto$r = Object.prototype;
var hasOwnProperty$n = objectProto$r.hasOwnProperty;
function template(string2, options12, guard) {
    var settings = templateSettings.imports._.templateSettings || templateSettings;
    if (guard && isIterateeCall(string2, options12, guard)) {
        options12 = void 0;
    }
    string2 = toString(string2);
    options12 = assignInWith({
    }, options12, settings, customDefaultsAssignIn);
    var imports = assignInWith({
    }, options12.imports, settings.imports, customDefaultsAssignIn), importsKeys = keys1(imports), importsValues = baseValues(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options12.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options12.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === reInterpolate ? reEsTemplate : reNoMatch).source + "|" + (options12.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty$n.call(options12, "sourceURL") ? "//# sourceURL=" + (options12.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string2.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string2.slice(index, offset).replace(reUnescapedString, escapeStringChar);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty$n.call(options12, "variable") && options12.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result2 = attempt(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result2.source = source;
    if (isError(result2)) {
        throw result2;
    }
    return result2;
}
var FUNC_ERROR_TEXT$b = "Expected a function";
function throttle(func2, wait, options12) {
    var leading = true, trailing = true;
    if (typeof func2 != "function") {
        throw new TypeError(FUNC_ERROR_TEXT$b);
    }
    if (isObject1(options12)) {
        leading = "leading" in options12 ? !!options12.leading : leading;
        trailing = "trailing" in options12 ? !!options12.trailing : trailing;
    }
    return debounce(func2, wait, {
        leading,
        maxWait: wait,
        trailing
    });
}
function thru(value24, interceptor) {
    return interceptor(value24);
}
var MAX_SAFE_INTEGER$4 = 9007199254740991;
var MAX_ARRAY_LENGTH$5 = 4294967295;
var nativeMin$b = Math.min;
function times(n, iteratee2) {
    n = toInteger(n);
    if (n < 1 || n > MAX_SAFE_INTEGER$4) {
        return [];
    }
    var index = MAX_ARRAY_LENGTH$5, length = nativeMin$b(n, MAX_ARRAY_LENGTH$5);
    iteratee2 = castFunction(iteratee2);
    n -= MAX_ARRAY_LENGTH$5;
    var result2 = baseTimes(length, iteratee2);
    while(++index < n){
        iteratee2(index);
    }
    return result2;
}
function wrapperToIterator() {
    return this;
}
function baseWrapperValue(value24, actions) {
    var result2 = value24;
    if (result2 instanceof LazyWrapper) {
        result2 = result2.value();
    }
    return arrayReduce(actions, function(result3, action) {
        return action.func.apply(action.thisArg, arrayPush([
            result3
        ], action.args));
    }, result2);
}
function wrapperValue() {
    return baseWrapperValue(this.__wrapped__, this.__actions__);
}
function toLower(value24) {
    return toString(value24).toLowerCase();
}
function toPath(value24) {
    if (isArray(value24)) {
        return arrayMap(value24, toKey);
    }
    return isSymbol(value24) ? [
        value24
    ] : copyArray(stringToPath(toString(value24)));
}
var MAX_SAFE_INTEGER$5 = 9007199254740991;
function toSafeInteger(value24) {
    return value24 ? baseClamp(toInteger(value24), -MAX_SAFE_INTEGER$5, MAX_SAFE_INTEGER$5) : value24 === 0 ? value24 : 0;
}
function toUpper(value24) {
    return toString(value24).toUpperCase();
}
function transform(object2, iteratee2, accumulator) {
    var isArr = isArray(object2), isArrLike = isArr || isBuffer(object2) || isTypedArray(object2);
    iteratee2 = baseIteratee(iteratee2);
    if (accumulator == null) {
        var Ctor = object2 && object2.constructor;
        if (isArrLike) {
            accumulator = isArr ? new Ctor() : [];
        } else if (isObject1(object2)) {
            accumulator = isFunction(Ctor) ? baseCreate(getPrototype(object2)) : {
            };
        } else {
            accumulator = {
            };
        }
    }
    (isArrLike ? arrayEach : baseForOwn)(object2, function(value24, index, object3) {
        return iteratee2(accumulator, value24, index, object3);
    });
    return accumulator;
}
function charsEndIndex(strSymbols, chrSymbols) {
    var index = strSymbols.length;
    while((index--) && baseIndexOf(chrSymbols, strSymbols[index], 0) > -1){
    }
    return index;
}
function charsStartIndex(strSymbols, chrSymbols) {
    var index = -1, length = strSymbols.length;
    while(++index < length && baseIndexOf(chrSymbols, strSymbols[index], 0) > -1){
    }
    return index;
}
function trim(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return baseTrim(string2);
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), chrSymbols = stringToArray(chars), start = charsStartIndex(strSymbols, chrSymbols), end = charsEndIndex(strSymbols, chrSymbols) + 1;
    return castSlice(strSymbols, start, end).join("");
}
function trimEnd(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return string2.slice(0, trimmedEndIndex(string2) + 1);
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), end = charsEndIndex(strSymbols, stringToArray(chars)) + 1;
    return castSlice(strSymbols, 0, end).join("");
}
var reTrimStart$2 = /^\s+/;
function trimStart(string2, chars, guard) {
    string2 = toString(string2);
    if (string2 && (guard || chars === void 0)) {
        return string2.replace(reTrimStart$2, "");
    }
    if (!string2 || !(chars = baseToString(chars))) {
        return string2;
    }
    var strSymbols = stringToArray(string2), start = charsStartIndex(strSymbols, stringToArray(chars));
    return castSlice(strSymbols, start).join("");
}
var DEFAULT_TRUNC_LENGTH = 30, DEFAULT_TRUNC_OMISSION = "...";
var reFlags$1 = /\w*$/;
function truncate(string2, options12) {
    var length = DEFAULT_TRUNC_LENGTH, omission = DEFAULT_TRUNC_OMISSION;
    if (isObject1(options12)) {
        var separator = "separator" in options12 ? options12.separator : separator;
        length = "length" in options12 ? toInteger(options12.length) : length;
        omission = "omission" in options12 ? baseToString(options12.omission) : omission;
    }
    string2 = toString(string2);
    var strLength = string2.length;
    if (hasUnicode(string2)) {
        var strSymbols = stringToArray(string2);
        strLength = strSymbols.length;
    }
    if (length >= strLength) {
        return string2;
    }
    var end = length - stringSize(omission);
    if (end < 1) {
        return omission;
    }
    var result2 = strSymbols ? castSlice(strSymbols, 0, end).join("") : string2.slice(0, end);
    if (separator === void 0) {
        return result2 + omission;
    }
    if (strSymbols) {
        end += result2.length - end;
    }
    if (isRegExp(separator)) {
        if (string2.slice(end).search(separator)) {
            var match, substring = result2;
            if (!separator.global) {
                separator = RegExp(separator.source, toString(reFlags$1.exec(separator)) + "g");
            }
            separator.lastIndex = 0;
            while(match = separator.exec(substring)){
                var newEnd = match.index;
            }
            result2 = result2.slice(0, newEnd === void 0 ? end : newEnd);
        }
    } else if (string2.indexOf(baseToString(separator), end) != end) {
        var index = result2.lastIndexOf(separator);
        if (index > -1) {
            result2 = result2.slice(0, index);
        }
    }
    return result2 + omission;
}
function unary(func2) {
    return ary(func2, 1);
}
var htmlUnescapes = {
    "&amp;": "&",
    "&lt;": "<",
    "&gt;": ">",
    "&quot;": '"',
    "&#39;": "'"
};
var unescapeHtmlChar = basePropertyOf(htmlUnescapes);
var reEscapedHtml = /&(?:amp|lt|gt|quot|#39);/g, reHasEscapedHtml = RegExp(reEscapedHtml.source);
function unescape1(string2) {
    string2 = toString(string2);
    return string2 && reHasEscapedHtml.test(string2) ? string2.replace(reEscapedHtml, unescapeHtmlChar) : string2;
}
var INFINITY$5 = 1 / 0;
var createSet = !(Set1 && 1 / setToArray(new Set1([
    ,
    -0
]))[1] == INFINITY$5) ? noop : function(values21) {
    return new Set1(values21);
};
var LARGE_ARRAY_SIZE$2 = 200;
function baseUniq(array2, iteratee2, comparator) {
    var index = -1, includes2 = arrayIncludes, length = array2.length, isCommon = true, result2 = [], seen = result2;
    if (comparator) {
        isCommon = false;
        includes2 = arrayIncludesWith;
    } else if (length >= LARGE_ARRAY_SIZE$2) {
        var set2 = iteratee2 ? null : createSet(array2);
        if (set2) {
            return setToArray(set2);
        }
        isCommon = false;
        includes2 = cacheHas;
        seen = new SetCache();
    } else {
        seen = iteratee2 ? [] : result2;
    }
    outer: while(++index < length){
        var value24 = array2[index], computed = iteratee2 ? iteratee2(value24) : value24;
        value24 = comparator || value24 !== 0 ? value24 : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee2) {
                seen.push(computed);
            }
            result2.push(value24);
        } else if (!includes2(seen, computed, comparator)) {
            if (seen !== result2) {
                seen.push(computed);
            }
            result2.push(value24);
        }
    }
    return result2;
}
var union = baseRest(function(arrays) {
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true));
});
var unionBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true), baseIteratee(iteratee2));
});
var unionWith = baseRest(function(arrays) {
    var comparator = last(arrays);
    comparator = typeof comparator == "function" ? comparator : void 0;
    return baseUniq(baseFlatten(arrays, 1, isArrayLikeObject, true), void 0, comparator);
});
function uniq(array2) {
    return array2 && array2.length ? baseUniq(array2) : [];
}
function uniqBy(array2, iteratee2) {
    return array2 && array2.length ? baseUniq(array2, baseIteratee(iteratee2)) : [];
}
function uniqWith(array2, comparator) {
    comparator = typeof comparator == "function" ? comparator : void 0;
    return array2 && array2.length ? baseUniq(array2, void 0, comparator) : [];
}
var idCounter = 0;
function uniqueId(prefix) {
    var id1 = ++idCounter;
    return toString(prefix) + id1;
}
function unset(object2, path4) {
    return object2 == null ? true : baseUnset(object2, path4);
}
var nativeMax$e = Math.max;
function unzip(array2) {
    if (!(array2 && array2.length)) {
        return [];
    }
    var length = 0;
    array2 = arrayFilter(array2, function(group) {
        if (isArrayLikeObject(group)) {
            length = nativeMax$e(group.length, length);
            return true;
        }
    });
    return baseTimes(length, function(index) {
        return arrayMap(array2, baseProperty(index));
    });
}
function unzipWith(array2, iteratee2) {
    if (!(array2 && array2.length)) {
        return [];
    }
    var result2 = unzip(array2);
    if (iteratee2 == null) {
        return result2;
    }
    return arrayMap(result2, function(group) {
        return apply(iteratee2, void 0, group);
    });
}
function baseUpdate(object2, path4, updater, customizer) {
    return baseSet(object2, path4, updater(baseGet(object2, path4)), customizer);
}
function update2(object2, path4, updater) {
    return object2 == null ? object2 : baseUpdate(object2, path4, castFunction(updater));
}
function updateWith(object2, path4, updater, customizer) {
    customizer = typeof customizer == "function" ? customizer : void 0;
    return object2 == null ? object2 : baseUpdate(object2, path4, castFunction(updater), customizer);
}
var upperCase = createCompounder(function(result2, word, index) {
    return result2 + (index ? " " : "") + word.toUpperCase();
});
function valuesIn(object2) {
    return object2 == null ? [] : baseValues(object2, keysIn(object2));
}
var without = baseRest(function(array2, values21) {
    return isArrayLikeObject(array2) ? baseDifference(array2, values21) : [];
});
function wrap(value25, wrapper) {
    return partial2(castFunction(wrapper), value25);
}
var wrapperAt = flatRest(function(paths) {
    var length = paths.length, start = length ? paths[0] : 0, value25 = this.__wrapped__, interceptor = function(object2) {
        return baseAt(object2, paths);
    };
    if (length > 1 || this.__actions__.length || !(value25 instanceof LazyWrapper) || !isIndex(start)) {
        return this.thru(interceptor);
    }
    value25 = value25.slice(start, +start + (length ? 1 : 0));
    value25.__actions__.push({
        func: thru,
        args: [
            interceptor
        ],
        thisArg: void 0
    });
    return new LodashWrapper(value25, this.__chain__).thru(function(array2) {
        if (length && !array2.length) {
            array2.push(void 0);
        }
        return array2;
    });
});
function wrapperChain() {
    return chain(this);
}
function wrapperReverse() {
    var value25 = this.__wrapped__;
    if (value25 instanceof LazyWrapper) {
        var wrapped = value25;
        if (this.__actions__.length) {
            wrapped = new LazyWrapper(this);
        }
        wrapped = wrapped.reverse();
        wrapped.__actions__.push({
            func: thru,
            args: [
                reverse
            ],
            thisArg: void 0
        });
        return new LodashWrapper(wrapped, this.__chain__);
    }
    return this.thru(reverse);
}
function baseXor(arrays, iteratee2, comparator) {
    var length = arrays.length;
    if (length < 2) {
        return length ? baseUniq(arrays[0]) : [];
    }
    var index = -1, result2 = Array(length);
    while(++index < length){
        var array2 = arrays[index], othIndex = -1;
        while(++othIndex < length){
            if (othIndex != index) {
                result2[index] = baseDifference(result2[index] || array2, arrays[othIndex], iteratee2, comparator);
            }
        }
    }
    return baseUniq(baseFlatten(result2, 1), iteratee2, comparator);
}
var xor = baseRest(function(arrays) {
    return baseXor(arrayFilter(arrays, isArrayLikeObject));
});
var xorBy = baseRest(function(arrays) {
    var iteratee2 = last(arrays);
    if (isArrayLikeObject(iteratee2)) {
        iteratee2 = void 0;
    }
    return baseXor(arrayFilter(arrays, isArrayLikeObject), baseIteratee(iteratee2));
});
var xorWith = baseRest(function(arrays) {
    var comparator = last(arrays);
    comparator = typeof comparator == "function" ? comparator : void 0;
    return baseXor(arrayFilter(arrays, isArrayLikeObject), void 0, comparator);
});
var zip = baseRest(unzip);
function baseZipObject(props, values21, assignFunc) {
    var index = -1, length = props.length, valsLength = values21.length, result2 = {
    };
    while(++index < length){
        var value25 = index < valsLength ? values21[index] : void 0;
        assignFunc(result2, props[index], value25);
    }
    return result2;
}
function zipObject(props, values21) {
    return baseZipObject(props || [], values21 || [], assignValue);
}
function zipObjectDeep(props, values21) {
    return baseZipObject(props || [], values21 || [], baseSet);
}
var zipWith = baseRest(function(arrays) {
    var length = arrays.length, iteratee2 = length > 1 ? arrays[length - 1] : void 0;
    iteratee2 = typeof iteratee2 == "function" ? (arrays.pop(), iteratee2) : void 0;
    return unzipWith(arrays, iteratee2);
});
var array = {
    chunk: chunk1,
    compact,
    concat: concat1,
    difference,
    differenceBy,
    differenceWith,
    drop,
    dropRight,
    dropRightWhile,
    dropWhile,
    fill,
    findIndex,
    findLastIndex,
    first: head1,
    flatten,
    flattenDeep,
    flattenDepth,
    fromPairs,
    head: head1,
    indexOf,
    initial,
    intersection,
    intersectionBy,
    intersectionWith,
    join: join4,
    last,
    lastIndexOf,
    nth,
    pull,
    pullAll,
    pullAllBy,
    pullAllWith,
    pullAt,
    remove,
    reverse,
    slice,
    sortedIndex,
    sortedIndexBy,
    sortedIndexOf,
    sortedLastIndex,
    sortedLastIndexBy,
    sortedLastIndexOf,
    sortedUniq,
    sortedUniqBy,
    tail,
    take,
    takeRight,
    takeRightWhile,
    takeWhile,
    union,
    unionBy,
    unionWith,
    uniq,
    uniqBy,
    uniqWith,
    unzip,
    unzipWith,
    without,
    xor,
    xorBy,
    xorWith,
    zip,
    zipObject,
    zipObjectDeep,
    zipWith
};
var collection = {
    countBy,
    each: forEach,
    eachRight: forEachRight,
    every,
    filter,
    find,
    findLast,
    flatMap,
    flatMapDeep,
    flatMapDepth,
    forEach,
    forEachRight,
    groupBy: groupBy1,
    includes,
    invokeMap,
    keyBy,
    map: map1,
    orderBy,
    partition,
    reduce,
    reduceRight,
    reject,
    sample,
    sampleSize,
    shuffle,
    size: size4,
    some,
    sortBy
};
var date = {
    now
};
var func1 = {
    after,
    ary,
    before,
    bind,
    bindKey,
    curry,
    curryRight,
    debounce,
    defer,
    delay,
    flip,
    memoize,
    negate,
    once,
    overArgs,
    partial: partial2,
    partialRight,
    rearg,
    rest,
    spread,
    throttle,
    unary,
    wrap
};
var lang = {
    castArray,
    clone,
    cloneDeep,
    cloneDeepWith,
    cloneWith,
    conformsTo,
    eq,
    gt,
    gte,
    isArguments,
    isArray,
    isArrayBuffer,
    isArrayLike,
    isArrayLikeObject,
    isBoolean: isBoolean1,
    isBuffer,
    isDate,
    isElement,
    isEmpty,
    isEqual,
    isEqualWith,
    isError,
    isFinite,
    isFunction,
    isInteger: isInteger1,
    isLength,
    isMap,
    isMatch,
    isMatchWith,
    isNaN: isNaN1,
    isNative,
    isNil,
    isNull: isNull1,
    isNumber,
    isObject: isObject1,
    isObjectLike,
    isPlainObject,
    isRegExp,
    isSafeInteger,
    isSet,
    isString,
    isSymbol,
    isTypedArray,
    isUndefined,
    isWeakMap,
    isWeakSet,
    lt,
    lte,
    toArray,
    toFinite,
    toInteger,
    toLength,
    toNumber,
    toPlainObject,
    toSafeInteger,
    toString
};
var math = {
    add,
    ceil,
    divide,
    floor,
    max,
    maxBy,
    mean,
    meanBy,
    min: min1,
    minBy,
    multiply,
    round,
    subtract,
    sum,
    sumBy
};
var number1 = {
    clamp,
    inRange,
    random: random1
};
var object = {
    assign,
    assignIn,
    assignInWith,
    assignWith,
    at,
    create,
    defaults: defaults1,
    defaultsDeep,
    entries: toPairs,
    entriesIn: toPairsIn,
    extend: assignIn,
    extendWith: assignInWith,
    findKey,
    findLastKey,
    forIn,
    forInRight,
    forOwn,
    forOwnRight,
    functions,
    functionsIn,
    get,
    has,
    hasIn,
    invert,
    invertBy,
    invoke,
    keys: keys1,
    keysIn,
    mapKeys,
    mapValues,
    merge: merge1,
    mergeWith,
    omit,
    omitBy,
    pick,
    pickBy,
    result: result1,
    set: set1,
    setWith,
    toPairs,
    toPairsIn,
    transform,
    unset,
    update: update2,
    updateWith,
    values: values2,
    valuesIn
};
var seq1 = {
    at: wrapperAt,
    chain,
    commit: wrapperCommit,
    lodash,
    next: wrapperNext,
    plant: wrapperPlant,
    reverse: wrapperReverse,
    tap,
    thru,
    toIterator: wrapperToIterator,
    toJSON: wrapperValue,
    value: wrapperValue,
    valueOf: wrapperValue,
    wrapperChain
};
var string1 = {
    camelCase,
    capitalize: capitalize1,
    deburr,
    endsWith,
    escape,
    escapeRegExp,
    kebabCase,
    lowerCase,
    lowerFirst,
    pad,
    padEnd,
    padStart,
    parseInt: parseInt$1,
    repeat: repeat1,
    replace,
    snakeCase,
    split: split1,
    startCase,
    startsWith,
    template,
    templateSettings,
    toLower,
    toUpper,
    trim,
    trimEnd,
    trimStart,
    truncate,
    unescape: unescape1,
    upperCase,
    upperFirst,
    words
};
var util1 = {
    attempt,
    bindAll,
    cond,
    conforms,
    constant,
    defaultTo,
    flow,
    flowRight,
    identity,
    iteratee,
    matches,
    matchesProperty,
    method: method1,
    methodOf,
    mixin,
    noop,
    nthArg,
    over,
    overEvery,
    overSome,
    property,
    propertyOf,
    range,
    rangeRight,
    stubArray,
    stubFalse,
    stubObject,
    stubString,
    stubTrue,
    times,
    toPath,
    uniqueId
};
function lazyClone() {
    var result2 = new LazyWrapper(this.__wrapped__);
    result2.__actions__ = copyArray(this.__actions__);
    result2.__dir__ = this.__dir__;
    result2.__filtered__ = this.__filtered__;
    result2.__iteratees__ = copyArray(this.__iteratees__);
    result2.__takeCount__ = this.__takeCount__;
    result2.__views__ = copyArray(this.__views__);
    return result2;
}
function lazyReverse() {
    if (this.__filtered__) {
        var result2 = new LazyWrapper(this);
        result2.__dir__ = -1;
        result2.__filtered__ = true;
    } else {
        result2 = this.clone();
        result2.__dir__ *= -1;
    }
    return result2;
}
var nativeMax$f = Math.max, nativeMin$c = Math.min;
function getView(start, end, transforms) {
    var index = -1, length = transforms.length;
    while(++index < length){
        var data = transforms[index], size21 = data.size;
        switch(data.type){
            case "drop":
                start += size21;
                break;
            case "dropRight":
                end -= size21;
                break;
            case "take":
                end = nativeMin$c(end, start + size21);
                break;
            case "takeRight":
                start = nativeMax$f(start, end - size21);
                break;
        }
    }
    return {
        start,
        end
    };
}
var LAZY_FILTER_FLAG = 1, LAZY_MAP_FLAG = 2;
var nativeMin$d = Math.min;
function lazyValue() {
    var array2 = this.__wrapped__.value(), dir = this.__dir__, isArr = isArray(array2), isRight = dir < 0, arrLength = isArr ? array2.length : 0, view = getView(0, arrLength, this.__views__), start = view.start, end = view.end, length = end - start, index = isRight ? end : start - 1, iteratees = this.__iteratees__, iterLength = iteratees.length, resIndex = 0, takeCount = nativeMin$d(length, this.__takeCount__);
    if (!isArr || !isRight && arrLength == length && takeCount == length) {
        return baseWrapperValue(array2, this.__actions__);
    }
    var result2 = [];
    outer: while((length--) && resIndex < takeCount){
        index += dir;
        var iterIndex = -1, value26 = array2[index];
        while(++iterIndex < iterLength){
            var data = iteratees[iterIndex], iteratee2 = data.iteratee, type5 = data.type, computed = iteratee2(value26);
            if (type5 == LAZY_MAP_FLAG) {
                value26 = computed;
            } else if (!computed) {
                if (type5 == LAZY_FILTER_FLAG) {
                    continue outer;
                } else {
                    break outer;
                }
            }
        }
        result2[resIndex++] = value26;
    }
    return result2;
}
var VERSION = "4.17.21";
var WRAP_BIND_KEY_FLAG$6 = 2;
var LAZY_FILTER_FLAG$1 = 1, LAZY_WHILE_FLAG = 3;
var MAX_ARRAY_LENGTH$6 = 4294967295;
var arrayProto$5 = Array.prototype, objectProto$s = Object.prototype;
var hasOwnProperty$o = objectProto$s.hasOwnProperty;
var symIterator$1 = Symbol1 ? Symbol1.iterator : void 0;
var nativeMax$g = Math.max, nativeMin$e = Math.min;
var mixin$1 = function(func2) {
    return function(object2, source, options12) {
        if (options12 == null) {
            var isObj = isObject1(source), props = isObj && keys1(source), methodNames = props && props.length && baseFunctions(source, props);
            if (!(methodNames ? methodNames.length : isObj)) {
                options12 = source;
                source = object2;
                object2 = this;
            }
        }
        return func2(object2, source, options12);
    };
}(mixin);
lodash.after = func1.after;
lodash.ary = func1.ary;
lodash.assign = object.assign;
lodash.assignIn = object.assignIn;
lodash.assignInWith = object.assignInWith;
lodash.assignWith = object.assignWith;
lodash.at = object.at;
lodash.before = func1.before;
lodash.bind = func1.bind;
lodash.bindAll = util1.bindAll;
lodash.bindKey = func1.bindKey;
lodash.castArray = lang.castArray;
lodash.chain = seq1.chain;
lodash.chunk = array.chunk;
lodash.compact = array.compact;
lodash.concat = array.concat;
lodash.cond = util1.cond;
lodash.conforms = util1.conforms;
lodash.constant = util1.constant;
lodash.countBy = collection.countBy;
lodash.create = object.create;
lodash.curry = func1.curry;
lodash.curryRight = func1.curryRight;
lodash.debounce = func1.debounce;
lodash.defaults = object.defaults;
lodash.defaultsDeep = object.defaultsDeep;
lodash.defer = func1.defer;
lodash.delay = func1.delay;
lodash.difference = array.difference;
lodash.differenceBy = array.differenceBy;
lodash.differenceWith = array.differenceWith;
lodash.drop = array.drop;
lodash.dropRight = array.dropRight;
lodash.dropRightWhile = array.dropRightWhile;
lodash.dropWhile = array.dropWhile;
lodash.fill = array.fill;
lodash.filter = collection.filter;
lodash.flatMap = collection.flatMap;
lodash.flatMapDeep = collection.flatMapDeep;
lodash.flatMapDepth = collection.flatMapDepth;
lodash.flatten = array.flatten;
lodash.flattenDeep = array.flattenDeep;
lodash.flattenDepth = array.flattenDepth;
lodash.flip = func1.flip;
lodash.flow = util1.flow;
lodash.flowRight = util1.flowRight;
lodash.fromPairs = array.fromPairs;
lodash.functions = object.functions;
lodash.functionsIn = object.functionsIn;
lodash.groupBy = collection.groupBy;
lodash.initial = array.initial;
lodash.intersection = array.intersection;
lodash.intersectionBy = array.intersectionBy;
lodash.intersectionWith = array.intersectionWith;
lodash.invert = object.invert;
lodash.invertBy = object.invertBy;
lodash.invokeMap = collection.invokeMap;
lodash.iteratee = util1.iteratee;
lodash.keyBy = collection.keyBy;
lodash.keys = keys1;
lodash.keysIn = object.keysIn;
lodash.map = collection.map;
lodash.mapKeys = object.mapKeys;
lodash.mapValues = object.mapValues;
lodash.matches = util1.matches;
lodash.matchesProperty = util1.matchesProperty;
lodash.memoize = func1.memoize;
lodash.merge = object.merge;
lodash.mergeWith = object.mergeWith;
lodash.method = util1.method;
lodash.methodOf = util1.methodOf;
lodash.mixin = mixin$1;
lodash.negate = negate;
lodash.nthArg = util1.nthArg;
lodash.omit = object.omit;
lodash.omitBy = object.omitBy;
lodash.once = func1.once;
lodash.orderBy = collection.orderBy;
lodash.over = util1.over;
lodash.overArgs = func1.overArgs;
lodash.overEvery = util1.overEvery;
lodash.overSome = util1.overSome;
lodash.partial = func1.partial;
lodash.partialRight = func1.partialRight;
lodash.partition = collection.partition;
lodash.pick = object.pick;
lodash.pickBy = object.pickBy;
lodash.property = util1.property;
lodash.propertyOf = util1.propertyOf;
lodash.pull = array.pull;
lodash.pullAll = array.pullAll;
lodash.pullAllBy = array.pullAllBy;
lodash.pullAllWith = array.pullAllWith;
lodash.pullAt = array.pullAt;
lodash.range = util1.range;
lodash.rangeRight = util1.rangeRight;
lodash.rearg = func1.rearg;
lodash.reject = collection.reject;
lodash.remove = array.remove;
lodash.rest = func1.rest;
lodash.reverse = array.reverse;
lodash.sampleSize = collection.sampleSize;
lodash.set = object.set;
lodash.setWith = object.setWith;
lodash.shuffle = collection.shuffle;
lodash.slice = array.slice;
lodash.sortBy = collection.sortBy;
lodash.sortedUniq = array.sortedUniq;
lodash.sortedUniqBy = array.sortedUniqBy;
lodash.split = string1.split;
lodash.spread = func1.spread;
lodash.tail = array.tail;
lodash.take = array.take;
lodash.takeRight = array.takeRight;
lodash.takeRightWhile = array.takeRightWhile;
lodash.takeWhile = array.takeWhile;
lodash.tap = seq1.tap;
lodash.throttle = func1.throttle;
lodash.thru = thru;
lodash.toArray = lang.toArray;
lodash.toPairs = object.toPairs;
lodash.toPairsIn = object.toPairsIn;
lodash.toPath = util1.toPath;
lodash.toPlainObject = lang.toPlainObject;
lodash.transform = object.transform;
lodash.unary = func1.unary;
lodash.union = array.union;
lodash.unionBy = array.unionBy;
lodash.unionWith = array.unionWith;
lodash.uniq = array.uniq;
lodash.uniqBy = array.uniqBy;
lodash.uniqWith = array.uniqWith;
lodash.unset = object.unset;
lodash.unzip = array.unzip;
lodash.unzipWith = array.unzipWith;
lodash.update = object.update;
lodash.updateWith = object.updateWith;
lodash.values = object.values;
lodash.valuesIn = object.valuesIn;
lodash.without = array.without;
lodash.words = string1.words;
lodash.wrap = func1.wrap;
lodash.xor = array.xor;
lodash.xorBy = array.xorBy;
lodash.xorWith = array.xorWith;
lodash.zip = array.zip;
lodash.zipObject = array.zipObject;
lodash.zipObjectDeep = array.zipObjectDeep;
lodash.zipWith = array.zipWith;
lodash.entries = object.toPairs;
lodash.entriesIn = object.toPairsIn;
lodash.extend = object.assignIn;
lodash.extendWith = object.assignInWith;
mixin$1(lodash, lodash);
lodash.add = math.add;
lodash.attempt = util1.attempt;
lodash.camelCase = string1.camelCase;
lodash.capitalize = string1.capitalize;
lodash.ceil = math.ceil;
lodash.clamp = number1.clamp;
lodash.clone = lang.clone;
lodash.cloneDeep = lang.cloneDeep;
lodash.cloneDeepWith = lang.cloneDeepWith;
lodash.cloneWith = lang.cloneWith;
lodash.conformsTo = lang.conformsTo;
lodash.deburr = string1.deburr;
lodash.defaultTo = util1.defaultTo;
lodash.divide = math.divide;
lodash.endsWith = string1.endsWith;
lodash.eq = lang.eq;
lodash.escape = string1.escape;
lodash.escapeRegExp = string1.escapeRegExp;
lodash.every = collection.every;
lodash.find = collection.find;
lodash.findIndex = array.findIndex;
lodash.findKey = object.findKey;
lodash.findLast = collection.findLast;
lodash.findLastIndex = array.findLastIndex;
lodash.findLastKey = object.findLastKey;
lodash.floor = math.floor;
lodash.forEach = collection.forEach;
lodash.forEachRight = collection.forEachRight;
lodash.forIn = object.forIn;
lodash.forInRight = object.forInRight;
lodash.forOwn = object.forOwn;
lodash.forOwnRight = object.forOwnRight;
lodash.get = object.get;
lodash.gt = lang.gt;
lodash.gte = lang.gte;
lodash.has = object.has;
lodash.hasIn = object.hasIn;
lodash.head = array.head;
lodash.identity = identity;
lodash.includes = collection.includes;
lodash.indexOf = array.indexOf;
lodash.inRange = number1.inRange;
lodash.invoke = object.invoke;
lodash.isArguments = lang.isArguments;
lodash.isArray = isArray;
lodash.isArrayBuffer = lang.isArrayBuffer;
lodash.isArrayLike = lang.isArrayLike;
lodash.isArrayLikeObject = lang.isArrayLikeObject;
lodash.isBoolean = lang.isBoolean;
lodash.isBuffer = lang.isBuffer;
lodash.isDate = lang.isDate;
lodash.isElement = lang.isElement;
lodash.isEmpty = lang.isEmpty;
lodash.isEqual = lang.isEqual;
lodash.isEqualWith = lang.isEqualWith;
lodash.isError = lang.isError;
lodash.isFinite = lang.isFinite;
lodash.isFunction = lang.isFunction;
lodash.isInteger = lang.isInteger;
lodash.isLength = lang.isLength;
lodash.isMap = lang.isMap;
lodash.isMatch = lang.isMatch;
lodash.isMatchWith = lang.isMatchWith;
lodash.isNaN = lang.isNaN;
lodash.isNative = lang.isNative;
lodash.isNil = lang.isNil;
lodash.isNull = lang.isNull;
lodash.isNumber = lang.isNumber;
lodash.isObject = isObject1;
lodash.isObjectLike = lang.isObjectLike;
lodash.isPlainObject = lang.isPlainObject;
lodash.isRegExp = lang.isRegExp;
lodash.isSafeInteger = lang.isSafeInteger;
lodash.isSet = lang.isSet;
lodash.isString = lang.isString;
lodash.isSymbol = lang.isSymbol;
lodash.isTypedArray = lang.isTypedArray;
lodash.isUndefined = lang.isUndefined;
lodash.isWeakMap = lang.isWeakMap;
lodash.isWeakSet = lang.isWeakSet;
lodash.join = array.join;
lodash.kebabCase = string1.kebabCase;
lodash.last = last;
lodash.lastIndexOf = array.lastIndexOf;
lodash.lowerCase = string1.lowerCase;
lodash.lowerFirst = string1.lowerFirst;
lodash.lt = lang.lt;
lodash.lte = lang.lte;
lodash.max = math.max;
lodash.maxBy = math.maxBy;
lodash.mean = math.mean;
lodash.meanBy = math.meanBy;
lodash.min = math.min;
lodash.minBy = math.minBy;
lodash.stubArray = util1.stubArray;
lodash.stubFalse = util1.stubFalse;
lodash.stubObject = util1.stubObject;
lodash.stubString = util1.stubString;
lodash.stubTrue = util1.stubTrue;
lodash.multiply = math.multiply;
lodash.nth = array.nth;
lodash.noop = util1.noop;
lodash.now = date.now;
lodash.pad = string1.pad;
lodash.padEnd = string1.padEnd;
lodash.padStart = string1.padStart;
lodash.parseInt = string1.parseInt;
lodash.random = number1.random;
lodash.reduce = collection.reduce;
lodash.reduceRight = collection.reduceRight;
lodash.repeat = string1.repeat;
lodash.replace = string1.replace;
lodash.result = object.result;
lodash.round = math.round;
lodash.sample = collection.sample;
lodash.size = collection.size;
lodash.snakeCase = string1.snakeCase;
lodash.some = collection.some;
lodash.sortedIndex = array.sortedIndex;
lodash.sortedIndexBy = array.sortedIndexBy;
lodash.sortedIndexOf = array.sortedIndexOf;
lodash.sortedLastIndex = array.sortedLastIndex;
lodash.sortedLastIndexBy = array.sortedLastIndexBy;
lodash.sortedLastIndexOf = array.sortedLastIndexOf;
lodash.startCase = string1.startCase;
lodash.startsWith = string1.startsWith;
lodash.subtract = math.subtract;
lodash.sum = math.sum;
lodash.sumBy = math.sumBy;
lodash.template = string1.template;
lodash.times = util1.times;
lodash.toFinite = lang.toFinite;
lodash.toInteger = toInteger;
lodash.toLength = lang.toLength;
lodash.toLower = string1.toLower;
lodash.toNumber = lang.toNumber;
lodash.toSafeInteger = lang.toSafeInteger;
lodash.toString = lang.toString;
lodash.toUpper = string1.toUpper;
lodash.trim = string1.trim;
lodash.trimEnd = string1.trimEnd;
lodash.trimStart = string1.trimStart;
lodash.truncate = string1.truncate;
lodash.unescape = string1.unescape;
lodash.uniqueId = util1.uniqueId;
lodash.upperCase = string1.upperCase;
lodash.upperFirst = string1.upperFirst;
lodash.each = collection.forEach;
lodash.eachRight = collection.forEachRight;
lodash.first = array.head;
mixin$1(lodash, function() {
    var source = {
    };
    baseForOwn(lodash, function(func2, methodName) {
        if (!hasOwnProperty$o.call(lodash.prototype, methodName)) {
            source[methodName] = func2;
        }
    });
    return source;
}(), {
    chain: false
});
lodash.VERSION = VERSION;
(lodash.templateSettings = string1.templateSettings).imports._ = lodash;
arrayEach([
    "bind",
    "bindKey",
    "curry",
    "curryRight",
    "partial",
    "partialRight"
], function(methodName) {
    lodash[methodName].placeholder = lodash;
});
arrayEach([
    "drop",
    "take"
], function(methodName, index) {
    LazyWrapper.prototype[methodName] = function(n) {
        n = n === void 0 ? 1 : nativeMax$g(toInteger(n), 0);
        var result2 = this.__filtered__ && !index ? new LazyWrapper(this) : this.clone();
        if (result2.__filtered__) {
            result2.__takeCount__ = nativeMin$e(n, result2.__takeCount__);
        } else {
            result2.__views__.push({
                size: nativeMin$e(n, MAX_ARRAY_LENGTH$6),
                type: methodName + (result2.__dir__ < 0 ? "Right" : "")
            });
        }
        return result2;
    };
    LazyWrapper.prototype[methodName + "Right"] = function(n) {
        return this.reverse()[methodName](n).reverse();
    };
});
arrayEach([
    "filter",
    "map",
    "takeWhile"
], function(methodName, index) {
    var type6 = index + 1, isFilter = type6 == LAZY_FILTER_FLAG$1 || type6 == LAZY_WHILE_FLAG;
    LazyWrapper.prototype[methodName] = function(iteratee2) {
        var result2 = this.clone();
        result2.__iteratees__.push({
            iteratee: baseIteratee(iteratee2),
            type: type6
        });
        result2.__filtered__ = result2.__filtered__ || isFilter;
        return result2;
    };
});
arrayEach([
    "head",
    "last"
], function(methodName, index) {
    var takeName = "take" + (index ? "Right" : "");
    LazyWrapper.prototype[methodName] = function() {
        return this[takeName](1).value()[0];
    };
});
arrayEach([
    "initial",
    "tail"
], function(methodName, index) {
    var dropName = "drop" + (index ? "" : "Right");
    LazyWrapper.prototype[methodName] = function() {
        return this.__filtered__ ? new LazyWrapper(this) : this[dropName](1);
    };
});
LazyWrapper.prototype.compact = function() {
    return this.filter(identity);
};
LazyWrapper.prototype.find = function(predicate) {
    return this.filter(predicate).head();
};
LazyWrapper.prototype.findLast = function(predicate) {
    return this.reverse().find(predicate);
};
LazyWrapper.prototype.invokeMap = baseRest(function(path4, args4) {
    if (typeof path4 == "function") {
        return new LazyWrapper(this);
    }
    return this.map(function(value27) {
        return baseInvoke(value27, path4, args4);
    });
});
LazyWrapper.prototype.reject = function(predicate) {
    return this.filter(negate(baseIteratee(predicate)));
};
LazyWrapper.prototype.slice = function(start, end) {
    start = toInteger(start);
    var result2 = this;
    if (result2.__filtered__ && (start > 0 || end < 0)) {
        return new LazyWrapper(result2);
    }
    if (start < 0) {
        result2 = result2.takeRight(-start);
    } else if (start) {
        result2 = result2.drop(start);
    }
    if (end !== void 0) {
        end = toInteger(end);
        result2 = end < 0 ? result2.dropRight(-end) : result2.take(end - start);
    }
    return result2;
};
LazyWrapper.prototype.takeRightWhile = function(predicate) {
    return this.reverse().takeWhile(predicate).reverse();
};
LazyWrapper.prototype.toArray = function() {
    return this.take(MAX_ARRAY_LENGTH$6);
};
baseForOwn(LazyWrapper.prototype, function(func2, methodName) {
    var checkIteratee = /^(?:filter|find|map|reject)|While$/.test(methodName), isTaker = /^(?:head|last)$/.test(methodName), lodashFunc = lodash[isTaker ? "take" + (methodName == "last" ? "Right" : "") : methodName], retUnwrapped = isTaker || /^find/.test(methodName);
    if (!lodashFunc) {
        return;
    }
    lodash.prototype[methodName] = function() {
        var value27 = this.__wrapped__, args4 = isTaker ? [
            1
        ] : arguments, isLazy = value27 instanceof LazyWrapper, iteratee2 = args4[0], useLazy = isLazy || isArray(value27);
        var interceptor = function(value28) {
            var result3 = lodashFunc.apply(lodash, arrayPush([
                value28
            ], args4));
            return isTaker && chainAll ? result3[0] : result3;
        };
        if (useLazy && checkIteratee && typeof iteratee2 == "function" && iteratee2.length != 1) {
            isLazy = useLazy = false;
        }
        var chainAll = this.__chain__, isHybrid = !!this.__actions__.length, isUnwrapped = retUnwrapped && !chainAll, onlyLazy = isLazy && !isHybrid;
        if (!retUnwrapped && useLazy) {
            value27 = onlyLazy ? value27 : new LazyWrapper(this);
            var result2 = func2.apply(value27, args4);
            result2.__actions__.push({
                func: thru,
                args: [
                    interceptor
                ],
                thisArg: void 0
            });
            return new LodashWrapper(result2, chainAll);
        }
        if (isUnwrapped && onlyLazy) {
            return func2.apply(this, args4);
        }
        result2 = this.thru(interceptor);
        return isUnwrapped ? isTaker ? result2.value()[0] : result2.value() : result2;
    };
});
arrayEach([
    "pop",
    "push",
    "shift",
    "sort",
    "splice",
    "unshift"
], function(methodName) {
    var func2 = arrayProto$5[methodName], chainName = /^(?:push|sort|unshift)$/.test(methodName) ? "tap" : "thru", retUnwrapped = /^(?:pop|shift)$/.test(methodName);
    lodash.prototype[methodName] = function() {
        var args4 = arguments;
        if (retUnwrapped && !this.__chain__) {
            var value27 = this.value();
            return func2.apply(isArray(value27) ? value27 : [], args4);
        }
        return this[chainName](function(value28) {
            return func2.apply(isArray(value28) ? value28 : [], args4);
        });
    };
});
baseForOwn(LazyWrapper.prototype, function(func2, methodName) {
    var lodashFunc = lodash[methodName];
    if (lodashFunc) {
        var key = lodashFunc.name + "";
        if (!hasOwnProperty$o.call(realNames, key)) {
            realNames[key] = [];
        }
        realNames[key].push({
            name: methodName,
            func: lodashFunc
        });
    }
});
realNames[createHybrid(void 0, WRAP_BIND_KEY_FLAG$6).name] = [
    {
        name: "wrapper",
        func: void 0
    }
];
LazyWrapper.prototype.clone = lazyClone;
LazyWrapper.prototype.reverse = lazyReverse;
LazyWrapper.prototype.value = lazyValue;
lodash.prototype.at = seq1.at;
lodash.prototype.chain = seq1.wrapperChain;
lodash.prototype.commit = seq1.commit;
lodash.prototype.next = seq1.next;
lodash.prototype.plant = seq1.plant;
lodash.prototype.reverse = seq1.reverse;
lodash.prototype.toJSON = lodash.prototype.valueOf = lodash.prototype.value = seq1.value;
lodash.prototype.first = lodash.prototype.head;
if (symIterator$1) {
    lodash.prototype[symIterator$1] = seq1.toIterator;
}
const ld = lodash;
function bytesToUuid(bytes) {
    const bits = [
        ...bytes
    ].map((bit)=>{
        const s = bit.toString(16);
        return bit < 16 ? "0" + s : s;
    });
    return [
        ...bits.slice(0, 4),
        "-",
        ...bits.slice(4, 6),
        "-",
        ...bits.slice(6, 8),
        "-",
        ...bits.slice(8, 10),
        "-",
        ...bits.slice(10, 16), 
    ].join("");
}
function generate() {
    const rnds = crypto.getRandomValues(new Uint8Array(16));
    rnds[6] = rnds[6] & 15 | 64;
    rnds[8] = rnds[8] & 63 | 128;
    return bytesToUuid(rnds);
}
function mergeConfigs(config, ...configs) {
    config = ld.cloneDeep(config);
    configs = ld.cloneDeep(configs);
    return lodash.mergeWith(config, ...configs, (objValue, srcValue)=>{
        if (lodash.isArray(objValue) || lodash.isArray(srcValue)) {
            if (!objValue) {
                return srcValue;
            } else if (!srcValue) {
                return objValue;
            } else {
                if (!lodash.isArray(objValue)) {
                    objValue = [
                        objValue
                    ];
                }
                if (!lodash.isArray(srcValue)) {
                    srcValue = [
                        srcValue
                    ];
                }
            }
            const combined = objValue.concat(srcValue);
            return lodash.uniqBy(combined, (value28)=>{
                if (typeof value28 === "function") {
                    return generate();
                } else {
                    return JSON.stringify(value28);
                }
            });
        }
    });
}
function camelToKebab(camel) {
    const kebab = [];
    for(let i5 = 0; i5 < camel.length; i5++){
        const ch = camel.charAt(i5);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i5 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i5 = 0; i5 < kebab.length; i5++){
        const ch = kebab.charAt(i5);
        if (ch === "-") {
            camel.push(kebab.charAt(++i5).toUpperCase());
        } else if (i5 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
async function includedMetadata(dir, baseMetadata, schema5) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join3(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join3(dir, file))
        );
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (exists(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema5, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error3("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeConfigs({
        }, ...filesMetadata),
        files: yamlFiles
    };
}
function formatFromMetadata(baseFormat, to, debug1) {
    const typedFormat = {
        render: {
        },
        execute: {
        },
        pandoc: {
        },
        language: {
        },
        metadata: {
        }
    };
    let format6 = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format6 = metadataAsFormat({
            });
        } else if (configFormat instanceof Object) {
            format6 = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeConfigs(baseFormat, format6);
    if (debug1) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
    }
    return mergedFormat;
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key);
}
function metadataAsFormat(metadata1) {
    const typedFormat = {
        render: {
        },
        execute: {
        },
        pandoc: {
        },
        language: {
        },
        metadata: {
        }
    };
    const format6 = typedFormat;
    Object.keys(metadata1).forEach((key)=>{
        if ([
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata, 
        ].includes(key)) {
            if (typeof metadata1[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format6[key] = format6[key] || {
                    };
                    format6[kExecuteDefaults][kExecuteEnabled] = metadata1[key];
                }
            } else {
                format6[key] = {
                    ...format6[key],
                    ...metadata1[key]
                };
            }
        } else {
            if (kRenderDefaultsKeys.includes(key)) {
                format6.render[key] = metadata1[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format6.execute[key] = metadata1[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format6.pandoc[key] = metadata1[key];
            } else if (kLanguageDefaultsKeys.includes(key)) {
                format6.language[key] = metadata1[key];
            } else {
                format6.metadata[key] = metadata1[key];
            }
        }
    });
    return typedFormat;
}
function metadataGetDeep(metadata1, property1) {
    let values3 = [];
    lodash.each(metadata1, (value28, key)=>{
        if (key === property1) {
            values3.push(value28);
        } else if (lodash.isObject(value28)) {
            values3 = values3.concat(metadataGetDeep(value28, property1));
        }
    });
    return values3;
}
const kStdOut = "-";
function parseRenderFlags(args4) {
    const flags = {
    };
    const argsStack = [
        ...args4
    ];
    let arg4 = argsStack.shift();
    while(arg4 !== undefined){
        switch(arg4){
            case "-t":
            case "--to":
                arg4 = argsStack.shift();
                if (arg4 && !arg4.startsWith("-")) {
                    flags.to = arg4;
                }
                break;
            case "-o":
            case "--output":
                arg4 = argsStack.shift();
                if (!arg4 || arg4.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    flags.output = arg4;
                }
                break;
            case "--output-dir":
                arg4 = argsStack.shift();
                flags.outputDir = arg4;
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg4 = argsStack.shift();
                break;
            case "--pdf-engine":
                arg4 = argsStack.shift();
                flags.pdfEngine = arg4;
                break;
            case "--pdf-engine-opt":
                arg4 = argsStack.shift();
                if (arg4) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg4);
                }
                break;
            case "--latex-makeindex-opt":
                arg4 = argsStack.shift();
                if (arg4) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg4);
                }
                break;
            case "--latex-tlmgr-opt":
                arg4 = argsStack.shift();
                if (arg4) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg4);
                }
                break;
            case "--natbib":
                arg4 = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg4 = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg4 = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg4 = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg4 = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg4 = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg4);
                break;
            case "--top-level-division":
                arg4 = argsStack.shift();
                flags[kTopLevelDivision] = arg4;
                break;
            case "--shift-heading-level-by":
                arg4 = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg4;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include2 = arg4.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include2] = includeFlags[include2] || [];
                    arg4 = argsStack.shift();
                    includeFlags[include2].push(arg4);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg4 = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg4 = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg4 = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg4 = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg4 = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg4 = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg4 = argsStack.shift();
                break;
            case "--execute-params":
                arg4 = argsStack.shift();
                flags.paramsFile = arg4;
                break;
            case "--execute-dir":
                arg4 = argsStack.shift();
                flags.executeDir = arg4;
                break;
            case "--execute-daemon":
                arg4 = argsStack.shift();
                flags.executeDaemon = parseInt(arg4, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg4 = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg4 = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg4 = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--cache":
                arg4 = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg4 = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg4 = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--debug":
                flags.debug = true;
                arg4 = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg4 = argsStack.shift();
                if (arg4) {
                    const param = parseMetadataFlagValue(arg4);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {
                            };
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg4 = argsStack.shift();
                if (arg4) {
                    const metadata1 = parseMetadataFlagValue(arg4);
                    if (metadata1) {
                        if (metadata1.value !== undefined) {
                            if (isQuartoMetadata(metadata1.name)) {
                                flags.metadata = flags.metadata || {
                                };
                                flags.metadata[metadata1.name] = metadata1.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {
                                };
                                flags.pandocMetadata[metadata1.name] = metadata1.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg4 = argsStack.shift();
                if (arg4) {
                    if (existsSync(arg4)) {
                        const metadata1 = readYamlFromMarkdownFile(arg4);
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata1
                        };
                    }
                }
                break;
            case "--reference-location":
                arg4 = argsStack.shift();
                if (arg4) {
                    flags[kReferenceLocation] = arg4;
                }
                break;
            default:
                arg4 = argsStack.shift();
                break;
        }
    }
    return flags;
}
function havePandocArg(pandocArgs, arg4) {
    return pandocArgs.indexOf(arg4) !== -1;
}
function replacePandocArg(pandocArgs, arg4, value28) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg4);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value28;
    } else {
        newArgs.push(arg4);
        newArgs.push(value28);
    }
    return newArgs;
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args4, arg4, index)=>{
        args4.push(arg4);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg4 === "-o" || arg4 === "--output")) {
            args4.push(kStdOut);
        }
        return args4;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    let removeNext = false;
    return pandocArgs.reduce((args4, arg4)=>{
        if (!removeArgs.has(arg4)) {
            if (!removeNext) {
                args4.push(arg4);
            }
            removeNext = false;
        } else {
            removeNext = removeArgs.get(arg4);
        }
        return args4;
    }, new Array());
}
function removePandocToArg(args4) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args4, removeArgs);
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args4 = [];
    for(let i5 = 0; i5 < pandocArgs.length; i5++){
        const arg4 = pandocArgs[i5];
        if (arg4 === "--metadata" || arg4 === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i5 + 1] || "");
            if (flagValue !== undefined && isQuartoMetadata(flagValue.name)) {
                i5++;
            } else {
                args4.push(arg4);
            }
        } else {
            args4.push(arg4);
        }
    }
    return args4;
}
function parseMetadataFlagValue(arg4) {
    const match = arg4.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {
        };
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value28) {
    if (value28) {
        const numbers = value28.split(/,/).map((number2)=>parseInt(number2.trim(), 10)
        ).filter((number2)=>!isNaN(number2)
        );
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
const importMeta = {
    url: "https://deno.land/x/deno_dom@v0.1.14-alpha/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder('utf-8');
const encodeString = typeof cachedTextEncoder.encodeInto === 'function' ? function(arg4, view) {
    return cachedTextEncoder.encodeInto(arg4, view);
} : function(arg4, view) {
    const buf = cachedTextEncoder.encode(arg4);
    view.set(buf);
    return {
        read: arg4.length,
        written: buf.length
    };
};
function passStringToWasm0(arg4, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg4);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len = arg4.length;
    let ptr = malloc(len);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len; offset++){
        const code3 = arg4.charCodeAt(offset);
        if (code3 > 127) break;
        mem[ptr + offset] = code3;
    }
    if (offset !== len) {
        if (offset !== 0) {
            arg4 = arg4.slice(offset);
        }
        ptr = realloc(ptr, len, len = offset + arg4.length * 3);
        const view = getUint8Memory0().subarray(ptr + offset, ptr + len);
        const ret = encodeString(arg4, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder('utf-8', {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len));
}
function parse4(html) {
    try {
        const retptr = wasm.__wbindgen_export_0.value - 16;
        wasm.__wbindgen_export_0.value = retptr;
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_export_0.value += 16;
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html) {
    try {
        const retptr = wasm.__wbindgen_export_0.value - 16;
        wasm.__wbindgen_export_0.value = retptr;
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_export_0.value += 16;
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load1(module, imports) {
    if (typeof Response === 'function' && module instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === 'function') {
            try {
                return await WebAssembly.instantiateStreaming(module, imports);
            } catch (e) {
                if (module.headers.get('Content-Type') != 'application/wasm') {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module
            };
        } else {
            return instance;
        }
    }
}
async function init1(input1) {
    if (typeof input1 === 'undefined') {
        input1 = importMeta.url.replace(/\.js$/, '_bg.wasm');
    }
    const imports = {
    };
    input1 = Uint8Array.from(atob("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