// Copyright Isaac Z. Schlueter and Contributors. All rights reserved. ISC license.
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * The semantic version parser.
 *
 * Adapted directly from [semver](https://github.com/npm/node-semver).
 *
 * This module is browser compatible.
 *
 * @module
 */ // Note: this is the semver.org version of the spec that it implements
// Not necessarily the package version of this code.
export const SEMVER_SPEC_VERSION = "2.0.0";
const MAX_LENGTH = 256;
// The actual regexps
const re = [];
const src = [];
let R = 0;
// The following Regular Expressions can be used for tokenizing,
// validating, and parsing SemVer version strings.
// ## Numeric Identifier
// A single `0`, or a non-zero digit followed by zero or more digits.
const NUMERICIDENTIFIER = R++;
src[NUMERICIDENTIFIER] = "0|[1-9]\\d*";
// ## Non-numeric Identifier
// Zero or more digits, followed by a letter or hyphen, and then zero or
// more letters, digits, or hyphens.
const NONNUMERICIDENTIFIER = R++;
src[NONNUMERICIDENTIFIER] = "\\d*[a-zA-Z-][a-zA-Z0-9-]*";
// ## Main Version
// Three dot-separated numeric identifiers.
const MAINVERSION = R++;
const nid = src[NUMERICIDENTIFIER];
src[MAINVERSION] = `(${nid})\\.(${nid})\\.(${nid})`;
// ## Pre-release Version Identifier
// A numeric identifier, or a non-numeric identifier.
const PRERELEASEIDENTIFIER = R++;
src[PRERELEASEIDENTIFIER] = "(?:" + src[NUMERICIDENTIFIER] + "|" + src[NONNUMERICIDENTIFIER] + ")";
// ## Pre-release Version
// Hyphen, followed by one or more dot-separated pre-release version
// identifiers.
const PRERELEASE = R++;
src[PRERELEASE] = "(?:-(" + src[PRERELEASEIDENTIFIER] + "(?:\\." + src[PRERELEASEIDENTIFIER] + ")*))";
// ## Build Metadata Identifier
// Any combination of digits, letters, or hyphens.
const BUILDIDENTIFIER = R++;
src[BUILDIDENTIFIER] = "[0-9A-Za-z-]+";
// ## Build Metadata
// Plus sign, followed by one or more period-separated build metadata
// identifiers.
const BUILD = R++;
src[BUILD] = "(?:\\+(" + src[BUILDIDENTIFIER] + "(?:\\." + src[BUILDIDENTIFIER] + ")*))";
// ## Full Version String
// A main version, followed optionally by a pre-release version and
// build metadata.
// Note that the only major, minor, patch, and pre-release sections of
// the version string are capturing groups.  The build metadata is not a
// capturing group, because it should not ever be used in version
// comparison.
const FULL = R++;
const FULLPLAIN = "v?" + src[MAINVERSION] + src[PRERELEASE] + "?" + src[BUILD] + "?";
src[FULL] = "^" + FULLPLAIN + "$";
const GTLT = R++;
src[GTLT] = "((?:<|>)?=?)";
// Something like "2.*" or "1.2.x".
// Note that "x.x" is a valid xRange identifer, meaning "any version"
// Only the first item is strictly required.
const XRANGEIDENTIFIER = R++;
src[XRANGEIDENTIFIER] = src[NUMERICIDENTIFIER] + "|x|X|\\*";
const XRANGEPLAIN = R++;
src[XRANGEPLAIN] = "[v=\\s]*(" + src[XRANGEIDENTIFIER] + ")" + "(?:\\.(" + src[XRANGEIDENTIFIER] + ")" + "(?:\\.(" + src[XRANGEIDENTIFIER] + ")" + "(?:" + src[PRERELEASE] + ")?" + src[BUILD] + "?" + ")?)?";
const XRANGE = R++;
src[XRANGE] = "^" + src[GTLT] + "\\s*" + src[XRANGEPLAIN] + "$";
// Tilde ranges.
// Meaning is "reasonably at or greater than"
const LONETILDE = R++;
src[LONETILDE] = "(?:~>?)";
const TILDE = R++;
src[TILDE] = "^" + src[LONETILDE] + src[XRANGEPLAIN] + "$";
// Caret ranges.
// Meaning is "at least and backwards compatible with"
const LONECARET = R++;
src[LONECARET] = "(?:\\^)";
const CARET = R++;
src[CARET] = "^" + src[LONECARET] + src[XRANGEPLAIN] + "$";
// A simple gt/lt/eq thing, or just "" to indicate "any version"
const COMPARATOR = R++;
src[COMPARATOR] = "^" + src[GTLT] + "\\s*(" + FULLPLAIN + ")$|^$";
// Something like `1.2.3 - 1.2.4`
const HYPHENRANGE = R++;
src[HYPHENRANGE] = "^\\s*(" + src[XRANGEPLAIN] + ")" + "\\s+-\\s+" + "(" + src[XRANGEPLAIN] + ")" + "\\s*$";
// Star ranges basically just allow anything at all.
const STAR = R++;
src[STAR] = "(<|>)?=?\\s*\\*";
// Compile to actual regexp objects.
// All are flag-free, unless they were created above with a flag.
for(let i = 0; i < R; i++){
    if (!re[i]) {
        re[i] = new RegExp(src[i]);
    }
}
export function parse(version, options) {
    if (typeof options !== "object") {
        options = {
            includePrerelease: false
        };
    }
    if (version instanceof SemVer) {
        return version;
    }
    if (typeof version !== "string") {
        return null;
    }
    if (version.length > MAX_LENGTH) {
        return null;
    }
    const r = re[FULL];
    if (!r.test(version)) {
        return null;
    }
    try {
        return new SemVer(version, options);
    } catch  {
        return null;
    }
}
export function valid(version, options) {
    if (version === null) return null;
    const v = parse(version, options);
    return v ? v.version : null;
}
export class SemVer {
    raw;
    options;
    major;
    minor;
    patch;
    version;
    build;
    prerelease;
    constructor(version, options){
        if (typeof options !== "object") {
            options = {
                includePrerelease: false
            };
        }
        if (version instanceof SemVer) {
            version = version.version;
        } else if (typeof version !== "string") {
            throw new TypeError("Invalid Version: " + version);
        }
        if (version.length > MAX_LENGTH) {
            throw new TypeError("version is longer than " + MAX_LENGTH + " characters");
        }
        if (!(this instanceof SemVer)) {
            return new SemVer(version, options);
        }
        this.options = options;
        const m = version.trim().match(re[FULL]);
        if (!m) {
            throw new TypeError("Invalid Version: " + version);
        }
        this.raw = version;
        // these are actually numbers
        this.major = +m[1];
        this.minor = +m[2];
        this.patch = +m[3];
        if (this.major > Number.MAX_SAFE_INTEGER || this.major < 0) {
            throw new TypeError("Invalid major version");
        }
        if (this.minor > Number.MAX_SAFE_INTEGER || this.minor < 0) {
            throw new TypeError("Invalid minor version");
        }
        if (this.patch > Number.MAX_SAFE_INTEGER || this.patch < 0) {
            throw new TypeError("Invalid patch version");
        }
        // numberify any prerelease numeric ids
        if (!m[4]) {
            this.prerelease = [];
        } else {
            this.prerelease = m[4].split(".").map((id)=>{
                if (/^[0-9]+$/.test(id)) {
                    const num = +id;
                    if (num >= 0 && num < Number.MAX_SAFE_INTEGER) {
                        return num;
                    }
                }
                return id;
            });
        }
        this.build = m[5] ? m[5].split(".") : [];
        this.format();
    }
    format() {
        this.version = this.major + "." + this.minor + "." + this.patch;
        if (this.prerelease.length) {
            this.version += "-" + this.prerelease.join(".");
        }
        return this.version;
    }
    compare(other) {
        if (!(other instanceof SemVer)) {
            other = new SemVer(other, this.options);
        }
        return this.compareMain(other) || this.comparePre(other);
    }
    compareMain(other) {
        if (!(other instanceof SemVer)) {
            other = new SemVer(other, this.options);
        }
        return compareIdentifiers(this.major, other.major) || compareIdentifiers(this.minor, other.minor) || compareIdentifiers(this.patch, other.patch);
    }
    comparePre(other) {
        if (!(other instanceof SemVer)) {
            other = new SemVer(other, this.options);
        }
        // NOT having a prerelease is > having one
        if (this.prerelease.length && !other.prerelease.length) {
            return -1;
        } else if (!this.prerelease.length && other.prerelease.length) {
            return 1;
        } else if (!this.prerelease.length && !other.prerelease.length) {
            return 0;
        }
        let i = 0;
        do {
            const a = this.prerelease[i];
            const b = other.prerelease[i];
            if (a === undefined && b === undefined) {
                return 0;
            } else if (b === undefined) {
                return 1;
            } else if (a === undefined) {
                return -1;
            } else if (a === b) {
                continue;
            } else {
                return compareIdentifiers(a, b);
            }
        }while (++i)
        return 1;
    }
    compareBuild(other) {
        if (!(other instanceof SemVer)) {
            other = new SemVer(other, this.options);
        }
        let i = 0;
        do {
            const a = this.build[i];
            const b = other.build[i];
            if (a === undefined && b === undefined) {
                return 0;
            } else if (b === undefined) {
                return 1;
            } else if (a === undefined) {
                return -1;
            } else if (a === b) {
                continue;
            } else {
                return compareIdentifiers(a, b);
            }
        }while (++i)
        return 1;
    }
    /**
   * @deprecated (will be removed after 0.165.0) use `increment` instead
   */ inc(release, identifier) {
        return this.increment(release, identifier);
    }
    increment(release, identifier) {
        switch(release){
            case "premajor":
                this.prerelease.length = 0;
                this.patch = 0;
                this.minor = 0;
                this.major++;
                this.increment("pre", identifier);
                break;
            case "preminor":
                this.prerelease.length = 0;
                this.patch = 0;
                this.minor++;
                this.increment("pre", identifier);
                break;
            case "prepatch":
                // If this is already a prerelease, it will bump to the next version
                // drop any prereleases that might already exist, since they are not
                // relevant at this point.
                this.prerelease.length = 0;
                this.increment("patch", identifier);
                this.increment("pre", identifier);
                break;
            // If the input is a non-prerelease version, this acts the same as
            // prepatch.
            case "prerelease":
                if (this.prerelease.length === 0) {
                    this.increment("patch", identifier);
                }
                this.increment("pre", identifier);
                break;
            case "major":
                // If this is a pre-major version, bump up to the same major version.
                // Otherwise increment major.
                // 1.0.0-5 bumps to 1.0.0
                // 1.1.0 bumps to 2.0.0
                if (this.minor !== 0 || this.patch !== 0 || this.prerelease.length === 0) {
                    this.major++;
                }
                this.minor = 0;
                this.patch = 0;
                this.prerelease = [];
                break;
            case "minor":
                // If this is a pre-minor version, bump up to the same minor version.
                // Otherwise increment minor.
                // 1.2.0-5 bumps to 1.2.0
                // 1.2.1 bumps to 1.3.0
                if (this.patch !== 0 || this.prerelease.length === 0) {
                    this.minor++;
                }
                this.patch = 0;
                this.prerelease = [];
                break;
            case "patch":
                // If this is not a pre-release version, it will increment the patch.
                // If it is a pre-release it will bump up to the same patch version.
                // 1.2.0-5 patches to 1.2.0
                // 1.2.0 patches to 1.2.1
                if (this.prerelease.length === 0) {
                    this.patch++;
                }
                this.prerelease = [];
                break;
            // This probably shouldn't be used publicly.
            // 1.0.0 "pre" would become 1.0.0-0 which is the wrong direction.
            case "pre":
                if (this.prerelease.length === 0) {
                    this.prerelease = [
                        0
                    ];
                } else {
                    let i = this.prerelease.length;
                    while(--i >= 0){
                        if (typeof this.prerelease[i] === "number") {
                            // deno-fmt-ignore
                            this.prerelease[i]++;
                            i = -2;
                        }
                    }
                    if (i === -1) {
                        // didn't increment anything
                        this.prerelease.push(0);
                    }
                }
                if (identifier) {
                    // 1.2.0-beta.1 bumps to 1.2.0-beta.2,
                    // 1.2.0-beta.fooblz or 1.2.0-beta bumps to 1.2.0-beta.0
                    if (this.prerelease[0] === identifier) {
                        if (isNaN(this.prerelease[1])) {
                            this.prerelease = [
                                identifier,
                                0
                            ];
                        }
                    } else {
                        this.prerelease = [
                            identifier,
                            0
                        ];
                    }
                }
                break;
            default:
                throw new Error("invalid increment argument: " + release);
        }
        this.format();
        this.raw = this.version;
        return this;
    }
    toString() {
        return this.version;
    }
}
/**
 * @deprecated (will be removed after 0.165.0) use `increment` instead
 */ export function inc(version, release, options, identifier) {
    return increment(version, release, options, identifier);
}
/**
 * Return the version incremented by the release type (major, minor, patch, or prerelease), or null if it's not valid.
 */ export function increment(version, release, options, identifier) {
    if (typeof options === "string") {
        identifier = options;
        options = undefined;
    }
    try {
        return new SemVer(version, options).increment(release, identifier).version;
    } catch  {
        return null;
    }
}
/**
 * @deprecated (will be removed after 0.165.0) use `difference` instead
 */ export function diff(version1, version2, options) {
    return difference(version1, version2, options);
}
export function difference(version1, version2, options) {
    if (eq(version1, version2, options)) {
        return null;
    } else {
        const v1 = parse(version1);
        const v2 = parse(version2);
        let prefix = "";
        let defaultResult = null;
        if (v1 && v2) {
            if (v1.prerelease.length || v2.prerelease.length) {
                prefix = "pre";
                defaultResult = "prerelease";
            }
            for(const key in v1){
                if (key === "major" || key === "minor" || key === "patch") {
                    if (v1[key] !== v2[key]) {
                        return prefix + key;
                    }
                }
            }
        }
        return defaultResult; // may be undefined
    }
}
const numeric = /^[0-9]+$/;
export function compareIdentifiers(a, b) {
    const anum = numeric.test(a);
    const bnum = numeric.test(b);
    if (a === null || b === null) throw "Comparison against null invalid";
    if (anum && bnum) {
        a = +a;
        b = +b;
    }
    return a === b ? 0 : anum && !bnum ? -1 : bnum && !anum ? 1 : a < b ? -1 : 1;
}
export function rcompareIdentifiers(a, b) {
    return compareIdentifiers(b, a);
}
/**
 * Return the major version number.
 */ export function major(v, options) {
    return new SemVer(v, options).major;
}
/**
 * Return the minor version number.
 */ export function minor(v, options) {
    return new SemVer(v, options).minor;
}
/**
 * Return the patch version number.
 */ export function patch(v, options) {
    return new SemVer(v, options).patch;
}
export function compare(v1, v2, options) {
    return new SemVer(v1, options).compare(new SemVer(v2, options));
}
export function compareBuild(a, b, options) {
    const versionA = new SemVer(a, options);
    const versionB = new SemVer(b, options);
    return versionA.compare(versionB) || versionA.compareBuild(versionB);
}
export function rcompare(v1, v2, options) {
    return compare(v2, v1, options);
}
export function sort(list, options) {
    return list.sort((a, b)=>{
        return compareBuild(a, b, options);
    });
}
export function rsort(list, options) {
    return list.sort((a, b)=>{
        return compareBuild(b, a, options);
    });
}
export function gt(v1, v2, options) {
    return compare(v1, v2, options) > 0;
}
export function lt(v1, v2, options) {
    return compare(v1, v2, options) < 0;
}
export function eq(v1, v2, options) {
    return compare(v1, v2, options) === 0;
}
export function neq(v1, v2, options) {
    return compare(v1, v2, options) !== 0;
}
export function gte(v1, v2, options) {
    return compare(v1, v2, options) >= 0;
}
export function lte(v1, v2, options) {
    return compare(v1, v2, options) <= 0;
}
export function cmp(v1, operator, v2, options) {
    switch(operator){
        case "===":
            if (typeof v1 === "object") v1 = v1.version;
            if (typeof v2 === "object") v2 = v2.version;
            return v1 === v2;
        case "!==":
            if (typeof v1 === "object") v1 = v1.version;
            if (typeof v2 === "object") v2 = v2.version;
            return v1 !== v2;
        case "":
        case "=":
        case "==":
            return eq(v1, v2, options);
        case "!=":
            return neq(v1, v2, options);
        case ">":
            return gt(v1, v2, options);
        case ">=":
            return gte(v1, v2, options);
        case "<":
            return lt(v1, v2, options);
        case "<=":
            return lte(v1, v2, options);
        default:
            throw new TypeError("Invalid operator: " + operator);
    }
}
const ANY = {};
export class Comparator {
    semver;
    operator;
    value;
    options;
    constructor(comp, options){
        if (typeof options !== "object") {
            options = {
                includePrerelease: false
            };
        }
        if (comp instanceof Comparator) {
            return comp;
        }
        if (!(this instanceof Comparator)) {
            return new Comparator(comp, options);
        }
        this.options = options;
        this.parse(comp);
        if (this.semver === ANY) {
            this.value = "";
        } else {
            this.value = this.operator + this.semver.version;
        }
    }
    parse(comp) {
        const r = re[COMPARATOR];
        const m = comp.match(r);
        if (!m) {
            throw new TypeError("Invalid comparator: " + comp);
        }
        const m1 = m[1];
        this.operator = m1 !== undefined ? m1 : "";
        if (this.operator === "=") {
            this.operator = "";
        }
        // if it literally is just '>' or '' then allow anything.
        if (!m[2]) {
            this.semver = ANY;
        } else {
            this.semver = new SemVer(m[2], this.options);
        }
    }
    test(version) {
        if (this.semver === ANY || version === ANY) {
            return true;
        }
        if (typeof version === "string") {
            version = new SemVer(version, this.options);
        }
        return cmp(version, this.operator, this.semver, this.options);
    }
    intersects(comp, options) {
        if (!(comp instanceof Comparator)) {
            throw new TypeError("a Comparator is required");
        }
        if (typeof options !== "object") {
            options = {
                includePrerelease: false
            };
        }
        let rangeTmp;
        if (this.operator === "") {
            if (this.value === "") {
                return true;
            }
            rangeTmp = new Range(comp.value, options);
            return satisfies(this.value, rangeTmp, options);
        } else if (comp.operator === "") {
            if (comp.value === "") {
                return true;
            }
            rangeTmp = new Range(this.value, options);
            return satisfies(comp.semver, rangeTmp, options);
        }
        const sameDirectionIncreasing = (this.operator === ">=" || this.operator === ">") && (comp.operator === ">=" || comp.operator === ">");
        const sameDirectionDecreasing = (this.operator === "<=" || this.operator === "<") && (comp.operator === "<=" || comp.operator === "<");
        const sameSemVer = this.semver.version === comp.semver.version;
        const differentDirectionsInclusive = (this.operator === ">=" || this.operator === "<=") && (comp.operator === ">=" || comp.operator === "<=");
        const oppositeDirectionsLessThan = cmp(this.semver, "<", comp.semver, options) && (this.operator === ">=" || this.operator === ">") && (comp.operator === "<=" || comp.operator === "<");
        const oppositeDirectionsGreaterThan = cmp(this.semver, ">", comp.semver, options) && (this.operator === "<=" || this.operator === "<") && (comp.operator === ">=" || comp.operator === ">");
        return sameDirectionIncreasing || sameDirectionDecreasing || sameSemVer && differentDirectionsInclusive || oppositeDirectionsLessThan || oppositeDirectionsGreaterThan;
    }
    toString() {
        return this.value;
    }
}
export class Range {
    range;
    raw;
    options;
    includePrerelease;
    set;
    constructor(range, options){
        if (typeof options !== "object") {
            options = {
                includePrerelease: false
            };
        }
        if (range instanceof Range) {
            if (range.includePrerelease === !!options.includePrerelease) {
                return range;
            } else {
                return new Range(range.raw, options);
            }
        }
        if (range instanceof Comparator) {
            return new Range(range.value, options);
        }
        if (!(this instanceof Range)) {
            return new Range(range, options);
        }
        this.options = options;
        this.includePrerelease = !!options.includePrerelease;
        // First, split based on boolean or ||
        this.raw = range;
        this.set = range.split(/\s*\|\|\s*/).map((range)=>this.parseRange(range.trim())).filter((c)=>{
            // throw out any that are not relevant for whatever reason
            return c.length;
        });
        if (!this.set.length) {
            throw new TypeError("Invalid SemVer Range: " + range);
        }
        this.format();
    }
    format() {
        this.range = this.set.map((comps)=>comps.join(" ").trim()).join("||").trim();
        return this.range;
    }
    parseRange(range) {
        range = range.trim();
        // `1.2.3 - 1.2.4` => `>=1.2.3 <=1.2.4`
        const hr = re[HYPHENRANGE];
        range = range.replace(hr, hyphenReplace);
        // normalize spaces
        range = range.split(/\s+/).join(" ");
        // At this point, the range is completely trimmed and
        // ready to be split into comparators.
        const set = range.split(" ").map((comp)=>parseComparator(comp, this.options)).join(" ").split(/\s+/);
        return set.map((comp)=>new Comparator(comp, this.options));
    }
    test(version) {
        if (typeof version === "string") {
            version = new SemVer(version, this.options);
        }
        for(let i = 0; i < this.set.length; i++){
            if (testSet(this.set[i], version, this.options)) {
                return true;
            }
        }
        return false;
    }
    intersects(range, options) {
        if (!(range instanceof Range)) {
            throw new TypeError("a Range is required");
        }
        return this.set.some((thisComparators)=>{
            return isSatisfiable(thisComparators, options) && range.set.some((rangeComparators)=>{
                return isSatisfiable(rangeComparators, options) && thisComparators.every((thisComparator)=>{
                    return rangeComparators.every((rangeComparator)=>{
                        return thisComparator.intersects(rangeComparator, options);
                    });
                });
            });
        });
    }
    toString() {
        return this.range;
    }
}
function testSet(set, version, options) {
    for(let i = 0; i < set.length; i++){
        if (!set[i].test(version)) {
            return false;
        }
    }
    if (version.prerelease.length && !options.includePrerelease) {
        // Find the set of versions that are allowed to have prereleases
        // For example, ^1.2.3-pr.1 desugars to >=1.2.3-pr.1 <2.0.0
        // That should allow `1.2.3-pr.2` to pass.
        // However, `1.2.4-alpha.notready` should NOT be allowed,
        // even though it's within the range set by the comparators.
        for(let i1 = 0; i1 < set.length; i1++){
            if (set[i1].semver === ANY) {
                continue;
            }
            if (set[i1].semver.prerelease.length > 0) {
                const allowed = set[i1].semver;
                if (allowed.major === version.major && allowed.minor === version.minor && allowed.patch === version.patch) {
                    return true;
                }
            }
        }
        // Version has a -pre, but it's not one of the ones we like.
        return false;
    }
    return true;
}
// take a set of comparators and determine whether there
// exists a version which can satisfy it
function isSatisfiable(comparators, options) {
    let result = true;
    const remainingComparators = comparators.slice();
    let testComparator = remainingComparators.pop();
    while(result && remainingComparators.length){
        result = remainingComparators.every((otherComparator)=>{
            return testComparator?.intersects(otherComparator, options);
        });
        testComparator = remainingComparators.pop();
    }
    return result;
}
// Mostly just for testing and legacy API reasons
export function toComparators(range, options) {
    return new Range(range, options).set.map((comp)=>{
        return comp.map((c)=>c.value).join(" ").trim().split(" ");
    });
}
// comprised of xranges, tildes, stars, and gtlt's at this point.
// already replaced the hyphen ranges
// turn into a set of JUST comparators.
function parseComparator(comp, options) {
    comp = replaceCarets(comp, options);
    comp = replaceTildes(comp, options);
    comp = replaceXRanges(comp, options);
    comp = replaceStars(comp, options);
    return comp;
}
function isX(id) {
    return !id || id.toLowerCase() === "x" || id === "*";
}
// ~, ~> --> * (any, kinda silly)
// ~2, ~2.x, ~2.x.x, ~>2, ~>2.x ~>2.x.x --> >=2.0.0 <3.0.0
// ~2.0, ~2.0.x, ~>2.0, ~>2.0.x --> >=2.0.0 <2.1.0
// ~1.2, ~1.2.x, ~>1.2, ~>1.2.x --> >=1.2.0 <1.3.0
// ~1.2.3, ~>1.2.3 --> >=1.2.3 <1.3.0
// ~1.2.0, ~>1.2.0 --> >=1.2.0 <1.3.0
function replaceTildes(comp, options) {
    return comp.trim().split(/\s+/).map((comp)=>replaceTilde(comp, options)).join(" ");
}
function replaceTilde(comp, _options) {
    const r = re[TILDE];
    return comp.replace(r, (_, M, m, p, pr)=>{
        let ret;
        if (isX(M)) {
            ret = "";
        } else if (isX(m)) {
            ret = ">=" + M + ".0.0 <" + (+M + 1) + ".0.0";
        } else if (isX(p)) {
            // ~1.2 == >=1.2.0 <1.3.0
            ret = ">=" + M + "." + m + ".0 <" + M + "." + (+m + 1) + ".0";
        } else if (pr) {
            ret = ">=" + M + "." + m + "." + p + "-" + pr + " <" + M + "." + (+m + 1) + ".0";
        } else {
            // ~1.2.3 == >=1.2.3 <1.3.0
            ret = ">=" + M + "." + m + "." + p + " <" + M + "." + (+m + 1) + ".0";
        }
        return ret;
    });
}
// ^ --> * (any, kinda silly)
// ^2, ^2.x, ^2.x.x --> >=2.0.0 <3.0.0
// ^2.0, ^2.0.x --> >=2.0.0 <3.0.0
// ^1.2, ^1.2.x --> >=1.2.0 <2.0.0
// ^1.2.3 --> >=1.2.3 <2.0.0
// ^1.2.0 --> >=1.2.0 <2.0.0
function replaceCarets(comp, options) {
    return comp.trim().split(/\s+/).map((comp)=>replaceCaret(comp, options)).join(" ");
}
function replaceCaret(comp, _options) {
    const r = re[CARET];
    return comp.replace(r, (_, M, m, p, pr)=>{
        let ret;
        if (isX(M)) {
            ret = "";
        } else if (isX(m)) {
            ret = ">=" + M + ".0.0 <" + (+M + 1) + ".0.0";
        } else if (isX(p)) {
            if (M === "0") {
                ret = ">=" + M + "." + m + ".0 <" + M + "." + (+m + 1) + ".0";
            } else {
                ret = ">=" + M + "." + m + ".0 <" + (+M + 1) + ".0.0";
            }
        } else if (pr) {
            if (M === "0") {
                if (m === "0") {
                    ret = ">=" + M + "." + m + "." + p + "-" + pr + " <" + M + "." + m + "." + (+p + 1);
                } else {
                    ret = ">=" + M + "." + m + "." + p + "-" + pr + " <" + M + "." + (+m + 1) + ".0";
                }
            } else {
                ret = ">=" + M + "." + m + "." + p + "-" + pr + " <" + (+M + 1) + ".0.0";
            }
        } else {
            if (M === "0") {
                if (m === "0") {
                    ret = ">=" + M + "." + m + "." + p + " <" + M + "." + m + "." + (+p + 1);
                } else {
                    ret = ">=" + M + "." + m + "." + p + " <" + M + "." + (+m + 1) + ".0";
                }
            } else {
                ret = ">=" + M + "." + m + "." + p + " <" + (+M + 1) + ".0.0";
            }
        }
        return ret;
    });
}
function replaceXRanges(comp, options) {
    return comp.split(/\s+/).map((comp)=>replaceXRange(comp, options)).join(" ");
}
function replaceXRange(comp, _options) {
    comp = comp.trim();
    const r = re[XRANGE];
    return comp.replace(r, (ret, gtlt, M, m, p, _pr)=>{
        const xM = isX(M);
        const xm = xM || isX(m);
        const xp = xm || isX(p);
        const anyX = xp;
        if (gtlt === "=" && anyX) {
            gtlt = "";
        }
        if (xM) {
            if (gtlt === ">" || gtlt === "<") {
                // nothing is allowed
                ret = "<0.0.0";
            } else {
                // nothing is forbidden
                ret = "*";
            }
        } else if (gtlt && anyX) {
            // we know patch is an x, because we have any x at all.
            // replace X with 0
            if (xm) {
                m = 0;
            }
            p = 0;
            if (gtlt === ">") {
                // >1 => >=2.0.0
                // >1.2 => >=1.3.0
                // >1.2.3 => >= 1.2.4
                gtlt = ">=";
                if (xm) {
                    M = +M + 1;
                    m = 0;
                    p = 0;
                } else {
                    m = +m + 1;
                    p = 0;
                }
            } else if (gtlt === "<=") {
                // <=0.7.x is actually <0.8.0, since any 0.7.x should
                // pass.  Similarly, <=7.x is actually <8.0.0, etc.
                gtlt = "<";
                if (xm) {
                    M = +M + 1;
                } else {
                    m = +m + 1;
                }
            }
            ret = gtlt + M + "." + m + "." + p;
        } else if (xm) {
            ret = ">=" + M + ".0.0 <" + (+M + 1) + ".0.0";
        } else if (xp) {
            ret = ">=" + M + "." + m + ".0 <" + M + "." + (+m + 1) + ".0";
        }
        return ret;
    });
}
// Because * is AND-ed with everything else in the comparator,
// and '' means "any version", just remove the *s entirely.
function replaceStars(comp, _options) {
    return comp.trim().replace(re[STAR], "");
}
// This function is passed to string.replace(re[HYPHENRANGE])
// M, m, patch, prerelease, build
// 1.2 - 3.4.5 => >=1.2.0 <=3.4.5
// 1.2.3 - 3.4 => >=1.2.0 <3.5.0 Any 3.4.x will do
// 1.2 - 3.4 => >=1.2.0 <3.5.0
function hyphenReplace(_$0, from, fM, fm, fp, _fpr, _fb, to, tM, tm, tp, tpr, _tb) {
    if (isX(fM)) {
        from = "";
    } else if (isX(fm)) {
        from = ">=" + fM + ".0.0";
    } else if (isX(fp)) {
        from = ">=" + fM + "." + fm + ".0";
    } else {
        from = ">=" + from;
    }
    if (isX(tM)) {
        to = "";
    } else if (isX(tm)) {
        to = "<" + (+tM + 1) + ".0.0";
    } else if (isX(tp)) {
        to = "<" + tM + "." + (+tm + 1) + ".0";
    } else if (tpr) {
        to = "<=" + tM + "." + tm + "." + tp + "-" + tpr;
    } else {
        to = "<=" + to;
    }
    return (from + " " + to).trim();
}
export function satisfies(version, range, options) {
    try {
        range = new Range(range, options);
    } catch  {
        return false;
    }
    return range.test(version);
}
export function maxSatisfying(versions, range, options) {
    //todo
    let max = null;
    let maxSV = null;
    let rangeObj;
    try {
        rangeObj = new Range(range, options);
    } catch  {
        return null;
    }
    versions.forEach((v)=>{
        if (rangeObj.test(v)) {
            // satisfies(v, range, options)
            if (!max || maxSV && maxSV.compare(v) === -1) {
                // compare(max, v, true)
                max = v;
                maxSV = new SemVer(max, options);
            }
        }
    });
    return max;
}
export function minSatisfying(versions, range, options) {
    //todo
    let min = null;
    let minSV = null;
    let rangeObj;
    try {
        rangeObj = new Range(range, options);
    } catch  {
        return null;
    }
    versions.forEach((v)=>{
        if (rangeObj.test(v)) {
            // satisfies(v, range, options)
            if (!min || minSV.compare(v) === 1) {
                // compare(min, v, true)
                min = v;
                minSV = new SemVer(min, options);
            }
        }
    });
    return min;
}
export function minVersion(range, options) {
    range = new Range(range, options);
    let minver = new SemVer("0.0.0");
    if (range.test(minver)) {
        return minver;
    }
    minver = new SemVer("0.0.0-0");
    if (range.test(minver)) {
        return minver;
    }
    minver = null;
    for(let i = 0; i < range.set.length; ++i){
        const comparators = range.set[i];
        comparators.forEach((comparator)=>{
            // Clone to avoid manipulating the comparator's semver object.
            const compver = new SemVer(comparator.semver.version);
            switch(comparator.operator){
                case ">":
                    if (compver.prerelease.length === 0) {
                        compver.patch++;
                    } else {
                        compver.prerelease.push(0);
                    }
                    compver.raw = compver.format();
                /* fallthrough */ case "":
                case ">=":
                    if (!minver || gt(minver, compver)) {
                        minver = compver;
                    }
                    break;
                case "<":
                case "<=":
                    break;
                /* istanbul ignore next */ default:
                    throw new Error("Unexpected operation: " + comparator.operator);
            }
        });
    }
    if (minver && range.test(minver)) {
        return minver;
    }
    return null;
}
export function validRange(range, options) {
    try {
        if (range === null) return null;
        // Return '*' instead of '' so that truthiness works.
        // This will throw if it's invalid anyway
        return new Range(range, options).range || "*";
    } catch  {
        return null;
    }
}
/**
 * Return true if version is less than all the versions possible in the range.
 */ export function ltr(version, range, options) {
    return outside(version, range, "<", options);
}
/**
 * Return true if version is greater than all the versions possible in the range.
 */ export function gtr(version, range, options) {
    return outside(version, range, ">", options);
}
/**
 * Return true if the version is outside the bounds of the range in either the high or low direction.
 * The hilo argument must be either the string '>' or '<'. (This is the function called by gtr and ltr.)
 */ export function outside(version, range, hilo, options) {
    version = new SemVer(version, options);
    range = new Range(range, options);
    let gtfn;
    let ltefn;
    let ltfn;
    let comp;
    let ecomp;
    switch(hilo){
        case ">":
            gtfn = gt;
            ltefn = lte;
            ltfn = lt;
            comp = ">";
            ecomp = ">=";
            break;
        case "<":
            gtfn = lt;
            ltefn = gte;
            ltfn = gt;
            comp = "<";
            ecomp = "<=";
            break;
        default:
            throw new TypeError('Must provide a hilo val of "<" or ">"');
    }
    // If it satisifes the range it is not outside
    if (satisfies(version, range, options)) {
        return false;
    }
    // From now on, variable terms are as if we're in "gtr" mode.
    // but note that everything is flipped for the "ltr" function.
    for(let i = 0; i < range.set.length; ++i){
        const comparators = range.set[i];
        let high = null;
        let low = null;
        for (let comparator of comparators){
            if (comparator.semver === ANY) {
                comparator = new Comparator(">=0.0.0");
            }
            high = high || comparator;
            low = low || comparator;
            if (gtfn(comparator.semver, high.semver, options)) {
                high = comparator;
            } else if (ltfn(comparator.semver, low.semver, options)) {
                low = comparator;
            }
        }
        if (high === null || low === null) return true;
        // If the edge version comparator has a operator then our version
        // isn't outside it
        if (high.operator === comp || high.operator === ecomp) {
            return false;
        }
        // If the lowest version comparator has an operator and our version
        // is less than it then it isn't higher than the range
        if ((!low.operator || low.operator === comp) && ltefn(version, low.semver)) {
            return false;
        } else if (low.operator === ecomp && ltfn(version, low.semver)) {
            return false;
        }
    }
    return true;
}
export function prerelease(version, options) {
    const parsed = parse(version, options);
    return parsed && parsed.prerelease.length ? parsed.prerelease : null;
}
/**
 * Return true if any of the ranges comparators intersect
 */ export function intersects(range1, range2, options) {
    range1 = new Range(range1, options);
    range2 = new Range(range2, options);
    return range1.intersects(range2);
}
export default SemVer;
//# sourceMappingURL=data:application/json;base64,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