// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/** {@linkcode parse} and {@linkcode stringify} for handling
 * [TOML](https://toml.io/en/) encoded data.
 *
 * This module is browser compatible.
 *
 * @module
 */ export { parse } from "./_toml/parser.ts";
// Bare keys may only contain ASCII letters,
// ASCII digits, underscores, and dashes (A-Za-z0-9_-).
function joinKeys(keys) {
    // Dotted keys are a sequence of bare or quoted keys joined with a dot.
    // This allows for grouping similar properties together:
    return keys.map((str)=>{
        return str.match(/[^A-Za-z0-9_-]/) ? JSON.stringify(str) : str;
    }).join(".");
}
var ArrayType;
(function(ArrayType) {
    ArrayType[ArrayType["ONLY_PRIMITIVE"] = 0] = "ONLY_PRIMITIVE";
    ArrayType[ArrayType["ONLY_OBJECT_EXCLUDING_ARRAY"] = 1] = "ONLY_OBJECT_EXCLUDING_ARRAY";
    ArrayType[ArrayType["MIXED"] = 2] = "MIXED";
})(ArrayType || (ArrayType = {}));
class Dumper {
    maxPad = 0;
    srcObject;
    output = [];
    #arrayTypeCache = new Map();
    constructor(srcObjc){
        this.srcObject = srcObjc;
    }
    dump(fmtOptions = {}) {
        // deno-lint-ignore no-explicit-any
        this.output = this.#printObject(this.srcObject);
        this.output = this.#format(fmtOptions);
        return this.output;
    }
    #printObject(obj, keys = []) {
        const out = [];
        const props = Object.keys(obj);
        const inlineProps = [];
        const multilineProps = [];
        for (const prop of props){
            if (this.#isSimplySerializable(obj[prop])) {
                inlineProps.push(prop);
            } else {
                multilineProps.push(prop);
            }
        }
        const sortedProps = inlineProps.concat(multilineProps);
        for(let i = 0; i < sortedProps.length; i++){
            const prop1 = sortedProps[i];
            const value = obj[prop1];
            if (value instanceof Date) {
                out.push(this.#dateDeclaration([
                    prop1
                ], value));
            } else if (typeof value === "string" || value instanceof RegExp) {
                out.push(this.#strDeclaration([
                    prop1
                ], value.toString()));
            } else if (typeof value === "number") {
                out.push(this.#numberDeclaration([
                    prop1
                ], value));
            } else if (typeof value === "boolean") {
                out.push(this.#boolDeclaration([
                    prop1
                ], value));
            } else if (value instanceof Array) {
                const arrayType = this.#getTypeOfArray(value);
                if (arrayType === ArrayType.ONLY_PRIMITIVE) {
                    out.push(this.#arrayDeclaration([
                        prop1
                    ], value));
                } else if (arrayType === ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY) {
                    // array of objects
                    for(let i1 = 0; i1 < value.length; i1++){
                        out.push("");
                        out.push(this.#headerGroup([
                            ...keys,
                            prop1
                        ]));
                        out.push(...this.#printObject(value[i1], [
                            ...keys,
                            prop1
                        ]));
                    }
                } else {
                    // this is a complex array, use the inline format.
                    const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
                    out.push(`${this.#declaration([
                        prop1
                    ])}[${str}]`);
                }
            } else if (typeof value === "object") {
                out.push("");
                out.push(this.#header([
                    ...keys,
                    prop1
                ]));
                if (value) {
                    const toParse = value;
                    out.push(...this.#printObject(toParse, [
                        ...keys,
                        prop1
                    ]));
                }
            // out.push(...this._parse(value, `${path}${prop}.`));
            }
        }
        out.push("");
        return out;
    }
    #isPrimitive(value1) {
        return value1 instanceof Date || value1 instanceof RegExp || [
            "string",
            "number",
            "boolean"
        ].includes(typeof value1);
    }
    #getTypeOfArray(arr) {
        if (this.#arrayTypeCache.has(arr)) {
            return this.#arrayTypeCache.get(arr);
        }
        const type = this.#doGetTypeOfArray(arr);
        this.#arrayTypeCache.set(arr, type);
        return type;
    }
    #doGetTypeOfArray(arr1) {
        if (!arr1.length) {
            // any type should be fine
            return ArrayType.ONLY_PRIMITIVE;
        }
        const onlyPrimitive = this.#isPrimitive(arr1[0]);
        if (arr1[0] instanceof Array) {
            return ArrayType.MIXED;
        }
        for(let i2 = 1; i2 < arr1.length; i2++){
            if (onlyPrimitive !== this.#isPrimitive(arr1[i2]) || arr1[i2] instanceof Array) {
                return ArrayType.MIXED;
            }
        }
        return onlyPrimitive ? ArrayType.ONLY_PRIMITIVE : ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY;
    }
    #printAsInlineValue(value2) {
        if (value2 instanceof Date) {
            return `"${this.#printDate(value2)}"`;
        } else if (typeof value2 === "string" || value2 instanceof RegExp) {
            return JSON.stringify(value2.toString());
        } else if (typeof value2 === "number") {
            return value2;
        } else if (typeof value2 === "boolean") {
            return value2.toString();
        } else if (value2 instanceof Array) {
            const str1 = value2.map((x)=>this.#printAsInlineValue(x)).join(",");
            return `[${str1}]`;
        } else if (typeof value2 === "object") {
            if (!value2) {
                throw new Error("should never reach");
            }
            const str2 = Object.keys(value2).map((key)=>{
                // deno-lint-ignore no-explicit-any
                return `${key} = ${this.#printAsInlineValue(value2[key])}`;
            }).join(",");
            return `{${str2}}`;
        }
        throw new Error("should never reach");
    }
    #isSimplySerializable(value3) {
        return typeof value3 === "string" || typeof value3 === "number" || typeof value3 === "boolean" || value3 instanceof RegExp || value3 instanceof Date || value3 instanceof Array && this.#getTypeOfArray(value3) !== ArrayType.ONLY_OBJECT_EXCLUDING_ARRAY;
    }
    #header(keys1) {
        return `[${joinKeys(keys1)}]`;
    }
    #headerGroup(keys2) {
        return `[[${joinKeys(keys2)}]]`;
    }
    #declaration(keys3) {
        const title = joinKeys(keys3);
        if (title.length > this.maxPad) {
            this.maxPad = title.length;
        }
        return `${title} = `;
    }
    #arrayDeclaration(keys4, value4) {
        return `${this.#declaration(keys4)}${JSON.stringify(value4)}`;
    }
    #strDeclaration(keys5, value5) {
        return `${this.#declaration(keys5)}${JSON.stringify(value5)}`;
    }
    #numberDeclaration(keys6, value6) {
        switch(value6){
            case Infinity:
                return `${this.#declaration(keys6)}inf`;
            case -Infinity:
                return `${this.#declaration(keys6)}-inf`;
            default:
                return `${this.#declaration(keys6)}${value6}`;
        }
    }
    #boolDeclaration(keys7, value7) {
        return `${this.#declaration(keys7)}${value7}`;
    }
    #printDate(value8) {
        function dtPad(v, lPad = 2) {
            return v.padStart(lPad, "0");
        }
        const m = dtPad((value8.getUTCMonth() + 1).toString());
        const d = dtPad(value8.getUTCDate().toString());
        const h = dtPad(value8.getUTCHours().toString());
        const min = dtPad(value8.getUTCMinutes().toString());
        const s = dtPad(value8.getUTCSeconds().toString());
        const ms = dtPad(value8.getUTCMilliseconds().toString(), 3);
        // formatted date
        const fData = `${value8.getUTCFullYear()}-${m}-${d}T${h}:${min}:${s}.${ms}`;
        return fData;
    }
    #dateDeclaration(keys8, value9) {
        return `${this.#declaration(keys8)}${this.#printDate(value9)}`;
    }
    #format(options = {}) {
        const { keyAlignment =false  } = options;
        const rDeclaration = /^(\".*\"|[^=]*)\s=/;
        const out1 = [];
        for(let i3 = 0; i3 < this.output.length; i3++){
            const l = this.output[i3];
            // we keep empty entry for array of objects
            if (l[0] === "[" && l[1] !== "[") {
                // empty object
                if (this.output[i3 + 1] === "") {
                    i3 += 1;
                    continue;
                }
                out1.push(l);
            } else {
                if (keyAlignment) {
                    const m1 = rDeclaration.exec(l);
                    if (m1) {
                        out1.push(l.replace(m1[1], m1[1].padEnd(this.maxPad)));
                    } else {
                        out1.push(l);
                    }
                } else {
                    out1.push(l);
                }
            }
        }
        // Cleaning multiple spaces
        const cleanedOutput = [];
        for(let i4 = 0; i4 < out1.length; i4++){
            const l1 = out1[i4];
            if (!(l1 === "" && out1[i4 + 1] === "")) {
                cleanedOutput.push(l1);
            }
        }
        return cleanedOutput;
    }
}
/**
 * Stringify dumps source object into TOML string and returns it.
 * @param srcObj
 * @param [fmtOptions] format options
 * @param [fmtOptions.keyAlignment] whether to algin key
 */ export function stringify(srcObj, fmtOptions) {
    return new Dumper(srcObj).dump(fmtOptions).join("\n");
}
//# sourceMappingURL=data:application/json;base64,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