package mods.immibis.redlogic.chips.ingame;

import mods.immibis.redlogic.Utils;
import mods.immibis.redlogic.api.wiring.*;
import mods.immibis.redlogic.chips.generated.CCOFactory;
import mods.immibis.redlogic.chips.generated.CompiledCircuitObject;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraftforge.common.ForgeDirection;

public class TileCustomCircuit extends TileEntity implements IRedstoneUpdatable, IRedstoneEmitter, IBundledUpdatable, IBundledEmitter, IConnectable {

	private String className = "";
	private CompiledCircuitObject circuit;
	private boolean failedCreatingCircuit;
	private byte[] serializedCircuit;
	
	private void createCircuitObject() {
		if(CCOFactory.instance == null)
			return;
		
		if(circuit == null && !failedCreatingCircuit) {
			try {
				if(serializedCircuit != null) {
					circuit = CCOFactory.deserialize(serializedCircuit);
					serializedCircuit = null;
					if(circuit._inputs == null || circuit._outputs == null)
						circuit = null;
				} else
					circuit = CCOFactory.instance.createObject(className);
			} catch(Throwable e) {
				new Exception("Failed to create circuit object at "+xCoord+","+yCoord+","+zCoord+" unknown dimension", e).printStackTrace();
				failedCreatingCircuit = true;
			}
		}
	}
	
	@Override
	public void updateEntity() {
		if(worldObj.isRemote) return;
		if(circuit == null)
			createCircuitObject();
		
		updateQueued = true;
		if(updateQueued && circuit != null) {
			updateQueued = false;
			
			try {
				circuit.update();
			} catch(Error e) {
				failedCreatingCircuit = true;
				circuit = null;
				e.printStackTrace();
				return;
			}
			
			notifyExtendedNeighbours();
			
			for(int k = 0; k < 6; k++) {
				ForgeDirection dir = ForgeDirection.VALID_DIRECTIONS[k];
				TileEntity te = worldObj.getBlockTileEntity(xCoord+dir.offsetX, yCoord+dir.offsetY, zCoord+dir.offsetZ);
				switch(circuit._outputs[k].length) {
				case 1:
					if(te instanceof IRedstoneUpdatable)
						((IRedstoneUpdatable)te).onRedstoneInputChanged();
					break;
				case 16:
					if(te instanceof IBundledUpdatable)
						((IBundledUpdatable)te).onBundledInputChanged();
					break;
				}
			}
		}
	}
	
	private boolean updateQueued = false;
	
	public void init(String className) {
		this.className = className;
		createCircuitObject();
		
		
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		onRedstoneInputChanged();
	}
	
	private short encodeBits(boolean[] a) {
		short rv = 0;
		for(int k = 0; k < a.length; k++)
			if(a[k])
				rv |= (short)(1 << k);
		return rv;
	}
	private void decodeBits(boolean[] a, short enc) {
		for(int k = 0; k < a.length; k++)
			a[k] = (enc & (1 << k)) != 0;
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setString("className", className);
		
		if(circuit != null) {
			tag.setByteArray("serialized", CCOFactory.serialize(circuit));
			
		} else if(serializedCircuit != null)
			// circuit failed to deserialize, so we keep the serialized bytes around
			tag.setByteArray("serialized", serializedCircuit);
		
		tag.setBoolean("uq", updateQueued);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		
		className = tag.getString("className");
		if(tag.hasKey("serialized"))
			serializedCircuit = tag.getByteArray("serialized");

		updateQueued = tag.getBoolean("uq");
	}

	@Override
	public short getEmittedSignalStrength(int blockFace, int toDirection) {
		if(circuit == null)
			return 0;
		if(circuit._outputs[toDirection].length != 1)
			return 0;
		return circuit._outputs[toDirection][0] ? (short)255 : 0;
	}
	
	@Override
	public byte[] getBundledCableStrength(int blockFace, int toDirection) {
		if(circuit == null)
			return null;
		
		boolean[] outputs = circuit._outputs[toDirection];
		if(outputs.length != 16)
			return null;
		
		byte[] b = new byte[16];
		for(int k = 0; k < 16; k++)
			b[k] = outputs[k] ? (byte)255 : 0;
		return b;
	}
	
	@Override
	public boolean connects(IWire wire, int blockFace, int fromDirection) {
		if(circuit == null) return false;
		int numIn = circuit._inputs[fromDirection].length;
		int numOut = circuit._outputs[fromDirection].length;
		
		if(wire instanceof IRedstoneWire)
			return numIn == 1 || numOut == 1;
		if(wire instanceof IBundledWire)
			return numIn == 16 || numOut == 16;
		return false;
	}
	
	@Override
	public boolean connectsAroundCorner(IWire wire, int blockFace, int fromDirection) {
		return false;
	}

	@Override
	public void onRedstoneInputChanged() {
		if(circuit == null) return;
		if(updateQueued) return;
		
		boolean anyChanged = false;
		
		for(int k = 0; k < 6; k++) {
			ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[k];
			if(circuit._inputs[k].length == 1) {
				boolean _new = Utils.getPowerStrength(worldObj, xCoord+fd.offsetX, yCoord+fd.offsetY, zCoord+fd.offsetZ, k^1, -1) > 0;
				
				if(!_new)
					for(int i = 0; i < 6; i++)
						if((i&6) != (k&6)) {
							_new = Utils.getPowerStrength(worldObj, xCoord+fd.offsetX, yCoord+fd.offsetY, zCoord+fd.offsetZ, k^1, i) > 0;
							if(_new)
								break;
						}
				
				if(circuit._inputs[k][0] != _new) {
					circuit._inputs[k][0] = _new;
					anyChanged = true;
				}
				
			} else if(circuit._inputs[k].length == 16) {
				TileEntity te = worldObj.getBlockTileEntity(xCoord+fd.offsetX, yCoord+fd.offsetY, zCoord+fd.offsetZ);
				if(te instanceof IBundledEmitter) {
					byte[] str = ((IBundledEmitter)te).getBundledCableStrength(-1, k^1);
					if(str == null)
						for(int i = 0; i < 6; i++)
							if((i&6) != (k&6)) {
								str = ((IBundledEmitter)te).getBundledCableStrength(i, k^1);
								if(str != null)
									break;
							}
					
					boolean[] inputs = circuit._inputs[k];
					if(str == null) {
						for(int i = 0; i < 16; i++) {
							if(inputs[i]) {
								inputs[i] = false;
								anyChanged = true;
							}
						}
						
					} else {	
						for(int i = 0; i < 16; i++) {
							if(inputs[i] != (str[i] != 0)) {
								inputs[i] = (str[i] != 0);
								anyChanged = true;
							}
						}
					}
				}
			}
		}
		
		updateQueued = anyChanged;
	}
	
	@Override
	public void onBundledInputChanged() {
		onRedstoneInputChanged();
	}
	
	private void notifyExtendedNeighbours() {
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord-1, yCoord, zCoord, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord+1, yCoord, zCoord, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord-1, zCoord, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord+1, zCoord, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord-1, getBlockType().blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord+1, getBlockType().blockID);
	}

	public String getClassName() {
		return className;
	}

}
