<?php

/**

 * Commonly used functions.

 */



/**

 * Includes the template file from "views" directory if found. 

 * 

 * @param string $template_name Template name ( exluding extension and "-template" postfix )

 * @param array $data Associative array of data that would be used in the template.

 * @return boolean False when template was not included.

 */

function wp_newswire_view( $template_name, $data = null ){

    $file_name = WP_NEWSWIRE_VIEWS_DIR . $template_name . '-template.php';

    if( ! is_file( $file_name ) ){

        return false;

    }

    

    if( $data !== null ){

        extract( (array) $data );

    }

    

    include $file_name;

}



/**

 * Use this function when checking nonce after form submission.

 * 

 * @param string $nonce_name Nonce name.

 * @param string $action_name Action name.

 * @see http://codex.wordpress.org/Function_Reference/wp_nonce_field

 */

function wp_newswire_check_nonce( $nonce_name, $action_name ){

    if ( 

        ! isset( $_POST[$nonce_name] ) 

        || ! wp_verify_nonce( $_POST[$nonce_name], $action_name ) 

    ) {

       print __( 'Sorry, your nonce did not verify.', 'wp_newswire' );

       exit;

    }

}



/**

 * Displays error / update messages wherever used.

 * 

 * @global object $wp_newswire Global $wp_newswire object.

 */

function wp_newswire_messages(){

    global $wp_newswire;

    if(  empty( $wp_newswire->messages ) ){

        return;

    } // no messages to display

    

    foreach( ( array) $wp_newswire->messages as $state => $messages ){

        $state = esc_attr($state);



        if( empty ( $messages ) ){

            continue;

        } // no messages for this state

        

        foreach ( (array) $messages as $message) {

            $html .= "<div id='message' class='$state'>" .

                        "<p>$message</p>" .

                    "</div>";

        }

    }

    

    echo $html;

}



/**

 * Adds a message into message stack for later display.

 * 

 * @global object $wp_newswire Global $wp_newswire object.

 * @param string $message Message content. HTML allowed.

 * @param string $state Message state error | update .

 */

function wp_newswire_add_message($message, $state = 'updated') { 

    global $wp_newswire;

    $wp_newswire->add_message($state, $message);

}



/**

 * Checks if the given content contains specified keywords. 

 * Keyword relation is AND .

 * 

 * @param string $keywords Keywords separated by comma.

 * @param string $content HTML content.

 * @return boolean True on success, false on failure.

 */

function wp_newswire_has_keywords( $keywords, $content ) {

    $keywords = explode(',', $keywords);

    

    foreach( (array) $keywords as $keyword ){

        $keyword = trim($keyword);

        if( $keyword === '' ){

            continue;

        } // empty keyword, skip

        

        if( stripos( $content, $keyword ) === false ){

            return false;

        } // keyword was not found

    }

    

    return true; // all keywords were found

}



/**

 * Trims the text with given number of sentences. The sentences are perceived as 

 * tokens separated with a dot sign. HTML tags are stripped.

 * 

 * @param string $text Text.

 * @param int $number Number of sentences.

 * @return string Trimmed text.

 */

function wp_newswire_trim_sentences( $text, $number ) {

    $text = wp_strip_all_tags( $text, true );

    $positions = array();

    

    preg_match_all( "/[.]+/", $text, $positions, PREG_OFFSET_CAPTURE ); // find dot positions

    if( empty( $positions ) ){

        return $text;

    } // the text contains less than 1 sentence

    

    $positions = $positions[0];

    if( count( $positions ) <= $number ){

        return $text;

    } // the text contains less sentences than needed to trim

    

    $last_position = $positions[$number-1][1];

    $text = substr( $text, 0, $last_position + 1 );

    return $text;

}



/**

 * Puts content pieces together.

 * 

 * @param string $content Original content.

 * @param int $length Number of sentences to trim the content with.

 * @param string $html_before HTML to insert before the content.

 * @param string $html_after HTML to insert after the content.

 * @param string $source URL to content source.

 * @return string Final HTML content.

 */

function wp_newswire_feed_item_content_html( $content, $length = false, $html_before, $html_after, $source = false ){

    if( $length ){

        $content = wp_newswire_trim_sentences( $content, $length );

    }

    $content_html = $html_before . $content;

    if( $source ){

        $source = '<a href="' . esc_attr( esc_url( $source ) ) . '" target="blank" title="' 

                . esc_attr( $title ) . '">' . esc_html( $source ) . '</a>';

        $content_html .= '<p class="wp-newswire-source">' . __( 'Source: ', 'wp_newswire' ) . $source . '</p>';

    }

    $content_html .= $html_after;

    

    return $content_html;

}



/**

 * Searches content for images and returns image source for the found image.

 * 

 * @param string $text Content HTML.

 * @return boolean | string Image source on success, false on failure.

 */

function wp_newswire_find_first_image( $text ) {

    $text = (string) $text;

    if( $text === '' ){

        return false;

    } // empty text

    

    $doc = new DOMDocument();

    if ( ! @$doc->loadHTML( $text ) ) {

        wp_newswire_add_message( __( 'Could not load HTML.', 'wp_newswire' ), 'error' );

        return false;

    }

    

    $xpath = new DOMXPath($doc);

    $imgs = $xpath->query("//img[1]");

    

    if( $imgs === false ){

        return false;

    }

    

    $img = $imgs->item( 0 );

    if( $img === null ){

        return false;

    }

    

    $src = $img->getAttribute('src');

    

    return $src;

}



/**

 * Sends error message to AJAX requests.

 * 

 * @param string $error_message Error message to send.

 */

function wp_newswire_ajax_error( $error_message ) {

    die( json_encode( array(

        'message' => $error_message,

        'status' => 'error'

    ) ) );

    

}



/**

 * Sends success message to AJAX requests.

 * 

 * @param string $success_message Success message to send.

 */

function wp_newswire_ajax_success( $success_message ) {

    die( json_encode( array(

        'message' => $success_message,

        'status' => 'success'

    ) ) );

}



/**

 * Returns an intance of WP_Newswire_Import_Manager based on the manager type.

 * 

 * @param string $manager_type rss | video | prwire

 * @param array $data Assoc array with necessary initialization data for the 

 * import manager.

 */

function wp_newswire_get_import_manager( $manager_type, $data ){

    switch( $manager_type ){

        case 'rss': 

            return new WP_Newswire_RSS_Import_Manager( $data['address'], $data['number'] , $data['keywords'] );

        case 'video':

            return new WP_Newswire_Video_Import_Manager( $data['number'] , $data['keywords'] );

        case 'prwire':

            return new WP_Newswire_PR_Wire_Import_Manager( $data['number'] , $data['keywords'] );

    } // pick appropriate importer

    

    return false;

}



function wp_newswire_get_setting( $setting_name ) {

    global $wp_newswire_settings_manager;

    if( isset( $wp_newswire_settings_manager->settings[$setting_name] ) ){

        return $wp_newswire_settings_manager->settings[$setting_name];

    }

}



function wp_newswire_count_seconds( $length, $unit ) {

    switch( $unit ){

        case 'min':

            return $length * 60;

        case 'hr':

            return $length * 60 * 60;

    }

    

    return $length;

}