<?php

/**
 * @package WP_Newswire
 * @subpackage Queue
 */

class WP_Newswire_Queue_Manager{
    
    public $update_time_interval = 60; // how many seconds to wait before another queue update
    public $chunks = 1; // how many items to fetch per one request
    public $queue_count = 0; // How many items are currently queued

# Initialization
    
    /**
     * Constructor.
     */
    function __construct() {
        $this->update_time_interval = WP_NEWSWIRE_QUEUE_TIME_INTERVAL;
        $this->chunks = WP_NEWSWIRE_QUEUE_CHUNKS;  
    }
    
    /**
     * Initializes all queue related modules.
     */
    public function init() {
        $this->register_queue_item_type();
        $this->update_queue_count();
        add_action( 'wp', array( 'WP_Newswire_Queue_Manager', 'setup_schedule' ) );        add_action( 'wpnwrqueueupdate', array( $this, 'update_queue' ) );
        add_filter( 'cron_schedules', array( $this, 'register_schedule_interval' ) );
        add_filter( 'heartbeat_send', array( $this, 'send_queue_count'), 10, 2 );
        add_filter( 'bulk_actions-edit-wp_nswr_queue_item', array( $this, 'bulk_actions' ) );
        add_filter( 'bulk_post_updated_messages', array( $this, 'bulk_post_updated_messages' ), 10, 2 );
        add_action( 'wp_ajax_wpnwr_approve_queued_posts', array( $this, 'approve_queued_posts') );
        add_action( 'publish_wp_nswr_queue_item', array( $this, 'change_post_type' ), 10, 2 );
    }
    
    public function change_post_type( $post_ID, $post_data ) {
        $this->approve_queue_item( $post_ID );
    }
    
    /**
     * Registers custom post type for queue items.
     */
    public function register_queue_item_type() {
        // @todo queue count into label through filters
        $queue_count_html = '<span class="wp_nwr_queue_count" style="color: rgb(255, 122, 0); font-style: italic;"></span>';
        
        $labels = array(
		'name'               => _x( 'Queue Items', 'post type general name', 'wp_newswire' ),
		'singular_name'      => _x( 'Queue Item', 'post type singular name', 'wp_newswire' ),
		'menu_name'          => _x( 'WP Newswire', 'admin menu', 'wp_newswire' ),
		'name_admin_bar'     => _x( 'WP Newswire', 'add new on admin bar', 'wp_newswire' ),
		'add_new'            => _x( 'Add New', 'queue item', 'wp_newswire' ),
		'add_new_item'       => __( 'Add New Queue Item', 'wp_newswire' ),
		'new_item'           => __( 'New Queue Item', 'wp_newswire' ),
		'edit_item'          => __( 'Edit Queue Item', 'wp_newswire' ),
		'view_item'          => __( 'View Queue Item', 'wp_newswire' ),
		'all_items'          => __( sprintf( 'Queue %s', $queue_count_html ), 'wp_newswire' ),
		'search_items'       => __( 'Search Queue Items', 'wp_newswire' ),
		'parent_item_colon'  => __( 'Parent Queue Items:', 'wp_newswire' ),
		'not_found'          => __( 'No queue items found.', 'wp_newswire' ),
		'not_found_in_trash' => __( 'No queue items in Trash.', 'wp_newswire' )
	);

	$args = array(
		'labels'             => $labels,
		'public'             => false,
		'publicly_queryable' => false,
		'show_ui'            => true,
		'show_in_menu'       => 'wp_newswire_add_rss',
		'query_var'          => true,
		'rewrite'            => array( 'slug' => 'queue_item' ),
		'capability_type'    => 'post',
                'capabilities' => array(
                    'create_posts' => false
                ),
                'map_meta_cap' => true,
		'has_archive'        => true,
		'hierarchical'       => false,
		'menu_position'      => WP_NEWSWIRE_QUEUE_MENU_POS,
		'supports'           => array( 'title', 'editor', 'thumbnail' )
	);

	register_post_type( 'wp_nswr_queue_item', $args );
    }
 
    
# Admin Display
    public function bulk_actions( $actions ) {
        unset( $actions['edit'] );
        return $actions;
    }
    
    public function bulk_post_updated_messages( $bulk_messages, $bulk_counts ) {
        if( ! isset( $_GET['wpnwr_status'] ) || ! isset( $_GET['updated'] ) ){
            return $bulk_messages;
        }
        $status = $_GET['wpnwr_status'];
        $messages = array(
            'error' => __( 'Some posts were not approved. Please try again.', 'wp_newswire' ),
            'success' => __( 'Posts approved.', 'wp_newswire' )
        );
        if( ! isset( $messages[$status] ) ){
            return $bulk_messages;
        }
        
        $bulk_messages['post']['updated'] = $messages[$status];
        return $bulk_messages;
    }
    
# Cron
    
    /**
     * Removes cron event for updating the queue / autoposting.
     */
    public static function reschedule_cron_event() {
        WP_Newswire_Queue_Manager::deschedule_cron_event();
        WP_Newswire_Queue_Manager::schedule_cron_event();
    }
       
    /**
     * Adds new cron event for updating the queue / autoposting.
     */
    public static function schedule_cron_event() {
        add_filter( 'cron_schedules', array( 'WP_Newswire_Queue_Manager', 'register_schedule_interval' ) );
        wp_schedule_event( time(), 'wp_nwr_interval', 'wpnwrqueueupdate' );
    }
    
    /**
     * Removes cron event. Should be called when plugin is deactivated.
     */
    public static function deschedule_cron_event() {
        wp_clear_scheduled_hook( 'wpnwrqueueupdate' );
    }
    
    /**
     * Double checks for the cron even being added.
     */
    public static function setup_schedule() {
        if ( ! wp_next_scheduled( 'wpnwrqueueupdate' ) ) {
            WP_Newswire_Queue_Manager::schedule_cron_event();
	}
    }
    
    /**
     * Registers time interval for the custom cron event for updating queue
     * / autoposting;
     * 
     * @param array $schedules Array of existing schedules.
     * @return array Array with new schedules.
     */
    public function register_schedule_interval( $schedules ) {
        $schedules['wp_nwr_interval'] = array(
            'interval' => WP_NEWSWIRE_QUEUE_TIME_INTERVAL,  
            'display' => __( 
                    sprintf( 'WP Newswire Queue Update: %d seconds', WP_NEWSWIRE_QUEUE_TIME_INTERVAL ),
                    'wp_newswire' )
            );

        return $schedules;
    }

# WP Heartbeat
    
    /**
     * Sends queue count to WP Heartbeat.
     * 
     * @global object $wp_newswire_queue_manager Queue manager.
     * @param array $response Data to be sent.
     * @param string $screen_id Current admin screen ID.
     * @return array New data to be sent.
     */
    public function send_queue_count( $response, $screen_id ) {
        global $wp_newswire_queue_manager;
        $response['wp-nwr-queue-count'] = $wp_newswire_queue_manager->queue_count;
        
        return $response;
    }

# Queue Update
    
    /**
     * Queues or autoposts feed items.
     * @todo use iterator on feed to fetch one at a time and start with next 
     * each time called
     * @return Array with newly added queue item IDs.
     */
    public function update_queue() {       
		global $wp_newswire_queue_manager;
        $queue_items = array(); // IDs for successfully fetched items
        $fetch_quantity = WP_NEWSWIRE_QUEUE_CHUNKS; // how many items to fetch
        
        $feed_iterator = new WP_Newswire_Feed_Iterator();
        $feed = $feed_iterator->current();       
		do{
            $import_manager = wp_newswire_get_import_manager( $feed->type, array(
                'address' => $feed->options['address'],
                'number' => $fetch_quantity,
                'keywords' => $feed->keywords
            ) );

            if( ! $import_manager || $import_manager->has_errors ){
                $feed = $feed_iterator->next();
                if( $this->empty_full_cycle( $feed_iterator, $fetch_quantity ) ){
                    break;
                }
                continue;
            } // could not init import manager
            
            $import_manager->filter_time = (int) mysql2date( 'U', $feed->time ); // feed's last fetch time
            $import_manager->prepare_items();	

            if( ! $import_manager->has_items() ){
                $feed = $feed_iterator->next();
                if( $this->empty_full_cycle( $feed_iterator, $fetch_quantity ) ){
                    break;
                }
                continue;
            }// no new items found

            // Import items to queue or autopost
            foreach( $import_manager->items as $item ){
                $item = $import_manager->prepare_item( $item, $feed->options );
                $queue_item = ( $feed->options['manually'] )
                        ? $wp_newswire_queue_manager->add( $item, $feed->options ) // add to queue
                        : $import_manager->insert_item( $item, $feed->options ); // autopost
                if( ! $queue_item ){
                    continue;
                } // could not add item to queue
                
                $queue_items[] = $queue_item; // store item ID
                $fetch_quantity --; // one more item fetched
            }
            
            if( ! empty( $queue_items ) ){
                $wp_newswire_queue_manager->update_queue_count( count( $queue_items ) );
                $feed->update_date( $item['date'] ); // update feed date
            } // update fetch date and queue count cache
            $feed = $feed_iterator->next();

            if( $this->empty_full_cycle( $feed_iterator, $fetch_quantity ) ){
                break;
            } // all feed up to date
        }
        while( $fetch_quantity > 0 );
        
        return $queue_items;
    }
    
    protected function empty_full_cycle( $feed_iterator, $fetch_quantity ) {
        return ( $feed_iterator->full_cycled && $fetch_quantity === WP_NEWSWIRE_QUEUE_CHUNKS );
    }
    
    /**
     * Given a count number will update cached queue_count variable of queue 
     * manager, else will fetch the count from database.
     * 
     * @param int $number Adds to current queue count. Can be a negative.
     */
    public function update_queue_count( $number = 0 ) {
        if( $number ){
            $this->queue_count += $number;
            return;
        }
        $this->queue_count = (int) wp_count_posts( 'wp_nswr_queue_item' )->pending;        
    }
    
    /**
     * Add item to queue.
     * 
     * @param array $item Prepared item.
     * @param array $args Extra arguments.
     */
    public function add( $item, $args ) {
        $args['_status'] = 'pending';
        $args['_type'] = 'wp_nswr_queue_item';
        $args['_hidden_insertion'] = true;
        return WP_Newswire_Import_Manager::insert_item( $item, $args );
    }
    
    /**
     * Approves current queue item and publishes the post.
     * 
     * @todo enable pages
     * @param int $post_ID Queue item ID to be approved.
     * @return int New post ID on success, 0 on failure.
     */
    public function approve_queue_item( $post_ID ) {
        $queue_meta = get_post_meta( $post_ID, 'wpnwr_queue_meta' );
        $post_type = isset( $queue_meta[0]['post_type'] ) ? $queue_meta[0]['post_type'] : 'post';
        $post_arr = array(
            'ID' => $post_ID,
            'post_status' => 'publish',
            'post_type' => $post_type
        ); // new post values

        $updated_post_ID = wp_update_post( $post_arr ); // try to update the post

        if( is_a( $updated_post_ID, 'WP_Error' ) ){
            return false;
        } // post not updated
        
        $cats = wp_set_post_categories( $updated_post_ID, array( $queue_meta[0]['post_category'] ) ); // try to update post category

        return $updated_post_ID;
    }
    
# AJAX
    
    public function approve_queued_posts() {
        if( empty( $_POST['ids']) ){
            wp_newswire_ajax_error( __( 'No data received.', 'wp_newswire' ) );
        }
        
        $not_approved = array();
        
        foreach ( (array) $_POST['ids'] as $post_id ){
            $updated = WP_Newswire_Queue_Manager::approve_queue_item( $post_id );
            if( ! $updated ){
                $not_approved[] = $post_id;
            }
        } // try to approve
        
        if( ! empty( $not_approved ) ){
            wp_newswire_ajax_error(
                    sprintf( __( 'The following posts failed to approve: %s', 'wp_newswire' ),
                            join(', ', $not_approved ) ) );
        } // error message for posts not approved
        
        wp_newswire_ajax_success( __( 'Posts successfully approved.', 'wp_newswire' ) ); // all approved
    }
}