<?php

/**

 * @package WP_Newswire

 * @subpackage Feed

 */



class WP_Newswire_Feed_Manager{

    

    /**

     * Gets all feed items from database as array of WP_Newswire_Feed_Item objects.

     * 

     * @global object $wpdb WP Database object.

     * @param string $type Feed type. Currently supported types are rss | video | prwire .

     * @param string $orderby Column name to order search results with.

     * @param string $order Order or results. ASC | DESC .

     * @return array\WP_Newswire_Feed_Item Array of WP_Newswire_Feed_Item objects.

     */

    public function get_all_items( $type = null, $orderby = null, $order = 'ASC' ) {

        global $wpdb;

        $items = array();

        $args = array();

        

        $query = " SELECT * FROM `" . WP_NEWSWIRE_TABLE_NAME . "`";

        if( $type !== null ){

            $query .= " WHERE `type` = %s";

            $args[] = $type;

        }

        if( $orderby !== null ){

            $query .= " ORDER BY `$orderby` $order";

        }

        $all_items = $wpdb->get_results( $wpdb->prepare( $query, $args ), ARRAY_A );

        

        foreach( (array) $all_items as $item_data ){

            $_item = new WP_Newswire_Feed_Item();

            $item_data['options'] = maybe_unserialize( $item_data['options'] );

            $_item->set_data($item_data);

            $items[] = $_item;

        }

                

        return $items;

    }

    

    public function get_feed_by_hash( $hash, $type ) {

        global $wpdb;

        $feed = $wpdb->get_results( $wpdb->prepare( 

                "SELECT * FROM `" . WP_NEWSWIRE_TABLE_NAME ."` WHERE `hash` = %s AND `type` = %s",

                array( $hash, $type ) ) );

        return $feed;

    }

    

    /**

     * Prepares POST-ed inputs for general feed items ( rss, videos etc )

     * 

     * @param array $data POST data

     * @return boolean | array Prepared data or false on failure. 

     */

    public function prepare_input( $data ){

        

        // Keywords sanitization

        $data['keywords'] = ( isset( $data['keywords'] ) && $data['keywords'] !== '' ) ? $data['keywords'] : '';



        // Length validation

        if ( ! preg_match( '/^[0-9]*$/', $data['length'] ) ) {

            wp_newswire_add_message( 

                    __( 'Please insert a number in "Snippet Length" field.', 'wp_newswire' ),

                    'error' );

            return false;

        }



        // Import validation

        if ( ! preg_match( '/^[0-9]*$/', $data['import_number'] ) ) {

            wp_newswire_add_message( 

                    __( 'Please insert a number in "Import" field.', 'wp_newswire' ),

                    'error' );

            return false;

        } 

        

        // Sanitize post type

        $allowed_post_types = array( 'post', 'page' );

        if( ! in_array( $data['post_type'], $allowed_post_types ) ){

            $data['post_type'] = 'post';

        }

        

        // Sanitize category

        $data['post_category'] = (int) $data['post_category'];



        // Sanitize checkboxes

        $data['include_images'] = (boolean) $data['include_images'];

        $data['manually'] = (boolean) $data['manually'];



        return $data;

    }

    

    /**

     * General function to handle new feed items ( validation, sanitization, 

     * insertion to database, import ).

     * 

     * @param string $feed_type Feed type [ rss | video | prwire | custom... ]

     * @param array $data Feed data

     * @return boolean True on success, false on failure.

     */

    public function add_feed( $feed_type, $data ) {
        $data = WP_Newswire_Feed_Manager::prepare_input( $data );

        if( $data === false ){
			
            return false;

        }

        

        extract($data);

        

        // Do import

        if( $import_number ){

            $import_manager = wp_newswire_get_import_manager( $feed_type, array(

                'address' => $data['address'],

                'number' => $import_number,

                'keywords' => $keywords

            ) );

            if( $import_manager->has_errors ){

                return false;

            } // Errors occured

            $import_manager->prepare_items();

            if( ! $import_manager->has_items() ){

                wp_newswire_add_message( 

                    __( 'No entries matching the specified criteria, try different keywords.', 'wp_newswire' ),

                    'error' );

            } // No items found

            else{

                $inserts = $import_manager->insert_all( $data );

                $errors = count( $import_manager->items ) - $inserts;

                if( $errors > 0 ){

                    wp_newswire_add_message( 

                        sprintf(__( '%d entries failed to save.', 'wp_newswire' ), $errors),

                        'error' );

                }

            }

        }

        

        // Do add feed item

        $item = new WP_Newswire_Feed_Item();

        $data['type'] = $feed_type;

        $data['posts'] = ( $import_number 

                && ! $import_manager->has_errors 

                && $import_manager->has_items() )

                ? $inserts : 0;

        $data['time'] = ( $import_number && $import_manager->last_inserted_item )

                ? $import_manager->last_inserted_item['date']

                : ( new DateTime( 'now' ) );

        

        $item->set_data( $data );

        if( $item->insert() === false ){

            wp_newswire_add_message( __( 'Feed could not be added.', 'wp_newswire' ), 'error' );

            return false;

        }

        return true;

    }



# RSS Feed

    

    /**

     * Checks RSS feed to be unique.

     * 

     * @param array $data RSS feed data.

     * @return boolean | array Sanitized data on success, false on failure.

     */

    public function add_rss_prepare_input( $data ) {

        // Address validation

        if( ! isset( $data['address'] ) || $data['address'] === '' ){

            wp_newswire_add_message( 

                    __( 'Please insert RSS feed address.', 'wp_newswire' ),

                    'error' );

            return false;

        }

        else{

            $feed = WP_Newswire_Feed_Manager::get_feed_by_hash( hash( 'md5', $data['address'] ), 'rss' );

            if( ! empty( $feed ) ){

                wp_newswire_add_message( 

                    __( 'This feed has already been added.', 'wp_newswire' ),

                    'error' );

                return false;

            }

        }

        

        $data['hash'] = $data['address']; // specify hash field

        $data['_options'] = array( 'address' => $data['address'] ); // extra data for options field

        

        return $data;

    }



    /**

     * Adds new RSS feed.

     * 

     * @param array $data Associative array with new RSS feed's data.

     * @return boolean True on success, false on failure.

     */

    public function add_rss_feed( $data ) {



        // Validate input

        $data = WP_Newswire_Feed_Manager::add_rss_prepare_input( $data );

        if( $data === false ){

            return false;

        }        

        

        return WP_Newswire_Feed_Manager::add_feed( 'rss', $data );      

    }



# Youtube Videos

    

    /**

     * Performs necessary checks and sanitization when keywords are the hash 

     * field for the feed.

     * 

     * @param array $data Feed data.

     * @param array $type Feed type ( rss | prwire | video )

     * @return boolean | array Sanitized data on success, false on failure.

     */

    public function keywords_hash( $data, $type ) {

        if( ! isset( $data['keywords'] ) || $data['keywords'] === '' ){

            wp_newswire_add_message( __( 'Please specify keywords.', 'wp_newswire' ), 'error' );

            return false;

        } // no keywords specified

        

        $feed = WP_Newswire_Feed_Manager::get_feed_by_hash( hash( 'md5', $data['keywords'] ), $type );

        if( ! empty( $feed ) ){

            wp_newswire_add_message( 

                __( 'This feed has already been added. Please try different keywords.', 'wp_newswire' ),

                'error' );

            return false;

        } // dublicate entries found

        

        $data['hash'] = $data['keywords']; // specify hash field

        

        return $data;

    }

    

    /**

     * Adds new video feed.

     * 

     * @param array $data Associative array with new video feed's data.

     * @return boolean True on success, false on failure.

     */

    public function add_video_feed( $data ) {



        // Validate input

        $data = WP_Newswire_Feed_Manager::keywords_hash( $data, 'video' );

        if( $data === false ){

            return false;

        }



        return WP_Newswire_Feed_Manager::add_feed( 'video', $data );      

    }

    

# PR Wire

    

    /**

     * Adds new PR Wire feed.

     * 

     * @param array $data Associative array with new video feed's data.

     * @return boolean True on success, false on failure.

     */

    public function add_prwire_feed( $data ) {

        // Validate input

        $data = WP_Newswire_Feed_Manager::keywords_hash( $data, 'prwire' );

        if( $data === false ){

		return false;

        }



        return WP_Newswire_Feed_Manager::add_feed( 'prwire', $data );      

    }

    

}